use std::borrow::Cow;

pub type Atom<'a> = Cow<'a, [u8]>;
pub type CowU8<'a> = Cow<'a, [u8]>;

#[derive(Clone, Debug)]
pub struct Tag<'a>(pub CowU8<'a>);

#[derive(Debug)]
#[non_exhaustive]
pub enum Response<'a> {
    Capabilities(Vec<Capability<'a>>),
    Continue(ResponseText<'a>),
    Condition(Condition<'a>),
    Done(ResponseDone<'a>),
    MailboxData(MailboxData<'a>),
    Fetch(u32, Vec<MessageAttribute<'a>>),
    Expunge(u32),
    Fatal(Condition<'a>),
    Tagged(Tag<'a>, Condition<'a>),
}

#[derive(Debug)]
pub struct ResponseText<'a> {
    pub code: Option<ResponseCode<'a>>,
    pub info: CowU8<'a>,
}

#[derive(Debug)]
pub enum MessageAttribute<'a> {
    Flags(Vec<Flag<'a>>),
    Envelope(Envelope),
}

#[derive(Debug)]
pub struct Envelope {}

#[derive(Debug)]
pub struct ResponseDone<'a> {
    pub tag: Tag<'a>,
    pub status: Status,
    pub code: Option<ResponseCode<'a>>,
    pub info: Option<Cow<'a, str>>,
}

#[derive(Debug)]
pub struct Condition<'a> {
    pub status: Status,
    pub code: Option<ResponseCode<'a>>,
    pub info: CowU8<'a>,
}

#[derive(Debug)]
pub enum Status {
    Ok,
    No,
    Bad,
    PreAuth,
    Bye,
}

#[derive(Debug)]
#[non_exhaustive]
pub enum ResponseCode<'a> {
    Alert,
    Capabilities(Vec<Capability<'a>>),
}

#[derive(Debug)]
pub enum Capability<'a> {
    Imap4rev1,
    Auth(Atom<'a>),
    Atom(Atom<'a>),
}

#[derive(Debug)]
pub enum MailboxData<'a> {
    Flags(Vec<Flag<'a>>),
    List(MailboxList<'a>),
}

#[derive(Debug)]
pub enum Mailbox<'a> {
    Inbox,
    Name(CowU8<'a>),
}

#[derive(Debug)]
pub enum Flag<'a> {
    Answered,
    Flagged,
    Deleted,
    Seen,
    Draft,
    Recent,
    Keyword(Atom<'a>),
    Extension(Atom<'a>),
}

#[derive(Debug)]
pub struct MailboxList<'a> {
    pub flags: Vec<MailboxListFlag<'a>>,
    pub delimiter: Option<u8>,
    pub mailbox: Mailbox<'a>,
}

#[derive(Debug)]
pub enum MailboxListFlag<'a> {
    NoInferiors,
    NoSelect,
    Marked,
    Unmarked,
    Extension(Atom<'a>),
}
