use std::ops::RangeFrom;

use nom::{
    error::{Error, ErrorKind, ParseError},
    Err, IResult, InputIter, Needed, Parser, Slice,
};

/// `sep_list!(t, d)` represents `t *(d t)` and automatically collapses it into `Vec<T>`.
///
/// Also `sep_list!(?t, d)` represents `[t *(d t)]` and will default to an empty vec.
///
/// If `d` is not passed then it defaults to `SP`.
macro_rules! sep_list {
    ($t:expr) => {
        map(
            pair($t, many0(preceded(crate::proto::rfc2234::SP, $t))),
            |(hd, mut tl)| {
                tl.insert(0, hd);
                tl
            },
        )
    };
    ($t:expr, $d:expr) => {
        map(pair($t, many0(preceded($d, $t))), |(hd, mut tl)| {
            tl.insert(0, hd);
            tl
        })
    };
    (? $t:expr) => {
        map(
            opt(pair($t, many0(preceded(crate::proto::rfc2234::SP, $t)))),
            |opt| {
                opt.map(|(hd, mut tl)| {
                    tl.insert(0, hd);
                    tl
                })
                .unwrap_or_else(Vec::new)
            },
        )
    };
    (? $t:expr, $d:expr) => {
        map(opt(pair($t, many0(preceded($d, $t)))), |opt| {
            opt.map(|(hd, mut tl)| {
                tl.insert(0, hd);
                tl
            })
            .unwrap_or_else(Vec::new)
        })
    };
}

/// Always fails, used as a no-op.
pub fn never<I, O>(i: I) -> IResult<I, O> { Err(Err::Error(Error::new(i, ErrorKind::Not))) }

/// Skip the part of the input matched by the given parser.
pub fn skip<E, F, I, O>(mut f: F) -> impl FnMut(I) -> IResult<I, (), E>
where
    I: Clone,
    F: Parser<I, O, E>,
{
    move |i: I| match f.parse(i.clone()) {
        Ok(_) => Ok((i, ())),
        Err(err) => Err(err),
    }
}

/// Same as nom satisfy, but operates on `&[u8]` instead of `&str`.
pub fn satisfy<F, I, E, T>(f: F) -> impl Fn(I) -> IResult<I, T, E>
where
    I: Slice<RangeFrom<usize>> + InputIter<Item = T>,
    F: Fn(T) -> bool,
    E: ParseError<I>,
    T: Copy,
{
    move |i: I| match i.iter_elements().next().map(|t| (f(t), t)) {
        Some((true, ft)) => Ok((i.slice(1..), ft)),
        Some((false, _)) => Err(Err::Error(E::from_error_kind(i, ErrorKind::Satisfy))),
        None => Err(Err::Incomplete(Needed::Unknown)),
    }
}

/// Match a single byte exactly.
pub fn byte<I, E: ParseError<I>>(b: u8) -> impl Fn(I) -> IResult<I, u8, E>
where
    I: Slice<RangeFrom<usize>> + InputIter<Item = u8>,
{
    satisfy(move |c| c == b)
}
