use anyhow::Result;
use futures::{future::FutureExt, stream::StreamExt};
use tokio::{
    io::{split, AsyncRead, AsyncWrite, ReadHalf, WriteHalf},
    sync::oneshot,
    task::JoinHandle,
};
use tokio_rustls::client::TlsStream;
use tokio_util::codec::FramedRead;

use crate::codec::ImapCodec;
use crate::proto::command::Command;

use super::upgrade::upgrade;

type ExitSender = oneshot::Sender<()>;
type ExitListener = oneshot::Receiver<()>;

pub struct Inner<C> {
    read_exit: ExitSender,
    read_handle: JoinHandle<ReadHalf<C>>,

    write_half: WriteHalf<C>,
}

impl<C> Inner<C>
where
    C: AsyncRead + AsyncWrite + Unpin + Send + 'static,
{
    pub async fn open(c: C) -> Result<Self> {
        // break the stream of bytes into a reader and a writer
        // the read_half represents the server->client connection
        // the write_half represents the client->server connection
        let (read_half, write_half) = split(c);

        // spawn the server->client loop
        let (read_exit, exit_rx) = oneshot::channel();
        let read_handle = tokio::spawn(read_loop(read_half, exit_rx));

        Ok(Inner {
            read_exit,
            read_handle,
            write_half,
        })
    }

    pub async fn execute<'a>(&mut self, _command: Command<'a>) {}

    pub async fn upgrade(self) -> Result<Inner<TlsStream<C>>> {
        // TODO: check that this capability exists??
        // issue exit to the read loop and retrieve the read half
        let _ = self.read_exit.send(());
        let read_half = self.read_handle.await?;
        let write_half = self.write_half;

        // put the read half and write half back together
        let stream = read_half.unsplit(write_half);
        let tls_stream = upgrade(stream, "hellosu").await?;

        Inner::open(tls_stream).await
    }
}

// exit is a channel that will notify this loop when some external
// even requires this loop to stop (for example, TLS upgrade).
//
// when the loop exits, the read half of the stream will be returned
async fn read_loop<C>(stream: ReadHalf<C>, exit: ExitListener) -> ReadHalf<C>
where
    C: AsyncRead,
{
    // set up framed communication
    let codec = ImapCodec::default();
    let mut framed = FramedRead::new(stream, codec);

    let exit = exit.fuse();
    pin_mut!(exit);
    loop {
        let next = framed.next().fuse();
        pin_mut!(next);

        select! {
            msg = next => {
                println!("hellosu {:?}", msg);
            }
            _ = exit => break,
        }
    }

    framed.into_inner()
}
