#![cfg_attr(not(feature = "std"), no_std)]

/// Edit this file to define custom logic or remove it if it is not needed.
/// Learn more about FRAME and the core library of Substrate FRAME pallets:
/// <https://substrate.dev/docs/en/knowledgebase/runtime/frame>
pub use pallet::*;

#[cfg(test)]
mod mock;

#[cfg(test)]
mod tests;

#[cfg(feature = "runtime-benchmarks")]
mod benchmarking;

#[frame_support::pallet]
pub mod pallet {
	use frame_support::dispatch::{DispatchResultWithPostInfo, Dispatchable};
	use frame_support::pallet_prelude::*;
	use frame_support::traits::schedule::*;
	use frame_system::pallet_prelude::*;

	#[pallet::config]
	pub trait Config: frame_system::Config {
		type Event: From<Event<Self>> + IsType<<Self as frame_system::Config>::Event>;

		type Proposal: Parameter + Dispatchable<Origin = Self::Origin> + From<Call<Self>>;
		type PalletsOrigin: From<frame_system::RawOrigin<Self::AccountId>>;
		type Scheduler: Anon<Self::BlockNumber, Self::Proposal, Self::PalletsOrigin>;

		#[pallet::constant]
		type Timeout: Get<Self::BlockNumber>;
	}

	#[pallet::pallet]
	#[pallet::generate_store(pub(super) trait Store)]
	pub struct Pallet<T>(_);

	#[pallet::storage]
	#[pallet::getter(fn pair_list)]
	pub type PairList<T: Config> = StorageMap<_, Blake2_128Concat, u32, T::AccountId>;

	#[pallet::event]
	#[pallet::metadata(T::AccountId = "AccountId")]
	#[pallet::generate_deposit(pub(super) fn deposit_event)]
	pub enum Event<T: Config> {
		/// finish pair. parameters: [PIN, sender1, sender2]
		Paired(u32, T::AccountId, T::AccountId),
		Timeout(u32),
	}

	#[pallet::error]
	pub enum Error<T> {
		Repeat,
	}

	#[pallet::hooks]
	impl<T: Config> Hooks<BlockNumberFor<T>> for Pallet<T> {}

	#[pallet::call]
	impl<T: Config> Pallet<T> {
		#[pallet::weight(10_000 + T::DbWeight::get().writes(1))]
		pub fn pair(origin: OriginFor<T>, pin: u32) -> DispatchResultWithPostInfo {
			let sender = ensure_signed(origin)?;
			if let Some(account) = <PairList<T>>::get(pin) {
				if account == sender {
					Err(Error::<T>::Repeat)?
				} else {
					<PairList<T>>::remove(pin);
					Self::deposit_event(Event::Paired(pin, sender, account));
					Ok(().into())
				}
			} else {
				<PairList<T>>::insert(pin, sender.clone());
				if T::Scheduler::schedule(
					DispatchTime::After(T::Timeout::get()),
					None,
					0,
					frame_system::RawOrigin::Root.into(),
					Call::timeout(pin).into(),
				)
				.is_err()
				{
					frame_support::print("scheduler cancel pin failed");
				}
				Ok(().into())
			}
		}

		#[pallet::weight(10_000 + T::DbWeight::get().writes(1))]
		/// PIN是私密的，永远返回成功，不会透露任何错误信息
		pub fn cancel(origin: OriginFor<T>, pin: u32) -> DispatchResultWithPostInfo {
			let sender = ensure_signed(origin)?;
			if let Some(account) = <PairList<T>>::get(pin) {
				if account == sender {
					<PairList<T>>::remove(pin);
					Ok(().into())
				} else {
					Ok(().into())
				}
			} else {
				Ok(().into())
			}
		}

		#[pallet::weight(10_000 + T::DbWeight::get().writes(1))]
		fn timeout(origin: OriginFor<T>, pin: u32) -> DispatchResultWithPostInfo {
			ensure_root(origin)?;
			<PairList<T>>::remove(pin);
			Self::deposit_event(Event::Timeout(pin));
			Ok(().into())
		}
	}
}
