use pktparse::{self, ethernet, ipv4::IPv4Header, ipv6::IPv6Header, arp::ArpPacket};
use pktparse::ip::IPProtocol;
use serde::{Serialize,Deserialize};


#[derive(Debug, Serialize, Deserialize, PartialEq)]
pub enum PktHeader {
    Ipv4(IPv4Header),
    Ipv6(IPv6Header),
    Arp(ArpPacket),
    Ether(ethernet::EthernetFrame),
    Tcp(pktparse::tcp::TcpHeader),
    Udp(pktparse::udp::UdpHeader),
}   

#[derive(Debug, Serialize, Deserialize)]
pub struct ParsedPkt {
    pub headers: Vec<PktHeader>,
    pub remaining: Vec<u8>,
}

impl ParsedPkt {
    pub fn new() -> ParsedPkt {
        ParsedPkt {
            headers: vec![],
            remaining: vec![],
        }
    }
}

impl ToString for PktHeader {
    fn to_string(&self) -> String {
        match self {
            PktHeader::Ipv4(_) => String::from("Ipv4"),
            PktHeader::Ipv6(_) => String::from("Ipv6"),
            PktHeader::Arp(_) => String::from("Arp"),
            PktHeader::Ether(_) => String::from("Ethernet"),
            PktHeader::Tcp(_) => String::from("TCP"),
            PktHeader::Udp(_) => String::from("UDP"),
        }
    }
}

#[derive(Clone,Copy,Debug)]
pub struct Parser {}

impl Parser {

    pub fn new() -> Parser {
        Parser {}
    }

    pub fn parse(&self, data: Vec<u8>) -> Result<ParsedPkt, String> {
        let parsed_pkt = self.parse_link(&data)?;
        Ok(parsed_pkt)
    } 

    pub fn parse_link(&self, data: &[u8]) -> Result<ParsedPkt, String> {
        let mut packet = ParsedPkt::new();
        match ethernet::parse_ethernet_frame(data) {
            Ok((content,headers)) => {
                match headers.ethertype {
                    pktparse::ethernet::EtherType::IPv4 => {
                        self.parse_ipv4_data(data, &mut packet)?
                    }
                    pktparse::ethernet::EtherType::IPv6 => {
                        self.parse_ipv6_data(data, &mut packet)?
                    }
                    pktparse::ethernet::EtherType::ARP => {
                        self.parse_arp_data(data, &mut packet)?
                    }
                    _ => {packet.remaining = content.to_owned();}
                }
                packet.headers.push(PktHeader::Ether(headers));
            }
            Err(_) => {packet.remaining = data.to_owned();}
        }
        Ok(packet)
    }   
    

    pub fn parse_ipv4_data(&self, content: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match pktparse::ipv4::parse_ipv4_header(content) {
            Err(err) => {
                packet.remaining = content.to_owned();
                Err(err.to_string())
            }
            Ok((_,headers)) => {
                self.parse_transports(&headers.protocol, content, packet)?;
                packet.headers.push(PktHeader::Ipv4(headers));
                Ok(())                
            }
        }        
    }

    pub fn parse_ipv6_data(&self, content: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match pktparse::ipv6::parse_ipv6_header(content) {
            Err(err) => {
                packet.remaining = content.to_owned();
                Err(err.to_string())
            }
            Ok((_,headers)) => {
                self.parse_transports(&headers.next_header, content, packet)?;
                packet.headers.push(PktHeader::Ipv6(headers));
                Ok(())                
            }
        }        
    }

    pub fn parse_arp_data(&self, content: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match pktparse::arp::parse_arp_pkt(content) {
            Err(err) => {
                packet.remaining = content.to_owned();
                Err(err.to_string())
            }
            Ok((_,headers)) => {
                packet.headers.push(PktHeader::Arp(headers));
                Ok(())                
            }
        }        
    }

    pub fn parse_transports(&self, protocol: &IPProtocol, data: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match protocol {
            IPProtocol::TCP => {
                self.parse_tcp_data(data, packet)?;
                Ok(())
            }
            IPProtocol::UDP => {
                self.parse_udp_data(data, packet)?;
                Ok(())
            }
            _ => {
                packet.remaining = data.to_owned();
                Ok(())
            }            
        }
    }

    pub fn parse_tcp_data(&self, content: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match pktparse::tcp::parse_tcp_header(content) {
            Err(err) => {
                packet.remaining = content.to_owned();
                Err(err.to_string())
            }
            Ok((_,headers)) => {
                packet.headers.push(PktHeader::Tcp(headers));
                Ok(())                
            }
        }        
    }

    pub fn parse_udp_data(&self, content: &[u8], packet: &mut ParsedPkt) -> Result<(), String> {
        match pktparse::udp::parse_udp_header(content) {
            Err(err) => {
                packet.remaining = content.to_owned();
                Err(err.to_string())
            }
            Ok((_,headers)) => {
                packet.headers.push(PktHeader::Udp(headers));
                Ok(())                
            }
        }        
    }
    
}