use pcap::{Active, Capture};
use pktparse;
use serde::{Serialize,Deserialize};
use crate::packets::parser;
use crate::packets;
use std::net::IpAddr;

#[derive(Debug,Serialize,Deserialize)]
pub struct IndiviualData {
    pub ipv4: pktparse::ipv4::IPv4Header,
    pub ipv6: pktparse::ipv6::IPv6Header,
    pub ether: pktparse::ethernet::EthernetFrame,
    pub arp: pktparse::arp::ArpPacket,    
}

#[derive(Debug,Serialize,Deserialize)]
pub struct SaveData {
    pub data: IndiviualData
}

#[derive(Debug,Serialize,Deserialize)]
pub struct JsonData {
    pub ipv4: Vec<Option<pktparse::ipv4::IPv4Header>>,
    pub ipv6: Vec<Option<pktparse::ipv6::IPv6Header>>,
    pub ether: Vec<Option<pktparse::ethernet::EthernetFrame>>,
    pub arp: Vec<Option<pktparse::arp::ArpPacket>>,
}

pub struct Handler {
    pub data: JsonData,
    pub parser: parser::Parser,
}

impl Handler {

    pub fn new() -> Handler {
        Handler {
            data: JsonData {
                ipv4: vec![],
                ipv6: vec![],
                ether: vec![],
                arp: vec![]     
            },
            parser: parser::Parser::new()
        }
    }


    pub fn get_packet_data(&self, packet: &packets::parser::ParsedPkt) -> (String, String, String, String) {
        let mut source_addr = "".to_string();
        let mut dest_addr = "".to_string();
        let mut source_port = "".to_string();
        let mut dest_port = "".to_string();

        packet.headers.iter().for_each(|pack|{
            match pack {
                parser::PktHeader::Tcp(packet) => {
                    source_port = packet.source_port.to_string();
                    dest_port = packet.dest_port.to_string();
                }
                parser::PktHeader::Udp(packet) => {
                    source_port = packet.source_port.to_string();
                    dest_port = packet.dest_port.to_string();
                }
                parser::PktHeader::Ipv4(packet) => {
                    source_addr = IpAddr::V4(packet.source_addr).to_string();
                    dest_addr = IpAddr::V4(packet.dest_addr).to_string();
                }
                parser::PktHeader::Ipv6(packet) => {
                    source_addr = IpAddr::V6(packet.source_addr).to_string();
                    dest_addr = IpAddr::V6(packet.dest_addr).to_string();
                }
                parser::PktHeader::Arp(packet) => {
                    source_addr = packet.src_addr.to_string();
                    dest_addr = packet.dest_addr.to_string();
                }
                _ => {}
            };
        });

        (source_addr, source_port, dest_addr, dest_port)
    }

    pub fn parse(&mut self, mut cap: Capture<Active>) {
        print!("\x1B[2J\x1B[1;1H");
        println!(
            "{0: <25} | {1: <15} | {2: <25} | {3: <15} | {4: <15}",
            "Source IP", "Source Port", "Dest IP", "Dest Port", "Protocol"
        );

        while let Ok(packet) = cap.next() {

            let parsed = self.parser.parse(packet.data.to_owned());
            match parsed {
                Ok(parsed) => {
                    let (source_addr, source_port, dest_addr, dest_port) = self.get_packet_data(&parsed);
                    println!(
                        "{0: <25} | {1: <15} | {2: <25} | {3: <15} | {4: <15}",
                        source_addr, source_port, dest_addr, dest_port, &parsed.headers[0].to_string()
                    );
                }
                Err(err) => {
                    //if err.to_lowercase() != "didn't match tcp or udp" {
                    println!("{}", err.to_string());
                    // } else {
                    //     println!("");
                    // }
                }
            }
        }
    }
}