use anyhow::Result;
use serde::Deserialize;
use std::collections::HashMap;
use std::{env, fs, process};
use url::Url;

#[derive(Deserialize)]
struct PackageLock {
    requires: bool,
    #[serde(rename = "lockfileVersion")]
    lockfile_version: u32,
    dependencies: HashMap<String, Dependency>,
}

impl PackageLock {
    fn check(&self) -> Vec<String> {
        self.dependencies
            .iter()
            .flat_map(|(name, dep)| dep.check(name))
            .collect()
    }
}

#[derive(Deserialize)]
struct Dependency {
    version: String,
    from: Option<String>,
    resolved: Option<String>,
    integrity: Option<String>,
    #[serde(default)]
    dev: bool,
    #[serde(default)]
    requires: HashMap<String, String>,
    #[serde(default)]
    dependencies: HashMap<String, Dependency>,
}

impl Dependency {
    fn check(&self, name: &str) -> Vec<String> {
        let mut resp = vec![];
        // DO CHECK
        if let Some(msg) = validate(name, self) {
            resp.push(format!("{}@{}: {}", name, self.version, msg));
        }

        for (name, dep) in self.dependencies.iter() {
            resp.extend(dep.check(name));
        }

        resp
    }
}

fn validate(_name: &str, dep: &Dependency) -> Option<String> {
    // If resolved is present, we check it first and rely on it
    if let Some(resolved) = &dep.resolved {
        let url = match Url::parse(resolved) {
            Ok(url) => url,
            Err(e) => {
                return Some(format!(
                    "\"resolved\" is not a valid URL: \"{}\" ({})",
                    resolved,
                    e.to_string()
                ))
            }
        };
        if url.scheme() != "https" {
            return Some(format!("\"resolved\" does not use HTTPS: {}", resolved));
        }
        if url.host_str() != Some("registry.npmjs.org") {
            return Some(format!(
                "\"resolved\" is not from registry.npmjs.org: {}",
                resolved
            ));
        }
        // resolved is OK, so this whole dep is OK.
        return None;
    }

    // version might be a valid URL
    if let Ok(url) = Url::parse(&dep.version) {
        if url.scheme() == "file" {
            // Part of repository, OK
            return None;
        }
        // git+ssh is bad but it's probably not insecure
        if !["git+https", "git+ssh", "https", "github"].contains(&url.scheme()) {
            return Some(format!("\"version\" does not use HTTPS: {}", url));
        }

        // TODO: Add some domain validation
    }
    None
}

fn main() -> Result<()> {
    let mut errors = false;
    for (index, path) in env::args().enumerate() {
        if index == 0 {
            continue;
        }
        let json: PackageLock = serde_json::from_str(&fs::read_to_string(&path)?)?;
        let deps = json.check();
        println!("{} issues found in: {}", deps.len(), &path);
        if !deps.is_empty() {
            println!("{}", deps.join("\n"));
            errors = true;
        }
    }
    if errors {
        process::exit(1);
    }
    Ok(())
}
