use elliptic_curve::generic_array::{ArrayLength, GenericArray};
use rand_core::{CryptoRng, RngCore};

pub use crate::{Error, Result};

pub trait System {
    type SecretKey: SecretKey;
}

pub trait SecretKey: Sized {
    type SIZE: ArrayLength<u8>;
    type PublicKey;
    type Signature;
    type SharedSecret;

    fn random(rng: impl CryptoRng + RngCore);

    fn public_key(&self) -> Self::PublicKey;

    fn from_bytes(bytes: impl AsRef<[u8]>) -> Result<Self>;
    fn to_bytes(bytes: impl AsRef<[u8]>) -> GenericArray<u8, Self::SIZE>;

    fn sign_prehashed(&self, prehashed_message: &[u8], rng: impl CryptoRng + RngCore) -> Self::Signature;
    fn sign(&self, message: &[u8], rng: impl CryptoRng + RngCore) -> Self::Signature;

    fn agree(&self, other: &Self::PublicKey) -> SharedSecret;
}

pub trait PublicKey: Sized {
    type SIZE: ArrayLength<u8>;
    type Signature;

    fn from_bytes(bytes: impl AsRef<[u8]>) -> Result<Self>;
    fn to_bytes(&self) -> GenericArray<u8, Self::SIZE>;

    fn verify_prehashed(&self, prehashed_message: &[u8], signature: &Self::Signature) -> Result<()>;
    fn verify(&self, message: &[u8], signature: &Self::Signature) -> Result<()>;
}
