/* entrypoint.rs
 *
 * Developed by Tim Walls <tim.walls@snowgoons.com>
 * Copyright (c) All Rights Reserved, Tim Walls
 */
//! Procedural macros to generate the main entry point for an AVRoxide
//! application.

// Imports ===================================================================
use proc_macro::TokenStream;
use proc_macro2::Span;
use quote::quote;

// Declarations ==============================================================
pub struct EntrypointConfig {
  _naked: bool,
  chip: String,
}

// Code ======================================================================
pub fn generate_main(args: TokenStream, item: TokenStream) -> TokenStream {
  let args = syn::parse_macro_input!(args as syn::AttributeArgs);
  let config = parse_args(args);

  match config {
    Ok(config) => {
      let input = syn::parse_macro_input!(item as syn::ItemFn);

      // Naked main function; just do enough to hook into the bootloader
      // @todo in future I plan to make this optional, with the default being
      //       something a little fancier that also takes care of the Oxide
      //       event loop initialisation
      if input.sig.ident != "main"
        || !input.sig.inputs.is_empty() {
        return syn::Error::new_spanned(&input.sig.ident, "the main function must have the signature `fn main() -> u8`").to_compile_error().into();
      }

      let bootlibname = format!("oxide-boot-{}", config.chip);
      let main_body = &input.block;


      let result = quote! {
        #[link(name=#bootlibname, kind="static")]
        extern {}
        #[no_mangle]
        pub fn __oxide_main() -> u8 #main_body
      };
      result.into()
    },
    Err(e) => {
      e.to_compile_error().into()
    }
  }
}

fn parse_string_literal(string: syn::Lit) -> Result<String,syn::Error> {
  match string {
    syn::Lit::Str(s) => Ok(s.value()),
    syn::Lit::Verbatim(s) => Ok(s.to_string()),
    _ => Err(syn::Error::new_spanned(string, "Cannot parse value as string"))
  }
}

pub fn parse_args(args: syn::AttributeArgs) -> Result<EntrypointConfig,syn::Error> {
  let mut naked:bool    = false;
  let mut chip:Option<String> = None;

  for arg in args {
    match arg {
      syn::NestedMeta::Meta(syn::Meta::NameValue(namevalue)) => {
        let key = namevalue.path.get_ident().unwrap().to_string().to_lowercase();
        match key.as_str() {
          "chip" => {
            chip = Some(parse_string_literal(namevalue.lit)?);
          },
          unknown => {
            return Err(syn::Error::new_spanned(namevalue, format!("Unknown attribute {}", unknown)));
          }
        }
      },
      syn::NestedMeta::Meta(syn::Meta::Path(path)) => {
        let key = path.get_ident().unwrap().to_string().to_lowercase();
        match key.as_str() {
          "naked" => {
            naked = true;
          },
          "chip" => {
            return Err(syn::Error::new_spanned(path, format!("The {} attribute requires a value", key)));
          },
          unknown => {
            return Err(syn::Error::new_spanned(path, format!("Unknown attribute {}", unknown)));
          }
        }
      },
      unknown => {
        return Err(syn::Error::new_spanned(unknown, "Unknown attribute"));
      }
    }
  }

  if chip.is_none() {
    return Err(syn::Error::new(Span::call_site(), "Chip variant must be specified"));
  }

  Ok(EntrypointConfig {
    _naked: naked,
    chip: chip.unwrap()
  })
}

// Tests =====================================================================
#[cfg(test)]
mod tests {
  #[allow(unused_imports)]
  use super::*;

  #[test]
  fn a_test() {}

  #[test]
  #[should_panic]
  fn a_failure_test() {
    panic!()
  }
}