use std::fmt::{Display, Formatter, Result as FmtResult};
use std::time::Duration;

use chrono::prelude::*;
use reqwest::blocking::Client;
use serde::{Deserialize, Serialize};
use structopt::StructOpt;
use sun_times::*;
use thiserror::Error;

#[derive(Error, Debug)]
pub enum AppError {
    #[error("failed request")]
    RequestError(#[from] reqwest::Error),
}

#[derive(Debug, Serialize, Deserialize, StructOpt)]
pub struct Options {
    #[structopt(short = "k", long = "api-key")]
    api_key: String,
    #[structopt(
        short = "u",
        long = "url-root",
        default_value = "api.openweathermap.org/data/2.5/weather"
    )]
    url_root: String,
    #[structopt(short = "f", long = "icon-font", default_value = "material-icons")]
    icon_font: IconFont,
}

#[derive(Debug, Serialize, Deserialize, Eq, PartialEq)]
pub enum IconFont {
    MaterialIcons,
}

impl IconFont {
    fn sun_icon(&self) -> &'static str {
        match self {
            Self::MaterialIcons => "\u{e518}",
        }
    }

    fn moon_icon(&self) -> &'static str {
        match self {
            Self::MaterialIcons => "\u{ef5e}",
        }
    }

    fn cloud_icon(&self) -> &'static str {
        match self {
            Self::MaterialIcons => "\u{e42d}",
        }
    }

    fn rain_icon(&self) -> &'static str {
        match self {
            Self::MaterialIcons => "\u{f1ad}",
        }
    }
}

impl std::str::FromStr for IconFont {
    type Err = &'static str;

    fn from_str(val: &str) -> Result<Self, Self::Err> {
        match val {
            "material-icons" => Ok(IconFont::MaterialIcons),
            _ => Err("Could not parse icon font name"),
        }
    }
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct GeoIpResponse {
    city: String,
    lat: f32,
    lon: f32,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct OpenWeatherMapResponse {
    weather: Vec<WeatherField>,
    main: TemperatureField,
}

pub struct OfflineWeather {
    city_name: String,
}

impl From<GeoIpResponse> for OfflineWeather {
    fn from(geoip: GeoIpResponse) -> Self {
        Self {
            city_name: geoip.city,
        }
    }
}

impl Display for OfflineWeather {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        write!(f, "{}, 666 ( revelations)", self.city_name)
    }
}

#[derive(Debug)]
pub struct Weather {
    key: String,
    lat: f32,
    lon: f32,
    city_name: String,
    temperature: f32,
    icon_font: IconFont,
}

impl From<(GeoIpResponse, TemperatureField, WeatherField, IconFont)> for Weather {
    fn from(
        (geoip, temp, weather, icon_font): (
            GeoIpResponse,
            TemperatureField,
            WeatherField,
            IconFont,
        ),
    ) -> Self {
        Weather {
            key: weather.description,
            lat: geoip.lat,
            lon: geoip.lon,
            city_name: geoip.city,
            temperature: temp.temp,
            icon_font,
        }
    }
}

impl Weather {
    pub fn celsius(&self) -> f32 {
        (self.temperature - 273.15).ceil()
    }

    pub fn is_day(&self) -> bool {
        let (start, end) = sun_times(Utc::today(), self.lat.into(), self.lon.into(), 1.0);
        let now = Utc::now();

        now >= start && now < end
    }
}

impl Display for Weather {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        let star_icon = if self.is_day() {
            self.icon_font.sun_icon()
        } else {
            self.icon_font.moon_icon()
        };
        let cloud_icon = self.icon_font.cloud_icon();

        let icon = match &*self.key {
            "clear sky" => star_icon,
            "few clouds" => star_icon,
            "broken clouds" => cloud_icon,
            "mist" => cloud_icon,
            "overcast clouds" => cloud_icon,
            "scattered clouds" => cloud_icon,
            "haze" => cloud_icon,
            "light rain" => self.icon_font.rain_icon(),
            _ => &self.key,
        };

        write!(
            f,
            "{}, {} °C ({} {})",
            self.city_name,
            self.celsius(),
            icon,
            self.key
        )
    }
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct WeatherField {
    main: String,
    description: String,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct TemperatureField {
    temp: f32,
    feels_like: f32,
    temp_min: f32,
    temp_max: f32,
    pressure: f32,
    humidity: f32,
}

fn get_location(_opts: &Options, client: &Client) -> Result<GeoIpResponse, AppError> {
    Ok(client.get("http://ip-api.com/json/").send()?.json()?)
}

fn get_weather(
    opts: &Options,
    client: &Client,
    loc: &GeoIpResponse,
) -> Result<OpenWeatherMapResponse, AppError> {
    let url = &format!(
        "https://{}?lat={}&lon={}&appid={}",
        opts.url_root, loc.lat, loc.lon, opts.api_key
    );

    Ok(client.get(url).send()?.json()?)
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
    env_logger::init();

    let client = Client::builder().timeout(Duration::from_secs(2)).build()?;
    let options = Options::from_args();

    let loc = match get_location(&options, &client) {
        Ok(loc) => loc,
        Err(e) => panic!("Failed to get user location from ip-api.com: {:?}", e),
    };

    let weather = match get_weather(&options, &client, &loc) {
        Ok(weather) => weather,
        Err(_) => {
            let weather: OfflineWeather = loc.into();
            println!("{}", weather);

            return Ok(());
        }
    };
    let w_field = weather.weather[0].clone();

    let weather: Weather = (loc, weather.main, w_field, options.icon_font).into();

    println!("{}", weather);

    Ok(())
}
