use std::fmt::{Display, Formatter, Result as FmtResult};
use std::time::Duration;

use chrono::prelude::*;
use reqwest::blocking::Client;
use serde::{Deserialize, Serialize};
use structopt::StructOpt;
use sun_times::*;
use thiserror::Error;

#[derive(Error, Debug)]
pub enum AppError {
    #[error("failed request")]
    RequestError(#[from] reqwest::Error),
}

#[derive(Debug, Serialize, Deserialize, StructOpt)]
pub struct Options {
    #[structopt(short = "k", long = "api-key")]
    api_key: String,
    #[structopt(
        short = "u",
        long = "url-root",
        default_value = "api.openweathermap.org/data/2.5/weather"
    )]
    url_root: String,
}

#[derive(Serialize, Deserialize, Debug)]
#[serde(rename_all = "camelCase")]
pub struct GeoIpResponse {
    city: String,
    lat: f32,
    lon: f32,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct OpenWeatherMapResponse {
    weather: Vec<WeatherField>,
    main: TemperatureField,
}

pub struct OfflineWeather {
    city_name: String,
}

impl From<GeoIpResponse> for OfflineWeather {
    fn from(geoip: GeoIpResponse) -> Self {
        Self {
            city_name: geoip.city,
        }
    }
}

impl Display for OfflineWeather {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        write!(f, "{}, 666 ( revelations)", self.city_name)
    }
}

#[derive(Debug)]
pub struct Weather {
    name: String,
    key: String,
    lat: f32,
    lon: f32,
    city_name: String,
    temperature: f32,
}

impl From<(GeoIpResponse, TemperatureField, WeatherField)> for Weather {
    fn from((geoip, temp, weather): (GeoIpResponse, TemperatureField, WeatherField)) -> Self {
        Weather {
            name: weather.main,
            key: weather.description,
            lat: geoip.lat,
            lon: geoip.lon,
            city_name: geoip.city,
            temperature: temp.temp,
        }
    }
}

impl Weather {
    pub fn celsius(&self) -> f32 {
        (self.temperature - 273.15).ceil()
    }

    pub fn is_day(&self) -> bool {
        let (start, end) = sun_times(Utc::today(), self.lat.into(), self.lon.into(), 1.0);
        let now = Utc::now();

        now >= start && now < end
    }
}

impl Display for Weather {
    fn fmt(&self, f: &mut Formatter) -> FmtResult {
        let star_icon = if self.is_day() { "" } else { "" };
        let cloud_icon = "";

        let icon = match &*self.key {
            "clear sky" => star_icon,
            "few clouds" => star_icon,
            "broken clouds" => cloud_icon,
            "mist" => cloud_icon,
            "overcast clouds" => cloud_icon,
            "scattered clouds" => cloud_icon,
            "haze" => cloud_icon,
            "light rain" => "",
            _ => &self.key,
        };

        write!(
            f,
            "{}, {} ({} {})",
            self.city_name,
            self.celsius(),
            icon,
            self.key
        )
    }
}

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct WeatherField {
    main: String,
    description: String,
}

#[derive(Serialize, Deserialize, Debug)]
pub struct TemperatureField {
    temp: f32,
    feels_like: f32,
    temp_min: f32,
    temp_max: f32,
    pressure: f32,
    humidity: f32,
}

fn get_location(_opts: &Options, client: &Client) -> Result<GeoIpResponse, AppError> {
    Ok(client.get("http://ip-api.com/json/").send()?.json()?)
}

fn get_weather(
    opts: &Options,
    client: &Client,
    loc: &GeoIpResponse,
) -> Result<OpenWeatherMapResponse, AppError> {
    let url = &format!(
        "https://{}?lat={}&lon={}&appid={}",
        opts.url_root, loc.lat, loc.lon, opts.api_key
    );

    Ok(client.get(url).send()?.json()?)
}

fn main() -> Result<(), Box<dyn std::error::Error>> {
    env_logger::init();

    let client = Client::builder().timeout(Duration::from_secs(2)).build()?;
    let options = Options::from_args();

    let loc = match get_location(&options, &client) {
        Ok(loc) => loc,
        Err(e) => panic!("Failed to get user location from ip-api.com: {:?}", e),
    };

    let weather = match get_weather(&options, &client, &loc) {
        Ok(weather) => weather,
        Err(_) => {
            let weather: OfflineWeather = loc.into();
            println!("{}", weather);

            return Ok(());
        }
    };
    let w_field = weather.weather[0].clone();

    let weather: Weather = (loc, weather.main, w_field).into();

    println!("{}", weather);

    Ok(())
}
