use crate::utils::filter_rs;
use colored::ColoredString;
use colored::Colorize;
use console::{Key, Term};
use core::fmt;
use derive_builder::Builder;
use dialoguer::theme::Theme;
use itertools::Itertools;
use lazy_static::lazy_static;
use std::collections::BTreeSet;
use std::fmt::Display;
use std::io;
use std::path::Path;
use std::path::PathBuf;
use thiserror::Error;
use walkdir::WalkDir;

lazy_static! {
    static ref STARTED: ColoredString = "✗".yellow();
    static ref COMPLETED: ColoredString = "✓".green();
}

#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub enum Status {
    NotStarted,
    Started,
    Completed,
}

impl Display for Status {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        use Status::*;
        match *self {
            NotStarted => Ok(()),
            Started => write!(f, "{}", *STARTED),
            Completed => write!(f, "{}", *COMPLETED),
        }
    }
}

/// An [`Item`] represents an item in the menu
pub trait Item {
    fn file_name(&self) -> &str;

    fn folder_name(&self) -> &str;

    fn status(&self) -> Status;
}

/// Lists exercises in the folder at `path`. Finds `.rs` files at
/// exactly two folders of depth. Returns a list of paths **without**
/// the `path` as a prefix.
///
/// # Examples
///
/// ```
/// # use otarustlings::menu::list_exercises;
/// let mut exercises = list_exercises("exercises");
///
/// // Should find `exercises/week1/01-quiz-compile.rs` first
/// assert_eq!(exercises.next(), Some("week1/01-quiz-compile.rs".into()));
/// ```
pub fn list_exercises(path: impl AsRef<Path>) -> impl Iterator<Item = PathBuf> {
    WalkDir::new(&path)
        .min_depth(2)
        .max_depth(2)
        .sort_by_file_name()
        .into_iter()
        .filter_map(Result::ok)
        .map(move |e| {
            e.into_path()
                .strip_prefix(&path)
                .expect("DirEntry to start with `path`")
                .to_owned()
        })
        .filter(|pb| filter_rs(pb))
}

/// Errors which [`Menu::interact`]
#[derive(Error, Debug)]
pub enum InteractionError {
    /// No items in the `Menu`
    #[error("no exercises found")]
    EmptyItems,

    /// An IO error happened
    #[error("IO error")]
    IOError(#[from] io::Error),
}

/// Otarustlings menu system
///
/// ## Specification
///
/// When selecting exercises only one folder (week) is shown at a time
///
///
/// ```ignore
/// week1
///     exercise1 DONE
///   > exercise2
/// week2
/// week3
/// ```
///
/// Moving down will result in week1 collapsing and week2 expanding
///
/// ```ignore
/// week1
/// week2
///   > hard_exercise
///     harder_quiz
/// week3
/// ```
///
///
#[derive(Clone, Builder)]
pub struct Menu<'menu, T>
where
    T: Item + PartialEq + fmt::Debug,
{
    #[builder(default = "0")]
    default_index: usize,
    items: &'menu BTreeSet<T>,
    theme: &'menu dyn Theme,
}

impl<'menu, T> Menu<'menu, T>
where
    T: Item + PartialEq + fmt::Debug,
{
    /// Enables user interaction and returns the index of the item in `self.items`.
    ///
    /// The user can select the items with the 'Space' bar or 'Enter'
    /// and the index of selected item will be returned.  The dialog
    /// is rendered on stderr.  Result contains `index` if user
    /// selected one of items using 'Enter'.
    ///
    /// # Errors
    ///
    /// - Returns a [`InteractionError::EmptyItems`] if the menu contains no items.
    /// - Returns a [`InteractionError::IOError`] if
    ///   - writing to the terminal fails
    ///   - reading a key from the terminal fails
    pub fn interact(&self, term: &Term) -> Result<Option<usize>, InteractionError> {
        if self.items.is_empty() {
            return Err(InteractionError::EmptyItems);
        }

        let mut render = TermThemeRenderer::new(term, self.theme);
        // Set the current selection to start at the default value
        let mut sel = self.default_index;

        term.hide_cursor()?;

        // Loop until user selects the item or exits
        loop {
            // gets the folder and file indexed by `sel`
            let selected_exercise = self.items.iter().nth(sel).expect("sel to be in range");

            let grouped = self.items.iter().group_by(|t| t.folder_name());

            // loop over each folder
            for (folder, exercises_in_folder) in &grouped {
                let exercises_in_folder: Vec<_> = exercises_in_folder.collect();

                let folder_statuses: Vec<_> =
                    exercises_in_folder.iter().map(|e| e.status()).collect();
                let contains_not_started = folder_statuses.contains(&Status::NotStarted);
                let contains_started = folder_statuses.contains(&Status::Started);
                let contains_completed = folder_statuses.contains(&Status::Completed);

                let folder_status =
                    match (contains_not_started, contains_started, contains_completed) {
                        // Folder contains no started or completed exercises
                        (true, false, false) => Status::NotStarted,
                        // Folder contains at least one not started exercises while not all beign not started
                        (true, _, _) => Status::Started,
                        // Folder contains at least one started exercise amidst only completed exercises
                        (false, true, _) => Status::Started,
                        (false, false, true) => Status::Completed,
                        (false, false, false) => unreachable!("no exercises in folder"),
                    };
                // render the week line always in non highlight mode
                render.render_item(&format!("{} {}", folder, folder_status), false)?;

                // if the week is the selected week
                if selected_exercise.folder_name() == folder {
                    // loop over all exercises in the week
                    for exercise in exercises_in_folder {
                        render.render_item(
                            &format!("  {} {}", exercise.file_name(), exercise.status()),
                            selected_exercise == exercise,
                        )?;
                    }
                }
            }

            // ensure everything that was just rendered is
            term.flush()?;

            // TODO implement actions elsewhere
            // let (next_sel, should_clear) = get_next_sel(term.read_key()?)?;
            match term.read_key()? {
                Key::ArrowDown | Key::Tab | Key::Char('j') => {
                    if sel == !0 {
                        sel = 0;
                    } else {
                        sel = (sel + 1) % self.items.len();
                    }
                }
                Key::Escape | Key::Char('q') => {
                    render.clear()?;

                    term.show_cursor()?;
                    term.flush()?;

                    return Ok(None);
                }
                Key::ArrowUp | Key::BackTab | Key::Char('k') => {
                    if sel == !0 {
                        sel = self.items.len() - 1;
                    } else {
                        sel = (sel + self.items.len() - 1) % (self.items.len());
                    }
                }

                Key::Enter | Key::Char(' ') if sel != !0 => {
                    render.clear()?;
                    term.show_cursor()?;
                    term.flush()?;

                    return Ok(Some(sel));
                }
                _ => {}
            }

            render.clear()?;
        }
    }
}

/// This renderer allows clearing all previously written lines
pub struct TermRenderer<'term> {
    term: &'term Term,
    height: usize,
}
impl<'a> TermRenderer<'a> {
    pub fn new(term: &'a Term) -> Self {
        Self { term, height: 0 }
    }

    pub fn write_formatted_line<F: FnOnce(&mut Self, &mut dyn fmt::Write) -> fmt::Result>(
        &mut self,
        f: F,
    ) -> io::Result<()> {
        let mut buf = String::new();
        f(self, &mut buf).map_err(|err| io::Error::new(io::ErrorKind::Other, err))?;
        self.height += buf.chars().filter(|&x| x == '\n').count() + 1;
        self.term.write_line(&buf)
    }

    /// Clears the previously rendered text.
    ///
    /// # Errors
    ///
    /// Return an [`io::Error`] if writing to the terminal fails.
    pub fn clear(&mut self) -> io::Result<()> {
        self.term.clear_last_lines(self.height)?;
        self.height = 0;
        Ok(())
    }
}

/// A terminal renderer that keeps track of the number of lines that
/// is currently rendered.
pub struct TermThemeRenderer<'a> {
    term: &'a Term,
    theme: &'a dyn Theme,
    height: usize,
}

impl<'a> TermThemeRenderer<'a> {
    pub fn new(term: &'a Term, theme: &'a dyn Theme) -> TermThemeRenderer<'a> {
        TermThemeRenderer {
            term,
            theme,
            height: 0,
        }
    }

    pub fn write_formatted_line<
        F: FnOnce(&mut TermThemeRenderer, &mut dyn fmt::Write) -> fmt::Result,
    >(
        &mut self,
        f: F,
    ) -> io::Result<()> {
        let mut buf = String::new();
        f(self, &mut buf).map_err(|err| io::Error::new(io::ErrorKind::Other, err))?;
        self.height += buf.chars().filter(|&x| x == '\n').count() + 1;
        buf = buf.replace('\n', "\r\n");
        self.term.write_line(&buf)
    }

    /// Renders `text` to the terminal. The line is highlighted if
    /// `active` is set.
    ///
    /// # Errors
    ///
    /// Returns an [`io::Error`] if writing to the terminal fails.
    pub fn render_item(&mut self, text: &str, active: bool) -> io::Result<()> {
        self.write_formatted_line(|this, buf| {
            this.theme.format_select_prompt_item(buf, text, active)
        })
    }

    /// Clears the previously rendered text.
    ///
    /// # Errors
    ///
    /// Return an [`io::Error`] if writing to the terminal fails.
    pub fn clear(&mut self) -> io::Result<()> {
        self.term.clear_last_lines(self.height)?;
        self.height = 0;
        Ok(())
    }
}
