use std::{
    fmt::{Display, Formatter},
    fs::{create_dir_all, File},
    io::{self, Write},
    path::Path,
    process,
    process::Command,
};

use rust_embed::RustEmbed;
use thiserror::Error;

use crate::{tempfile::TempFile, RUSTC_COLOR_ARGS};

/// `Output` contains stdout and stderr strings of some executed process.
#[derive(Debug, Clone)]
pub struct Output {
    pub stdout: String,
    pub stderr: String,
}

impl Output {
    pub fn from_cmd_output(output: &process::Output) -> Self {
        Self {
            stdout: String::from_utf8_lossy(&output.stdout).to_string(),
            stderr: String::from_utf8_lossy(&output.stderr).to_string(),
        }
    }
}

impl Display for Output {
    fn fmt(&self, formatter: &mut Formatter) -> Result<(), std::fmt::Error> {
        write!(formatter, "{}", self.stdout)
    }
}

/// Represents different errors from [`compile`]
#[derive(Error, Debug)]
pub enum CompileError {
    /// Compilation failed due to an IO error
    #[error("compilation IO error")]
    IOError(#[from] io::Error),

    /// Compilation error which contains compiler output
    #[error("compilation failed")]
    OutputError(Output),
}

/// Represents different errors from [`run`]
#[derive(Error, Debug)]
pub enum RunError {
    /// Execution failed due to an IO error
    #[error("execution IO error")]
    IOError(#[from] io::Error),

    /// Execution failed at runtime. For example the tests may have
    /// failed or the application may have crashed
    #[error("execution failed")]
    OutputError(Output),
}

/// Compiles the Rust source code file with `rustc` and returns a
/// [`TempFile`] guarding the resultant binary.
///
/// # Panics
///
/// Panics if path contains invalid UTF-8.
///
/// # Errors
///
/// If `rustc` exited with non-zero status, it is assumed to be a
/// compilation IO error [`CompileError::IOError`] which is then returned.
pub fn compile(path: impl AsRef<Path>) -> Result<TempFile, CompileError> {
    let temp_file = TempFile::create();
    let cmd_output = Command::new("rustc")
        .args(&[
            "--test",
            path.as_ref().to_str().unwrap(),
            "-o",
            temp_file.path().as_os_str().to_str().unwrap(),
        ])
        .args(RUSTC_COLOR_ARGS)
        .output()?;

    if cmd_output.status.success() {
        Ok(temp_file)
    } else {
        Err(CompileError::OutputError(Output::from_cmd_output(
            &cmd_output,
        )))
    }
}

/// Executes the file.
///
/// # Errors
///
/// If the execution failed due to an IO error a [`RunError::IOError`]
/// is returned. If the execution failed at runtime a
/// [`RunError::OutputError`].
pub fn run<T: AsRef<Path>>(path: T) -> Result<Output, RunError> {
    let cmd_output = Command::new(path.as_ref())
        .arg("--show-output")
        .args(RUSTC_COLOR_ARGS)
        .output()?;

    let output = Output::from_cmd_output(&cmd_output);
    if cmd_output.status.success() {
        Ok(output)
    } else {
        Err(RunError::OutputError(output))
    }
}

/// Structure for scanning all of the exercises from the `exercises/`
/// folder at compile time
#[derive(RustEmbed)]
#[folder = "exercises"]
pub struct Exercises;

pub fn init() -> io::Result<()> {
    for embedded_file in Exercises::iter() {
        if embedded_file == "state.toml" {
            continue;
        }
        let path = Path::new("exercises").join(embedded_file.as_ref());
        if !path.exists() {
            // Not overwriting old exercises
            create_dir_all(path.parent().expect("path to be inside exercises"))?;
            let mut file = File::create(&path)?;
            let exercise = Exercises::get(&embedded_file).unwrap();
            let data = std::str::from_utf8(&exercise.data).expect("files to contain valid utf8");
            file.write_all(data.as_bytes())?;
            println!(
                "New exercise: {}",
                path.to_str().expect("path to be valid str")
            );
        }
    }
    println!("Creating exercises finished! Run `otarustlings start` to start.");

    Ok(())
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn compile_success() {
        let compile = compile("tests/fixture/compileSuccess.rs").unwrap();
        let path = compile.path();
        let out = run(path.to_str().unwrap()).unwrap();
        assert!(out.stdout.contains("Hello, world!"));
        assert_eq!(out.stderr.len(), 0);
    }

    #[test]
    fn compile_failure() {
        let out = compile("tests/fixture/compileFailure.rs").unwrap_err();
        if let CompileError::OutputError(output) = out {
            assert!(output.stderr.contains("aborting due to 2 previous errors"));
            assert!(output.stderr.contains("not a function"));
            assert_eq!(output.stdout.len(), 0);
        } else {
            unreachable!();
        }
    }

    #[test]
    fn should_delete_temp_file() {
        let compiled_handle = compile("tests/fixture/compileSuccess.rs").unwrap();
        let path = compiled_handle.path().clone();
        assert!(path.exists());
        drop(compiled_handle);
        assert!(!path.exists());
    }
}
