use console::{Key, Term};
use dialoguer::theme::ColorfulTheme;
use notify::{DebouncedEvent, RecommendedWatcher, RecursiveMode, Watcher};
use otarustlings::{
    exercise::{compile, init, run, CompileError, RunError},
    menu::MenuBuilder,
    state::{Exercise, State},
};
use std::{
    path::{Path, PathBuf},
    sync::{mpsc::channel, Arc, Mutex},
    thread::{self, JoinHandle},
    time::Duration,
};
use structopt::StructOpt;

#[derive(StructOpt)]
/// Otarustlings exercise platform.
///
/// Otarustlings is used for learning hands on Rust with great
/// examples and interesting challenges. To get started, run
/// `otarustlings start`. When new exercises are available, run
/// `otarustlings init` to get them.
enum Opt {
    /// Writes the exercises to `exercises/` folder. Doesn't overwrite
    /// old exercises.
    Init,
    /// Start otarustlings. Asks if you want initialize first.
    Start,
}

fn main() -> anyhow::Result<()> {
    // when starting the app a tree of exercises is shown from which
    // the user can select to continue where they left off or manually
    // select which exercise to start on.
    //
    // after selecting the exercise the screen is cleared and
    // otarustligs starts watching changes to that file and
    // automatically recompiles on changes.  pressing 'q' or ESC will
    // lead back to the menu.  pressing '?' will show keyboard
    // shortcuts.  once the exercise is finished, it gets marked in
    // the state and otarustlings continues with the next exercise
    // except for last exercise of the week where the menu of the next
    // week is shown.
    //
    // all folders and exercises have a marker next to them to show if
    // they have been started or are done
    //
    // otarustlings has a --help which explains how to do the exercises

    let opt = Opt::from_args();
    match opt {
        Opt::Init => init().unwrap(),
        Opt::Start => {
            // TODO check if initialized
            let (tx, rx) = channel();

            let mut watcher: RecommendedWatcher = Watcher::new(tx, Duration::from_secs(1))?;
            watcher.watch("exercises", RecursiveMode::Recursive)?;

            #[derive(Debug, Clone)]
            enum Message {
                Notify(PathBuf),
                TestExercise,
                SelectExercise(Exercise),
                ExitExercise,
            }

            let (sender, receiver) = channel();

            let watch_sender = sender.clone();
            let _todo_fix_watch_handle = thread::spawn(move || loop {
                if let DebouncedEvent::Write(b) = rx.recv().unwrap() {
                    watch_sender.send(Message::Notify(b)).unwrap();
                }
            });

            let state = Arc::new(Mutex::new(State::scan_open_default("exercises").unwrap()));

            let input_sender = sender.clone();
            let input_state = Arc::clone(&state);
            let input_handle = thread::spawn(move || {
                // Move a clone of the Arc to this closure
                let state = input_state;

                let mut prev_sel = 0;
                loop {
                    let theme = ColorfulTheme::default();

                    let sel = {
                        let state = state.lock().unwrap();
                        let menu = MenuBuilder::default()
                            .default_index(prev_sel)
                            .items(&state.0)
                            .theme(&theme)
                            .build()
                            .unwrap();
                        // FIXME menu interact blocks the lock
                        if let Some(sel) = menu.interact(&Term::stderr()).unwrap() {
                            sel
                        } else {
                            // Exit the menu
                            break;
                        }
                    };
                    let exercise = state
                        .lock()
                        .unwrap()
                        .0
                        .iter()
                        .nth(sel)
                        .expect("sel to be within range")
                        .clone();

                    // compile and run exercise
                    input_sender.send(Message::SelectExercise(exercise.clone()))?;
                    input_sender.send(Message::TestExercise)?;
                    prev_sel = sel;

                    let term = Term::stdout();
                    loop {
                        // wait and recompile on changes
                        match term.read_key()? {
                            Key::Char('?') => {
                                println!("Help: press Q or ESC to exit.")
                            }
                            Key::Escape | Key::Char('q') => {
                                input_sender.send(Message::ExitExercise)?;
                                break;
                            }
                            Key::Enter => {
                                // TODO call state.get_next_exercise(exercise)
                            }
                            _ => {}
                        }
                    }

                    // do expensive calculation
                }

                Ok::<(), anyhow::Error>(())
            });

            let compile_state = Arc::clone(&state);
            let _todo_fix_compile_handle: JoinHandle<anyhow::Result<()>> = thread::spawn(move || {
                // Move a clone of the Arc to this closure
                let state = compile_state;
                let mut current_exercise = None::<Exercise>;

                let term = Term::stdout();

                loop {
                    match receiver.recv().unwrap() {
                        Message::Notify(b) => {
                            if let Some(a) = &current_exercise {
                                if std::env::current_dir().unwrap().join("exercises").join(&a.path) == b {
                                    sender.send(Message::TestExercise)?;
                                }
                            }
                        }
                        Message::TestExercise => {
                            term.hide_cursor().unwrap();

                            term.clear_screen().unwrap();

                            let mut exercise = current_exercise.clone().unwrap();

                            {
                                let mut state = state.lock().unwrap();

                                assert!(
                                    state.0.remove(&exercise), // remove returns true if the item existed
                                    "state did not contain the exercise"
                                );
                                exercise.exercise_state.start_exercise();
                                state.0.insert(exercise);

                                state.save().unwrap();
                            }

                            match compile(
                                Path::new("exercises").join(
                                    current_exercise
                                        .clone()
                                        .expect("current exercise to be Some")
                                        .path,
                                ),
                            ) {
                                Ok(tempfile) => match run(tempfile) {
                                    Ok(output) => {
                                        println!("{}", output.stdout.replace('\n', "\r\n"));
                                        println!("Press enter to start the next exercise or Q to return to menu.\r");

                                        {
                                            let mut state = state.lock().unwrap();
                                            // exercise.exercise_state.complete_exercise();
                                            let mut exercise = current_exercise.clone().unwrap();
                                            assert!(
                                                state.0.remove(&exercise), // remove returns true if the item existed
                                                "state did not contain the exercise"
                                            );
                                            exercise.exercise_state.complete_exercise();
                                            state.0.insert(exercise);

                                            state.save().unwrap();
                                        }
                                    }
                                    Err(RunError::OutputError(output)) => {
                                        println!("{}", output.stdout.replace('\n', "\r\n"))
                                    }
                                    Err(e) => {
                                        panic!("{e:?}");
                                    }
                                },
                                Err(CompileError::OutputError(output)) => {
                                    println!("{}", output.stderr.replace('\n', "\r\n"))
                                }
                                Err(CompileError::IOError(output)) => {
                                    println!("{}", output);
                                }
                            }
                        }
                        Message::SelectExercise(b) => {
                            current_exercise = Some(b);
                        }
                        Message::ExitExercise => {
                            current_exercise = None;

                            term.clear_screen().unwrap();

                            term.show_cursor().unwrap();
                            term.flush().unwrap();
                        }
                    }
                }
            });

            input_handle.join().unwrap()?;
        }
    };
    Ok(())
}
