use crate::menu::{self, Item, Status};
use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};
use std::collections::{BTreeMap, BTreeSet};
use std::default::Default;
use std::io::{self, ErrorKind};
use std::mem::swap;
use std::path::{Path, PathBuf};
use tokio::fs::{read_to_string, File};
use tokio::io::AsyncWriteExt;

/// `Exercise` represents an exercise.
#[derive(Debug, Clone)]
pub struct Exercise {
    pub path: PathBuf,
    pub exercise_state: ExerciseState,
}

impl PartialEq for Exercise {
    fn eq(&self, other: &Self) -> bool {
        self.path == other.path
    }
}

impl Eq for Exercise {}

impl PartialOrd for Exercise {
    fn partial_cmp(&self, other: &Self) -> Option<std::cmp::Ordering> {
        self.path.partial_cmp(&other.path)
    }
}

impl Ord for Exercise {
    fn cmp(&self, other: &Self) -> std::cmp::Ordering {
        self.path.cmp(&other.path)
    }
}

impl From<(PathBuf, ExerciseState)> for Exercise {
    fn from(from: (PathBuf, ExerciseState)) -> Self {
        Self {
            path: from.0,
            exercise_state: from.1,
        }
    }
}

impl Item for Exercise {
    fn file_name(&self) -> &str {
        self.path.file_name().unwrap().to_str().unwrap()
    }

    fn folder_name(&self) -> &str {
        self.path
            .parent()
            .unwrap()
            .file_name()
            .unwrap()
            .to_str()
            .unwrap()
    }

    fn status(&self) -> Status {
        match (self.exercise_state.started, self.exercise_state.completed) {
            (_, Some(_)) => Status::Completed,
            (Some(_), None) => Status::Started,
            (_, _) => Status::NotStarted,
        }
    }
}

#[derive(
    Copy, Clone, Serialize, Deserialize, Hash, Default, Debug, PartialEq, Eq, PartialOrd, Ord,
)]
pub struct ExerciseState {
    pub started: Option<DateTime<Utc>>,
    pub completed: Option<DateTime<Utc>>,
}

impl ExerciseState {
    pub fn start_exercise(&mut self) -> DateTime<Utc> {
        if let Some(date) = self.started {
            date
        } else {
            let now = Utc::now();
            self.started = Some(now);
            now
        }
    }

    pub fn complete_exercise(&mut self) -> DateTime<Utc> {
        if let Some(date) = self.completed {
            date
        } else {
            let now = Utc::now();
            self.completed = Some(now);
            now
        }
    }
}

/// State of the exercises.
#[derive(Debug, Default, Clone)]
pub struct State(pub BTreeSet<Exercise>);

impl From<BTreeMap<PathBuf, ExerciseState>> for State {
    fn from(map: BTreeMap<PathBuf, ExerciseState>) -> Self {
        Self(map.into_iter().map(Exercise::from).collect())
    }
}

impl State {
    /// Creates a new empty state.
    pub fn new() -> Self {
        Self::default()
    }

    /// Collects all exercises under `path` as their own default [`ExerciseState`].
    ///
    /// If the folder at `path` is empty, the state will be empty too. TODO Should error.
    pub fn scan(path: impl AsRef<Path>) -> Self {
        let exercises = menu::list_exercises(path)
            .map(|path| (path, ExerciseState::default()).into())
            .collect();
        Self(exercises)
    }

    /// Tries to deserialize `exercises/state.toml` as [`Self`].
    ///
    /// # Errors
    ///
    /// Returns an [`io::Error`] if reading the state file fails or if
    /// the file contains invalid toml.
    pub async fn open(path: impl AsRef<Path>) -> io::Result<Self> {
        let map: BTreeMap<PathBuf, ExerciseState> =
            toml::from_str(&read_to_string(path.as_ref().join("state.toml")).await?)?;
        Ok(map.into())
    }

    /// Reads `exercises/state.toml` if it exists and scans the
    /// `exercises` directory extending the state from
    /// `state.toml`. This is the logical option in most cases.
    ///
    /// # Errors
    ///
    /// Same errors as [`Self::open`].
    pub async fn scan_open_default(path: impl AsRef<Path>) -> io::Result<Self> {
        let path = path.as_ref();
        let mut state = match Self::open(path).await {
            Err(e) => {
                if e.kind() == ErrorKind::NotFound {
                    Self::default()
                } else {
                    return Err(e);
                }
            }
            Ok(state) => state,
        };
        let other = Self::scan(path);
        state.extend(other);
        Ok(state)
    }

    /// Saves the state to `exercises/state.toml`.
    ///
    /// # Errors
    ///
    /// Returns an [`io::Error`] if creating or writing to the state
    /// file fails.
    pub async fn save(&self) -> io::Result<()> {
        let mut file = File::create("exercises/state.toml").await?;
        let mut map = BTreeMap::new();

        for ex in &self.0 {
            map.insert(&ex.path, &ex.exercise_state);
        }
        file.write_all(
            toml::to_string(&map)
                .expect("state to be serializable")
                .as_bytes(),
        )
        .await?;
        Ok(())
    }

    /// Joins two states favoring self's exercises. Useful for adding
    /// missing exercises with:
    ///
    /// ```
    /// # use otarustlings::state::{State, Exercise, ExerciseState};
    /// let mut left = State::new();
    /// # left.0.insert(Exercise { path: "old_week1/ex1.rs".into(), exercise_state: ExerciseState::default() });
    /// left.extend(State::scan("exercises"));
    /// ```
    ///
    /// # Example
    ///
    /// ```
    /// # use chrono::Utc;
    /// # use otarustlings::state::{State, Exercise, ExerciseState};
    /// let started = ExerciseState {
    ///     started: Some(Utc::now()),
    ///     ..ExerciseState::default()
    /// };
    /// // left:
    /// //   week1
    /// //     ex1.rs (started)
    /// //   week2
    /// //     ex1.rs (default)
    /// //
    /// // right:
    /// //   week1
    /// //     ex1.rs (default)
    /// //   week3
    /// //     ex1.rs (default)
    ///
    /// let mut left = State::new();
    /// left.0.insert(Exercise { path: "week1/ex1.rs".into(), exercise_state: started.clone() });
    /// left.0.insert(Exercise { path: "week2/ex1.rs".into(), exercise_state: ExerciseState::default() });
    ///
    /// let mut right = State::new();
    /// right.0.insert(Exercise { path: "week1/ex1.rs".into(), exercise_state: ExerciseState::default() });
    /// right.0.insert(Exercise { path: "week3/ex1.rs".into(), exercise_state: ExerciseState::default() });
    ///
    /// // Extending `left` with `right` should give
    /// // left:
    /// //   week1
    /// //     ex1.rs (started)
    /// //   week2
    /// //     ex1.rs (default)
    /// //   week3
    /// //     ex1.rs (default)
    ///
    /// left.extend(right);
    ///
    /// assert_eq!(left.0.iter().nth(0).unwrap().exercise_state, started.clone());
    /// ```
    pub fn extend(&mut self, mut other: Self) {
        swap(&mut self.0, &mut other.0); // swapping sets inverts the priority
        self.0.append(&mut other.0);
        // let mut extended = BinaryHeap::with_capacity(self.0.capacity() + other.0.capacity());
        // while let Some(self_next) = self.0.pop() {
        //     while let Some(other_next) = other.0.peek() {
        //         if other_next > &self_next {
        //             extended.push(other.0.pop().expect("peek and pop to give same value"));
        //         } else {
        //             break;
        //         }
        //     }
        //     extended.push(self_next);
        // }
        // Self(extended)
    }

    /// Returns an iterator yielding [`Exercise`]s.
    pub fn iter(&self) -> impl Iterator<Item = &'_ Exercise> {
        self.0.iter()
    }
}
