use crate::menu::{self, Item};
use chrono::{DateTime, Utc};
use serde::{Deserialize, Serialize};
use std::collections::BTreeMap;
use std::default::Default;
use std::io::{self, ErrorKind};
use std::mem::swap;
use std::path::{Path, PathBuf};
use tokio::fs::{read_to_string, File};
use tokio::io::AsyncWriteExt;

/// `Exercise` represents an exercise.
#[derive(Debug, Clone, Copy, Hash, PartialEq, Eq, PartialOrd, Ord)]
pub struct Exercise<'ex> {
    pub path: &'ex Path,
    pub exercise_state: &'ex ExerciseState,
}

impl<'ex> From<(&'ex PathBuf, &'ex ExerciseState)> for Exercise<'ex> {
    fn from(from: (&'ex PathBuf, &'ex ExerciseState)) -> Self {
        Self {
            path: from.0,
            exercise_state: from.1,
        }
    }
}

impl Item for Exercise<'_> {
    fn file_name(&self) -> &str {
        self.path.file_name().unwrap().to_str().unwrap()
    }

    fn folder_name(&self) -> &str {
        self.path
            .parent()
            .unwrap()
            .file_name()
            .unwrap()
            .to_str()
            .unwrap()
    }

    fn label(&self) -> &str {
        match (self.exercise_state.started, self.exercise_state.completed) {
            (Some(_), None) => "in progress",
            (_, Some(_)) => "done",
            (_, _) => "",
        }
    }
}

#[derive(
    Copy, Clone, Serialize, Deserialize, Hash, Default, Debug, PartialEq, Eq, PartialOrd, Ord,
)]
pub struct ExerciseState {
    pub started: Option<DateTime<Utc>>,
    pub completed: Option<DateTime<Utc>>,
}

impl ExerciseState {
    pub fn start_exercise(&mut self) -> DateTime<Utc> {
        if let Some(date) = self.started {
            date
        } else {
            let now = Utc::now();
            self.started = Some(now);
            now
        }
    }

    pub fn complete_exercise(&mut self) -> DateTime<Utc> {
        if let Some(date) = self.completed {
            date
        } else {
            let now = Utc::now();
            self.completed = Some(now);
            now
        }
    }
}

/// State of the exercises.
#[derive(Serialize, Deserialize, Debug, Default, Clone, PartialEq)]
#[serde(transparent)]
pub struct State(pub BTreeMap<PathBuf, ExerciseState>);

impl State {
    /// Creates a new empty state
    pub fn new() -> Self {
        Self::default()
    }

    /// Collects all exercises under `path` as their own default [`ExerciseState`].
    ///
    /// If the folder at `path` is empty, the state will be empty too. TODO Should error
    pub fn scan(path: impl AsRef<Path>) -> Self {
        let exercises = menu::list_exercises(path)
            .map(|path| (path, ExerciseState::default()))
            .collect();
        Self(exercises)
    }

    /// Tries to deserialize `exercises/state.toml` as [`Self`].
    ///
    /// # Errors
    ///
    /// Returns an [`io::Error`] if reading the state file fails or if
    /// the file contains invalid toml.
    pub async fn open(path: impl AsRef<Path>) -> io::Result<Self> {
        let state = toml::from_str(&read_to_string(path.as_ref().join("state.toml")).await?)?;
        Ok(state)
    }

    /// Reads `exercises/state.toml` if it exists and scans the
    /// `exercises` directory extending the state from
    /// `state.toml`. This is the logical option in most cases.
    ///
    /// # Errors
    ///
    /// Same errors as [`Self::open`].
    pub async fn scan_open_default(path: impl AsRef<Path>) -> io::Result<Self> {
        let path = path.as_ref();
        let mut state = match Self::open(path).await {
            Err(e) => {
                if e.kind() == ErrorKind::NotFound {
                    Self::default()
                } else {
                    return Err(e);
                }
            }
            Ok(state) => state,
        };
        state.extend(Self::scan(path));
        Ok(state)
    }

    /// Saves the state to `exercises/state.toml`.
    ///
    /// # Errors
    ///
    /// Returns an [`io::Error`] if creating or writing to the state
    /// file fails.
    pub async fn save(&self) -> io::Result<()> {
        let mut file = File::create("exercises/state.toml").await?;
        file.write_all(
            toml::to_string(self)
                .expect("state to be serializable")
                .as_bytes(),
        )
        .await?;
        Ok(())
    }

    /// Joins two states favoring self's exercises. Useful for adding
    /// missing exercises with:
    ///
    /// ```
    /// # use otarustlings::state::{State, ExerciseState};
    /// let mut left = State::new();
    /// # left.0.insert("old_week/ex1.rs".into(), ExerciseState::default());
    /// left.extend(State::scan("exercises"));
    /// ```
    ///
    /// # Example
    ///
    /// ```
    /// # use chrono::Utc;
    /// # use otarustlings::state::{State, ExerciseState};
    /// let started = ExerciseState {
    ///     started: Some(Utc::now()),
    ///     ..ExerciseState::default()
    /// };
    /// // left:
    /// //   week1
    /// //     ex1.rs (started)
    /// //   week2
    /// //     ex1.rs (default)
    /// //
    /// // right:
    /// //   week1
    /// //     ex1.rs (default)
    /// //   week3
    /// //     ex1.rs (default)
    ///
    /// let mut left = State::new();
    /// left.0.insert("week1/ex1.rs".into(), started.clone());
    /// left.0.insert("week2/ex1.rs".into(), ExerciseState::default());
    ///
    /// let mut right = State::new();
    /// right.0.insert("week1/ex1.rs".into(), ExerciseState::default());
    /// right.0.insert("week3/ex1.rs".into(), ExerciseState::default());
    ///
    /// // Extending `left` with `right` should give
    /// // left:
    /// //   week1
    /// //     ex1.rs (started)
    /// //   week2
    /// //     ex1.rs (default)
    /// //   week3
    /// //     ex1.rs (default)
    ///
    /// left.extend(right);
    ///
    /// let mut expected = State::new();
    /// expected.0.insert("week1/ex1.rs".into(), started.clone());
    /// expected.0.insert("week2/ex1.rs".into(), ExerciseState::default());
    /// expected.0.insert("week3/ex1.rs".into(), ExerciseState::default());
    ///
    /// assert_eq!(left, expected)
    /// ```
    pub fn extend(&mut self, mut other: Self) {
        swap(self, &mut other);
        self.0.extend(other.0);
    }

    /// Returns an iterator yielding [`Exercise`]s.
    pub fn iter_exercises(&self) -> impl Iterator<Item = Exercise<'_>> {
        self.0.iter().map(Exercise::from)
    }
}
