use byteorder::{BigEndian, ByteOrder, LittleEndian};
use std::io::{Error, ErrorKind, Result};

pub struct ByteBuffer {
    pub data: Vec<u8>,
    pub read_pos: usize,
    pub write_pos: usize,
}

impl ByteBuffer {
    pub fn new(size: usize) -> ByteBuffer {
        ByteBuffer {
            data: vec![0; size],
            read_pos: 0,
            write_pos: 0,
        }
    }

    pub fn from_buf(vec: Vec<u8>) -> ByteBuffer {
        ByteBuffer {
            data: vec,
            read_pos: 0,
            write_pos: 0,
        }
    }

    pub fn clear(&mut self) {
        self.read_pos = 0;
        self.write_pos = 0;
    }

    pub fn skip(&mut self, skip: usize) {
        self.read_pos += skip;
        self.write_pos += skip;
    }

    // Write operations

    pub fn write_bytes(&mut self, bytes: &[u8]) {
        //self.flush_bit();

        let size = bytes.len() + self.write_pos;

        if size > self.data.len() {
            panic!("write size greater than bytes length")
        }

        for v in bytes {
            self.data[self.write_pos] = *v;
            self.write_pos += 1;
        }
    }

    pub fn write_bool(&mut self, val: bool) {
        self.write_bytes(&[val as u8]);
    }

    pub fn write_u8(&mut self, val: u8) {
        self.write_bytes(&[val]);
    }

    pub fn write_i8(&mut self, val: i8) {
        self.write_u8(val as u8);
    }

    pub fn write_i8_sub(&mut self, val: i8) {
        self.write_u8(128 - val as u8);
    }

    pub fn write_u16(&mut self, val: u16) {
        let mut buf = [0; 2];

        BigEndian::write_u16(&mut buf, val);

        self.write_bytes(&buf);
    }

    pub fn write_u16_smart(&mut self, val: u16) {
        match val {
            0..=127 => self.write_u8(val as u8),
            128..=32767 => self.write_u16(val + 32768),
            _ => panic!("Value {} is too big for write_i16_smart", val),
        }
    }

    pub fn write_i16(&mut self, val: i16) {
        self.write_u16(val as u16);
    }

    pub fn write_i16_add(&mut self, val: i16) {
        self.write_i8((val >> 8) as i8);
        self.write_i8((val + 128) as i8);
    }

    pub fn write_i16_le_add(&mut self, val: i16) {
        self.write_i8((val + 128) as i8);
        self.write_i8((val >> 8) as i8);
    }

    pub fn write_u16_le(&mut self, val: u16) {
        let mut buf = [0; 2];

        LittleEndian::write_u16(&mut buf, val);

        self.write_bytes(&buf);
    }

    pub fn write_i16_le(&mut self, val: i16) {
        self.write_u16_le(val as u16);
    }

    pub fn write_u32(&mut self, val: u32) {
        let mut buf = [0; 4];

        BigEndian::write_u32(&mut buf, val);

        self.write_bytes(&buf);
    }

    pub fn write_i32(&mut self, val: i32) {
        self.write_u32(val as u32);
    }

    pub fn write_i32_ime(&mut self, val: i32) {
        self.write_i16(val as i16);
        self.write_i16((val >> 16) as i16);
    }

    pub fn write_u64(&mut self, val: u64) {
        let mut buf = [0; 8];

        BigEndian::write_u64(&mut buf, val);

        self.write_bytes(&buf);
    }

    pub fn write_i64(&mut self, val: i64) {
        self.write_u64(val as u64);
    }

    pub fn write_string_null_terminated(&mut self, s: &str) {
        // TODO: Check here for out of length

        for x in s.as_bytes() {
            self.data[self.write_pos] = *x;
            self.write_pos += 1;
        }
        self.write_i8(0);
    }

    pub fn write_bytes_reversed_add(&mut self, buf: &ByteBuffer) {
        for i in (buf.read_pos..buf.write_pos).rev() {
            self.write_i8(buf.data.get(i).unwrap().wrapping_add(128) as i8);
        }
    }

    // Read operation
    pub fn read_bytes(&mut self, buf: &mut ByteBuffer, size: usize) -> Result<()> {
        if self.read_pos + size > self.data.len() {
            return Err(Error::new(
                ErrorKind::UnexpectedEof,
                "could not read enough bytes from buffer",
            ));
        }
        let range = self.read_pos..self.read_pos + size;
        buf.write_bytes(&self.data[range]);
        self.read_pos += size;
        Ok(())
    }

    pub fn read_bool(&mut self) -> bool {
        let value = self
            .data
            .get(self.read_pos)
            .expect("failed reading 1 byte for bool");
        self.read_pos += 1;
        return *value == 1;
    }

    pub fn read_u8(&mut self) -> u8 {
        let value = self
            .data
            .get(self.read_pos)
            .expect("failed reading 1 byte for u8");
        self.read_pos += 1;
        return *value;
    }

    pub fn read_i8(&mut self) -> i8 {
        return self.read_u8() as i8;
    }

    pub fn read_u16(&mut self) -> u16 {
        let bytes = self
            .data
            .get(self.read_pos..self.read_pos + 2)
            .expect("failed reading 2 bytes for u16");
        self.read_pos += 2;
        let value = BigEndian::read_u16(bytes);
        return value;
    }

    pub fn read_i16(&mut self) -> i16 {
        return self.read_u16() as i16;
    }

    pub fn read_u32(&mut self) -> u32 {
        let bytes = self
            .data
            .get(self.read_pos..self.read_pos + 4)
            .expect("failed reading 4 bytes for u32");
        self.read_pos += 4;
        let value = BigEndian::read_u32(bytes);
        return value;
    }

    pub fn read_i32(&mut self) -> i32 {
        return self.read_u32() as i32;
    }

    pub fn read_u64(&mut self) -> u64 {
        let bytes = self
            .data
            .get(self.read_pos..self.read_pos + 8)
            .expect("failed reading 8 bytes for u64");
        self.read_pos += 8;
        let value = BigEndian::read_u64(bytes);
        return value;
    }

    pub fn read_i64(&mut self) -> i64 {
        return self.read_u64() as i64;
    }

    pub fn read_string(&mut self) -> String {
        let mut str = Vec::new();

        while let Some(x) = self.data.get(self.read_pos) {
            if *x != 0 {
                str.push(*x);
                self.read_pos += 1;
            } else {
                break;
            }
        }

        let s = match std::str::from_utf8(&str) {
            Ok(v) => v,
            Err(e) => panic!("Invalid UTF-8 sequence: {}", e),
        };

        let string = s.to_owned();

        return string;
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn test_write_u8() {
        let mut buf = ByteBuffer::new(1);
        buf.write_u8(155);
        assert_eq!(buf.data[0], 155);
        assert_eq!(buf.write_pos, 1);
    }

    #[test]
    fn test_read_u8() {
        let mut buf = ByteBuffer::new(1);
        buf.data[0] = 231;
        assert_eq!(buf.read_u8(), 231);
        assert_eq!(buf.read_pos, 1);
    }

    #[test]
    fn test_write_i8() {
        let mut buf = ByteBuffer::new(1);
        buf.write_i8(101);
        assert_eq!(buf.data[0], 101);
        assert_eq!(buf.write_pos, 1);
    }

    #[test]
    fn test_read_i8() {
        let mut buf = ByteBuffer::new(1);
        buf.data[0] = 35;
        assert_eq!(buf.read_i8(), 35);
        assert_eq!(buf.read_pos, 1);
    }

    #[test]
    fn test_write_u16() {
        let mut buf = ByteBuffer::new(2);
        buf.write_u16(18953);
        assert_eq!(buf.data[0], 74);
        assert_eq!(buf.data[1], 9);
        assert_eq!(buf.write_pos, 2);
    }

}
