use bytes::BytesMut;
use proptest::prelude::*;

use originz::codec::{character_string, decoder, domain_name};

proptest! {
    #[test]
    fn message_decode_doesnt_crash(s in ".*") {
        // ignore any io errors, they're expected
        let _ = decoder::DNSMessageDecoder::new().decode(s.as_bytes());
    }

    #[test]
    fn characterstring_read_doesnt_crash(s in ".*") {
        // ignore any io errors, they're expected
        let _ = character_string::read(s.as_bytes(), 0, "test data");
    }

    #[test]
    fn characterstring_encode_decode(val in ".{0,255}", before in ".*", after in ".*") {
        // Add some garbage before and after the encoded data
        let mut content = Vec::from(before.as_bytes());
        let encoded_offset = content.len();

        // Ensure that we have <=255 bytes: the above 255-char limit doesn't account for multibyte chars
        let mut val_bytes = Vec::with_capacity(val.len());
        val_bytes.extend(val.as_bytes());
        val_bytes.truncate(255);

        let mut encoded = BytesMut::new();
        character_string::write(&val_bytes, &mut encoded, "test data").expect("write failed");
        content.extend(encoded);
        content.extend(after.as_bytes());

        let (consumed_len, decoded) = character_string::read(&content, encoded_offset, "test data").expect("read failed");
        assert_eq!(consumed_len, 1+val_bytes.len());
        assert_eq!(decoded, val_bytes);
    }

    #[test]
    fn domainname_read_doesnt_crash(s in ".*") {
        // ignore any io errors, they're expected
        let _ = domain_name::read(s.as_bytes(), 0, "test data");
    }

    #[test]
    fn domainname_readnoptr_doesnt_crash(s in ".*") {
        // ignore any io errors, they're expected
        let _ = domain_name::read(s.as_bytes(), 0, "test data");
    }

    #[test]
    fn domainname_encode_decode(val in "[a-zA-Z0-9]([a-zA-Z0-9-]{0,61}[a-zA-Z0-9])?\\.", before in ".*", after in ".*") {
        // Add some garbage before and after the encoded data
        let mut content = Vec::from(before.as_bytes());
        let encoded_offset = content.len();

        let mut encoded = BytesMut::new();
        let mut ptr_offsets = domain_name::LabelOffsets::new();
        domain_name::write(&val, &mut encoded, &mut ptr_offsets, "test data").expect("write failed");
        content.extend(encoded);
        content.extend(after.as_bytes());

        let (consumed_len, decoded) = domain_name::read(&content, encoded_offset, "test data").expect("read failed");
        assert_eq!(consumed_len, 1+val.len());
        assert_eq!(decoded, val);
    }
}
