use bytecheck::CheckBytes;
use rkyv::{Archive, Serialize, Deserialize};

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct A {
  /// The first segment of an IPv4 address
  pub address1: u8,
  /// The second segment of an IPv4 address
  pub address2: u8,
  /// The third segment of an IPv4 address
  pub address3: u8,
  /// The fourth segment of an IPv4 address
  pub address4: u8,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct NS {
  /// A <domain-name> which specifies a host which should be
  /// authoritative for the specified class and domain.
  pub nsdname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct CNAME {
  /// A <domain-name> which specifies the canonical or primary
  /// name for the owner.  The owner name is an alias.
  pub cname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct SOA {
  /// The <domain-name> of the name server that was the
  /// original or primary source of data for this zone.
  pub mname: String,

  /// A <domain-name> which specifies the mailbox of the
  /// person responsible for this zone.
  pub rname: String,

  /// The unsigned 32 bit version number of the original copy
  /// of the zone.  Zone transfers preserve this value.  This
  /// value wraps and should be compared using sequence space
  /// arithmetic.
  pub serial: u32,

  /// A 32 bit time interval before the zone should be
  /// refreshed.
  pub refresh: u32,

  /// A 32 bit time interval that should elapse before a
  /// failed refresh should be retried.
  pub retry: u32,

  /// A 32 bit time value that specifies the upper limit on
  /// the time interval that can elapse before the zone is no
  /// longer authoritative.
  pub expire: u32,

  /// The unsigned 32 bit minimum TTL field that should be
  /// exported with any RR from this zone.
  pub minimum: u32,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct PTR {
  /// A <domain-name> which points to some location in the
  /// domain name space.
  pub ptrdname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct HINFO {
  /// A <character-string> which specifies the CPU type.
  pub cpu: Vec<u8>,

  /// A <character-string> which specifies the operating
  /// system type.
  pub os: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct MX {
  /// A 16 bit integer which specifies the preference given to
  /// this RR among others at the same owner.  Lower values
  /// are preferred.
  pub preference: u16,

  /// A <domain-name> which specifies a host willing to act as
  /// a mail exchange for the owner name.
  pub exchange: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct TXT {
  /// One or more <character-string>s.
  pub entries: Vec<TXTEntry>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct TXTEntry {
  /// A <character-string>.
  pub data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct RP {
  /// The first field, <mbox-dname>, is a domain name that specifies the
  /// mailbox for the responsible person.  Its format in master files uses
  /// the DNS convention for mailbox encoding, identical to that used for
  /// the RNAME mailbox field in the SOA RR.  The root domain name (just
  /// ".") may be specified for <mbox-dname> to indicate that no mailbox is
  /// available.
  pub mbox_dname: String,

  /// The second field, <txt-dname>, is a domain name for which TXT RR's
  /// exist.  A subsequent query can be performed to retrieve the
  /// associated TXT resource records at <txt-dname>.  This provides a
  /// level of indirection so that the entity can be referred to from
  /// multiple places in the DNS.  The root domain name (just ".") may be
  /// specified for <txt-dname> to indicate that the TXT_DNAME is absent,
  /// and no associated TXT RR exists.
  pub txt_dname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct AFSDB {
  ///  The <subtype> field is a 16 bit integer.
  pub subtype: u16,

  /// The <hostname> field is a domain name of a host that has
  /// a server for the cell named by the owner name of the RR.
  pub hostname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct SIG {
  /// The type of the other RRs covered by this SIG.
  pub type_covered: u16,

  /// The key algorithm parallel to the same field for the SIG resource.
  pub algorithm: u8,

  /// An unsigned count of how many labels there are in the original SIG
  /// RR owner name not counting the null label for root and not
  /// counting any initial * for a wildcard.
  pub labels: u8,

  /// This original TTL is protected by the signature while the current
  /// TTL field is not.
  pub original_ttl: u32,

  /// The SIG is valid from the signature inception time until the
  /// signature expiration time.  Both are unsigned numbers of seconds
  /// since the start of 1 January 1970, GMT, ignoring leap seconds.
  pub signature_expiration: u32,
  pub signature_inception: u32,

  /// Used to efficiently select the applicable key among multiple.
  pub key_tag: u16,

  /// The domain name of the signer generating the SIG RR.
  pub signers_name: String,

  /// Binds the other RDATA fields to the RRset of the type covered RRs
  /// with that owner name and class.  This covered RRset is thereby
  /// authenticated.
  pub signature: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct KEY {
  /// Flags for key type and what key should be used for.
  pub flags: u16, // 12 bit

  /// Protocol to use the key with: TLS, email, dnssec, ipsec, etc.
  pub protocol: u8,

  /// The key algorithm parallel to the same field for the SIG resource.
  pub algorithm: u8,

  /// The key data.
  pub public_key: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct AAAA {
  /// The first segment of an IPv6 address
  pub address1: u16,
  /// The second segment of an IPv6 address
  pub address2: u16,
  /// The third segment of an IPv6 address
  pub address3: u16,
  /// The fourth segment of an IPv6 address
  pub address4: u16,
  /// The fifth segment of an IPv6 address
  pub address5: u16,
  /// The sixth segment of an IPv6 address
  pub address6: u16,
  /// The seventh segment of an IPv6 address
  pub address7: u16,
  /// The eighth segment of an IPv6 address
  pub address8: u16,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct LOC {
  /// Version number of the representation.  This must be zero.
  pub version: u8,

  /// The diameter of a sphere enclosing the described entity, in
  /// centimeters, expressed as a pair of four-bit unsigned
  /// integers, each ranging from zero to nine, with the most
  /// significant four bits representing the base and the second
  /// number representing the power of ten by which to multiply
  /// the base. Range: 0e0 to 9e9 (90000 km)
  pub size: u8,

  /// The horizontal precision of the data, in centimeters,
  /// expressed using the same representation as SIZE.
  pub horiz_pre: u8,

  /// The vertical precision of the data, in centimeters,
  /// expressed using the sane representation as for SIZE.
  pub vert_pre: u8,

  /// The latitude of the center of the sphere described by the
  /// SIZE field, in thousandths of a second of arc. 2^31 represents
  /// the equator, numbers above that are north latitude.
  pub latitude: i32,

  /// The longitude of the center of the sphere described by the
  /// SIZE field, in thousandths of a second of arc, rounded away from
  /// the prime meridian. 2^31 represents the prime meridian; numbers
  /// above that are east longitude.
  pub longitude: i32,

  /// The altitude of the center of the sphere described by the SIZE
  /// field, in centimeters, from a base of 100,000m below the
  /// [WGS 84] reference spheroid used by GPS (semimajor axis
  /// a=6378137.0, reciprocal flattening rf=298.257223563).
  pub altitude: i32,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct SRV {
  /// The priority of this target host.  A client MUST attempt to
  /// contact the target host with the lowest-numbered priority it can
  /// reach; target hosts with the same priority SHOULD be tried in an
  /// order defined by the weight field.
  pub priority: u16,

  /// A server selection mechanism.  The weight field specifies a
  /// relative weight for entries with the same priority. Larger
  /// weights SHOULD be given a proportionately higher probability of
  /// being selected.
  pub weight: u16,

  /// The port on this target host of this service.
  pub port: u16,

  /// The domain name of the target host.  There MUST be one or more
  /// address records for this name, the name MUST NOT be an alias (in
  /// the sense of RFC 1034 or RFC 2181).
  pub target: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct NAPTR {
  /// A 16-bit unsigned integer specifying the order in which the NAPTR
  /// records MUST be processed in order to accurately represent the
  /// ordered list of Rules.
  pub order: u16,

  /// A 16-bit unsigned integer that specifies the order in which NAPTR
  /// records with equal Order values SHOULD be processed, low numbers
  /// being processed before high numbers.
  pub preference: u16,

  /// A <character-string> containing flags to control aspects of the
  /// rewriting and interpretation of the fields in the record.
  pub flags: Vec<u8>,

  /// A <character-string> that specifies the Service Parameters
  /// applicable to this this delegation path.
  pub services: Vec<u8>,

  /// A <character-string> containing a substitution expression that is
  /// applied to the original string held by the client in order to
  /// construct the next domain name to lookup.
  pub regexp: Vec<u8>,

  /// A <domain-name> which is the next domain-name to query for
  /// depending on the potential values found in the flags field.
  pub replacement: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct KX {
  /// Specifies the preference given to this RR among other KX records
  /// at the same owner.  Lower values are preferred.
  pub preference: u16,

  /// A <domain-name> which specifies a host willing to act as a mail
  /// exchange for the owner name. (MUST NOT be compressed)
  pub exchanger: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct CERT {
  /// The type field is the certificate type as defined in RFC4398 2.1
  pub type_: u16,

  /// The key tag field is the 16-bit value computed for the key embedded
  /// in the certificate, using the RRSIG Key Tag algorithm.
  pub key_tag: u16,

  /// The algorithm field has the same meaning as the algorithm field in
  /// DNSKEY and RRSIG RRs, except 0 => unknown.
  pub algorithm: u8,

  pub certificate: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct DNAME {
  /// Alias for a name and all its subnames, similar to CNAME which is
  /// exact match only.
  /// The DNAME RDATA target name MUST NOT be sent out in compressed form
  /// and MUST be downcased for DNS Security Extensions (DNSSEC)
  /// validation.
  pub dname: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct APL {
  /// May have zero or more items
  pub item: Vec<APLItem>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct APLItem {
  /// 1 for IPv4, 2 for IPv6
  pub address_family: u16,

  /// Coded prefix length, IPv4: 0-32, IPv6: 0-128
  pub prefix: u8,

  /// Negation flag, indicates the presence of the
  /// ! character in the textual format.
  pub negation: bool,

  /// Address family dependent part. IPv4: 0-4 bytes, IPv6: 0-16 bytes
  pub afd_part: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct DS {
  /// The Key Tag field lists the key tag of the DNSKEY RR referred to by
  /// the DS record, in network byte order.
  pub key_tag: u16,

  /// The Algorithm field lists the algorithm number of the DNSKEY RR
  /// referred to by the DS record.
  pub algorithm: u8,

  /// The DS RR refers to a DNSKEY RR by including a digest of that DNSKEY
  /// RR.  The Digest Type field identifies the algorithm used to construct
  /// the digest.
  pub digest_type: u8,

  /// The DS record refers to a DNSKEY RR by including a digest of that
  /// DNSKEY RR.
  pub digest: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct SSHFP {
  /// The algorithm of the public key, 0=reserved, 1=rsa, 2=dss
  pub algorithm: u8,

  /// The digest algorithm for the key fingerprint. 0=reserved, 1=sha1
  pub fp_type: u8,

  /// The message-digest algorithm is presumed to produce an opaque octet
  /// string output, which is placed as-is in the RDATA fingerprint field.
  pub fingerprint: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct IPSECKEY {
  /// Gateways listed in IPSECKEY records with lower precedence are to be
  /// attempted first.  Where there is a tie in precedence, the order
  /// should be non-deterministic.
  pub precedence: u8,

  /// The gateway type field indicates the format of the information that
  /// is stored in the gateway field. 0=none, 1=ipv4, 2=ipv6, 3=domain
  pub gateway_type: u8,

  /// The public key's cryptographic algorithm and determines the format
  /// of the public key field. 0=none, 1=dsa, 2=rsa
  pub algorithm: u8,

  /// The gateway field indicates a gateway to which an IPsec tunnel may be
  /// created in order to reach the entity named by this resource record.
  pub gateway: Vec<u8>,

  /// The public key with format defined by the 'algorithm'.
  pub public_key: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct RRSIG {
  /// The Type Covered field identifies the type of the RRset that is
  /// covered by this RRSIG record.
  pub type_covered: u16,

  /// The Algorithm Number field identifies the cryptographic algorithm
  /// used to create the signature.
  pub algorithm: u8,

  /// The Labels field specifies the number of labels in the original RRSIG
  /// RR owner name.
  pub labels: u8,

  /// The Original TTL field specifies the TTL of the covered RRset as it
  /// appears in the authoritative zone.
  pub original_ttl: u32,

  /// The Signature Expiration and Inception fields specify a validity
  /// period for the signature.
  pub signature_expiration: u32,
  pub signature_inception: u32,

  /// The Key Tag field contains the key tag value of the DNSKEY RR that
  /// validates this signature, in network byte order.
  pub key_tag: u16,

  /// The Signer's Name field value identifies the owner name of the DNSKEY
  /// RR that a validator is supposed to use to validate this signature.
  pub signers_name: String,

  /// The Signature field contains the cryptographic signature that covers
  /// the RRSIG RDATA (excluding the Signature field) and the RRset
  /// specified by the RRSIG owner name, RRSIG class, and RRSIG Type
  /// Covered field.
  pub signature: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct NSEC {
  /// The Next Domain field contains the next owner name (in the canonical
  /// ordering of the zone) that has authoritative data or contains a
  /// delegation point NS RRset.
  /// A sender MUST NOT use DNS name compression on the Next Domain Name
  /// field when transmitting an NSEC RR.
  pub next_domain_name: String,

  /// The Type Bit Maps field identifies the RRset types that exist at the
  /// NSEC RR's owner name.
  pub type_bit_maps: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct DNSKEY {
  /// Bit 7 of the Flags field is the Zone Key flag: 1 = zone, 0 = other
  /// Bit 15 of the Flags field is the Secure Entry Point flag, described
  /// in [RFC3757].
  pub flags: u16,

  /// The Protocol Field MUST have value 3
  pub protocol: u8,

  /// The Algorithm field identifies the public key's cryptographic
  /// algorithm and determines the format of the Public Key field.
  pub algorithm: u8,

  /// The Public Key Field holds the public key material.
  pub public_key: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct DHCID {
  /// Opaque DHCP data, to be interpreted by DHCP clients and servers
  pub dhcp_data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct NSEC3 {
  /// The Hash Algorithm field identifies the cryptographic hash algorithm
  /// used to construct the hash-value.
  pub hash_algorithm: u8,

  /// The Flags field contains 8 one-bit flags that can be used to indicate
  /// different processing.  All undefined flags must be zero.  The only
  /// flag defined by this specification is the Opt-Out flag.
  pub flags: u8,

  /// The Iterations field defines the number of additional times the hash
  /// function has been performed.
  pub iterations: u16,

  /// The Salt field is appended to the original owner name before hashing
  /// in order to defend against pre-calculated dictionary attacks.
  pub salt: Vec<u8>,

  /// The Next Hashed Owner Name field contains the next hashed owner name
  /// in hash order.
  pub next_hashed_owner_name: Vec<u8>,

  /// The Type Bit Maps field identifies the RRSet types that exist at the
  /// original owner name of the NSEC3 RR.
  pub type_bit_maps: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct NSEC3PARAM {
  /// The Hash Algorithm field identifies the cryptographic hash algorithm
  /// used to construct the hash-value.
  pub hash_algorithm: u8,

  /// The Flags field contains 8 one-bit flags that can be used to indicate
  /// different processing.  All undefined flags must be zero.  The only
  /// flag defined by this specification is the Opt-Out flag.
  pub flags: u8,

  /// The Iterations field defines the number of additional times the hash
  /// function has been performed.
  pub iterations: u16,

  /// The Salt field is appended to the original owner name before hashing
  /// in order to defend against pre-calculated dictionary attacks.
  pub salt: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct TLSA {
  /// Specifies the provided association that will be used to match the
  /// certificate presented in the TLS handshake.
  pub certificate_usage: u8,

  /// Specifies which part of the TLS certificate presented by the server
  /// will be matched against the association data
  pub selector: u8,

  /// Specifies how the certificate association is presented.
  pub matching_type: u8,

  /// The data to be matched: certificate, SubjectPublicKeyInfo, hash, etc
  pub certificate_association_data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct SMIMEA {
  /// The SMIMEA wire format and presentation format are the same as for
  /// the TLSA record as described in Section 2.1 of [RFC6698].

  /// Specifies the provided association that will be used to match the
  /// certificate presented in the TLS handshake.
  pub certificate_usage: u8,

  /// Specifies which part of the TLS certificate presented by the server
  /// will be matched against the association data
  pub selector: u8,

  /// Specifies how the certificate association is presented.
  pub matching_type: u8,

  /// The data to be matched: certificate, SubjectPublicKeyInfo, hash, etc
  pub certificate_association_data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct HIP {
  /// The PK cryptographic algorithm and the implied Public Key field format.
  pub public_key_algorithm: u8,

  /// The HIT is stored as a binary value in network byte order.
  pub hit: Vec<u8>,

  /// DSA (RFC2536), RSA (RFC3110), or ECDSA (RFC6605)
  pub public_key: Vec<u8>,

  /// One or more variable length domain-encoded values, which MUST NOT be
  /// compressed. Listed in order of preference.
  pub rendezvous_servers: Vec<String>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct OPENPGPKEY {
  /// The RDATA portion of an OPENPGPKEY resource record contains a single
  /// value consisting of a Transferable Public Key formatted as specified
  /// in [RFC4880].
  pub public_key: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct CSYNC {
  /// A copy of the 32-bit SOA serial number from the child zone.
  pub soa_serial: u32,

  /// Booleans that define operations affecting CSYNC record processing.
  pub flags: u16,

  /// The record types to be processed by the parental agent.
  pub type_bit_map: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct TKEY {
  /// The algorithm name is in the form of a domain name with the same
  /// meaning as in [RFC 2845].
  pub algorithm: String,

  /// The inception time and expiration times are in number of seconds
  /// since the beginning of 1 January 1970 GMT ignoring leap seconds
  /// treated as modulo 2**32 using ring arithmetic [RFC 1982].
  pub inception: u32,
  pub expiration: u32,

  /// The mode field specifies the general scheme for key agreement or the
  /// purpose of the TKEY DNS message.
  pub mode: u16,

  /// The error code field is an extended RCODE.
  pub error: u16,

  /// The key exchange data field. Depends on the mode.
  pub key_data: Vec<u8>,

  /// May be used in future extensions.
  pub other_data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct TSIG {
  /// Name of the algorithm in domain name syntax.
  pub algorithm_name: String,

  /// Seconds since 1-Jan-70 UTC.
  pub time_signed: u64, // 48 bits

  /// Seconds of error permitted in Time Signed.
  pub fudge: u16,

  /// Defined by Algorithm Name.
  pub mac: Vec<u8>,

  /// Empty unless Error == BADTIME
  pub other_data: Vec<u8>,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct URI {
  /// A client MUST attempt to contact the URI with the lowest-numbered
  /// priority it can reach, URIs with the same priority SHOULD be
  /// selected according to probabilities defined by the weight field.
  pub priority: u16,

  /// Specifies a relative weight for entries with the same priority.
  /// Larger weights SHOULD be given a proportionately higher probability
  /// of being selected.
  pub weight: u16,

  /// The URI of the target, enclosed in double-quotes, using the format
  /// specified in RFC3986
  pub target: String,
}

#[derive(Archive, Deserialize, Serialize, Clone)]
#[archive_attr(derive(CheckBytes))]
pub struct CAA {
  /// Issuer critical flag at bit 0, all others reserved for future use.
  pub flags: u8,

  /// The property identifier, a sequence of US-ASCII characters.
  pub tag: String,

  /// A sequence of octets representing the property value.
  pub value: Vec<u8>,
}
