use std::collections::HashMap;
use std::convert::TryFrom;
use std::fs;
use std::net::{SocketAddr, ToSocketAddrs};
use std::path::PathBuf;
use std::sync::{Arc, Mutex};
use std::thread;
use std::time::Duration;

use anyhow::{bail, Context, Result};
use async_lock::Barrier;
use async_net::{TcpListener, TcpStream, UdpSocket};
use bytes::BytesMut;
use hyper::{Client, Uri};
use smol::Task;
use tracing::{self, debug, info, warn};

use crate::filter::{filter, reader};
use crate::{cache, client, config, hyper_smol, listen_tcp, listen_udp, lookup};

/// TCP size header is 16 bits, so max theoretical size is 64k
static MAX_TCP_BYTES: u16 = 65535;

/// Hardcoded hostnames that we should block for technical reasons
static HARDCODED_BLOCKED_HOSTS: &'static [&'static str] = &[
    // Placeholder domain for users to check that Originz is working
    "test-blocked.origi.nz",
    // See https://support.mozilla.org/en-US/kb/canary-domain-use-application-dnsnet
    "use-application-dns.net",
];

/// Runs the server. Separate from main.rs to simplify testing in benchmarks
pub struct Runner {
    config: config::Config,
    storage_dir: PathBuf,
    tcp_listener: TcpListener,
    udp_sock: UdpSocket,
}

/// The data associated with an incoming request,
/// either a bytes payload for UDP or a stream (with response socket) for TCP
#[derive(Debug)]
pub enum RequestData {
    Udp(BytesMut),
    Tcp(TcpStream),
}

/// The request and the source that sent the request.
#[derive(Debug)]
pub struct RequestMsg {
    pub src: SocketAddr,
    pub data: RequestData,
}

impl Runner {
    /// Creates a new `Runner` instance after setting up any listen sockets.
    pub async fn new(config_path: String, config: config::Config) -> Result<Runner> {
        // Initialize listen socket up-front so that upstream can quickly downgrade the user to non-root if needed.
        let dns_listen_host = config.listen_dns.trim();
        let dns_listen_addr = dns_listen_host
            .to_socket_addrs()?
            .next()
            .with_context(|| format!("Invalid listen_dns address: {}", dns_listen_host))?;

        let storage_dir = PathBuf::from(config.storage.trim());
        if !storage_dir.exists() {
            fs::create_dir(&storage_dir).with_context(|| {
                format!("Failed to create storage directory: {:?}", storage_dir)
            })?;
        } else if storage_dir.is_file() {
            bail!(
                "Specified .storage path in {} is a regular file: {:?}",
                config_path,
                storage_dir
            );
        }

        // Set up sockets up-front. This is mainly to support listening on an ephemeral port (:0),
        // where tcp_addr/udp_addr are unknown until the listeners have been initialized.
        let tcp_listener = TcpListener::bind(dns_listen_addr)
            .await
            .with_context(|| format!("Failed to listen on TCP {}", dns_listen_addr))?;
        let udp_sock = UdpSocket::bind(dns_listen_addr)
            .await
            .with_context(|| format!("Failed to listen on UDP {}", dns_listen_addr))?;

        Ok(Runner {
            config,
            storage_dir,
            tcp_listener,
            udp_sock,
        })
    }

    /// Returns the listen endpoint for the TCP socket.
    /// This is for testing cases, where an ephemeral listen port is being used.
    pub fn get_tcp_endpoint(self: &Runner) -> Result<SocketAddr> {
        self.tcp_listener
            .local_addr()
            .with_context(|| "Couldn't get local TCP socket address")
    }

    /// Returns the listen endpoint for the UDP socket.
    /// This is for testing cases, where an ephemeral listen port is being used.
    pub fn get_udp_endpoint(self: &Runner) -> Result<SocketAddr> {
        self.udp_sock
            .local_addr()
            .with_context(|| "Couldn't get local UDP socket address")
    }

    /// Runs (and consumes) the server. This should block until one of the following occurs:
    /// - A fatal error
    /// - `stop` has received an event or the associated sender has been closed
    pub async fn run(self, stop: Arc<Barrier>) -> Result<()> {
        // Set up a channel for handling cache lookups/updates
        // We use the channel pattern here to avoid issues around a shared/mutexed cache object, which
        // internally may need to make await calls within the mutex lock guard. Rust doesn't like that.
        let (cache_tx, cache_task) = cache::task::start_cache(&self.config)?;

        // Set up a channel for receiving requests from listen sockets and making them availaboe for worker threads.
        let (mut server_query_tx, server_query_rx): (
            async_channel::Sender<RequestMsg>,
            async_channel::Receiver<RequestMsg>,
        ) = async_channel::bounded(32);
        // Lock the receive end: Shared by worker threads
        let server_query_rx = Arc::new(Mutex::new(server_query_rx));

        // Spawn task to listen for incoming TCP requests
        let tcp_listener_task: Task<()>;
        {
            let mut tcp_listener_move = self.tcp_listener;
            let mut server_query_tx_copy = server_query_tx.clone();
            tcp_listener_task = smol::spawn(async move {
                listen_tcp::listen_tcp(&mut tcp_listener_move, &mut server_query_tx_copy)
                    .await
                    .expect("TCP listen failed");
            });
        }

        // Spawn task to listen for incoming UDP requests
        let udp_listener_task: Task<()>;
        let udp_endpoint = self.udp_sock.local_addr()?;
        {
            let mut udp_sock_copy = self.udp_sock.clone();
            udp_listener_task = smol::spawn(async move {
                // Move the original server_query_tx rather than creating another clone.
                // This ensures that the queue will close when the listeners are gone.
                listen_udp::listen_udp(&mut udp_sock_copy, &mut server_query_tx)
                    .await
                    .expect("UDP listen failed");
            });
        }

        // Spawn task to trigger timer-based filter refreshes
        let (filter_refresh_tx, filter_refresh_rx): (
            async_channel::Sender<()>,
            async_channel::Receiver<()>,
        ) = async_channel::bounded(1);

        // Kick an initial refresh notification
        filter_refresh_tx.send(()).await?;
        let filter_refresh_timer_task = if self.config.filter_refresh_seconds == 0 {
            info!("Skipping filter refresh loop: filter_refresh_seconds=0");
            None
        } else {
            let filter_refresh = Duration::from_secs(self.config.filter_refresh_seconds);
            Some(smol::spawn(async move {
                loop {
                    async_io::Timer::after(filter_refresh).await;
                    debug!("Notifying filter refresh");
                    if let Err(e) = filter_refresh_tx.send(()).await {
                        warn!("Got error when sending refresh to queue: {}", e);
                        // Continue loop in case this is some kind of temporary issue...
                    }
                }
            }))
        };

        let filters_dir = self.storage_dir.join("filters");
        if !filters_dir.exists() {
            fs::create_dir(&filters_dir).with_context(|| {
                format!(
                    "Failed to create filter download directory: {:?}",
                    filters_dir
                )
            })?;
        } else if filters_dir.is_file() {
            bail!(
                "Filter download directory configured storage path is a regular file: {:?}",
                filters_dir
            );
        }

        let mut filter = filter::Filter::new();
        // Set up the hardcoded values first - for now they take priority over any manual configuration.
        // There isn't a good reason for a user to override a Originz test domain, for example.
        filter.set_hardcoded_block(HARDCODED_BLOCKED_HOSTS.into())?;
        let filter = Arc::new(Mutex::new(filter));

        let mut thread_handles = Vec::new();

        // Set up a thread for periodically reloading filters.
        // Avoids only using a async task because they dislike interacting with mutexes.
        {
            let resolver = client::upstream::parse_upstreams(cache_tx.clone(), &self.config.upstreams)?;
            let filter_copy = filter.clone();
            // For now we assume that the list of filters is constant.
            // Once we start supporting live updates of the filter list, this will need to be restructured.
            let mut filters_copy = self.config.filters.clone();
            if !self.config.overrides.is_empty() || !self.config.blocks.is_empty() || !self.config.allows.is_empty() {
                filters_copy.insert(
                    "".to_string(),
                    config::ConfigFilter{
                        overrides: self.config.overrides,
                        blocks: self.config.blocks,
                        allows: self.config.allows,
                        applies_to: "".to_string(),
                    }
                );
            }
            thread_handles.push(
                thread::Builder::new()
                    .name("filter-worker".to_string())
                    .spawn(move || {
                        smol::block_on(async move {
                            let fetch_client = hyper_smol::client_originz(resolver, false, false, 4096);
                            let span = tracing::info_span!("filter-worker");
                            let _enter = span.enter();
                            // Ensure that the filters are loaded in on the first pass, even if they
                            // weren't redownloaded due to an up-to-date local cached copy.
                            let mut force_load = true;
                            loop {
                                if let Ok(_) = filter_refresh_rx.recv().await {
                                    debug!("Refreshing filters with force_load={}", force_load);
                                    refresh_filters(
                                        &filters_dir,
                                        &filter_copy,
                                        &filters_copy,
                                        &fetch_client,
                                        force_load
                                    ).await;
                                    force_load = false;
                                    debug!("Filters refreshed");
                                } else {
                                    info!("Exiting filter refresh thread: timer queue has closed.");
                                    break;
                                }
                            }
                        });
                    })?
            );
        }

        // Start independent threads to handle received requests, query upstreams, and send back responses
        let response_timeout = Duration::from_millis(1000);
        for i in 0..self.config.workers {
            let mut lookup = lookup::Lookup::new(
                client::upstream::parse_upstreams(cache_tx.clone(), &self.config.upstreams)?,
                filter.clone(),
            );
            let server_query_rx_copy = server_query_rx.clone();
            let udp_sock_copy = self.udp_sock.clone();
            let idx = i;
            let thread_fn = move || {
                smol::block_on(async move {
                    let mut tcp_buf = BytesMut::with_capacity(MAX_TCP_BYTES as usize);
                    // Shows up as 'query-worker{idx=5}' in logs
                    let span = tracing::info_span!("query-worker", idx);
                    let _enter = span.enter();
                    loop {
                        if !handle_next_request(
                            &server_query_rx_copy,
                            &mut lookup,
                            &mut tcp_buf,
                            &udp_sock_copy,
                            &response_timeout
                        ).await {
                            warn!("Worker thread exiting, this is expected only if we're shutting down");
                            break;
                        }
                    }
                });
            };
            thread_handles.push(
                thread::Builder::new()
                    .name(format!("query-worker-{}", i))
                    .spawn(thread_fn)?,
            );
        }

        // We just log the UDP endpoint - it should be the same as the TCP endpoint unless port 0 was used
        info!("Waiting for clients at {:?}", udp_endpoint);

        // Wait indefinitely for the stop barrier to receive an event or be closed
        stop.wait().await;
        info!("Shutting down: stop signal received");

        // First, drop the task handles for the TCP/UDP listeners and filter refresh timer so that they stop.
        drop(tcp_listener_task);
        drop(udp_listener_task);
        if let Some(task) = filter_refresh_timer_task {
            drop(task);
        }

        // Now that the listeners have stopped, the processing queue should eventually empty.
        // Once empty, the queue should return None to the threads since its inputs (the listeners) have all been dropped.
        // At that point the threads should exit on their own.
        for thread in thread_handles {
            let thread_name = if let Some(name) = &thread.thread().name() {
                name.to_string()
            } else {
                "???".to_string()
            };
            info!("Waiting for thread to exit: {}", thread_name);
            // Would use with_context but the error type is weird
            match thread.join().err() {
                Some(e) => bail!("Failed to wait for thread to exit: {} {:?}", thread_name, e),
                None => {}
            }
        }

        // Finally shut down the cache once nobody should be using it.
        drop(cache_task);

        info!("Shutdown complete");
        Ok(())
    }
}

/// Sets up a thread for periodically reloading filters.
/// Avoids using an async task because they dislike interacting with mutexes.
async fn refresh_filters(
    filters_dir: &PathBuf,
    filter: &Arc<Mutex<filter::Filter>>,
    filter_configs: &HashMap<String, config::ConfigFilter>,
    fetch_client: &Client<hyper_smol::SmolConnector>,
    force_update: bool
) {
    for (_name, conf) in filter_configs {
        let mut filters = Vec::new();
        // For each override/block, check the URL (or do nothing for a local path)
        // Then re-read the result from disk.
        for entry in &conf.overrides {
            if let Some(filter) = refresh_filter(
                fetch_client,
                filters_dir,
                entry,
                reader::FilterType::OVERRIDE,
                force_update
            ).await {
                filters.push(filter);
            }
        }
        for entry in &conf.blocks {
            if let Some(filter) = refresh_filter(
                fetch_client,
                filters_dir,
                entry,
                reader::FilterType::BLOCK,
                force_update
            ).await {
                filters.push(filter);
            }
        }
        if let Ok(mut filter_locked) = filter.lock() {
            filter_locked.update_entries(filters);
        } else {
            warn!("Failed to lock filter for entry update");
        }
    }
}

async fn refresh_filter(
    fetch_client: &Client<hyper_smol::SmolConnector>,
    fetch_dir: &PathBuf,
    filter_path_or_url: &String,
    filter_type: reader::FilterType,
    force_update: bool,
) -> Option<reader::FilterEntries> {
    if let Ok(filter_uri) = Uri::try_from(filter_path_or_url) {
        // Filesystem paths can get parsed as URLs with no scheme
        if filter_uri.scheme() != None {
            // Looks like a URL, check for updated version to download
            if let Ok((local_path, updated)) = filter::update_url(
                fetch_client,
                fetch_dir,
                filter_path_or_url,
                10000,
            ).await {
                if updated || force_update {
                    // The file had an update to download, or an update/read is being forced
                    if let Ok(filter) = reader::read(
                        filter_type,
                        reader::FileInfo {
                            source_path: filter_path_or_url.clone(),
                            local_path
                        }
                    ) {
                        return Some(filter);
                    }
                }
            }
            return None;
        }
    }

    if let Ok(filter) = reader::read(
        filter_type,
        reader::FileInfo {
            source_path: filter_path_or_url.clone(),
            local_path: filter_path_or_url.clone(),
        }
    ) {
        return Some(filter);
    }
    None
}

async fn handle_next_request(
    server_query_rx: &Mutex<async_channel::Receiver<RequestMsg>>,
    lookup: &mut lookup::Lookup,
    tcp_buf: &mut BytesMut,
    udp_sock: &UdpSocket,
    response_timeout: &Duration
) -> bool {
    let request: RequestMsg;
    match server_query_rx.lock() {
        Err(e) => {
            // Not expecting this to happen, but just in case lets try to keep going
            warn!("Failed to lock receive queue, trying again: {:?}", e);
            return true; // Continue thread
        }
        Ok(server_query_rx_lock) => {
            // Grab a request from the queue, then release the lock
            if let Ok(got_request) = server_query_rx_lock.recv().await {
                request = got_request;
            } else {
                info!("Exiting handler thread: request queue has closed.");
                return false; // EXIT THREAD
            }
        }
    }

    match request.data {
        RequestData::Udp(buf) => {
            listen_udp::handle_udp_request(lookup, request.src, buf, udp_sock)
                .await
        }
        RequestData::Tcp(tcp_stream) => {
            listen_tcp::handle_tcp_request(
                lookup,
                response_timeout,
                request.src,
                tcp_stream,
                tcp_buf,
            )
                .await;
            // Reset buffer size afterwards (capacity should stay the same)
            tcp_buf.resize(MAX_TCP_BYTES as usize, 0);
        }
    }
    true // Continue thread
}
