use std::path::{Path, PathBuf};
use std::vec::Vec;

use anyhow::{Context, Result};
use hyper::Client;
use sha2::{Digest, Sha256};

use crate::filter::{path, reader, updater};
use crate::{fetcher, hyper_smol};

/// An iterator that goes over the parent domains of a provided child domain.
/// For example, www.domain.com => [www.domain.com, domain.com, com]
struct DomainParentIter<'a> {
    domain: &'a String,
    start_idx: usize,
}

impl<'a> DomainParentIter<'a> {
    fn new(domain: &'a String) -> DomainParentIter<'a> {
        DomainParentIter {
            domain,
            start_idx: 0,
        }
    }
}

impl<'a> Iterator for DomainParentIter<'a> {
    type Item = &'a str;

    fn next(&mut self) -> Option<&'a str> {
        if self.start_idx >= self.domain.len() {
            // Seeked past end of domain string, nothing left
            None
        } else {
            // Collect this result: everything from start_idx
            let remainder = &self.domain[self.start_idx..];
            // Update start for next result
            match remainder.find('.') {
                Some(idx) => {
                    // idx is within remainder's address space, which starts at start_idx
                    // add 1 to seek past the '.' itself
                    self.start_idx += idx + 1;
                }
                None => {
                    self.start_idx = self.domain.len();
                }
            }
            Some(remainder)
        }
    }
}

pub struct Filter {
    overrides: Vec<reader::FilterEntries>,
    blocks: Vec<reader::FilterEntries>,
}

impl Filter {
    pub fn new() -> Filter {
        Filter {
            overrides: vec![],
            blocks: vec![],
        }
    }

    pub fn update_entries(self: &mut Filter, entrieses: Vec<reader::FilterEntries>) {
        for entries in entrieses {
            match entries.filter_type {
                reader::FilterType::BLOCK => upsert_entries(&mut self.blocks, entries),
                reader::FilterType::OVERRIDE => upsert_entries(&mut self.overrides, entries),
            }
        }
    }

    pub fn set_hardcoded_block(self: &mut Filter, block_names: &[&str]) -> Result<()> {
        let hardcoded_entries = reader::block_hardcoded(block_names)?;
        upsert_entries(&mut self.blocks, hardcoded_entries);
        Ok(())
    }

    pub fn check(self: &Filter, host: &String) -> Option<(&Option<reader::FileInfo>, &reader::FilterEntry)> {
        // Go over domains in ancestor order, checking all blocks for each ancestor.
        // For example check all files for 'www.example.com', then each again for 'example.com'.
        // This allows file B with 'www.example.com' to take precedence over file A with 'example.com'

        // Meanwhile if two files mention the exact same domain then the first file in the list wins.
        // So if file A says "127.0.0.1" and file B says "172.16.0.1" then "127.0.0.1" wins.

        for domain_str in DomainParentIter::new(&host) {
            let domain = domain_str.to_string();
            for override_entry in &self.overrides {
                match override_entry.get(&domain) {
                    // Found in an override file: Tell upstream to let it through or use provided override value
                    Some(entry) => return Some((&override_entry.info, entry)),
                    None => {}
                }
            }
            for block in &self.blocks {
                match block.get(&domain) {
                    // Found in block: Tell upstream to block it or use filter-provided override
                    Some(entry) => return Some((&block.info, entry)),
                    None => {}
                }
            }
        }

        return None;
    }
}

/// Returns the local path where the file was downloaded,
/// and whether the file was updated (true) or the update was skipped (false)
pub async fn update_url(
    fetch_client: &Client<hyper_smol::SmolConnector>,
    filters_dir: &PathBuf,
    uri_string: &String,
    timeout_ms: u64,
) -> Result<(String, bool)> {
    let fetcher = fetcher::Fetcher::new(10 * 1024 * 1024, None);
    // We download files to the exact SHA of the URL string we were provided.
    // This is an easy way to avoid filename collisions in URLs: example1.com/hosts vs example2.com/hosts
    // If the user changes the URL string then that changes the SHA, perfect for "cache invalidation" purposes.
    let hosts_path_sha = Sha256::digest(uri_string.as_bytes());
    let download_path = Path::new(filters_dir).join(format!(
        "{:x}.sha256.{}",
        hosts_path_sha,
        path::ZSTD_EXTENSION
    ));
    let downloaded = updater::update_file(
        fetch_client,
        &fetcher,
        uri_string,
        download_path.as_path(),
        timeout_ms,
    )
        .await?;
    Ok((
        download_path
            .to_str()
            .with_context(|| format!("busted download path: {:?}", download_path))?
        .to_string(),
        downloaded
    ))
}

fn upsert_entries(entries: &mut Vec<reader::FilterEntries>, new_entry: reader::FilterEntries) {
    if let Some(new_file_info) = &new_entry.info {
        // Before adding a new file entry, check for an existing file entry to be replaced/updated.
        for i in 0..entries.len() {
            let entry = entries.get(i).expect("incoherent vector size");
            if let Some(existing_file_info) = &entry.info {
                if existing_file_info.local_path == new_file_info.local_path {
                    // Delete or replace existing version
                    if new_entry.is_empty() {
                        entries.remove(i);
                    } else {
                        entries.insert(i, new_entry);
                    }
                    return;
                }
            }
        }
    }
    // Add new entry
    if !new_entry.is_empty() {
        entries.push(new_entry);
    }
}

#[cfg(test)]
mod tests {
    use super::*;

    #[test]
    fn iter_empty() {
        let domain = "".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(None, iter.next());
    }

    #[test]
    fn iter_com() {
        let domain = "com".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(Some("com"), iter.next());
        assert_eq!(None, iter.next());
    }

    #[test]
    fn iter_domaincom() {
        let domain = "domain.com".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(Some("domain.com"), iter.next());
        assert_eq!(Some("com"), iter.next());
        assert_eq!(None, iter.next());
    }

    #[test]
    fn iter_wwwdomaincom() {
        let domain = "www.domain.com".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(Some("www.domain.com"), iter.next());
        assert_eq!(Some("domain.com"), iter.next());
        assert_eq!(Some("com"), iter.next());
        assert_eq!(None, iter.next());
    }

    #[test]
    fn iter_wwwngeeknz() {
        let domain = "www.n.geek.nz".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(Some("www.n.geek.nz"), iter.next());
        assert_eq!(Some("n.geek.nz"), iter.next());
        assert_eq!(Some("geek.nz"), iter.next());
        assert_eq!(Some("nz"), iter.next());
        assert_eq!(None, iter.next());
    }

    #[test]
    fn iter_averylongteststringwithmanysegments() {
        let domain = "a.very-long.test.string.with-many.segments".to_string();
        let mut iter = DomainParentIter::new(&domain);
        assert_eq!(
            Some("a.very-long.test.string.with-many.segments"),
            iter.next()
        );
        assert_eq!(
            Some("very-long.test.string.with-many.segments"),
            iter.next()
        );
        assert_eq!(Some("test.string.with-many.segments"), iter.next());
        assert_eq!(Some("string.with-many.segments"), iter.next());
        assert_eq!(Some("with-many.segments"), iter.next());
        assert_eq!(Some("segments"), iter.next());
        assert_eq!(None, iter.next());
    }
}
