use std::convert::TryFrom;
use std::fmt;
use std::iter::Iterator;
use std::net::SocketAddr;
use std::vec::Vec;

use anyhow::{bail, Context, Result};
use hyper::Uri;
use tracing::{self, trace};

use crate::cache;
use crate::client::{https, system, tcp, udp, DnsClient};
use crate::resolver::Resolver;

static DEFAULT_UDP_TCP_PORT: u16 = 53;
static DEFAULT_TIMEOUT_MS: u64 = 10000;

#[derive(Clone, Debug, PartialEq)]
enum UpstreamType {
    /// System resolver
    System,
    /// UDP-only
    Udp,
    /// TCP-only
    Tcp,
    /// DoH
    Https,
}

struct UpstreamInfo {
    uri: Uri,
    upstream_type: UpstreamType,
}

impl fmt::Debug for UpstreamInfo {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        f.write_str(self.uri.to_string().as_str())
    }
}

impl UpstreamInfo {
    fn new(upstream_orig: &String) -> Result<(UpstreamInfo, Option<UpstreamInfo>)> {
        let upstream = upstream_orig.trim();
        let uri = Uri::try_from(upstream)
            .with_context(|| format!("Failed to parse upstream URI: {}", upstream))?;
        match uri.scheme_str() {
            None => {
                if let Some("system") = uri.host() {
                    // URI as 'system'
                    Ok((UpstreamInfo {
                        uri,
                        upstream_type: UpstreamType::System,
                    }, None))
                } else {
                    // URI as '<ip>': assume udp+tcp
                    Ok((
                        UpstreamInfo {
                            uri: uri.clone(),
                            upstream_type: UpstreamType::Udp,
                        },
                        Some(UpstreamInfo {
                            uri,
                            upstream_type: UpstreamType::Tcp,
                        })
                    ))
                }
            },
            Some(scheme) => match scheme {
                "udp+tcp" => Ok((
                    UpstreamInfo {
                        uri: uri.clone(),
                        upstream_type: UpstreamType::Udp,
                    },
                    Some(UpstreamInfo {
                        uri,
                        upstream_type: UpstreamType::Tcp,
                    })
                )),
                "udp" => Ok((UpstreamInfo {
                    uri,
                    upstream_type: UpstreamType::Udp,
                }, None)),
                "tcp" => Ok((UpstreamInfo {
                    uri,
                    upstream_type: UpstreamType::Tcp,
                }, None)),
                "https" => Ok((UpstreamInfo {
                    uri,
                    upstream_type: UpstreamType::Https,
                }, None)),
                "tls" => bail!("TODO(#5) DNS-over-TLS is not yet supported for upstream: {}", upstream),
                other => bail!(format!("Unsupported upstream URI scheme '{}', expected one of: 'system', '<ip>', 'udp+tcp://<ip>', 'udp://<ip>', 'tcp://<ip>', or 'https://<ip/host>'", other)),
            },
        }
    }
}

/// Convert the config-provided upstream strings into a list of prepared client instances.
pub fn parse_upstreams(
    cache_tx: async_channel::Sender<cache::task::CacheMsg>,
    config_upstreams: &Vec<String>,
) -> Result<Resolver> {
    if config_upstreams.is_empty() {
        bail!("Upstreams list is empty, at least one upstream is required");
    }
    let mut upstreams = Vec::new();
    // Parse upstreams, check if URIs are provided as an IP or a hostname
    for config_upstream in config_upstreams {
        match UpstreamInfo::new(config_upstream)? {
            (upstream_info1, Some(upstream_info2)) => {
                trace!(
                    "Parsed upstreams: {:?} + {:?}",
                    upstream_info1, upstream_info2
                );
                upstreams.push(upstream_info1);
                upstreams.push(upstream_info2);
            }
            (upstream_info, None) => {
                trace!("Parsed upstream: {:?}", upstream_info);
                upstreams.push(upstream_info);
            }
        }
    }

    // "Bootstrap" upstreams are used for internal hostname resolutions for client endpoints.
    // For example, if a DoH client is specified as a hostname (https://example.com/dns-query), we need a separate "bootstrap" client for resolving that hostname.
    // Meanwhile, other internal lookups for e.g. filter downloads can use the full list of upstreams.
    // We also use these for other internet access, such as when downloading filters that were specified as URLs.
    let mut ip_client_found = false;
    for upstream_info in &mut upstreams {
        if upstream_info.upstream_type != UpstreamType::Https {
            ip_client_found = true;
        }
    }
    if !ip_client_found {
        bail!(
            r#"At least one upstream server must be specified as an IP.
For example, given a primary DoH upstream of 'https://example.com', add a secondary upstream of e.g. '1.1.1.1' which can then be used to resolve 'example.com'.
Configured upstreams are: {:?}"#,
            upstreams
        );
    }

    let mut clients: Vec<Box<dyn DnsClient + Send + 'static>> = Vec::new();
    for upstream_info in &upstreams {
        if upstream_info.upstream_type == UpstreamType::Https {
            // Build a separate copy of the IP-based "bootstrap" clients so that this https client can look up its upstream as described above.
            let mut bootstrap_clients: Vec<Box<dyn DnsClient + Send + 'static>> = Vec::new();
            for upstream_info in &upstreams {
                if upstream_info.upstream_type != UpstreamType::Https {
                    bootstrap_clients.push(to_client(upstream_info)?);
                }
            }
            let internal_resolver = Resolver::new(cache_tx.clone(), bootstrap_clients);
            clients.push(Box::new(https::Client::new(
                upstream_info.uri.clone(),
                internal_resolver,
                DEFAULT_TIMEOUT_MS,
            )?));
        } else {
            clients.push(to_client(upstream_info)?);
        }
    }
    Ok(Resolver::new(cache_tx, clients))
}

fn to_client(upstream_info: &UpstreamInfo) -> Result<Box<dyn DnsClient + Send>> {
    match &upstream_info.upstream_type {
        UpstreamType::System => Ok(Box::new(system::Client::new())),
        UpstreamType::Udp => Ok(Box::new(udp::Client::new(
            to_addr(&upstream_info.uri, DEFAULT_UDP_TCP_PORT)?.with_context(|| {
                format!(
                    "UDP upstreams must be specified as an IP address: {:?}",
                    upstream_info.uri
                )
            })?,
            DEFAULT_TIMEOUT_MS,
        ))),
        UpstreamType::Tcp => Ok(Box::new(tcp::Client::new(
            to_addr(&upstream_info.uri, DEFAULT_UDP_TCP_PORT)?.with_context(|| {
                format!(
                    "TCP upstreams must be specified as an IP address: {:?}",
                    upstream_info.uri
                )
            })?,
            DEFAULT_TIMEOUT_MS,
        ))),
        other => bail!("{:?} client requires a bootstrap resolver", other),
    }
}

/// If the URI is provided as an IP, extract it, otherwise return `None`.
fn to_addr(uri: &Uri, default_port: u16) -> Result<Option<SocketAddr>> {
    let authority = uri
        .authority()
        .with_context(|| format!("Missing authority in upstream: {}", uri))?;
    let ip_addr = authority.host().parse();
    Ok(ip_addr.map_or(None, |ip| {
        Some(SocketAddr::new(
            ip,
            authority.port_u16().unwrap_or(default_port),
        ))
    }))
}
