use crate::{OrdesCons, OrdesPop, OrdesPush, OrdesRest};
// use std::array::IntoIter;
use core::array::IntoIter;

pub struct ConstCheck<const CHECK: bool>;

pub trait True {}
impl True for ConstCheck<true> {}

impl<T, const N: usize> OrdesPush<T> for [T; N]
where
    ConstCheck<{ N < core::usize::MAX }>: True,
    [T; N + 1]: Sized,
{
    type Output = [T; N + 1];

    fn push(self, input: T) -> Self::Output {
        let mut iter = IntoIter::new(self).chain(IntoIter::new([input]));
        [(); N + 1].map(move |()| iter.next().unwrap())
    }
}

impl<T, const N: usize> OrdesCons<T> for [T; N]
where
    ConstCheck<{ N < core::usize::MAX }>: True,
    [T; N + 1]: Sized,
{
    type Output = [T; N + 1];

    fn cons(self, input: T) -> Self::Output {
        let mut iter = IntoIter::new([input]).chain(IntoIter::new(self));
        [(); N + 1].map(|()| iter.next().unwrap())
    }
}

impl<T, const N: usize> OrdesPop for [T; N]
where
    ConstCheck<{ N > 1 }>: True,
    [T; N - 1]: Sized,
{
    type Newlen = [T; N - 1];
    type Output = T;

    fn pop(self) -> (Self::Newlen, Self::Output) {
        let mut iter = IntoIter::new(self);
        (
            [(); N - 1].map(|()| iter.next().unwrap()),
            iter.next().unwrap(),
        )
    }
}

impl<T, const N: usize> OrdesRest for [T; N]
where
    ConstCheck<{ N > 1 }>: True,
    [T; N - 1]: Sized,
{
    type Newlen = [T; N - 1];
    type Output = T;

    fn rest(self) -> (Self::Newlen, Self::Output) {
        let mut iter = IntoIter::new(self);
        let first = iter.next().unwrap();
        ([(); N - 1].map(|()| iter.next().unwrap()), first)
    }
}

impl<T> OrdesPop for [T; 1] {
    type Newlen = ();
    type Output = T;

    fn pop(self) -> ((), T) {
        ((), IntoIter::new(self).next().unwrap())
    }
}

impl<T> OrdesRest for [T; 1] {
    type Newlen = ();
    type Output = T;

    fn rest(self) -> ((), T) {
        ((), IntoIter::new(self).next().unwrap())
    }
}

pub trait OrdesInsert<const I: usize, Input> {
    type Output;

    unsafe fn insert(self, input: Input) -> Self::Output;
}

impl<T, const N: usize, const I: usize> OrdesInsert<I, T> for [T; N]
where
    ConstCheck<{ 0 < N }>: True,
    ConstCheck<{ N < core::usize::MAX }>: True,
    ConstCheck<{ I < N }>: True,
    [T; N + 1]: Sized,
    [T; I]: Sized,
    [T; N - I]: Sized,
{
    type Output = [T; N + 1];

    unsafe fn insert(self, input: T) -> Self::Output {
        let mut self_iter = IntoIter::new(self);
        let first_part = collect_into_array_copy::<_, I>(&mut self_iter).unwrap();
        let second_part = collect_into_array_copy::<_, { N - I }>(&mut self_iter).unwrap();
        let mut iter = IntoIter::new(first_part)
            .chain(IntoIter::new([input]))
            .chain(IntoIter::new(second_part));
        [(); N + 1].map(move |()| iter.next().unwrap())
    }
}

use core::mem::MaybeUninit;

fn collect_into_array_copy<I: Iterator, const N: usize>(iter: &mut I) -> Option<[I::Item; N]> {
    struct Guard<T, const N: usize> {
        ptr: *mut T,
        initialised: usize,
    }

    impl<T, const N: usize> Drop for Guard<T, N> {
        fn drop(&mut self) {
            debug_assert!(self.initialised <= N);
            let initialised_part = core::ptr::slice_from_raw_parts_mut(self.ptr, self.initialised);
            unsafe {
                core::ptr::drop_in_place(initialised_part);
            }
        }
    }

    let mut array = MaybeUninit::uninit_array::<N>();
    let mut guard: Guard<_, N> = Guard {
        ptr: MaybeUninit::slice_as_mut_ptr(&mut array),
        initialised: 0,
    };
    while let Some(item) = iter.next() {
        unsafe {
            array.get_unchecked_mut(guard.initialised).write(item);
        }
        guard.initialised += 1;
        if guard.initialised == N {
            core::mem::forget(guard);
            let out = unsafe { MaybeUninit::array_assume_init(array) };
            return Some(out);
        }
    }
    None
}
