use group::ff::PrimeField;
use pasta_curves::{arithmetic::CurveAffine, pallas};

/// Generator used in SinsemillaCommit randomness for note commitment
pub const GENERATOR: ([u8; 32], [u8; 32]) = (
    [
        19, 110, 252, 15, 72, 44, 2, 44, 124, 164, 20, 252, 92, 197, 158, 35, 242, 61, 111, 147,
        171, 159, 35, 205, 51, 69, 169, 40, 195, 6, 178, 38,
    ],
    [
        227, 55, 207, 34, 198, 83, 65, 195, 246, 220, 224, 35, 120, 192, 199, 121, 198, 225, 102,
        35, 47, 26, 84, 169, 76, 192, 109, 104, 75, 86, 222, 60,
    ],
);

/// Full-width z-values for GENERATOR
pub const Z: [u64; super::NUM_WINDOWS] = [
    253356, 149209, 114903, 10575, 6973, 30969, 55415, 206450, 18453, 24528, 13099, 213949, 29959,
    49929, 80867, 17465, 43715, 80241, 55983, 132629, 66101, 24136, 31372, 107975, 161748, 24107,
    72184, 9338, 232543, 13519, 33536, 32530, 130885, 41578, 18166, 91947, 59796, 35560, 5631,
    158600, 24695, 42654, 138331, 11268, 54733, 92869, 33770, 169166, 94853, 7006, 117687, 8073,
    11865, 15349, 186445, 7696, 25167, 30146, 277659, 53921, 19594, 41306, 30172, 8124, 46133,
    38659, 61965, 92134, 43958, 86662, 2047, 3542, 20976, 7411, 53574, 38271, 48233, 65338, 30516,
    41201, 40964, 8563, 36035, 6334, 176,
];

/// Full-width u-values for GENERATOR
pub const U: [[[u8; 32]; super::H]; super::NUM_WINDOWS] = [
    [
        [
            114, 245, 9, 0, 5, 43, 7, 118, 244, 43, 15, 200, 116, 210, 0, 24, 161, 28, 185, 152,
            216, 107, 69, 189, 146, 74, 72, 71, 79, 117, 226, 21,
        ],
        [
            198, 234, 165, 117, 60, 32, 158, 128, 245, 251, 167, 51, 94, 222, 84, 164, 93, 252, 6,
            172, 219, 79, 238, 23, 10, 251, 127, 190, 92, 247, 84, 7,
        ],
        [
            247, 1, 1, 220, 151, 35, 250, 127, 20, 73, 141, 158, 41, 239, 31, 53, 87, 128, 115, 67,
            105, 106, 106, 227, 24, 13, 193, 27, 11, 59, 156, 12,
        ],
        [
            182, 18, 28, 164, 222, 215, 93, 217, 191, 112, 205, 228, 10, 41, 225, 115, 16, 106,
            253, 86, 190, 158, 170, 163, 242, 182, 199, 91, 54, 66, 87, 45,
        ],
        [
            85, 233, 2, 251, 58, 85, 151, 41, 171, 17, 163, 129, 14, 53, 149, 170, 49, 242, 59, 95,
            195, 86, 152, 140, 191, 194, 135, 96, 61, 247, 224, 19,
        ],
        [
            9, 26, 141, 163, 30, 34, 167, 73, 162, 45, 13, 19, 213, 4, 222, 207, 61, 172, 202, 243,
            122, 40, 68, 21, 175, 213, 86, 213, 254, 227, 101, 15,
        ],
        [
            86, 112, 227, 11, 213, 132, 173, 162, 105, 136, 218, 34, 136, 252, 237, 146, 166, 198,
            142, 26, 59, 222, 162, 186, 51, 155, 102, 60, 62, 47, 222, 4,
        ],
        [
            86, 92, 45, 23, 81, 81, 61, 194, 172, 21, 21, 83, 38, 217, 43, 44, 235, 231, 88, 89,
            249, 41, 181, 159, 83, 22, 75, 138, 70, 126, 207, 16,
        ],
    ],
    [
        [
            54, 21, 251, 114, 55, 147, 20, 211, 33, 240, 173, 252, 164, 174, 133, 65, 206, 140, 93,
            20, 42, 49, 181, 104, 21, 120, 234, 196, 70, 100, 73, 42,
        ],
        [
            238, 4, 102, 31, 21, 150, 35, 139, 86, 204, 115, 87, 148, 16, 158, 189, 161, 53, 20,
            94, 238, 188, 243, 68, 91, 3, 117, 235, 245, 98, 110, 2,
        ],
        [
            71, 0, 3, 234, 114, 242, 86, 47, 159, 35, 98, 250, 218, 203, 192, 75, 15, 24, 202, 7,
            142, 99, 58, 79, 13, 158, 200, 188, 138, 62, 146, 46,
        ],
        [
            110, 160, 81, 239, 206, 110, 48, 235, 48, 53, 5, 146, 193, 225, 227, 102, 163, 114,
            159, 188, 166, 215, 24, 17, 124, 185, 233, 142, 97, 16, 247, 53,
        ],
        [
            5, 74, 101, 186, 155, 171, 161, 12, 114, 102, 185, 130, 202, 125, 97, 157, 214, 109,
            90, 55, 78, 151, 78, 233, 114, 32, 38, 167, 230, 47, 114, 43,
        ],
        [
            43, 145, 156, 91, 52, 42, 252, 255, 165, 106, 190, 113, 140, 48, 74, 56, 231, 74, 233,
            182, 111, 200, 49, 231, 139, 33, 38, 157, 185, 84, 190, 60,
        ],
        [
            35, 64, 214, 183, 170, 176, 163, 222, 136, 23, 64, 62, 90, 18, 25, 78, 171, 169, 203,
            242, 161, 64, 191, 204, 85, 79, 54, 130, 184, 99, 198, 38,
        ],
        [
            6, 238, 153, 49, 156, 138, 236, 90, 155, 111, 98, 17, 43, 249, 193, 244, 33, 204, 245,
            189, 174, 217, 152, 5, 133, 155, 139, 239, 164, 120, 180, 8,
        ],
    ],
    [
        [
            233, 129, 153, 212, 169, 73, 235, 118, 37, 78, 183, 222, 115, 222, 111, 212, 174, 24,
            102, 40, 189, 47, 223, 108, 201, 62, 19, 230, 93, 6, 207, 6,
        ],
        [
            10, 125, 165, 221, 218, 11, 142, 131, 27, 0, 193, 244, 189, 149, 150, 127, 192, 16,
            251, 0, 79, 224, 118, 106, 158, 229, 58, 190, 27, 199, 70, 44,
        ],
        [
            47, 93, 253, 7, 216, 219, 104, 144, 107, 172, 234, 91, 107, 52, 121, 5, 45, 214, 148,
            39, 118, 243, 128, 223, 42, 130, 88, 198, 89, 37, 10, 40,
        ],
        [
            24, 208, 204, 230, 65, 96, 241, 136, 121, 22, 15, 171, 27, 48, 41, 212, 157, 228, 185,
            202, 141, 243, 68, 217, 60, 108, 115, 188, 211, 131, 67, 41,
        ],
        [
            190, 202, 105, 68, 170, 47, 201, 132, 190, 190, 166, 225, 7, 107, 65, 69, 109, 193, 47,
            14, 223, 26, 63, 185, 78, 230, 218, 0, 146, 143, 144, 0,
        ],
        [
            166, 38, 45, 231, 156, 245, 176, 191, 52, 176, 212, 147, 13, 162, 132, 4, 8, 238, 204,
            24, 112, 250, 234, 42, 224, 153, 250, 69, 56, 41, 114, 34,
        ],
        [
            171, 36, 217, 169, 17, 204, 156, 28, 33, 25, 229, 73, 250, 40, 65, 166, 92, 68, 77,
            151, 246, 87, 239, 237, 47, 250, 65, 58, 41, 64, 29, 13,
        ],
        [
            17, 52, 107, 231, 186, 222, 120, 227, 138, 150, 6, 62, 222, 26, 253, 222, 252, 69, 140,
            137, 103, 210, 248, 27, 241, 171, 195, 56, 114, 101, 22, 6,
        ],
    ],
    [
        [
            167, 249, 74, 41, 45, 5, 43, 197, 190, 158, 48, 58, 214, 194, 196, 48, 163, 111, 197,
            83, 203, 83, 193, 101, 43, 228, 53, 170, 170, 190, 223, 22,
        ],
        [
            162, 191, 69, 35, 204, 166, 151, 23, 131, 224, 254, 194, 233, 234, 193, 6, 70, 175, 56,
            93, 247, 40, 95, 84, 133, 73, 73, 210, 110, 187, 131, 18,
        ],
        [
            86, 251, 193, 249, 206, 104, 190, 79, 14, 228, 247, 226, 227, 1, 142, 53, 81, 147, 128,
            242, 205, 74, 216, 186, 139, 15, 7, 103, 248, 58, 67, 38,
        ],
        [
            141, 211, 222, 46, 115, 150, 90, 79, 200, 197, 108, 95, 129, 77, 116, 216, 231, 125,
            21, 48, 248, 132, 112, 71, 112, 37, 146, 179, 113, 163, 176, 52,
        ],
        [
            118, 76, 20, 77, 119, 186, 56, 50, 222, 250, 197, 240, 220, 202, 194, 207, 254, 50, 5,
            161, 165, 0, 49, 111, 45, 160, 220, 7, 202, 110, 89, 50,
        ],
        [
            102, 37, 174, 173, 77, 58, 49, 245, 204, 1, 199, 86, 184, 110, 17, 46, 126, 117, 147,
            199, 52, 11, 59, 7, 23, 192, 10, 183, 9, 203, 47, 63,
        ],
        [
            80, 3, 3, 232, 195, 68, 22, 157, 76, 193, 50, 194, 229, 96, 221, 86, 166, 92, 177, 234,
            210, 181, 234, 159, 182, 146, 34, 172, 4, 135, 41, 44,
        ],
        [
            251, 99, 183, 95, 90, 221, 38, 54, 26, 124, 224, 152, 204, 56, 15, 10, 123, 235, 151,
            73, 105, 106, 24, 173, 164, 213, 225, 110, 98, 59, 133, 2,
        ],
    ],
    [
        [
            164, 185, 168, 158, 213, 198, 142, 64, 83, 152, 213, 11, 100, 189, 236, 113, 45, 181,
            112, 106, 135, 24, 198, 226, 203, 63, 78, 3, 226, 180, 74, 10,
        ],
        [
            189, 212, 60, 126, 110, 69, 152, 122, 85, 34, 9, 19, 22, 248, 13, 32, 135, 61, 19, 215,
            200, 213, 219, 231, 220, 119, 92, 75, 219, 5, 120, 40,
        ],
        [
            146, 245, 8, 160, 80, 125, 123, 131, 78, 50, 225, 183, 20, 25, 103, 1, 230, 84, 156,
            252, 244, 204, 172, 164, 252, 118, 213, 130, 204, 37, 91, 11,
        ],
        [
            142, 4, 206, 226, 62, 227, 19, 91, 196, 199, 182, 237, 73, 168, 222, 152, 4, 82, 70,
            114, 57, 114, 229, 181, 134, 59, 0, 208, 176, 183, 218, 30,
        ],
        [
            64, 30, 209, 160, 131, 75, 126, 41, 130, 105, 95, 99, 233, 182, 221, 1, 39, 231, 206,
            9, 47, 25, 170, 24, 14, 249, 62, 68, 42, 68, 198, 6,
        ],
        [
            171, 100, 218, 125, 75, 158, 24, 198, 25, 157, 232, 195, 15, 68, 34, 201, 219, 130,
            198, 107, 2, 101, 70, 250, 161, 168, 99, 10, 18, 233, 13, 24,
        ],
        [
            93, 226, 47, 214, 188, 46, 157, 237, 214, 49, 235, 173, 68, 33, 184, 57, 185, 165, 77,
            217, 205, 103, 160, 63, 235, 254, 153, 164, 215, 239, 24, 53,
        ],
        [
            168, 254, 143, 159, 235, 96, 14, 119, 152, 26, 156, 14, 149, 106, 16, 223, 73, 161,
            133, 220, 175, 139, 213, 210, 56, 30, 214, 161, 51, 20, 128, 17,
        ],
    ],
    [
        [
            90, 157, 237, 146, 6, 11, 72, 108, 27, 34, 77, 120, 242, 127, 35, 117, 189, 64, 45, 84,
            233, 127, 132, 119, 128, 130, 141, 201, 107, 122, 33, 62,
        ],
        [
            205, 110, 229, 214, 93, 107, 183, 211, 172, 82, 194, 10, 68, 115, 182, 64, 59, 197,
            153, 86, 34, 235, 12, 128, 63, 71, 215, 108, 109, 223, 137, 51,
        ],
        [
            215, 181, 7, 95, 143, 147, 43, 211, 208, 240, 5, 116, 105, 62, 141, 149, 67, 190, 237,
            211, 147, 81, 206, 163, 206, 127, 90, 89, 234, 173, 18, 43,
        ],
        [
            171, 227, 107, 54, 113, 175, 138, 78, 159, 26, 107, 241, 64, 57, 153, 114, 237, 92,
            166, 85, 165, 104, 112, 131, 66, 27, 243, 179, 236, 14, 13, 39,
        ],
        [
            74, 37, 91, 124, 254, 2, 191, 78, 0, 103, 90, 169, 99, 225, 232, 176, 213, 32, 172,
            134, 17, 146, 120, 74, 78, 79, 132, 142, 122, 222, 157, 45,
        ],
        [
            95, 6, 241, 174, 101, 194, 254, 44, 170, 149, 90, 122, 227, 137, 151, 143, 227, 35,
            116, 144, 162, 87, 53, 70, 57, 37, 5, 66, 238, 197, 203, 6,
        ],
        [
            114, 141, 200, 255, 187, 228, 255, 185, 248, 223, 241, 204, 71, 85, 159, 221, 108, 199,
            184, 101, 44, 141, 20, 17, 147, 245, 108, 154, 109, 126, 173, 52,
        ],
        [
            176, 218, 48, 106, 6, 126, 6, 176, 103, 206, 215, 132, 17, 182, 174, 147, 102, 173,
            109, 141, 172, 86, 46, 28, 60, 224, 186, 110, 209, 106, 131, 9,
        ],
    ],
    [
        [
            5, 64, 1, 52, 210, 89, 32, 189, 231, 103, 255, 195, 129, 226, 88, 38, 122, 54, 105, 83,
            102, 200, 148, 152, 55, 224, 238, 104, 64, 140, 40, 44,
        ],
        [
            18, 82, 230, 31, 144, 31, 240, 18, 127, 237, 48, 165, 173, 125, 216, 42, 162, 13, 28,
            124, 153, 18, 17, 45, 44, 176, 214, 60, 102, 153, 177, 18,
        ],
        [
            10, 199, 20, 92, 238, 49, 31, 43, 38, 104, 38, 113, 88, 32, 207, 71, 229, 34, 106, 126,
            43, 249, 24, 17, 160, 207, 133, 37, 148, 115, 158, 50,
        ],
        [
            221, 204, 123, 82, 227, 236, 230, 64, 31, 215, 60, 62, 158, 143, 0, 64, 155, 245, 217,
            56, 122, 32, 0, 101, 97, 50, 195, 253, 180, 102, 116, 54,
        ],
        [
            197, 157, 252, 45, 151, 7, 74, 165, 127, 57, 241, 132, 58, 148, 23, 25, 109, 28, 57,
            57, 13, 248, 90, 58, 251, 218, 152, 10, 130, 155, 145, 47,
        ],
        [
            206, 10, 180, 190, 173, 179, 139, 102, 34, 2, 20, 48, 56, 111, 227, 181, 51, 19, 215,
            182, 56, 33, 176, 226, 168, 79, 157, 199, 157, 171, 52, 36,
        ],
        [
            137, 140, 87, 36, 181, 252, 224, 59, 74, 44, 185, 157, 233, 12, 11, 156, 101, 254, 117,
            18, 170, 253, 63, 99, 86, 216, 76, 79, 41, 143, 158, 48,
        ],
        [
            125, 189, 147, 43, 201, 233, 79, 46, 181, 15, 208, 71, 166, 151, 8, 176, 113, 207, 120,
            143, 4, 233, 139, 219, 201, 39, 208, 28, 175, 67, 198, 9,
        ],
    ],
    [
        [
            22, 37, 53, 160, 237, 140, 57, 123, 148, 134, 144, 163, 194, 30, 81, 169, 197, 228, 71,
            38, 137, 208, 166, 114, 242, 79, 76, 62, 162, 120, 252, 14,
        ],
        [
            190, 105, 199, 169, 19, 226, 70, 173, 140, 172, 222, 75, 102, 132, 53, 177, 25, 6, 31,
            202, 0, 6, 243, 47, 53, 103, 175, 18, 201, 165, 9, 12,
        ],
        [
            106, 129, 69, 5, 170, 151, 240, 186, 237, 175, 20, 253, 40, 36, 236, 149, 101, 108,
            117, 163, 42, 3, 159, 177, 223, 198, 149, 23, 235, 129, 28, 59,
        ],
        [
            133, 72, 190, 211, 56, 227, 246, 179, 101, 144, 136, 137, 144, 142, 226, 105, 34, 78,
            109, 103, 159, 87, 255, 103, 169, 40, 239, 110, 131, 156, 159, 12,
        ],
        [
            245, 215, 211, 148, 84, 236, 188, 64, 63, 204, 85, 19, 186, 176, 159, 77, 234, 187, 87,
            224, 232, 22, 67, 84, 104, 130, 82, 181, 103, 186, 96, 36,
        ],
        [
            81, 145, 145, 5, 218, 219, 75, 245, 176, 51, 163, 96, 41, 53, 68, 66, 178, 46, 132,
            255, 156, 136, 171, 119, 128, 182, 9, 125, 159, 93, 6, 24,
        ],
        [
            174, 192, 45, 222, 156, 253, 141, 108, 134, 16, 64, 169, 140, 214, 116, 85, 206, 19,
            143, 90, 191, 122, 191, 50, 120, 242, 12, 250, 50, 89, 166, 1,
        ],
        [
            145, 122, 108, 141, 163, 122, 96, 198, 250, 29, 56, 173, 56, 129, 5, 192, 14, 164, 151,
            18, 232, 155, 23, 242, 150, 129, 184, 71, 42, 211, 34, 37,
        ],
    ],
    [
        [
            189, 187, 66, 97, 29, 216, 85, 51, 152, 120, 220, 104, 253, 201, 53, 132, 65, 240, 252,
            220, 227, 35, 99, 171, 171, 146, 144, 19, 202, 90, 91, 41,
        ],
        [
            72, 152, 255, 229, 122, 94, 187, 103, 248, 26, 154, 126, 54, 68, 234, 8, 202, 208, 155,
            99, 159, 223, 101, 8, 54, 217, 49, 98, 108, 149, 98, 24,
        ],
        [
            50, 232, 175, 207, 73, 18, 225, 218, 85, 236, 151, 102, 191, 232, 207, 250, 161, 234,
            137, 249, 42, 243, 65, 15, 150, 29, 21, 86, 102, 192, 26, 32,
        ],
        [
            40, 193, 12, 48, 4, 212, 39, 67, 231, 182, 218, 43, 11, 231, 75, 82, 24, 190, 132, 82,
            27, 169, 140, 214, 162, 122, 191, 110, 225, 207, 185, 43,
        ],
        [
            177, 1, 100, 87, 228, 136, 102, 8, 248, 170, 10, 58, 139, 21, 103, 168, 205, 234, 0,
            28, 226, 183, 78, 51, 60, 54, 35, 247, 130, 199, 233, 59,
        ],
        [
            181, 222, 23, 124, 138, 101, 61, 15, 49, 131, 173, 47, 224, 128, 118, 125, 66, 45, 68,
            180, 215, 53, 226, 11, 226, 177, 103, 246, 43, 29, 102, 5,
        ],
        [
            64, 123, 98, 247, 117, 240, 66, 217, 174, 246, 81, 150, 232, 133, 87, 229, 154, 43,
            124, 143, 114, 78, 96, 39, 149, 89, 66, 131, 238, 217, 127, 8,
        ],
        [
            115, 22, 38, 108, 115, 205, 105, 132, 105, 23, 113, 198, 46, 49, 30, 219, 57, 177, 189,
            77, 143, 24, 58, 146, 219, 20, 25, 218, 0, 190, 172, 23,
        ],
    ],
    [
        [
            166, 199, 164, 173, 14, 58, 189, 169, 19, 8, 20, 217, 59, 6, 227, 254, 3, 142, 27, 136,
            252, 193, 94, 100, 59, 223, 58, 195, 186, 240, 4, 53,
        ],
        [
            157, 20, 32, 77, 161, 28, 25, 124, 24, 76, 50, 122, 214, 108, 34, 240, 252, 230, 170,
            213, 153, 250, 131, 108, 176, 84, 218, 246, 108, 45, 198, 10,
        ],
        [
            124, 234, 207, 196, 216, 43, 64, 63, 61, 246, 48, 162, 149, 214, 6, 195, 194, 116, 146,
            100, 169, 48, 64, 96, 106, 124, 215, 221, 87, 77, 185, 63,
        ],
        [
            180, 35, 0, 153, 18, 168, 15, 109, 149, 209, 175, 145, 171, 236, 80, 140, 70, 253, 113,
            127, 49, 137, 73, 116, 18, 135, 81, 199, 186, 178, 36, 18,
        ],
        [
            124, 24, 76, 143, 58, 111, 89, 209, 97, 164, 199, 85, 178, 211, 185, 32, 102, 11, 228,
            36, 191, 135, 71, 111, 163, 74, 109, 215, 152, 144, 47, 41,
        ],
        [
            150, 183, 97, 110, 78, 33, 125, 152, 170, 254, 153, 118, 243, 79, 96, 147, 244, 171,
            107, 192, 173, 19, 8, 252, 153, 34, 220, 25, 32, 238, 234, 20,
        ],
        [
            142, 28, 82, 170, 186, 139, 148, 129, 61, 162, 120, 138, 168, 8, 244, 53, 249, 80, 4,
            142, 211, 117, 188, 204, 11, 255, 178, 96, 230, 193, 245, 0,
        ],
        [
            198, 214, 148, 238, 200, 11, 130, 194, 168, 46, 236, 82, 165, 75, 3, 19, 160, 232, 216,
            70, 191, 29, 164, 144, 57, 60, 58, 212, 128, 226, 152, 24,
        ],
    ],
    [
        [
            6, 94, 153, 100, 45, 112, 206, 104, 159, 95, 4, 92, 175, 185, 180, 113, 250, 110, 67,
            166, 94, 228, 126, 186, 236, 64, 252, 84, 134, 208, 99, 17,
        ],
        [
            134, 224, 207, 111, 61, 213, 5, 165, 226, 79, 208, 198, 130, 192, 89, 149, 242, 47, 53,
            50, 206, 109, 157, 243, 99, 129, 215, 75, 181, 144, 117, 58,
        ],
        [
            155, 174, 69, 67, 171, 111, 186, 235, 235, 192, 162, 14, 114, 66, 8, 99, 226, 82, 31,
            134, 213, 214, 115, 222, 214, 211, 229, 184, 153, 7, 31, 13,
        ],
        [
            181, 100, 30, 150, 176, 114, 167, 164, 173, 245, 138, 179, 201, 206, 116, 110, 33, 245,
            104, 230, 172, 144, 182, 39, 160, 1, 55, 170, 171, 14, 97, 36,
        ],
        [
            27, 182, 3, 193, 6, 114, 63, 211, 213, 111, 123, 204, 119, 182, 64, 168, 191, 189, 236,
            169, 62, 102, 46, 10, 66, 233, 108, 95, 156, 5, 2, 17,
        ],
        [
            252, 184, 10, 162, 39, 90, 220, 172, 255, 246, 129, 43, 70, 137, 87, 55, 3, 197, 79,
            110, 208, 179, 213, 197, 8, 97, 207, 114, 63, 16, 35, 19,
        ],
        [
            114, 84, 84, 248, 14, 96, 220, 105, 180, 236, 142, 134, 58, 183, 46, 126, 117, 238, 59,
            233, 31, 32, 166, 80, 31, 170, 87, 81, 239, 167, 243, 20,
        ],
        [
            70, 228, 140, 132, 194, 68, 145, 7, 226, 237, 165, 98, 214, 101, 204, 127, 42, 36, 184,
            82, 253, 192, 72, 225, 215, 205, 242, 162, 212, 113, 64, 21,
        ],
    ],
    [
        [
            88, 195, 125, 176, 127, 155, 224, 36, 205, 80, 207, 184, 175, 242, 195, 101, 204, 77,
            215, 121, 230, 220, 29, 57, 131, 164, 115, 230, 76, 211, 215, 23,
        ],
        [
            213, 195, 189, 198, 199, 49, 198, 15, 92, 101, 87, 99, 207, 99, 186, 213, 229, 179, 69,
            71, 41, 140, 203, 153, 22, 155, 214, 226, 130, 175, 156, 62,
        ],
        [
            166, 117, 91, 218, 251, 30, 251, 111, 174, 234, 219, 158, 177, 73, 166, 88, 204, 53,
            27, 49, 235, 95, 205, 250, 155, 196, 152, 217, 130, 56, 145, 51,
        ],
        [
            12, 180, 50, 125, 157, 91, 123, 81, 184, 183, 48, 213, 87, 185, 233, 171, 55, 204, 113,
            238, 197, 226, 30, 163, 183, 131, 36, 135, 210, 12, 197, 42,
        ],
        [
            175, 205, 39, 140, 182, 198, 178, 47, 247, 108, 237, 249, 97, 233, 180, 95, 123, 109,
            1, 227, 242, 234, 128, 66, 217, 225, 8, 216, 228, 135, 20, 46,
        ],
        [
            204, 48, 117, 11, 21, 101, 12, 185, 206, 97, 152, 180, 12, 104, 28, 71, 58, 176, 116,
            10, 174, 103, 201, 82, 118, 74, 125, 197, 149, 216, 176, 4,
        ],
        [
            150, 144, 226, 186, 64, 17, 254, 26, 21, 248, 118, 91, 180, 255, 125, 61, 67, 113, 25,
            170, 155, 206, 110, 113, 66, 64, 88, 174, 131, 38, 196, 55,
        ],
        [
            157, 181, 209, 189, 59, 76, 227, 183, 0, 161, 70, 233, 68, 56, 230, 107, 139, 252, 93,
            167, 174, 72, 190, 77, 142, 215, 228, 199, 1, 60, 136, 56,
        ],
    ],
    [
        [
            190, 92, 13, 8, 22, 229, 140, 76, 14, 27, 58, 56, 121, 150, 136, 228, 72, 103, 71, 115,
            243, 253, 53, 81, 123, 38, 199, 219, 123, 19, 174, 14,
        ],
        [
            7, 188, 24, 186, 0, 251, 54, 130, 213, 106, 143, 83, 155, 119, 206, 253, 249, 191, 54,
            233, 191, 176, 42, 51, 93, 9, 163, 121, 219, 200, 214, 12,
        ],
        [
            198, 163, 145, 80, 22, 184, 116, 216, 156, 139, 34, 249, 166, 116, 103, 145, 120, 158,
            85, 102, 25, 27, 113, 121, 102, 190, 55, 151, 48, 240, 71, 32,
        ],
        [
            204, 58, 122, 178, 78, 187, 138, 17, 175, 30, 44, 72, 40, 61, 247, 156, 165, 100, 129,
            171, 82, 152, 58, 208, 228, 249, 238, 15, 169, 34, 87, 9,
        ],
        [
            134, 168, 115, 14, 77, 100, 63, 169, 243, 160, 196, 220, 14, 1, 189, 106, 39, 109, 74,
            95, 69, 13, 80, 153, 149, 228, 76, 44, 189, 217, 174, 14,
        ],
        [
            159, 106, 117, 192, 70, 92, 68, 153, 97, 21, 78, 158, 38, 251, 73, 250, 232, 212, 81,
            218, 132, 210, 141, 52, 154, 109, 180, 102, 26, 46, 237, 42,
        ],
        [
            12, 163, 20, 229, 61, 228, 45, 24, 204, 20, 16, 67, 244, 76, 55, 249, 80, 75, 21, 138,
            253, 198, 255, 253, 92, 251, 90, 177, 226, 0, 189, 42,
        ],
        [
            122, 234, 40, 6, 65, 195, 69, 149, 59, 88, 230, 156, 205, 111, 77, 80, 8, 79, 75, 165,
            237, 130, 130, 223, 115, 216, 16, 83, 191, 48, 199, 35,
        ],
    ],
    [
        [
            120, 206, 29, 206, 231, 94, 118, 231, 142, 12, 21, 55, 86, 131, 101, 140, 149, 39, 143,
            47, 153, 183, 84, 124, 235, 181, 72, 47, 36, 106, 118, 26,
        ],
        [
            228, 137, 53, 107, 172, 135, 152, 135, 60, 72, 14, 251, 236, 200, 210, 82, 179, 124,
            204, 92, 40, 155, 7, 230, 141, 192, 49, 132, 97, 89, 119, 22,
        ],
        [
            207, 29, 225, 197, 86, 135, 65, 28, 17, 153, 15, 136, 28, 124, 240, 84, 14, 201, 29,
            194, 140, 205, 231, 108, 243, 100, 134, 59, 67, 249, 95, 34,
        ],
        [
            234, 84, 54, 160, 181, 164, 65, 6, 15, 193, 237, 165, 148, 82, 153, 142, 39, 239, 134,
            40, 164, 122, 235, 29, 102, 186, 216, 29, 103, 239, 85, 50,
        ],
        [
            58, 194, 136, 114, 154, 163, 206, 220, 211, 122, 144, 102, 109, 22, 30, 145, 234, 170,
            24, 252, 71, 139, 166, 193, 151, 227, 118, 9, 63, 209, 24, 52,
        ],
        [
            69, 95, 192, 151, 92, 184, 83, 24, 138, 189, 254, 51, 192, 141, 121, 211, 54, 121, 107,
            109, 187, 240, 161, 244, 54, 11, 205, 41, 49, 186, 58, 21,
        ],
        [
            174, 72, 170, 165, 97, 229, 216, 108, 52, 175, 217, 144, 65, 170, 178, 72, 57, 244,
            217, 93, 90, 59, 133, 150, 65, 222, 78, 168, 142, 10, 65, 57,
        ],
        [
            64, 55, 77, 9, 97, 224, 252, 144, 123, 19, 154, 107, 90, 176, 121, 15, 30, 43, 94, 233,
            86, 28, 92, 121, 22, 29, 194, 229, 26, 46, 23, 24,
        ],
    ],
    [
        [
            76, 60, 120, 126, 240, 28, 211, 52, 21, 189, 35, 153, 162, 242, 177, 158, 251, 118,
            212, 37, 42, 83, 34, 73, 67, 232, 149, 198, 196, 87, 158, 15,
        ],
        [
            157, 158, 65, 83, 61, 63, 54, 50, 65, 74, 210, 115, 194, 223, 77, 138, 137, 78, 188,
            14, 117, 203, 104, 45, 48, 199, 130, 1, 54, 149, 206, 35,
        ],
        [
            231, 191, 185, 71, 252, 217, 195, 188, 72, 190, 22, 41, 42, 147, 166, 69, 42, 126, 125,
            168, 48, 63, 127, 44, 181, 189, 110, 141, 72, 178, 125, 29,
        ],
        [
            94, 64, 125, 170, 1, 131, 212, 24, 147, 151, 15, 73, 177, 8, 135, 160, 201, 115, 241,
            136, 247, 184, 211, 13, 64, 206, 126, 239, 224, 73, 215, 55,
        ],
        [
            12, 37, 188, 206, 231, 212, 14, 180, 159, 225, 82, 255, 232, 11, 128, 28, 50, 95, 1,
            131, 203, 4, 17, 66, 239, 193, 151, 123, 253, 68, 20, 40,
        ],
        [
            48, 80, 138, 22, 87, 172, 103, 208, 138, 14, 164, 65, 191, 235, 51, 248, 21, 117, 129,
            248, 199, 128, 126, 210, 189, 57, 67, 150, 178, 8, 188, 62,
        ],
        [
            98, 115, 183, 93, 165, 143, 231, 214, 229, 168, 250, 179, 9, 43, 102, 129, 189, 8, 138,
            149, 215, 140, 100, 168, 27, 152, 140, 252, 36, 109, 62, 49,
        ],
        [
            82, 184, 171, 186, 244, 111, 194, 186, 63, 45, 8, 90, 97, 130, 102, 112, 185, 70, 182,
            92, 197, 253, 122, 124, 196, 164, 170, 25, 180, 117, 94, 2,
        ],
    ],
    [
        [
            209, 225, 106, 206, 153, 240, 154, 163, 143, 89, 129, 19, 20, 45, 44, 223, 64, 38, 34,
            208, 9, 151, 85, 171, 182, 89, 137, 36, 163, 41, 84, 42,
        ],
        [
            146, 188, 65, 87, 122, 69, 148, 30, 80, 176, 184, 33, 3, 77, 156, 203, 73, 230, 231,
            12, 22, 232, 26, 143, 91, 35, 206, 246, 43, 55, 3, 9,
        ],
        [
            25, 114, 90, 10, 192, 14, 159, 234, 190, 73, 105, 222, 143, 2, 98, 30, 107, 18, 49, 35,
            100, 255, 185, 20, 52, 155, 150, 175, 35, 209, 41, 39,
        ],
        [
            191, 153, 7, 52, 5, 119, 8, 21, 199, 194, 163, 24, 199, 76, 185, 128, 49, 18, 54, 206,
            252, 101, 181, 105, 101, 3, 93, 250, 139, 110, 105, 17,
        ],
        [
            187, 224, 140, 3, 152, 10, 215, 106, 92, 32, 53, 193, 164, 49, 85, 248, 152, 217, 148,
            69, 206, 67, 202, 77, 40, 134, 1, 233, 36, 57, 255, 60,
        ],
        [
            174, 28, 43, 241, 197, 16, 52, 191, 5, 12, 100, 203, 128, 228, 95, 21, 139, 30, 27,
            151, 49, 131, 188, 119, 245, 70, 113, 118, 64, 72, 182, 6,
        ],
        [
            48, 252, 34, 196, 173, 44, 133, 85, 67, 13, 237, 191, 29, 5, 184, 67, 133, 144, 192,
            38, 19, 37, 175, 108, 156, 242, 21, 223, 11, 119, 165, 14,
        ],
        [
            100, 197, 0, 213, 89, 62, 88, 131, 238, 25, 108, 172, 133, 12, 34, 14, 9, 130, 39, 54,
            188, 4, 61, 243, 131, 108, 22, 70, 116, 37, 210, 34,
        ],
    ],
    [
        [
            170, 138, 180, 84, 35, 21, 13, 177, 134, 125, 153, 82, 70, 64, 27, 75, 162, 161, 10,
            198, 105, 36, 120, 166, 182, 214, 103, 117, 229, 138, 80, 59,
        ],
        [
            84, 95, 68, 119, 210, 84, 69, 165, 136, 57, 55, 153, 132, 67, 66, 82, 213, 80, 68, 108,
            192, 184, 190, 202, 40, 243, 116, 24, 142, 112, 174, 57,
        ],
        [
            88, 41, 176, 199, 215, 98, 157, 224, 41, 6, 229, 129, 207, 52, 114, 203, 182, 27, 82,
            185, 68, 102, 126, 6, 49, 224, 172, 179, 169, 23, 67, 50,
        ],
        [
            13, 39, 165, 156, 39, 15, 143, 254, 19, 44, 221, 98, 41, 248, 51, 162, 124, 244, 145,
            66, 77, 238, 2, 159, 167, 130, 5, 165, 148, 217, 126, 39,
        ],
        [
            243, 178, 255, 129, 211, 199, 156, 239, 252, 157, 180, 156, 233, 49, 20, 122, 235, 112,
            107, 104, 128, 111, 10, 39, 225, 187, 190, 216, 237, 133, 116, 27,
        ],
        [
            62, 140, 130, 27, 124, 177, 228, 42, 137, 63, 23, 176, 132, 100, 20, 235, 149, 6, 30,
            226, 243, 86, 218, 62, 39, 167, 124, 40, 114, 50, 151, 44,
        ],
        [
            153, 186, 80, 123, 183, 144, 118, 245, 93, 213, 158, 238, 76, 61, 218, 113, 80, 97,
            193, 199, 84, 3, 88, 133, 168, 218, 248, 41, 216, 231, 132, 54,
        ],
        [
            24, 13, 231, 102, 64, 93, 142, 32, 95, 124, 165, 112, 146, 124, 94, 71, 147, 194, 180,
            81, 138, 95, 198, 173, 135, 219, 180, 238, 55, 175, 0, 49,
        ],
    ],
    [
        [
            191, 67, 109, 233, 253, 99, 144, 145, 112, 41, 12, 226, 34, 38, 59, 104, 32, 46, 165,
            134, 63, 12, 240, 107, 124, 170, 157, 95, 123, 77, 51, 11,
        ],
        [
            253, 176, 178, 255, 113, 102, 61, 37, 130, 48, 154, 203, 106, 34, 89, 133, 223, 16, 46,
            161, 228, 11, 168, 58, 124, 111, 180, 95, 205, 178, 201, 8,
        ],
        [
            45, 51, 116, 58, 87, 136, 35, 53, 226, 3, 240, 236, 196, 23, 79, 255, 236, 129, 129,
            153, 110, 21, 32, 121, 214, 66, 53, 39, 32, 121, 155, 26,
        ],
        [
            253, 55, 59, 249, 128, 61, 95, 130, 99, 194, 200, 12, 129, 70, 56, 130, 60, 163, 84,
            203, 119, 27, 142, 56, 171, 128, 144, 207, 208, 220, 33, 18,
        ],
        [
            255, 59, 215, 178, 200, 172, 206, 53, 251, 202, 242, 148, 156, 247, 117, 226, 60, 145,
            122, 154, 15, 205, 138, 253, 158, 133, 46, 103, 33, 12, 85, 26,
        ],
        [
            51, 26, 129, 30, 187, 55, 234, 9, 36, 14, 6, 19, 54, 209, 189, 240, 226, 122, 219, 186,
            62, 106, 52, 89, 242, 4, 177, 214, 20, 228, 188, 18,
        ],
        [
            43, 93, 69, 179, 7, 206, 234, 199, 244, 76, 82, 200, 246, 125, 21, 1, 124, 142, 3, 76,
            21, 1, 237, 236, 151, 249, 71, 38, 159, 159, 239, 48,
        ],
        [
            230, 77, 223, 241, 62, 17, 127, 49, 233, 34, 99, 218, 95, 132, 99, 158, 50, 81, 121,
            208, 225, 109, 68, 170, 238, 26, 147, 195, 178, 33, 157, 16,
        ],
    ],
    [
        [
            52, 186, 178, 89, 121, 159, 161, 108, 165, 177, 92, 184, 149, 101, 171, 153, 70, 215,
            46, 200, 193, 14, 188, 34, 192, 141, 24, 210, 156, 81, 217, 41,
        ],
        [
            205, 99, 134, 116, 97, 224, 2, 77, 29, 77, 88, 53, 0, 111, 15, 56, 74, 172, 166, 150,
            133, 93, 146, 253, 3, 100, 190, 133, 11, 46, 40, 62,
        ],
        [
            202, 120, 227, 229, 35, 128, 155, 243, 255, 108, 76, 195, 219, 96, 86, 121, 217, 111,
            11, 121, 77, 228, 175, 2, 23, 102, 78, 193, 249, 10, 139, 10,
        ],
        [
            128, 219, 91, 150, 191, 196, 135, 27, 205, 245, 32, 8, 15, 40, 90, 201, 73, 241, 31,
            207, 216, 225, 129, 90, 173, 32, 108, 126, 59, 32, 11, 19,
        ],
        [
            31, 172, 118, 187, 18, 112, 69, 208, 250, 115, 34, 50, 221, 213, 209, 55, 44, 127, 130,
            196, 190, 155, 247, 254, 2, 195, 231, 155, 169, 158, 198, 60,
        ],
        [
            205, 109, 204, 106, 223, 114, 189, 146, 15, 175, 189, 220, 55, 7, 242, 114, 99, 96, 81,
            181, 204, 132, 253, 3, 52, 216, 47, 37, 58, 52, 227, 21,
        ],
        [
            232, 76, 28, 95, 190, 76, 135, 35, 133, 206, 252, 224, 237, 123, 84, 47, 188, 253, 5,
            178, 145, 176, 232, 227, 7, 230, 89, 213, 199, 47, 66, 32,
        ],
        [
            71, 151, 218, 212, 133, 48, 92, 62, 72, 154, 75, 10, 186, 210, 198, 146, 53, 18, 173,
            172, 60, 110, 21, 166, 252, 53, 239, 211, 36, 189, 156, 1,
        ],
    ],
    [
        [
            33, 207, 156, 240, 223, 153, 52, 29, 223, 44, 214, 249, 177, 30, 23, 32, 99, 199, 159,
            161, 175, 245, 85, 144, 14, 118, 163, 72, 111, 135, 0, 41,
        ],
        [
            110, 244, 91, 45, 63, 22, 57, 91, 73, 106, 99, 88, 146, 118, 14, 176, 186, 228, 14, 83,
            140, 189, 143, 197, 194, 135, 97, 41, 116, 113, 163, 30,
        ],
        [
            137, 86, 251, 125, 228, 181, 116, 52, 60, 174, 181, 79, 70, 211, 93, 128, 143, 17, 107,
            48, 241, 58, 140, 58, 240, 34, 167, 4, 67, 26, 212, 29,
        ],
        [
            108, 217, 222, 116, 191, 191, 47, 61, 96, 84, 106, 166, 253, 5, 212, 60, 163, 168, 156,
            145, 186, 129, 244, 187, 152, 235, 40, 36, 173, 231, 185, 56,
        ],
        [
            144, 139, 139, 57, 138, 79, 237, 42, 243, 255, 137, 46, 60, 236, 0, 124, 151, 172, 130,
            131, 4, 37, 255, 157, 112, 141, 248, 116, 32, 21, 169, 1,
        ],
        [
            109, 149, 1, 115, 108, 63, 68, 192, 91, 49, 234, 90, 143, 98, 248, 92, 202, 207, 144,
            86, 0, 26, 65, 235, 156, 131, 11, 74, 112, 175, 247, 14,
        ],
        [
            18, 194, 249, 56, 144, 151, 10, 107, 87, 48, 179, 117, 7, 100, 38, 220, 26, 2, 18, 160,
            171, 125, 140, 213, 171, 82, 189, 98, 96, 3, 235, 5,
        ],
        [
            44, 108, 31, 136, 128, 150, 86, 40, 251, 99, 8, 150, 157, 135, 28, 214, 158, 60, 242,
            29, 123, 252, 85, 65, 172, 61, 241, 230, 140, 29, 224, 27,
        ],
    ],
    [
        [
            72, 96, 81, 44, 138, 199, 176, 84, 134, 126, 188, 123, 88, 83, 37, 233, 31, 237, 230,
            160, 15, 57, 252, 28, 79, 176, 8, 136, 160, 168, 180, 54,
        ],
        [
            236, 204, 160, 217, 80, 141, 110, 37, 12, 106, 216, 168, 97, 147, 225, 178, 243, 91,
            144, 177, 162, 143, 29, 238, 45, 191, 217, 169, 17, 6, 82, 15,
        ],
        [
            151, 37, 164, 32, 139, 47, 139, 135, 64, 90, 53, 139, 250, 31, 129, 155, 38, 132, 99,
            202, 75, 25, 233, 151, 238, 122, 231, 45, 178, 23, 36, 7,
        ],
        [
            145, 211, 54, 13, 50, 237, 143, 95, 233, 219, 250, 67, 255, 72, 244, 202, 26, 178, 61,
            233, 52, 109, 106, 30, 246, 249, 34, 126, 72, 113, 240, 4,
        ],
        [
            113, 166, 236, 160, 205, 43, 161, 37, 126, 197, 172, 121, 235, 101, 42, 43, 143, 199,
            111, 177, 23, 42, 51, 220, 232, 188, 92, 235, 139, 183, 170, 3,
        ],
        [
            236, 93, 71, 61, 20, 98, 70, 157, 157, 108, 146, 80, 225, 245, 166, 244, 14, 223, 41,
            191, 206, 212, 140, 195, 99, 13, 112, 217, 191, 189, 57, 22,
        ],
        [
            110, 80, 126, 40, 242, 214, 43, 231, 72, 57, 159, 88, 36, 146, 13, 25, 198, 55, 233,
            134, 49, 87, 14, 198, 40, 68, 245, 50, 182, 195, 172, 32,
        ],
        [
            138, 47, 67, 28, 155, 85, 91, 243, 199, 109, 115, 119, 160, 15, 14, 10, 252, 133, 3,
            75, 35, 17, 189, 159, 27, 1, 163, 166, 144, 211, 110, 11,
        ],
    ],
    [
        [
            31, 79, 187, 42, 20, 82, 51, 24, 96, 195, 51, 150, 237, 25, 238, 211, 26, 151, 165,
            162, 158, 98, 38, 255, 134, 178, 213, 60, 249, 77, 5, 2,
        ],
        [
            97, 204, 217, 104, 57, 122, 48, 148, 131, 78, 36, 83, 200, 202, 85, 8, 163, 120, 189,
            42, 109, 236, 11, 100, 221, 32, 84, 136, 26, 126, 107, 53,
        ],
        [
            80, 195, 31, 107, 117, 133, 195, 140, 109, 176, 18, 91, 213, 153, 80, 1, 68, 254, 42,
            238, 24, 189, 171, 105, 94, 59, 219, 230, 22, 176, 107, 9,
        ],
        [
            141, 33, 117, 234, 88, 61, 132, 175, 21, 196, 141, 130, 123, 155, 29, 70, 66, 189, 3,
            171, 12, 150, 97, 145, 117, 192, 81, 68, 6, 28, 96, 28,
        ],
        [
            200, 160, 161, 9, 64, 94, 205, 144, 62, 116, 255, 185, 94, 175, 20, 167, 217, 78, 234,
            1, 1, 122, 238, 184, 69, 233, 201, 85, 254, 138, 212, 39,
        ],
        [
            109, 185, 122, 128, 51, 220, 226, 181, 98, 114, 82, 142, 251, 224, 39, 161, 4, 27, 155,
            185, 175, 78, 71, 250, 103, 160, 191, 44, 50, 253, 150, 48,
        ],
        [
            63, 60, 25, 196, 216, 52, 106, 139, 111, 73, 150, 237, 22, 80, 63, 12, 198, 131, 217,
            24, 217, 222, 1, 154, 107, 121, 42, 206, 71, 231, 85, 24,
        ],
        [
            110, 208, 119, 152, 113, 49, 42, 110, 242, 125, 32, 238, 131, 123, 132, 102, 46, 60,
            111, 90, 37, 46, 79, 200, 142, 181, 136, 54, 102, 115, 166, 45,
        ],
    ],
    [
        [
            65, 117, 127, 35, 217, 86, 9, 239, 179, 241, 114, 13, 75, 195, 58, 239, 152, 49, 225,
            92, 93, 1, 35, 31, 117, 45, 132, 152, 184, 111, 227, 62,
        ],
        [
            28, 50, 160, 235, 129, 142, 99, 208, 77, 124, 228, 209, 205, 199, 158, 149, 244, 14,
            22, 129, 93, 116, 205, 167, 237, 75, 238, 150, 8, 21, 160, 10,
        ],
        [
            240, 31, 142, 228, 114, 27, 182, 196, 65, 166, 251, 35, 97, 252, 58, 32, 207, 217, 94,
            11, 22, 44, 35, 143, 234, 143, 112, 21, 225, 43, 124, 35,
        ],
        [
            84, 215, 38, 31, 55, 68, 5, 227, 30, 70, 197, 135, 157, 55, 132, 110, 17, 239, 55, 150,
            66, 94, 111, 156, 61, 134, 129, 252, 80, 136, 109, 3,
        ],
        [
            244, 5, 160, 73, 65, 202, 125, 239, 79, 248, 118, 146, 198, 46, 179, 29, 27, 183, 162,
            224, 163, 29, 88, 196, 95, 77, 15, 162, 245, 208, 121, 58,
        ],
        [
            33, 142, 135, 112, 143, 47, 193, 39, 170, 223, 34, 145, 60, 197, 160, 23, 233, 22, 82,
            189, 91, 251, 186, 21, 238, 69, 58, 194, 34, 58, 40, 9,
        ],
        [
            46, 205, 15, 203, 124, 46, 93, 43, 68, 29, 165, 195, 225, 198, 170, 60, 23, 136, 96,
            65, 54, 231, 238, 94, 94, 2, 16, 171, 167, 138, 105, 16,
        ],
        [
            10, 116, 71, 4, 152, 209, 8, 163, 77, 5, 23, 146, 141, 186, 60, 94, 39, 117, 33, 20, 5,
            3, 156, 67, 221, 79, 57, 111, 72, 37, 126, 51,
        ],
    ],
    [
        [
            191, 166, 185, 94, 206, 142, 77, 46, 132, 158, 245, 234, 48, 226, 49, 249, 39, 60, 88,
            14, 224, 91, 91, 163, 161, 166, 103, 20, 190, 13, 116, 19,
        ],
        [
            51, 216, 164, 99, 24, 233, 245, 224, 152, 116, 137, 45, 246, 190, 47, 222, 130, 135,
            57, 245, 95, 18, 139, 248, 200, 158, 124, 75, 9, 72, 222, 22,
        ],
        [
            11, 200, 108, 211, 187, 171, 183, 33, 91, 144, 221, 192, 112, 246, 92, 195, 14, 243,
            116, 64, 122, 238, 227, 49, 72, 133, 228, 133, 36, 105, 65, 44,
        ],
        [
            97, 255, 190, 195, 103, 126, 180, 211, 73, 73, 8, 155, 205, 46, 78, 148, 212, 87, 193,
            107, 121, 190, 164, 242, 251, 238, 87, 198, 180, 148, 171, 30,
        ],
        [
            146, 130, 197, 213, 1, 97, 12, 6, 210, 55, 145, 213, 55, 243, 71, 83, 117, 160, 62,
            201, 72, 75, 160, 185, 107, 235, 22, 189, 164, 222, 197, 10,
        ],
        [
            187, 238, 229, 171, 0, 101, 27, 153, 124, 37, 3, 122, 43, 192, 182, 104, 220, 157, 180,
            252, 88, 120, 29, 197, 199, 109, 94, 10, 196, 187, 183, 9,
        ],
        [
            240, 83, 124, 167, 254, 134, 93, 87, 81, 207, 252, 115, 47, 49, 33, 65, 200, 33, 186,
            176, 72, 31, 149, 189, 19, 122, 236, 228, 129, 163, 231, 41,
        ],
        [
            8, 130, 118, 127, 122, 91, 123, 43, 98, 65, 35, 242, 64, 136, 15, 55, 191, 136, 4, 26,
            157, 157, 138, 218, 75, 204, 240, 144, 61, 53, 99, 62,
        ],
    ],
    [
        [
            160, 12, 147, 196, 134, 124, 67, 246, 5, 2, 69, 220, 231, 138, 227, 129, 126, 185, 74,
            155, 236, 207, 218, 212, 238, 22, 0, 157, 94, 85, 24, 6,
        ],
        [
            138, 4, 163, 194, 25, 156, 50, 92, 111, 33, 7, 26, 69, 110, 26, 194, 70, 115, 87, 232,
            187, 92, 183, 40, 206, 229, 191, 48, 9, 75, 8, 42,
        ],
        [
            144, 61, 87, 235, 101, 165, 69, 137, 69, 225, 109, 196, 47, 204, 127, 42, 158, 121, 50,
            27, 129, 66, 148, 82, 79, 154, 189, 5, 115, 100, 17, 35,
        ],
        [
            87, 218, 178, 199, 26, 202, 96, 153, 160, 84, 208, 0, 84, 46, 32, 221, 97, 63, 161,
            235, 61, 34, 111, 119, 91, 179, 51, 175, 158, 71, 156, 34,
        ],
        [
            189, 150, 38, 95, 146, 100, 0, 105, 233, 4, 173, 80, 206, 50, 53, 160, 40, 167, 33,
            234, 80, 54, 99, 4, 41, 55, 46, 209, 140, 210, 143, 57,
        ],
        [
            25, 185, 233, 107, 108, 66, 45, 50, 152, 110, 164, 20, 199, 231, 35, 48, 66, 116, 149,
            211, 214, 134, 127, 152, 63, 102, 27, 33, 90, 16, 216, 50,
        ],
        [
            77, 105, 46, 102, 27, 172, 194, 72, 49, 15, 105, 118, 41, 16, 161, 168, 226, 172, 47,
            111, 115, 141, 123, 82, 160, 242, 69, 32, 125, 21, 44, 14,
        ],
        [
            198, 193, 51, 219, 107, 174, 135, 49, 208, 46, 134, 81, 121, 111, 246, 128, 198, 48,
            158, 216, 160, 175, 107, 116, 51, 156, 87, 45, 238, 142, 66, 0,
        ],
    ],
    [
        [
            228, 186, 23, 81, 165, 67, 155, 142, 246, 246, 111, 141, 210, 130, 233, 51, 211, 146,
            202, 26, 110, 46, 212, 39, 154, 170, 63, 110, 31, 238, 80, 0,
        ],
        [
            140, 202, 146, 110, 94, 220, 222, 80, 20, 41, 163, 129, 65, 188, 13, 58, 224, 0, 215,
            183, 33, 118, 158, 219, 197, 78, 110, 143, 63, 70, 21, 15,
        ],
        [
            71, 155, 232, 76, 80, 9, 234, 144, 163, 24, 181, 206, 34, 218, 60, 75, 159, 106, 75,
            94, 218, 241, 17, 5, 169, 237, 87, 239, 51, 126, 174, 12,
        ],
        [
            68, 14, 217, 104, 199, 117, 220, 170, 82, 113, 240, 86, 146, 132, 62, 68, 71, 63, 72,
            67, 170, 58, 28, 113, 136, 167, 66, 29, 90, 185, 131, 10,
        ],
        [
            124, 223, 201, 195, 99, 243, 62, 202, 157, 49, 112, 118, 126, 180, 56, 109, 13, 58, 38,
            146, 240, 112, 157, 47, 149, 136, 68, 241, 122, 150, 37, 23,
        ],
        [
            19, 217, 47, 84, 231, 168, 141, 240, 13, 102, 26, 80, 70, 109, 70, 142, 83, 149, 253,
            102, 136, 214, 198, 184, 89, 232, 43, 180, 94, 57, 237, 59,
        ],
        [
            2, 164, 113, 50, 147, 55, 23, 208, 109, 197, 30, 13, 248, 111, 194, 13, 229, 73, 91,
            166, 195, 175, 210, 198, 179, 60, 221, 87, 104, 70, 218, 26,
        ],
        [
            17, 160, 102, 238, 191, 46, 46, 161, 64, 238, 175, 98, 139, 219, 113, 56, 87, 78, 228,
            246, 100, 155, 192, 172, 221, 72, 238, 31, 204, 49, 35, 54,
        ],
    ],
    [
        [
            42, 238, 87, 78, 76, 206, 132, 112, 92, 251, 61, 98, 151, 105, 49, 204, 234, 104, 116,
            42, 176, 84, 72, 114, 111, 247, 18, 125, 165, 27, 250, 60,
        ],
        [
            192, 195, 201, 56, 49, 145, 231, 71, 125, 128, 185, 1, 116, 183, 81, 142, 207, 148,
            159, 32, 184, 160, 8, 39, 8, 100, 43, 218, 96, 50, 107, 57,
        ],
        [
            161, 28, 139, 96, 155, 238, 224, 98, 1, 109, 199, 72, 193, 233, 77, 40, 223, 139, 126,
            165, 233, 112, 9, 39, 61, 37, 172, 130, 252, 255, 231, 55,
        ],
        [
            148, 189, 23, 30, 46, 255, 1, 123, 106, 99, 45, 7, 65, 202, 55, 188, 236, 162, 182, 70,
            38, 204, 120, 93, 114, 107, 159, 91, 250, 140, 57, 57,
        ],
        [
            51, 190, 71, 10, 22, 62, 37, 160, 168, 87, 199, 129, 183, 99, 225, 142, 247, 162, 123,
            57, 44, 239, 49, 212, 91, 166, 5, 116, 83, 223, 225, 22,
        ],
        [
            105, 203, 12, 226, 205, 192, 204, 244, 44, 237, 186, 143, 3, 235, 29, 157, 45, 95, 190,
            177, 140, 218, 246, 125, 107, 136, 39, 11, 171, 15, 165, 15,
        ],
        [
            40, 63, 140, 161, 128, 218, 185, 176, 105, 10, 235, 239, 69, 132, 86, 207, 77, 47, 176,
            71, 250, 98, 164, 8, 158, 67, 139, 106, 185, 21, 241, 45,
        ],
        [
            113, 187, 121, 32, 76, 47, 222, 172, 149, 197, 166, 55, 51, 145, 35, 53, 127, 181, 124,
            87, 228, 146, 246, 19, 131, 251, 25, 177, 8, 37, 166, 53,
        ],
    ],
    [
        [
            18, 1, 130, 209, 3, 70, 234, 9, 136, 82, 114, 124, 75, 221, 153, 20, 31, 238, 248, 78,
            148, 136, 42, 184, 143, 43, 26, 126, 200, 61, 80, 57,
        ],
        [
            225, 98, 72, 37, 222, 154, 100, 214, 73, 206, 129, 29, 27, 131, 81, 32, 182, 243, 185,
            93, 84, 0, 151, 201, 243, 111, 89, 42, 117, 167, 99, 34,
        ],
        [
            118, 227, 216, 225, 147, 147, 53, 116, 60, 220, 139, 31, 1, 124, 185, 10, 157, 49, 42,
            132, 235, 10, 161, 152, 137, 48, 32, 206, 86, 213, 221, 32,
        ],
        [
            254, 51, 99, 58, 209, 148, 243, 37, 169, 11, 22, 249, 168, 97, 56, 119, 21, 77, 50,
            144, 156, 65, 220, 104, 229, 213, 153, 30, 42, 240, 156, 31,
        ],
        [
            15, 208, 192, 113, 116, 160, 10, 101, 148, 128, 253, 44, 233, 9, 78, 22, 56, 25, 98,
            54, 63, 55, 113, 147, 151, 161, 116, 34, 245, 39, 75, 3,
        ],
        [
            210, 150, 121, 196, 177, 196, 184, 180, 88, 219, 232, 40, 89, 169, 87, 199, 227, 151,
            251, 74, 249, 12, 128, 2, 208, 145, 98, 160, 150, 112, 200, 5,
        ],
        [
            253, 175, 188, 74, 246, 2, 160, 24, 157, 187, 41, 151, 103, 79, 161, 226, 146, 151, 46,
            208, 37, 5, 86, 126, 191, 153, 231, 94, 147, 101, 189, 5,
        ],
        [
            15, 30, 205, 151, 176, 243, 96, 206, 110, 125, 116, 143, 234, 172, 254, 151, 148, 18,
            25, 224, 83, 167, 228, 13, 47, 241, 58, 218, 76, 104, 93, 3,
        ],
    ],
    [
        [
            72, 10, 136, 27, 24, 16, 41, 48, 68, 158, 58, 254, 35, 250, 247, 107, 227, 74, 234, 98,
            97, 242, 9, 158, 70, 85, 49, 198, 203, 11, 230, 20,
        ],
        [
            214, 243, 43, 26, 120, 31, 250, 32, 6, 32, 255, 63, 75, 36, 248, 24, 236, 39, 227, 136,
            236, 50, 219, 231, 16, 127, 24, 172, 146, 96, 244, 59,
        ],
        [
            96, 191, 165, 84, 42, 112, 128, 149, 53, 170, 114, 219, 238, 65, 138, 252, 190, 104,
            95, 28, 80, 239, 185, 250, 14, 0, 70, 112, 95, 96, 37, 48,
        ],
        [
            23, 136, 200, 26, 146, 148, 183, 211, 0, 39, 126, 89, 25, 17, 253, 149, 215, 73, 33,
            67, 99, 1, 207, 32, 180, 34, 81, 116, 182, 105, 39, 33,
        ],
        [
            87, 95, 184, 76, 228, 106, 105, 36, 159, 100, 80, 25, 234, 27, 214, 63, 43, 106, 183,
            209, 28, 154, 57, 252, 1, 33, 107, 197, 64, 155, 206, 45,
        ],
        [
            61, 125, 226, 15, 206, 94, 85, 192, 206, 85, 168, 220, 114, 189, 128, 250, 80, 50, 167,
            65, 231, 176, 99, 21, 82, 43, 218, 99, 153, 198, 55, 56,
        ],
        [
            21, 41, 65, 249, 187, 151, 1, 15, 58, 96, 102, 96, 128, 158, 199, 143, 110, 200, 182,
            55, 15, 40, 227, 249, 70, 24, 155, 205, 177, 248, 68, 36,
        ],
        [
            153, 93, 124, 9, 139, 102, 83, 232, 141, 176, 183, 178, 149, 222, 54, 14, 181, 4, 83,
            113, 81, 218, 84, 31, 86, 230, 20, 226, 169, 169, 147, 20,
        ],
    ],
    [
        [
            42, 6, 69, 242, 47, 170, 149, 121, 142, 185, 225, 212, 167, 145, 100, 105, 224, 168,
            223, 75, 70, 49, 78, 136, 148, 163, 129, 189, 213, 99, 57, 7,
        ],
        [
            202, 48, 49, 166, 179, 127, 171, 43, 179, 215, 185, 98, 122, 169, 76, 163, 162, 152,
            151, 234, 229, 1, 174, 31, 106, 210, 184, 103, 68, 174, 231, 14,
        ],
        [
            4, 1, 223, 186, 118, 214, 99, 238, 170, 254, 46, 156, 164, 66, 220, 145, 35, 249, 225,
            0, 146, 146, 32, 90, 63, 225, 4, 213, 120, 168, 202, 31,
        ],
        [
            191, 2, 7, 207, 2, 3, 132, 59, 8, 17, 130, 119, 167, 244, 65, 80, 33, 93, 217, 36, 196,
            247, 121, 191, 101, 80, 78, 103, 89, 253, 243, 4,
        ],
        [
            9, 202, 97, 46, 9, 57, 25, 238, 13, 173, 253, 219, 178, 11, 191, 62, 82, 207, 231, 62,
            8, 85, 176, 21, 98, 73, 141, 255, 69, 71, 22, 2,
        ],
        [
            60, 6, 122, 230, 204, 92, 159, 86, 44, 51, 165, 228, 157, 166, 23, 89, 139, 113, 97,
            190, 96, 11, 249, 114, 210, 177, 241, 181, 193, 129, 248, 45,
        ],
        [
            127, 187, 205, 131, 198, 160, 221, 214, 70, 29, 3, 78, 25, 70, 87, 81, 163, 105, 247,
            244, 39, 47, 27, 201, 134, 108, 160, 68, 208, 167, 154, 19,
        ],
        [
            88, 198, 1, 133, 122, 100, 181, 13, 161, 66, 143, 128, 97, 99, 180, 200, 72, 50, 81,
            197, 140, 51, 43, 34, 107, 170, 115, 22, 89, 214, 148, 19,
        ],
    ],
    [
        [
            209, 79, 29, 202, 130, 199, 138, 237, 198, 43, 240, 165, 246, 59, 173, 155, 149, 89,
            176, 38, 98, 32, 55, 57, 108, 38, 39, 35, 198, 30, 70, 19,
        ],
        [
            62, 138, 183, 26, 93, 155, 177, 41, 76, 46, 239, 209, 50, 0, 254, 176, 243, 143, 41,
            55, 205, 156, 52, 194, 186, 253, 134, 3, 91, 137, 64, 60,
        ],
        [
            80, 193, 154, 56, 1, 162, 178, 97, 39, 58, 89, 158, 81, 12, 66, 239, 39, 78, 240, 163,
            245, 197, 106, 91, 36, 144, 179, 175, 242, 170, 115, 9,
        ],
        [
            88, 53, 242, 253, 61, 8, 226, 3, 32, 199, 199, 180, 38, 237, 102, 177, 91, 144, 79, 16,
            43, 224, 47, 42, 149, 106, 48, 178, 185, 78, 194, 13,
        ],
        [
            185, 170, 69, 129, 68, 165, 94, 97, 241, 117, 27, 40, 168, 74, 208, 125, 41, 231, 207,
            184, 204, 188, 239, 161, 170, 101, 109, 96, 40, 127, 9, 47,
        ],
        [
            81, 215, 213, 32, 75, 212, 14, 16, 55, 113, 242, 220, 197, 92, 200, 96, 231, 48, 87,
            21, 233, 175, 160, 133, 37, 83, 165, 1, 105, 134, 137, 53,
        ],
        [
            42, 175, 116, 152, 103, 108, 52, 117, 235, 100, 55, 253, 99, 9, 170, 189, 196, 204,
            243, 121, 96, 28, 164, 2, 84, 75, 93, 32, 35, 74, 106, 19,
        ],
        [
            205, 37, 25, 223, 168, 4, 64, 239, 149, 231, 158, 24, 100, 88, 21, 68, 153, 174, 171,
            142, 203, 23, 79, 195, 120, 135, 251, 184, 199, 198, 118, 59,
        ],
    ],
    [
        [
            236, 72, 236, 182, 97, 100, 170, 247, 194, 142, 191, 163, 104, 232, 55, 188, 179, 3,
            241, 161, 130, 187, 201, 135, 36, 107, 194, 11, 58, 37, 198, 21,
        ],
        [
            162, 108, 210, 213, 229, 7, 184, 128, 48, 218, 122, 152, 34, 44, 255, 129, 8, 7, 70,
            159, 165, 102, 133, 198, 89, 136, 9, 210, 81, 209, 125, 62,
        ],
        [
            19, 122, 208, 53, 54, 195, 80, 105, 130, 108, 101, 232, 213, 250, 94, 143, 34, 16, 123,
            22, 85, 24, 222, 116, 81, 77, 146, 123, 63, 109, 155, 25,
        ],
        [
            20, 202, 194, 34, 23, 228, 108, 225, 197, 169, 220, 145, 3, 87, 252, 183, 201, 134, 9,
            70, 26, 33, 233, 245, 90, 40, 9, 193, 145, 177, 137, 30,
        ],
        [
            142, 118, 192, 79, 106, 86, 122, 98, 133, 200, 92, 17, 203, 195, 232, 141, 214, 22,
            103, 255, 60, 51, 44, 88, 64, 130, 184, 251, 182, 235, 174, 33,
        ],
        [
            136, 174, 114, 92, 123, 204, 141, 133, 43, 185, 26, 57, 221, 186, 86, 185, 32, 212, 97,
            202, 226, 129, 224, 47, 143, 83, 70, 209, 87, 203, 198, 21,
        ],
        [
            59, 223, 14, 205, 86, 189, 127, 244, 44, 140, 133, 52, 255, 246, 253, 24, 44, 25, 191,
            198, 49, 247, 200, 84, 150, 16, 205, 200, 110, 226, 60, 51,
        ],
        [
            196, 65, 164, 48, 29, 211, 202, 47, 236, 1, 149, 74, 82, 175, 50, 154, 47, 128, 24,
            144, 152, 172, 164, 95, 48, 1, 175, 159, 21, 247, 184, 45,
        ],
    ],
    [
        [
            113, 63, 185, 131, 212, 136, 143, 154, 164, 166, 141, 78, 33, 22, 153, 205, 188, 108,
            173, 204, 209, 27, 18, 248, 69, 250, 136, 134, 86, 66, 2, 25,
        ],
        [
            81, 213, 197, 160, 99, 146, 255, 95, 58, 208, 199, 37, 174, 131, 30, 27, 44, 119, 238,
            23, 8, 85, 177, 30, 23, 198, 236, 206, 160, 196, 192, 37,
        ],
        [
            137, 140, 108, 240, 119, 166, 103, 173, 83, 168, 239, 88, 58, 195, 143, 255, 23, 197,
            34, 204, 167, 220, 180, 119, 136, 63, 224, 235, 87, 32, 22, 19,
        ],
        [
            9, 120, 1, 228, 172, 242, 68, 238, 34, 31, 73, 46, 221, 161, 229, 216, 212, 51, 236,
            60, 129, 118, 235, 30, 173, 98, 105, 92, 38, 48, 179, 63,
        ],
        [
            101, 254, 224, 108, 2, 7, 51, 58, 155, 127, 90, 231, 44, 99, 139, 228, 104, 107, 248,
            235, 10, 231, 131, 133, 26, 9, 5, 238, 79, 173, 149, 53,
        ],
        [
            173, 174, 13, 8, 187, 48, 186, 134, 32, 196, 154, 196, 109, 220, 41, 70, 235, 190, 7,
            157, 94, 233, 100, 151, 26, 193, 252, 252, 204, 166, 215, 19,
        ],
        [
            28, 95, 198, 85, 93, 171, 70, 119, 154, 50, 27, 16, 190, 246, 143, 130, 168, 183, 89,
            119, 4, 58, 205, 161, 105, 28, 156, 158, 83, 20, 197, 49,
        ],
        [
            113, 106, 199, 144, 147, 223, 231, 150, 7, 222, 112, 180, 120, 139, 99, 219, 8, 246,
            89, 224, 255, 179, 139, 176, 16, 103, 178, 74, 156, 40, 204, 37,
        ],
    ],
    [
        [
            38, 34, 221, 71, 220, 112, 148, 120, 84, 22, 39, 92, 53, 80, 244, 119, 64, 234, 90, 97,
            135, 161, 115, 110, 22, 189, 44, 36, 23, 85, 38, 60,
        ],
        [
            213, 170, 91, 62, 172, 9, 75, 190, 253, 21, 22, 180, 194, 51, 94, 46, 65, 244, 206,
            220, 81, 152, 136, 195, 45, 33, 133, 242, 112, 223, 144, 7,
        ],
        [
            29, 76, 103, 10, 234, 122, 243, 58, 113, 123, 43, 25, 116, 213, 214, 178, 247, 113,
            135, 238, 148, 58, 116, 97, 83, 203, 1, 208, 121, 52, 38, 13,
        ],
        [
            23, 152, 157, 192, 12, 28, 52, 28, 155, 182, 184, 142, 66, 105, 69, 154, 59, 92, 117,
            37, 115, 67, 17, 111, 24, 253, 41, 147, 51, 203, 39, 46,
        ],
        [
            255, 207, 178, 103, 215, 12, 12, 158, 83, 240, 123, 146, 137, 235, 116, 25, 102, 144,
            56, 182, 234, 251, 238, 201, 56, 41, 137, 96, 239, 216, 162, 34,
        ],
        [
            148, 146, 139, 236, 159, 116, 15, 54, 141, 119, 201, 163, 202, 166, 244, 120, 14, 220,
            191, 209, 31, 121, 208, 243, 120, 106, 102, 3, 129, 135, 29, 36,
        ],
        [
            80, 37, 232, 226, 149, 80, 204, 156, 22, 11, 19, 34, 221, 211, 204, 75, 173, 150, 68,
            253, 109, 0, 156, 187, 182, 100, 79, 74, 250, 255, 2, 62,
        ],
        [
            155, 207, 110, 75, 63, 173, 133, 240, 21, 108, 245, 233, 80, 114, 0, 221, 114, 148, 5,
            141, 127, 163, 96, 9, 253, 75, 91, 74, 137, 126, 94, 59,
        ],
    ],
    [
        [
            235, 150, 195, 122, 252, 131, 216, 100, 102, 193, 233, 212, 79, 117, 151, 50, 246, 124,
            8, 1, 247, 254, 208, 90, 247, 231, 240, 3, 86, 82, 75, 58,
        ],
        [
            41, 49, 135, 170, 77, 96, 175, 215, 190, 42, 192, 62, 122, 169, 219, 246, 241, 143, 42,
            119, 172, 141, 224, 125, 128, 75, 40, 154, 90, 163, 236, 9,
        ],
        [
            206, 138, 164, 80, 147, 233, 240, 203, 111, 52, 22, 4, 38, 199, 245, 224, 203, 7, 14,
            101, 212, 245, 30, 178, 177, 76, 194, 55, 43, 124, 155, 15,
        ],
        [
            93, 30, 127, 191, 82, 233, 253, 219, 58, 221, 171, 157, 164, 145, 223, 121, 11, 152,
            114, 200, 186, 150, 146, 87, 11, 48, 157, 85, 162, 36, 127, 8,
        ],
        [
            95, 35, 201, 218, 74, 87, 138, 236, 183, 10, 32, 124, 190, 37, 47, 8, 136, 96, 117,
            164, 62, 4, 77, 251, 40, 70, 172, 22, 186, 85, 29, 30,
        ],
        [
            133, 23, 191, 232, 143, 103, 124, 29, 186, 196, 156, 170, 254, 197, 48, 205, 114, 31,
            159, 157, 72, 26, 86, 115, 177, 128, 75, 211, 207, 109, 187, 40,
        ],
        [
            129, 18, 198, 231, 250, 247, 65, 206, 32, 191, 126, 179, 56, 192, 26, 255, 151, 215,
            103, 101, 40, 182, 203, 39, 100, 126, 202, 129, 173, 114, 208, 46,
        ],
        [
            232, 235, 95, 198, 201, 94, 246, 81, 82, 104, 27, 178, 133, 79, 120, 171, 186, 171, 89,
            233, 83, 28, 26, 22, 73, 23, 229, 8, 239, 200, 19, 27,
        ],
    ],
    [
        [
            210, 71, 108, 215, 152, 32, 62, 185, 111, 112, 16, 4, 13, 227, 37, 156, 126, 114, 174,
            38, 138, 245, 46, 170, 103, 115, 52, 113, 17, 234, 43, 34,
        ],
        [
            210, 57, 57, 191, 219, 197, 56, 74, 5, 219, 191, 250, 122, 229, 56, 27, 192, 255, 143,
            226, 134, 81, 82, 20, 228, 137, 206, 11, 203, 171, 194, 2,
        ],
        [
            2, 78, 48, 95, 121, 129, 100, 124, 153, 168, 144, 254, 51, 226, 117, 249, 102, 93, 236,
            209, 195, 58, 248, 173, 65, 106, 174, 194, 116, 87, 96, 57,
        ],
        [
            241, 60, 10, 54, 190, 93, 196, 234, 99, 112, 230, 21, 141, 232, 41, 30, 250, 202, 112,
            58, 179, 160, 100, 1, 209, 6, 141, 184, 63, 124, 137, 38,
        ],
        [
            58, 21, 254, 142, 179, 18, 16, 67, 218, 111, 138, 59, 181, 29, 207, 76, 39, 76, 177,
            120, 66, 64, 22, 178, 208, 40, 155, 120, 211, 84, 207, 21,
        ],
        [
            174, 159, 153, 25, 242, 221, 84, 226, 243, 117, 42, 141, 100, 128, 90, 79, 193, 255,
            179, 12, 31, 233, 79, 219, 143, 35, 12, 139, 59, 105, 189, 25,
        ],
        [
            218, 68, 43, 94, 237, 195, 110, 37, 78, 248, 223, 30, 109, 144, 98, 161, 59, 211, 240,
            180, 222, 116, 134, 183, 183, 166, 97, 66, 241, 222, 53, 18,
        ],
        [
            245, 148, 191, 198, 199, 1, 171, 246, 89, 150, 11, 34, 102, 248, 117, 200, 37, 182,
            103, 126, 70, 226, 218, 203, 11, 125, 121, 32, 18, 230, 243, 18,
        ],
    ],
    [
        [
            255, 144, 208, 132, 52, 254, 117, 148, 102, 19, 153, 74, 6, 56, 14, 56, 0, 104, 184,
            201, 182, 204, 210, 221, 180, 73, 127, 109, 252, 126, 24, 33,
        ],
        [
            47, 153, 208, 169, 51, 149, 24, 41, 69, 165, 93, 193, 209, 66, 187, 89, 18, 193, 252,
            170, 55, 6, 107, 77, 170, 123, 155, 171, 170, 110, 41, 13,
        ],
        [
            168, 134, 125, 156, 48, 7, 212, 22, 221, 205, 11, 8, 187, 85, 52, 124, 218, 128, 24,
            106, 88, 190, 65, 166, 34, 120, 160, 186, 50, 209, 57, 27,
        ],
        [
            12, 230, 242, 95, 27, 44, 255, 38, 151, 185, 224, 26, 193, 155, 36, 198, 219, 80, 217,
            78, 29, 16, 14, 160, 95, 137, 114, 98, 32, 28, 117, 16,
        ],
        [
            19, 221, 130, 39, 185, 145, 167, 178, 12, 122, 174, 141, 109, 15, 59, 92, 217, 193, 8,
            140, 77, 168, 45, 8, 43, 157, 217, 73, 46, 223, 82, 23,
        ],
        [
            11, 218, 183, 42, 61, 124, 53, 72, 153, 53, 169, 155, 23, 136, 33, 20, 216, 88, 174,
            50, 165, 138, 100, 174, 31, 151, 244, 200, 102, 163, 211, 45,
        ],
        [
            40, 208, 183, 220, 99, 217, 214, 147, 127, 3, 227, 197, 133, 174, 177, 28, 89, 48, 195,
            227, 55, 51, 79, 76, 252, 236, 8, 4, 74, 196, 102, 57,
        ],
        [
            217, 25, 168, 251, 68, 161, 167, 143, 58, 241, 41, 14, 245, 47, 92, 191, 48, 195, 26,
            228, 238, 180, 236, 8, 105, 149, 218, 119, 31, 56, 177, 56,
        ],
    ],
    [
        [
            31, 98, 35, 5, 253, 140, 112, 75, 250, 170, 119, 208, 186, 119, 54, 79, 34, 155, 132,
            123, 178, 111, 84, 7, 45, 231, 74, 65, 216, 235, 174, 16,
        ],
        [
            197, 28, 157, 33, 185, 182, 123, 128, 223, 137, 177, 196, 237, 219, 169, 53, 32, 100,
            251, 238, 92, 182, 118, 231, 8, 2, 229, 32, 143, 187, 243, 11,
        ],
        [
            53, 49, 78, 10, 50, 233, 255, 107, 193, 56, 21, 50, 236, 45, 145, 145, 147, 62, 44,
            111, 70, 71, 175, 53, 203, 148, 52, 67, 100, 19, 26, 3,
        ],
        [
            161, 103, 224, 78, 117, 78, 234, 195, 253, 216, 60, 242, 228, 181, 84, 115, 93, 60, 78,
            177, 233, 239, 55, 51, 87, 29, 91, 73, 160, 22, 171, 28,
        ],
        [
            106, 94, 122, 161, 158, 123, 47, 190, 164, 230, 236, 39, 112, 190, 86, 119, 220, 119,
            136, 5, 205, 153, 136, 144, 195, 36, 8, 31, 156, 77, 197, 7,
        ],
        [
            84, 189, 79, 111, 149, 133, 18, 214, 25, 129, 102, 3, 255, 214, 137, 117, 145, 219,
            122, 117, 251, 157, 67, 103, 112, 156, 101, 60, 196, 24, 228, 9,
        ],
        [
            221, 128, 184, 19, 209, 70, 94, 235, 168, 5, 193, 131, 176, 112, 12, 162, 60, 52, 126,
            165, 78, 106, 205, 247, 39, 129, 58, 66, 183, 218, 92, 34,
        ],
        [
            37, 90, 158, 71, 194, 0, 179, 71, 130, 53, 224, 14, 163, 30, 186, 122, 241, 4, 160,
            224, 157, 159, 58, 19, 72, 213, 191, 207, 214, 79, 97, 7,
        ],
    ],
    [
        [
            17, 77, 140, 150, 199, 124, 25, 200, 242, 238, 45, 141, 236, 5, 130, 183, 54, 235, 128,
            228, 212, 54, 81, 133, 39, 76, 208, 190, 12, 132, 119, 11,
        ],
        [
            107, 215, 140, 18, 7, 122, 91, 92, 146, 39, 245, 151, 85, 68, 229, 96, 154, 146, 33,
            101, 126, 111, 236, 5, 204, 109, 157, 205, 70, 121, 217, 0,
        ],
        [
            70, 94, 179, 106, 229, 170, 170, 47, 55, 65, 230, 43, 24, 82, 194, 151, 155, 198, 165,
            155, 205, 28, 14, 104, 20, 71, 197, 140, 250, 71, 125, 17,
        ],
        [
            106, 252, 217, 245, 224, 67, 205, 117, 213, 48, 7, 135, 102, 189, 26, 18, 4, 54, 28,
            251, 124, 84, 93, 0, 40, 66, 115, 117, 204, 79, 228, 22,
        ],
        [
            242, 35, 187, 88, 191, 59, 140, 236, 161, 109, 107, 209, 136, 246, 243, 27, 223, 98,
            151, 100, 212, 119, 181, 232, 39, 21, 93, 43, 28, 54, 121, 28,
        ],
        [
            71, 151, 100, 227, 73, 205, 19, 3, 130, 251, 173, 106, 89, 57, 39, 39, 122, 133, 165,
            156, 3, 206, 44, 215, 128, 92, 207, 248, 158, 242, 254, 0,
        ],
        [
            135, 68, 93, 82, 180, 183, 2, 251, 43, 188, 239, 107, 205, 155, 212, 29, 221, 31, 106,
            214, 183, 206, 119, 193, 233, 125, 198, 221, 69, 92, 33, 62,
        ],
        [
            30, 16, 32, 89, 21, 239, 233, 59, 207, 112, 12, 6, 215, 78, 169, 126, 167, 25, 242, 72,
            91, 134, 70, 202, 21, 211, 175, 45, 241, 46, 126, 19,
        ],
    ],
    [
        [
            239, 132, 33, 38, 102, 212, 72, 89, 11, 15, 97, 253, 10, 0, 7, 0, 193, 147, 176, 43,
            213, 38, 141, 110, 55, 61, 127, 15, 217, 159, 199, 41,
        ],
        [
            214, 253, 151, 31, 243, 238, 244, 4, 240, 234, 253, 169, 181, 192, 6, 57, 219, 163,
            185, 123, 103, 228, 127, 129, 214, 214, 43, 225, 63, 75, 182, 53,
        ],
        [
            32, 79, 73, 179, 13, 71, 33, 110, 122, 154, 127, 57, 134, 49, 208, 108, 5, 172, 153,
            147, 192, 46, 212, 252, 33, 114, 20, 39, 244, 63, 96, 59,
        ],
        [
            235, 135, 102, 44, 198, 110, 108, 192, 212, 172, 181, 10, 197, 131, 11, 220, 40, 83,
            211, 8, 182, 147, 94, 90, 76, 129, 90, 194, 77, 213, 126, 55,
        ],
        [
            134, 43, 3, 60, 183, 128, 197, 163, 170, 11, 127, 13, 12, 251, 24, 190, 45, 105, 181,
            131, 97, 232, 88, 155, 212, 247, 110, 50, 247, 242, 78, 11,
        ],
        [
            107, 99, 135, 228, 201, 238, 31, 15, 7, 159, 61, 63, 134, 124, 107, 227, 233, 133, 57,
            126, 106, 183, 205, 196, 222, 230, 130, 213, 208, 64, 99, 20,
        ],
        [
            136, 191, 122, 221, 143, 194, 149, 206, 210, 101, 122, 87, 41, 73, 39, 0, 203, 249,
            255, 105, 157, 67, 114, 227, 107, 204, 112, 134, 137, 141, 192, 40,
        ],
        [
            53, 251, 202, 6, 165, 211, 68, 162, 219, 79, 119, 243, 221, 34, 46, 97, 240, 59, 81,
            62, 192, 122, 250, 188, 247, 91, 173, 184, 16, 162, 228, 42,
        ],
    ],
    [
        [
            207, 188, 203, 116, 227, 47, 15, 94, 109, 85, 244, 221, 60, 114, 166, 91, 220, 124,
            217, 87, 15, 10, 223, 41, 201, 185, 195, 188, 26, 179, 0, 42,
        ],
        [
            169, 222, 42, 237, 79, 30, 175, 161, 31, 217, 5, 161, 189, 73, 91, 163, 120, 84, 89,
            232, 151, 51, 37, 39, 130, 149, 209, 68, 215, 4, 169, 47,
        ],
        [
            227, 83, 192, 240, 71, 1, 222, 202, 123, 190, 215, 154, 182, 192, 122, 211, 174, 143,
            187, 185, 69, 74, 141, 161, 31, 218, 147, 239, 72, 246, 120, 4,
        ],
        [
            137, 27, 69, 79, 29, 209, 12, 234, 186, 93, 226, 104, 95, 211, 110, 68, 30, 100, 164,
            236, 245, 45, 0, 29, 59, 22, 176, 196, 130, 184, 217, 27,
        ],
        [
            26, 62, 9, 239, 203, 49, 46, 166, 96, 81, 208, 1, 72, 8, 94, 190, 19, 237, 242, 213,
            15, 157, 31, 240, 39, 122, 94, 8, 171, 16, 159, 28,
        ],
        [
            191, 213, 238, 242, 171, 187, 13, 177, 204, 117, 29, 64, 237, 164, 165, 236, 142, 220,
            60, 63, 61, 57, 146, 155, 58, 206, 128, 15, 73, 253, 188, 9,
        ],
        [
            184, 134, 37, 128, 143, 22, 57, 204, 184, 89, 54, 236, 170, 123, 109, 20, 114, 172,
            121, 243, 37, 217, 219, 38, 71, 67, 51, 49, 13, 31, 79, 36,
        ],
        [
            223, 219, 38, 58, 177, 64, 40, 107, 71, 58, 211, 112, 35, 73, 154, 52, 180, 252, 153,
            239, 154, 5, 210, 15, 116, 102, 250, 226, 17, 79, 2, 0,
        ],
    ],
    [
        [
            86, 108, 216, 34, 154, 182, 132, 72, 67, 161, 20, 204, 92, 218, 109, 31, 146, 215, 73,
            85, 186, 63, 178, 83, 74, 63, 115, 248, 215, 203, 131, 31,
        ],
        [
            68, 146, 252, 110, 32, 80, 133, 118, 93, 102, 51, 124, 114, 76, 141, 132, 243, 175, 35,
            2, 151, 83, 90, 137, 188, 59, 33, 45, 144, 241, 77, 59,
        ],
        [
            199, 168, 25, 87, 146, 91, 7, 45, 196, 18, 34, 77, 121, 59, 49, 171, 218, 111, 121,
            144, 67, 17, 163, 60, 41, 184, 123, 35, 107, 158, 204, 42,
        ],
        [
            81, 34, 122, 126, 218, 213, 211, 210, 132, 204, 13, 131, 85, 187, 237, 173, 173, 72,
            210, 134, 63, 167, 97, 139, 126, 221, 254, 217, 147, 76, 48, 36,
        ],
        [
            212, 17, 78, 238, 49, 160, 218, 85, 86, 250, 124, 160, 63, 77, 80, 153, 20, 197, 70,
            88, 8, 92, 28, 108, 34, 214, 138, 234, 220, 191, 6, 42,
        ],
        [
            121, 219, 39, 10, 207, 76, 241, 73, 84, 53, 101, 47, 129, 70, 245, 235, 249, 253, 118,
            193, 89, 63, 27, 201, 16, 103, 193, 199, 206, 22, 188, 4,
        ],
        [
            32, 135, 154, 90, 39, 88, 11, 21, 28, 118, 172, 80, 157, 117, 227, 194, 95, 21, 193,
            244, 134, 48, 20, 209, 135, 51, 22, 226, 172, 35, 170, 32,
        ],
        [
            43, 221, 151, 128, 141, 210, 207, 198, 229, 190, 198, 122, 10, 175, 184, 64, 55, 212,
            147, 45, 67, 156, 35, 168, 175, 147, 89, 191, 250, 231, 99, 62,
        ],
    ],
    [
        [
            243, 133, 64, 65, 91, 46, 144, 139, 138, 115, 84, 217, 89, 221, 170, 23, 2, 205, 113,
            152, 46, 159, 81, 126, 63, 7, 137, 165, 116, 51, 40, 11,
        ],
        [
            109, 94, 105, 247, 230, 206, 57, 126, 225, 207, 205, 33, 125, 9, 155, 204, 119, 58,
            221, 209, 138, 16, 203, 96, 189, 26, 45, 92, 3, 36, 48, 9,
        ],
        [
            254, 204, 107, 219, 208, 50, 17, 142, 30, 121, 107, 37, 62, 51, 152, 159, 53, 134, 215,
            86, 73, 214, 25, 237, 64, 69, 212, 112, 191, 153, 246, 44,
        ],
        [
            99, 148, 101, 253, 131, 235, 77, 241, 22, 107, 74, 217, 103, 175, 136, 203, 69, 61,
            221, 193, 214, 8, 102, 175, 83, 121, 218, 128, 206, 159, 25, 23,
        ],
        [
            111, 64, 79, 248, 50, 79, 231, 252, 52, 151, 216, 18, 195, 193, 18, 109, 126, 164, 43,
            102, 23, 39, 211, 247, 10, 29, 172, 211, 182, 31, 245, 13,
        ],
        [
            3, 86, 28, 24, 154, 35, 148, 150, 230, 16, 146, 211, 24, 184, 121, 224, 111, 36, 128,
            45, 67, 71, 97, 234, 173, 172, 203, 44, 219, 134, 47, 22,
        ],
        [
            62, 127, 47, 223, 234, 188, 44, 197, 94, 21, 47, 75, 38, 226, 46, 82, 214, 220, 94,
            205, 176, 252, 226, 208, 85, 7, 191, 28, 217, 158, 83, 13,
        ],
        [
            96, 225, 72, 121, 83, 10, 196, 69, 54, 68, 161, 232, 58, 131, 166, 200, 208, 196, 134,
            18, 85, 234, 37, 244, 239, 42, 197, 250, 60, 118, 37, 4,
        ],
    ],
    [
        [
            100, 154, 11, 180, 188, 20, 222, 46, 29, 254, 104, 147, 128, 159, 209, 246, 95, 79, 6,
            158, 211, 154, 74, 202, 45, 8, 199, 59, 67, 89, 31, 36,
        ],
        [
            59, 218, 141, 48, 106, 107, 91, 63, 20, 57, 51, 66, 177, 136, 20, 151, 170, 87, 166,
            122, 26, 134, 223, 101, 144, 92, 104, 253, 117, 115, 198, 48,
        ],
        [
            6, 234, 190, 8, 45, 67, 212, 14, 215, 229, 43, 147, 98, 177, 241, 248, 142, 172, 9, 27,
            42, 223, 91, 180, 75, 219, 159, 182, 205, 5, 110, 1,
        ],
        [
            139, 41, 229, 150, 94, 20, 37, 79, 90, 170, 31, 21, 106, 60, 63, 40, 225, 44, 156, 254,
            70, 77, 196, 147, 128, 130, 180, 151, 41, 75, 158, 33,
        ],
        [
            139, 111, 3, 218, 41, 122, 103, 61, 112, 106, 17, 227, 74, 93, 112, 194, 70, 21, 101,
            116, 48, 164, 124, 20, 170, 241, 78, 147, 89, 107, 12, 37,
        ],
        [
            135, 10, 234, 73, 60, 129, 112, 145, 165, 224, 166, 60, 131, 105, 236, 222, 39, 59,
            227, 161, 113, 246, 119, 103, 24, 162, 5, 55, 245, 221, 19, 11,
        ],
        [
            79, 204, 186, 89, 248, 255, 253, 34, 43, 99, 55, 53, 162, 27, 15, 51, 206, 113, 185,
            154, 102, 75, 154, 80, 115, 51, 169, 203, 111, 57, 253, 15,
        ],
        [
            241, 70, 150, 96, 14, 181, 110, 248, 200, 60, 151, 32, 56, 236, 1, 160, 207, 239, 204,
            96, 180, 20, 114, 141, 204, 79, 222, 42, 233, 93, 86, 4,
        ],
    ],
    [
        [
            133, 42, 108, 146, 9, 233, 159, 193, 246, 146, 58, 201, 35, 57, 51, 79, 31, 110, 185,
            80, 198, 123, 90, 212, 230, 44, 5, 125, 69, 245, 133, 8,
        ],
        [
            229, 72, 83, 102, 213, 54, 195, 176, 137, 144, 238, 18, 31, 192, 209, 173, 251, 255,
            140, 176, 244, 204, 213, 213, 163, 231, 2, 147, 229, 252, 61, 45,
        ],
        [
            71, 127, 51, 61, 4, 61, 154, 153, 6, 190, 184, 184, 246, 195, 97, 120, 73, 143, 135,
            84, 154, 23, 119, 144, 120, 38, 55, 242, 88, 180, 27, 19,
        ],
        [
            75, 52, 60, 75, 156, 190, 238, 12, 77, 247, 183, 176, 64, 156, 173, 107, 27, 154, 11,
            252, 58, 116, 8, 98, 74, 169, 127, 147, 196, 228, 107, 32,
        ],
        [
            148, 235, 145, 32, 227, 216, 189, 85, 81, 174, 122, 74, 29, 157, 193, 199, 131, 131,
            215, 200, 40, 168, 165, 204, 234, 158, 148, 212, 238, 62, 183, 18,
        ],
        [
            154, 181, 252, 201, 70, 152, 237, 106, 10, 69, 44, 207, 168, 233, 5, 104, 110, 68, 207,
            203, 85, 242, 149, 191, 125, 123, 34, 246, 37, 233, 36, 44,
        ],
        [
            211, 231, 62, 59, 210, 172, 126, 106, 177, 163, 74, 94, 201, 226, 57, 90, 138, 213,
            177, 154, 128, 11, 241, 24, 122, 183, 243, 189, 220, 68, 234, 7,
        ],
        [
            56, 235, 124, 129, 101, 66, 101, 95, 248, 36, 151, 219, 71, 118, 126, 54, 161, 159,
            130, 62, 236, 239, 129, 232, 227, 45, 180, 127, 204, 15, 191, 41,
        ],
    ],
    [
        [
            58, 7, 140, 180, 114, 238, 131, 202, 207, 64, 62, 207, 210, 71, 132, 77, 52, 11, 81,
            73, 215, 25, 106, 0, 178, 254, 238, 8, 223, 22, 116, 55,
        ],
        [
            223, 58, 89, 147, 0, 219, 48, 190, 127, 47, 35, 39, 120, 55, 84, 162, 29, 15, 239, 70,
            105, 190, 179, 179, 161, 77, 68, 62, 74, 245, 2, 0,
        ],
        [
            119, 42, 91, 164, 172, 227, 112, 30, 94, 87, 129, 68, 103, 131, 240, 24, 28, 173, 61,
            51, 123, 176, 110, 204, 238, 162, 214, 123, 67, 43, 77, 61,
        ],
        [
            234, 137, 155, 145, 183, 245, 247, 147, 139, 194, 96, 144, 204, 154, 159, 94, 97, 150,
            59, 129, 113, 213, 0, 184, 73, 215, 212, 114, 207, 10, 31, 30,
        ],
        [
            42, 90, 73, 230, 113, 98, 152, 59, 179, 244, 222, 133, 142, 178, 172, 225, 57, 210, 95,
            96, 162, 203, 63, 245, 108, 203, 167, 94, 125, 54, 162, 58,
        ],
        [
            240, 196, 76, 186, 213, 181, 225, 51, 212, 188, 155, 194, 158, 183, 234, 110, 181, 97,
            93, 116, 201, 160, 208, 67, 218, 169, 210, 210, 11, 33, 212, 7,
        ],
        [
            68, 105, 62, 121, 37, 69, 237, 189, 194, 25, 197, 246, 64, 28, 252, 193, 66, 110, 140,
            255, 218, 86, 133, 31, 198, 252, 165, 237, 73, 22, 15, 60,
        ],
        [
            210, 33, 63, 244, 100, 40, 160, 16, 81, 38, 120, 211, 176, 13, 155, 255, 113, 214, 10,
            73, 204, 178, 186, 189, 207, 91, 142, 112, 152, 74, 42, 17,
        ],
    ],
    [
        [
            65, 190, 3, 42, 54, 225, 195, 120, 113, 25, 19, 40, 59, 78, 139, 70, 53, 104, 84, 240,
            93, 38, 233, 201, 29, 218, 43, 14, 16, 238, 1, 3,
        ],
        [
            94, 85, 50, 4, 205, 48, 215, 233, 248, 185, 111, 219, 70, 219, 70, 1, 2, 75, 187, 253,
            5, 210, 180, 219, 163, 252, 251, 224, 27, 158, 205, 35,
        ],
        [
            145, 45, 21, 30, 132, 8, 164, 37, 104, 121, 240, 200, 86, 247, 240, 166, 128, 202, 205,
            12, 252, 53, 120, 60, 180, 106, 148, 158, 111, 18, 126, 40,
        ],
        [
            53, 47, 28, 89, 156, 147, 71, 72, 158, 73, 217, 107, 73, 132, 49, 120, 42, 30, 182,
            213, 150, 114, 90, 223, 215, 71, 92, 251, 21, 142, 200, 0,
        ],
        [
            21, 187, 146, 84, 186, 214, 154, 62, 147, 23, 51, 157, 249, 201, 137, 79, 187, 184,
            129, 149, 53, 241, 85, 154, 207, 58, 122, 229, 14, 246, 154, 7,
        ],
        [
            71, 228, 225, 174, 83, 69, 84, 246, 152, 5, 21, 70, 130, 252, 61, 162, 9, 28, 95, 31,
            115, 251, 224, 36, 219, 224, 124, 5, 249, 147, 129, 52,
        ],
        [
            99, 84, 54, 113, 53, 228, 153, 72, 229, 33, 119, 65, 136, 127, 25, 133, 51, 92, 140,
            149, 34, 123, 82, 14, 117, 69, 132, 126, 250, 215, 130, 47,
        ],
        [
            158, 108, 91, 43, 179, 230, 110, 55, 17, 10, 220, 62, 145, 238, 91, 226, 158, 83, 51,
            101, 189, 13, 71, 78, 184, 42, 243, 17, 201, 181, 140, 11,
        ],
    ],
    [
        [
            194, 241, 4, 81, 158, 51, 209, 249, 35, 115, 149, 92, 55, 91, 174, 251, 202, 167, 82,
            52, 253, 32, 43, 121, 206, 222, 164, 43, 248, 220, 122, 12,
        ],
        [
            99, 154, 211, 41, 184, 45, 22, 111, 12, 195, 48, 243, 13, 253, 200, 229, 24, 205, 149,
            165, 148, 197, 251, 209, 182, 19, 159, 152, 94, 232, 156, 4,
        ],
        [
            0, 100, 188, 149, 90, 163, 159, 111, 0, 244, 218, 168, 18, 83, 125, 222, 206, 144, 226,
            94, 118, 121, 221, 75, 14, 17, 94, 243, 202, 242, 253, 27,
        ],
        [
            240, 46, 2, 247, 182, 224, 144, 79, 143, 124, 223, 40, 51, 131, 25, 96, 132, 163, 48,
            24, 133, 37, 180, 92, 120, 224, 236, 140, 128, 1, 189, 40,
        ],
        [
            119, 221, 95, 119, 141, 17, 159, 165, 72, 131, 170, 156, 220, 250, 5, 111, 209, 99, 9,
            38, 230, 163, 50, 149, 172, 255, 56, 203, 136, 161, 48, 21,
        ],
        [
            116, 77, 45, 31, 77, 80, 76, 36, 84, 112, 49, 174, 101, 182, 116, 15, 15, 242, 172,
            150, 114, 212, 46, 120, 187, 34, 145, 212, 0, 158, 239, 36,
        ],
        [
            234, 22, 164, 104, 165, 197, 122, 166, 202, 123, 178, 133, 187, 135, 235, 194, 206,
            219, 58, 179, 30, 245, 147, 110, 141, 124, 206, 1, 148, 209, 101, 4,
        ],
        [
            242, 73, 177, 217, 169, 198, 140, 205, 216, 73, 28, 38, 255, 90, 182, 211, 219, 192,
            108, 40, 247, 162, 57, 211, 108, 105, 233, 230, 138, 168, 92, 63,
        ],
    ],
    [
        [
            173, 95, 90, 146, 119, 75, 98, 69, 145, 115, 239, 18, 233, 216, 75, 241, 80, 100, 96,
            37, 221, 214, 199, 94, 228, 153, 135, 100, 155, 155, 216, 2,
        ],
        [
            150, 79, 236, 24, 163, 31, 70, 201, 143, 232, 204, 60, 220, 195, 134, 231, 11, 12, 144,
            0, 193, 32, 61, 209, 239, 139, 83, 197, 150, 64, 148, 59,
        ],
        [
            131, 108, 159, 162, 122, 223, 253, 251, 52, 16, 253, 139, 103, 224, 54, 119, 11, 188,
            86, 103, 124, 47, 156, 143, 37, 30, 210, 118, 193, 210, 84, 34,
        ],
        [
            43, 173, 230, 170, 177, 240, 16, 6, 56, 146, 203, 212, 135, 78, 45, 36, 207, 117, 225,
            215, 190, 197, 187, 20, 2, 61, 52, 114, 11, 136, 193, 10,
        ],
        [
            13, 77, 75, 121, 118, 79, 49, 124, 11, 205, 176, 65, 144, 48, 236, 222, 67, 180, 109,
            168, 43, 77, 168, 27, 2, 222, 123, 135, 11, 103, 6, 22,
        ],
        [
            214, 232, 233, 41, 25, 210, 118, 90, 85, 18, 143, 98, 244, 50, 251, 225, 101, 170, 229,
            58, 207, 175, 175, 233, 208, 5, 52, 27, 209, 230, 169, 31,
        ],
        [
            129, 131, 148, 184, 182, 36, 79, 246, 200, 243, 153, 36, 187, 3, 93, 208, 186, 153,
            173, 154, 142, 119, 197, 27, 57, 122, 240, 85, 105, 206, 17, 50,
        ],
        [
            241, 75, 9, 108, 156, 226, 162, 237, 221, 84, 169, 167, 11, 25, 191, 150, 233, 203, 61,
            64, 95, 176, 96, 79, 39, 143, 83, 172, 33, 238, 82, 10,
        ],
    ],
    [
        [
            249, 64, 46, 51, 246, 45, 133, 33, 65, 155, 225, 183, 65, 194, 179, 149, 118, 11, 201,
            218, 20, 8, 7, 98, 19, 54, 211, 59, 226, 174, 181, 30,
        ],
        [
            78, 52, 33, 54, 74, 139, 14, 51, 38, 151, 205, 116, 207, 250, 247, 61, 67, 250, 220,
            248, 184, 120, 243, 69, 81, 143, 46, 190, 151, 25, 209, 7,
        ],
        [
            117, 44, 92, 67, 68, 250, 114, 230, 86, 95, 107, 31, 210, 141, 32, 48, 43, 80, 5, 102,
            72, 39, 215, 25, 246, 7, 20, 183, 170, 130, 203, 40,
        ],
        [
            248, 52, 236, 99, 245, 230, 43, 187, 159, 164, 252, 246, 45, 35, 142, 166, 203, 99,
            243, 180, 131, 234, 104, 61, 198, 159, 140, 136, 45, 120, 20, 55,
        ],
        [
            197, 252, 78, 164, 142, 18, 216, 81, 116, 199, 65, 0, 35, 6, 95, 105, 67, 83, 162, 56,
            231, 180, 36, 75, 119, 162, 211, 252, 162, 108, 87, 7,
        ],
        [
            67, 43, 183, 166, 242, 83, 163, 0, 72, 129, 240, 196, 178, 90, 3, 33, 144, 139, 80,
            173, 176, 17, 119, 45, 223, 30, 157, 64, 198, 232, 11, 8,
        ],
        [
            236, 216, 184, 177, 72, 88, 104, 89, 162, 120, 101, 134, 110, 49, 69, 253, 105, 171,
            109, 13, 86, 253, 38, 253, 135, 146, 38, 248, 160, 88, 12, 6,
        ],
        [
            87, 44, 230, 170, 54, 93, 209, 204, 5, 240, 55, 102, 218, 228, 105, 33, 171, 112, 84,
            227, 131, 207, 119, 121, 115, 23, 26, 179, 8, 111, 253, 3,
        ],
    ],
    [
        [
            100, 157, 27, 13, 92, 74, 138, 226, 131, 160, 172, 107, 65, 128, 45, 170, 169, 90, 155,
            25, 167, 166, 136, 89, 232, 252, 203, 155, 70, 6, 182, 32,
        ],
        [
            159, 197, 68, 197, 232, 244, 71, 69, 251, 104, 129, 20, 231, 138, 116, 216, 109, 77,
            148, 92, 134, 5, 135, 227, 146, 116, 246, 221, 236, 104, 83, 26,
        ],
        [
            149, 158, 108, 59, 214, 113, 52, 126, 198, 118, 186, 164, 142, 109, 2, 1, 124, 233, 0,
            84, 130, 41, 182, 172, 37, 217, 212, 126, 222, 145, 202, 4,
        ],
        [
            165, 213, 205, 252, 170, 149, 33, 185, 40, 200, 132, 94, 128, 188, 2, 243, 141, 30,
            126, 208, 209, 234, 78, 69, 70, 69, 205, 216, 154, 51, 86, 10,
        ],
        [
            153, 122, 142, 76, 38, 219, 144, 223, 227, 232, 9, 56, 225, 5, 191, 92, 100, 51, 155,
            107, 205, 202, 74, 170, 103, 189, 30, 159, 68, 61, 202, 13,
        ],
        [
            134, 38, 203, 216, 65, 140, 214, 94, 24, 224, 177, 151, 140, 7, 41, 124, 40, 41, 1, 51,
            59, 89, 30, 130, 108, 87, 127, 214, 162, 183, 246, 22,
        ],
        [
            187, 193, 99, 160, 244, 73, 163, 177, 215, 30, 255, 31, 114, 223, 43, 85, 84, 226, 220,
            164, 67, 0, 238, 185, 249, 231, 198, 124, 172, 35, 149, 60,
        ],
        [
            45, 186, 4, 252, 12, 240, 229, 162, 18, 91, 136, 107, 201, 227, 77, 8, 175, 8, 198,
            103, 34, 168, 221, 193, 176, 128, 72, 93, 195, 26, 172, 3,
        ],
    ],
    [
        [
            91, 46, 86, 226, 133, 67, 28, 86, 126, 29, 234, 89, 216, 65, 179, 40, 36, 237, 114,
            194, 111, 26, 250, 5, 252, 24, 131, 155, 178, 92, 10, 11,
        ],
        [
            67, 208, 75, 127, 148, 190, 127, 53, 27, 219, 88, 183, 139, 50, 201, 19, 250, 158, 128,
            169, 150, 92, 76, 50, 89, 132, 55, 122, 219, 124, 16, 11,
        ],
        [
            140, 159, 86, 156, 195, 225, 138, 7, 78, 241, 245, 181, 207, 152, 209, 52, 121, 95, 35,
            65, 101, 20, 101, 88, 9, 130, 39, 135, 135, 74, 9, 1,
        ],
        [
            253, 181, 175, 91, 246, 184, 164, 214, 93, 116, 66, 47, 57, 58, 32, 0, 45, 28, 106, 79,
            214, 62, 176, 226, 255, 241, 191, 101, 224, 247, 9, 29,
        ],
        [
            233, 235, 128, 119, 38, 23, 25, 207, 212, 163, 148, 62, 177, 73, 115, 228, 237, 228,
            174, 133, 58, 20, 187, 192, 111, 239, 218, 119, 132, 233, 147, 28,
        ],
        [
            102, 191, 62, 245, 131, 141, 134, 94, 211, 147, 155, 226, 34, 19, 146, 177, 54, 235,
            23, 15, 176, 169, 246, 97, 178, 252, 103, 39, 105, 178, 162, 51,
        ],
        [
            7, 50, 217, 129, 106, 152, 234, 81, 110, 193, 51, 82, 62, 50, 179, 59, 89, 116, 206,
            117, 204, 73, 35, 115, 85, 176, 39, 231, 60, 170, 116, 47,
        ],
        [
            75, 196, 187, 39, 241, 38, 201, 59, 230, 232, 134, 175, 19, 66, 71, 243, 254, 64, 242,
            14, 160, 95, 24, 130, 33, 51, 183, 120, 216, 162, 193, 32,
        ],
    ],
    [
        [
            80, 139, 231, 43, 42, 117, 92, 23, 23, 28, 89, 92, 0, 118, 252, 26, 197, 165, 70, 191,
            196, 203, 169, 72, 141, 197, 13, 103, 216, 197, 184, 13,
        ],
        [
            171, 152, 143, 232, 240, 5, 166, 106, 27, 18, 57, 228, 232, 129, 41, 209, 77, 30, 210,
            74, 209, 33, 45, 74, 163, 209, 238, 156, 80, 77, 158, 15,
        ],
        [
            96, 136, 152, 220, 61, 122, 124, 180, 101, 158, 18, 10, 19, 70, 9, 13, 228, 26, 191,
            48, 235, 92, 34, 90, 241, 180, 94, 57, 155, 70, 144, 42,
        ],
        [
            179, 252, 92, 89, 59, 253, 147, 136, 110, 199, 105, 197, 235, 217, 159, 216, 232, 248,
            23, 191, 13, 61, 223, 38, 104, 236, 204, 89, 222, 220, 240, 61,
        ],
        [
            217, 91, 66, 69, 68, 44, 20, 147, 18, 87, 88, 66, 142, 170, 173, 93, 77, 15, 82, 94,
            26, 144, 31, 224, 164, 45, 203, 84, 24, 139, 14, 23,
        ],
        [
            61, 139, 126, 42, 167, 228, 59, 215, 79, 212, 163, 10, 48, 212, 246, 81, 180, 25, 156,
            101, 209, 148, 57, 138, 78, 49, 193, 14, 101, 158, 189, 55,
        ],
        [
            159, 6, 43, 236, 178, 210, 174, 111, 168, 153, 94, 23, 184, 185, 247, 237, 207, 133,
            190, 220, 41, 179, 2, 74, 193, 144, 244, 247, 129, 121, 73, 41,
        ],
        [
            73, 81, 174, 197, 26, 219, 123, 128, 63, 219, 134, 33, 79, 124, 244, 18, 207, 94, 155,
            181, 248, 239, 24, 32, 62, 101, 30, 162, 200, 148, 17, 59,
        ],
    ],
    [
        [
            21, 11, 152, 124, 138, 60, 93, 36, 7, 60, 214, 84, 45, 154, 172, 52, 35, 153, 17, 9,
            203, 118, 210, 182, 10, 95, 57, 145, 126, 8, 5, 42,
        ],
        [
            71, 101, 167, 15, 118, 90, 63, 8, 124, 156, 180, 28, 65, 161, 49, 86, 131, 205, 169,
            69, 84, 231, 193, 228, 214, 212, 145, 49, 194, 101, 242, 29,
        ],
        [
            7, 34, 97, 70, 161, 98, 83, 134, 241, 74, 132, 136, 64, 214, 70, 28, 200, 90, 12, 111,
            86, 123, 8, 251, 164, 2, 58, 177, 195, 146, 182, 21,
        ],
        [
            164, 230, 11, 145, 185, 76, 76, 97, 238, 93, 215, 40, 162, 213, 31, 242, 100, 67, 139,
            137, 118, 188, 204, 150, 253, 175, 94, 60, 210, 138, 9, 32,
        ],
        [
            108, 48, 162, 95, 78, 234, 152, 114, 166, 28, 255, 6, 184, 151, 224, 174, 128, 95, 140,
            153, 230, 85, 248, 241, 64, 210, 182, 139, 32, 250, 41, 32,
        ],
        [
            154, 194, 221, 96, 105, 253, 154, 78, 151, 82, 214, 33, 56, 41, 200, 160, 107, 92, 70,
            233, 152, 50, 229, 239, 233, 141, 44, 101, 191, 20, 37, 8,
        ],
        [
            87, 249, 195, 51, 136, 74, 211, 133, 253, 248, 97, 146, 179, 28, 66, 93, 36, 153, 66,
            239, 16, 69, 135, 216, 35, 151, 73, 200, 68, 9, 32, 13,
        ],
        [
            184, 229, 139, 19, 69, 111, 241, 199, 83, 10, 138, 24, 203, 217, 63, 209, 81, 219, 24,
            37, 49, 241, 79, 9, 138, 240, 48, 107, 29, 46, 55, 4,
        ],
    ],
    [
        [
            222, 18, 27, 133, 130, 121, 92, 228, 235, 31, 219, 249, 77, 230, 39, 140, 187, 30, 119,
            112, 172, 56, 28, 218, 242, 51, 136, 143, 49, 179, 27, 25,
        ],
        [
            2, 99, 67, 80, 156, 33, 160, 127, 126, 26, 10, 176, 146, 164, 212, 207, 172, 162, 165,
            143, 181, 41, 188, 59, 128, 2, 140, 243, 6, 191, 47, 50,
        ],
        [
            80, 146, 67, 231, 83, 95, 75, 75, 198, 78, 97, 42, 71, 41, 87, 178, 100, 226, 240, 179,
            51, 94, 229, 133, 55, 227, 162, 161, 155, 14, 160, 51,
        ],
        [
            184, 82, 151, 20, 59, 61, 157, 47, 88, 40, 125, 76, 132, 98, 13, 220, 157, 43, 174,
            135, 113, 131, 232, 159, 134, 112, 131, 61, 229, 56, 187, 31,
        ],
        [
            103, 159, 75, 105, 229, 173, 33, 147, 222, 198, 216, 51, 158, 126, 253, 155, 209, 71,
            97, 213, 227, 160, 59, 169, 255, 4, 158, 246, 202, 37, 41, 55,
        ],
        [
            70, 44, 245, 153, 162, 248, 57, 34, 159, 124, 179, 43, 174, 127, 126, 62, 239, 103, 71,
            230, 168, 6, 1, 249, 202, 249, 58, 64, 50, 84, 166, 49,
        ],
        [
            24, 11, 169, 224, 141, 175, 98, 63, 54, 81, 30, 229, 141, 220, 77, 48, 160, 66, 234,
            155, 148, 214, 56, 206, 162, 192, 93, 21, 37, 68, 223, 20,
        ],
        [
            12, 206, 30, 240, 150, 168, 254, 195, 216, 250, 139, 93, 74, 204, 177, 206, 186, 183,
            11, 216, 8, 219, 75, 55, 158, 241, 27, 121, 158, 126, 89, 37,
        ],
    ],
    [
        [
            126, 244, 161, 84, 141, 63, 108, 50, 97, 130, 49, 143, 229, 206, 114, 37, 17, 244, 30,
            5, 5, 65, 90, 231, 102, 176, 228, 110, 42, 206, 59, 7,
        ],
        [
            171, 250, 110, 27, 19, 51, 98, 97, 147, 101, 9, 200, 255, 24, 103, 220, 213, 243, 63,
            100, 129, 94, 129, 237, 192, 53, 44, 54, 81, 150, 163, 49,
        ],
        [
            207, 235, 227, 240, 49, 150, 106, 200, 76, 72, 90, 46, 38, 172, 83, 132, 225, 207, 25,
            95, 29, 88, 178, 163, 130, 5, 41, 2, 35, 56, 242, 5,
        ],
        [
            150, 235, 254, 218, 81, 133, 9, 45, 179, 227, 128, 87, 212, 188, 250, 90, 210, 132, 40,
            16, 118, 238, 125, 165, 27, 219, 224, 239, 34, 178, 94, 39,
        ],
        [
            104, 208, 155, 48, 14, 222, 119, 22, 228, 238, 214, 217, 249, 175, 117, 113, 143, 198,
            122, 25, 87, 198, 212, 60, 57, 28, 213, 213, 91, 130, 12, 38,
        ],
        [
            201, 85, 96, 155, 78, 135, 127, 238, 44, 177, 100, 78, 247, 155, 175, 223, 170, 205,
            210, 140, 238, 45, 94, 95, 244, 19, 5, 32, 43, 139, 214, 48,
        ],
        [
            249, 240, 53, 215, 1, 213, 54, 151, 143, 107, 23, 223, 161, 19, 249, 213, 102, 137,
            228, 184, 2, 78, 237, 177, 23, 49, 218, 85, 55, 62, 215, 20,
        ],
        [
            96, 201, 226, 19, 157, 94, 105, 10, 170, 204, 220, 109, 21, 94, 224, 26, 132, 56, 89,
            173, 176, 87, 145, 180, 174, 90, 228, 72, 206, 159, 48, 45,
        ],
    ],
    [
        [
            113, 76, 55, 16, 2, 198, 219, 166, 149, 128, 101, 84, 86, 171, 12, 5, 131, 54, 86, 113,
            196, 45, 45, 11, 118, 194, 183, 17, 58, 196, 31, 10,
        ],
        [
            172, 125, 221, 60, 54, 229, 101, 125, 102, 252, 27, 200, 177, 87, 186, 178, 60, 164,
            109, 220, 15, 121, 133, 105, 9, 238, 96, 11, 36, 59, 9, 30,
        ],
        [
            183, 108, 71, 108, 153, 232, 119, 177, 137, 136, 110, 110, 3, 219, 142, 115, 255, 120,
            114, 121, 148, 17, 97, 46, 218, 123, 63, 232, 214, 16, 147, 41,
        ],
        [
            173, 231, 224, 166, 86, 142, 54, 70, 179, 225, 252, 2, 244, 75, 83, 118, 10, 20, 156,
            247, 202, 76, 237, 164, 61, 66, 13, 122, 141, 245, 173, 44,
        ],
        [
            108, 239, 77, 249, 18, 117, 63, 85, 58, 86, 175, 32, 30, 153, 255, 100, 84, 29, 190,
            31, 229, 96, 136, 89, 133, 204, 51, 185, 137, 216, 164, 20,
        ],
        [
            59, 220, 40, 70, 32, 241, 241, 203, 179, 230, 248, 247, 43, 169, 181, 150, 156, 133,
            147, 122, 253, 248, 32, 86, 24, 136, 5, 111, 169, 164, 215, 56,
        ],
        [
            98, 209, 76, 142, 205, 4, 81, 26, 135, 170, 10, 201, 252, 197, 53, 94, 254, 105, 146,
            81, 160, 43, 12, 97, 126, 94, 64, 186, 207, 201, 68, 56,
        ],
        [
            201, 190, 210, 46, 86, 174, 226, 108, 128, 111, 77, 178, 211, 163, 110, 111, 62, 61,
            153, 63, 216, 125, 9, 48, 127, 164, 43, 106, 64, 147, 129, 57,
        ],
    ],
    [
        [
            255, 240, 106, 196, 39, 210, 196, 148, 144, 253, 91, 141, 51, 240, 165, 236, 108, 169,
            69, 223, 235, 23, 160, 124, 2, 48, 44, 255, 167, 134, 112, 57,
        ],
        [
            124, 68, 80, 75, 141, 236, 245, 254, 13, 176, 89, 204, 71, 154, 156, 56, 117, 53, 210,
            213, 47, 41, 149, 84, 28, 125, 7, 64, 72, 79, 177, 38,
        ],
        [
            61, 71, 251, 185, 213, 100, 253, 192, 180, 4, 68, 195, 221, 45, 113, 19, 246, 221, 88,
            76, 165, 136, 14, 114, 197, 112, 100, 42, 89, 152, 91, 5,
        ],
        [
            42, 226, 253, 216, 123, 40, 107, 109, 166, 66, 10, 217, 98, 141, 231, 69, 243, 226, 46,
            114, 123, 117, 180, 33, 18, 136, 116, 160, 222, 248, 162, 11,
        ],
        [
            34, 220, 174, 123, 63, 171, 95, 68, 127, 18, 102, 15, 252, 35, 255, 63, 161, 228, 65,
            19, 218, 181, 243, 128, 113, 164, 244, 94, 67, 79, 147, 56,
        ],
        [
            209, 56, 211, 64, 210, 9, 133, 186, 36, 212, 196, 71, 50, 82, 23, 95, 76, 75, 155, 87,
            37, 250, 226, 170, 148, 75, 75, 60, 138, 186, 40, 37,
        ],
        [
            220, 163, 215, 26, 244, 235, 220, 165, 176, 44, 86, 158, 231, 85, 52, 25, 149, 119,
            195, 200, 49, 40, 140, 123, 192, 128, 4, 162, 187, 171, 90, 9,
        ],
        [
            225, 7, 90, 21, 111, 77, 213, 55, 198, 207, 93, 231, 90, 26, 147, 18, 234, 190, 254,
            163, 41, 26, 104, 7, 234, 53, 80, 54, 100, 142, 14, 48,
        ],
    ],
    [
        [
            103, 49, 41, 63, 34, 209, 87, 117, 41, 188, 144, 57, 24, 205, 210, 209, 219, 178, 104,
            6, 21, 18, 145, 241, 202, 73, 252, 3, 136, 255, 16, 36,
        ],
        [
            126, 141, 157, 92, 89, 211, 65, 46, 189, 255, 55, 182, 185, 222, 254, 89, 158, 15, 249,
            216, 37, 35, 39, 19, 124, 89, 81, 246, 62, 174, 73, 47,
        ],
        [
            224, 229, 224, 38, 180, 80, 136, 160, 85, 100, 115, 226, 227, 6, 130, 184, 52, 91, 188,
            70, 78, 48, 79, 177, 179, 45, 224, 193, 145, 116, 220, 6,
        ],
        [
            240, 89, 82, 251, 24, 123, 117, 60, 200, 187, 3, 66, 226, 200, 129, 133, 208, 196, 24,
            206, 61, 85, 56, 87, 169, 251, 87, 152, 109, 102, 86, 10,
        ],
        [
            219, 120, 137, 97, 77, 109, 105, 128, 53, 170, 171, 221, 142, 205, 205, 243, 140, 73,
            150, 192, 58, 248, 218, 204, 10, 22, 80, 10, 145, 140, 23, 7,
        ],
        [
            97, 62, 153, 212, 157, 7, 197, 172, 187, 181, 216, 197, 154, 222, 232, 25, 162, 34,
            115, 231, 82, 58, 59, 23, 244, 49, 112, 205, 47, 222, 162, 59,
        ],
        [
            67, 55, 27, 153, 104, 30, 226, 63, 37, 218, 100, 170, 15, 229, 63, 147, 159, 80, 136,
            152, 199, 99, 121, 227, 100, 146, 169, 44, 76, 162, 53, 15,
        ],
        [
            44, 140, 65, 137, 229, 185, 246, 138, 6, 180, 68, 32, 124, 9, 94, 142, 211, 153, 126,
            82, 82, 90, 75, 97, 238, 201, 97, 201, 147, 106, 102, 45,
        ],
    ],
    [
        [
            154, 78, 13, 175, 144, 144, 181, 254, 86, 176, 140, 173, 101, 106, 138, 95, 233, 97,
            107, 87, 110, 243, 211, 168, 113, 244, 0, 143, 106, 163, 103, 49,
        ],
        [
            113, 86, 6, 11, 223, 129, 113, 20, 87, 6, 149, 189, 142, 165, 81, 44, 77, 211, 189,
            211, 155, 63, 227, 198, 235, 245, 238, 220, 145, 222, 131, 0,
        ],
        [
            239, 34, 158, 133, 28, 172, 184, 246, 211, 95, 144, 164, 222, 37, 57, 217, 243, 104,
            162, 216, 71, 180, 230, 178, 65, 102, 237, 102, 33, 27, 64, 29,
        ],
        [
            189, 2, 47, 8, 126, 160, 149, 81, 2, 192, 56, 35, 16, 216, 1, 185, 78, 101, 205, 142,
            248, 26, 138, 78, 19, 227, 132, 76, 211, 101, 143, 36,
        ],
        [
            9, 244, 15, 160, 155, 180, 106, 160, 138, 78, 142, 12, 224, 162, 60, 80, 62, 44, 201,
            200, 85, 40, 69, 242, 28, 63, 119, 55, 55, 1, 187, 48,
        ],
        [
            230, 52, 100, 80, 122, 148, 40, 24, 136, 15, 108, 33, 241, 1, 150, 97, 30, 12, 157, 14,
            247, 86, 117, 42, 218, 238, 93, 130, 148, 50, 76, 9,
        ],
        [
            204, 114, 216, 224, 115, 175, 115, 198, 33, 211, 13, 20, 101, 254, 128, 182, 219, 99,
            49, 136, 34, 104, 131, 176, 57, 127, 217, 238, 87, 74, 149, 41,
        ],
        [
            252, 28, 170, 15, 158, 141, 226, 80, 34, 113, 118, 234, 44, 161, 4, 180, 81, 42, 246,
            29, 137, 65, 208, 21, 184, 245, 74, 248, 211, 96, 223, 42,
        ],
    ],
    [
        [
            99, 73, 57, 46, 43, 160, 177, 18, 52, 120, 173, 225, 193, 243, 102, 8, 209, 4, 27, 211,
            18, 8, 26, 208, 100, 9, 229, 162, 211, 226, 100, 8,
        ],
        [
            89, 148, 122, 14, 37, 150, 254, 164, 84, 248, 1, 148, 99, 59, 136, 157, 50, 211, 11,
            131, 52, 223, 87, 90, 47, 199, 113, 72, 114, 117, 27, 16,
        ],
        [
            54, 224, 16, 104, 41, 121, 198, 104, 97, 16, 200, 211, 177, 6, 144, 107, 166, 233, 125,
            93, 81, 144, 234, 166, 100, 27, 56, 198, 215, 110, 191, 8,
        ],
        [
            120, 49, 186, 107, 174, 192, 21, 80, 127, 55, 201, 252, 245, 21, 234, 199, 197, 0, 127,
            79, 165, 21, 163, 227, 148, 6, 201, 182, 212, 74, 85, 60,
        ],
        [
            32, 206, 13, 185, 245, 1, 247, 16, 12, 178, 178, 51, 66, 213, 224, 253, 141, 191, 123,
            215, 206, 96, 242, 167, 21, 252, 216, 198, 200, 113, 138, 23,
        ],
        [
            214, 146, 47, 116, 13, 140, 129, 199, 78, 218, 105, 47, 184, 183, 37, 176, 138, 116,
            29, 152, 211, 234, 236, 128, 221, 19, 16, 180, 5, 31, 38, 52,
        ],
        [
            53, 22, 251, 54, 25, 10, 180, 124, 152, 90, 251, 137, 121, 218, 152, 77, 36, 39, 110,
            131, 149, 164, 253, 218, 195, 159, 8, 17, 163, 180, 48, 54,
        ],
        [
            79, 222, 175, 84, 68, 207, 171, 218, 9, 173, 100, 196, 4, 209, 60, 217, 95, 220, 33,
            210, 226, 131, 0, 236, 216, 61, 217, 44, 220, 178, 178, 22,
        ],
    ],
    [
        [
            107, 105, 93, 217, 54, 82, 75, 109, 105, 29, 83, 80, 248, 146, 161, 232, 150, 85, 45,
            8, 3, 22, 194, 208, 55, 67, 26, 125, 7, 148, 129, 37,
        ],
        [
            29, 95, 48, 152, 141, 85, 76, 104, 69, 146, 226, 5, 55, 108, 221, 235, 242, 85, 175,
            127, 92, 127, 161, 193, 47, 225, 40, 74, 235, 163, 158, 32,
        ],
        [
            227, 244, 182, 232, 84, 114, 73, 222, 186, 128, 35, 45, 188, 135, 174, 135, 186, 197,
            229, 218, 9, 125, 136, 27, 131, 145, 163, 103, 77, 34, 43, 33,
        ],
        [
            195, 68, 118, 100, 210, 211, 111, 3, 155, 3, 156, 224, 132, 248, 95, 134, 129, 212, 28,
            125, 239, 111, 143, 163, 225, 20, 226, 153, 87, 30, 30, 19,
        ],
        [
            142, 241, 239, 103, 84, 171, 22, 132, 95, 60, 94, 44, 167, 21, 215, 45, 59, 120, 47, 0,
            131, 89, 140, 239, 48, 106, 10, 231, 219, 66, 142, 39,
        ],
        [
            198, 220, 99, 166, 20, 56, 174, 133, 222, 155, 196, 81, 30, 232, 17, 138, 240, 13, 195,
            17, 216, 102, 91, 153, 195, 251, 189, 122, 157, 150, 7, 5,
        ],
        [
            13, 239, 171, 80, 251, 163, 115, 201, 8, 163, 26, 149, 243, 103, 167, 155, 194, 39,
            226, 136, 8, 190, 219, 117, 165, 246, 254, 248, 156, 148, 87, 30,
        ],
        [
            104, 157, 26, 155, 195, 128, 35, 82, 221, 218, 54, 250, 194, 68, 185, 187, 136, 145,
            79, 166, 157, 58, 86, 9, 247, 239, 103, 175, 218, 226, 137, 19,
        ],
    ],
    [
        [
            76, 222, 120, 1, 103, 34, 213, 86, 193, 6, 130, 96, 165, 215, 27, 94, 77, 215, 179,
            150, 125, 6, 171, 65, 24, 8, 61, 139, 250, 144, 31, 31,
        ],
        [
            134, 23, 114, 162, 211, 19, 156, 246, 34, 170, 93, 193, 218, 142, 88, 4, 248, 192, 40,
            118, 53, 217, 131, 17, 168, 129, 200, 206, 4, 196, 235, 20,
        ],
        [
            181, 35, 108, 194, 46, 27, 101, 145, 74, 17, 49, 227, 145, 110, 34, 95, 112, 64, 9,
            241, 215, 205, 232, 135, 35, 194, 234, 141, 80, 6, 89, 37,
        ],
        [
            184, 177, 62, 50, 1, 237, 106, 151, 102, 29, 108, 160, 83, 99, 212, 156, 164, 218, 16,
            75, 167, 201, 62, 156, 91, 188, 199, 8, 195, 74, 196, 28,
        ],
        [
            17, 89, 59, 91, 140, 2, 44, 45, 192, 13, 217, 135, 26, 70, 171, 156, 117, 198, 240, 1,
            218, 24, 48, 197, 51, 45, 56, 194, 94, 94, 51, 40,
        ],
        [
            158, 122, 35, 55, 187, 102, 254, 58, 132, 214, 105, 217, 35, 171, 216, 216, 152, 172,
            126, 106, 107, 206, 127, 41, 164, 33, 165, 71, 247, 92, 143, 14,
        ],
        [
            120, 71, 69, 120, 156, 189, 181, 43, 176, 232, 130, 201, 238, 12, 22, 58, 133, 25, 155,
            146, 9, 188, 14, 153, 53, 156, 38, 212, 195, 134, 95, 7,
        ],
        [
            157, 96, 229, 115, 219, 58, 67, 16, 224, 9, 44, 240, 209, 213, 218, 68, 247, 237, 57,
            28, 228, 198, 146, 99, 140, 224, 145, 231, 119, 223, 118, 48,
        ],
    ],
    [
        [
            213, 203, 57, 233, 35, 161, 91, 226, 27, 23, 16, 217, 18, 194, 126, 205, 148, 142, 190,
            53, 58, 59, 253, 86, 70, 219, 212, 125, 153, 224, 6, 9,
        ],
        [
            155, 137, 19, 65, 34, 199, 103, 121, 244, 81, 27, 130, 6, 116, 234, 102, 44, 199, 4,
            19, 80, 173, 42, 110, 70, 143, 130, 125, 159, 106, 236, 39,
        ],
        [
            122, 10, 176, 127, 83, 36, 22, 172, 72, 92, 139, 199, 151, 96, 70, 203, 154, 81, 240,
            22, 29, 6, 119, 248, 46, 45, 132, 169, 166, 30, 90, 22,
        ],
        [
            26, 13, 5, 45, 142, 71, 73, 137, 254, 3, 246, 242, 178, 98, 165, 194, 210, 131, 252,
            15, 162, 70, 72, 143, 67, 161, 142, 16, 152, 79, 80, 61,
        ],
        [
            19, 13, 118, 209, 24, 155, 176, 112, 201, 217, 188, 152, 118, 164, 78, 191, 232, 62,
            216, 249, 142, 131, 126, 180, 90, 1, 150, 49, 78, 96, 75, 31,
        ],
        [
            26, 242, 23, 163, 221, 125, 25, 226, 219, 86, 141, 10, 85, 252, 140, 215, 130, 142,
            159, 229, 169, 103, 118, 242, 183, 177, 223, 236, 90, 61, 233, 55,
        ],
        [
            10, 208, 13, 145, 192, 166, 233, 172, 141, 58, 55, 234, 155, 69, 41, 247, 190, 194, 81,
            157, 179, 146, 197, 146, 251, 187, 218, 11, 60, 144, 237, 58,
        ],
        [
            240, 206, 55, 40, 176, 39, 191, 197, 6, 96, 136, 229, 46, 153, 100, 47, 62, 60, 214,
            146, 206, 84, 114, 69, 26, 45, 157, 102, 40, 72, 91, 28,
        ],
    ],
    [
        [
            43, 161, 254, 90, 188, 123, 114, 200, 17, 242, 129, 50, 94, 237, 177, 114, 185, 147,
            161, 2, 170, 118, 128, 252, 95, 241, 149, 1, 49, 204, 201, 59,
        ],
        [
            122, 217, 240, 150, 125, 123, 87, 55, 111, 22, 243, 27, 215, 190, 60, 91, 116, 68, 96,
            197, 18, 223, 203, 149, 121, 151, 239, 104, 126, 189, 132, 5,
        ],
        [
            121, 57, 216, 35, 130, 22, 227, 222, 233, 39, 46, 77, 2, 167, 64, 44, 59, 50, 64, 48,
            204, 175, 113, 77, 238, 211, 91, 23, 98, 243, 90, 14,
        ],
        [
            90, 181, 29, 181, 36, 252, 218, 201, 12, 206, 43, 100, 88, 235, 1, 117, 73, 51, 208,
            240, 30, 133, 64, 151, 40, 151, 233, 173, 66, 146, 212, 22,
        ],
        [
            85, 121, 202, 247, 237, 164, 240, 148, 100, 11, 187, 140, 202, 183, 225, 244, 244, 207,
            198, 197, 209, 28, 26, 58, 202, 83, 113, 214, 181, 78, 222, 41,
        ],
        [
            180, 231, 142, 160, 125, 98, 227, 7, 143, 3, 69, 171, 93, 224, 114, 156, 125, 17, 181,
            18, 0, 115, 49, 15, 15, 34, 81, 186, 185, 117, 184, 45,
        ],
        [
            66, 196, 173, 255, 17, 239, 59, 139, 157, 5, 120, 107, 8, 206, 0, 133, 65, 23, 102, 73,
            131, 86, 172, 35, 99, 27, 238, 142, 167, 99, 224, 29,
        ],
        [
            87, 234, 21, 65, 77, 92, 71, 146, 253, 59, 106, 150, 122, 185, 113, 28, 4, 114, 34,
            172, 19, 85, 11, 106, 118, 8, 215, 10, 29, 202, 109, 56,
        ],
    ],
    [
        [
            204, 119, 78, 168, 255, 167, 165, 38, 156, 195, 194, 168, 90, 72, 102, 86, 246, 39, 82,
            194, 147, 87, 28, 41, 208, 198, 242, 252, 144, 50, 134, 56,
        ],
        [
            89, 168, 85, 62, 11, 67, 112, 59, 249, 65, 53, 37, 136, 204, 207, 189, 131, 40, 126,
            21, 99, 186, 208, 46, 113, 66, 38, 133, 191, 32, 81, 56,
        ],
        [
            59, 65, 171, 57, 1, 188, 192, 11, 248, 82, 69, 83, 153, 237, 51, 109, 39, 20, 9, 93,
            219, 37, 123, 106, 255, 133, 228, 77, 110, 85, 86, 12,
        ],
        [
            220, 71, 156, 57, 178, 156, 43, 207, 183, 201, 186, 96, 127, 23, 55, 116, 0, 21, 158,
            116, 250, 252, 242, 252, 253, 8, 81, 42, 200, 209, 23, 7,
        ],
        [
            152, 231, 245, 77, 243, 134, 75, 92, 238, 114, 243, 21, 28, 94, 81, 19, 67, 215, 7,
            155, 62, 252, 92, 137, 224, 23, 81, 196, 41, 120, 45, 31,
        ],
        [
            88, 86, 212, 205, 201, 188, 34, 237, 245, 154, 110, 89, 224, 247, 247, 53, 251, 109,
            178, 217, 100, 216, 154, 42, 254, 31, 115, 102, 112, 207, 154, 22,
        ],
        [
            227, 134, 163, 64, 117, 82, 220, 198, 240, 28, 86, 151, 125, 205, 72, 70, 186, 42, 142,
            20, 138, 239, 66, 221, 249, 247, 198, 177, 8, 226, 29, 62,
        ],
        [
            90, 225, 249, 139, 11, 83, 149, 107, 183, 239, 107, 72, 71, 9, 216, 151, 44, 79, 63,
            221, 56, 152, 219, 200, 185, 214, 29, 93, 202, 116, 166, 47,
        ],
    ],
    [
        [
            228, 25, 203, 131, 72, 219, 121, 155, 238, 243, 156, 151, 72, 209, 251, 93, 197, 32,
            228, 115, 26, 13, 254, 51, 249, 232, 59, 57, 46, 81, 82, 55,
        ],
        [
            78, 128, 240, 159, 205, 127, 13, 51, 72, 25, 28, 151, 53, 55, 190, 169, 240, 58, 252,
            198, 109, 187, 49, 106, 176, 72, 218, 91, 146, 127, 109, 52,
        ],
        [
            240, 19, 94, 221, 101, 239, 76, 3, 110, 229, 128, 156, 74, 63, 200, 43, 28, 157, 127,
            164, 254, 141, 137, 22, 137, 103, 214, 120, 105, 215, 42, 31,
        ],
        [
            218, 88, 204, 255, 86, 211, 52, 151, 180, 145, 248, 64, 196, 167, 68, 246, 119, 170,
            90, 21, 120, 194, 93, 3, 12, 145, 33, 24, 91, 50, 116, 19,
        ],
        [
            148, 134, 195, 91, 255, 65, 238, 49, 239, 141, 111, 9, 94, 137, 1, 152, 121, 224, 25,
            170, 66, 243, 161, 10, 215, 177, 161, 60, 36, 9, 75, 20,
        ],
        [
            184, 101, 185, 170, 85, 116, 0, 47, 172, 203, 160, 18, 109, 98, 250, 162, 255, 213,
            245, 164, 53, 87, 62, 30, 115, 76, 152, 245, 10, 217, 214, 43,
        ],
        [
            111, 35, 157, 172, 67, 135, 219, 111, 41, 3, 184, 115, 38, 234, 133, 124, 249, 119,
            160, 155, 221, 133, 28, 122, 176, 170, 19, 151, 104, 79, 85, 51,
        ],
        [
            181, 239, 109, 111, 233, 193, 199, 150, 223, 181, 215, 54, 196, 216, 251, 35, 21, 10,
            6, 44, 120, 98, 170, 250, 231, 90, 99, 189, 50, 189, 19, 60,
        ],
    ],
    [
        [
            40, 12, 194, 46, 247, 222, 58, 252, 87, 215, 47, 154, 251, 107, 52, 13, 128, 24, 213,
            13, 195, 35, 191, 235, 68, 93, 189, 189, 252, 252, 167, 24,
        ],
        [
            38, 105, 73, 132, 137, 11, 105, 85, 16, 244, 223, 124, 93, 158, 118, 81, 177, 159, 1,
            69, 113, 28, 17, 50, 79, 17, 28, 208, 35, 218, 212, 25,
        ],
        [
            76, 198, 33, 207, 191, 213, 53, 3, 200, 102, 78, 223, 93, 40, 61, 123, 106, 236, 185,
            131, 214, 106, 80, 153, 115, 228, 225, 71, 31, 140, 113, 57,
        ],
        [
            214, 79, 107, 70, 115, 107, 122, 106, 0, 239, 195, 79, 62, 111, 169, 40, 204, 73, 217,
            164, 131, 200, 145, 244, 54, 62, 40, 186, 182, 44, 248, 23,
        ],
        [
            242, 254, 205, 32, 90, 19, 103, 203, 170, 64, 162, 67, 182, 13, 75, 230, 28, 194, 34,
            240, 204, 197, 198, 219, 78, 74, 72, 236, 11, 206, 32, 46,
        ],
        [
            152, 23, 75, 211, 211, 169, 146, 12, 223, 100, 17, 184, 84, 89, 76, 194, 212, 15, 78,
            245, 16, 56, 80, 218, 164, 120, 72, 246, 19, 28, 116, 28,
        ],
        [
            74, 5, 117, 170, 126, 141, 251, 3, 12, 184, 128, 229, 31, 147, 243, 221, 79, 250, 241,
            162, 81, 183, 76, 169, 164, 167, 195, 37, 111, 145, 214, 51,
        ],
        [
            71, 243, 35, 221, 187, 141, 25, 109, 57, 157, 201, 174, 128, 157, 50, 164, 198, 111,
            235, 150, 36, 252, 127, 131, 76, 90, 169, 91, 222, 26, 201, 6,
        ],
    ],
    [
        [
            213, 238, 194, 146, 94, 121, 106, 132, 205, 35, 198, 73, 149, 185, 16, 238, 184, 3, 36,
            162, 134, 91, 66, 150, 229, 202, 155, 71, 36, 244, 31, 18,
        ],
        [
            114, 6, 174, 91, 0, 229, 59, 73, 36, 11, 45, 159, 105, 145, 142, 172, 102, 234, 203,
            14, 22, 138, 48, 23, 194, 172, 140, 16, 65, 157, 119, 42,
        ],
        [
            242, 66, 98, 57, 205, 27, 28, 168, 126, 141, 54, 41, 206, 178, 252, 138, 29, 199, 151,
            192, 196, 176, 69, 22, 45, 185, 84, 64, 200, 215, 82, 19,
        ],
        [
            50, 189, 133, 225, 116, 129, 14, 69, 207, 164, 13, 13, 250, 172, 186, 243, 244, 127,
            185, 242, 1, 232, 229, 7, 170, 2, 10, 118, 44, 62, 29, 63,
        ],
        [
            134, 20, 230, 242, 6, 31, 69, 27, 232, 249, 58, 102, 106, 148, 1, 224, 138, 125, 66,
            223, 207, 45, 175, 235, 24, 152, 126, 136, 97, 120, 156, 26,
        ],
        [
            206, 69, 248, 240, 137, 112, 230, 15, 55, 6, 9, 161, 176, 42, 199, 151, 177, 92, 10,
            76, 38, 22, 158, 101, 58, 246, 68, 112, 208, 242, 223, 31,
        ],
        [
            9, 185, 158, 109, 54, 50, 170, 95, 16, 229, 14, 119, 229, 171, 193, 73, 205, 98, 125,
            87, 205, 43, 194, 214, 47, 121, 248, 196, 175, 28, 71, 18,
        ],
        [
            166, 6, 125, 245, 207, 115, 3, 74, 255, 191, 88, 77, 92, 152, 122, 155, 252, 125, 203,
            199, 219, 49, 179, 95, 85, 60, 174, 52, 41, 46, 122, 57,
        ],
    ],
    [
        [
            215, 197, 242, 170, 148, 210, 105, 84, 28, 246, 154, 48, 114, 11, 47, 104, 198, 29, 79,
            121, 85, 192, 83, 226, 104, 221, 36, 167, 10, 230, 9, 44,
        ],
        [
            38, 101, 71, 27, 235, 109, 39, 104, 162, 219, 2, 102, 130, 75, 149, 224, 208, 106, 36,
            35, 126, 77, 200, 120, 248, 70, 189, 98, 147, 56, 196, 33,
        ],
        [
            43, 254, 197, 70, 182, 96, 195, 113, 109, 4, 37, 174, 86, 22, 45, 155, 149, 166, 47,
            41, 235, 155, 60, 143, 107, 202, 246, 174, 99, 81, 182, 37,
        ],
        [
            204, 133, 71, 6, 137, 11, 247, 251, 200, 146, 71, 8, 134, 144, 30, 115, 180, 230, 96,
            37, 126, 8, 96, 241, 78, 104, 95, 13, 231, 243, 172, 0,
        ],
        [
            19, 155, 215, 193, 160, 46, 131, 2, 59, 139, 238, 194, 57, 178, 12, 140, 3, 243, 64,
            237, 112, 40, 81, 250, 124, 241, 12, 117, 130, 116, 254, 40,
        ],
        [
            137, 165, 36, 240, 128, 172, 202, 88, 237, 127, 204, 45, 69, 177, 242, 25, 100, 146,
            236, 198, 192, 92, 215, 18, 69, 4, 245, 4, 1, 10, 43, 52,
        ],
        [
            3, 158, 53, 179, 240, 14, 146, 201, 182, 180, 67, 159, 181, 33, 165, 229, 217, 247, 33,
            27, 167, 152, 110, 131, 85, 43, 206, 221, 226, 250, 123, 23,
        ],
        [
            229, 92, 186, 133, 10, 181, 79, 50, 154, 37, 248, 83, 217, 228, 33, 159, 103, 149, 143,
            33, 8, 183, 95, 113, 63, 102, 224, 73, 156, 251, 98, 37,
        ],
    ],
    [
        [
            247, 74, 62, 239, 168, 228, 32, 209, 222, 23, 144, 181, 39, 128, 160, 88, 15, 214, 249,
            216, 183, 217, 18, 79, 72, 114, 111, 253, 179, 143, 225, 48,
        ],
        [
            40, 173, 194, 76, 40, 155, 20, 143, 254, 219, 116, 6, 91, 177, 224, 105, 43, 133, 196,
            4, 242, 87, 73, 118, 249, 232, 136, 223, 69, 92, 238, 24,
        ],
        [
            158, 53, 48, 117, 246, 139, 220, 102, 7, 22, 194, 218, 130, 117, 34, 146, 1, 31, 87,
            139, 146, 132, 54, 142, 233, 247, 135, 110, 93, 193, 162, 4,
        ],
        [
            134, 20, 204, 240, 252, 33, 224, 236, 178, 214, 249, 240, 220, 247, 221, 114, 160, 124,
            110, 165, 70, 19, 132, 54, 244, 161, 6, 225, 112, 95, 154, 35,
        ],
        [
            106, 46, 14, 29, 169, 215, 146, 162, 206, 26, 154, 130, 201, 59, 173, 149, 168, 203,
            105, 14, 148, 193, 107, 250, 103, 1, 79, 222, 39, 226, 119, 0,
        ],
        [
            234, 18, 118, 231, 14, 17, 232, 50, 40, 103, 81, 18, 143, 207, 224, 52, 62, 0, 102,
            217, 182, 165, 217, 235, 230, 180, 13, 68, 2, 18, 40, 40,
        ],
        [
            75, 123, 96, 255, 76, 203, 72, 134, 54, 123, 235, 42, 70, 231, 240, 72, 34, 190, 255,
            142, 236, 80, 214, 84, 148, 191, 46, 41, 191, 9, 249, 49,
        ],
        [
            81, 106, 77, 254, 224, 248, 86, 236, 175, 228, 23, 60, 12, 21, 156, 181, 16, 96, 194,
            65, 43, 72, 84, 213, 169, 23, 235, 175, 17, 160, 101, 40,
        ],
    ],
    [
        [
            255, 202, 243, 40, 59, 189, 65, 206, 36, 58, 220, 59, 32, 50, 215, 251, 250, 199, 144,
            167, 60, 147, 124, 145, 211, 191, 40, 253, 45, 57, 65, 62,
        ],
        [
            216, 142, 222, 34, 95, 115, 109, 52, 136, 6, 48, 112, 55, 233, 214, 65, 188, 63, 63,
            188, 30, 125, 219, 228, 96, 39, 2, 232, 216, 236, 36, 24,
        ],
        [
            49, 214, 133, 89, 17, 232, 45, 73, 222, 70, 231, 0, 157, 115, 175, 226, 197, 144, 161,
            60, 109, 17, 96, 18, 208, 193, 222, 87, 232, 106, 203, 63,
        ],
        [
            222, 205, 168, 98, 100, 250, 218, 211, 223, 20, 227, 19, 84, 40, 183, 228, 179, 174,
            193, 9, 150, 36, 54, 235, 138, 48, 91, 217, 90, 2, 68, 12,
        ],
        [
            248, 49, 13, 135, 252, 104, 109, 197, 85, 103, 6, 201, 50, 224, 152, 124, 78, 50, 46,
            240, 133, 124, 241, 227, 25, 35, 192, 2, 228, 37, 30, 58,
        ],
        [
            14, 157, 212, 178, 33, 141, 195, 233, 148, 209, 99, 31, 162, 136, 5, 142, 120, 233,
            220, 15, 79, 175, 177, 241, 146, 49, 102, 211, 252, 130, 121, 9,
        ],
        [
            221, 245, 13, 62, 16, 26, 233, 246, 17, 220, 110, 118, 6, 251, 172, 236, 231, 108, 147,
            73, 160, 123, 205, 72, 54, 107, 216, 172, 212, 227, 197, 56,
        ],
        [
            26, 183, 159, 163, 99, 211, 238, 142, 63, 162, 131, 254, 192, 50, 122, 30, 11, 165, 0,
            34, 137, 201, 18, 44, 35, 67, 75, 29, 91, 3, 64, 24,
        ],
    ],
    [
        [
            217, 170, 198, 82, 164, 156, 48, 78, 243, 8, 162, 5, 161, 214, 26, 30, 105, 121, 221,
            15, 178, 235, 126, 91, 240, 231, 234, 164, 129, 163, 23, 54,
        ],
        [
            254, 124, 232, 46, 154, 234, 41, 132, 147, 167, 82, 156, 255, 152, 165, 188, 194, 112,
            118, 106, 96, 179, 109, 52, 246, 42, 79, 35, 107, 57, 212, 26,
        ],
        [
            38, 115, 40, 37, 250, 17, 254, 56, 6, 112, 253, 115, 15, 36, 210, 211, 238, 33, 204,
            253, 199, 143, 3, 65, 134, 74, 79, 33, 60, 28, 170, 43,
        ],
        [
            19, 156, 2, 238, 173, 43, 201, 213, 118, 138, 29, 250, 53, 73, 7, 189, 20, 61, 113,
            139, 64, 230, 97, 1, 225, 169, 162, 11, 77, 134, 136, 53,
        ],
        [
            78, 97, 56, 58, 166, 149, 113, 8, 98, 0, 137, 188, 8, 124, 162, 118, 177, 116, 251,
            177, 116, 210, 102, 31, 183, 243, 26, 52, 83, 79, 186, 33,
        ],
        [
            51, 200, 124, 61, 128, 141, 234, 223, 108, 47, 120, 168, 202, 74, 72, 146, 10, 40, 157,
            61, 25, 196, 20, 146, 157, 45, 194, 230, 131, 171, 53, 16,
        ],
        [
            181, 85, 36, 16, 18, 181, 21, 254, 141, 102, 213, 159, 132, 1, 57, 130, 9, 143, 121,
            68, 15, 119, 7, 171, 143, 183, 193, 1, 249, 253, 36, 20,
        ],
        [
            107, 77, 204, 52, 72, 81, 22, 230, 177, 54, 238, 49, 183, 222, 112, 78, 164, 255, 134,
            65, 123, 8, 4, 195, 184, 24, 216, 143, 48, 172, 114, 26,
        ],
    ],
    [
        [
            224, 165, 239, 181, 244, 151, 129, 54, 85, 87, 124, 240, 174, 1, 70, 118, 30, 155, 51,
            250, 142, 68, 46, 119, 171, 232, 52, 161, 115, 45, 192, 28,
        ],
        [
            177, 83, 89, 10, 98, 235, 250, 204, 137, 157, 85, 57, 213, 67, 50, 214, 196, 32, 115,
            160, 55, 99, 37, 105, 50, 158, 157, 95, 87, 89, 12, 6,
        ],
        [
            31, 181, 62, 233, 183, 232, 177, 247, 241, 220, 131, 89, 222, 45, 15, 22, 2, 29, 216,
            31, 154, 176, 224, 163, 152, 55, 230, 74, 194, 64, 29, 1,
        ],
        [
            174, 31, 125, 191, 176, 122, 35, 41, 104, 113, 68, 147, 119, 17, 103, 38, 222, 44, 81,
            3, 250, 74, 9, 106, 176, 7, 241, 139, 171, 202, 152, 32,
        ],
        [
            54, 165, 3, 154, 4, 5, 149, 0, 125, 146, 154, 23, 61, 148, 76, 158, 82, 143, 229, 163,
            204, 218, 102, 166, 23, 255, 201, 209, 86, 160, 68, 15,
        ],
        [
            107, 49, 88, 210, 239, 78, 195, 211, 215, 223, 154, 25, 177, 128, 228, 39, 237, 211,
            251, 201, 51, 177, 160, 171, 26, 199, 234, 13, 143, 210, 6, 19,
        ],
        [
            129, 220, 49, 186, 238, 46, 18, 145, 214, 116, 177, 103, 238, 253, 156, 98, 208, 242,
            61, 98, 82, 77, 93, 237, 42, 106, 139, 37, 40, 119, 94, 21,
        ],
        [
            99, 86, 197, 177, 154, 78, 39, 72, 132, 166, 108, 253, 48, 59, 67, 185, 2, 130, 56,
            151, 148, 78, 188, 50, 138, 40, 127, 115, 8, 30, 195, 47,
        ],
    ],
    [
        [
            161, 97, 38, 183, 81, 130, 85, 252, 94, 241, 250, 149, 204, 211, 68, 21, 144, 223, 76,
            73, 67, 83, 162, 151, 197, 108, 157, 32, 148, 143, 62, 26,
        ],
        [
            186, 237, 123, 78, 107, 54, 30, 146, 192, 222, 165, 177, 2, 125, 183, 167, 236, 168,
            96, 121, 6, 147, 225, 117, 86, 8, 15, 115, 149, 199, 1, 31,
        ],
        [
            91, 208, 64, 25, 137, 126, 121, 244, 6, 205, 13, 56, 5, 211, 30, 139, 68, 85, 197, 221,
            100, 14, 18, 32, 193, 3, 151, 227, 148, 4, 192, 0,
        ],
        [
            132, 233, 13, 214, 23, 47, 150, 6, 226, 216, 88, 203, 152, 212, 8, 231, 1, 141, 112,
            240, 220, 127, 199, 201, 254, 112, 119, 135, 253, 128, 237, 24,
        ],
        [
            208, 154, 28, 21, 219, 23, 84, 233, 114, 177, 88, 89, 112, 204, 130, 28, 13, 124, 13,
            71, 67, 197, 83, 80, 157, 25, 105, 158, 2, 150, 133, 47,
        ],
        [
            102, 50, 242, 246, 255, 157, 245, 139, 43, 72, 245, 232, 239, 122, 214, 227, 114, 31,
            59, 175, 126, 249, 229, 161, 47, 124, 232, 45, 116, 187, 11, 6,
        ],
        [
            195, 235, 184, 23, 101, 52, 32, 120, 163, 252, 87, 10, 118, 21, 95, 10, 118, 95, 109,
            88, 251, 173, 11, 86, 187, 194, 230, 32, 85, 54, 187, 6,
        ],
        [
            11, 77, 255, 109, 67, 103, 249, 253, 225, 160, 107, 82, 170, 72, 28, 221, 50, 158, 118,
            22, 15, 51, 157, 150, 64, 174, 9, 18, 110, 94, 134, 63,
        ],
    ],
    [
        [
            22, 249, 212, 177, 47, 64, 198, 103, 181, 243, 4, 164, 46, 243, 160, 120, 81, 116, 50,
            224, 30, 154, 199, 23, 49, 128, 92, 209, 125, 1, 248, 10,
        ],
        [
            202, 149, 25, 118, 221, 245, 101, 65, 28, 151, 69, 128, 19, 207, 9, 86, 54, 248, 254,
            163, 143, 209, 12, 53, 158, 149, 44, 216, 140, 192, 33, 1,
        ],
        [
            160, 220, 56, 72, 111, 111, 70, 75, 244, 213, 20, 118, 166, 117, 67, 124, 122, 251, 51,
            98, 117, 157, 56, 173, 9, 19, 156, 208, 121, 109, 33, 7,
        ],
        [
            88, 192, 255, 45, 177, 228, 254, 60, 125, 5, 56, 103, 211, 6, 132, 123, 20, 169, 138,
            89, 208, 137, 112, 92, 166, 113, 164, 47, 31, 242, 69, 19,
        ],
        [
            71, 100, 47, 249, 20, 104, 251, 81, 0, 130, 176, 233, 221, 69, 63, 21, 95, 229, 105,
            213, 176, 43, 152, 148, 63, 203, 237, 97, 68, 27, 5, 46,
        ],
        [
            213, 68, 150, 155, 57, 103, 131, 237, 51, 253, 41, 186, 132, 242, 66, 88, 43, 204, 111,
            137, 65, 204, 245, 92, 57, 182, 47, 87, 206, 38, 232, 56,
        ],
        [
            5, 222, 1, 109, 48, 37, 104, 111, 196, 51, 129, 84, 228, 187, 45, 176, 46, 194, 26,
            108, 193, 235, 216, 205, 73, 148, 115, 216, 187, 176, 202, 33,
        ],
        [
            52, 125, 84, 43, 125, 255, 223, 79, 212, 184, 34, 97, 130, 37, 146, 155, 142, 123, 197,
            113, 16, 190, 149, 192, 3, 69, 64, 162, 2, 247, 186, 39,
        ],
    ],
    [
        [
            100, 246, 118, 128, 108, 93, 29, 54, 43, 222, 151, 194, 22, 167, 184, 64, 178, 30, 254,
            104, 231, 101, 219, 176, 100, 66, 227, 21, 60, 56, 141, 30,
        ],
        [
            25, 82, 5, 57, 97, 112, 70, 14, 177, 181, 80, 33, 243, 121, 214, 125, 15, 56, 54, 178,
            173, 129, 105, 53, 204, 161, 47, 31, 104, 212, 113, 30,
        ],
        [
            165, 241, 162, 162, 146, 225, 123, 214, 227, 9, 94, 232, 224, 33, 220, 126, 140, 230,
            100, 6, 146, 206, 32, 87, 34, 226, 9, 137, 94, 24, 29, 56,
        ],
        [
            210, 204, 192, 45, 111, 180, 6, 42, 143, 142, 209, 66, 70, 70, 165, 125, 177, 136, 61,
            170, 218, 137, 47, 140, 145, 98, 97, 201, 242, 38, 159, 13,
        ],
        [
            145, 131, 248, 82, 176, 63, 234, 187, 225, 244, 68, 228, 160, 221, 168, 229, 196, 111,
            210, 249, 182, 190, 50, 110, 79, 189, 198, 201, 186, 177, 167, 22,
        ],
        [
            64, 115, 113, 246, 255, 216, 234, 56, 117, 195, 120, 55, 240, 134, 160, 169, 39, 37,
            80, 157, 183, 5, 183, 141, 102, 99, 209, 75, 179, 115, 212, 40,
        ],
        [
            168, 252, 46, 212, 207, 192, 61, 237, 74, 29, 184, 78, 230, 155, 18, 104, 17, 182, 192,
            138, 131, 165, 221, 197, 28, 227, 210, 173, 81, 160, 166, 58,
        ],
        [
            77, 53, 146, 43, 89, 3, 223, 140, 112, 124, 116, 255, 250, 126, 220, 113, 46, 150, 26,
            249, 193, 109, 169, 1, 91, 73, 98, 140, 174, 75, 63, 15,
        ],
    ],
    [
        [
            27, 205, 44, 170, 203, 156, 219, 207, 220, 128, 50, 89, 195, 109, 65, 110, 108, 8, 55,
            253, 180, 73, 204, 109, 234, 238, 167, 255, 123, 234, 84, 24,
        ],
        [
            127, 219, 25, 52, 81, 140, 114, 75, 213, 68, 108, 1, 95, 51, 226, 118, 149, 197, 171,
            220, 251, 33, 88, 169, 50, 122, 38, 100, 3, 102, 47, 31,
        ],
        [
            116, 246, 165, 168, 166, 54, 240, 34, 61, 44, 75, 36, 18, 76, 242, 135, 212, 88, 133,
            169, 34, 137, 102, 214, 6, 14, 5, 61, 212, 36, 247, 51,
        ],
        [
            181, 3, 99, 237, 164, 47, 170, 88, 174, 148, 98, 149, 163, 57, 231, 192, 105, 214, 110,
            164, 11, 208, 98, 230, 184, 137, 159, 127, 173, 163, 196, 36,
        ],
        [
            62, 15, 190, 131, 36, 208, 162, 22, 187, 30, 242, 168, 211, 115, 26, 134, 153, 102, 79,
            19, 117, 123, 31, 37, 200, 237, 190, 49, 222, 82, 194, 6,
        ],
        [
            130, 193, 193, 11, 20, 203, 1, 24, 41, 18, 242, 33, 158, 52, 92, 183, 150, 111, 127,
            153, 106, 13, 55, 96, 121, 87, 76, 174, 91, 50, 8, 21,
        ],
        [
            25, 206, 189, 81, 111, 245, 100, 197, 220, 52, 174, 171, 137, 120, 124, 247, 234, 110,
            216, 252, 100, 114, 82, 52, 23, 134, 212, 218, 74, 119, 94, 20,
        ],
        [
            222, 24, 83, 2, 137, 110, 39, 174, 67, 16, 243, 169, 132, 178, 196, 239, 153, 218, 85,
            165, 204, 160, 193, 202, 190, 116, 134, 156, 227, 79, 126, 31,
        ],
    ],
    [
        [
            78, 127, 250, 132, 73, 120, 250, 80, 235, 105, 226, 127, 217, 64, 142, 70, 102, 179,
            220, 176, 43, 150, 213, 22, 220, 27, 16, 110, 202, 206, 150, 2,
        ],
        [
            59, 135, 36, 89, 92, 38, 243, 89, 157, 94, 193, 187, 36, 198, 31, 210, 144, 172, 169,
            226, 100, 166, 162, 107, 179, 124, 242, 58, 192, 181, 120, 31,
        ],
        [
            244, 226, 171, 215, 32, 82, 150, 4, 29, 112, 228, 54, 44, 49, 219, 85, 165, 229, 235,
            179, 29, 52, 144, 132, 17, 254, 63, 120, 60, 4, 33, 28,
        ],
        [
            94, 250, 234, 126, 51, 236, 214, 132, 207, 148, 71, 2, 99, 254, 12, 161, 167, 46, 79,
            54, 232, 205, 208, 140, 236, 34, 79, 62, 195, 152, 152, 44,
        ],
        [
            236, 54, 144, 91, 214, 215, 143, 43, 147, 135, 206, 43, 142, 33, 216, 123, 160, 20,
            128, 138, 19, 57, 31, 213, 191, 145, 230, 241, 196, 67, 228, 45,
        ],
        [
            209, 246, 124, 2, 205, 158, 164, 155, 232, 52, 224, 121, 49, 167, 125, 186, 64, 190,
            110, 213, 253, 154, 230, 8, 203, 72, 208, 156, 224, 14, 26, 36,
        ],
        [
            2, 163, 112, 13, 243, 117, 175, 20, 37, 74, 70, 184, 159, 43, 64, 58, 196, 60, 58, 97,
            217, 56, 155, 232, 182, 122, 231, 235, 63, 244, 85, 54,
        ],
        [
            198, 207, 127, 56, 2, 173, 97, 18, 140, 235, 98, 17, 74, 58, 229, 135, 107, 233, 107,
            97, 101, 167, 19, 153, 50, 196, 142, 26, 74, 12, 31, 24,
        ],
    ],
    [
        [
            27, 63, 161, 142, 183, 196, 17, 99, 100, 98, 158, 20, 142, 103, 84, 220, 11, 21, 113,
            159, 248, 107, 32, 78, 227, 20, 95, 12, 63, 74, 188, 50,
        ],
        [
            202, 124, 66, 161, 185, 74, 107, 125, 210, 209, 173, 82, 129, 191, 17, 80, 68, 23, 17,
            109, 138, 102, 163, 219, 39, 177, 231, 178, 186, 189, 100, 41,
        ],
        [
            165, 199, 235, 218, 57, 201, 68, 38, 225, 8, 202, 8, 122, 10, 9, 3, 108, 9, 192, 10,
            77, 228, 225, 137, 192, 34, 172, 150, 3, 144, 72, 50,
        ],
        [
            243, 144, 241, 83, 19, 129, 69, 136, 39, 17, 21, 64, 217, 91, 137, 38, 123, 162, 240,
            193, 242, 181, 40, 21, 87, 126, 38, 49, 184, 28, 16, 7,
        ],
        [
            238, 195, 172, 123, 53, 111, 247, 70, 103, 77, 54, 97, 229, 183, 174, 143, 101, 174,
            158, 2, 5, 2, 30, 76, 208, 223, 129, 66, 119, 241, 118, 17,
        ],
        [
            94, 251, 39, 193, 48, 215, 167, 73, 107, 136, 251, 46, 162, 178, 167, 131, 148, 160,
            247, 72, 132, 67, 69, 217, 32, 144, 252, 60, 155, 172, 213, 11,
        ],
        [
            136, 203, 58, 211, 248, 4, 150, 29, 122, 226, 157, 183, 129, 29, 48, 15, 99, 138, 246,
            99, 67, 146, 78, 105, 219, 58, 247, 132, 205, 250, 234, 13,
        ],
        [
            131, 246, 106, 110, 134, 217, 32, 180, 116, 227, 242, 87, 223, 171, 248, 156, 56, 249,
            118, 66, 102, 195, 13, 122, 117, 15, 166, 104, 223, 110, 200, 38,
        ],
    ],
    [
        [
            248, 41, 85, 64, 78, 103, 70, 171, 95, 20, 190, 196, 220, 19, 20, 34, 203, 240, 224,
            22, 20, 168, 71, 249, 105, 188, 34, 115, 181, 131, 207, 26,
        ],
        [
            130, 35, 15, 37, 46, 188, 165, 135, 236, 67, 238, 130, 228, 16, 12, 182, 223, 202, 150,
            47, 130, 124, 225, 5, 183, 157, 197, 97, 133, 28, 77, 6,
        ],
        [
            200, 136, 113, 135, 204, 108, 67, 234, 187, 200, 171, 194, 9, 124, 29, 232, 155, 249,
            23, 190, 69, 94, 168, 236, 104, 122, 134, 228, 92, 98, 135, 15,
        ],
        [
            40, 64, 67, 103, 146, 167, 207, 42, 161, 242, 39, 181, 77, 11, 239, 91, 33, 194, 4,
            143, 210, 181, 34, 134, 240, 253, 73, 170, 44, 240, 204, 60,
        ],
        [
            251, 77, 3, 112, 185, 64, 10, 145, 1, 112, 49, 180, 162, 46, 158, 133, 19, 73, 150,
            189, 121, 207, 13, 244, 198, 103, 144, 22, 66, 97, 93, 20,
        ],
        [
            127, 204, 42, 168, 86, 226, 173, 153, 6, 106, 143, 186, 137, 126, 128, 239, 134, 108,
            77, 33, 189, 136, 242, 212, 142, 45, 206, 210, 159, 136, 78, 12,
        ],
        [
            15, 253, 114, 72, 101, 192, 37, 62, 200, 113, 200, 20, 58, 212, 84, 36, 71, 29, 81,
            225, 177, 24, 48, 98, 16, 4, 235, 72, 37, 79, 154, 60,
        ],
        [
            153, 113, 168, 211, 229, 185, 26, 190, 217, 41, 18, 202, 163, 69, 193, 138, 93, 70, 45,
            180, 29, 242, 126, 146, 176, 107, 25, 21, 253, 61, 191, 36,
        ],
    ],
    [
        [
            246, 253, 20, 245, 213, 217, 176, 5, 165, 120, 172, 185, 67, 63, 124, 64, 27, 206, 210,
            204, 46, 74, 6, 24, 213, 23, 24, 173, 66, 6, 232, 7,
        ],
        [
            186, 109, 215, 209, 176, 2, 5, 129, 174, 195, 12, 193, 252, 117, 125, 60, 202, 235, 90,
            58, 214, 95, 228, 254, 112, 219, 85, 209, 23, 135, 95, 56,
        ],
        [
            148, 69, 223, 63, 194, 200, 74, 105, 138, 211, 153, 65, 156, 196, 218, 23, 227, 33,
            197, 145, 61, 189, 254, 92, 219, 252, 232, 247, 238, 135, 155, 25,
        ],
        [
            16, 74, 65, 243, 138, 145, 137, 152, 228, 22, 218, 130, 235, 43, 33, 254, 132, 238,
            221, 161, 148, 92, 148, 30, 29, 84, 186, 39, 228, 47, 56, 44,
        ],
        [
            114, 111, 3, 54, 230, 233, 73, 242, 188, 183, 225, 49, 201, 217, 39, 206, 215, 138, 72,
            191, 187, 215, 115, 117, 156, 251, 68, 31, 186, 218, 56, 62,
        ],
        [
            174, 161, 85, 223, 12, 89, 158, 140, 24, 219, 99, 174, 136, 215, 56, 209, 179, 109, 87,
            118, 194, 55, 18, 230, 34, 87, 235, 151, 247, 40, 15, 9,
        ],
        [
            188, 199, 8, 222, 231, 93, 94, 85, 25, 248, 96, 68, 134, 135, 102, 132, 189, 31, 19,
            187, 49, 97, 47, 124, 106, 216, 69, 133, 160, 167, 141, 62,
        ],
        [
            158, 55, 224, 15, 244, 146, 13, 184, 185, 172, 203, 161, 20, 202, 230, 1, 78, 165, 72,
            73, 127, 2, 36, 55, 110, 119, 241, 238, 23, 86, 241, 23,
        ],
    ],
    [
        [
            135, 127, 214, 177, 50, 175, 15, 156, 209, 192, 69, 139, 50, 0, 233, 92, 181, 39, 10,
            69, 65, 8, 254, 137, 208, 22, 174, 245, 220, 39, 224, 49,
        ],
        [
            89, 164, 61, 68, 166, 224, 4, 27, 209, 222, 1, 46, 57, 173, 182, 122, 135, 238, 56,
            170, 111, 216, 230, 184, 201, 196, 196, 111, 52, 252, 149, 31,
        ],
        [
            44, 67, 177, 56, 207, 234, 197, 117, 53, 180, 148, 180, 138, 63, 54, 133, 255, 80, 171,
            53, 5, 133, 46, 157, 122, 49, 143, 26, 210, 189, 190, 56,
        ],
        [
            97, 11, 237, 129, 221, 220, 1, 91, 199, 236, 87, 251, 50, 166, 242, 156, 29, 149, 195,
            211, 215, 89, 109, 129, 216, 39, 165, 199, 238, 33, 118, 18,
        ],
        [
            173, 167, 107, 119, 184, 254, 187, 104, 208, 100, 103, 46, 4, 153, 8, 135, 55, 157, 79,
            182, 182, 244, 11, 141, 124, 43, 93, 138, 221, 251, 107, 61,
        ],
        [
            35, 15, 241, 4, 169, 175, 97, 122, 64, 8, 38, 135, 20, 128, 36, 135, 7, 241, 86, 36,
            227, 18, 58, 52, 156, 65, 251, 13, 51, 40, 1, 62,
        ],
        [
            51, 214, 155, 250, 163, 83, 17, 145, 93, 141, 84, 108, 188, 234, 73, 63, 21, 220, 19,
            70, 34, 254, 118, 13, 101, 146, 8, 159, 246, 201, 62, 9,
        ],
        [
            42, 79, 34, 69, 94, 116, 129, 61, 198, 162, 167, 131, 186, 165, 197, 204, 39, 114, 19,
            170, 141, 174, 239, 143, 233, 204, 27, 176, 172, 62, 211, 19,
        ],
    ],
    [
        [
            11, 146, 12, 170, 203, 61, 124, 253, 186, 30, 3, 132, 3, 164, 20, 218, 152, 38, 222,
            27, 241, 100, 118, 24, 108, 196, 164, 220, 168, 167, 65, 43,
        ],
        [
            56, 137, 74, 113, 79, 61, 238, 6, 241, 48, 30, 175, 59, 105, 39, 66, 42, 235, 181, 21,
            93, 90, 98, 121, 104, 211, 8, 30, 95, 51, 143, 17,
        ],
        [
            60, 233, 97, 79, 255, 211, 206, 175, 40, 250, 215, 143, 238, 202, 102, 55, 30, 120, 62,
            152, 92, 135, 110, 174, 8, 230, 16, 108, 211, 6, 76, 48,
        ],
        [
            150, 98, 90, 96, 10, 194, 174, 119, 245, 44, 32, 222, 185, 134, 169, 110, 150, 234,
            207, 3, 247, 241, 123, 96, 227, 44, 194, 88, 206, 67, 79, 60,
        ],
        [
            231, 190, 200, 68, 220, 134, 95, 81, 234, 146, 172, 193, 126, 114, 165, 46, 10, 32,
            188, 216, 92, 197, 110, 121, 224, 230, 85, 185, 195, 221, 182, 43,
        ],
        [
            136, 228, 204, 68, 47, 198, 41, 175, 24, 45, 174, 97, 56, 168, 205, 226, 158, 67, 146,
            78, 214, 241, 137, 115, 247, 95, 3, 43, 191, 50, 72, 8,
        ],
        [
            252, 139, 167, 182, 91, 54, 101, 249, 218, 222, 216, 145, 121, 148, 42, 239, 125, 253,
            192, 157, 169, 4, 134, 248, 242, 176, 20, 70, 253, 215, 235, 10,
        ],
        [
            122, 194, 254, 40, 70, 70, 110, 39, 34, 206, 80, 159, 44, 21, 73, 57, 96, 107, 154, 37,
            162, 44, 188, 179, 64, 145, 122, 9, 170, 208, 180, 42,
        ],
    ],
    [
        [
            140, 93, 7, 187, 156, 138, 71, 223, 25, 250, 16, 248, 240, 31, 202, 232, 177, 222, 30,
            229, 189, 102, 196, 144, 189, 121, 65, 16, 249, 168, 130, 42,
        ],
        [
            111, 180, 120, 71, 245, 37, 82, 193, 217, 93, 132, 203, 59, 47, 30, 247, 231, 188, 194,
            215, 197, 25, 185, 86, 89, 33, 212, 138, 138, 126, 129, 40,
        ],
        [
            156, 59, 188, 165, 192, 116, 8, 175, 249, 205, 27, 121, 79, 17, 148, 80, 75, 228, 1,
            107, 20, 121, 226, 166, 82, 171, 242, 115, 149, 163, 96, 60,
        ],
        [
            45, 235, 1, 13, 44, 236, 199, 11, 58, 20, 32, 81, 144, 62, 102, 97, 31, 14, 155, 161,
            193, 6, 141, 114, 110, 193, 54, 254, 148, 188, 62, 25,
        ],
        [
            121, 92, 222, 72, 218, 89, 183, 106, 4, 64, 223, 224, 171, 98, 19, 105, 146, 15, 179,
            92, 172, 246, 197, 249, 51, 137, 124, 218, 98, 21, 105, 15,
        ],
        [
            171, 221, 122, 139, 130, 21, 237, 201, 5, 138, 198, 110, 44, 60, 79, 49, 185, 87, 190,
            128, 122, 241, 184, 31, 77, 60, 167, 95, 209, 122, 194, 49,
        ],
        [
            3, 199, 82, 229, 54, 214, 97, 104, 6, 223, 90, 80, 207, 165, 154, 190, 228, 253, 248,
            89, 72, 20, 183, 139, 172, 180, 81, 7, 179, 47, 163, 43,
        ],
        [
            23, 206, 138, 108, 6, 214, 33, 144, 242, 253, 75, 234, 133, 244, 122, 168, 41, 181,
            167, 62, 99, 40, 18, 100, 20, 61, 142, 234, 208, 41, 17, 34,
        ],
    ],
];

pub fn generator() -> pallas::Affine {
    pallas::Affine::from_xy(
        pallas::Base::from_repr(GENERATOR.0).unwrap(),
        pallas::Base::from_repr(GENERATOR.1).unwrap(),
    )
    .unwrap()
}

#[cfg(test)]
mod tests {
    use super::super::{NOTE_COMMITMENT_PERSONALIZATION, NUM_WINDOWS};
    use super::*;
    use halo2_gadgets::{
        ecc::chip::constants::{test_lagrange_coeffs, test_zs_and_us},
        primitives::sinsemilla::CommitDomain,
    };

    use group::Curve;
    use pasta_curves::{arithmetic::CurveAffine, pallas};

    #[test]
    fn generator() {
        let domain = CommitDomain::new(NOTE_COMMITMENT_PERSONALIZATION);
        let point = domain.R();
        let coords = point.to_affine().coordinates().unwrap();

        assert_eq!(*coords.x(), pallas::Base::from_repr(GENERATOR.0).unwrap());
        assert_eq!(*coords.y(), pallas::Base::from_repr(GENERATOR.1).unwrap());
    }

    #[test]
    fn lagrange_coeffs() {
        let base = super::generator();
        test_lagrange_coeffs(base, NUM_WINDOWS);
    }

    #[test]
    fn z() {
        let base = super::generator();
        test_zs_and_us(base, &Z, &U, NUM_WINDOWS);
    }
}
