use group::ff::PrimeField;
use pasta_curves::{arithmetic::CurveAffine, pallas};

/// Generator used in SinsemillaCommit randomness for IVK commitment
pub const GENERATOR: ([u8; 32], [u8; 32]) = (
    [
        24, 161, 248, 95, 110, 72, 35, 152, 199, 237, 26, 211, 226, 127, 149, 2, 72, 137, 128, 64,
        10, 41, 52, 22, 78, 19, 112, 80, 205, 44, 162, 37,
    ],
    [
        169, 221, 127, 227, 179, 147, 231, 63, 199, 166, 88, 27, 251, 66, 68, 107, 148, 87, 75, 40,
        196, 144, 200, 194, 235, 250, 162, 102, 153, 210, 207, 41,
    ],
);

/// Full-width z-values for GENERATOR
pub const Z: [u64; super::NUM_WINDOWS] = [
    18172, 17390, 61749, 65182, 33835, 155942, 26189, 52444, 40096, 139582, 99218, 20669, 291337,
    12465, 132211, 75527, 68003, 95835, 237325, 21348, 35494, 215451, 49456, 6332, 99036, 224845,
    25324, 23649, 83567, 20531, 9280, 72505, 136089, 21180, 132741, 32676, 18421, 107173, 45630,
    24851, 53914, 156083, 104170, 103364, 25728, 9482, 140699, 42185, 285585, 342, 78646, 326807,
    68908, 10376, 335378, 138003, 41031, 105432, 37682, 15886, 9325, 42470, 27439, 11884, 13979,
    214340, 53073, 76228, 67906, 44696, 178502, 130216, 4242, 142464, 211101, 13210, 66616, 103624,
    7870, 143575, 13058, 27070, 30734, 41157, 2955,
];

/// Full-width u-values for GENERATOR
pub const U: [[[u8; 32]; super::H]; super::NUM_WINDOWS] = [
    [
        [
            38, 43, 235, 53, 200, 116, 153, 194, 47, 92, 18, 54, 194, 221, 170, 155, 205, 143, 92,
            174, 249, 178, 182, 216, 217, 200, 148, 84, 184, 201, 111, 24,
        ],
        [
            57, 210, 96, 16, 151, 52, 218, 250, 249, 5, 128, 232, 65, 231, 239, 167, 102, 8, 198,
            79, 116, 245, 42, 45, 237, 3, 166, 7, 179, 115, 76, 47,
        ],
        [
            148, 246, 234, 157, 72, 45, 221, 176, 144, 235, 23, 51, 13, 230, 80, 196, 122, 188, 56,
            231, 230, 193, 9, 121, 49, 208, 79, 184, 14, 56, 178, 28,
        ],
        [
            139, 148, 48, 95, 182, 176, 253, 110, 107, 242, 54, 21, 66, 103, 169, 9, 207, 182, 149,
            92, 151, 154, 80, 129, 107, 53, 158, 194, 114, 17, 67, 9,
        ],
        [
            194, 138, 153, 65, 192, 221, 4, 110, 115, 182, 50, 98, 165, 161, 49, 119, 63, 24, 149,
            79, 48, 173, 90, 248, 95, 228, 20, 248, 76, 46, 207, 2,
        ],
        [
            16, 137, 162, 110, 105, 192, 29, 133, 191, 64, 219, 94, 186, 194, 0, 7, 173, 221, 156,
            4, 123, 86, 65, 138, 26, 16, 223, 218, 186, 247, 121, 47,
        ],
        [
            247, 8, 62, 207, 128, 127, 80, 169, 185, 32, 34, 54, 198, 4, 121, 42, 44, 173, 233,
            161, 24, 123, 170, 24, 37, 186, 147, 80, 34, 6, 65, 9,
        ],
        [
            184, 197, 131, 138, 111, 145, 67, 140, 183, 100, 227, 73, 20, 230, 2, 163, 56, 202, 35,
            231, 166, 47, 139, 177, 79, 128, 79, 50, 69, 94, 128, 11,
        ],
    ],
    [
        [
            228, 55, 222, 252, 163, 143, 71, 34, 215, 169, 224, 143, 57, 193, 87, 79, 96, 233, 88,
            62, 140, 245, 144, 229, 235, 154, 159, 134, 176, 74, 10, 13,
        ],
        [
            203, 65, 45, 88, 76, 0, 165, 247, 212, 123, 12, 178, 115, 129, 81, 211, 5, 234, 61,
            193, 134, 73, 14, 55, 219, 239, 40, 178, 81, 127, 137, 47,
        ],
        [
            118, 42, 192, 222, 58, 225, 233, 65, 254, 82, 102, 94, 121, 192, 112, 149, 5, 80, 129,
            239, 214, 4, 25, 194, 238, 75, 208, 226, 21, 125, 139, 21,
        ],
        [
            12, 65, 189, 92, 187, 162, 133, 145, 218, 177, 133, 115, 1, 196, 23, 21, 197, 179, 85,
            84, 108, 33, 144, 51, 214, 196, 130, 16, 7, 67, 46, 10,
        ],
        [
            194, 202, 169, 139, 155, 118, 15, 169, 51, 217, 118, 205, 203, 108, 45, 37, 73, 184,
            203, 217, 143, 127, 54, 171, 255, 88, 9, 116, 206, 71, 209, 47,
        ],
        [
            15, 52, 133, 149, 237, 80, 61, 200, 227, 82, 203, 228, 135, 59, 240, 163, 219, 1, 185,
            17, 13, 248, 85, 164, 159, 169, 73, 215, 117, 212, 30, 40,
        ],
        [
            30, 200, 206, 82, 181, 95, 234, 33, 234, 33, 165, 40, 175, 71, 221, 104, 194, 48, 80,
            67, 53, 13, 117, 131, 33, 22, 81, 190, 171, 25, 115, 24,
        ],
        [
            163, 65, 127, 103, 209, 178, 109, 122, 8, 2, 239, 83, 7, 176, 66, 181, 240, 17, 213,
            28, 162, 225, 177, 182, 94, 255, 64, 60, 88, 209, 168, 15,
        ],
    ],
    [
        [
            201, 133, 116, 99, 229, 230, 38, 216, 86, 54, 178, 27, 247, 159, 97, 189, 201, 167, 43,
            60, 241, 97, 140, 243, 94, 64, 91, 28, 229, 55, 135, 52,
        ],
        [
            113, 170, 86, 193, 75, 222, 86, 137, 7, 231, 101, 244, 158, 115, 245, 209, 214, 66,
            163, 228, 218, 85, 56, 43, 164, 132, 221, 237, 72, 184, 91, 37,
        ],
        [
            131, 115, 214, 52, 233, 217, 201, 116, 101, 213, 199, 176, 74, 97, 217, 152, 232, 87,
            103, 124, 132, 104, 134, 160, 61, 107, 131, 81, 105, 121, 76, 41,
        ],
        [
            172, 44, 224, 71, 175, 155, 74, 150, 250, 191, 110, 29, 54, 146, 12, 247, 71, 51, 30,
            163, 203, 51, 46, 255, 155, 118, 193, 185, 138, 20, 142, 10,
        ],
        [
            228, 98, 86, 115, 115, 98, 150, 181, 175, 218, 75, 241, 35, 254, 191, 123, 237, 163,
            215, 117, 6, 203, 219, 65, 245, 161, 225, 48, 67, 224, 161, 21,
        ],
        [
            239, 23, 50, 140, 192, 70, 215, 235, 223, 250, 6, 174, 207, 249, 166, 70, 92, 226, 138,
            237, 234, 146, 19, 108, 181, 80, 254, 80, 222, 78, 170, 13,
        ],
        [
            105, 88, 220, 64, 178, 53, 69, 119, 203, 32, 253, 120, 201, 121, 5, 44, 163, 120, 62,
            197, 13, 211, 222, 212, 161, 35, 191, 40, 81, 239, 98, 45,
        ],
        [
            225, 167, 50, 48, 66, 64, 142, 173, 238, 171, 131, 2, 178, 131, 193, 75, 44, 33, 113,
            143, 159, 81, 201, 165, 3, 132, 91, 69, 72, 122, 5, 34,
        ],
    ],
    [
        [
            48, 191, 86, 130, 85, 144, 157, 35, 36, 188, 210, 224, 234, 51, 152, 200, 136, 137,
            168, 97, 97, 189, 232, 39, 51, 165, 73, 68, 233, 37, 14, 0,
        ],
        [
            6, 86, 175, 220, 24, 9, 216, 103, 64, 7, 180, 171, 14, 90, 234, 37, 65, 169, 198, 195,
            250, 79, 151, 139, 35, 113, 228, 26, 127, 86, 124, 12,
        ],
        [
            118, 194, 93, 189, 16, 152, 182, 130, 136, 79, 99, 200, 174, 187, 181, 234, 104, 247,
            239, 41, 176, 229, 214, 219, 110, 234, 253, 107, 53, 222, 57, 12,
        ],
        [
            159, 96, 55, 104, 236, 61, 119, 214, 121, 9, 146, 92, 89, 231, 232, 121, 66, 195, 105,
            66, 174, 191, 212, 118, 92, 30, 223, 222, 54, 171, 24, 58,
        ],
        [
            78, 244, 3, 237, 108, 152, 99, 240, 195, 201, 58, 149, 136, 57, 170, 126, 90, 168, 8,
            145, 242, 28, 58, 135, 222, 98, 198, 255, 151, 192, 140, 59,
        ],
        [
            4, 123, 44, 17, 130, 193, 135, 237, 165, 215, 129, 100, 179, 215, 150, 150, 49, 171,
            181, 27, 231, 149, 64, 82, 184, 41, 59, 255, 227, 23, 132, 45,
        ],
        [
            249, 11, 208, 25, 21, 210, 141, 96, 112, 168, 221, 247, 108, 62, 216, 218, 77, 3, 13,
            218, 133, 242, 76, 40, 72, 115, 242, 250, 118, 60, 40, 53,
        ],
        [
            18, 205, 34, 103, 209, 200, 230, 196, 181, 56, 100, 22, 64, 56, 135, 88, 22, 193, 121,
            61, 158, 23, 86, 241, 129, 216, 243, 16, 137, 32, 59, 15,
        ],
    ],
    [
        [
            67, 188, 36, 61, 110, 49, 190, 197, 209, 218, 102, 48, 19, 161, 118, 249, 7, 228, 251,
            198, 69, 36, 45, 30, 122, 113, 228, 247, 161, 90, 215, 62,
        ],
        [
            230, 163, 230, 185, 126, 85, 51, 224, 220, 60, 117, 31, 116, 82, 180, 90, 35, 181, 23,
            207, 130, 176, 146, 246, 149, 100, 35, 41, 211, 116, 76, 21,
        ],
        [
            104, 218, 189, 68, 148, 109, 54, 65, 126, 148, 202, 72, 11, 178, 190, 241, 252, 42, 57,
            119, 160, 71, 9, 113, 164, 74, 107, 108, 230, 197, 56, 34,
        ],
        [
            168, 67, 227, 161, 138, 101, 117, 205, 122, 79, 124, 156, 61, 254, 239, 74, 52, 246,
            136, 119, 100, 235, 111, 76, 249, 228, 33, 181, 76, 48, 244, 8,
        ],
        [
            57, 190, 34, 170, 85, 104, 175, 192, 138, 247, 198, 225, 120, 123, 182, 167, 155, 77,
            185, 94, 189, 249, 189, 192, 116, 210, 42, 8, 39, 107, 228, 40,
        ],
        [
            137, 250, 104, 160, 92, 133, 234, 253, 148, 230, 125, 116, 77, 156, 46, 97, 98, 93, 98,
            162, 16, 246, 14, 156, 232, 142, 228, 184, 47, 18, 109, 51,
        ],
        [
            181, 234, 243, 86, 130, 53, 164, 12, 44, 2, 138, 234, 32, 159, 29, 199, 83, 104, 6,
            251, 157, 245, 119, 243, 192, 124, 83, 141, 84, 7, 253, 37,
        ],
        [
            184, 204, 52, 142, 167, 35, 251, 253, 44, 239, 94, 43, 119, 65, 99, 140, 248, 55, 32,
            41, 123, 37, 159, 252, 228, 94, 125, 9, 94, 123, 196, 11,
        ],
    ],
    [
        [
            220, 27, 17, 56, 42, 160, 65, 249, 171, 103, 2, 144, 119, 111, 173, 243, 169, 246, 74,
            94, 178, 136, 81, 131, 137, 145, 8, 199, 129, 187, 82, 40,
        ],
        [
            15, 14, 95, 84, 183, 63, 5, 26, 28, 13, 152, 174, 134, 53, 110, 63, 236, 51, 115, 96,
            128, 222, 243, 133, 220, 138, 207, 69, 248, 68, 163, 54,
        ],
        [
            157, 113, 146, 113, 88, 59, 70, 92, 105, 214, 135, 187, 108, 247, 196, 101, 243, 124,
            113, 202, 145, 193, 137, 12, 189, 232, 61, 152, 47, 0, 60, 46,
        ],
        [
            121, 124, 78, 224, 66, 132, 151, 208, 89, 48, 210, 229, 250, 246, 187, 115, 132, 246,
            165, 83, 116, 100, 176, 192, 157, 149, 229, 90, 164, 247, 106, 48,
        ],
        [
            179, 178, 102, 86, 109, 190, 43, 234, 151, 18, 121, 247, 58, 220, 212, 196, 22, 102,
            246, 146, 53, 145, 90, 244, 127, 11, 114, 56, 230, 49, 239, 3,
        ],
        [
            151, 130, 174, 163, 229, 52, 14, 12, 85, 143, 129, 148, 137, 192, 85, 164, 198, 87,
            204, 79, 250, 32, 221, 149, 48, 124, 96, 176, 121, 125, 189, 42,
        ],
        [
            153, 91, 143, 221, 234, 183, 165, 196, 154, 195, 52, 111, 21, 112, 103, 209, 234, 104,
            139, 132, 183, 44, 110, 33, 16, 107, 25, 181, 58, 37, 166, 25,
        ],
        [
            47, 67, 89, 104, 59, 192, 183, 21, 49, 222, 239, 60, 234, 11, 121, 71, 117, 4, 106, 20,
            226, 178, 1, 44, 19, 241, 180, 203, 241, 195, 228, 53,
        ],
    ],
    [
        [
            123, 55, 47, 2, 155, 105, 127, 151, 194, 54, 66, 196, 200, 140, 171, 69, 209, 52, 213,
            234, 88, 138, 177, 18, 190, 175, 136, 118, 192, 121, 3, 60,
        ],
        [
            40, 147, 60, 109, 168, 148, 50, 213, 164, 67, 54, 126, 99, 78, 221, 27, 65, 208, 53,
            147, 192, 130, 242, 63, 97, 75, 31, 131, 139, 114, 19, 54,
        ],
        [
            18, 174, 193, 9, 97, 164, 97, 44, 90, 32, 54, 8, 223, 18, 97, 244, 115, 94, 114, 178,
            66, 198, 193, 209, 161, 56, 189, 97, 138, 189, 172, 7,
        ],
        [
            73, 30, 174, 174, 85, 131, 112, 210, 126, 211, 184, 128, 29, 229, 154, 14, 238, 77, 33,
            219, 178, 24, 163, 22, 187, 213, 72, 13, 97, 249, 53, 35,
        ],
        [
            194, 188, 159, 242, 52, 203, 166, 37, 70, 147, 45, 10, 253, 195, 93, 160, 97, 99, 166,
            192, 248, 92, 6, 73, 85, 223, 58, 113, 183, 44, 39, 24,
        ],
        [
            194, 181, 84, 60, 45, 220, 54, 229, 21, 95, 31, 114, 26, 125, 202, 172, 39, 215, 50,
            55, 67, 247, 154, 124, 28, 205, 103, 132, 150, 24, 115, 7,
        ],
        [
            58, 18, 125, 164, 3, 135, 144, 38, 72, 55, 238, 146, 28, 64, 157, 12, 194, 159, 47,
            147, 152, 202, 216, 171, 147, 169, 147, 158, 15, 210, 190, 40,
        ],
        [
            17, 195, 245, 108, 190, 67, 179, 94, 90, 53, 44, 36, 195, 149, 189, 61, 45, 129, 211,
            47, 82, 12, 48, 12, 239, 162, 216, 173, 129, 242, 89, 41,
        ],
    ],
    [
        [
            87, 63, 207, 4, 57, 122, 127, 205, 17, 114, 67, 49, 57, 125, 176, 23, 124, 186, 31, 23,
            240, 37, 248, 255, 196, 215, 78, 156, 76, 11, 208, 11,
        ],
        [
            198, 119, 167, 135, 159, 121, 168, 1, 179, 169, 15, 36, 219, 227, 120, 22, 57, 156,
            149, 193, 81, 160, 60, 172, 173, 107, 136, 2, 207, 75, 109, 0,
        ],
        [
            147, 190, 194, 154, 209, 165, 211, 180, 54, 99, 243, 144, 148, 239, 140, 181, 45, 138,
            133, 128, 39, 218, 159, 246, 111, 140, 48, 146, 24, 179, 55, 30,
        ],
        [
            163, 218, 69, 41, 175, 237, 150, 201, 166, 234, 156, 55, 5, 169, 73, 51, 248, 20, 47,
            40, 18, 54, 245, 235, 23, 99, 169, 52, 95, 112, 90, 41,
        ],
        [
            111, 154, 122, 13, 187, 181, 205, 140, 152, 97, 152, 244, 166, 102, 113, 80, 215, 209,
            114, 78, 159, 55, 132, 147, 138, 70, 3, 99, 86, 96, 240, 43,
        ],
        [
            244, 247, 239, 143, 87, 37, 51, 146, 36, 1, 168, 235, 223, 145, 44, 138, 215, 83, 247,
            146, 205, 128, 244, 70, 39, 193, 87, 202, 180, 14, 22, 48,
        ],
        [
            214, 99, 131, 88, 28, 66, 84, 178, 111, 245, 18, 26, 116, 39, 160, 226, 138, 120, 140,
            162, 102, 240, 180, 96, 55, 0, 160, 136, 163, 92, 123, 12,
        ],
        [
            180, 185, 68, 236, 220, 8, 73, 25, 32, 188, 243, 53, 80, 7, 122, 72, 114, 205, 148,
            232, 148, 125, 144, 253, 16, 191, 120, 5, 110, 160, 179, 4,
        ],
    ],
    [
        [
            215, 36, 133, 246, 146, 247, 247, 148, 226, 94, 231, 75, 215, 107, 94, 114, 33, 123,
            159, 150, 147, 196, 251, 228, 202, 91, 248, 52, 181, 225, 3, 12,
        ],
        [
            151, 217, 103, 127, 100, 72, 73, 217, 249, 232, 233, 179, 164, 101, 133, 105, 70, 186,
            220, 239, 179, 151, 224, 24, 166, 8, 77, 37, 104, 146, 35, 8,
        ],
        [
            236, 156, 172, 137, 14, 45, 196, 85, 119, 4, 3, 132, 17, 254, 213, 253, 82, 236, 229,
            61, 129, 51, 56, 44, 145, 202, 139, 85, 193, 218, 144, 49,
        ],
        [
            84, 3, 20, 83, 134, 163, 190, 186, 93, 102, 59, 195, 164, 237, 231, 68, 181, 194, 48,
            66, 161, 190, 118, 16, 255, 5, 182, 100, 50, 129, 103, 35,
        ],
        [
            174, 23, 251, 144, 221, 20, 104, 213, 183, 219, 143, 177, 241, 88, 90, 160, 82, 237,
            109, 168, 6, 241, 168, 159, 172, 32, 71, 95, 195, 34, 17, 40,
        ],
        [
            201, 152, 156, 13, 83, 24, 2, 213, 237, 49, 196, 157, 27, 13, 179, 251, 252, 54, 214,
            118, 248, 220, 194, 210, 106, 126, 60, 148, 62, 216, 181, 42,
        ],
        [
            13, 123, 103, 146, 116, 115, 226, 70, 243, 30, 134, 205, 127, 136, 233, 193, 46, 136,
            185, 99, 224, 56, 117, 170, 127, 41, 79, 144, 143, 66, 178, 54,
        ],
        [
            106, 68, 95, 199, 8, 128, 1, 227, 77, 41, 100, 89, 12, 106, 31, 249, 50, 31, 179, 199,
            97, 200, 187, 138, 173, 68, 55, 220, 204, 113, 185, 17,
        ],
    ],
    [
        [
            69, 139, 165, 232, 189, 151, 135, 228, 146, 203, 4, 199, 79, 88, 205, 208, 219, 227,
            134, 54, 21, 219, 234, 102, 79, 40, 154, 105, 27, 90, 75, 54,
        ],
        [
            106, 255, 138, 45, 182, 156, 63, 135, 105, 239, 114, 52, 216, 133, 125, 168, 94, 216,
            27, 167, 135, 95, 103, 111, 114, 247, 194, 132, 111, 90, 246, 48,
        ],
        [
            194, 9, 225, 228, 74, 52, 98, 220, 90, 99, 240, 151, 235, 87, 61, 11, 72, 37, 144, 202,
            67, 171, 76, 97, 148, 76, 131, 111, 128, 79, 83, 26,
        ],
        [
            101, 226, 33, 108, 207, 233, 24, 201, 236, 245, 69, 101, 214, 18, 21, 16, 193, 128,
            124, 109, 38, 27, 197, 241, 67, 19, 82, 53, 4, 248, 223, 13,
        ],
        [
            198, 98, 126, 137, 89, 45, 3, 147, 114, 128, 182, 243, 147, 132, 201, 109, 58, 39, 166,
            193, 81, 8, 209, 235, 10, 108, 253, 67, 151, 135, 161, 30,
        ],
        [
            119, 53, 78, 105, 73, 215, 248, 66, 16, 213, 74, 178, 30, 136, 181, 111, 193, 70, 85,
            19, 183, 100, 90, 56, 178, 95, 217, 97, 223, 243, 11, 52,
        ],
        [
            11, 121, 245, 230, 235, 125, 64, 10, 52, 221, 159, 37, 113, 169, 8, 44, 95, 145, 89,
            93, 160, 190, 68, 235, 48, 159, 108, 13, 41, 197, 147, 51,
        ],
        [
            189, 245, 38, 171, 5, 217, 150, 237, 129, 197, 196, 176, 83, 68, 183, 90, 243, 92, 121,
            113, 83, 130, 251, 91, 115, 197, 4, 59, 35, 255, 56, 39,
        ],
    ],
    [
        [
            250, 115, 123, 108, 134, 106, 234, 116, 143, 122, 119, 128, 48, 173, 72, 120, 37, 103,
            242, 160, 140, 207, 135, 191, 7, 13, 97, 18, 161, 131, 6, 2,
        ],
        [
            236, 123, 232, 92, 251, 155, 53, 38, 1, 189, 120, 44, 155, 102, 233, 118, 75, 140, 244,
            133, 61, 215, 248, 63, 84, 66, 43, 167, 68, 91, 80, 20,
        ],
        [
            103, 218, 138, 213, 206, 237, 104, 38, 243, 214, 154, 113, 158, 50, 252, 11, 142, 32,
            87, 99, 211, 75, 60, 169, 53, 209, 7, 76, 78, 225, 215, 46,
        ],
        [
            194, 226, 180, 141, 202, 156, 21, 38, 187, 135, 186, 116, 99, 202, 19, 181, 81, 234, 7,
            108, 23, 13, 38, 68, 196, 172, 225, 178, 152, 58, 83, 7,
        ],
        [
            160, 134, 246, 154, 234, 171, 190, 13, 94, 84, 40, 86, 32, 145, 212, 199, 66, 176, 119,
            189, 174, 251, 168, 28, 205, 119, 7, 250, 253, 231, 138, 39,
        ],
        [
            134, 77, 107, 31, 55, 174, 187, 91, 199, 170, 82, 87, 159, 255, 112, 212, 32, 53, 61,
            0, 17, 237, 216, 205, 247, 143, 120, 131, 2, 204, 154, 60,
        ],
        [
            201, 181, 59, 200, 44, 104, 196, 246, 253, 204, 123, 20, 169, 127, 230, 157, 145, 197,
            218, 90, 99, 224, 11, 221, 2, 72, 99, 53, 193, 146, 156, 47,
        ],
        [
            41, 245, 146, 251, 215, 173, 135, 54, 143, 105, 8, 15, 38, 154, 64, 148, 54, 118, 107,
            148, 197, 91, 216, 16, 224, 57, 25, 131, 187, 219, 186, 27,
        ],
    ],
    [
        [
            221, 254, 116, 99, 181, 55, 18, 178, 66, 57, 174, 12, 24, 26, 136, 35, 149, 84, 253,
            62, 207, 82, 150, 156, 131, 17, 13, 156, 174, 72, 179, 29,
        ],
        [
            210, 46, 48, 176, 52, 56, 63, 149, 44, 238, 40, 236, 155, 80, 243, 29, 111, 36, 240,
            29, 67, 187, 45, 146, 116, 237, 205, 52, 176, 254, 75, 17,
        ],
        [
            35, 250, 61, 135, 124, 240, 188, 158, 248, 141, 147, 223, 240, 184, 92, 228, 49, 49,
            72, 46, 108, 235, 136, 230, 93, 215, 199, 242, 178, 43, 47, 7,
        ],
        [
            27, 112, 31, 19, 47, 93, 177, 182, 109, 126, 14, 233, 59, 218, 60, 87, 133, 223, 62,
            136, 167, 36, 184, 169, 221, 172, 74, 199, 168, 126, 247, 45,
        ],
        [
            80, 135, 205, 38, 126, 77, 134, 91, 95, 2, 64, 226, 227, 181, 167, 114, 56, 63, 119,
            60, 209, 210, 203, 55, 79, 59, 78, 6, 222, 71, 207, 7,
        ],
        [
            113, 129, 202, 15, 125, 102, 224, 36, 204, 5, 170, 228, 146, 244, 105, 149, 99, 250,
            57, 123, 198, 56, 145, 25, 6, 30, 112, 221, 201, 228, 136, 37,
        ],
        [
            225, 128, 150, 104, 49, 57, 121, 187, 66, 93, 43, 114, 206, 64, 81, 11, 22, 40, 30,
            152, 2, 85, 134, 241, 144, 41, 217, 215, 240, 196, 178, 28,
        ],
        [
            46, 50, 121, 120, 185, 246, 24, 62, 72, 218, 18, 73, 186, 112, 101, 125, 62, 187, 9,
            87, 184, 138, 229, 211, 213, 196, 116, 71, 119, 146, 214, 30,
        ],
    ],
    [
        [
            213, 25, 243, 89, 153, 12, 106, 58, 2, 63, 93, 221, 49, 252, 84, 150, 204, 15, 253,
            151, 32, 140, 131, 150, 37, 245, 125, 223, 136, 115, 26, 17,
        ],
        [
            167, 41, 163, 182, 251, 78, 242, 173, 178, 104, 236, 137, 143, 17, 73, 129, 241, 171,
            222, 153, 144, 64, 170, 21, 95, 62, 172, 107, 30, 237, 51, 12,
        ],
        [
            206, 103, 30, 156, 70, 3, 208, 135, 21, 109, 32, 63, 76, 96, 112, 150, 219, 199, 141,
            194, 110, 229, 101, 4, 11, 185, 50, 63, 80, 240, 133, 44,
        ],
        [
            174, 29, 247, 44, 246, 20, 223, 133, 6, 122, 102, 210, 91, 66, 105, 178, 55, 227, 112,
            89, 29, 77, 237, 38, 219, 12, 127, 136, 117, 27, 195, 49,
        ],
        [
            74, 141, 53, 90, 118, 93, 102, 49, 234, 115, 93, 27, 174, 29, 70, 255, 188, 50, 130,
            41, 76, 244, 205, 198, 192, 226, 205, 227, 48, 174, 136, 62,
        ],
        [
            129, 45, 8, 72, 246, 235, 199, 178, 112, 64, 206, 81, 216, 82, 19, 228, 89, 200, 21,
            102, 175, 15, 239, 69, 246, 47, 112, 164, 203, 196, 205, 60,
        ],
        [
            26, 222, 152, 124, 45, 123, 245, 33, 71, 30, 177, 117, 217, 100, 245, 240, 159, 9, 153,
            130, 155, 25, 51, 236, 251, 29, 196, 228, 149, 241, 148, 47,
        ],
        [
            206, 140, 210, 217, 54, 219, 195, 231, 81, 215, 183, 190, 86, 184, 157, 184, 46, 131,
            138, 159, 41, 166, 150, 180, 101, 138, 137, 131, 117, 112, 78, 28,
        ],
    ],
    [
        [
            32, 245, 224, 194, 45, 105, 155, 242, 83, 25, 197, 160, 98, 50, 249, 46, 199, 214, 54,
            83, 43, 191, 245, 97, 162, 116, 22, 166, 95, 158, 36, 3,
        ],
        [
            92, 39, 151, 43, 3, 140, 150, 234, 15, 221, 112, 255, 2, 238, 89, 228, 133, 243, 245,
            71, 224, 137, 8, 137, 190, 6, 26, 149, 173, 204, 57, 1,
        ],
        [
            64, 18, 216, 246, 50, 188, 216, 96, 35, 148, 164, 152, 224, 77, 82, 121, 239, 17, 89,
            103, 117, 43, 110, 154, 115, 83, 163, 27, 129, 99, 231, 13,
        ],
        [
            102, 185, 48, 229, 26, 28, 30, 88, 191, 9, 87, 48, 209, 98, 186, 2, 241, 1, 154, 148,
            50, 219, 151, 236, 144, 38, 10, 96, 168, 89, 234, 6,
        ],
        [
            219, 25, 241, 139, 137, 23, 114, 87, 186, 197, 152, 231, 115, 5, 180, 151, 13, 42, 143,
            196, 195, 63, 191, 225, 55, 185, 80, 252, 168, 220, 249, 43,
        ],
        [
            56, 39, 181, 129, 25, 70, 151, 78, 137, 135, 43, 71, 194, 175, 188, 81, 187, 23, 190,
            30, 99, 212, 106, 103, 189, 224, 21, 62, 46, 202, 112, 17,
        ],
        [
            66, 143, 116, 167, 150, 120, 18, 169, 89, 188, 184, 176, 213, 238, 77, 55, 5, 116, 177,
            74, 74, 103, 149, 26, 85, 123, 133, 205, 163, 105, 200, 13,
        ],
        [
            152, 99, 40, 98, 102, 86, 25, 59, 40, 40, 161, 67, 165, 68, 132, 79, 23, 246, 234, 78,
            171, 191, 53, 141, 162, 249, 128, 33, 235, 138, 202, 45,
        ],
    ],
    [
        [
            156, 90, 57, 85, 160, 134, 120, 45, 141, 95, 233, 198, 87, 48, 28, 164, 86, 0, 216,
            168, 23, 88, 84, 27, 191, 41, 191, 214, 134, 146, 125, 16,
        ],
        [
            202, 213, 237, 17, 167, 12, 116, 168, 19, 16, 201, 1, 174, 151, 151, 109, 96, 38, 70,
            163, 159, 105, 232, 241, 143, 122, 123, 74, 176, 146, 46, 53,
        ],
        [
            86, 35, 102, 245, 218, 90, 113, 176, 240, 10, 35, 199, 207, 154, 109, 97, 55, 206, 79,
            25, 161, 155, 108, 28, 71, 143, 98, 25, 41, 208, 15, 38,
        ],
        [
            21, 105, 152, 96, 135, 12, 77, 15, 38, 243, 33, 154, 192, 46, 89, 206, 173, 203, 165,
            27, 53, 154, 68, 226, 247, 152, 10, 215, 107, 10, 82, 20,
        ],
        [
            169, 199, 174, 125, 91, 102, 240, 104, 65, 246, 97, 246, 21, 94, 61, 40, 70, 166, 200,
            43, 59, 181, 54, 81, 65, 252, 130, 95, 103, 237, 172, 26,
        ],
        [
            82, 236, 145, 30, 120, 228, 221, 15, 244, 173, 130, 104, 246, 245, 234, 230, 19, 52,
            247, 195, 227, 206, 247, 161, 160, 242, 57, 139, 131, 232, 217, 19,
        ],
        [
            66, 96, 175, 203, 229, 111, 75, 250, 199, 105, 135, 242, 35, 238, 196, 110, 62, 122,
            118, 141, 104, 14, 152, 242, 122, 177, 206, 36, 47, 96, 80, 35,
        ],
        [
            44, 44, 255, 47, 209, 96, 218, 60, 192, 105, 16, 204, 135, 134, 96, 245, 21, 175, 105,
            29, 246, 177, 111, 60, 156, 25, 120, 138, 101, 6, 90, 2,
        ],
    ],
    [
        [
            31, 184, 49, 71, 143, 43, 71, 222, 169, 53, 216, 191, 22, 249, 138, 207, 21, 24, 3, 86,
            46, 69, 90, 63, 151, 252, 63, 56, 97, 158, 177, 47,
        ],
        [
            80, 11, 236, 123, 242, 1, 232, 3, 112, 167, 77, 43, 73, 167, 74, 53, 127, 83, 151, 237,
            182, 3, 198, 28, 92, 233, 64, 159, 129, 241, 31, 35,
        ],
        [
            164, 68, 201, 171, 117, 151, 76, 22, 43, 3, 101, 209, 79, 199, 10, 242, 68, 41, 82,
            158, 19, 203, 6, 85, 240, 109, 130, 203, 49, 23, 236, 19,
        ],
        [
            200, 91, 200, 245, 221, 228, 3, 161, 167, 14, 150, 242, 143, 230, 90, 251, 97, 215,
            108, 155, 196, 17, 25, 66, 60, 39, 65, 19, 221, 235, 105, 21,
        ],
        [
            38, 200, 246, 45, 43, 12, 141, 145, 41, 87, 40, 36, 165, 94, 46, 223, 254, 245, 129,
            61, 36, 195, 1, 60, 183, 142, 10, 11, 175, 127, 244, 50,
        ],
        [
            202, 242, 204, 124, 201, 204, 110, 3, 156, 77, 23, 67, 177, 83, 55, 126, 93, 132, 159,
            253, 43, 207, 216, 254, 65, 208, 218, 161, 36, 155, 215, 41,
        ],
        [
            196, 246, 176, 207, 138, 104, 9, 125, 222, 187, 126, 223, 6, 40, 98, 57, 3, 154, 54,
            192, 46, 76, 238, 121, 186, 37, 110, 93, 80, 189, 123, 46,
        ],
        [
            2, 55, 61, 20, 25, 205, 105, 149, 79, 130, 87, 233, 206, 198, 185, 72, 135, 84, 237,
            91, 194, 22, 122, 251, 239, 110, 119, 81, 185, 19, 172, 26,
        ],
    ],
    [
        [
            137, 32, 193, 249, 241, 195, 226, 227, 9, 156, 214, 152, 183, 175, 233, 7, 144, 225,
            117, 40, 17, 144, 155, 184, 73, 22, 186, 201, 126, 48, 73, 19,
        ],
        [
            4, 229, 38, 227, 150, 237, 160, 16, 19, 8, 50, 139, 36, 140, 14, 168, 176, 170, 247,
            251, 117, 175, 3, 26, 75, 51, 177, 30, 213, 155, 100, 1,
        ],
        [
            38, 62, 6, 144, 171, 163, 7, 83, 176, 32, 82, 125, 170, 197, 243, 129, 78, 13, 230, 36,
            58, 77, 17, 66, 97, 78, 237, 162, 166, 120, 64, 0,
        ],
        [
            173, 217, 143, 101, 223, 2, 93, 90, 207, 15, 89, 140, 23, 62, 71, 51, 73, 66, 1, 243,
            181, 155, 44, 210, 198, 184, 102, 129, 193, 66, 153, 53,
        ],
        [
            46, 132, 212, 140, 34, 60, 168, 43, 71, 11, 78, 130, 37, 88, 126, 124, 34, 140, 178,
            169, 206, 180, 230, 6, 220, 121, 69, 2, 158, 251, 104, 52,
        ],
        [
            25, 88, 235, 52, 117, 217, 117, 152, 20, 28, 113, 243, 252, 122, 60, 178, 249, 235,
            104, 54, 139, 106, 153, 45, 215, 237, 205, 54, 177, 250, 154, 60,
        ],
        [
            144, 20, 151, 189, 75, 212, 154, 40, 119, 94, 131, 185, 43, 52, 37, 243, 134, 85, 176,
            15, 165, 233, 201, 115, 24, 114, 213, 43, 157, 18, 200, 13,
        ],
        [
            62, 211, 60, 152, 90, 25, 180, 150, 167, 108, 41, 45, 59, 205, 58, 4, 58, 110, 221,
            131, 148, 83, 110, 196, 178, 38, 115, 216, 192, 159, 9, 1,
        ],
    ],
    [
        [
            120, 126, 7, 168, 237, 253, 5, 172, 68, 40, 41, 161, 97, 41, 131, 250, 140, 183, 32,
            45, 234, 188, 137, 104, 201, 78, 209, 11, 146, 115, 26, 38,
        ],
        [
            125, 25, 235, 249, 4, 225, 85, 246, 226, 249, 209, 153, 173, 235, 39, 138, 126, 187,
            144, 237, 237, 238, 7, 242, 28, 27, 0, 172, 65, 228, 114, 33,
        ],
        [
            134, 96, 205, 140, 233, 160, 20, 132, 91, 169, 133, 20, 3, 101, 164, 137, 57, 128, 66,
            255, 186, 31, 215, 57, 244, 144, 111, 29, 208, 113, 70, 8,
        ],
        [
            77, 134, 125, 207, 98, 204, 79, 2, 151, 187, 122, 89, 19, 157, 8, 229, 65, 231, 71,
            186, 149, 131, 45, 52, 44, 229, 213, 117, 124, 97, 224, 47,
        ],
        [
            22, 31, 170, 116, 29, 149, 167, 52, 101, 81, 99, 102, 23, 54, 43, 88, 219, 239, 124,
            120, 110, 103, 95, 93, 114, 82, 235, 254, 74, 162, 12, 43,
        ],
        [
            157, 112, 208, 244, 62, 51, 73, 171, 174, 66, 197, 67, 115, 203, 126, 35, 72, 131, 50,
            156, 249, 148, 69, 112, 107, 76, 103, 122, 52, 224, 209, 34,
        ],
        [
            244, 70, 79, 63, 227, 180, 125, 21, 167, 95, 230, 153, 64, 29, 5, 96, 83, 108, 176,
            197, 206, 46, 250, 57, 137, 57, 65, 75, 89, 64, 29, 27,
        ],
        [
            215, 163, 172, 33, 213, 111, 149, 27, 88, 233, 32, 105, 3, 129, 48, 145, 235, 26, 134,
            7, 186, 59, 56, 176, 166, 104, 40, 0, 201, 231, 243, 26,
        ],
    ],
    [
        [
            87, 35, 156, 129, 118, 185, 139, 129, 110, 245, 61, 122, 247, 13, 221, 235, 250, 173,
            136, 136, 56, 228, 183, 140, 161, 37, 46, 237, 158, 248, 157, 13,
        ],
        [
            74, 120, 21, 253, 130, 99, 169, 173, 59, 189, 108, 55, 191, 30, 46, 129, 218, 43, 47,
            42, 77, 169, 140, 23, 93, 188, 9, 170, 61, 248, 34, 57,
        ],
        [
            80, 93, 107, 130, 213, 98, 234, 84, 99, 7, 36, 75, 206, 150, 12, 208, 63, 247, 223, 3,
            162, 231, 236, 66, 187, 224, 84, 96, 153, 103, 18, 50,
        ],
        [
            235, 121, 156, 218, 55, 8, 17, 101, 33, 51, 218, 31, 10, 163, 169, 198, 12, 140, 123,
            213, 24, 143, 38, 73, 123, 129, 151, 98, 62, 250, 17, 26,
        ],
        [
            152, 84, 193, 18, 114, 227, 10, 46, 250, 144, 81, 78, 116, 12, 132, 137, 13, 227, 147,
            39, 169, 173, 126, 199, 152, 63, 187, 69, 207, 151, 152, 23,
        ],
        [
            142, 128, 46, 231, 126, 20, 245, 111, 146, 158, 213, 71, 174, 100, 72, 13, 94, 126,
            127, 238, 83, 30, 6, 250, 78, 248, 90, 42, 65, 81, 114, 29,
        ],
        [
            252, 29, 137, 205, 37, 85, 213, 156, 238, 200, 193, 190, 5, 140, 230, 216, 163, 239,
            18, 210, 130, 93, 88, 105, 99, 242, 155, 92, 108, 125, 154, 11,
        ],
        [
            138, 93, 184, 46, 124, 53, 181, 41, 164, 228, 229, 50, 252, 30, 84, 156, 203, 236, 235,
            67, 175, 177, 58, 126, 90, 46, 76, 174, 16, 102, 35, 50,
        ],
    ],
    [
        [
            66, 109, 120, 10, 148, 162, 208, 189, 108, 229, 165, 252, 148, 105, 186, 23, 216, 168,
            56, 226, 106, 133, 249, 188, 193, 32, 132, 189, 136, 203, 245, 38,
        ],
        [
            212, 196, 252, 200, 36, 140, 206, 76, 62, 118, 3, 112, 247, 254, 149, 13, 67, 213, 158,
            104, 163, 214, 32, 99, 168, 108, 141, 18, 67, 85, 161, 35,
        ],
        [
            117, 70, 191, 243, 235, 232, 147, 89, 59, 150, 37, 79, 22, 49, 139, 155, 203, 203, 79,
            3, 54, 87, 18, 48, 35, 139, 5, 51, 152, 20, 224, 35,
        ],
        [
            146, 35, 71, 13, 6, 47, 60, 35, 176, 230, 253, 114, 246, 137, 83, 82, 255, 112, 139,
            147, 189, 153, 219, 44, 90, 58, 79, 48, 155, 97, 192, 50,
        ],
        [
            176, 96, 137, 224, 105, 17, 148, 136, 14, 4, 146, 64, 224, 173, 196, 12, 117, 17, 203,
            113, 202, 147, 125, 248, 46, 138, 240, 187, 211, 244, 126, 51,
        ],
        [
            10, 238, 39, 125, 20, 139, 248, 221, 187, 58, 244, 18, 33, 146, 43, 25, 68, 128, 171,
            75, 102, 157, 103, 134, 66, 213, 155, 96, 206, 101, 216, 51,
        ],
        [
            235, 115, 86, 91, 137, 109, 145, 181, 76, 249, 206, 228, 197, 247, 27, 239, 54, 57,
            250, 235, 43, 205, 177, 56, 13, 244, 118, 196, 197, 47, 152, 56,
        ],
        [
            239, 14, 150, 59, 172, 245, 69, 211, 26, 66, 243, 223, 239, 135, 123, 167, 246, 75, 72,
            217, 29, 85, 66, 136, 66, 252, 212, 164, 82, 83, 8, 17,
        ],
    ],
    [
        [
            240, 64, 94, 89, 242, 218, 143, 212, 160, 227, 13, 154, 211, 1, 137, 199, 87, 102, 155,
            2, 238, 22, 135, 90, 1, 31, 3, 72, 120, 186, 216, 26,
        ],
        [
            17, 177, 67, 243, 214, 113, 198, 12, 36, 183, 67, 197, 88, 118, 47, 156, 54, 81, 244,
            19, 84, 250, 44, 63, 91, 1, 122, 131, 32, 140, 156, 50,
        ],
        [
            18, 131, 239, 7, 227, 62, 66, 246, 185, 91, 15, 20, 138, 127, 72, 108, 118, 1, 95, 101,
            106, 122, 213, 159, 177, 149, 158, 26, 156, 255, 164, 23,
        ],
        [
            156, 143, 11, 217, 149, 102, 38, 136, 195, 206, 130, 158, 51, 7, 238, 132, 145, 89, 41,
            105, 4, 223, 136, 187, 185, 14, 96, 203, 124, 211, 232, 27,
        ],
        [
            243, 48, 57, 73, 175, 83, 255, 118, 109, 101, 251, 204, 97, 18, 136, 142, 244, 232,
            140, 69, 137, 12, 243, 55, 6, 193, 102, 18, 105, 85, 212, 32,
        ],
        [
            106, 72, 21, 34, 123, 67, 121, 212, 206, 35, 224, 6, 234, 148, 88, 70, 102, 72, 78,
            176, 241, 69, 245, 42, 124, 102, 122, 228, 78, 213, 225, 15,
        ],
        [
            138, 76, 147, 188, 209, 14, 97, 83, 234, 74, 217, 13, 10, 52, 222, 208, 215, 107, 249,
            45, 197, 115, 144, 10, 242, 244, 80, 14, 188, 122, 243, 46,
        ],
        [
            135, 1, 59, 246, 113, 195, 253, 191, 137, 145, 155, 220, 69, 4, 208, 168, 251, 143, 83,
            125, 181, 243, 142, 133, 187, 154, 95, 35, 242, 96, 22, 46,
        ],
    ],
    [
        [
            212, 106, 147, 198, 73, 242, 158, 22, 55, 148, 138, 51, 81, 214, 51, 215, 54, 228, 134,
            212, 164, 23, 101, 39, 93, 111, 222, 219, 34, 238, 210, 63,
        ],
        [
            44, 206, 102, 234, 64, 229, 63, 6, 217, 121, 61, 251, 91, 93, 218, 58, 191, 201, 249,
            245, 197, 66, 190, 84, 163, 131, 52, 51, 114, 184, 27, 15,
        ],
        [
            205, 245, 64, 10, 119, 181, 177, 248, 219, 83, 73, 222, 242, 114, 221, 64, 189, 254,
            61, 12, 69, 85, 230, 143, 26, 57, 38, 144, 125, 98, 225, 38,
        ],
        [
            5, 110, 248, 163, 158, 65, 46, 204, 22, 73, 246, 174, 157, 32, 211, 0, 111, 157, 247,
            206, 108, 48, 18, 203, 229, 181, 78, 228, 19, 89, 228, 59,
        ],
        [
            0, 8, 189, 101, 170, 252, 8, 211, 161, 154, 10, 147, 125, 49, 249, 48, 10, 107, 162,
            132, 35, 169, 223, 151, 106, 194, 200, 251, 251, 243, 74, 43,
        ],
        [
            57, 109, 106, 128, 160, 236, 72, 207, 149, 229, 48, 9, 124, 193, 39, 216, 210, 42, 9,
            11, 140, 49, 187, 136, 128, 240, 253, 205, 220, 185, 90, 48,
        ],
        [
            250, 254, 164, 4, 48, 165, 213, 187, 138, 16, 165, 199, 100, 173, 32, 33, 61, 217, 156,
            149, 30, 82, 185, 0, 28, 228, 249, 137, 247, 85, 65, 58,
        ],
        [
            2, 242, 14, 70, 214, 3, 108, 124, 224, 207, 16, 138, 171, 60, 1, 115, 217, 89, 60, 55,
            211, 209, 142, 93, 114, 250, 87, 5, 60, 224, 180, 60,
        ],
    ],
    [
        [
            144, 93, 21, 236, 191, 116, 211, 41, 163, 54, 250, 228, 3, 160, 70, 250, 130, 42, 231,
            138, 180, 120, 229, 225, 127, 183, 94, 185, 13, 125, 155, 50,
        ],
        [
            174, 14, 254, 1, 121, 129, 94, 156, 14, 188, 103, 7, 151, 42, 251, 49, 236, 235, 33, 2,
            232, 148, 92, 233, 238, 1, 73, 88, 79, 141, 33, 28,
        ],
        [
            20, 177, 228, 62, 142, 155, 143, 176, 245, 117, 129, 68, 2, 249, 175, 67, 80, 238, 156,
            251, 77, 24, 244, 122, 60, 46, 151, 109, 189, 248, 51, 11,
        ],
        [
            215, 91, 221, 163, 212, 172, 203, 236, 2, 3, 122, 182, 171, 138, 216, 28, 45, 39, 185,
            150, 7, 23, 146, 119, 165, 97, 100, 61, 2, 10, 29, 59,
        ],
        [
            161, 113, 88, 99, 211, 225, 154, 79, 233, 24, 96, 184, 88, 15, 238, 149, 42, 252, 65,
            124, 40, 195, 78, 249, 189, 126, 145, 137, 156, 145, 116, 14,
        ],
        [
            178, 124, 248, 37, 22, 236, 110, 226, 71, 15, 113, 237, 84, 182, 95, 165, 164, 66, 48,
            122, 177, 201, 165, 216, 111, 161, 33, 61, 119, 252, 191, 35,
        ],
        [
            197, 120, 10, 208, 220, 138, 250, 233, 8, 158, 226, 151, 22, 31, 183, 174, 135, 125,
            16, 153, 88, 248, 53, 63, 179, 179, 189, 168, 188, 51, 30, 47,
        ],
        [
            112, 82, 66, 68, 97, 15, 114, 12, 48, 43, 167, 112, 15, 4, 2, 42, 184, 147, 23, 151,
            76, 93, 254, 182, 206, 61, 4, 232, 155, 28, 126, 9,
        ],
    ],
    [
        [
            145, 216, 180, 132, 60, 223, 148, 216, 34, 180, 242, 242, 176, 250, 130, 154, 231, 192,
            2, 180, 19, 252, 149, 36, 226, 139, 218, 152, 23, 71, 186, 60,
        ],
        [
            145, 23, 19, 74, 196, 74, 253, 57, 75, 126, 188, 188, 85, 221, 184, 189, 145, 217, 9,
            29, 92, 142, 118, 170, 123, 227, 2, 13, 45, 133, 10, 34,
        ],
        [
            92, 162, 150, 39, 198, 231, 165, 20, 239, 143, 146, 25, 70, 213, 227, 42, 36, 193, 233,
            136, 227, 72, 41, 211, 119, 192, 56, 46, 178, 115, 249, 24,
        ],
        [
            184, 106, 63, 59, 43, 244, 197, 133, 94, 98, 229, 89, 233, 225, 58, 207, 125, 248, 175,
            61, 6, 114, 65, 195, 188, 135, 168, 132, 163, 118, 73, 40,
        ],
        [
            66, 44, 95, 80, 162, 127, 4, 166, 224, 150, 7, 215, 213, 240, 116, 133, 187, 123, 218,
            65, 44, 202, 58, 102, 60, 67, 131, 28, 33, 251, 226, 35,
        ],
        [
            89, 165, 126, 219, 26, 99, 11, 186, 78, 27, 5, 40, 149, 62, 109, 179, 48, 182, 87, 56,
            220, 63, 161, 160, 189, 95, 152, 215, 116, 208, 97, 29,
        ],
        [
            133, 68, 102, 106, 161, 231, 146, 41, 123, 189, 139, 12, 188, 95, 100, 65, 47, 225,
            146, 16, 211, 141, 113, 140, 204, 214, 183, 85, 130, 129, 161, 29,
        ],
        [
            95, 28, 16, 201, 220, 92, 90, 223, 184, 54, 43, 27, 180, 222, 76, 220, 109, 213, 67,
            82, 180, 223, 14, 245, 218, 208, 56, 221, 129, 236, 198, 0,
        ],
    ],
    [
        [
            153, 10, 78, 169, 254, 76, 133, 187, 164, 212, 220, 235, 99, 248, 60, 54, 222, 32, 177,
            2, 73, 110, 14, 138, 64, 89, 32, 200, 132, 63, 10, 50,
        ],
        [
            179, 107, 129, 58, 191, 91, 211, 38, 108, 98, 114, 39, 138, 2, 84, 136, 107, 195, 35,
            63, 195, 194, 129, 119, 78, 121, 24, 254, 124, 168, 253, 7,
        ],
        [
            201, 134, 252, 140, 79, 1, 46, 243, 125, 150, 69, 149, 201, 31, 145, 207, 118, 58, 214,
            92, 165, 7, 121, 55, 206, 133, 90, 169, 62, 238, 88, 7,
        ],
        [
            124, 231, 176, 78, 128, 30, 182, 95, 20, 20, 221, 90, 225, 166, 82, 43, 34, 235, 98,
            104, 118, 65, 207, 222, 48, 245, 244, 162, 126, 27, 250, 40,
        ],
        [
            84, 234, 22, 3, 167, 3, 19, 186, 163, 10, 6, 176, 16, 2, 161, 151, 132, 147, 110, 76,
            120, 111, 18, 171, 200, 18, 253, 159, 59, 248, 0, 19,
        ],
        [
            254, 251, 230, 244, 114, 117, 80, 226, 27, 33, 7, 106, 164, 188, 7, 234, 73, 93, 111,
            95, 173, 159, 172, 176, 78, 88, 34, 64, 114, 144, 186, 63,
        ],
        [
            229, 103, 0, 114, 148, 250, 213, 136, 168, 246, 230, 255, 196, 152, 208, 58, 26, 126,
            104, 80, 173, 54, 87, 171, 231, 128, 129, 86, 200, 28, 6, 50,
        ],
        [
            71, 246, 84, 162, 124, 177, 159, 98, 133, 205, 218, 133, 162, 133, 203, 151, 169, 145,
            33, 18, 215, 185, 35, 218, 20, 37, 27, 12, 221, 78, 110, 28,
        ],
    ],
    [
        [
            7, 43, 88, 155, 62, 53, 76, 25, 49, 176, 58, 33, 114, 15, 198, 248, 238, 191, 77, 235,
            180, 168, 67, 37, 211, 154, 76, 91, 80, 200, 190, 14,
        ],
        [
            187, 200, 252, 223, 36, 168, 140, 211, 167, 54, 64, 182, 227, 140, 67, 105, 66, 243,
            132, 46, 25, 121, 138, 58, 231, 59, 241, 145, 174, 45, 158, 35,
        ],
        [
            89, 201, 179, 68, 45, 205, 62, 252, 44, 171, 52, 118, 98, 252, 186, 237, 78, 237, 215,
            210, 27, 28, 21, 138, 153, 72, 20, 129, 215, 14, 65, 13,
        ],
        [
            207, 114, 96, 244, 251, 228, 50, 157, 218, 5, 140, 218, 244, 124, 241, 35, 105, 25, 78,
            38, 32, 226, 11, 67, 158, 60, 40, 0, 90, 112, 228, 59,
        ],
        [
            44, 23, 234, 166, 86, 53, 31, 5, 209, 239, 37, 13, 176, 86, 138, 220, 117, 172, 6, 218,
            199, 181, 172, 183, 114, 122, 70, 147, 235, 69, 171, 58,
        ],
        [
            251, 128, 29, 190, 110, 201, 163, 80, 112, 7, 226, 118, 202, 186, 171, 62, 201, 161,
            102, 127, 140, 103, 55, 8, 163, 137, 161, 2, 98, 190, 60, 53,
        ],
        [
            3, 77, 209, 74, 105, 88, 148, 168, 132, 184, 149, 244, 245, 31, 174, 162, 175, 102, 37,
            175, 138, 139, 97, 68, 68, 228, 100, 89, 250, 92, 100, 19,
        ],
        [
            176, 186, 169, 162, 147, 182, 155, 40, 7, 23, 44, 21, 227, 13, 45, 120, 242, 1, 27,
            167, 111, 224, 74, 125, 25, 20, 161, 138, 25, 183, 192, 2,
        ],
    ],
    [
        [
            151, 231, 53, 20, 217, 87, 127, 197, 27, 224, 167, 107, 88, 0, 77, 97, 49, 29, 3, 37,
            59, 191, 47, 129, 60, 176, 200, 144, 3, 31, 120, 6,
        ],
        [
            50, 17, 211, 29, 50, 226, 181, 186, 111, 15, 189, 130, 136, 8, 77, 8, 126, 228, 213, 4,
            51, 193, 84, 83, 191, 254, 71, 223, 190, 13, 92, 14,
        ],
        [
            235, 208, 23, 65, 116, 24, 230, 63, 109, 34, 215, 81, 39, 184, 79, 123, 190, 212, 36,
            43, 97, 68, 108, 184, 171, 6, 18, 67, 172, 104, 210, 45,
        ],
        [
            43, 61, 160, 187, 51, 200, 189, 10, 55, 89, 135, 156, 56, 115, 195, 45, 133, 30, 229,
            182, 30, 93, 188, 158, 86, 180, 239, 225, 39, 203, 9, 12,
        ],
        [
            112, 93, 244, 17, 221, 137, 126, 198, 97, 242, 109, 37, 18, 53, 243, 111, 32, 52, 168,
            88, 8, 159, 133, 250, 72, 148, 15, 165, 34, 206, 4, 54,
        ],
        [
            179, 38, 41, 29, 122, 240, 170, 53, 168, 87, 73, 225, 55, 212, 84, 215, 69, 250, 197,
            81, 2, 46, 21, 71, 104, 92, 233, 125, 159, 174, 73, 28,
        ],
        [
            215, 147, 127, 39, 190, 39, 215, 114, 1, 88, 192, 44, 188, 41, 189, 122, 215, 120, 26,
            136, 42, 161, 32, 238, 123, 125, 99, 139, 123, 160, 126, 35,
        ],
        [
            139, 58, 166, 224, 210, 76, 220, 214, 58, 146, 172, 75, 147, 130, 101, 34, 87, 114,
            134, 231, 57, 106, 228, 24, 95, 150, 248, 196, 166, 157, 47, 49,
        ],
    ],
    [
        [
            154, 44, 154, 17, 14, 119, 215, 198, 46, 175, 171, 157, 193, 22, 57, 134, 26, 142, 53,
            232, 209, 83, 26, 15, 60, 5, 155, 141, 102, 218, 186, 5,
        ],
        [
            169, 131, 113, 37, 98, 173, 190, 221, 20, 16, 88, 92, 67, 160, 25, 246, 246, 66, 31,
            127, 43, 253, 88, 247, 40, 171, 221, 39, 50, 166, 101, 49,
        ],
        [
            6, 232, 55, 227, 94, 249, 213, 167, 115, 81, 188, 73, 113, 54, 198, 181, 173, 30, 217,
            186, 78, 230, 90, 133, 14, 129, 56, 51, 127, 107, 163, 51,
        ],
        [
            120, 238, 215, 74, 28, 49, 35, 46, 98, 63, 95, 51, 241, 195, 25, 78, 248, 132, 5, 211,
            1, 112, 122, 98, 21, 113, 125, 61, 64, 232, 110, 55,
        ],
        [
            33, 74, 172, 61, 251, 174, 136, 52, 115, 195, 205, 228, 151, 213, 16, 9, 226, 190, 182,
            100, 137, 243, 46, 178, 149, 220, 214, 116, 250, 124, 18, 35,
        ],
        [
            23, 44, 183, 43, 181, 227, 174, 141, 120, 243, 138, 225, 133, 125, 53, 145, 97, 115,
            48, 126, 152, 117, 171, 71, 162, 165, 176, 51, 32, 27, 188, 5,
        ],
        [
            166, 99, 229, 130, 156, 150, 80, 56, 81, 46, 155, 102, 207, 204, 193, 88, 50, 104, 126,
            37, 181, 161, 160, 7, 9, 133, 241, 123, 104, 67, 180, 56,
        ],
        [
            5, 115, 17, 153, 100, 192, 255, 228, 78, 171, 112, 78, 174, 8, 248, 132, 165, 207, 86,
            60, 87, 149, 165, 37, 147, 158, 161, 120, 86, 60, 84, 63,
        ],
    ],
    [
        [
            140, 37, 93, 56, 153, 182, 130, 1, 95, 53, 224, 183, 222, 82, 182, 84, 241, 179, 104,
            7, 205, 30, 188, 106, 227, 160, 151, 228, 251, 87, 188, 18,
        ],
        [
            69, 75, 168, 95, 88, 120, 61, 139, 158, 178, 85, 80, 96, 116, 164, 31, 85, 37, 223, 70,
            123, 211, 247, 151, 37, 26, 163, 44, 169, 26, 236, 52,
        ],
        [
            253, 240, 81, 251, 222, 234, 85, 234, 243, 202, 195, 107, 86, 77, 4, 133, 3, 164, 49,
            116, 28, 179, 60, 189, 129, 176, 72, 62, 103, 84, 130, 30,
        ],
        [
            130, 7, 87, 230, 130, 156, 229, 136, 96, 182, 48, 237, 135, 249, 123, 77, 66, 215, 128,
            140, 39, 78, 101, 221, 17, 128, 157, 194, 242, 184, 120, 39,
        ],
        [
            241, 198, 194, 238, 100, 111, 211, 220, 28, 45, 77, 201, 225, 153, 45, 12, 63, 142,
            104, 206, 74, 0, 59, 81, 227, 124, 44, 156, 149, 228, 177, 22,
        ],
        [
            33, 24, 148, 45, 69, 92, 133, 68, 104, 193, 9, 44, 233, 181, 230, 248, 164, 241, 31, 3,
            96, 4, 13, 178, 22, 115, 206, 184, 78, 66, 211, 24,
        ],
        [
            147, 117, 96, 107, 211, 62, 174, 184, 36, 61, 56, 45, 148, 144, 27, 195, 147, 103, 188,
            0, 153, 134, 134, 5, 144, 137, 176, 132, 146, 192, 72, 28,
        ],
        [
            41, 43, 175, 189, 159, 31, 156, 63, 221, 19, 178, 175, 103, 67, 43, 177, 46, 76, 65,
            225, 94, 126, 57, 99, 211, 22, 241, 242, 69, 188, 253, 53,
        ],
    ],
    [
        [
            16, 84, 117, 4, 29, 151, 25, 85, 212, 201, 246, 140, 236, 125, 51, 153, 82, 180, 2,
            136, 97, 13, 69, 179, 52, 2, 34, 99, 205, 11, 178, 37,
        ],
        [
            50, 5, 86, 25, 191, 2, 103, 33, 91, 250, 57, 64, 191, 115, 148, 97, 73, 135, 186, 105,
            51, 8, 93, 163, 185, 149, 97, 252, 14, 23, 223, 33,
        ],
        [
            204, 37, 164, 107, 54, 185, 240, 186, 162, 231, 206, 164, 103, 105, 235, 250, 204, 56,
            83, 83, 244, 139, 64, 169, 89, 86, 141, 122, 209, 83, 180, 2,
        ],
        [
            214, 15, 207, 13, 33, 170, 216, 82, 19, 202, 172, 127, 163, 185, 70, 191, 50, 74, 81,
            120, 137, 250, 94, 106, 89, 21, 176, 84, 116, 5, 70, 45,
        ],
        [
            228, 181, 219, 48, 206, 214, 218, 233, 172, 15, 42, 210, 93, 26, 106, 40, 46, 38, 31,
            173, 37, 220, 70, 249, 110, 228, 34, 161, 212, 145, 74, 0,
        ],
        [
            116, 63, 120, 41, 190, 77, 110, 48, 6, 95, 248, 65, 28, 254, 120, 111, 18, 232, 153,
            161, 194, 81, 56, 69, 191, 12, 8, 76, 239, 88, 105, 60,
        ],
        [
            159, 17, 123, 163, 202, 118, 227, 194, 63, 51, 216, 161, 216, 223, 247, 250, 58, 64,
            100, 242, 254, 203, 170, 226, 19, 185, 208, 69, 18, 10, 43, 52,
        ],
        [
            248, 196, 92, 216, 67, 195, 218, 121, 90, 231, 175, 194, 85, 169, 109, 214, 35, 240,
            86, 175, 175, 119, 7, 27, 43, 147, 31, 23, 64, 47, 182, 29,
        ],
    ],
    [
        [
            168, 89, 212, 27, 44, 101, 233, 29, 222, 148, 126, 207, 237, 33, 171, 118, 157, 87, 82,
            122, 63, 158, 78, 26, 156, 224, 204, 238, 11, 236, 75, 39,
        ],
        [
            210, 162, 30, 236, 158, 198, 77, 145, 133, 121, 228, 222, 39, 19, 50, 157, 87, 102,
            214, 144, 85, 84, 147, 165, 91, 172, 106, 113, 236, 192, 86, 9,
        ],
        [
            98, 136, 193, 144, 162, 192, 99, 58, 224, 77, 23, 246, 58, 162, 203, 65, 143, 113, 124,
            58, 65, 216, 85, 213, 87, 31, 156, 110, 167, 37, 247, 16,
        ],
        [
            114, 57, 34, 60, 250, 39, 223, 43, 226, 103, 214, 214, 185, 36, 210, 248, 181, 101, 81,
            91, 172, 174, 198, 64, 69, 215, 182, 122, 121, 48, 177, 53,
        ],
        [
            54, 247, 9, 251, 221, 228, 245, 195, 31, 168, 79, 197, 35, 1, 155, 159, 197, 197, 79,
            233, 113, 217, 39, 135, 103, 121, 171, 104, 219, 209, 151, 30,
        ],
        [
            67, 76, 134, 87, 87, 24, 179, 36, 45, 144, 205, 160, 105, 47, 150, 23, 0, 205, 234, 75,
            10, 133, 108, 14, 1, 91, 13, 238, 142, 215, 243, 58,
        ],
        [
            111, 211, 5, 125, 150, 54, 126, 4, 9, 226, 217, 30, 137, 148, 156, 22, 66, 243, 179,
            181, 153, 228, 127, 248, 2, 80, 29, 233, 179, 230, 245, 9,
        ],
        [
            196, 202, 65, 82, 146, 164, 175, 9, 27, 44, 243, 201, 113, 37, 164, 51, 220, 51, 110,
            110, 153, 72, 88, 87, 128, 158, 118, 53, 81, 212, 138, 1,
        ],
    ],
    [
        [
            240, 27, 128, 83, 62, 106, 95, 39, 98, 33, 131, 9, 133, 210, 62, 78, 76, 240, 235, 156,
            80, 124, 193, 74, 142, 209, 160, 60, 151, 106, 192, 17,
        ],
        [
            10, 234, 151, 181, 184, 48, 26, 195, 34, 48, 69, 70, 88, 128, 36, 171, 37, 190, 63,
            230, 131, 119, 96, 168, 221, 7, 3, 182, 61, 119, 146, 45,
        ],
        [
            1, 177, 14, 77, 209, 225, 50, 37, 44, 178, 155, 185, 255, 53, 136, 39, 217, 175, 86,
            46, 129, 51, 73, 213, 96, 70, 110, 128, 6, 209, 107, 40,
        ],
        [
            1, 41, 114, 5, 32, 182, 148, 153, 210, 242, 118, 122, 146, 225, 50, 237, 138, 117, 64,
            250, 6, 133, 64, 151, 239, 232, 112, 166, 222, 103, 192, 62,
        ],
        [
            40, 11, 170, 150, 94, 39, 123, 218, 115, 43, 29, 105, 156, 208, 30, 91, 242, 19, 160,
            156, 212, 127, 176, 42, 33, 71, 160, 76, 130, 160, 20, 36,
        ],
        [
            207, 175, 212, 51, 120, 120, 218, 173, 25, 249, 187, 76, 135, 243, 179, 80, 223, 23,
            120, 149, 107, 52, 156, 41, 255, 100, 193, 238, 248, 149, 153, 44,
        ],
        [
            122, 35, 185, 54, 102, 87, 34, 91, 232, 2, 170, 41, 84, 191, 167, 96, 141, 242, 39,
            237, 220, 115, 229, 243, 211, 216, 17, 222, 146, 109, 149, 48,
        ],
        [
            135, 20, 109, 135, 214, 156, 229, 98, 173, 213, 226, 44, 144, 56, 252, 137, 190, 235,
            120, 163, 206, 253, 181, 103, 130, 174, 201, 205, 4, 53, 152, 46,
        ],
    ],
    [
        [
            10, 142, 209, 19, 89, 150, 192, 43, 239, 151, 170, 251, 128, 226, 243, 190, 235, 26,
            66, 71, 55, 221, 236, 98, 149, 171, 88, 172, 230, 5, 233, 2,
        ],
        [
            156, 168, 79, 29, 44, 17, 130, 180, 80, 222, 115, 221, 227, 217, 195, 91, 218, 216,
            133, 2, 171, 15, 111, 82, 76, 171, 109, 215, 44, 206, 95, 60,
        ],
        [
            82, 50, 79, 123, 243, 18, 97, 3, 135, 72, 145, 152, 246, 155, 56, 167, 69, 232, 236,
            222, 17, 80, 228, 123, 110, 71, 187, 132, 180, 157, 94, 43,
        ],
        [
            133, 56, 222, 62, 206, 68, 120, 247, 78, 228, 79, 25, 173, 18, 206, 23, 199, 33, 94,
            146, 109, 86, 181, 57, 106, 232, 166, 187, 47, 45, 190, 31,
        ],
        [
            255, 251, 31, 52, 33, 199, 123, 217, 174, 217, 232, 117, 186, 143, 26, 30, 39, 117,
            165, 210, 103, 18, 34, 222, 59, 180, 171, 73, 64, 118, 74, 49,
        ],
        [
            68, 91, 156, 155, 35, 223, 48, 21, 231, 13, 153, 12, 213, 186, 1, 241, 5, 0, 203, 212,
            90, 49, 200, 64, 219, 252, 147, 46, 215, 151, 236, 7,
        ],
        [
            247, 226, 71, 82, 215, 139, 160, 131, 30, 130, 152, 222, 210, 5, 133, 197, 241, 86, 45,
            116, 53, 190, 25, 140, 158, 87, 213, 121, 12, 172, 199, 24,
        ],
        [
            229, 141, 95, 11, 163, 54, 149, 119, 4, 118, 188, 195, 249, 198, 228, 223, 95, 167,
            233, 188, 125, 155, 186, 86, 58, 4, 89, 10, 107, 36, 172, 16,
        ],
    ],
    [
        [
            253, 165, 27, 216, 77, 164, 49, 188, 140, 85, 119, 2, 143, 21, 241, 7, 224, 99, 213,
            140, 91, 71, 51, 39, 118, 152, 202, 160, 251, 21, 198, 32,
        ],
        [
            0, 210, 144, 78, 21, 82, 237, 163, 173, 160, 244, 242, 249, 95, 121, 211, 200, 10, 2,
            85, 106, 3, 176, 46, 36, 207, 155, 196, 110, 124, 128, 53,
        ],
        [
            85, 250, 155, 123, 178, 197, 121, 15, 218, 152, 126, 42, 137, 33, 202, 101, 178, 22,
            103, 192, 64, 139, 5, 127, 67, 94, 221, 100, 185, 73, 108, 10,
        ],
        [
            15, 53, 88, 70, 174, 250, 13, 25, 24, 206, 211, 240, 226, 219, 163, 151, 54, 247, 178,
            117, 151, 211, 152, 203, 237, 144, 163, 41, 173, 69, 195, 59,
        ],
        [
            242, 209, 249, 152, 137, 32, 207, 20, 121, 123, 210, 175, 104, 77, 55, 143, 115, 224,
            47, 4, 147, 97, 154, 50, 129, 125, 15, 23, 186, 161, 25, 29,
        ],
        [
            210, 47, 108, 11, 41, 196, 180, 206, 209, 19, 36, 27, 105, 0, 242, 184, 75, 46, 130,
            150, 119, 203, 105, 86, 64, 122, 246, 157, 103, 45, 7, 62,
        ],
        [
            120, 226, 121, 154, 242, 62, 202, 9, 87, 218, 91, 88, 3, 245, 76, 76, 40, 26, 68, 80,
            180, 66, 58, 139, 88, 100, 199, 100, 36, 216, 191, 58,
        ],
        [
            160, 142, 135, 30, 128, 186, 93, 94, 200, 52, 246, 68, 226, 51, 140, 36, 58, 238, 64,
            48, 165, 135, 28, 66, 183, 186, 4, 230, 104, 111, 70, 17,
        ],
    ],
    [
        [
            169, 163, 197, 216, 116, 65, 109, 237, 51, 192, 24, 202, 186, 54, 64, 16, 135, 35, 119,
            123, 30, 27, 149, 113, 152, 27, 213, 235, 74, 228, 25, 50,
        ],
        [
            94, 146, 241, 96, 243, 211, 176, 157, 5, 20, 143, 58, 78, 196, 233, 156, 201, 33, 237,
            201, 216, 30, 138, 99, 241, 75, 94, 195, 201, 115, 90, 57,
        ],
        [
            223, 25, 95, 223, 236, 31, 251, 252, 110, 101, 210, 88, 132, 5, 20, 60, 19, 196, 188,
            207, 184, 129, 218, 204, 171, 58, 88, 159, 254, 52, 84, 16,
        ],
        [
            167, 254, 178, 116, 5, 197, 144, 192, 29, 226, 111, 206, 183, 152, 229, 102, 244, 61,
            231, 113, 249, 238, 28, 117, 83, 196, 204, 98, 168, 252, 8, 4,
        ],
        [
            116, 228, 134, 78, 53, 106, 91, 46, 222, 231, 2, 212, 189, 226, 173, 168, 212, 157,
            120, 28, 166, 17, 232, 145, 36, 121, 221, 35, 72, 66, 196, 34,
        ],
        [
            20, 133, 122, 73, 166, 68, 157, 123, 246, 7, 77, 139, 203, 121, 158, 76, 116, 100, 127,
            230, 51, 212, 112, 138, 193, 207, 153, 250, 158, 18, 75, 51,
        ],
        [
            165, 172, 206, 156, 241, 81, 188, 251, 147, 161, 222, 196, 133, 69, 112, 59, 135, 119,
            47, 33, 85, 179, 164, 232, 120, 182, 104, 5, 196, 161, 182, 56,
        ],
        [
            167, 142, 23, 10, 197, 190, 129, 124, 191, 14, 167, 7, 54, 72, 167, 134, 252, 194, 121,
            224, 7, 189, 197, 147, 198, 62, 216, 75, 238, 55, 124, 8,
        ],
    ],
    [
        [
            169, 7, 115, 210, 156, 19, 28, 45, 159, 241, 95, 115, 25, 97, 41, 82, 205, 96, 92, 201,
            180, 27, 188, 101, 78, 34, 164, 62, 39, 77, 225, 44,
        ],
        [
            96, 87, 81, 178, 227, 39, 109, 76, 110, 243, 25, 223, 80, 132, 214, 188, 215, 193, 106,
            192, 165, 53, 20, 194, 37, 17, 243, 21, 41, 22, 18, 54,
        ],
        [
            228, 207, 168, 250, 67, 67, 250, 225, 163, 246, 3, 206, 243, 52, 113, 22, 21, 184, 62,
            177, 122, 200, 204, 165, 238, 201, 224, 62, 72, 33, 81, 36,
        ],
        [
            62, 26, 98, 19, 20, 210, 218, 57, 126, 87, 89, 162, 47, 111, 194, 12, 214, 134, 160,
            32, 229, 220, 107, 171, 18, 207, 106, 82, 189, 204, 141, 1,
        ],
        [
            17, 26, 209, 134, 88, 233, 110, 84, 43, 104, 90, 54, 64, 165, 108, 53, 251, 129, 213,
            254, 251, 101, 145, 40, 7, 231, 242, 249, 72, 214, 39, 42,
        ],
        [
            161, 17, 192, 70, 218, 119, 88, 156, 78, 162, 251, 1, 234, 203, 81, 56, 126, 194, 126,
            42, 165, 66, 66, 219, 157, 74, 203, 143, 18, 47, 51, 52,
        ],
        [
            43, 157, 147, 232, 6, 52, 131, 52, 72, 123, 249, 157, 60, 89, 230, 143, 115, 50, 38,
            134, 111, 206, 30, 175, 65, 36, 76, 157, 164, 65, 201, 38,
        ],
        [
            206, 22, 236, 47, 122, 166, 215, 229, 161, 250, 56, 20, 180, 86, 60, 90, 251, 130, 72,
            167, 145, 105, 142, 204, 131, 93, 69, 55, 146, 165, 221, 40,
        ],
    ],
    [
        [
            249, 242, 21, 59, 135, 24, 245, 119, 127, 198, 118, 52, 239, 218, 112, 56, 29, 37, 147,
            112, 46, 128, 19, 82, 194, 81, 245, 247, 54, 65, 29, 14,
        ],
        [
            36, 83, 140, 239, 206, 62, 115, 139, 236, 215, 196, 82, 95, 248, 71, 245, 58, 78, 174,
            33, 182, 29, 67, 2, 49, 46, 188, 235, 242, 85, 224, 3,
        ],
        [
            62, 123, 204, 165, 50, 119, 248, 120, 89, 102, 76, 10, 251, 221, 182, 243, 92, 147, 22,
            111, 30, 132, 4, 19, 86, 16, 201, 53, 98, 18, 43, 48,
        ],
        [
            51, 40, 231, 19, 115, 64, 149, 172, 249, 156, 238, 143, 245, 188, 130, 96, 154, 29,
            135, 109, 46, 254, 185, 130, 61, 28, 67, 8, 172, 115, 79, 58,
        ],
        [
            54, 123, 99, 202, 255, 50, 82, 189, 246, 133, 142, 66, 22, 38, 76, 253, 238, 34, 88,
            116, 77, 242, 84, 108, 243, 231, 178, 151, 103, 255, 89, 9,
        ],
        [
            80, 138, 121, 15, 93, 11, 24, 112, 146, 63, 198, 155, 12, 127, 20, 15, 94, 69, 39, 221,
            107, 251, 182, 184, 16, 84, 90, 69, 159, 213, 39, 54,
        ],
        [
            139, 70, 202, 84, 209, 56, 254, 84, 216, 195, 84, 158, 124, 98, 163, 218, 94, 11, 163,
            9, 42, 83, 159, 244, 179, 75, 190, 59, 46, 184, 30, 41,
        ],
        [
            104, 75, 123, 111, 5, 75, 204, 36, 182, 49, 51, 126, 209, 201, 143, 115, 154, 53, 116,
            250, 77, 65, 69, 226, 128, 142, 205, 101, 163, 217, 15, 42,
        ],
    ],
    [
        [
            12, 0, 40, 89, 200, 146, 231, 32, 7, 210, 86, 77, 206, 245, 234, 19, 172, 170, 58, 108,
            36, 183, 180, 153, 155, 161, 74, 78, 18, 213, 106, 45,
        ],
        [
            54, 121, 227, 245, 13, 91, 172, 234, 241, 135, 83, 92, 132, 87, 223, 225, 170, 82, 147,
            106, 56, 152, 196, 15, 110, 16, 220, 210, 167, 132, 147, 46,
        ],
        [
            191, 194, 84, 235, 59, 153, 63, 90, 249, 253, 195, 202, 19, 162, 43, 210, 241, 180, 81,
            251, 7, 129, 85, 120, 223, 64, 187, 56, 7, 190, 168, 34,
        ],
        [
            134, 60, 171, 29, 83, 57, 134, 240, 159, 75, 121, 97, 53, 145, 10, 97, 60, 41, 64, 4,
            212, 180, 179, 52, 1, 58, 78, 23, 120, 40, 35, 54,
        ],
        [
            184, 110, 240, 2, 73, 205, 194, 246, 210, 166, 203, 213, 175, 225, 47, 211, 142, 183,
            222, 180, 253, 7, 251, 237, 181, 54, 129, 165, 27, 43, 117, 39,
        ],
        [
            192, 117, 239, 105, 11, 250, 37, 116, 110, 41, 150, 214, 18, 127, 6, 3, 95, 10, 113,
            251, 138, 117, 88, 98, 219, 30, 92, 111, 47, 121, 62, 49,
        ],
        [
            106, 41, 188, 0, 79, 170, 110, 58, 210, 56, 221, 140, 97, 225, 151, 132, 115, 192, 82,
            181, 19, 163, 13, 17, 236, 205, 15, 30, 3, 211, 249, 6,
        ],
        [
            87, 144, 20, 67, 36, 215, 225, 190, 145, 89, 16, 213, 18, 121, 231, 161, 91, 162, 36,
            252, 20, 120, 174, 234, 203, 86, 225, 106, 52, 13, 38, 45,
        ],
    ],
    [
        [
            238, 133, 103, 91, 99, 161, 201, 157, 122, 248, 132, 37, 241, 150, 151, 26, 199, 151,
            164, 97, 34, 160, 197, 202, 52, 162, 229, 43, 237, 227, 136, 45,
        ],
        [
            162, 65, 50, 209, 33, 252, 250, 222, 24, 113, 30, 11, 36, 85, 185, 206, 100, 108, 219,
            183, 221, 16, 178, 179, 163, 18, 187, 82, 67, 198, 198, 57,
        ],
        [
            69, 233, 89, 202, 122, 215, 104, 132, 206, 156, 152, 147, 197, 210, 27, 231, 183, 186,
            92, 17, 120, 187, 143, 163, 225, 81, 21, 72, 93, 7, 196, 27,
        ],
        [
            38, 237, 19, 29, 117, 255, 144, 122, 171, 181, 247, 94, 88, 101, 213, 241, 186, 225,
            134, 4, 80, 120, 94, 234, 102, 183, 99, 164, 131, 187, 132, 55,
        ],
        [
            155, 156, 64, 199, 160, 171, 202, 191, 152, 58, 169, 151, 175, 213, 147, 132, 46, 164,
            196, 7, 196, 183, 186, 71, 114, 71, 184, 126, 44, 47, 24, 62,
        ],
        [
            47, 44, 188, 236, 111, 225, 46, 128, 76, 235, 139, 140, 205, 235, 29, 129, 198, 199,
            202, 172, 93, 81, 72, 26, 142, 31, 99, 48, 152, 248, 28, 40,
        ],
        [
            36, 151, 70, 226, 44, 150, 152, 239, 239, 233, 50, 179, 161, 120, 65, 135, 195, 27,
            135, 103, 192, 203, 73, 94, 21, 30, 227, 41, 40, 148, 225, 0,
        ],
        [
            155, 17, 42, 85, 170, 76, 233, 176, 16, 190, 197, 57, 30, 42, 33, 223, 101, 22, 18,
            206, 171, 83, 208, 216, 82, 178, 213, 81, 74, 255, 95, 38,
        ],
    ],
    [
        [
            54, 8, 224, 104, 248, 77, 99, 60, 13, 208, 245, 117, 189, 100, 102, 86, 46, 130, 26,
            21, 173, 106, 18, 60, 169, 20, 254, 83, 119, 0, 95, 5,
        ],
        [
            230, 178, 120, 49, 250, 42, 251, 98, 39, 3, 136, 65, 253, 165, 204, 239, 163, 148, 51,
            169, 122, 182, 72, 197, 222, 128, 147, 72, 237, 156, 152, 8,
        ],
        [
            19, 86, 176, 28, 57, 166, 19, 232, 42, 211, 58, 20, 150, 66, 80, 17, 109, 238, 228, 68,
            184, 14, 15, 150, 165, 6, 102, 79, 156, 6, 3, 1,
        ],
        [
            253, 225, 19, 134, 49, 103, 63, 151, 7, 151, 27, 174, 176, 155, 244, 254, 80, 10, 77,
            13, 125, 6, 0, 159, 174, 133, 128, 162, 163, 54, 32, 7,
        ],
        [
            20, 122, 210, 59, 32, 118, 92, 7, 227, 125, 245, 98, 228, 150, 121, 191, 233, 127, 217,
            33, 245, 24, 145, 160, 149, 249, 30, 132, 54, 226, 121, 9,
        ],
        [
            33, 175, 62, 170, 114, 81, 193, 177, 6, 202, 21, 242, 244, 183, 86, 196, 52, 220, 193,
            134, 119, 61, 215, 155, 24, 133, 127, 204, 95, 47, 77, 59,
        ],
        [
            189, 6, 50, 74, 18, 109, 39, 182, 206, 232, 250, 242, 210, 131, 138, 112, 119, 166, 76,
            26, 90, 52, 105, 200, 244, 199, 32, 40, 98, 228, 167, 37,
        ],
        [
            56, 113, 154, 35, 162, 76, 47, 212, 228, 242, 71, 241, 126, 255, 187, 178, 64, 126, 81,
            245, 80, 182, 200, 85, 172, 97, 35, 77, 184, 251, 84, 25,
        ],
    ],
    [
        [
            12, 230, 104, 29, 206, 194, 74, 78, 85, 208, 57, 110, 166, 4, 210, 246, 253, 77, 87,
            139, 137, 151, 107, 151, 193, 139, 48, 7, 131, 243, 165, 56,
        ],
        [
            16, 106, 200, 3, 167, 12, 130, 120, 148, 233, 201, 119, 44, 221, 151, 90, 228, 85, 241,
            0, 215, 129, 245, 242, 36, 55, 115, 198, 144, 158, 69, 18,
        ],
        [
            28, 44, 187, 226, 84, 6, 135, 126, 182, 209, 160, 144, 14, 180, 21, 44, 8, 15, 244, 88,
            212, 42, 24, 105, 125, 35, 172, 66, 47, 113, 133, 33,
        ],
        [
            9, 189, 144, 175, 191, 136, 166, 240, 19, 237, 73, 19, 206, 196, 9, 199, 5, 25, 202,
            165, 13, 171, 240, 169, 192, 122, 100, 129, 176, 192, 234, 55,
        ],
        [
            192, 225, 147, 209, 101, 193, 163, 144, 93, 123, 172, 39, 143, 71, 136, 63, 99, 52,
            106, 194, 237, 87, 89, 235, 146, 135, 72, 96, 74, 2, 53, 37,
        ],
        [
            0, 89, 253, 106, 180, 242, 187, 141, 158, 209, 229, 100, 132, 116, 24, 10, 23, 103,
            207, 141, 111, 230, 149, 18, 30, 13, 97, 180, 135, 73, 5, 0,
        ],
        [
            81, 161, 223, 168, 32, 133, 55, 175, 229, 60, 247, 13, 214, 85, 147, 136, 166, 182,
            207, 165, 70, 5, 246, 18, 121, 61, 124, 98, 243, 138, 30, 51,
        ],
        [
            57, 191, 83, 37, 102, 159, 132, 165, 105, 85, 174, 131, 158, 51, 155, 73, 53, 161, 34,
            45, 154, 71, 201, 15, 41, 223, 46, 21, 189, 231, 40, 19,
        ],
    ],
    [
        [
            138, 114, 183, 238, 119, 69, 77, 194, 154, 214, 49, 36, 14, 183, 188, 15, 31, 242, 222,
            54, 184, 157, 231, 0, 48, 45, 248, 77, 28, 231, 154, 36,
        ],
        [
            179, 41, 18, 93, 149, 95, 227, 27, 14, 200, 100, 184, 90, 178, 30, 127, 59, 195, 253,
            44, 236, 163, 110, 9, 21, 169, 129, 4, 69, 121, 147, 30,
        ],
        [
            105, 82, 83, 247, 98, 229, 63, 229, 249, 250, 153, 84, 251, 79, 191, 217, 11, 54, 72,
            161, 77, 13, 8, 84, 146, 179, 195, 130, 120, 104, 253, 33,
        ],
        [
            79, 107, 10, 22, 0, 150, 204, 208, 40, 254, 8, 117, 61, 94, 43, 129, 204, 43, 103, 68,
            128, 53, 5, 45, 217, 46, 73, 163, 205, 197, 162, 32,
        ],
        [
            32, 137, 210, 135, 2, 141, 240, 12, 66, 216, 159, 1, 91, 255, 208, 69, 249, 108, 78,
            199, 222, 246, 75, 30, 19, 10, 100, 23, 19, 55, 158, 38,
        ],
        [
            225, 215, 161, 47, 217, 150, 60, 35, 109, 2, 159, 109, 167, 241, 193, 228, 243, 128,
            84, 155, 136, 95, 205, 141, 23, 244, 34, 211, 191, 43, 165, 56,
        ],
        [
            229, 240, 112, 132, 192, 72, 52, 216, 10, 111, 211, 17, 255, 129, 216, 0, 160, 71, 253,
            197, 23, 15, 88, 86, 119, 22, 7, 215, 96, 165, 238, 51,
        ],
        [
            41, 241, 99, 40, 68, 20, 115, 88, 65, 43, 219, 96, 252, 18, 192, 3, 209, 196, 244, 173,
            167, 101, 107, 131, 145, 136, 227, 57, 29, 168, 114, 11,
        ],
    ],
    [
        [
            154, 239, 49, 188, 250, 102, 141, 162, 103, 128, 214, 237, 112, 83, 82, 39, 169, 38,
            186, 224, 83, 17, 44, 13, 89, 204, 82, 168, 153, 47, 177, 32,
        ],
        [
            44, 230, 100, 6, 130, 117, 77, 153, 54, 35, 247, 189, 194, 8, 16, 63, 21, 142, 72, 135,
            168, 206, 177, 164, 49, 230, 99, 225, 60, 67, 155, 39,
        ],
        [
            169, 141, 219, 61, 4, 46, 12, 93, 167, 222, 157, 101, 225, 66, 40, 200, 99, 167, 104,
            200, 201, 121, 56, 131, 9, 149, 7, 96, 78, 147, 45, 35,
        ],
        [
            60, 73, 142, 226, 0, 134, 72, 110, 138, 195, 66, 163, 232, 76, 236, 197, 119, 245, 33,
            176, 176, 193, 191, 82, 93, 99, 200, 122, 214, 4, 106, 62,
        ],
        [
            33, 168, 2, 134, 195, 243, 192, 108, 200, 39, 203, 209, 184, 80, 67, 36, 36, 133, 17,
            84, 252, 62, 107, 24, 128, 56, 93, 32, 88, 187, 74, 20,
        ],
        [
            219, 20, 232, 167, 231, 189, 251, 75, 141, 188, 40, 255, 113, 169, 75, 126, 51, 153,
            150, 144, 3, 2, 48, 195, 95, 106, 157, 67, 187, 122, 83, 58,
        ],
        [
            188, 190, 244, 180, 101, 55, 139, 190, 117, 226, 19, 119, 22, 66, 0, 180, 238, 67, 105,
            113, 146, 196, 115, 38, 40, 228, 250, 123, 178, 244, 232, 50,
        ],
        [
            128, 9, 198, 45, 101, 40, 185, 231, 79, 41, 73, 11, 16, 195, 192, 252, 134, 134, 140,
            132, 52, 99, 62, 164, 180, 33, 52, 159, 66, 196, 79, 35,
        ],
    ],
    [
        [
            10, 239, 94, 184, 118, 1, 158, 108, 158, 169, 76, 121, 236, 36, 25, 221, 105, 129, 115,
            149, 209, 44, 31, 83, 91, 26, 78, 126, 255, 38, 25, 56,
        ],
        [
            207, 84, 209, 233, 227, 197, 135, 68, 247, 42, 105, 196, 115, 52, 31, 57, 58, 7, 51,
            12, 151, 197, 15, 178, 131, 83, 160, 90, 179, 238, 0, 1,
        ],
        [
            88, 114, 54, 87, 151, 143, 109, 8, 32, 131, 174, 143, 36, 0, 241, 67, 69, 39, 183, 24,
            46, 190, 165, 142, 35, 116, 35, 242, 200, 167, 172, 63,
        ],
        [
            33, 70, 50, 238, 103, 165, 48, 92, 4, 221, 0, 157, 84, 240, 144, 240, 18, 63, 226, 1,
            151, 3, 49, 247, 129, 248, 23, 106, 90, 166, 204, 44,
        ],
        [
            106, 134, 57, 46, 200, 54, 110, 138, 86, 96, 6, 197, 4, 183, 174, 208, 239, 184, 201,
            108, 12, 179, 36, 69, 218, 9, 165, 76, 112, 142, 76, 14,
        ],
        [
            97, 3, 42, 225, 95, 159, 129, 254, 99, 46, 181, 184, 192, 3, 168, 169, 118, 8, 164, 8,
            94, 203, 221, 208, 205, 41, 189, 179, 30, 177, 251, 44,
        ],
        [
            53, 132, 132, 52, 92, 180, 183, 130, 219, 108, 144, 28, 118, 56, 255, 151, 102, 154,
            205, 137, 43, 116, 29, 96, 143, 6, 143, 205, 163, 245, 106, 44,
        ],
        [
            218, 157, 183, 63, 44, 213, 213, 83, 210, 222, 174, 20, 246, 248, 126, 87, 221, 23,
            142, 136, 120, 246, 43, 32, 148, 58, 248, 160, 123, 58, 221, 12,
        ],
    ],
    [
        [
            149, 137, 111, 83, 176, 241, 115, 102, 142, 224, 16, 10, 96, 228, 141, 172, 140, 55,
            25, 116, 233, 91, 178, 226, 61, 228, 239, 3, 161, 147, 55, 0,
        ],
        [
            224, 0, 124, 93, 127, 173, 200, 210, 76, 186, 63, 249, 145, 142, 191, 53, 164, 52, 12,
            246, 5, 147, 5, 219, 84, 33, 67, 17, 12, 186, 56, 6,
        ],
        [
            141, 150, 107, 181, 39, 27, 6, 86, 75, 78, 249, 120, 1, 70, 150, 239, 62, 91, 105, 27,
            207, 74, 134, 102, 24, 119, 82, 203, 42, 91, 232, 27,
        ],
        [
            98, 255, 117, 104, 87, 175, 148, 205, 89, 247, 46, 53, 8, 41, 74, 139, 233, 18, 235,
            173, 199, 208, 211, 168, 2, 68, 169, 180, 164, 20, 97, 42,
        ],
        [
            142, 31, 62, 232, 242, 54, 172, 213, 218, 45, 139, 176, 152, 33, 55, 113, 31, 236, 64,
            163, 4, 159, 145, 68, 47, 187, 118, 126, 73, 172, 66, 51,
        ],
        [
            177, 99, 95, 196, 0, 236, 92, 199, 227, 64, 111, 143, 71, 195, 178, 33, 230, 212, 13,
            0, 88, 120, 166, 23, 56, 146, 248, 71, 64, 129, 231, 28,
        ],
        [
            211, 107, 48, 102, 191, 222, 34, 82, 109, 123, 188, 56, 84, 50, 158, 35, 226, 179, 252,
            12, 108, 246, 213, 136, 254, 116, 42, 71, 83, 167, 134, 38,
        ],
        [
            60, 43, 191, 53, 154, 254, 231, 249, 60, 152, 183, 226, 99, 111, 57, 196, 8, 126, 203,
            215, 228, 240, 139, 196, 74, 217, 66, 180, 5, 49, 10, 61,
        ],
    ],
    [
        [
            20, 60, 106, 50, 215, 140, 210, 245, 242, 191, 232, 212, 151, 142, 214, 148, 65, 129,
            147, 86, 16, 226, 222, 240, 178, 198, 208, 215, 155, 63, 153, 3,
        ],
        [
            231, 76, 69, 20, 14, 9, 126, 164, 67, 97, 38, 101, 31, 238, 75, 208, 65, 146, 153, 62,
            115, 249, 153, 193, 36, 89, 201, 158, 168, 83, 14, 3,
        ],
        [
            172, 28, 102, 166, 48, 244, 128, 113, 84, 168, 154, 152, 71, 47, 14, 238, 173, 216,
            173, 84, 87, 152, 229, 74, 214, 142, 86, 0, 218, 201, 76, 38,
        ],
        [
            250, 251, 83, 44, 140, 74, 35, 121, 221, 192, 158, 22, 150, 145, 158, 182, 199, 58,
            139, 95, 104, 115, 78, 251, 35, 114, 123, 157, 107, 127, 141, 2,
        ],
        [
            15, 141, 54, 127, 192, 139, 29, 132, 103, 155, 18, 77, 69, 91, 246, 255, 11, 5, 164,
            52, 118, 220, 215, 136, 109, 14, 70, 118, 147, 212, 180, 52,
        ],
        [
            162, 34, 139, 38, 22, 144, 76, 40, 62, 100, 16, 96, 187, 133, 251, 207, 24, 229, 28,
            15, 243, 126, 192, 49, 78, 19, 171, 244, 38, 152, 49, 48,
        ],
        [
            156, 79, 249, 126, 214, 71, 52, 167, 23, 74, 34, 53, 116, 119, 101, 73, 139, 0, 214, 8,
            165, 21, 237, 91, 41, 154, 105, 148, 158, 109, 86, 4,
        ],
        [
            18, 190, 253, 27, 131, 121, 226, 105, 125, 62, 67, 179, 22, 245, 106, 80, 132, 191, 71,
            72, 139, 255, 44, 115, 58, 245, 52, 70, 181, 3, 70, 21,
        ],
    ],
    [
        [
            80, 124, 240, 232, 204, 160, 223, 47, 212, 70, 254, 52, 246, 144, 208, 188, 234, 135,
            228, 159, 159, 194, 192, 107, 188, 57, 154, 163, 184, 53, 100, 44,
        ],
        [
            75, 85, 94, 47, 222, 218, 25, 221, 103, 44, 223, 208, 96, 167, 80, 244, 3, 145, 108,
            246, 106, 250, 68, 7, 167, 208, 175, 48, 117, 132, 62, 56,
        ],
        [
            86, 43, 192, 127, 103, 49, 141, 242, 26, 223, 123, 81, 193, 204, 142, 184, 208, 72, 98,
            101, 163, 68, 149, 183, 184, 29, 148, 26, 79, 89, 63, 7,
        ],
        [
            13, 17, 229, 153, 40, 187, 221, 180, 216, 40, 169, 214, 113, 40, 104, 151, 96, 61, 225,
            245, 220, 170, 198, 151, 134, 249, 146, 22, 49, 18, 117, 23,
        ],
        [
            24, 160, 53, 180, 65, 171, 149, 169, 240, 40, 53, 91, 160, 6, 73, 4, 74, 136, 214, 119,
            226, 139, 89, 65, 171, 116, 135, 48, 173, 31, 151, 21,
        ],
        [
            87, 245, 10, 110, 108, 73, 107, 12, 238, 92, 189, 142, 60, 156, 208, 149, 246, 203,
            116, 204, 81, 86, 83, 23, 216, 84, 14, 59, 69, 226, 129, 53,
        ],
        [
            136, 212, 47, 189, 103, 45, 150, 40, 232, 21, 226, 251, 185, 20, 204, 133, 61, 33, 35,
            12, 250, 121, 142, 19, 135, 13, 175, 97, 211, 151, 76, 59,
        ],
        [
            94, 35, 203, 250, 198, 153, 2, 78, 51, 187, 241, 100, 16, 204, 18, 163, 4, 164, 181,
            48, 70, 13, 140, 179, 246, 138, 46, 169, 239, 63, 83, 54,
        ],
    ],
    [
        [
            196, 215, 172, 247, 59, 103, 86, 156, 251, 251, 189, 44, 121, 171, 238, 170, 90, 94,
            229, 210, 2, 28, 116, 152, 175, 132, 150, 248, 247, 33, 93, 21,
        ],
        [
            11, 90, 134, 193, 86, 130, 247, 159, 151, 103, 235, 179, 36, 64, 31, 5, 216, 136, 218,
            242, 78, 34, 94, 97, 216, 226, 91, 32, 255, 21, 194, 8,
        ],
        [
            90, 247, 208, 172, 150, 176, 5, 164, 130, 56, 121, 166, 192, 17, 147, 181, 227, 173,
            129, 60, 164, 158, 107, 41, 196, 190, 42, 251, 119, 141, 10, 39,
        ],
        [
            251, 222, 252, 102, 190, 25, 50, 11, 77, 246, 85, 34, 74, 50, 187, 142, 67, 52, 11,
            106, 12, 141, 82, 182, 116, 253, 163, 140, 96, 98, 49, 24,
        ],
        [
            8, 73, 143, 244, 252, 146, 102, 150, 89, 157, 174, 71, 145, 173, 140, 93, 252, 86, 188,
            244, 50, 29, 20, 215, 83, 180, 168, 192, 50, 174, 135, 35,
        ],
        [
            9, 29, 218, 176, 253, 21, 178, 210, 244, 103, 160, 163, 241, 203, 37, 18, 141, 143, 66,
            247, 138, 118, 238, 64, 213, 141, 137, 207, 192, 146, 60, 16,
        ],
        [
            23, 112, 67, 207, 126, 97, 125, 148, 105, 113, 57, 100, 159, 254, 187, 221, 179, 75,
            84, 174, 136, 217, 67, 60, 52, 107, 254, 187, 54, 211, 173, 20,
        ],
        [
            197, 164, 148, 248, 41, 239, 242, 160, 160, 121, 202, 77, 129, 207, 71, 251, 23, 90,
            94, 78, 15, 206, 195, 162, 59, 131, 154, 91, 51, 112, 146, 20,
        ],
    ],
    [
        [
            197, 194, 236, 107, 71, 31, 175, 12, 97, 227, 76, 119, 30, 218, 41, 255, 125, 91, 56,
            190, 6, 1, 9, 248, 146, 9, 190, 14, 239, 199, 13, 35,
        ],
        [
            26, 55, 245, 109, 77, 123, 10, 248, 62, 199, 41, 101, 47, 206, 8, 105, 156, 69, 125, 8,
            50, 122, 23, 226, 145, 12, 219, 99, 67, 3, 191, 61,
        ],
        [
            109, 119, 98, 22, 28, 134, 177, 75, 25, 239, 54, 177, 169, 132, 116, 64, 4, 126, 221,
            82, 209, 126, 208, 215, 116, 246, 75, 215, 125, 163, 194, 7,
        ],
        [
            206, 15, 193, 174, 47, 41, 191, 214, 49, 210, 206, 7, 100, 245, 30, 170, 116, 39, 148,
            121, 53, 80, 201, 222, 6, 85, 55, 205, 72, 174, 6, 30,
        ],
        [
            229, 61, 220, 101, 100, 229, 212, 61, 175, 75, 4, 139, 117, 135, 63, 236, 252, 252,
            226, 155, 85, 116, 194, 245, 243, 209, 32, 159, 63, 234, 30, 37,
        ],
        [
            101, 33, 62, 209, 69, 119, 42, 92, 72, 89, 169, 145, 81, 100, 138, 196, 41, 253, 17,
            139, 60, 148, 139, 33, 255, 217, 8, 120, 230, 207, 192, 0,
        ],
        [
            39, 245, 116, 42, 45, 45, 44, 76, 177, 117, 99, 172, 86, 59, 13, 153, 100, 22, 9, 34,
            175, 154, 45, 229, 115, 193, 15, 182, 165, 125, 221, 1,
        ],
        [
            31, 54, 118, 58, 10, 66, 64, 174, 93, 114, 215, 142, 4, 206, 248, 153, 215, 197, 227,
            232, 157, 63, 47, 149, 123, 20, 49, 177, 74, 71, 82, 47,
        ],
    ],
    [
        [
            101, 117, 124, 209, 188, 36, 221, 34, 18, 6, 100, 248, 247, 169, 215, 102, 222, 48, 63,
            163, 182, 198, 83, 29, 203, 38, 22, 215, 118, 207, 177, 24,
        ],
        [
            84, 130, 80, 94, 13, 54, 109, 181, 221, 55, 251, 15, 143, 19, 87, 37, 148, 115, 213,
            160, 66, 83, 85, 162, 181, 179, 123, 17, 167, 247, 169, 37,
        ],
        [
            48, 59, 190, 61, 155, 234, 182, 117, 7, 244, 103, 194, 137, 218, 193, 172, 133, 66,
            223, 190, 111, 184, 35, 252, 102, 241, 89, 202, 143, 204, 240, 20,
        ],
        [
            119, 50, 120, 161, 215, 33, 56, 98, 241, 90, 240, 196, 248, 176, 102, 160, 235, 212,
            132, 151, 245, 11, 100, 223, 178, 133, 219, 85, 167, 246, 224, 21,
        ],
        [
            229, 144, 255, 154, 253, 52, 218, 12, 50, 165, 137, 170, 99, 196, 224, 186, 205, 85,
            190, 222, 118, 171, 254, 22, 7, 161, 255, 63, 75, 232, 3, 40,
        ],
        [
            65, 180, 60, 254, 30, 123, 254, 166, 174, 240, 212, 209, 13, 250, 250, 250, 145, 169,
            183, 194, 250, 165, 207, 148, 223, 58, 237, 44, 189, 216, 54, 42,
        ],
        [
            226, 251, 98, 219, 120, 82, 25, 105, 66, 253, 36, 140, 6, 244, 78, 64, 192, 2, 213,
            254, 215, 31, 221, 88, 6, 22, 221, 114, 145, 79, 29, 5,
        ],
        [
            113, 197, 52, 189, 31, 251, 25, 215, 225, 220, 50, 184, 252, 46, 191, 8, 128, 233, 194,
            71, 209, 60, 75, 190, 18, 160, 165, 185, 140, 84, 132, 54,
        ],
    ],
    [
        [
            142, 77, 52, 88, 247, 15, 6, 176, 107, 248, 160, 21, 209, 145, 80, 129, 225, 111, 95,
            237, 154, 77, 137, 253, 198, 71, 191, 114, 103, 5, 177, 39,
        ],
        [
            100, 4, 219, 69, 66, 64, 242, 131, 146, 201, 195, 14, 122, 183, 133, 62, 93, 159, 223,
            121, 254, 112, 235, 156, 41, 123, 165, 215, 129, 120, 57, 57,
        ],
        [
            11, 207, 121, 20, 205, 231, 207, 28, 66, 107, 148, 129, 36, 231, 40, 57, 75, 223, 109,
            181, 67, 177, 222, 11, 49, 217, 251, 164, 148, 153, 149, 7,
        ],
        [
            162, 102, 217, 50, 109, 94, 110, 154, 241, 207, 28, 240, 23, 129, 206, 78, 234, 74,
            156, 43, 11, 214, 17, 71, 48, 2, 64, 148, 241, 130, 185, 3,
        ],
        [
            142, 69, 138, 197, 77, 73, 50, 67, 52, 165, 96, 106, 125, 213, 111, 77, 92, 113, 130,
            95, 184, 18, 167, 102, 222, 99, 108, 45, 158, 82, 209, 35,
        ],
        [
            87, 67, 165, 162, 27, 141, 113, 255, 161, 65, 129, 182, 154, 52, 53, 145, 58, 102, 217,
            92, 230, 24, 189, 13, 111, 139, 65, 225, 173, 227, 67, 28,
        ],
        [
            202, 80, 111, 163, 121, 60, 211, 55, 216, 58, 50, 109, 16, 57, 8, 8, 46, 72, 193, 153,
            65, 167, 132, 192, 231, 21, 247, 69, 33, 20, 58, 23,
        ],
        [
            249, 173, 126, 139, 157, 178, 129, 165, 40, 169, 0, 114, 71, 157, 88, 113, 122, 40,
            191, 200, 72, 90, 5, 41, 121, 7, 152, 40, 85, 132, 39, 4,
        ],
    ],
    [
        [
            199, 216, 114, 251, 140, 210, 175, 171, 220, 148, 69, 166, 128, 95, 74, 205, 248, 245,
            19, 125, 233, 17, 142, 185, 113, 146, 57, 42, 0, 120, 198, 4,
        ],
        [
            250, 9, 179, 31, 23, 73, 162, 25, 28, 253, 80, 9, 232, 137, 137, 43, 17, 241, 71, 178,
            151, 218, 104, 93, 121, 153, 27, 40, 120, 219, 43, 47,
        ],
        [
            23, 231, 27, 9, 75, 177, 179, 234, 202, 55, 195, 150, 120, 172, 62, 156, 183, 43, 84,
            109, 173, 197, 148, 245, 133, 189, 127, 174, 82, 189, 115, 25,
        ],
        [
            9, 143, 201, 210, 39, 51, 154, 239, 84, 93, 95, 93, 3, 91, 159, 65, 0, 98, 184, 97,
            108, 24, 36, 170, 223, 25, 245, 14, 144, 196, 66, 47,
        ],
        [
            94, 162, 20, 133, 117, 34, 246, 81, 24, 117, 207, 237, 217, 57, 146, 137, 143, 215,
            159, 112, 85, 221, 211, 241, 61, 19, 20, 198, 223, 206, 104, 19,
        ],
        [
            89, 188, 251, 72, 124, 111, 97, 181, 107, 226, 217, 177, 180, 22, 182, 252, 46, 174,
            117, 232, 43, 159, 47, 17, 108, 190, 117, 228, 186, 121, 102, 56,
        ],
        [
            243, 167, 49, 44, 140, 139, 230, 208, 31, 131, 170, 189, 192, 4, 203, 255, 24, 221,
            194, 244, 109, 141, 22, 224, 138, 88, 234, 70, 99, 44, 158, 16,
        ],
        [
            136, 211, 151, 83, 245, 38, 217, 39, 240, 140, 51, 223, 35, 33, 225, 23, 89, 102, 73,
            255, 73, 0, 214, 171, 6, 199, 255, 162, 247, 240, 205, 26,
        ],
    ],
    [
        [
            254, 77, 97, 198, 201, 60, 159, 34, 206, 67, 175, 168, 187, 6, 194, 218, 205, 50, 36,
            161, 65, 69, 61, 160, 250, 92, 190, 125, 252, 88, 68, 25,
        ],
        [
            133, 196, 181, 170, 255, 139, 126, 115, 14, 8, 201, 55, 127, 0, 78, 159, 174, 128, 246,
            25, 157, 67, 55, 224, 177, 43, 162, 37, 70, 56, 4, 27,
        ],
        [
            112, 166, 24, 255, 18, 180, 244, 235, 31, 244, 66, 170, 197, 32, 253, 64, 16, 184, 86,
            163, 30, 235, 54, 178, 201, 177, 153, 71, 176, 227, 211, 26,
        ],
        [
            151, 222, 230, 164, 11, 133, 141, 108, 118, 190, 234, 99, 15, 33, 229, 82, 88, 14, 9,
            120, 72, 207, 4, 120, 64, 65, 41, 222, 46, 203, 18, 9,
        ],
        [
            103, 80, 114, 196, 74, 31, 54, 48, 10, 95, 29, 137, 192, 2, 117, 62, 170, 67, 233, 14,
            92, 90, 108, 161, 78, 71, 153, 237, 75, 227, 166, 5,
        ],
        [
            212, 150, 143, 103, 12, 222, 171, 219, 8, 32, 230, 183, 124, 27, 227, 126, 179, 47,
            222, 140, 182, 172, 225, 22, 227, 50, 79, 237, 228, 201, 201, 57,
        ],
        [
            70, 84, 206, 124, 223, 76, 35, 210, 59, 152, 102, 132, 156, 238, 248, 117, 141, 11, 8,
            13, 50, 184, 50, 10, 163, 255, 207, 54, 255, 18, 3, 15,
        ],
        [
            6, 252, 192, 97, 81, 78, 102, 178, 50, 96, 125, 90, 116, 56, 139, 196, 202, 88, 121,
            186, 27, 23, 112, 100, 105, 28, 30, 44, 39, 83, 71, 14,
        ],
    ],
    [
        [
            251, 94, 232, 178, 20, 76, 56, 34, 195, 179, 50, 154, 255, 237, 147, 119, 150, 15, 199,
            76, 170, 185, 174, 58, 52, 46, 50, 22, 155, 174, 2, 29,
        ],
        [
            213, 156, 33, 83, 188, 31, 83, 136, 209, 126, 89, 13, 129, 191, 217, 248, 192, 156,
            175, 74, 138, 54, 211, 158, 215, 21, 159, 116, 54, 167, 90, 3,
        ],
        [
            144, 92, 139, 79, 131, 242, 162, 182, 206, 192, 184, 68, 136, 217, 253, 55, 94, 203,
            200, 248, 114, 55, 235, 153, 179, 204, 123, 113, 118, 90, 213, 59,
        ],
        [
            91, 135, 225, 99, 39, 50, 171, 68, 85, 153, 75, 163, 52, 245, 81, 102, 116, 162, 178,
            207, 99, 156, 37, 226, 47, 164, 207, 174, 132, 132, 133, 60,
        ],
        [
            54, 214, 3, 118, 0, 225, 55, 117, 205, 163, 115, 249, 200, 246, 140, 108, 76, 169, 50,
            122, 155, 118, 18, 154, 88, 129, 196, 0, 233, 169, 115, 0,
        ],
        [
            124, 176, 25, 175, 103, 157, 153, 64, 64, 22, 197, 54, 192, 232, 233, 212, 146, 16,
            183, 52, 141, 90, 80, 128, 237, 67, 145, 229, 89, 9, 192, 46,
        ],
        [
            213, 174, 134, 198, 210, 44, 68, 2, 36, 211, 34, 63, 65, 177, 47, 211, 85, 214, 83, 15,
            158, 245, 58, 134, 23, 194, 25, 245, 254, 120, 158, 15,
        ],
        [
            81, 6, 176, 10, 91, 61, 188, 113, 152, 41, 236, 6, 95, 20, 71, 148, 23, 21, 40, 26,
            240, 110, 136, 133, 26, 121, 121, 214, 161, 97, 185, 30,
        ],
    ],
    [
        [
            71, 167, 42, 102, 126, 239, 68, 9, 111, 196, 137, 241, 163, 6, 156, 107, 138, 134, 74,
            132, 121, 33, 184, 172, 186, 130, 126, 62, 237, 118, 242, 26,
        ],
        [
            130, 92, 111, 138, 18, 54, 180, 172, 235, 236, 195, 254, 31, 119, 107, 236, 128, 170,
            143, 88, 90, 255, 23, 114, 193, 147, 139, 82, 215, 223, 190, 60,
        ],
        [
            85, 115, 103, 249, 139, 252, 172, 223, 249, 30, 175, 150, 183, 241, 226, 250, 69, 42,
            207, 248, 102, 215, 134, 31, 198, 184, 222, 211, 172, 176, 22, 10,
        ],
        [
            124, 104, 228, 185, 245, 31, 226, 5, 192, 123, 102, 59, 179, 106, 171, 111, 10, 29, 79,
            223, 174, 80, 224, 180, 71, 96, 52, 20, 213, 116, 245, 4,
        ],
        [
            204, 7, 183, 204, 52, 91, 161, 202, 255, 249, 153, 146, 3, 171, 245, 206, 34, 133, 131,
            3, 52, 31, 56, 253, 203, 81, 176, 126, 113, 38, 164, 52,
        ],
        [
            170, 200, 186, 32, 27, 90, 197, 215, 251, 122, 225, 34, 45, 175, 154, 199, 161, 24,
            103, 6, 195, 12, 138, 54, 210, 118, 197, 114, 60, 16, 192, 6,
        ],
        [
            168, 15, 89, 40, 153, 35, 233, 107, 133, 214, 104, 196, 21, 161, 5, 77, 119, 71, 1,
            140, 212, 160, 104, 252, 2, 166, 70, 57, 231, 28, 116, 4,
        ],
        [
            228, 252, 183, 21, 218, 84, 226, 14, 150, 183, 222, 17, 159, 46, 189, 140, 36, 60, 42,
            103, 42, 207, 72, 169, 51, 212, 213, 209, 67, 129, 17, 17,
        ],
    ],
    [
        [
            229, 94, 4, 4, 255, 10, 228, 111, 228, 106, 39, 196, 52, 15, 132, 47, 69, 132, 17, 208,
            13, 31, 167, 122, 184, 4, 212, 237, 111, 253, 11, 21,
        ],
        [
            53, 100, 74, 54, 153, 221, 88, 228, 247, 9, 37, 119, 44, 212, 219, 189, 28, 56, 188,
            198, 217, 1, 176, 232, 60, 123, 225, 127, 139, 193, 6, 31,
        ],
        [
            186, 168, 160, 166, 117, 235, 200, 114, 87, 89, 231, 247, 221, 235, 74, 28, 189, 19,
            28, 250, 46, 18, 120, 213, 193, 165, 126, 113, 134, 33, 32, 1,
        ],
        [
            50, 59, 63, 152, 85, 72, 221, 20, 198, 176, 87, 118, 81, 128, 237, 174, 255, 75, 155,
            53, 225, 109, 108, 226, 71, 88, 113, 119, 79, 43, 101, 38,
        ],
        [
            247, 92, 39, 118, 214, 109, 43, 18, 244, 113, 180, 251, 7, 111, 140, 206, 180, 97, 232,
            118, 86, 133, 251, 155, 26, 165, 139, 163, 215, 90, 77, 32,
        ],
        [
            149, 133, 199, 150, 17, 179, 149, 210, 101, 143, 83, 108, 98, 253, 84, 88, 141, 25,
            138, 95, 17, 3, 61, 100, 3, 78, 192, 208, 117, 129, 2, 6,
        ],
        [
            91, 178, 133, 154, 63, 174, 196, 94, 152, 232, 192, 125, 37, 28, 44, 110, 38, 62, 21,
            218, 235, 32, 32, 202, 98, 230, 161, 4, 128, 239, 125, 12,
        ],
        [
            151, 89, 41, 182, 150, 178, 27, 24, 82, 77, 206, 134, 252, 28, 50, 228, 213, 182, 197,
            209, 49, 213, 175, 72, 189, 253, 249, 88, 108, 236, 168, 58,
        ],
    ],
    [
        [
            38, 21, 85, 239, 52, 220, 50, 246, 95, 229, 244, 243, 173, 123, 102, 19, 164, 131, 228,
            132, 241, 24, 151, 136, 224, 25, 54, 189, 80, 222, 246, 22,
        ],
        [
            244, 175, 133, 128, 140, 88, 63, 118, 85, 87, 53, 206, 87, 43, 61, 129, 118, 243, 9,
            191, 248, 248, 70, 141, 80, 49, 149, 101, 197, 108, 2, 16,
        ],
        [
            228, 89, 135, 108, 54, 96, 202, 77, 40, 125, 27, 170, 68, 170, 35, 209, 21, 253, 54,
            24, 198, 125, 164, 90, 226, 96, 191, 107, 23, 182, 237, 44,
        ],
        [
            60, 224, 131, 3, 20, 51, 245, 142, 136, 95, 102, 66, 216, 168, 75, 59, 218, 9, 12, 179,
            110, 203, 136, 209, 124, 209, 123, 238, 155, 141, 119, 30,
        ],
        [
            56, 124, 170, 94, 110, 39, 171, 145, 126, 39, 53, 218, 222, 52, 177, 152, 181, 192, 92,
            212, 138, 130, 233, 100, 78, 160, 236, 36, 101, 9, 253, 47,
        ],
        [
            73, 38, 150, 82, 121, 254, 45, 212, 3, 18, 24, 130, 113, 56, 74, 110, 9, 206, 34, 91,
            223, 241, 108, 114, 238, 211, 171, 30, 137, 150, 59, 0,
        ],
        [
            72, 66, 80, 129, 28, 93, 228, 162, 207, 163, 104, 93, 73, 170, 53, 116, 34, 97, 34, 28,
            147, 33, 119, 42, 3, 2, 144, 113, 205, 233, 98, 6,
        ],
        [
            152, 136, 201, 11, 112, 141, 110, 36, 28, 222, 78, 38, 88, 212, 172, 192, 168, 212,
            248, 176, 243, 215, 39, 54, 166, 119, 60, 182, 84, 44, 132, 1,
        ],
    ],
    [
        [
            166, 20, 55, 181, 238, 166, 209, 247, 197, 203, 233, 189, 89, 232, 139, 118, 203, 26,
            245, 159, 234, 187, 122, 104, 95, 200, 221, 233, 216, 232, 137, 44,
        ],
        [
            149, 226, 202, 252, 150, 254, 162, 47, 186, 188, 250, 215, 65, 25, 65, 199, 156, 231,
            254, 234, 86, 188, 133, 32, 37, 32, 138, 174, 186, 151, 23, 59,
        ],
        [
            123, 18, 173, 246, 115, 145, 29, 164, 82, 83, 79, 117, 140, 85, 179, 255, 80, 78, 124,
            154, 179, 168, 22, 216, 87, 39, 12, 224, 219, 14, 237, 47,
        ],
        [
            198, 76, 74, 220, 44, 216, 61, 154, 238, 101, 198, 27, 202, 26, 80, 54, 22, 209, 105,
            210, 163, 61, 69, 57, 165, 232, 132, 92, 116, 105, 213, 62,
        ],
        [
            184, 32, 212, 62, 67, 44, 136, 203, 226, 214, 152, 150, 241, 162, 174, 69, 254, 15, 17,
            28, 234, 162, 154, 43, 55, 57, 48, 146, 13, 170, 146, 36,
        ],
        [
            40, 10, 127, 190, 248, 155, 217, 21, 147, 185, 101, 235, 133, 96, 152, 7, 89, 74, 127,
            120, 205, 71, 181, 117, 141, 158, 100, 225, 86, 35, 84, 30,
        ],
        [
            59, 0, 26, 9, 117, 171, 252, 142, 194, 159, 19, 148, 53, 2, 243, 200, 116, 95, 59, 43,
            156, 118, 8, 146, 31, 235, 125, 67, 226, 193, 192, 14,
        ],
        [
            152, 174, 144, 126, 124, 71, 171, 85, 237, 165, 158, 214, 209, 62, 250, 85, 162, 49,
            129, 79, 191, 21, 192, 223, 222, 118, 173, 150, 164, 212, 184, 41,
        ],
    ],
    [
        [
            217, 100, 143, 202, 28, 1, 60, 244, 252, 12, 56, 235, 162, 250, 194, 0, 158, 25, 148,
            62, 226, 184, 144, 86, 154, 54, 77, 122, 47, 47, 37, 47,
        ],
        [
            202, 69, 190, 148, 247, 74, 31, 98, 96, 125, 31, 19, 13, 131, 46, 232, 247, 75, 226,
            211, 35, 173, 66, 190, 8, 160, 57, 35, 181, 203, 33, 57,
        ],
        [
            105, 168, 39, 76, 17, 201, 8, 85, 160, 1, 108, 17, 65, 17, 143, 13, 26, 237, 93, 205,
            40, 188, 41, 247, 82, 200, 105, 236, 20, 96, 99, 16,
        ],
        [
            8, 225, 80, 33, 92, 253, 80, 149, 76, 124, 194, 200, 247, 158, 230, 212, 154, 151, 45,
            60, 189, 91, 94, 248, 13, 166, 178, 71, 207, 40, 252, 34,
        ],
        [
            112, 102, 155, 6, 178, 250, 238, 207, 237, 193, 217, 248, 147, 24, 131, 76, 225, 178,
            237, 111, 231, 44, 13, 218, 28, 166, 69, 107, 158, 18, 239, 14,
        ],
        [
            9, 175, 91, 204, 143, 242, 215, 115, 39, 213, 85, 208, 155, 76, 228, 114, 126, 170,
            228, 103, 118, 64, 196, 226, 35, 184, 202, 195, 205, 36, 213, 24,
        ],
        [
            121, 166, 135, 131, 76, 123, 114, 93, 221, 160, 197, 146, 126, 212, 52, 241, 95, 56,
            127, 65, 155, 66, 25, 216, 43, 42, 163, 49, 131, 212, 76, 7,
        ],
        [
            99, 210, 39, 55, 89, 68, 174, 153, 25, 69, 37, 70, 180, 15, 215, 92, 30, 95, 85, 74,
            115, 111, 15, 166, 74, 203, 79, 168, 21, 176, 80, 27,
        ],
    ],
    [
        [
            57, 146, 226, 221, 107, 225, 126, 141, 182, 27, 144, 50, 73, 165, 21, 228, 91, 125, 53,
            198, 13, 183, 251, 163, 208, 52, 135, 192, 136, 144, 143, 33,
        ],
        [
            123, 7, 46, 76, 70, 183, 31, 50, 102, 73, 52, 69, 42, 212, 19, 239, 207, 121, 226, 48,
            7, 124, 101, 156, 27, 108, 250, 115, 203, 80, 158, 54,
        ],
        [
            105, 12, 215, 144, 236, 236, 57, 169, 134, 16, 123, 228, 143, 246, 201, 34, 194, 94,
            87, 66, 109, 43, 155, 104, 47, 0, 200, 97, 218, 162, 17, 13,
        ],
        [
            124, 237, 179, 116, 163, 119, 114, 172, 236, 205, 105, 103, 1, 74, 29, 243, 205, 244,
            232, 249, 52, 251, 59, 108, 46, 186, 212, 23, 85, 250, 38, 2,
        ],
        [
            169, 136, 44, 150, 8, 97, 227, 243, 193, 218, 181, 203, 176, 162, 70, 93, 239, 130,
            230, 67, 9, 57, 26, 124, 59, 229, 159, 92, 24, 98, 77, 53,
        ],
        [
            81, 1, 244, 212, 4, 189, 139, 212, 63, 219, 239, 218, 254, 209, 167, 118, 214, 108, 63,
            241, 3, 215, 184, 41, 148, 192, 183, 121, 48, 82, 1, 25,
        ],
        [
            77, 157, 101, 207, 121, 249, 208, 27, 219, 162, 235, 48, 152, 43, 15, 71, 93, 19, 134,
            30, 228, 115, 246, 243, 146, 135, 193, 55, 199, 21, 26, 58,
        ],
        [
            158, 13, 107, 51, 93, 172, 190, 196, 224, 143, 126, 198, 215, 92, 139, 136, 230, 209,
            26, 247, 99, 138, 255, 219, 115, 248, 137, 42, 148, 138, 164, 35,
        ],
    ],
    [
        [
            144, 129, 1, 246, 175, 28, 39, 247, 112, 135, 137, 167, 136, 48, 66, 164, 151, 73, 8,
            241, 210, 54, 55, 165, 84, 252, 199, 16, 160, 37, 145, 23,
        ],
        [
            192, 125, 70, 179, 34, 242, 184, 201, 4, 245, 132, 241, 154, 195, 44, 156, 117, 250,
            176, 65, 83, 229, 1, 161, 53, 116, 194, 235, 72, 95, 130, 10,
        ],
        [
            228, 8, 155, 222, 133, 218, 125, 194, 220, 72, 46, 118, 157, 179, 153, 160, 190, 219,
            166, 145, 183, 114, 184, 90, 206, 47, 43, 46, 190, 180, 243, 14,
        ],
        [
            144, 1, 212, 200, 34, 67, 61, 193, 101, 41, 199, 112, 121, 197, 30, 248, 172, 240, 1,
            23, 210, 116, 54, 82, 203, 153, 141, 24, 223, 253, 48, 25,
        ],
        [
            97, 162, 113, 178, 80, 172, 128, 13, 18, 247, 1, 132, 162, 145, 152, 16, 26, 44, 104,
            137, 200, 140, 57, 188, 181, 173, 186, 77, 241, 70, 103, 62,
        ],
        [
            98, 180, 95, 72, 251, 169, 134, 210, 225, 93, 191, 188, 89, 165, 113, 10, 20, 55, 249,
            100, 231, 105, 100, 74, 66, 30, 50, 169, 116, 236, 28, 25,
        ],
        [
            106, 166, 231, 140, 101, 154, 108, 162, 240, 137, 97, 132, 90, 169, 10, 153, 166, 136,
            135, 146, 102, 93, 177, 86, 64, 92, 168, 72, 31, 104, 20, 33,
        ],
        [
            188, 211, 184, 142, 84, 192, 128, 168, 155, 47, 14, 126, 83, 126, 152, 130, 211, 224,
            8, 217, 108, 82, 178, 193, 83, 42, 129, 40, 74, 255, 26, 21,
        ],
    ],
    [
        [
            27, 111, 120, 34, 70, 220, 74, 211, 225, 132, 62, 224, 106, 147, 107, 170, 65, 245,
            126, 162, 179, 236, 82, 239, 235, 57, 126, 146, 208, 181, 25, 25,
        ],
        [
            67, 51, 145, 136, 236, 14, 221, 242, 49, 186, 236, 69, 247, 188, 190, 22, 49, 49, 193,
            147, 62, 22, 207, 31, 233, 100, 182, 124, 70, 69, 83, 19,
        ],
        [
            25, 40, 159, 124, 230, 217, 113, 135, 193, 155, 204, 66, 147, 111, 238, 31, 231, 110,
            72, 68, 189, 205, 241, 130, 156, 50, 58, 137, 248, 179, 113, 34,
        ],
        [
            199, 7, 17, 175, 48, 117, 199, 19, 50, 252, 37, 72, 70, 122, 69, 90, 19, 57, 128, 68,
            235, 160, 101, 40, 62, 245, 236, 132, 63, 145, 10, 11,
        ],
        [
            238, 244, 156, 113, 5, 156, 173, 22, 95, 73, 65, 155, 53, 223, 225, 255, 58, 104, 112,
            89, 167, 166, 114, 249, 45, 217, 195, 168, 12, 31, 14, 5,
        ],
        [
            168, 30, 153, 147, 149, 139, 195, 4, 163, 70, 46, 36, 180, 109, 86, 197, 142, 11, 142,
            250, 72, 198, 51, 173, 202, 162, 153, 85, 2, 145, 110, 10,
        ],
        [
            36, 27, 187, 193, 56, 99, 159, 134, 253, 3, 229, 231, 249, 147, 129, 252, 156, 85, 84,
            57, 241, 140, 251, 174, 135, 138, 176, 221, 73, 222, 159, 14,
        ],
        [
            206, 186, 15, 163, 104, 0, 19, 188, 128, 226, 81, 186, 190, 214, 39, 110, 123, 9, 147,
            66, 66, 64, 78, 14, 58, 49, 165, 149, 192, 4, 67, 13,
        ],
    ],
    [
        [
            32, 250, 229, 202, 183, 206, 20, 75, 220, 120, 12, 190, 111, 12, 98, 111, 61, 146, 165,
            166, 133, 60, 94, 136, 61, 223, 154, 129, 90, 109, 230, 10,
        ],
        [
            147, 107, 125, 22, 133, 54, 201, 8, 20, 118, 236, 210, 106, 113, 137, 34, 237, 206,
            128, 160, 184, 214, 23, 43, 145, 48, 157, 54, 59, 100, 162, 8,
        ],
        [
            13, 212, 17, 152, 59, 216, 167, 254, 78, 26, 36, 129, 32, 206, 173, 158, 23, 4, 56,
            205, 192, 12, 1, 27, 227, 234, 12, 233, 104, 82, 182, 61,
        ],
        [
            7, 17, 169, 161, 14, 21, 162, 94, 106, 66, 217, 14, 218, 63, 127, 131, 90, 30, 175,
            208, 7, 209, 226, 5, 137, 240, 116, 204, 192, 76, 138, 51,
        ],
        [
            190, 206, 82, 115, 96, 94, 88, 62, 148, 244, 93, 133, 55, 16, 9, 115, 17, 170, 88, 254,
            150, 87, 214, 42, 60, 43, 247, 235, 111, 159, 60, 42,
        ],
        [
            253, 2, 221, 158, 160, 73, 222, 22, 213, 157, 119, 31, 16, 236, 85, 113, 78, 159, 158,
            85, 126, 31, 31, 136, 220, 205, 114, 13, 54, 102, 96, 10,
        ],
        [
            11, 138, 36, 180, 3, 0, 209, 232, 70, 200, 182, 250, 148, 237, 249, 181, 32, 214, 7,
            223, 165, 74, 141, 2, 237, 247, 107, 67, 252, 175, 58, 44,
        ],
        [
            228, 122, 212, 211, 34, 224, 37, 28, 89, 232, 71, 201, 243, 32, 242, 117, 54, 221, 62,
            243, 217, 189, 0, 215, 68, 28, 192, 112, 142, 49, 211, 12,
        ],
    ],
    [
        [
            146, 43, 218, 86, 28, 225, 144, 208, 46, 241, 190, 206, 28, 110, 180, 198, 132, 130,
            253, 202, 21, 111, 227, 246, 195, 55, 14, 125, 40, 137, 176, 30,
        ],
        [
            95, 71, 151, 226, 250, 85, 113, 233, 79, 7, 178, 203, 58, 47, 152, 144, 214, 1, 202,
            245, 234, 85, 58, 45, 235, 255, 6, 174, 119, 7, 46, 30,
        ],
        [
            164, 42, 64, 130, 86, 86, 18, 10, 84, 13, 223, 98, 150, 57, 219, 189, 41, 25, 54, 97,
            151, 202, 192, 203, 75, 104, 152, 221, 238, 70, 27, 45,
        ],
        [
            127, 13, 100, 237, 255, 54, 232, 43, 143, 73, 168, 109, 176, 49, 8, 120, 57, 130, 244,
            217, 26, 122, 135, 110, 64, 85, 16, 224, 79, 231, 138, 62,
        ],
        [
            194, 131, 166, 241, 159, 28, 76, 114, 253, 239, 213, 220, 254, 100, 77, 64, 91, 238,
            87, 44, 36, 163, 148, 132, 32, 1, 178, 89, 210, 240, 134, 48,
        ],
        [
            123, 225, 66, 102, 92, 204, 94, 145, 164, 245, 23, 107, 190, 226, 54, 77, 213, 179,
            125, 75, 186, 59, 70, 150, 179, 74, 227, 71, 147, 60, 139, 16,
        ],
        [
            39, 243, 105, 150, 174, 74, 30, 84, 233, 29, 218, 218, 73, 42, 20, 214, 172, 15, 236,
            147, 47, 157, 17, 153, 87, 209, 198, 9, 91, 84, 88, 35,
        ],
        [
            45, 154, 254, 153, 15, 16, 141, 166, 168, 13, 22, 76, 212, 147, 189, 192, 47, 5, 41,
            249, 28, 251, 60, 130, 52, 148, 200, 198, 95, 188, 221, 30,
        ],
    ],
    [
        [
            107, 243, 152, 171, 58, 95, 246, 5, 41, 34, 30, 145, 136, 116, 75, 1, 3, 253, 5, 12,
            117, 143, 210, 29, 168, 141, 105, 141, 77, 72, 157, 45,
        ],
        [
            35, 114, 206, 37, 84, 242, 230, 119, 141, 159, 167, 71, 206, 102, 55, 140, 204, 137,
            151, 85, 36, 29, 196, 49, 40, 43, 108, 83, 211, 90, 13, 33,
        ],
        [
            15, 177, 11, 152, 142, 9, 201, 105, 21, 183, 41, 239, 124, 235, 47, 81, 143, 50, 62,
            229, 121, 136, 146, 46, 59, 244, 139, 24, 170, 178, 240, 19,
        ],
        [
            165, 123, 207, 244, 153, 87, 143, 86, 122, 246, 136, 169, 101, 176, 217, 9, 5, 212,
            168, 27, 145, 3, 197, 154, 62, 69, 109, 182, 149, 23, 91, 21,
        ],
        [
            189, 137, 178, 165, 178, 87, 211, 192, 102, 167, 225, 100, 139, 160, 16, 87, 22, 204,
            105, 99, 131, 153, 108, 49, 34, 187, 49, 6, 185, 187, 30, 43,
        ],
        [
            27, 145, 78, 59, 46, 36, 132, 75, 120, 159, 51, 37, 37, 192, 108, 126, 73, 189, 174,
            82, 29, 111, 65, 80, 154, 218, 41, 64, 68, 29, 165, 30,
        ],
        [
            98, 187, 214, 47, 4, 74, 164, 184, 225, 129, 120, 133, 86, 231, 75, 237, 176, 49, 53,
            169, 90, 4, 18, 9, 167, 71, 22, 156, 13, 39, 15, 3,
        ],
        [
            112, 124, 173, 252, 97, 191, 76, 94, 83, 160, 143, 31, 249, 204, 154, 11, 144, 153, 84,
            26, 22, 65, 50, 243, 26, 43, 2, 117, 66, 61, 109, 33,
        ],
    ],
    [
        [
            48, 179, 105, 154, 137, 101, 207, 31, 159, 178, 200, 35, 0, 58, 168, 178, 174, 225, 29,
            180, 161, 43, 109, 228, 221, 204, 109, 67, 166, 195, 179, 63,
        ],
        [
            192, 220, 245, 60, 113, 224, 39, 140, 250, 15, 194, 10, 179, 68, 226, 120, 139, 19, 98,
            15, 83, 170, 245, 23, 120, 70, 92, 174, 64, 64, 106, 61,
        ],
        [
            224, 5, 162, 145, 63, 125, 41, 116, 214, 111, 234, 50, 80, 95, 211, 97, 214, 145, 238,
            22, 96, 42, 196, 79, 43, 20, 88, 179, 203, 22, 113, 46,
        ],
        [
            213, 81, 107, 202, 117, 25, 111, 182, 105, 138, 27, 113, 154, 75, 57, 246, 246, 83, 99,
            217, 179, 112, 69, 133, 173, 21, 93, 29, 205, 161, 76, 60,
        ],
        [
            53, 44, 98, 79, 115, 175, 162, 136, 175, 194, 206, 120, 151, 114, 188, 68, 142, 30, 96,
            184, 78, 24, 83, 6, 158, 185, 44, 71, 0, 93, 104, 0,
        ],
        [
            250, 21, 67, 252, 44, 115, 18, 82, 222, 64, 36, 75, 249, 107, 180, 166, 243, 36, 0,
            117, 175, 185, 120, 145, 155, 116, 188, 143, 17, 46, 165, 56,
        ],
        [
            241, 212, 155, 63, 65, 50, 226, 122, 207, 162, 225, 154, 169, 53, 14, 221, 96, 76, 136,
            178, 90, 164, 121, 103, 108, 127, 54, 163, 96, 85, 77, 6,
        ],
        [
            19, 175, 186, 97, 92, 236, 91, 61, 140, 4, 133, 53, 194, 213, 65, 250, 230, 238, 235,
            203, 46, 65, 35, 16, 182, 249, 176, 229, 175, 138, 30, 25,
        ],
    ],
    [
        [
            193, 173, 135, 142, 136, 208, 142, 206, 234, 104, 90, 172, 21, 141, 16, 215, 245, 110,
            59, 172, 92, 70, 107, 247, 27, 166, 143, 181, 155, 250, 106, 53,
        ],
        [
            40, 195, 37, 64, 237, 12, 144, 62, 106, 178, 195, 117, 222, 163, 229, 91, 210, 2, 145,
            237, 136, 209, 62, 140, 230, 8, 101, 56, 25, 178, 213, 58,
        ],
        [
            101, 129, 170, 93, 188, 56, 112, 107, 211, 10, 183, 62, 68, 200, 64, 130, 73, 142, 202,
            103, 69, 40, 144, 172, 242, 250, 22, 131, 198, 103, 108, 58,
        ],
        [
            194, 190, 28, 152, 159, 2, 136, 216, 228, 235, 50, 39, 214, 18, 140, 207, 80, 219, 220,
            101, 153, 79, 16, 105, 188, 49, 203, 108, 66, 225, 90, 59,
        ],
        [
            153, 64, 128, 222, 2, 153, 45, 105, 116, 220, 68, 127, 162, 151, 86, 187, 167, 170, 11,
            233, 28, 199, 145, 241, 243, 234, 10, 120, 92, 219, 209, 43,
        ],
        [
            50, 89, 222, 249, 125, 86, 27, 111, 222, 37, 116, 131, 25, 18, 181, 68, 48, 103, 126,
            135, 57, 250, 15, 250, 177, 17, 31, 129, 239, 252, 162, 28,
        ],
        [
            114, 163, 205, 167, 236, 34, 31, 190, 190, 105, 34, 205, 130, 240, 231, 205, 50, 231,
            88, 79, 27, 132, 231, 176, 11, 46, 129, 157, 254, 249, 246, 15,
        ],
        [
            93, 238, 47, 126, 190, 204, 194, 137, 218, 143, 52, 16, 199, 227, 130, 21, 153, 75,
            233, 47, 222, 61, 182, 45, 223, 200, 222, 178, 182, 170, 228, 31,
        ],
    ],
    [
        [
            12, 17, 186, 64, 81, 161, 220, 198, 174, 199, 6, 92, 215, 23, 232, 34, 246, 12, 107,
            125, 15, 107, 2, 242, 20, 8, 246, 125, 121, 99, 80, 3,
        ],
        [
            89, 57, 179, 171, 146, 205, 229, 64, 221, 171, 206, 90, 241, 174, 37, 148, 197, 159,
            18, 100, 219, 205, 52, 254, 192, 164, 85, 30, 65, 213, 210, 54,
        ],
        [
            62, 137, 99, 6, 120, 112, 106, 227, 209, 75, 212, 194, 210, 14, 2, 30, 195, 242, 218,
            76, 57, 40, 167, 185, 142, 96, 69, 77, 167, 67, 209, 9,
        ],
        [
            180, 9, 228, 211, 34, 32, 178, 132, 120, 119, 122, 62, 191, 119, 141, 172, 218, 172,
            109, 180, 94, 64, 162, 121, 52, 14, 121, 66, 59, 109, 245, 59,
        ],
        [
            249, 3, 248, 71, 44, 145, 178, 67, 137, 38, 251, 235, 157, 68, 254, 171, 27, 162, 133,
            219, 234, 216, 112, 152, 25, 58, 135, 251, 66, 71, 156, 26,
        ],
        [
            95, 35, 211, 78, 202, 25, 45, 202, 135, 35, 113, 93, 165, 64, 194, 144, 175, 17, 228,
            210, 224, 254, 58, 112, 227, 81, 126, 97, 86, 65, 29, 62,
        ],
        [
            221, 226, 149, 102, 167, 18, 52, 77, 5, 186, 64, 11, 100, 216, 208, 10, 248, 189, 34,
            212, 242, 213, 9, 142, 115, 97, 101, 65, 220, 76, 136, 18,
        ],
        [
            46, 234, 59, 13, 34, 48, 213, 65, 235, 58, 247, 80, 169, 39, 199, 103, 208, 23, 231,
            159, 190, 5, 184, 75, 56, 226, 248, 207, 69, 148, 211, 39,
        ],
    ],
    [
        [
            32, 230, 133, 110, 165, 142, 6, 236, 172, 169, 47, 58, 149, 196, 9, 122, 18, 15, 198,
            9, 254, 57, 37, 7, 223, 227, 136, 12, 203, 242, 43, 24,
        ],
        [
            42, 147, 110, 109, 50, 65, 17, 222, 30, 153, 56, 60, 204, 134, 182, 170, 16, 57, 253,
            161, 24, 121, 0, 24, 141, 195, 159, 253, 145, 154, 147, 9,
        ],
        [
            14, 162, 52, 18, 158, 156, 230, 183, 175, 194, 121, 255, 238, 127, 155, 250, 246, 125,
            169, 227, 72, 68, 0, 27, 232, 41, 17, 161, 76, 66, 20, 53,
        ],
        [
            50, 22, 182, 33, 162, 102, 95, 212, 128, 44, 7, 71, 26, 32, 114, 117, 49, 127, 40, 178,
            231, 245, 1, 106, 238, 134, 100, 231, 254, 238, 176, 5,
        ],
        [
            50, 237, 91, 136, 123, 1, 96, 84, 250, 240, 150, 12, 248, 226, 90, 37, 68, 177, 48,
            242, 94, 102, 174, 114, 222, 126, 153, 41, 195, 54, 126, 56,
        ],
        [
            133, 193, 10, 246, 10, 82, 239, 122, 163, 93, 110, 94, 60, 44, 177, 207, 189, 174, 87,
            89, 69, 129, 226, 121, 140, 169, 190, 105, 76, 58, 86, 8,
        ],
        [
            23, 43, 215, 223, 60, 59, 224, 124, 151, 95, 219, 161, 57, 159, 207, 111, 242, 135, 40,
            239, 253, 77, 19, 184, 64, 186, 150, 176, 66, 238, 67, 40,
        ],
        [
            126, 167, 187, 171, 240, 55, 84, 223, 134, 218, 13, 244, 77, 58, 112, 57, 26, 208, 108,
            78, 36, 188, 140, 87, 207, 217, 22, 144, 157, 20, 32, 54,
        ],
    ],
    [
        [
            93, 37, 10, 228, 50, 61, 16, 189, 13, 195, 139, 25, 160, 47, 200, 124, 186, 134, 98,
            124, 16, 208, 15, 234, 76, 94, 7, 231, 28, 219, 8, 55,
        ],
        [
            240, 37, 251, 82, 107, 129, 109, 226, 56, 158, 226, 245, 117, 81, 14, 88, 89, 159, 94,
            112, 221, 109, 148, 109, 96, 111, 107, 193, 65, 177, 88, 61,
        ],
        [
            66, 195, 4, 90, 120, 185, 19, 112, 65, 178, 63, 160, 49, 116, 10, 220, 176, 189, 95,
            102, 60, 97, 194, 253, 126, 162, 104, 217, 4, 50, 147, 37,
        ],
        [
            238, 163, 121, 161, 177, 233, 63, 78, 235, 119, 237, 197, 121, 164, 83, 212, 6, 208,
            230, 24, 249, 141, 95, 163, 28, 102, 175, 11, 75, 81, 160, 0,
        ],
        [
            229, 143, 0, 254, 88, 138, 193, 137, 148, 92, 219, 194, 181, 122, 207, 15, 16, 193,
            188, 217, 68, 31, 11, 158, 167, 27, 64, 136, 203, 117, 161, 12,
        ],
        [
            11, 49, 214, 250, 109, 158, 144, 42, 1, 206, 122, 94, 165, 157, 202, 213, 96, 180, 33,
            20, 22, 177, 7, 163, 28, 139, 104, 251, 198, 9, 214, 33,
        ],
        [
            116, 243, 168, 236, 137, 191, 154, 52, 41, 230, 77, 98, 8, 184, 250, 38, 141, 230, 158,
            5, 39, 9, 89, 103, 230, 79, 166, 50, 205, 4, 255, 57,
        ],
        [
            48, 98, 65, 66, 216, 38, 51, 172, 142, 86, 116, 5, 120, 215, 156, 152, 106, 238, 101,
            242, 53, 120, 56, 97, 38, 217, 38, 74, 226, 28, 214, 1,
        ],
    ],
    [
        [
            105, 231, 75, 196, 42, 78, 208, 145, 101, 74, 247, 42, 206, 96, 9, 165, 101, 89, 102,
            246, 47, 7, 215, 185, 122, 16, 186, 27, 10, 246, 28, 23,
        ],
        [
            217, 126, 103, 98, 167, 172, 142, 251, 248, 148, 70, 136, 6, 254, 190, 127, 239, 148,
            0, 199, 6, 165, 124, 27, 43, 237, 41, 234, 88, 83, 203, 32,
        ],
        [
            107, 239, 45, 27, 57, 6, 26, 100, 184, 166, 49, 44, 51, 34, 71, 119, 233, 175, 17, 249,
            180, 107, 41, 96, 188, 234, 154, 63, 206, 106, 120, 38,
        ],
        [
            18, 121, 183, 245, 156, 149, 212, 126, 222, 16, 47, 234, 70, 173, 22, 213, 183, 77, 82,
            53, 36, 126, 75, 154, 210, 154, 27, 140, 64, 60, 177, 57,
        ],
        [
            129, 228, 124, 136, 140, 181, 45, 123, 137, 7, 141, 185, 55, 235, 55, 18, 222, 8, 68,
            17, 115, 177, 15, 70, 229, 2, 114, 211, 234, 30, 197, 41,
        ],
        [
            228, 45, 55, 6, 192, 96, 178, 131, 218, 145, 217, 97, 91, 39, 160, 64, 227, 19, 66,
            189, 251, 188, 240, 67, 146, 48, 200, 76, 18, 250, 8, 30,
        ],
        [
            220, 15, 205, 81, 21, 46, 62, 121, 239, 193, 97, 108, 94, 74, 148, 33, 244, 134, 30,
            132, 132, 215, 205, 182, 150, 14, 115, 31, 182, 72, 94, 20,
        ],
        [
            69, 231, 138, 117, 173, 221, 41, 131, 254, 155, 191, 117, 134, 185, 244, 239, 223, 169,
            119, 108, 64, 191, 142, 111, 73, 108, 160, 106, 155, 102, 205, 46,
        ],
    ],
    [
        [
            216, 228, 186, 25, 141, 114, 50, 180, 248, 77, 180, 135, 54, 130, 143, 37, 195, 92, 84,
            224, 181, 77, 69, 208, 172, 57, 221, 164, 6, 233, 98, 49,
        ],
        [
            169, 136, 52, 59, 237, 211, 56, 94, 189, 86, 149, 191, 67, 97, 7, 175, 106, 166, 35,
            82, 86, 61, 133, 188, 53, 17, 216, 207, 65, 229, 194, 2,
        ],
        [
            255, 84, 1, 221, 103, 57, 196, 217, 196, 185, 158, 106, 101, 162, 235, 202, 135, 134,
            88, 167, 150, 148, 218, 54, 255, 108, 49, 166, 102, 176, 84, 30,
        ],
        [
            92, 224, 203, 243, 67, 151, 137, 209, 206, 249, 185, 243, 233, 175, 220, 244, 175, 1,
            122, 2, 95, 243, 93, 187, 239, 23, 255, 69, 41, 120, 38, 38,
        ],
        [
            128, 21, 223, 166, 79, 166, 187, 242, 137, 152, 56, 128, 243, 30, 183, 245, 180, 123,
            32, 118, 207, 24, 147, 57, 157, 179, 123, 151, 104, 69, 46, 52,
        ],
        [
            68, 43, 112, 45, 194, 157, 39, 211, 168, 167, 179, 209, 157, 101, 27, 43, 230, 76, 176,
            78, 232, 203, 95, 253, 116, 237, 107, 32, 48, 158, 58, 50,
        ],
        [
            100, 239, 210, 82, 79, 131, 253, 105, 235, 193, 69, 45, 187, 62, 95, 152, 54, 234, 64,
            114, 234, 169, 168, 87, 150, 35, 127, 246, 74, 17, 148, 2,
        ],
        [
            75, 112, 183, 215, 233, 223, 247, 91, 99, 90, 35, 216, 185, 170, 80, 82, 85, 71, 162,
            41, 97, 124, 189, 217, 53, 205, 136, 253, 224, 194, 79, 1,
        ],
    ],
    [
        [
            53, 21, 40, 125, 210, 154, 222, 229, 87, 242, 184, 188, 80, 173, 32, 106, 137, 190,
            140, 153, 255, 209, 126, 13, 117, 250, 37, 153, 245, 4, 245, 2,
        ],
        [
            228, 81, 243, 233, 86, 117, 4, 73, 124, 54, 11, 93, 190, 50, 53, 159, 145, 61, 152,
            111, 240, 113, 23, 252, 69, 88, 36, 226, 4, 109, 229, 3,
        ],
        [
            145, 141, 131, 86, 137, 206, 97, 36, 227, 136, 52, 93, 108, 105, 103, 81, 237, 131,
            235, 148, 131, 81, 85, 254, 247, 197, 252, 157, 10, 55, 29, 3,
        ],
        [
            103, 142, 61, 123, 169, 170, 61, 44, 87, 32, 62, 112, 84, 138, 187, 144, 188, 38, 202,
            94, 221, 200, 202, 244, 109, 138, 121, 56, 154, 45, 76, 0,
        ],
        [
            69, 127, 85, 137, 95, 27, 254, 15, 93, 42, 143, 67, 31, 140, 206, 8, 214, 151, 217, 69,
            145, 105, 169, 116, 238, 37, 55, 137, 215, 80, 158, 8,
        ],
        [
            22, 207, 101, 165, 241, 155, 248, 78, 60, 23, 127, 195, 251, 193, 217, 78, 162, 158,
            177, 146, 112, 183, 237, 32, 1, 138, 101, 44, 157, 172, 149, 36,
        ],
        [
            76, 137, 207, 192, 160, 232, 61, 242, 123, 239, 139, 70, 152, 11, 242, 137, 58, 104,
            228, 63, 142, 23, 137, 23, 37, 253, 40, 100, 183, 97, 87, 40,
        ],
        [
            34, 111, 225, 153, 243, 185, 42, 1, 25, 11, 157, 125, 204, 60, 23, 123, 187, 200, 70,
            228, 192, 25, 145, 43, 170, 247, 123, 157, 50, 180, 67, 12,
        ],
    ],
    [
        [
            139, 112, 152, 150, 233, 77, 13, 21, 174, 183, 70, 63, 135, 81, 128, 72, 144, 58, 49,
            65, 191, 20, 146, 19, 26, 255, 133, 93, 112, 124, 207, 62,
        ],
        [
            2, 63, 203, 234, 101, 157, 7, 25, 149, 165, 34, 204, 219, 25, 49, 15, 142, 143, 122,
            136, 92, 218, 149, 250, 192, 211, 144, 125, 34, 237, 151, 21,
        ],
        [
            188, 61, 75, 218, 124, 179, 96, 22, 138, 217, 109, 25, 125, 27, 24, 96, 239, 246, 167,
            201, 19, 9, 166, 15, 88, 21, 233, 112, 123, 235, 164, 21,
        ],
        [
            219, 142, 147, 199, 38, 55, 203, 214, 230, 204, 36, 59, 30, 239, 102, 148, 0, 108, 173,
            188, 32, 144, 135, 194, 254, 50, 3, 35, 201, 218, 230, 21,
        ],
        [
            68, 146, 51, 190, 170, 51, 35, 111, 126, 119, 58, 209, 251, 55, 62, 91, 158, 203, 122,
            233, 55, 43, 185, 64, 118, 142, 220, 200, 199, 254, 47, 41,
        ],
        [
            24, 58, 100, 185, 6, 101, 241, 244, 103, 176, 10, 237, 176, 190, 212, 66, 82, 150, 241,
            230, 149, 43, 245, 7, 25, 251, 229, 179, 221, 123, 159, 63,
        ],
        [
            28, 224, 32, 241, 13, 252, 30, 52, 182, 163, 237, 107, 166, 132, 135, 68, 78, 190, 35,
            235, 117, 126, 20, 138, 61, 223, 23, 145, 57, 74, 120, 4,
        ],
        [
            31, 99, 79, 38, 16, 74, 27, 159, 179, 143, 60, 173, 93, 70, 72, 38, 183, 41, 238, 168,
            71, 38, 4, 101, 78, 155, 224, 181, 93, 127, 35, 11,
        ],
    ],
    [
        [
            60, 137, 224, 53, 215, 197, 25, 63, 90, 33, 63, 146, 157, 176, 68, 250, 232, 147, 24,
            191, 26, 117, 147, 165, 101, 139, 54, 222, 124, 240, 79, 13,
        ],
        [
            134, 236, 133, 85, 112, 78, 54, 159, 209, 197, 57, 233, 145, 194, 146, 20, 103, 18,
            176, 88, 222, 218, 76, 231, 199, 242, 225, 215, 238, 66, 98, 55,
        ],
        [
            182, 198, 172, 117, 4, 72, 70, 255, 122, 154, 82, 83, 111, 6, 200, 7, 183, 198, 100,
            13, 210, 31, 60, 128, 75, 234, 201, 180, 70, 102, 30, 36,
        ],
        [
            153, 192, 32, 150, 124, 34, 86, 143, 118, 191, 78, 155, 197, 139, 103, 250, 187, 136,
            219, 207, 254, 39, 230, 186, 100, 89, 110, 143, 183, 188, 81, 49,
        ],
        [
            231, 119, 201, 218, 80, 86, 142, 21, 12, 24, 57, 147, 136, 40, 228, 215, 158, 224, 139,
            236, 214, 47, 59, 208, 17, 48, 224, 21, 169, 129, 224, 17,
        ],
        [
            100, 226, 103, 188, 57, 117, 183, 147, 141, 126, 101, 99, 181, 52, 203, 220, 157, 213,
            75, 185, 16, 251, 107, 65, 118, 234, 204, 254, 112, 123, 90, 5,
        ],
        [
            109, 233, 111, 207, 104, 219, 124, 6, 23, 121, 255, 210, 34, 191, 93, 83, 152, 185, 83,
            105, 217, 198, 181, 246, 226, 112, 155, 92, 188, 101, 111, 39,
        ],
        [
            179, 174, 143, 91, 160, 49, 243, 140, 151, 237, 93, 224, 177, 144, 185, 60, 240, 162,
            55, 226, 253, 59, 107, 70, 83, 152, 148, 243, 130, 180, 241, 17,
        ],
    ],
    [
        [
            1, 195, 47, 208, 103, 47, 214, 177, 87, 185, 90, 187, 162, 188, 124, 200, 242, 52, 134,
            234, 83, 113, 107, 123, 170, 90, 81, 163, 132, 119, 35, 14,
        ],
        [
            155, 171, 18, 109, 236, 115, 191, 12, 58, 167, 16, 164, 181, 122, 238, 96, 155, 233,
            253, 23, 109, 140, 48, 21, 65, 109, 224, 117, 176, 241, 254, 18,
        ],
        [
            70, 96, 64, 75, 143, 232, 226, 103, 52, 104, 0, 69, 83, 125, 202, 106, 161, 191, 177,
            102, 45, 23, 238, 204, 53, 40, 78, 59, 242, 35, 159, 19,
        ],
        [
            111, 81, 99, 34, 235, 124, 190, 219, 217, 172, 34, 122, 197, 157, 192, 78, 195, 143,
            238, 204, 77, 161, 107, 102, 74, 229, 67, 12, 117, 122, 246, 0,
        ],
        [
            144, 191, 125, 138, 68, 229, 183, 225, 230, 10, 142, 95, 44, 186, 213, 104, 243, 32,
            18, 104, 187, 179, 37, 101, 2, 120, 216, 48, 170, 4, 158, 38,
        ],
        [
            217, 133, 55, 67, 171, 207, 107, 237, 48, 161, 178, 178, 184, 86, 122, 8, 47, 50, 129,
            108, 208, 22, 31, 13, 123, 72, 240, 226, 193, 85, 76, 7,
        ],
        [
            143, 182, 97, 62, 51, 201, 49, 9, 30, 111, 130, 124, 184, 183, 215, 198, 43, 236, 200,
            126, 218, 175, 240, 182, 161, 238, 80, 220, 13, 1, 11, 36,
        ],
        [
            167, 187, 116, 148, 117, 166, 104, 241, 228, 188, 41, 158, 140, 164, 46, 222, 157, 136,
            217, 42, 198, 235, 34, 14, 130, 24, 192, 142, 220, 217, 48, 38,
        ],
    ],
    [
        [
            4, 57, 211, 98, 189, 189, 205, 184, 46, 220, 130, 21, 68, 2, 10, 29, 48, 61, 199, 226,
            206, 227, 52, 249, 60, 176, 140, 23, 177, 67, 92, 15,
        ],
        [
            181, 234, 40, 253, 161, 44, 200, 120, 241, 178, 215, 196, 222, 102, 210, 0, 182, 155,
            34, 174, 18, 66, 1, 124, 173, 154, 113, 56, 25, 13, 138, 18,
        ],
        [
            55, 134, 66, 34, 135, 181, 181, 9, 132, 109, 116, 206, 120, 178, 211, 47, 53, 148, 149,
            115, 173, 120, 236, 5, 58, 69, 176, 237, 73, 147, 106, 47,
        ],
        [
            158, 35, 158, 179, 82, 219, 196, 82, 60, 250, 48, 96, 3, 113, 101, 66, 35, 143, 232,
            102, 166, 223, 93, 32, 125, 45, 235, 133, 189, 37, 139, 18,
        ],
        [
            10, 247, 120, 27, 215, 183, 216, 124, 175, 48, 103, 247, 34, 165, 86, 118, 173, 252,
            154, 253, 25, 249, 106, 175, 73, 29, 109, 255, 8, 131, 131, 10,
        ],
        [
            157, 228, 177, 8, 79, 14, 148, 44, 112, 58, 0, 43, 124, 26, 238, 145, 31, 240, 212,
            214, 50, 56, 8, 37, 162, 19, 47, 237, 128, 152, 97, 12,
        ],
        [
            89, 136, 71, 53, 83, 53, 243, 10, 123, 130, 170, 24, 167, 121, 41, 40, 148, 89, 117,
            122, 123, 30, 110, 13, 240, 219, 58, 200, 65, 26, 61, 44,
        ],
        [
            156, 55, 245, 59, 198, 161, 112, 207, 14, 229, 66, 65, 138, 39, 231, 216, 200, 235, 25,
            219, 204, 38, 32, 167, 134, 247, 41, 188, 166, 147, 85, 55,
        ],
    ],
    [
        [
            233, 92, 75, 182, 82, 75, 137, 98, 145, 215, 248, 100, 63, 227, 162, 169, 113, 40, 221,
            59, 2, 250, 220, 112, 132, 129, 250, 177, 8, 101, 164, 8,
        ],
        [
            31, 38, 71, 251, 116, 153, 63, 32, 236, 124, 90, 46, 197, 165, 160, 131, 250, 189, 241,
            82, 157, 233, 201, 163, 118, 11, 216, 15, 33, 90, 101, 5,
        ],
        [
            82, 156, 93, 235, 57, 86, 36, 76, 10, 134, 151, 132, 254, 98, 48, 157, 111, 45, 76,
            195, 253, 218, 46, 2, 173, 142, 75, 218, 42, 54, 73, 15,
        ],
        [
            59, 182, 126, 198, 69, 64, 199, 105, 134, 90, 186, 27, 217, 143, 228, 145, 122, 150,
            91, 236, 213, 134, 172, 191, 182, 40, 246, 159, 31, 217, 88, 27,
        ],
        [
            196, 28, 79, 113, 123, 215, 117, 42, 34, 17, 152, 218, 135, 254, 95, 214, 11, 224, 139,
            61, 142, 110, 220, 197, 153, 144, 149, 233, 74, 146, 63, 7,
        ],
        [
            27, 38, 138, 60, 32, 242, 80, 37, 33, 219, 27, 181, 40, 238, 146, 248, 109, 102, 248,
            210, 191, 96, 45, 228, 120, 76, 244, 62, 177, 201, 55, 17,
        ],
        [
            81, 219, 130, 197, 159, 89, 150, 166, 78, 19, 4, 125, 2, 112, 255, 111, 2, 158, 160,
            81, 195, 82, 139, 20, 156, 54, 0, 128, 78, 241, 177, 46,
        ],
        [
            6, 236, 47, 125, 243, 52, 250, 71, 17, 75, 21, 146, 229, 56, 210, 203, 231, 210, 39,
            170, 202, 114, 14, 199, 67, 74, 203, 97, 155, 186, 171, 20,
        ],
    ],
    [
        [
            158, 124, 26, 65, 178, 248, 144, 218, 104, 25, 157, 123, 70, 74, 14, 224, 180, 141, 73,
            53, 170, 128, 16, 99, 206, 127, 195, 128, 130, 243, 78, 48,
        ],
        [
            36, 86, 7, 238, 107, 51, 95, 192, 105, 21, 3, 175, 244, 132, 71, 95, 176, 177, 37, 191,
            229, 238, 174, 51, 24, 167, 55, 176, 224, 205, 82, 43,
        ],
        [
            136, 128, 95, 139, 143, 114, 236, 145, 6, 225, 197, 17, 55, 175, 53, 58, 25, 247, 29,
            140, 117, 43, 202, 117, 245, 202, 62, 46, 137, 57, 77, 9,
        ],
        [
            127, 245, 39, 235, 242, 22, 144, 158, 100, 135, 219, 155, 91, 192, 73, 60, 225, 231,
            25, 84, 215, 52, 125, 43, 20, 45, 223, 178, 108, 59, 203, 16,
        ],
        [
            148, 178, 74, 27, 251, 145, 203, 17, 208, 119, 184, 84, 117, 0, 241, 206, 166, 222,
            187, 106, 157, 202, 125, 129, 14, 89, 175, 187, 23, 159, 126, 55,
        ],
        [
            114, 4, 75, 157, 83, 233, 5, 108, 158, 88, 53, 99, 8, 186, 204, 172, 120, 171, 248, 43,
            48, 207, 38, 111, 138, 225, 143, 83, 209, 105, 105, 37,
        ],
        [
            92, 59, 72, 113, 151, 252, 46, 138, 65, 212, 75, 197, 200, 119, 255, 6, 66, 114, 189,
            247, 23, 249, 124, 92, 47, 173, 213, 55, 128, 69, 145, 0,
        ],
        [
            22, 114, 108, 240, 146, 117, 172, 126, 46, 221, 36, 250, 171, 77, 187, 18, 38, 212,
            212, 52, 42, 251, 64, 90, 240, 227, 253, 15, 18, 125, 168, 1,
        ],
    ],
    [
        [
            186, 129, 70, 211, 209, 123, 234, 4, 142, 210, 52, 75, 0, 165, 181, 253, 213, 180, 1,
            26, 175, 236, 137, 95, 51, 205, 8, 153, 161, 113, 70, 48,
        ],
        [
            211, 79, 54, 21, 140, 133, 28, 234, 69, 33, 148, 67, 237, 83, 252, 208, 124, 149, 83,
            176, 134, 112, 148, 157, 239, 4, 185, 88, 111, 40, 62, 7,
        ],
        [
            180, 162, 213, 99, 227, 21, 244, 143, 151, 86, 246, 138, 9, 110, 92, 15, 219, 95, 149,
            64, 82, 58, 85, 201, 2, 168, 105, 198, 252, 15, 21, 33,
        ],
        [
            238, 242, 147, 203, 56, 47, 195, 175, 121, 150, 66, 19, 29, 125, 247, 37, 225, 233,
            248, 120, 250, 40, 194, 36, 124, 245, 20, 70, 213, 211, 179, 27,
        ],
        [
            242, 105, 72, 107, 136, 229, 105, 29, 230, 84, 124, 18, 45, 141, 155, 6, 97, 175, 95,
            95, 0, 4, 19, 166, 109, 241, 230, 239, 164, 149, 186, 52,
        ],
        [
            1, 82, 33, 143, 210, 61, 135, 113, 67, 115, 94, 12, 152, 184, 122, 63, 9, 113, 248, 26,
            121, 49, 198, 28, 239, 249, 37, 44, 227, 142, 11, 63,
        ],
        [
            141, 102, 219, 141, 89, 219, 109, 104, 45, 114, 10, 166, 184, 102, 70, 195, 113, 68,
            13, 96, 100, 28, 99, 64, 139, 172, 150, 189, 54, 35, 5, 16,
        ],
        [
            226, 23, 65, 200, 163, 177, 166, 155, 32, 24, 126, 141, 136, 199, 29, 214, 185, 111,
            48, 116, 38, 107, 66, 40, 42, 151, 106, 247, 65, 198, 12, 18,
        ],
    ],
    [
        [
            11, 57, 245, 10, 120, 212, 154, 224, 122, 18, 200, 9, 192, 66, 247, 130, 124, 226, 39,
            100, 91, 114, 249, 21, 221, 158, 71, 64, 87, 180, 147, 61,
        ],
        [
            37, 54, 176, 141, 253, 46, 101, 178, 149, 5, 94, 50, 150, 19, 8, 243, 238, 54, 66, 215,
            149, 62, 233, 236, 150, 146, 115, 42, 56, 15, 139, 52,
        ],
        [
            54, 70, 131, 252, 109, 145, 35, 237, 41, 156, 62, 251, 130, 172, 131, 215, 50, 13, 168,
            3, 149, 189, 158, 245, 40, 211, 35, 247, 190, 218, 68, 31,
        ],
        [
            97, 18, 186, 126, 155, 90, 103, 122, 159, 64, 119, 250, 254, 3, 125, 97, 78, 132, 97,
            20, 24, 36, 13, 6, 160, 106, 10, 38, 137, 196, 82, 59,
        ],
        [
            83, 183, 38, 131, 224, 55, 36, 131, 250, 50, 98, 62, 53, 31, 186, 3, 59, 193, 44, 178,
            167, 60, 7, 31, 17, 197, 225, 87, 77, 201, 175, 50,
        ],
        [
            166, 65, 53, 61, 2, 156, 230, 20, 187, 218, 252, 134, 254, 199, 189, 164, 61, 159, 23,
            37, 253, 230, 212, 38, 189, 174, 98, 199, 120, 113, 184, 46,
        ],
        [
            38, 212, 175, 135, 83, 126, 242, 192, 211, 177, 211, 133, 235, 210, 36, 196, 46, 174,
            42, 9, 38, 72, 180, 212, 25, 76, 60, 244, 160, 252, 226, 52,
        ],
        [
            9, 186, 189, 28, 93, 244, 73, 59, 139, 98, 224, 159, 137, 187, 244, 145, 248, 78, 122,
            176, 76, 77, 126, 35, 71, 70, 152, 33, 154, 40, 47, 8,
        ],
    ],
    [
        [
            94, 74, 53, 248, 123, 40, 164, 191, 101, 131, 134, 77, 92, 11, 113, 23, 119, 101, 67,
            184, 14, 138, 91, 152, 10, 50, 206, 226, 114, 50, 153, 38,
        ],
        [
            70, 24, 160, 108, 135, 70, 121, 130, 110, 115, 249, 98, 163, 92, 182, 121, 140, 170,
            43, 109, 119, 200, 93, 198, 144, 155, 40, 165, 116, 110, 83, 59,
        ],
        [
            13, 111, 207, 211, 162, 206, 0, 87, 64, 221, 186, 206, 200, 25, 249, 40, 163, 157, 217,
            108, 151, 40, 35, 149, 112, 144, 3, 223, 7, 239, 84, 27,
        ],
        [
            123, 7, 143, 59, 246, 209, 168, 75, 236, 100, 37, 50, 57, 242, 6, 74, 175, 206, 53,
            213, 100, 11, 124, 103, 34, 250, 34, 134, 171, 72, 96, 26,
        ],
        [
            60, 211, 255, 163, 224, 254, 242, 177, 84, 221, 102, 62, 71, 95, 142, 146, 101, 6, 167,
            153, 65, 139, 49, 248, 206, 27, 116, 138, 191, 104, 173, 61,
        ],
        [
            117, 248, 184, 111, 92, 89, 176, 208, 186, 103, 174, 104, 91, 39, 254, 211, 76, 235,
            186, 171, 140, 209, 163, 40, 128, 223, 238, 208, 174, 221, 7, 32,
        ],
        [
            15, 175, 163, 23, 221, 182, 136, 48, 98, 253, 118, 47, 232, 19, 69, 81, 113, 107, 230,
            69, 64, 87, 123, 226, 59, 240, 192, 34, 182, 24, 75, 54,
        ],
        [
            200, 246, 135, 47, 106, 173, 187, 66, 101, 71, 151, 84, 100, 193, 113, 156, 71, 60,
            224, 247, 238, 27, 214, 105, 25, 8, 215, 175, 211, 72, 152, 50,
        ],
    ],
    [
        [
            11, 177, 54, 174, 134, 45, 96, 29, 9, 32, 202, 41, 30, 87, 38, 235, 103, 175, 59, 65,
            34, 62, 230, 27, 51, 229, 143, 20, 250, 88, 49, 13,
        ],
        [
            87, 251, 224, 97, 90, 123, 60, 173, 91, 250, 215, 196, 210, 111, 58, 198, 29, 127, 17,
            192, 116, 130, 106, 223, 111, 179, 160, 19, 165, 77, 156, 35,
        ],
        [
            228, 124, 114, 35, 183, 22, 90, 239, 18, 23, 66, 210, 98, 54, 229, 152, 173, 58, 197,
            212, 179, 59, 46, 118, 71, 8, 205, 134, 221, 142, 49, 36,
        ],
        [
            97, 172, 167, 10, 48, 141, 55, 10, 206, 107, 10, 234, 141, 228, 243, 71, 219, 231, 255,
            127, 65, 87, 122, 172, 51, 255, 202, 73, 57, 233, 54, 56,
        ],
        [
            218, 201, 81, 10, 208, 235, 255, 195, 207, 199, 247, 55, 152, 186, 219, 173, 46, 254,
            116, 54, 25, 159, 121, 178, 6, 197, 64, 24, 165, 163, 254, 27,
        ],
        [
            199, 51, 104, 153, 21, 83, 213, 132, 151, 210, 221, 160, 191, 61, 32, 39, 116, 202,
            190, 68, 168, 8, 206, 34, 76, 101, 91, 56, 234, 195, 177, 36,
        ],
        [
            200, 156, 175, 110, 211, 162, 86, 81, 86, 181, 99, 37, 39, 70, 77, 2, 181, 145, 157,
            169, 145, 213, 47, 162, 60, 26, 123, 27, 69, 95, 111, 43,
        ],
        [
            30, 40, 191, 205, 195, 141, 30, 180, 227, 90, 168, 109, 254, 194, 197, 47, 47, 158,
            151, 92, 119, 93, 22, 48, 71, 132, 50, 156, 11, 56, 60, 24,
        ],
    ],
    [
        [
            97, 236, 10, 82, 54, 165, 11, 222, 119, 226, 142, 250, 85, 73, 71, 5, 130, 185, 152,
            10, 38, 191, 159, 228, 39, 47, 7, 123, 206, 29, 63, 38,
        ],
        [
            63, 93, 78, 222, 254, 85, 222, 188, 119, 75, 202, 250, 75, 37, 247, 183, 250, 248, 44,
            137, 207, 242, 186, 236, 175, 255, 203, 108, 22, 25, 86, 3,
        ],
        [
            66, 175, 216, 225, 118, 129, 106, 102, 143, 170, 241, 201, 21, 1, 14, 89, 249, 49, 143,
            24, 17, 12, 166, 116, 29, 109, 134, 183, 99, 174, 81, 37,
        ],
        [
            197, 95, 177, 171, 196, 1, 233, 30, 98, 64, 186, 21, 34, 102, 54, 128, 110, 217, 3, 49,
            99, 164, 192, 114, 104, 45, 7, 144, 94, 62, 16, 61,
        ],
        [
            98, 86, 153, 20, 228, 157, 201, 70, 53, 155, 58, 84, 87, 140, 55, 16, 52, 29, 87, 106,
            98, 147, 35, 71, 160, 18, 82, 216, 95, 244, 124, 45,
        ],
        [
            242, 37, 85, 138, 93, 10, 117, 237, 63, 244, 73, 17, 187, 70, 150, 137, 148, 180, 139,
            39, 229, 169, 1, 46, 243, 136, 170, 65, 51, 64, 34, 8,
        ],
        [
            106, 96, 176, 162, 206, 47, 28, 119, 20, 102, 105, 144, 240, 117, 54, 120, 21, 220, 87,
            50, 191, 53, 87, 35, 135, 229, 145, 33, 96, 101, 129, 38,
        ],
        [
            166, 50, 224, 123, 255, 237, 250, 219, 171, 90, 189, 230, 77, 15, 184, 248, 42, 245,
            240, 71, 209, 160, 244, 146, 72, 164, 139, 171, 242, 118, 62, 26,
        ],
    ],
    [
        [
            176, 240, 195, 230, 203, 40, 73, 83, 30, 77, 37, 58, 215, 19, 113, 147, 218, 57, 68,
            214, 129, 183, 139, 254, 131, 136, 216, 77, 185, 153, 140, 12,
        ],
        [
            174, 253, 75, 3, 63, 87, 254, 53, 165, 243, 14, 41, 65, 7, 178, 91, 20, 111, 153, 155,
            23, 89, 87, 37, 105, 23, 204, 147, 189, 113, 94, 2,
        ],
        [
            218, 122, 7, 174, 69, 151, 16, 100, 230, 16, 187, 104, 238, 221, 124, 137, 157, 148,
            222, 48, 86, 3, 71, 115, 64, 244, 81, 206, 205, 128, 41, 38,
        ],
        [
            105, 132, 64, 84, 221, 69, 102, 124, 171, 223, 164, 67, 156, 97, 234, 192, 157, 110,
            215, 207, 127, 75, 159, 90, 243, 153, 13, 168, 51, 78, 121, 21,
        ],
        [
            124, 100, 211, 110, 134, 174, 177, 107, 84, 71, 156, 178, 27, 156, 130, 141, 190, 112,
            114, 107, 8, 10, 227, 230, 169, 165, 46, 162, 126, 47, 219, 21,
        ],
        [
            174, 109, 63, 1, 208, 183, 47, 213, 241, 116, 48, 79, 112, 250, 39, 77, 205, 232, 183,
            125, 65, 147, 180, 37, 118, 143, 209, 157, 0, 165, 131, 35,
        ],
        [
            171, 239, 114, 71, 235, 47, 49, 6, 184, 165, 133, 4, 30, 222, 236, 244, 145, 233, 148,
            50, 228, 226, 223, 139, 224, 213, 38, 123, 117, 85, 230, 42,
        ],
        [
            153, 172, 236, 117, 32, 68, 60, 120, 241, 40, 227, 98, 64, 137, 114, 1, 21, 216, 57,
            244, 116, 137, 172, 13, 181, 160, 230, 67, 60, 188, 218, 58,
        ],
    ],
];

pub fn generator() -> pallas::Affine {
    pallas::Affine::from_xy(
        pallas::Base::from_repr(GENERATOR.0).unwrap(),
        pallas::Base::from_repr(GENERATOR.1).unwrap(),
    )
    .unwrap()
}

#[cfg(test)]
mod tests {
    use super::super::{COMMIT_IVK_PERSONALIZATION, NUM_WINDOWS};
    use super::*;
    use group::Curve;
    use halo2_gadgets::{
        ecc::chip::constants::{test_lagrange_coeffs, test_zs_and_us},
        primitives::sinsemilla::CommitDomain,
    };
    use pasta_curves::{arithmetic::CurveAffine, pallas};

    #[test]
    fn generator() {
        let domain = CommitDomain::new(COMMIT_IVK_PERSONALIZATION);
        let point = domain.R();
        let coords = point.to_affine().coordinates().unwrap();

        assert_eq!(*coords.x(), pallas::Base::from_repr(GENERATOR.0).unwrap());
        assert_eq!(*coords.y(), pallas::Base::from_repr(GENERATOR.1).unwrap());
    }

    #[test]
    fn lagrange_coeffs() {
        let base = super::generator();
        test_lagrange_coeffs(base, NUM_WINDOWS);
    }

    #[test]
    fn z() {
        let base = super::generator();
        test_zs_and_us(base, &Z, &U, NUM_WINDOWS);
    }
}
