use group::ff::PrimeField;
use pasta_curves::{arithmetic::CurveAffine, pallas};

/// The value commitment is used to check balance between inputs and outputs. The value is
/// placed over this generator.
pub const GENERATOR: ([u8; 32], [u8; 32]) = (
    [
        145, 90, 60, 136, 104, 198, 195, 14, 47, 128, 144, 238, 69, 215, 110, 64, 72, 32, 141, 234,
        91, 35, 102, 79, 187, 9, 164, 15, 85, 68, 244, 7,
    ],
    [
        202, 217, 114, 167, 200, 70, 47, 14, 70, 129, 146, 243, 241, 41, 146, 39, 194, 183, 111,
        228, 201, 44, 86, 33, 140, 98, 38, 175, 119, 103, 19, 36,
    ],
);

/// Full-width z-values for GENERATOR
pub const Z: [u64; super::NUM_WINDOWS] = [
    181916, 22148, 340526, 80718, 104958, 86894, 43381, 1060, 82130, 4741, 55897, 4304, 114469,
    20503, 25001, 62408, 52978, 35893, 72071, 154369, 67304, 7299, 27960, 42929, 51869, 89967,
    62210, 59433, 47868, 32536, 105000, 1546, 2116, 18717, 50694, 22864, 254428, 54966, 108762,
    46706, 65730, 45555, 7376, 50051, 24773, 74636, 44806, 23223, 78561, 50668, 7380, 13697,
    171970, 269484, 25534, 5098, 79584, 6889, 21432, 73095, 36745, 37350, 6274, 5179, 50216, 12007,
    44029, 88199, 70401, 14120, 19017, 2423, 26494, 34954, 126293, 167379, 136922, 45619, 30331,
    22632, 163228, 12997, 4461, 32320, 13430,
];

/// Full-width u-values for GENERATOR
pub const U: [[[u8; 32]; super::H]; super::NUM_WINDOWS] = [
    [
        [
            139, 239, 3, 113, 200, 111, 72, 118, 105, 23, 186, 243, 234, 10, 173, 186, 53, 143,
            244, 250, 24, 189, 161, 158, 43, 13, 143, 252, 82, 104, 150, 55,
        ],
        [
            102, 10, 45, 173, 93, 142, 120, 21, 105, 140, 183, 3, 198, 201, 198, 184, 15, 206, 177,
            131, 9, 25, 103, 232, 162, 26, 45, 202, 9, 125, 93, 49,
        ],
        [
            7, 184, 84, 133, 195, 104, 173, 245, 198, 202, 50, 197, 58, 26, 187, 116, 72, 224, 153,
            67, 200, 174, 32, 60, 195, 80, 152, 192, 131, 168, 208, 13,
        ],
        [
            62, 23, 97, 17, 76, 95, 147, 158, 244, 165, 111, 203, 162, 25, 80, 235, 63, 80, 45, 12,
            254, 236, 154, 135, 211, 75, 44, 122, 125, 197, 9, 41,
        ],
        [
            99, 80, 223, 192, 185, 128, 95, 248, 127, 225, 142, 107, 179, 40, 8, 112, 59, 60, 81,
            89, 87, 153, 126, 140, 190, 187, 139, 9, 24, 21, 104, 5,
        ],
        [
            39, 201, 225, 147, 38, 14, 83, 216, 168, 53, 60, 41, 194, 52, 37, 141, 20, 20, 10, 242,
            173, 169, 194, 72, 182, 95, 173, 186, 229, 115, 178, 37,
        ],
        [
            234, 29, 76, 124, 112, 42, 82, 57, 181, 205, 61, 90, 58, 224, 227, 21, 83, 58, 30, 216,
            161, 10, 104, 83, 34, 113, 192, 146, 141, 215, 219, 44,
        ],
        [
            218, 183, 114, 168, 64, 166, 144, 113, 219, 75, 64, 32, 252, 13, 10, 111, 244, 194, 79,
            70, 7, 209, 129, 66, 232, 180, 217, 248, 61, 191, 59, 51,
        ],
    ],
    [
        [
            96, 67, 69, 38, 122, 193, 223, 87, 223, 138, 202, 119, 235, 141, 70, 229, 166, 135,
            208, 39, 220, 143, 203, 102, 106, 99, 115, 39, 8, 60, 31, 30,
        ],
        [
            207, 136, 59, 50, 178, 209, 184, 137, 238, 170, 63, 147, 102, 56, 194, 30, 152, 87,
            165, 250, 203, 71, 154, 64, 41, 216, 239, 143, 9, 94, 33, 52,
        ],
        [
            158, 74, 177, 224, 26, 70, 115, 103, 211, 95, 237, 92, 72, 210, 216, 225, 165, 20, 251,
            241, 200, 140, 178, 182, 154, 65, 105, 144, 85, 65, 240, 46,
        ],
        [
            11, 99, 19, 189, 201, 13, 110, 245, 240, 219, 4, 55, 24, 81, 54, 2, 82, 54, 171, 29,
            203, 99, 234, 64, 175, 154, 87, 112, 249, 29, 20, 30,
        ],
        [
            213, 177, 148, 202, 251, 114, 75, 8, 198, 148, 44, 102, 185, 120, 226, 222, 251, 65,
            206, 38, 99, 180, 62, 13, 221, 122, 206, 148, 10, 212, 234, 4,
        ],
        [
            191, 60, 118, 185, 153, 148, 200, 78, 50, 10, 66, 97, 26, 79, 240, 136, 20, 115, 39,
            240, 166, 237, 87, 98, 223, 84, 197, 151, 230, 139, 235, 7,
        ],
        [
            170, 86, 142, 224, 231, 59, 184, 35, 237, 52, 13, 72, 108, 156, 227, 229, 214, 103,
            175, 139, 231, 148, 146, 51, 195, 157, 221, 189, 122, 27, 227, 37,
        ],
        [
            88, 29, 201, 124, 227, 97, 36, 193, 215, 180, 137, 50, 249, 137, 10, 118, 67, 200, 35,
            42, 181, 217, 213, 146, 210, 159, 47, 194, 165, 101, 241, 20,
        ],
    ],
    [
        [
            147, 4, 192, 237, 120, 223, 231, 71, 128, 237, 0, 110, 243, 11, 132, 152, 102, 150,
            212, 251, 145, 56, 21, 247, 78, 232, 88, 204, 26, 139, 75, 1,
        ],
        [
            242, 104, 68, 11, 78, 77, 207, 142, 185, 45, 95, 183, 81, 22, 8, 125, 247, 66, 152, 59,
            191, 118, 51, 215, 85, 0, 51, 170, 147, 239, 110, 17,
        ],
        [
            148, 204, 16, 59, 17, 174, 121, 72, 218, 232, 205, 215, 128, 255, 187, 112, 141, 188,
            30, 2, 176, 122, 78, 158, 147, 150, 228, 25, 253, 68, 254, 43,
        ],
        [
            30, 207, 11, 138, 80, 55, 51, 156, 139, 83, 138, 33, 115, 159, 197, 188, 79, 31, 104,
            111, 62, 80, 183, 143, 105, 189, 238, 79, 147, 212, 156, 11,
        ],
        [
            162, 59, 128, 0, 14, 229, 58, 101, 219, 135, 2, 93, 2, 254, 14, 43, 143, 219, 76, 121,
            233, 148, 54, 51, 227, 115, 38, 253, 95, 103, 191, 21,
        ],
        [
            117, 191, 41, 120, 209, 97, 234, 18, 169, 17, 7, 120, 210, 68, 139, 185, 133, 131, 43,
            106, 10, 107, 18, 21, 57, 8, 190, 228, 149, 37, 158, 51,
        ],
        [
            31, 174, 69, 213, 168, 110, 159, 179, 18, 94, 76, 220, 112, 128, 252, 184, 236, 100,
            185, 238, 191, 51, 204, 100, 107, 218, 101, 96, 106, 166, 70, 6,
        ],
        [
            123, 139, 4, 10, 146, 168, 123, 135, 46, 115, 8, 11, 195, 106, 68, 214, 136, 168, 177,
            1, 184, 13, 51, 185, 184, 187, 68, 26, 157, 183, 41, 62,
        ],
    ],
    [
        [
            153, 31, 98, 227, 163, 137, 0, 129, 133, 57, 178, 132, 239, 46, 204, 124, 73, 65, 29,
            216, 30, 37, 56, 132, 241, 220, 62, 78, 187, 62, 240, 45,
        ],
        [
            125, 116, 131, 117, 46, 193, 254, 93, 79, 76, 177, 57, 31, 200, 120, 132, 186, 53, 66,
            220, 163, 178, 5, 230, 188, 212, 210, 25, 188, 103, 93, 44,
        ],
        [
            49, 162, 67, 249, 87, 145, 89, 247, 245, 104, 208, 165, 219, 100, 16, 187, 142, 200,
            66, 47, 38, 77, 245, 3, 108, 74, 62, 106, 50, 208, 84, 52,
        ],
        [
            201, 221, 108, 129, 170, 111, 64, 73, 140, 255, 129, 243, 252, 87, 129, 226, 170, 210,
            141, 213, 121, 212, 34, 184, 139, 157, 198, 98, 40, 50, 160, 59,
        ],
        [
            238, 218, 145, 5, 148, 79, 125, 6, 93, 100, 166, 138, 241, 213, 11, 178, 96, 251, 113,
            138, 47, 97, 33, 222, 224, 129, 174, 130, 217, 164, 140, 29,
        ],
        [
            212, 178, 199, 211, 213, 183, 6, 187, 140, 241, 90, 125, 180, 76, 46, 158, 109, 136,
            103, 168, 66, 53, 177, 127, 32, 212, 197, 52, 144, 147, 136, 45,
        ],
        [
            103, 249, 188, 165, 128, 120, 137, 119, 158, 154, 30, 17, 38, 169, 19, 33, 53, 148, 37,
            218, 95, 85, 148, 115, 41, 250, 150, 226, 101, 212, 163, 30,
        ],
        [
            230, 249, 111, 35, 125, 100, 99, 9, 198, 41, 46, 115, 83, 8, 31, 196, 164, 188, 94, 8,
            109, 175, 164, 169, 64, 5, 78, 121, 162, 246, 59, 53,
        ],
    ],
    [
        [
            64, 172, 254, 150, 194, 0, 229, 126, 227, 156, 144, 238, 66, 221, 251, 17, 212, 189,
            75, 173, 183, 141, 105, 90, 211, 214, 215, 190, 220, 218, 32, 2,
        ],
        [
            250, 142, 28, 190, 160, 225, 213, 205, 14, 211, 102, 52, 153, 255, 151, 209, 225, 2,
            33, 25, 213, 238, 238, 198, 18, 154, 228, 241, 241, 54, 157, 60,
        ],
        [
            75, 214, 166, 211, 115, 128, 32, 28, 77, 81, 147, 56, 39, 155, 109, 112, 10, 160, 98,
            103, 72, 95, 27, 59, 207, 229, 174, 25, 4, 251, 92, 60,
        ],
        [
            34, 113, 137, 237, 47, 68, 116, 169, 227, 218, 209, 74, 23, 162, 182, 147, 221, 151,
            145, 79, 131, 95, 193, 132, 143, 130, 130, 227, 245, 198, 206, 27,
        ],
        [
            127, 167, 50, 210, 223, 36, 251, 32, 21, 244, 240, 170, 220, 7, 86, 197, 230, 237, 222,
            25, 156, 33, 93, 227, 95, 118, 148, 123, 226, 170, 189, 62,
        ],
        [
            172, 190, 180, 238, 11, 198, 98, 110, 7, 7, 137, 152, 215, 206, 40, 220, 218, 85, 9,
            52, 172, 74, 5, 214, 152, 191, 17, 48, 100, 173, 54, 26,
        ],
        [
            247, 12, 146, 226, 110, 132, 132, 139, 243, 207, 204, 228, 47, 186, 231, 240, 231, 176,
            133, 36, 206, 22, 119, 135, 22, 145, 197, 100, 20, 199, 15, 58,
        ],
        [
            175, 19, 90, 7, 34, 58, 172, 219, 22, 67, 115, 16, 50, 51, 173, 31, 133, 161, 1, 229,
            178, 133, 102, 152, 55, 78, 45, 110, 75, 239, 234, 38,
        ],
    ],
    [
        [
            145, 188, 217, 28, 211, 109, 158, 69, 114, 242, 67, 177, 86, 231, 235, 110, 226, 248,
            92, 192, 231, 217, 250, 30, 54, 110, 42, 17, 4, 123, 63, 41,
        ],
        [
            239, 149, 56, 141, 40, 202, 6, 176, 150, 138, 225, 8, 95, 21, 41, 149, 135, 163, 132,
            41, 2, 43, 139, 171, 27, 196, 102, 64, 33, 218, 153, 59,
        ],
        [
            203, 34, 210, 53, 46, 172, 48, 78, 145, 63, 217, 49, 77, 236, 1, 154, 200, 72, 24, 2,
            222, 189, 166, 166, 132, 245, 177, 199, 89, 54, 126, 25,
        ],
        [
            86, 6, 180, 69, 62, 170, 243, 248, 240, 30, 241, 155, 22, 79, 11, 110, 178, 64, 195,
            62, 30, 75, 182, 14, 113, 220, 96, 15, 168, 107, 37, 60,
        ],
        [
            119, 113, 192, 238, 13, 95, 60, 229, 22, 93, 102, 163, 170, 144, 188, 6, 194, 160, 69,
            116, 94, 199, 26, 192, 249, 116, 128, 72, 184, 45, 162, 35,
        ],
        [
            94, 223, 14, 35, 38, 97, 179, 224, 57, 126, 170, 251, 91, 103, 84, 23, 224, 151, 106,
            199, 236, 106, 163, 141, 58, 175, 81, 80, 40, 214, 201, 59,
        ],
        [
            132, 112, 204, 9, 226, 55, 185, 75, 231, 21, 48, 249, 198, 110, 57, 82, 123, 164, 91,
            16, 85, 246, 121, 57, 4, 132, 38, 133, 2, 196, 123, 63,
        ],
        [
            10, 232, 165, 228, 159, 111, 77, 10, 124, 168, 59, 111, 134, 139, 175, 115, 181, 41,
            195, 207, 122, 3, 181, 1, 233, 171, 144, 20, 209, 170, 250, 28,
        ],
    ],
    [
        [
            31, 220, 178, 109, 251, 49, 176, 112, 11, 205, 236, 79, 93, 93, 209, 18, 54, 37, 19,
            194, 83, 223, 91, 242, 233, 83, 141, 144, 135, 244, 105, 56,
        ],
        [
            81, 231, 94, 170, 107, 127, 203, 163, 10, 183, 25, 198, 4, 61, 67, 154, 163, 90, 74, 9,
            200, 29, 145, 141, 166, 123, 62, 13, 104, 194, 93, 50,
        ],
        [
            172, 180, 2, 27, 47, 80, 235, 164, 224, 160, 254, 247, 147, 236, 137, 122, 237, 65, 98,
            174, 224, 209, 61, 144, 152, 43, 50, 80, 206, 128, 29, 7,
        ],
        [
            96, 37, 53, 71, 220, 216, 77, 204, 62, 14, 73, 192, 136, 118, 209, 107, 116, 89, 247,
            117, 41, 254, 157, 253, 146, 83, 61, 134, 172, 217, 167, 61,
        ],
        [
            104, 117, 19, 237, 94, 120, 199, 136, 125, 131, 228, 144, 28, 10, 148, 119, 106, 17,
            250, 90, 94, 194, 167, 240, 113, 247, 183, 253, 160, 184, 27, 9,
        ],
        [
            192, 24, 53, 127, 235, 143, 177, 125, 190, 143, 56, 228, 130, 68, 241, 104, 203, 233,
            225, 7, 148, 95, 154, 210, 22, 127, 67, 100, 19, 0, 18, 49,
        ],
        [
            236, 129, 24, 57, 236, 189, 67, 7, 7, 189, 46, 125, 107, 150, 52, 83, 225, 198, 101,
            47, 123, 198, 8, 88, 198, 83, 121, 40, 186, 113, 242, 42,
        ],
        [
            120, 71, 188, 215, 73, 12, 233, 32, 183, 162, 175, 38, 132, 27, 32, 249, 176, 102, 92,
            123, 174, 249, 228, 134, 124, 231, 187, 191, 240, 9, 236, 25,
        ],
    ],
    [
        [
            44, 238, 221, 171, 60, 205, 205, 235, 28, 12, 166, 250, 253, 81, 171, 208, 182, 24, 91,
            210, 129, 225, 102, 183, 208, 189, 114, 152, 151, 157, 226, 51,
        ],
        [
            103, 138, 231, 149, 85, 124, 78, 82, 40, 17, 231, 154, 156, 3, 30, 222, 245, 52, 94,
            21, 104, 35, 253, 234, 63, 93, 118, 157, 170, 22, 250, 37,
        ],
        [
            40, 91, 164, 160, 164, 242, 12, 226, 36, 210, 221, 128, 70, 87, 209, 36, 35, 189, 44,
            85, 110, 100, 212, 166, 132, 244, 106, 243, 103, 75, 243, 20,
        ],
        [
            159, 108, 152, 218, 145, 14, 59, 25, 174, 120, 118, 20, 221, 214, 134, 127, 8, 84, 14,
            151, 206, 249, 43, 127, 219, 205, 103, 185, 160, 23, 8, 55,
        ],
        [
            248, 129, 230, 218, 151, 177, 113, 167, 115, 89, 111, 199, 44, 100, 229, 186, 76, 136,
            78, 174, 14, 239, 223, 152, 0, 35, 134, 159, 69, 198, 100, 37,
        ],
        [
            86, 62, 241, 104, 218, 62, 10, 203, 109, 39, 47, 242, 110, 34, 17, 62, 188, 251, 109,
            224, 224, 49, 232, 249, 158, 44, 131, 184, 232, 169, 151, 11,
        ],
        [
            230, 160, 217, 171, 53, 116, 55, 143, 30, 111, 3, 92, 203, 224, 216, 180, 33, 218, 240,
            25, 183, 233, 55, 238, 165, 146, 168, 141, 40, 92, 37, 22,
        ],
        [
            209, 112, 179, 95, 105, 7, 163, 138, 141, 97, 118, 174, 120, 237, 72, 230, 79, 172, 90,
            219, 26, 192, 38, 228, 122, 28, 72, 80, 235, 193, 127, 34,
        ],
    ],
    [
        [
            254, 28, 193, 110, 41, 72, 172, 152, 87, 109, 233, 222, 140, 147, 159, 124, 148, 182,
            94, 179, 206, 34, 33, 50, 172, 30, 12, 199, 87, 72, 165, 41,
        ],
        [
            245, 173, 241, 93, 248, 176, 51, 64, 144, 104, 208, 216, 16, 206, 94, 152, 167, 89,
            137, 196, 246, 240, 87, 167, 171, 104, 27, 116, 108, 217, 22, 9,
        ],
        [
            132, 156, 222, 191, 17, 62, 197, 49, 34, 216, 214, 33, 47, 82, 224, 216, 170, 239, 241,
            126, 234, 212, 199, 91, 138, 110, 27, 128, 117, 50, 242, 32,
        ],
        [
            64, 5, 181, 26, 70, 28, 184, 194, 87, 116, 43, 93, 67, 92, 174, 29, 11, 79, 243, 240,
            11, 121, 157, 62, 58, 33, 181, 90, 211, 131, 7, 8,
        ],
        [
            230, 49, 144, 6, 195, 18, 172, 214, 238, 213, 186, 116, 144, 69, 231, 252, 84, 102, 91,
            94, 200, 169, 229, 178, 99, 245, 15, 119, 190, 194, 95, 54,
        ],
        [
            41, 252, 108, 140, 207, 174, 78, 195, 183, 186, 125, 24, 131, 106, 216, 108, 184, 153,
            249, 79, 94, 210, 124, 195, 115, 224, 63, 192, 184, 83, 30, 26,
        ],
        [
            208, 125, 51, 179, 127, 202, 37, 211, 44, 72, 27, 164, 254, 137, 58, 148, 0, 63, 41,
            234, 1, 158, 72, 25, 169, 132, 1, 189, 215, 2, 64, 14,
        ],
        [
            46, 111, 37, 127, 176, 137, 50, 89, 246, 138, 208, 203, 106, 195, 157, 231, 240, 144,
            240, 225, 64, 198, 233, 167, 127, 40, 231, 104, 82, 155, 45, 18,
        ],
    ],
    [
        [
            25, 54, 51, 222, 26, 219, 162, 250, 134, 244, 11, 167, 206, 151, 28, 254, 63, 143, 163,
            169, 151, 94, 5, 207, 253, 137, 198, 25, 88, 102, 129, 24,
        ],
        [
            173, 42, 240, 164, 55, 124, 163, 247, 2, 94, 188, 244, 114, 36, 131, 79, 243, 126, 25,
            60, 70, 209, 150, 121, 9, 44, 80, 224, 183, 131, 133, 40,
        ],
        [
            246, 58, 162, 49, 244, 161, 222, 142, 77, 187, 24, 76, 41, 16, 76, 147, 43, 143, 173,
            142, 112, 194, 31, 105, 27, 195, 126, 227, 23, 231, 208, 8,
        ],
        [
            35, 200, 84, 68, 62, 127, 102, 254, 54, 172, 151, 193, 222, 181, 190, 204, 95, 57, 33,
            225, 15, 79, 237, 101, 28, 239, 186, 29, 41, 69, 98, 21,
        ],
        [
            100, 64, 24, 112, 156, 153, 245, 233, 166, 182, 114, 120, 185, 207, 43, 155, 145, 73,
            64, 215, 81, 169, 35, 240, 69, 116, 5, 34, 240, 79, 105, 1,
        ],
        [
            226, 59, 11, 185, 102, 13, 42, 109, 44, 163, 130, 161, 218, 147, 189, 18, 60, 61, 123,
            242, 172, 208, 131, 138, 118, 95, 21, 111, 46, 74, 82, 1,
        ],
        [
            217, 244, 22, 128, 132, 230, 210, 181, 187, 251, 65, 195, 81, 129, 222, 206, 97, 14,
            178, 243, 19, 113, 209, 11, 142, 86, 248, 246, 229, 152, 162, 17,
        ],
        [
            59, 102, 139, 93, 161, 101, 23, 36, 218, 70, 246, 128, 98, 147, 174, 207, 84, 139, 39,
            67, 227, 7, 130, 64, 37, 22, 194, 218, 191, 43, 34, 45,
        ],
    ],
    [
        [
            243, 215, 252, 180, 108, 191, 218, 211, 153, 49, 144, 152, 90, 95, 76, 233, 106, 222,
            156, 245, 61, 172, 45, 242, 58, 167, 202, 227, 245, 236, 247, 20,
        ],
        [
            2, 213, 207, 34, 65, 186, 12, 228, 150, 46, 40, 66, 72, 196, 244, 151, 80, 62, 116,
            141, 36, 52, 180, 11, 184, 13, 175, 50, 172, 66, 85, 29,
        ],
        [
            185, 172, 242, 99, 188, 15, 56, 93, 53, 83, 208, 174, 10, 71, 76, 36, 98, 173, 159, 81,
            53, 129, 157, 220, 61, 93, 12, 123, 21, 2, 123, 46,
        ],
        [
            195, 200, 168, 165, 148, 30, 154, 31, 130, 39, 105, 199, 45, 122, 208, 75, 43, 40, 223,
            149, 143, 209, 121, 207, 176, 192, 237, 254, 133, 131, 212, 14,
        ],
        [
            243, 120, 106, 116, 193, 82, 61, 254, 19, 211, 26, 13, 30, 197, 160, 76, 142, 143, 228,
            54, 49, 114, 253, 198, 190, 58, 118, 175, 105, 244, 57, 1,
        ],
        [
            187, 65, 118, 89, 48, 103, 234, 64, 137, 212, 20, 130, 63, 41, 167, 40, 204, 105, 151,
            188, 106, 116, 212, 8, 33, 220, 86, 126, 174, 117, 241, 11,
        ],
        [
            23, 36, 152, 55, 25, 64, 51, 241, 153, 94, 53, 240, 244, 26, 191, 249, 76, 47, 167, 53,
            103, 179, 87, 200, 85, 57, 187, 73, 87, 186, 71, 5,
        ],
        [
            3, 245, 6, 142, 151, 192, 17, 62, 8, 136, 223, 178, 122, 228, 234, 192, 18, 12, 162,
            186, 249, 173, 183, 20, 190, 238, 56, 35, 142, 237, 243, 2,
        ],
    ],
    [
        [
            214, 249, 38, 19, 194, 4, 20, 205, 62, 197, 25, 3, 103, 4, 128, 12, 222, 100, 14, 15,
            9, 188, 229, 54, 182, 46, 18, 0, 53, 137, 27, 32,
        ],
        [
            1, 248, 42, 1, 186, 186, 208, 3, 200, 86, 156, 177, 16, 214, 197, 100, 46, 136, 45,
            186, 221, 12, 30, 122, 141, 141, 242, 37, 9, 23, 50, 31,
        ],
        [
            30, 193, 71, 240, 123, 34, 96, 160, 159, 81, 65, 89, 85, 224, 39, 182, 188, 43, 148, 4,
            178, 136, 230, 179, 120, 157, 237, 153, 144, 203, 166, 18,
        ],
        [
            23, 153, 154, 149, 96, 81, 58, 206, 78, 42, 131, 36, 180, 230, 152, 240, 188, 196, 75,
            191, 187, 191, 201, 29, 248, 169, 1, 249, 194, 90, 146, 13,
        ],
        [
            172, 165, 50, 18, 118, 97, 254, 130, 3, 56, 196, 176, 230, 203, 95, 140, 114, 140, 53,
            45, 184, 55, 58, 243, 32, 82, 182, 61, 121, 144, 156, 59,
        ],
        [
            117, 176, 227, 84, 115, 78, 150, 191, 31, 133, 143, 105, 89, 128, 249, 146, 113, 16,
            24, 46, 121, 132, 4, 240, 59, 5, 228, 223, 10, 28, 17, 45,
        ],
        [
            90, 222, 163, 229, 143, 243, 155, 195, 207, 44, 38, 111, 136, 71, 10, 65, 141, 20, 56,
            224, 173, 128, 45, 220, 236, 250, 2, 212, 132, 245, 86, 20,
        ],
        [
            132, 121, 145, 157, 175, 243, 206, 52, 195, 75, 219, 105, 139, 21, 43, 111, 235, 75,
            244, 41, 179, 111, 142, 241, 56, 139, 199, 215, 155, 155, 24, 3,
        ],
    ],
    [
        [
            170, 160, 77, 190, 31, 71, 130, 150, 13, 82, 166, 134, 114, 218, 143, 234, 103, 193,
            31, 79, 160, 143, 224, 137, 153, 223, 83, 95, 86, 15, 224, 44,
        ],
        [
            247, 66, 144, 180, 80, 213, 107, 248, 33, 247, 21, 40, 162, 43, 229, 130, 164, 122,
            155, 92, 221, 188, 252, 170, 102, 151, 54, 204, 150, 59, 194, 56,
        ],
        [
            99, 55, 144, 231, 101, 168, 114, 245, 137, 185, 40, 155, 159, 157, 78, 143, 41, 227,
            125, 81, 209, 82, 71, 23, 225, 95, 242, 152, 61, 106, 114, 58,
        ],
        [
            251, 171, 20, 73, 200, 63, 174, 145, 92, 225, 76, 78, 213, 212, 87, 243, 42, 186, 194,
            98, 80, 131, 184, 206, 237, 248, 143, 235, 185, 66, 225, 15,
        ],
        [
            41, 63, 235, 60, 242, 30, 193, 127, 243, 81, 133, 103, 18, 74, 224, 246, 14, 15, 81,
            128, 139, 171, 164, 159, 217, 64, 66, 253, 203, 171, 90, 44,
        ],
        [
            107, 47, 2, 101, 9, 174, 191, 51, 206, 178, 45, 59, 2, 48, 86, 193, 88, 190, 79, 94,
            255, 5, 32, 137, 160, 237, 0, 50, 20, 131, 108, 8,
        ],
        [
            149, 251, 115, 8, 57, 217, 50, 3, 163, 192, 38, 199, 174, 92, 109, 92, 246, 242, 44,
            56, 193, 10, 51, 113, 130, 201, 253, 190, 30, 240, 63, 40,
        ],
        [
            110, 22, 68, 198, 134, 118, 180, 38, 89, 200, 93, 253, 239, 27, 222, 170, 177, 213, 83,
            119, 187, 33, 203, 82, 158, 22, 19, 215, 138, 247, 189, 51,
        ],
    ],
    [
        [
            117, 125, 50, 157, 49, 226, 157, 252, 37, 2, 218, 196, 79, 208, 204, 122, 201, 107,
            136, 149, 22, 194, 213, 162, 66, 123, 132, 115, 191, 180, 83, 48,
        ],
        [
            210, 19, 161, 243, 101, 177, 187, 16, 121, 148, 102, 119, 222, 135, 3, 199, 122, 151,
            166, 149, 140, 235, 66, 227, 102, 181, 160, 237, 201, 144, 179, 11,
        ],
        [
            62, 110, 127, 32, 13, 245, 200, 232, 109, 85, 27, 78, 90, 105, 51, 16, 209, 176, 250,
            162, 217, 67, 168, 73, 143, 84, 9, 68, 225, 83, 207, 18,
        ],
        [
            178, 159, 32, 19, 97, 131, 45, 209, 170, 100, 118, 101, 225, 22, 117, 148, 47, 171,
            123, 52, 52, 31, 197, 175, 63, 62, 53, 66, 159, 86, 101, 18,
        ],
        [
            32, 79, 44, 69, 64, 154, 0, 68, 192, 225, 40, 51, 90, 68, 250, 187, 184, 212, 134, 241,
            224, 157, 188, 102, 132, 155, 170, 242, 5, 149, 109, 41,
        ],
        [
            37, 1, 13, 187, 23, 69, 249, 107, 244, 132, 14, 31, 32, 38, 13, 196, 203, 148, 105, 80,
            38, 185, 27, 200, 4, 42, 54, 47, 69, 35, 139, 47,
        ],
        [
            59, 145, 155, 161, 230, 222, 9, 75, 18, 137, 208, 130, 222, 156, 205, 35, 60, 164, 30,
            87, 239, 251, 197, 42, 175, 198, 243, 52, 153, 33, 82, 6,
        ],
        [
            114, 241, 55, 63, 242, 105, 93, 232, 30, 144, 103, 214, 145, 231, 247, 5, 39, 81, 213,
            224, 233, 54, 143, 86, 24, 239, 152, 241, 22, 3, 88, 48,
        ],
    ],
    [
        [
            183, 73, 223, 229, 216, 181, 22, 14, 160, 107, 224, 122, 232, 129, 32, 164, 185, 28,
            237, 169, 30, 220, 109, 105, 222, 16, 150, 199, 169, 98, 178, 63,
        ],
        [
            254, 59, 102, 135, 221, 217, 120, 25, 70, 142, 180, 143, 18, 32, 76, 155, 27, 126, 85,
            209, 54, 41, 226, 146, 61, 60, 90, 93, 148, 65, 13, 54,
        ],
        [
            45, 111, 189, 160, 179, 174, 50, 59, 44, 37, 174, 101, 210, 22, 94, 248, 230, 247, 218,
            200, 245, 206, 106, 137, 52, 245, 230, 142, 169, 163, 134, 18,
        ],
        [
            217, 173, 14, 202, 213, 119, 164, 85, 177, 132, 83, 40, 249, 158, 172, 93, 210, 45,
            244, 253, 247, 227, 38, 252, 226, 240, 65, 163, 106, 29, 22, 19,
        ],
        [
            249, 69, 144, 252, 245, 20, 125, 73, 213, 219, 120, 12, 11, 216, 108, 111, 107, 17, 24,
            126, 242, 50, 77, 245, 180, 66, 194, 195, 136, 0, 179, 36,
        ],
        [
            110, 175, 70, 254, 167, 151, 104, 91, 126, 243, 201, 150, 21, 231, 172, 247, 188, 206,
            194, 88, 239, 32, 83, 81, 239, 114, 164, 154, 166, 82, 47, 46,
        ],
        [
            1, 177, 232, 3, 140, 36, 220, 195, 191, 180, 103, 48, 18, 19, 97, 145, 66, 252, 19,
            194, 79, 177, 227, 57, 214, 205, 227, 144, 105, 243, 49, 52,
        ],
        [
            3, 30, 79, 200, 159, 73, 74, 224, 211, 118, 77, 119, 49, 6, 200, 123, 189, 29, 240, 75,
            143, 113, 248, 126, 157, 34, 181, 40, 71, 104, 40, 29,
        ],
    ],
    [
        [
            72, 187, 142, 33, 65, 74, 73, 237, 236, 55, 32, 61, 31, 150, 208, 1, 154, 10, 254, 203,
            12, 136, 113, 130, 120, 149, 192, 62, 188, 20, 167, 15,
        ],
        [
            34, 157, 37, 82, 105, 170, 4, 202, 92, 83, 161, 163, 61, 113, 24, 211, 35, 231, 97, 61,
            163, 180, 214, 182, 24, 32, 57, 234, 100, 78, 115, 10,
        ],
        [
            222, 232, 162, 244, 15, 70, 7, 177, 125, 48, 236, 76, 215, 103, 30, 192, 174, 174, 204,
            26, 189, 219, 24, 120, 164, 196, 186, 112, 232, 17, 41, 50,
        ],
        [
            55, 110, 18, 103, 80, 146, 54, 15, 36, 53, 114, 18, 77, 124, 197, 206, 106, 215, 86,
            197, 102, 199, 200, 42, 27, 9, 93, 246, 79, 0, 67, 25,
        ],
        [
            121, 169, 86, 133, 108, 129, 219, 189, 34, 211, 101, 130, 74, 34, 13, 81, 14, 110, 214,
            84, 27, 61, 84, 26, 29, 50, 181, 52, 167, 142, 120, 34,
        ],
        [
            124, 171, 39, 168, 104, 172, 63, 79, 108, 178, 191, 149, 140, 79, 124, 216, 131, 183,
            186, 128, 242, 182, 167, 255, 103, 148, 242, 122, 64, 230, 71, 7,
        ],
        [
            56, 174, 182, 231, 135, 215, 66, 103, 46, 120, 92, 118, 249, 203, 107, 149, 58, 138,
            148, 91, 58, 39, 142, 88, 196, 91, 44, 213, 184, 39, 132, 40,
        ],
        [
            185, 195, 19, 33, 60, 2, 123, 172, 104, 179, 183, 244, 19, 202, 190, 127, 167, 133,
            136, 102, 193, 29, 212, 214, 123, 183, 156, 205, 192, 198, 35, 43,
        ],
    ],
    [
        [
            205, 83, 71, 183, 1, 24, 115, 206, 228, 111, 5, 241, 177, 42, 216, 40, 155, 197, 81,
            205, 133, 76, 236, 36, 117, 185, 253, 103, 120, 35, 65, 21,
        ],
        [
            166, 215, 18, 12, 156, 184, 129, 59, 129, 189, 191, 109, 255, 117, 116, 255, 94, 69,
            20, 76, 50, 5, 87, 157, 165, 64, 83, 3, 196, 27, 19, 22,
        ],
        [
            140, 139, 130, 83, 133, 157, 114, 126, 234, 212, 116, 154, 64, 173, 80, 56, 16, 61,
            108, 202, 102, 96, 21, 247, 144, 174, 45, 27, 201, 147, 225, 41,
        ],
        [
            62, 87, 186, 240, 132, 207, 51, 133, 39, 103, 81, 145, 73, 139, 31, 131, 59, 209, 176,
            139, 208, 186, 233, 247, 155, 252, 144, 3, 44, 85, 252, 49,
        ],
        [
            138, 130, 218, 32, 216, 172, 47, 195, 15, 48, 92, 183, 94, 116, 144, 79, 223, 127, 155,
            136, 39, 119, 64, 30, 182, 98, 54, 118, 54, 223, 150, 27,
        ],
        [
            206, 25, 234, 10, 64, 70, 6, 217, 31, 179, 247, 5, 7, 239, 139, 168, 33, 169, 35, 190,
            246, 31, 225, 27, 22, 85, 217, 211, 42, 112, 105, 39,
        ],
        [
            216, 56, 84, 211, 144, 154, 176, 88, 174, 154, 110, 176, 208, 88, 132, 35, 157, 174,
            230, 171, 136, 232, 84, 14, 237, 218, 16, 49, 179, 160, 234, 50,
        ],
        [
            183, 99, 245, 70, 190, 152, 145, 87, 35, 163, 196, 71, 172, 58, 205, 155, 160, 103, 8,
            85, 57, 175, 88, 31, 88, 49, 206, 162, 16, 155, 23, 25,
        ],
    ],
    [
        [
            74, 149, 239, 189, 114, 65, 210, 77, 46, 82, 98, 194, 14, 133, 94, 49, 40, 152, 91, 50,
            5, 164, 18, 226, 135, 94, 2, 125, 175, 41, 120, 51,
        ],
        [
            181, 143, 178, 114, 48, 133, 23, 9, 63, 217, 111, 235, 184, 251, 15, 60, 148, 101, 59,
            45, 88, 219, 28, 83, 26, 229, 124, 153, 156, 77, 168, 40,
        ],
        [
            171, 22, 216, 53, 16, 224, 251, 108, 191, 93, 174, 150, 237, 239, 25, 5, 125, 60, 159,
            39, 193, 135, 180, 109, 3, 224, 127, 206, 115, 24, 224, 45,
        ],
        [
            200, 41, 189, 233, 227, 16, 222, 59, 153, 138, 79, 11, 196, 111, 162, 29, 146, 83, 220,
            255, 1, 10, 130, 58, 243, 8, 93, 230, 33, 87, 48, 41,
        ],
        [
            181, 159, 33, 60, 68, 174, 209, 207, 203, 92, 245, 243, 74, 170, 154, 1, 43, 160, 52,
            194, 210, 27, 145, 30, 99, 172, 27, 249, 71, 223, 104, 50,
        ],
        [
            175, 5, 60, 58, 189, 14, 246, 123, 78, 49, 84, 221, 109, 65, 185, 93, 36, 88, 96, 86,
            61, 236, 119, 23, 27, 100, 94, 165, 136, 114, 128, 4,
        ],
        [
            150, 74, 80, 139, 229, 38, 87, 70, 245, 190, 92, 10, 211, 22, 165, 73, 143, 143, 126,
            177, 23, 53, 219, 247, 31, 38, 25, 182, 158, 244, 20, 50,
        ],
        [
            18, 209, 127, 187, 128, 0, 119, 101, 205, 210, 31, 119, 197, 0, 221, 229, 203, 212, 62,
            55, 237, 254, 101, 158, 122, 176, 42, 150, 232, 45, 7, 41,
        ],
    ],
    [
        [
            36, 137, 204, 219, 0, 154, 63, 247, 215, 165, 137, 105, 28, 105, 38, 4, 193, 135, 137,
            132, 73, 32, 75, 1, 172, 186, 105, 126, 155, 9, 234, 27,
        ],
        [
            79, 123, 106, 34, 65, 148, 102, 212, 37, 226, 26, 177, 218, 65, 92, 191, 248, 14, 27,
            239, 91, 204, 175, 13, 75, 172, 17, 69, 134, 139, 96, 52,
        ],
        [
            78, 188, 220, 59, 5, 149, 96, 230, 10, 117, 245, 26, 174, 111, 27, 186, 139, 178, 228,
            214, 35, 5, 223, 223, 57, 100, 118, 222, 159, 1, 163, 53,
        ],
        [
            73, 71, 233, 100, 247, 29, 45, 142, 28, 138, 200, 24, 17, 39, 182, 140, 255, 159, 152,
            94, 75, 170, 3, 92, 75, 65, 27, 18, 222, 2, 33, 3,
        ],
        [
            13, 222, 43, 60, 116, 40, 239, 164, 101, 201, 242, 163, 153, 35, 237, 50, 242, 156,
            223, 180, 148, 126, 6, 177, 165, 110, 139, 226, 231, 8, 145, 59,
        ],
        [
            246, 117, 29, 220, 85, 113, 129, 150, 184, 4, 247, 10, 172, 74, 103, 73, 108, 4, 201,
            86, 25, 60, 29, 231, 244, 96, 57, 30, 146, 122, 223, 12,
        ],
        [
            23, 172, 62, 0, 141, 188, 233, 62, 34, 177, 96, 95, 247, 236, 50, 170, 113, 178, 151,
            44, 48, 184, 93, 237, 208, 5, 168, 49, 109, 16, 21, 4,
        ],
        [
            143, 110, 28, 202, 250, 200, 195, 38, 9, 224, 117, 239, 53, 6, 222, 247, 231, 173, 228,
            210, 251, 2, 14, 133, 86, 139, 194, 14, 184, 178, 31, 42,
        ],
    ],
    [
        [
            105, 82, 47, 89, 25, 103, 11, 249, 66, 111, 188, 87, 78, 135, 150, 100, 113, 159, 209,
            41, 242, 37, 25, 215, 250, 132, 205, 49, 100, 113, 42, 32,
        ],
        [
            124, 198, 51, 229, 215, 184, 224, 251, 79, 26, 84, 35, 213, 164, 170, 39, 215, 255,
            210, 173, 169, 81, 244, 105, 176, 249, 37, 191, 25, 6, 190, 16,
        ],
        [
            253, 151, 83, 116, 230, 52, 110, 206, 147, 81, 170, 96, 182, 189, 181, 35, 18, 118,
            175, 63, 93, 201, 248, 55, 129, 200, 104, 11, 160, 47, 4, 30,
        ],
        [
            28, 3, 12, 229, 152, 219, 186, 100, 206, 36, 225, 30, 34, 30, 108, 68, 21, 196, 136,
            120, 225, 83, 116, 203, 0, 85, 160, 6, 105, 42, 13, 61,
        ],
        [
            191, 116, 153, 77, 233, 2, 241, 126, 83, 78, 152, 251, 93, 112, 49, 221, 130, 19, 135,
            243, 251, 24, 111, 131, 88, 152, 29, 20, 167, 247, 113, 7,
        ],
        [
            124, 91, 25, 231, 114, 60, 237, 33, 75, 74, 25, 161, 83, 19, 39, 229, 183, 41, 5, 224,
            4, 248, 170, 42, 77, 249, 106, 245, 84, 62, 169, 48,
        ],
        [
            91, 157, 52, 245, 165, 35, 240, 158, 137, 167, 246, 18, 171, 56, 55, 19, 42, 145, 226,
            167, 56, 65, 165, 215, 46, 42, 69, 158, 208, 251, 37, 10,
        ],
        [
            89, 198, 136, 103, 162, 233, 140, 72, 172, 240, 202, 112, 189, 124, 240, 96, 250, 141,
            169, 63, 77, 47, 203, 146, 57, 89, 123, 126, 54, 233, 83, 45,
        ],
    ],
    [
        [
            178, 16, 104, 36, 221, 99, 174, 215, 73, 246, 184, 112, 42, 211, 233, 124, 101, 17,
            215, 164, 187, 122, 244, 216, 15, 188, 20, 190, 255, 89, 108, 13,
        ],
        [
            211, 185, 13, 171, 251, 160, 220, 120, 214, 182, 153, 128, 218, 152, 221, 223, 133,
            255, 100, 20, 188, 224, 61, 176, 23, 208, 137, 116, 1, 33, 3, 53,
        ],
        [
            39, 18, 157, 171, 253, 120, 243, 71, 172, 2, 132, 76, 83, 210, 103, 104, 67, 120, 60,
            58, 145, 252, 20, 90, 44, 3, 245, 247, 95, 6, 243, 2,
        ],
        [
            228, 125, 115, 84, 131, 131, 169, 92, 51, 15, 82, 112, 8, 204, 91, 222, 203, 208, 109,
            111, 14, 59, 28, 244, 46, 35, 144, 69, 212, 66, 200, 44,
        ],
        [
            124, 115, 67, 210, 197, 53, 11, 153, 55, 71, 208, 74, 42, 181, 140, 0, 21, 155, 176,
            20, 255, 136, 77, 238, 59, 70, 25, 25, 41, 180, 112, 22,
        ],
        [
            124, 61, 201, 115, 14, 189, 176, 115, 124, 24, 53, 163, 232, 123, 121, 50, 30, 56, 161,
            52, 120, 136, 33, 134, 184, 16, 182, 31, 222, 26, 199, 41,
        ],
        [
            151, 34, 213, 29, 165, 116, 149, 163, 134, 111, 30, 119, 195, 252, 137, 89, 146, 206,
            18, 82, 141, 247, 3, 130, 205, 69, 224, 189, 21, 55, 114, 62,
        ],
        [
            241, 243, 198, 132, 63, 204, 196, 163, 230, 17, 248, 158, 98, 188, 208, 23, 173, 12,
            187, 227, 73, 242, 34, 237, 83, 236, 73, 86, 135, 201, 77, 14,
        ],
    ],
    [
        [
            80, 161, 16, 201, 18, 107, 180, 237, 109, 13, 229, 253, 142, 231, 128, 185, 249, 180,
            210, 192, 126, 126, 137, 247, 97, 233, 118, 230, 208, 44, 147, 42,
        ],
        [
            65, 106, 220, 93, 202, 7, 140, 37, 78, 54, 13, 227, 12, 189, 35, 252, 207, 100, 130,
            146, 10, 104, 219, 121, 127, 229, 170, 22, 88, 108, 181, 14,
        ],
        [
            2, 157, 209, 112, 135, 200, 85, 220, 218, 10, 173, 98, 35, 244, 80, 103, 141, 49, 254,
            61, 207, 4, 33, 214, 195, 150, 60, 163, 222, 156, 239, 42,
        ],
        [
            37, 211, 37, 147, 184, 166, 133, 18, 236, 217, 52, 134, 76, 157, 193, 88, 141, 231, 1,
            140, 111, 174, 181, 143, 125, 159, 79, 19, 199, 246, 202, 46,
        ],
        [
            110, 207, 189, 175, 189, 39, 143, 75, 162, 13, 195, 173, 70, 132, 116, 211, 227, 67,
            222, 87, 179, 3, 179, 208, 95, 26, 97, 57, 79, 162, 109, 4,
        ],
        [
            190, 104, 106, 225, 88, 186, 197, 168, 57, 237, 224, 17, 25, 24, 12, 68, 144, 195, 156,
            140, 139, 148, 115, 239, 237, 123, 99, 173, 243, 43, 36, 50,
        ],
        [
            195, 129, 0, 72, 233, 178, 136, 47, 186, 117, 240, 197, 103, 82, 186, 53, 108, 10, 230,
            147, 5, 91, 174, 204, 254, 223, 40, 243, 120, 135, 80, 5,
        ],
        [
            114, 139, 243, 114, 81, 82, 185, 45, 160, 204, 55, 76, 56, 73, 96, 173, 89, 180, 187,
            248, 217, 29, 209, 242, 49, 222, 223, 187, 79, 31, 171, 18,
        ],
    ],
    [
        [
            145, 189, 93, 77, 84, 187, 100, 38, 15, 134, 76, 143, 117, 175, 58, 183, 67, 205, 108,
            48, 80, 218, 140, 116, 209, 197, 219, 206, 218, 56, 117, 43,
        ],
        [
            109, 207, 168, 80, 126, 16, 71, 172, 63, 120, 145, 32, 80, 61, 208, 2, 58, 77, 68, 17,
            180, 162, 200, 232, 13, 185, 148, 170, 212, 169, 215, 35,
        ],
        [
            116, 165, 185, 42, 50, 205, 109, 251, 128, 151, 207, 144, 214, 147, 123, 160, 138, 32,
            10, 144, 97, 19, 182, 103, 58, 146, 207, 54, 85, 210, 161, 37,
        ],
        [
            190, 55, 212, 35, 164, 202, 202, 234, 154, 4, 6, 112, 209, 150, 66, 114, 203, 36, 203,
            24, 252, 175, 4, 184, 42, 158, 195, 232, 251, 59, 242, 4,
        ],
        [
            204, 229, 51, 68, 126, 85, 72, 53, 106, 2, 172, 19, 37, 238, 62, 101, 89, 96, 245, 177,
            177, 40, 170, 245, 147, 72, 141, 93, 233, 146, 36, 41,
        ],
        [
            75, 158, 246, 21, 51, 72, 3, 237, 77, 75, 117, 247, 101, 125, 14, 129, 23, 180, 190,
            49, 57, 32, 223, 224, 65, 205, 113, 48, 119, 203, 73, 13,
        ],
        [
            208, 136, 26, 159, 242, 113, 111, 175, 113, 163, 69, 66, 131, 170, 224, 157, 61, 229,
            222, 185, 61, 153, 133, 149, 167, 215, 195, 134, 97, 137, 168, 7,
        ],
        [
            35, 33, 104, 95, 155, 119, 101, 133, 53, 48, 169, 224, 62, 138, 152, 3, 245, 213, 187,
            253, 52, 0, 26, 40, 13, 50, 41, 224, 161, 121, 194, 25,
        ],
    ],
    [
        [
            47, 30, 245, 11, 248, 255, 82, 126, 62, 107, 70, 78, 38, 130, 69, 145, 206, 99, 157,
            171, 184, 143, 155, 226, 182, 81, 109, 192, 179, 104, 10, 5,
        ],
        [
            146, 17, 30, 231, 169, 68, 47, 50, 8, 250, 194, 198, 195, 213, 98, 37, 31, 151, 214,
            53, 11, 76, 106, 27, 111, 214, 28, 89, 231, 124, 141, 52,
        ],
        [
            95, 57, 186, 20, 37, 9, 249, 188, 111, 199, 138, 164, 67, 210, 204, 91, 242, 253, 155,
            129, 139, 172, 35, 247, 62, 243, 90, 81, 121, 132, 189, 6,
        ],
        [
            105, 51, 164, 127, 16, 30, 56, 76, 170, 213, 193, 77, 27, 136, 146, 19, 173, 117, 146,
            9, 37, 22, 57, 185, 117, 113, 162, 145, 197, 99, 233, 17,
        ],
        [
            18, 116, 253, 121, 151, 86, 62, 152, 186, 6, 124, 197, 101, 205, 63, 23, 49, 25, 157,
            126, 62, 113, 25, 189, 85, 126, 171, 171, 18, 221, 252, 14,
        ],
        [
            220, 42, 220, 82, 18, 237, 116, 250, 56, 130, 90, 202, 56, 16, 179, 88, 145, 64, 230,
            133, 33, 198, 236, 106, 36, 38, 14, 226, 74, 203, 58, 31,
        ],
        [
            210, 176, 102, 53, 106, 223, 246, 149, 22, 171, 229, 252, 194, 239, 188, 94, 104, 4,
            85, 107, 98, 2, 197, 254, 9, 22, 10, 5, 27, 190, 151, 23,
        ],
        [
            0, 241, 127, 167, 92, 61, 119, 20, 244, 167, 64, 241, 70, 140, 148, 165, 70, 67, 44,
            248, 244, 214, 61, 129, 117, 226, 122, 157, 119, 133, 48, 47,
        ],
    ],
    [
        [
            4, 166, 155, 39, 140, 2, 252, 205, 171, 177, 125, 47, 117, 177, 191, 55, 138, 148, 78,
            33, 229, 205, 154, 132, 232, 72, 20, 229, 53, 63, 67, 1,
        ],
        [
            43, 67, 59, 76, 240, 46, 123, 207, 242, 176, 122, 211, 255, 12, 219, 60, 231, 75, 3,
            190, 18, 41, 212, 251, 73, 151, 103, 120, 1, 191, 0, 61,
        ],
        [
            164, 33, 177, 27, 105, 174, 60, 4, 88, 15, 206, 165, 57, 125, 25, 243, 39, 137, 109,
            142, 161, 29, 42, 203, 207, 57, 81, 16, 220, 148, 100, 21,
        ],
        [
            86, 4, 190, 246, 52, 129, 210, 172, 1, 30, 55, 220, 143, 98, 235, 140, 53, 209, 102,
            196, 193, 10, 74, 160, 177, 42, 223, 127, 16, 244, 124, 43,
        ],
        [
            225, 192, 141, 226, 121, 140, 180, 82, 60, 53, 207, 158, 159, 8, 14, 9, 102, 180, 3,
            194, 72, 45, 23, 138, 183, 92, 239, 11, 125, 218, 124, 43,
        ],
        [
            210, 37, 140, 143, 98, 19, 14, 69, 7, 66, 252, 22, 62, 1, 105, 251, 62, 64, 112, 99,
            44, 135, 68, 122, 198, 118, 193, 181, 255, 3, 27, 44,
        ],
        [
            109, 112, 251, 144, 181, 127, 116, 134, 187, 207, 207, 1, 197, 252, 45, 76, 205, 109,
            209, 56, 246, 39, 46, 24, 168, 117, 187, 51, 19, 29, 254, 16,
        ],
        [
            41, 4, 157, 245, 247, 94, 64, 172, 73, 69, 97, 224, 245, 58, 118, 249, 70, 15, 240,
            218, 103, 192, 63, 244, 57, 65, 17, 184, 101, 73, 54, 23,
        ],
    ],
    [
        [
            223, 121, 179, 93, 127, 216, 86, 9, 57, 189, 231, 203, 53, 145, 180, 21, 71, 194, 48,
            133, 129, 176, 172, 166, 123, 28, 42, 37, 31, 173, 212, 39,
        ],
        [
            97, 252, 223, 243, 58, 160, 128, 88, 43, 185, 8, 44, 130, 103, 177, 237, 69, 157, 111,
            78, 63, 57, 32, 114, 219, 163, 133, 222, 61, 151, 15, 1,
        ],
        [
            87, 137, 120, 152, 128, 145, 204, 40, 45, 237, 31, 59, 55, 93, 175, 210, 241, 170, 176,
            158, 7, 255, 88, 212, 241, 219, 18, 54, 48, 199, 196, 44,
        ],
        [
            81, 107, 25, 136, 211, 131, 229, 141, 146, 243, 163, 237, 205, 129, 199, 145, 87, 14,
            174, 129, 183, 238, 77, 141, 121, 253, 106, 221, 225, 141, 103, 17,
        ],
        [
            120, 243, 252, 105, 134, 235, 87, 145, 113, 190, 201, 93, 193, 90, 249, 234, 193, 81,
            34, 140, 7, 203, 62, 86, 125, 80, 123, 82, 45, 120, 93, 36,
        ],
        [
            176, 225, 11, 228, 191, 111, 178, 190, 132, 153, 247, 238, 219, 222, 128, 45, 71, 169,
            21, 91, 143, 139, 185, 103, 36, 80, 67, 247, 92, 81, 114, 60,
        ],
        [
            46, 51, 237, 55, 50, 4, 48, 16, 121, 192, 74, 147, 225, 127, 46, 7, 30, 33, 157, 42,
            180, 248, 143, 133, 220, 169, 31, 97, 123, 19, 197, 59,
        ],
        [
            53, 95, 71, 61, 252, 230, 199, 192, 101, 235, 194, 218, 126, 23, 13, 236, 119, 146,
            143, 214, 170, 112, 252, 186, 39, 146, 13, 77, 80, 26, 43, 2,
        ],
    ],
    [
        [
            195, 118, 63, 33, 37, 85, 102, 80, 239, 150, 164, 82, 197, 241, 150, 118, 81, 249, 223,
            22, 176, 48, 64, 204, 129, 146, 89, 37, 67, 13, 74, 58,
        ],
        [
            112, 91, 136, 133, 7, 92, 221, 205, 91, 29, 20, 95, 248, 102, 106, 68, 55, 33, 118,
            193, 186, 96, 165, 163, 182, 85, 13, 172, 167, 209, 67, 42,
        ],
        [
            185, 151, 20, 57, 185, 10, 140, 226, 229, 3, 62, 149, 145, 79, 245, 205, 231, 76, 63,
            185, 204, 149, 10, 177, 137, 198, 113, 73, 27, 48, 163, 7,
        ],
        [
            230, 103, 185, 235, 217, 108, 170, 98, 162, 251, 153, 175, 197, 188, 220, 123, 235,
            208, 223, 14, 130, 63, 50, 43, 131, 150, 88, 170, 22, 35, 87, 0,
        ],
        [
            81, 131, 168, 216, 91, 18, 186, 77, 201, 236, 192, 208, 33, 151, 172, 185, 157, 9, 217,
            168, 236, 212, 186, 230, 81, 30, 93, 12, 24, 4, 131, 26,
        ],
        [
            64, 149, 92, 20, 249, 203, 232, 169, 227, 78, 195, 114, 204, 188, 117, 59, 24, 140,
            204, 4, 219, 168, 195, 253, 1, 160, 217, 236, 163, 20, 243, 25,
        ],
        [
            82, 173, 48, 198, 187, 239, 155, 12, 9, 47, 144, 166, 182, 114, 36, 252, 94, 193, 225,
            248, 141, 255, 176, 14, 91, 174, 31, 140, 18, 42, 62, 3,
        ],
        [
            238, 79, 14, 238, 178, 202, 236, 102, 201, 183, 56, 152, 187, 34, 28, 114, 182, 183,
            87, 218, 33, 167, 213, 196, 48, 234, 241, 70, 89, 76, 111, 7,
        ],
    ],
    [
        [
            37, 232, 202, 95, 118, 254, 243, 205, 180, 127, 216, 17, 97, 148, 112, 42, 250, 84,
            232, 146, 58, 213, 87, 229, 9, 88, 246, 40, 136, 125, 78, 10,
        ],
        [
            188, 137, 225, 229, 196, 113, 43, 196, 9, 6, 41, 56, 117, 157, 202, 248, 247, 199, 69,
            162, 105, 147, 9, 112, 133, 64, 224, 1, 2, 62, 174, 51,
        ],
        [
            101, 201, 205, 179, 172, 35, 58, 167, 99, 12, 7, 229, 249, 222, 20, 75, 31, 85, 8, 65,
            156, 35, 22, 144, 248, 121, 38, 221, 230, 17, 107, 22,
        ],
        [
            210, 64, 16, 86, 168, 14, 161, 143, 205, 219, 249, 21, 227, 148, 168, 102, 248, 119,
            181, 212, 225, 211, 79, 88, 203, 13, 216, 120, 51, 200, 192, 7,
        ],
        [
            166, 155, 235, 41, 141, 202, 9, 206, 75, 25, 104, 97, 28, 239, 71, 34, 217, 225, 35,
            10, 104, 30, 83, 184, 227, 67, 76, 14, 117, 11, 49, 4,
        ],
        [
            80, 29, 16, 98, 226, 236, 172, 96, 143, 116, 127, 164, 211, 167, 195, 144, 90, 35, 30,
            37, 175, 148, 5, 193, 47, 74, 7, 127, 248, 111, 208, 46,
        ],
        [
            222, 13, 22, 172, 64, 195, 247, 45, 137, 255, 220, 206, 106, 138, 183, 118, 8, 57, 185,
            88, 175, 128, 155, 135, 117, 242, 66, 145, 188, 95, 9, 0,
        ],
        [
            52, 0, 238, 239, 30, 191, 93, 152, 16, 76, 27, 30, 98, 136, 166, 19, 138, 0, 139, 247,
            149, 8, 220, 50, 201, 182, 229, 239, 136, 191, 87, 4,
        ],
    ],
    [
        [
            105, 215, 131, 127, 67, 108, 143, 18, 162, 181, 248, 61, 218, 82, 232, 43, 171, 70,
            102, 12, 197, 231, 57, 224, 87, 132, 189, 40, 174, 67, 205, 23,
        ],
        [
            193, 35, 208, 49, 156, 76, 127, 125, 37, 174, 206, 28, 23, 74, 214, 223, 88, 221, 47,
            248, 110, 251, 2, 41, 38, 65, 45, 19, 179, 187, 63, 24,
        ],
        [
            178, 98, 110, 6, 17, 23, 79, 37, 105, 148, 237, 89, 194, 184, 226, 85, 19, 158, 126,
            112, 157, 16, 240, 153, 60, 195, 46, 172, 150, 229, 162, 31,
        ],
        [
            167, 160, 123, 55, 171, 70, 72, 120, 175, 85, 129, 30, 118, 187, 31, 35, 13, 168, 186,
            193, 81, 56, 60, 32, 132, 242, 55, 150, 126, 46, 209, 22,
        ],
        [
            216, 130, 207, 144, 87, 74, 175, 79, 231, 2, 123, 17, 64, 227, 169, 21, 190, 76, 35,
            230, 219, 83, 211, 189, 109, 218, 239, 234, 76, 117, 33, 56,
        ],
        [
            118, 63, 187, 184, 88, 54, 195, 72, 172, 236, 147, 107, 35, 109, 68, 57, 16, 172, 137,
            76, 60, 158, 232, 92, 36, 25, 85, 135, 175, 81, 53, 26,
        ],
        [
            187, 33, 86, 227, 140, 59, 212, 98, 155, 221, 32, 11, 9, 134, 134, 226, 5, 117, 206,
            129, 177, 250, 255, 43, 238, 111, 101, 79, 223, 221, 163, 29,
        ],
        [
            77, 49, 10, 237, 205, 191, 201, 163, 163, 225, 93, 46, 239, 70, 220, 239, 203, 63, 156,
            198, 130, 51, 176, 113, 190, 109, 84, 182, 74, 59, 114, 39,
        ],
    ],
    [
        [
            74, 223, 30, 129, 173, 226, 128, 237, 41, 212, 70, 197, 194, 114, 220, 197, 31, 248,
            240, 233, 234, 74, 76, 155, 238, 120, 92, 217, 121, 41, 54, 37,
        ],
        [
            51, 18, 252, 47, 138, 218, 156, 211, 146, 28, 232, 59, 4, 122, 229, 8, 158, 30, 122,
            176, 61, 43, 138, 102, 233, 3, 30, 53, 210, 16, 238, 55,
        ],
        [
            13, 204, 107, 141, 43, 181, 190, 161, 141, 7, 221, 249, 6, 181, 19, 141, 27, 119, 205,
            25, 130, 213, 121, 24, 22, 224, 51, 59, 144, 90, 149, 49,
        ],
        [
            133, 62, 165, 170, 251, 128, 218, 36, 124, 28, 187, 255, 156, 154, 117, 121, 8, 219,
            124, 166, 67, 248, 133, 158, 87, 87, 170, 11, 207, 38, 240, 58,
        ],
        [
            199, 143, 63, 27, 146, 11, 253, 73, 185, 122, 150, 33, 197, 181, 179, 42, 189, 194,
            184, 127, 35, 251, 33, 206, 226, 117, 218, 153, 62, 40, 165, 27,
        ],
        [
            221, 43, 193, 158, 219, 220, 86, 127, 115, 33, 220, 90, 187, 231, 111, 152, 51, 78,
            106, 32, 181, 71, 184, 57, 181, 23, 124, 156, 31, 201, 147, 36,
        ],
        [
            64, 246, 138, 146, 102, 47, 250, 226, 46, 41, 105, 241, 112, 20, 248, 250, 148, 165, 0,
            187, 7, 146, 141, 190, 104, 164, 180, 143, 101, 173, 100, 1,
        ],
        [
            201, 115, 121, 111, 136, 21, 12, 57, 218, 66, 99, 202, 42, 227, 3, 27, 78, 20, 74, 12,
            214, 4, 15, 4, 124, 34, 162, 49, 225, 225, 84, 35,
        ],
    ],
    [
        [
            159, 111, 158, 169, 189, 141, 8, 1, 137, 192, 3, 165, 87, 99, 174, 203, 182, 162, 63,
            5, 133, 2, 254, 170, 26, 95, 119, 225, 150, 113, 216, 42,
        ],
        [
            191, 94, 75, 194, 137, 49, 195, 153, 50, 60, 153, 207, 98, 138, 2, 142, 34, 80, 150,
            253, 239, 160, 29, 220, 153, 59, 121, 254, 188, 74, 125, 24,
        ],
        [
            65, 69, 79, 44, 99, 131, 67, 225, 79, 150, 152, 86, 182, 50, 214, 170, 248, 1, 143,
            244, 5, 44, 114, 55, 193, 251, 160, 170, 252, 233, 58, 6,
        ],
        [
            125, 12, 108, 193, 44, 207, 226, 157, 243, 134, 14, 10, 152, 189, 13, 84, 231, 189,
            248, 124, 69, 182, 227, 163, 51, 137, 150, 100, 10, 149, 241, 47,
        ],
        [
            119, 43, 37, 67, 59, 238, 239, 138, 18, 145, 98, 57, 167, 86, 44, 100, 66, 200, 25, 99,
            156, 250, 80, 209, 18, 132, 112, 85, 118, 64, 126, 52,
        ],
        [
            148, 161, 79, 36, 118, 111, 65, 248, 70, 153, 76, 152, 1, 51, 200, 80, 24, 160, 250,
            83, 1, 173, 246, 44, 246, 212, 29, 165, 76, 238, 138, 29,
        ],
        [
            224, 67, 53, 101, 225, 107, 169, 247, 117, 223, 10, 22, 162, 0, 68, 71, 8, 158, 253,
            123, 54, 205, 198, 242, 213, 57, 146, 153, 136, 212, 158, 38,
        ],
        [
            79, 7, 45, 28, 3, 130, 10, 184, 57, 175, 24, 207, 238, 145, 124, 114, 193, 80, 179,
            205, 12, 85, 182, 192, 23, 119, 23, 39, 77, 105, 242, 1,
        ],
    ],
    [
        [
            28, 128, 149, 54, 197, 184, 228, 248, 116, 22, 130, 218, 24, 0, 138, 240, 183, 99, 240,
            12, 76, 193, 95, 150, 233, 40, 47, 24, 242, 12, 142, 47,
        ],
        [
            123, 40, 155, 92, 163, 34, 144, 128, 132, 10, 205, 21, 139, 110, 152, 94, 154, 247,
            184, 195, 162, 72, 42, 127, 9, 174, 92, 158, 164, 26, 100, 12,
        ],
        [
            14, 7, 174, 73, 235, 206, 1, 187, 94, 147, 150, 246, 59, 146, 80, 158, 157, 229, 197,
            155, 62, 159, 22, 78, 40, 29, 26, 202, 230, 40, 230, 13,
        ],
        [
            195, 197, 87, 192, 158, 100, 3, 125, 177, 187, 3, 10, 2, 65, 28, 26, 39, 28, 197, 134,
            229, 2, 112, 21, 37, 217, 82, 158, 116, 16, 94, 49,
        ],
        [
            172, 162, 178, 197, 3, 150, 82, 223, 22, 71, 179, 84, 232, 56, 180, 151, 162, 65, 162,
            143, 184, 35, 8, 61, 198, 127, 43, 26, 133, 4, 118, 13,
        ],
        [
            147, 99, 145, 67, 186, 57, 231, 205, 201, 14, 157, 179, 190, 189, 149, 228, 28, 139,
            204, 174, 82, 37, 211, 157, 181, 226, 62, 7, 23, 186, 132, 9,
        ],
        [
            63, 13, 19, 112, 241, 89, 54, 26, 108, 209, 48, 226, 212, 239, 141, 183, 182, 53, 89,
            85, 39, 159, 182, 132, 216, 155, 31, 24, 24, 227, 110, 24,
        ],
        [
            132, 212, 75, 16, 39, 54, 191, 5, 93, 187, 110, 211, 100, 141, 231, 28, 174, 152, 111,
            97, 79, 208, 247, 137, 227, 192, 138, 12, 245, 160, 126, 58,
        ],
    ],
    [
        [
            57, 61, 34, 232, 122, 223, 251, 191, 227, 31, 48, 207, 80, 199, 85, 61, 111, 163, 170,
            122, 178, 118, 171, 213, 158, 236, 149, 204, 21, 247, 230, 45,
        ],
        [
            201, 11, 237, 229, 154, 110, 13, 228, 79, 89, 39, 129, 202, 197, 20, 88, 156, 67, 37,
            139, 244, 101, 143, 48, 191, 179, 87, 56, 51, 163, 254, 35,
        ],
        [
            189, 59, 221, 156, 135, 170, 47, 88, 37, 153, 124, 23, 240, 241, 143, 128, 239, 84,
            129, 74, 49, 122, 130, 42, 224, 225, 49, 79, 155, 231, 138, 56,
        ],
        [
            102, 84, 213, 34, 181, 199, 13, 179, 15, 120, 75, 185, 191, 243, 159, 9, 133, 227, 154,
            101, 229, 174, 195, 60, 255, 132, 107, 167, 95, 69, 33, 4,
        ],
        [
            25, 164, 67, 51, 123, 35, 90, 91, 146, 113, 171, 160, 223, 19, 224, 92, 86, 80, 199,
            245, 3, 142, 211, 254, 49, 213, 47, 192, 189, 203, 38, 25,
        ],
        [
            182, 20, 78, 172, 95, 91, 88, 232, 255, 5, 234, 10, 37, 83, 71, 199, 185, 253, 140, 85,
            3, 212, 11, 220, 17, 49, 139, 176, 192, 166, 204, 20,
        ],
        [
            45, 29, 246, 97, 225, 42, 223, 183, 126, 52, 58, 176, 37, 70, 206, 18, 3, 164, 120,
            116, 79, 221, 183, 150, 196, 254, 148, 43, 137, 252, 28, 38,
        ],
        [
            227, 92, 88, 54, 209, 207, 103, 86, 224, 243, 183, 129, 108, 83, 157, 39, 226, 85, 248,
            33, 168, 76, 25, 24, 108, 14, 210, 193, 108, 94, 200, 37,
        ],
    ],
    [
        [
            38, 187, 218, 84, 238, 63, 122, 136, 4, 166, 244, 7, 44, 171, 39, 199, 254, 104, 13,
            76, 206, 106, 51, 146, 31, 98, 45, 93, 157, 232, 88, 24,
        ],
        [
            50, 151, 220, 232, 125, 38, 99, 189, 183, 233, 229, 180, 209, 248, 94, 202, 124, 98,
            68, 113, 182, 128, 225, 187, 104, 97, 240, 97, 23, 2, 78, 40,
        ],
        [
            185, 234, 164, 16, 117, 218, 213, 138, 219, 251, 188, 24, 130, 13, 8, 132, 114, 179,
            127, 8, 199, 15, 108, 123, 248, 125, 98, 114, 199, 49, 151, 29,
        ],
        [
            197, 33, 58, 236, 135, 100, 72, 143, 82, 48, 152, 244, 244, 128, 244, 135, 11, 49, 99,
            223, 13, 28, 192, 60, 14, 249, 122, 63, 10, 229, 13, 48,
        ],
        [
            34, 105, 164, 85, 31, 66, 87, 61, 158, 209, 230, 70, 247, 84, 104, 45, 105, 105, 183,
            10, 12, 208, 229, 125, 81, 87, 184, 49, 231, 76, 2, 0,
        ],
        [
            203, 119, 128, 246, 182, 199, 111, 67, 47, 244, 4, 182, 195, 130, 47, 70, 1, 125, 199,
            21, 58, 32, 136, 148, 237, 67, 244, 121, 78, 17, 162, 62,
        ],
        [
            168, 51, 187, 80, 154, 99, 32, 100, 2, 198, 3, 45, 40, 246, 36, 220, 72, 108, 194, 78,
            218, 158, 38, 131, 247, 253, 100, 240, 49, 249, 198, 23,
        ],
        [
            229, 127, 250, 99, 148, 180, 224, 70, 25, 12, 182, 194, 5, 32, 145, 6, 168, 60, 183,
            43, 56, 235, 99, 57, 251, 84, 71, 154, 221, 208, 35, 17,
        ],
    ],
    [
        [
            135, 144, 34, 253, 72, 230, 20, 130, 152, 143, 241, 69, 97, 189, 237, 136, 2, 47, 52,
            84, 89, 203, 68, 175, 237, 219, 177, 177, 133, 204, 76, 38,
        ],
        [
            77, 97, 144, 253, 187, 106, 149, 179, 219, 51, 135, 69, 243, 163, 159, 228, 37, 201,
            106, 187, 166, 131, 150, 20, 114, 178, 182, 177, 70, 99, 99, 53,
        ],
        [
            147, 219, 248, 110, 243, 241, 185, 70, 69, 79, 14, 33, 101, 231, 193, 76, 65, 21, 253,
            231, 70, 24, 233, 13, 131, 80, 209, 106, 209, 167, 92, 9,
        ],
        [
            53, 151, 111, 202, 18, 147, 183, 105, 4, 44, 90, 43, 95, 62, 49, 201, 64, 71, 81, 226,
            235, 224, 82, 105, 110, 95, 41, 230, 242, 67, 234, 60,
        ],
        [
            175, 101, 205, 91, 4, 241, 57, 85, 193, 202, 5, 102, 39, 134, 177, 26, 219, 161, 126,
            10, 101, 104, 156, 171, 95, 235, 65, 35, 42, 178, 106, 14,
        ],
        [
            111, 115, 75, 121, 96, 212, 237, 44, 87, 85, 98, 147, 98, 58, 146, 199, 151, 7, 148,
            211, 251, 253, 34, 26, 119, 43, 225, 118, 17, 152, 137, 4,
        ],
        [
            94, 127, 124, 42, 65, 235, 150, 132, 214, 217, 183, 223, 216, 49, 160, 129, 91, 51,
            108, 194, 1, 248, 216, 206, 78, 155, 58, 141, 66, 191, 240, 52,
        ],
        [
            243, 42, 140, 79, 233, 110, 28, 104, 82, 74, 76, 228, 123, 35, 55, 147, 149, 179, 155,
            6, 149, 83, 50, 91, 175, 204, 211, 53, 217, 190, 246, 54,
        ],
    ],
    [
        [
            102, 58, 156, 115, 172, 173, 89, 128, 136, 214, 38, 179, 183, 156, 21, 250, 113, 241,
            164, 127, 126, 160, 49, 210, 173, 235, 100, 32, 139, 122, 165, 35,
        ],
        [
            214, 61, 123, 53, 199, 34, 204, 112, 236, 150, 157, 131, 97, 109, 159, 184, 204, 152,
            112, 42, 171, 10, 229, 65, 150, 74, 199, 231, 228, 205, 18, 17,
        ],
        [
            143, 44, 3, 188, 135, 28, 111, 222, 202, 125, 214, 208, 209, 38, 114, 207, 38, 62, 183,
            254, 238, 120, 166, 219, 143, 85, 84, 37, 103, 96, 97, 21,
        ],
        [
            148, 177, 168, 127, 63, 64, 150, 26, 15, 73, 199, 97, 136, 44, 223, 51, 113, 222, 76,
            61, 194, 129, 250, 38, 142, 42, 116, 132, 186, 150, 171, 36,
        ],
        [
            174, 26, 21, 203, 190, 33, 43, 56, 157, 185, 148, 218, 225, 226, 57, 196, 252, 222, 33,
            9, 77, 119, 244, 185, 160, 238, 70, 231, 143, 213, 138, 23,
        ],
        [
            101, 125, 198, 161, 7, 227, 60, 17, 249, 182, 87, 179, 32, 124, 66, 138, 40, 159, 28,
            75, 177, 137, 158, 141, 246, 1, 191, 232, 164, 198, 247, 21,
        ],
        [
            71, 82, 237, 89, 81, 217, 75, 98, 104, 141, 129, 130, 83, 240, 123, 156, 241, 48, 199,
            246, 16, 136, 10, 104, 198, 185, 106, 103, 17, 47, 82, 26,
        ],
        [
            129, 197, 126, 12, 63, 25, 235, 35, 94, 93, 239, 89, 127, 214, 155, 151, 103, 54, 34,
            189, 143, 198, 178, 1, 201, 250, 74, 210, 69, 22, 148, 39,
        ],
    ],
    [
        [
            53, 13, 65, 249, 224, 244, 4, 181, 148, 177, 238, 124, 28, 105, 189, 23, 127, 239, 162,
            120, 47, 162, 0, 111, 49, 187, 200, 30, 158, 169, 27, 4,
        ],
        [
            248, 118, 85, 63, 106, 89, 195, 84, 176, 202, 161, 63, 252, 62, 59, 40, 88, 158, 170,
            94, 6, 72, 41, 21, 123, 85, 249, 32, 173, 229, 161, 2,
        ],
        [
            103, 42, 197, 89, 189, 15, 89, 67, 61, 197, 214, 87, 86, 184, 14, 73, 217, 129, 96,
            108, 150, 253, 197, 204, 243, 238, 213, 142, 181, 53, 122, 49,
        ],
        [
            53, 243, 169, 200, 115, 180, 90, 42, 194, 45, 81, 249, 67, 229, 85, 242, 10, 164, 168,
            32, 190, 172, 71, 122, 128, 240, 32, 64, 163, 92, 204, 18,
        ],
        [
            96, 253, 241, 14, 76, 21, 60, 76, 38, 190, 1, 3, 132, 19, 108, 102, 203, 218, 19, 35,
            233, 105, 192, 81, 249, 230, 216, 144, 18, 105, 20, 4,
        ],
        [
            238, 144, 36, 34, 174, 202, 176, 61, 148, 100, 1, 105, 59, 86, 166, 59, 95, 28, 8, 172,
            145, 46, 63, 159, 175, 159, 77, 231, 12, 155, 103, 3,
        ],
        [
            201, 112, 31, 187, 183, 207, 186, 133, 24, 209, 163, 240, 219, 228, 133, 40, 55, 195,
            146, 4, 32, 38, 192, 110, 182, 143, 71, 136, 27, 200, 108, 34,
        ],
        [
            254, 24, 24, 47, 115, 110, 172, 93, 17, 177, 4, 93, 201, 121, 119, 183, 19, 248, 52,
            86, 115, 74, 223, 106, 119, 22, 187, 194, 36, 41, 19, 57,
        ],
    ],
    [
        [
            173, 129, 213, 38, 131, 94, 122, 115, 246, 58, 95, 40, 216, 85, 100, 239, 194, 10, 35,
            101, 93, 146, 185, 130, 211, 233, 3, 128, 56, 90, 21, 49,
        ],
        [
            86, 52, 235, 254, 233, 233, 159, 162, 150, 66, 178, 36, 195, 157, 26, 251, 29, 233,
            171, 56, 147, 154, 194, 16, 174, 161, 56, 198, 46, 150, 78, 10,
        ],
        [
            155, 220, 126, 74, 39, 181, 166, 17, 12, 176, 143, 186, 34, 74, 38, 241, 35, 77, 7, 66,
            90, 91, 119, 241, 235, 16, 35, 188, 240, 12, 225, 44,
        ],
        [
            26, 128, 31, 93, 94, 250, 172, 66, 207, 87, 203, 139, 18, 19, 180, 251, 45, 132, 118,
            214, 142, 184, 170, 6, 129, 111, 154, 229, 206, 93, 43, 62,
        ],
        [
            74, 234, 78, 87, 247, 140, 239, 158, 208, 124, 182, 64, 135, 123, 226, 208, 143, 9, 28,
            47, 76, 49, 147, 165, 163, 173, 186, 221, 64, 18, 242, 4,
        ],
        [
            102, 15, 228, 6, 109, 75, 103, 53, 235, 105, 148, 112, 14, 209, 118, 15, 206, 38, 211,
            191, 45, 59, 17, 164, 5, 97, 207, 172, 113, 174, 151, 12,
        ],
        [
            182, 198, 44, 72, 215, 80, 214, 74, 36, 187, 96, 36, 75, 122, 251, 79, 111, 141, 104,
            211, 96, 27, 154, 137, 81, 155, 237, 118, 198, 220, 240, 36,
        ],
        [
            210, 226, 153, 190, 119, 238, 200, 137, 186, 126, 198, 229, 234, 35, 61, 181, 214, 235,
            147, 87, 205, 50, 66, 220, 36, 197, 22, 203, 224, 11, 10, 60,
        ],
    ],
    [
        [
            9, 195, 159, 120, 189, 176, 219, 20, 99, 101, 117, 114, 93, 166, 142, 90, 52, 200, 98,
            114, 97, 126, 34, 96, 118, 185, 215, 251, 148, 77, 87, 19,
        ],
        [
            50, 130, 54, 156, 8, 243, 245, 21, 115, 243, 141, 50, 57, 163, 190, 162, 186, 254, 96,
            21, 221, 183, 159, 19, 150, 49, 249, 178, 169, 131, 75, 56,
        ],
        [
            109, 254, 22, 3, 199, 134, 177, 80, 204, 161, 10, 173, 237, 126, 178, 229, 70, 30, 3,
            246, 202, 161, 90, 229, 160, 16, 169, 202, 104, 39, 197, 1,
        ],
        [
            95, 179, 233, 6, 144, 172, 174, 142, 67, 4, 33, 137, 72, 48, 108, 152, 88, 201, 234,
            221, 234, 154, 179, 39, 209, 113, 0, 188, 222, 32, 97, 9,
        ],
        [
            175, 222, 53, 19, 156, 69, 205, 48, 91, 68, 159, 242, 127, 200, 159, 55, 242, 116, 13,
            2, 159, 225, 193, 234, 222, 33, 114, 12, 105, 215, 17, 51,
        ],
        [
            123, 223, 171, 72, 167, 121, 200, 176, 52, 160, 97, 178, 109, 5, 35, 224, 48, 106, 237,
            8, 72, 160, 22, 211, 106, 246, 196, 217, 48, 220, 133, 7,
        ],
        [
            179, 44, 209, 79, 15, 63, 82, 9, 228, 138, 76, 79, 216, 127, 147, 79, 251, 107, 235,
            27, 89, 218, 138, 32, 150, 255, 58, 171, 187, 54, 11, 59,
        ],
        [
            135, 219, 120, 88, 47, 222, 125, 235, 159, 179, 137, 207, 239, 76, 254, 182, 26, 255,
            137, 9, 114, 170, 242, 59, 147, 178, 82, 105, 120, 163, 165, 56,
        ],
    ],
    [
        [
            49, 245, 74, 15, 79, 151, 114, 223, 88, 204, 54, 17, 223, 211, 56, 243, 140, 147, 231,
            194, 145, 209, 115, 122, 133, 53, 65, 38, 227, 232, 194, 1,
        ],
        [
            139, 234, 196, 211, 16, 253, 195, 182, 204, 43, 70, 215, 131, 12, 156, 176, 1, 181,
            206, 180, 185, 145, 168, 245, 102, 147, 163, 131, 54, 15, 80, 23,
        ],
        [
            165, 107, 0, 13, 216, 120, 63, 220, 127, 136, 126, 22, 114, 49, 104, 34, 144, 83, 117,
            103, 73, 69, 67, 4, 94, 122, 92, 49, 45, 19, 203, 1,
        ],
        [
            57, 67, 199, 232, 154, 171, 244, 107, 180, 16, 176, 181, 58, 191, 139, 1, 236, 109,
            133, 154, 204, 81, 197, 250, 102, 135, 191, 107, 255, 138, 122, 2,
        ],
        [
            131, 246, 101, 113, 122, 177, 203, 36, 187, 253, 75, 137, 138, 91, 125, 116, 201, 131,
            188, 164, 176, 158, 228, 168, 44, 79, 52, 106, 14, 190, 26, 15,
        ],
        [
            151, 24, 233, 35, 43, 183, 63, 142, 143, 124, 217, 64, 36, 225, 65, 100, 127, 232, 86,
            46, 96, 33, 120, 128, 99, 189, 77, 103, 75, 50, 238, 6,
        ],
        [
            34, 188, 40, 2, 222, 186, 165, 32, 19, 108, 101, 182, 124, 106, 225, 221, 244, 29, 251,
            184, 252, 107, 39, 20, 41, 194, 2, 225, 147, 17, 254, 11,
        ],
        [
            163, 210, 154, 39, 136, 44, 144, 169, 139, 196, 107, 36, 58, 100, 18, 249, 234, 224,
            179, 133, 22, 54, 204, 86, 254, 146, 34, 170, 164, 208, 141, 21,
        ],
    ],
    [
        [
            212, 57, 196, 255, 116, 10, 198, 162, 235, 70, 204, 155, 155, 252, 182, 251, 108, 89,
            15, 66, 200, 235, 157, 30, 193, 55, 170, 25, 48, 150, 174, 46,
        ],
        [
            73, 15, 246, 147, 31, 114, 184, 23, 189, 135, 237, 223, 46, 217, 219, 63, 199, 115,
            114, 227, 255, 93, 219, 162, 47, 89, 95, 196, 97, 180, 109, 13,
        ],
        [
            45, 49, 142, 52, 248, 150, 56, 109, 99, 149, 25, 160, 166, 240, 96, 241, 94, 77, 44,
            217, 50, 231, 124, 213, 185, 3, 125, 168, 173, 6, 149, 32,
        ],
        [
            244, 199, 196, 141, 35, 8, 94, 117, 100, 81, 106, 254, 233, 180, 12, 237, 206, 142,
            228, 141, 79, 50, 119, 16, 94, 134, 158, 85, 242, 31, 30, 28,
        ],
        [
            250, 118, 28, 156, 124, 151, 194, 243, 252, 135, 49, 133, 184, 151, 1, 52, 72, 62, 209,
            219, 159, 239, 199, 118, 233, 81, 13, 246, 154, 95, 91, 43,
        ],
        [
            148, 41, 120, 124, 79, 58, 248, 13, 19, 168, 125, 163, 96, 7, 180, 190, 86, 81, 43,
            249, 183, 109, 169, 50, 192, 65, 153, 58, 251, 95, 239, 21,
        ],
        [
            5, 221, 39, 209, 61, 5, 170, 169, 239, 185, 230, 104, 97, 144, 169, 124, 53, 203, 90,
            37, 53, 149, 7, 245, 142, 65, 195, 158, 58, 92, 201, 37,
        ],
        [
            109, 213, 234, 137, 252, 160, 129, 210, 249, 154, 254, 3, 240, 28, 166, 22, 170, 196,
            203, 240, 186, 131, 234, 216, 184, 66, 6, 144, 171, 241, 92, 1,
        ],
    ],
    [
        [
            146, 223, 54, 195, 85, 234, 248, 34, 125, 125, 215, 25, 174, 180, 216, 60, 247, 5, 128,
            182, 151, 217, 164, 205, 118, 170, 184, 236, 149, 62, 161, 31,
        ],
        [
            158, 58, 122, 20, 197, 199, 64, 124, 63, 206, 24, 117, 236, 141, 106, 97, 204, 4, 166,
            196, 2, 159, 166, 146, 177, 180, 47, 37, 223, 86, 168, 50,
        ],
        [
            191, 163, 218, 160, 251, 127, 243, 25, 144, 88, 178, 19, 43, 249, 66, 139, 164, 15,
            131, 116, 254, 213, 218, 126, 7, 46, 54, 49, 182, 86, 54, 49,
        ],
        [
            176, 149, 25, 128, 24, 242, 7, 223, 3, 207, 221, 176, 138, 82, 192, 185, 45, 4, 156,
            186, 96, 62, 244, 74, 238, 175, 172, 183, 208, 96, 65, 23,
        ],
        [
            134, 58, 211, 81, 238, 63, 222, 132, 104, 11, 49, 26, 146, 71, 92, 37, 4, 252, 51, 65,
            9, 121, 189, 239, 11, 47, 203, 247, 143, 50, 132, 54,
        ],
        [
            7, 85, 156, 94, 147, 75, 20, 40, 58, 92, 141, 16, 69, 205, 224, 177, 209, 40, 15, 100,
            7, 13, 58, 29, 145, 223, 215, 21, 191, 226, 26, 26,
        ],
        [
            105, 133, 138, 132, 202, 221, 113, 164, 179, 179, 245, 43, 54, 164, 13, 169, 7, 52,
            252, 75, 143, 31, 125, 18, 194, 160, 85, 116, 60, 199, 14, 15,
        ],
        [
            222, 114, 216, 202, 53, 243, 194, 143, 86, 50, 1, 196, 113, 133, 21, 229, 241, 197,
            143, 128, 57, 76, 134, 162, 241, 28, 204, 249, 86, 196, 47, 10,
        ],
    ],
    [
        [
            140, 34, 41, 75, 234, 68, 6, 132, 20, 133, 22, 50, 10, 116, 214, 160, 101, 67, 255,
            164, 173, 182, 183, 76, 45, 64, 169, 10, 74, 111, 151, 52,
        ],
        [
            71, 77, 105, 249, 106, 189, 164, 179, 236, 36, 83, 6, 53, 110, 222, 110, 253, 41, 11,
            117, 83, 57, 16, 185, 170, 25, 13, 28, 45, 60, 212, 15,
        ],
        [
            0, 204, 125, 241, 248, 116, 213, 191, 28, 250, 46, 137, 128, 107, 187, 108, 190, 128,
            27, 164, 49, 254, 86, 174, 177, 238, 60, 155, 114, 217, 225, 5,
        ],
        [
            5, 158, 38, 49, 215, 186, 131, 56, 21, 200, 199, 31, 227, 172, 45, 39, 28, 45, 55, 159,
            55, 126, 34, 40, 255, 233, 126, 173, 178, 210, 125, 17,
        ],
        [
            113, 248, 161, 198, 225, 135, 51, 248, 203, 94, 37, 238, 150, 41, 243, 222, 105, 48,
            92, 81, 134, 6, 41, 57, 203, 211, 57, 112, 143, 221, 240, 31,
        ],
        [
            82, 243, 29, 216, 118, 139, 147, 98, 208, 80, 146, 98, 142, 127, 201, 193, 94, 5, 240,
            169, 154, 187, 169, 187, 181, 108, 16, 176, 52, 153, 185, 40,
        ],
        [
            180, 4, 216, 113, 228, 196, 165, 187, 181, 24, 182, 129, 143, 90, 54, 102, 39, 216,
            246, 28, 202, 130, 130, 164, 164, 23, 204, 3, 93, 252, 160, 19,
        ],
        [
            236, 183, 190, 138, 148, 47, 164, 106, 37, 140, 60, 83, 100, 87, 25, 94, 3, 155, 36,
            217, 2, 9, 122, 48, 59, 74, 238, 62, 235, 254, 76, 63,
        ],
    ],
    [
        [
            145, 2, 88, 168, 72, 120, 32, 138, 82, 71, 120, 197, 202, 217, 99, 230, 103, 161, 20,
            69, 139, 231, 10, 20, 223, 19, 104, 101, 97, 42, 121, 13,
        ],
        [
            249, 255, 74, 127, 202, 45, 113, 91, 135, 69, 14, 75, 180, 46, 26, 41, 213, 68, 168,
            203, 107, 4, 15, 223, 81, 162, 37, 185, 230, 130, 218, 24,
        ],
        [
            229, 23, 249, 95, 149, 172, 105, 186, 130, 22, 109, 90, 114, 118, 91, 227, 63, 199,
            106, 119, 238, 16, 55, 220, 45, 208, 52, 63, 237, 194, 126, 22,
        ],
        [
            236, 224, 208, 190, 253, 62, 123, 39, 67, 207, 40, 133, 104, 99, 14, 59, 29, 45, 212,
            230, 238, 156, 59, 211, 201, 28, 6, 135, 6, 170, 41, 2,
        ],
        [
            117, 186, 35, 47, 195, 87, 43, 185, 50, 136, 121, 104, 52, 109, 241, 221, 160, 100, 34,
            72, 217, 36, 69, 240, 185, 82, 33, 24, 82, 146, 129, 22,
        ],
        [
            40, 216, 70, 160, 218, 185, 168, 230, 67, 74, 241, 72, 30, 56, 73, 212, 29, 131, 210,
            56, 61, 138, 16, 72, 74, 208, 175, 95, 233, 23, 68, 50,
        ],
        [
            250, 249, 247, 170, 88, 230, 224, 198, 25, 123, 18, 15, 231, 56, 213, 167, 70, 89, 25,
            233, 30, 104, 240, 125, 246, 136, 189, 24, 2, 217, 184, 57,
        ],
        [
            80, 221, 4, 106, 97, 0, 93, 65, 187, 49, 74, 27, 59, 42, 216, 255, 178, 111, 209, 168,
            148, 213, 149, 222, 244, 237, 45, 14, 247, 140, 147, 24,
        ],
    ],
    [
        [
            103, 89, 59, 35, 57, 15, 73, 153, 88, 110, 137, 148, 85, 109, 160, 59, 21, 153, 234,
            11, 211, 42, 66, 211, 68, 38, 15, 82, 233, 242, 246, 56,
        ],
        [
            96, 192, 241, 71, 162, 188, 189, 54, 176, 170, 74, 155, 210, 182, 205, 213, 187, 53,
            60, 231, 143, 251, 199, 78, 247, 216, 140, 90, 128, 175, 11, 44,
        ],
        [
            10, 136, 130, 80, 243, 131, 6, 45, 149, 177, 198, 249, 148, 33, 121, 208, 241, 179,
            254, 129, 63, 128, 87, 98, 82, 74, 178, 206, 136, 237, 177, 54,
        ],
        [
            140, 174, 124, 27, 24, 16, 89, 115, 91, 12, 227, 201, 107, 37, 48, 219, 159, 96, 137,
            160, 17, 96, 226, 109, 216, 54, 158, 68, 1, 57, 62, 21,
        ],
        [
            184, 104, 44, 147, 115, 104, 146, 195, 48, 83, 156, 79, 62, 14, 137, 213, 20, 2, 64,
            71, 19, 82, 188, 223, 162, 170, 239, 238, 32, 92, 100, 27,
        ],
        [
            207, 172, 15, 212, 186, 64, 30, 3, 26, 41, 32, 48, 108, 230, 73, 168, 130, 44, 140,
            210, 212, 140, 38, 79, 89, 154, 72, 38, 234, 29, 111, 46,
        ],
        [
            169, 251, 74, 195, 182, 161, 15, 245, 9, 175, 45, 68, 112, 77, 254, 175, 101, 57, 117,
            229, 114, 240, 162, 49, 79, 195, 76, 222, 24, 29, 205, 49,
        ],
        [
            66, 42, 240, 58, 215, 243, 13, 29, 121, 8, 251, 212, 227, 43, 108, 116, 167, 146, 112,
            29, 157, 28, 232, 63, 40, 136, 60, 181, 113, 116, 136, 9,
        ],
    ],
    [
        [
            222, 250, 15, 159, 22, 192, 63, 242, 244, 9, 170, 199, 36, 12, 22, 135, 36, 103, 27,
            30, 210, 19, 220, 227, 198, 202, 170, 11, 171, 226, 120, 36,
        ],
        [
            13, 233, 60, 66, 94, 62, 80, 17, 51, 96, 92, 52, 194, 151, 201, 43, 145, 244, 136, 54,
            62, 228, 178, 17, 237, 50, 43, 86, 226, 186, 205, 39,
        ],
        [
            17, 178, 141, 143, 150, 198, 188, 109, 202, 15, 246, 81, 81, 23, 194, 33, 100, 215, 33,
            38, 128, 94, 173, 211, 250, 255, 185, 53, 26, 15, 100, 1,
        ],
        [
            29, 13, 207, 26, 98, 99, 13, 249, 166, 223, 134, 57, 202, 243, 56, 134, 47, 86, 77,
            174, 126, 82, 11, 136, 206, 200, 80, 142, 133, 59, 226, 45,
        ],
        [
            30, 246, 189, 45, 47, 225, 241, 181, 46, 77, 149, 202, 125, 19, 150, 189, 61, 110, 166,
            235, 18, 47, 82, 23, 31, 253, 45, 168, 199, 197, 100, 13,
        ],
        [
            163, 144, 73, 57, 237, 138, 87, 249, 81, 92, 134, 158, 7, 198, 181, 121, 235, 165, 27,
            22, 195, 193, 81, 182, 82, 147, 125, 185, 230, 127, 212, 41,
        ],
        [
            99, 53, 81, 32, 186, 67, 123, 181, 118, 216, 51, 213, 48, 132, 99, 168, 17, 212, 8,
            180, 183, 132, 100, 75, 201, 55, 13, 103, 31, 134, 12, 29,
        ],
        [
            204, 31, 149, 83, 95, 245, 53, 231, 200, 193, 124, 32, 42, 222, 91, 84, 37, 117, 53,
            116, 166, 28, 165, 6, 7, 88, 156, 0, 217, 37, 205, 21,
        ],
    ],
    [
        [
            133, 125, 183, 101, 61, 147, 72, 149, 191, 124, 2, 158, 189, 180, 148, 247, 242, 130,
            157, 154, 138, 32, 39, 113, 18, 103, 212, 2, 52, 42, 150, 9,
        ],
        [
            155, 153, 34, 249, 57, 253, 43, 212, 131, 227, 138, 238, 250, 38, 95, 115, 158, 179,
            220, 114, 87, 215, 143, 24, 139, 211, 161, 25, 165, 127, 26, 46,
        ],
        [
            248, 24, 188, 125, 43, 117, 180, 162, 154, 30, 111, 77, 63, 80, 222, 213, 39, 94, 105,
            244, 205, 160, 240, 125, 127, 22, 68, 60, 4, 159, 101, 12,
        ],
        [
            202, 182, 101, 247, 202, 184, 249, 165, 48, 157, 78, 244, 161, 129, 213, 241, 85, 55,
            1, 148, 152, 20, 217, 134, 226, 216, 98, 32, 63, 115, 205, 2,
        ],
        [
            12, 227, 112, 199, 148, 87, 71, 67, 63, 109, 117, 99, 32, 35, 55, 108, 13, 99, 205,
            173, 201, 27, 223, 79, 14, 136, 251, 75, 29, 113, 7, 9,
        ],
        [
            53, 122, 33, 1, 148, 172, 48, 164, 95, 139, 231, 211, 62, 152, 233, 208, 6, 67, 40,
            162, 85, 247, 207, 27, 71, 148, 238, 1, 166, 101, 154, 35,
        ],
        [
            161, 30, 55, 238, 200, 130, 23, 62, 183, 234, 222, 236, 186, 80, 177, 71, 226, 12, 18,
            136, 149, 181, 12, 186, 197, 176, 30, 124, 178, 222, 31, 26,
        ],
        [
            220, 42, 168, 151, 75, 170, 70, 232, 113, 157, 111, 72, 209, 182, 214, 18, 203, 62,
            193, 190, 220, 44, 144, 225, 236, 250, 122, 80, 83, 193, 125, 22,
        ],
    ],
    [
        [
            170, 49, 110, 163, 118, 15, 74, 16, 140, 62, 228, 247, 212, 253, 62, 59, 14, 30, 160,
            139, 165, 77, 221, 157, 50, 217, 113, 202, 56, 132, 252, 23,
        ],
        [
            204, 248, 203, 88, 186, 179, 102, 131, 176, 162, 204, 93, 132, 134, 140, 44, 190, 111,
            170, 182, 182, 154, 34, 182, 72, 145, 249, 218, 227, 222, 127, 57,
        ],
        [
            243, 181, 217, 123, 251, 210, 151, 17, 234, 18, 151, 41, 199, 190, 68, 25, 143, 119,
            47, 117, 196, 126, 170, 4, 2, 92, 76, 45, 165, 7, 67, 32,
        ],
        [
            78, 151, 250, 251, 145, 247, 127, 21, 8, 78, 75, 203, 87, 88, 88, 192, 254, 127, 173,
            160, 123, 221, 115, 242, 177, 65, 250, 109, 60, 243, 66, 50,
        ],
        [
            163, 204, 67, 168, 29, 250, 242, 18, 179, 78, 156, 63, 83, 213, 155, 169, 201, 141,
            147, 39, 67, 201, 27, 185, 12, 136, 169, 136, 90, 250, 59, 61,
        ],
        [
            217, 130, 254, 58, 106, 222, 13, 57, 188, 196, 184, 159, 16, 125, 101, 76, 237, 202,
            20, 154, 36, 171, 17, 203, 30, 174, 71, 28, 49, 14, 11, 40,
        ],
        [
            103, 104, 113, 172, 174, 8, 232, 248, 176, 114, 142, 202, 124, 57, 147, 248, 183, 13,
            100, 169, 5, 175, 203, 142, 150, 76, 116, 217, 74, 87, 193, 61,
        ],
        [
            106, 70, 255, 164, 171, 32, 41, 157, 50, 76, 235, 158, 79, 146, 163, 103, 55, 210, 247,
            65, 12, 143, 226, 228, 134, 223, 111, 57, 117, 0, 158, 19,
        ],
    ],
    [
        [
            209, 91, 126, 65, 66, 75, 243, 162, 155, 127, 161, 141, 1, 207, 59, 241, 167, 100, 208,
            242, 202, 136, 148, 80, 250, 199, 246, 122, 195, 21, 17, 50,
        ],
        [
            160, 23, 252, 146, 181, 244, 35, 83, 49, 80, 175, 137, 187, 14, 153, 154, 210, 170,
            253, 239, 189, 159, 66, 83, 46, 47, 203, 221, 74, 96, 231, 31,
        ],
        [
            255, 28, 169, 35, 131, 9, 128, 83, 53, 153, 78, 112, 201, 215, 113, 204, 14, 161, 70,
            21, 43, 234, 162, 8, 50, 202, 178, 229, 163, 231, 75, 0,
        ],
        [
            151, 7, 75, 95, 206, 3, 186, 193, 10, 216, 191, 56, 5, 241, 211, 41, 176, 163, 112,
            226, 119, 198, 67, 29, 86, 77, 125, 189, 42, 244, 37, 63,
        ],
        [
            22, 69, 53, 55, 24, 217, 190, 106, 17, 30, 46, 219, 19, 13, 91, 88, 52, 239, 230, 65,
            172, 249, 117, 14, 212, 76, 50, 219, 63, 85, 85, 44,
        ],
        [
            95, 67, 36, 155, 161, 152, 90, 49, 98, 250, 31, 40, 227, 83, 13, 63, 152, 78, 52, 55,
            179, 174, 109, 115, 237, 188, 143, 119, 39, 205, 240, 52,
        ],
        [
            191, 218, 230, 242, 222, 153, 9, 6, 184, 144, 39, 153, 17, 47, 235, 11, 156, 237, 106,
            25, 226, 22, 204, 91, 174, 13, 118, 159, 67, 192, 180, 11,
        ],
        [
            64, 139, 49, 104, 152, 44, 115, 188, 115, 244, 108, 247, 152, 36, 76, 87, 197, 145,
            122, 252, 164, 179, 33, 139, 178, 33, 91, 180, 55, 198, 116, 28,
        ],
    ],
    [
        [
            121, 138, 217, 165, 137, 32, 102, 122, 2, 132, 235, 209, 189, 143, 126, 154, 255, 192,
            11, 18, 192, 180, 141, 158, 58, 213, 167, 150, 144, 12, 137, 32,
        ],
        [
            37, 68, 105, 192, 246, 223, 110, 73, 212, 170, 113, 192, 94, 48, 75, 145, 180, 144, 57,
            168, 175, 175, 116, 14, 144, 230, 2, 71, 231, 227, 93, 22,
        ],
        [
            55, 239, 250, 161, 54, 154, 33, 206, 228, 249, 45, 126, 12, 187, 171, 233, 179, 141,
            113, 128, 26, 208, 66, 163, 119, 5, 134, 70, 212, 91, 126, 25,
        ],
        [
            211, 218, 167, 184, 136, 234, 50, 153, 235, 240, 121, 105, 75, 92, 134, 203, 54, 9,
            240, 37, 132, 51, 94, 62, 227, 206, 227, 26, 232, 80, 50, 58,
        ],
        [
            247, 153, 165, 91, 186, 187, 160, 101, 181, 94, 109, 64, 195, 54, 10, 53, 122, 95, 94,
            223, 224, 187, 3, 166, 202, 12, 226, 250, 194, 230, 14, 14,
        ],
        [
            84, 40, 61, 75, 7, 180, 84, 252, 149, 110, 235, 145, 230, 102, 43, 59, 84, 177, 15, 72,
            245, 45, 100, 181, 159, 135, 73, 70, 45, 130, 23, 15,
        ],
        [
            176, 95, 212, 132, 225, 23, 127, 234, 119, 13, 97, 153, 63, 184, 203, 20, 158, 129, 71,
            212, 187, 114, 5, 66, 8, 255, 197, 184, 58, 87, 53, 46,
        ],
        [
            16, 127, 94, 241, 202, 213, 70, 104, 66, 253, 252, 251, 32, 90, 82, 235, 212, 171, 176,
            158, 111, 171, 170, 223, 21, 135, 124, 8, 2, 163, 132, 60,
        ],
    ],
    [
        [
            227, 106, 232, 144, 99, 103, 224, 175, 110, 11, 138, 168, 36, 37, 51, 178, 223, 20,
            214, 254, 112, 70, 96, 185, 211, 87, 83, 197, 104, 243, 216, 52,
        ],
        [
            130, 229, 53, 255, 238, 87, 247, 194, 175, 43, 49, 133, 15, 9, 230, 116, 200, 212, 229,
            179, 231, 4, 15, 203, 110, 47, 4, 130, 136, 86, 48, 49,
        ],
        [
            47, 243, 195, 44, 235, 13, 162, 169, 83, 152, 125, 54, 110, 131, 234, 38, 185, 159,
            162, 155, 244, 43, 221, 221, 205, 171, 95, 242, 148, 151, 11, 35,
        ],
        [
            167, 90, 16, 27, 245, 22, 86, 47, 185, 27, 129, 85, 243, 61, 176, 192, 183, 132, 79,
            230, 218, 19, 237, 23, 105, 124, 208, 38, 178, 255, 128, 42,
        ],
        [
            254, 42, 152, 192, 222, 80, 229, 159, 80, 221, 180, 50, 154, 115, 132, 186, 232, 133,
            81, 52, 49, 138, 192, 67, 159, 157, 6, 159, 63, 244, 18, 39,
        ],
        [
            82, 238, 44, 142, 105, 250, 102, 225, 160, 86, 185, 241, 244, 228, 45, 98, 14, 6, 186,
            3, 103, 72, 115, 129, 220, 107, 87, 25, 18, 239, 41, 61,
        ],
        [
            38, 96, 137, 193, 31, 196, 122, 212, 249, 106, 137, 240, 9, 1, 244, 43, 246, 195, 28,
            79, 40, 193, 249, 111, 177, 171, 178, 146, 5, 130, 199, 38,
        ],
        [
            49, 138, 50, 43, 88, 47, 129, 40, 253, 221, 234, 179, 108, 110, 16, 164, 236, 146, 134,
            226, 92, 142, 158, 18, 75, 148, 196, 99, 142, 193, 32, 32,
        ],
    ],
    [
        [
            136, 222, 31, 32, 158, 114, 150, 51, 255, 249, 195, 251, 49, 203, 33, 38, 143, 117,
            158, 139, 35, 54, 168, 163, 36, 5, 4, 24, 21, 107, 32, 8,
        ],
        [
            139, 142, 184, 112, 128, 119, 77, 133, 53, 144, 84, 23, 177, 2, 187, 179, 30, 93, 90,
            138, 73, 70, 129, 214, 185, 249, 38, 44, 114, 66, 31, 60,
        ],
        [
            70, 234, 44, 237, 63, 86, 138, 80, 38, 77, 0, 237, 105, 160, 161, 106, 73, 199, 70,
            235, 14, 227, 136, 202, 144, 196, 163, 81, 111, 106, 82, 47,
        ],
        [
            40, 88, 203, 143, 239, 58, 0, 59, 249, 62, 40, 239, 19, 92, 207, 51, 252, 63, 205, 29,
            238, 11, 59, 181, 106, 136, 22, 181, 36, 21, 177, 16,
        ],
        [
            75, 228, 4, 173, 170, 140, 175, 235, 102, 195, 159, 243, 64, 41, 21, 2, 246, 29, 242,
            42, 219, 89, 14, 181, 195, 64, 36, 206, 219, 19, 201, 43,
        ],
        [
            99, 70, 84, 229, 213, 64, 250, 175, 183, 122, 91, 174, 129, 41, 175, 77, 133, 253, 111,
            83, 20, 196, 125, 248, 230, 8, 189, 219, 198, 132, 117, 49,
        ],
        [
            146, 31, 155, 47, 167, 160, 234, 249, 196, 38, 83, 35, 96, 191, 69, 232, 164, 232, 241,
            225, 121, 189, 98, 1, 45, 228, 7, 52, 137, 87, 156, 55,
        ],
        [
            160, 123, 219, 27, 246, 32, 134, 120, 16, 76, 98, 152, 240, 140, 17, 236, 43, 181, 91,
            154, 12, 80, 78, 141, 209, 68, 179, 131, 134, 238, 214, 29,
        ],
    ],
    [
        [
            79, 127, 50, 0, 17, 192, 72, 76, 17, 57, 216, 82, 157, 95, 86, 152, 6, 69, 246, 36,
            130, 33, 231, 72, 62, 180, 179, 180, 172, 195, 145, 54,
        ],
        [
            218, 244, 56, 14, 177, 227, 196, 223, 249, 8, 61, 252, 143, 208, 236, 117, 16, 219, 26,
            78, 111, 76, 253, 21, 89, 97, 133, 31, 219, 214, 130, 53,
        ],
        [
            224, 191, 66, 62, 230, 117, 56, 49, 22, 179, 57, 155, 161, 72, 246, 216, 122, 170, 226,
            178, 168, 99, 213, 40, 172, 104, 214, 235, 100, 7, 215, 26,
        ],
        [
            97, 165, 191, 166, 199, 215, 159, 154, 9, 200, 98, 51, 193, 16, 119, 135, 8, 248, 235,
            220, 91, 233, 56, 169, 209, 80, 67, 28, 187, 214, 7, 1,
        ],
        [
            164, 25, 209, 61, 23, 143, 33, 36, 199, 146, 9, 119, 86, 130, 161, 162, 230, 143, 172,
            107, 133, 19, 27, 111, 25, 73, 186, 44, 49, 170, 143, 4,
        ],
        [
            127, 165, 126, 108, 249, 181, 86, 104, 103, 188, 164, 130, 127, 57, 191, 189, 34, 108,
            230, 187, 61, 35, 37, 253, 242, 59, 58, 175, 13, 120, 254, 56,
        ],
        [
            29, 179, 113, 204, 88, 62, 212, 46, 48, 15, 107, 28, 234, 136, 13, 178, 148, 96, 68,
            217, 137, 66, 14, 78, 65, 58, 220, 221, 241, 155, 202, 12,
        ],
        [
            104, 142, 48, 244, 1, 124, 44, 185, 200, 192, 86, 131, 156, 158, 4, 187, 199, 14, 236,
            54, 44, 245, 21, 182, 113, 231, 13, 142, 135, 88, 169, 36,
        ],
    ],
    [
        [
            179, 250, 191, 143, 76, 85, 98, 204, 132, 233, 251, 175, 209, 74, 137, 39, 203, 149,
            63, 252, 246, 49, 40, 30, 148, 93, 65, 201, 161, 49, 187, 27,
        ],
        [
            203, 211, 179, 86, 16, 67, 209, 167, 214, 211, 109, 229, 33, 25, 211, 188, 130, 46,
            186, 119, 41, 173, 245, 62, 7, 55, 108, 180, 101, 115, 125, 40,
        ],
        [
            107, 22, 94, 96, 173, 204, 250, 22, 118, 142, 197, 250, 65, 193, 108, 142, 129, 40, 6,
            13, 218, 17, 207, 193, 75, 1, 195, 27, 7, 231, 199, 17,
        ],
        [
            167, 104, 62, 170, 140, 224, 10, 224, 33, 72, 34, 192, 184, 14, 139, 230, 42, 70, 2,
            10, 88, 218, 169, 123, 65, 198, 119, 16, 205, 62, 64, 8,
        ],
        [
            222, 193, 21, 182, 77, 38, 168, 6, 156, 213, 49, 108, 180, 47, 54, 151, 129, 51, 137,
            226, 103, 85, 108, 252, 94, 11, 25, 7, 56, 87, 205, 46,
        ],
        [
            55, 109, 243, 206, 205, 71, 42, 29, 223, 73, 27, 225, 255, 37, 205, 42, 134, 63, 249,
            2, 91, 152, 44, 111, 202, 23, 173, 135, 200, 207, 39, 48,
        ],
        [
            80, 30, 91, 2, 97, 101, 54, 60, 211, 197, 233, 83, 35, 26, 41, 184, 198, 11, 20, 81,
            130, 203, 153, 178, 149, 187, 189, 223, 113, 175, 177, 27,
        ],
        [
            215, 49, 5, 95, 100, 130, 208, 94, 159, 34, 242, 80, 221, 34, 75, 165, 167, 184, 118,
            14, 8, 33, 83, 229, 192, 200, 102, 128, 209, 220, 60, 27,
        ],
    ],
    [
        [
            19, 126, 36, 206, 173, 162, 196, 15, 59, 89, 70, 64, 231, 230, 176, 226, 171, 99, 59,
            85, 145, 57, 65, 100, 8, 220, 66, 151, 211, 189, 26, 60,
        ],
        [
            115, 58, 171, 209, 149, 12, 201, 125, 168, 145, 30, 99, 185, 185, 212, 83, 153, 170,
            149, 248, 139, 170, 213, 67, 185, 31, 106, 163, 137, 210, 53, 40,
        ],
        [
            34, 169, 41, 154, 168, 16, 208, 29, 254, 75, 80, 21, 20, 65, 127, 18, 19, 245, 106,
            199, 144, 115, 206, 170, 49, 72, 36, 89, 242, 157, 222, 24,
        ],
        [
            10, 164, 200, 254, 5, 101, 144, 74, 139, 202, 71, 210, 151, 43, 63, 228, 174, 239, 5,
            186, 214, 250, 6, 184, 96, 120, 9, 180, 157, 21, 232, 36,
        ],
        [
            75, 221, 109, 210, 3, 45, 141, 250, 189, 180, 105, 1, 74, 134, 4, 161, 180, 240, 1, 39,
            29, 186, 183, 100, 136, 177, 251, 3, 22, 195, 45, 33,
        ],
        [
            38, 154, 140, 7, 121, 70, 69, 229, 20, 92, 70, 49, 117, 230, 240, 101, 67, 40, 78, 8,
            39, 107, 90, 224, 243, 224, 223, 88, 164, 130, 120, 59,
        ],
        [
            160, 144, 216, 94, 228, 217, 167, 0, 168, 204, 30, 88, 222, 158, 9, 43, 233, 216, 234,
            64, 127, 232, 242, 72, 60, 34, 181, 252, 16, 95, 161, 14,
        ],
        [
            195, 165, 176, 79, 109, 182, 190, 127, 55, 224, 143, 82, 207, 108, 164, 169, 220, 92,
            27, 213, 76, 95, 22, 114, 127, 200, 126, 188, 126, 178, 102, 63,
        ],
    ],
    [
        [
            20, 117, 67, 54, 142, 82, 77, 181, 169, 181, 133, 0, 187, 243, 217, 179, 224, 159, 213,
            10, 96, 54, 130, 18, 6, 84, 239, 195, 151, 65, 242, 59,
        ],
        [
            66, 142, 85, 142, 44, 64, 147, 62, 104, 65, 106, 202, 192, 153, 90, 55, 96, 24, 20, 38,
            217, 213, 123, 31, 237, 188, 139, 9, 182, 167, 141, 41,
        ],
        [
            207, 161, 20, 7, 36, 203, 12, 135, 141, 208, 187, 91, 21, 156, 252, 230, 93, 132, 146,
            151, 159, 100, 206, 7, 135, 147, 64, 130, 252, 100, 23, 60,
        ],
        [
            228, 233, 251, 200, 117, 97, 78, 30, 62, 137, 82, 213, 118, 245, 200, 117, 106, 9, 38,
            253, 58, 144, 26, 79, 166, 109, 144, 198, 249, 23, 101, 28,
        ],
        [
            21, 198, 96, 23, 63, 237, 51, 143, 1, 191, 95, 232, 147, 215, 221, 203, 87, 163, 194,
            245, 250, 167, 167, 1, 180, 181, 225, 154, 151, 182, 130, 37,
        ],
        [
            162, 42, 88, 75, 53, 39, 117, 224, 28, 248, 100, 248, 181, 211, 65, 195, 192, 24, 84,
            43, 12, 6, 16, 188, 98, 89, 177, 228, 205, 219, 60, 15,
        ],
        [
            197, 110, 140, 5, 162, 128, 77, 192, 185, 62, 133, 220, 126, 39, 219, 148, 75, 228, 59,
            187, 14, 230, 2, 226, 221, 103, 44, 215, 203, 191, 70, 37,
        ],
        [
            212, 74, 149, 23, 162, 38, 156, 114, 238, 30, 148, 201, 74, 99, 236, 44, 254, 224, 30,
            239, 35, 57, 37, 65, 27, 132, 100, 122, 31, 58, 109, 27,
        ],
    ],
    [
        [
            103, 60, 139, 126, 17, 213, 197, 175, 109, 73, 72, 103, 183, 2, 168, 147, 229, 146,
            163, 240, 237, 251, 74, 204, 173, 237, 2, 81, 217, 38, 198, 26,
        ],
        [
            148, 142, 217, 114, 107, 45, 140, 245, 45, 85, 63, 184, 18, 254, 33, 98, 248, 215, 31,
            13, 147, 228, 90, 147, 57, 118, 190, 54, 109, 166, 232, 40,
        ],
        [
            111, 126, 218, 163, 215, 86, 243, 206, 178, 189, 67, 107, 168, 148, 148, 150, 207, 229,
            162, 190, 99, 21, 204, 46, 137, 115, 14, 38, 124, 45, 7, 31,
        ],
        [
            120, 94, 71, 150, 58, 223, 76, 228, 24, 222, 106, 138, 131, 141, 201, 114, 133, 81,
            208, 146, 222, 53, 190, 188, 243, 101, 130, 69, 176, 0, 144, 19,
        ],
        [
            75, 62, 208, 13, 106, 100, 175, 135, 189, 103, 31, 136, 188, 184, 155, 122, 111, 105,
            240, 140, 7, 105, 177, 110, 64, 11, 23, 49, 104, 166, 225, 29,
        ],
        [
            221, 18, 92, 158, 90, 112, 188, 28, 106, 101, 91, 243, 93, 143, 20, 223, 203, 242, 47,
            39, 63, 111, 126, 205, 146, 55, 207, 185, 179, 128, 14, 17,
        ],
        [
            120, 22, 161, 120, 221, 121, 204, 161, 46, 187, 5, 183, 89, 114, 196, 224, 8, 38, 101,
            86, 52, 196, 8, 105, 71, 97, 245, 117, 232, 101, 1, 35,
        ],
        [
            97, 176, 100, 220, 11, 5, 167, 93, 125, 197, 189, 160, 99, 38, 156, 187, 219, 103, 121,
            132, 60, 6, 17, 51, 131, 182, 75, 89, 53, 115, 247, 33,
        ],
    ],
    [
        [
            207, 159, 115, 155, 49, 140, 88, 106, 204, 148, 127, 110, 102, 239, 71, 50, 116, 241,
            161, 21, 115, 188, 195, 172, 48, 25, 135, 77, 103, 11, 250, 8,
        ],
        [
            53, 226, 49, 81, 131, 154, 93, 210, 99, 6, 241, 86, 41, 242, 116, 183, 23, 71, 226,
            225, 139, 35, 254, 155, 10, 61, 253, 220, 96, 27, 229, 44,
        ],
        [
            86, 73, 95, 110, 144, 237, 118, 227, 89, 133, 209, 162, 107, 24, 108, 6, 152, 8, 73,
            47, 239, 83, 251, 156, 36, 137, 65, 184, 20, 7, 184, 25,
        ],
        [
            73, 67, 2, 194, 3, 175, 147, 62, 215, 31, 179, 37, 248, 247, 228, 219, 86, 172, 138,
            78, 149, 151, 231, 210, 135, 83, 48, 88, 17, 243, 157, 19,
        ],
        [
            119, 233, 103, 230, 0, 17, 17, 12, 79, 202, 7, 142, 66, 32, 12, 216, 107, 223, 194, 82,
            131, 194, 68, 181, 60, 242, 17, 198, 45, 147, 255, 22,
        ],
        [
            13, 69, 173, 185, 175, 48, 175, 137, 106, 63, 5, 27, 54, 95, 201, 207, 229, 228, 54,
            164, 143, 58, 78, 64, 12, 51, 195, 210, 204, 146, 64, 18,
        ],
        [
            234, 84, 133, 114, 188, 84, 22, 238, 15, 80, 147, 93, 113, 197, 232, 9, 61, 181, 84, 9,
            100, 102, 8, 232, 54, 143, 11, 7, 152, 206, 5, 40,
        ],
        [
            57, 77, 249, 157, 194, 147, 228, 76, 88, 15, 59, 18, 116, 226, 216, 213, 113, 122, 13,
            119, 238, 232, 215, 176, 140, 152, 158, 170, 241, 251, 236, 22,
        ],
    ],
    [
        [
            254, 181, 68, 223, 236, 39, 90, 99, 37, 74, 79, 37, 4, 69, 15, 85, 227, 255, 6, 143,
            164, 197, 83, 0, 74, 59, 37, 114, 82, 160, 183, 3,
        ],
        [
            205, 219, 115, 69, 73, 42, 198, 241, 206, 18, 124, 136, 252, 166, 211, 154, 58, 167,
            23, 6, 57, 205, 200, 210, 73, 181, 175, 33, 118, 232, 152, 61,
        ],
        [
            131, 71, 36, 249, 192, 54, 50, 65, 9, 141, 95, 165, 255, 154, 54, 217, 250, 127, 68,
            177, 131, 207, 66, 187, 46, 48, 135, 44, 198, 173, 172, 20,
        ],
        [
            22, 128, 64, 66, 38, 59, 150, 29, 112, 190, 217, 120, 239, 211, 204, 253, 122, 62, 11,
            237, 33, 29, 119, 205, 235, 253, 180, 87, 199, 230, 183, 60,
        ],
        [
            122, 104, 29, 128, 154, 86, 116, 112, 241, 40, 109, 16, 144, 82, 222, 122, 172, 201,
            75, 214, 205, 168, 32, 91, 203, 28, 46, 114, 218, 82, 18, 48,
        ],
        [
            221, 10, 12, 130, 141, 9, 26, 13, 195, 232, 15, 60, 197, 192, 117, 229, 184, 110, 170,
            92, 191, 138, 41, 66, 27, 60, 210, 156, 199, 115, 164, 56,
        ],
        [
            121, 35, 54, 98, 55, 143, 52, 179, 2, 61, 161, 109, 169, 148, 239, 238, 9, 164, 231,
            216, 160, 162, 173, 134, 231, 226, 130, 102, 239, 169, 44, 16,
        ],
        [
            198, 57, 213, 12, 44, 150, 213, 163, 167, 39, 246, 18, 39, 137, 192, 35, 136, 231, 107,
            189, 61, 27, 163, 87, 214, 83, 119, 127, 246, 239, 39, 52,
        ],
    ],
    [
        [
            185, 73, 33, 65, 211, 14, 33, 86, 138, 6, 164, 187, 255, 69, 78, 253, 83, 184, 44, 255,
            21, 48, 149, 155, 121, 24, 91, 115, 54, 24, 84, 62,
        ],
        [
            105, 91, 120, 139, 39, 32, 121, 140, 89, 109, 59, 171, 95, 234, 21, 41, 12, 240, 246,
            153, 66, 6, 226, 31, 80, 147, 53, 111, 250, 209, 72, 46,
        ],
        [
            218, 148, 82, 157, 209, 222, 86, 248, 254, 4, 160, 225, 150, 222, 127, 75, 233, 239,
            136, 156, 200, 19, 170, 55, 117, 74, 48, 67, 43, 33, 174, 19,
        ],
        [
            182, 8, 141, 160, 94, 156, 111, 237, 26, 83, 42, 26, 164, 150, 120, 210, 90, 32, 233,
            151, 101, 230, 235, 243, 79, 180, 152, 117, 104, 105, 2, 2,
        ],
        [
            226, 88, 147, 59, 28, 120, 10, 106, 17, 3, 181, 165, 169, 42, 157, 103, 86, 16, 57, 32,
            242, 38, 118, 199, 209, 132, 108, 29, 13, 36, 66, 45,
        ],
        [
            59, 134, 13, 128, 49, 120, 104, 220, 205, 203, 247, 211, 109, 54, 2, 104, 55, 55, 174,
            137, 145, 197, 13, 223, 14, 212, 158, 87, 174, 226, 202, 2,
        ],
        [
            80, 17, 130, 225, 209, 179, 88, 96, 37, 152, 203, 70, 100, 161, 154, 37, 135, 123, 176,
            148, 25, 121, 9, 199, 73, 179, 219, 18, 85, 186, 178, 39,
        ],
        [
            20, 29, 148, 97, 111, 108, 49, 113, 63, 110, 194, 53, 35, 251, 5, 121, 46, 86, 171,
            199, 5, 203, 87, 40, 199, 71, 83, 96, 64, 218, 212, 60,
        ],
    ],
    [
        [
            6, 109, 206, 92, 9, 160, 131, 155, 216, 126, 46, 253, 160, 152, 216, 161, 49, 172, 17,
            0, 184, 177, 139, 130, 101, 15, 177, 181, 10, 38, 241, 31,
        ],
        [
            126, 33, 243, 169, 172, 127, 107, 116, 0, 75, 122, 15, 4, 153, 153, 58, 56, 252, 29,
            237, 214, 219, 218, 7, 246, 151, 151, 130, 54, 99, 251, 49,
        ],
        [
            111, 55, 110, 109, 255, 254, 236, 86, 100, 82, 150, 127, 142, 198, 80, 148, 106, 154,
            232, 11, 60, 246, 170, 188, 81, 74, 56, 104, 71, 90, 205, 20,
        ],
        [
            173, 19, 142, 233, 210, 176, 247, 226, 195, 87, 177, 70, 135, 96, 23, 240, 206, 171,
            84, 205, 110, 227, 148, 138, 6, 249, 91, 77, 147, 33, 177, 45,
        ],
        [
            128, 134, 78, 206, 171, 42, 66, 88, 247, 175, 4, 234, 240, 38, 148, 150, 88, 191, 245,
            50, 103, 223, 60, 37, 82, 65, 44, 217, 181, 245, 124, 5,
        ],
        [
            115, 230, 148, 128, 84, 122, 60, 46, 27, 232, 127, 51, 8, 133, 215, 66, 89, 222, 154,
            252, 9, 233, 194, 58, 174, 236, 197, 21, 44, 241, 241, 38,
        ],
        [
            67, 214, 36, 107, 4, 73, 244, 168, 44, 72, 167, 224, 3, 55, 62, 70, 237, 129, 75, 17,
            125, 236, 204, 239, 239, 10, 117, 243, 61, 71, 202, 56,
        ],
        [
            202, 195, 57, 169, 77, 192, 73, 210, 151, 66, 40, 168, 248, 195, 215, 133, 244, 89,
            224, 48, 36, 108, 160, 23, 179, 91, 99, 229, 93, 44, 20, 45,
        ],
    ],
    [
        [
            163, 121, 71, 96, 36, 196, 194, 47, 226, 26, 109, 145, 34, 27, 140, 139, 12, 15, 202,
            197, 210, 27, 46, 82, 159, 203, 189, 65, 158, 83, 130, 46,
        ],
        [
            253, 14, 139, 66, 95, 63, 237, 107, 113, 207, 46, 231, 223, 172, 187, 60, 60, 219, 175,
            123, 1, 152, 96, 201, 148, 199, 2, 178, 151, 212, 109, 35,
        ],
        [
            88, 58, 174, 25, 43, 75, 80, 228, 87, 94, 26, 207, 255, 176, 227, 78, 203, 97, 202,
            141, 70, 72, 158, 251, 237, 140, 156, 1, 12, 219, 235, 44,
        ],
        [
            159, 231, 8, 161, 182, 250, 9, 245, 215, 155, 99, 201, 83, 210, 129, 2, 164, 67, 100,
            189, 238, 22, 210, 69, 176, 190, 243, 205, 207, 56, 14, 41,
        ],
        [
            55, 189, 29, 92, 22, 22, 208, 205, 64, 236, 5, 66, 118, 71, 45, 43, 21, 173, 195, 90,
            197, 129, 14, 33, 18, 29, 222, 45, 46, 196, 207, 7,
        ],
        [
            82, 104, 247, 77, 232, 36, 116, 179, 196, 226, 64, 108, 3, 127, 74, 9, 64, 70, 35, 252,
            216, 18, 8, 100, 90, 47, 119, 210, 178, 45, 195, 42,
        ],
        [
            186, 14, 254, 60, 145, 38, 37, 216, 115, 22, 239, 124, 112, 128, 82, 62, 32, 124, 198,
            228, 53, 1, 153, 169, 177, 68, 208, 228, 211, 95, 7, 9,
        ],
        [
            43, 127, 253, 68, 34, 27, 115, 203, 227, 127, 190, 105, 183, 240, 215, 39, 180, 235,
            52, 211, 22, 217, 143, 62, 206, 95, 95, 21, 179, 173, 187, 40,
        ],
    ],
    [
        [
            131, 114, 135, 36, 105, 222, 23, 214, 185, 164, 251, 205, 92, 193, 72, 141, 238, 95,
            30, 7, 205, 137, 109, 67, 6, 211, 254, 238, 205, 93, 248, 8,
        ],
        [
            84, 246, 47, 201, 191, 63, 93, 82, 192, 92, 116, 175, 63, 184, 93, 178, 218, 82, 132,
            246, 186, 116, 87, 8, 183, 49, 227, 163, 38, 253, 108, 26,
        ],
        [
            6, 251, 41, 69, 223, 49, 29, 206, 58, 21, 142, 51, 233, 183, 17, 7, 217, 0, 81, 233,
            202, 48, 47, 101, 165, 192, 221, 106, 85, 1, 119, 47,
        ],
        [
            155, 201, 217, 190, 29, 250, 17, 180, 157, 67, 107, 96, 78, 170, 220, 10, 105, 107, 45,
            150, 231, 202, 86, 245, 215, 155, 214, 166, 248, 64, 145, 30,
        ],
        [
            83, 7, 177, 230, 141, 158, 93, 202, 158, 43, 163, 0, 254, 82, 139, 239, 126, 115, 210,
            50, 149, 165, 220, 201, 209, 48, 24, 226, 159, 173, 118, 48,
        ],
        [
            82, 210, 142, 117, 9, 95, 90, 109, 253, 87, 151, 157, 67, 189, 66, 204, 182, 59, 48,
            38, 145, 140, 176, 206, 55, 97, 116, 204, 28, 205, 89, 24,
        ],
        [
            220, 202, 253, 28, 242, 130, 86, 193, 187, 216, 73, 198, 231, 44, 233, 93, 247, 37,
            224, 33, 48, 43, 155, 204, 152, 21, 118, 161, 73, 135, 91, 39,
        ],
        [
            187, 254, 171, 174, 249, 248, 213, 106, 175, 179, 210, 131, 184, 126, 37, 35, 202, 9,
            240, 29, 24, 239, 213, 132, 138, 116, 32, 156, 7, 113, 226, 25,
        ],
    ],
    [
        [
            156, 164, 8, 226, 97, 207, 91, 167, 225, 181, 182, 160, 132, 78, 236, 30, 215, 20, 95,
            9, 227, 77, 117, 68, 12, 178, 75, 43, 119, 151, 125, 3,
        ],
        [
            88, 89, 242, 116, 112, 190, 26, 203, 76, 128, 41, 80, 241, 253, 69, 188, 11, 123, 28,
            138, 47, 217, 131, 246, 104, 174, 71, 39, 91, 137, 238, 27,
        ],
        [
            6, 243, 135, 48, 85, 60, 99, 53, 2, 117, 68, 224, 154, 116, 145, 216, 177, 236, 240,
            177, 21, 13, 251, 225, 143, 176, 93, 143, 112, 171, 130, 11,
        ],
        [
            165, 207, 217, 234, 178, 110, 151, 80, 150, 134, 43, 61, 131, 99, 184, 162, 158, 113,
            155, 239, 130, 171, 157, 168, 231, 45, 113, 248, 80, 213, 51, 2,
        ],
        [
            128, 168, 127, 174, 75, 22, 175, 44, 133, 227, 50, 26, 252, 142, 227, 90, 32, 56, 94,
            252, 53, 254, 0, 174, 65, 18, 199, 215, 241, 120, 110, 55,
        ],
        [
            211, 242, 54, 2, 188, 143, 153, 96, 178, 248, 47, 63, 90, 109, 94, 60, 8, 61, 25, 205,
            123, 32, 217, 91, 125, 237, 195, 199, 225, 228, 97, 59,
        ],
        [
            208, 167, 59, 171, 42, 184, 236, 23, 17, 52, 52, 174, 35, 212, 29, 143, 18, 174, 99,
            48, 179, 44, 237, 46, 119, 4, 170, 48, 30, 79, 182, 10,
        ],
        [
            106, 121, 190, 22, 158, 183, 33, 248, 175, 196, 212, 4, 160, 136, 121, 1, 110, 225,
            143, 145, 248, 216, 225, 198, 45, 119, 247, 143, 160, 247, 172, 61,
        ],
    ],
    [
        [
            170, 197, 210, 231, 188, 33, 170, 201, 233, 126, 60, 21, 94, 154, 254, 1, 196, 112, 19,
            33, 164, 39, 47, 136, 105, 87, 42, 209, 126, 179, 4, 53,
        ],
        [
            188, 4, 106, 142, 39, 183, 58, 78, 91, 77, 33, 182, 34, 195, 233, 151, 102, 209, 51,
            195, 23, 95, 51, 237, 30, 112, 16, 207, 245, 125, 187, 59,
        ],
        [
            15, 242, 145, 233, 209, 153, 20, 138, 166, 217, 11, 161, 243, 43, 241, 80, 125, 65, 1,
            235, 179, 76, 8, 72, 23, 184, 85, 248, 167, 133, 69, 2,
        ],
        [
            225, 27, 224, 155, 142, 0, 95, 230, 169, 31, 72, 118, 204, 184, 84, 219, 150, 161, 137,
            193, 136, 180, 112, 183, 197, 184, 226, 97, 174, 180, 24, 11,
        ],
        [
            199, 134, 139, 145, 225, 180, 173, 107, 79, 68, 52, 186, 212, 235, 203, 133, 237, 28,
            129, 57, 178, 111, 10, 128, 182, 15, 240, 22, 78, 20, 144, 37,
        ],
        [
            124, 87, 4, 136, 241, 14, 185, 35, 104, 27, 231, 201, 32, 180, 69, 222, 208, 122, 144,
            183, 68, 218, 190, 211, 97, 52, 193, 1, 239, 155, 129, 35,
        ],
        [
            196, 160, 5, 223, 59, 150, 79, 242, 92, 80, 44, 206, 218, 37, 78, 177, 242, 104, 141,
            77, 227, 103, 140, 229, 134, 180, 34, 11, 45, 120, 81, 53,
        ],
        [
            99, 157, 16, 77, 119, 202, 154, 196, 15, 121, 133, 20, 68, 176, 29, 23, 176, 41, 217,
            190, 25, 99, 134, 214, 160, 62, 79, 119, 238, 131, 33, 36,
        ],
    ],
    [
        [
            82, 63, 89, 161, 22, 25, 171, 246, 76, 224, 209, 2, 192, 99, 241, 215, 61, 51, 46, 111,
            168, 154, 99, 217, 0, 4, 43, 53, 18, 209, 229, 54,
        ],
        [
            167, 96, 77, 229, 175, 88, 227, 207, 122, 115, 183, 68, 181, 58, 37, 192, 5, 113, 227,
            193, 160, 30, 159, 155, 194, 116, 35, 238, 138, 4, 110, 1,
        ],
        [
            208, 142, 104, 151, 195, 118, 201, 104, 233, 25, 4, 134, 154, 101, 184, 186, 176, 240,
            88, 47, 60, 63, 85, 49, 28, 180, 85, 205, 229, 172, 224, 23,
        ],
        [
            225, 179, 202, 188, 22, 119, 248, 74, 28, 29, 26, 29, 213, 85, 201, 119, 250, 208, 90,
            186, 188, 163, 67, 225, 7, 105, 115, 238, 100, 73, 15, 43,
        ],
        [
            94, 130, 151, 41, 5, 147, 117, 3, 249, 79, 58, 213, 193, 248, 185, 183, 10, 21, 35, 63,
            25, 84, 177, 253, 165, 102, 91, 81, 138, 219, 39, 54,
        ],
        [
            176, 233, 181, 59, 196, 247, 219, 24, 149, 28, 113, 113, 133, 66, 180, 213, 204, 65,
            88, 21, 171, 131, 99, 218, 15, 224, 29, 102, 207, 240, 157, 40,
        ],
        [
            148, 66, 102, 220, 47, 53, 236, 254, 85, 226, 162, 110, 94, 77, 10, 149, 173, 16, 70,
            31, 31, 87, 110, 6, 57, 165, 232, 79, 119, 39, 176, 12,
        ],
        [
            23, 109, 174, 136, 38, 90, 212, 253, 239, 204, 23, 11, 76, 147, 151, 42, 129, 24, 229,
            90, 202, 179, 169, 199, 57, 95, 105, 207, 109, 231, 64, 23,
        ],
    ],
    [
        [
            210, 218, 63, 9, 74, 89, 145, 23, 196, 76, 249, 167, 105, 125, 253, 20, 49, 203, 12,
            184, 203, 120, 138, 93, 91, 46, 186, 251, 218, 25, 70, 4,
        ],
        [
            139, 120, 71, 74, 206, 5, 119, 147, 208, 109, 12, 36, 182, 63, 104, 121, 35, 248, 122,
            36, 26, 179, 98, 242, 96, 103, 157, 145, 215, 160, 238, 37,
        ],
        [
            119, 7, 93, 55, 33, 255, 127, 106, 216, 22, 144, 55, 139, 80, 136, 132, 240, 202, 42,
            225, 208, 127, 220, 53, 244, 237, 71, 189, 88, 189, 163, 5,
        ],
        [
            170, 190, 180, 72, 134, 101, 250, 194, 105, 240, 171, 146, 156, 166, 210, 64, 172, 238,
            90, 149, 94, 59, 190, 252, 76, 134, 126, 243, 154, 205, 102, 13,
        ],
        [
            61, 238, 119, 250, 17, 114, 41, 23, 62, 28, 235, 146, 48, 100, 181, 149, 254, 237, 241,
            138, 188, 225, 84, 16, 184, 30, 56, 107, 147, 250, 66, 2,
        ],
        [
            28, 109, 249, 181, 203, 172, 134, 246, 128, 130, 89, 122, 40, 47, 34, 151, 64, 90, 103,
            136, 81, 219, 41, 175, 233, 47, 111, 72, 168, 30, 5, 56,
        ],
        [
            104, 186, 155, 55, 148, 38, 106, 255, 151, 132, 138, 199, 113, 77, 212, 162, 185, 6,
            79, 180, 229, 49, 60, 105, 198, 127, 91, 244, 214, 186, 62, 61,
        ],
        [
            29, 254, 87, 7, 165, 114, 18, 76, 54, 39, 67, 130, 22, 232, 156, 152, 108, 111, 76,
            242, 206, 188, 3, 128, 125, 61, 185, 187, 198, 48, 252, 44,
        ],
    ],
    [
        [
            187, 50, 8, 176, 60, 67, 135, 89, 94, 138, 218, 198, 22, 101, 177, 175, 102, 240, 126,
            241, 82, 144, 251, 102, 72, 83, 107, 48, 146, 17, 213, 28,
        ],
        [
            66, 28, 29, 99, 204, 226, 10, 124, 168, 193, 84, 224, 148, 6, 106, 116, 159, 95, 220,
            57, 114, 182, 252, 121, 106, 183, 91, 112, 135, 212, 111, 32,
        ],
        [
            74, 224, 165, 3, 122, 142, 237, 166, 242, 106, 187, 40, 75, 60, 48, 81, 110, 173, 246,
            152, 66, 206, 150, 22, 173, 227, 82, 47, 214, 31, 65, 25,
        ],
        [
            241, 175, 29, 66, 85, 85, 198, 49, 73, 114, 59, 93, 17, 168, 52, 23, 229, 172, 95, 205,
            71, 36, 103, 168, 242, 113, 13, 36, 233, 30, 70, 12,
        ],
        [
            237, 63, 61, 13, 103, 181, 237, 70, 72, 231, 126, 24, 8, 136, 64, 33, 107, 250, 198,
            207, 196, 179, 175, 29, 224, 86, 78, 2, 162, 171, 147, 0,
        ],
        [
            11, 93, 3, 235, 245, 18, 55, 148, 155, 241, 206, 53, 156, 60, 107, 8, 149, 42, 57, 117,
            150, 13, 151, 213, 8, 223, 186, 100, 167, 163, 208, 4,
        ],
        [
            101, 234, 84, 111, 45, 42, 194, 165, 16, 135, 160, 248, 114, 253, 44, 141, 91, 202, 39,
            81, 92, 168, 4, 92, 33, 196, 29, 132, 94, 49, 171, 36,
        ],
        [
            170, 78, 106, 212, 212, 78, 86, 18, 151, 82, 184, 101, 228, 37, 105, 224, 43, 102, 141,
            27, 19, 208, 35, 122, 31, 89, 86, 64, 192, 221, 17, 17,
        ],
    ],
    [
        [
            151, 50, 9, 177, 173, 66, 105, 137, 98, 60, 243, 94, 35, 26, 166, 35, 163, 143, 255,
            135, 238, 117, 197, 96, 41, 128, 136, 136, 152, 185, 32, 22,
        ],
        [
            106, 14, 42, 81, 127, 116, 49, 48, 154, 161, 77, 30, 178, 74, 202, 5, 124, 177, 27,
            169, 164, 70, 166, 152, 138, 194, 203, 198, 134, 203, 255, 40,
        ],
        [
            47, 96, 56, 43, 93, 62, 87, 53, 208, 6, 248, 168, 93, 211, 251, 240, 46, 138, 239, 76,
            20, 142, 20, 116, 0, 124, 15, 225, 30, 217, 116, 37,
        ],
        [
            231, 205, 171, 128, 77, 59, 170, 67, 8, 186, 39, 75, 19, 129, 239, 181, 145, 141, 177,
            29, 7, 232, 153, 69, 218, 26, 205, 164, 206, 207, 0, 54,
        ],
        [
            229, 107, 81, 81, 205, 122, 120, 80, 1, 96, 13, 166, 168, 148, 0, 240, 16, 207, 156,
            84, 125, 0, 207, 196, 242, 124, 208, 133, 52, 32, 182, 9,
        ],
        [
            179, 184, 212, 223, 250, 116, 72, 192, 9, 113, 91, 215, 203, 32, 185, 213, 167, 149,
            210, 44, 74, 255, 114, 55, 2, 218, 189, 211, 188, 57, 154, 59,
        ],
        [
            162, 165, 77, 233, 58, 220, 93, 70, 103, 253, 243, 4, 206, 70, 34, 10, 214, 54, 189,
            252, 222, 176, 215, 185, 142, 65, 240, 131, 61, 234, 236, 13,
        ],
        [
            48, 73, 161, 183, 218, 189, 224, 124, 241, 181, 230, 36, 73, 203, 90, 15, 116, 0, 69,
            237, 156, 181, 231, 180, 84, 197, 234, 181, 190, 199, 255, 52,
        ],
    ],
    [
        [
            220, 111, 247, 64, 46, 37, 189, 99, 253, 91, 162, 238, 241, 140, 108, 174, 90, 146,
            190, 18, 224, 248, 2, 146, 96, 236, 243, 212, 16, 103, 66, 30,
        ],
        [
            31, 37, 121, 78, 67, 203, 24, 7, 252, 180, 126, 16, 235, 113, 134, 22, 212, 79, 77,
            205, 114, 166, 90, 126, 71, 121, 244, 35, 135, 130, 158, 2,
        ],
        [
            122, 50, 153, 247, 80, 50, 170, 82, 111, 229, 167, 139, 213, 253, 253, 149, 1, 132,
            230, 42, 168, 20, 94, 119, 109, 143, 55, 222, 234, 138, 53, 52,
        ],
        [
            22, 219, 32, 98, 222, 149, 154, 11, 103, 155, 91, 234, 214, 98, 113, 121, 194, 38, 112,
            245, 228, 153, 212, 138, 113, 239, 46, 112, 14, 169, 217, 57,
        ],
        [
            218, 47, 129, 55, 195, 205, 29, 175, 252, 108, 76, 237, 220, 134, 47, 151, 189, 218,
            53, 45, 238, 145, 156, 13, 203, 143, 8, 75, 11, 81, 68, 27,
        ],
        [
            165, 15, 17, 124, 219, 84, 3, 50, 247, 108, 34, 94, 139, 246, 71, 153, 118, 224, 119,
            186, 40, 94, 45, 102, 25, 209, 86, 62, 6, 174, 151, 53,
        ],
        [
            128, 13, 54, 223, 254, 21, 137, 124, 31, 192, 179, 70, 118, 72, 142, 187, 167, 0, 178,
            120, 165, 60, 204, 21, 18, 202, 138, 190, 17, 246, 100, 51,
        ],
        [
            148, 186, 124, 117, 46, 54, 68, 248, 71, 170, 136, 117, 249, 134, 99, 43, 192, 98, 121,
            63, 207, 156, 75, 54, 185, 158, 212, 22, 18, 211, 51, 52,
        ],
    ],
    [
        [
            24, 52, 228, 114, 155, 108, 214, 40, 54, 236, 37, 7, 134, 141, 237, 159, 77, 185, 13,
            39, 195, 44, 58, 34, 229, 109, 3, 4, 68, 221, 190, 1,
        ],
        [
            246, 210, 179, 250, 192, 219, 151, 191, 86, 214, 163, 157, 76, 7, 169, 170, 142, 35,
            76, 33, 26, 34, 184, 11, 1, 201, 80, 190, 175, 176, 222, 22,
        ],
        [
            45, 230, 47, 218, 24, 154, 184, 70, 125, 235, 25, 236, 196, 31, 216, 134, 216, 186,
            232, 169, 254, 217, 128, 228, 248, 58, 65, 17, 151, 230, 252, 39,
        ],
        [
            49, 249, 191, 147, 138, 148, 49, 217, 180, 143, 197, 86, 126, 226, 166, 187, 86, 230,
            21, 120, 10, 5, 224, 33, 168, 69, 24, 177, 196, 212, 19, 56,
        ],
        [
            78, 3, 109, 136, 87, 121, 66, 254, 234, 100, 221, 179, 88, 91, 164, 100, 110, 130, 124,
            91, 196, 38, 192, 144, 172, 166, 18, 172, 247, 58, 40, 14,
        ],
        [
            214, 225, 119, 224, 102, 139, 164, 249, 12, 11, 182, 68, 37, 36, 143, 165, 110, 236,
            160, 96, 231, 183, 237, 46, 33, 87, 251, 228, 234, 141, 224, 8,
        ],
        [
            41, 23, 61, 155, 61, 57, 2, 172, 82, 226, 110, 224, 155, 186, 1, 116, 244, 90, 184,
            218, 91, 241, 142, 128, 196, 137, 32, 56, 165, 182, 15, 35,
        ],
        [
            134, 195, 91, 252, 76, 154, 223, 242, 155, 67, 75, 113, 30, 185, 151, 122, 108, 161,
            234, 74, 245, 112, 142, 141, 211, 94, 34, 237, 227, 172, 133, 1,
        ],
    ],
    [
        [
            167, 255, 57, 210, 245, 166, 106, 113, 15, 165, 122, 64, 80, 166, 212, 184, 73, 180,
            113, 173, 253, 39, 112, 70, 8, 20, 6, 65, 24, 62, 64, 5,
        ],
        [
            203, 107, 134, 205, 3, 48, 37, 121, 106, 5, 121, 146, 237, 41, 10, 95, 169, 198, 158,
            124, 91, 231, 131, 169, 13, 208, 34, 9, 160, 185, 35, 62,
        ],
        [
            159, 157, 209, 73, 242, 210, 214, 163, 71, 138, 37, 146, 109, 231, 120, 218, 9, 240,
            166, 57, 3, 33, 3, 133, 32, 211, 190, 239, 180, 72, 251, 14,
        ],
        [
            235, 185, 248, 188, 219, 240, 28, 178, 142, 217, 194, 95, 153, 114, 200, 233, 230, 77,
            203, 129, 201, 87, 115, 29, 42, 80, 160, 207, 197, 103, 161, 18,
        ],
        [
            64, 41, 171, 141, 241, 81, 18, 17, 80, 24, 42, 140, 182, 191, 240, 165, 69, 60, 26, 10,
            218, 36, 147, 62, 195, 1, 37, 201, 79, 111, 117, 48,
        ],
        [
            99, 149, 18, 72, 7, 45, 206, 70, 179, 31, 195, 173, 136, 27, 178, 204, 81, 105, 144,
            191, 206, 91, 40, 189, 8, 32, 92, 108, 138, 209, 85, 23,
        ],
        [
            122, 133, 113, 198, 75, 111, 114, 28, 82, 109, 46, 198, 163, 186, 185, 113, 163, 16,
            25, 119, 248, 50, 124, 124, 202, 61, 194, 124, 64, 125, 8, 3,
        ],
        [
            231, 74, 52, 247, 27, 78, 86, 39, 140, 9, 146, 193, 50, 231, 242, 134, 74, 134, 57,
            174, 105, 148, 90, 9, 117, 29, 5, 224, 77, 120, 94, 43,
        ],
    ],
    [
        [
            66, 125, 134, 52, 113, 121, 1, 157, 97, 185, 99, 106, 44, 43, 60, 127, 106, 209, 19,
            201, 122, 218, 244, 45, 241, 86, 143, 244, 245, 208, 162, 7,
        ],
        [
            18, 80, 241, 94, 225, 206, 164, 202, 189, 9, 154, 17, 214, 79, 229, 176, 191, 112, 95,
            5, 52, 43, 102, 83, 249, 151, 222, 109, 132, 233, 251, 3,
        ],
        [
            253, 91, 9, 76, 243, 26, 104, 70, 78, 24, 99, 82, 248, 129, 141, 73, 122, 142, 83, 21,
            218, 240, 118, 200, 240, 184, 227, 38, 137, 46, 190, 3,
        ],
        [
            155, 143, 123, 196, 198, 125, 209, 238, 161, 6, 7, 95, 47, 74, 163, 223, 247, 184, 193,
            90, 102, 129, 37, 25, 201, 201, 101, 71, 81, 31, 49, 54,
        ],
        [
            229, 127, 64, 113, 252, 102, 170, 69, 29, 131, 215, 235, 16, 35, 215, 100, 234, 250,
            171, 224, 254, 50, 218, 177, 59, 199, 79, 8, 85, 162, 106, 42,
        ],
        [
            48, 238, 75, 148, 191, 96, 178, 107, 185, 175, 86, 29, 33, 61, 66, 179, 234, 82, 134,
            19, 254, 213, 129, 75, 188, 155, 24, 164, 137, 200, 190, 41,
        ],
        [
            220, 31, 55, 254, 233, 65, 55, 210, 170, 104, 85, 6, 115, 71, 143, 106, 177, 34, 156,
            96, 46, 5, 122, 164, 33, 122, 36, 249, 146, 49, 169, 44,
        ],
        [
            222, 65, 234, 61, 23, 51, 29, 154, 52, 159, 192, 26, 238, 127, 189, 106, 219, 161, 17,
            81, 68, 54, 147, 136, 210, 156, 120, 36, 171, 255, 57, 58,
        ],
    ],
    [
        [
            71, 251, 242, 20, 210, 72, 251, 220, 14, 8, 19, 169, 220, 25, 39, 174, 75, 44, 133, 22,
            105, 236, 21, 103, 212, 100, 5, 34, 121, 79, 162, 0,
        ],
        [
            194, 41, 180, 100, 238, 236, 100, 134, 89, 82, 152, 106, 230, 80, 7, 146, 66, 87, 9,
            42, 24, 87, 242, 96, 92, 63, 56, 169, 156, 190, 229, 34,
        ],
        [
            141, 254, 94, 59, 138, 160, 119, 154, 211, 60, 56, 145, 249, 153, 182, 182, 220, 6,
            168, 138, 212, 237, 122, 17, 116, 242, 63, 233, 22, 145, 242, 30,
        ],
        [
            153, 109, 113, 106, 252, 209, 225, 86, 244, 176, 84, 140, 44, 71, 175, 77, 108, 18, 39,
            175, 183, 234, 142, 163, 210, 63, 38, 103, 232, 88, 223, 17,
        ],
        [
            109, 169, 28, 78, 240, 241, 115, 94, 37, 155, 187, 62, 78, 248, 41, 16, 199, 207, 47,
            59, 100, 234, 147, 113, 185, 30, 204, 174, 216, 101, 217, 26,
        ],
        [
            199, 76, 237, 233, 188, 41, 236, 124, 20, 78, 61, 129, 251, 10, 106, 218, 166, 189, 16,
            189, 63, 30, 242, 111, 51, 164, 134, 147, 135, 167, 20, 32,
        ],
        [
            246, 126, 31, 131, 87, 207, 67, 228, 92, 46, 120, 73, 48, 7, 53, 2, 228, 91, 167, 249,
            224, 145, 38, 140, 49, 110, 9, 198, 28, 166, 199, 23,
        ],
        [
            231, 83, 169, 70, 52, 62, 177, 207, 175, 89, 75, 217, 161, 59, 56, 2, 169, 1, 9, 205,
            52, 49, 201, 25, 166, 146, 234, 172, 244, 231, 106, 47,
        ],
    ],
    [
        [
            194, 228, 232, 201, 33, 16, 85, 164, 105, 160, 141, 18, 249, 59, 102, 48, 193, 116,
            209, 147, 10, 211, 226, 31, 157, 35, 94, 220, 99, 198, 20, 42,
        ],
        [
            173, 35, 189, 24, 85, 198, 154, 168, 25, 220, 226, 85, 221, 15, 58, 3, 224, 4, 58, 168,
            58, 29, 56, 245, 186, 223, 226, 181, 18, 185, 172, 40,
        ],
        [
            27, 18, 59, 17, 10, 72, 50, 106, 165, 134, 74, 100, 153, 117, 121, 253, 16, 86, 246,
            110, 187, 230, 25, 216, 83, 224, 186, 141, 45, 119, 33, 49,
        ],
        [
            244, 45, 103, 199, 97, 180, 35, 18, 240, 147, 30, 233, 41, 208, 61, 151, 65, 157, 96,
            171, 100, 18, 176, 152, 225, 175, 193, 244, 215, 35, 160, 5,
        ],
        [
            28, 222, 44, 27, 46, 254, 161, 252, 108, 206, 27, 127, 81, 6, 137, 238, 194, 214, 249,
            85, 148, 234, 179, 183, 90, 25, 164, 214, 151, 128, 32, 50,
        ],
        [
            125, 65, 228, 176, 203, 94, 50, 79, 49, 124, 4, 14, 170, 253, 17, 191, 75, 224, 55, 44,
            99, 162, 131, 127, 176, 147, 248, 78, 201, 196, 241, 48,
        ],
        [
            117, 21, 105, 66, 55, 43, 28, 99, 109, 184, 3, 147, 39, 208, 145, 27, 14, 225, 196,
            196, 154, 197, 190, 14, 72, 118, 238, 54, 18, 187, 54, 26,
        ],
        [
            70, 174, 45, 224, 63, 252, 216, 207, 239, 185, 167, 156, 199, 168, 127, 109, 175, 46,
            99, 100, 169, 75, 223, 35, 161, 102, 242, 103, 16, 8, 209, 55,
        ],
    ],
    [
        [
            13, 194, 138, 145, 78, 219, 22, 182, 229, 0, 1, 81, 185, 166, 160, 114, 3, 35, 63, 253,
            99, 163, 28, 47, 140, 78, 62, 65, 234, 25, 70, 37,
        ],
        [
            61, 96, 17, 118, 88, 237, 59, 158, 157, 55, 155, 76, 10, 185, 215, 169, 151, 212, 195,
            26, 97, 8, 212, 102, 242, 134, 252, 227, 178, 5, 199, 22,
        ],
        [
            34, 135, 203, 237, 55, 68, 123, 190, 69, 155, 72, 68, 155, 82, 131, 168, 8, 208, 132,
            197, 218, 130, 132, 104, 97, 217, 13, 156, 76, 112, 181, 16,
        ],
        [
            95, 192, 150, 61, 45, 125, 142, 207, 41, 12, 219, 19, 191, 159, 133, 179, 135, 190,
            152, 115, 226, 94, 245, 57, 244, 14, 94, 122, 221, 36, 84, 14,
        ],
        [
            60, 90, 132, 102, 233, 42, 52, 168, 255, 226, 18, 254, 36, 107, 88, 73, 146, 120, 103,
            244, 75, 107, 77, 25, 143, 238, 53, 95, 194, 155, 31, 40,
        ],
        [
            138, 52, 172, 249, 115, 57, 8, 171, 31, 81, 27, 131, 252, 161, 40, 171, 234, 110, 66,
            119, 79, 189, 7, 252, 119, 55, 152, 241, 39, 80, 8, 4,
        ],
        [
            14, 102, 234, 194, 179, 112, 30, 38, 179, 183, 161, 70, 57, 1, 209, 189, 106, 139, 242,
            192, 200, 245, 190, 143, 182, 219, 93, 128, 9, 37, 227, 63,
        ],
        [
            122, 219, 120, 166, 254, 238, 26, 106, 128, 225, 64, 176, 209, 13, 159, 175, 107, 206,
            89, 142, 159, 70, 79, 142, 28, 220, 139, 218, 34, 215, 125, 63,
        ],
    ],
    [
        [
            172, 21, 136, 167, 111, 36, 168, 90, 36, 244, 208, 208, 34, 42, 91, 158, 121, 11, 214,
            17, 150, 220, 169, 137, 174, 234, 191, 231, 228, 166, 244, 39,
        ],
        [
            247, 84, 154, 37, 49, 122, 163, 113, 225, 52, 106, 231, 138, 127, 51, 135, 176, 76,
            175, 217, 247, 243, 84, 86, 27, 91, 136, 106, 29, 19, 70, 32,
        ],
        [
            183, 140, 119, 40, 234, 18, 150, 188, 245, 145, 29, 134, 108, 239, 3, 75, 85, 135, 174,
            215, 248, 64, 240, 135, 80, 107, 134, 242, 237, 181, 254, 34,
        ],
        [
            157, 21, 160, 26, 117, 32, 17, 245, 64, 238, 132, 120, 163, 148, 190, 127, 55, 73, 224,
            219, 166, 32, 236, 26, 161, 153, 206, 37, 218, 158, 115, 50,
        ],
        [
            102, 128, 174, 242, 77, 163, 172, 154, 138, 244, 102, 17, 122, 197, 231, 16, 67, 251,
            112, 69, 234, 176, 66, 247, 133, 177, 21, 204, 174, 237, 4, 49,
        ],
        [
            239, 96, 167, 25, 10, 118, 236, 57, 32, 103, 160, 19, 209, 187, 230, 224, 85, 134, 110,
            238, 216, 152, 208, 109, 0, 188, 254, 240, 93, 53, 109, 23,
        ],
        [
            108, 28, 154, 95, 92, 197, 178, 24, 211, 70, 120, 40, 247, 84, 168, 186, 7, 171, 75,
            120, 151, 200, 22, 59, 21, 170, 251, 131, 56, 235, 175, 30,
        ],
        [
            167, 85, 243, 125, 181, 241, 122, 202, 14, 50, 200, 228, 168, 106, 9, 243, 21, 134,
            155, 179, 123, 98, 111, 203, 96, 238, 117, 102, 244, 82, 32, 3,
        ],
    ],
    [
        [
            81, 194, 89, 56, 62, 155, 254, 27, 234, 67, 235, 139, 206, 253, 178, 220, 111, 228,
            143, 76, 232, 237, 223, 210, 227, 76, 234, 83, 158, 41, 165, 30,
        ],
        [
            231, 16, 196, 172, 106, 248, 216, 179, 18, 11, 178, 169, 44, 149, 81, 254, 108, 74,
            207, 135, 95, 26, 32, 80, 180, 103, 196, 109, 126, 18, 241, 26,
        ],
        [
            25, 140, 4, 125, 204, 220, 57, 121, 155, 173, 62, 202, 248, 55, 151, 70, 91, 173, 109,
            160, 23, 245, 116, 228, 68, 199, 22, 29, 170, 0, 10, 36,
        ],
        [
            170, 226, 29, 101, 137, 91, 118, 169, 27, 250, 42, 0, 135, 49, 146, 159, 97, 105, 241,
            55, 67, 34, 72, 162, 122, 167, 18, 120, 202, 9, 82, 39,
        ],
        [
            130, 23, 40, 251, 34, 205, 36, 24, 34, 193, 84, 226, 32, 35, 205, 182, 98, 64, 218, 99,
            223, 29, 193, 200, 187, 51, 94, 127, 85, 136, 106, 63,
        ],
        [
            20, 174, 125, 99, 207, 8, 50, 43, 203, 1, 165, 193, 43, 225, 16, 254, 207, 164, 119,
            162, 140, 94, 66, 208, 113, 7, 89, 200, 55, 53, 169, 24,
        ],
        [
            114, 146, 145, 183, 20, 171, 214, 161, 11, 200, 195, 62, 124, 57, 225, 210, 37, 202,
            108, 255, 246, 167, 197, 3, 205, 66, 206, 131, 22, 60, 71, 51,
        ],
        [
            96, 247, 39, 97, 175, 212, 113, 61, 218, 136, 61, 57, 34, 15, 242, 54, 133, 211, 151,
            28, 183, 159, 134, 188, 192, 104, 174, 54, 236, 231, 26, 5,
        ],
    ],
    [
        [
            159, 212, 195, 141, 31, 242, 46, 244, 34, 181, 77, 152, 114, 59, 111, 27, 127, 224,
            253, 146, 51, 221, 241, 189, 211, 26, 127, 150, 115, 126, 255, 8,
        ],
        [
            166, 51, 28, 89, 18, 65, 151, 110, 54, 47, 186, 87, 146, 233, 13, 36, 99, 46, 194, 38,
            248, 235, 17, 179, 225, 64, 174, 192, 149, 76, 193, 57,
        ],
        [
            124, 227, 148, 2, 195, 251, 0, 91, 99, 91, 4, 61, 36, 61, 60, 185, 160, 8, 216, 162,
            255, 163, 156, 212, 189, 234, 160, 131, 80, 233, 69, 31,
        ],
        [
            228, 5, 75, 168, 81, 179, 20, 142, 53, 62, 19, 152, 211, 120, 178, 172, 214, 30, 69,
            85, 124, 165, 73, 139, 93, 24, 36, 46, 199, 96, 40, 30,
        ],
        [
            252, 20, 32, 26, 246, 30, 192, 50, 136, 144, 68, 182, 232, 174, 73, 4, 252, 168, 180,
            46, 189, 66, 59, 7, 189, 2, 72, 92, 118, 220, 240, 43,
        ],
        [
            61, 140, 148, 113, 74, 83, 138, 136, 151, 151, 156, 219, 38, 244, 28, 186, 62, 172, 40,
            239, 14, 13, 100, 187, 136, 215, 226, 156, 167, 53, 46, 22,
        ],
        [
            137, 240, 137, 254, 218, 224, 225, 48, 30, 129, 224, 75, 203, 160, 113, 2, 238, 6, 205,
            79, 230, 109, 24, 44, 5, 147, 43, 123, 120, 33, 176, 16,
        ],
        [
            252, 48, 156, 11, 223, 23, 120, 45, 236, 97, 109, 212, 138, 155, 248, 142, 17, 14, 185,
            199, 52, 218, 56, 244, 239, 186, 59, 155, 82, 48, 200, 39,
        ],
    ],
    [
        [
            5, 220, 5, 128, 148, 124, 206, 148, 180, 145, 11, 158, 188, 112, 206, 237, 160, 123,
            218, 124, 1, 54, 164, 128, 18, 95, 83, 183, 248, 29, 228, 12,
        ],
        [
            214, 168, 154, 183, 172, 78, 16, 52, 50, 33, 14, 96, 238, 44, 248, 23, 157, 35, 215,
            83, 221, 57, 161, 29, 193, 11, 161, 179, 229, 48, 198, 21,
        ],
        [
            133, 164, 224, 39, 107, 6, 45, 223, 31, 74, 95, 252, 109, 219, 126, 219, 154, 63, 190,
            71, 193, 16, 23, 146, 1, 24, 128, 52, 193, 130, 160, 23,
        ],
        [
            121, 231, 94, 211, 129, 23, 194, 169, 190, 169, 243, 221, 172, 104, 207, 254, 157, 174,
            139, 227, 208, 20, 188, 82, 82, 190, 139, 190, 72, 23, 205, 20,
        ],
        [
            91, 172, 54, 154, 40, 43, 90, 99, 254, 241, 239, 151, 148, 167, 189, 194, 248, 200, 56,
            103, 126, 85, 184, 208, 17, 168, 236, 248, 153, 208, 179, 31,
        ],
        [
            205, 123, 215, 56, 96, 17, 215, 28, 168, 216, 148, 55, 15, 133, 60, 195, 164, 134, 17,
            234, 32, 36, 207, 201, 14, 112, 211, 78, 203, 7, 230, 52,
        ],
        [
            93, 248, 248, 21, 87, 109, 35, 58, 212, 50, 248, 161, 115, 63, 189, 191, 135, 115, 223,
            208, 78, 213, 39, 118, 91, 211, 131, 192, 33, 234, 149, 37,
        ],
        [
            35, 215, 54, 145, 79, 225, 122, 226, 127, 245, 11, 190, 251, 200, 1, 47, 80, 179, 224,
            211, 121, 180, 217, 47, 108, 41, 65, 53, 77, 93, 205, 47,
        ],
    ],
    [
        [
            121, 229, 227, 180, 214, 66, 64, 75, 82, 108, 25, 2, 216, 161, 23, 42, 45, 75, 106,
            187, 162, 63, 32, 6, 45, 189, 38, 184, 218, 18, 6, 43,
        ],
        [
            65, 81, 179, 34, 92, 125, 223, 47, 254, 226, 115, 185, 128, 172, 68, 24, 9, 184, 57,
            129, 234, 174, 232, 12, 171, 145, 73, 72, 18, 195, 42, 11,
        ],
        [
            14, 28, 16, 216, 210, 137, 236, 100, 226, 55, 225, 147, 154, 252, 211, 181, 106, 243,
            161, 79, 247, 183, 91, 55, 111, 18, 181, 172, 154, 5, 151, 53,
        ],
        [
            59, 222, 201, 101, 150, 48, 2, 24, 246, 2, 107, 177, 44, 141, 54, 27, 244, 1, 91, 52,
            200, 187, 171, 156, 135, 208, 249, 222, 154, 42, 16, 33,
        ],
        [
            104, 150, 234, 215, 26, 74, 45, 216, 243, 25, 122, 153, 44, 247, 128, 15, 164, 80, 95,
            105, 74, 164, 174, 150, 221, 180, 164, 117, 94, 121, 184, 21,
        ],
        [
            244, 3, 63, 14, 247, 139, 139, 24, 56, 152, 122, 211, 220, 149, 32, 72, 173, 50, 233,
            95, 192, 61, 115, 222, 6, 198, 23, 145, 220, 92, 168, 14,
        ],
        [
            123, 105, 48, 25, 179, 136, 51, 189, 62, 161, 38, 50, 41, 89, 206, 31, 92, 164, 227,
            29, 69, 233, 226, 53, 29, 17, 61, 221, 213, 199, 213, 36,
        ],
        [
            10, 180, 201, 130, 164, 6, 35, 113, 109, 164, 34, 227, 3, 202, 86, 70, 95, 236, 68, 10,
            198, 16, 204, 49, 49, 232, 145, 2, 205, 237, 135, 33,
        ],
    ],
    [
        [
            164, 61, 189, 226, 93, 173, 150, 18, 110, 21, 99, 125, 250, 45, 117, 50, 250, 157, 212,
            9, 169, 14, 232, 53, 168, 110, 221, 253, 98, 44, 184, 55,
        ],
        [
            10, 200, 231, 57, 40, 195, 109, 223, 230, 107, 35, 167, 70, 250, 255, 175, 106, 241,
            35, 163, 24, 65, 178, 218, 181, 40, 201, 150, 29, 181, 120, 1,
        ],
        [
            23, 147, 76, 178, 80, 155, 42, 165, 82, 142, 164, 81, 100, 180, 173, 9, 26, 9, 233, 6,
            67, 3, 203, 70, 213, 195, 1, 93, 123, 169, 4, 12,
        ],
        [
            15, 24, 204, 61, 20, 190, 93, 198, 16, 129, 215, 142, 41, 33, 219, 224, 34, 154, 94,
            234, 109, 21, 251, 232, 198, 62, 4, 235, 236, 16, 3, 45,
        ],
        [
            143, 177, 225, 157, 173, 54, 188, 123, 41, 91, 240, 220, 223, 81, 70, 194, 27, 147,
            141, 230, 175, 43, 25, 222, 7, 143, 81, 185, 252, 252, 13, 63,
        ],
        [
            2, 130, 165, 36, 212, 154, 222, 127, 66, 123, 167, 90, 242, 142, 84, 162, 154, 244, 10,
            228, 7, 110, 213, 182, 236, 30, 202, 0, 248, 20, 219, 31,
        ],
        [
            91, 141, 198, 216, 211, 179, 193, 160, 134, 90, 212, 128, 85, 185, 8, 213, 156, 178,
            84, 94, 33, 60, 104, 15, 76, 132, 101, 139, 16, 202, 39, 7,
        ],
        [
            40, 21, 104, 114, 55, 102, 78, 10, 8, 169, 121, 156, 143, 255, 176, 144, 216, 134, 19,
            27, 70, 151, 163, 149, 21, 220, 239, 65, 109, 137, 163, 36,
        ],
    ],
    [
        [
            177, 102, 25, 89, 51, 205, 218, 46, 239, 16, 165, 74, 131, 94, 82, 248, 57, 135, 156,
            160, 63, 57, 210, 21, 30, 42, 243, 39, 215, 37, 128, 28,
        ],
        [
            105, 52, 188, 198, 150, 99, 113, 232, 68, 219, 243, 180, 210, 216, 187, 179, 27, 118,
            63, 211, 239, 147, 39, 75, 243, 209, 136, 247, 151, 84, 146, 32,
        ],
        [
            114, 57, 148, 130, 175, 207, 73, 255, 149, 172, 194, 92, 217, 117, 177, 223, 52, 37,
            51, 57, 169, 149, 212, 248, 73, 221, 164, 0, 186, 245, 112, 23,
        ],
        [
            57, 72, 243, 117, 146, 120, 9, 114, 211, 234, 216, 19, 43, 149, 3, 198, 27, 4, 239, 93,
            46, 75, 219, 149, 112, 40, 54, 28, 48, 234, 66, 57,
        ],
        [
            67, 161, 31, 89, 164, 242, 224, 205, 207, 9, 154, 77, 198, 216, 166, 150, 167, 168, 60,
            83, 229, 54, 120, 248, 82, 105, 61, 130, 142, 46, 193, 18,
        ],
        [
            45, 128, 84, 140, 202, 147, 79, 95, 51, 250, 148, 114, 188, 145, 61, 29, 62, 132, 115,
            244, 102, 11, 0, 96, 85, 30, 86, 136, 195, 144, 247, 18,
        ],
        [
            7, 154, 159, 131, 116, 203, 12, 167, 60, 97, 53, 6, 77, 133, 249, 117, 25, 204, 243,
            118, 175, 154, 233, 70, 243, 130, 27, 127, 179, 205, 253, 15,
        ],
        [
            106, 108, 100, 123, 231, 100, 59, 207, 108, 100, 34, 155, 222, 229, 31, 255, 98, 181,
            171, 122, 9, 114, 3, 101, 255, 72, 161, 61, 8, 219, 123, 27,
        ],
    ],
    [
        [
            138, 91, 16, 195, 19, 101, 70, 243, 47, 48, 96, 34, 191, 184, 150, 226, 190, 100, 88,
            119, 234, 152, 160, 152, 104, 219, 29, 239, 19, 148, 9, 38,
        ],
        [
            246, 211, 73, 28, 125, 13, 63, 122, 5, 88, 167, 151, 179, 186, 84, 13, 111, 163, 102,
            113, 126, 241, 32, 50, 224, 13, 146, 15, 230, 150, 11, 52,
        ],
        [
            154, 150, 57, 187, 127, 246, 180, 239, 190, 165, 103, 233, 89, 183, 5, 175, 150, 93,
            87, 51, 167, 126, 178, 169, 66, 28, 9, 161, 218, 239, 123, 25,
        ],
        [
            185, 229, 244, 166, 244, 110, 77, 101, 167, 108, 163, 229, 108, 174, 226, 103, 89, 36,
            200, 164, 214, 148, 49, 212, 242, 197, 121, 167, 97, 157, 74, 17,
        ],
        [
            52, 58, 191, 122, 149, 177, 220, 26, 97, 103, 175, 44, 100, 111, 151, 136, 79, 54, 93,
            185, 65, 182, 102, 131, 165, 82, 46, 91, 114, 18, 91, 6,
        ],
        [
            107, 19, 80, 165, 166, 216, 121, 157, 215, 92, 135, 201, 246, 234, 30, 122, 36, 27,
            203, 88, 216, 113, 237, 199, 111, 120, 35, 119, 8, 205, 32, 46,
        ],
        [
            20, 119, 36, 113, 135, 162, 9, 149, 247, 211, 154, 128, 212, 137, 68, 212, 28, 87, 145,
            233, 53, 162, 167, 72, 114, 252, 255, 10, 110, 30, 117, 41,
        ],
        [
            69, 67, 187, 172, 32, 58, 82, 131, 180, 241, 128, 245, 245, 178, 215, 244, 40, 37, 49,
            66, 218, 99, 232, 17, 201, 214, 84, 241, 29, 215, 217, 16,
        ],
    ],
    [
        [
            8, 217, 27, 9, 205, 139, 87, 170, 175, 101, 13, 65, 42, 153, 124, 195, 221, 110, 147,
            106, 107, 50, 112, 113, 79, 85, 128, 96, 221, 122, 88, 37,
        ],
        [
            62, 174, 26, 84, 3, 154, 76, 218, 161, 234, 202, 236, 43, 222, 190, 214, 40, 177, 81,
            209, 138, 91, 117, 25, 45, 250, 233, 13, 223, 81, 61, 34,
        ],
        [
            112, 4, 242, 252, 45, 111, 178, 214, 78, 207, 68, 208, 81, 104, 157, 45, 160, 124, 213,
            193, 38, 80, 58, 205, 140, 9, 107, 41, 196, 141, 223, 40,
        ],
        [
            54, 79, 54, 153, 234, 76, 173, 78, 107, 19, 23, 164, 223, 21, 74, 147, 230, 20, 103,
            113, 131, 60, 33, 233, 200, 146, 23, 144, 189, 71, 94, 24,
        ],
        [
            113, 192, 128, 244, 20, 55, 236, 149, 20, 196, 221, 99, 71, 23, 208, 187, 92, 158, 5,
            73, 239, 163, 254, 39, 142, 161, 51, 180, 159, 209, 209, 12,
        ],
        [
            204, 251, 47, 230, 61, 123, 221, 149, 69, 100, 194, 77, 56, 250, 81, 132, 163, 128,
            181, 208, 116, 94, 151, 79, 31, 139, 66, 154, 91, 182, 52, 28,
        ],
        [
            83, 105, 243, 244, 101, 204, 108, 161, 15, 204, 51, 165, 200, 209, 129, 177, 30, 110,
            84, 4, 65, 203, 125, 221, 234, 8, 100, 183, 47, 52, 49, 25,
        ],
        [
            201, 43, 72, 102, 184, 137, 0, 15, 252, 35, 236, 135, 16, 241, 233, 164, 218, 184, 188,
            92, 174, 121, 33, 207, 166, 154, 39, 203, 16, 89, 89, 46,
        ],
    ],
];

pub fn generator() -> pallas::Affine {
    pallas::Affine::from_xy(
        pallas::Base::from_repr(GENERATOR.0).unwrap(),
        pallas::Base::from_repr(GENERATOR.1).unwrap(),
    )
    .unwrap()
}

#[cfg(test)]
mod tests {
    use super::super::{NUM_WINDOWS, VALUE_COMMITMENT_PERSONALIZATION};
    use super::*;
    use group::Curve;
    use halo2_gadgets::ecc::chip::constants::{test_lagrange_coeffs, test_zs_and_us};
    use pasta_curves::{
        arithmetic::{CurveAffine, CurveExt},
        pallas,
    };

    #[test]
    fn generator() {
        let hasher = pallas::Point::hash_to_curve(VALUE_COMMITMENT_PERSONALIZATION);
        let point = hasher(b"r");
        let coords = point.to_affine().coordinates().unwrap();

        assert_eq!(*coords.x(), pallas::Base::from_repr(GENERATOR.0).unwrap());
        assert_eq!(*coords.y(), pallas::Base::from_repr(GENERATOR.1).unwrap());
    }

    #[test]
    fn lagrange_coeffs() {
        let base = super::generator();
        test_lagrange_coeffs(base, NUM_WINDOWS);
    }

    #[test]
    fn z() {
        let base = super::generator();
        test_zs_and_us(base, &Z, &U, NUM_WINDOWS);
    }
}
