use pasta_curves::{
    arithmetic::{CurveAffine, FieldExt},
    pallas,
};

/// The value commitment is used to check balance between inputs and outputs. The value is
/// placed over this generator.
pub const GENERATOR: ([u8; 32], [u8; 32]) = (
    [
        99, 201, 117, 184, 132, 114, 26, 141, 12, 161, 112, 123, 227, 12, 127, 12, 95, 68, 95, 62,
        124, 24, 141, 59, 6, 214, 241, 40, 179, 35, 85, 55,
    ],
    [
        201, 59, 12, 123, 129, 62, 227, 76, 216, 189, 5, 192, 254, 20, 201, 223, 251, 36, 214, 254,
        252, 188, 16, 123, 219, 102, 26, 223, 127, 53, 208, 26,
    ],
);

/// Full-width z-values for GENERATOR
pub const Z: [u64; super::NUM_WINDOWS] = [
    49707, 15701, 45931, 163127, 41654, 212130, 34473, 25205, 4118, 10240, 12264, 22866, 203610,
    18808, 13851, 62448, 62380, 94497, 39496, 73216, 32037, 32774, 61690, 39173, 74580, 84678,
    23418, 103090, 34763, 19801, 54976, 196082, 131117, 20556, 58936, 139049, 49530, 488, 2129,
    44219, 64328, 38875, 58430, 34536, 84014, 15455, 38059, 15915, 26893, 100337, 120701, 98937,
    37075, 35293, 8351, 8361, 273432, 717, 3253, 40140, 28024, 95195, 41937, 200127, 95471, 103562,
    75737, 4182, 362357, 15219, 136680, 168274, 25085, 5925, 254392, 93041, 56204, 46757, 109788,
    100797, 80349, 87315, 77372, 96572, 18965,
];

/// Full-width u-values for GENERATOR
pub const U: [[[u8; 32]; super::H]; super::NUM_WINDOWS] = [
    [
        [
            134, 139, 159, 167, 179, 203, 183, 86, 54, 69, 108, 127, 183, 40, 226, 188, 34, 72,
            235, 174, 120, 30, 214, 202, 46, 166, 177, 78, 133, 80, 221, 35,
        ],
        [
            128, 147, 194, 62, 34, 230, 94, 209, 44, 242, 88, 94, 91, 252, 184, 217, 72, 173, 32,
            128, 118, 158, 176, 67, 255, 192, 227, 150, 189, 56, 205, 10,
        ],
        [
            89, 235, 150, 72, 148, 202, 175, 172, 88, 248, 184, 63, 149, 194, 33, 149, 187, 218,
            51, 194, 131, 184, 34, 138, 62, 112, 115, 0, 197, 109, 145, 37,
        ],
        [
            55, 241, 4, 146, 234, 109, 232, 212, 90, 95, 84, 170, 84, 254, 20, 89, 80, 100, 143, 3,
            30, 229, 117, 35, 32, 249, 43, 36, 200, 74, 200, 13,
        ],
        [
            194, 23, 77, 68, 227, 113, 225, 181, 201, 67, 219, 25, 234, 73, 160, 104, 12, 6, 180,
            204, 185, 16, 224, 62, 83, 132, 193, 185, 69, 90, 67, 29,
        ],
        [
            156, 240, 85, 68, 24, 96, 228, 37, 250, 218, 25, 99, 51, 50, 37, 27, 23, 76, 42, 243,
            126, 23, 236, 249, 179, 94, 59, 83, 197, 193, 215, 32,
        ],
        [
            73, 145, 163, 254, 185, 142, 57, 198, 25, 48, 144, 127, 17, 171, 210, 170, 164, 88, 3,
            235, 37, 45, 124, 125, 44, 113, 141, 119, 93, 91, 163, 47,
        ],
        [
            144, 156, 74, 131, 227, 73, 40, 222, 73, 67, 210, 157, 154, 161, 96, 112, 61, 251, 210,
            2, 2, 199, 107, 190, 39, 133, 107, 181, 235, 217, 1, 32,
        ],
    ],
    [
        [
            204, 30, 193, 82, 109, 194, 174, 212, 203, 182, 251, 224, 85, 31, 240, 246, 163, 165,
            69, 160, 26, 78, 143, 50, 115, 92, 109, 71, 48, 205, 160, 15,
        ],
        [
            20, 52, 64, 133, 30, 26, 4, 37, 88, 2, 107, 192, 179, 203, 130, 109, 203, 163, 224, 31,
            238, 207, 238, 0, 79, 55, 35, 8, 69, 238, 155, 60,
        ],
        [
            129, 188, 56, 216, 72, 61, 212, 36, 171, 128, 228, 90, 229, 75, 126, 41, 186, 124, 100,
            62, 161, 224, 105, 69, 196, 160, 131, 35, 39, 217, 149, 40,
        ],
        [
            168, 4, 93, 175, 37, 163, 30, 50, 27, 141, 194, 23, 206, 18, 65, 85, 150, 36, 51, 169,
            206, 47, 128, 65, 151, 181, 117, 50, 74, 251, 37, 14,
        ],
        [
            143, 167, 209, 67, 187, 208, 101, 235, 49, 77, 89, 225, 7, 243, 50, 56, 150, 233, 186,
            94, 0, 161, 136, 54, 163, 101, 23, 180, 199, 82, 163, 32,
        ],
        [
            83, 85, 245, 163, 247, 47, 226, 3, 90, 35, 254, 250, 71, 176, 18, 214, 121, 5, 11, 162,
            55, 2, 116, 46, 36, 20, 81, 19, 209, 79, 84, 53,
        ],
        [
            181, 129, 189, 44, 171, 86, 12, 145, 21, 185, 255, 145, 238, 46, 220, 242, 242, 239, 5,
            39, 176, 87, 6, 205, 209, 13, 190, 26, 72, 252, 247, 47,
        ],
        [
            169, 151, 215, 50, 18, 225, 124, 186, 49, 5, 84, 177, 139, 69, 189, 88, 237, 39, 196,
            111, 251, 146, 226, 12, 207, 215, 254, 0, 246, 155, 86, 31,
        ],
    ],
    [
        [
            178, 207, 217, 160, 25, 246, 180, 59, 161, 29, 253, 55, 84, 14, 217, 170, 120, 167,
            203, 21, 147, 2, 190, 236, 198, 41, 99, 194, 47, 97, 213, 20,
        ],
        [
            45, 217, 42, 12, 249, 206, 77, 249, 191, 69, 78, 30, 201, 54, 60, 175, 218, 187, 222,
            73, 28, 240, 105, 118, 228, 192, 184, 161, 65, 230, 115, 35,
        ],
        [
            145, 233, 255, 221, 141, 221, 216, 67, 77, 53, 229, 49, 167, 130, 235, 235, 55, 39, 88,
            70, 146, 70, 243, 31, 156, 138, 246, 251, 13, 41, 2, 16,
        ],
        [
            44, 14, 167, 217, 244, 57, 222, 12, 243, 116, 172, 17, 218, 246, 161, 124, 209, 248,
            240, 97, 59, 176, 138, 242, 76, 23, 207, 179, 136, 243, 79, 12,
        ],
        [
            120, 239, 19, 236, 60, 149, 184, 181, 237, 83, 242, 22, 122, 53, 211, 43, 195, 217,
            208, 185, 53, 156, 105, 158, 161, 156, 47, 90, 67, 139, 159, 60,
        ],
        [
            249, 224, 188, 87, 191, 134, 192, 15, 207, 29, 167, 27, 149, 161, 44, 61, 81, 55, 15,
            30, 78, 203, 150, 136, 55, 222, 237, 245, 39, 227, 177, 29,
        ],
        [
            91, 228, 118, 231, 230, 113, 246, 243, 112, 73, 19, 146, 105, 173, 81, 28, 27, 59, 168,
            92, 82, 198, 192, 111, 208, 156, 210, 129, 148, 194, 13, 14,
        ],
        [
            224, 185, 193, 29, 240, 177, 244, 134, 56, 252, 24, 119, 213, 110, 191, 221, 105, 122,
            89, 30, 114, 196, 53, 33, 168, 48, 77, 85, 241, 128, 168, 51,
        ],
    ],
    [
        [
            79, 58, 209, 170, 171, 147, 19, 156, 23, 243, 181, 231, 92, 215, 234, 118, 17, 151,
            212, 115, 164, 220, 114, 178, 18, 147, 101, 201, 228, 162, 126, 54,
        ],
        [
            235, 245, 230, 232, 32, 231, 43, 206, 119, 97, 30, 131, 10, 3, 195, 93, 78, 158, 42,
            62, 53, 26, 42, 26, 123, 39, 164, 39, 149, 119, 79, 39,
        ],
        [
            78, 161, 190, 242, 222, 143, 16, 193, 35, 198, 206, 67, 37, 101, 231, 6, 62, 205, 36,
            2, 146, 11, 191, 111, 78, 107, 222, 19, 32, 194, 31, 28,
        ],
        [
            189, 36, 36, 147, 50, 162, 17, 179, 250, 219, 167, 100, 114, 56, 90, 184, 42, 49, 101,
            155, 213, 179, 140, 62, 245, 55, 81, 218, 11, 116, 203, 32,
        ],
        [
            170, 122, 46, 42, 82, 84, 63, 207, 174, 176, 7, 139, 74, 49, 198, 238, 243, 217, 155,
            175, 31, 22, 42, 49, 125, 6, 210, 80, 124, 138, 184, 42,
        ],
        [
            30, 149, 102, 115, 26, 193, 146, 161, 139, 248, 18, 9, 55, 231, 50, 78, 236, 158, 74,
            204, 173, 228, 73, 125, 75, 127, 42, 21, 202, 128, 69, 46,
        ],
        [
            33, 237, 143, 1, 212, 254, 16, 231, 202, 39, 42, 28, 134, 77, 246, 51, 31, 146, 160,
            11, 178, 194, 169, 251, 50, 131, 241, 132, 81, 122, 244, 58,
        ],
        [
            197, 18, 235, 234, 189, 238, 252, 93, 244, 246, 76, 251, 21, 250, 8, 118, 38, 40, 39,
            103, 96, 147, 112, 103, 188, 219, 33, 183, 6, 139, 186, 47,
        ],
    ],
    [
        [
            27, 179, 1, 219, 5, 237, 22, 195, 49, 31, 125, 207, 224, 185, 249, 141, 188, 16, 209,
            225, 213, 167, 101, 63, 245, 229, 177, 242, 28, 121, 227, 4,
        ],
        [
            209, 90, 156, 216, 151, 223, 46, 129, 150, 195, 74, 57, 16, 102, 32, 110, 156, 108, 48,
            128, 236, 231, 35, 169, 60, 210, 63, 240, 252, 175, 28, 22,
        ],
        [
            133, 6, 154, 178, 236, 58, 163, 25, 85, 197, 169, 75, 26, 103, 112, 168, 225, 245, 103,
            211, 223, 163, 79, 47, 253, 17, 114, 233, 127, 217, 94, 51,
        ],
        [
            252, 137, 21, 60, 226, 116, 238, 97, 66, 128, 128, 213, 51, 35, 35, 130, 77, 160, 200,
            226, 91, 25, 22, 133, 27, 188, 198, 159, 155, 4, 106, 4,
        ],
        [
            76, 129, 23, 15, 64, 93, 179, 128, 41, 202, 40, 164, 7, 199, 172, 203, 98, 179, 243,
            28, 66, 149, 80, 177, 112, 51, 81, 231, 178, 208, 16, 54,
        ],
        [
            100, 99, 114, 178, 31, 130, 16, 122, 18, 47, 103, 243, 55, 84, 112, 92, 156, 246, 119,
            91, 4, 173, 157, 48, 246, 132, 129, 162, 33, 42, 164, 38,
        ],
        [
            152, 167, 13, 134, 122, 121, 84, 244, 205, 49, 40, 241, 68, 223, 97, 107, 112, 216,
            124, 98, 232, 219, 187, 158, 28, 95, 222, 129, 219, 240, 233, 13,
        ],
        [
            255, 191, 150, 9, 220, 109, 94, 217, 202, 19, 194, 216, 152, 81, 167, 107, 204, 138,
            104, 200, 200, 253, 210, 14, 87, 133, 149, 93, 119, 137, 129, 45,
        ],
    ],
    [
        [
            162, 107, 4, 19, 49, 176, 93, 99, 15, 133, 159, 53, 191, 0, 79, 228, 236, 96, 145, 53,
            120, 10, 150, 94, 198, 49, 253, 119, 229, 116, 150, 13,
        ],
        [
            205, 164, 35, 49, 168, 151, 217, 138, 164, 143, 237, 170, 140, 153, 15, 171, 74, 89,
            60, 195, 13, 20, 170, 36, 66, 199, 45, 252, 18, 47, 204, 8,
        ],
        [
            95, 149, 177, 178, 217, 143, 235, 251, 203, 227, 33, 62, 147, 134, 228, 83, 84, 170,
            118, 91, 103, 94, 41, 99, 1, 95, 59, 67, 253, 146, 165, 5,
        ],
        [
            19, 184, 182, 93, 1, 4, 202, 21, 119, 77, 151, 12, 98, 255, 223, 34, 54, 162, 44, 199,
            13, 217, 159, 37, 147, 185, 89, 21, 251, 128, 21, 25,
        ],
        [
            190, 179, 248, 211, 188, 233, 181, 149, 178, 232, 28, 253, 234, 103, 160, 153, 252,
            188, 19, 132, 143, 227, 31, 109, 134, 85, 21, 104, 133, 39, 148, 13,
        ],
        [
            149, 40, 255, 15, 159, 19, 43, 100, 31, 196, 107, 209, 83, 25, 77, 122, 211, 27, 237,
            158, 74, 188, 198, 73, 86, 171, 73, 163, 110, 176, 37, 55,
        ],
        [
            25, 194, 88, 151, 108, 22, 157, 145, 11, 87, 6, 215, 227, 141, 203, 218, 34, 25, 236,
            49, 69, 170, 243, 13, 240, 210, 161, 186, 45, 32, 239, 12,
        ],
        [
            100, 133, 83, 207, 52, 184, 61, 180, 226, 133, 42, 161, 249, 216, 158, 39, 5, 185, 52,
            8, 212, 185, 49, 234, 32, 200, 23, 158, 19, 111, 74, 50,
        ],
    ],
    [
        [
            215, 206, 249, 236, 176, 248, 217, 12, 0, 204, 155, 112, 248, 215, 200, 248, 8, 250,
            61, 104, 22, 109, 196, 54, 27, 3, 0, 203, 187, 116, 69, 45,
        ],
        [
            82, 29, 122, 199, 191, 205, 105, 136, 96, 93, 249, 65, 104, 197, 48, 64, 166, 175, 144,
            96, 114, 186, 147, 65, 1, 35, 87, 134, 90, 145, 230, 24,
        ],
        [
            154, 172, 87, 147, 10, 166, 129, 223, 224, 162, 95, 194, 248, 41, 234, 174, 177, 221,
            64, 244, 202, 201, 225, 138, 179, 110, 80, 104, 84, 39, 26, 43,
        ],
        [
            75, 226, 94, 205, 107, 248, 174, 164, 95, 115, 20, 62, 241, 202, 80, 182, 248, 213,
            115, 216, 142, 165, 182, 8, 151, 156, 170, 110, 219, 237, 63, 3,
        ],
        [
            49, 234, 64, 230, 146, 237, 191, 49, 112, 45, 60, 43, 252, 216, 195, 70, 153, 146, 225,
            144, 9, 226, 23, 158, 136, 113, 247, 108, 62, 144, 108, 9,
        ],
        [
            103, 21, 127, 107, 59, 109, 133, 160, 143, 88, 92, 94, 63, 60, 176, 30, 201, 130, 24,
            59, 22, 197, 68, 223, 207, 57, 45, 122, 91, 202, 196, 35,
        ],
        [
            52, 102, 244, 254, 244, 16, 81, 190, 173, 244, 154, 38, 64, 214, 63, 97, 169, 9, 167,
            51, 246, 33, 109, 247, 50, 220, 114, 64, 13, 179, 177, 30,
        ],
        [
            144, 221, 225, 200, 23, 166, 32, 166, 176, 75, 198, 254, 19, 193, 245, 128, 7, 12, 208,
            85, 254, 120, 147, 143, 63, 184, 236, 60, 26, 253, 251, 35,
        ],
    ],
    [
        [
            75, 83, 105, 13, 6, 123, 42, 108, 212, 123, 233, 193, 139, 45, 132, 88, 44, 242, 39,
            25, 1, 235, 41, 93, 161, 218, 156, 90, 105, 43, 114, 46,
        ],
        [
            113, 216, 156, 147, 163, 145, 14, 155, 250, 78, 89, 11, 16, 228, 91, 223, 129, 87, 8,
            161, 204, 28, 0, 75, 59, 172, 225, 121, 76, 165, 42, 36,
        ],
        [
            195, 34, 156, 131, 184, 36, 15, 228, 23, 198, 79, 88, 138, 166, 168, 21, 16, 35, 82,
            123, 87, 247, 59, 42, 195, 113, 128, 240, 223, 247, 79, 25,
        ],
        [
            197, 57, 189, 78, 132, 12, 150, 148, 108, 113, 199, 232, 67, 154, 65, 201, 151, 136,
            175, 146, 0, 126, 215, 68, 183, 100, 207, 132, 39, 238, 74, 28,
        ],
        [
            213, 35, 155, 170, 77, 61, 206, 127, 232, 164, 11, 133, 55, 141, 203, 73, 112, 37, 27,
            8, 250, 18, 169, 233, 171, 174, 218, 9, 15, 227, 192, 24,
        ],
        [
            134, 99, 90, 204, 109, 34, 78, 239, 97, 108, 119, 18, 165, 166, 98, 20, 155, 15, 193,
            71, 34, 134, 45, 232, 229, 4, 167, 6, 75, 99, 2, 54,
        ],
        [
            4, 136, 7, 177, 132, 128, 188, 128, 221, 121, 213, 198, 133, 42, 219, 156, 145, 179,
            35, 254, 88, 233, 96, 90, 206, 150, 181, 191, 254, 218, 217, 31,
        ],
        [
            183, 202, 252, 66, 36, 234, 250, 116, 96, 15, 72, 88, 124, 136, 8, 156, 204, 151, 89,
            171, 185, 139, 47, 215, 117, 207, 44, 117, 226, 84, 248, 5,
        ],
    ],
    [
        [
            48, 78, 243, 80, 211, 5, 172, 75, 113, 98, 154, 247, 185, 52, 36, 155, 5, 244, 149, 7,
            84, 162, 39, 0, 233, 219, 188, 229, 84, 18, 154, 34,
        ],
        [
            135, 176, 3, 156, 231, 35, 196, 24, 109, 104, 169, 210, 121, 247, 72, 214, 141, 15,
            155, 221, 249, 30, 196, 117, 108, 229, 219, 116, 233, 89, 208, 61,
        ],
        [
            108, 238, 126, 36, 119, 110, 54, 206, 165, 24, 218, 88, 206, 201, 226, 190, 28, 76,
            125, 62, 87, 21, 15, 158, 2, 233, 41, 18, 242, 223, 192, 6,
        ],
        [
            169, 95, 111, 93, 78, 167, 161, 2, 54, 29, 206, 209, 119, 196, 7, 36, 248, 233, 68, 45,
            243, 58, 155, 147, 32, 175, 221, 168, 144, 79, 126, 18,
        ],
        [
            229, 0, 65, 205, 148, 179, 229, 116, 253, 143, 119, 111, 142, 63, 235, 124, 140, 243,
            182, 164, 15, 57, 219, 23, 114, 237, 116, 213, 200, 235, 227, 55,
        ],
        [
            9, 233, 193, 136, 60, 112, 74, 107, 87, 12, 113, 20, 8, 8, 94, 195, 227, 83, 178, 77,
            143, 104, 211, 169, 253, 62, 8, 196, 81, 39, 96, 13,
        ],
        [
            223, 12, 147, 224, 113, 27, 48, 142, 243, 199, 244, 62, 255, 20, 63, 79, 12, 37, 159,
            81, 80, 172, 136, 23, 107, 186, 180, 33, 148, 49, 128, 10,
        ],
        [
            100, 248, 42, 239, 155, 107, 80, 255, 126, 96, 195, 152, 204, 166, 14, 46, 114, 152,
            200, 111, 94, 233, 117, 247, 89, 83, 34, 2, 189, 3, 245, 4,
        ],
    ],
    [
        [
            26, 254, 208, 112, 167, 15, 77, 90, 124, 14, 227, 31, 204, 94, 253, 228, 223, 252, 114,
            141, 234, 26, 108, 144, 115, 131, 19, 142, 25, 148, 80, 53,
        ],
        [
            95, 154, 56, 85, 66, 246, 97, 207, 254, 117, 98, 51, 204, 68, 9, 1, 157, 25, 113, 175,
            12, 172, 33, 67, 212, 17, 238, 76, 166, 43, 6, 30,
        ],
        [
            48, 204, 33, 34, 141, 223, 27, 151, 244, 225, 206, 129, 54, 211, 251, 18, 168, 186,
            214, 141, 37, 229, 219, 62, 187, 87, 66, 140, 181, 233, 161, 35,
        ],
        [
            143, 158, 2, 174, 102, 73, 205, 247, 110, 78, 131, 38, 192, 195, 70, 193, 14, 166, 232,
            42, 142, 222, 35, 147, 189, 77, 50, 220, 23, 56, 126, 38,
        ],
        [
            10, 84, 87, 89, 169, 77, 97, 156, 0, 100, 41, 191, 232, 88, 138, 82, 220, 123, 231, 59,
            142, 115, 196, 116, 217, 80, 178, 117, 237, 250, 11, 32,
        ],
        [
            152, 88, 228, 103, 64, 4, 197, 110, 204, 4, 0, 131, 250, 205, 206, 83, 180, 179, 98,
            51, 213, 11, 237, 149, 150, 226, 123, 62, 129, 198, 28, 58,
        ],
        [
            188, 254, 66, 154, 178, 61, 236, 232, 27, 107, 131, 38, 104, 160, 49, 57, 3, 111, 18,
            106, 158, 91, 57, 220, 174, 59, 138, 227, 176, 49, 171, 15,
        ],
        [
            92, 116, 148, 172, 212, 215, 137, 153, 48, 241, 40, 25, 45, 204, 141, 58, 235, 146,
            155, 248, 50, 112, 4, 206, 228, 70, 94, 117, 128, 150, 213, 18,
        ],
    ],
    [
        [
            155, 96, 199, 129, 161, 131, 9, 109, 43, 95, 79, 108, 230, 168, 172, 51, 244, 183, 203,
            211, 198, 95, 64, 184, 12, 184, 190, 250, 108, 77, 202, 61,
        ],
        [
            151, 114, 132, 115, 19, 130, 213, 126, 140, 64, 58, 149, 171, 26, 245, 47, 253, 9, 212,
            80, 154, 3, 206, 254, 158, 241, 87, 90, 67, 8, 29, 59,
        ],
        [
            231, 100, 245, 101, 53, 146, 235, 219, 228, 106, 101, 51, 85, 110, 51, 78, 221, 149,
            223, 78, 184, 47, 144, 234, 161, 28, 6, 48, 41, 228, 253, 57,
        ],
        [
            239, 82, 160, 74, 136, 231, 198, 194, 99, 192, 170, 182, 36, 125, 149, 58, 9, 239, 91,
            146, 54, 104, 94, 42, 147, 4, 61, 79, 185, 11, 51, 1,
        ],
        [
            56, 89, 137, 88, 196, 32, 115, 15, 6, 53, 123, 43, 42, 27, 226, 220, 196, 131, 211,
            126, 14, 174, 191, 229, 172, 6, 210, 189, 92, 66, 79, 26,
        ],
        [
            236, 153, 243, 131, 143, 123, 164, 95, 214, 160, 21, 150, 12, 71, 118, 170, 92, 166,
            223, 162, 133, 58, 93, 2, 31, 57, 197, 190, 66, 169, 112, 18,
        ],
        [
            137, 13, 132, 54, 102, 120, 119, 177, 34, 217, 123, 89, 76, 242, 218, 72, 236, 129, 33,
            42, 148, 242, 221, 33, 252, 74, 219, 142, 25, 82, 162, 62,
        ],
        [
            0, 132, 243, 70, 85, 72, 216, 242, 225, 170, 56, 212, 52, 143, 154, 202, 248, 33, 148,
            236, 117, 24, 29, 63, 148, 233, 28, 74, 59, 44, 221, 6,
        ],
    ],
    [
        [
            122, 45, 241, 68, 72, 20, 58, 84, 181, 215, 45, 12, 74, 54, 17, 144, 144, 97, 82, 103,
            81, 123, 158, 238, 167, 173, 212, 8, 90, 63, 188, 59,
        ],
        [
            190, 69, 211, 175, 56, 60, 42, 23, 208, 231, 241, 96, 205, 55, 222, 210, 178, 240, 230,
            118, 210, 253, 59, 206, 104, 165, 221, 249, 237, 74, 42, 33,
        ],
        [
            69, 202, 255, 56, 181, 188, 225, 92, 19, 165, 74, 192, 67, 249, 139, 35, 87, 125, 27,
            228, 61, 78, 56, 144, 42, 255, 159, 217, 53, 99, 10, 62,
        ],
        [
            17, 213, 190, 3, 139, 251, 81, 49, 133, 14, 185, 61, 240, 132, 117, 99, 240, 125, 104,
            191, 248, 27, 77, 11, 130, 122, 237, 184, 35, 100, 180, 3,
        ],
        [
            236, 29, 195, 209, 34, 84, 144, 163, 206, 89, 168, 108, 27, 11, 70, 115, 204, 213, 254,
            154, 127, 24, 198, 242, 246, 223, 181, 250, 165, 119, 22, 62,
        ],
        [
            60, 247, 191, 168, 239, 211, 60, 148, 241, 31, 18, 170, 16, 159, 36, 114, 42, 255, 166,
            224, 203, 137, 29, 159, 207, 121, 83, 18, 65, 177, 147, 58,
        ],
        [
            142, 74, 131, 209, 60, 216, 199, 80, 41, 112, 74, 182, 191, 182, 44, 195, 255, 209,
            212, 108, 139, 121, 174, 211, 221, 158, 186, 94, 7, 171, 195, 6,
        ],
        [
            198, 195, 122, 214, 128, 112, 1, 64, 196, 79, 62, 9, 3, 224, 192, 175, 23, 151, 79,
            177, 86, 124, 241, 131, 111, 212, 44, 236, 166, 31, 192, 15,
        ],
    ],
    [
        [
            50, 31, 67, 146, 113, 240, 176, 252, 99, 195, 220, 195, 236, 33, 235, 44, 22, 161, 130,
            84, 214, 230, 9, 201, 254, 31, 226, 194, 208, 156, 234, 31,
        ],
        [
            130, 252, 165, 204, 53, 14, 76, 171, 205, 52, 8, 140, 156, 25, 1, 79, 106, 128, 120,
            230, 96, 4, 65, 143, 120, 86, 95, 116, 178, 151, 251, 21,
        ],
        [
            82, 176, 110, 60, 13, 107, 15, 237, 57, 201, 226, 28, 131, 232, 139, 0, 26, 117, 17,
            21, 90, 52, 206, 160, 31, 184, 67, 211, 119, 126, 23, 2,
        ],
        [
            86, 78, 25, 205, 17, 108, 177, 47, 226, 5, 14, 124, 217, 100, 164, 146, 67, 56, 65,
            155, 54, 2, 97, 66, 248, 244, 46, 71, 35, 246, 90, 35,
        ],
        [
            28, 122, 89, 60, 230, 19, 230, 132, 129, 117, 243, 172, 180, 95, 210, 183, 231, 73,
            194, 50, 123, 93, 40, 116, 179, 64, 48, 127, 37, 161, 133, 1,
        ],
        [
            72, 176, 170, 234, 97, 26, 154, 13, 139, 23, 69, 194, 241, 59, 143, 29, 131, 72, 236,
            123, 90, 116, 132, 207, 226, 63, 190, 11, 9, 159, 225, 18,
        ],
        [
            71, 3, 178, 208, 101, 43, 142, 1, 34, 183, 67, 144, 188, 255, 243, 77, 160, 17, 3, 90,
            104, 102, 15, 237, 159, 17, 119, 223, 119, 180, 184, 11,
        ],
        [
            198, 247, 138, 113, 71, 170, 58, 158, 166, 18, 216, 133, 230, 166, 87, 243, 180, 129,
            59, 227, 21, 156, 206, 171, 200, 29, 193, 191, 35, 144, 110, 35,
        ],
    ],
    [
        [
            143, 29, 76, 191, 243, 57, 156, 88, 188, 187, 113, 179, 47, 12, 12, 80, 210, 133, 232,
            199, 196, 174, 105, 166, 185, 201, 72, 33, 182, 107, 10, 14,
        ],
        [
            164, 93, 48, 22, 226, 133, 56, 19, 13, 143, 250, 49, 96, 151, 50, 136, 186, 199, 102,
            135, 44, 77, 221, 203, 66, 145, 199, 234, 71, 151, 237, 60,
        ],
        [
            40, 169, 150, 23, 171, 18, 88, 233, 196, 71, 216, 65, 86, 136, 191, 225, 157, 148, 204,
            166, 81, 32, 185, 159, 33, 211, 227, 110, 49, 200, 239, 6,
        ],
        [
            177, 51, 226, 132, 191, 36, 206, 168, 107, 159, 196, 253, 111, 164, 203, 30, 229, 157,
            238, 168, 231, 1, 231, 132, 124, 170, 113, 72, 205, 71, 179, 61,
        ],
        [
            173, 21, 156, 246, 171, 56, 11, 126, 227, 206, 228, 132, 13, 35, 56, 214, 167, 169,
            199, 247, 183, 237, 175, 164, 56, 38, 12, 149, 16, 165, 206, 41,
        ],
        [
            128, 227, 45, 193, 60, 153, 0, 6, 8, 102, 44, 127, 61, 224, 183, 230, 212, 141, 1, 51,
            225, 100, 204, 170, 122, 148, 93, 219, 56, 143, 127, 14,
        ],
        [
            70, 116, 18, 220, 241, 134, 196, 152, 34, 203, 160, 47, 133, 155, 48, 238, 3, 231, 142,
            250, 130, 80, 127, 29, 47, 30, 247, 166, 7, 84, 97, 32,
        ],
        [
            198, 120, 45, 251, 227, 181, 90, 244, 88, 96, 139, 170, 238, 2, 214, 157, 142, 252, 38,
            70, 46, 231, 140, 254, 133, 163, 174, 239, 236, 31, 248, 47,
        ],
    ],
    [
        [
            55, 136, 210, 179, 46, 105, 249, 203, 6, 77, 242, 11, 176, 133, 40, 140, 94, 213, 221,
            116, 54, 114, 154, 208, 221, 104, 34, 129, 102, 30, 159, 58,
        ],
        [
            79, 251, 68, 141, 49, 255, 119, 149, 245, 56, 33, 15, 177, 119, 71, 34, 2, 157, 154,
            100, 0, 130, 254, 89, 119, 244, 255, 136, 145, 170, 75, 22,
        ],
        [
            246, 138, 15, 233, 177, 157, 214, 220, 249, 137, 175, 49, 64, 49, 163, 203, 16, 5, 156,
            137, 236, 177, 51, 237, 183, 248, 106, 64, 95, 201, 63, 60,
        ],
        [
            169, 216, 126, 241, 88, 115, 61, 5, 60, 196, 46, 248, 192, 24, 133, 247, 107, 35, 176,
            62, 3, 251, 187, 51, 131, 108, 26, 58, 71, 173, 69, 50,
        ],
        [
            84, 151, 140, 102, 38, 240, 56, 110, 112, 245, 122, 78, 225, 149, 139, 150, 7, 90, 124,
            202, 208, 243, 85, 76, 216, 157, 7, 245, 29, 230, 202, 57,
        ],
        [
            151, 205, 82, 162, 218, 18, 210, 30, 96, 49, 203, 180, 161, 163, 135, 58, 156, 223,
            248, 97, 155, 19, 244, 20, 204, 201, 109, 159, 51, 166, 157, 53,
        ],
        [
            12, 187, 38, 15, 49, 56, 192, 212, 23, 177, 153, 244, 192, 65, 217, 112, 210, 82, 187,
            254, 42, 203, 172, 84, 39, 84, 171, 8, 145, 13, 6, 50,
        ],
        [
            61, 241, 209, 199, 182, 87, 14, 9, 24, 44, 194, 13, 4, 85, 186, 9, 120, 82, 120, 0,
            167, 113, 231, 252, 218, 210, 112, 246, 163, 9, 247, 30,
        ],
    ],
    [
        [
            196, 38, 105, 221, 18, 136, 108, 66, 223, 180, 237, 185, 161, 102, 180, 224, 164, 195,
            0, 254, 210, 238, 158, 146, 253, 119, 73, 191, 148, 17, 152, 50,
        ],
        [
            76, 130, 205, 97, 172, 181, 61, 253, 66, 7, 37, 249, 147, 121, 205, 101, 67, 50, 84,
            125, 60, 49, 65, 55, 48, 130, 37, 86, 92, 89, 181, 35,
        ],
        [
            206, 105, 119, 230, 215, 16, 15, 177, 186, 198, 98, 203, 210, 114, 246, 76, 112, 132,
            63, 157, 216, 253, 212, 195, 36, 11, 58, 200, 70, 165, 187, 21,
        ],
        [
            95, 85, 152, 202, 120, 158, 2, 113, 140, 9, 152, 197, 28, 23, 243, 98, 30, 116, 241,
            59, 150, 179, 40, 13, 119, 61, 108, 115, 88, 216, 66, 62,
        ],
        [
            108, 254, 117, 155, 108, 68, 3, 230, 25, 36, 106, 204, 158, 15, 114, 204, 134, 182,
            239, 235, 104, 144, 35, 31, 84, 246, 38, 252, 87, 60, 20, 12,
        ],
        [
            14, 76, 210, 134, 98, 224, 219, 180, 88, 134, 93, 30, 34, 159, 112, 85, 114, 46, 244,
            59, 169, 188, 235, 108, 254, 236, 101, 86, 97, 132, 255, 62,
        ],
        [
            23, 167, 12, 59, 95, 158, 244, 22, 231, 148, 247, 162, 226, 95, 136, 77, 62, 234, 119,
            179, 29, 62, 160, 116, 94, 240, 7, 113, 75, 241, 201, 62,
        ],
        [
            233, 142, 117, 40, 158, 166, 10, 78, 255, 44, 235, 32, 185, 102, 244, 23, 251, 127, 12,
            249, 120, 18, 196, 115, 151, 251, 236, 217, 97, 244, 107, 52,
        ],
    ],
    [
        [
            107, 155, 149, 167, 240, 119, 46, 213, 7, 178, 44, 247, 191, 163, 247, 255, 109, 160,
            87, 134, 236, 13, 49, 223, 245, 64, 194, 243, 220, 166, 203, 59,
        ],
        [
            119, 76, 9, 136, 64, 42, 217, 106, 111, 151, 245, 107, 101, 191, 56, 195, 239, 79, 134,
            239, 31, 37, 60, 160, 131, 101, 90, 38, 70, 213, 103, 5,
        ],
        [
            184, 158, 152, 93, 132, 93, 66, 54, 106, 79, 134, 202, 210, 72, 74, 25, 96, 93, 178,
            109, 231, 248, 173, 100, 154, 238, 170, 161, 194, 75, 148, 35,
        ],
        [
            156, 17, 84, 74, 233, 248, 203, 84, 127, 132, 30, 245, 52, 4, 27, 53, 231, 22, 75, 33,
            66, 120, 175, 149, 77, 114, 237, 18, 99, 189, 126, 31,
        ],
        [
            78, 144, 141, 232, 241, 133, 132, 75, 214, 189, 122, 225, 96, 155, 59, 205, 144, 243,
            176, 140, 253, 16, 152, 59, 80, 164, 165, 153, 15, 89, 219, 50,
        ],
        [
            62, 16, 233, 111, 18, 125, 232, 184, 252, 2, 110, 148, 127, 176, 119, 162, 66, 211,
            198, 136, 64, 217, 189, 197, 121, 66, 32, 71, 147, 13, 146, 8,
        ],
        [
            2, 45, 226, 216, 24, 120, 51, 196, 19, 16, 152, 7, 178, 30, 88, 194, 255, 186, 231, 81,
            217, 67, 36, 199, 41, 12, 135, 142, 160, 247, 199, 34,
        ],
        [
            80, 193, 196, 122, 181, 155, 245, 75, 185, 218, 222, 58, 140, 110, 234, 148, 30, 19,
            243, 175, 188, 1, 57, 85, 163, 253, 211, 210, 200, 7, 249, 43,
        ],
    ],
    [
        [
            187, 240, 200, 140, 119, 234, 17, 25, 208, 192, 198, 107, 40, 102, 42, 212, 229, 1,
            231, 121, 19, 35, 249, 226, 185, 185, 225, 179, 84, 18, 191, 15,
        ],
        [
            249, 21, 206, 133, 57, 77, 143, 126, 192, 252, 36, 62, 173, 16, 52, 11, 7, 198, 149,
            220, 48, 193, 40, 253, 217, 235, 145, 96, 241, 215, 193, 16,
        ],
        [
            203, 97, 200, 83, 165, 80, 108, 220, 108, 185, 123, 227, 143, 247, 12, 200, 9, 195, 68,
            46, 238, 32, 161, 42, 247, 167, 117, 159, 191, 89, 194, 21,
        ],
        [
            200, 138, 253, 89, 202, 154, 62, 252, 110, 107, 99, 189, 49, 33, 69, 220, 211, 185, 89,
            62, 101, 231, 203, 68, 45, 5, 73, 39, 144, 230, 186, 43,
        ],
        [
            31, 42, 63, 181, 31, 131, 126, 192, 52, 179, 2, 178, 71, 193, 225, 189, 52, 153, 62,
            166, 207, 192, 10, 174, 67, 228, 150, 28, 169, 63, 242, 60,
        ],
        [
            228, 40, 179, 0, 71, 223, 234, 22, 20, 72, 239, 146, 142, 7, 168, 47, 102, 186, 137,
            113, 248, 114, 204, 153, 12, 22, 145, 205, 143, 109, 169, 3,
        ],
        [
            232, 227, 9, 11, 117, 128, 157, 126, 209, 210, 249, 174, 93, 37, 76, 150, 114, 36, 225,
            75, 198, 10, 191, 167, 6, 2, 244, 29, 191, 7, 21, 44,
        ],
        [
            86, 226, 203, 210, 152, 57, 49, 226, 171, 172, 86, 225, 126, 215, 249, 166, 100, 119,
            54, 180, 193, 199, 146, 147, 253, 16, 107, 16, 22, 21, 106, 31,
        ],
    ],
    [
        [
            249, 155, 151, 211, 192, 36, 207, 21, 36, 141, 169, 175, 223, 248, 235, 62, 161, 214,
            236, 25, 35, 202, 64, 158, 84, 97, 121, 1, 190, 142, 23, 56,
        ],
        [
            79, 129, 15, 128, 17, 234, 8, 173, 53, 238, 129, 60, 40, 6, 33, 216, 0, 171, 82, 80,
            227, 3, 219, 184, 48, 231, 244, 213, 115, 224, 162, 49,
        ],
        [
            246, 219, 42, 28, 42, 134, 38, 100, 60, 131, 51, 159, 148, 5, 219, 203, 159, 51, 175,
            176, 154, 20, 79, 168, 197, 140, 200, 98, 192, 113, 179, 48,
        ],
        [
            87, 7, 235, 9, 102, 19, 100, 91, 214, 230, 74, 213, 117, 120, 7, 239, 137, 255, 158,
            243, 160, 169, 30, 105, 28, 72, 188, 245, 76, 187, 227, 46,
        ],
        [
            158, 245, 74, 150, 32, 197, 25, 173, 237, 231, 201, 44, 210, 117, 134, 203, 75, 108,
            102, 63, 124, 33, 47, 100, 56, 65, 129, 102, 161, 211, 117, 22,
        ],
        [
            204, 70, 46, 194, 74, 212, 80, 20, 239, 11, 98, 124, 117, 38, 63, 246, 222, 191, 44,
            185, 131, 146, 129, 92, 158, 2, 87, 240, 150, 133, 57, 19,
        ],
        [
            236, 136, 27, 147, 102, 12, 170, 62, 254, 16, 6, 217, 27, 178, 115, 247, 60, 158, 45,
            219, 224, 99, 111, 154, 253, 46, 247, 200, 138, 156, 26, 26,
        ],
        [
            75, 211, 65, 63, 194, 212, 191, 3, 100, 173, 100, 101, 62, 25, 72, 147, 13, 148, 111,
            14, 157, 82, 27, 241, 125, 154, 243, 137, 14, 135, 241, 26,
        ],
    ],
    [
        [
            218, 202, 124, 110, 86, 218, 195, 2, 64, 68, 69, 220, 16, 26, 0, 40, 224, 238, 59, 69,
            246, 226, 3, 27, 219, 85, 17, 245, 133, 220, 230, 63,
        ],
        [
            212, 226, 110, 162, 36, 55, 75, 198, 129, 197, 158, 79, 126, 204, 159, 71, 33, 60, 121,
            48, 127, 147, 250, 124, 60, 87, 179, 173, 160, 204, 32, 53,
        ],
        [
            137, 110, 13, 7, 108, 200, 154, 204, 18, 86, 44, 221, 40, 171, 207, 5, 25, 42, 108, 2,
            147, 18, 42, 89, 101, 50, 48, 7, 233, 123, 235, 8,
        ],
        [
            206, 247, 190, 175, 201, 90, 146, 123, 203, 83, 104, 46, 243, 131, 105, 131, 39, 186,
            205, 103, 203, 204, 186, 196, 251, 101, 101, 250, 186, 160, 211, 12,
        ],
        [
            237, 149, 65, 13, 108, 153, 242, 14, 61, 62, 156, 46, 128, 126, 158, 246, 226, 128,
            243, 193, 10, 165, 120, 64, 72, 91, 32, 215, 90, 235, 82, 24,
        ],
        [
            185, 204, 139, 200, 5, 172, 176, 126, 105, 190, 243, 150, 126, 26, 28, 74, 143, 149,
            118, 82, 210, 254, 70, 30, 81, 23, 128, 89, 255, 171, 137, 63,
        ],
        [
            169, 75, 113, 101, 245, 119, 251, 230, 244, 201, 79, 85, 164, 86, 81, 155, 57, 44, 203,
            143, 173, 62, 78, 180, 1, 48, 216, 129, 223, 34, 105, 33,
        ],
        [
            71, 92, 139, 39, 87, 72, 88, 2, 1, 5, 154, 14, 244, 138, 253, 116, 113, 203, 7, 174,
            49, 216, 88, 94, 233, 251, 220, 17, 19, 62, 142, 41,
        ],
    ],
    [
        [
            61, 222, 98, 187, 178, 246, 136, 235, 234, 162, 29, 132, 16, 87, 58, 249, 140, 210, 62,
            204, 34, 166, 135, 164, 118, 21, 181, 46, 215, 232, 234, 41,
        ],
        [
            108, 124, 187, 66, 180, 167, 233, 17, 56, 166, 172, 61, 198, 167, 53, 66, 58, 24, 86,
            29, 24, 30, 59, 213, 140, 66, 162, 145, 212, 204, 115, 25,
        ],
        [
            126, 189, 246, 53, 52, 94, 147, 57, 67, 135, 51, 106, 225, 56, 234, 69, 109, 0, 84, 85,
            158, 111, 234, 117, 134, 231, 202, 117, 69, 62, 231, 62,
        ],
        [
            180, 135, 221, 126, 92, 15, 183, 27, 71, 155, 203, 35, 217, 20, 196, 191, 166, 148, 12,
            90, 70, 8, 55, 187, 227, 234, 199, 6, 170, 31, 164, 35,
        ],
        [
            127, 229, 190, 196, 138, 191, 99, 200, 2, 94, 33, 220, 198, 94, 247, 56, 92, 235, 5,
            231, 0, 215, 115, 237, 123, 128, 163, 62, 245, 136, 81, 22,
        ],
        [
            227, 192, 203, 190, 191, 238, 65, 46, 225, 246, 98, 119, 248, 140, 53, 18, 29, 161, 15,
            186, 236, 103, 170, 73, 194, 235, 181, 196, 130, 245, 221, 13,
        ],
        [
            64, 205, 137, 125, 6, 95, 1, 144, 47, 13, 117, 175, 2, 2, 233, 104, 21, 195, 159, 219,
            27, 30, 12, 229, 218, 26, 207, 101, 32, 16, 167, 2,
        ],
        [
            34, 165, 74, 35, 215, 50, 17, 254, 127, 186, 142, 239, 125, 1, 238, 118, 184, 35, 133,
            15, 46, 87, 123, 153, 143, 75, 66, 12, 100, 48, 121, 48,
        ],
    ],
    [
        [
            214, 114, 165, 15, 16, 78, 181, 140, 200, 65, 29, 172, 151, 153, 42, 42, 237, 0, 16,
            125, 100, 98, 184, 223, 117, 20, 178, 157, 216, 44, 213, 15,
        ],
        [
            132, 159, 1, 46, 5, 227, 173, 110, 40, 63, 114, 61, 174, 128, 176, 65, 210, 40, 113,
            98, 56, 97, 31, 79, 27, 108, 73, 89, 187, 91, 212, 56,
        ],
        [
            172, 60, 113, 153, 79, 201, 88, 244, 49, 183, 22, 155, 50, 177, 233, 114, 239, 171, 11,
            254, 167, 190, 175, 168, 186, 77, 225, 135, 173, 40, 55, 48,
        ],
        [
            125, 213, 133, 173, 206, 217, 164, 129, 242, 146, 105, 114, 154, 236, 10, 141, 30, 90,
            218, 141, 173, 47, 144, 222, 136, 142, 38, 52, 26, 23, 14, 60,
        ],
        [
            139, 80, 72, 222, 254, 102, 93, 93, 171, 176, 185, 71, 208, 45, 194, 155, 138, 10, 119,
            216, 70, 65, 186, 101, 170, 173, 40, 248, 151, 68, 36, 2,
        ],
        [
            217, 169, 205, 250, 22, 76, 171, 72, 234, 51, 110, 48, 151, 167, 126, 164, 6, 208, 212,
            219, 0, 239, 167, 53, 192, 217, 172, 45, 11, 255, 36, 8,
        ],
        [
            185, 240, 107, 188, 203, 252, 145, 109, 83, 50, 19, 103, 221, 108, 214, 191, 76, 193,
            213, 76, 178, 238, 191, 96, 148, 240, 145, 197, 57, 223, 91, 38,
        ],
        [
            135, 117, 125, 1, 188, 32, 35, 122, 175, 114, 34, 107, 83, 189, 14, 252, 187, 207, 56,
            81, 179, 150, 83, 169, 41, 7, 153, 219, 41, 249, 137, 36,
        ],
    ],
    [
        [
            130, 161, 187, 43, 82, 167, 198, 157, 80, 21, 74, 98, 249, 108, 140, 91, 167, 175, 83,
            166, 160, 83, 226, 11, 105, 178, 43, 93, 152, 98, 39, 60,
        ],
        [
            103, 60, 46, 185, 126, 175, 11, 242, 101, 195, 88, 74, 137, 157, 7, 133, 136, 190, 182,
            15, 95, 81, 82, 60, 238, 203, 94, 197, 58, 235, 78, 26,
        ],
        [
            125, 109, 188, 77, 254, 92, 214, 193, 55, 226, 214, 209, 14, 61, 76, 62, 1, 184, 215,
            234, 137, 221, 36, 165, 14, 51, 144, 85, 103, 230, 139, 54,
        ],
        [
            31, 80, 113, 254, 151, 75, 72, 118, 184, 169, 88, 45, 176, 244, 185, 244, 116, 20, 254,
            25, 46, 209, 230, 101, 108, 212, 145, 109, 19, 203, 52, 23,
        ],
        [
            223, 26, 38, 148, 184, 152, 184, 143, 7, 88, 24, 23, 10, 222, 186, 75, 235, 47, 152,
            166, 150, 194, 251, 172, 244, 48, 213, 91, 185, 90, 135, 6,
        ],
        [
            49, 201, 141, 151, 38, 161, 147, 187, 9, 184, 151, 218, 50, 193, 207, 31, 125, 243, 44,
            73, 133, 109, 219, 122, 48, 232, 143, 23, 243, 253, 230, 4,
        ],
        [
            226, 81, 244, 145, 117, 92, 163, 152, 234, 27, 48, 188, 216, 35, 238, 160, 132, 170,
            116, 231, 72, 202, 3, 185, 230, 77, 245, 56, 30, 214, 138, 48,
        ],
        [
            203, 71, 145, 89, 101, 217, 131, 63, 86, 163, 54, 136, 121, 122, 155, 211, 113, 235,
            21, 121, 67, 53, 217, 167, 76, 158, 51, 115, 145, 91, 31, 17,
        ],
    ],
    [
        [
            163, 118, 62, 49, 37, 114, 17, 221, 161, 123, 183, 175, 99, 19, 182, 11, 127, 236, 183,
            255, 144, 45, 151, 179, 76, 158, 3, 244, 18, 197, 192, 15,
        ],
        [
            161, 140, 144, 70, 95, 191, 18, 250, 95, 202, 117, 12, 81, 145, 232, 37, 148, 192, 220,
            126, 52, 144, 121, 51, 101, 197, 209, 187, 62, 95, 123, 6,
        ],
        [
            218, 235, 88, 200, 137, 249, 88, 23, 9, 21, 41, 221, 173, 222, 183, 202, 244, 51, 59,
            81, 157, 46, 225, 92, 104, 138, 194, 120, 192, 35, 255, 55,
        ],
        [
            183, 236, 50, 252, 158, 165, 145, 185, 126, 97, 212, 66, 97, 197, 185, 125, 85, 60,
            239, 123, 147, 234, 128, 46, 113, 127, 136, 143, 12, 160, 244, 14,
        ],
        [
            229, 188, 4, 248, 56, 209, 163, 222, 126, 202, 169, 221, 208, 239, 56, 204, 25, 214,
            222, 140, 83, 2, 164, 52, 212, 80, 104, 109, 32, 204, 13, 39,
        ],
        [
            124, 94, 218, 55, 16, 89, 156, 202, 79, 167, 11, 70, 95, 50, 198, 187, 51, 105, 69, 76,
            83, 72, 44, 143, 64, 60, 19, 14, 192, 196, 80, 27,
        ],
        [
            97, 209, 31, 19, 192, 81, 91, 247, 188, 17, 0, 21, 75, 229, 246, 158, 136, 76, 210,
            207, 249, 143, 152, 221, 229, 242, 59, 114, 206, 95, 96, 47,
        ],
        [
            11, 206, 70, 250, 78, 150, 63, 79, 249, 28, 67, 124, 107, 222, 13, 23, 17, 68, 53, 122,
            184, 139, 37, 94, 6, 39, 35, 160, 66, 221, 110, 57,
        ],
    ],
    [
        [
            223, 134, 239, 176, 66, 205, 170, 212, 138, 20, 104, 236, 148, 255, 100, 186, 129, 240,
            197, 226, 26, 15, 254, 102, 80, 192, 154, 239, 139, 23, 121, 63,
        ],
        [
            5, 119, 208, 59, 122, 3, 98, 77, 4, 190, 57, 200, 218, 45, 203, 72, 213, 82, 131, 107,
            169, 215, 115, 204, 1, 224, 230, 159, 198, 109, 107, 54,
        ],
        [
            104, 158, 197, 37, 221, 147, 75, 28, 103, 145, 43, 10, 92, 236, 156, 128, 161, 166, 38,
            21, 61, 23, 31, 28, 253, 61, 17, 246, 189, 252, 175, 43,
        ],
        [
            24, 105, 76, 61, 63, 75, 50, 76, 51, 192, 208, 63, 88, 229, 128, 119, 220, 237, 199,
            84, 244, 89, 108, 230, 177, 251, 241, 178, 198, 176, 83, 21,
        ],
        [
            95, 224, 156, 151, 85, 5, 100, 100, 241, 158, 113, 178, 102, 190, 76, 193, 236, 80,
            137, 95, 138, 94, 146, 3, 76, 157, 86, 102, 237, 20, 176, 63,
        ],
        [
            135, 94, 202, 65, 67, 250, 57, 207, 182, 240, 92, 3, 232, 237, 207, 226, 96, 251, 5,
            14, 56, 84, 221, 45, 50, 21, 23, 38, 125, 141, 125, 33,
        ],
        [
            1, 250, 154, 59, 193, 213, 14, 252, 103, 49, 176, 144, 111, 21, 26, 187, 8, 115, 236,
            31, 221, 198, 37, 92, 194, 32, 42, 236, 140, 108, 35, 39,
        ],
        [
            103, 78, 38, 12, 33, 172, 47, 58, 164, 230, 77, 235, 219, 159, 138, 135, 17, 66, 215,
            244, 183, 189, 78, 154, 114, 85, 131, 71, 221, 132, 206, 51,
        ],
    ],
    [
        [
            74, 237, 137, 138, 7, 144, 141, 134, 245, 201, 75, 46, 40, 46, 38, 183, 163, 73, 201,
            165, 151, 236, 247, 214, 6, 44, 36, 27, 225, 75, 97, 48,
        ],
        [
            38, 39, 238, 243, 197, 119, 161, 224, 249, 104, 44, 35, 221, 140, 86, 236, 9, 96, 12,
            2, 73, 135, 165, 30, 67, 208, 15, 210, 52, 18, 118, 6,
        ],
        [
            143, 224, 93, 228, 86, 241, 188, 45, 229, 9, 195, 9, 209, 118, 156, 40, 7, 200, 138,
            200, 192, 83, 107, 147, 195, 86, 84, 8, 93, 62, 87, 54,
        ],
        [
            212, 96, 74, 248, 166, 217, 63, 214, 37, 93, 202, 74, 239, 98, 42, 253, 228, 212, 156,
            203, 141, 14, 197, 19, 234, 148, 17, 40, 141, 236, 35, 49,
        ],
        [
            173, 148, 235, 124, 235, 146, 153, 90, 15, 115, 1, 233, 109, 117, 216, 239, 79, 144,
            231, 37, 188, 189, 78, 29, 133, 26, 176, 109, 177, 234, 202, 59,
        ],
        [
            10, 38, 232, 241, 157, 38, 214, 76, 198, 109, 162, 131, 127, 3, 40, 15, 48, 195, 47,
            211, 52, 8, 0, 189, 126, 120, 13, 42, 238, 226, 225, 3,
        ],
        [
            243, 73, 44, 63, 255, 180, 74, 70, 135, 143, 219, 51, 6, 159, 80, 154, 210, 51, 241,
            230, 40, 249, 80, 106, 102, 166, 7, 155, 194, 38, 33, 17,
        ],
        [
            191, 105, 232, 173, 147, 80, 10, 21, 203, 53, 211, 74, 36, 15, 3, 188, 128, 47, 9, 186,
            83, 202, 36, 81, 138, 63, 148, 167, 137, 35, 147, 19,
        ],
    ],
    [
        [
            236, 196, 10, 96, 182, 79, 154, 55, 78, 33, 43, 127, 92, 81, 212, 66, 30, 0, 86, 117,
            207, 159, 3, 186, 59, 148, 154, 200, 141, 92, 147, 27,
        ],
        [
            225, 90, 178, 233, 102, 159, 82, 225, 166, 129, 39, 34, 244, 250, 228, 100, 134, 200,
            148, 85, 91, 43, 105, 75, 200, 87, 81, 59, 40, 193, 211, 55,
        ],
        [
            59, 148, 63, 224, 145, 150, 218, 137, 164, 83, 27, 147, 217, 75, 108, 182, 36, 204,
            121, 133, 12, 8, 235, 64, 128, 244, 34, 13, 9, 72, 63, 28,
        ],
        [
            212, 212, 49, 80, 78, 45, 56, 203, 40, 249, 128, 137, 185, 87, 92, 47, 146, 177, 132,
            62, 215, 25, 127, 115, 205, 101, 6, 154, 76, 80, 114, 18,
        ],
        [
            176, 192, 244, 210, 165, 138, 232, 75, 159, 72, 99, 239, 121, 37, 50, 114, 181, 220,
            176, 22, 164, 127, 221, 10, 193, 99, 140, 15, 204, 86, 212, 53,
        ],
        [
            159, 147, 161, 250, 222, 214, 217, 27, 87, 5, 8, 226, 156, 162, 178, 255, 170, 187,
            200, 161, 26, 195, 99, 180, 246, 102, 210, 123, 107, 242, 173, 35,
        ],
        [
            147, 2, 204, 135, 133, 234, 183, 133, 186, 151, 248, 180, 75, 13, 40, 56, 145, 134,
            209, 237, 43, 6, 107, 52, 33, 206, 165, 226, 205, 75, 28, 21,
        ],
        [
            30, 67, 172, 233, 205, 173, 245, 113, 133, 159, 28, 77, 133, 209, 85, 173, 163, 43,
            138, 208, 96, 215, 181, 142, 133, 151, 156, 36, 126, 32, 68, 7,
        ],
    ],
    [
        [
            12, 123, 146, 187, 79, 71, 121, 126, 110, 67, 163, 209, 185, 8, 43, 250, 139, 95, 132,
            9, 141, 123, 120, 25, 113, 59, 253, 85, 72, 170, 199, 27,
        ],
        [
            148, 38, 17, 234, 63, 253, 30, 242, 241, 113, 82, 233, 175, 17, 227, 75, 172, 25, 1,
            230, 87, 107, 49, 184, 36, 27, 255, 135, 253, 15, 42, 18,
        ],
        [
            57, 122, 227, 248, 85, 40, 238, 252, 56, 46, 59, 209, 214, 24, 129, 169, 59, 35, 169,
            94, 189, 126, 146, 19, 162, 186, 148, 249, 170, 190, 218, 36,
        ],
        [
            72, 154, 129, 64, 173, 23, 154, 53, 215, 48, 178, 84, 152, 160, 105, 56, 49, 219, 11,
            6, 196, 221, 55, 85, 72, 129, 149, 148, 178, 165, 20, 61,
        ],
        [
            211, 89, 31, 7, 22, 111, 165, 192, 156, 164, 251, 69, 251, 72, 208, 157, 53, 57, 144,
            72, 12, 146, 11, 14, 145, 181, 14, 68, 237, 120, 174, 4,
        ],
        [
            18, 115, 60, 134, 217, 114, 253, 177, 223, 163, 10, 83, 103, 37, 178, 69, 3, 78, 164,
            2, 35, 65, 2, 219, 93, 141, 186, 80, 20, 166, 103, 43,
        ],
        [
            110, 16, 223, 29, 240, 124, 98, 122, 115, 98, 200, 21, 240, 17, 173, 209, 114, 172,
            198, 226, 17, 201, 34, 247, 39, 15, 246, 75, 212, 48, 39, 11,
        ],
        [
            249, 114, 161, 95, 41, 76, 181, 17, 119, 208, 229, 87, 84, 28, 170, 246, 144, 202, 188,
            5, 49, 53, 77, 232, 8, 6, 247, 81, 111, 232, 35, 57,
        ],
    ],
    [
        [
            166, 191, 70, 138, 155, 151, 236, 84, 96, 249, 99, 236, 45, 137, 97, 90, 167, 20, 174,
            149, 119, 233, 203, 65, 37, 72, 157, 86, 249, 111, 56, 60,
        ],
        [
            43, 38, 90, 143, 119, 212, 89, 41, 208, 243, 146, 176, 197, 213, 11, 165, 57, 6, 3,
            139, 45, 156, 142, 42, 177, 203, 207, 69, 121, 180, 159, 39,
        ],
        [
            55, 217, 98, 174, 52, 172, 195, 38, 151, 240, 237, 116, 26, 43, 119, 0, 126, 67, 123,
            173, 195, 172, 254, 130, 16, 111, 177, 190, 128, 161, 192, 9,
        ],
        [
            73, 205, 70, 151, 40, 112, 216, 117, 3, 99, 187, 10, 34, 53, 109, 94, 0, 225, 52, 200,
            127, 182, 30, 45, 188, 66, 50, 192, 179, 24, 235, 42,
        ],
        [
            8, 249, 190, 160, 146, 39, 217, 65, 210, 62, 89, 36, 4, 149, 125, 101, 32, 202, 88, 37,
            16, 29, 219, 53, 204, 32, 1, 228, 3, 121, 6, 21,
        ],
        [
            205, 41, 152, 28, 50, 10, 231, 209, 244, 103, 159, 116, 223, 92, 92, 246, 192, 171, 38,
            66, 116, 104, 90, 75, 238, 94, 179, 10, 71, 73, 189, 57,
        ],
        [
            28, 180, 207, 160, 147, 83, 83, 148, 184, 113, 208, 92, 58, 143, 129, 211, 108, 26, 86,
            119, 117, 184, 154, 40, 181, 24, 6, 90, 11, 150, 28, 38,
        ],
        [
            144, 187, 91, 40, 224, 180, 119, 167, 173, 26, 227, 221, 172, 189, 180, 197, 52, 233,
            39, 112, 179, 40, 22, 182, 41, 225, 104, 21, 8, 96, 166, 51,
        ],
    ],
    [
        [
            118, 105, 175, 214, 145, 170, 223, 7, 51, 179, 133, 78, 187, 138, 233, 70, 53, 182,
            238, 228, 35, 55, 107, 18, 52, 34, 86, 113, 121, 120, 97, 8,
        ],
        [
            136, 52, 133, 99, 235, 32, 133, 156, 69, 187, 203, 147, 116, 160, 103, 12, 154, 202,
            135, 55, 163, 214, 96, 56, 11, 53, 23, 149, 77, 160, 189, 36,
        ],
        [
            46, 168, 103, 19, 16, 88, 71, 160, 190, 241, 72, 249, 169, 10, 141, 151, 42, 14, 129,
            36, 184, 162, 230, 186, 51, 234, 169, 202, 221, 129, 188, 47,
        ],
        [
            94, 22, 119, 86, 162, 42, 146, 178, 116, 23, 102, 176, 181, 124, 222, 250, 98, 251,
            121, 116, 211, 152, 216, 137, 68, 158, 133, 226, 124, 183, 135, 22,
        ],
        [
            247, 213, 253, 212, 198, 6, 78, 30, 223, 210, 220, 205, 100, 129, 193, 27, 248, 74, 60,
            30, 35, 4, 143, 42, 249, 183, 78, 209, 154, 199, 38, 53,
        ],
        [
            226, 200, 233, 87, 125, 230, 129, 209, 4, 2, 105, 213, 112, 223, 41, 44, 9, 120, 65,
            10, 131, 60, 123, 29, 20, 11, 33, 217, 153, 65, 91, 39,
        ],
        [
            155, 153, 145, 173, 223, 141, 69, 150, 77, 108, 185, 23, 158, 178, 160, 123, 90, 218,
            166, 40, 80, 206, 252, 227, 104, 248, 194, 61, 88, 204, 154, 8,
        ],
        [
            142, 151, 153, 200, 56, 160, 131, 14, 164, 106, 30, 83, 29, 118, 48, 143, 222, 57, 93,
            202, 102, 18, 253, 130, 35, 72, 38, 19, 42, 58, 106, 41,
        ],
    ],
    [
        [
            23, 79, 174, 233, 153, 156, 213, 212, 214, 92, 159, 141, 29, 229, 223, 133, 240, 209,
            79, 184, 66, 96, 27, 101, 214, 54, 86, 241, 99, 56, 51, 46,
        ],
        [
            119, 55, 21, 173, 29, 13, 198, 88, 92, 46, 235, 147, 188, 255, 90, 77, 8, 136, 101,
            250, 53, 209, 119, 162, 7, 76, 17, 162, 29, 199, 118, 39,
        ],
        [
            8, 253, 207, 82, 23, 175, 92, 88, 155, 246, 195, 146, 200, 200, 105, 255, 61, 115, 151,
            166, 124, 64, 220, 17, 100, 86, 20, 203, 125, 3, 221, 36,
        ],
        [
            174, 75, 215, 152, 74, 10, 18, 7, 252, 162, 23, 11, 68, 157, 89, 226, 95, 128, 7, 78,
            23, 165, 37, 157, 141, 175, 117, 23, 234, 123, 72, 42,
        ],
        [
            133, 181, 221, 30, 165, 125, 199, 148, 83, 133, 250, 164, 209, 152, 68, 72, 34, 196,
            80, 12, 224, 123, 47, 42, 221, 191, 74, 161, 197, 203, 235, 32,
        ],
        [
            27, 29, 142, 238, 251, 254, 200, 233, 42, 77, 207, 93, 95, 197, 163, 242, 93, 134, 230,
            18, 210, 156, 93, 167, 113, 16, 49, 46, 123, 218, 178, 61,
        ],
        [
            118, 213, 164, 221, 213, 78, 38, 138, 116, 99, 222, 183, 249, 126, 137, 103, 13, 10,
            153, 232, 108, 250, 220, 187, 195, 166, 238, 223, 42, 124, 91, 10,
        ],
        [
            25, 143, 13, 156, 46, 146, 125, 162, 179, 244, 52, 49, 47, 142, 84, 241, 15, 96, 35,
            122, 26, 175, 155, 30, 133, 224, 76, 77, 136, 128, 198, 6,
        ],
    ],
    [
        [
            228, 4, 100, 254, 64, 122, 195, 101, 45, 240, 33, 189, 172, 246, 68, 35, 204, 229, 42,
            174, 120, 197, 247, 7, 61, 33, 16, 165, 182, 175, 213, 43,
        ],
        [
            95, 82, 224, 168, 92, 69, 52, 224, 169, 53, 233, 52, 82, 158, 169, 80, 71, 142, 135,
            177, 107, 103, 114, 1, 245, 144, 160, 123, 8, 185, 118, 12,
        ],
        [
            67, 145, 142, 236, 153, 196, 179, 180, 101, 10, 152, 214, 130, 7, 187, 152, 147, 177,
            234, 96, 10, 16, 126, 208, 89, 163, 239, 38, 237, 148, 11, 29,
        ],
        [
            10, 227, 59, 211, 209, 116, 53, 110, 240, 229, 85, 169, 227, 37, 187, 141, 76, 185,
            215, 142, 202, 158, 100, 104, 52, 107, 179, 66, 237, 8, 203, 51,
        ],
        [
            158, 60, 113, 87, 219, 161, 5, 234, 31, 203, 166, 8, 226, 230, 59, 175, 238, 85, 94,
            39, 197, 52, 79, 172, 93, 55, 160, 241, 119, 43, 48, 27,
        ],
        [
            214, 132, 49, 107, 226, 92, 16, 179, 190, 128, 224, 62, 190, 5, 135, 116, 189, 57, 80,
            246, 109, 33, 9, 96, 231, 206, 202, 64, 169, 175, 222, 2,
        ],
        [
            211, 188, 126, 110, 204, 200, 213, 50, 216, 49, 204, 224, 253, 245, 57, 139, 185, 75,
            226, 85, 128, 149, 32, 213, 127, 7, 127, 106, 139, 76, 93, 42,
        ],
        [
            38, 227, 189, 95, 248, 213, 125, 206, 75, 148, 138, 153, 134, 92, 243, 67, 101, 42, 86,
            23, 28, 105, 5, 30, 72, 228, 21, 251, 56, 15, 104, 44,
        ],
    ],
    [
        [
            37, 51, 111, 189, 64, 200, 241, 35, 170, 63, 123, 104, 60, 201, 92, 104, 78, 13, 168,
            90, 133, 18, 74, 125, 137, 233, 244, 183, 194, 166, 136, 4,
        ],
        [
            135, 115, 58, 240, 172, 155, 138, 72, 201, 245, 165, 132, 213, 230, 175, 122, 80, 137,
            12, 202, 51, 147, 254, 83, 83, 12, 24, 196, 127, 65, 29, 9,
        ],
        [
            102, 193, 159, 222, 145, 157, 184, 248, 13, 50, 103, 129, 22, 72, 211, 169, 252, 59,
            21, 179, 8, 224, 191, 155, 105, 80, 205, 20, 126, 53, 167, 51,
        ],
        [
            248, 10, 36, 176, 206, 117, 68, 20, 29, 159, 234, 75, 123, 229, 186, 211, 197, 38, 74,
            210, 209, 247, 75, 138, 105, 11, 213, 184, 88, 146, 169, 63,
        ],
        [
            72, 92, 136, 57, 104, 244, 192, 189, 92, 229, 248, 238, 206, 124, 141, 216, 65, 129,
            74, 239, 127, 25, 111, 173, 5, 103, 127, 62, 158, 173, 150, 14,
        ],
        [
            137, 66, 178, 24, 248, 195, 72, 192, 193, 118, 248, 34, 201, 117, 22, 68, 132, 3, 194,
            91, 96, 246, 119, 95, 73, 112, 14, 241, 140, 190, 209, 1,
        ],
        [
            189, 118, 10, 191, 62, 184, 239, 63, 140, 57, 145, 76, 51, 251, 162, 162, 37, 175, 50,
            68, 158, 17, 225, 79, 180, 153, 166, 114, 251, 155, 69, 33,
        ],
        [
            59, 192, 204, 241, 201, 5, 189, 79, 249, 152, 117, 146, 123, 43, 151, 111, 131, 23,
            129, 192, 167, 112, 132, 40, 171, 94, 90, 212, 100, 52, 48, 28,
        ],
    ],
    [
        [
            160, 245, 228, 171, 90, 165, 111, 25, 114, 50, 116, 39, 201, 216, 29, 129, 112, 207,
            152, 165, 48, 118, 172, 101, 106, 76, 4, 16, 248, 172, 170, 39,
        ],
        [
            60, 126, 38, 248, 48, 18, 168, 77, 10, 58, 62, 36, 45, 84, 12, 6, 72, 185, 147, 255,
            88, 135, 221, 226, 229, 114, 39, 186, 151, 16, 175, 17,
        ],
        [
            62, 55, 177, 217, 205, 252, 191, 205, 39, 137, 103, 195, 252, 240, 216, 28, 24, 138,
            162, 54, 48, 159, 17, 103, 110, 255, 103, 3, 150, 20, 31, 15,
        ],
        [
            120, 251, 98, 40, 80, 237, 108, 131, 14, 65, 116, 12, 242, 115, 26, 181, 106, 240, 90,
            13, 158, 21, 68, 3, 88, 14, 59, 191, 63, 253, 247, 19,
        ],
        [
            218, 240, 72, 194, 44, 92, 246, 87, 69, 50, 34, 84, 102, 185, 46, 200, 68, 10, 145, 56,
            10, 240, 100, 73, 125, 33, 207, 254, 198, 215, 119, 23,
        ],
        [
            166, 98, 74, 208, 109, 114, 207, 207, 253, 137, 146, 124, 252, 37, 92, 142, 72, 195,
            79, 128, 83, 3, 62, 182, 252, 140, 69, 112, 81, 35, 14, 63,
        ],
        [
            216, 80, 1, 36, 96, 224, 105, 59, 214, 215, 53, 144, 31, 78, 236, 102, 94, 23, 217,
            195, 69, 74, 56, 14, 28, 241, 94, 33, 216, 166, 153, 47,
        ],
        [
            239, 70, 89, 170, 122, 62, 9, 125, 238, 216, 0, 168, 95, 170, 65, 156, 31, 4, 6, 163,
            64, 110, 235, 190, 227, 59, 9, 254, 83, 42, 179, 19,
        ],
    ],
    [
        [
            231, 7, 127, 64, 136, 255, 178, 190, 36, 15, 199, 117, 145, 167, 218, 125, 217, 43,
            195, 87, 29, 177, 173, 97, 128, 42, 111, 135, 122, 245, 128, 55,
        ],
        [
            35, 175, 64, 138, 189, 200, 135, 138, 203, 147, 145, 199, 126, 81, 114, 39, 30, 64, 73,
            90, 94, 109, 230, 69, 49, 71, 190, 180, 50, 10, 203, 20,
        ],
        [
            110, 75, 166, 243, 255, 128, 242, 208, 67, 78, 166, 29, 124, 54, 56, 160, 106, 113,
            190, 155, 58, 241, 21, 161, 93, 198, 173, 46, 251, 76, 109, 34,
        ],
        [
            18, 157, 21, 73, 163, 78, 187, 222, 155, 123, 56, 158, 35, 155, 108, 36, 154, 199, 10,
            39, 245, 212, 70, 250, 249, 38, 212, 21, 226, 178, 136, 44,
        ],
        [
            135, 162, 176, 158, 247, 168, 215, 162, 0, 67, 153, 248, 4, 101, 176, 233, 224, 73,
            213, 227, 132, 34, 140, 44, 207, 79, 194, 198, 151, 218, 52, 50,
        ],
        [
            201, 18, 231, 111, 222, 85, 53, 39, 138, 60, 182, 53, 132, 16, 164, 42, 57, 56, 88,
            234, 13, 162, 66, 189, 142, 130, 211, 204, 252, 71, 47, 16,
        ],
        [
            91, 194, 190, 127, 41, 158, 183, 169, 31, 35, 96, 249, 162, 84, 132, 230, 161, 243,
            170, 226, 210, 24, 14, 186, 83, 31, 210, 85, 195, 185, 24, 20,
        ],
        [
            217, 59, 106, 21, 76, 11, 79, 85, 36, 138, 162, 208, 13, 136, 63, 206, 136, 158, 34,
            207, 169, 168, 252, 135, 67, 172, 77, 234, 167, 29, 87, 45,
        ],
    ],
    [
        [
            228, 252, 249, 53, 147, 111, 53, 172, 223, 102, 206, 74, 150, 3, 55, 233, 88, 23, 159,
            60, 57, 46, 186, 171, 76, 170, 249, 69, 244, 79, 192, 30,
        ],
        [
            128, 217, 236, 6, 89, 109, 19, 109, 108, 57, 35, 181, 116, 110, 215, 182, 43, 137, 252,
            11, 156, 212, 132, 209, 86, 87, 8, 233, 186, 24, 134, 41,
        ],
        [
            159, 52, 52, 246, 254, 33, 247, 217, 161, 118, 36, 12, 186, 90, 199, 184, 112, 54, 159,
            71, 21, 210, 38, 191, 179, 162, 228, 205, 173, 111, 182, 43,
        ],
        [
            19, 68, 116, 77, 123, 197, 44, 57, 140, 130, 231, 218, 37, 95, 228, 61, 81, 117, 38,
            22, 118, 84, 173, 119, 161, 183, 187, 24, 201, 83, 166, 40,
        ],
        [
            203, 74, 138, 253, 8, 155, 80, 99, 82, 194, 145, 185, 32, 193, 80, 85, 61, 166, 71, 92,
            106, 41, 55, 60, 40, 237, 54, 145, 112, 90, 104, 41,
        ],
        [
            152, 100, 177, 4, 228, 177, 130, 52, 85, 136, 118, 163, 98, 102, 225, 225, 113, 159,
            224, 37, 25, 80, 112, 245, 96, 183, 221, 146, 174, 25, 48, 41,
        ],
        [
            124, 166, 27, 180, 114, 17, 98, 30, 99, 232, 181, 40, 65, 191, 114, 222, 48, 209, 53,
            93, 98, 38, 42, 146, 68, 143, 229, 208, 46, 178, 238, 62,
        ],
        [
            238, 242, 49, 191, 50, 111, 197, 182, 193, 77, 103, 223, 113, 224, 125, 195, 23, 135,
            50, 145, 77, 171, 82, 132, 173, 196, 80, 142, 107, 157, 142, 29,
        ],
    ],
    [
        [
            82, 195, 210, 133, 249, 13, 111, 212, 92, 164, 28, 91, 160, 60, 34, 35, 243, 234, 176,
            255, 53, 140, 151, 76, 109, 87, 182, 16, 14, 250, 248, 2,
        ],
        [
            130, 201, 107, 39, 163, 233, 135, 216, 65, 13, 199, 247, 155, 234, 100, 133, 254, 230,
            170, 109, 170, 222, 252, 40, 68, 48, 107, 211, 83, 112, 240, 60,
        ],
        [
            30, 28, 129, 71, 31, 137, 206, 248, 195, 232, 69, 203, 96, 43, 67, 68, 33, 187, 167,
            235, 204, 62, 103, 193, 206, 147, 89, 203, 190, 181, 114, 10,
        ],
        [
            127, 36, 5, 200, 56, 245, 144, 50, 154, 33, 53, 7, 127, 232, 151, 11, 124, 180, 160,
            89, 172, 138, 214, 238, 131, 54, 191, 199, 206, 111, 61, 23,
        ],
        [
            146, 197, 178, 24, 254, 19, 211, 232, 156, 144, 13, 104, 87, 52, 25, 6, 187, 206, 19,
            91, 43, 70, 93, 18, 227, 52, 139, 210, 183, 88, 127, 36,
        ],
        [
            113, 170, 45, 153, 118, 29, 249, 241, 96, 85, 78, 221, 223, 55, 10, 199, 137, 15, 230,
            113, 214, 92, 169, 32, 58, 251, 62, 186, 233, 37, 32, 1,
        ],
        [
            47, 80, 145, 109, 177, 172, 202, 217, 197, 228, 249, 136, 223, 48, 136, 55, 244, 227,
            206, 93, 78, 84, 183, 243, 203, 98, 169, 196, 103, 39, 76, 48,
        ],
        [
            32, 251, 111, 146, 107, 57, 148, 119, 188, 206, 74, 49, 89, 16, 106, 149, 111, 90, 210,
            38, 142, 50, 196, 143, 191, 133, 221, 123, 219, 205, 157, 15,
        ],
    ],
    [
        [
            124, 193, 251, 119, 28, 89, 69, 168, 245, 164, 145, 5, 34, 127, 237, 40, 9, 194, 222,
            92, 5, 254, 145, 164, 238, 22, 168, 97, 39, 63, 63, 48,
        ],
        [
            128, 14, 241, 12, 198, 51, 53, 31, 29, 56, 116, 167, 33, 172, 77, 124, 84, 230, 106,
            120, 94, 231, 26, 253, 29, 246, 251, 225, 144, 208, 93, 41,
        ],
        [
            145, 233, 236, 72, 57, 76, 123, 17, 55, 100, 127, 228, 145, 32, 165, 62, 23, 123, 121,
            2, 13, 178, 199, 180, 214, 163, 228, 157, 63, 135, 187, 30,
        ],
        [
            76, 75, 200, 240, 37, 16, 184, 100, 38, 65, 206, 138, 92, 149, 194, 84, 241, 42, 194,
            161, 149, 93, 182, 230, 110, 171, 51, 65, 219, 158, 252, 19,
        ],
        [
            169, 156, 202, 117, 173, 222, 62, 242, 203, 66, 26, 216, 9, 92, 112, 184, 46, 165, 52,
            160, 114, 38, 68, 43, 148, 62, 30, 121, 26, 192, 152, 0,
        ],
        [
            38, 77, 105, 144, 232, 160, 9, 77, 14, 96, 247, 59, 109, 53, 101, 114, 55, 121, 243,
            111, 90, 240, 61, 220, 55, 0, 114, 64, 49, 194, 37, 63,
        ],
        [
            94, 199, 2, 248, 34, 232, 68, 130, 1, 237, 3, 73, 251, 153, 188, 183, 202, 133, 201,
            127, 244, 156, 184, 58, 22, 53, 115, 184, 107, 251, 115, 41,
        ],
        [
            44, 105, 2, 121, 133, 252, 4, 51, 135, 185, 18, 240, 184, 64, 39, 128, 126, 216, 168,
            25, 54, 230, 96, 205, 156, 5, 228, 79, 121, 245, 0, 37,
        ],
    ],
    [
        [
            26, 227, 128, 219, 221, 39, 112, 88, 84, 130, 216, 87, 157, 119, 89, 202, 85, 179, 237,
            107, 51, 211, 185, 30, 143, 196, 239, 4, 187, 89, 157, 46,
        ],
        [
            11, 54, 239, 254, 23, 23, 81, 152, 249, 219, 38, 144, 14, 66, 140, 19, 125, 160, 65,
            238, 136, 168, 185, 167, 131, 60, 217, 246, 38, 125, 58, 35,
        ],
        [
            157, 212, 24, 185, 100, 207, 250, 73, 82, 53, 146, 208, 36, 19, 63, 130, 58, 99, 61,
            79, 153, 247, 2, 65, 234, 255, 31, 19, 13, 168, 227, 45,
        ],
        [
            36, 219, 99, 139, 180, 218, 207, 143, 166, 191, 127, 239, 60, 76, 145, 170, 56, 128,
            112, 71, 49, 92, 121, 144, 212, 14, 21, 80, 66, 58, 141, 30,
        ],
        [
            36, 182, 110, 54, 159, 120, 85, 112, 6, 158, 77, 194, 157, 33, 36, 82, 2, 190, 84, 155,
            101, 244, 186, 202, 94, 98, 115, 72, 231, 202, 129, 55,
        ],
        [
            39, 152, 205, 65, 57, 155, 123, 160, 72, 205, 121, 246, 119, 90, 39, 77, 49, 95, 30,
            159, 243, 241, 69, 154, 109, 150, 235, 177, 140, 158, 197, 50,
        ],
        [
            101, 168, 129, 120, 249, 152, 206, 97, 236, 88, 65, 123, 54, 174, 153, 187, 243, 114,
            75, 24, 105, 33, 133, 230, 209, 159, 244, 92, 139, 79, 29, 60,
        ],
        [
            159, 56, 171, 163, 194, 74, 206, 171, 192, 125, 50, 168, 220, 194, 217, 246, 53, 51,
            143, 119, 97, 16, 102, 154, 202, 171, 112, 0, 9, 76, 52, 33,
        ],
    ],
    [
        [
            109, 67, 166, 112, 18, 88, 181, 61, 137, 12, 191, 223, 140, 243, 174, 222, 244, 102,
            170, 101, 139, 159, 219, 120, 143, 31, 69, 70, 172, 184, 26, 47,
        ],
        [
            49, 226, 39, 237, 179, 21, 55, 83, 139, 192, 65, 216, 225, 192, 239, 65, 235, 37, 109,
            139, 101, 80, 151, 15, 183, 1, 40, 226, 236, 136, 173, 62,
        ],
        [
            132, 87, 121, 118, 184, 56, 103, 98, 191, 218, 47, 214, 204, 128, 57, 24, 141, 86, 189,
            135, 180, 249, 214, 40, 206, 182, 57, 238, 90, 133, 168, 5,
        ],
        [
            60, 151, 209, 75, 237, 97, 152, 67, 235, 169, 140, 65, 35, 251, 107, 207, 254, 72, 81,
            3, 244, 168, 187, 246, 125, 37, 167, 104, 81, 165, 105, 34,
        ],
        [
            231, 77, 104, 60, 44, 124, 228, 21, 207, 19, 248, 89, 180, 91, 64, 64, 251, 244, 223,
            207, 76, 3, 207, 142, 92, 16, 177, 109, 3, 141, 225, 16,
        ],
        [
            229, 156, 173, 75, 130, 73, 204, 213, 61, 179, 238, 170, 15, 77, 113, 178, 84, 151,
            137, 159, 149, 224, 204, 243, 232, 113, 208, 238, 82, 107, 180, 32,
        ],
        [
            136, 153, 152, 220, 13, 5, 102, 224, 162, 188, 90, 143, 119, 179, 243, 104, 91, 134,
            56, 22, 5, 152, 153, 17, 234, 167, 107, 234, 16, 66, 85, 34,
        ],
        [
            130, 45, 164, 141, 146, 197, 199, 12, 51, 82, 156, 32, 239, 243, 29, 170, 63, 17, 153,
            176, 150, 114, 38, 219, 27, 107, 187, 72, 32, 106, 68, 20,
        ],
    ],
    [
        [
            51, 54, 182, 167, 147, 255, 80, 103, 13, 213, 237, 114, 233, 242, 173, 177, 188, 54, 5,
            143, 165, 220, 114, 249, 175, 171, 173, 188, 4, 162, 62, 26,
        ],
        [
            241, 98, 202, 23, 140, 169, 52, 94, 94, 232, 69, 90, 53, 215, 112, 146, 164, 223, 204,
            33, 208, 138, 238, 181, 48, 56, 243, 170, 171, 35, 126, 48,
        ],
        [
            56, 210, 32, 249, 23, 217, 198, 162, 239, 118, 198, 97, 177, 72, 116, 153, 131, 248,
            194, 171, 180, 33, 195, 66, 159, 125, 110, 230, 56, 89, 15, 9,
        ],
        [
            5, 133, 98, 216, 11, 182, 6, 102, 218, 117, 248, 20, 242, 170, 0, 183, 178, 30, 67,
            211, 90, 255, 24, 70, 135, 65, 114, 85, 238, 29, 70, 40,
        ],
        [
            102, 170, 201, 230, 61, 107, 34, 64, 162, 224, 15, 128, 93, 20, 103, 143, 19, 116, 25,
            123, 93, 16, 154, 219, 136, 216, 69, 185, 35, 126, 242, 30,
        ],
        [
            204, 190, 215, 249, 104, 142, 254, 130, 121, 238, 163, 234, 63, 91, 230, 251, 121, 201,
            127, 69, 0, 121, 61, 203, 137, 2, 77, 227, 203, 127, 89, 44,
        ],
        [
            159, 236, 163, 243, 192, 6, 217, 134, 210, 91, 90, 137, 46, 126, 9, 76, 108, 26, 31,
            241, 181, 180, 29, 63, 62, 119, 7, 143, 166, 173, 8, 6,
        ],
        [
            8, 238, 253, 27, 28, 102, 219, 25, 139, 220, 55, 32, 203, 141, 243, 93, 224, 72, 195,
            75, 45, 78, 151, 142, 71, 137, 38, 141, 205, 102, 85, 35,
        ],
    ],
    [
        [
            98, 124, 121, 75, 138, 47, 120, 198, 57, 252, 117, 98, 53, 109, 168, 113, 235, 254,
            170, 23, 167, 88, 117, 230, 246, 7, 189, 223, 21, 12, 28, 43,
        ],
        [
            172, 101, 158, 162, 23, 10, 251, 252, 92, 152, 13, 48, 159, 45, 112, 195, 149, 53, 198,
            137, 77, 177, 135, 21, 86, 161, 36, 67, 10, 84, 115, 59,
        ],
        [
            89, 60, 73, 153, 180, 157, 56, 238, 205, 197, 102, 16, 177, 170, 181, 232, 46, 160,
            135, 229, 62, 155, 44, 183, 1, 70, 211, 211, 124, 120, 222, 26,
        ],
        [
            126, 135, 102, 76, 24, 172, 65, 102, 115, 181, 92, 106, 34, 101, 203, 56, 249, 148, 93,
            125, 212, 127, 82, 177, 180, 49, 119, 240, 169, 234, 9, 49,
        ],
        [
            71, 143, 202, 34, 243, 214, 106, 210, 230, 70, 197, 6, 173, 230, 236, 14, 39, 19, 42,
            121, 111, 148, 97, 89, 223, 152, 217, 77, 66, 193, 126, 54,
        ],
        [
            81, 149, 56, 199, 215, 100, 87, 71, 249, 207, 114, 45, 137, 99, 209, 60, 26, 92, 202,
            131, 184, 151, 170, 22, 219, 93, 71, 234, 107, 142, 232, 59,
        ],
        [
            17, 166, 47, 181, 47, 121, 77, 103, 36, 162, 150, 155, 54, 127, 29, 40, 219, 189, 57,
            150, 76, 147, 110, 67, 241, 212, 104, 10, 154, 50, 48, 17,
        ],
        [
            20, 246, 109, 111, 72, 251, 94, 213, 187, 51, 196, 220, 212, 10, 167, 224, 102, 27,
            115, 235, 106, 242, 214, 243, 148, 66, 111, 9, 221, 230, 224, 13,
        ],
    ],
    [
        [
            158, 218, 203, 21, 240, 183, 105, 204, 138, 99, 207, 223, 104, 252, 85, 27, 195, 215,
            225, 142, 69, 132, 177, 232, 49, 174, 246, 78, 103, 125, 17, 29,
        ],
        [
            228, 101, 236, 222, 176, 167, 152, 154, 183, 104, 237, 123, 248, 121, 232, 69, 215,
            181, 73, 228, 140, 179, 94, 40, 67, 69, 67, 102, 154, 223, 122, 51,
        ],
        [
            248, 30, 67, 220, 168, 33, 81, 46, 96, 22, 94, 129, 31, 130, 56, 223, 184, 1, 198, 98,
            212, 233, 165, 210, 145, 4, 100, 44, 234, 75, 14, 6,
        ],
        [
            176, 203, 91, 145, 241, 154, 103, 144, 187, 208, 156, 255, 219, 145, 244, 86, 128, 136,
            123, 221, 50, 148, 156, 138, 187, 15, 153, 186, 86, 77, 138, 8,
        ],
        [
            180, 19, 118, 126, 101, 180, 248, 122, 194, 255, 63, 91, 186, 33, 147, 163, 36, 254,
            211, 96, 175, 110, 18, 140, 9, 76, 246, 237, 146, 1, 231, 12,
        ],
        [
            88, 60, 192, 62, 79, 179, 180, 200, 248, 127, 196, 132, 80, 39, 113, 255, 229, 201,
            241, 38, 97, 34, 4, 49, 232, 50, 107, 64, 4, 224, 224, 37,
        ],
        [
            149, 121, 11, 212, 194, 55, 251, 28, 119, 100, 176, 84, 249, 60, 182, 190, 221, 64,
            228, 2, 214, 39, 56, 220, 227, 136, 228, 13, 141, 159, 245, 42,
        ],
        [
            60, 243, 115, 233, 255, 140, 63, 231, 210, 14, 155, 130, 118, 194, 7, 141, 48, 173,
            107, 162, 177, 75, 123, 169, 232, 103, 73, 240, 67, 80, 57, 24,
        ],
    ],
    [
        [
            123, 136, 142, 35, 186, 163, 74, 144, 91, 217, 190, 33, 143, 34, 55, 166, 163, 87, 199,
            199, 13, 152, 48, 89, 144, 123, 47, 91, 121, 216, 134, 26,
        ],
        [
            91, 28, 158, 185, 86, 38, 1, 159, 40, 166, 63, 248, 93, 73, 118, 55, 180, 26, 78, 86,
            42, 5, 98, 205, 83, 26, 145, 23, 158, 205, 197, 54,
        ],
        [
            13, 183, 119, 158, 77, 201, 152, 68, 105, 136, 252, 223, 120, 231, 59, 72, 171, 229,
            90, 75, 140, 89, 17, 82, 61, 117, 213, 226, 128, 253, 101, 39,
        ],
        [
            59, 213, 226, 8, 10, 97, 178, 127, 43, 50, 225, 84, 92, 71, 0, 6, 25, 254, 192, 225,
            129, 154, 45, 197, 149, 89, 51, 131, 185, 23, 99, 50,
        ],
        [
            86, 45, 93, 113, 243, 120, 77, 157, 17, 22, 33, 162, 239, 86, 34, 58, 187, 204, 242,
            211, 193, 192, 149, 103, 35, 85, 43, 239, 101, 191, 214, 23,
        ],
        [
            46, 196, 68, 68, 11, 189, 17, 137, 247, 165, 190, 8, 171, 95, 251, 9, 230, 51, 75, 122,
            196, 72, 18, 142, 118, 216, 157, 154, 149, 160, 68, 23,
        ],
        [
            135, 237, 213, 40, 201, 43, 82, 54, 223, 17, 46, 232, 102, 235, 40, 83, 119, 29, 149,
            23, 11, 62, 225, 95, 65, 46, 134, 46, 176, 242, 155, 0,
        ],
        [
            253, 9, 210, 43, 163, 50, 199, 172, 36, 116, 174, 131, 178, 56, 211, 11, 88, 67, 68,
            216, 138, 157, 28, 47, 254, 216, 114, 210, 142, 206, 184, 55,
        ],
    ],
    [
        [
            197, 81, 131, 228, 67, 23, 186, 51, 177, 152, 255, 231, 198, 185, 86, 51, 221, 245,
            167, 67, 74, 178, 231, 210, 217, 140, 72, 202, 217, 176, 139, 60,
        ],
        [
            84, 252, 34, 127, 193, 224, 200, 198, 234, 98, 124, 60, 4, 168, 73, 124, 79, 120, 198,
            114, 70, 123, 112, 62, 82, 53, 64, 80, 174, 183, 98, 6,
        ],
        [
            199, 164, 163, 248, 113, 221, 67, 237, 201, 176, 1, 167, 144, 149, 235, 184, 83, 244,
            12, 92, 104, 253, 162, 59, 22, 201, 221, 234, 163, 224, 89, 5,
        ],
        [
            87, 154, 186, 165, 223, 109, 104, 245, 249, 189, 175, 226, 10, 125, 198, 87, 129, 98,
            86, 13, 230, 38, 3, 66, 58, 175, 193, 243, 130, 7, 197, 41,
        ],
        [
            71, 25, 97, 21, 122, 68, 82, 109, 212, 193, 136, 140, 58, 0, 12, 145, 215, 119, 7, 158,
            82, 133, 73, 16, 20, 124, 236, 178, 228, 255, 82, 37,
        ],
        [
            3, 127, 251, 18, 54, 47, 117, 7, 70, 9, 116, 157, 145, 150, 42, 199, 157, 49, 154, 22,
            251, 13, 55, 97, 62, 218, 210, 141, 253, 191, 74, 6,
        ],
        [
            173, 242, 97, 188, 114, 16, 6, 168, 246, 152, 247, 172, 229, 126, 102, 1, 15, 91, 193,
            129, 169, 34, 47, 166, 229, 92, 100, 13, 226, 74, 56, 52,
        ],
        [
            37, 169, 210, 23, 92, 170, 67, 52, 168, 83, 127, 25, 142, 75, 150, 126, 218, 154, 237,
            239, 54, 45, 19, 253, 219, 252, 48, 37, 104, 56, 2, 21,
        ],
    ],
    [
        [
            60, 125, 249, 106, 109, 26, 169, 114, 202, 228, 129, 132, 38, 39, 147, 68, 43, 242, 79,
            218, 20, 31, 165, 156, 137, 220, 49, 241, 151, 247, 105, 53,
        ],
        [
            212, 200, 140, 161, 15, 200, 20, 128, 243, 18, 105, 104, 217, 199, 147, 246, 93, 207,
            207, 122, 198, 145, 173, 157, 243, 84, 236, 192, 37, 75, 26, 18,
        ],
        [
            48, 74, 217, 159, 193, 212, 181, 11, 109, 13, 103, 211, 74, 136, 28, 55, 187, 15, 70,
            196, 203, 198, 192, 71, 116, 32, 179, 94, 46, 123, 243, 62,
        ],
        [
            32, 118, 247, 159, 253, 204, 153, 215, 217, 29, 212, 104, 196, 46, 55, 76, 139, 95,
            221, 156, 28, 206, 109, 199, 191, 207, 48, 128, 215, 199, 153, 14,
        ],
        [
            243, 149, 62, 162, 172, 158, 84, 72, 94, 135, 29, 157, 48, 234, 186, 120, 34, 167, 217,
            200, 135, 119, 88, 173, 184, 86, 133, 70, 192, 84, 196, 57,
        ],
        [
            244, 67, 190, 58, 76, 41, 196, 149, 51, 29, 81, 123, 148, 148, 73, 244, 118, 244, 250,
            41, 239, 117, 59, 26, 104, 33, 23, 149, 65, 244, 39, 33,
        ],
        [
            54, 60, 215, 246, 79, 251, 104, 178, 207, 103, 237, 158, 80, 0, 147, 131, 213, 98, 195,
            57, 104, 115, 125, 246, 1, 99, 10, 173, 43, 139, 242, 1,
        ],
        [
            84, 50, 191, 103, 217, 208, 123, 53, 160, 95, 108, 77, 170, 154, 202, 199, 67, 242, 75,
            192, 30, 171, 48, 213, 177, 52, 208, 103, 137, 76, 9, 0,
        ],
    ],
    [
        [
            198, 88, 29, 105, 10, 101, 59, 211, 37, 76, 97, 63, 188, 168, 187, 252, 174, 75, 103,
            28, 207, 172, 19, 47, 89, 231, 179, 10, 117, 142, 240, 27,
        ],
        [
            37, 124, 164, 233, 8, 135, 109, 1, 136, 169, 167, 146, 195, 44, 71, 214, 193, 210, 10,
            250, 119, 80, 176, 97, 1, 122, 97, 4, 251, 47, 251, 14,
        ],
        [
            189, 244, 239, 17, 181, 148, 184, 159, 229, 49, 128, 129, 26, 100, 215, 116, 27, 55,
            169, 152, 190, 16, 82, 32, 145, 95, 121, 181, 178, 107, 32, 38,
        ],
        [
            208, 57, 233, 100, 166, 183, 132, 121, 242, 57, 169, 22, 247, 39, 94, 133, 137, 112,
            113, 87, 193, 38, 162, 181, 160, 250, 164, 231, 50, 189, 83, 18,
        ],
        [
            89, 30, 89, 240, 145, 188, 55, 190, 206, 229, 82, 37, 221, 78, 109, 41, 114, 35, 237,
            224, 11, 114, 82, 220, 237, 91, 24, 25, 209, 240, 157, 54,
        ],
        [
            53, 11, 143, 37, 127, 190, 2, 16, 220, 176, 53, 153, 217, 128, 213, 211, 52, 117, 20,
            40, 23, 78, 48, 220, 36, 118, 56, 67, 198, 220, 105, 61,
        ],
        [
            130, 17, 90, 245, 190, 206, 74, 238, 215, 29, 201, 98, 113, 136, 240, 128, 69, 26, 86,
            198, 78, 240, 146, 69, 242, 168, 245, 180, 73, 231, 119, 3,
        ],
        [
            117, 183, 82, 52, 195, 130, 3, 41, 1, 9, 249, 106, 246, 164, 128, 188, 215, 143, 162,
            187, 35, 36, 150, 143, 43, 50, 156, 147, 232, 175, 97, 49,
        ],
    ],
    [
        [
            38, 224, 69, 88, 100, 38, 34, 196, 95, 198, 226, 152, 100, 82, 198, 120, 108, 7, 59,
            14, 129, 184, 42, 54, 141, 209, 250, 70, 224, 158, 238, 62,
        ],
        [
            39, 179, 16, 131, 80, 58, 46, 177, 58, 161, 92, 239, 226, 252, 174, 213, 245, 185, 34,
            111, 151, 92, 102, 16, 136, 219, 119, 113, 139, 136, 34, 3,
        ],
        [
            150, 71, 69, 203, 65, 36, 72, 94, 92, 38, 188, 129, 142, 8, 151, 232, 181, 160, 72,
            169, 34, 41, 159, 165, 68, 204, 138, 253, 86, 27, 121, 25,
        ],
        [
            39, 162, 63, 3, 162, 220, 95, 191, 231, 84, 23, 145, 162, 179, 153, 113, 214, 161, 86,
            3, 160, 28, 114, 83, 175, 233, 79, 73, 219, 69, 177, 9,
        ],
        [
            137, 4, 29, 218, 129, 49, 228, 150, 66, 226, 72, 29, 194, 103, 28, 253, 238, 18, 6, 77,
            175, 84, 92, 130, 186, 220, 67, 10, 191, 26, 234, 28,
        ],
        [
            84, 223, 200, 75, 221, 183, 164, 115, 0, 67, 189, 87, 45, 218, 79, 28, 148, 155, 97,
            88, 245, 71, 78, 252, 147, 23, 126, 73, 127, 176, 246, 40,
        ],
        [
            179, 241, 1, 84, 103, 179, 254, 187, 112, 110, 174, 233, 19, 158, 3, 90, 250, 147, 45,
            63, 243, 166, 155, 159, 8, 234, 67, 205, 55, 245, 253, 14,
        ],
        [
            226, 134, 21, 50, 73, 160, 26, 45, 255, 234, 88, 153, 31, 125, 250, 213, 120, 231, 112,
            91, 28, 234, 92, 44, 231, 118, 31, 130, 206, 179, 242, 8,
        ],
    ],
    [
        [
            167, 13, 48, 42, 131, 107, 78, 98, 138, 196, 106, 57, 105, 59, 218, 50, 239, 183, 45,
            39, 30, 212, 206, 92, 65, 91, 170, 52, 34, 57, 174, 11,
        ],
        [
            217, 73, 127, 55, 81, 100, 78, 66, 157, 217, 177, 137, 100, 148, 138, 154, 9, 201, 139,
            236, 209, 100, 133, 99, 114, 87, 106, 9, 109, 201, 195, 11,
        ],
        [
            94, 10, 206, 24, 189, 217, 152, 247, 221, 25, 15, 16, 3, 132, 132, 84, 101, 19, 137,
            74, 224, 150, 101, 239, 114, 174, 156, 74, 119, 58, 68, 13,
        ],
        [
            23, 77, 197, 220, 177, 194, 206, 23, 139, 196, 0, 238, 80, 128, 118, 20, 243, 149, 74,
            32, 204, 153, 72, 26, 160, 54, 66, 168, 143, 169, 204, 51,
        ],
        [
            40, 217, 104, 147, 156, 74, 178, 30, 32, 214, 13, 145, 178, 253, 53, 41, 125, 142, 47,
            40, 15, 49, 207, 85, 55, 217, 159, 250, 157, 91, 55, 55,
        ],
        [
            40, 244, 153, 231, 80, 142, 238, 13, 177, 131, 169, 177, 186, 16, 151, 255, 8, 111, 47,
            152, 52, 15, 83, 240, 84, 197, 239, 128, 202, 202, 76, 30,
        ],
        [
            250, 60, 71, 175, 180, 159, 32, 159, 225, 187, 30, 160, 169, 116, 171, 153, 23, 152,
            172, 89, 254, 12, 18, 217, 158, 115, 139, 123, 180, 230, 119, 58,
        ],
        [
            208, 102, 76, 132, 87, 190, 6, 202, 244, 78, 197, 130, 183, 148, 55, 26, 61, 211, 35,
            148, 181, 36, 158, 12, 157, 120, 21, 100, 45, 85, 114, 5,
        ],
    ],
    [
        [
            154, 225, 115, 161, 18, 54, 207, 2, 188, 4, 87, 220, 139, 224, 192, 9, 203, 28, 219,
            127, 130, 225, 41, 129, 213, 5, 243, 222, 219, 195, 53, 54,
        ],
        [
            112, 249, 215, 104, 105, 10, 110, 38, 76, 233, 154, 73, 92, 73, 120, 254, 31, 78, 198,
            139, 142, 165, 130, 70, 118, 175, 226, 90, 139, 203, 142, 30,
        ],
        [
            98, 135, 112, 214, 115, 38, 47, 143, 75, 8, 199, 93, 243, 142, 80, 205, 242, 230, 144,
            69, 57, 131, 221, 123, 154, 254, 70, 177, 38, 5, 179, 55,
        ],
        [
            172, 90, 158, 114, 79, 59, 0, 48, 148, 66, 138, 218, 129, 224, 130, 181, 91, 214, 102,
            138, 186, 20, 248, 111, 189, 6, 113, 155, 152, 70, 184, 53,
        ],
        [
            93, 132, 123, 157, 80, 196, 33, 122, 74, 252, 53, 16, 109, 153, 14, 47, 185, 36, 0,
            205, 112, 169, 47, 118, 85, 10, 88, 148, 215, 102, 50, 49,
        ],
        [
            227, 29, 11, 192, 11, 129, 121, 0, 224, 246, 251, 181, 43, 252, 124, 103, 158, 178,
            116, 105, 135, 194, 215, 0, 74, 24, 4, 103, 198, 99, 92, 62,
        ],
        [
            215, 88, 93, 169, 126, 62, 12, 33, 41, 174, 153, 42, 94, 183, 54, 237, 185, 27, 82,
            137, 5, 203, 193, 5, 43, 214, 218, 227, 158, 86, 67, 11,
        ],
        [
            204, 198, 113, 206, 88, 212, 218, 203, 76, 170, 88, 64, 85, 144, 199, 183, 19, 103,
            204, 12, 154, 29, 84, 152, 96, 147, 233, 62, 34, 37, 252, 19,
        ],
    ],
    [
        [
            22, 87, 1, 109, 71, 145, 157, 11, 187, 118, 148, 132, 206, 10, 133, 13, 217, 56, 66,
            83, 203, 55, 30, 139, 161, 158, 238, 242, 100, 169, 53, 2,
        ],
        [
            71, 213, 215, 26, 60, 9, 211, 106, 70, 217, 178, 104, 196, 36, 165, 198, 160, 50, 164,
            0, 229, 135, 70, 162, 214, 3, 75, 70, 207, 163, 75, 56,
        ],
        [
            221, 79, 237, 22, 189, 144, 124, 70, 190, 2, 159, 160, 123, 87, 13, 127, 179, 26, 237,
            107, 37, 252, 71, 33, 180, 38, 63, 45, 24, 235, 161, 20,
        ],
        [
            187, 190, 72, 212, 201, 195, 98, 232, 141, 33, 40, 137, 167, 195, 147, 44, 89, 98, 212,
            216, 94, 167, 189, 122, 222, 141, 116, 37, 88, 168, 103, 17,
        ],
        [
            239, 90, 233, 200, 169, 16, 109, 99, 224, 159, 160, 124, 212, 51, 244, 255, 153, 179,
            203, 234, 125, 186, 86, 67, 157, 144, 19, 254, 145, 81, 67, 37,
        ],
        [
            43, 152, 102, 183, 158, 246, 197, 209, 214, 63, 126, 228, 123, 198, 231, 33, 187, 146,
            175, 159, 120, 2, 176, 113, 101, 202, 195, 61, 128, 217, 57, 41,
        ],
        [
            34, 53, 55, 23, 239, 129, 15, 228, 220, 113, 238, 34, 251, 139, 231, 49, 202, 252, 146,
            71, 231, 123, 197, 152, 214, 254, 26, 213, 42, 141, 44, 59,
        ],
        [
            14, 199, 227, 234, 179, 149, 80, 241, 36, 197, 33, 50, 41, 101, 228, 198, 1, 235, 176,
            65, 116, 150, 209, 35, 170, 142, 31, 4, 168, 133, 28, 25,
        ],
    ],
    [
        [
            227, 69, 192, 97, 61, 2, 159, 47, 31, 246, 219, 46, 10, 9, 197, 91, 223, 218, 48, 88,
            68, 196, 183, 101, 146, 42, 88, 243, 6, 163, 85, 24,
        ],
        [
            133, 181, 41, 19, 10, 65, 213, 69, 31, 230, 153, 104, 99, 35, 68, 61, 213, 42, 252,
            215, 14, 69, 37, 225, 58, 113, 145, 234, 102, 36, 26, 61,
        ],
        [
            36, 174, 201, 204, 110, 25, 137, 23, 88, 63, 173, 206, 140, 217, 221, 34, 191, 239,
            200, 133, 87, 234, 201, 114, 100, 21, 214, 213, 227, 116, 87, 43,
        ],
        [
            33, 254, 187, 9, 6, 142, 196, 236, 174, 73, 115, 38, 11, 202, 94, 21, 232, 88, 10, 16,
            86, 137, 2, 114, 74, 232, 75, 217, 55, 154, 163, 36,
        ],
        [
            61, 43, 128, 223, 251, 52, 151, 169, 224, 228, 34, 213, 36, 60, 191, 138, 119, 196,
            244, 87, 107, 167, 139, 227, 227, 36, 95, 70, 145, 206, 23, 59,
        ],
        [
            149, 239, 176, 117, 82, 16, 109, 249, 224, 163, 81, 48, 91, 82, 238, 212, 205, 250,
            105, 102, 169, 8, 45, 218, 154, 106, 204, 233, 104, 179, 21, 47,
        ],
        [
            136, 82, 212, 89, 173, 244, 117, 33, 77, 197, 211, 141, 19, 126, 62, 14, 18, 88, 118,
            61, 46, 233, 207, 23, 181, 122, 253, 31, 223, 68, 115, 44,
        ],
        [
            230, 82, 213, 56, 121, 61, 11, 146, 68, 52, 147, 115, 18, 230, 149, 148, 48, 88, 173,
            238, 218, 98, 191, 129, 200, 222, 212, 245, 210, 215, 171, 28,
        ],
    ],
    [
        [
            191, 225, 193, 65, 70, 115, 255, 253, 40, 60, 196, 222, 142, 70, 14, 97, 163, 100, 23,
            145, 35, 160, 154, 227, 141, 93, 182, 222, 47, 174, 56, 40,
        ],
        [
            197, 21, 27, 76, 137, 232, 211, 79, 171, 229, 90, 184, 215, 43, 72, 109, 244, 111, 249,
            227, 72, 76, 67, 98, 93, 71, 22, 15, 118, 88, 255, 26,
        ],
        [
            171, 251, 162, 17, 102, 160, 158, 12, 84, 9, 1, 167, 213, 63, 157, 165, 63, 95, 171,
            233, 58, 186, 255, 70, 112, 8, 197, 221, 248, 175, 3, 11,
        ],
        [
            211, 3, 175, 11, 139, 74, 240, 174, 110, 157, 211, 68, 153, 75, 254, 118, 192, 108,
            206, 103, 55, 115, 235, 149, 79, 220, 166, 192, 115, 153, 230, 48,
        ],
        [
            27, 45, 52, 129, 100, 72, 202, 230, 149, 80, 18, 81, 42, 120, 156, 241, 218, 16, 49,
            65, 215, 124, 129, 56, 207, 166, 117, 199, 29, 53, 222, 26,
        ],
        [
            76, 70, 58, 161, 96, 99, 111, 185, 67, 54, 34, 220, 49, 46, 125, 120, 115, 149, 96,
            111, 185, 212, 0, 186, 22, 117, 140, 245, 135, 179, 42, 19,
        ],
        [
            66, 230, 5, 90, 123, 140, 234, 5, 139, 250, 44, 103, 84, 127, 174, 178, 194, 65, 195,
            50, 213, 77, 153, 212, 244, 209, 216, 149, 55, 234, 135, 6,
        ],
        [
            254, 64, 206, 41, 117, 44, 153, 71, 0, 204, 67, 65, 231, 116, 35, 8, 99, 69, 207, 34,
            15, 233, 184, 86, 39, 186, 205, 238, 233, 245, 250, 44,
        ],
    ],
    [
        [
            199, 157, 28, 88, 230, 49, 91, 18, 24, 87, 45, 179, 63, 135, 122, 54, 172, 223, 233,
            189, 237, 197, 203, 20, 126, 47, 99, 38, 69, 226, 255, 14,
        ],
        [
            166, 224, 94, 181, 103, 110, 4, 185, 117, 168, 188, 254, 60, 122, 232, 159, 147, 129,
            120, 140, 200, 103, 249, 27, 133, 22, 186, 19, 170, 127, 231, 35,
        ],
        [
            148, 181, 107, 54, 45, 247, 223, 58, 41, 106, 43, 34, 157, 54, 190, 18, 237, 98, 10,
            159, 193, 210, 95, 129, 44, 182, 121, 37, 250, 123, 200, 56,
        ],
        [
            165, 157, 120, 229, 210, 117, 15, 122, 29, 182, 94, 216, 135, 219, 142, 206, 235, 38,
            6, 254, 112, 192, 172, 33, 177, 179, 58, 161, 33, 231, 226, 44,
        ],
        [
            22, 188, 212, 0, 63, 2, 102, 220, 41, 147, 252, 183, 186, 222, 65, 133, 223, 104, 245,
            251, 67, 50, 241, 141, 16, 153, 53, 36, 223, 92, 56, 33,
        ],
        [
            3, 171, 26, 109, 247, 144, 210, 147, 173, 157, 69, 27, 151, 43, 191, 184, 230, 134,
            121, 109, 159, 59, 207, 83, 160, 231, 1, 16, 121, 248, 25, 2,
        ],
        [
            187, 76, 191, 137, 218, 205, 64, 191, 85, 92, 226, 13, 159, 50, 44, 20, 106, 248, 106,
            104, 38, 147, 26, 206, 72, 152, 219, 227, 39, 111, 153, 22,
        ],
        [
            37, 88, 160, 215, 216, 90, 55, 159, 157, 19, 83, 137, 20, 17, 97, 87, 108, 194, 57,
            124, 237, 155, 251, 143, 155, 24, 194, 250, 107, 157, 108, 6,
        ],
    ],
    [
        [
            179, 238, 187, 220, 201, 52, 151, 82, 231, 246, 234, 157, 60, 224, 148, 107, 52, 158,
            184, 135, 41, 91, 13, 157, 216, 87, 79, 70, 110, 154, 19, 9,
        ],
        [
            79, 134, 215, 209, 161, 228, 162, 74, 17, 227, 169, 26, 65, 40, 189, 70, 253, 44, 93,
            128, 39, 142, 12, 97, 154, 111, 201, 52, 11, 129, 200, 9,
        ],
        [
            73, 100, 11, 140, 141, 41, 181, 27, 253, 130, 136, 154, 107, 9, 135, 217, 16, 152, 64,
            77, 175, 250, 164, 120, 110, 201, 44, 94, 10, 74, 233, 42,
        ],
        [
            9, 25, 17, 237, 209, 6, 7, 44, 161, 60, 19, 113, 80, 235, 188, 115, 251, 74, 128, 65,
            255, 21, 126, 249, 164, 59, 22, 226, 177, 71, 22, 24,
        ],
        [
            34, 100, 239, 148, 80, 7, 119, 217, 234, 64, 187, 147, 240, 200, 246, 154, 249, 100,
            209, 249, 185, 119, 23, 202, 156, 219, 19, 22, 127, 127, 212, 25,
        ],
        [
            239, 117, 59, 251, 76, 110, 242, 157, 3, 197, 97, 180, 211, 12, 224, 189, 252, 196,
            171, 31, 104, 172, 244, 239, 53, 75, 208, 126, 41, 144, 59, 49,
        ],
        [
            29, 222, 224, 229, 25, 2, 93, 176, 51, 17, 100, 137, 64, 86, 76, 200, 138, 21, 29, 217,
            63, 159, 144, 129, 8, 201, 76, 236, 91, 62, 67, 50,
        ],
        [
            24, 21, 241, 5, 210, 159, 87, 84, 200, 22, 175, 154, 87, 212, 37, 148, 23, 160, 175,
            200, 118, 117, 174, 229, 11, 130, 157, 255, 26, 52, 11, 8,
        ],
    ],
    [
        [
            252, 153, 100, 93, 87, 56, 92, 116, 12, 45, 12, 57, 66, 110, 227, 147, 147, 101, 100,
            0, 10, 181, 178, 164, 21, 13, 180, 13, 89, 248, 205, 7,
        ],
        [
            86, 69, 138, 31, 50, 117, 37, 245, 147, 126, 71, 200, 159, 103, 75, 102, 204, 178, 135,
            180, 3, 63, 115, 110, 254, 22, 32, 8, 50, 111, 104, 22,
        ],
        [
            66, 78, 49, 221, 153, 235, 211, 155, 58, 202, 119, 231, 206, 30, 248, 215, 223, 171,
            150, 81, 46, 255, 66, 79, 74, 248, 252, 151, 251, 131, 160, 7,
        ],
        [
            153, 143, 67, 122, 83, 102, 174, 152, 47, 152, 114, 206, 200, 235, 113, 13, 45, 164,
            174, 214, 19, 129, 73, 211, 103, 185, 237, 210, 68, 157, 11, 24,
        ],
        [
            83, 217, 27, 197, 244, 71, 107, 145, 132, 40, 44, 124, 230, 82, 21, 131, 249, 225, 116,
            102, 163, 121, 50, 142, 20, 27, 72, 67, 183, 28, 169, 46,
        ],
        [
            56, 15, 117, 133, 19, 226, 237, 94, 34, 153, 237, 147, 143, 59, 163, 126, 83, 107, 102,
            25, 204, 62, 208, 59, 218, 48, 126, 107, 79, 164, 3, 3,
        ],
        [
            127, 209, 7, 206, 197, 21, 110, 43, 111, 178, 96, 254, 183, 112, 65, 204, 159, 42, 129,
            127, 147, 172, 175, 22, 114, 1, 144, 109, 70, 51, 246, 28,
        ],
        [
            126, 247, 38, 212, 155, 106, 76, 120, 152, 196, 61, 109, 174, 141, 172, 42, 144, 51,
            35, 254, 134, 214, 219, 152, 80, 109, 121, 166, 91, 226, 119, 18,
        ],
    ],
    [
        [
            131, 89, 210, 253, 224, 36, 151, 136, 156, 86, 88, 23, 205, 83, 177, 59, 106, 12, 16,
            233, 183, 197, 90, 203, 29, 196, 1, 237, 224, 223, 99, 15,
        ],
        [
            224, 204, 224, 17, 99, 37, 205, 45, 255, 41, 69, 6, 204, 226, 63, 14, 180, 116, 248,
            184, 82, 55, 19, 160, 151, 69, 187, 35, 87, 120, 132, 15,
        ],
        [
            125, 177, 114, 23, 209, 68, 219, 141, 142, 186, 25, 218, 57, 226, 190, 210, 45, 94,
            196, 39, 20, 233, 122, 93, 225, 23, 220, 158, 192, 78, 152, 24,
        ],
        [
            57, 56, 40, 61, 97, 131, 104, 4, 240, 109, 227, 75, 152, 44, 242, 243, 188, 45, 229,
            118, 46, 27, 180, 167, 240, 224, 124, 15, 143, 50, 225, 6,
        ],
        [
            53, 16, 46, 178, 58, 24, 197, 247, 42, 220, 177, 47, 133, 95, 81, 61, 134, 28, 220,
            184, 45, 12, 121, 18, 174, 234, 250, 124, 46, 190, 104, 19,
        ],
        [
            38, 156, 61, 146, 163, 231, 91, 107, 19, 123, 200, 192, 85, 173, 156, 34, 44, 66, 161,
            83, 221, 166, 70, 51, 85, 232, 243, 252, 162, 18, 15, 56,
        ],
        [
            123, 18, 87, 200, 196, 100, 121, 223, 151, 195, 24, 144, 131, 124, 56, 229, 199, 83,
            183, 137, 62, 154, 247, 40, 76, 135, 146, 9, 103, 5, 68, 33,
        ],
        [
            184, 165, 230, 0, 109, 19, 115, 70, 25, 102, 2, 138, 190, 210, 62, 196, 27, 76, 57, 31,
            127, 78, 172, 163, 114, 236, 24, 224, 54, 31, 4, 45,
        ],
    ],
    [
        [
            138, 133, 77, 92, 247, 250, 141, 156, 66, 158, 130, 208, 243, 85, 97, 12, 47, 187, 101,
            220, 195, 14, 119, 254, 22, 0, 119, 0, 247, 177, 238, 18,
        ],
        [
            202, 128, 126, 221, 182, 213, 55, 69, 34, 2, 42, 251, 22, 177, 35, 130, 132, 199, 33,
            5, 210, 99, 140, 214, 159, 25, 136, 202, 160, 255, 153, 29,
        ],
        [
            8, 156, 183, 206, 52, 122, 127, 243, 45, 216, 68, 38, 215, 206, 139, 166, 196, 43, 58,
            250, 200, 141, 58, 74, 28, 236, 244, 39, 167, 34, 55, 26,
        ],
        [
            174, 91, 180, 80, 186, 115, 38, 145, 115, 89, 5, 138, 189, 53, 49, 203, 52, 86, 205,
            123, 164, 146, 169, 110, 119, 202, 249, 86, 19, 179, 89, 60,
        ],
        [
            135, 230, 4, 37, 38, 213, 28, 78, 101, 215, 177, 207, 184, 98, 242, 130, 215, 189, 209,
            198, 171, 165, 39, 79, 18, 196, 43, 75, 85, 32, 191, 46,
        ],
        [
            121, 226, 59, 93, 13, 120, 205, 138, 103, 215, 23, 193, 170, 191, 103, 45, 154, 1, 76,
            213, 25, 118, 23, 228, 233, 180, 229, 181, 61, 104, 52, 14,
        ],
        [
            4, 161, 12, 147, 13, 60, 20, 82, 53, 224, 72, 90, 102, 123, 77, 85, 31, 11, 131, 50,
            41, 88, 50, 22, 149, 35, 29, 142, 247, 84, 135, 16,
        ],
        [
            196, 43, 149, 199, 167, 192, 232, 239, 9, 9, 110, 4, 148, 64, 58, 97, 207, 253, 80,
            247, 228, 179, 121, 4, 183, 162, 169, 120, 200, 254, 109, 35,
        ],
    ],
    [
        [
            217, 164, 120, 31, 248, 243, 75, 91, 94, 55, 138, 226, 220, 92, 79, 18, 59, 193, 156,
            126, 26, 222, 103, 8, 250, 221, 211, 68, 177, 71, 133, 11,
        ],
        [
            34, 78, 160, 92, 3, 181, 43, 166, 227, 79, 153, 125, 39, 253, 0, 135, 22, 125, 102,
            161, 212, 195, 132, 46, 25, 132, 4, 234, 198, 43, 85, 49,
        ],
        [
            133, 39, 163, 244, 250, 113, 200, 90, 202, 172, 126, 52, 182, 135, 223, 43, 6, 214,
            135, 140, 239, 172, 1, 36, 37, 93, 212, 96, 74, 51, 189, 10,
        ],
        [
            126, 177, 236, 150, 50, 183, 198, 85, 154, 215, 89, 75, 4, 57, 211, 33, 179, 220, 253,
            182, 13, 65, 218, 80, 132, 54, 222, 214, 52, 39, 198, 7,
        ],
        [
            133, 246, 57, 187, 96, 68, 33, 138, 75, 185, 97, 210, 2, 89, 238, 63, 73, 39, 51, 58,
            242, 125, 16, 9, 91, 94, 243, 69, 35, 42, 136, 9,
        ],
        [
            149, 191, 120, 245, 46, 159, 66, 53, 67, 200, 107, 196, 86, 191, 234, 195, 140, 23,
            152, 5, 213, 125, 61, 250, 113, 227, 62, 153, 193, 198, 62, 23,
        ],
        [
            181, 92, 7, 115, 101, 203, 87, 170, 108, 11, 133, 229, 209, 88, 235, 9, 211, 0, 34,
            128, 143, 32, 158, 16, 55, 100, 222, 242, 247, 39, 217, 41,
        ],
        [
            176, 9, 186, 146, 2, 201, 229, 116, 61, 190, 159, 129, 11, 22, 215, 50, 181, 105, 128,
            91, 161, 104, 19, 214, 144, 254, 225, 169, 186, 237, 100, 32,
        ],
    ],
    [
        [
            40, 7, 199, 148, 140, 28, 231, 90, 252, 187, 127, 106, 225, 109, 28, 74, 174, 202, 162,
            233, 111, 25, 39, 121, 86, 188, 166, 60, 26, 192, 125, 23,
        ],
        [
            29, 191, 141, 163, 214, 37, 63, 14, 117, 75, 249, 36, 245, 31, 107, 34, 130, 14, 5,
            129, 145, 43, 255, 54, 59, 211, 77, 194, 248, 213, 122, 10,
        ],
        [
            90, 72, 79, 149, 87, 28, 38, 255, 171, 49, 255, 93, 202, 66, 30, 225, 2, 33, 50, 59,
            167, 22, 89, 84, 195, 250, 231, 41, 48, 59, 54, 54,
        ],
        [
            52, 242, 43, 225, 153, 145, 156, 46, 4, 211, 112, 65, 19, 141, 52, 99, 139, 149, 190,
            143, 143, 57, 228, 85, 0, 67, 199, 163, 60, 171, 182, 2,
        ],
        [
            6, 134, 104, 87, 111, 169, 74, 7, 110, 63, 55, 7, 135, 154, 171, 179, 33, 64, 254, 160,
            90, 79, 218, 218, 147, 45, 244, 115, 21, 71, 63, 54,
        ],
        [
            182, 244, 16, 171, 5, 16, 178, 183, 14, 152, 205, 29, 128, 93, 92, 17, 206, 47, 111,
            155, 27, 164, 23, 21, 13, 125, 119, 240, 170, 148, 16, 53,
        ],
        [
            78, 4, 253, 34, 167, 129, 30, 5, 1, 146, 243, 55, 152, 9, 74, 73, 26, 199, 175, 138,
            162, 181, 235, 238, 158, 40, 157, 162, 8, 163, 63, 29,
        ],
        [
            159, 213, 249, 32, 115, 186, 250, 55, 240, 55, 181, 229, 106, 166, 99, 149, 114, 141,
            239, 154, 192, 4, 127, 64, 247, 172, 240, 23, 243, 109, 102, 16,
        ],
    ],
    [
        [
            208, 62, 192, 255, 179, 46, 237, 29, 32, 212, 88, 59, 32, 109, 116, 45, 150, 40, 251,
            132, 136, 88, 63, 10, 111, 34, 102, 42, 7, 106, 48, 30,
        ],
        [
            235, 194, 35, 149, 132, 244, 37, 88, 78, 71, 224, 63, 246, 169, 172, 201, 131, 7, 57,
            88, 91, 199, 146, 170, 188, 167, 141, 1, 40, 121, 209, 61,
        ],
        [
            50, 65, 193, 19, 111, 120, 148, 118, 147, 240, 246, 104, 72, 174, 113, 54, 201, 144,
            17, 98, 92, 11, 113, 250, 183, 114, 0, 74, 134, 20, 217, 47,
        ],
        [
            87, 190, 24, 114, 241, 193, 159, 122, 102, 3, 11, 48, 61, 248, 75, 21, 92, 19, 224,
            163, 186, 64, 204, 3, 37, 86, 103, 31, 53, 49, 204, 62,
        ],
        [
            137, 116, 208, 75, 219, 108, 60, 61, 85, 214, 112, 149, 190, 75, 153, 174, 134, 62,
            187, 183, 63, 151, 51, 41, 250, 231, 194, 99, 131, 71, 91, 57,
        ],
        [
            214, 76, 152, 156, 14, 135, 169, 54, 244, 66, 31, 124, 203, 249, 203, 238, 149, 193,
            165, 38, 141, 78, 15, 161, 59, 180, 218, 91, 96, 75, 3, 42,
        ],
        [
            9, 219, 89, 115, 107, 140, 184, 162, 172, 175, 14, 145, 179, 159, 157, 5, 36, 112, 255,
            25, 134, 192, 252, 32, 94, 23, 81, 146, 187, 238, 88, 59,
        ],
        [
            226, 191, 50, 112, 133, 132, 199, 142, 252, 89, 92, 227, 45, 129, 181, 217, 107, 158,
            105, 62, 76, 104, 15, 6, 219, 239, 88, 28, 104, 248, 0, 39,
        ],
    ],
    [
        [
            135, 78, 215, 34, 105, 20, 88, 1, 116, 251, 173, 74, 81, 101, 102, 140, 171, 42, 13,
            54, 241, 73, 104, 41, 94, 167, 87, 232, 13, 15, 171, 6,
        ],
        [
            18, 88, 231, 231, 168, 128, 81, 148, 23, 38, 172, 207, 34, 73, 241, 215, 221, 119, 64,
            126, 107, 176, 138, 251, 112, 72, 252, 250, 245, 48, 36, 32,
        ],
        [
            20, 44, 38, 7, 55, 71, 118, 91, 105, 195, 150, 14, 32, 211, 187, 89, 108, 224, 153,
            109, 204, 72, 160, 152, 23, 199, 155, 84, 53, 183, 30, 17,
        ],
        [
            244, 232, 165, 147, 219, 157, 69, 181, 173, 36, 94, 183, 49, 16, 68, 180, 203, 232,
            159, 198, 107, 174, 162, 163, 135, 203, 109, 106, 71, 28, 132, 54,
        ],
        [
            128, 32, 218, 132, 20, 69, 199, 166, 54, 131, 78, 244, 246, 161, 200, 50, 62, 90, 228,
            51, 26, 1, 225, 211, 41, 51, 115, 34, 4, 184, 7, 29,
        ],
        [
            77, 46, 225, 194, 127, 232, 30, 240, 158, 194, 26, 28, 221, 172, 60, 118, 248, 174,
            218, 183, 52, 107, 113, 40, 234, 106, 137, 58, 251, 190, 14, 47,
        ],
        [
            84, 208, 14, 230, 33, 132, 17, 151, 159, 240, 47, 66, 226, 210, 183, 101, 220, 32, 19,
            175, 47, 156, 100, 55, 2, 45, 224, 187, 154, 232, 63, 45,
        ],
        [
            38, 3, 199, 23, 232, 121, 90, 88, 192, 71, 149, 125, 160, 140, 92, 248, 244, 52, 250,
            237, 159, 240, 87, 131, 206, 103, 57, 231, 240, 126, 44, 8,
        ],
    ],
    [
        [
            121, 185, 230, 118, 95, 189, 254, 13, 234, 203, 79, 140, 250, 63, 61, 205, 108, 239,
            89, 178, 242, 36, 237, 131, 191, 178, 198, 57, 106, 146, 208, 2,
        ],
        [
            254, 175, 112, 227, 155, 33, 74, 141, 22, 57, 159, 5, 129, 57, 239, 27, 96, 22, 86, 8,
            223, 241, 173, 226, 171, 232, 32, 45, 158, 169, 2, 25,
        ],
        [
            21, 46, 93, 77, 133, 253, 127, 147, 1, 240, 85, 211, 36, 215, 94, 13, 26, 253, 25, 72,
            67, 90, 229, 7, 174, 236, 233, 250, 65, 217, 124, 51,
        ],
        [
            213, 61, 22, 32, 180, 27, 96, 10, 216, 12, 135, 161, 36, 223, 176, 137, 197, 99, 83,
            215, 58, 242, 191, 192, 84, 236, 195, 12, 141, 45, 215, 23,
        ],
        [
            147, 199, 116, 205, 253, 64, 71, 177, 48, 220, 174, 145, 113, 191, 60, 31, 60, 171, 79,
            86, 141, 122, 89, 188, 252, 100, 42, 127, 193, 98, 172, 43,
        ],
        [
            83, 40, 233, 182, 200, 135, 94, 11, 109, 77, 241, 71, 248, 64, 10, 227, 84, 146, 205,
            168, 77, 57, 138, 34, 128, 201, 16, 115, 35, 244, 147, 0,
        ],
        [
            8, 124, 97, 197, 83, 77, 42, 95, 91, 69, 173, 108, 237, 215, 44, 211, 28, 202, 237, 0,
            22, 248, 104, 213, 146, 245, 52, 221, 218, 26, 160, 62,
        ],
        [
            46, 200, 28, 76, 156, 255, 239, 240, 233, 131, 252, 96, 12, 144, 56, 163, 52, 116, 162,
            59, 149, 1, 133, 107, 190, 132, 233, 235, 230, 244, 231, 49,
        ],
    ],
    [
        [
            131, 244, 147, 237, 104, 133, 117, 169, 82, 239, 134, 219, 21, 106, 10, 250, 191, 18,
            211, 97, 10, 161, 87, 187, 16, 85, 107, 192, 9, 17, 208, 44,
        ],
        [
            129, 156, 210, 98, 148, 44, 136, 246, 152, 130, 74, 18, 3, 30, 242, 12, 155, 8, 49,
            245, 49, 215, 93, 167, 0, 8, 213, 68, 9, 38, 113, 53,
        ],
        [
            114, 214, 69, 108, 113, 73, 81, 37, 251, 91, 43, 99, 248, 39, 99, 40, 85, 223, 101, 78,
            243, 170, 160, 149, 155, 104, 14, 78, 97, 248, 37, 57,
        ],
        [
            77, 232, 118, 221, 50, 130, 124, 90, 91, 140, 68, 158, 193, 229, 180, 176, 151, 167,
            148, 141, 46, 104, 179, 55, 57, 142, 216, 23, 141, 196, 236, 54,
        ],
        [
            198, 180, 66, 160, 132, 49, 199, 243, 237, 240, 166, 225, 99, 37, 155, 24, 179, 113,
            199, 80, 80, 132, 32, 52, 251, 211, 162, 116, 170, 203, 81, 42,
        ],
        [
            241, 79, 65, 222, 239, 74, 64, 122, 34, 93, 203, 102, 251, 103, 65, 233, 204, 203, 192,
            245, 189, 251, 1, 151, 128, 157, 182, 203, 193, 102, 27, 23,
        ],
        [
            60, 20, 8, 87, 8, 19, 211, 166, 63, 239, 146, 164, 49, 67, 11, 133, 95, 9, 182, 199,
            17, 86, 113, 170, 168, 83, 77, 230, 81, 201, 238, 13,
        ],
        [
            68, 167, 166, 81, 81, 66, 93, 73, 185, 91, 178, 215, 84, 163, 122, 201, 149, 247, 161,
            87, 127, 103, 79, 29, 14, 199, 44, 247, 33, 123, 183, 38,
        ],
    ],
    [
        [
            143, 198, 147, 9, 216, 184, 32, 238, 248, 95, 47, 180, 248, 117, 252, 150, 150, 89,
            148, 161, 187, 47, 61, 225, 246, 154, 106, 101, 128, 51, 151, 12,
        ],
        [
            38, 250, 166, 101, 235, 31, 63, 68, 193, 170, 168, 34, 52, 63, 97, 98, 214, 229, 31,
            74, 233, 172, 22, 81, 196, 245, 11, 111, 44, 98, 185, 1,
        ],
        [
            141, 209, 110, 148, 171, 39, 157, 235, 89, 173, 140, 216, 148, 1, 175, 157, 227, 104,
            210, 112, 66, 160, 156, 124, 27, 102, 120, 142, 149, 110, 148, 12,
        ],
        [
            71, 30, 90, 107, 232, 6, 151, 1, 6, 25, 188, 6, 31, 64, 184, 104, 184, 216, 57, 92,
            230, 236, 169, 238, 77, 38, 45, 165, 162, 162, 252, 7,
        ],
        [
            136, 10, 204, 99, 83, 109, 73, 101, 61, 41, 237, 197, 0, 232, 185, 168, 79, 128, 191,
            219, 87, 14, 163, 244, 0, 145, 72, 199, 11, 55, 218, 38,
        ],
        [
            246, 115, 92, 44, 64, 216, 74, 82, 132, 50, 82, 211, 56, 171, 176, 137, 24, 110, 103,
            158, 46, 158, 151, 154, 132, 254, 220, 180, 5, 120, 72, 10,
        ],
        [
            30, 159, 81, 104, 246, 32, 96, 43, 152, 44, 150, 221, 165, 14, 162, 61, 30, 166, 133,
            194, 104, 127, 76, 214, 122, 89, 109, 2, 219, 232, 185, 5,
        ],
        [
            138, 10, 208, 162, 118, 236, 11, 68, 180, 213, 38, 108, 145, 79, 115, 185, 231, 51, 84,
            160, 104, 136, 76, 142, 201, 121, 111, 41, 3, 126, 71, 52,
        ],
    ],
    [
        [
            146, 172, 178, 233, 212, 141, 180, 76, 211, 173, 0, 136, 102, 215, 35, 18, 210, 57,
            230, 203, 52, 90, 73, 253, 90, 56, 84, 206, 187, 227, 139, 51,
        ],
        [
            12, 172, 149, 245, 3, 108, 117, 73, 133, 51, 161, 59, 203, 181, 112, 71, 115, 193, 65,
            135, 234, 163, 80, 138, 170, 33, 62, 175, 158, 128, 5, 62,
        ],
        [
            133, 76, 138, 32, 247, 244, 21, 235, 18, 19, 89, 83, 65, 116, 151, 3, 0, 132, 125, 21,
            220, 165, 25, 140, 77, 25, 235, 191, 41, 144, 142, 54,
        ],
        [
            234, 91, 142, 174, 106, 23, 132, 53, 16, 96, 217, 122, 84, 117, 226, 87, 131, 59, 157,
            78, 201, 217, 103, 134, 14, 141, 193, 42, 17, 135, 118, 3,
        ],
        [
            60, 209, 64, 222, 188, 72, 228, 235, 152, 155, 14, 252, 73, 49, 37, 168, 181, 101, 219,
            232, 169, 55, 171, 90, 138, 159, 181, 180, 41, 195, 41, 48,
        ],
        [
            153, 0, 87, 75, 203, 95, 71, 181, 161, 183, 42, 85, 14, 77, 220, 46, 138, 157, 229,
            178, 159, 144, 57, 24, 238, 125, 128, 79, 134, 222, 64, 61,
        ],
        [
            51, 123, 20, 217, 230, 53, 195, 7, 147, 10, 229, 20, 147, 206, 100, 197, 88, 73, 14,
            211, 138, 72, 211, 100, 184, 62, 149, 254, 53, 140, 211, 38,
        ],
        [
            125, 218, 238, 203, 197, 139, 19, 248, 119, 73, 85, 241, 205, 20, 187, 100, 110, 150,
            80, 176, 39, 83, 85, 44, 160, 209, 47, 240, 119, 195, 98, 9,
        ],
    ],
    [
        [
            225, 193, 188, 61, 63, 105, 192, 160, 101, 114, 91, 50, 187, 102, 91, 202, 144, 99, 48,
            178, 235, 57, 151, 190, 202, 221, 26, 129, 39, 222, 144, 13,
        ],
        [
            214, 4, 119, 107, 105, 182, 45, 196, 47, 123, 78, 115, 118, 142, 134, 160, 176, 161,
            175, 251, 249, 25, 123, 177, 39, 236, 198, 98, 166, 14, 35, 47,
        ],
        [
            227, 146, 231, 46, 217, 182, 26, 134, 224, 88, 174, 142, 210, 131, 27, 217, 90, 20,
            169, 93, 202, 238, 30, 17, 75, 25, 218, 94, 117, 74, 98, 28,
        ],
        [
            211, 62, 181, 29, 241, 231, 74, 132, 58, 71, 13, 180, 216, 120, 215, 208, 191, 166,
            110, 238, 239, 129, 248, 13, 11, 70, 25, 173, 118, 194, 208, 38,
        ],
        [
            66, 6, 102, 187, 1, 88, 189, 123, 109, 200, 226, 193, 103, 248, 134, 154, 16, 188, 82,
            145, 255, 211, 203, 237, 25, 60, 201, 58, 159, 125, 47, 51,
        ],
        [
            92, 174, 73, 174, 127, 71, 106, 23, 242, 24, 143, 255, 40, 70, 242, 75, 236, 158, 165,
            248, 142, 147, 103, 167, 116, 17, 171, 193, 247, 38, 196, 17,
        ],
        [
            195, 200, 231, 140, 158, 101, 132, 162, 20, 214, 89, 132, 144, 2, 45, 206, 161, 228,
            31, 197, 107, 84, 171, 48, 231, 66, 103, 57, 182, 100, 148, 9,
        ],
        [
            91, 239, 228, 202, 109, 58, 224, 173, 255, 33, 238, 51, 252, 38, 7, 104, 68, 71, 61,
            50, 61, 234, 52, 125, 225, 246, 120, 198, 204, 251, 42, 21,
        ],
    ],
    [
        [
            227, 156, 120, 101, 157, 6, 24, 17, 138, 120, 0, 7, 99, 255, 30, 170, 247, 10, 28, 78,
            237, 183, 120, 0, 36, 150, 114, 5, 52, 27, 118, 43,
        ],
        [
            89, 179, 59, 45, 37, 224, 161, 148, 170, 242, 21, 93, 17, 112, 56, 87, 146, 224, 32,
            16, 205, 72, 120, 24, 201, 226, 101, 138, 140, 191, 35, 60,
        ],
        [
            101, 246, 160, 36, 45, 192, 37, 250, 141, 131, 166, 105, 246, 253, 183, 78, 45, 17,
            224, 198, 219, 4, 184, 41, 159, 25, 227, 42, 184, 110, 168, 47,
        ],
        [
            254, 68, 240, 20, 227, 67, 252, 227, 74, 173, 78, 4, 237, 230, 36, 46, 168, 242, 97,
            165, 41, 179, 234, 114, 25, 118, 188, 115, 134, 176, 144, 56,
        ],
        [
            136, 19, 168, 231, 61, 23, 128, 30, 218, 186, 214, 182, 64, 202, 240, 242, 242, 27,
            190, 68, 139, 239, 118, 209, 45, 102, 252, 42, 203, 47, 190, 46,
        ],
        [
            43, 185, 227, 112, 96, 91, 133, 107, 201, 158, 93, 223, 207, 3, 3, 113, 129, 81, 103,
            112, 200, 142, 133, 213, 205, 249, 61, 35, 223, 149, 94, 46,
        ],
        [
            21, 9, 170, 210, 27, 219, 51, 217, 231, 84, 155, 179, 113, 211, 235, 30, 198, 178, 115,
            212, 243, 76, 239, 4, 80, 205, 216, 210, 216, 162, 54, 16,
        ],
        [
            241, 46, 223, 153, 180, 201, 195, 235, 159, 17, 89, 184, 132, 27, 147, 66, 9, 118, 156,
            32, 63, 60, 231, 176, 17, 181, 136, 199, 185, 156, 85, 49,
        ],
    ],
    [
        [
            16, 63, 29, 234, 36, 248, 169, 46, 117, 122, 16, 213, 80, 112, 195, 180, 35, 236, 79,
            11, 161, 39, 231, 149, 76, 182, 68, 223, 211, 132, 105, 60,
        ],
        [
            90, 185, 155, 250, 193, 255, 242, 96, 46, 32, 211, 85, 104, 146, 242, 203, 136, 215,
            207, 249, 156, 165, 186, 101, 172, 94, 196, 196, 215, 90, 242, 18,
        ],
        [
            197, 35, 232, 78, 110, 178, 187, 0, 86, 140, 77, 150, 2, 34, 10, 28, 36, 222, 5, 126,
            195, 84, 209, 117, 171, 224, 207, 178, 246, 55, 19, 23,
        ],
        [
            7, 92, 45, 201, 159, 244, 184, 81, 84, 15, 16, 47, 29, 88, 220, 15, 84, 114, 193, 25,
            178, 137, 226, 153, 49, 173, 132, 75, 132, 245, 35, 46,
        ],
        [
            72, 45, 4, 235, 41, 35, 82, 6, 47, 12, 192, 121, 4, 244, 109, 216, 187, 189, 132, 96,
            53, 230, 169, 219, 247, 118, 128, 25, 76, 227, 229, 21,
        ],
        [
            11, 120, 128, 125, 181, 136, 90, 219, 125, 105, 13, 176, 160, 59, 54, 96, 25, 244, 122,
            151, 1, 182, 7, 118, 193, 158, 96, 250, 212, 93, 136, 17,
        ],
        [
            133, 10, 151, 153, 67, 142, 158, 17, 179, 30, 169, 96, 248, 213, 174, 250, 243, 228,
            222, 11, 57, 109, 208, 34, 21, 91, 222, 40, 95, 116, 158, 42,
        ],
        [
            144, 89, 121, 13, 187, 123, 87, 232, 28, 81, 15, 255, 80, 167, 8, 245, 134, 94, 176,
            29, 69, 112, 191, 43, 174, 0, 147, 222, 94, 154, 107, 31,
        ],
    ],
    [
        [
            30, 133, 165, 235, 190, 92, 40, 23, 196, 48, 17, 234, 31, 38, 10, 244, 66, 89, 244, 71,
            136, 178, 252, 70, 108, 251, 8, 218, 62, 6, 12, 13,
        ],
        [
            181, 217, 82, 12, 171, 165, 231, 168, 153, 34, 248, 106, 105, 24, 128, 34, 58, 33, 186,
            213, 180, 153, 142, 141, 191, 219, 10, 88, 38, 80, 62, 2,
        ],
        [
            3, 56, 194, 11, 68, 15, 132, 139, 250, 136, 196, 236, 244, 49, 232, 78, 78, 146, 217,
            164, 195, 179, 210, 41, 198, 109, 232, 241, 142, 158, 120, 60,
        ],
        [
            253, 122, 255, 208, 56, 178, 103, 118, 227, 193, 99, 164, 224, 85, 254, 159, 53, 45,
            22, 186, 84, 156, 59, 10, 161, 181, 242, 40, 37, 167, 101, 14,
        ],
        [
            170, 6, 86, 213, 88, 160, 229, 166, 21, 23, 82, 239, 180, 219, 65, 204, 51, 125, 82,
            37, 217, 255, 248, 143, 97, 148, 22, 63, 59, 77, 172, 52,
        ],
        [
            7, 111, 62, 154, 249, 48, 164, 29, 197, 139, 4, 132, 101, 228, 166, 111, 64, 212, 112,
            92, 112, 133, 134, 100, 31, 126, 163, 217, 28, 154, 7, 10,
        ],
        [
            16, 149, 13, 107, 255, 13, 114, 211, 25, 178, 249, 249, 126, 248, 167, 165, 239, 25,
            152, 131, 140, 1, 109, 226, 30, 201, 210, 45, 192, 187, 195, 60,
        ],
        [
            61, 150, 128, 112, 101, 124, 178, 16, 66, 126, 64, 241, 36, 2, 50, 130, 115, 227, 191,
            51, 67, 137, 99, 202, 20, 115, 217, 8, 39, 161, 233, 10,
        ],
    ],
    [
        [
            7, 91, 172, 227, 110, 194, 188, 240, 45, 58, 80, 180, 3, 254, 196, 195, 236, 211, 65,
            120, 5, 170, 255, 3, 147, 252, 203, 94, 252, 67, 122, 45,
        ],
        [
            101, 45, 123, 17, 211, 232, 240, 59, 210, 231, 30, 102, 247, 0, 185, 30, 255, 221, 153,
            5, 173, 39, 166, 43, 26, 10, 173, 70, 17, 22, 169, 37,
        ],
        [
            244, 80, 93, 226, 138, 172, 42, 204, 223, 176, 65, 190, 234, 140, 110, 167, 78, 238,
            121, 100, 83, 24, 22, 229, 43, 230, 35, 182, 193, 93, 241, 13,
        ],
        [
            40, 243, 125, 169, 93, 224, 111, 219, 8, 17, 241, 43, 153, 177, 12, 105, 193, 98, 163,
            245, 54, 245, 241, 129, 42, 96, 240, 40, 196, 125, 248, 24,
        ],
        [
            222, 158, 22, 92, 0, 102, 141, 6, 180, 153, 231, 232, 45, 5, 103, 172, 163, 27, 143,
            240, 1, 243, 81, 122, 56, 142, 106, 205, 104, 219, 225, 40,
        ],
        [
            113, 135, 24, 191, 205, 226, 63, 197, 101, 190, 0, 172, 160, 174, 236, 156, 179, 148,
            128, 21, 127, 99, 19, 124, 27, 134, 120, 87, 48, 237, 244, 8,
        ],
        [
            165, 118, 169, 41, 44, 57, 144, 54, 148, 117, 151, 165, 73, 209, 197, 162, 122, 129,
            197, 73, 137, 248, 65, 137, 77, 187, 30, 54, 229, 62, 134, 9,
        ],
        [
            235, 36, 130, 170, 81, 34, 149, 86, 235, 75, 240, 162, 216, 136, 174, 84, 169, 62, 208,
            43, 212, 112, 204, 11, 242, 100, 111, 183, 20, 81, 103, 21,
        ],
    ],
    [
        [
            69, 81, 105, 6, 179, 69, 132, 229, 251, 152, 83, 207, 7, 150, 148, 76, 118, 19, 77,
            127, 197, 153, 52, 135, 5, 173, 199, 50, 172, 148, 240, 6,
        ],
        [
            169, 73, 220, 22, 85, 117, 145, 40, 30, 80, 252, 140, 241, 71, 106, 187, 255, 19, 109,
            24, 119, 194, 152, 102, 216, 102, 146, 61, 127, 23, 154, 8,
        ],
        [
            67, 230, 84, 76, 200, 132, 151, 89, 246, 255, 51, 98, 217, 150, 32, 12, 179, 115, 164,
            159, 240, 221, 100, 73, 215, 168, 61, 234, 33, 105, 80, 53,
        ],
        [
            155, 123, 14, 229, 134, 242, 1, 66, 61, 136, 191, 36, 194, 117, 41, 185, 52, 239, 241,
            89, 111, 60, 84, 126, 97, 45, 52, 194, 115, 90, 244, 41,
        ],
        [
            47, 224, 100, 95, 184, 27, 152, 43, 14, 105, 24, 229, 40, 94, 231, 253, 136, 67, 97,
            19, 139, 79, 52, 55, 208, 164, 206, 103, 93, 179, 49, 37,
        ],
        [
            137, 28, 107, 166, 19, 3, 10, 137, 179, 248, 144, 129, 62, 138, 174, 40, 237, 112, 154,
            187, 244, 126, 56, 209, 254, 195, 153, 32, 248, 146, 55, 25,
        ],
        [
            28, 146, 146, 196, 134, 69, 193, 92, 102, 51, 189, 99, 24, 109, 22, 88, 189, 170, 218,
            201, 146, 81, 245, 19, 95, 168, 181, 193, 4, 35, 77, 13,
        ],
        [
            231, 152, 17, 26, 137, 49, 95, 140, 143, 240, 151, 215, 118, 240, 153, 94, 214, 99, 17,
            203, 166, 250, 120, 226, 231, 74, 141, 20, 212, 20, 198, 10,
        ],
    ],
    [
        [
            201, 36, 247, 251, 228, 99, 121, 198, 220, 187, 83, 50, 200, 54, 242, 88, 65, 214, 227,
            76, 161, 238, 84, 184, 149, 41, 12, 124, 34, 236, 97, 59,
        ],
        [
            126, 74, 151, 14, 16, 110, 166, 2, 235, 52, 43, 240, 222, 178, 145, 101, 104, 153, 171,
            153, 93, 82, 21, 76, 37, 55, 5, 176, 118, 67, 19, 27,
        ],
        [
            208, 113, 45, 31, 250, 171, 238, 119, 191, 57, 114, 133, 71, 167, 168, 207, 97, 125,
            235, 44, 135, 224, 75, 206, 215, 143, 56, 111, 117, 215, 25, 21,
        ],
        [
            247, 158, 25, 107, 7, 121, 254, 119, 228, 136, 246, 2, 15, 160, 69, 180, 190, 134, 179,
            204, 64, 242, 118, 216, 121, 118, 35, 55, 57, 95, 110, 25,
        ],
        [
            149, 84, 210, 56, 93, 110, 28, 177, 219, 155, 235, 168, 137, 141, 71, 114, 129, 55,
            141, 177, 20, 180, 47, 26, 123, 157, 159, 133, 225, 81, 103, 47,
        ],
        [
            215, 28, 126, 174, 218, 158, 19, 78, 226, 193, 168, 64, 223, 102, 211, 44, 25, 205, 14,
            129, 65, 95, 189, 107, 107, 229, 97, 5, 86, 67, 37, 60,
        ],
        [
            23, 91, 47, 107, 133, 237, 134, 217, 123, 232, 136, 140, 222, 76, 114, 80, 248, 160,
            164, 212, 125, 157, 97, 81, 215, 159, 52, 135, 100, 153, 7, 26,
        ],
        [
            192, 235, 48, 8, 201, 70, 70, 145, 16, 213, 165, 6, 236, 57, 71, 204, 39, 134, 61, 192,
            139, 24, 175, 21, 108, 244, 90, 67, 206, 102, 17, 20,
        ],
    ],
    [
        [
            59, 227, 100, 216, 15, 248, 10, 182, 89, 182, 173, 216, 232, 218, 171, 125, 44, 162,
            99, 204, 165, 177, 157, 62, 219, 33, 124, 190, 160, 248, 193, 60,
        ],
        [
            138, 137, 214, 237, 224, 227, 132, 212, 18, 215, 128, 173, 12, 134, 162, 2, 89, 17, 69,
            88, 50, 193, 21, 158, 85, 194, 228, 207, 209, 15, 48, 56,
        ],
        [
            235, 131, 4, 17, 94, 133, 194, 205, 154, 120, 81, 80, 60, 240, 201, 125, 250, 1, 149,
            179, 58, 228, 68, 71, 224, 15, 212, 181, 170, 9, 135, 47,
        ],
        [
            30, 162, 68, 164, 134, 183, 89, 52, 42, 138, 32, 99, 53, 176, 204, 80, 128, 32, 205,
            64, 177, 197, 186, 93, 239, 127, 63, 219, 19, 200, 132, 5,
        ],
        [
            255, 30, 56, 241, 97, 130, 220, 188, 77, 45, 179, 149, 97, 23, 65, 112, 235, 24, 127,
            22, 204, 191, 181, 164, 28, 238, 158, 188, 52, 52, 239, 9,
        ],
        [
            117, 70, 55, 141, 159, 39, 198, 73, 18, 25, 171, 142, 26, 7, 61, 178, 86, 181, 136, 11,
            80, 110, 226, 235, 152, 116, 119, 80, 9, 77, 171, 1,
        ],
        [
            145, 166, 166, 110, 183, 126, 211, 28, 74, 99, 88, 1, 60, 143, 245, 37, 61, 107, 94,
            172, 101, 87, 64, 150, 89, 87, 23, 159, 223, 49, 13, 22,
        ],
        [
            102, 212, 116, 255, 19, 29, 250, 185, 75, 229, 219, 55, 220, 164, 195, 61, 45, 46, 245,
            119, 79, 113, 47, 229, 242, 74, 172, 24, 189, 248, 222, 35,
        ],
    ],
    [
        [
            178, 128, 66, 22, 136, 89, 154, 163, 11, 60, 6, 113, 119, 121, 188, 80, 127, 170, 10,
            61, 221, 229, 163, 217, 4, 206, 79, 88, 105, 222, 53, 9,
        ],
        [
            185, 239, 123, 80, 135, 216, 219, 192, 225, 181, 44, 19, 223, 11, 150, 61, 21, 13, 170,
            149, 8, 147, 248, 57, 43, 1, 222, 231, 81, 172, 49, 31,
        ],
        [
            88, 32, 154, 70, 204, 46, 188, 217, 185, 179, 122, 163, 239, 139, 136, 119, 231, 117,
            199, 74, 239, 108, 234, 25, 21, 246, 75, 183, 71, 172, 247, 25,
        ],
        [
            143, 4, 10, 225, 34, 215, 168, 147, 44, 185, 131, 183, 230, 6, 125, 173, 71, 57, 255,
            57, 45, 203, 159, 231, 97, 15, 80, 248, 189, 162, 230, 32,
        ],
        [
            166, 187, 119, 117, 197, 168, 67, 253, 47, 170, 125, 165, 17, 15, 149, 44, 146, 58,
            198, 97, 128, 144, 7, 40, 95, 143, 147, 172, 152, 219, 236, 28,
        ],
        [
            28, 176, 50, 123, 16, 49, 98, 25, 185, 200, 78, 192, 135, 9, 17, 249, 80, 80, 148, 70,
            197, 44, 151, 13, 252, 245, 170, 173, 184, 192, 179, 50,
        ],
        [
            7, 203, 97, 103, 251, 87, 201, 89, 46, 123, 27, 243, 73, 93, 22, 209, 139, 173, 145,
            228, 215, 23, 133, 50, 142, 179, 0, 115, 169, 54, 54, 36,
        ],
        [
            103, 67, 224, 228, 216, 51, 67, 191, 241, 190, 231, 167, 239, 122, 145, 135, 27, 27,
            121, 169, 32, 249, 121, 157, 76, 209, 1, 125, 128, 109, 122, 9,
        ],
    ],
    [
        [
            148, 192, 149, 231, 219, 175, 220, 161, 114, 143, 131, 199, 122, 47, 237, 202, 210,
            139, 29, 217, 170, 108, 115, 249, 68, 146, 176, 108, 222, 159, 233, 14,
        ],
        [
            85, 172, 51, 155, 179, 59, 252, 236, 149, 163, 120, 147, 116, 175, 86, 1, 40, 81, 47,
            133, 109, 255, 74, 131, 19, 129, 194, 4, 159, 185, 31, 3,
        ],
        [
            209, 194, 182, 196, 5, 180, 67, 30, 36, 61, 3, 166, 211, 15, 123, 155, 128, 13, 120,
            142, 137, 166, 245, 147, 78, 249, 140, 255, 203, 212, 85, 6,
        ],
        [
            127, 131, 57, 200, 151, 146, 45, 152, 85, 62, 96, 86, 213, 36, 238, 237, 132, 44, 127,
            149, 182, 216, 127, 58, 114, 211, 189, 141, 202, 194, 62, 62,
        ],
        [
            99, 71, 21, 6, 166, 145, 151, 160, 236, 240, 86, 161, 124, 186, 117, 20, 238, 65, 92,
            94, 164, 101, 27, 72, 90, 199, 127, 238, 171, 190, 35, 55,
        ],
        [
            3, 197, 168, 159, 131, 53, 238, 255, 128, 42, 6, 120, 7, 172, 39, 108, 174, 46, 155,
            47, 131, 151, 235, 200, 222, 175, 119, 50, 78, 128, 153, 3,
        ],
        [
            116, 66, 37, 31, 30, 169, 73, 64, 193, 70, 88, 219, 59, 172, 96, 110, 243, 160, 105,
            191, 2, 226, 190, 27, 52, 202, 22, 33, 25, 38, 220, 31,
        ],
        [
            49, 200, 93, 193, 225, 104, 64, 102, 3, 223, 45, 31, 211, 145, 210, 114, 132, 27, 240,
            232, 189, 127, 87, 226, 125, 18, 170, 133, 195, 198, 153, 16,
        ],
    ],
    [
        [
            121, 53, 200, 131, 196, 84, 198, 124, 113, 102, 142, 218, 224, 247, 145, 131, 186, 103,
            190, 167, 247, 147, 224, 16, 99, 39, 56, 226, 46, 1, 49, 59,
        ],
        [
            22, 50, 40, 200, 159, 100, 214, 73, 77, 69, 163, 2, 110, 139, 27, 106, 154, 229, 182,
            137, 7, 168, 187, 202, 17, 19, 160, 25, 162, 17, 89, 48,
        ],
        [
            164, 44, 83, 249, 124, 41, 132, 94, 96, 167, 55, 81, 204, 189, 223, 195, 250, 225, 16,
            205, 0, 3, 130, 36, 165, 36, 166, 132, 92, 91, 67, 63,
        ],
        [
            74, 250, 76, 30, 147, 188, 232, 26, 69, 89, 236, 235, 117, 199, 87, 31, 17, 223, 161,
            198, 244, 125, 157, 12, 237, 115, 191, 191, 71, 169, 248, 8,
        ],
        [
            199, 31, 247, 17, 166, 210, 80, 23, 60, 82, 40, 177, 153, 123, 9, 176, 227, 228, 80,
            110, 191, 176, 172, 159, 234, 227, 133, 97, 160, 201, 8, 45,
        ],
        [
            76, 132, 117, 193, 103, 234, 31, 11, 42, 125, 132, 21, 248, 178, 136, 13, 191, 20, 46,
            6, 95, 231, 230, 95, 29, 223, 70, 180, 205, 75, 70, 29,
        ],
        [
            252, 10, 50, 84, 120, 20, 243, 48, 252, 113, 112, 249, 255, 250, 144, 178, 245, 108,
            209, 169, 102, 254, 1, 34, 254, 141, 201, 39, 18, 66, 213, 37,
        ],
        [
            216, 43, 15, 51, 184, 164, 7, 189, 92, 219, 230, 192, 138, 49, 17, 116, 209, 106, 26,
            208, 1, 207, 209, 130, 200, 28, 194, 68, 159, 151, 92, 59,
        ],
    ],
    [
        [
            130, 227, 156, 234, 103, 13, 183, 24, 209, 108, 219, 46, 240, 126, 63, 210, 226, 125,
            54, 73, 152, 148, 106, 3, 165, 179, 217, 171, 135, 178, 187, 25,
        ],
        [
            249, 123, 230, 251, 192, 186, 61, 109, 88, 101, 236, 21, 188, 172, 133, 102, 200, 159,
            0, 83, 252, 97, 5, 193, 171, 41, 51, 40, 232, 17, 197, 47,
        ],
        [
            230, 209, 96, 28, 233, 75, 176, 21, 83, 187, 216, 27, 42, 240, 129, 127, 124, 226, 24,
            47, 194, 225, 110, 93, 108, 50, 169, 242, 24, 163, 130, 38,
        ],
        [
            133, 62, 95, 11, 93, 161, 179, 247, 32, 25, 89, 217, 173, 147, 63, 62, 162, 228, 178,
            45, 146, 101, 13, 17, 171, 246, 125, 143, 194, 108, 30, 58,
        ],
        [
            246, 216, 189, 144, 11, 240, 154, 125, 249, 154, 103, 59, 40, 198, 29, 207, 162, 186,
            17, 160, 63, 218, 136, 217, 105, 228, 71, 36, 45, 244, 135, 30,
        ],
        [
            219, 147, 219, 20, 99, 77, 100, 59, 113, 73, 90, 255, 236, 117, 72, 139, 43, 36, 149,
            4, 112, 41, 98, 157, 60, 2, 220, 9, 118, 181, 189, 46,
        ],
        [
            4, 109, 33, 143, 207, 231, 44, 77, 113, 223, 222, 247, 110, 52, 82, 202, 123, 213, 168,
            77, 249, 194, 9, 158, 7, 158, 246, 204, 165, 198, 9, 63,
        ],
        [
            84, 44, 128, 238, 129, 148, 169, 143, 203, 17, 195, 37, 152, 129, 162, 175, 39, 140,
            217, 131, 143, 238, 67, 49, 14, 235, 52, 228, 147, 63, 54, 35,
        ],
    ],
    [
        [
            191, 200, 71, 7, 129, 246, 124, 225, 156, 59, 22, 9, 148, 195, 71, 78, 147, 151, 234,
            102, 64, 121, 202, 224, 245, 27, 80, 223, 176, 227, 26, 17,
        ],
        [
            157, 66, 240, 109, 254, 70, 22, 27, 239, 237, 53, 115, 128, 155, 233, 187, 164, 148,
            232, 0, 41, 157, 202, 126, 95, 253, 77, 150, 3, 247, 218, 58,
        ],
        [
            228, 151, 252, 188, 162, 217, 229, 108, 43, 184, 35, 147, 108, 83, 94, 28, 151, 48,
            103, 150, 181, 98, 147, 212, 14, 134, 229, 37, 192, 112, 24, 16,
        ],
        [
            83, 208, 193, 201, 178, 111, 198, 23, 126, 42, 70, 132, 206, 197, 226, 122, 89, 140,
            131, 139, 135, 216, 52, 136, 76, 217, 147, 230, 129, 245, 176, 59,
        ],
        [
            244, 45, 69, 232, 83, 171, 220, 246, 253, 139, 1, 48, 15, 253, 251, 59, 222, 181, 17,
            129, 4, 237, 150, 59, 232, 244, 106, 254, 81, 244, 254, 18,
        ],
        [
            188, 199, 14, 197, 255, 152, 21, 31, 227, 236, 205, 79, 82, 104, 52, 214, 2, 218, 180,
            90, 116, 189, 195, 244, 167, 62, 194, 36, 107, 184, 11, 43,
        ],
        [
            91, 136, 16, 71, 244, 127, 17, 108, 158, 203, 244, 225, 158, 108, 74, 143, 209, 88,
            216, 76, 60, 162, 103, 218, 6, 86, 220, 238, 109, 6, 112, 28,
        ],
        [
            210, 145, 15, 69, 76, 180, 168, 134, 197, 134, 121, 248, 13, 45, 162, 58, 191, 117,
            233, 85, 161, 132, 243, 111, 26, 158, 207, 5, 173, 101, 203, 39,
        ],
    ],
    [
        [
            91, 229, 184, 51, 211, 97, 48, 184, 113, 20, 161, 108, 173, 224, 151, 36, 98, 116, 190,
            236, 239, 208, 45, 65, 41, 218, 22, 78, 21, 159, 68, 26,
        ],
        [
            154, 217, 229, 180, 214, 167, 164, 163, 231, 131, 232, 179, 73, 218, 83, 161, 164, 155,
            38, 217, 38, 83, 173, 123, 185, 195, 136, 28, 222, 105, 248, 44,
        ],
        [
            239, 18, 84, 25, 24, 214, 58, 20, 152, 95, 143, 93, 97, 219, 36, 109, 132, 217, 17, 55,
            49, 1, 18, 121, 131, 145, 4, 217, 250, 134, 88, 14,
        ],
        [
            253, 67, 150, 157, 135, 33, 201, 53, 183, 45, 159, 132, 171, 98, 41, 58, 140, 134, 11,
            145, 25, 241, 71, 213, 88, 135, 55, 47, 244, 112, 235, 43,
        ],
        [
            203, 78, 141, 190, 49, 209, 195, 175, 45, 64, 143, 120, 102, 235, 159, 161, 245, 198,
            29, 126, 9, 86, 92, 60, 63, 99, 217, 169, 217, 25, 97, 57,
        ],
        [
            233, 223, 245, 147, 242, 164, 19, 19, 137, 245, 19, 241, 36, 159, 136, 111, 211, 191,
            185, 125, 216, 176, 166, 33, 218, 175, 100, 57, 180, 168, 33, 21,
        ],
        [
            191, 125, 108, 252, 109, 113, 20, 141, 133, 231, 150, 20, 101, 219, 222, 38, 99, 160,
            59, 92, 97, 181, 71, 115, 51, 165, 119, 92, 118, 44, 127, 60,
        ],
        [
            113, 88, 23, 85, 9, 181, 218, 205, 196, 189, 218, 69, 235, 215, 199, 177, 88, 46, 80,
            177, 124, 126, 188, 209, 80, 154, 254, 177, 249, 6, 34, 45,
        ],
    ],
    [
        [
            49, 171, 43, 141, 243, 56, 203, 14, 8, 164, 117, 204, 211, 2, 98, 215, 247, 219, 40,
            10, 65, 77, 198, 28, 168, 165, 187, 204, 93, 58, 182, 38,
        ],
        [
            76, 41, 75, 64, 104, 7, 90, 39, 75, 212, 84, 97, 109, 16, 221, 105, 175, 57, 203, 170,
            160, 193, 20, 250, 41, 48, 135, 95, 210, 60, 239, 18,
        ],
        [
            11, 48, 231, 185, 24, 69, 14, 2, 119, 209, 228, 104, 103, 135, 156, 250, 144, 116, 172,
            233, 164, 53, 176, 174, 97, 86, 41, 34, 69, 64, 80, 45,
        ],
        [
            206, 62, 81, 203, 222, 175, 138, 11, 37, 103, 16, 242, 2, 233, 61, 206, 100, 119, 153,
            92, 27, 168, 76, 186, 61, 115, 161, 236, 188, 105, 53, 39,
        ],
        [
            184, 121, 255, 112, 144, 174, 21, 210, 228, 37, 104, 55, 37, 190, 74, 219, 138, 97,
            143, 89, 102, 179, 145, 166, 45, 48, 254, 158, 171, 154, 179, 29,
        ],
        [
            6, 225, 64, 102, 192, 218, 125, 198, 151, 71, 85, 123, 214, 160, 161, 13, 126, 187,
            146, 117, 142, 61, 67, 33, 247, 101, 92, 227, 237, 171, 178, 46,
        ],
        [
            214, 175, 191, 1, 36, 44, 95, 190, 226, 11, 238, 183, 89, 58, 32, 159, 7, 43, 143, 126,
            30, 154, 77, 11, 225, 220, 100, 206, 57, 69, 147, 59,
        ],
        [
            132, 71, 64, 13, 36, 95, 197, 166, 181, 130, 188, 161, 21, 134, 156, 60, 229, 202, 115,
            252, 142, 193, 144, 9, 148, 49, 152, 217, 94, 10, 200, 16,
        ],
    ],
    [
        [
            169, 185, 107, 159, 236, 0, 234, 249, 163, 24, 243, 22, 89, 24, 72, 31, 99, 20, 210,
            52, 203, 224, 58, 131, 135, 126, 139, 241, 11, 223, 180, 41,
        ],
        [
            100, 111, 163, 64, 52, 18, 196, 37, 86, 14, 150, 151, 192, 192, 213, 66, 5, 175, 208,
            15, 99, 225, 99, 23, 255, 115, 41, 53, 244, 241, 232, 62,
        ],
        [
            203, 172, 162, 221, 207, 144, 7, 120, 39, 125, 39, 231, 4, 167, 167, 147, 197, 4, 221,
            168, 11, 32, 252, 185, 253, 56, 198, 125, 180, 217, 48, 14,
        ],
        [
            151, 198, 41, 131, 85, 85, 153, 167, 214, 190, 207, 57, 232, 116, 153, 110, 242, 143,
            87, 149, 148, 79, 202, 50, 254, 93, 181, 118, 241, 36, 47, 30,
        ],
        [
            169, 132, 214, 252, 163, 226, 40, 236, 102, 76, 39, 197, 196, 64, 76, 131, 131, 239,
            125, 186, 226, 201, 133, 168, 63, 255, 217, 34, 32, 14, 87, 27,
        ],
        [
            152, 171, 139, 173, 78, 7, 26, 26, 41, 52, 127, 102, 29, 98, 192, 1, 192, 245, 215,
            144, 80, 63, 44, 223, 64, 223, 246, 42, 138, 196, 82, 56,
        ],
        [
            181, 29, 217, 137, 178, 103, 44, 244, 146, 108, 167, 194, 209, 98, 153, 191, 200, 209,
            197, 121, 30, 99, 180, 251, 16, 70, 185, 48, 28, 225, 29, 4,
        ],
        [
            6, 12, 109, 251, 112, 60, 35, 12, 42, 207, 70, 3, 142, 241, 196, 126, 220, 203, 68,
            249, 158, 166, 46, 67, 32, 23, 192, 239, 36, 44, 206, 25,
        ],
    ],
    [
        [
            94, 19, 121, 58, 184, 157, 237, 59, 227, 107, 130, 65, 168, 219, 76, 58, 226, 235, 146,
            109, 35, 2, 8, 52, 110, 145, 135, 173, 45, 18, 40, 18,
        ],
        [
            73, 20, 152, 96, 153, 114, 82, 42, 234, 214, 83, 47, 117, 62, 120, 141, 43, 72, 95, 94,
            222, 242, 227, 94, 53, 78, 58, 66, 227, 164, 108, 36,
        ],
        [
            15, 203, 0, 206, 129, 153, 3, 46, 221, 28, 232, 197, 225, 202, 69, 64, 15, 85, 203, 4,
            165, 83, 252, 98, 130, 126, 118, 166, 132, 113, 254, 3,
        ],
        [
            27, 247, 153, 83, 213, 195, 61, 103, 248, 197, 36, 170, 167, 29, 42, 61, 58, 108, 9,
            162, 63, 63, 78, 234, 102, 112, 223, 115, 109, 89, 205, 11,
        ],
        [
            230, 87, 100, 238, 166, 118, 76, 72, 206, 16, 128, 19, 196, 235, 19, 85, 25, 180, 143,
            88, 79, 67, 74, 18, 42, 0, 182, 167, 5, 82, 141, 58,
        ],
        [
            242, 225, 221, 181, 10, 42, 4, 246, 169, 149, 20, 171, 248, 249, 181, 206, 171, 85,
            200, 89, 186, 144, 18, 113, 211, 76, 54, 13, 175, 5, 23, 57,
        ],
        [
            124, 90, 7, 105, 198, 10, 101, 152, 215, 14, 162, 242, 28, 138, 77, 181, 216, 134, 209,
            125, 207, 40, 23, 247, 132, 101, 202, 184, 108, 234, 30, 23,
        ],
        [
            113, 101, 162, 46, 79, 205, 213, 191, 24, 206, 128, 228, 42, 66, 147, 106, 8, 211, 6,
            57, 131, 108, 24, 46, 80, 206, 180, 58, 149, 204, 118, 61,
        ],
    ],
    [
        [
            251, 19, 185, 65, 96, 235, 128, 183, 255, 41, 247, 142, 148, 237, 175, 76, 168, 5, 48,
            90, 246, 162, 79, 83, 10, 170, 36, 227, 89, 13, 167, 62,
        ],
        [
            56, 166, 248, 151, 90, 187, 224, 187, 156, 41, 254, 99, 77, 25, 30, 149, 121, 164, 64,
            17, 55, 129, 219, 217, 125, 235, 138, 13, 220, 168, 152, 1,
        ],
        [
            165, 106, 214, 186, 223, 37, 21, 54, 151, 110, 55, 189, 197, 11, 61, 209, 156, 123,
            169, 180, 84, 10, 149, 134, 72, 186, 24, 177, 182, 246, 164, 19,
        ],
        [
            73, 95, 93, 223, 114, 216, 130, 77, 25, 212, 228, 134, 143, 89, 68, 90, 11, 213, 48,
            205, 242, 51, 247, 151, 216, 45, 159, 152, 184, 225, 221, 41,
        ],
        [
            40, 171, 76, 37, 66, 223, 82, 113, 211, 247, 233, 118, 209, 240, 160, 244, 13, 51, 194,
            152, 58, 168, 231, 12, 185, 25, 186, 175, 64, 12, 231, 6,
        ],
        [
            179, 90, 55, 174, 72, 152, 204, 148, 24, 250, 29, 50, 143, 98, 133, 199, 241, 225, 133,
            171, 33, 49, 178, 167, 35, 162, 79, 159, 26, 232, 0, 8,
        ],
        [
            125, 6, 204, 17, 50, 201, 101, 165, 28, 107, 109, 206, 143, 215, 105, 123, 175, 164,
            63, 159, 21, 11, 234, 96, 159, 43, 88, 18, 159, 33, 88, 38,
        ],
        [
            221, 140, 167, 49, 28, 116, 206, 121, 79, 136, 80, 60, 181, 7, 95, 177, 32, 126, 87,
            128, 168, 124, 129, 166, 189, 214, 255, 44, 221, 83, 103, 48,
        ],
    ],
    [
        [
            100, 19, 245, 99, 127, 23, 39, 14, 128, 145, 57, 217, 20, 253, 209, 74, 223, 24, 220,
            142, 19, 167, 130, 239, 220, 226, 211, 66, 177, 44, 34, 25,
        ],
        [
            253, 114, 38, 89, 223, 86, 164, 50, 35, 154, 30, 159, 194, 152, 119, 36, 118, 25, 244,
            85, 63, 29, 152, 56, 150, 14, 250, 231, 157, 94, 53, 1,
        ],
        [
            122, 133, 104, 47, 162, 156, 2, 203, 65, 211, 188, 21, 95, 196, 233, 189, 85, 11, 82,
            62, 121, 185, 128, 178, 117, 221, 41, 246, 177, 224, 168, 53,
        ],
        [
            15, 47, 113, 92, 78, 129, 107, 39, 106, 27, 10, 4, 193, 207, 48, 209, 184, 128, 128,
            54, 167, 37, 99, 169, 88, 131, 209, 43, 191, 232, 246, 43,
        ],
        [
            151, 100, 55, 80, 35, 25, 142, 78, 150, 69, 94, 12, 237, 225, 126, 108, 106, 133, 11,
            118, 58, 8, 195, 87, 124, 129, 2, 8, 228, 8, 235, 41,
        ],
        [
            154, 75, 9, 198, 211, 209, 244, 131, 49, 126, 187, 157, 54, 92, 229, 166, 249, 72, 229,
            168, 92, 24, 17, 154, 4, 213, 211, 179, 26, 60, 115, 57,
        ],
        [
            148, 168, 89, 36, 108, 105, 22, 194, 209, 232, 199, 64, 79, 100, 96, 148, 205, 137, 20,
            98, 107, 79, 12, 200, 56, 4, 93, 151, 129, 18, 90, 2,
        ],
        [
            173, 118, 232, 220, 163, 182, 149, 237, 184, 151, 108, 87, 139, 99, 14, 144, 174, 120,
            101, 207, 198, 189, 108, 99, 145, 73, 114, 141, 90, 97, 136, 36,
        ],
    ],
];

pub fn generator() -> pallas::Affine {
    pallas::Affine::from_xy(
        pallas::Base::from_bytes(&GENERATOR.0).unwrap(),
        pallas::Base::from_bytes(&GENERATOR.1).unwrap(),
    )
    .unwrap()
}

#[cfg(test)]
mod tests {
    use super::super::{
        test_lagrange_coeffs, test_zs_and_us, NUM_WINDOWS, ORCHARD_PERSONALIZATION,
    };
    use super::*;
    use group::Curve;
    use pasta_curves::{
        arithmetic::{CurveAffine, CurveExt, FieldExt},
        pallas,
    };

    #[test]
    fn generator() {
        let hasher = pallas::Point::hash_to_curve(ORCHARD_PERSONALIZATION);
        let point = hasher(b"G");
        let coords = point.to_affine().coordinates().unwrap();

        assert_eq!(*coords.x(), pallas::Base::from_bytes(&GENERATOR.0).unwrap());
        assert_eq!(*coords.y(), pallas::Base::from_bytes(&GENERATOR.1).unwrap());
    }

    #[test]
    fn lagrange_coeffs() {
        let base = super::generator();
        test_lagrange_coeffs(base, NUM_WINDOWS);
    }

    #[test]
    fn z() {
        let base = super::generator();
        test_zs_and_us(base, &Z, &U, NUM_WINDOWS);
    }
}
