use pasta_curves::{
    arithmetic::{CurveAffine, FieldExt},
    pallas,
};

pub const GENERATOR: ([u8; 32], [u8; 32]) = (
    [
        117, 202, 71, 228, 167, 106, 111, 211, 155, 219, 181, 204, 146, 177, 126, 94, 207, 201,
        244, 250, 113, 85, 55, 46, 141, 25, 168, 156, 22, 170, 231, 37,
    ],
    [
        204, 1, 224, 26, 226, 220, 184, 45, 51, 180, 193, 193, 158, 164, 224, 95, 117, 255, 8, 32,
        68, 115, 4, 136, 132, 51, 26, 27, 133, 31, 92, 21,
    ],
);

/// Full-width z-values for GENERATOR
pub const Z: [u64; super::NUM_WINDOWS] = [
    34374, 173069, 40776, 220066, 45494, 37762, 5245, 11979, 33386, 238556, 128731, 12128, 89982,
    85351, 9804, 12820, 80455, 100009, 24382, 17854, 26367, 7067, 102106, 64293, 114999, 172304,
    36687, 11287, 66386, 41470, 182654, 12214, 36528, 16257, 26179, 15660, 106189, 211703, 12936,
    2506, 149799, 82965, 117810, 98881, 296, 146201, 63200, 31766, 78221, 6587, 27974, 126041,
    19927, 79339, 210060, 127148, 10109, 19815, 107452, 10296, 642, 11828, 3985, 2984, 30806,
    12554, 1815, 19894, 16790, 33748, 12879, 1742, 30858, 118563, 26855, 75617, 10167, 17660,
    33638, 89236, 50234, 30489, 67488, 50229, 29277,
];

/// Full-width u-values for GENERATOR
pub const U: [[[u8; 32]; super::H]; super::NUM_WINDOWS] = [
    [
        [
            8, 177, 117, 90, 104, 127, 169, 131, 254, 128, 23, 151, 181, 247, 22, 208, 55, 1, 147,
            216, 84, 35, 129, 16, 134, 195, 231, 78, 90, 27, 48, 24,
        ],
        [
            85, 52, 30, 137, 203, 92, 107, 164, 157, 0, 160, 220, 77, 151, 245, 43, 236, 219, 6,
            127, 136, 136, 255, 80, 178, 129, 223, 223, 100, 205, 206, 60,
        ],
        [
            180, 42, 81, 35, 92, 89, 115, 34, 230, 154, 57, 180, 27, 241, 82, 7, 29, 18, 87, 188,
            15, 115, 37, 92, 226, 71, 117, 25, 58, 188, 84, 2,
        ],
        [
            120, 152, 127, 75, 237, 238, 241, 80, 87, 202, 187, 80, 67, 210, 115, 228, 187, 183,
            190, 104, 143, 201, 79, 59, 88, 235, 35, 30, 112, 142, 245, 41,
        ],
        [
            137, 203, 204, 3, 111, 36, 152, 243, 202, 11, 171, 157, 74, 86, 136, 224, 134, 228, 83,
            142, 77, 3, 87, 124, 174, 254, 51, 139, 199, 223, 108, 33,
        ],
        [
            253, 61, 90, 128, 49, 47, 137, 144, 47, 195, 161, 160, 78, 225, 222, 129, 15, 194, 150,
            16, 41, 55, 82, 238, 34, 87, 98, 170, 213, 158, 15, 50,
        ],
        [
            47, 235, 116, 15, 252, 144, 163, 83, 0, 21, 138, 234, 11, 181, 189, 10, 246, 121, 216,
            134, 47, 95, 213, 3, 238, 46, 179, 91, 217, 89, 45, 51,
        ],
        [
            14, 37, 210, 28, 30, 181, 151, 46, 9, 92, 16, 9, 143, 179, 209, 165, 243, 236, 31, 73,
            1, 41, 26, 125, 225, 42, 241, 153, 7, 72, 154, 28,
        ],
    ],
    [
        [
            155, 23, 107, 105, 142, 88, 223, 1, 145, 75, 68, 196, 138, 223, 209, 168, 178, 183,
            154, 186, 235, 94, 192, 81, 170, 238, 26, 126, 159, 204, 87, 18,
        ],
        [
            23, 143, 212, 151, 42, 93, 52, 146, 4, 240, 71, 229, 183, 45, 75, 117, 31, 86, 162,
            145, 194, 110, 118, 145, 36, 1, 110, 255, 119, 221, 191, 4,
        ],
        [
            43, 199, 119, 40, 42, 43, 118, 185, 214, 192, 11, 20, 35, 124, 64, 65, 126, 39, 23,
            136, 102, 196, 220, 99, 190, 252, 163, 189, 215, 110, 5, 26,
        ],
        [
            231, 221, 137, 18, 190, 54, 27, 37, 198, 161, 251, 131, 26, 85, 24, 253, 119, 85, 63,
            73, 4, 205, 90, 160, 68, 17, 124, 250, 142, 144, 165, 7,
        ],
        [
            83, 50, 206, 53, 156, 115, 230, 212, 31, 155, 204, 33, 132, 233, 228, 213, 249, 252,
            224, 9, 157, 163, 184, 220, 25, 118, 167, 180, 145, 86, 59, 27,
        ],
        [
            245, 214, 106, 201, 143, 76, 98, 94, 184, 16, 72, 167, 42, 124, 18, 182, 124, 0, 122,
            151, 24, 228, 104, 144, 81, 240, 105, 126, 158, 79, 213, 60,
        ],
        [
            117, 59, 119, 9, 69, 161, 159, 137, 154, 195, 28, 54, 240, 214, 26, 97, 138, 121, 199,
            4, 28, 17, 159, 186, 52, 216, 186, 6, 65, 17, 155, 53,
        ],
        [
            131, 74, 9, 131, 251, 243, 140, 39, 209, 183, 173, 30, 223, 144, 252, 210, 71, 196,
            218, 92, 149, 114, 87, 13, 176, 167, 135, 21, 79, 51, 166, 21,
        ],
    ],
    [
        [
            106, 53, 57, 143, 70, 101, 179, 227, 157, 251, 249, 89, 23, 241, 33, 172, 138, 109,
            112, 10, 239, 116, 201, 204, 70, 88, 47, 122, 115, 244, 190, 1,
        ],
        [
            209, 215, 139, 155, 229, 13, 70, 83, 197, 215, 97, 80, 247, 86, 167, 76, 230, 225, 158,
            160, 192, 121, 46, 212, 197, 12, 107, 161, 225, 174, 226, 6,
        ],
        [
            112, 160, 94, 22, 110, 32, 8, 135, 32, 51, 220, 110, 15, 225, 140, 230, 76, 199, 123,
            151, 87, 211, 19, 62, 21, 129, 16, 60, 12, 108, 165, 31,
        ],
        [
            204, 30, 64, 88, 48, 86, 118, 103, 250, 219, 102, 20, 56, 126, 52, 226, 0, 61, 113,
            192, 116, 132, 99, 78, 111, 44, 138, 247, 154, 143, 219, 27,
        ],
        [
            234, 175, 176, 17, 157, 210, 254, 209, 26, 205, 38, 46, 13, 146, 162, 67, 133, 126, 0,
            137, 173, 137, 151, 97, 177, 226, 59, 43, 155, 198, 51, 45,
        ],
        [
            197, 96, 86, 193, 80, 242, 4, 255, 18, 52, 82, 105, 29, 253, 234, 3, 160, 233, 99, 224,
            58, 44, 97, 40, 194, 99, 173, 127, 62, 47, 208, 37,
        ],
        [
            22, 149, 249, 163, 1, 218, 101, 70, 94, 230, 144, 57, 241, 33, 6, 232, 196, 115, 108,
            223, 76, 188, 1, 17, 42, 214, 155, 247, 149, 148, 238, 35,
        ],
        [
            13, 87, 74, 130, 84, 86, 209, 2, 50, 148, 189, 154, 138, 103, 165, 188, 10, 34, 149,
            21, 75, 83, 63, 88, 162, 137, 118, 232, 60, 157, 137, 53,
        ],
    ],
    [
        [
            233, 74, 133, 175, 252, 244, 95, 128, 64, 52, 236, 30, 62, 251, 20, 39, 136, 210, 13,
            125, 220, 122, 12, 168, 132, 0, 203, 121, 166, 114, 89, 55,
        ],
        [
            23, 16, 81, 254, 130, 74, 96, 191, 239, 107, 112, 232, 193, 231, 115, 168, 222, 76,
            200, 25, 29, 74, 120, 123, 219, 38, 10, 170, 159, 38, 151, 59,
        ],
        [
            123, 54, 39, 213, 60, 136, 244, 129, 49, 166, 51, 223, 82, 7, 141, 210, 150, 80, 71,
            91, 48, 195, 52, 32, 17, 149, 25, 63, 179, 211, 116, 44,
        ],
        [
            109, 212, 117, 221, 214, 8, 16, 202, 176, 94, 85, 170, 181, 234, 43, 86, 151, 193, 42,
            9, 185, 141, 187, 197, 32, 149, 233, 82, 135, 196, 226, 55,
        ],
        [
            167, 160, 99, 140, 33, 119, 82, 182, 166, 50, 48, 229, 46, 170, 178, 22, 52, 2, 60,
            138, 235, 97, 75, 141, 32, 166, 120, 45, 103, 212, 212, 54,
        ],
        [
            127, 109, 31, 40, 88, 39, 93, 49, 112, 214, 201, 84, 181, 82, 212, 26, 236, 67, 108,
            50, 59, 188, 242, 182, 163, 230, 153, 128, 207, 86, 214, 18,
        ],
        [
            246, 232, 168, 240, 223, 17, 25, 238, 68, 106, 81, 83, 149, 255, 179, 85, 215, 221,
            161, 98, 211, 255, 34, 100, 244, 171, 34, 77, 13, 97, 7, 31,
        ],
        [
            203, 28, 143, 74, 174, 27, 47, 59, 154, 247, 128, 8, 70, 108, 73, 167, 25, 171, 132,
            99, 192, 80, 53, 69, 93, 132, 35, 224, 141, 157, 182, 27,
        ],
    ],
    [
        [
            158, 175, 208, 26, 183, 200, 147, 77, 15, 124, 87, 17, 47, 238, 228, 184, 132, 87, 24,
            159, 255, 90, 200, 42, 118, 110, 207, 49, 100, 156, 40, 17,
        ],
        [
            214, 80, 143, 90, 122, 14, 85, 237, 43, 157, 103, 184, 14, 213, 121, 149, 38, 12, 185,
            162, 65, 76, 175, 234, 33, 6, 128, 70, 87, 64, 121, 32,
        ],
        [
            239, 204, 8, 172, 229, 121, 210, 167, 213, 113, 34, 212, 58, 196, 246, 231, 235, 249,
            10, 1, 133, 180, 234, 122, 185, 209, 137, 42, 151, 187, 90, 49,
        ],
        [
            38, 201, 204, 153, 118, 114, 213, 247, 168, 32, 231, 172, 56, 74, 104, 185, 82, 10,
            189, 39, 0, 120, 76, 60, 187, 40, 207, 49, 112, 7, 226, 13,
        ],
        [
            49, 138, 24, 179, 17, 17, 139, 34, 66, 7, 206, 250, 69, 221, 34, 216, 233, 153, 230,
            61, 129, 139, 206, 131, 48, 182, 148, 13, 71, 243, 111, 22,
        ],
        [
            196, 88, 22, 46, 64, 213, 113, 115, 213, 31, 243, 86, 240, 86, 2, 152, 250, 26, 105,
            171, 184, 61, 159, 95, 65, 29, 25, 78, 240, 245, 10, 21,
        ],
        [
            78, 28, 8, 195, 230, 66, 59, 242, 40, 196, 240, 152, 11, 217, 164, 30, 69, 14, 45, 198,
            239, 214, 233, 45, 101, 211, 98, 235, 100, 118, 55, 4,
        ],
        [
            78, 224, 209, 126, 212, 30, 129, 15, 115, 153, 242, 115, 131, 124, 100, 161, 132, 191,
            92, 106, 202, 67, 75, 214, 38, 188, 240, 124, 54, 224, 93, 45,
        ],
    ],
    [
        [
            217, 244, 110, 79, 182, 134, 95, 114, 224, 131, 47, 250, 120, 210, 60, 25, 170, 192,
            236, 37, 233, 77, 18, 29, 113, 70, 184, 210, 226, 211, 36, 23,
        ],
        [
            97, 25, 216, 18, 187, 188, 60, 255, 57, 173, 195, 70, 147, 75, 123, 96, 32, 186, 235,
            203, 168, 246, 177, 206, 183, 164, 219, 209, 70, 100, 221, 31,
        ],
        [
            103, 65, 104, 143, 235, 96, 30, 14, 136, 100, 168, 208, 183, 133, 61, 209, 88, 171,
            234, 144, 128, 186, 154, 57, 115, 217, 218, 65, 208, 15, 164, 5,
        ],
        [
            129, 17, 125, 139, 223, 114, 198, 101, 118, 190, 54, 211, 209, 193, 119, 254, 191, 151,
            210, 201, 167, 151, 199, 55, 144, 146, 28, 125, 15, 140, 64, 50,
        ],
        [
            157, 114, 191, 73, 254, 85, 156, 148, 193, 78, 64, 118, 237, 231, 95, 96, 156, 223,
            251, 161, 108, 104, 68, 68, 208, 108, 52, 36, 193, 20, 209, 39,
        ],
        [
            153, 221, 73, 244, 97, 41, 144, 181, 6, 255, 117, 181, 73, 60, 182, 130, 126, 179, 61,
            86, 141, 96, 33, 233, 132, 47, 145, 215, 219, 54, 147, 11,
        ],
        [
            94, 107, 185, 219, 70, 23, 205, 149, 216, 21, 53, 23, 98, 243, 185, 235, 221, 198, 49,
            177, 48, 56, 251, 180, 141, 119, 212, 20, 227, 194, 64, 19,
        ],
        [
            85, 148, 121, 226, 36, 182, 97, 6, 122, 1, 31, 39, 92, 65, 36, 147, 205, 69, 204, 90,
            204, 166, 179, 223, 5, 125, 68, 63, 64, 128, 63, 21,
        ],
    ],
    [
        [
            2, 169, 25, 203, 212, 185, 124, 61, 150, 239, 166, 57, 177, 48, 11, 166, 89, 251, 110,
            226, 49, 69, 226, 224, 250, 219, 252, 137, 18, 129, 43, 16,
        ],
        [
            232, 38, 209, 98, 87, 176, 166, 69, 137, 12, 213, 116, 97, 229, 25, 53, 64, 231, 213,
            255, 182, 234, 179, 255, 159, 162, 70, 169, 254, 193, 83, 31,
        ],
        [
            120, 35, 170, 169, 134, 229, 114, 38, 97, 2, 84, 36, 156, 80, 255, 130, 168, 156, 195,
            7, 97, 120, 214, 215, 145, 152, 211, 201, 5, 36, 62, 32,
        ],
        [
            57, 117, 255, 121, 168, 160, 43, 168, 42, 239, 58, 130, 73, 244, 242, 252, 9, 27, 79,
            245, 226, 201, 189, 164, 57, 142, 150, 162, 209, 17, 121, 53,
        ],
        [
            36, 52, 2, 158, 240, 153, 97, 93, 33, 133, 153, 174, 3, 23, 116, 181, 157, 223, 70, 59,
            100, 101, 102, 195, 6, 148, 63, 228, 72, 6, 27, 35,
        ],
        [
            52, 197, 226, 140, 56, 213, 200, 104, 117, 7, 101, 250, 224, 176, 188, 134, 69, 59, 23,
            121, 181, 197, 236, 229, 24, 199, 79, 243, 120, 143, 120, 4,
        ],
        [
            148, 181, 32, 129, 234, 255, 140, 131, 155, 227, 16, 106, 1, 185, 148, 195, 165, 0,
            152, 130, 138, 134, 208, 12, 171, 88, 26, 142, 46, 36, 140, 52,
        ],
        [
            184, 10, 44, 43, 76, 195, 89, 67, 120, 15, 19, 199, 96, 229, 109, 115, 237, 50, 51,
            156, 230, 142, 3, 95, 109, 79, 20, 130, 54, 41, 225, 62,
        ],
    ],
    [
        [
            171, 227, 38, 163, 220, 64, 126, 70, 193, 140, 185, 136, 97, 117, 204, 177, 109, 139,
            25, 38, 25, 5, 27, 31, 169, 30, 18, 188, 243, 99, 125, 29,
        ],
        [
            22, 191, 88, 67, 150, 57, 68, 66, 95, 65, 24, 208, 188, 6, 1, 8, 62, 195, 83, 224, 113,
            136, 61, 75, 90, 155, 228, 209, 155, 239, 250, 52,
        ],
        [
            64, 110, 241, 54, 230, 252, 13, 172, 5, 83, 159, 231, 249, 166, 47, 196, 203, 230, 82,
            234, 108, 75, 104, 217, 140, 93, 142, 42, 54, 10, 218, 13,
        ],
        [
            119, 77, 69, 84, 160, 177, 65, 143, 32, 247, 4, 101, 44, 47, 38, 175, 53, 8, 239, 115,
            177, 68, 166, 173, 245, 168, 188, 129, 49, 220, 136, 0,
        ],
        [
            95, 4, 124, 89, 22, 199, 81, 164, 42, 73, 149, 244, 67, 141, 249, 132, 13, 55, 180, 2,
            70, 238, 186, 129, 130, 196, 191, 36, 188, 89, 173, 3,
        ],
        [
            185, 209, 66, 253, 215, 62, 233, 72, 67, 105, 81, 154, 41, 127, 66, 152, 203, 206, 232,
            49, 28, 14, 132, 182, 245, 30, 42, 158, 61, 84, 55, 34,
        ],
        [
            168, 36, 132, 217, 188, 230, 252, 8, 63, 232, 169, 205, 203, 35, 80, 7, 166, 179, 165,
            178, 34, 108, 179, 252, 12, 77, 183, 64, 21, 134, 195, 58,
        ],
        [
            167, 154, 142, 35, 246, 107, 255, 85, 225, 7, 179, 39, 11, 163, 211, 139, 38, 5, 130,
            124, 212, 111, 10, 235, 206, 93, 247, 118, 97, 74, 76, 3,
        ],
    ],
    [
        [
            17, 35, 231, 232, 187, 165, 56, 74, 129, 69, 48, 249, 182, 107, 89, 163, 51, 84, 170,
            211, 242, 113, 166, 225, 78, 206, 206, 92, 51, 71, 168, 33,
        ],
        [
            218, 32, 99, 108, 201, 243, 16, 42, 119, 238, 180, 204, 217, 64, 11, 181, 159, 187,
            222, 68, 31, 23, 215, 14, 135, 43, 60, 234, 199, 136, 209, 56,
        ],
        [
            229, 89, 20, 107, 93, 96, 226, 179, 225, 22, 59, 133, 30, 79, 105, 26, 42, 251, 35, 41,
            45, 151, 73, 127, 72, 33, 32, 44, 114, 183, 200, 12,
        ],
        [
            0, 214, 119, 87, 158, 203, 58, 209, 100, 45, 242, 75, 5, 118, 29, 75, 239, 159, 30,
            192, 196, 58, 48, 65, 129, 192, 13, 14, 115, 79, 219, 26,
        ],
        [
            98, 88, 236, 178, 70, 249, 24, 151, 194, 204, 95, 251, 142, 94, 94, 218, 19, 253, 216,
            155, 109, 25, 166, 168, 213, 209, 169, 135, 244, 2, 153, 63,
        ],
        [
            56, 137, 153, 59, 70, 236, 157, 224, 248, 77, 167, 238, 185, 216, 239, 54, 176, 54,
            164, 172, 129, 29, 243, 226, 56, 188, 149, 113, 66, 53, 111, 29,
        ],
        [
            175, 39, 133, 206, 20, 108, 5, 54, 187, 154, 174, 144, 23, 193, 58, 173, 223, 19, 181,
            161, 102, 248, 146, 180, 241, 58, 121, 96, 8, 218, 201, 61,
        ],
        [
            214, 255, 102, 226, 32, 197, 112, 142, 159, 244, 213, 201, 228, 8, 141, 112, 112, 6,
            182, 31, 68, 215, 126, 66, 131, 203, 253, 206, 102, 155, 100, 9,
        ],
    ],
    [
        [
            134, 183, 10, 76, 61, 159, 117, 15, 17, 79, 26, 161, 72, 129, 77, 56, 221, 0, 23, 32,
            255, 12, 114, 208, 110, 133, 136, 7, 222, 63, 99, 63,
        ],
        [
            144, 188, 53, 201, 167, 105, 34, 93, 26, 15, 121, 27, 37, 68, 230, 134, 119, 6, 76,
            159, 120, 62, 200, 59, 205, 110, 251, 54, 132, 113, 216, 47,
        ],
        [
            226, 242, 186, 127, 110, 179, 100, 138, 238, 127, 104, 131, 32, 98, 96, 244, 121, 154,
            129, 71, 34, 157, 103, 53, 124, 131, 128, 159, 113, 77, 226, 52,
        ],
        [
            149, 188, 192, 214, 255, 107, 1, 216, 118, 134, 46, 23, 126, 73, 234, 175, 39, 77, 232,
            99, 36, 76, 27, 215, 134, 227, 12, 255, 98, 120, 110, 13,
        ],
        [
            55, 19, 111, 30, 3, 196, 248, 249, 103, 123, 237, 222, 156, 152, 143, 182, 93, 209,
            227, 184, 97, 233, 110, 31, 198, 108, 2, 144, 202, 161, 246, 48,
        ],
        [
            89, 105, 183, 28, 13, 186, 20, 45, 187, 147, 254, 9, 196, 222, 10, 212, 208, 35, 59,
            168, 234, 91, 53, 115, 119, 28, 240, 140, 226, 217, 16, 13,
        ],
        [
            71, 90, 196, 184, 172, 101, 82, 49, 204, 76, 80, 24, 30, 132, 224, 136, 92, 236, 73,
            240, 43, 152, 36, 240, 204, 168, 202, 133, 47, 56, 86, 21,
        ],
        [
            137, 123, 99, 242, 15, 214, 23, 223, 103, 70, 184, 82, 212, 43, 109, 198, 1, 146, 88,
            223, 141, 176, 98, 56, 8, 226, 123, 144, 58, 148, 80, 34,
        ],
    ],
    [
        [
            112, 203, 118, 169, 67, 201, 121, 80, 95, 50, 206, 193, 132, 19, 44, 16, 100, 210, 169,
            174, 43, 239, 197, 42, 153, 77, 189, 127, 134, 228, 51, 1,
        ],
        [
            67, 82, 109, 220, 119, 200, 223, 134, 201, 18, 211, 184, 106, 158, 243, 17, 160, 51,
            223, 227, 252, 83, 229, 84, 109, 162, 231, 240, 207, 99, 145, 49,
        ],
        [
            127, 38, 205, 72, 71, 81, 43, 26, 72, 50, 220, 151, 66, 66, 41, 27, 99, 47, 96, 142, 1,
            83, 210, 246, 201, 190, 8, 34, 223, 215, 84, 33,
        ],
        [
            196, 65, 248, 47, 253, 158, 13, 126, 69, 77, 144, 88, 158, 188, 200, 138, 70, 7, 237,
            81, 207, 18, 52, 77, 131, 187, 61, 208, 53, 15, 87, 28,
        ],
        [
            142, 69, 186, 111, 164, 34, 17, 159, 95, 229, 106, 79, 178, 236, 40, 122, 205, 228, 56,
            78, 187, 203, 150, 216, 228, 220, 77, 102, 101, 57, 73, 22,
        ],
        [
            34, 97, 52, 100, 250, 150, 39, 93, 193, 188, 241, 25, 57, 178, 210, 228, 116, 102, 120,
            194, 43, 92, 58, 223, 79, 172, 233, 206, 42, 252, 212, 23,
        ],
        [
            255, 63, 76, 185, 74, 23, 208, 38, 220, 177, 232, 82, 75, 190, 205, 7, 107, 83, 37, 46,
            184, 118, 15, 69, 3, 148, 140, 160, 31, 107, 64, 19,
        ],
        [
            12, 74, 20, 81, 11, 141, 102, 144, 211, 239, 203, 214, 114, 193, 197, 27, 220, 229,
            202, 253, 48, 109, 46, 125, 149, 151, 47, 224, 141, 201, 39, 41,
        ],
    ],
    [
        [
            48, 45, 200, 86, 94, 46, 69, 245, 90, 200, 213, 189, 150, 174, 57, 226, 248, 5, 169,
            162, 250, 242, 92, 40, 218, 149, 132, 31, 157, 36, 254, 60,
        ],
        [
            81, 121, 54, 199, 38, 109, 10, 32, 0, 15, 80, 210, 86, 161, 85, 51, 217, 129, 254, 180,
            183, 108, 178, 137, 49, 248, 205, 212, 55, 77, 123, 17,
        ],
        [
            37, 248, 159, 204, 26, 229, 145, 238, 125, 74, 43, 139, 95, 23, 214, 19, 87, 91, 160,
            213, 235, 11, 65, 199, 200, 90, 231, 254, 125, 91, 125, 49,
        ],
        [
            178, 12, 107, 228, 177, 201, 160, 30, 244, 72, 143, 232, 142, 179, 253, 92, 117, 50,
            44, 170, 122, 254, 72, 125, 220, 234, 140, 49, 161, 58, 153, 32,
        ],
        [
            247, 237, 132, 219, 101, 146, 116, 139, 68, 147, 48, 176, 77, 83, 30, 181, 165, 91, 62,
            137, 228, 23, 138, 14, 88, 132, 111, 111, 140, 123, 228, 58,
        ],
        [
            18, 233, 193, 160, 178, 238, 235, 150, 3, 31, 61, 226, 24, 146, 32, 3, 177, 46, 49, 89,
            49, 45, 71, 83, 119, 186, 203, 200, 163, 237, 116, 21,
        ],
        [
            115, 144, 78, 114, 159, 231, 131, 65, 180, 121, 24, 139, 139, 29, 182, 131, 218, 81,
            58, 73, 107, 254, 32, 46, 177, 216, 202, 137, 182, 124, 221, 10,
        ],
        [
            223, 14, 173, 171, 179, 199, 71, 176, 71, 161, 186, 240, 0, 32, 204, 124, 160, 63, 112,
            86, 71, 213, 245, 55, 138, 233, 228, 28, 149, 8, 9, 41,
        ],
    ],
    [
        [
            21, 6, 131, 62, 42, 77, 174, 152, 2, 182, 147, 70, 52, 196, 221, 208, 21, 218, 63, 239,
            142, 100, 80, 227, 206, 92, 178, 63, 213, 66, 228, 42,
        ],
        [
            234, 98, 104, 233, 255, 69, 42, 117, 242, 174, 71, 237, 128, 208, 210, 92, 215, 80, 68,
            54, 109, 133, 117, 120, 11, 193, 130, 150, 123, 248, 52, 50,
        ],
        [
            42, 97, 215, 197, 142, 218, 220, 4, 99, 17, 179, 35, 227, 236, 159, 122, 227, 16, 41,
            191, 142, 113, 7, 118, 168, 0, 200, 69, 87, 232, 203, 18,
        ],
        [
            204, 101, 216, 98, 75, 52, 7, 133, 107, 63, 36, 58, 158, 102, 166, 65, 187, 189, 120,
            36, 170, 181, 122, 231, 195, 77, 77, 164, 123, 199, 51, 45,
        ],
        [
            69, 211, 219, 4, 90, 40, 61, 238, 40, 23, 38, 242, 82, 66, 124, 186, 78, 178, 158, 75,
            9, 78, 88, 175, 124, 209, 52, 139, 154, 205, 152, 1,
        ],
        [
            40, 177, 100, 175, 71, 133, 71, 147, 120, 232, 113, 156, 156, 171, 231, 14, 14, 48,
            254, 27, 109, 239, 11, 41, 141, 122, 231, 120, 162, 61, 172, 53,
        ],
        [
            138, 3, 230, 77, 88, 246, 242, 91, 53, 109, 136, 187, 226, 159, 63, 30, 210, 149, 85,
            132, 19, 243, 108, 225, 16, 67, 206, 187, 22, 250, 158, 38,
        ],
        [
            54, 230, 61, 20, 21, 69, 248, 130, 44, 13, 179, 216, 126, 95, 167, 29, 128, 81, 136,
            81, 198, 181, 167, 108, 21, 92, 89, 163, 48, 9, 25, 49,
        ],
    ],
    [
        [
            213, 68, 175, 186, 156, 48, 75, 216, 75, 120, 118, 229, 190, 251, 69, 197, 207, 233,
            255, 33, 12, 44, 176, 102, 172, 203, 246, 139, 9, 148, 188, 21,
        ],
        [
            59, 116, 78, 231, 0, 190, 4, 183, 254, 176, 57, 241, 16, 160, 195, 151, 109, 113, 65,
            227, 234, 248, 162, 150, 59, 223, 226, 68, 211, 220, 124, 35,
        ],
        [
            157, 241, 116, 218, 111, 20, 63, 220, 180, 253, 193, 124, 53, 174, 146, 115, 25, 21, 4,
            184, 28, 62, 68, 1, 136, 40, 112, 99, 155, 130, 29, 25,
        ],
        [
            107, 14, 195, 187, 170, 217, 205, 148, 49, 208, 203, 147, 220, 1, 216, 9, 235, 2, 23,
            137, 226, 242, 113, 173, 119, 193, 192, 240, 96, 34, 203, 27,
        ],
        [
            220, 189, 158, 255, 179, 2, 118, 209, 172, 32, 234, 150, 180, 140, 186, 88, 11, 146,
            216, 229, 126, 96, 132, 108, 80, 11, 133, 21, 208, 31, 104, 23,
        ],
        [
            34, 104, 190, 51, 173, 94, 133, 55, 100, 80, 120, 221, 114, 236, 129, 217, 2, 20, 117,
            96, 59, 162, 69, 216, 80, 193, 215, 191, 222, 251, 213, 54,
        ],
        [
            217, 173, 46, 115, 62, 139, 158, 81, 137, 145, 61, 242, 63, 26, 154, 80, 87, 23, 255,
            158, 21, 151, 39, 16, 176, 188, 206, 50, 91, 132, 244, 48,
        ],
        [
            125, 136, 4, 138, 60, 132, 107, 72, 160, 2, 185, 213, 99, 167, 64, 247, 133, 173, 139,
            25, 250, 132, 174, 254, 31, 85, 94, 85, 52, 212, 209, 61,
        ],
    ],
    [
        [
            220, 160, 48, 26, 85, 79, 138, 23, 139, 157, 143, 9, 40, 55, 28, 86, 243, 144, 211,
            131, 58, 6, 188, 247, 0, 45, 174, 144, 19, 188, 181, 57,
        ],
        [
            103, 128, 25, 9, 206, 137, 248, 46, 126, 157, 159, 0, 236, 255, 135, 58, 181, 227, 135,
            176, 113, 94, 237, 66, 47, 151, 18, 229, 19, 72, 170, 8,
        ],
        [
            46, 122, 169, 159, 37, 178, 130, 155, 49, 215, 123, 43, 108, 225, 225, 91, 74, 95, 1,
            190, 16, 230, 9, 231, 61, 153, 72, 119, 42, 120, 10, 41,
        ],
        [
            189, 179, 165, 203, 192, 23, 235, 85, 2, 155, 19, 118, 235, 123, 65, 219, 126, 163,
            149, 241, 213, 44, 143, 103, 139, 91, 15, 175, 111, 183, 238, 35,
        ],
        [
            52, 114, 250, 93, 151, 139, 135, 118, 40, 231, 118, 234, 63, 201, 15, 42, 126, 157,
            136, 113, 54, 209, 76, 78, 136, 13, 28, 3, 57, 119, 28, 32,
        ],
        [
            241, 223, 85, 153, 43, 88, 1, 195, 181, 239, 22, 16, 218, 11, 3, 91, 249, 42, 20, 25,
            215, 41, 132, 194, 106, 145, 165, 100, 33, 215, 243, 54,
        ],
        [
            185, 84, 49, 216, 187, 52, 40, 181, 184, 77, 168, 77, 5, 229, 36, 180, 82, 129, 99,
            126, 222, 17, 2, 185, 176, 18, 198, 220, 41, 192, 84, 1,
        ],
        [
            116, 111, 197, 38, 37, 6, 27, 181, 167, 142, 204, 148, 127, 82, 201, 197, 192, 133, 68,
            126, 57, 93, 146, 225, 67, 223, 187, 252, 131, 16, 136, 46,
        ],
    ],
    [
        [
            229, 194, 151, 215, 0, 49, 54, 48, 137, 138, 168, 12, 253, 114, 171, 220, 131, 85, 61,
            41, 90, 111, 235, 175, 142, 32, 64, 15, 171, 10, 37, 55,
        ],
        [
            179, 30, 248, 30, 85, 253, 191, 10, 66, 20, 9, 239, 72, 78, 126, 155, 5, 22, 16, 55,
            236, 155, 48, 104, 120, 209, 179, 94, 72, 144, 136, 36,
        ],
        [
            5, 85, 228, 82, 72, 151, 179, 40, 184, 104, 86, 64, 251, 178, 255, 222, 120, 33, 172,
            13, 70, 48, 74, 179, 143, 139, 203, 101, 242, 164, 233, 11,
        ],
        [
            155, 215, 194, 45, 223, 134, 9, 251, 102, 98, 255, 26, 99, 131, 158, 41, 28, 241, 244,
            3, 14, 14, 150, 5, 198, 23, 195, 55, 67, 107, 145, 50,
        ],
        [
            242, 66, 220, 115, 24, 115, 105, 247, 185, 102, 202, 149, 182, 74, 79, 71, 118, 46, 12,
            174, 168, 154, 12, 49, 8, 175, 173, 107, 106, 231, 34, 46,
        ],
        [
            218, 10, 168, 153, 218, 247, 27, 186, 177, 229, 129, 137, 17, 34, 173, 65, 77, 164,
            106, 206, 109, 156, 174, 78, 141, 110, 2, 177, 136, 189, 222, 27,
        ],
        [
            63, 206, 227, 252, 14, 115, 85, 145, 147, 2, 233, 37, 99, 52, 187, 61, 160, 216, 254,
            113, 47, 202, 231, 30, 8, 95, 186, 3, 95, 93, 92, 46,
        ],
        [
            194, 115, 177, 48, 84, 53, 255, 14, 255, 104, 86, 24, 225, 135, 136, 207, 223, 82, 248,
            120, 107, 19, 233, 239, 239, 174, 182, 203, 11, 70, 33, 29,
        ],
    ],
    [
        [
            255, 67, 169, 212, 128, 117, 218, 146, 73, 118, 113, 48, 51, 232, 221, 22, 69, 49, 118,
            127, 159, 164, 25, 203, 45, 231, 32, 76, 20, 3, 182, 35,
        ],
        [
            128, 192, 45, 222, 4, 111, 75, 67, 105, 135, 179, 9, 231, 6, 181, 241, 43, 60, 145,
            129, 101, 107, 239, 197, 249, 142, 15, 73, 190, 55, 8, 32,
        ],
        [
            75, 171, 221, 252, 200, 45, 18, 225, 108, 75, 63, 243, 49, 199, 160, 123, 50, 138, 70,
            237, 85, 182, 107, 126, 116, 200, 23, 254, 122, 68, 35, 39,
        ],
        [
            220, 204, 97, 165, 14, 201, 153, 93, 114, 166, 64, 88, 30, 193, 77, 84, 243, 228, 14,
            33, 240, 75, 108, 168, 10, 68, 170, 110, 117, 133, 17, 14,
        ],
        [
            7, 8, 119, 58, 124, 100, 81, 248, 160, 28, 127, 136, 223, 190, 172, 109, 44, 130, 189,
            200, 120, 37, 118, 199, 93, 174, 13, 227, 228, 109, 77, 24,
        ],
        [
            117, 48, 90, 251, 34, 191, 185, 106, 156, 198, 3, 244, 147, 116, 106, 125, 9, 98, 103,
            25, 67, 247, 161, 20, 215, 143, 98, 56, 111, 192, 80, 12,
        ],
        [
            75, 150, 146, 192, 192, 73, 4, 222, 31, 115, 161, 55, 189, 28, 254, 27, 140, 18, 245,
            197, 156, 136, 196, 80, 24, 215, 69, 183, 144, 198, 218, 25,
        ],
        [
            4, 59, 137, 5, 85, 187, 155, 254, 174, 111, 148, 55, 146, 77, 162, 139, 54, 133, 46,
            24, 60, 78, 15, 39, 58, 21, 189, 95, 85, 66, 75, 11,
        ],
    ],
    [
        [
            38, 12, 159, 91, 61, 126, 7, 197, 252, 179, 45, 40, 117, 226, 17, 100, 55, 150, 227,
            217, 209, 62, 51, 50, 221, 205, 172, 219, 49, 51, 83, 32,
        ],
        [
            57, 122, 44, 109, 85, 120, 81, 245, 24, 47, 86, 189, 244, 231, 243, 83, 100, 219, 109,
            83, 65, 99, 251, 105, 85, 7, 198, 149, 255, 76, 29, 29,
        ],
        [
            83, 248, 191, 219, 235, 44, 62, 170, 200, 75, 122, 207, 106, 115, 35, 184, 127, 201,
            143, 180, 11, 25, 135, 78, 235, 106, 174, 203, 163, 11, 14, 1,
        ],
        [
            3, 65, 229, 5, 70, 240, 200, 34, 11, 204, 14, 118, 49, 109, 108, 179, 116, 175, 211,
            123, 3, 227, 30, 105, 244, 63, 147, 137, 174, 6, 58, 49,
        ],
        [
            255, 44, 184, 49, 125, 73, 101, 79, 205, 191, 13, 181, 32, 246, 234, 221, 236, 168,
            174, 34, 80, 203, 242, 216, 243, 28, 146, 97, 71, 119, 33, 37,
        ],
        [
            66, 233, 66, 56, 132, 114, 180, 163, 103, 56, 133, 98, 139, 1, 125, 189, 147, 192, 40,
            64, 96, 55, 15, 157, 50, 250, 230, 167, 42, 104, 5, 31,
        ],
        [
            6, 183, 137, 208, 244, 75, 158, 179, 218, 29, 142, 60, 229, 162, 211, 232, 18, 152, 34,
            85, 133, 122, 3, 251, 178, 144, 51, 171, 190, 238, 27, 3,
        ],
        [
            129, 246, 203, 236, 115, 97, 172, 103, 209, 116, 141, 209, 172, 34, 244, 120, 19, 127,
            229, 88, 113, 135, 129, 213, 94, 46, 29, 66, 93, 142, 188, 36,
        ],
    ],
    [
        [
            194, 202, 86, 162, 6, 28, 7, 93, 217, 68, 231, 140, 53, 182, 117, 79, 205, 232, 162,
            118, 84, 16, 245, 15, 199, 85, 234, 133, 4, 11, 210, 35,
        ],
        [
            157, 145, 197, 39, 234, 104, 156, 193, 122, 16, 169, 90, 174, 93, 243, 193, 116, 187,
            240, 105, 166, 97, 121, 195, 247, 81, 101, 251, 216, 51, 21, 54,
        ],
        [
            61, 143, 204, 33, 38, 137, 156, 7, 19, 73, 71, 241, 59, 229, 100, 52, 53, 38, 189, 124,
            89, 219, 37, 80, 179, 34, 169, 198, 3, 223, 233, 10,
        ],
        [
            204, 47, 178, 44, 209, 241, 91, 185, 131, 159, 137, 31, 191, 153, 83, 229, 11, 23, 175,
            81, 72, 20, 30, 65, 99, 191, 42, 182, 180, 250, 108, 44,
        ],
        [
            223, 219, 203, 230, 21, 177, 178, 244, 62, 182, 248, 215, 254, 41, 219, 104, 64, 104,
            105, 86, 234, 81, 171, 153, 179, 116, 16, 103, 192, 250, 50, 38,
        ],
        [
            179, 15, 111, 91, 124, 136, 230, 38, 62, 52, 144, 140, 164, 57, 139, 56, 98, 70, 189,
            251, 255, 55, 115, 97, 196, 57, 221, 165, 145, 80, 26, 46,
        ],
        [
            21, 103, 9, 56, 143, 106, 41, 164, 169, 198, 100, 253, 71, 31, 238, 128, 240, 35, 71,
            86, 44, 61, 61, 237, 182, 110, 207, 53, 238, 201, 246, 40,
        ],
        [
            204, 177, 216, 11, 64, 47, 94, 27, 176, 61, 203, 218, 169, 142, 37, 95, 21, 172, 57, 9,
            223, 126, 135, 241, 193, 203, 213, 144, 92, 248, 186, 35,
        ],
    ],
    [
        [
            228, 75, 109, 106, 255, 254, 130, 13, 9, 11, 211, 140, 45, 161, 63, 69, 215, 182, 22,
            59, 74, 170, 177, 7, 2, 115, 14, 109, 242, 13, 68, 25,
        ],
        [
            6, 222, 179, 66, 171, 58, 140, 170, 192, 130, 16, 180, 206, 179, 253, 194, 92, 54, 196,
            149, 116, 69, 233, 218, 100, 13, 63, 9, 144, 54, 85, 33,
        ],
        [
            95, 33, 20, 56, 49, 62, 85, 56, 154, 215, 173, 198, 73, 95, 232, 119, 212, 167, 148,
            120, 147, 204, 164, 27, 136, 69, 78, 130, 7, 206, 240, 36,
        ],
        [
            109, 38, 215, 164, 103, 152, 172, 47, 244, 227, 72, 242, 229, 117, 198, 228, 89, 185,
            207, 83, 21, 173, 187, 111, 41, 54, 183, 144, 61, 53, 193, 59,
        ],
        [
            67, 218, 51, 136, 12, 221, 238, 123, 201, 104, 185, 13, 178, 54, 4, 239, 242, 103, 151,
            232, 203, 231, 207, 63, 60, 240, 41, 187, 153, 2, 249, 22,
        ],
        [
            175, 206, 167, 190, 93, 40, 120, 180, 117, 248, 121, 191, 230, 16, 16, 194, 194, 12,
            146, 20, 128, 18, 44, 31, 48, 232, 16, 11, 84, 110, 68, 8,
        ],
        [
            136, 156, 120, 102, 134, 205, 174, 55, 52, 71, 4, 165, 3, 97, 61, 249, 52, 110, 212,
            234, 83, 197, 244, 93, 107, 88, 47, 120, 63, 80, 100, 28,
        ],
        [
            237, 132, 106, 72, 25, 163, 185, 172, 196, 20, 116, 129, 247, 220, 212, 128, 76, 154,
            10, 50, 26, 220, 50, 179, 252, 180, 239, 44, 147, 71, 71, 38,
        ],
    ],
    [
        [
            195, 126, 226, 101, 232, 151, 83, 172, 247, 79, 222, 218, 214, 224, 68, 44, 234, 50,
            156, 252, 136, 137, 56, 174, 225, 55, 8, 111, 153, 167, 160, 43,
        ],
        [
            241, 114, 11, 165, 35, 58, 224, 96, 147, 243, 117, 49, 124, 82, 194, 138, 217, 81, 168,
            235, 194, 7, 11, 112, 218, 164, 244, 225, 127, 222, 190, 7,
        ],
        [
            210, 109, 146, 106, 46, 44, 15, 251, 15, 32, 213, 72, 138, 6, 104, 145, 147, 59, 192,
            77, 53, 213, 36, 69, 191, 247, 20, 79, 43, 242, 181, 47,
        ],
        [
            204, 85, 163, 37, 144, 235, 182, 50, 84, 214, 30, 210, 246, 198, 20, 48, 217, 247, 117,
            180, 207, 150, 47, 224, 35, 81, 66, 231, 203, 126, 250, 7,
        ],
        [
            251, 235, 48, 197, 117, 61, 99, 74, 158, 107, 9, 243, 137, 155, 185, 232, 53, 14, 195,
            20, 2, 83, 162, 104, 55, 30, 229, 35, 167, 108, 91, 63,
        ],
        [
            230, 107, 185, 57, 172, 171, 223, 240, 186, 182, 106, 85, 77, 252, 75, 255, 117, 50,
            88, 60, 41, 76, 188, 39, 194, 16, 46, 117, 220, 130, 64, 10,
        ],
        [
            228, 156, 144, 195, 138, 103, 97, 79, 225, 101, 43, 50, 232, 2, 152, 215, 232, 116,
            177, 254, 13, 17, 224, 87, 217, 149, 255, 187, 18, 241, 16, 15,
        ],
        [
            83, 173, 113, 34, 69, 81, 9, 148, 205, 79, 152, 7, 254, 207, 74, 31, 247, 197, 139, 66,
            18, 219, 218, 37, 222, 252, 172, 80, 184, 147, 51, 41,
        ],
    ],
    [
        [
            0, 89, 55, 87, 138, 18, 8, 243, 120, 2, 157, 224, 104, 227, 73, 235, 249, 187, 159,
            136, 40, 1, 155, 200, 193, 125, 4, 81, 219, 169, 8, 56,
        ],
        [
            116, 29, 157, 39, 107, 56, 158, 126, 218, 221, 193, 170, 23, 95, 163, 26, 206, 168,
            133, 96, 36, 195, 59, 152, 44, 66, 146, 48, 61, 175, 172, 43,
        ],
        [
            213, 48, 102, 181, 10, 10, 192, 234, 79, 4, 153, 169, 240, 170, 155, 171, 56, 254, 3,
            110, 104, 41, 167, 162, 135, 231, 107, 109, 118, 222, 146, 3,
        ],
        [
            165, 81, 105, 154, 149, 116, 129, 132, 226, 15, 34, 154, 113, 82, 142, 224, 187, 254,
            157, 40, 127, 53, 13, 50, 6, 116, 44, 28, 171, 25, 57, 11,
        ],
        [
            137, 173, 137, 148, 183, 45, 212, 173, 208, 103, 238, 190, 209, 217, 109, 115, 131,
            170, 167, 157, 48, 105, 2, 225, 33, 144, 56, 104, 74, 90, 101, 45,
        ],
        [
            37, 210, 32, 187, 166, 239, 15, 248, 101, 84, 62, 170, 65, 179, 205, 5, 155, 124, 68,
            191, 56, 128, 71, 92, 106, 68, 1, 162, 177, 208, 131, 32,
        ],
        [
            20, 40, 151, 188, 73, 233, 86, 220, 88, 77, 88, 18, 101, 114, 204, 242, 37, 68, 158,
            216, 166, 213, 147, 74, 136, 177, 146, 127, 183, 9, 51, 51,
        ],
        [
            35, 250, 252, 114, 37, 150, 90, 62, 182, 153, 4, 114, 89, 207, 137, 211, 221, 60, 38,
            181, 33, 54, 11, 251, 225, 50, 39, 247, 36, 64, 59, 60,
        ],
    ],
    [
        [
            3, 44, 176, 61, 155, 104, 167, 47, 197, 180, 112, 16, 213, 166, 109, 139, 47, 145, 213,
            181, 28, 249, 251, 77, 66, 197, 100, 68, 46, 151, 197, 40,
        ],
        [
            229, 35, 247, 127, 1, 20, 236, 233, 188, 17, 88, 74, 87, 15, 3, 29, 133, 223, 178, 245,
            50, 63, 172, 25, 149, 238, 2, 154, 212, 51, 108, 23,
        ],
        [
            186, 13, 97, 147, 88, 88, 217, 127, 47, 123, 233, 132, 148, 151, 181, 146, 56, 81, 20,
            162, 72, 61, 131, 202, 209, 20, 131, 189, 7, 97, 130, 13,
        ],
        [
            139, 153, 27, 151, 116, 240, 246, 218, 212, 68, 91, 52, 237, 242, 111, 7, 137, 76, 231,
            210, 148, 216, 28, 108, 27, 231, 141, 34, 80, 138, 59, 56,
        ],
        [
            253, 100, 45, 165, 2, 73, 249, 202, 254, 16, 147, 135, 31, 124, 231, 212, 212, 0, 79,
            103, 51, 195, 6, 238, 158, 204, 61, 69, 129, 149, 202, 11,
        ],
        [
            103, 190, 49, 80, 44, 51, 242, 50, 73, 31, 21, 63, 205, 25, 99, 107, 172, 204, 132, 57,
            220, 222, 181, 200, 245, 232, 39, 33, 20, 227, 78, 40,
        ],
        [
            85, 212, 71, 43, 217, 39, 72, 56, 143, 82, 141, 13, 180, 31, 10, 76, 187, 243, 38, 201,
            119, 39, 155, 236, 217, 54, 223, 154, 194, 164, 134, 62,
        ],
        [
            77, 5, 128, 184, 241, 69, 158, 253, 102, 134, 157, 97, 98, 242, 100, 188, 115, 143,
            210, 202, 43, 167, 1, 228, 159, 25, 90, 97, 135, 29, 156, 31,
        ],
    ],
    [
        [
            36, 72, 161, 85, 169, 138, 112, 236, 110, 59, 12, 214, 194, 22, 221, 138, 62, 109, 108,
            237, 74, 210, 136, 75, 62, 166, 199, 109, 221, 86, 136, 29,
        ],
        [
            140, 2, 200, 184, 189, 53, 142, 10, 137, 117, 249, 87, 40, 107, 43, 120, 46, 109, 193,
            103, 206, 125, 50, 217, 13, 183, 23, 93, 30, 254, 235, 19,
        ],
        [
            135, 255, 108, 83, 58, 92, 159, 201, 196, 233, 222, 140, 225, 114, 46, 196, 1, 113,
            133, 89, 242, 84, 195, 115, 216, 206, 91, 136, 138, 11, 182, 40,
        ],
        [
            43, 56, 228, 25, 206, 194, 106, 233, 211, 143, 222, 188, 69, 112, 67, 95, 44, 35, 141,
            217, 190, 181, 140, 62, 217, 117, 135, 249, 33, 200, 62, 24,
        ],
        [
            27, 86, 219, 191, 131, 228, 7, 155, 210, 136, 213, 240, 244, 130, 179, 191, 182, 222,
            17, 154, 6, 97, 15, 123, 71, 182, 172, 41, 208, 234, 109, 21,
        ],
        [
            223, 22, 81, 97, 125, 169, 6, 44, 41, 71, 184, 66, 241, 108, 28, 88, 120, 93, 200, 86,
            150, 164, 255, 54, 241, 182, 103, 35, 29, 246, 255, 23,
        ],
        [
            210, 52, 232, 184, 171, 196, 52, 49, 91, 18, 72, 164, 48, 119, 62, 14, 82, 171, 43, 64,
            136, 163, 208, 132, 255, 220, 172, 227, 37, 160, 4, 13,
        ],
        [
            158, 181, 213, 223, 176, 57, 134, 98, 137, 31, 26, 52, 93, 80, 212, 202, 96, 189, 24,
            72, 94, 211, 23, 8, 135, 87, 25, 138, 244, 103, 178, 10,
        ],
    ],
    [
        [
            240, 216, 75, 131, 74, 33, 115, 241, 209, 7, 112, 77, 251, 35, 7, 247, 247, 86, 143,
            130, 144, 13, 181, 122, 153, 89, 125, 105, 99, 213, 33, 55,
        ],
        [
            145, 235, 201, 158, 59, 22, 0, 15, 228, 200, 174, 78, 89, 87, 26, 249, 11, 155, 71, 79,
            3, 34, 190, 254, 234, 73, 46, 92, 61, 216, 41, 17,
        ],
        [
            43, 56, 20, 48, 194, 81, 86, 213, 164, 121, 220, 75, 76, 201, 36, 142, 56, 70, 75, 194,
            140, 87, 136, 174, 210, 127, 214, 65, 193, 41, 107, 63,
        ],
        [
            213, 92, 219, 226, 204, 5, 20, 13, 110, 238, 153, 209, 70, 249, 219, 212, 107, 56, 7,
            179, 87, 36, 234, 35, 89, 18, 114, 25, 230, 192, 242, 11,
        ],
        [
            226, 223, 162, 191, 16, 10, 133, 242, 11, 71, 12, 55, 220, 63, 111, 23, 30, 154, 90,
            151, 140, 23, 118, 109, 212, 51, 99, 222, 98, 158, 237, 48,
        ],
        [
            20, 38, 109, 230, 153, 122, 68, 236, 179, 59, 137, 231, 25, 246, 164, 211, 103, 164,
            113, 21, 139, 125, 13, 45, 47, 207, 173, 18, 205, 128, 112, 16,
        ],
        [
            133, 0, 26, 33, 217, 221, 37, 96, 168, 164, 192, 185, 154, 124, 126, 12, 113, 167, 105,
            37, 99, 38, 76, 76, 244, 244, 98, 94, 185, 223, 178, 50,
        ],
        [
            120, 250, 66, 188, 11, 58, 177, 140, 133, 4, 35, 210, 16, 107, 110, 139, 177, 78, 164,
            108, 156, 193, 85, 154, 42, 199, 216, 70, 150, 47, 234, 38,
        ],
    ],
    [
        [
            91, 250, 197, 122, 129, 9, 102, 216, 154, 110, 193, 132, 68, 224, 180, 95, 240, 62,
            136, 1, 176, 233, 188, 19, 194, 229, 140, 112, 44, 133, 125, 24,
        ],
        [
            249, 15, 38, 45, 54, 238, 216, 27, 31, 126, 65, 99, 176, 76, 76, 214, 121, 5, 114, 71,
            130, 26, 237, 49, 235, 91, 254, 1, 144, 4, 163, 45,
        ],
        [
            255, 229, 192, 133, 154, 196, 136, 105, 117, 96, 229, 23, 193, 198, 227, 47, 103, 66,
            232, 204, 152, 70, 129, 24, 164, 199, 122, 63, 191, 92, 135, 20,
        ],
        [
            145, 44, 90, 2, 241, 40, 255, 122, 79, 26, 250, 253, 187, 122, 164, 178, 138, 125, 75,
            226, 146, 177, 191, 224, 93, 101, 229, 157, 138, 155, 12, 16,
        ],
        [
            139, 122, 121, 78, 6, 21, 126, 230, 160, 18, 97, 42, 15, 118, 106, 37, 136, 251, 240,
            227, 57, 51, 233, 199, 158, 70, 127, 104, 78, 185, 127, 8,
        ],
        [
            200, 199, 55, 107, 21, 71, 88, 106, 158, 74, 81, 41, 158, 199, 169, 181, 24, 9, 235,
            60, 232, 189, 129, 3, 44, 218, 123, 77, 232, 208, 128, 9,
        ],
        [
            87, 199, 112, 222, 83, 57, 76, 182, 157, 113, 119, 150, 108, 143, 237, 75, 56, 50, 38,
            66, 172, 194, 17, 165, 112, 37, 228, 185, 86, 2, 145, 62,
        ],
        [
            140, 36, 76, 105, 203, 131, 114, 245, 40, 53, 57, 156, 212, 119, 252, 184, 23, 242, 29,
            102, 122, 221, 83, 101, 87, 196, 251, 116, 162, 218, 192, 1,
        ],
    ],
    [
        [
            97, 64, 127, 237, 110, 192, 220, 140, 59, 142, 248, 62, 234, 132, 188, 82, 67, 201,
            217, 7, 201, 173, 228, 72, 105, 116, 5, 79, 175, 99, 110, 47,
        ],
        [
            150, 89, 171, 150, 182, 170, 198, 17, 165, 5, 128, 18, 226, 2, 104, 242, 43, 208, 9,
            148, 202, 37, 98, 172, 180, 251, 0, 83, 175, 210, 66, 58,
        ],
        [
            17, 173, 81, 22, 153, 10, 186, 31, 241, 216, 106, 49, 89, 190, 84, 172, 253, 183, 24,
            143, 238, 231, 209, 108, 168, 173, 98, 162, 198, 181, 254, 51,
        ],
        [
            137, 210, 105, 132, 29, 175, 97, 107, 112, 74, 210, 47, 145, 197, 47, 143, 80, 208,
            204, 127, 23, 42, 5, 235, 125, 56, 241, 210, 114, 164, 16, 20,
        ],
        [
            132, 20, 239, 85, 238, 127, 107, 177, 97, 117, 240, 180, 115, 92, 140, 47, 146, 146,
            209, 250, 105, 68, 254, 108, 179, 193, 247, 12, 193, 30, 102, 43,
        ],
        [
            145, 126, 226, 38, 179, 157, 78, 13, 224, 130, 207, 125, 26, 174, 162, 43, 138, 2, 152,
            36, 231, 146, 204, 225, 233, 239, 158, 247, 28, 56, 142, 26,
        ],
        [
            91, 164, 60, 87, 26, 175, 68, 99, 255, 62, 87, 54, 247, 40, 87, 155, 223, 249, 133, 28,
            117, 204, 149, 227, 61, 21, 58, 52, 75, 149, 250, 10,
        ],
        [
            237, 191, 63, 71, 77, 84, 26, 130, 131, 163, 245, 220, 12, 111, 61, 79, 234, 238, 214,
            176, 160, 20, 135, 93, 135, 186, 136, 38, 105, 93, 100, 15,
        ],
    ],
    [
        [
            223, 170, 66, 190, 77, 208, 106, 211, 248, 2, 37, 34, 99, 11, 115, 188, 248, 104, 129,
            217, 110, 246, 30, 138, 250, 119, 187, 198, 184, 231, 12, 7,
        ],
        [
            120, 133, 72, 207, 98, 167, 236, 27, 52, 201, 169, 82, 14, 47, 143, 2, 249, 62, 196,
            64, 24, 11, 206, 173, 64, 91, 37, 228, 197, 230, 184, 38,
        ],
        [
            213, 11, 163, 125, 42, 99, 72, 223, 58, 183, 0, 153, 96, 92, 125, 212, 108, 65, 112,
            252, 99, 25, 113, 161, 55, 130, 32, 116, 104, 18, 188, 46,
        ],
        [
            142, 172, 196, 64, 209, 79, 68, 196, 33, 58, 63, 86, 184, 182, 233, 48, 38, 46, 86,
            198, 52, 81, 36, 136, 62, 49, 52, 132, 26, 44, 174, 8,
        ],
        [
            79, 79, 157, 236, 132, 179, 39, 72, 58, 22, 193, 191, 5, 165, 177, 232, 113, 76, 196,
            135, 205, 206, 127, 137, 17, 121, 184, 142, 66, 50, 42, 30,
        ],
        [
            140, 147, 23, 191, 59, 161, 110, 195, 158, 68, 129, 248, 198, 61, 3, 54, 198, 151, 148,
            60, 165, 221, 201, 66, 76, 191, 167, 108, 16, 122, 2, 60,
        ],
        [
            230, 31, 126, 106, 18, 212, 227, 114, 225, 54, 2, 184, 55, 2, 36, 148, 145, 243, 68,
            157, 10, 231, 99, 227, 102, 179, 220, 245, 187, 202, 90, 3,
        ],
        [
            131, 100, 10, 199, 130, 82, 185, 106, 12, 181, 129, 230, 230, 197, 136, 203, 147, 216,
            212, 52, 200, 130, 43, 61, 218, 76, 160, 84, 61, 47, 4, 9,
        ],
    ],
    [
        [
            55, 204, 232, 108, 99, 18, 152, 227, 32, 80, 114, 26, 128, 128, 125, 191, 206, 224,
            241, 115, 26, 136, 84, 154, 148, 210, 33, 18, 177, 227, 22, 13,
        ],
        [
            144, 114, 124, 2, 51, 175, 130, 107, 154, 122, 141, 251, 83, 165, 109, 160, 140, 155,
            37, 33, 174, 241, 100, 50, 181, 99, 106, 235, 13, 20, 231, 49,
        ],
        [
            216, 224, 156, 16, 159, 31, 195, 82, 159, 97, 74, 220, 80, 110, 13, 143, 225, 137, 212,
            157, 20, 130, 117, 243, 6, 194, 113, 149, 109, 157, 117, 48,
        ],
        [
            203, 152, 160, 212, 237, 116, 129, 128, 172, 159, 96, 132, 111, 87, 224, 76, 161, 186,
            43, 113, 253, 239, 239, 247, 147, 215, 35, 210, 24, 189, 172, 42,
        ],
        [
            127, 6, 186, 229, 204, 141, 254, 171, 184, 224, 28, 52, 14, 248, 229, 15, 103, 100,
            189, 80, 135, 173, 32, 47, 15, 100, 230, 42, 42, 133, 116, 62,
        ],
        [
            98, 43, 219, 88, 207, 92, 180, 93, 183, 130, 209, 52, 218, 197, 222, 121, 202, 82, 89,
            163, 146, 136, 199, 138, 92, 229, 97, 45, 202, 14, 107, 3,
        ],
        [
            240, 123, 106, 60, 50, 212, 31, 248, 89, 162, 207, 92, 59, 93, 219, 186, 97, 70, 183,
            76, 226, 50, 124, 180, 170, 240, 86, 206, 208, 73, 143, 58,
        ],
        [
            68, 34, 149, 125, 18, 250, 72, 119, 93, 236, 204, 129, 252, 238, 125, 83, 69, 110, 14,
            140, 202, 10, 54, 156, 156, 245, 188, 186, 65, 147, 214, 58,
        ],
    ],
    [
        [
            100, 148, 161, 249, 209, 68, 164, 245, 215, 120, 36, 73, 193, 207, 199, 211, 164, 112,
            198, 111, 27, 1, 28, 72, 43, 84, 188, 15, 250, 159, 188, 26,
        ],
        [
            253, 232, 39, 91, 86, 20, 213, 250, 200, 253, 181, 117, 94, 30, 220, 129, 132, 201,
            156, 56, 249, 165, 70, 192, 24, 61, 24, 145, 187, 24, 7, 13,
        ],
        [
            235, 160, 136, 133, 75, 13, 222, 161, 238, 179, 2, 106, 74, 63, 0, 248, 96, 26, 138,
            196, 53, 150, 8, 73, 124, 157, 77, 23, 208, 245, 181, 10,
        ],
        [
            224, 180, 160, 165, 136, 230, 75, 44, 173, 23, 88, 47, 214, 41, 208, 76, 49, 85, 171,
            140, 128, 17, 207, 88, 157, 169, 101, 251, 90, 52, 114, 26,
        ],
        [
            251, 247, 129, 11, 189, 220, 160, 126, 110, 241, 216, 126, 145, 176, 90, 92, 255, 164,
            67, 155, 143, 105, 231, 221, 19, 4, 209, 247, 129, 79, 58, 49,
        ],
        [
            55, 58, 1, 79, 198, 128, 28, 83, 10, 33, 83, 28, 178, 155, 41, 216, 238, 187, 174, 61,
            201, 86, 87, 236, 92, 163, 208, 44, 86, 114, 54, 54,
        ],
        [
            57, 169, 95, 175, 193, 90, 221, 121, 48, 104, 22, 110, 222, 88, 201, 207, 154, 16, 36,
            26, 255, 141, 57, 222, 7, 189, 115, 238, 20, 167, 80, 40,
        ],
        [
            47, 181, 105, 56, 58, 145, 115, 74, 79, 83, 212, 58, 126, 171, 44, 110, 219, 142, 60,
            180, 182, 14, 17, 229, 248, 82, 81, 14, 170, 126, 147, 54,
        ],
    ],
    [
        [
            128, 20, 45, 17, 172, 230, 162, 59, 42, 143, 8, 76, 33, 186, 248, 116, 42, 73, 85, 228,
            40, 39, 54, 88, 181, 90, 54, 87, 55, 177, 229, 48,
        ],
        [
            102, 173, 199, 114, 55, 226, 31, 35, 68, 39, 12, 40, 107, 135, 27, 73, 112, 191, 44,
            71, 245, 185, 21, 25, 255, 200, 88, 114, 110, 229, 161, 22,
        ],
        [
            242, 13, 62, 36, 95, 118, 193, 245, 50, 127, 84, 98, 54, 154, 247, 225, 103, 249, 157,
            187, 31, 199, 119, 56, 69, 142, 166, 156, 124, 182, 243, 30,
        ],
        [
            169, 44, 97, 74, 69, 161, 72, 96, 128, 26, 55, 131, 150, 94, 84, 17, 234, 77, 2, 152,
            17, 35, 18, 95, 191, 22, 140, 116, 110, 72, 104, 40,
        ],
        [
            23, 99, 66, 21, 11, 139, 224, 24, 118, 30, 242, 229, 10, 83, 69, 203, 47, 187, 184,
            170, 182, 204, 151, 96, 55, 203, 201, 45, 173, 76, 21, 62,
        ],
        [
            174, 56, 175, 37, 18, 168, 92, 220, 126, 59, 122, 128, 188, 14, 11, 76, 224, 185, 52,
            56, 171, 1, 76, 94, 170, 201, 52, 223, 143, 143, 65, 0,
        ],
        [
            105, 252, 232, 233, 233, 184, 240, 251, 105, 190, 199, 191, 40, 128, 70, 127, 50, 153,
            218, 78, 159, 152, 131, 91, 144, 74, 196, 12, 72, 3, 98, 60,
        ],
        [
            86, 118, 204, 12, 44, 119, 238, 58, 14, 159, 108, 2, 205, 230, 12, 174, 118, 229, 14,
            199, 87, 166, 74, 165, 221, 17, 87, 56, 146, 78, 20, 38,
        ],
    ],
    [
        [
            17, 210, 205, 128, 203, 153, 80, 160, 236, 118, 114, 81, 57, 77, 237, 168, 120, 203,
            243, 6, 165, 10, 37, 16, 19, 82, 193, 5, 125, 101, 1, 1,
        ],
        [
            52, 57, 24, 226, 28, 210, 52, 197, 135, 76, 224, 159, 200, 11, 210, 84, 143, 91, 58,
            149, 199, 147, 56, 115, 35, 77, 56, 29, 112, 225, 160, 21,
        ],
        [
            103, 157, 74, 50, 168, 153, 10, 113, 111, 250, 147, 120, 97, 105, 30, 162, 84, 110,
            181, 117, 25, 90, 57, 255, 1, 241, 61, 16, 55, 61, 65, 52,
        ],
        [
            241, 161, 9, 88, 151, 232, 118, 42, 51, 245, 132, 81, 90, 123, 243, 30, 223, 21, 254,
            64, 128, 203, 22, 56, 162, 74, 101, 8, 215, 253, 75, 7,
        ],
        [
            168, 52, 111, 97, 141, 41, 127, 84, 216, 177, 7, 162, 249, 219, 97, 204, 187, 22, 8,
            167, 81, 147, 177, 55, 76, 215, 129, 233, 167, 122, 131, 60,
        ],
        [
            148, 157, 151, 154, 244, 40, 17, 161, 198, 20, 105, 212, 156, 158, 39, 23, 231, 137,
            222, 84, 68, 27, 3, 207, 199, 230, 18, 44, 157, 202, 81, 52,
        ],
        [
            77, 77, 79, 101, 89, 159, 81, 121, 236, 91, 174, 188, 38, 33, 130, 156, 128, 26, 22,
            145, 39, 255, 241, 196, 151, 255, 82, 91, 148, 232, 71, 18,
        ],
        [
            194, 196, 111, 238, 57, 75, 10, 99, 212, 187, 129, 199, 119, 92, 1, 108, 241, 170, 157,
            183, 229, 228, 101, 27, 52, 174, 69, 134, 147, 83, 58, 54,
        ],
    ],
    [
        [
            241, 63, 197, 8, 100, 206, 130, 195, 128, 98, 20, 252, 192, 60, 31, 154, 23, 124, 103,
            182, 200, 154, 206, 243, 229, 5, 113, 213, 164, 112, 185, 55,
        ],
        [
            105, 242, 123, 46, 87, 11, 173, 139, 0, 221, 84, 124, 26, 7, 229, 104, 187, 91, 104,
            81, 219, 204, 240, 1, 1, 235, 27, 29, 117, 155, 29, 38,
        ],
        [
            225, 247, 40, 180, 114, 90, 146, 118, 40, 231, 57, 24, 23, 219, 32, 253, 2, 110, 91,
            34, 56, 181, 2, 115, 172, 228, 119, 92, 167, 155, 176, 58,
        ],
        [
            249, 104, 21, 229, 236, 207, 33, 6, 53, 166, 64, 148, 104, 232, 112, 223, 106, 6, 118,
            70, 13, 100, 73, 127, 195, 76, 19, 138, 175, 115, 228, 1,
        ],
        [
            195, 144, 80, 114, 29, 145, 101, 51, 84, 237, 157, 117, 160, 225, 241, 235, 114, 170,
            30, 122, 200, 158, 177, 41, 229, 85, 234, 198, 76, 168, 52, 24,
        ],
        [
            120, 210, 255, 49, 191, 96, 63, 157, 96, 72, 69, 219, 183, 118, 127, 68, 51, 117, 157,
            15, 174, 235, 120, 43, 212, 104, 132, 191, 34, 171, 203, 33,
        ],
        [
            44, 169, 60, 45, 196, 253, 165, 116, 12, 163, 75, 217, 85, 188, 185, 137, 45, 249, 112,
            151, 138, 217, 14, 129, 30, 158, 215, 254, 84, 42, 228, 42,
        ],
        [
            203, 168, 249, 62, 65, 9, 40, 183, 128, 84, 88, 173, 178, 248, 224, 77, 145, 33, 193,
            92, 50, 207, 141, 24, 15, 52, 252, 243, 246, 98, 19, 49,
        ],
    ],
    [
        [
            237, 54, 165, 102, 136, 30, 200, 45, 73, 254, 49, 68, 66, 86, 185, 7, 241, 116, 123,
            60, 28, 190, 126, 70, 234, 142, 64, 228, 224, 138, 208, 27,
        ],
        [
            99, 57, 207, 132, 197, 103, 87, 84, 55, 138, 63, 103, 194, 83, 185, 46, 158, 91, 214,
            20, 95, 59, 228, 214, 143, 132, 0, 23, 253, 230, 26, 48,
        ],
        [
            123, 239, 222, 226, 21, 242, 243, 197, 129, 206, 152, 129, 16, 20, 211, 42, 176, 65,
            242, 192, 66, 69, 81, 3, 131, 15, 216, 202, 2, 243, 218, 7,
        ],
        [
            31, 26, 204, 209, 239, 190, 83, 225, 196, 75, 45, 121, 3, 102, 101, 131, 44, 31, 218,
            48, 73, 17, 134, 242, 247, 29, 221, 207, 227, 5, 52, 25,
        ],
        [
            202, 14, 80, 192, 246, 207, 239, 108, 96, 56, 18, 233, 77, 173, 33, 8, 59, 253, 229,
            142, 225, 80, 41, 255, 246, 3, 81, 26, 35, 44, 231, 16,
        ],
        [
            121, 74, 250, 232, 223, 85, 200, 144, 168, 194, 169, 93, 4, 215, 101, 243, 38, 138, 47,
            65, 199, 166, 45, 18, 118, 239, 185, 189, 156, 25, 107, 24,
        ],
        [
            78, 176, 119, 91, 128, 226, 57, 17, 143, 199, 58, 244, 198, 201, 19, 241, 221, 217,
            241, 225, 34, 210, 174, 2, 205, 104, 225, 0, 230, 178, 142, 57,
        ],
        [
            148, 105, 249, 45, 180, 221, 60, 123, 112, 195, 248, 88, 13, 52, 126, 194, 131, 236,
            52, 91, 170, 222, 209, 94, 201, 73, 253, 213, 21, 196, 23, 26,
        ],
    ],
    [
        [
            182, 212, 91, 5, 64, 193, 110, 29, 140, 166, 154, 43, 144, 226, 204, 76, 157, 57, 210,
            77, 119, 255, 110, 232, 205, 123, 162, 24, 193, 148, 79, 12,
        ],
        [
            36, 238, 108, 197, 169, 69, 71, 57, 159, 134, 64, 215, 134, 142, 123, 7, 230, 54, 52,
            211, 182, 48, 157, 191, 50, 219, 222, 147, 205, 0, 190, 11,
        ],
        [
            35, 5, 133, 217, 180, 43, 56, 102, 117, 79, 198, 160, 251, 151, 49, 167, 97, 168, 236,
            153, 46, 53, 97, 136, 177, 22, 144, 248, 209, 112, 2, 22,
        ],
        [
            189, 185, 158, 102, 165, 81, 53, 43, 92, 59, 187, 63, 53, 181, 50, 254, 28, 27, 251,
            184, 213, 163, 78, 124, 194, 6, 198, 11, 104, 110, 190, 58,
        ],
        [
            116, 193, 244, 154, 127, 157, 31, 127, 143, 165, 156, 200, 172, 124, 18, 75, 72, 222,
            71, 165, 110, 23, 37, 39, 40, 221, 208, 104, 143, 242, 10, 48,
        ],
        [
            183, 141, 193, 42, 36, 65, 222, 109, 178, 190, 244, 230, 12, 133, 148, 135, 165, 30,
            186, 27, 85, 88, 243, 247, 39, 160, 74, 215, 156, 107, 27, 18,
        ],
        [
            178, 132, 174, 93, 169, 172, 0, 113, 42, 161, 34, 61, 153, 62, 171, 231, 99, 205, 151,
            136, 195, 41, 104, 69, 129, 94, 118, 232, 230, 30, 41, 17,
        ],
        [
            168, 254, 122, 49, 132, 197, 175, 154, 5, 26, 244, 178, 204, 59, 101, 158, 244, 142,
            21, 29, 108, 129, 119, 165, 179, 98, 124, 213, 19, 29, 89, 20,
        ],
    ],
    [
        [
            58, 113, 74, 241, 171, 205, 88, 69, 237, 115, 76, 207, 136, 236, 87, 123, 162, 139,
            106, 6, 32, 222, 117, 231, 41, 193, 154, 129, 90, 25, 140, 32,
        ],
        [
            230, 114, 111, 31, 111, 230, 200, 253, 222, 93, 21, 215, 146, 39, 154, 45, 176, 255,
            252, 145, 47, 232, 32, 241, 128, 121, 239, 189, 41, 31, 178, 36,
        ],
        [
            174, 68, 205, 209, 74, 239, 125, 252, 66, 216, 148, 150, 245, 194, 226, 111, 75, 204,
            15, 141, 251, 66, 147, 233, 225, 226, 32, 128, 223, 237, 242, 59,
        ],
        [
            3, 121, 18, 77, 255, 155, 204, 130, 228, 64, 12, 91, 2, 202, 55, 137, 126, 167, 237,
            179, 121, 95, 97, 237, 79, 161, 223, 249, 20, 104, 224, 38,
        ],
        [
            200, 199, 195, 205, 182, 178, 120, 64, 102, 134, 15, 97, 173, 148, 179, 163, 187, 169,
            183, 49, 254, 95, 207, 243, 104, 207, 129, 137, 204, 95, 68, 40,
        ],
        [
            205, 24, 7, 183, 41, 150, 50, 243, 52, 134, 179, 221, 154, 3, 50, 15, 125, 141, 44,
            229, 174, 113, 51, 207, 142, 158, 157, 165, 171, 33, 167, 38,
        ],
        [
            93, 0, 212, 151, 114, 45, 134, 198, 142, 228, 90, 196, 237, 19, 187, 131, 51, 19, 28,
            175, 104, 227, 105, 196, 131, 167, 79, 38, 138, 196, 51, 2,
        ],
        [
            59, 126, 116, 119, 162, 37, 37, 83, 6, 89, 187, 30, 31, 41, 207, 146, 78, 210, 47, 181,
            12, 196, 180, 125, 230, 122, 253, 44, 241, 73, 220, 41,
        ],
    ],
    [
        [
            93, 227, 53, 151, 63, 84, 124, 154, 237, 167, 104, 227, 134, 9, 69, 169, 213, 20, 129,
            164, 5, 139, 107, 214, 219, 181, 245, 2, 133, 23, 167, 51,
        ],
        [
            52, 208, 36, 128, 213, 207, 33, 248, 5, 208, 172, 141, 116, 118, 32, 61, 70, 206, 220,
            2, 177, 33, 220, 229, 217, 39, 243, 191, 204, 59, 103, 13,
        ],
        [
            137, 0, 31, 220, 219, 195, 242, 66, 222, 9, 30, 49, 67, 147, 242, 237, 184, 155, 92,
            127, 0, 138, 163, 239, 31, 52, 29, 248, 179, 142, 178, 51,
        ],
        [
            9, 247, 72, 70, 229, 95, 189, 205, 92, 112, 59, 237, 203, 45, 169, 254, 130, 56, 222,
            97, 91, 53, 5, 79, 254, 144, 144, 139, 249, 44, 112, 10,
        ],
        [
            142, 182, 137, 135, 132, 12, 193, 117, 33, 4, 3, 131, 45, 71, 220, 57, 220, 109, 192,
            168, 19, 114, 226, 245, 71, 204, 104, 87, 86, 129, 252, 48,
        ],
        [
            155, 212, 212, 132, 134, 186, 221, 90, 141, 120, 16, 198, 25, 43, 18, 134, 198, 62,
            173, 11, 168, 215, 104, 237, 241, 111, 108, 112, 148, 121, 90, 25,
        ],
        [
            237, 57, 126, 105, 21, 13, 14, 166, 233, 74, 163, 1, 182, 44, 234, 186, 213, 219, 38,
            5, 226, 108, 110, 255, 36, 207, 85, 60, 152, 169, 241, 14,
        ],
        [
            219, 43, 21, 203, 72, 178, 62, 213, 60, 192, 3, 20, 90, 10, 186, 134, 181, 154, 0, 53,
            75, 214, 211, 194, 227, 32, 13, 159, 93, 183, 134, 29,
        ],
    ],
    [
        [
            49, 16, 184, 146, 99, 19, 153, 194, 123, 41, 24, 235, 243, 73, 9, 215, 195, 186, 58,
            253, 33, 124, 76, 111, 151, 179, 252, 251, 103, 1, 100, 19,
        ],
        [
            15, 173, 71, 88, 104, 55, 212, 113, 228, 43, 4, 246, 137, 5, 251, 183, 137, 34, 16,
            234, 130, 208, 240, 188, 128, 118, 175, 127, 198, 150, 228, 54,
        ],
        [
            19, 194, 219, 217, 24, 146, 37, 223, 241, 118, 130, 17, 106, 48, 243, 74, 75, 45, 249,
            82, 100, 196, 236, 235, 144, 119, 189, 233, 209, 224, 89, 37,
        ],
        [
            5, 204, 210, 20, 79, 207, 213, 247, 252, 195, 1, 101, 247, 54, 70, 221, 19, 126, 205,
            230, 224, 154, 57, 24, 251, 138, 104, 107, 145, 143, 240, 22,
        ],
        [
            178, 233, 194, 212, 193, 74, 25, 74, 15, 201, 106, 114, 6, 248, 134, 168, 129, 110,
            148, 125, 208, 42, 9, 90, 43, 255, 71, 206, 153, 173, 12, 42,
        ],
        [
            119, 68, 2, 6, 31, 24, 160, 117, 87, 143, 120, 3, 215, 255, 247, 207, 125, 158, 23, 39,
            73, 106, 15, 240, 170, 155, 168, 142, 89, 225, 234, 63,
        ],
        [
            2, 252, 41, 89, 60, 106, 201, 142, 88, 102, 71, 228, 123, 206, 24, 191, 1, 253, 100,
            123, 184, 7, 217, 120, 110, 91, 223, 31, 201, 144, 243, 15,
        ],
        [
            93, 40, 93, 41, 170, 15, 155, 192, 244, 60, 25, 163, 9, 222, 26, 176, 200, 141, 148,
            145, 57, 182, 188, 135, 57, 156, 54, 242, 130, 162, 218, 45,
        ],
    ],
    [
        [
            214, 69, 246, 81, 233, 237, 251, 120, 2, 67, 240, 114, 30, 75, 200, 44, 1, 53, 148,
            130, 104, 10, 26, 136, 234, 145, 255, 246, 127, 216, 194, 32,
        ],
        [
            8, 65, 67, 164, 14, 155, 196, 111, 151, 196, 74, 150, 62, 189, 131, 110, 205, 122, 128,
            186, 236, 28, 51, 10, 206, 109, 134, 57, 87, 101, 93, 63,
        ],
        [
            68, 251, 222, 27, 177, 128, 75, 246, 128, 26, 58, 21, 141, 73, 219, 42, 229, 3, 99, 79,
            130, 94, 144, 140, 141, 29, 53, 13, 199, 208, 48, 59,
        ],
        [
            121, 169, 146, 165, 201, 195, 186, 28, 83, 140, 53, 245, 69, 22, 126, 233, 44, 39, 159,
            191, 14, 124, 20, 39, 211, 99, 13, 89, 204, 83, 245, 48,
        ],
        [
            241, 91, 55, 71, 30, 143, 172, 30, 17, 71, 66, 49, 156, 12, 144, 152, 162, 58, 65, 31,
            79, 2, 122, 225, 125, 125, 153, 221, 22, 197, 140, 37,
        ],
        [
            124, 252, 189, 185, 58, 35, 47, 92, 69, 63, 199, 229, 172, 244, 39, 99, 171, 41, 82,
            29, 253, 54, 46, 8, 131, 43, 68, 128, 109, 196, 72, 49,
        ],
        [
            151, 169, 183, 67, 74, 131, 44, 160, 250, 82, 177, 35, 52, 186, 12, 2, 39, 80, 220,
            239, 222, 107, 96, 203, 129, 128, 190, 117, 141, 234, 237, 7,
        ],
        [
            251, 84, 120, 50, 59, 6, 204, 109, 171, 19, 119, 102, 17, 119, 90, 250, 186, 203, 7,
            226, 58, 117, 211, 98, 120, 62, 139, 222, 92, 87, 237, 59,
        ],
    ],
    [
        [
            191, 132, 136, 12, 31, 227, 14, 77, 18, 230, 96, 126, 159, 194, 114, 113, 28, 229, 95,
            102, 160, 121, 135, 222, 1, 204, 166, 147, 27, 78, 94, 63,
        ],
        [
            41, 189, 153, 118, 73, 216, 74, 136, 32, 117, 103, 71, 64, 171, 143, 4, 185, 64, 223,
            191, 207, 225, 91, 3, 220, 58, 84, 161, 4, 186, 255, 61,
        ],
        [
            220, 110, 238, 123, 194, 162, 87, 36, 98, 216, 59, 236, 84, 52, 113, 46, 60, 29, 97,
            191, 141, 77, 148, 189, 79, 189, 144, 241, 49, 163, 107, 17,
        ],
        [
            12, 58, 75, 102, 5, 119, 46, 98, 201, 99, 246, 5, 148, 216, 73, 204, 226, 111, 142, 35,
            12, 56, 248, 38, 114, 250, 183, 85, 199, 74, 219, 21,
        ],
        [
            84, 181, 225, 125, 40, 19, 155, 140, 118, 241, 20, 21, 158, 167, 93, 244, 2, 17, 21,
            91, 2, 83, 1, 94, 22, 55, 135, 167, 223, 24, 32, 42,
        ],
        [
            6, 100, 231, 118, 1, 253, 62, 137, 179, 245, 83, 61, 171, 224, 96, 238, 76, 134, 31,
            107, 133, 47, 160, 13, 10, 43, 78, 93, 177, 12, 195, 17,
        ],
        [
            81, 34, 138, 58, 81, 179, 240, 49, 216, 93, 153, 221, 161, 206, 140, 17, 40, 75, 244,
            16, 220, 69, 242, 179, 209, 48, 169, 69, 163, 216, 79, 19,
        ],
        [
            60, 205, 53, 247, 139, 167, 157, 115, 224, 197, 206, 220, 238, 82, 218, 253, 74, 237,
            137, 50, 31, 151, 177, 175, 197, 44, 58, 147, 109, 119, 51, 1,
        ],
    ],
    [
        [
            27, 15, 21, 190, 82, 194, 14, 56, 128, 44, 111, 155, 238, 130, 188, 19, 119, 238, 0, 4,
            91, 86, 254, 61, 166, 67, 119, 22, 106, 244, 56, 47,
        ],
        [
            23, 52, 161, 183, 10, 194, 104, 78, 22, 66, 219, 166, 215, 76, 71, 106, 108, 153, 245,
            170, 105, 53, 197, 234, 219, 226, 234, 184, 86, 189, 222, 10,
        ],
        [
            231, 5, 115, 95, 58, 66, 208, 135, 209, 47, 123, 58, 134, 156, 88, 202, 13, 79, 225, 8,
            164, 39, 127, 246, 43, 136, 188, 177, 254, 151, 70, 23,
        ],
        [
            86, 248, 109, 23, 128, 55, 179, 214, 159, 18, 33, 106, 34, 109, 190, 0, 110, 224, 62,
            210, 87, 190, 246, 206, 196, 141, 133, 226, 167, 133, 19, 47,
        ],
        [
            70, 58, 113, 121, 237, 219, 30, 160, 229, 88, 95, 214, 8, 232, 108, 98, 14, 233, 43,
            189, 225, 16, 247, 36, 111, 122, 199, 25, 205, 44, 112, 42,
        ],
        [
            184, 97, 40, 59, 116, 147, 139, 60, 30, 250, 67, 11, 242, 111, 104, 150, 15, 215, 245,
            99, 31, 128, 240, 115, 18, 249, 99, 114, 186, 65, 76, 26,
        ],
        [
            183, 90, 33, 10, 78, 210, 54, 79, 156, 8, 176, 24, 54, 209, 203, 157, 151, 134, 88,
            104, 211, 62, 38, 197, 117, 122, 207, 15, 163, 5, 59, 8,
        ],
        [
            41, 14, 171, 211, 125, 45, 149, 132, 101, 32, 35, 142, 108, 194, 13, 37, 122, 27, 219,
            223, 32, 41, 153, 199, 167, 246, 214, 240, 222, 131, 85, 9,
        ],
    ],
    [
        [
            146, 73, 118, 250, 189, 115, 96, 188, 162, 168, 121, 1, 55, 221, 181, 187, 145, 13,
            213, 160, 250, 117, 252, 42, 161, 41, 42, 60, 45, 193, 25, 30,
        ],
        [
            38, 126, 2, 131, 102, 66, 220, 138, 83, 239, 80, 46, 245, 63, 94, 233, 192, 15, 173,
            155, 6, 154, 124, 236, 99, 151, 106, 21, 10, 55, 175, 45,
        ],
        [
            147, 163, 56, 143, 203, 206, 180, 175, 15, 39, 185, 183, 254, 241, 235, 182, 62, 116,
            27, 66, 73, 66, 55, 190, 19, 68, 189, 214, 46, 89, 13, 49,
        ],
        [
            121, 248, 233, 93, 196, 140, 233, 226, 177, 11, 177, 78, 181, 222, 38, 101, 216, 68,
            10, 26, 212, 185, 105, 39, 89, 190, 15, 184, 32, 157, 147, 35,
        ],
        [
            67, 108, 70, 93, 37, 202, 52, 114, 7, 214, 41, 246, 220, 152, 65, 195, 75, 216, 193,
            159, 110, 113, 85, 105, 182, 55, 126, 48, 92, 74, 41, 13,
        ],
        [
            66, 55, 15, 17, 96, 43, 142, 225, 55, 202, 242, 132, 16, 128, 198, 64, 57, 51, 173,
            153, 18, 21, 74, 117, 208, 85, 87, 200, 8, 25, 30, 53,
        ],
        [
            142, 173, 137, 206, 188, 2, 155, 106, 110, 80, 138, 211, 142, 216, 9, 126, 115, 161,
            60, 119, 251, 199, 29, 163, 69, 166, 106, 200, 115, 12, 27, 42,
        ],
        [
            5, 28, 251, 152, 246, 99, 74, 222, 86, 91, 4, 141, 212, 14, 200, 222, 129, 133, 91,
            117, 4, 233, 235, 254, 214, 85, 56, 28, 211, 167, 164, 7,
        ],
    ],
    [
        [
            199, 248, 56, 7, 116, 43, 199, 175, 73, 62, 19, 67, 250, 144, 43, 227, 245, 85, 242,
            25, 74, 33, 89, 208, 243, 137, 220, 136, 0, 79, 235, 17,
        ],
        [
            141, 87, 177, 94, 249, 243, 42, 2, 191, 181, 12, 101, 121, 76, 243, 215, 5, 157, 170,
            98, 185, 102, 238, 161, 179, 91, 225, 102, 17, 243, 151, 18,
        ],
        [
            5, 165, 124, 37, 121, 192, 201, 221, 50, 34, 204, 74, 249, 97, 138, 157, 183, 155, 191,
            222, 146, 82, 129, 195, 236, 32, 110, 47, 32, 32, 212, 59,
        ],
        [
            9, 21, 25, 227, 120, 92, 35, 43, 121, 43, 234, 147, 157, 23, 218, 121, 248, 28, 137,
            152, 121, 231, 163, 87, 212, 168, 222, 177, 182, 50, 19, 26,
        ],
        [
            214, 240, 41, 113, 104, 195, 60, 163, 221, 18, 241, 139, 35, 166, 231, 241, 137, 56,
            217, 182, 177, 214, 146, 157, 176, 171, 13, 41, 204, 80, 94, 53,
        ],
        [
            5, 160, 27, 29, 63, 102, 96, 140, 192, 60, 193, 95, 114, 182, 5, 102, 115, 144, 60, 92,
            83, 249, 85, 161, 172, 2, 96, 118, 95, 245, 201, 58,
        ],
        [
            35, 161, 31, 237, 123, 141, 123, 90, 233, 91, 250, 0, 189, 31, 19, 179, 134, 219, 230,
            7, 54, 236, 238, 251, 183, 245, 255, 172, 233, 240, 4, 27,
        ],
        [
            2, 165, 109, 76, 190, 46, 187, 163, 33, 215, 48, 19, 0, 240, 15, 120, 7, 191, 169, 213,
            118, 162, 0, 79, 58, 146, 185, 165, 77, 215, 43, 16,
        ],
    ],
    [
        [
            139, 175, 132, 197, 168, 169, 208, 139, 136, 29, 122, 201, 243, 76, 13, 159, 57, 180,
            235, 124, 157, 231, 66, 8, 233, 163, 50, 145, 44, 186, 159, 39,
        ],
        [
            18, 215, 223, 109, 50, 114, 37, 123, 232, 3, 220, 242, 61, 30, 200, 62, 103, 11, 171,
            156, 78, 117, 46, 234, 174, 247, 58, 238, 51, 180, 32, 19,
        ],
        [
            101, 233, 249, 129, 23, 236, 206, 133, 169, 247, 127, 5, 25, 115, 137, 232, 198, 119,
            126, 232, 19, 27, 53, 153, 229, 149, 149, 92, 202, 120, 23, 34,
        ],
        [
            195, 226, 216, 96, 174, 47, 198, 78, 76, 136, 67, 64, 61, 50, 64, 28, 70, 131, 105, 96,
            54, 197, 98, 18, 197, 130, 92, 155, 116, 104, 213, 28,
        ],
        [
            4, 165, 219, 75, 186, 168, 40, 93, 124, 34, 52, 233, 148, 177, 130, 40, 113, 173, 207,
            243, 123, 185, 251, 0, 111, 209, 114, 114, 0, 165, 47, 21,
        ],
        [
            15, 211, 38, 24, 236, 47, 246, 65, 238, 204, 84, 203, 250, 41, 211, 141, 195, 16, 196,
            184, 251, 47, 229, 78, 183, 45, 82, 65, 59, 228, 223, 38,
        ],
        [
            226, 208, 172, 170, 113, 92, 148, 103, 53, 253, 157, 15, 1, 15, 188, 215, 214, 21, 122,
            252, 33, 72, 164, 8, 8, 129, 2, 81, 175, 88, 16, 42,
        ],
        [
            120, 112, 122, 188, 218, 169, 97, 108, 132, 61, 177, 126, 146, 248, 48, 72, 231, 104,
            58, 58, 245, 127, 126, 68, 82, 142, 8, 42, 111, 149, 188, 32,
        ],
    ],
    [
        [
            95, 8, 182, 234, 116, 76, 223, 240, 103, 146, 57, 57, 56, 106, 176, 215, 49, 209, 32,
            158, 35, 28, 130, 7, 201, 178, 230, 212, 200, 223, 206, 62,
        ],
        [
            239, 66, 31, 68, 227, 197, 138, 254, 87, 118, 79, 186, 88, 181, 83, 240, 159, 11, 15,
            255, 84, 228, 69, 3, 205, 39, 11, 178, 223, 249, 193, 56,
        ],
        [
            159, 153, 62, 236, 201, 9, 233, 213, 145, 93, 104, 128, 1, 156, 105, 1, 141, 33, 122,
            15, 197, 68, 57, 25, 48, 67, 17, 219, 227, 252, 72, 62,
        ],
        [
            73, 22, 127, 172, 163, 146, 170, 68, 75, 175, 220, 162, 160, 27, 99, 36, 24, 11, 219,
            20, 38, 174, 178, 85, 133, 51, 30, 87, 210, 46, 109, 49,
        ],
        [
            37, 59, 243, 109, 246, 173, 114, 98, 148, 92, 118, 145, 45, 46, 163, 208, 44, 248, 6,
            192, 35, 46, 20, 0, 78, 120, 119, 214, 200, 148, 11, 45,
        ],
        [
            142, 201, 84, 209, 45, 193, 134, 251, 249, 3, 16, 208, 69, 91, 69, 255, 76, 221, 82,
            169, 30, 218, 46, 183, 12, 35, 214, 84, 23, 245, 74, 34,
        ],
        [
            169, 46, 120, 5, 117, 166, 109, 155, 218, 69, 132, 230, 139, 49, 118, 147, 121, 143,
            14, 200, 114, 128, 170, 214, 123, 83, 79, 227, 170, 87, 149, 60,
        ],
        [
            63, 140, 50, 114, 17, 41, 117, 40, 247, 224, 249, 168, 254, 222, 74, 122, 134, 252, 35,
            27, 213, 180, 92, 240, 255, 145, 13, 103, 152, 170, 236, 50,
        ],
    ],
    [
        [
            215, 154, 109, 148, 179, 184, 4, 246, 126, 31, 67, 190, 52, 170, 95, 63, 21, 74, 205,
            159, 226, 106, 95, 251, 151, 33, 95, 9, 8, 51, 54, 16,
        ],
        [
            125, 211, 129, 224, 62, 115, 145, 137, 123, 63, 37, 176, 121, 83, 200, 198, 200, 63,
            199, 7, 111, 61, 16, 189, 147, 128, 57, 111, 140, 117, 192, 13,
        ],
        [
            62, 162, 174, 44, 90, 198, 105, 23, 32, 95, 170, 43, 95, 202, 41, 143, 5, 120, 103,
            243, 69, 109, 70, 226, 206, 144, 32, 187, 168, 156, 74, 2,
        ],
        [
            74, 155, 142, 102, 116, 7, 191, 222, 129, 124, 146, 194, 182, 170, 38, 29, 14, 163,
            145, 158, 214, 61, 192, 123, 190, 109, 174, 52, 88, 52, 154, 21,
        ],
        [
            210, 110, 65, 168, 192, 245, 112, 81, 43, 66, 225, 107, 42, 112, 186, 67, 146, 66, 69,
            141, 69, 125, 61, 226, 103, 26, 102, 60, 12, 207, 245, 34,
        ],
        [
            0, 109, 234, 239, 154, 137, 77, 134, 209, 28, 113, 244, 198, 228, 214, 139, 52, 33, 86,
            69, 51, 218, 213, 211, 31, 28, 195, 77, 222, 90, 231, 0,
        ],
        [
            62, 169, 126, 192, 19, 148, 244, 128, 245, 147, 155, 85, 225, 132, 157, 156, 111, 26,
            103, 87, 100, 195, 177, 53, 58, 124, 93, 38, 187, 56, 176, 59,
        ],
        [
            224, 216, 241, 35, 55, 36, 233, 159, 110, 128, 30, 204, 91, 50, 227, 153, 76, 84, 209,
            62, 140, 19, 74, 222, 67, 215, 86, 219, 28, 6, 127, 26,
        ],
    ],
    [
        [
            196, 239, 163, 163, 226, 62, 34, 86, 120, 237, 194, 94, 87, 123, 11, 194, 71, 232, 84,
            212, 162, 54, 89, 87, 229, 244, 147, 80, 81, 231, 144, 59,
        ],
        [
            3, 183, 214, 69, 55, 0, 243, 100, 210, 173, 147, 22, 164, 222, 22, 236, 41, 16, 23,
            212, 64, 207, 16, 104, 62, 156, 117, 194, 143, 252, 114, 30,
        ],
        [
            83, 242, 71, 100, 1, 160, 144, 96, 22, 203, 210, 122, 123, 222, 221, 58, 212, 89, 96,
            210, 172, 144, 144, 217, 3, 21, 57, 248, 252, 36, 168, 45,
        ],
        [
            162, 85, 218, 40, 224, 40, 96, 145, 190, 124, 73, 217, 211, 89, 160, 248, 157, 203,
            174, 61, 201, 102, 43, 232, 246, 173, 125, 249, 96, 147, 84, 28,
        ],
        [
            127, 164, 182, 128, 219, 61, 29, 77, 246, 20, 106, 153, 148, 217, 135, 214, 48, 10,
            100, 82, 166, 117, 154, 82, 30, 129, 66, 224, 97, 15, 108, 42,
        ],
        [
            44, 25, 145, 64, 165, 137, 106, 115, 213, 161, 96, 217, 45, 60, 88, 226, 124, 219, 43,
            200, 254, 197, 37, 111, 72, 159, 204, 121, 132, 253, 215, 63,
        ],
        [
            130, 113, 153, 54, 123, 127, 228, 89, 131, 168, 79, 57, 6, 226, 59, 134, 92, 149, 83,
            208, 71, 194, 150, 140, 231, 16, 60, 214, 214, 93, 159, 34,
        ],
        [
            198, 56, 77, 220, 125, 205, 170, 180, 213, 78, 72, 22, 51, 57, 27, 7, 9, 212, 232, 210,
            105, 136, 110, 193, 162, 159, 52, 143, 41, 167, 211, 29,
        ],
    ],
    [
        [
            160, 165, 254, 36, 87, 6, 88, 21, 72, 216, 230, 247, 2, 55, 242, 73, 36, 213, 230, 8,
            90, 80, 77, 158, 60, 59, 121, 94, 59, 96, 94, 21,
        ],
        [
            45, 176, 13, 86, 112, 155, 195, 206, 252, 211, 160, 213, 218, 77, 219, 237, 222, 47,
            75, 7, 27, 201, 208, 42, 78, 5, 247, 152, 111, 163, 28, 19,
        ],
        [
            199, 71, 216, 75, 50, 105, 91, 201, 53, 124, 235, 0, 108, 165, 181, 70, 42, 153, 163,
            201, 11, 54, 41, 233, 146, 38, 184, 139, 125, 91, 209, 10,
        ],
        [
            86, 137, 234, 158, 219, 91, 233, 65, 189, 187, 15, 107, 19, 111, 12, 249, 160, 129,
            246, 146, 62, 110, 87, 239, 99, 96, 143, 149, 115, 75, 64, 52,
        ],
        [
            98, 114, 143, 6, 251, 147, 133, 224, 83, 35, 220, 23, 118, 87, 253, 157, 125, 207, 180,
            57, 5, 127, 14, 24, 191, 18, 189, 133, 24, 63, 98, 21,
        ],
        [
            7, 253, 227, 108, 211, 101, 224, 147, 234, 204, 96, 88, 155, 213, 73, 12, 78, 63, 195,
            95, 231, 53, 32, 242, 35, 185, 9, 226, 163, 182, 106, 3,
        ],
        [
            25, 246, 155, 59, 73, 164, 80, 198, 181, 76, 138, 14, 37, 122, 78, 226, 38, 183, 227,
            225, 221, 67, 143, 190, 95, 113, 73, 57, 112, 85, 221, 21,
        ],
        [
            149, 94, 77, 90, 128, 207, 67, 153, 17, 196, 236, 238, 199, 44, 10, 245, 147, 164, 221,
            226, 230, 123, 118, 33, 85, 34, 246, 225, 65, 251, 160, 26,
        ],
    ],
    [
        [
            198, 30, 82, 10, 148, 182, 240, 154, 54, 243, 247, 229, 181, 129, 228, 22, 161, 215,
            77, 251, 121, 190, 107, 85, 56, 41, 194, 91, 243, 88, 14, 58,
        ],
        [
            202, 246, 182, 231, 195, 153, 88, 238, 60, 84, 67, 178, 248, 59, 105, 49, 195, 130,
            228, 104, 204, 136, 202, 84, 168, 146, 5, 208, 62, 85, 25, 17,
        ],
        [
            119, 64, 199, 157, 4, 43, 91, 198, 55, 245, 237, 178, 21, 83, 35, 100, 246, 74, 152,
            132, 234, 245, 211, 248, 224, 198, 126, 176, 196, 121, 100, 50,
        ],
        [
            101, 85, 104, 70, 109, 112, 93, 153, 253, 233, 162, 249, 142, 220, 208, 19, 165, 218,
            66, 255, 125, 124, 227, 42, 223, 153, 75, 69, 155, 156, 31, 57,
        ],
        [
            63, 55, 139, 112, 194, 107, 121, 53, 129, 202, 204, 46, 248, 109, 34, 241, 53, 227,
            206, 243, 178, 228, 215, 86, 153, 195, 47, 114, 97, 169, 5, 27,
        ],
        [
            53, 75, 214, 129, 18, 225, 191, 83, 159, 41, 109, 40, 176, 57, 43, 164, 161, 185, 225,
            136, 251, 243, 193, 42, 95, 115, 32, 171, 12, 213, 167, 3,
        ],
        [
            235, 36, 235, 197, 64, 34, 134, 89, 21, 195, 40, 218, 162, 235, 91, 98, 189, 163, 189,
            186, 29, 184, 247, 93, 83, 134, 190, 214, 231, 255, 49, 5,
        ],
        [
            36, 53, 226, 5, 178, 106, 251, 156, 151, 206, 13, 255, 45, 147, 244, 47, 145, 220, 199,
            136, 195, 250, 170, 26, 117, 196, 227, 93, 178, 18, 178, 54,
        ],
    ],
    [
        [
            91, 68, 35, 111, 130, 120, 151, 184, 175, 224, 185, 116, 137, 151, 158, 155, 189, 79,
            221, 201, 73, 145, 154, 99, 84, 54, 37, 92, 60, 129, 214, 44,
        ],
        [
            231, 28, 49, 225, 237, 154, 194, 248, 14, 192, 101, 134, 78, 32, 217, 133, 35, 203,
            126, 217, 65, 210, 226, 254, 67, 48, 218, 213, 163, 219, 160, 50,
        ],
        [
            195, 124, 201, 154, 77, 19, 27, 255, 51, 178, 233, 32, 112, 70, 138, 116, 23, 99, 22,
            170, 23, 205, 137, 213, 91, 207, 198, 118, 66, 21, 222, 50,
        ],
        [
            149, 89, 237, 156, 70, 100, 84, 58, 87, 158, 41, 169, 249, 94, 134, 84, 109, 110, 191,
            60, 29, 237, 6, 215, 116, 124, 158, 30, 58, 205, 244, 17,
        ],
        [
            80, 179, 104, 84, 95, 131, 131, 215, 129, 166, 153, 81, 167, 251, 66, 242, 254, 45,
            100, 1, 161, 77, 234, 73, 72, 85, 152, 84, 251, 23, 114, 26,
        ],
        [
            122, 218, 71, 58, 4, 142, 173, 59, 70, 145, 210, 95, 79, 43, 69, 197, 200, 23, 165,
            251, 43, 41, 216, 32, 187, 240, 177, 81, 137, 144, 198, 61,
        ],
        [
            40, 250, 126, 111, 65, 53, 57, 76, 207, 56, 219, 216, 128, 31, 63, 72, 34, 174, 157,
            129, 159, 143, 16, 42, 162, 60, 231, 208, 218, 112, 16, 63,
        ],
        [
            203, 16, 248, 1, 77, 5, 210, 197, 69, 134, 121, 102, 244, 42, 182, 244, 143, 71, 179,
            152, 34, 139, 52, 89, 180, 232, 152, 56, 149, 0, 97, 37,
        ],
    ],
    [
        [
            5, 89, 207, 4, 33, 94, 36, 209, 180, 97, 214, 235, 236, 90, 149, 192, 144, 15, 127,
            233, 150, 20, 79, 10, 72, 207, 61, 43, 159, 78, 107, 53,
        ],
        [
            79, 186, 12, 1, 247, 170, 13, 98, 141, 222, 29, 5, 91, 198, 14, 85, 242, 207, 252, 151,
            191, 37, 3, 221, 27, 83, 31, 250, 45, 99, 122, 16,
        ],
        [
            137, 124, 62, 217, 230, 197, 251, 81, 224, 75, 111, 172, 69, 51, 172, 138, 87, 147, 33,
            128, 200, 169, 159, 176, 204, 10, 225, 2, 139, 30, 160, 49,
        ],
        [
            102, 7, 138, 62, 135, 48, 100, 156, 184, 139, 98, 191, 161, 89, 48, 104, 32, 241, 60,
            135, 3, 71, 117, 99, 5, 215, 250, 18, 212, 34, 64, 20,
        ],
        [
            197, 161, 228, 116, 31, 179, 142, 77, 140, 217, 94, 108, 137, 35, 39, 126, 84, 201, 71,
            58, 10, 52, 7, 184, 156, 65, 185, 188, 182, 98, 65, 43,
        ],
        [
            207, 153, 115, 73, 75, 200, 35, 16, 32, 82, 68, 212, 14, 206, 254, 63, 56, 140, 59,
            185, 68, 204, 73, 67, 27, 95, 207, 228, 74, 72, 18, 2,
        ],
        [
            185, 232, 177, 216, 67, 26, 72, 76, 157, 100, 17, 181, 161, 81, 150, 245, 177, 250, 28,
            58, 130, 154, 25, 212, 136, 67, 140, 88, 161, 41, 232, 7,
        ],
        [
            81, 249, 170, 40, 188, 151, 82, 232, 199, 203, 11, 66, 67, 143, 120, 54, 251, 4, 180,
            183, 125, 142, 17, 249, 162, 6, 177, 157, 59, 88, 176, 60,
        ],
    ],
    [
        [
            5, 39, 116, 70, 87, 101, 17, 78, 253, 249, 241, 133, 179, 20, 136, 35, 91, 141, 137,
            92, 41, 59, 37, 107, 234, 106, 220, 49, 41, 20, 228, 50,
        ],
        [
            239, 224, 34, 219, 49, 32, 61, 153, 160, 42, 234, 0, 74, 179, 81, 112, 224, 236, 181,
            23, 23, 195, 36, 114, 166, 23, 23, 86, 131, 165, 107, 5,
        ],
        [
            157, 92, 182, 140, 251, 56, 162, 62, 72, 102, 59, 203, 169, 127, 177, 139, 173, 223,
            193, 220, 173, 108, 133, 146, 220, 50, 125, 176, 182, 249, 232, 4,
        ],
        [
            63, 192, 18, 230, 178, 221, 22, 4, 11, 89, 10, 235, 5, 15, 142, 128, 116, 213, 251, 13,
            33, 120, 164, 251, 113, 177, 195, 141, 146, 137, 224, 10,
        ],
        [
            146, 171, 247, 18, 200, 10, 47, 184, 29, 129, 50, 237, 55, 8, 93, 157, 184, 118, 158,
            97, 116, 202, 135, 240, 162, 226, 131, 135, 146, 30, 31, 55,
        ],
        [
            240, 233, 41, 237, 47, 32, 136, 122, 213, 32, 160, 3, 47, 216, 193, 98, 240, 97, 202,
            50, 205, 150, 40, 36, 87, 8, 132, 22, 53, 105, 118, 49,
        ],
        [
            58, 230, 109, 52, 109, 115, 162, 116, 80, 181, 12, 159, 59, 246, 184, 98, 98, 100, 163,
            231, 95, 152, 12, 195, 216, 26, 110, 75, 162, 52, 155, 36,
        ],
        [
            1, 160, 91, 80, 221, 176, 161, 223, 120, 239, 80, 39, 92, 193, 131, 72, 216, 141, 107,
            157, 188, 163, 246, 59, 97, 109, 38, 108, 175, 125, 229, 25,
        ],
    ],
    [
        [
            46, 149, 1, 150, 69, 179, 76, 218, 4, 111, 196, 93, 98, 116, 217, 54, 62, 78, 228, 34,
            223, 95, 111, 200, 16, 142, 189, 142, 54, 126, 218, 4,
        ],
        [
            251, 202, 109, 62, 44, 11, 142, 106, 121, 128, 123, 3, 9, 30, 36, 167, 102, 117, 14,
            76, 11, 184, 80, 177, 118, 164, 162, 108, 33, 215, 19, 25,
        ],
        [
            198, 206, 168, 7, 85, 212, 232, 75, 69, 188, 47, 243, 205, 196, 159, 134, 115, 51, 173,
            117, 92, 119, 112, 120, 247, 137, 248, 56, 24, 89, 164, 58,
        ],
        [
            10, 143, 238, 110, 161, 146, 112, 53, 117, 115, 105, 110, 154, 126, 142, 110, 236, 6,
            24, 240, 220, 69, 251, 167, 134, 148, 63, 30, 137, 42, 107, 5,
        ],
        [
            143, 210, 102, 57, 97, 48, 92, 46, 119, 14, 133, 224, 196, 245, 141, 4, 132, 34, 58, 8,
            68, 179, 49, 1, 109, 89, 134, 116, 7, 181, 84, 44,
        ],
        [
            154, 170, 62, 94, 138, 234, 199, 118, 107, 57, 142, 126, 220, 120, 9, 134, 2, 37, 17,
            163, 72, 232, 113, 113, 29, 210, 2, 39, 104, 121, 17, 22,
        ],
        [
            50, 131, 237, 169, 131, 43, 30, 246, 70, 95, 0, 147, 100, 233, 236, 114, 161, 10, 83,
            70, 68, 14, 36, 144, 2, 37, 135, 16, 147, 70, 231, 2,
        ],
        [
            141, 174, 115, 187, 181, 226, 19, 76, 147, 156, 204, 125, 181, 255, 94, 24, 99, 152,
            170, 194, 18, 9, 52, 211, 183, 62, 3, 121, 150, 15, 160, 26,
        ],
    ],
    [
        [
            10, 39, 254, 165, 143, 140, 105, 76, 102, 131, 235, 58, 71, 161, 130, 71, 222, 115, 68,
            185, 196, 102, 223, 155, 199, 84, 188, 119, 208, 10, 24, 3,
        ],
        [
            251, 47, 0, 26, 188, 43, 51, 170, 4, 147, 19, 172, 125, 166, 44, 76, 174, 27, 174, 68,
            192, 97, 20, 120, 89, 203, 21, 165, 193, 221, 112, 19,
        ],
        [
            103, 90, 73, 145, 111, 139, 91, 135, 197, 12, 29, 183, 228, 247, 237, 160, 61, 197,
            123, 235, 42, 8, 238, 111, 114, 33, 99, 55, 90, 181, 9, 59,
        ],
        [
            136, 249, 201, 5, 56, 155, 117, 156, 164, 153, 151, 157, 204, 1, 122, 15, 169, 219,
            108, 95, 137, 140, 225, 50, 8, 254, 128, 176, 245, 148, 231, 47,
        ],
        [
            185, 227, 120, 188, 133, 61, 3, 169, 174, 45, 248, 48, 52, 20, 250, 157, 124, 212, 209,
            192, 13, 22, 22, 102, 11, 236, 194, 236, 0, 22, 55, 44,
        ],
        [
            172, 216, 57, 7, 93, 154, 231, 155, 201, 255, 159, 145, 116, 135, 127, 173, 185, 188,
            237, 178, 10, 184, 107, 170, 165, 137, 247, 107, 147, 237, 232, 3,
        ],
        [
            94, 197, 39, 16, 162, 134, 70, 119, 155, 56, 251, 82, 45, 190, 106, 99, 114, 38, 217,
            242, 188, 178, 248, 123, 15, 14, 239, 147, 109, 15, 127, 16,
        ],
        [
            152, 158, 191, 195, 22, 237, 186, 168, 85, 70, 228, 13, 111, 109, 5, 52, 154, 109, 80,
            252, 147, 178, 64, 196, 184, 138, 225, 26, 247, 183, 189, 9,
        ],
    ],
    [
        [
            153, 70, 167, 177, 229, 151, 123, 186, 51, 190, 231, 196, 226, 132, 37, 12, 27, 171,
            67, 175, 188, 253, 127, 246, 196, 250, 200, 188, 89, 51, 90, 35,
        ],
        [
            7, 47, 63, 135, 76, 144, 203, 18, 252, 212, 29, 27, 170, 100, 14, 233, 90, 100, 182,
            36, 77, 55, 117, 100, 149, 210, 106, 18, 54, 194, 40, 45,
        ],
        [
            145, 138, 183, 17, 90, 5, 158, 103, 234, 188, 115, 38, 10, 195, 12, 19, 111, 232, 36,
            96, 216, 151, 218, 196, 23, 234, 114, 63, 112, 245, 52, 17,
        ],
        [
            188, 191, 226, 36, 30, 19, 172, 66, 73, 35, 237, 184, 220, 146, 166, 50, 67, 252, 228,
            206, 137, 25, 124, 209, 115, 233, 176, 38, 97, 194, 168, 55,
        ],
        [
            156, 84, 233, 11, 139, 39, 117, 26, 180, 77, 208, 247, 156, 213, 39, 71, 105, 47, 14,
            93, 84, 220, 66, 221, 67, 31, 239, 193, 137, 75, 98, 4,
        ],
        [
            19, 1, 39, 184, 88, 13, 233, 13, 98, 92, 56, 181, 205, 181, 92, 50, 63, 164, 56, 233,
            101, 251, 60, 126, 212, 238, 191, 231, 248, 115, 174, 31,
        ],
        [
            45, 142, 9, 191, 151, 248, 78, 181, 181, 4, 226, 246, 186, 243, 96, 205, 28, 234, 235,
            103, 49, 205, 114, 132, 136, 106, 50, 51, 220, 56, 185, 54,
        ],
        [
            230, 1, 231, 54, 221, 188, 3, 230, 233, 72, 235, 37, 100, 117, 179, 200, 146, 196, 117,
            150, 221, 59, 169, 110, 142, 254, 169, 129, 192, 210, 124, 2,
        ],
    ],
    [
        [
            113, 127, 193, 36, 33, 98, 37, 229, 172, 76, 71, 176, 107, 47, 84, 130, 144, 108, 249,
            204, 247, 50, 143, 242, 185, 160, 207, 41, 83, 184, 30, 1,
        ],
        [
            60, 188, 219, 67, 6, 66, 64, 62, 105, 32, 121, 184, 30, 137, 100, 255, 176, 188, 74,
            132, 73, 99, 168, 95, 30, 231, 55, 219, 80, 28, 236, 57,
        ],
        [
            162, 221, 96, 144, 26, 59, 193, 103, 61, 148, 12, 148, 65, 41, 57, 22, 198, 112, 0, 25,
            116, 202, 173, 254, 225, 89, 14, 113, 132, 126, 173, 25,
        ],
        [
            211, 146, 138, 171, 246, 25, 51, 48, 83, 26, 132, 149, 160, 115, 242, 27, 161, 12, 76,
            130, 46, 16, 213, 54, 134, 95, 115, 43, 91, 197, 223, 46,
        ],
        [
            164, 147, 245, 58, 228, 119, 37, 72, 47, 194, 171, 113, 130, 106, 53, 214, 61, 56, 100,
            200, 159, 224, 18, 246, 94, 215, 48, 69, 190, 169, 51, 42,
        ],
        [
            132, 253, 120, 207, 143, 101, 57, 58, 167, 185, 23, 252, 157, 221, 248, 157, 46, 68,
            122, 151, 192, 184, 194, 142, 98, 70, 189, 107, 61, 76, 149, 61,
        ],
        [
            227, 38, 171, 227, 71, 147, 6, 219, 173, 77, 149, 187, 130, 150, 56, 156, 189, 132, 44,
            201, 189, 2, 133, 21, 174, 175, 141, 89, 76, 98, 35, 39,
        ],
        [
            112, 33, 226, 167, 129, 192, 243, 199, 106, 110, 29, 184, 195, 62, 72, 51, 203, 147,
            66, 200, 234, 235, 45, 248, 118, 78, 37, 131, 2, 115, 240, 35,
        ],
    ],
    [
        [
            29, 94, 98, 16, 75, 79, 201, 48, 252, 129, 82, 87, 55, 155, 214, 178, 0, 13, 80, 120,
            252, 26, 199, 98, 112, 117, 251, 132, 9, 222, 88, 1,
        ],
        [
            183, 159, 59, 28, 80, 19, 246, 226, 204, 42, 87, 98, 227, 225, 239, 180, 98, 43, 241,
            205, 139, 238, 215, 120, 185, 226, 163, 198, 184, 182, 245, 34,
        ],
        [
            197, 11, 112, 174, 119, 138, 42, 144, 151, 132, 47, 51, 241, 212, 173, 241, 153, 176,
            171, 224, 252, 197, 125, 153, 40, 30, 3, 243, 250, 146, 89, 33,
        ],
        [
            229, 206, 81, 251, 52, 183, 144, 121, 141, 120, 80, 108, 133, 246, 20, 114, 51, 177,
            66, 218, 113, 1, 198, 68, 86, 224, 175, 59, 85, 237, 146, 25,
        ],
        [
            199, 176, 127, 191, 59, 34, 251, 44, 251, 172, 34, 15, 151, 54, 230, 62, 38, 208, 221,
            40, 244, 213, 69, 138, 161, 105, 65, 136, 129, 153, 158, 14,
        ],
        [
            82, 194, 114, 87, 225, 102, 24, 119, 238, 85, 93, 21, 89, 149, 104, 144, 151, 19, 52,
            96, 84, 71, 69, 160, 87, 245, 239, 208, 59, 88, 92, 19,
        ],
        [
            182, 89, 228, 186, 60, 45, 23, 77, 190, 131, 89, 134, 109, 200, 222, 153, 254, 217,
            191, 37, 100, 96, 85, 165, 64, 182, 141, 244, 208, 85, 89, 42,
        ],
        [
            193, 216, 142, 50, 41, 162, 188, 146, 206, 33, 83, 35, 144, 127, 148, 243, 129, 98,
            209, 161, 59, 180, 96, 133, 225, 53, 12, 187, 177, 207, 80, 34,
        ],
    ],
    [
        [
            147, 41, 219, 169, 88, 177, 67, 111, 1, 56, 197, 17, 47, 232, 125, 5, 29, 159, 89, 40,
            154, 98, 87, 141, 85, 150, 231, 16, 222, 124, 173, 37,
        ],
        [
            202, 211, 119, 159, 104, 213, 253, 76, 62, 219, 254, 66, 220, 221, 192, 56, 122, 124,
            42, 139, 242, 215, 178, 229, 246, 79, 21, 121, 31, 183, 27, 48,
        ],
        [
            126, 218, 96, 65, 191, 72, 25, 144, 186, 197, 91, 220, 79, 245, 85, 156, 37, 197, 30,
            94, 10, 170, 200, 6, 38, 97, 135, 84, 253, 214, 251, 14,
        ],
        [
            48, 6, 251, 73, 216, 94, 185, 113, 9, 130, 250, 198, 78, 214, 160, 76, 245, 192, 171,
            66, 30, 225, 79, 58, 153, 29, 135, 234, 76, 130, 57, 3,
        ],
        [
            178, 120, 202, 176, 134, 4, 205, 224, 149, 195, 40, 140, 155, 181, 194, 86, 200, 34,
            38, 203, 133, 233, 248, 87, 190, 214, 183, 33, 226, 170, 80, 35,
        ],
        [
            19, 135, 66, 237, 59, 74, 11, 20, 115, 128, 39, 102, 218, 68, 225, 172, 232, 233, 63,
            231, 110, 210, 108, 51, 25, 43, 231, 125, 230, 47, 145, 32,
        ],
        [
            236, 165, 166, 249, 54, 207, 68, 121, 176, 134, 177, 143, 7, 104, 156, 1, 57, 62, 158,
            204, 51, 205, 187, 225, 56, 95, 255, 145, 207, 19, 137, 25,
        ],
        [
            193, 100, 46, 208, 77, 17, 64, 1, 103, 156, 154, 222, 122, 153, 178, 11, 101, 138, 156,
            74, 247, 158, 1, 74, 139, 90, 207, 2, 47, 131, 87, 29,
        ],
    ],
    [
        [
            9, 151, 216, 97, 151, 239, 26, 211, 241, 12, 192, 250, 40, 134, 192, 184, 76, 241, 13,
            93, 189, 139, 9, 63, 165, 232, 209, 97, 134, 128, 47, 25,
        ],
        [
            6, 215, 38, 225, 140, 33, 72, 85, 69, 32, 15, 206, 114, 187, 55, 198, 11, 4, 119, 45,
            48, 105, 157, 118, 14, 168, 134, 79, 202, 86, 107, 54,
        ],
        [
            155, 143, 59, 197, 19, 114, 1, 51, 150, 66, 32, 45, 71, 46, 23, 139, 173, 182, 83, 56,
            51, 209, 56, 207, 199, 111, 208, 234, 115, 93, 56, 6,
        ],
        [
            69, 81, 31, 176, 185, 244, 148, 134, 139, 89, 254, 245, 123, 72, 132, 8, 143, 17, 136,
            40, 232, 236, 23, 16, 49, 201, 214, 155, 143, 214, 68, 46,
        ],
        [
            103, 7, 147, 218, 180, 175, 110, 155, 209, 66, 81, 13, 47, 84, 202, 205, 59, 40, 230,
            40, 60, 237, 124, 116, 178, 122, 45, 76, 26, 168, 166, 4,
        ],
        [
            252, 98, 53, 28, 192, 10, 195, 191, 117, 188, 21, 45, 1, 14, 222, 141, 58, 57, 250,
            153, 163, 4, 227, 41, 61, 128, 72, 245, 187, 192, 246, 59,
        ],
        [
            226, 85, 143, 113, 156, 101, 10, 58, 234, 10, 73, 60, 40, 247, 222, 193, 27, 221, 7,
            61, 57, 100, 158, 132, 84, 101, 53, 162, 83, 170, 215, 49,
        ],
        [
            129, 32, 252, 254, 63, 237, 242, 129, 12, 162, 24, 70, 1, 58, 54, 56, 170, 225, 83,
            125, 77, 125, 105, 162, 239, 120, 30, 118, 26, 46, 45, 21,
        ],
    ],
    [
        [
            83, 141, 231, 4, 150, 71, 0, 103, 215, 244, 1, 244, 23, 161, 130, 12, 169, 207, 32, 79,
            56, 140, 75, 104, 119, 157, 228, 240, 79, 73, 51, 59,
        ],
        [
            234, 101, 136, 42, 80, 209, 52, 16, 174, 200, 12, 76, 182, 75, 79, 226, 173, 229, 58,
            57, 151, 88, 7, 252, 145, 51, 6, 142, 162, 20, 130, 29,
        ],
        [
            129, 10, 76, 145, 229, 149, 209, 78, 93, 209, 163, 248, 222, 215, 207, 48, 177, 196,
            136, 196, 57, 240, 145, 134, 0, 59, 120, 123, 58, 72, 103, 34,
        ],
        [
            159, 127, 28, 87, 113, 72, 104, 203, 67, 58, 211, 169, 138, 254, 31, 168, 1, 12, 221,
            178, 227, 218, 109, 93, 161, 116, 98, 24, 54, 136, 33, 11,
        ],
        [
            173, 29, 146, 100, 140, 178, 50, 51, 128, 61, 123, 161, 60, 75, 103, 218, 56, 121, 24,
            43, 147, 99, 210, 193, 172, 188, 106, 182, 135, 222, 57, 20,
        ],
        [
            166, 29, 73, 107, 24, 198, 36, 111, 3, 25, 183, 183, 143, 82, 96, 98, 203, 35, 146, 82,
            49, 121, 135, 63, 191, 33, 165, 223, 230, 26, 71, 51,
        ],
        [
            140, 119, 109, 157, 12, 201, 64, 255, 148, 64, 154, 55, 156, 173, 187, 3, 77, 206, 106,
            33, 181, 22, 193, 88, 126, 68, 56, 236, 204, 241, 206, 59,
        ],
        [
            223, 213, 234, 5, 113, 245, 89, 108, 98, 154, 67, 165, 131, 55, 1, 68, 214, 224, 106,
            243, 170, 193, 222, 162, 232, 153, 246, 35, 149, 43, 120, 5,
        ],
    ],
    [
        [
            90, 122, 239, 39, 194, 205, 160, 37, 139, 132, 43, 234, 78, 48, 93, 144, 182, 51, 6,
            196, 190, 232, 79, 172, 38, 94, 38, 172, 54, 233, 92, 49,
        ],
        [
            158, 106, 252, 23, 153, 30, 201, 88, 50, 108, 217, 90, 33, 91, 67, 52, 33, 27, 134, 67,
            250, 80, 151, 7, 119, 5, 89, 110, 23, 214, 67, 17,
        ],
        [
            223, 251, 16, 98, 252, 239, 182, 12, 255, 242, 67, 31, 99, 169, 10, 148, 70, 186, 236,
            158, 230, 79, 2, 61, 166, 121, 72, 159, 81, 207, 43, 10,
        ],
        [
            88, 65, 196, 114, 188, 103, 151, 218, 161, 219, 137, 9, 51, 81, 215, 64, 102, 55, 235,
            96, 48, 54, 52, 217, 192, 125, 147, 89, 141, 194, 69, 14,
        ],
        [
            146, 75, 25, 162, 47, 21, 77, 168, 14, 191, 35, 24, 27, 59, 14, 51, 198, 52, 130, 41,
            223, 79, 45, 216, 248, 178, 250, 45, 220, 124, 24, 56,
        ],
        [
            141, 246, 176, 10, 226, 240, 16, 2, 3, 42, 222, 107, 203, 96, 22, 128, 170, 251, 19,
            224, 63, 228, 160, 116, 247, 60, 194, 164, 213, 90, 228, 41,
        ],
        [
            193, 89, 149, 53, 35, 2, 173, 192, 228, 106, 131, 55, 10, 216, 235, 148, 81, 38, 218,
            206, 47, 201, 1, 76, 165, 237, 8, 181, 51, 182, 179, 44,
        ],
        [
            208, 110, 172, 179, 135, 41, 146, 198, 71, 234, 170, 157, 98, 177, 254, 237, 235, 236,
            170, 248, 184, 158, 182, 109, 20, 53, 20, 230, 235, 161, 47, 58,
        ],
    ],
    [
        [
            173, 212, 202, 158, 121, 32, 179, 110, 216, 160, 75, 127, 248, 141, 186, 129, 102, 227,
            204, 97, 88, 186, 60, 229, 229, 253, 167, 213, 175, 226, 127, 47,
        ],
        [
            150, 23, 120, 218, 29, 242, 9, 232, 28, 28, 210, 109, 85, 108, 243, 26, 30, 119, 124,
            117, 125, 222, 224, 22, 235, 55, 57, 228, 153, 209, 124, 48,
        ],
        [
            167, 39, 173, 99, 186, 157, 146, 31, 20, 26, 63, 62, 185, 254, 57, 13, 117, 96, 73, 25,
            182, 97, 42, 160, 92, 242, 26, 54, 145, 29, 31, 29,
        ],
        [
            27, 117, 138, 76, 21, 193, 135, 154, 46, 130, 187, 36, 2, 196, 200, 196, 114, 110, 210,
            92, 51, 12, 40, 19, 62, 34, 201, 44, 159, 3, 144, 4,
        ],
        [
            210, 235, 204, 146, 85, 169, 236, 3, 193, 188, 201, 172, 79, 164, 148, 207, 40, 103,
            56, 206, 91, 104, 249, 17, 75, 159, 27, 103, 101, 72, 100, 37,
        ],
        [
            129, 52, 24, 42, 26, 131, 207, 198, 211, 136, 212, 66, 40, 68, 181, 251, 95, 157, 84,
            218, 27, 199, 150, 194, 62, 78, 46, 197, 253, 50, 6, 16,
        ],
        [
            218, 196, 228, 200, 220, 170, 106, 13, 0, 28, 246, 133, 25, 240, 21, 184, 58, 238, 128,
            219, 223, 224, 8, 226, 193, 129, 230, 131, 177, 116, 189, 3,
        ],
        [
            184, 75, 112, 127, 251, 197, 87, 114, 3, 208, 126, 142, 219, 226, 17, 61, 237, 126,
            201, 155, 23, 146, 209, 142, 168, 108, 160, 39, 43, 57, 93, 40,
        ],
    ],
    [
        [
            95, 161, 42, 151, 11, 124, 239, 193, 114, 247, 72, 186, 128, 141, 47, 108, 197, 47,
            226, 180, 118, 22, 163, 147, 103, 103, 121, 44, 149, 7, 225, 52,
        ],
        [
            197, 245, 125, 151, 243, 170, 39, 79, 128, 110, 215, 131, 244, 219, 188, 110, 33, 84,
            142, 213, 174, 251, 94, 60, 6, 133, 32, 235, 44, 57, 127, 13,
        ],
        [
            252, 8, 154, 162, 113, 63, 242, 246, 220, 245, 78, 4, 239, 88, 232, 208, 96, 142, 94,
            85, 216, 210, 72, 102, 140, 127, 171, 225, 68, 126, 75, 24,
        ],
        [
            86, 231, 20, 187, 67, 55, 132, 4, 133, 210, 120, 101, 238, 222, 7, 144, 48, 63, 42,
            118, 223, 206, 56, 246, 33, 231, 203, 252, 190, 167, 135, 6,
        ],
        [
            94, 214, 184, 9, 110, 119, 88, 199, 41, 223, 151, 204, 14, 1, 22, 72, 205, 44, 201,
            199, 49, 104, 118, 205, 219, 194, 27, 249, 202, 206, 11, 38,
        ],
        [
            238, 189, 11, 100, 86, 83, 110, 193, 217, 127, 43, 219, 33, 236, 60, 155, 46, 210, 95,
            165, 211, 55, 114, 120, 151, 4, 84, 199, 188, 80, 111, 24,
        ],
        [
            75, 132, 96, 67, 196, 109, 201, 195, 202, 133, 205, 157, 180, 194, 218, 28, 161, 176,
            208, 36, 13, 53, 120, 238, 165, 67, 123, 43, 49, 230, 118, 16,
        ],
        [
            216, 53, 14, 231, 149, 30, 238, 198, 91, 0, 75, 182, 248, 127, 51, 15, 249, 81, 128,
            49, 85, 26, 247, 233, 71, 61, 204, 240, 198, 44, 253, 4,
        ],
    ],
    [
        [
            117, 79, 104, 186, 22, 240, 96, 245, 133, 124, 14, 208, 175, 92, 38, 203, 96, 167, 168,
            74, 132, 255, 239, 51, 230, 9, 37, 100, 77, 221, 86, 56,
        ],
        [
            124, 56, 154, 244, 207, 39, 218, 208, 77, 128, 167, 52, 127, 189, 228, 183, 179, 179,
            19, 142, 235, 99, 1, 109, 232, 184, 146, 75, 251, 180, 168, 34,
        ],
        [
            38, 81, 241, 77, 207, 198, 52, 63, 104, 175, 182, 68, 16, 186, 231, 19, 112, 112, 137,
            169, 174, 109, 153, 170, 164, 156, 204, 14, 86, 138, 183, 10,
        ],
        [
            224, 100, 144, 95, 68, 53, 51, 69, 233, 19, 14, 192, 229, 240, 1, 145, 122, 174, 244,
            98, 18, 76, 248, 132, 254, 161, 221, 175, 254, 139, 28, 44,
        ],
        [
            106, 24, 183, 172, 72, 180, 89, 112, 169, 107, 39, 253, 193, 177, 132, 118, 27, 212,
            94, 108, 143, 97, 7, 222, 16, 135, 55, 59, 108, 184, 225, 39,
        ],
        [
            185, 92, 139, 189, 50, 105, 41, 186, 189, 124, 23, 212, 220, 255, 191, 28, 156, 70,
            224, 206, 235, 149, 177, 221, 61, 166, 2, 189, 157, 71, 9, 10,
        ],
        [
            46, 205, 220, 212, 230, 14, 222, 10, 86, 30, 250, 186, 42, 16, 214, 159, 31, 106, 154,
            168, 68, 126, 110, 221, 254, 185, 231, 235, 185, 145, 232, 50,
        ],
        [
            158, 96, 218, 239, 191, 119, 59, 99, 248, 159, 160, 48, 59, 14, 160, 80, 7, 149, 110,
            144, 96, 78, 250, 254, 213, 73, 119, 133, 151, 128, 130, 57,
        ],
    ],
    [
        [
            55, 146, 3, 40, 40, 129, 157, 149, 185, 250, 175, 142, 28, 65, 11, 98, 180, 41, 163,
            240, 126, 177, 8, 171, 37, 81, 46, 151, 115, 201, 64, 4,
        ],
        [
            33, 171, 216, 52, 120, 12, 136, 183, 174, 39, 212, 234, 107, 120, 254, 152, 217, 105,
            17, 158, 47, 221, 66, 80, 173, 143, 0, 245, 2, 54, 105, 20,
        ],
        [
            165, 77, 224, 22, 86, 173, 23, 3, 158, 217, 230, 209, 104, 123, 51, 124, 229, 70, 86,
            113, 69, 124, 202, 89, 130, 165, 9, 143, 13, 117, 251, 41,
        ],
        [
            150, 0, 141, 72, 1, 45, 247, 44, 239, 25, 127, 126, 167, 217, 60, 19, 87, 40, 41, 53,
            154, 65, 36, 62, 224, 245, 133, 23, 62, 113, 68, 10,
        ],
        [
            240, 173, 98, 79, 231, 216, 199, 13, 115, 239, 49, 44, 70, 131, 132, 17, 181, 178, 255,
            187, 26, 212, 140, 247, 181, 66, 131, 122, 34, 232, 212, 51,
        ],
        [
            128, 97, 150, 86, 107, 88, 9, 39, 221, 108, 40, 17, 141, 189, 176, 112, 245, 115, 102,
            222, 41, 103, 116, 83, 45, 41, 79, 189, 123, 128, 221, 28,
        ],
        [
            124, 71, 144, 26, 166, 147, 228, 121, 160, 123, 202, 119, 231, 221, 98, 239, 232, 206,
            68, 253, 178, 155, 183, 151, 255, 14, 63, 178, 233, 48, 7, 45,
        ],
        [
            247, 149, 217, 167, 89, 183, 230, 75, 8, 65, 241, 170, 95, 83, 4, 74, 30, 184, 105,
            224, 114, 65, 153, 101, 169, 46, 55, 94, 151, 150, 156, 62,
        ],
    ],
    [
        [
            224, 154, 142, 251, 230, 202, 38, 207, 234, 62, 53, 21, 186, 231, 57, 92, 134, 174, 48,
            226, 161, 198, 9, 186, 167, 197, 249, 112, 127, 236, 188, 40,
        ],
        [
            71, 140, 229, 89, 155, 109, 149, 82, 250, 226, 71, 125, 62, 174, 128, 67, 205, 8, 237,
            7, 131, 223, 182, 207, 162, 132, 28, 3, 34, 239, 115, 1,
        ],
        [
            2, 52, 130, 240, 234, 192, 5, 100, 224, 251, 234, 21, 75, 255, 150, 249, 123, 6, 111,
            69, 38, 123, 57, 240, 142, 106, 115, 11, 208, 182, 76, 1,
        ],
        [
            195, 55, 130, 187, 251, 59, 51, 201, 72, 12, 61, 186, 107, 186, 8, 121, 240, 11, 205,
            41, 41, 96, 48, 97, 225, 55, 86, 127, 182, 47, 73, 26,
        ],
        [
            252, 85, 10, 142, 41, 197, 192, 149, 233, 59, 135, 90, 245, 1, 2, 82, 149, 230, 105,
            50, 180, 40, 246, 255, 235, 38, 246, 220, 26, 71, 108, 58,
        ],
        [
            45, 234, 112, 242, 241, 115, 242, 151, 74, 199, 165, 59, 232, 121, 140, 143, 72, 21,
            222, 198, 250, 27, 82, 71, 136, 229, 46, 114, 215, 58, 80, 59,
        ],
        [
            76, 30, 114, 108, 66, 237, 123, 62, 251, 89, 191, 57, 10, 24, 103, 125, 62, 176, 243,
            156, 26, 71, 128, 100, 216, 96, 134, 238, 25, 255, 52, 18,
        ],
        [
            121, 9, 205, 213, 115, 82, 104, 88, 4, 94, 64, 39, 52, 23, 184, 59, 249, 173, 51, 174,
            29, 29, 93, 209, 180, 25, 30, 248, 200, 217, 8, 57,
        ],
    ],
    [
        [
            29, 243, 202, 191, 232, 38, 147, 197, 36, 165, 153, 225, 248, 25, 108, 49, 193, 125,
            147, 152, 224, 204, 29, 99, 149, 18, 80, 202, 126, 240, 240, 44,
        ],
        [
            251, 175, 93, 27, 1, 223, 152, 144, 40, 199, 0, 117, 26, 185, 199, 115, 43, 5, 28, 50,
            87, 195, 19, 177, 129, 150, 12, 27, 27, 106, 104, 1,
        ],
        [
            229, 107, 241, 196, 196, 75, 154, 193, 28, 194, 91, 92, 211, 30, 203, 193, 132, 50,
            220, 88, 9, 89, 31, 83, 156, 14, 131, 253, 144, 77, 180, 21,
        ],
        [
            254, 86, 225, 99, 239, 66, 104, 173, 130, 44, 9, 3, 102, 150, 131, 27, 222, 8, 11, 91,
            248, 133, 165, 168, 173, 243, 131, 231, 0, 73, 179, 7,
        ],
        [
            188, 14, 111, 31, 108, 93, 253, 223, 11, 168, 85, 111, 196, 44, 54, 85, 185, 179, 22,
            134, 219, 228, 190, 119, 109, 175, 172, 63, 43, 197, 129, 28,
        ],
        [
            100, 77, 224, 123, 66, 253, 55, 225, 81, 92, 1, 112, 220, 61, 123, 40, 141, 117, 186,
            45, 66, 37, 141, 226, 222, 227, 252, 18, 235, 1, 150, 55,
        ],
        [
            21, 60, 39, 162, 247, 118, 245, 247, 37, 78, 193, 16, 37, 78, 38, 239, 102, 205, 145,
            92, 94, 149, 21, 90, 29, 2, 226, 14, 63, 94, 18, 55,
        ],
        [
            32, 30, 94, 116, 13, 47, 190, 17, 138, 146, 169, 220, 7, 166, 141, 201, 122, 50, 211,
            216, 192, 170, 170, 171, 30, 44, 106, 143, 15, 47, 230, 25,
        ],
    ],
    [
        [
            77, 240, 65, 69, 55, 114, 61, 35, 5, 23, 224, 54, 74, 102, 26, 45, 204, 151, 93, 176,
            44, 214, 39, 238, 92, 55, 135, 235, 175, 159, 199, 5,
        ],
        [
            122, 13, 98, 228, 133, 218, 65, 35, 16, 73, 59, 73, 0, 127, 167, 126, 175, 104, 228,
            96, 221, 56, 145, 137, 246, 33, 195, 5, 77, 54, 190, 9,
        ],
        [
            174, 226, 237, 252, 72, 86, 217, 57, 42, 131, 198, 224, 42, 59, 77, 64, 103, 3, 231,
            53, 203, 219, 45, 142, 33, 249, 24, 163, 139, 52, 71, 22,
        ],
        [
            165, 193, 99, 157, 238, 70, 47, 182, 129, 207, 77, 193, 167, 206, 41, 170, 8, 91, 51,
            163, 137, 67, 22, 73, 127, 207, 133, 240, 97, 233, 40, 62,
        ],
        [
            68, 84, 199, 125, 172, 196, 145, 144, 127, 29, 232, 25, 215, 220, 132, 99, 86, 52, 46,
            227, 101, 72, 70, 193, 3, 12, 225, 172, 53, 10, 55, 1,
        ],
        [
            3, 230, 66, 237, 23, 28, 118, 88, 23, 25, 96, 176, 203, 97, 140, 92, 216, 153, 34, 166,
            23, 137, 59, 172, 172, 144, 164, 12, 90, 33, 51, 50,
        ],
        [
            208, 145, 77, 172, 10, 38, 208, 225, 138, 7, 140, 175, 49, 114, 231, 203, 45, 54, 2,
            248, 110, 69, 166, 65, 113, 253, 131, 119, 63, 153, 168, 8,
        ],
        [
            64, 27, 162, 156, 179, 22, 43, 27, 133, 222, 33, 161, 107, 141, 133, 53, 253, 214, 128,
            156, 65, 14, 138, 233, 169, 145, 68, 139, 139, 152, 17, 21,
        ],
    ],
    [
        [
            60, 31, 22, 89, 116, 180, 216, 149, 0, 129, 148, 46, 128, 169, 173, 130, 198, 155, 20,
            77, 15, 33, 67, 126, 183, 233, 15, 232, 67, 208, 104, 2,
        ],
        [
            155, 169, 114, 191, 1, 9, 32, 205, 125, 117, 59, 132, 211, 82, 6, 93, 8, 189, 13, 64,
            142, 81, 52, 199, 215, 64, 188, 226, 85, 66, 199, 4,
        ],
        [
            135, 136, 62, 242, 84, 7, 79, 221, 122, 126, 6, 216, 66, 57, 49, 81, 215, 223, 105,
            207, 3, 255, 74, 17, 23, 227, 139, 51, 56, 70, 173, 59,
        ],
        [
            180, 159, 125, 70, 170, 143, 19, 162, 132, 18, 31, 117, 102, 35, 244, 188, 51, 218,
            241, 248, 199, 96, 59, 221, 138, 234, 184, 20, 66, 172, 108, 63,
        ],
        [
            64, 182, 108, 185, 163, 65, 242, 40, 229, 179, 230, 48, 115, 253, 64, 30, 243, 239,
            110, 178, 250, 53, 51, 179, 73, 51, 100, 19, 25, 117, 241, 22,
        ],
        [
            231, 35, 183, 229, 255, 134, 248, 75, 161, 139, 166, 135, 137, 49, 205, 77, 3, 250,
            194, 31, 188, 120, 169, 46, 149, 249, 55, 90, 229, 74, 4, 49,
        ],
        [
            24, 229, 105, 198, 193, 222, 104, 210, 123, 28, 79, 249, 84, 41, 18, 53, 10, 226, 217,
            78, 205, 25, 72, 174, 239, 206, 106, 238, 80, 64, 200, 22,
        ],
        [
            222, 176, 15, 182, 226, 196, 43, 186, 234, 151, 194, 218, 55, 176, 128, 110, 23, 3,
            111, 85, 205, 226, 16, 195, 224, 5, 52, 202, 233, 194, 144, 32,
        ],
    ],
    [
        [
            65, 188, 25, 218, 142, 199, 65, 90, 223, 99, 127, 135, 228, 118, 83, 193, 167, 156,
            245, 70, 139, 254, 210, 121, 117, 108, 221, 63, 29, 183, 68, 23,
        ],
        [
            45, 246, 56, 33, 68, 250, 251, 144, 133, 241, 149, 206, 109, 107, 232, 244, 134, 26,
            11, 23, 246, 208, 237, 5, 5, 70, 130, 91, 33, 64, 201, 53,
        ],
        [
            202, 192, 115, 229, 84, 103, 107, 253, 178, 17, 39, 9, 36, 148, 157, 247, 239, 229, 28,
            93, 207, 6, 42, 165, 6, 122, 151, 176, 77, 47, 252, 7,
        ],
        [
            207, 85, 81, 110, 169, 61, 244, 254, 205, 23, 156, 177, 253, 223, 114, 177, 39, 174,
            107, 106, 136, 3, 157, 197, 104, 218, 228, 151, 68, 138, 26, 52,
        ],
        [
            50, 91, 21, 55, 91, 236, 143, 193, 156, 32, 162, 37, 68, 242, 34, 132, 56, 188, 117,
            252, 205, 61, 217, 205, 22, 224, 120, 2, 142, 91, 239, 7,
        ],
        [
            98, 217, 244, 140, 150, 75, 199, 231, 239, 148, 141, 95, 83, 68, 152, 85, 142, 239,
            212, 124, 160, 95, 115, 128, 194, 250, 115, 140, 161, 21, 111, 40,
        ],
        [
            103, 19, 40, 8, 83, 195, 218, 97, 78, 185, 136, 249, 38, 37, 51, 245, 34, 59, 230, 66,
            108, 128, 156, 149, 116, 19, 215, 251, 124, 252, 120, 31,
        ],
        [
            220, 0, 174, 242, 25, 60, 52, 199, 2, 18, 234, 211, 205, 186, 197, 58, 213, 209, 37,
            91, 68, 41, 212, 206, 113, 9, 4, 54, 157, 125, 222, 22,
        ],
    ],
    [
        [
            60, 95, 106, 48, 229, 64, 114, 62, 218, 8, 178, 224, 211, 189, 87, 226, 33, 44, 116,
            179, 48, 119, 6, 223, 75, 208, 59, 198, 250, 3, 96, 17,
        ],
        [
            95, 180, 11, 184, 37, 224, 28, 70, 179, 135, 225, 145, 85, 105, 148, 132, 31, 49, 15,
            181, 204, 100, 143, 221, 64, 231, 81, 201, 255, 93, 18, 46,
        ],
        [
            163, 206, 214, 36, 248, 250, 138, 215, 102, 145, 187, 127, 207, 228, 179, 30, 237, 175,
            69, 58, 117, 224, 107, 128, 0, 10, 199, 204, 144, 221, 111, 21,
        ],
        [
            65, 194, 195, 213, 229, 186, 100, 176, 8, 52, 52, 67, 60, 46, 97, 32, 154, 236, 10, 44,
            194, 188, 140, 207, 58, 214, 36, 231, 249, 44, 20, 55,
        ],
        [
            21, 231, 150, 194, 145, 213, 144, 44, 76, 189, 202, 14, 62, 107, 105, 136, 153, 154,
            16, 176, 139, 115, 87, 62, 16, 249, 72, 237, 131, 75, 114, 59,
        ],
        [
            221, 180, 148, 120, 116, 48, 217, 169, 177, 254, 80, 199, 131, 151, 79, 41, 76, 181,
            126, 86, 146, 180, 54, 66, 34, 169, 24, 95, 230, 127, 206, 42,
        ],
        [
            153, 198, 204, 12, 57, 10, 116, 176, 197, 116, 227, 193, 115, 216, 137, 230, 206, 161,
            169, 203, 53, 115, 126, 94, 225, 40, 47, 121, 180, 193, 191, 14,
        ],
        [
            132, 55, 75, 201, 131, 204, 219, 8, 118, 98, 65, 32, 145, 169, 179, 73, 224, 100, 168,
            94, 47, 22, 227, 203, 68, 128, 36, 32, 112, 241, 126, 41,
        ],
    ],
    [
        [
            185, 130, 202, 63, 39, 212, 225, 220, 185, 162, 38, 43, 70, 142, 46, 167, 158, 231,
            248, 162, 242, 228, 11, 205, 122, 190, 138, 146, 68, 200, 133, 25,
        ],
        [
            213, 163, 16, 131, 225, 53, 231, 167, 142, 135, 65, 194, 29, 18, 178, 96, 58, 186, 75,
            238, 6, 177, 23, 109, 14, 47, 134, 55, 211, 252, 146, 24,
        ],
        [
            9, 164, 244, 123, 78, 17, 135, 83, 123, 116, 140, 84, 200, 32, 129, 123, 112, 125, 127,
            102, 9, 77, 120, 106, 205, 83, 139, 201, 178, 212, 206, 52,
        ],
        [
            2, 154, 191, 181, 97, 225, 10, 225, 173, 78, 34, 92, 171, 14, 150, 110, 158, 229, 90,
            223, 81, 134, 2, 162, 176, 233, 223, 254, 63, 229, 137, 6,
        ],
        [
            58, 166, 77, 108, 142, 242, 100, 146, 162, 37, 77, 174, 129, 3, 223, 75, 28, 158, 118,
            86, 133, 195, 136, 246, 117, 173, 109, 37, 117, 87, 8, 48,
        ],
        [
            89, 201, 176, 246, 163, 171, 140, 250, 101, 230, 141, 183, 75, 25, 28, 67, 247, 45,
            213, 84, 241, 33, 235, 165, 130, 125, 107, 163, 46, 28, 74, 13,
        ],
        [
            102, 44, 233, 80, 122, 63, 170, 158, 150, 164, 86, 224, 152, 129, 198, 100, 247, 121,
            85, 67, 184, 35, 97, 152, 12, 86, 232, 241, 177, 38, 19, 16,
        ],
        [
            128, 70, 1, 11, 18, 206, 21, 3, 196, 188, 66, 134, 153, 133, 140, 52, 70, 114, 252, 17,
            143, 212, 210, 62, 45, 114, 178, 57, 254, 119, 152, 62,
        ],
    ],
    [
        [
            86, 29, 41, 210, 20, 163, 246, 97, 255, 42, 82, 153, 58, 80, 69, 223, 114, 218, 207,
            112, 216, 174, 191, 209, 200, 188, 254, 134, 214, 30, 232, 13,
        ],
        [
            146, 74, 34, 8, 35, 218, 132, 100, 104, 27, 60, 156, 232, 162, 104, 71, 255, 94, 4,
            125, 63, 59, 17, 91, 86, 65, 207, 33, 111, 195, 21, 61,
        ],
        [
            83, 90, 45, 62, 199, 173, 172, 28, 233, 71, 22, 4, 87, 56, 218, 182, 125, 246, 86, 201,
            211, 184, 158, 40, 99, 41, 70, 52, 68, 70, 33, 11,
        ],
        [
            160, 62, 174, 112, 183, 78, 236, 128, 32, 30, 87, 94, 5, 220, 26, 132, 201, 247, 134,
            39, 177, 19, 78, 198, 18, 118, 167, 237, 23, 1, 25, 47,
        ],
        [
            181, 137, 97, 42, 173, 195, 208, 246, 200, 53, 2, 111, 52, 198, 106, 192, 198, 223,
            132, 39, 226, 208, 93, 252, 43, 174, 146, 151, 35, 70, 71, 17,
        ],
        [
            185, 213, 47, 92, 58, 84, 148, 151, 165, 115, 236, 139, 63, 3, 108, 13, 100, 13, 73,
            104, 142, 21, 241, 166, 37, 188, 69, 96, 115, 201, 132, 37,
        ],
        [
            227, 170, 112, 80, 230, 140, 124, 73, 86, 12, 122, 47, 236, 177, 87, 54, 38, 180, 198,
            34, 36, 211, 183, 241, 169, 23, 98, 119, 106, 231, 182, 46,
        ],
        [
            211, 204, 104, 127, 81, 137, 162, 175, 132, 17, 114, 150, 104, 121, 23, 138, 87, 2,
            118, 223, 47, 149, 65, 61, 225, 158, 188, 62, 132, 24, 89, 1,
        ],
    ],
    [
        [
            61, 236, 235, 15, 173, 143, 20, 200, 109, 66, 144, 52, 144, 222, 213, 48, 46, 246, 142,
            179, 144, 182, 34, 247, 30, 74, 122, 105, 62, 148, 121, 55,
        ],
        [
            138, 247, 228, 197, 108, 167, 21, 234, 145, 197, 219, 29, 69, 246, 61, 188, 46, 76,
            255, 106, 41, 192, 16, 159, 198, 14, 32, 15, 185, 106, 162, 61,
        ],
        [
            221, 232, 255, 202, 222, 138, 118, 47, 195, 220, 232, 160, 178, 195, 18, 192, 134, 252,
            39, 157, 86, 11, 116, 44, 110, 103, 166, 191, 244, 217, 176, 62,
        ],
        [
            103, 49, 138, 113, 133, 245, 37, 35, 191, 47, 54, 201, 112, 220, 191, 103, 254, 100,
            32, 246, 92, 171, 179, 209, 2, 98, 151, 243, 9, 160, 7, 17,
        ],
        [
            194, 218, 101, 98, 161, 81, 251, 151, 95, 207, 172, 101, 69, 59, 19, 129, 145, 90, 58,
            131, 112, 62, 201, 181, 12, 218, 118, 57, 128, 243, 18, 61,
        ],
        [
            136, 114, 9, 162, 245, 108, 250, 167, 107, 12, 53, 121, 162, 86, 72, 82, 193, 173, 193,
            3, 112, 153, 247, 130, 215, 3, 83, 160, 130, 183, 171, 36,
        ],
        [
            72, 255, 40, 62, 196, 43, 119, 217, 123, 93, 7, 133, 137, 86, 165, 234, 226, 235, 82,
            36, 205, 38, 94, 234, 141, 70, 115, 139, 150, 35, 3, 15,
        ],
        [
            137, 194, 240, 168, 61, 140, 100, 71, 151, 99, 232, 226, 1, 177, 155, 111, 105, 172,
            167, 186, 232, 216, 142, 252, 199, 238, 27, 190, 112, 115, 218, 15,
        ],
    ],
    [
        [
            210, 180, 82, 123, 75, 247, 236, 159, 89, 130, 67, 24, 48, 93, 251, 167, 49, 138, 99,
            231, 31, 23, 106, 47, 29, 153, 248, 255, 6, 64, 114, 37,
        ],
        [
            213, 231, 86, 118, 67, 132, 251, 112, 123, 171, 149, 118, 232, 231, 88, 39, 110, 109,
            67, 159, 116, 246, 137, 172, 209, 103, 210, 207, 92, 19, 65, 11,
        ],
        [
            159, 0, 20, 44, 28, 39, 141, 233, 233, 7, 150, 247, 77, 58, 84, 79, 162, 237, 150, 180,
            146, 15, 95, 110, 111, 147, 65, 35, 109, 145, 56, 3,
        ],
        [
            8, 60, 212, 124, 61, 67, 66, 57, 221, 27, 31, 245, 118, 82, 175, 250, 190, 11, 19, 136,
            111, 10, 148, 232, 184, 161, 35, 134, 69, 73, 154, 17,
        ],
        [
            145, 127, 187, 184, 48, 200, 18, 166, 176, 76, 149, 227, 216, 148, 148, 30, 55, 241,
            93, 246, 183, 184, 74, 100, 56, 253, 80, 102, 120, 224, 96, 52,
        ],
        [
            198, 162, 119, 153, 73, 75, 95, 48, 9, 209, 11, 141, 127, 201, 187, 106, 244, 206, 122,
            104, 124, 120, 171, 12, 234, 209, 27, 130, 143, 214, 21, 55,
        ],
        [
            187, 102, 233, 179, 50, 210, 197, 38, 215, 193, 25, 172, 131, 167, 24, 42, 181, 204,
            231, 27, 240, 186, 96, 154, 187, 91, 0, 202, 139, 153, 89, 22,
        ],
        [
            139, 6, 212, 90, 73, 253, 231, 41, 155, 82, 25, 222, 118, 132, 114, 172, 73, 33, 16,
            106, 255, 202, 122, 2, 32, 23, 219, 145, 34, 3, 127, 17,
        ],
    ],
    [
        [
            15, 218, 1, 4, 47, 88, 165, 70, 251, 169, 133, 238, 213, 60, 36, 16, 38, 25, 239, 203,
            119, 241, 177, 190, 214, 205, 67, 143, 237, 93, 65, 12,
        ],
        [
            139, 246, 38, 149, 143, 160, 39, 39, 68, 154, 125, 174, 15, 68, 25, 53, 134, 207, 82,
            209, 173, 211, 1, 232, 108, 205, 236, 153, 156, 188, 123, 49,
        ],
        [
            166, 203, 125, 197, 10, 162, 54, 101, 73, 138, 132, 192, 94, 134, 184, 43, 163, 125,
            62, 136, 192, 111, 81, 244, 96, 91, 107, 230, 39, 174, 221, 50,
        ],
        [
            203, 17, 199, 92, 248, 138, 46, 153, 243, 134, 224, 231, 126, 165, 12, 83, 132, 100,
            210, 177, 219, 175, 207, 80, 243, 78, 207, 168, 26, 150, 59, 16,
        ],
        [
            210, 115, 149, 113, 141, 81, 10, 215, 232, 146, 149, 148, 242, 200, 224, 251, 2, 113,
            208, 123, 121, 60, 91, 36, 120, 173, 36, 166, 222, 85, 146, 17,
        ],
        [
            15, 39, 251, 185, 192, 28, 201, 215, 69, 23, 154, 118, 71, 80, 71, 200, 44, 28, 159,
            144, 72, 122, 131, 183, 155, 235, 245, 162, 246, 224, 195, 30,
        ],
        [
            192, 99, 56, 232, 172, 239, 89, 241, 68, 215, 74, 219, 17, 83, 149, 196, 212, 141, 89,
            77, 176, 112, 208, 14, 102, 155, 163, 223, 19, 237, 240, 11,
        ],
        [
            11, 134, 188, 142, 141, 48, 24, 249, 107, 250, 136, 137, 50, 137, 122, 132, 89, 184,
            91, 134, 18, 58, 197, 81, 29, 107, 53, 52, 51, 190, 47, 41,
        ],
    ],
    [
        [
            216, 96, 15, 16, 34, 37, 7, 137, 147, 242, 29, 102, 222, 190, 103, 228, 196, 43, 184,
            68, 207, 200, 152, 23, 166, 13, 103, 72, 250, 77, 223, 19,
        ],
        [
            17, 5, 52, 91, 113, 213, 231, 173, 71, 80, 14, 50, 250, 23, 255, 204, 19, 119, 18, 63,
            0, 183, 109, 238, 188, 128, 216, 227, 125, 77, 56, 39,
        ],
        [
            137, 69, 254, 160, 111, 117, 176, 50, 113, 241, 185, 88, 252, 132, 100, 222, 204, 231,
            208, 220, 80, 201, 73, 143, 0, 36, 187, 40, 29, 227, 128, 15,
        ],
        [
            85, 19, 126, 169, 120, 52, 94, 181, 83, 59, 160, 181, 85, 188, 159, 229, 226, 209, 209,
            144, 162, 129, 42, 221, 116, 26, 39, 226, 41, 116, 138, 37,
        ],
        [
            136, 39, 38, 138, 243, 149, 227, 50, 119, 42, 102, 230, 95, 128, 172, 244, 44, 121, 53,
            67, 27, 74, 67, 37, 254, 30, 37, 122, 3, 93, 83, 5,
        ],
        [
            18, 180, 139, 65, 80, 118, 148, 154, 67, 83, 242, 248, 63, 17, 141, 98, 105, 98, 77,
            39, 125, 125, 132, 187, 192, 110, 216, 193, 129, 161, 141, 26,
        ],
        [
            12, 175, 122, 170, 199, 194, 225, 179, 142, 228, 183, 160, 252, 228, 14, 93, 28, 232,
            62, 123, 252, 13, 7, 73, 178, 111, 126, 69, 82, 83, 167, 56,
        ],
        [
            34, 98, 50, 126, 175, 145, 47, 127, 51, 121, 100, 251, 5, 193, 176, 185, 129, 28, 77,
            14, 239, 31, 114, 104, 157, 135, 243, 15, 194, 186, 247, 47,
        ],
    ],
    [
        [
            234, 212, 198, 174, 82, 170, 174, 229, 248, 183, 186, 230, 189, 253, 193, 136, 113,
            147, 209, 239, 130, 31, 109, 6, 2, 116, 245, 23, 87, 89, 104, 63,
        ],
        [
            254, 234, 202, 192, 38, 123, 22, 81, 79, 136, 11, 185, 106, 85, 172, 180, 130, 174,
            183, 251, 175, 203, 79, 183, 69, 186, 217, 218, 118, 131, 45, 36,
        ],
        [
            156, 17, 73, 41, 203, 16, 153, 119, 186, 222, 153, 124, 230, 146, 87, 29, 7, 191, 190,
            168, 102, 105, 238, 126, 148, 93, 158, 1, 200, 176, 0, 61,
        ],
        [
            28, 160, 75, 140, 67, 66, 166, 123, 64, 37, 80, 129, 123, 96, 55, 239, 245, 242, 176,
            30, 43, 167, 123, 30, 176, 235, 103, 119, 245, 43, 21, 44,
        ],
        [
            163, 35, 23, 88, 71, 238, 67, 76, 77, 179, 194, 119, 140, 158, 137, 93, 247, 67, 243,
            45, 252, 94, 13, 74, 38, 133, 49, 175, 68, 253, 135, 60,
        ],
        [
            82, 94, 126, 254, 52, 64, 115, 123, 76, 161, 251, 209, 137, 97, 118, 74, 81, 1, 125,
            193, 173, 11, 246, 66, 249, 206, 159, 206, 86, 37, 57, 17,
        ],
        [
            51, 57, 178, 131, 146, 150, 143, 91, 101, 103, 151, 109, 209, 27, 102, 100, 179, 202,
            11, 153, 116, 142, 71, 129, 40, 20, 255, 203, 12, 76, 77, 53,
        ],
        [
            242, 188, 216, 208, 193, 32, 161, 163, 164, 220, 120, 103, 93, 135, 13, 181, 119, 237,
            107, 48, 156, 126, 194, 143, 116, 254, 41, 3, 168, 47, 144, 52,
        ],
    ],
    [
        [
            11, 17, 2, 160, 102, 157, 163, 62, 254, 29, 138, 65, 230, 250, 58, 107, 50, 68, 184,
            225, 156, 103, 209, 236, 196, 36, 246, 79, 147, 27, 42, 53,
        ],
        [
            33, 33, 59, 118, 107, 125, 175, 118, 27, 201, 46, 161, 241, 95, 54, 7, 100, 3, 105,
            194, 176, 145, 178, 241, 147, 26, 225, 149, 225, 48, 203, 10,
        ],
        [
            66, 12, 165, 163, 5, 18, 193, 183, 234, 11, 228, 173, 190, 240, 194, 125, 104, 253,
            185, 96, 65, 0, 33, 143, 94, 152, 3, 80, 191, 234, 246, 41,
        ],
        [
            106, 111, 228, 171, 155, 251, 129, 82, 97, 39, 36, 150, 52, 230, 2, 162, 124, 1, 114,
            127, 6, 185, 190, 77, 105, 246, 148, 200, 181, 122, 152, 31,
        ],
        [
            76, 128, 201, 228, 223, 71, 179, 217, 129, 188, 155, 222, 98, 245, 140, 135, 197, 35,
            131, 14, 99, 164, 68, 80, 61, 181, 56, 111, 34, 113, 62, 28,
        ],
        [
            161, 162, 61, 111, 161, 188, 162, 168, 143, 68, 187, 26, 105, 142, 203, 40, 131, 84,
            161, 4, 23, 197, 49, 163, 197, 175, 44, 66, 178, 49, 34, 44,
        ],
        [
            11, 182, 20, 161, 20, 121, 201, 228, 240, 159, 79, 73, 227, 220, 151, 16, 152, 49, 190,
            6, 54, 180, 221, 163, 21, 37, 241, 190, 21, 124, 22, 12,
        ],
        [
            30, 201, 178, 243, 23, 101, 121, 64, 115, 58, 150, 24, 168, 137, 3, 5, 21, 6, 45, 177,
            142, 173, 190, 90, 208, 48, 226, 34, 43, 38, 248, 22,
        ],
    ],
    [
        [
            157, 196, 42, 177, 43, 61, 107, 82, 52, 60, 246, 13, 114, 134, 5, 254, 222, 212, 147,
            7, 79, 65, 89, 206, 136, 72, 110, 59, 237, 163, 232, 55,
        ],
        [
            128, 212, 62, 148, 105, 186, 184, 3, 0, 157, 76, 76, 177, 17, 197, 42, 170, 97, 125,
            102, 222, 54, 95, 255, 85, 220, 105, 248, 152, 208, 45, 8,
        ],
        [
            227, 105, 171, 85, 47, 42, 190, 109, 30, 44, 51, 6, 165, 123, 172, 160, 211, 16, 230,
            26, 236, 178, 84, 233, 75, 144, 170, 176, 212, 45, 8, 49,
        ],
        [
            71, 24, 160, 130, 125, 205, 239, 123, 250, 248, 212, 253, 150, 47, 54, 228, 246, 98,
            93, 10, 67, 26, 177, 163, 16, 162, 188, 21, 125, 105, 201, 0,
        ],
        [
            196, 159, 136, 205, 253, 178, 38, 61, 249, 18, 42, 38, 104, 83, 32, 105, 105, 218, 166,
            88, 116, 188, 68, 103, 186, 14, 155, 32, 90, 204, 134, 19,
        ],
        [
            220, 208, 84, 205, 93, 246, 24, 240, 195, 19, 108, 146, 148, 239, 196, 44, 229, 2, 201,
            158, 233, 172, 117, 23, 108, 146, 186, 58, 130, 170, 197, 47,
        ],
        [
            48, 27, 108, 232, 165, 0, 125, 145, 138, 3, 23, 211, 187, 17, 166, 226, 107, 197, 242,
            35, 26, 41, 1, 72, 51, 128, 184, 217, 227, 132, 211, 35,
        ],
        [
            92, 66, 232, 101, 143, 36, 76, 151, 59, 49, 148, 158, 200, 199, 162, 104, 255, 186,
            143, 97, 68, 243, 94, 86, 178, 221, 95, 134, 100, 235, 198, 22,
        ],
    ],
    [
        [
            38, 139, 223, 198, 218, 129, 87, 92, 118, 134, 243, 198, 64, 44, 189, 253, 40, 56, 192,
            128, 135, 20, 106, 32, 231, 124, 210, 0, 196, 153, 140, 43,
        ],
        [
            184, 194, 151, 179, 50, 163, 11, 229, 169, 173, 159, 39, 249, 174, 118, 91, 223, 168,
            50, 101, 209, 0, 165, 156, 165, 7, 122, 158, 112, 227, 29, 27,
        ],
        [
            64, 156, 174, 42, 252, 153, 144, 240, 138, 152, 130, 113, 25, 91, 2, 73, 78, 22, 219,
            226, 9, 62, 212, 218, 201, 191, 178, 186, 43, 11, 237, 28,
        ],
        [
            233, 131, 26, 245, 82, 167, 210, 217, 33, 231, 113, 122, 73, 198, 60, 177, 210, 216,
            218, 1, 171, 161, 181, 219, 224, 166, 231, 9, 144, 173, 215, 55,
        ],
        [
            1, 121, 8, 158, 236, 71, 214, 244, 87, 104, 58, 32, 244, 25, 55, 245, 102, 235, 161,
            181, 162, 41, 179, 252, 157, 112, 182, 24, 11, 162, 12, 3,
        ],
        [
            135, 220, 222, 191, 198, 97, 139, 12, 21, 185, 161, 58, 235, 130, 145, 103, 216, 73,
            24, 149, 54, 221, 204, 44, 29, 105, 75, 241, 59, 92, 14, 1,
        ],
        [
            113, 99, 3, 22, 170, 61, 230, 135, 157, 116, 193, 60, 246, 113, 193, 150, 160, 211, 86,
            48, 135, 216, 226, 81, 111, 68, 71, 29, 111, 78, 39, 7,
        ],
        [
            76, 15, 116, 174, 6, 23, 26, 143, 29, 83, 201, 35, 202, 41, 125, 55, 43, 33, 17, 230,
            142, 183, 55, 241, 144, 102, 49, 101, 206, 238, 122, 52,
        ],
    ],
    [
        [
            117, 135, 48, 149, 94, 206, 192, 251, 95, 172, 33, 214, 122, 5, 176, 120, 216, 116, 99,
            41, 31, 203, 126, 237, 42, 244, 207, 224, 245, 150, 117, 23,
        ],
        [
            215, 141, 63, 69, 188, 158, 241, 182, 219, 150, 206, 78, 163, 60, 134, 169, 32, 124,
            131, 192, 65, 85, 99, 147, 172, 71, 89, 31, 146, 237, 99, 6,
        ],
        [
            112, 236, 159, 17, 39, 131, 26, 30, 73, 94, 167, 165, 101, 55, 30, 223, 226, 53, 245,
            230, 110, 29, 243, 20, 140, 0, 239, 202, 7, 190, 74, 27,
        ],
        [
            43, 45, 16, 196, 170, 112, 210, 218, 217, 27, 243, 24, 116, 156, 168, 216, 232, 51,
            233, 255, 29, 133, 105, 96, 40, 72, 39, 238, 122, 241, 20, 62,
        ],
        [
            112, 97, 10, 55, 63, 180, 243, 106, 97, 140, 110, 206, 228, 135, 105, 109, 12, 145,
            104, 189, 222, 92, 227, 145, 83, 23, 123, 58, 80, 63, 119, 49,
        ],
        [
            222, 111, 84, 201, 220, 211, 85, 66, 2, 80, 166, 99, 203, 7, 253, 212, 8, 120, 179,
            163, 48, 117, 246, 142, 242, 83, 91, 35, 65, 145, 20, 43,
        ],
        [
            13, 98, 129, 119, 108, 221, 125, 217, 152, 161, 126, 199, 233, 58, 76, 137, 123, 187,
            202, 234, 56, 36, 240, 90, 181, 55, 176, 89, 27, 84, 134, 12,
        ],
        [
            196, 96, 10, 230, 5, 3, 194, 234, 175, 221, 145, 93, 40, 98, 252, 129, 107, 230, 72, 7,
            205, 13, 181, 56, 107, 83, 102, 7, 241, 38, 44, 28,
        ],
    ],
    [
        [
            49, 255, 63, 55, 167, 241, 86, 34, 231, 35, 195, 153, 186, 81, 209, 131, 82, 204, 179,
            49, 181, 172, 52, 196, 139, 145, 206, 177, 48, 174, 77, 6,
        ],
        [
            173, 191, 127, 104, 192, 251, 10, 96, 24, 209, 39, 36, 112, 169, 200, 142, 24, 169, 11,
            184, 106, 152, 170, 237, 234, 8, 49, 65, 20, 150, 61, 59,
        ],
        [
            205, 20, 22, 50, 121, 235, 201, 249, 50, 21, 200, 211, 220, 198, 103, 170, 248, 76,
            117, 126, 81, 201, 134, 212, 205, 162, 229, 180, 51, 19, 237, 5,
        ],
        [
            169, 152, 197, 220, 162, 201, 73, 12, 53, 71, 243, 206, 218, 32, 232, 63, 161, 163, 81,
            59, 103, 25, 235, 175, 156, 58, 131, 72, 120, 135, 220, 40,
        ],
        [
            174, 157, 241, 32, 126, 124, 61, 18, 85, 161, 222, 239, 104, 112, 181, 58, 17, 10, 192,
            128, 159, 167, 150, 102, 32, 158, 191, 58, 196, 214, 211, 36,
        ],
        [
            212, 189, 202, 43, 164, 106, 115, 57, 56, 129, 143, 0, 92, 174, 37, 114, 194, 97, 201,
            67, 83, 60, 242, 21, 255, 122, 109, 95, 55, 136, 212, 12,
        ],
        [
            41, 172, 202, 39, 28, 52, 243, 63, 155, 161, 82, 6, 78, 68, 236, 168, 171, 107, 200,
            81, 37, 82, 194, 11, 74, 23, 40, 145, 40, 163, 196, 31,
        ],
        [
            119, 146, 247, 74, 70, 181, 50, 90, 212, 1, 145, 188, 149, 47, 106, 9, 44, 225, 113,
            77, 103, 73, 250, 25, 15, 46, 111, 20, 202, 135, 106, 58,
        ],
    ],
    [
        [
            214, 208, 212, 237, 91, 53, 93, 29, 73, 180, 213, 18, 238, 127, 24, 92, 242, 232, 88,
            248, 28, 20, 126, 241, 2, 100, 46, 30, 212, 70, 50, 2,
        ],
        [
            75, 62, 19, 16, 103, 247, 66, 81, 143, 208, 79, 175, 38, 117, 164, 226, 213, 10, 65,
            48, 228, 244, 51, 138, 66, 240, 74, 112, 189, 152, 168, 22,
        ],
        [
            137, 246, 59, 222, 246, 36, 63, 111, 128, 237, 151, 78, 199, 142, 182, 92, 117, 227,
            249, 16, 135, 236, 15, 7, 222, 205, 206, 235, 59, 21, 248, 22,
        ],
        [
            134, 58, 60, 194, 199, 50, 43, 29, 94, 67, 39, 243, 47, 72, 136, 20, 166, 189, 195,
            227, 151, 138, 193, 6, 141, 87, 134, 214, 107, 227, 179, 25,
        ],
        [
            240, 65, 105, 174, 82, 73, 62, 176, 32, 143, 83, 50, 246, 33, 143, 191, 153, 108, 79,
            227, 137, 196, 65, 7, 202, 251, 72, 151, 51, 138, 67, 6,
        ],
        [
            240, 164, 171, 188, 152, 96, 15, 167, 244, 151, 116, 123, 174, 150, 114, 219, 203, 190,
            60, 64, 123, 164, 24, 213, 22, 120, 151, 169, 147, 87, 14, 54,
        ],
        [
            237, 220, 160, 30, 10, 28, 230, 104, 44, 38, 128, 21, 205, 37, 7, 127, 217, 55, 245,
            49, 101, 127, 156, 214, 36, 193, 135, 216, 188, 175, 29, 13,
        ],
        [
            159, 14, 150, 98, 114, 124, 115, 108, 53, 17, 110, 73, 228, 238, 220, 104, 221, 43, 35,
            200, 148, 231, 94, 27, 93, 87, 208, 147, 133, 248, 182, 22,
        ],
    ],
    [
        [
            195, 102, 107, 183, 172, 226, 147, 203, 62, 64, 248, 145, 214, 32, 229, 17, 52, 12,
            239, 111, 133, 247, 141, 219, 112, 72, 179, 205, 47, 104, 154, 28,
        ],
        [
            154, 83, 5, 251, 52, 174, 96, 151, 148, 226, 184, 227, 233, 61, 78, 47, 189, 59, 196,
            225, 49, 253, 53, 118, 250, 113, 105, 195, 118, 18, 59, 58,
        ],
        [
            254, 53, 230, 124, 224, 107, 238, 47, 232, 196, 210, 155, 253, 125, 87, 42, 243, 249,
            247, 197, 210, 242, 181, 193, 98, 105, 25, 167, 203, 190, 56, 25,
        ],
        [
            119, 44, 168, 159, 70, 8, 158, 167, 237, 214, 246, 240, 186, 191, 26, 102, 222, 77,
            142, 206, 192, 203, 225, 105, 135, 210, 127, 165, 15, 239, 31, 45,
        ],
        [
            71, 8, 243, 147, 142, 135, 167, 118, 218, 47, 132, 51, 79, 183, 217, 131, 101, 83, 223,
            206, 34, 158, 147, 137, 194, 202, 7, 29, 82, 158, 123, 25,
        ],
        [
            153, 113, 163, 1, 79, 119, 160, 95, 221, 126, 184, 114, 24, 24, 253, 225, 39, 126, 110,
            215, 105, 198, 155, 244, 127, 181, 251, 237, 65, 154, 157, 21,
        ],
        [
            135, 239, 95, 132, 96, 211, 235, 194, 241, 123, 41, 109, 18, 212, 73, 170, 223, 91,
            248, 7, 78, 221, 20, 42, 22, 16, 11, 32, 47, 37, 78, 55,
        ],
        [
            207, 147, 78, 128, 25, 37, 55, 174, 70, 71, 43, 31, 237, 22, 121, 86, 23, 22, 145, 84,
            77, 46, 59, 0, 250, 91, 15, 235, 56, 172, 105, 45,
        ],
    ],
];

pub fn generator() -> pallas::Affine {
    pallas::Affine::from_xy(
        pallas::Base::from_bytes(&GENERATOR.0).unwrap(),
        pallas::Base::from_bytes(&GENERATOR.1).unwrap(),
    )
    .unwrap()
}

#[cfg(test)]
mod tests {
    use super::super::{
        test_lagrange_coeffs, test_zs_and_us, NUM_WINDOWS, ORCHARD_PERSONALIZATION,
    };
    use super::*;
    use group::Curve;
    use pasta_curves::{
        arithmetic::{CurveExt, FieldExt},
        pallas,
    };

    #[test]
    fn generator() {
        let hasher = pallas::Point::hash_to_curve(ORCHARD_PERSONALIZATION);
        let point = hasher(b"K");
        let coords = point.to_affine().coordinates().unwrap();

        assert_eq!(*coords.x(), pallas::Base::from_bytes(&GENERATOR.0).unwrap());
        assert_eq!(*coords.y(), pallas::Base::from_bytes(&GENERATOR.1).unwrap());
    }

    #[test]
    fn lagrange_coeffs() {
        let base = super::generator();
        test_lagrange_coeffs(base, NUM_WINDOWS);
    }

    #[test]
    fn z() {
        let base = super::generator();
        test_zs_and_us(base, &Z, &U, NUM_WINDOWS);
    }
}
