/********************************************************************************************
* SIDH: an efficient supersingular isogeny cryptography library
*
* Abstract: precomputed tables for Pohlig-Hellman when using compression
*********************************************************************************************/ 

// This is for \ell=2 case. Two different cases must be handled: w divides e, and not.
#if defined(COMPRESSED_TABLES)
	#ifdef ELL2_TORUS
		#if (W_2 == 4)
	
			// Texp table for ell=2, W2=4, W2_1=3
			static const uint64_t ph2_Texp[((1<<(W_2_1-2))-1)*NWORDS64_FIELD] = {
			0xC4C747BB43797A73,0x7109BBD6F8A84A02,0x9613C602E4869C7C,0xCF019E7E6794965A,0xF58E1C18639C04E9,0x326F0B99394C7FBD,0xF02EE25B6C66
			};

			// Log table for ell=2, W2_1=3
			static const int ph2_Log[(1<<(W_2_1))-1] = {
			4, -2, 2, 3, -1, 1, -3,
			};

			// G table for ell=2, W2=W2_1+W2_2=4
			static const uint64_t ph2_G[(1<<(W_2-2))*NWORDS64_FIELD] = {
			0xFFFFFFFFFFFF8BD3,0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFF,0x44B18275E6FFFFFF,0xA3C4B77CDBB901CF,0x83CA0B82224DDF49,0x147307F5A4469,0xC4C747BB43790647,0x7109BBD6F8A84A02,0x9613C602E4869C7C,0x15F1AA796B94965A,0x1D8C771D0DFC5815,0x493CB744D9D53EB1,0x3403A9E3D8B,0x5E31296681E19E92,0xFE6B47AA0389E55,0x3FEB6DE7F12F392E,0x6D8CD5E1A32285EA,0x1164ED2C4A5B9CAD,0x4639CBB2691F8613,0x583D0FD0B9DE,0x9972685E88C3C753,0x76828CB7798F26E2,0x7F0B38D03478E300,0x24DC43AFCED1715,0x31D422CA93C55EFD,0xC6C9936EF3ED3D5A,0x40AAF45C27F9
			};

			// TORUS + SIGNED W2=4
			static const uint64_t ph2_CT[DLEN_2*(ELL2_W >> 1)*NWORDS64_FIELD] = {
				0xE9503B1B62B781AB,0x2584495189C604C8,0xABCA57E776688301,0x93BF6D939E7A0A4D,0xFC6AFC5732330B6F,0xAA0981A01F0BEBB9,0x1521062ECFCD2,0x9D25566076ABA1E1,0x25FC093C1B6C647C,0x5426A38D46CBB2E1,0x502C1AB24327BB09,0xC02FB1D682F4BFB1,0x9D2BEDF0FD11B7F,0x64F202F65706,0x3D66E290370D3C8C,0xCB7058BB676891AF,0x90BFDF60C8479451,0x8A86DDA0889CA2E7,0xC9C56DAB6207D3E1,0xDCAF253D2914CA12,0x5E42C3069CB9,0xB36CD51E4B56229D,0xBA43A9AE16B2221D,0xEABA75E4B479AD53,0xA548ABE7141BE015,0xB11C81170366830D,0x463365C88B3C52BC,0x4F4FAF97A94A,0x99E9AE92FFE89DC0,0xA18224D7C4A916A,0x5BE94965F6D624F6,0xCAEB795F5917243A,0x9FC852B477571AC1,0x736E14771A5459D3,0x6AE531D311A1,0x88581326F6584F8F,0x977E350D66B48266,0x638839803B7854,0xF7551B524A2902A4,0x690B975049DC5119,0xD93BCEF59488614,0xFC6366CA9D0B,0xD16348CDE6F7E97,0xEC2F21471639D3C3,0x897BB1C805439F8F,0xDB8553D763419535,0x55819BCE2A098979,0xE7A1438C17A99D42,0x10001DA95CA19,0x730BC19E7787D905,0x452FB03E8C8EA4DC,0x5F09C36489BD680B,0xAA9964636CFA6085,0x10F1002EE63C27E,0x7C8501DA1EDAA757,0x16CB016D36738,0xC28EB9AAD06B87F3,0x2360AB2A839407C3,0x7171D33791836587,0x88BA0279A502191F,0xF947A1630946BD29,0x326EBD103A6BBA5E,0x14783FA37A404,0xD75251A555582FE7,0x658A4079122CDE69,0x7EF399938EFA570E,0xE01F6577B0368AD9,0xD7A5B1E48CF8CFA8,0xA693C773233A502,0x222F8E44F638,0xB81369459AE1B8BC,0xF9AD7B7145FEACDE,0xBBD0E11F72896F2,0xE6CFA1D3C9ECFED1,0xB2EBAC60CDBD7438,0xEF7EE41340D91A0A,0x20589456CCA6D,0x811E2C43F0AB6AD7,0x77D652171F66A84D,0xD7AEBC31C31C934,0x7ED3CB88C1B70CD5,0x4A160766AC05A25E,0xDCA45F2B2A735041,0x198699B8D08F1,0x6A0EEC9CA30D9C8F,0xA411F87790EE20B6,0x28065EC27EF7F6FF,0xF80E16530F0825D,0x366DAF53ED894198,0x5751B0BD74D68AB5,0x111CFD20CC969,0xE2B91C853C16AFB2,0xDF8944F28342C790,0x1DCBFE80DABFEF3D,0x2CD22F8038747A1F,0x64B241F02CF0EF63,0x44AD1A433E31CC54,0x22D184B32B104,0x3E8175D999E7F1F7,0x8D7C657913094D17,0x53D62AA4351AC045,0x89A49C400935B2F9,0xB1AA43277C76E3A6,0x743CB2EB731E06,0x18B8BC57DEFDE,0xD48302A369E0B6EE,0xB2379C3EE36AFEA7,0xD8FDCC80010EB503,0xF980ABFF70D01A0,0x53E7259C7ED6F33D,0x6611BFC936BB21D1,0xC91106DEEA56,0xF1DE8F43E189237,0xFD5BFBD1FD9A4B15,0x6D32F75D665DF161,0xD0CE41230DE6D5B4,0x1698EDE544FA61E5,0xE9E8E582B6B8370A,0x1412896A6D325,0x926EB07D09A6A2AE,0xED1C53E42C0C6202,0xA0960BE43AE2E541,0x88C09BAD06F37888,0x7ED888AB48C3ABB5,0x85A21D78BAD9AADD,0x12DDFA8708D81,0x67FA0D9C0BDF33A7,0x5A1C3D0435532EB6,0x7B82033DD09A40DD,0xD2BA0A4B86AE8982,0x85BFC5C8B80AC3FA,0x26814128BE87FC84,0xA1E451267946,0xB578F00FF22AA94F,0xA51ED87F95C11159,0x15CD56D5E85D3079,0xF1AF5815D0ED8E0E,0x4AE15AD82698913F,0xFEB4E871919887DC,0x17418E0C28F09,0x14D07363621B3D6E,0xD0DC60786B10C73E,0xB2940CFB649BDB14,0xBF2FF4F867C9BC4E,0x964CF4286731FC50,0xB203C6A1F6C61B6C,0x1E5D70F33DDB6,0x4B731116E201422A,0xFAD272CD5525729A,0xFD0853D28A243AB2,0xC29F33912C28FBC,0xDB88ABF46E7EB755,0x52100042A06698AD,0x1AE59A362D490,0x59F1DB2D9B051856,0x8147AF3D1B381D74,0xA3E631840E7B68BA,0x48391DA085619C6A,0xF9627342909279F8,0x3601A92EFFCA9312,0x7CD1B51109FF,0x58EA6661DD290A9B,0x5D643FC591192511,0xD31A5B0F1FFDFE3B,0xA24E89FD88607A57,0xE0DAED087B08FB13,0xC1A5040FFF1ACF5D,0x4455E9A52119,0x715349693E321B5,0x1624030F4E38654E,0x14CD15E88ED9A9F5,0xF6D134BB40450BC4,0x22489AE5F690499,0x4A208DDF0C93411C,0x1CF603D9DDC06,0x5FE42FC374ADB1EB,0x927BEEB400C4EE56,0x5E9251271BDE4CBC,0x21DA48B7407D4CE8,0xBC029D17AA6CD556,0x7364A9BF0F209BD3,0xBD69BC280AD5,0xA2428EC4E4E5419E,0x427F0D08ED5C7A6E,0xCCC7D384BE0BD37,0xFF7B14F817A661B3,0x9DAA0FC55D4B1CC0,0xE2FBD1E1ED945B6A,0xB1BB5CDAE454,0x7DDE28590DED3D05,0x568D036391F4175D,0xF18C3F48D9A695E8,0xA5CA316DB4B06FFD,0xC5FC351CE963BFD7,0xA30B379A465BA4C3,0xFA63FD27D80E,0xB8E91AEB1ACA4351,0x972DBCE3E1D33FE9,0xFFF565D3D7340354,0x4E5363313A5F1BEA,0x5BB96AEFDCAD125F,0xFE20C2CD431828AF,0xEB9059FD58AE,0x823B2679077573F,0x14E89356E88EDF01,0xEEE602A10390D71,0x6575707B90A4FF83,0x24FEFEFF0E49ED98,0x31EDEB8737B1A258,0x3583CD3EAAA1,0xA98110F9F9CD0160,0xBAC059D66230AA3E,0x9780F5E28D2494D2,0xE07CD76A0BF92EF7,0xB4B05F535A83E4DB,0xC3458D466DC25E79,0xA33F13F22E40,0x2969CE04F99782FE,0x9FB567A0FF03B065,0xC2D66700F2CB2DEA,0x2B98924B7EE5B338,0x606971614F647EDF,0x6505CE1AD7BAD70D,0x1C7134FA35946,0x870DC3B40322BA2C,0xD85F7D33ED22542,0x99C4ACF9CAF16019,0x16B78CAC738A0170,0x316453A62426C4E8,0xE8FD9B55B7100E1,0x1F62B124EAAD3,0x61A81C100C77C467,0x2F3327E55079CB19,0x63876DFA0DEDF3AB,0x2A30FB5E049FC139,0xADA23240320FC1A,0xB02B041D36F816A4,0xC4D1462FCA21,0xAF7C2F6954DDA65E,0xB8B6101CD3687276,0x54B89E23AFEC0692,0x66757611AF5983C0,0x932B03532F177F27,0xFCED39909B53A94A,0x18C44C21CD0E3,0xF6C38332443C9C27,0x3FDF94B0570E7EF0,0x388FA651C84BAE9E,0x9479EF32E8398D0C,0x8EBBF079100E3CAA,0xCBC51BD366E3E269,0x21CA55175D420,0xCEE5458711D3CF11,0x94CDDE272BB44E24,0x3ED2FF9B586A7818,0x6C686A596B6E4CA2,0xDE6BBD54CC52D2FA,0xC5E9125B3CB2597E,0x42FA565A1F9C,0xD112D5B41E6B59D7,0x5F1D3C6D82B25399,0x27D2EA701D9E08CF,0x38C2262AB8D944DB,0xC1350E0F414FA53B,0xEA35D2ACBD50EBE8,0x172593E8D230E,0x471B0AE21A986A4A,0xCF7FD7D2FE060726,0xF42524AE091F342C,0xD238E12877203BB8,0x9DA3AB14729348DA,0x5D08BCC1620452A1,0x4952CADD7CF,0x89A353FD3C85001,0xD0A29E2C537FE0BA,0xEE4B80422A5EF788,0xC1A975387549E33C,0xD7C8B3059BD2A696,0xDF96CC36F2DCAD7D,0x1107A9E86BBD6,0xF631863860C52EDD,0x5E50F477CE2F9F34,0x36387E0EBEC600A3,0xD4E3A1741E01F609,0x7425F63D66558F04,0xE458D32CD843A55,0xFD64433B9E3,0x134A8782FF731451,0x316CF9BE95210F4E,0xD5A00D2B40665ABD,0x967CA6D258B05A,0x7AFCB59D01434ECA,0x9D0E3830D8526E0C,0xD34B19E71DE0,0x7F8C71E77033040,0xFE0D225D8527CF96,0x2BF28799D8DD90C3,0x54330934056B11FD,0x547796C3337B3072,0x15CA20ECF85822DD,0x3C64726CF8D4,0xBD6FA77EA4AD9C5,0xC0E7D6FEBDE58A25,0x60AC151E39E3849D,0x7E9C474C75462A91,0xBADDC1B3E4D218C5,0xB827BF11F7DDAE60,0x600E9C0574F,0xF3F005AB05B48456,0x29ABF152AD53AD5,0x1978FE893A170A53,0xA3FFE11EBB162CC6,0xFDD138CDCAE32A13,0x3029BA5F361E8921,0x1EB7B1F4FEAEC,0xEBA4A7CE7B6C1A79,0x5151B617AE0E3080,0xB0878E4FA7A5C563,0xE3F51F81901ACD00,0xEF404AD7ED1C040,0xDFE24DA5CECDC503,0x130F68DE5CA42,0x3360AE6DDAD85F8E,0x19625905A4C62597,0xA50BACE9CDC654D3,0x704EE670E2C566C1,0xDC1CBDA02050957,0xB61B3C6699677856,0xB3ED7EED5499,0xD739292502D011CD,0x3EDD004C8C0CD7BB,0x63090029404C4F6,0xB7177FA88E23B3FA,0x579D5A4BF002F260,0xBCFDE2BEF19D3C36,0x4C960C237B8B,0xF8D1458BF19F69C7,0xEBF47DEEC16B5648,0xA7B5508E61A2270E,0xC1437E1B481D2935,0x874C15FA0A277BBD,0xEBF0FAA7CF1D687F,0x15D4B5C5508C3,0xF780101A6DCE50E1,0x595619FF46841D6,0x223F35E38EA1293D,0x121FA7881F6FF537,0x8BB4E37099507793,0xF58F157E6568D84A,0xB7A5604E2FAC,0x3E2B97F9B2018B0,0xE60890AA5564BD2,0x26257DED16F1F794,0xECFA94DB481773FA,0xD3FFEA4B03F83323,0xE9496679546D0A3C,0x20CFC05773A98,0x60C95462852406C4,0x67087CEF9E4A9008,0xD519D72CC698134F,0x9F0E1C4BE0592F82,0x172932B982FF6552,0x900D9CAB8E25E861,0xE864187BCD9D,0xFB12B4FE0B5D9B7E,0xB01EDBCCEA071B7E,0xB3D74B38CAA53C4B,0xBDEAFDC9253A19B1,0xD6A7D7BCB10A53B9,0xC138A37BD8147D43,0x11DDAB50B0F1,0xAA7D825A5DDA5264,0xE5F5DDCFF4B56362,0x23C51DD0FA1FE287,0xFA52B13D488F69D,0x2969FF070DA08CCD,0x54F9833F77DA5D69,0x5FB311567D25,0xFDF368C490DC2DC7,0x6A8E93BFA25108F6,0x58451EC4A45E9DF0,0x976453D0C79C365C,0x8CB52C266BF56BBD,0x286CFF8C55EE87C5,0x1E4A84D74DBE5,0x53450E506A4A041A,0x5B3DFB952EDA758E,0x8B6D2BF58E97A073,0x5C9B3151F55FECBA,0x3F75A6DE8056E3A3,0x3E273F1D4B792BD,0x1231ABC8670E2,0x69FE6AF84EFFE627,0xF93498BFD6B77439,0xFBCAE33D818698E4,0x8E009B3FDB4E73E5,0x89AB6CC070E4178A,0x915A67AFB213855B,0x1AE0280840C01,0x462A41A83FDEB611,0x22896C290C4859BE,0xDDC749F322A3D11E,0x7CEDCA9196469F21,0x5A83FC7483FAC67B,0xB2BD898762657D55,0x19DE49AA99C2D,0xF3B13E773D96088,0x4016A396671A613B,0xC25D2A0CEF21F8E7,0xA93DDCBAEB6546FD,0xE40FBA5F0D795AE7,0xEFCB36E9BB5A7315,0x7ED2AA61D5E,0x28522016DC21032D,0x3732F9B076532365,0x286B6FCD6B4835CC,0xF59DCE6E09F7FAA2,0x2FC9E505697F681F,0xCCD164779C9632CA,0x19E486720C304,0x811FA5D63BAEB549,0x6131DE3914958DB5,0xADA0336F4ABE8B93,0x2068014756426D43,0xC65E76228C220011,0x86AC202860A85BDF,0xA2D9C00AC742,0x6C8A61A0D775C1E5,0x4C20BD1ED4AA3DA6,0xDC27B565B06D8D1B,0x89C8C22C60F64E34,0x5ACD9C9C27ED5A14,0x7896E19EFD503C1D,0xB1F5D8F5FFE1,0x31F3F6A2A25A8C81,0x4A008C33176336F3,0x5D9C432FE1AC769B,0x72E7E31566D902E,0x98F933607A94E5F3,0x25F1E73177AC0570,0x17C4336348DE2,0x8A4D1C8D73D4A84D,0x7090469E85E2748,0x6B873AFCF0F7CBC5,0x30520B2B8905E3B4,0x4FC901E498F9CF1,0x1B22ACFE7B550496,0x15B01DC6BE13D,0x7A34CB552E024F80,0x6D5562FE14265456,0x3CAFAE6610B224D8,0x408C4B12D570DD13,0x8BDEBA2AB8EF88BB,0x73E5D5D0480474E7,0x1D222E2D63118,0xDA98B429D9F49A2C,0xCFE6680EC65F78B2,0xCFD9490F488F8372,0x9234D4B3EB63959A,0x28C052636B762B9E,0xF389847C8C2A4630,0x1EE26142D48C2,0xEE7FE37F545B347,0x53CAD59912C335A3,0xEFB53941BECF40BE,0xC6ED4ED0AA926402,0x41DBB88919D1FB57,0x7C1381F3DDD5AFE7,0x13F0854A4CB59,0xF22D21A62B467955,0x325F39F20CD51A36,0xD38BBEFFFFC637F5,0xE370366D70F1589B,0x9CC812C2AB558C4F,0x7F79217010EE4E40,0x1808149B3F21F,0xC56E7CDB16A9DDFD,0x6E34D208AFD23AB5,0x7BF0FCEF6E95F014,0x404773FF0EA04CB1,0xD9BD230C6DE2DB4B,0x6CE18EEB333CA746,0x130CD6008969,0x60C51F096DFC87C0,0x9FFC22CC8DC97F18,0x70A811C628150DD5,0x1E5900C76BC93D6A,0x9B2C65E7A2B12D29,0x591577A55870E8F1,0x1D4BF4F594549,0xA9711DAFC3FC7D70,0x9EE1D054F4D39DEE,0x7D686689384E4661,0x4E8F784B746EE271,0x447207F171C760AB,0x22FDF9D804DDC347,0x1D5BD716A07D6,0xFCF8ED906712DB84,0x5F04AFBC30534B1F,0x235F7D6C4BEA2C58,0x7B8B9DFA8FF497C6,0x50CAD89DA7DD9119,0xE08BEFF479B10D21,0x1CAAAA747BDE2,0xBA66D4840967E3F8,0x5B77F86C1C20115D,0x515BFBDBA05D8FF8,0xA6A7E27E77BD13E3,0x56DCDDEBA0338E46,0x65032146985E7ED0,0xEB23D391752D,0x72D1298EC94797BC,0xBFC61047D81F13E3,0xD7D11680F2149977,0x22332FC8479936E3,0xB1D0DC7738729C80,0xB72594284D4CCFD1,0xBA3B2E9ABEC6,0x4421C5EA2FB8F20D,0x5BE64B56E86ECEF5,0xCD07FE5111BEB813,0xCBC3B6D6A47F310C,0xAF25C2D7A94418E0,0x1D5A73156512E570,0x212B1DD0858EB,0x8708D8DFDB8FA481,0x89A0C5FEE6E26D03,0xBF93CAE89147A35C,0xF36B950DFD4CC4C7,0x724D670E39AC50E2,0x1CF0F624F6523E31,0x1992440AF25A5,0xF3A4B2BDC281C13C,0xCD099479D3D2B62C,0xAFBEC94945726FD0,0xF865E8816F4C1AFA,0xF66663F44D6F0C2F,0x81A9306801842950,0x4F98F13B585D,0x7E9035C69D468953,0xE1BD085EDDC6359A,0x602D7982EA1A7F32,0x7C6EF20561F6A738,0x45E78BC9112A90A2,0xF13F2013D6F5EE69,0x1A6D53BA47434,0x1C1F43858DA14238,0x6EA464B84343EBCD,0x76EEFD8AA59B668,0xF2F37FC99E361801,0xFE196F190E54EAAC,0xBD3CB671766DB425,0xE5062C91991E,0xCB13033213DA3931,0xBFD5F7C5552476CE,0xF64A925670AC01FB,0xBF43387D02FBB37,0x1D84C0935AD88855,0x5A6130C87C5FA632,0x1FB45A0758C02,0x67445574FA914F9A,0xA286E3114EA2D3DA,0xAE931F9876B537FC,0xEF193C20BB073626,0xBF8B86CF92C04FE8,0x23DC111F9C87F974,0x1D87A1E3B1FA9,0xFD670CBC4C15826A,0x3B480FB3A39CA86B,0xF77C65DCA10C6119,0x960D9ECFF0FC80DD,0x77555B47AE02E314,0x3B9A51EBBC63F100,0x217AFA32712B7,0x82485EA326BC92C1,0x23706F4CFE849229,0x4A1D72E39C63D976,0xB6826F3732D3B07D,0x37DBA91AAE2C1A76,0xBC3DB6A9659D868C,0x19B169FDF9F20,0x6818B2B3FD6AE224,0xECDA8ED9B791F8B3,0x44A71CF1B0A875F0,0x796D5E3FC8F886AA,0x53A9662A6B74990,0x44D923D8F6CE76A6,0x1F699F2C88A2C,0x500E0238EA6724E7,0x5FD65E9918B25F93,0xB7A4378C8F12AEB9,0x95424185A342F828,0xAEC34329D19353FB,0xCF672149FA732D89,0x153D7D90A449C,0xB8525D7E2B4FF9F9,0xDB109CA652E6A115,0x97B80782CAB504C,0xED0AE4D66C2551E9,0xD7FA7966E7BF5484,0x141BBF23524ADEEF,0x1DDD411D853E9,0xCDF01BC07F89D7A4,0x27DDE723C4EB7B94,0x9B923A2D678302EC,0xE99F503677741E9F,0xF5953ED7C0CA8238,0x8A4EDFC1A6AE84F7,0xB9A202A50FEA,0x28C4A5BED721DFF5,0x7437E3C01865264F,0x95E4DF6EDE5589BE,0x382C093AA779FE9D,0xFB87DD73A441548F,0x21629B687A906F1D,0xC50A1AC5BB1B,0xD9087CDA74C99B88,0xCC9A4B5B445B3CE1,0xD96A0486687F4D1C,0x468B64ECA7BE9BDF,0xB8F49888FC1F7050,0xFC29C94F55B6B38B,0x9AED67647E3,0x6A61E2907E96BE48,0x80B5BFCE3281CBD5,0x6005AFD4D4C43C0,0x796E4E89FA0076C5,0x6AC9892941E2A27A,0x5ADC4F797F2EE4DE,0x1B98A79DC7A8A,0x9B84571FFAE415E4,0x69182F4CBAB89F47,0xC4CC7C76D521BDE5,0x9525703CBA33FE6E,0x50EFA6B2A846D21B,0xA931D7971562D247,0x1F64BCC7A280A,0x914DC50CD57BAE3,0x44E5848889941646,0x1973D46A2FCC5E65,0xA0B8967A01E0613A,0x4F83BB1713CE0EF5,0xC90523ACF58A1C20,0x20E4ACD463C8A,0xD0C507E6BF3DAE43,0xEDD6B9AE49751317,0x1128345ACA5FC095,0xE63BF4A9CF8485AB,0x844859A09461A36,0x1E7EEE349C9D9859,0x84250AD7BAA8,0x7029E55671496051,0x23752B9AB8B1528B,0x7081DA7B4C39720C,0xE894F229F169E089,0x883FB87D21992668,0xBF1B05A46E9BD3E2,0x47A07A0DBCAE,0x10B2BE1C4D37987D,0x99132D5300EAC26D,0x4FD9C91F4D5D6DB2,0x5031D5F3AB1EE57C,0x97F0C6A93C819AEC,0x23E82FA00B2CE850,0x11EF0BAC64EB3,0xC8F7B15BD673B5C7,0xAE7CC60881FBDBDC,0xC35B90C1FE47212A,0xAA2C1C8FD4ECD75C,0x56E140E2DD02AB88,0x2C7A72F25E288CFA,0x762418676542,0x3C80DBFCC950197,0xCF4B0548C06D8B,0x9A2DEA594F29ED96,0x5F43C07101E4CDB2,0x2ADA7D027DC39140,0x7D1A30DFC50B9741,0x83BCC8A7ABB1,0xAD2FBDAE951D3A0C,0xB932CF4701165CF5,0xF573810D56097E88,0x64E39A4011AB381A,0xB711383B6FF5A903,0x29B1E25EA83EB71,0x1F67E75127FB9,0x2CA5FEBE7FA3E9A,0x624F053488D73ACA,0xFA92BDE088CE68C1,0x85D0A8DA01EDE1E0,0xF28C053E8ED67E9,0x9E734C9B3CC274C2,0x13E260D188C5A,0xFFFA3C775F4DCBD0,0x1F0182649A6FE88F,0x48CEF60A8A864817,0x19D45800BC2ACC1D,0x206DD2F795943071,0xC697F6E9C86D5596,0x6F59A305D06E,0x7D36E0E792EC3FDB,0x7010CBDE737C0682,0xFEE3587D9F86ABB3,0xA6625D47CD40871C,0x4994BA665B309F8A,0x9E384751DE14CCC0,0xCF42739830C7,0x3BC37AFB52EEB2E3,0x258030816B5BC23F,0x86ABA64E2960CC8B,0xEEE47CD5D647C32,0x34F39F2C6FC1BBAB,0x68466801B5313B9B,0x128407BFAF9B,0x81D069BEA585C825,0x50DA52119625165E,0x964FEE0E43D5D809,0x57DF566310691047,0x1FAB6568060FAE81,0x96E9C7C80B39769E,0x1195B1C94EAF8,0xB62D60F1470CF385,0x1E967C6BCD9E6877,0x8F5421A1DA957492,0x8BEFF1620D7B91C9,0xFACF99B27DE96763,0x14FFF28C638B2383,0x16E2DD13EDD6C,0xAB7048CF494ACF5,0xF55F49CF62F97E84,0xB2DEE9C996AA16BD,0xB4353D50B31883CC,0x64911D7715A240B1,0xE355E49C039DF3B7,0xCC69209D9E30,0x4C19DEC1ED51ED4B,0xF70AB30D80F8132C,0xF2C113C96363291C,0xAA731F505FC9A9F5,0xBD3D334585A47312,0x549DD34BA5D95A19,0xF111E59D2DC6,0x4825BAA0C6C1D801,0x5CA9E4404CA46A97,0x7E05DDFD508DBC45,0xEDA164B00C49F16A,0x466DDE790345820F,0x202496604C5002DB,0x1515F145985A5,0xF114BD58D05093DE,0xFB66930AFF2A7601,0x9E771AEAE9D7827E,0x6C0E1F115FAD38B1,0x9B0C6AA0B25F2FB7,0xAFF64EDAB9F8B3DC,0x114F6407B8326,0x82177257F0C8A6E1,0x1C07FB330457832B,0xFF9DC4690F507689,0x445AE0F3F0877339,0x251BF19DBC62AE17,0x4EC811D90E173439,0x1794FCA845F4B,0x3536D44904E070C8,0x70DB206D44B516A,0xAA18D03C0FF30A61,0x7E899D9D4312CF73,0x3D1CCB473F20BE7B,0x557121FA9EB46685,0x48E29EEEC574,0x300AD1ED9D23F2CD,0x3570AF102337BED6,0x6B98FE2066BE5894,0x82C68CCF7134B052,0x89709EEE56B00DE,0x3D8D0C29C826B7B,0x965E97BE1F38,0x1B6521A69F8CA022,0xFF10F9BFCD615D13,0x18978D88D7665FE0,0x29A2F4DADF534514,0xEC26BAE83E1CD2EB,0x3956FCB6F649D7DB,0xEC9F4750EEAC,0x4F3258A2EFD08E2D,0x47CBF6C428B89EEA,0x8159BDD2AD6D2B60,0x523138C85CEB06CB,0xDA7C7ABF583ECA0C,0x16A7D3E536C60914,0x2BD234698F93,0x7D9F734E01A53E3A,0x93AFB4FDE75BC3AA,0x52372A957F8DBB58,0x4B953963BA18396E,0x76D1345BCABEA1D1,0x845F6DC9C8B64B93,0x720A18F3BD01,0xAC9B25FDC7E1A08E,0xB44859F2051A010B,0x3845BDA8FB61BF60,0x2D2274C2DAB0A1AD,0xBF898635846F71E7,0xACBE47E9BC5F31A8,0xC610619F568E,0xBBF56604AF9D1EE9,0x1EF490728E884A18,0x89C206E24A9FDB89,0xDCCCE71800BFACE4,0xF6EE88F73510C6DB,0x89AA3B11E1459ED,0x89325CAEDE0C,0x3BC6C7F203B7ECA4,0x293DDC591F3877F7,0x19B0884A7A3C303F,0xCF47E021A2F1B5BF,0x5F061536808A4375,0xD186152EF6C0298D,0x1194BC6591431,0x384AC0412F92E501,0xDEF07B6444ACF715,0xBDD68F68E2E966A1,0x26D14BD343AFD0A2,0x2C1C515815EE779,0xAAE66252B5CCC7AB,0x10C900560968E,0x2B36C9FEDA4FBD28,0x37935FBCCE20B88C,0x7EC9EB07F935FE5E,0x2DE280FCC649EC90,0xADDAE96F842EF629,0x91993457C2B9D77B,0xD1A0E62F0431,0xDB821EF7F59DCBB2,0x87A2A0EFE0CFF0C2,0x40EE1A94CED9A1DB,0x45AB50686B197183,0x2E14C237B9FC94D4,0x1B8D0F73C33169C3,0x6BFCA66FD872,0x89C14DBFA20E02AA,0x65A0DD6E5E4B2F31,0x66BBA841B67ADF7B,0xC7A5B8A8DBE57C16,0x307342DAC6037E51,0xB7E7B92F556A9692,0x1311EB5BAAEA,0x478E5B566EAE31F,0x4206C484AAFA04E,0x2574C29A5A8A48D4,0x562E08031818D08F,0x57200B1D5DFAB40E,0x96AB3086C8759528,0x1950C87A90CA5,0xABF7721B5B2E2862,0x5783914DE7EF62D,0x4E4F8F0AF71E6FBB,0x53F4396C7BE3FD11,0x5A7463D5DDFDB886,0xB86157123E20B3E,0x85CC7FCE007A,0xDB6587300C2801C9,0xF99E56F2EDCCC69D,0x6F3EE74517AEE2CF,0x5BB4E587B0ECEAD4,0x764BFFD758FF83DD,0x59255242E22C0A1,0xF8AF7566C4F4,0xEC77AE00B5BDCC9D,0x88D9A5BB23E12963,0x544BDB365EE32DBD,0x18C25A883CC197AF,0x71BD81897D2493A0,0x1604500DA2607D18,0x15BC55CF95551,0x66EA7012A5733FDC,0xF1BA6BF1A7395595,0xB2612160C13B0D55,0xDD4584EB55D1B534,0xCD448F582C1110A2,0x72844F16A549C367,0xA69B88E12064,0xD82377EB46A9FD85,0xC159E7F206F5C18B,0x3FABF805EA1B1C18,0xBFADC39BAD71BD76,0xCABAAFA5A9BAEBB,0x38B7F997D1D0CD23,0xAADF279E4398,0x4C802301B8863B2C,0xA37A0BAE9FDE20A,0x90C11387D0055D2,0x90C48A79DEA8E1B,0x469887845868E39,0xCD8C8FDA3DB00192,0x1D3B1462D4283,0xEC7633369EE2A8DE,0xBC03BC60E7A00DA,0xB84E5E725D85EA94,0x9EA82D43B53E9C0,0x430DCCC38EAFFE29,0xAF4D88F5265AFA43,0xAFEC88BFBC5F,0xB2CE38BE24788251,0x9140206C5AA97925,0x7CCA07578BCB4A75,0x6E4F2C982FFE380C,0x283997B4592C46D3,0xEB402FA841D2975E,0x13A5F0900A4E4,0x85649535F0487DEC,0x8B8243DEC8241E33,0x7C7ED681281749C3,0xD7031B566C81A841,0xDE198ED6A4FEB1D9,0xB517DCD405CD1801,0xA219AFFCA673,0x4D0E3A7B5B7F9DA6,0xFD3AC255B3D82AFC,0xAA937946FDEF79BE,0x8BF54BA2D9F7DEF2,0x38B2DEE0CF6B924D,0x415B7BFA654504CE,0x20863728555F0,0x4AC9ECB29C8D85CE,0xF7A1F0B5FB38D99F,0x3BD0951D535FE06B,0x6BA24F0FEE01A059,0x77F836CC8E7BB92,0x1248BD4D0634424,0x17BEB8CC5B729,0x3BB7E213CFDAE97C,0x8272CD982F96657E,0x4C94EECCBA58377C,0xC86C969337C573C9,0xDF0B6227D50C4E52,0xB24261B33B9C1EF7,0x10249CD082EA7,0x8739382ECE9564B,0xB6C23BCC5B41504F,0x8BC66FCCF39B1A1C,0x1F51DE49F0098658,0x89D2C3D4A9E32C95,0xA263ADF49BC28334,0x1A5A9B9B41016,0x76E6ADB59CF8292E,0xBAC974121C1F0A94,0xD7CDD2F488C9D5F6,0xDD758E502C0C73AD,0xD62B08C3544F35B8,0x8BBA5F1AE2EF997,0x1ACAC122A589C,0xEE33B07BB95F715,0x70698218D87631CD,0xFC705C510AAB2933,0xF1A2149EA4E76A6C,0xD4F33449C2CB6E14,0xF13C1A96204C0F22,0x154710A8EEBFF,0x872151BAC138C038,0x5B708F8077E7A43C,0x81C052719D2F933B,0x9EECC5BEAEB3F743,0x542D43F2C3F2263B,0xC773DB8028A146E1,0x95B561556BEA,0x8C253ECE0AFB54EE,0x8A0D0EB7F61B49CD,0xAD9E76D70106C53D,0xFA60211E6E7EDBC8,0xAC5FB8C4ECDEE19E,0x4E3EF868AC3D43D0,0xB7F491C9AF3F,0x77877C1843CE59DB,0xFB796E7BB9AEC7E1,0xC6D59DB88DDFCB41,0x4063E424FDCAA827,0x13C48F0C95302B6F,0xFAB7D8A82E347A0D,0x134F3FC2B0553,0x8024B1CA2EB5516C,0x73EC47C60B3DC828,0xA95DE0ED9D4399F4,0x5231D1877DF13DD7,0x4593B4ABB32A2759,0x6144BD84D16B507A,0x23EEC9CA70FA,0xFBF3A154C6DA6960,0x454EE55AE259215,0x2C57285C581BA881,0x17B1A08AD5D4E72F,0xF44DDBC4157396BE,0xD6AF531BEC3266C2,0x11004C75F09FC,0xE84A4D0343064658,0x89293D291C96589,0x44D22FB1E5A2C707,0x8BA5F41B0ABFF6CD,0xA8BA470D99420C12,0x4F6A68ED4A96AD3B,0x610C98653A50,0xAA2A15BA833FF72B,0xD760AD8443381F1F,0xE34BA08C8CA088DE,0x74F293C86E89C60B,0x4F96F5E6702889A3,0x38F2A4DF7595777B,0x23026B7BF86AA,0x5E8C04E49685F8E9,0x6953FF0B3143415,0x16771B8D1C7EBDAF,0x675CE00AB0D7485A,0x6AD1071137BE197F,0x623B6BEA7966473F,0xA397D190071D,0xD166DC2E9F26E33,0x74BF6538BC05BD25,0x720B101901F06F32,0x60E6689AA83C7DB2,0x6285B911EEACBD2C,0xE26131028CFEF794,0x1E20A0FD4D01B,0x66E993D84D0CAD70,0xE26C573C0D71A2F4,0x3B8E80A47909173C,0x632E5B6C780BD529,0x89378DC80607055C,0x12EE7F319C3B9515,0x21B1EEA22C07A,0x45A12E4234B8A160,0x3735B15CF1B13C3,0x3CEE11835A600DDB,0xC8E03C580A41417D,0x3B33AF81E13A6A3E,0xDC1CE1719D7B47BB,0xF4A238B18361,0x623886A2F2146B98,0x2D16B9C53E76DBEF,0x3AF2727F29F2DADA,0x984B45C56918B353,0x37C81C5A2209E7D3,0x97016F54DAF398B5,0x3447581D232A,0x3E807867CEB45F47,0xBC0F26EDC054A5A9,0x2214237DCEDD5D2F,0x3BD7AEF4CBAD1824,0x856DAA2D910FFE24,0x24F6E85D7F021745,0x7D311BBD0053,0x7D5065C4F9A50A1A,0x71B21FAC253A664D,0x67450CDA784FCD3A,0x17E1D2BD27F96864,0xC25CF6477E4D7FB,0x9362A790C4B2863F,0x1C3FF3FACB806,0x76CAFABE7CA00D9D,0x16C921A829EBEA16,0xE1C9848B9172A61F,0x3523242F942885B8,0x1459A24944BEAC2A,0x3E18328E121EB194,0x1FB674A2C2864,0xE35367580B08DAE6,0x139C17F150AF234B,0x7D96432428778CD4,0x1BABD5A28E66BC54,0x81D2ED38C7CF0249,0x82128944EA4E4E21,0x930CFA8B3815,0xBCEEA42B46D40010,0x7A9E32155A49B1D2,0xD98481448CA22FCB,0xDB00536FD618070E,0x1CAA9DD6BDDE91AA,0xAA9530C915F5615F,0x22004456DABF6,0x478BE07931781563,0xF92A9226ABD883C5,0xF009B14970FD2D43,0x3D78FD6D51D70566,0x37BA0F6C56A45D67,0xA98CDE230C79A97F,0xC0E7C4001146,0xCBB92EAF50E3270E,0x2C270DC78F5FD150,0xA16D3B520722B1B6,0xBB06A781EB545C32,0x6DF19292F7613249,0x27F7D39E91CA7960,0xACBF3CDC9D18,0x6C24C5B14B715067,0x7338B339A52E1E5C,0xD74D6AF9B3A4196A,0x2507E0718C24516,0x993B19A2AEB404CD,0x319518BEA9CF3AA,0x80CCC7E333CA,0x1D5A311656557DC8,0x50248DF3F8BFAB49,0xBE6DE66CA3627270,0x4E8CCD89D70A2361,0x7E92FBB02CDE605C,0x3D3D928076D1783F,0x65938433BCB9,0x3C169F251C740E24,0x45ED709DACCEBA18,0x14673B549421712F,0x8878848E616E1052,0xC21CC6AA55B7C4A,0x21488470128D7140,0x1EF3067A4A60,0xCD77694EC2A2DC3C,0xC3031B66ABF540C3,0x42E5E6E224AB08A8,0x4BF3782485941333,0x4E8AB58C7BE30A5,0x47CB5BE8433453CF,0x27586F74436A,0x7A72A8CE92345563,0x5C915E31869FFDB8,0xC8BDE8B035E84EF7,0xAC41C89A7E163D91,0xBC64D0E9EF9094B2,0x8F680410546BDC1,0x183699797E09B,0x5E3C7A8202DB27A9,0x5078582FA0C61545,0xF8034C90C2901377,0x225882CBF4379289,0x679F75E824597CE8,0x4207926E24DDDF12,0x8A60995305E5,0x89A57A19B8E4CEC3,0xF1FB9FCE99A5F3E6,0xE7FCDF9B4CC49F4A,0x873E61E291216D35,0x2BCFC2D887847688,0x57659E7711E2F8EA,0x1D45861E18188,0xE7E9449796FECFDD,0xDAFFA9E4A672A0F,0x82FB1EDFEE120A14,0xFB6CC419B9E86256,0x5CDE4AA77F71FE0,0x99B64E8E0E3E62AC,0x1B556BFCFFA87,0x71DCF31E304AF921,0xE777DDA355531DF1,0x91647D4B29C75838,0x350FAA4AEA04AF17,0xA12CFE504FE40AA1,0xEEF7D506B490A50C,0x121CB39F53C61,0xA6792D5A986C56F6,0xCE146AEE753F4ECA,0xDD8470DEF334D31E,0x115568BDC760ADD0,0x5D71678677D8B769,0x326BD721419AC637,0x765C297444B0,0x525807C7193F7304,0x6C25E33983E714E3,0xDA46623DE92C0B81,0xD65E14C5F72BC45B,0x85F248A4F4E29682,0x49F218B552B5CA89,0x18AD8F074BCD7,0x4357F6C0FFF5F3E3,0x948F6585CEE50EC6,0x4337AB801A8E7E8B,0xF978958713E383BB,0x3A25F5056B19CB3E,0xBED0FB2B6B043833,0xAE7343554826,0x2DA10B3C612DDE5E,0xD62DE83EAE25255E,0xF6C1688E6BC9952E,0x89E384011B283A36,0x2C8A8AD10ED5C10F,0x4B8B1E08F085A057,0x73E4108CFA27,0x2A04FD6F2BE91317,0x17553EBCAC5499F,0x2847742E01DAE68E,0xEB5C65C98089FB06,0x3E6934E7DE72C8F9,0x692583C7769B56E,0x2213D5EC2929B,0xD0D728BC0AF74199,0xAAB068E6AE609E4B,0x76D109E8EA4CEC0C,0xEE9CC7673F2E18F8,0x4AFAD33DDB7AA968,0x15B69BDC5A0BD64C,0x1387FC89DF28A,0x74C7BD021F9BBBAE,0x173A8DE0B3733905,0xC2CC683146FFC765,0xDAFEA8B8169FE78E,0x9C4F7A79202923D3,0xF80B65951D228E1B,0x16999ADC3A5D8,0xE10BA2FD2CB60D21,0x5194DB871878015C,0xDFD5A9D0A4F00D19,0xA6BE5343E2C757BE,0xCE91D788C1C018EB,0x13F21D7EA0A2AB5B,0x10F09344B37D1,0x44FB71C89699B435,0x3F89E35B9D1BDB08,0x7C8AA49731BA0BF8,0x5501A7D218CA0127,0xE31EAA82AA9B86BF,0x1D97A4642FF433F0,0x20B2291284773,0x867E224C9A486C46,0xE86CB5A39A4F10C2,0x2E8662AAF7691355,0xE6867E91FCA5FBDB,0x7A2DB70399C587F1,0x7946079D1CA73C9,0x43FE49044D70,0x9494A12CED37EA3E,0x5077BDFD2CA9C4D2,0x759BB16515819FC0,0x6EEEFD51B41A9136,0x7AA2074738E8D852,0xB9D44DC4390EEF8,0x50960B8ADCCB,0xCD4715F66BE3F52F,0xF37F656725DC0084,0x1A27AEDFA32803A6,0x797DD2EF4745A0C7,0x937ED5AB561CB0B1,0x89039E32C0859237,0xE2F9C64B4CE8,0x54DECC262889D9BF,0x74AFE7B02759D454,0x3F9FBCF8F35D186E,0x30ECBDB78EE009C4,0x839391B3B051C8E6,0x3056CFA0147324DE,0x153FE5D6BEA13,0x9637C00981E395EE,0x6BB1FF9458A4D5A1,0xAA1D25978F5CA05D,0x96505D7A3928C85D,0xEDE506D8281BBEDB,0xA92E85292FB240EC,0x1FA150EF990FF,0xD2E6DDC0FB8299E1,0x25439F5EC5A22ADD,0x3B60757C689050AE,0xB7612854BD4D8244,0x736E81A1F1327FFD,0x85669385D4C95187,0xBFBBF0B2BDD4,0x92DCB5574E6BC458,0x470B6D78C6A382F4,0x9F54571320B85D32,0x23896E34FDB3C084,0x305A1F58B77B9AA8,0xDEC1688CAFE3E69,0xD0CC9AD366F2,0xE2D1360BB47FD748,0x2D44674ABAD23B8D,0x2A4B2873FB41EAE3,0x6D4859C7363DD9B8,0xF37F816DF3FCF888,0xE0477D3484FC0456,0x293A8326D945,0xE0D4217A4842C5A8,0x29631C26C2C053FC,0xABFC63D480C37321,0x6DCF467652FAAE99,0xCA028B32CB2671DA,0x6E19D3F87E214E9B,0x18CF55D248B0D,0xB0649CFDDED5F60A,0x46C8B3129A9290C5,0x27CD2883DEABDF74,0x33881051CAE8ABE3,0xFC9CBAFB9820C528,0x75B27A70CFAF518D,0x2CE03E3ED012,0xDA19D075D175E772,0xBDA7225C2EEC66B,0xC00659BBC702D604,0x450234D2720D90D5,0xC4B33483B95C13B5,0xD754E7EBFD32B18C,0x5D1CDC8E19E1,0x475449CAB7F19AC1,0x6123011D52187FDF,0x39A4BC2401A05037,0x69FB97B9ED4D7E18,0xAA3B16B702FCA682,0xBF32E92B272BDB39,0x749F0C0E7381,0xB15FE5069561F658,0x90796EED58FEB224,0xE1598824895F161F,0x5A60A354D92800A4,0x800C0D1F593EE77F,0x7316868A1CC6BA54,0x1B1DCABF99E06,0xC0804F571009D1BB,0x7FC108A121922BDD,0x1EC025084BAD8D2,0x2BA19C87EBDA52FB,0x7918E5679D7DEC35,0xD5CFDBD7C5E36866,0x177C5749C7810,0x451C4C0965D6B189,0xCBCBD8EE0252BCBD,0x1678099265583389,0x440C9E406F762994,0x690474D6301A60C6,0xFF1B7D1879EEE8BE,0xF5593F11FFC6,0xF95E89ECD5719905,0x43807570CA2FDA4A,0xAB20D6D6B3981B1,0xC36B172014C835E6,0xF79CF8D7580341EA,0x5B46CF0F99B4D9FF,0x5C1DB2F17C6E,0x6B05890C27B023B9,0x3C2908CED7A02CE0,0x5B06236DF663B5F9,0x4CF6DD81594A39B7,0xD9A1F4FF499E36A9,0x2FBA855CAEBC2D19,0xA4E7D7BA081D,0xDC2D1E76C19C11C3,0x9693B542577E8C94,0x83531BE399656DE2,0x3EC797184285BCA8,0xFA6E0A0BE882B41E,0xCAEE58F9F7E64E82,0x1DA191FC78A1B,0x7AD28200A53EDDF6,0x758CF4F667DCC812,0xE3D78D5ACE59A3B7,0x1C16FCCABFF89EB5,0xDB45414846334D45,0xBCA41A0848D67EFE,0x1B78BFDFB06D2,0x9C63377217DA7E0B,0x719760FE724A02E,0x5E03B8E9B0C0F0FE,0x5358FB6F808F7D19,0x41C55BDDE587A7F1,0x595766583AD502C7,0x682EF0D91903,0x1A9FAFAE8C7AE9D1,0x4DC9BADDD74E7EB3,0xCB7B906D4DF01017,0x389F37661749D1A7,0x159E17B2E7B9E351,0x11FECBE7F5565ACD,0x1D32A111BDF0,0xDEE8507D7DF168F,0xFB1BEB256880AEC8,0xF3D33A81CA1A103,0x8C9BCBEABD3CD1C,0x815810164999AA31,0xF8C1BB4CC41FB7CD,0x114E5440C85D6,0x10E90FA55705C86B,0xF49C4CDADE740B61,0x166C2851A4A43E14,0x8C64E708282DFB27,0xE3C6C4C9237AF699,0x4320325DC684B438,0x120A916B52968,0x511B88241BD28D62,0x1932520B0CB03F1B,0xACBADF9898D38318,0x4BF6A3CDC691BA04,0xDAA6571EDB30322,0x805EC9D75CFE7FA5,0x2030336D8F852,0x94FF4015F58C5055,0xEB94DB4F47393C32,0x654465FDA06AF8D4,0x843BB8AF4A2ED807,0x60502A23A7ABB7B8,0x79F3CF2E6677F29A,0xCCE506D0F4D0,0xAB6CEA5FDA2818E9,0x447705A86ABC0544,0xF818C2107B20EE82,0xEE39D5CB3C71AEB3,0x59120C1DD150D852,0xA2FD06D17BFE0ED5,0x1C12962052891,0xE6912D582E5DE00C,0xC4ACE0625D12CFD5,0x47B038C7D55FD148,0x5ECE15D49E9A9B9D,0xE99C999F9C7BDF4,0x304B114A013017BE,0x12861199AA0E2,0x45E66C2943B4797,0x6F68F7D684B095F3,0x7A42C2A5F3806851,0x44CFBC45CD58069D,0xB581C7EB31E8CB6A,0x173028EDB12D1F81,0x11A103ECCF3DB,0xDD6A8857884F3F9F,0xA28545F3E9E2998C,0xA8810E7E69ADD2BB,0x8199E044B4BF9AE4,0x277BAF9434225AF,0xBB32C54F7289799E,0x3F9DE55DF24,0x6B3C45B07ACD4C38,0x78045F498E831937,0x1BBBBBCA187552EB,0x19B746F388FFA9C0,0xBB138DAF516D470F,0x2748E742650A81E6,0x1D7F07C8AE161,0x2EBADB714512E3FA,0x52A6DD105A79A62,0x2B20B2B4326F498C,0x6AE2C5A5605C2DED,0xF14DD00233FB0639,0xC267A7288C55DE0E,0x9B96CD0D6D87,0x11F424B523335301,0xCE8A1E400044AAF6,0x75B519C4BCC66336,0x3274DAAEDFA93270,0x9AAB5426D44E2B4A,0x8C00C8E0D2286AC6,0x1E0980868C7A7,0x50EEAA573BEF3EC7,0x5BC202174B5C5576,0x83989E848CBCEA80,0x1C906A23CAE80AC5,0xB8C67F3E88DD57F6,0x5E2937DF8CC74264,0xB8F399292DC0,0xDE90F317F08693A4,0xDB7B34A5C96BD139,0xDC416D33DA44BAB2,0xAAE9CFF565F59707,0x732BEBB72BD90E2,0x20D2EFEE5B6F943,0x21397DF88A03F,0xF8A62156B0368D5F,0xDA87A5120DC14EDE,0x43BA6D5CFC117CD3,0x58F905BB85B611B5,0x18F4E53BE8267601,0x2DC68C73DA725E87,0xA93AB7E555E3,0xC83CC3B8E19705C4,0x17ECD7B60AAAF6C5,0x1E7DEE95F199C725,0xDFFE0EF87D631301,0x5AE7C8B860D4B73F,0x1327A66587F96244,0xB013A0EA7643,0x97197E6DEBE643F4,0x5792BA668EC835FC,0xE3982E77347BB74B,0x5DCAD5BE139627BF,0x1464C160FA82102A,0x760D9084550FDD9E,0x1784AA9B31798,0x8608C16953267C7B,0x4BEDC9AFEAEE6FA,0xA8E715C15A273D8F,0xA524B3323D5023BB,0xCFB7220A8E8A61E1,0x2D4045B242FEEE6A,0xE6AF2D70876A,0x5BEB7A2DB2C17C4B,0xEDF2E7B5CBB40D2E,0x1FD355B4A9BD0272,0x540D587F49B01B1F,0xF5CD1CF08F8BE260,0x58BCF8D42BC092F0,0x1D85DFD580CF3,0xA569798BA2B1289F,0x3D3E575562D7EF40,0x2008A351749FCDBC,0xEC0C9AB88E345BCC,0x8252DF4E120B6F40,0x99F9894FF20EA7D4,0x163F5148B86FA,0x8B5972F80FFBCEE0,0xD8ED9F8DBE1AD7BE,0x466BC7314F8CAD0A,0xD921722CC0DC90F5,0x7CB1385D6D62A732,0xBF496556FD7B6C5B,0x1E3B560350E06,0x42D381BEB44C0C16,0xF309533834F2DED7,0xBCDB0FC5912558B0,0x54ACA00444678595,0x849BEA2D93C0CB08,0x77FB3B48122144BB,0x144A7E2DEF9C6,0xEDDF25E8FA5ABD31,0xAE24252374647083,0x64A75A3A09FB281C,0xF38F9BB656D48F88,0xF4A75E55CB1725B8,0xBD8DA66C8A41AFD6,0x13C546580157A,0x9D5B7F30F99BED5,0x833DBF3993B01E0B,0x447BFF13A7C453E2,0xC732381E2F9F33F0,0xF26636683F495439,0xC89531A136E4BE61,0x15DD9A4E5513,0x1B1D70651CDCC789,0xB332C2BDCED06B39,0x8F54C111C393B743,0x136E1CF8E1DB566B,0xB4D965109498796D,0xE842260CF9EDA2F8,0x168F556831EB7,0xCA683952B5C92A99,0x38B5F0CE5FED1600,0xDAE90F1F1539BED2,0x435E0FABEDA6AE51,0x16F83DFF118591A4,0xCDDA13395966922,0x18A276007553B,0x5B350D96065486FB,0xF4811C2E4AB3EA67,0xA783541898D12EC0,0x826C0E30A6E1D778,0x5E428DF2154C848F,0x905EF7760427F8FB,0x1CE7E92B03CC5,0xBCCB0A872F1E6F1B,0x8273CCB3F79B9481,0x4F22B1CC669E8AE1,0xA47DA449DC3C60CA,0x49A2A43CBE4EC3A6,0x41343F89FE9B7564,0x4E9E51274683,0xFE3951E7E8A06D70,0xD250978885F69392,0x39C94AB48B268956,0x438548020A43D990,0xA4387181A250E5BA,0x590E63F0B36CBCD,0x18EF3BE3D48D9,0xD63D0A350423E0C2,0x737ACAB238F1956C,0xAB5AB648FA4E82C8,0x1BF6B343574F57EE,0x5EDF811E68EC9FC7,0xA7F24263B48A2FE8,0x44D278FEA754,0x1F84DCC56B286883,0xD6746D8170089EB7,0x77263C58330D79F1,0x4245A3195BB0DFC0,0x4AC05927F99E007,0x33200638E6769004,0x3752917E56A7,0x29A226FBAB8B01F1,0x3865F449B72F9847,0xCDD517DA41E90CF3,0xCCD2941EFC805009,0xF425552B5FEDF8BC,0x52A2B1E02BFB0370,0x19D6B3FBDEFB,0xC68C5D1C5E282FAB,0xFDA531C88FFB4F7B,0x5AA2557390E746EA,0x544A1F189D86AD25,0x94461CFAE26D6190,0x20D8C570EF45F4E4,0x92A7A6C27BD6,0x262A48B93D47AFF1,0x292D1E55B0F60CEF,0xB234241984DF3A64,0x218A31F030E90054,0xDAE73BC1C4F98507,0x9F97BED12A64E86F,0x1AB63C38240C4,0x335EDC3CD5F9936,0x98F694E17535D753,0x3870AAEC6F53C8A5,0xA486CB7A1913AE09,0xDBD5FEF6A41E29EA,0x3DADD7BD21E339CE,0x10E7361492D,0x5095977BDEB1B36C,0x357FE8B37579158E,0x385CA177BF816FCB,0x8F08E526E1DC77A5,0x16B7FB99FC6A014B,0xA7264DBD7FA6E58,0x181B3AF74C554,0x5A3607372124190E,0xC1A280D2D109C80D,0x4362F03886804EB5,0x78292D62C6A3CD8E,0xBCB91F20C2ED6BB9,0x4E1064B8492F2D41,0x22C00D97001B0,0x234C74EDC4142408,0x628945864F2E94C3,0xC4C469959854AFC,0xADCD7C36DD1C4E66,0x179D081C1F21D075,0xCFD6D6CC5DD8DD23,0x18B7D04B880ED,0xCFFF90AE4C25171C,0x74F86028A40BB25D,0x3B68BFE499280CAE,0xBFD8625FDC85EC9F,0xE0CB680758145DA0,0xB62D0E45C4C9CF0C,0x3F98100E7AB2,0xF15E1ECBBEA945F4,0xEFAB4EEA5FEDED0F,0xF2CF754ECF898640,0x7F076FEC12D4B55E,0x648342D624F7447E,0x9D97CA409913964,0x1B3FCE1227EC1,0xE575D42A223EA064,0x59FC4D7A5000EC75,0xD507B99DB1979308,0xCA39CAA96AE62B90,0x4079CF243CDA9F5F,0x7C6F41E10C337562,0x18916F5705A05,0xD2B2B5405B409DB8,0xE24430C9946160F6,0xA80A1E0E4F933EF1,0xEE342DBCECECAFFF,0x48D75C2F1025874B,0xE6A399FB36491A7,0x1803600BFD56E,0x335A0B3DEBD77BDE,0xAEBCB4CAA302204D,0x4372C361B5E88093,0x1DFB6A22A735B063,0xC663C469398EB683,0xC36A47FDFE8DA729,0xFA3409DAF087,0xD46E8AF99E0A7507,0xB912510F967B2222,0xFD7514B8ADBF1FFD,0xBF0B38FEDF7C485E,0x34E415C0E1D11F9D,0xC1734B56131B1A0D,0x1164695919A90,0x3D7BDB50F36CC367,0xDC909A6A2062AD6C,0xE9CA98A3C2E5612,0xFAAB136D5459ACCF,0x6AC1FBF6B86F810B,0x12DFAD1C0D971EA1,0x1CEB5EB055D8F,0x7878FDCE9D8D1326,0x6E77B0EF83B6CD74,0x120953B2DA5355B7,0x6C90352F433F8231,0xDA52D620FBEE2E2D,0x1CFAD28C4FB6D282,0x2141C56FCEF2D,0x31C6B070C5AA396A,0x777BEC4CF965FCDA,0x87C9332479689CF2,0xCEF4AB47EA284896,0xA0AA9BD8CAF5B4C,0xAE55F35DD6D0BA49,0x62CEEDB0A074,0x38EA80B7B8816CED,0xAA9F1A4843F8FDE0,0xC1FBA0970CADCDAF,0x6234AA1D594718C,0x575E80581F5197FB,0x837A50BB9163678,0x617D6A3B5189,0x441AEB9ED7B63AA4,0x450FCC1286BDAF19,0xEEC928E70B519D6D,0x244638D26CF6ABB6,0x72BC30E830993CD9,0xEDC5385A21AC61A6,0x19E420F2590C,0x2FEA141E2D0F2CE6,0x6687ECC658352F54,0x3A2776E7A28D327D,0xD7C619C47C053E72,0x88E3521BCAE52EF6,0x4226CAC9EC6A5AD9,0x130BDB9E0CF87,0xE28F6F7A9F333C1F,0xB5C4DBBFDF30DAFB,0xDCCC3CC4F0DE9374,0x6F06C6BAB3BEA310,0x8422E66D7183A750,0x103404DC6C2501CF,0x127AE009A631F,0x5F0019831237C303,0xD8B7C43A933F66D2,0x8D82E5C774EE2E84,0xD86799796B4A0940,0x9D28E1AF4D952061,0xE8BC4028C7F1AF56,0x230DD4A408184,0x2FEE7F36EB0B9677,0xE88058D3094FFAD3,0x3E5C400D29DDC528,0x70194309A36630F7,0xD5136F76FF1965EB,0x74D5BD83226082F9,0x3DAE7C010312,0xCA8EDB0D37D985EA,0x71F65BA6C423D6DE,0xC1B9C2B4F8EF6805,0xE2C6DF92A5910B9,0xB6B97DE36A12162B,0xC42A85E32C00D36E,0x16929E327D7F7,0xC446F608236B2BA3,0x885AB185C618756D,0xBE468E30541D34F0,0x6C0F33DAC7AA4AAA,0x325F5886A7F0F2FB,0x72BE4788C5E42F0A,0x11B122AD8C47A,0x3B5ADE51D0C8E0F3,0xF323EE00980240D3,0x65A15F21471DF7C3,0x3B85651CC1837805,0x4B06C77D18D1C90D,0x10A726E757FF02A0,0x3CCAA8B96F29,0x306DF77D2304820B,0xEBFD8781F18CB8C8,0xB82E97944B2C1A0D,0xFC84555DC206140B,0xE571378E8776F933,0x67C11EC2BC3440D8,0x386ED1902D08,0xFEC0BE7582F61246,0x464328AA03178D37,0xCCAE6196409EBC06,0xDCCE072E286EA86C,0xDC8E292F47D970C6,0xC62E831077BA26F1,0xC89A678E069A,0x29059E708B51A44D,0x9437D0EA6860623,0x87D7311187C47BD2,0xB5A5AC2D339398F6,0xC6197FC769D1D118,0x2274012853D09C82,0x10A217E3FC31,0x793F86A6CF142777,0x1C924FA5245FB9F,0xC29D1F6E479356FA,0x6B1207E06AF63A5C,0x29D60CD1BD4D4D8,0x46BFB6F8D8185C7D,0x19EFD294110D5,0x404D828968A7F1F,0x6DE984E4AC8E2121,0x7BB38DB646B6B612,0xAD1E45D8C97697AD,0xD8C6B5A1A129C061,0xE237C734EDE03AF,0x1A0DA68CB63D1,0x386EA9DEB6A12817,0x28F85B2EC109CFA8,0x4BD50B85412EF6B9,0x61E4B5FDFB4693A7,0xBA93077BD48896B1,0xC4D94C44B758C1E9,0x12D905E078360,0x7757708F1110E38B,0xB15D30AC75B642E8,0x44AB6126EC2752DD,0x2892BAA4777C5F91,0xF4697CE4BBA3DE8B,0x5DBA81FB7346754A,0x173864847C34C,0x6CDB54DDE7F6DC27,0xCAB8F6461FB80473,0x11E56D8ED2C1C47D,0x6C8ADA408290E1D0,0x10A50B18CFD12BC9,0xE884EC25EA8FA2C5,0x18ECF74179DD1,0xC39DDD8B9A6D44A9,0x69985C858A854A35,0x6730A1C478F393A4,0x9049E0490FDD25F9,0x430C13912435751E,0xED913176C7C47ACA,0x179DC264E3430,0xB62F21CAEC27FCF,0xF998D259FAF13AEA,0xBB50271D2847CD5C,0x5BFC6242EF65896B,0xA08CB626DC7A6FC3,0xC49D8F18F1FCAB16,0x1EF16424AF603,0x8648146350391E6C,0x1BE6E175CA8B2A1D,0xE2CABD65209150DE,0x11128F40DD056347,0x54178DCEA475BC2A,0x9DD2F67EBDF8CC21,0xF15C42BE9C53,0xA8336A19A6255BED,0xC88C342BCDDD841A,0x92A4DA6C2D78263F,0xB4528340EF1A315A,0x7ED6E9B8B3E3FC63,0x252B3C5CD1123CE6,0xF9A13241E635,0x7141C7E35F86724E,0x2F2F0CD9F0A07E46,0xE586233D8B55F990,0x922CCFCCD2AE4F60,0x6B8CA701918A378,0x33117B7667318280,0x106EE48F4511F,0xB99BFCAD16AA90F1,0x40C4D5D14931D4EE,0xE79B2C89E1697157,0x85EADFA0ED7FC53A,0xD6CB2244A8EBF831,0xE83976D95713A858,0x19FB582C0B588,0x5B40C855606E793F,0x1D5186594AACC33F,0xA5B7890A822F0ED7,0x4ADADD58B13E5A6,0xFDF7F3D15813B471,0x67D45697556C6251,0x5E66D10BAB15,0x581843781165B731,0xC4D04E3A796DB6B0,0xD466DE1829ED4FA1,0x50B190309747AD0F,0x384193B2628D0488,0xA11B1B0DC65A4A8B,0x2113C9ACA2FA2,0x23D27D978DCDA935,0xD28DAF552E85379C,0x3348D478B4374D9B,0x447DB2FBC0B4F753,0x3229249E31BAB724,0x70F244C7E59B0AC7,0x2988CAB809CC,0x6D65C8999C1C51,0x5D67DFD509522CC1,0xA7E76C93A52A8E63,0x8D9B8048B06D07EB,0x92AF279C120DC0A2,0xC608B65343FEBFF3,0x8BE008912055,0xEEAB7C24AC7B80A,0x8DC0B43B4C0E62D,0x2FAE92A84E3D67F7,0xCC1818AC89DE1362,0x46E5E8AD179EF1FA,0x8EDA28B2DFD6504F,0x1E91AF2D53CF8,0x9B925BA5415CD0B7,0x5A55E24B2AFE521B,0x26D064B479BEC1A4,0x33C88FD416A4E0E2,0x342FE1E66238278,0xF589912D111F2A6D,0x29C7817C5F70,0x39074838E8CD106D,0x24D49791E171F276,0x77476D7A7EDBCCBB,0xED5581F9B091D6D9,0xAB9839F581DAF42A,0x591AED8F18A770CA,0x34F142E891A4,0x9E478A5200E3B1A8,0x53AF3CE3D7E03832,0x1403276224EED7F,0xA0025BECB1780C09,0x2CB3010B43E0DE6E,0x35E1A266F37D0786,0x227783FC0A67A,0x3EE79DE2F5731884,0xA0652B4DF6540AD3,0x5D60739A0801E84C,0x9634D066F67DA69,0xDE46F099F716E6DC,0x1394C77D01422B62,0x14F8174FCD4B4,0x99939C50AA474AEC,0xE45FE5F4C7C316A6,0x57CE98D6506FCD7F,0xE411A61988581617,0x36F85E66BE89FE6A,0x6E0A07EE5A9527C0,0x200FDEC9109F3,0xE12ACE513BA60112,0x8C1C44DD055F5CDC,0x84A4D795A7E60132,0x6641FE5C06A2465A,0x88AD5DF59DBCC772,0x16CAA3BB05AF398A,0x185A58117E9B9,0xAE440E68D7F21145,0xBC8C55A7410A1EE5,0x478571C982DD320E,0x49AF688F35B250BF,0xED922B78F9AFEA43,0x79A3D01CB9E97C93,0xBC5658C1A379,0xB4C731B3C8441561,0x81CA268A00726DA5,0xB5F63352F8946954,0xF00068717EBD898F,0x488D37A19868517,0x839F0A324FCDC63C,0x2160365995965,0x4438DA172F815F7A,0x9A9298A70DDA83E0,0x808DB5F2EA9A713C,0x97BBF594C29DC301,0xCFA43611BEEE73FC,0x3E9719AAF9FCA231,0xB86F6CEE9435,0xC344026AD67BF65,0xE0E76982A139A6F6,0xF9A349B6152147CC,0x1558F511CF9B742B,0xD0DECFAA2295EB22,0xC9F63FBF65D53777,0x15D289D7C39DB,0xE452DFA30435C03C,0x74404AB8CC40F04E,0x9437C391ACE50954,0x83BF94D049887654,0x79136D44787E150,0x6E24E6614D3D88C0,0x1E921FFDD3FB0,0xA1F7993D6D7358C2,0xE5F0EACD4C14CE77,0xA884F1F2184F11BD,0x2DF2A84CF5AFBA3C,0x42C4D2DED216A0B3,0xBFE4B5EE84F27955,0xAA35D9E513AA,0x4E8BF2D44EE2A912,0xCD29A3437B21AB08,0x1A8F02B5FDED0086,0x6CB3C90A4A1A4820,0x7C9A516BC51E76AC,0x9806D151CD47B28D,0x207238D50A001,0xA31B1AB02D7364BD,0x1C50A6D7DD8B5D16,0xAC2E067D5BA5B520,0x30E232E2F4FCC3FC,0xA2C3D6C80D01F4C2,0xAFAFE79BAE663362,0x194483729EFE,0x53F09274C09CE4C1,0x175C011DA6786529,0x370FA6C609ACCFE7,0x650C3AEBE84BCF2D,0xAA2F381CB765BAAA,0xF21750798A196EC5,0x1FBD8E55213A5,0xDF92AC0FF6BAC480,0x99A7E77BB08A4170,0xB7A4E6906032C9C2,0xE00CC936CD926D9F,0x2024E20EA1A45948,0xA7DE05053409558F,0x535138DB51B9,0xC57DC9B1B7EA96BF,0xC24D3F78B0D052F0,0x90BDFD7CE4D22BAB,0x86C2E1AF5E204EE7,0xE3176F4BFA7F3F51,0x5A329556369ECE14,0x20A578DAE9072,0x456067BF588A859D,0xBC1A60D726A9A06F,0x7E9519C94B8AD362,0x734C45A7E3A0A19B,0xB6855342A518DF03,0xD345168EF326BCDC,0x99C518E859B8,0x36C8F5A58558EA7A,0x53991D7B0B8B35CD,0x4D5935A8008DB481,0x3C3535BCE8BAEF76,0x6CF0B22F92982AD1,0x95028DA2B15832A4,0x1D978CA851450,0x78C70B9F5C40FE1B,0x87FEA1B6837FA2CE,0xC1609402682C3148,0xF1B008E538041204,0xD01BD7DBEE2E89CB,0x57741F8769C45525,0x6BBD5BF19149,0x6B311AF32B5F2638,0x7A2D1D8AA8D41B89,0xD7577F45C3C68201,0x7E0994FBF26B2FBE,0x2744ACE52651A57A,0x388CED78F4040D51,0x7DB57A654650,0xB9E6F8D7F49C4753,0xAB1E9E630CD69AC1,0x9452C0481E39C330,0x648AFDF16C80D566,0xAA2C31B00310C8B9,0xAB425FD9933B43DD,0x177E67D9150A1,0x4F07C92C3F88FED4,0x2E6D28DCE21852C3,0xD7D30F2B78F7EE63,0xAD2DA5167A6D12CA,0x9C73191BF346109B,0xBEFAE479B662F262,0x397D9073C12C,0x28E9065CE04A19E0,0xB1B10917C0C5CFBF,0xDBA7F084B0777F1F,0xF6A31A04DA166FCE,0x1F08564D8B70B361,0xD87C7BEAC0E934BB,0x116F26AF8269C,0x90659DAF3C968EC5,0x4A9A6AC04310D479,0xF92DB12D03BF2613,0x144169716C2E5CA6,0x148A35D90E1E02C0,0xC407F85ABCCB6AE7,0x1E5E2F24390D2,0xFAAEEA4B9EBE993A,0x9774E4BA6CE74326,0x6E291FB25A946664,0xD56000FF00D9A443,0x13F018BB96420111,0x522C0271DA7EA156,0x498A28D03A19,0x2824FD2A26001707,0xE163DFE9AE3F1B81,0x6AE1EC7ED96AA01E,0xE885523C6904C5B3,0xE7BF7FC0F7D7E3BA,0x975FA9563D1752B7,0x12F95951FB4BA,0x5B7D898BA4023748,0xF8F761FEA05A6D00,0x8AA748EA32532A42,0x831D59C8A18D40C6,0xDD824A3AD1A66D48,0xC5E2FB862DE035F3,0x1ADBE4506A161,0xCC59580F01374B3A,0x8B9DC7275DF1F06F,0xD3D28CCE201D27D2,0xC3929A0915F4D32B,0x20D34D6D102CCAEF,0x175D6078407F766F,0x192757543A82B,0xA4283B496E06969D,0x686BEBED27A01967,0xFD6EFDA84AA9D9EB,0x8A471641AB3E6AE8,0x66630367388F8DAC,0x6607C1D910814078,0xBC14119BA0DE,0xD0DBB25E190D9F85,0xE08C081B2385D539,0x10A991FCADB4800,0x61E7B01387A1E19F,0xB4683FF4214AF6FA,0x37829CE5155F6B62,0xF8EC03E7A9FF,0x865437BB03531A8F,0xF1507D37A2A75889,0x22CE8B0872393F4F,0x29CCD96871C30F9C,0x94EDCFE3306137A6,0xDB3E9A01C7405BA9,0x215C6985AB039,0xABB82784E49EFAFA,0x8D38BABD54C11993,0x67323BA43E55997A,0x3D74BD7611C11390,0x13AB8309A344D65C,0x9A918AD939D06E00,0x399E6F795B36,0x9F7790031BD2B4E0,0xA64027C2F4CBAC5F,0x7D70AFD3176CD163,0xFB527A9571A89BE5,0xD72011FF8EF7426,0x1EF27367E20D00D9,0x2EC41B54AE9,0x4250C0C66E966276,0x80A9C34AB54673E9,0x684EBDB26C9BD958,0x33DE7202F3E1C374,0x9442113D28D8B5A0,0xE1A4ACC8A40956FD,0x111F543849C49,0x46275E46E466394F,0x451E75DA1168B58B,0x4BC019643A2AFB5F,0xAC10A69F587EB711,0x6A041280FAE40091,0x2241DFAE7D9548A8,0x9013BB6A265F,0x1F3FF329096DD1C1,0xD26E9D80D0418116,0x2D091C6B90BF5751,0x329475075B7D2257,0x29D4D296E2DC4575,0x92CA013C03AE4C25,0x1B45C6EDDBA2,0xBF10CE7722A1AC69,0xB5E11D1E3DBF741D,0xD6F7CC10E711F83A,0x73F0E323AD6D01B7,0xF02B90B47903F211,0x403A4C7FEBBBFC9F,0x1025F65E44D59,0xAC93A2A866514811,0xB38B3D0DA05135E6,0x5E1EE416D89DC1CE,0x3DF3B049721885DF,0xFF8269EF5B29D198,0xC164472D76A6C3BB,0x1EF52B16B0AF7,0x73010501C92C8D4F,0x13E977CDA821A895,0x1534EE4C7C53F76B,0x582121ED15833561,0x3676A4F21A9242B,0xDCBC9027BF85C48F,0x175B02634F8CD,0x58564637464DF314,0x27CEAE73F9836D8A,0xD62C4072B3E9D5A3,0x47CAE822F777E855,0xA58F6C7B5E43AF39,0xC1EBF5899867D24F,0x20184ABF59935,0x1646D0CA4B0B16D7,0xF4FA6B308FFAD224,0xC2D6A15F67C9AE81,0x88E05AD0B038B7C6,0x9878DA01D49AE46F,0xA068AB73BC495E8E,0x6833E0323E96,0xA46EF2C0DF52060B,0xFF70FD801CB800C5,0x6F6A015B58008B4D,0x717336FB0602ACA9,0x36ECE64B7F9E760A,0x3C6BCE38ECDDC2CB,0x1976BBE3C2323,0xD78335274E11901A,0x8EA65248D62F31F2,0x62B86631BA95DF32,0x201DC7FA0960B38,0x20F810D36C847C3B,0xC4606C63FFCE2DD2,0x1F2CD1D3E90FE,0x22E4AC84C301BA36,0x2D244CF1AB757BD7,0x1E8F3D77CE1709B4,0x5B41C4FC85AD1A97,0x267F541BA447509A,0x92A4637BB5A9591E,0x1454D698D1042,0x3A198AE0EC2E88F4,0x8665CABC7B009EF6,0x85C04D5216D11358,0x632C4A7810145F33,0x436C29D66C230B17,0xC0CF85F5DDE51A17,0x16654E3A8193E,0xD7162F3C30A001C6,0xCAC7C4D46E1C479C,0xCB91C1B4682ADF9E,0xCEEA01E6533CB01,0xC5D4F7EF442DC085,0x179C6618526CE3E8,0x8070F4783AC4,0x3192CDD1DA9668F8,0xAA311F4C8D9CDDAB,0x26EE433140CC17C1,0x89C5E712D5284CD,0x5C13DEA2FFB03967,0xABA8C766E2DFC668,0x1FE787134248F,0x8925BCFDBF25BF7E,0x67F1D9F8FB4B9312,0x1EA1CCB09EB538A7,0xE91330D143D33DFA,0xADCAD580ED3A71EB,0x7944054AE551DB7D,0x6F416A28A88A,0xB65462AA9956E85E,0x6132E06D79D5626F,0x394AE20125818D3C,0x5FA82C06AFD3A642,0x3A619194D4217DA1,0x3CBA78883DF2A412,0x15F1AC93C2051,0x31A509DC673855B3,0x9A48C72B31A50FAC,0x206D12668E7A935F,0x158DB8D58587F76A,0x607D651D7F91A9F3,0x14BA3459FB931EB6,0x13873696947AC,0x577B849454CBB4F6,0x3DB8A0A99E0B306E,0xDCF6F8192582DBC4,0x5DC2089915661BD5,0x1B9267A295DC6ADA,0xAA233AA42A9EB6F8,0xD56E9A9DE97E,0xFBA786F17CE3A13C,0xB839EC93107A564C,0x15CBFB34ADB564BF,0xE45E143E0B614912,0xFFC0F9DA37E54277,0xA990BAE8495710F6,0x7CF3D77B8DEE,0xC1304F16528B8467,0xD423F4694BA6F018,0x1AFE849AF2D576BE,0x41EF815C755D2C72,0xAC4F8B38694070B8,0x50D56A2FCD9BA669,0x1BB4237E30060,0x347C0554E44E8392,0xFBFE97E88098EA34,0xEDA0AFC1A5623E72,0xB8FAF31E3C1704C1,0xEF42CAA663DD290E,0xCB3CFC5928058B85,0x389C65F969F8,0xEEC8FCE8023EEE7D,0x5B9029B2E486F9EE,0xB8CE6E6246CD8662,0x2B2BFC7F09DB83FF,0x2FF8674588E5D9C6,0xDB3B44D1E653B173,0x17650922AE262,0xD30168CCDF8D8A1C,0xA95ED190B906F575,0x71E5252322CBC69E,0x3098900141328F2A,0x2303AC755EBE1636,0xFE3D06D0F50740A,0x12B5014EB58D6,0x76B8236FAC4A7E44,0xDDA8C2466247AEBC,0x407E65EB76839605,0x436E7143A8142BAA,0x2E39E56BE2EB57EE,0x34743111B8B37335,0x2250E74EAAF4E,0x9D9ED252A8A6FE5F,0x3AF11AC4CA5FB2,0x48F5687A5F19E7E6,0x72147050EB31B14F,0xC9B6BF533445830E,0x1F5C147903A999A9,0xECB8FEB65A8,0x712DCB7C1124AACB,0x61AF05A80673AEC4,0x73D123A384DEA768,0x1B5B97AFA1659D1,0x1394525963CE7A87,0x67E3B9B2AE6BF4C5,0xB33795491594,0x9E8DBBDE515C3AA9,0xD3C7E5038A867E2E,0x831C97670DE8F05B,0xEC305D6C1E276987,0x4D9B9087E8AD749B,0x4EA2B16C25A128C,0x13CECB1D576,0xE9B8759143C844B4,0x4787724B5D9F2DAE,0x117C739D7D33E551,0xF741C65424EE6FE8,0x9D225AB76137F7FC,0x578A7D5ADFDB6256,0xCE7156EA95D9,0xE907FF8A3EF62B96,0xB910BB8DDFAEB456,0xE4DD23F424C0FC81,0xFC66A00FC84EB749,0x106CF1E35D99F9E6,0x36257454BA9C169C,0x1B444514E6B8C,0x863DA4810C1D6ABC,0xB8FC097D2D4B5512,0xD46402C6B32FB6C8,0xC68D97D86FB2176C,0x7EAD69B1F6199019,0xA44CDAF3728026A5,0xF6B31A2BF270,0x11C2F12B710020AC,0x80AB4034E2F60F4A,0x104829A2D4A52544,0x45891F9E0A65ACF2,0xF955CC567A1A86D2,0x85CC6FFE3D011718,0x8CBF46AC4A49,0x7AEF623491062AAA,0x7AF428601BFA4FE2,0xBF152599AFA6E62F,0xEC3B83B36AC42B72,0x46DFA95019F5F487,0x24B60EE1162205B8,0x20A43DD904A51,0xC592C3C05623CDCC,0x1E815B465076EF14,0xC1171B3751C8DEFF,0xEACF90488E5A9BA3,0x78EB4748039A0BAC,0x5F61D53E5D2A70AD,0xB245790BD8FB,0xDA8FB6A069ABBA0E,0x6AEBB07FE4F716F5,0xDA77072D704EEEA6,0xDD78DAE0A4BBFC33,0xCF9C1E74CFB5B45F,0xACCA07FA439E57D1,0x10584A8D5ECEC,0x80D1A1B9DC8E0C49,0xD4C1425C506A55B1,0xE70594A0A3F4F160,0xC1F5A532354ACA62,0xA71A13C92F59727A,0xDE383FFBBE424F,0x1A7BD4AB0EF18,0xEB67A9B344194695,0xFB188F8881511B42,0x99BB3B872E844587,0x6938B1F6F947602D,0x9667C52622D22548,0x6EF59FAD94B23065,0x9B5298A3F207,0x418220ADF7A87A6B,0x92B55FAFD4D8B057,0x53F3B4DAF32FDA0F,0xC4A03D9099BBF94D,0xD1A43EF55111CC5,0x3633F6AEBE6D89FC,0x192D5127E050E,0x443D58BF31CADFF1,0x50D8F8671F961D96,0x9845F2316078A440,0xDB6CBDED339D6C69,0x1D0B0F302C327741,0xA28D839FB77C3C2B,0x95841E27514B,0xAA3F5E1CEF0B697D,0xB54E055626557407,0xC9EE962BE3B4632E,0x33E49D164FB00427,0x8EA058B1A611FCCB,0xAB587243026DC033,0x195B19A981969,0xD9D15BF9DB80293A,0xE3BC8F2C74C12C6,0xEC76C8F4200CEBC0,0xE1DECE62C2950F47,0x2577DA28AB1AC78B,0x8CAE1CB049E318F2,0x14F64792A01EC,0x20425E02F2F53363,0x67370FFD7DC903CC,0x96C219678C2826FC,0x978FB6DEFA85FB84,0xB83D390F20E04424,0x2D14DC4FC99751CF,0x1331F3414F24C,0x740FA14257345D45,0x9957D0B9D45CD2DC,0x6C282DB4C0C271AE,0x259373F62F714D92,0xD155BB3EB7D27C5D,0x4601B0EA3ECB9FB1,0x1CB2DFFDD49A6,0xB760909FA3916E6E,0xFB21744262386A3,0x113B96CB303BA1A0,0xAE499D36C68931F1,0xE463AA8CF1E8DC1F,0x81251B4C4E340B4,0x2315609F2869D,0xB36F319042F6E814,0x6421B7F4DDAF2079,0xAF781D68F9BF462D,0x2AF04838014895DC,0x8D4B094F0F4DB77A,0xE779E0037D6EF6A8,0x11037337FD0C8,0x85CD378EF948B4B8,0xFCA02F570A4B9047,0xB7A91C6A89BDCB77,0xDA289D013BCBD439,0x9E840713E21F97F2,0x9ED1F96F4CBE746D,0xB5DD1562DCEE,0x277857B9F5F27259,0xA3ABB182FA4590AE,0x86792C42525EC39E,0x563CBE81D855BCD7,0x981894E6F68B43F,0xC5F59C64E5256771,0x1C0DC1D10BF1,0x87D2FD04A9519D76,0xFAB910EC4A1EC482,0xB1DE3F593FC89293,0x8B4ABCC6AB54418D,0xE05E52B5E14C5E08,0x3D3E3BE34B357BFD,0x4DCFBEE0426C,0x4773BC055166759D,0xCBA0F8E305338A22,0x6636C3B588DDAD04,0x4A121AC1FFAD6212,0xC85F608030CD1BAD,0xF9F6FCD1296B289,0x22313AAE157AF,0x552C62667F62DE48,0xD41DC057E49115A,0xB97CFA75F9E38D25,0x1AAD77DE32FFC67A,0xE91AA9BB83F27063,0xD98C9619304EA8C,0x21D6F7066A190,0xA854183F6CBA17E,0xFC21A7A7394A05AD,0x40B75871F11C3846,0xCD28B840B72934B5,0x8B9B780729EA75AE,0x8B893B16D4FCB09A,0x129D66B8D586B,0x4D48E19BB21A4C0B,0x255CFFC9D78E9DDD,0x52E95354B0F87797,0x6983A6F0895CCFCC,0xCA764830D98D6846,0x83764223D5774C30,0x1A90A7582BCAE,0x913117D413CD2E6D,0xAAA61B04BBA78845,0xBAFF79E08A77D6AD,0x9D7457B77E900A2D,0x5A92E2AADC36E92F,0xDA8DDF71AD5E839C,0x12E48D814BF59,0x4CD294DDE764BE3B,0x900DAD3529C3C64A,0x5041DF6210F30E15,0x4C65BF5B20269B2E,0x70F9F0564F830E84,0xDE54D53F042229D9,0x1BB5793F9C856,0xB648B48A4DC0955F,0x67F77308ABA50028,0x2E225ECDB0EAAB6E,0x3320574D1E4D281E,0x420C818AFF0765EC,0xB396F0D5A481F31B,0x835D0EDD552A,0xC90B2F3451266BC1,0x5D0C60364542D92D,0xB9CB80309EF6DD7,0xCD3753A62A9904E5,0xF27744A488E8995B,0x3CD07DB4F260D1FD,0x76DF7F499956,0x428CA25C13E9C311,0xB8F3F150D6E42A00,0x3C3CCF91BF73E25F,0x96D304F3B573E459,0x965B5DAE7EDB64,0xDD62F21AF2CEF962,0x16C51150B147F,0x92454ADEC8A1DD07,0x6DDDDC5C6FEF8914,0xD345A481030F91A5,0xE6E21308A3C2E142,0xBE01B09DA419CA56,0x38DA7C81E9E067EB,0x1722505C61C52,0x67449C46A016331C,0x224C5474B6F8E972,0xAF57EDAF79F073E3,0x2CEE48392904DDD2,0x6B33C0C7E3400D79,0x86CEBE5366B636E3,0x1015CED6E3D96,0xAF3586CF8EDE8AFC,0xA6FEC3DAEA8858DD,0xFB5CD1131375D56D,0x921DD2D8FF873CA8,0x3B98C5F5C2892702,0xD42996623D59C597,0xFEA749A1BEBE,0x3FA59CD4D3A20BE5,0x3F7940B13BC95DCF,0x57CEB2C2C707EBC3,0xE05274FB0DEC5DA0,0x29631725544A4AB,0x428BD889919D6302,0x1D0034A3C6B84,0xCCA1D858CBB6FA47,0x7751FCE30E1AE52E,0xD89C41368F7AE249,0x8A18F5102B3F8973,0x495D9933D4A3B21F,0x8F69A3DBAAA812B9,0x11ABA3F9C8E83,0xCC3131F049B00200,0x13203AC9FD0A2A49,0xA965D73CF9AB76,0x4B24792423172FF4,0x6044536491E5697F,0x453CE5EE76C6B556,0xC517FE0CFE0B,0x869B5492E1B118E5,0x5D99A5B386CD6ECB,0xEA235AB4F9D1CD49,0xBD9C647630654E64,0x2A819F227BC4B45A,0xD62B1FCCF76A993,0xF431B425427,0xE9A31DA13F341C2B,0x6BD02B17CCEEBFF4,0x38491141F0AC3AD4,0xEAA0A80E882A310A,0xFB0C06E1FF3CDD56,0x9D8888B642EDB483,0x9C135300BA92,0x2185C02047751FCF,0x8DEBB18A5DB406FB,0x31425A6DAEB6C4B,0x222A4F80CBFFBDF2,0xF8E7D78540D23416,0x7CC495160E718BD0,0x207908B1E1912,0x694E045EB19A0962,0x993AAEF3FCA8F5B7,0xAADC8B7768C16F3C,0xC35EAD601E0D10B2,0xF9E0CD8A5CAB3E47,0xC1D626AF17A8899A,0xE9F4A0418991,0xB647C7ABAAC8B49E,0x1DFE09978C7F6D97,0x52A3B1448F4A7977,0xD1F7385BD785F28F,0x2E16EEEB893281A1,0x898EBDEE1BDF9936,0x1317BFB10392B,0x3C9137905622C07D,0x674C10210C28E8EB,0x39DC7D3C840A473B,0xCDDE0C65F1FD153C,0xF0B604009FD5F540,0xF12F7967DCD8B32E,0xE1E0B26AAA54,0xE3971738F3F543BD,0xF66B58A78C2F84CE,0xCEA27500018FB501,0x96AF6DC88BE595A3,0x1F665AD45EAAA598,0x2E442426F22C3465,0xF6447D10E692,0xCFAC850B88F2EE6B,0xE5149943A94B6D86,0x5F161FCAED3D68F3,0xA4B8437A2C59B019,0x9CD5076385FF36D5,0x382AB02F21F32D9E,0x105F294B8B1EB,0xE3716E262F25B84B,0x610AE02D0437B63,0x6A9F4549668B7682,0x751E9D1723AD8D06,0x9ECF9442C1C3F2F6,0xCE6B82C9A1784B7B,0x1071D2991F0FD,0xFC52C76FF2E37753,0x7BB3DE9C57FAF10F,0xA72DE46DA2DC8075,0xBE5C789D014000DF,0x774A978B9F2A13B1,0x2A69DA1DF98D485B,0x137F01B0AB8E0,0x8B0CF4D812C1D7E0,0xBC658AD5D8356473,0x1C6D7E562456569C,0x128989A252C3956B,0xD17ED1255B113B10,0x80F7C0186D6B765C,0x1FDB866B1BAFB,0xA190593B4EE9182A,0x245A33F8A1E3D97E,0x209CAE71A7424CAB,0xCE8F72220E4CB52,0xB88859A309648D3F,0xBE0CFAE14975507F,0x1E46936B19009,0xBDCBA527CE6B8413,0x5B1D23B9AD0F44F4,0xCB54A742841405C7,0x98006C059320B118,0x75D8491111CDEBDA,0x4FEA0F57E5D7D97E,0x11FD00B648E54,0xE7D4809079565757,0x23DC253E0911A698,0x1437E22AB6A3E7B1,0x6DF1C259B20C7637,0x4C03904AE34B3D69,0xD7DD4C03E321DBB5,0xF16BF181C8B1,0xC884A57FBE9ADFC8,0xCB5874F3418874A1,0xFA5FF16489EA9FD7,0xE9AB315040CD40D9,0x99C3D6487099E9F3,0xBD50404CA678DC56,0x4C7BC778D26D,0x74DD972AF3E6DC8E,0x4E0D4E0DD0900E82,0xA70C400743BA5C82,0xE3A398CD794ECC06,0xC01D5C2BBAB7CCCC,0x41CAB3D1CFC1F17F,0x1BF9D20BE8DA8,0x966696094743BB9C,0xDD486486099A8AAD,0x742BAE9E6BFCC5B7,0x39A3FE82E19F8A52,0xE59212BC67F163C8,0x6A33DA5236B841A6,0x5B455D4C57B4,0xE546E4477EB9878A,0x43A8D6B631BA776B,0x6ED8BC769375F0C9,0x1C99454B5B7122CD,0xFC17BAE82A002161,0x8C11BFEEAF73B2A7,0x180A993129DA6,0xAA4E3F8B8AB80F4B,0x445DD58738644348,0xDF93E2F1164760CE,0x57F5D539862E5210,0xDFCC200704A3E4EB,0x1DAD8D019BEABECC,0x1B7A804C3F2E0,0x8C18CC1D904DC445,0x3FE21169F37F3EC8,0x711F1E5E5E303A1F,0x203D5F13ED96EF57,0x20396CFDAD6DF82B,0xFE9BE047EE9001D6,0x212AB3D848CB9,0x5F3768C526A87085,0xF8F198E509C57474,0xB5F0A57E9D0D7FAA,0x3C6096ACE60E7EA4,0xEB1B73D4296DD246,0xBC025A0DEFA13AA2,0x6CF96BBC5B1E,0x1C8D08F6C98071FB,0x34DEFCCBD835BDD9,0x9B9A90D4754B45B1,0x234552EE248E795,0x30F2E4BCDF275A57,0x6A8CF82FDFAFEBD9,0x2035CB20FC178,0x3F3E18C03B26E0E7,0xF4F2E77BD78633B3,0x181DF7C49426E87A,0xFCE5135CA0289A9A,0x87427B8C76ECF4AB,0xF28DB6926AB2A3E7,0xC5CD6976B23B,0xD8118E6ACE18D4B0,0xDAC080643AB50196,0x615E30AFBC4BC2C,0xB5F2A8D589020481,0x142B9595BF689B17,0x4607C67F1F3A1B8D,0x16C6440B4AB39,0xCDBBAE2AA8831CA9,0xABE1F0C11ED47671,0xA337F8DD2D900D07,0xABFBB98B1D69D7D0,0x35D23945C5EFBD59,0xA2714EA242E013DD,0x1FA68D419A07C,0xD622CD7D81CD8101,0xFC1EB8478FCA0425,0x3E782B95286051F5,0x434CDCD147C0D343,0x95622D3291CDF047,0x72EE497F1E713EA0,0x215B42A1F8C66,0x20838C756C04A60B,0xD8C7906FAD7278A7,0x2DEE65F4267D5F21,0x5DFF0B7EC706E682,0x1BCB77F2E21D7CA1,0xEEA67E8C8FFF675C,0x85445654D95A,0x708C97B35DC0B30E,0x856F27F0C7B02698,0xED9E9993E679A27C,0x56D39ABE70BE6DC2,0x8D5038314B6A6E6,0x78A652E0E4ACB558,0xBEBD767678D3,0x4BB952A9DCD0B41B,0x1B2DFC9E144DECEC,0xD6EA616D4D89F413,0x4C14FE95B41CE7F2,0xFA237EB1783714CF,0xC73AFAD601ED1FFB,0x205652ABE3E04,0x7EF73A2A48665575,0xDB30EBC666E2E20E,0xF54439761F83D8FC,0x9FEC87C2BE559840,0x3AAE150F120EAEE3,0x1309C3708A93A125,0x1633B491C46DF,0x1359FBD9486B3D15,0xEFC5C9516810418,0x9AEF3CD0BE31BF0B,0x333B9E1F0E2F771E,0xF3F506802AAE748A,0x706DA13191D6C76D,0x636D58C25E2,0x41CDFBBE8A65F209,0x574D1449657D21A9,0x81C59647116457B0,0x110E38CD7E6386BF,0xFF8E0D93125AF60B,0xAE30186BF3D1FA32,0x8CD500951EB1,0xB32936A2BF9BE1AD,0xE5D1716EE811D7C5,0xAB15964DABB82F2E,0x52D2F98C449059BD,0x242EC45FD44ABECD,0xA0AFE7520548DAF8,0x165AFA3BE7B7D,0xD3776F1D0209FE64,0x2B925D03A9482AB7,0x4662D2F42BB1812E,0xBAA5E56593AB3055,0x7545D79D2D7C473B,0xE98AA8D2892D582A,0x15333406F9F87,0xF7E7C2613FCE2B49,0xE9F1B696F3584C98,0xFFB295E75AAFBBF5,0xD4E6ED907BBE9A0F,0xF9A0DD71ADDA4D2C,0x1BD249C63E36AF6C,0x19EEA131BE37C,0x8480BCB226B6FFE,0x63DDB72DCFC65ED,0x6BCC5AD62D988E1D,0x2FEBF0F62AA509AE,0x2C15433EAA5F6972,0x951AE6A141062506,0xE231460E90F3,0x2E0856A52B22486D,0xC2D4B1072B186F9E,0x2A3EF5DCA9E33E56,0x53372BB1C0AD7BCC,0x5BDB230FF5311D7A,0xE6D6CDF5F0C56372,0x220ADF7DA2E5F,0x40AB3A624F7DACFD,0x6DB27BDF03B178BF,0x2AAA0792EFA4801B,0x4583CF951804F804,0xBAFB1F415C38B671,0xD53A4D262C5547C1,0xE5661DFE34CA,0x563391139C76871A,0xD2586D57ABB8CB25,0xFB963C05513A78FC,0x7326072AD1EE1836,0xA131647BFDADD34B,0x9670E7DFEB921A9D,0x1D9D7AA5A931,0x10B9397CB85B31A2,0xD8A4D7CB51F3099E,0x83B1874B75CC4D91,0x2A1CF139B441D72,0x9F0CC6517F8EFA71,0xAB6F6185C7264E9C,0xF1CC75A08E6D,0xEBBE92ACDF3E7833,0x9B3378D13DF17792,0xD7AA0C8943EE19AA,0xDED5787631BA9C7D,0x7026083FE5E3276D,0xD6217768EBE676EA,0x1A6C299D76D85,0xE065EC83C74E16C,0x2F41AFDF4E22CC43,0x43DEDB3658BB85C8,0x5EE1E9C8644A8E5E,0x217C7917C7624860,0xCB6A7F1A786BDBAD,0x701DF96AE336,0x75FD2ABF6F932376,0xA22F061CA117762A,0xEB9DFA4FB50F0BD9,0xC5008D646FC03D3E,0xA9EE540A32ABCF78,0xC2BAAAE47BB1FC52,0x377002CA63D7,0xC160A75F28BA4EBD,0x3774CB7265DAB49B,0x6F80D514AFE8372,0xE8266B085164CC14,0x743F1A43F381492E,0xB78B2A176BAF8D34,0x1BD9DA7F0191,0xF74D083A33C05210,0x5E735A23499B2D1E,0xA7A3093FB20EEC4A,0xD9EECB2669E367C1,0xCC257E05DAFBE0FC,0xB48E2F6643E60440,0xB517FE161728,0x5352D687D2510AB7,0x300AD3D04C8E512F,0x8FA84C8E19C739A9,0xEE94FED2A789848D,0xA2A9A351FD54B74B,0x6F41241AEFB06501,0x20EEEE38ABF8D,0x85F04CC4C3DDAFFE,0xE779E846BA97030,0xEF30088BBD7B736F,0xF61C162727C3A0D,0xD5C4F2BBB32502F8,0x47A51EB186423207,0x13CB6DC7A4A92,0xA9990B1F8E785ECB,0xF417D9B72F52BD63,0x6F52309FCA8F7FFA,0xA07C760077BB010F,0xD0AE526AB7E4CB8A,0x4B69624C4046813A,0xCCA19F8A33DE,0x6FEF289703DB3567,0x2E32B7D31DC712BD,0xF334BBF7FB47AC30,0x3E5A2A7CA4175191,0xC748BFC1E8A65069,0x1B2F4D376F84070,0x1780FE5F39E8E,0x9F364488A1A492C1,0x2AC455DD2E82A729,0xDA71DCF3B9FEC76C,0x147586E8DD870127,0x6FD5F5BC988AD56B,0x211DAA57A535A155,0x35EB2666B71C,0xEAD4B628B1EABDE0,0x2A524207E671E5F0,0xA79D67AD76873C3B,0xAEC934BD282DB79A,0x5217D197504F56D8,0x8E9222DC1AB2BF90,0x195F02B111320,0x6B1D3DF21FEE0632,0xB9E857421D11D0A7,0x5D713FCCF68FF85A,0x793D265E2A8C8E1,0x8FB7E7F11B8E01E9,0x9F6E444C6D276AA7,0x17EFE9E8649E8,0xF0BA71AAE2B489A4,0x21C1EB478277DAAB,0xE34C90CBF6069D96,0x4BF72DBE0DCFE7B0,0x4F03208BE71DF98D,0x4DC356E22E98D230,0x1CD4425C8F592,0x17A12AB328661F1C,0x7524C1B2FE8E041B,0xE5DF0C1E4A03A46E,0xE203536BB31E7F12,0x4F7D0C6F6762CEBF,0xB02DFFF2480A0915,0x16E4E24E1126A,0x3E0AB6ED4DBB93CF,0xDF215197477509C7,0x1BD87127AA60EAE9,0x71632196C386A436,0x679800CD6928F3C,0xDF0BD050AB99014E,0x55052B731B61,0x428FDABF22A4140C,0x72D7C5E0BFAE62FB,0x58AA6CDC7BB7259D,0x9BCB9CD069CC97D3,0x91E15CB7CBCA26D4,0xF3734067A644BDE7,0x1199F5FC7154D,0x8CADC685B14C9CB8,0xBBF6AA8E7C08D2C1,0xAC44559034D27E52,0x2961A3197B5356BD,0xFA7EBE75B306D670,0x4773929CE348DB78,0x13AB6F6565F0F,0x41C77BA1635BA62,0x66906ECBB79F1D69,0x2A2FAC37C37665E5,0x4C7424CBBCB37C59,0x90759CF374112CF6,0x68C03241FBDAAD94,0x11B413D0370D5,0xEA7E00851C1420C,0x92A26D03190ECF62,0x3D644DE05B33F8AD,0xA4B55631C7367EF4,0x8B874C7AD90FF68F,0x82B3AD9C85EEC970,0x51F160A5904C,0xA1CED6997E1E616D,0xF0194B855FC761AA,0xC01492180ED0C6D1,0x9034A0993FDD7A15,0x6A616F4BE6FD11F6,0x26C2942418A59A43,0x1DBE21746B966,0x3B38B844BC86F9B8,0x8EF644290757B5FD,0x69EC39FD1B796383,0xE7CFCC01776B69A5,0x5E39E55B235C568E,0x23BFA891A7EFE1A5,0x230DEEC7935B9,0xDCFF082AF0485B6D,0xA769FB9A95204517,0x54CD3B2A0279E406,0xE91203B401E9BC35,0x2A9913FE5C749ACD,0xDBEC6C62DE1581BD,0x6D5845A1880C,0x742C,0x0,0x0,0xB90FF404FC000000,0xD801A4FB559FACD4,0xE93254545F77410C,0xECEEA7BD2EDA,0x668D97A1773C38AC,0x897D73488670D91D,0x80F4C72FCB871CFF,0xFB73B23FE612E8EA,0x49F239AD9D934FA6,0xA632CC678DD7E2FC,0x1F37432BB4B4A,0x3B38B844BC861160,0x8EF644290757B5FD,0x69EC39FD1B796383,0x75AFE3F77F6B69A5,0xAE369B64781CFCE5,0x515AFFE8E9015F8B,0x57019CFED803,0x2B5D661005106DFB,0xD22CC3335117F5B0,0xEC3C1E1DD7DDFFCA,0xBC17F58C1706A6CA,0xA57A5D8602F5C220,0xB93C02ADCC5017A5,0x1E2628C83347C,0x0,0x0,0x0,0x0,0x0,0x0,0x0
			};			

		#endif				
	#endif	
#endif

// This is for \ell=3 case. Note: Two different cases must be handled: w divides e, and not.
#if defined(COMPRESSED_TABLES)
	#if defined(ELL3_FULL_SIGNED)
		#if (W_3 == 3)
/* OQS note: unused
			static const uint64_t *ph3_T = {0};
*/
			static const uint64_t ph3_T1[DLEN_3*(ELL3_W>>1)*2*NWORDS64_FIELD] = {
				0x37ED7978F7323982,0x4936C76D51F21F5,0x16050C18A973EE88,0x72F7BF23AB9375D3,0x47C989C9C28BA7B6,0xDF611D74933C983D,0x1BC0CBF2A04F4,0x8A0EAC4B85F5303,0xB686A96CB86BBC16,0x8F36EBD4A443E09C,0x948222B2B2D156F6,0x8A66ACFAE2CCD48B,0x98DEEDA161F27B50,0x103F846A96DB5,0x694F92DFDA192A6A,0x53DB650F045FBB94,0x914D7DD5D24B8A7D,0x332A4681B4537855,0x77B22ED3C97AA0A8,0x29F51E43F01EEBF0,0x2BE99CC2EA3F,0x50AADD31483272BD,0xEF71AFF18F5CD2A7,0x2718B25973322EE9,0x4AA70233D7135E34,0x18E840E9D3D8137D,0xCC10E80D37099E81,0x611A286EE0D1,0xA9354A35B269FF9F,0x6FD04B074DF96AF6,0xC3106A5031DDC3E7,0x96D97DB085DE52CB,0x52E69CD3E0656B33,0x413B47B6FD5C630A,0x99B4D7906FA4,0x4773AACD7C006827,0x6E0D9C18E227CEB2,0x970AA29DDA4FC705,0x7DB2A73DC99EB,0x503DA274C8A57B9C,0x456519435726C6BD,0x1269589CECCC9,0xF03E85B4AA764A,0xD6A2D3F58CFCB874,0x92DA0409FD01266B,0xADCC6564A5D4F6F7,0xC02F15A74EC243E8,0x7730B32C01DE2F53,0x7E846A2516F7,0xE133F07C2ED3E1D8,0xB4326C2931139F04,0x6A51505413E95C84,0x736ACF58BEEF524F,0x538B2008AF8272BD,0x4BA606901671E978,0x177FF06B895C4,0xFDC3EBCDD657CD6E,0xA247E9F561464943,0x3C1009CB51346175,0xF002A2B6FA005AEF,0xEDDF64F46D2A177C,0x83518EC200172743,0xCCB04068DD44,0x45C5EF8ED846E9A1,0xC594E2935A404A15,0xF1FA86D9065998F9,0x3177F334A98AB972,0x683137B06187B6C5,0x763E818699AF9A82,0x25CDBC8D6DE3,0x27AE2A62883696B6,0x591D09EF41B582E3,0xAA5B4530E1582194,0x99CCB2BCAD4E2F47,0xAF7B2BBC1BFAEF2A,0xD2D3BCB820C00A6B,0x184CF997B843D,0xD149B9FF09F549CC,0xE7CFAEAB0F338F18,0x9EA6958AE87055F5,0x408E4C22C8263BC8,0x17E57438EEDDC7DB,0xD58BB12D4423F92F,0x1D7D0C433AD30,0xC7DCBE1AEC6D7C2D,0x8F6FB8EDACEED8A3,0xA92B7BA065B357F1,0xF6A92D93E4F369B4,0xD0B517306D4B0BAF,0xF53ABA5B52BEFB27,0xC56412AA646B,0x4B826A4831175010,0x67D08097812EF865,0x1950DA79789DA00D,0xED0B99011BB1215C,0x5E5B32AC7AE8C701,0x561D7DE509792047,0x1C21010F2BCCF,0xF057F3BC88DF9B21,0x8B8EE060C16EE48F,0x2FDAC5C9B23B3ECB,0x6AAEE97E18D02869,0x5EF58C8866A37082,0x9D42D0D7D776AAF7,0x9106AA9E5BF6,0x7F7F72CA5B66613B,0x49FF121C8796E2,0x446921D8A9193C4D,0xC46710A67EF0094C,0xEFEA22E111B76E16,0x2132DDE3208096C8,0x158537695C848,0xC9B7E0C49271E5E6,0x9232D78410486561,0x15CB972066F892E8,0xA221C57C5BC43FDF,0x3CD91966B44E4005,0x4313785071F6AAD4,0xCC4419E3E98C,0x83EF71190034677F,0xB73C0DA961FBF58C,0x2D2FA4D06846EF77,0x249169E3A3B98786,0xBC1076683BEB81A4,0x45D0E9EB5D1324FE,0x627AFD457D2E,0xA2D211FD38E81A4E,0xE74DF34D9BDAAD83,0xFA9276E47ABCF316,0xE306ACDDA72C24FA,0x5B68BF41E7F5BF84,0x322D98FD53020106,0x8902F84E1521,0x8C44089102E680E8,0x4CE3AD9309B0D800,0xE2AFBFE63C21F2F6,0x51D7493A7E0E5F24,0x5F4EBD5D1CCA229E,0x6B49826312FF74D8,0x91AB97F84071,0xC363D921AA5766B4,0x5F78F0E41B0F5359,0x4A43F68C67C5BCE2,0xFEAC89F6BF98C204,0x9D7AC06DBF4FC7B4,0x26989B3B2D44C774,0x1B6C805550CB5,0xD253F7739CCDB075,0x8318538FAD5D33A2,0xDBF3CCADFDE3674E,0xC6773568F794E12D,0xD6A71F2C69C17A5B,0x53FD7701967BFE2A,0x5CB62ADF7B66,0xF1876400E4AEBCEC,0xDB7D4068365A6399,0x8D2B3C07E3AA1D93,0xB763B70480DB937E,0x6B209F9E89C1C38E,0x99F233803462481C,0x1703585A8F941,0x53651286AF21B7BD,0xB5DC8DE712CB9D2B,0x8C8653F83462375E,0xA13993C7C4E915F2,0x964055D5E5966731,0x357C020389033773,0xCC534E50DED,0xFF72F3C9C8A650F0,0xCE49F4C2868EF761,0x458103FE62D5D7EC,0x5EDF0259B91D5E5F,0x1D08EB6816BD60E5,0xA946F4C5D79C5C1B,0x223E0CC250BC,0x2B51A049820E158D,0xB0AE40314ADDBCED,0x7ADA8CE98928AE66,0x54CB91E1C6E32237,0xC5BA6BBE489D3E92,0xC61076FBB6204480,0x16725A031428C,0x19CC0BB9E81D0F2D,0xC74613BF54DB4817,0x372D926DB39C3CDF,0x2CE188DAAE451706,0x2E9D1F86190D2D03,0x7AF6FEBA83E7E98,0x1239E39C886C9,0x18016CB681973207,0xFF2EEE394C8FA492,0x7015BF66C6DA1CB8,0x682A804E6C62CDBA,0x9ADB8BA7AC3527D2,0x5CD6BD2C7F57E672,0x1F7FCBBEAF368,0xC9615E77B9A331CD,0x796546F9415693B6,0x49E2874740F92975,0xEEF96C473880186C,0x6BED96B40F3F4146,0x6F0183BCE2AC1B17,0x22050F405E829,0x65707E635ECB4E9D,0x8B6CB4F957CB894B,0xA548A3A39C9F6A88,0x4FBE409960FB15CB,0x2122ACF1EF2316A0,0xF5A6FB61350D3FD4,0x133795A80A6D8,0x4CB72711CA850294,0xC211F90452DE238A,0x80E6D14309CB28FA,0xA30C6A97C9FF1E8A,0x2430A4CF8D66F69C,0x80AFE7DB642FD042,0x6868939979C,0x459FD7E2EF627373,0xBCE705FA699DA4ED,0x361BA49E03059B10,0xBA3AE86694E7AFC9,0xAECCF50BB95E2BD4,0xBD9B1FD6D6AFC93C,0x144E505989609,0xE8C5A33992BDE5A,0x1E43F6F8ADC4687B,0xCEB1842C9087709B,0xEDB68CD7E5AD47E,0x6EE1502AB55169CE,0xA8AB5BF4CF196E9F,0xB16B329F570B,0xBBD392B020E491FD,0x94E0A3821045F058,0x4507C63B48F72909,0x7F06CB6D6E81C97D,0x8F1BC7547C099C62,0xA9EE5E02C3A26F1B,0x17005DFFDF107,0xDDE3DB271B9A4338,0xA5FD982FD0B4AF0,0xE31686BEC642E12E,0x7A80DB5D4B32EC9,0xC5151C20FB123F75,0x8252EA8E4768B901,0x1ACFDC9EBFDAC,0xF9F03DC5A7959190,0xE5758D844E7AE926,0x8407306CBE092609,0xB6FF0103DFC5E9E9,0xA4B827DEE62D3AF0,0xC358FDC46FE676A2,0x31ACEF775F8B,0x3AEE9AFA0A75690F,0xF727CFAF23834AF1,0x4D60704B124B5887,0x8F2C54DCA0B62387,0xC199B08D1BF36133,0xC8497DF8D26F0A3A,0x6EDF8A6FF561,0xE3DC50CC98B2DB44,0xEE6D280D20D7AF8A,0xAB02DF10F3EB2C6,0xF45C684A268969C6,0x64609DE2083F62A7,0xD5F322894E61F36F,0x163A65A090CEF,0xBE1D168EAF5630DA,0xDBE8169263A15123,0xB93503EF4F1EA8C3,0x52200F4FD9684DFE,0x133D324EF92F3352,0xBA8FCD8662861855,0x1F6D78A773767,0x9C82B7C999E240A4,0x1220E79AAC60F412,0x2F696A5BE3B50E85,0xB35D254F7941B586,0xC98A7E825C600FF4,0x2984553C9C9C35E7,0x70D98ABB8FF9,0x3BF03B0B965023B1,0x19516B395D8DA1F2,0x7464C1F243A64333,0x42B77F8AA757596B,0xF4DEA8A5A296386F,0x907D14439E00BC3A,0x1FE7146A8996B,0xD7A71267F37A1505,0x2AF0E82C15DF8276,0x52B00A1A190E239A,0xF6046122222A2E85,0xABCF14323CBA69CB,0x9D196AF9DC177905,0x50E3901669FC,0xCF6D9E37ACA7FC41,0xE3BB6F4674DA80C7,0xDAAC5E9A0A44C2ED,0x7D5307AB1434786E,0x622342575645CD0F,0xE28C64BD43BFED4,0xB39968FC5CFB,0x938BBBEF5B9D2EDC,0x364AD80C20AFC700,0xEE2A7B39C1AC7E9A,0x66A182254E550751,0xD29EA3C001982C8B,0xBB51E4F503A3C906,0x1958524411AFC,0xE8428B7A729110B9,0x2E8097378786016D,0x2F0328778BBE0ADE,0xB14FC0435B4DB89A,0x772113481A7CC660,0x986D7F22ACF37EF6,0x11737B1BC4D2D,0x355E1DF04317F5A6,0x5CD277CA467DD1BF,0x4F6EC2BDBFDB407F,0x15CCF1A664666C48,0x5CCBA353639CB77C,0x91ACF89E2581873F,0x4580EE384271,0x41E94A2CCE86F736,0x4C8C15CF21972171,0xE8E30CEEA70D832F,0x3568E1E0B9408B85,0x6137CC06ECFA3C39,0xF1874FAC5763CA25,0x6B28FE92E335,0x4079F2E96D4C37A,0xADEFBF0D4E81AA76,0x73C93F65C7AC3C1A,0xC413FEA1F936A2DC,0x898653899E3635C6,0x86CFE3C4D9012325,0x52C45C688B16,0xA463DE01D72C1A70,0x29A864C11EBC266,0xFDF2F62E382A27C8,0xCADBE05B2D3FC1D2,0x7BCADE04573915AD,0x9B4C5206CD9A7A56,0x13CAF24B79DA6,0x7A4F90F0CDF4AA5C,0x60213E8E82C6A788,0x4C6A474E2FC805FC,0xB976FD938065427E,0x4EF436FF608A0849,0x51833803AEA9329,0x654F30F79524,0x96EFB0DF2DFE2C8E,0x3A06FBB27B9EF0A5,0x14E37AC90144C0C7,0x464DD0DA0ABBD703,0xAE734F2423D4926A,0x1422C19B36EA6180,0x1098DC36CC91D,0x3700F973B254499A,0xBB004B4BF0D3B116,0x90B567FE1A266551,0xA2FBF99C37634018,0xF758C30D975543F6,0xD0D48AC9501A4C71,0xB9AD678DBDD9,0x99718695CF56D1F3,0xFA0CF4A65E2853B,0x6BFC00CE66341DB5,0x9A1FD3D8BDE2168E,0xA66312AB6D1D1429,0xC4F6C545D87F3AB7,0x684ED280FF51,0xBB4C6122596AAFB4,0x9975728556A3FC4,0xF737FEBEFB08A1B9,0xE2179CF8F34CA308,0x1F733D8647D99B33,0x8F161334971B8D14,0x302DCAACE7F5,0xE4E770D3A1998E55,0xE50F7A43CB8F3B7,0xCC0EE4F49DEE0608,0x696621C1C702829,0xEAAF7372E46AD1B1,0x6A697C9793AE126C,0xDCD786C808DD,0x62BC51904C27EC96,0x98659E35ED3CA154,0xCBEE68DBA5AEF1A9,0x84A6828DBE8486AD,0xB62C8CEBDA58823E,0xC0505E7F1EF6C439,0x3E7290CF7FA3,0x259120C9EABD2A04,0x296350D8CC35E881,0x86F68443CA640C6F,0xA973BF6907A3D5D2,0x14037612F335A0E5,0xD8D2268298E642DF,0xD8266EEAE590,0x5249141997855FB9,0x9AC7EE090E433A1,0xED35F7C7E9A6E973,0x436D4670CC885A0C,0xEAE89063A007DE81,0x60485A975654EE3E,0x8A62063C2F5C,0x20D3CA8D414DAB5E,0xF0258C41FC6BBB86,0x43433C374F87A54E,0x4774088F8B175EE5,0x213EA6CCF17658C1,0x227216743DCDE7C,0x155CE7D303C44,0xFD853D5D19CF4854,0x392508A64BDD3A54,0x894B13B458027F1D,0x8CA48D136C13B85B,0x5731520BD0C4AA8A,0x52CAA13C82DBE553,0x68EB8E1B6529,0xF2CD11D60502840C,0x2F2239288190E77E,0xC54FB6F43DDEE33,0x8432E6AB6C48DD03,0xBC1C5C175CC8B07C,0xBA9377935A979713,0xAD160C55FE9E,0xE5D9A636DB5A8277,0x3B299FD3A9E82550,0x3F160DC2C89EC1F3,0x423C8BE2392F9BC5,0x3367279C42919FF,0x4ADF68450E4ACBCC,0x12346785644F9,0xD68C754289408DC,0x51DAAB92F41618D0,0x64D36752180AB1C3,0x8B5ED8FBD1A334CF,0x40F949620608B5AD,0x58C02DDCF3E3CCD4,0x16309E490B72,0xB9174D2B0DA5B62E,0x29BF6AA15EF6BA92,0x5547D71A78FF4258,0xDABA5F729726FC62,0xD48DF9C370793FAB,0x244C37BE2D9550A8,0x1608063A660F3,0x2454EB43EAEDD43F,0x3C7F0615382A914D,0x1D3B977EE568FE40,0x2433C72B68D09881,0x5D90C300127B5FF8,0x35B5BFA968DFA206,0x16B62FBFB5093,0x1C28CD2BEBC85AAD,0x1EA92D8C711B11F9,0xF1441C840CFEDDDA,0x91F3632F3411D040,0x82CDC128C068940E,0x3ED528A97689B92,0x103D7C12C5935,0x9F4122BEA9516475,0x38FDEC71E35AC118,0x5637F04F82B9D34A,0x618C456E7AFF7415,0xD228B1403859497A,0x972F88F825FFFF45,0x3599BD2EBDE7,0x451721B7E01921AF,0x1324771B92077467,0x7987B70AFA46143A,0x207658160847F295,0x45AAE41A8EF23AB5,0xA3404D8312AAC63A,0x22DE6B7333E2B,0x8ADDEE86131054D7,0xE97A6178E6D79720,0x8EA7ABCFD67A1D62,0x2D22A45E719681C0,0x72AFBB8AE11C6C0A,0x3CA2C1D95465587A,0x1E3C708CF09EF,0xA2A1EAEF86053BA1,0xFD69E28F3A2E9CA8,0xFD6031FC1739E60C,0xBEDF49AE806DE468,0x506A2982151B920C,0x5BC815D2DD99C568,0x2DB77C0FCB84,0x670B253E6A9CAC62,0xC65C04F49DCDFBF7,0xBC23E4CB94E6E9A,0xA9CB01141278AAC9,0x1A523BEEF1D3AD67,0x47700A8558454256,0xFF2A84448B0,0x5AAA8BD39B7CB73E,0xDDEBED4AE2361213,0x30B66468B222DC24,0x605D92C7BC8BB2DB,0x8835E0F870F0AFFA,0x2A778A5FA977EF15,0x20050B6EE9C8,0x25CEED71C603C446,0x98486A1D16B12A09,0xECBC943445FAE29D,0x819990A109E73276,0x4EF0DE8C559A5DF3,0x3C7F22A2234AA455,0x496F3FC1AA49,0x8927A98470215420,0x6EC8A85311A84229,0x247F01F36031D3BF,0x354AE03DDB636697,0xF2D8C34BA94295B7,0xBCC65F1B5CC8D5A4,0x94F581B780BF,0x45DD1EF3E0779BEE,0xBA84378054966FD9,0x81D263C16CCECBC9,0xA77B15A2F2171E68,0xC46434F71B72F2E7,0x77AB7265E70D339E,0x56978FE3DA58,0x5179D2522980E2E9,0x2391C1952821720A,0x37BBA3B0E6699D9F,0x3D4937335A8E7160,0xE3C4F8CDDD171F4B,0xE1F519FC408928A4,0x597AEA1A835E,0x3E1E36EB2B476825,0xC0AD39C5CA941EF4,0xDC84844D2A04A14F,0x1630CA91B9EA7F6E,0xA408CD429FD64C5E,0xAAF1E240CA528C20,0xD81C2DEACC48,0x91E1900B00E86936,0x7BE0E846D5CE97FF,0x721E690F017839F,0x3B4039E977C30C12,0xA3971F3A83BA6BC5,0xDF15512A4D70A592,0x19E62F93D4F22,0xE43EC119B8E89DFB,0x4DEB15DD1B9A0AB2,0x64829D8F572AD605,0x940936CCC608F92C,0x51F47E94B0861D24,0x492F7EBEAD3CCE12,0x1FC3876F37613,0x25C4A6F996DC1323,0x680A8A80AB9132FE,0x8E3C791DEB467E64,0x428699677462FC06,0x2D843757B703FADA,0x51B28D04B3643CFD,0x1850339234F22,0x17A3112EF9C267C8,0x5BEF4AEC399A35F2,0xD9D791D66F35CD4B,0x4BFE5DA02ADD348B,0xC235D0C6A8BAB24B,0xFF93B782B6B7B3FA,0x156430C23B60D,0xB6C4FCDE05363EFB,0x332D1402A997931B,0x580795E607F457DB,0x37483F148577486A,0x9EBCBF2A142B5717,0x1FB805332A8E4DA7,0x1DAAE25FF7308,0xD1D273C0403DE823,0x781B73D97BB67E1,0x51F4D81C1E0AC129,0x9F339FE1A254FC7,0xB2996EE412AC0B92,0x1760A8A6E2FE0990,0xFBFC00C9B60D,0x4C91E4B4B33A0C23,0xFC39A597CC1E04FD,0xA7F7658285DB1A43,0x2A2B5772E3B3162D,0x1E4A550641B5E3FE,0x33C3B321B9B99780,0x17D6DE7225983,0xECA480ACB43BCC1E,0xDD01AD6A9ED2785,0x92828D3EEABD6021,0x11DB3994B9A081A5,0x3984BA2D4AC653DF,0x44335D2844147169,0x9B7D558EC564,0x1F838D429EA62CC4,0x4A0D87DEEFC95AE6,0x4F8B5D39F65C7140,0x6AF51F2CF4EB68FB,0x4798FE25308722C4,0x3677CEA9F214ADF7,0x4A559E293A03,0x97E9229297FA5A1A,0x87CE5DF0DA35988F,0xD09A7A325EF512CD,0x97FB235B1CC676D,0x65DD05A89495B0BE,0x20677EF46C7104B2,0x24B8A8A7B794,0x3E8B19B1784E40DC,0xC77EE84ED7CD2798,0x129A212EDE8B5FCF,0xDC03C11C78A5FC0,0xC0F0742520A5FFE2,0x80CE2428457EA6C6,0x1ECF7ECCD9512,0x20A948B07F395A6B,0x83D3D210248DCE7E,0x70179AB1B0C9818D,0x4381BCE5DD6D7DD0,0x520356837E06C429,0xC5BE680016032F02,0x10F187A80F430,0xEB8F79468A4C59DA,0xA18525DB1CF634E4,0x789C47A5ED8622FD,0xCBC952513A63262E,0x5AB4C39DF2DB191B,0x94817CCD0F5F6D2B,0x65826F8F95B5,0xD42FA3AAC3418712,0x124B5D5AC5D6531E,0x3619D10136E3E4C8,0x665ED1D7B581123F,0xCB56E76A6C76094,0x42757942896F00BA,0xEE77BD90E0D4,0x89B316FBCF89B2F4,0x551950A7C90336BE,0x73F3639F5F84540A,0x383A92452C6FA028,0x7BA3ECBD34060959,0x9678C50D21B17FE,0x5717FC46BEE,0x6BBB9E10FBEB1058,0x8646FE3C8B810DE8,0x40FE97C5432F17E7,0x9CCA279822B33A8F,0x2E7005316443BD05,0x4DF6C576A4D283B1,0x2258D129F20CF,0x3ED4B4E97F8EEFD,0xD200955A58549DE8,0xF383C4814573B1E1,0xE035730952374D99,0xE69C22AE2E87D840,0xA998D0A4F7842ED3,0x1ADBECAA65361,0x1E90562B19A2C767,0x90916D779EC73093,0xBDC75EFA2077D0A,0xDBB814AC43FA58B1,0x25C788853A25949F,0x5696D094A05F13B6,0x23EE47B42381,0x2896837F061E5C95,0x18E00DFE453167BA,0x2187CAEB2813E16E,0x2CD5DD9E3855305F,0x8B53B210903EC228,0x40FCC08C8E186CDB,0x196AB10027441,0xDC0A94B96A6B56B,0xD9D9057B0CE59143,0x3B488D345718AB36,0x2EB6C75F6022472B,0x58799F84F8270C65,0x396208EDA10B86A,0xF4A31E58A5D6,0xCFE883E4454B525B,0xB01B46BC9B92AEA,0x15C9F9B353D8A7DF,0x7CB934FB3F7E377F,0x5A624D1FBE9D50D,0x4AE48CE6866E6C11,0x9D77893330C7,0xC2229E88BA598531,0xFA7888F566F6AA32,0xF1D88C2022B21A54,0x1C992666343F1DA,0x320FFE3076C76244,0xEEC1C3215138A03D,0xAA6538747325,0x613DFED2CA10388C,0xC2CEB089A68F20AF,0x3CE79BC8488D3C97,0xDF5609301E603558,0x3EA599D90048596E,0x90ABB280F8BF1ED2,0x10A0C816EA3C4,0x7B76D2528746143F,0x91734621CD60A92C,0x9AE08AC37D292EDA,0x632B3469BDECD530,0x9347186D2B907B16,0x48B3607A705F3171,0x11E69A0D1E233,0xCFF102CAC6A9BC25,0xE7F88CFD6E795F59,0xBF66AA5A23854B27,0x4816668703F43242,0x20C296E0791828BF,0x2758CC11FA1C07BA,0x1681CE0B76110,0x597DC92287F3F03D,0xB851067F414A56DF,0x53DCE090BEE9E880,0xE817868EE914C688,0x137D938724BDC4C0,0x36488FC78115D859,0x1187641368CDC,0x57F41714259A0FD7,0x9AAA59713B7D7377,0x2243E12268D3FC69,0xC6EA6C147D363FFF,0xF43706D18618A2B6,0x2D980F2F037DA8DD,0x168B1B42BBC20,0x55E7DF5904F12E85,0x4819B3ABE47DA6F8,0xC8E940549D5B8CF5,0xDC83DDCCE72D9B52,0xB49A25AD17733066,0x5FBF9F9E3CC8A1F8,0x1A9D39A30116F,0x84D90559C8E97FC8,0x938FA91412A2D817,0xE2315CF1C1E69A96,0x69DFC18EEBB3511D,0xA409D353FD784C84,0x819AE43CB2377373,0x1DF9845A995E4,0xB606B23D1A86DB65,0x8D5539CDBC6A4B05,0x5725B9F80B025881,0x374D5B24F660E747,0x75868CD69C75EE8B,0xE7D8DE35FA168C4E,0x1AC4AA828122B,0x48CE050DF7D161B5,0xD5E1F6ACBA619A11,0x244E813E118FA869,0x28003E5D99A3A441,0x3B68B7D319069EA1,0x983688F4C9C84166,0x168125925D30E,0xB13947C39CF64B19,0x28674DEF15D919BA,0x5EDEAA1540F3E427,0xD6F262E411B3D89,0xBA7A2674093131FE,0x2BDE12EB794F83DB,0x118AD6EA8806,0x3CEF3CD3D46DD518,0x1EAE2B7C77E22AB4,0xD82F3F0CE40D71E,0x19FF46EE106BB60A,0x3097D41585C6E4AE,0x39E6556A40136136,0x1D1774685C42D,0x5C89902264D1CFF9,0xC98B69C47844418,0xC0635FD0E1A62316,0x95A8D4D2B332F943,0xC81D1D5859CD2276,0xD93A88F8F0690296,0x449124833F5E,0x3397CCC0C112AABD,0x8C6F06C424A734EE,0xA622809EC0EDF21E,0x552E40F79EC3F856,0x9FA096D2631D94B2,0xB272AF9DCB59F80F,0x17855976ACFA3,0x1D862A062339F457,0xC5D1B98E02E88616,0xF5459B8185444631,0x504EAD6371C25CF6,0xB29D04BC820695,0x5794C4FB95A7F311,0x82801BAC29F9,0x9D4F6C6F41845388,0x5E348455073CD766,0x6E21BCB1C8EA53E5,0x25D211AB4F104D00,0x7EA2248DBB70B2A1,0x89880CFCDD0A7E93,0x1646CBBA9AE48,0xD3DA92D4101A2AF7,0x4F1D8E3A1E1E7BC8,0x13D4142B1419779E,0x6E437CB6563AF5A1,0xB0568DE6A3EF073,0x8088ED7AE3DF1CA9,0x754BD3A02819,0x50E7764AC0F64C,0xFC3C9F27856DCD42,0xFC8E1436B052F2D7,0x6FE5E4A462BE5FCB,0x602A7EBD505C9D00,0xE9147A4887A5D929,0x68FD9755A3FE,0x617D377B1A9BBB33,0x1F85942061080213,0xE8F463C000B0DD2E,0xAF3356113AD75BEB,0xDAB7A3C7CEF996D2,0xDA81F226E1956921,0x195C1D56FEE23,0x82A7DD5BCC269CF0,0xCD80708AFAF58D3D,0x479AABE4B45B6561,0xA7BAE7550B9055C3,0x86ED14C80C55F460,0xB5B50238884E44D,0x164BB65DC3439,0xE0B05FB9DE4F451,0x9EDC42096981875A,0x3CF17A6D14332878,0xBF6DD99555893F63,0xAD7E442476DA6E00,0xBE8425BF5BD75214,0x16AF9FDB9F4E2,0xC3787CECDE613015,0x4C14CAB66A68A17,0x450D11B9FE410E37,0xD5D3ED15557BFC96,0xDCDE6EFE5D90481E,0x559B33FA0FBB3F9,0x58305F213789,0x89C1297040D3A474,0xFF71C4A765B4B4E7,0xCE64AA377C9CD1BF,0x639DBE419BE2C45F,0x4F98329F7FCA5FC8,0x665157168B2E4BC,0x210B727EBAA37,0x4E2EA413C071AC7,0xC9D6FC0B84D652CB,0xE7DF5DD3B5C472A5,0x17D11EB4637491BA,0xA95AA02F7E9B40EA,0xBCC421D0B684C06B,0x12A14E8F4040A,0xFC2DE4B969CE1150,0xD9C7FAD15E714C8D,0x19DE7663B89A7E46,0xBB462108A114ED6E,0x1D45364BCEE30A86,0xC3044F5F24B1CBA5,0x1C4C5F668A933,0x29965AB0B440852D,0x81411C622890FD31,0x1DA061B9880278C7,0xEFE99ACD902E1828,0xF2533DEF24CC6E17,0xF9BE61936AED9F4A,0x1EF2CAE490C7E,0xBB244C3DF0B84B59,0x8453C34FFF6E0DDA,0x90BAEBF94D38BDA8,0x1C5ED74AF26D103D,0x97F75C348B9CDCC4,0xA648A4AE830475C,0x5FB38D4F1B80,0xFBF59A236A6141DF,0x6F63ED2E63B0C79C,0x19E851A7899F601C,0x9AA728BD5EEBD222,0x61DDEFAB4B9E0B54,0xAA86F9B844C0443D,0x11211185AAAE1,0x971218D2939FB319,0xA5FDF0D7D36A2D58,0x917CC7C553F8E12B,0x1F1A460486DEDF97,0x2079D4BB740F9668,0xFCC675EE444D44E,0x1CAFD996F01C8,0x23632D87580CE030,0x2DBD4286178EB228,0x77B34FA5D1FE4719,0xFDA468D28A460BC4,0x86C0BBDE29AB9BF,0x6CC22E4CD2A2A44A,0x11C8160486014,0x3325F6FD96F566B2,0x48CC62C72370D834,0x205628177B556618,0xDDB990ACD89DE18,0x9F690C91970B4C8E,0x6FF323BA69EDBA0D,0x775C14E1E99B,0xD5C4E990EC943659,0xABEAA9368F1FBFA2,0x90A500ED99759515,0x773999243294A74B,0x338DEC509B073E66,0x37DE0EE14866D62,0xA5D875D8C502,0x610DC00581A673FF,0x7A2252C9C9C59EC1,0x68F4CAB28B0BDEA0,0xF09470EB9591CBFF,0xD613524F623913A1,0x392EAE8EE930C83A,0x2301ED4962761,0xDAC1D5074C7FAC11,0x1201094EA3567DBA,0x6035395B9EF45314,0x448EC3463D66D1BE,0xF87A2045402F682B,0xF1DA3881061A01A6,0x103C46E7BA7D4,0xFFB1C7745471413D,0x82814E0EA40A1DB2,0x6894EA66A68AA1AF,0x8208154BB53BC1FA,0x8F04BB227BBB6A2D,0x6A4F34802D71546B,0xCB6543ACEC72,0x3257407C69EAE7B8,0xCC142AF9C4F9ED2E,0x795AF9CF87455853,0x50B2ECCE460CC358,0xB4A3A4DD5B7116B,0x2A90E327E0A13F6D,0x105559901176,0x993B0EBEFB56A8C4,0xC08562C18402597C,0x591E067D5C6FB9DD,0xA28E4AE7A5E491F8,0x41F2D1732A9F9D8E,0x89567F6E064E8F56,0x160EDDB96DB62,0x95F031DEAB939ED3,0x68BDBB684EA98320,0x827CB6B6BF2C9C58,0xB7FC54DEBC89C71D,0xB51E046855C44A5B,0x938ECE19F1A2DAC6,0xBC7E6DFAF66A,0x6DAF1543650FE939,0x56E3F5B5A2FB1756,0x7EDE9FEAE3701221,0x11E76CD322EA40B5,0xA1654212EA5451B3,0x626666D3D8DE14C1,0x96CAAF3C17D8,0x615E38D241FD05FC,0x3C0062D34EB7DE11,0x7D2F5F0FB354BBAE,0x85E2C91EC41BCD3F,0x9952EC9522F453E0,0x3A72483C32D12963,0x973ECBC9FA6F,0x809EC00ECF39DE1B,0xFF5F3A35E04124AA,0xE968E96D29CD7AA0,0x491ABA6854EA6425,0x2BCE677F4F7D197E,0x14B2B9E382ABDB4F,0xC9AFE7DF6771,0xA45B92839DF3F1AD,0xC373F0DDA9F8B386,0x59112C7216E69A30,0xC5AFE58EA0734181,0x665E85979C5F0B59,0x7E856B620359BFFF,0x18BFCC3BCC553,0x47E98CB9F11B756,0x90E1F82E53F51078,0x2E428ED4D50F5C15,0x71B359D96627CA39,0x2C7A36D50507AF03,0x57C01F032A8B493B,0x167580C80140F,0x9C6989438DC6B8F1,0x3ED036CE665A34A0,0x96E1CA7F38265B72,0x4D25A8A4FE908242,0xE5DC97CD31A61CB8,0xBA1F39AF3C58DBA7,0x199895DB72E62,0x19F28BB8F8803F53,0x7E9231E9AB7FD718,0x45CB9ACC479BC93A,0xD3547EAE190152F0,0x86837DECBACC9A0F,0xEB6E221561EC393,0x1EF169CA564BE,0x6C32A6342FEFD3AC,0x2C2FF5277ABA1650,0xF12E3C56F074F51B,0xE55D6B3AF77E4335,0xEC227BB8310EB596,0x2C2DDBD0C0AAD614,0x202CD6DF23EA,0x18ED542129F57537,0x1A0915DC0B1022A,0xABF7DC242D16E6B2,0xEC4DD12DE1262D06,0x10B9483ADCFCAEDC,0x44824E123E29B7A9,0x8EE0734F08BF,0xCD09735185E3B3E3,0xE222A1F2B3E86541,0x6A847937235CCB2E,0x410080FA880548C3,0x97F0543C2C8ADEB,0xF39260F113B1ED70,0xA0C713636AE1,0xAC5DB3F8A9DDEC46,0x9244396D61E5957A,0xE0EE324364425AF8,0xBE71B718B4692F0B,0xDB0BCDCB37795CE9,0x1FF9B333707DA3FC,0xA507737ED80,0xBC2F4DEF28503752,0xDBCB89B2D7CF897,0x66451D05753258FA,0x7D285958C7A09C70,0x5617C75F685D49D3,0xC812CF22C0564F82,0x1C735413CB2EB,0xDB08D5E1A19E3C31,0x431350D75CE35AE3,0x1230A01410558003,0xE056964F7537108C,0x9F480DEC8F44D3AB,0x80AB18CB3DC14F71,0xFB93C00AB4CE,0xA4DA93C29946146C,0x167F004A73405FA2,0xDEC708A7EC726135,0x3B0F3CC805BAE24C,0x801B84A811A14545,0xB2C65CF2DB0245F,0x1781B2CFDF0C9,0x21E7AFAD2369B5CA,0xD0B6EE6F5A6B206C,0x5959BB0563A18B7B,0xAF861CD2988CDD3B,0x8F12C543169352CD,0xD0372DF1DBBABC45,0x19706C59D2A61,0xE45AAB7BA40DDA80,0xFED9168291C6620F,0x80FFB8F004B0D0E9,0x4D9A92884ED930C1,0x27CB1BD879FEED90,0xA9C77814A3E3C2FD,0x219FAF6FD07A1,0xB6EC30D266920532,0x8CB29A1369E7FFF6,0x7D5808C355431989,0x788A390BFD47B3B8,0xFA716A6AADB1D448,0xA3154330FF5EEF6F,0x4B9DFA5CF38E,0xD603415471971675,0x105436BA268C0F11,0x2A2F580A31A6F644,0x3481734388B1E918,0x2C622457F2D99B28,0xC8F255807F1B71FE,0x183F9510A06EC,0xFDB24066CE968C82,0xA6C866A07A877453,0xB6E7BD82B015C816,0x62E112927722DACF,0xFB7D539133CFDDEF,0x414DE44E3A26E950,0x169C9691B344A,0xF6719926CB461426,0x876E28158BCC1ECA,0x3E18904712CD6739,0xA6EABDFB4F49CA53,0xE831912E525E0B4A,0x7B21681082748C2D,0x14CE489ABA0F8,0xEC2DE2829A236D76,0xA89B5061BC31D514,0x4D30FEB3F4B046D8,0xEFC79370701D6D99,0xF6B38D4F70A430EF,0x88A3C8920DD3DD17,0x1AACB7AD6D963,0x108FC2682F8B1309,0xF0EBFA41C1CF1296,0xD47CB36AD7B53903,0xD5DE9B9AED7EF00B,0xB9BA0155897C6D43,0x1A56AE3261D669AC,0x11DB629347049,0x49FD8CDB826492BB,0xE7FEE9E8A1A31984,0x5FA80EA5ADCA30F0,0x6BB0607E3B5445D8,0x59460FBA1C339609,0xDBB9FDAA8D1886F2,0x2253D19B47F6F,0x9C2D01F96C71C0EE,0x986698E464F17F4B,0xA2377AE4588DFA3F,0xCA1A99EAA06D4452,0xC094406AF2F7D088,0xF852B42D12C2E0,0x455ABFC4A688,0x8CC232D0029FAE41,0x1FF0EE8967F2B586,0x89A2096A166B9876,0xE51E7BC5430ADD25,0x14E96D994B63501B,0x5A13BAB060EB7E1D,0x1526C87977071,0x18CDCDB0CA56A7F3,0x19B29E0E63247456,0x1072F30249E8642D,0xB27BDBDD8458B19F,0xE2BB359016E681CD,0xF7349DC8464DA410,0x122530AF1B8C2,0xD4DB80B99E71F195,0x1FD6C73CEC487A44,0xCB8EC53D4D98D038,0x8F9599E7AF896C9D,0x9DE70E37D7D3404F,0x25918F1C61F2E8A3,0x2EE4FA576F89,0x6C3404ACA8916BED,0xB3C0CC23AD3DD17,0xB147775302AD42D7,0xC830CAD9EE1DE07,0xF74B80B2AD672AB7,0xF21C66A894C741D2,0x169F44056AACE,0xF5A07BDB3E01FAF6,0x49B7C9537B05DB06,0xD21B12C2060C99DA,0x41A3540698821CD5,0x26C54378DDD7F95F,0x20898686440A01D1,0x6EA345275737,0xB235CA9C5AF7606C,0xEFF72146ECC8B508,0x50634D2D69AD9BE0,0xB9848AFC80BC6477,0xF46113E8EEC7D3F3,0xE753C3F5FA03EA7F,0x1E5229C55FE1D,0xFB743DFA8B66FEAD,0xACD236F2AD5CC4B0,0xEA6E365C8BB2CEA2,0x173FE0D096FBE5ED,0xCF2C70C39B949170,0x3BB345C521819CDA,0x1BAF42688ACE6,0x4E86B73ED656AA5D,0x37E91829C4A0C577,0x86ADBDA10FE51D3E,0x6E8FB8254861C352,0xB55ED210FE373B92,0x57834442773BFEB8,0x232B5A5CE84F1,0xECE750A6D35A3CBB,0xE58307B01454B86C,0x6A06F8016C4E9BAC,0x809461E1A9ACFD1E,0xFF5238035FAC462A,0xD05720F7747F325A,0x2469F6C5F0CF,0x47E969466F0ED526,0x24C3896E3E2B8D4,0xCF8A698EF0FED870,0x7760BD23A0353DF2,0x983A71E99B3B872,0x1FC5D0AD83A9FC48,0xCB2111A4556E,0xA66195CD209E1B6B,0x4D67A89DA9917AEF,0x7394D4F03F7EB4BF,0x183F3BD60624A393,0xF6435CB2CB67B30E,0xDD9ADBA7C44438B2,0xAD10EA2280E3,0xEA0B386349B03B05,0x3ADCB8969D5FB401,0xEDBDBD1AF3BB46D,0xCA1DFBE7074248D9,0xE224390160186622,0x1973576A009DED2,0x7DAF1E640C47,0x9B3F61FEE0ABFB03,0xBF70F2BC3C323D4C,0xC1528598CE32C6BD,0x7DEF265161A307DA,0xF2886238D52CEC15,0xE1E9BB4BDD68A269,0x92588332E5D,0x5CC77FA16AEF2681,0xB67178E6C7513C27,0x225B2E4C08339FDB,0xC29432766F501C05,0x9BF2D9DEC3BF2D80,0x891CCB6F95243C0F,0x1A0D680E7376B,0x75B15F3C1272D5CD,0x4C8BCC658C60B127,0xB989A0CC76F1BC75,0xD29411D2802DDD7,0xC7C40C240248DEC4,0x383D441CCAA4EECD,0x47C493965C78,0xC06EE26FB9D32912,0xCDC32FC1A2276D27,0x723E492881AD3C0A,0x45AE685988E9F088,0x4855FCCB7DDE60D4,0x8015FA22A5A34595,0x39E7B204AAED,0xE5BB13F23A0AE22B,0xB1E46565710307B2,0x1EEF46ACE694C9C,0x84891549DF15261D,0xE2D7CD5499E0E766,0xA668541CBE3359C4,0x27BE122F22F,0x138797F397066281,0xD06BA590B498ED35,0x99BDE9DDA97E22FA,0xA830D66B2161BCD7,0x8D1B45C8C31AB861,0x47D2C92785B22CA2,0x1B4B394100A9,0x64FCECA5BB3232E7,0xA5352B77344F7308,0xD3BA633E275428FB,0xB1A84342DD44742,0xB69C8E78EFD015DC,0xD717E610CE2873C8,0x1009C130CF577,0x1E4E88F673D5DAF0,0x38FDD95E57DB4DA2,0xAAAFB03981004F96,0x8666E4010E1ED8C5,0x97A66F193ADFFF0B,0xB2ABC6ED8AE967FF,0x166DE6B0BD190,0x984F2596723D638C,0xA08A1C9590C07523,0x872DD9B23C61C4E5,0xD038422958E2FB88,0x36F3ACA0FEADAF7B,0x884C0C9BDF225963,0x210D4FD75B710,0xAF194496793F529E,0xCA64AB5739F1473F,0x16F5B776A24F33F2,0xBB6A017596C98D96,0x92E8E631E1DFBC91,0x43CE5C45D4477B2F,0x777F053580E4,0xAE1A278CFD8D503,0x646C05085881F413,0x95AEEC497420FFB7,0x44432EAACF0C5BC,0x6D0C27F428482A7B,0x36BF0F73795AA970,0x19B5D7D1CB06A,0x8CF1AAAA1528F977,0x30E061E79074CB0C,0xA707BCACAF089926,0xC4433A96484B2488,0x3EFC2B893CC11E21,0x3E9C16A82D794CEA,0x1AA78C0619F6,0x6081FF811443FFA9,0x350E77FDCAB2D6FD,0x59AB0E00DCE80940,0xCA3DDE7ED63EE6E9,0xBC6DBA8E805469F4,0x9E00842703BDF5CE,0x12E18815CBAB0,0xDD8EC29C8CA1B3E,0x294FDFB431DF7FF7,0x5161BE8DC4F9B74E,0x2D1DA9FADA3A0D19,0x6F8E5E8E319CDC8D,0x141B4E90AB5C35D7,0x209DCD53ADEC5,0x48BAEDD5FD2B1800,0xF0169DBE8161293E,0xB6552BB0BDD53BDB,0xE65FE0A7DA66E23D,0x1E1131F481DD48A9,0x2D1D20B8601744B0,0x5C6C4116B51D,0xAF3F77272DBFC895,0x5566C60CE4E13FF2,0x6A2F921B830DD150,0x9713F5E579DB2A7B,0xEB5DB9E5173B29D5,0xE149A82C1F4354DD,0x229123D77F20D,0xB67EF8C12A0F95BD,0x7D3E0BDECBDF5615,0x2708B77D19CF90E,0x23E07BCCACF17B47,0x8D47EB82DC6A1F7C,0x10A673E279430B65,0x14CF38522E666,0x1A096E1EEB270528,0x933038843B893F38,0x88B613C19DBB7467,0xFEC6DA1CA0713103,0x240180E56E9B7710,0xBF2E5E38057F4D63,0xAB82D1ADAF55,0x64B0D1361328A2FC,0x9FD323059E7DEC31,0x176507762516522B,0xA298240070E60293,0xC678C08972E3D95,0x6542786D3EF630C6,0xBD131F0FFF97,0x759C2CC404CF2525,0x1F18C42BB1269C0B,0xA93FF49AB4FC2F16,0xACEA1D2F89F846B4,0x55D0B3D1B30B72B9,0xC02DD5194D73EE73,0x1B8051FBB465C,0x21AD4D0C8B3B4E9E,0x1D4938F15687ED62,0xC490E543CDE63812,0x3D6AFEEC9FD48E99,0x404BDCF332999FFE,0xC049F5C227B38EB1,0xD4AF9FA64189,0x4C6F285F6239C182,0xC66EF01936A7320D,0xC4FE328637979F04,0x56D57EDF7CF75EEB,0xBFCB5B4E5547393,0x9B9C7A301E3D6EE5,0x1C1B26ED0C82B,0x4CDBE84D24218BE8,0xB808B0CCACD871FF,0x1145DEB3732F69AE,0x1BDF419C628D4BE0,0xA17C8038D3E66E3B,0xCE1DFB832A44C778,0x1D94BA500B74A,0x76DBC6BF1905DD32,0x2634EF48A254C112,0xD6CA51CAA0EABF78,0xC6328E6DABA4709,0x1A12187881954B43,0xBF62D0F7A9F849C6,0x20A02C731C77D,0xEBD258D94F44FB25,0x4E423AE47CA55F8C,0x593C2E2DAF3904B1,0xA6730BC41DBFB765,0x5ABCBA39E82B35F,0x684486408C99FEA7,0xC0791F43934B,0x7028BCB2B280628B,0xE85E508AA59D338F,0x75D883E5EF7A65DF,0x2703E3E40C7C183A,0xB960C62F477E2EF2,0xED805CF81622C636,0x1F33CE668DA27,0xD24EFCEFACBB4507,0xE859FEA5FF78FD89,0x920BB19909373B75,0x7FFF9DF9C2465F5,0xC307CDC3BC25FC30,0xA63D94EF63913399,0xED7041483B84,0x2A113CD82EF74316,0x25F3CD1F99B6A469,0x91F37212B85EC349,0x41469F7DA76CA706,0x66CC52A81908A5D,0x4E7666AFC4487BA0,0xA6A111E280D7,0x53157DA977573097,0xE2E41727861A0F67,0xC2DBC4398C642C6B,0x52CD3F578EDAE27,0xDC92BAEFDA69BA88,0x3E1170C7A23382C3,0x1220D031995E3,0x48920CFA2960125F,0x3CF3F278AC0415D7,0xAC6612B4FF9DD17C,0x958CF8566E1E0A4F,0xE38F6D00C4CFA722,0x44DD4D60F7A1FE34,0xC3E9D4249E77,0x33F4899BFC7C3EE6,0xC47156BDF2864716,0x728E6FF200D9FA3D,0xB5460A8F51DF05E7,0xF293AF33AFF6951A,0x85A506D7608A1D87,0x10AEA53AAB584,0x888D19D68AF4E1AD,0xD25B0BEB0232714,0x2DD2984B27DF8BB8,0x2DDF53B7AE50CC22,0x1F9EC9C49B76DD65,0xAA1D9F1F17D33215,0xF0917A6C7542,0xFBDA2145C797476F,0x6CD5C403A1CD6D3B,0x89F56C9964461B50,0x23F5B252926D5E3F,0xD4258EAD67277D86,0xD5539A71F8344291,0xE737BB5EFBAC,0x90F367CC36133E88,0xB98A3144A93CC999,0x440C4DAF9E85252B,0x1C5EB877144AE6E8,0xF0DDFC2DF4195E5B,0xAB7A30B97FD198EE,0x1C4B221F528C0,0x4DB27151549BB609,0x54736A4DCC4764B3,0x39B740256064F4AF,0x61CAE44808C102CE,0x7876252A843EAD60,0x289228F92C875623,0x1754EF33BEE8D,0x404EDA8D39F641CE,0xD870F99C9F39FFE1,0x2DE9B0EB596C7B5A,0x22B0EFA0177E2C43,0x27449E7B89A6C38D,0xC7B53FC84AC70FB6,0x1740C1CD62EAE,0xD3F26F1F9527B83D,0x850B832791ACB96A,0xFFBDE8F888140FC,0x3CE0491FA5944FF3,0xFC36A1819BF21EEB,0xC0B781BE5BD3AC3,0xEFA0DB34FD95,0xFF6345E89328BCE0,0x8D79BB07F0AD184E,0x35139A24BD461173,0x1734448FC20A9C8B,0xBF667217CA7A09A1,0xE8EFB503E7196C5B,0x22E314F4872E1,0x6EB8AF1144E66885,0xA0664F384B2C74A1,0xBE5C5B9C67C50569,0x7D763D5E67E0301,0x97D9BB30D21837FE,0x7B8E0AC0EBB6AA84,0x1D3E30FEC320C,0x386503F306755803,0xF57F11BE69F5A7F,0x5AD5D1AF653FC2C1,0xEF66E6557A3CB2B2,0x97E4B94FF03DA531,0x1386737CE662693B,0xF8971469D755,0xF04D0E8F962ECD7,0xB56E44FE43EAA2C2,0x170AFB64F110B4A8,0x7A8306ABE1822C25,0x7CB27D274B8FF21,0xE3FE38F8CE6A7B52,0x172EA02AA50B4,0x643C9B06E08CC66,0x2B87BDAF4E9388DC,0xD71922CF1672D873,0xE28B6BF1D657B51E,0x4EE86292969E26C8,0x774FA57CEFDBEBD3,0x22356CA9281DC,0xCF4DDC5549300A,0xE9413CA6F4CD4100,0xF7A542F8CF3404BA,0x439D8AC2A6FE8751,0x20ED88D8D653E5A6,0x3A1D37F43C306D15,0x19CB3FB40E64E,0x30EF0546FB21C004,0x3B818951DE4FB322,0xC765E13150B1C2D5,0xD678825CF33E22EF,0x32982733B22CB893,0x103D3E24F664B502,0x1C359727DB042,0xDF109C4B080306CE,0x1A578D1CCF3FF3D2,0x55602552343FD214,0x7538DD131E9ABB7C,0xD72C3130CB9A9C17,0xBA59DB23F49F0A09,0xCD2C2968CC5D,0x7414CC5229185524,0xACBFC0F4E77368BC,0xD9E98100179F527D,0x9E769DB09985FE03,0xC782F37B113AFEF5,0xDDF36B5E5F45F31C,0x103405AB2D43C,0x40DE33CAC0717450,0xC3B2E413E3A0F123,0xE53E09E084E1A6DE,0xDE1546CAB7A06ABB,0x7A16FFE13407D29,0x492B4FC12399CF6F,0x1219277B863A,0xFB9A3F094C746CA5,0x81E2E7491C7872E3,0xEE7B2250BD534FCA,0x601224D10EA0C48D,0x2F34BA04B8B07DD8,0x6DCFFB47B6412E5E,0x152AA294A9DB6,0x3DA4FB2FBD3C0BFF,0xF0D755D71384CCA9,0xBA2A155866B5DFDD,0xAF48DAD304B56D63,0x7D3C7D260FA4B19B,0xB06EC96A2148DE61,0x15C585D390AC,0x8EA7A8FB6C35919B,0x7C75E88ADCB355E2,0x2849C06FC41098DE,0xF29D73C749CDCD65,0x8FF0B7A11F3AD0EE,0x729A27BFB0EB9D73,0x1ADE07D8F5DB4,0xCD3F00D2031410DF,0x71A4FB9A0CE0800D,0x2B3BBF9AAED9501,0xCDBED9DD561A6924,0xB4BA31215E9AF8B7,0xE671C61CEE66F24C,0x1360681BB1629,0xC4E35287B435A223,0x5686889DEA700377,0xD887EF0013EA1713,0xFD36C4988796A889,0xDFB874596A4296A8,0xC15C9FABE5C014E2,0x16DC390DADF51,0xEF6420DB718E8CCA,0x22FA53308FA56AB,0x942E1FABB5C82FBC,0x1F024937482AC7B8,0xA13BA11324609DB1,0xEFDB248CA66B462A,0x72F06D79F8E3,0x9A4DB208DB96B3C6,0xF193B9FD64A6E6E0,0x681FA9DC20DB5D3E,0x104B991877536BAA,0x6F1EFDFB41FEDF60,0xDF88781860A4D51B,0x22A463C0BEBCB,0xED6E3F02C7C46FE6,0x4C7AB61159472360,0x96E35359DE718AFA,0xCFDEEDD57314E67A,0x294929D4C604D0E3,0x1F26CED647677727,0x219FC25DB3665,0xCFDC8FE63B395FB2,0x479356ED76587F0,0x7941A8AA36E594C5,0x9DB6379AA5BD5E8C,0xB5EEEE5C3AC553A7,0xCF2C83741C73FEC1,0x21559AC288838,0x3555F7322975B35E,0xD2BB2813B031EDD4,0xD383D98C41C4AC95,0x71E5B04EC5DB2483,0xC1ACE99F40E7C9EE,0xD127B2220B5C328F,0x8EF07D64B36,0x8F9C314CD1EA55DD,0x694392EBF98B8A2,0xB251111C6791930,0x4603DA5542CCF7A3,0xA8C11246DB27A30A,0x1882391142DFF5E5,0x14ACB5655C154,0x89C7E784BA9DFCAC,0x7AC5FD56D285ECF0,0x1EED5838A9744BF7,0x411FAF6DBE63D539,0xA6E7B243074FE4C6,0xB37D3A6746DEFF6,0x69B843253C29,0xCBB6CCDAD699F147,0xE407FD6CD2429B71,0xFFD8F22806765E1B,0x9BAAA7A037FC0C2E,0xF8373E117EDA3D78,0xCEEBFFF90BAF3B03,0x80E00A8A1753,0x1DC7261C69DEA34C,0x2D366AB3AA66E40F,0x8EDFCF071C904EFC,0x60C3E72B5BB70A9C,0x4B56A53C01649C2B,0xDEADE364E0023CA7,0x21D6660A39A3A,0x177D4F641E8C4445,0xE9F8A0EA417FCBB9,0xA39BCBB585D54F6C,0xAD7B9AC3AA5193F4,0xBA8B961E256977BD,0xBA756133BC0CB8B4,0x813871F29AC3,0x5DB62A851952AC1B,0x9B3F62695628E9AE,0x6E6B76578038E04C,0xE21CA7DDBDD5830C,0xA3D21C6FECCFBFD4,0x48CF797C096E0740,0x80A32146236A,0x7CE03DBFEE575E24,0xB5502FA503B9635C,0xF6D7020410A9C247,0xAE731401F0227220,0x626FB4B1155D4A2C,0xD3B4121F356F92B6,0x118FCAED5598B,0x85B3E3F08124B64E,0xF104133C18139ACC,0x28902AADDF6361F3,0xE8DD45B08B900D27,0x9FFDBAB1C0E62AA4,0x264C51B5BE14E180,0x69BDB668EDA8,0x719DE1EDC7D0A421,0x6BC959BD69519685,0xB107B97645155E0A,0xACC868A7936881,0x50F46B9C022275E,0x2C3D1284963EF67A,0x21ACE55D2850A,0x876A9B62899AE70A,0xA597032DBCEE2638,0xF80BDFDD8D6D24D9,0x6E8C7C9277CA1A72,0x9F6B8CF39318805A,0xDFBABD16529CB1A2,0x12CECB1521941,0xE5E32920DCF5C7E2,0x977B6818D9225CF8,0x4C2C1A90AA10045,0x3F6BB1CDAE29B31,0x616EA6FF7C682817,0xF6520D1AFB4CBBB8,0x543E4A838328,0x2D820EDDC0651F7B,0x62013B028741C437,0x1B35E956F9177AEC,0x935EFADB19A405C0,0x12DFDAF2C5348559,0x5279F1FE21C8FBE,0x363BB39BE8AF,0x6CD3519BAFED8441,0xB38340789DEB45BD,0xAED44626481A1A53,0x46B118FCABF566E1,0x29AFBF3B872A027B,0x8E2B400D687BF795,0x146EA999A9953,0x8D8E09D61F8B04BE,0x3BFF5233A5B23972,0x876234D2414CAB73,0x30F22C61DC9E5B6B,0xC9BE7B9418C51E7B,0x669DD4A7F3D7D45C,0x109D73460E8D4,0x87A996347FFB42C3,0x1FDF5D79B2F4160A,0xBEF1646A2FE69D72,0xA8AFEA914DDE982D,0xDDA0F66F8ECCBBFE,0x7142C3A7533328FB,0xFD86470B37CC,0x7BE54C5FBF393E42,0xAD8EEAD71A1B0DF2,0xC7DF639A4B3E6D59,0x5CF51F3FC9BCF254,0x924BF07DEBF89205,0xB1FA9B5DF653DFEB,0x9AA47F4C66E6,0x6B65C09FF1BF37E4,0x8B67A6412C1396DE,0xB1C3DB99CFC79ECE,0x2AF2FEB2FFD763CB,0x8CDA48BEFFC89561,0x7EF4ACB8820EB11E,0xC295B5E3C215,0x4FBFBD071811C44,0x7F9B332378872F9C,0x3122325CD5EFF137,0x890B5D9091E1FA1B,0x1AF7B7698754926D,0xBDEFC6942C1332CD,0x18170143CDD6F,0x37B7B6E086E1259,0x2CDAC4FA9084F90C,0x7395E53E950697D,0xE193731660AB74EB,0xD2D9F2B85BB07E5A,0x9A24A36BED7E7169,0x1D7D39DB0756A,0xC53D6B47AA1A22D9,0x4C31DE2344308D97,0xB5095B8FD03E4AAF,0x47AFFD594571C275,0x44D4DECE6B311FA4,0x6C04202276C75ED6,0x31107804501,0x5C2E02BB72CC67C4,0x199056847A147B95,0x1F5E769F5581E04B,0x2AB2DBC5984A080B,0xE892CB86C4C3332E,0x81D1F64438ACEA48,0x189D5B472319C,0xCE16FD8336A5F12B,0x2D0BD200C05DCF17,0x759D564FCC3D1D5F,0xB050DA5152AC93D2,0x4A05AA3503015256,0xA9C2FC326D770AB7,0x1F88396C8570F,0xFDBF0DA4DDB14AEF,0xE31C97CAAC2B1AE9,0xC1EE53A7AD179B6A,0xEAFBF7A9FAFFAF8B,0xFA0A8C02A4F7B0B3,0xB763F72ECB7E482B,0x1E53148120D31,0x4D4CE2364C6A6F44,0x18F9CD80BDED669A,0xA8BD152FF08103AA,0x3873E8B339B5EF4A,0x18995B6BEF1E37A3,0xA651B0D801AFD341,0x674753C6055B,0x71B25620BA2351E9,0x470B289271A90100,0x5AC9214438915FDD,0x4EC4A230188AB8D6,0x23B64C76C199BC5,0x98DC0D68F585345E,0x1A3AFF392A192,0x51B86B9C5BD849AA,0x86066077994AFBDE,0xBFA165C507A052B3,0xF666F3D99F8DDF0,0x3DB1410FEC0802D3,0xEA21E37FF1B89A72,0x181B3FA7BEDDB,0x7CF9C503C698917F,0x6D29EAC059A3FB6F,0xEC171C079D4F79C,0x3A63EE1ACE6FD938,0x1F7CEE09CB71B0BA,0x735FC8D24266D78F,0x16BD0946D1735,0x9B1F80F7C1A38D1D,0x2B00C41A24736AC,0xFF73205A93F649C0,0xEE455544367650C,0x81B91666845B9B7C,0x65A4B598599C6F28,0x17A4167D393CB,0x5FEA7615E1586961,0xF6CFC55A4E3FF600,0xAD12626592D51E7E,0xE4E9A2F63398AA6B,0x660564A431C03F58,0x80D3F23AB2BC5441,0xBD5D2F62FB42,0x7B8D6C84C1C173BC,0x1F855FBC9D7A2626,0x8A43C1EB4B2A1186,0x7E05D617AE981155,0x8C6BEDB65D55B368,0x8ABEC0F8DA0E19BB,0x179B122228D23,0x74F79E8BC6EBDC73,0xEBBAB125D14022B2,0x2ACFA700BB2E4D24,0x3260C9E2A9346D87,0x12DE20BE9B6EF154,0x9AD472813D28322F,0x1A843F9E12AF5,0x57FDAE3875A8750D,0x6D18736EA187ED61,0x6C8AFDCA13C92443,0xF118921FBBFEE1ED,0x1D5626A7695F376B,0x599DE0B91A589A2A,0x1FBAD61F71587,0x72A9B33F06B83EE7,0x3D791D5E45E1824C,0x2D533C8F6861664D,0xFBF508F845035F20,0x7E4C27072918DBA0,0xEBE602F6C29E73E1,0x22FCC4BDD7B42,0x78CDE4F9D6051327,0xCF10B1620B661D24,0x98D4FB1F5781A67C,0x909C5A7FB50BDDCF,0xFF01BF5791090F1D,0x8DFF2D3118FBCB63,0x7F70C09D2EFD,0xCF2EDABA8295FD20,0xA6F8DF8CC583746,0xFDA751D82BF865CA,0x2FBE1C53DE2C0424,0x4D55ED1A61339F79,0xB34234F2BFE697C7,0x2E5856979B09,0x8816C79C8ADB0ED4,0x9339FF345CA7C8A8,0x535DC632C3F597BB,0x3EF3D718CACCF0AE,0x89B774C401AB4891,0xB5934191E05CB2BC,0xEC1FA11D8B68,0x8AB1A1F70385C52D,0xB94556CF9BF3D0AC,0xB4AA88E6819E5E0A,0x6D319409884BFA33,0xE4189021F087C029,0x15A89A91B12344D8,0x1B2710E8F329D,0x5298F797A27F5150,0x92CA69210336675D,0x5DB8C32FAB48C020,0x1480138C4FC22048,0xE6325C0E69BE2148,0xEE1FDA3493C7CF34,0x197B8AD2FB15,0x287360F85DE134D0,0xF7BF6A242A2034F4,0x9398781FB7AA3F70,0x5B9D567F433BA060,0xC330328C0593C768,0xCE880153ADCDB51E,0x4BBE3955FFAF,0x2BD285A7330F2945,0x9CEF6B1356206C9D,0xEE6C9617BBF264EE,0xB06956627D3BA3BA,0xB94687FADF39CFBF,0x117E6CA9CBD434EA,0xD4FE1A24A740,0xDC514777F5FCEEDC,0x51CB00476BB7B247,0x8D5E114884D6551,0x20A4959B8C8F1761,0x2A43692EC48A6EB4,0xC00EDA522A0E1380,0x6C4DC102325B,0xAF0633E8A9E5527F,0xE2EEA66F7250EC0A,0x6CE7F2C3CECFDA4A,0x7BE5543EFC7E0572,0x1BD3EACAA7CB25FE,0x62B3D7DDAD19EC4B,0xF99063F2F137,0x450DB5A186E9D34B,0x8F37B9F2425D8EE6,0x8893C0DA2AAB45C,0xFF81BFDBA648AE6,0xA40FD5C504D362EF,0xC04BC3C3CC54B8B5,0x1D5C05322BADB,0x1F72B005F6500AF,0x718EFF708D70552F,0x3C8A47DAB7154E8B,0x16434A662EF12A6F,0xC2F96FAF1A3FD0F0,0xB3B3C304A6AD0B20,0x1D6C6319FE55A,0xA09ED6D834081483,0x3D961759E80AACCD,0xB9E183993FAF1B95,0x3ACBA59ADDE57342,0x568DCB21DFF38F13,0x677FBEA6796492A4,0x128AAC311C758,0x308133220F4A5675,0xD672D989540215C6,0x9869F3FBCE75871F,0xB7708AC22FBDB5E4,0x9CCF68316022F1B0,0xDEE2C6A84E2C5D87,0x19878F83A0035,0x9310906A4C863A71,0x7DCABE89A78E922A,0x539E8976A185C160,0x6282B352E69009FE,0xEC000B175141C6E4,0x93E7BDF374F90EF1,0x1628788CA99BE,0xB7F9BB65D8BAF28C,0x4C7BFA14B8D3156E,0x34FA213E1E1075F2,0x1F94CDD1AB61A46B,0x9D63052543E45312,0x2C9AC5A3C4E98741,0x1079E5A2858B0,0x286787778402A292,0x145C63904754316C,0x1FACFCC71C294471,0x14E7B408D394AF92,0x1843A9224D6B7912,0xE88598FF4C7A96BF,0x22F1F3142A93F,0x10174710B4AC3AD0,0x7AA8AAB51CB04D78,0x23C9486761FB2E2,0xF22EA5ACB02192D,0xC47A9ED354B2C950,0x30F44322C9CC2354,0x1D2AEAEFA4BC7,0x46DF8F21E4F31262,0x2222F9F0CFFA1405,0x9E47AC6AA5BDEC06,0xE0FF3ECBDC073EB8,0x4B3C858154C05798,0xC81999835CECA0F1,0x8110F1B23837,0x3920DF6E58D73FFA,0x3770DD18D63F632F,0x5D39D84AFBF42931,0x2B67E29F9E13A52,0x95804445B117CC6B,0xEC4645C1B095E737,0x18C16B829E158,0xB65AF2AA6DF1D920,0x8B26EAD8E215174B,0xB427002CA16FBB0C,0x7E64DB0FD00946E7,0x59A510B5763F161B,0x4094A25A1C2F480F,0x1EA94B1C47662,0xB91425437778E7F7,0x8E2EDB1C49C16B0E,0x7A2785F78BBC5293,0x3E1A368AE6C66246,0x8161D1C4013AC402,0x301D1316C90FB6B6,0x1BE2D2E82E769,0x1016DE3C0F13542F,0x109182BC1B760757,0x14680DF44F5A0C26,0xC8390A3962F895E3,0x9BFD11CAB43F31C8,0xDE7CAC87490CDB0F,0x87163AD58642,0xECE7A259AB7107E3,0x2084E61BA6A31A74,0xF1DB96C020F9434D,0x472BD63069CDFB41,0xB3012F8E23E8CBE6,0x138D20EB3D3F4516,0x140F7C34576DD,0x71D64143FD756C8C,0x737F9C70493DC8C9,0xBFA869BD0077D5AB,0x32F7C76C19C2B3F8,0x185B8DE9CF33138C,0x4146A5ED121604F,0x199DC7DF43EC5,0x1153FFFCCB348CD5,0xCEBEFAEA41DF75C6,0xCA8748480B814868,0x79E2C99FBE9E5880,0x2759ADF9E306DFE7,0x5446CBC43774951B,0x1C2895200D4DC,0xDACECCAAA3C2A7CD,0xE56712DACAB019F6,0x7955D43F30F61063,0x9AC9F05FA4ED2841,0x65B5339DB6048BE2,0x260CC3B4AF77C5A,0x2F347551F8DE,0x8D58D309EEDC486F,0x112C5BD86E9E0FB1,0x858BBAA717F59232,0x658B8AF8E7B62FBF,0x88FB5C91F4AE85EA,0x22A6C230F9C90815,0x11C19CAB72E67,0x1D90480AC175AFAF,0x7FFC8D58B9BC7EB3,0x827BD1874A612D07,0x9908D51007A0EBA5,0x2A5D23B758579A71,0xBDFEFC70358620A0,0x1738C2B2FF3D1,0xC983335328420E5D,0x79C286A5071BF4D2,0xDFBDC55D634338E8,0x697511718C4AA293,0xAF5CF0B482CC875C,0x1EF2E2D292D0C957,0x977C97D9279E,0xC9F66A0683B8C3A,0x54EEE9D8BDE994E5,0x7ED22AFD0860F6AD,0x711346E4E1C858C0,0xA692118CF50453E3,0x3036C4A5D963CA83,0x1A132C155E430,0xE6861A1FADEB275,0x2075B10C5ADE9821,0x46A86544757505B1,0x6BBB476E6583AA41,0x31223859B2B39372,0x19F39E6BEC13D373,0x8C68B695D2AA,0xCC2FED47AE78DC2A,0xBC9A2CA6BD18F4C2,0xCF9E50B65018980A,0xBC6A2DF109C0CA18,0x2D664A7D1B84B2EE,0x25D97DBC2F591D05,0x10EFBE11F08C8,0x4ED30A44A7D583B9,0x6DEA0899A47B6EEC,0x65A7C972D43E26BE,0x1D88B5C2F65B6318,0x7AA7EFB82F1BC251,0x17BA08A2FC3B1B68,0x13F416D8C9998,0x6C6AB7915DC56FA7,0x44A62D6C0DBFDDA9,0x932849230E2C950E,0xF5515D4E0A038407,0xB700FB5318C1C881,0x68D705F045535BC9,0x10C67C833867B,0x614078FE2C65F581,0xD10FC22198596A48,0xC12E3D2A7AF855BE,0x19E0DBF641D0D61A,0x95BE419AE23B904E,0xEAC47BD7C8A0C503,0xCEC5ED309E4B,0xDBCAAD67468C1E3F,0xE3868C7D24892DD8,0x8532C53E9F064304,0x5A84E56B3DB5641B,0x8F7271A31C4328E4,0x1D12EA24F6EB934E,0x13D49DA82F2CF,0x95F6F7D6C4F9A54,0x82CFDA459B67E23,0x71B78784D9AEE8C6,0x537179EAD94CBAC2,0xB259E260ECAC9463,0x776C673D5A369033,0xEF62D95139EC,0x375D907A4DB2A4A5,0x8361B6F14332735B,0x339259421FAC2B08,0xE71E76267B6A20F8,0xE373D3575AB2C700,0xEA061F809BE71774,0x1F8714D51E389,0x9DBF97926CC0CD11,0x7F4D4B16BFFAE26C,0x1A1C19B192F95A1D,0xC24B558E10922174,0x43B91A4D085C41C6,0xD0D9DA2EEB1F8E32,0x17F3E0119DFC5,0x952FDD73F0B5DECC,0x1F510CA832EE938C,0x76D42E3E7895522C,0x654856DD5E83684,0xB1F7480F90591155,0x6E41A748338C099D,0x18D9BD8953D3C,0xDA29A1DD58C7AE34,0x977E44C3210B70B4,0xC51126523427AA05,0xA2B05B44C4803A63,0x378EAE846D0312BD,0xD72C2F48D8745C23,0x613C40A1117A,0x9F9D362976F40D96,0xD99BBE928EFAE86F,0x77C1E91EFA831B,0x7E8A0807C5E5500E,0xFAEF3C566C9D024F,0xB9329BD4D006F227,0x117B661F0BB6A,0x6F9BEFA1F740BD1F,0xFC9D3D553FC31FFF,0xF233FB831A6D605D,0xE35458D0041A4005,0x7EE3214D99F6A4A3,0x4C3FC466835214FB,0xCFA3F7A992E3,0x75225C779135BFC5,0x3AC1A77E6211EEB2,0xAA3E38A29008064B,0xFACEB3724A426F9E,0xE5B617D1110215A2,0xF5AB629A850999E7,0x12263BCDA0259,0x21AB8166177C053D,0xD97C2DE0F5066E60,0x815E0FBEE13FAFAF,0xC8C1A4D7A2F9A10C,0x7700DA6B212FBEAB,0xD2E5C0650D9D2329,0x2A8FDEF774F9,0x6EE28B87068A2E09,0xE1D44A98F93681FB,0x9E9F44B4F47ED0A6,0xAE8834EEBCA4A3C3,0x8E24EBD9DF32342A,0x50F0E7697122BC91,0x839792CF2644,0x297043419A52B7B0,0x6D6105739A378A07,0x423D295874F14F92,0xC3250F49F78A9844,0x3A5A24F25B2AD265,0x42BC8ACC48453D02,0x22C6A5F6A79D9,0xD13E6CA77320AF5E,0x79BE8D435E44B0CE,0x133C68BE90E8DF37,0x568DF78A4A6002E8,0x5ECD671867C665D2,0xE0437E79726F7321,0x13FAF53501647,0xF838BCEDB3D4B830,0xA8855EC221E60485,0x507CE532CDDEBA60,0x757A9CDDCF653E57,0x2E544610E9D02CDE,0x717C3C2A4663E432,0xB304FB9B47E3,0x6A7A2678C565779A,0xA2A8C43404EC85CE,0xDA6C3C4727D8C1BD,0x95C40C06C2D61FC3,0x344D31268A5E09CE,0xC5F675964FF880A5,0x1A6C80FD4975D,0x5A8F68B8B0E78675,0xF92F400CBFCE87DE,0x6D35E78E1BDE0830,0x3C1AB45B5BC2EDF,0xD1E2E249C9988A2,0x420461DEA4EA9A6B,0xABD4794DD096,0x535590A143DB7CEA,0xF61CEA217FC37F9F,0x6DB8E94682A16E04,0x21E012004C8470B1,0x6A070AB20BA37D2C,0x6C9460E143D3AA3A,0xA37222A352A5,0xD86C4BDE8031F281,0x6BF12C0B987168F8,0xA7F557020FE57AF,0x69FB1836B6380366,0xCBE1E7CA931C623D,0x323D1C387A1B4789,0xB0F1296A5E2B,0x7338F8B5C7E9B29D,0x426B5C75DE8C8DCE,0xB67063F4FDC3F23D,0x7C5B6B7971793736,0xCC4208DEEFDE9453,0x25166CFEF257497,0x132B040B9AB59,0x8736318961941A53,0xED6E0E9635E86131,0x7CA40E67B2E9AB36,0xBE606F2AC109A825,0x4F2B1198250D908E,0x3225091C04B09195,0xEC693D0AC3E3,0x4F277C9E73F2A111,0xAB607A5008F2D8C2,0x69A89D6B649D42FB,0x88E482741647DEE0,0x371280C3D5B669C8,0x8DD7727C23D5AFE8,0x5D2544E7489D,0x9754E0126F38BF6A,0xF390592EEBBD8E61,0xA2402F58D7DF42BD,0x1CDC8CAC3014CAAB,0x8A2BA7F2186E0731,0xFF5703BD43832136,0x13E474D088E22,0x34DA9AB4DE5F1482,0x9F66274A5100F430,0x3D1F1DB437CEF4FA,0xD949DCA4ADD35587,0x30B90F9AAF9B6AE3,0xFECE70370DEAC3C7,0x3C9F1FCAEFA2,0x325D970D05B82478,0x6616D62DAF53E39C,0x13EEAAE4BD9A136E,0x7EC1490B48D5EB2A,0xE41006D47269FDD5,0x54ACE7AD07CF749C,0x212DC1E595ADF,0x4F1521AC6710C6B1,0x4A66FC8804A125B5,0xB8A76D2289435294,0xEF9B239F99D9F582,0xEA91B734AB8307D3,0x653A66567FF7747,0x15C168BEF0B77,0x50C01DCF3905A757,0xE1CE4320D521C745,0x53F0757590043A6B,0x1A71A86EB8A6FD2B,0x14E72C72B290D75,0x7630A002EA8B0F0F,0xCB8375A779E0,0x3590ECC749A88583,0x63D247632DBF7A30,0xE841493B52CEB63E,0x165DDE01BF4E7E48,0x166EFEA23F42EB28,0x5E6CA655F80BEB35,0x1AE8F655CCF19,0x59CE81A200D56B02,0x9B309C8DB394666A,0x6CCB522E3C19ABE2,0xAF3CA598140F2A92,0x79E3695A778ED089,0x57B667DAD3903DCD,0x17EE2D370992F,0x7404181C945C5E07,0xC0F768D3F50EBE6C,0xD718798D962EDE01,0xB78D513962391606,0xA4C2DBAF6EC8B84A,0x45174F93F593AA97,0xCC1E32D55183,0x984C581089CE807D,0xA518A730544F9DB4,0x7C5AD283E4A282D5,0x2BF5DA606E8A633C,0x11237320F34C1EBF,0x780EBA8DFEFC73E9,0xDBC1A9E0D852,0x468AFBFB812D248F,0x4AC18AD78729B300,0x4A018CE7BC3039A5,0x4E85B312F13EA169,0x897EF8CC903BD43,0xBF44CBC3ED75C48F,0x17A185433CB1A,0xE1A67052E24D71DB,0x47D6B754C481AC8F,0x30E28090C771C4CB,0x48BFA5E75E58CD12,0xB396C1633BB3D263,0x9265A047DD665B54,0x1F9D27150DEC,0x6A6A33A4EB026227,0x4F502D7EAFFE6205,0x29909B58EA9795CD,0x2E5A376EC650CF1C,0x8481563236BDAE35,0x5D75DB648B1E8C91,0x5B21E1AF5724,0xDC685D0DC3E26B9B,0xDC9FA533B337DFB0,0xFE9946BC9C7C7AD9,0xA179D6F9A6433F8E,0x1ED2A05EB6E2BA47,0x9E7B699B1B65CAF9,0x1215D31C38F6A,0x5BEFF278C7DA88BD,0xC839AD06BE540ED5,0x58B153B8FB3C09C1,0x1D80D8A58B989AB8,0xD16F6E81D3CA2D28,0x113CEDF44C82AFD,0x210BD230441BE,0x120350D86A72504,0xDD751D0B26754680,0xBE37CF40A62027B0,0x565CA751D58D03D2,0x74A4C1E6AB6E70BD,0x300A8360CBE9D081,0xEF6AD7B8693F,0x6370ABC4412F5D23,0x795A34DB98CA5043,0x72A3CA17EA7CBCFF,0x1FDCBA4A976A0E6A,0xBB23AD9481B50D15,0xBC36ACB5975F75FB,0x206ED60C2A89F,0x592E0D514B16C38F,0x767D73365858A10C,0x7055B50F355D6327,0x5AEF8B49E5D98C3F,0x9425F76AC4FDA23F,0x5FEB2CD69843E5CF,0xB7D4BB0CD8D4,0x6C1CD6002AF4A7B0,0x678E5C40558133A3,0x65CDF7E824C8B295,0x1AC3487DDE039037,0xE65C273A7AB63496,0x212DB48306A125A4,0x136B991A79A71,0xF272CBA3CD86A1CA,0xF822B5DF9F50F36A,0xCD8149DE291894F6,0xC216223FEDBB77F4,0x5DBC34EA9A932BB0,0x6C41B441100ED30,0x80A945B1C46D,0x21C81793E09D942D,0x5AB9011B38386BD4,0x4E726ED3AEF00762,0x758C643254DCD59D,0x3245B16C92BD61D0,0x1EE134BFC232D322,0x5EFF9210CFBD,0x1358569E74C3819,0x61701E48C29C038A,0xF15BCE69DCAC9B95,0x2F0CE8A2759C728F,0x70FA1C5C1D3A1D73,0x53A549584551E036,0x2E7F470BCB03,0xA22689292CB603F4,0xA2AB8B080CB6D5CB,0xCC062DFB57B7F088,0x94AD1D973A7E748,0x9A6541E2DB7EE6F1,0x8FCA855F8D648D9F,0x13B33A2C2C92A,0xCF19198A0C367FAC,0x3C90232BBAC99377,0xEB13848694390E2,0x4B85229FEC74DB8,0x21978AAF0E812A2A,0xF249CB7045C3B1E1,0x7B170A4480A6,0x2A14A627C8E7D3A3,0x9592D3608A2D2640,0xCFDC65013201699,0xFFD90242B7637E39,0x216E72B056ED0F5C,0xDDBD4E097A99BEE7,0xF086C3AAF972,0x24F22285DAC19C7E,0xF3777471B8DFB6C9,0x8F7845AEF2416084,0x72D3161A169957CC,0x87EDF60EC7CCDE6A,0x31976374B60FA750,0x290B66B05C0D,0x9120C07565546675,0xCE2891EDEE9A1CA2,0xD5676DA332F6BE73,0x671E4541BCEBFF35,0x242AED038DFF4C5C,0x3A69E28E95A7592D,0x77A380B1599C,0xBB948AC7647F95DB,0x2BC3E282693CFE73,0x39D6DA4D7A206F07,0x845F4C1ED9901CD5,0x630F130A33A6E54E,0x962599D90A70697F,0x1014F0432FBBD,0x9CF0E7C4B8A5465F,0x50350A811AC57590,0x884C66425C9C9259,0x3F1FDCE4565E1A04,0x75FF00229018CFBD,0x429253DBA75C36BA,0xEE660DB6E671,0xA42C5B2FE885B3D3,0xC0C4F6C0CCC3698,0xB9B17104FB1E1DEC,0xCAAAF27A2F8CB69A,0x8CA66EB775641EEA,0xEC6A789BFA6E1503,0x2441FCBA8AA1,0xDCA499773B929C45,0xC3A20FE6438E8B5A,0x45A5AFEDCC29BAF8,0xAB9F24A88C1E411C,0x34EDA03583918C8F,0xF9CC8280894539A1,0xEDCF244632AA,0x228B427F42466077,0x8BF746F0541874BF,0x5522B86B746370A0,0xBF3327C5BD107DE9,0xFCB7CE79B7F2CD69,0x4E17124213832619,0xEC8562F276E8,0x7A437C2D203EBE34,0x684466D2F9E5DB9,0x56035D1EADCA8502,0xA33CC63BF68423A0,0x8D610AD2CC887089,0x7F2BBC9CA0872EE3,0x301FA550FDD1,0x6DC6E08748E8C9C4,0x535413D9D8D41468,0x2CA20531A67DA368,0x600A164B7CCA24DE,0xB8FACED9E033DB54,0x75B945573D382744,0x12EFBCD32F2A8,0xB0EF8889C1E7E333,0xE225C16CFF150372,0x414E8527AEB622EE,0xFFE846B1DBB2EC29,0x7F46867FB4E22F04,0x3E53D47B5FE1D5CD,0x3FA5E1A6A210,0xD0EA73BC61DF2CEA,0xF453129680FAC2AF,0x8BE5081C22CA1130,0x49346C79985BA836,0xC42043D67EF0F7E6,0xD5E436C0F8DF0D4E,0x400FFD3E2529,0x250EB74C7EE09775,0xA8FEEEB668E7F1AB,0x306C611C86A9D870,0xFBB26E1F1A9D12C0,0xCA35EB147792F79F,0x5F6060F64E6770FB,0x14120133F3C09,0xCCAA61E9864982A5,0xF43B9182A0814DB2,0xE819AB728F47ED5,0x7948D9D986F98292,0xF7D58765948EDCA5,0x32CCEEA126B2BE6,0x2052F68023386,0xEEE200809F944896,0x26AA48F4A6AD5970,0xB8026C1A283E771C,0xCE2C17A3453E89EA,0xE137224507DCB55E,0x1C4C29E6F74A5D43,0x171D10B64B3C6,0x96DFD30EFFCF90B0,0x9E25F49D1C9C8D8,0x4DA91F6504292BC4,0x8E19DD5670172626,0x3B7C8C2ABAE79EDB,0x36AC6C6D5D51C864,0x207194D669579,0x95EC8BA3E346CFC,0xC1CBD6AF75C19593,0x488A8A16DFB80C71,0x1608F63CEBF02B48,0x3E8E77FA840F9AE,0x92644EE655F4D242,0x1DFCACAF898BF,0xA3EF5E88A4AF28D,0xC1733034606F3272,0x614E54173F0FF070,0xA5FCFE30F26F1DCE,0xADD2C87C76B4E14,0x94B1C85E8CB2AD91,0x117BB53B682AB,0x2B388F44027F1CAA,0x21029FEA571C7627,0xE58FAD890CC9DC2B,0x8749FA1063E1F2E6,0x44468E8D0CCCAD8A,0x879C0BC0AC9DF4A1,0x13390FCCED3B3,0x7089F618679FD6F1,0x27BE7E43D5F65F4A,0x395B1F2E138815A2,0xF5A8B1FDE031B0AC,0x33704C8B3A534F2F,0x443BB569E914F1FF,0x125BB443493AE,0x1D1973AA826AA5FA,0xAA10043558E12935,0x50368312F6EFBF3A,0x6132A94DFD9B6F7C,0xFA9DCF0023C8AD70,0xFB7E426FD9942777,0x8FBB48069507,0x2BC090C68BD8EE0E,0x764CE58DBC86290C,0x342F2B4087315509,0x6D6C844D366792D7,0xDF4B07FCBA13221,0x2EE780CD2361198E,0x15B362A1BE646,0xE7F17209CB8E1A36,0xDB89851D0CB308DB,0x6F04DFB4ED40D4E3,0xF44ED784565BB9A3,0x9D5C4E1C365D6A00,0x6A0E3B4F7B902C74,0x1BF0885702CA6,0x182935E4F6661A44,0xED1FA57AD444D8B6,0xB2817CFC86EB7B9D,0xF27BA78E19D3F6D,0x8CA6988F25CB8F50,0xA62E1A0F6B46F3B8,0x1D0E7C858F6B6,0x8D1C7818D04382EA,0x19A4DE6D0E912AC3,0x5C7BF61F80264FD3,0xA650F9F54BC86321,0x8F3C6400DFC78293,0x1CFEEC739B8AD6EE,0x80F69214F75C,0xE85D6930BE6FE72C,0xC80B8F443322C878,0xE9C2D55B597F1E69,0x5B9B098E9B057487,0xDDD1A8FD742EE48A,0x2D9C81DB9A3CC4DD,0x2341876BBA9AF,0xA2D713CA4DA69538,0xEFA983630A202D49,0xF58532E4A24C3A1F,0xA07DBA43E5532E40,0xAF7492709CE1A512,0x2A26EEEF9B7C9FC4,0x13E1530C78719,0xE0E26F23FF29A66F,0x8FE04D27039CAE77,0x8E7F0F5F6E1D63BD,0x50A05F46864DCC87,0x5A20616366659FBF,0xF98F63A203063AE,0x2296B7724AE91,0xCDF1E9A176A9FB83,0xD0D821C77909DA59,0xB95C5CC74F1513D9,0x1899086C3FD8A3B9,0xE5B4E4F7862DEDEC,0x72227ADF6DD4E281,0x136FC0FBC098C,0x2E089CF9935A4672,0x7FC8F4556BB1194,0x7AD2837DECC6176,0x1EE96BA08F355F31,0x189FFCD45A5E4ED3,0xE9BC5D7F4E0E3231,0xB675D32F43E7,0x722682B08E26249B,0x625C205CB137373C,0x773850FB6267C440,0x117CAE101635E96F,0x400FA1C9D733496D,0x43CB9098CA3E5E7C,0x23A7CFBCA4A8,0x55F7AC19EC3417B,0xDAE84EFBB5BA989A,0x606A3DE840E41724,0x6113727829E7C873,0x4D60A8D053982161,0x234FF0FE7B22D35A,0x11B714C308B88,0x977EFD821E662F10,0x4ACC6A30C04E0052,0x4699AC4BF5ACBF3C,0xAA51FB8D509B24DD,0xD7067667009B25EC,0x2DE710ECDB091B4E,0xCE727DDFBC0A,0x841BFD1407E4E26E,0x340A234FF09F43C6,0x6BA9B5F55257DC10,0x17D609C8CA9EA848,0x4B0B823418FD637F,0x80862168F9703C21,0x1275AEFF12068,0xFAD6AC9BF8CEDF9F,0x3303560E613096C1,0x10BFB22C07D6BF59,0xBD46013D14408CC5,0x4F95F951D67E12BB,0x23854556E0A53CDB,0xA43C1453AACA,0x4DE8107878D93522,0x4671522019FB4088,0x698A1F6DA7A13F7A,0xFAE56702E5E45235,0x9D7730557B28D017,0x99A66B9119BBE749,0xAEE6C5B38462,0x4FB4DAB428834060,0xE543D0DDF03C2DF1,0xC29FE99C65FB781C,0x2DD5B12060A707FC,0x34569F0A8CA5EC2,0x9E3E88A0B8A07525,0xB0BD165A2610,0xAF3FB7E7716795A5,0x7AD36A2F1935F422,0xB1F82BBE3B41B53D,0x691D602490E09FAD,0xD65651575044CBFB,0xA5812AC05BC3698B,0x23B85A4441E9,0x80D9DE02084A59EF,0x23F12E8C98479A6,0xEB37B64C82DA5A36,0xDF12F6FAF10BDAF8,0x189B10B584B9104E,0xC35E24E6F1AD4D55,0xF0FFDEBD1E60,0x6CDE861409C8C890,0x14FACB12CB84AC04,0xAC4045A116D2BE7D,0xEDB99688BBC3B9D3,0x53C6C6365329DE0E,0x5368F48EC3966CFE,0x39378A557AE,0x2799F97DF597D592,0x49C07A3BB43ACA35,0xADE8D932A952DED5,0xDE1624F6D647D759,0x4DD12913D2175293,0x84239BF5DA2FBC54,0x185717BC7D9CF,0x1D6F5733908AA93D,0xA378028911B50EE1,0xD9A075B956A69DD0,0xE5B9503F40C8BA64,0xAA44787CAF25FA2D,0xD8757CFD85C5AA92,0x9189A25D54E9,0xF7C883444A28CCE7,0x4CCFA6EFEBC19E46,0xF4803BF962B82674,0x39733C4E3D599BC9,0xF6D74E81D87FDC20,0xB3B7229C82E0685A,0x79DE2551A5AC,0xCA683D19CE892DEE,0x8D97C4CE7B878968,0xC5B3CF751B05B6A,0xA92027EBB3339D0F,0x84011C11D8EBC1A9,0x5D6C8BF7A1071EA1,0x93CEDA0D129D,0xD85A0BD98A863DD2,0x71963A806670B5B4,0x3D6639FA66817A9D,0x40119E8675733FBE,0x492297233F38615F,0x6B7F3CD9035A9B56,0x199C23F41964,0xCA0E1F60591CEABD,0x40FDB462A7BFDE71,0x42C032A44663A55B,0x76D408F7E692ADB4,0xB1C39A4E139318A9,0xC5B8FAA418E24952,0x17B161902517D,0xA9477207388956DA,0x1F0B590B9476C90,0xA4FD636879065157,0x727D597D39B692AB,0xE9436DFA169AF61,0x5FD1A0A263D94695,0x1A2D645863983,0x910C7514D7F54929,0xA7C9EB6AFF759BE6,0xB24ECDD9800CD022,0x57FF6ABE68AE3D0E,0xD4A23AB50E0CFB4F,0x357D9A617B66CC73,0xDB843312FB2E,0x309DB7349207FFDE,0x25FB41DFBCAD1316,0x925C0748631137BC,0x6C279324C574D9D0,0xFD3BEF269FA2DC49,0xDEF90D1F10221D9,0x8E6479BDCAC3,0x6C1384F34C25765F,0x98448695EB511E99,0x23E92E9E05006990,0xA3912E8922FC4E24,0xEA4AA2E4CB19C120,0x17232578D35C569C,0x192BB46F70239,0x64CC26D0FA313211,0x9B9BA33A5D033059,0x184062EAB4B72582,0x3679E17D8C6EAD22,0x3591B6809D2D5AE5,0x5BA93B62EFA11615,0xA6EBC90BF042,0x50E4F4B98821D4D1,0xB575CB814F310365,0x2ADC07558EC10159,0xA6A89B5061EE63C5,0xABE4C0C41FFEF67F,0x2AE3D2951337E71C,0xF6050E9D52F4,0x513998B93FB7B8F4,0xF6BF005E2B8A1C25,0xFA82E07DEE27C541,0x887FC0B87FD94250,0xDF1F205D4C14A304,0x54C72C8DEF7202BF,0x12F43D5E5181D,0x3CFB605CE4A17332,0xAAEA040B4B405D5D,0x878034393AB5BFCB,0x58A78023319CB60,0x7A59CBFDF4CBDE89,0xCBF4F757C7001F7A,0x199287F695931,0xE3354273A95E742F,0x72A24FE402EEE0D8,0x5CBFFD0C1E600473,0x46CA38E9B3DAEB14,0xE14C54A93AA29A51,0x25FC090248E113A7,0x116C76B8D8C5A,0xBC31759460FEEC2B,0x34F66F615176310A,0x568703F737BE1B08,0x178950B77374354A,0x1ECD99422103E64B,0x3C3641CABB5FD1F3,0xAD35C5A69ED5,0x4B92C984A3361EA4,0x3BB460616BD402D7,0x83740FCCB35EFB85,0xECE0BF66C6349E56,0x45C583D7E4787BEC,0x93DC001AC9B3041F,0x1FD22E6A24FCB,0xA23DEA0CAD67572D,0x8966D8DE6459FB88,0xCA42D02A039FE5E3,0xAA6532B6D3A687AD,0x5F949B6DECCA82F7,0xDE95845ADE1BF7E,0x1F8DDE7640597,0xAFE2BFC74BC9F271,0xA504017123B59104,0x8A6C96AD0066E3F4,0x60EE8BC87541C843,0x8FE520155C87252F,0x38264C323532C9CB,0x88EC9277AA85,0x85AE1C149882AA10,0x53AC6DAA0AD9DCC9,0x494A49D85759FD3A,0xA65763F99403B901,0x276C02FE797FBCE5,0xFC31EA317B686AD6,0x8D1B52DEDCB0,0x957C53280ECED01E,0x7278BF64F740CE3F,0x992D7106E08BBCDE,0xE913042400567511,0xB99702DD2C75E3E2,0x20EF8C192234B24C,0xA2B8C667DF0C,0x8AC50CA4B2143036,0x1295237A142E7000,0x3204E27D925A8DAE,0x2FE7B760278B2CD5,0xB1882B7B4DD1B04D,0xDE89A5C6629BC2AD,0xB57E7666984C,0xA98D39C823B86B43,0xDEA7852265C9E9AB,0x1EF85EF85888505E,0x867F986D3A44FB75,0xBC5B3116D1C5C5C9,0x81B27862A099ECE5,0xE68313271EE9,0x83033369198620B9,0xEEFC56A3108A80C,0x5D9C10DD74B01223,0xD25B0E6697161B70,0x4477C5EB2E96358A,0x6E4C1324BB16DCB5,0x2ABB5B332528,0x10711A48DA40C4D7,0xB874DD6F38501AC7,0x39913033073E8F61,0xCDACE4665A65DAB7,0x381F9C945A37CDB7,0x74A3708E395D3B59,0xFD205E728FE5,0x450C732825D39F42,0x6DA82924CDDA3398,0x92533B055624D79D,0x97540199CF72E85F,0xC5CFBCCE9E3D8665,0x3741EC05F2B37C8D,0x110AC4C8AC67D,0xA7C03B7A794A2C0,0x3900722857C30881,0xA8C441EF2E641D12,0x753C800DA8B95AAE,0x2E33B7FB49ED6FE2,0xEA829A0C10C83864,0xCFA2133FD629,0x99056B00E46EDA03,0xC7CB58BE96C3CBBE,0xAF9644CBCA7DCCDA,0xDA0CCCD04F8C7042,0x1C0C954C2DC9533D,0xFE8946BD5038C081,0xE7E5B4C3C495,0x8D1E10617AE6018C,0x3B2C930A125E3A88,0x60C44C8EC1B16FEF,0x4F211CFE26163B5A,0xF258F29D9DE7A575,0x6D64EDFC6902E2AA,0x10C573A373BE7,0x8E548774845DB5A5,0x4345637A44BC2699,0xDA037714FA5B8B1C,0xB2C8E6856A6C0F7B,0x38011D2909351F9B,0xF057512080B9ECDB,0x1B53EFEEA2992,0x5C647139F93E8911,0x213C195F24FDE0C7,0x40EC8ED568954FC9,0x39B6C759D4FE0378,0xDBD178BF289668AA,0x1EC3C7E2D8C8E446,0x1A2A40734EA3C,0xB13209BA1A8B4519,0xC9FEAA926625A353,0xE94B61E435D811BF,0x48C20CFDC389AA22,0x2463892FAB4155C7,0x1BF053B27A757810,0x138C6F4BA0B64,0xF659121E9F61C2E6,0x7E8970135AF94330,0x696F8055BA3BE107,0x5A3CFD16826D2976,0xF2DCB66A041236FA,0x8E0A81EE383A4DF3,0x182E1A9EBB67F,0xFC481FFC677E1635,0xC678EBD18FAC00A7,0x97E8679A96B4962,0xCC075D6731B2FE8B,0x394FF1008650CE2,0x8FE6D6037DE4392B,0xBCADE8B2A492,0xA131FE830394236C,0x641A4A6CFC021622,0xAB8145DA582C3F53,0xEDDD7188C5BFBA9D,0x643D219CA38C2A4E,0x758E45E65255DE5D,0xF7E62A420001,0x1ED7C0957C3538E5,0x78BA6FF2201FB760,0x481BC0C01367CCF9,0xB865DAECCC43959C,0x77F1F7806DC8C70F,0x1C2EC92AE51CEAED,0xA5C02B6C10BA,0xF31CC9E6FD0813E8,0x2344462F5EAF3A07,0x751319931D916906,0xAA885BC474B9245F,0x1C9C806D7D7DE4A2,0xAAA760C170A5B67F,0x58178A7C5D25,0xD6BBD20CE1C81E15,0xEC53B43C0463B2FC,0x9C4E4E29E2EE4238,0xC58788FD508558E3,0x793D47E589D42F02,0xF17E896FE3E8B436,0x211B7687797D8,0x62845FBFDD3605D6,0x9F9AB96CB199CEA0,0xFF027B962B5355EF,0xDFE590443F995C37,0x48A2C8C8C081BDB6,0x576E85E1A9492922,0x4485D2DE7492,0x441AAF50A7765ADC,0xC85FFB9509936729,0x1B483388F704517D,0xD9B3181A1C1604CC,0xF77B1FA22F46E894,0x9F9DD17A3BB7C5E4,0x2227E953EC04C,0xDA9A4A4A683A9E56,0xD65D4EFA9745D31C,0x18C3B0D12319785,0xA52A2F09E88C3E12,0x7921B8F458A4AD9C,0xD2FBBE08F54B445C,0xECEFD08658B2,0x1AF9CCF2CA16A3B,0xC6ACBF40B74B1B79,0x1A1AADC8E4AFD078,0xA73825A7A740B8FA,0x4ECFD80154708C14,0x9A05F2F76E39048B,0x199EDAC6297BF,0xF26CE6B3C9F32FD4,0x30EED38F8D91A2A9,0xA296F524BF913859,0x889ED41B52B989B8,0xF1CFF0B92ED0CB5A,0x30CD703206864ED6,0x190640EDCF66B,0xE5FE2CCF9E20DA1D,0x2FB60FCC57F7F655,0x637083BADE7F8BFE,0xE9427D7D504BEB6A,0x300BEB63D44C6DEE,0xEB2938D89370C908,0x30934DFCB34A,0x9E039137470B,0x6F3DCC21AAF903F,0x6ACB3ABAF5F2ACCB,0xB6BDFDAD68FB4728,0x5DBC896D04CB6362,0xCA64E67B2CA76FCD,0x1AA100509452F,0xF994305EAF53C5BF,0x4E05C37A0CBC0CB5,0x17768346A726A28A,0x11CF6926E3EA4A2,0x649C8BEF51BA9FE,0xA74EC9A5A3C0E08B,0x160FDB48B4304,0x9604E7530B283841,0xDE5D3AEC663A212D,0xB95145F1B99A99C7,0x90EAB5ECD4DE4247,0xD502799FE0311ED6,0x9ED934F7466777,0xBE94220B09DF,0x3A5D790A6FFEC7C9,0x8D9F3DA1C5B82737,0x97E32C881A0D83E8,0xDAA42FDED89BD1C3,0x46C45E92E997E66,0xD8D48CDAE7E7BA79,0x674C440D2866,0xBF5215000F10A15E,0x40E34A1F28B599C6,0xC78D2D89475A068C,0x9D63FAE954EA83FE,0x48DFE44E002499,0x19AF3A7D8043C1A3,0x572446EC8AF2,0xB55AA3B660F15138,0x45B02987EF451A5D,0xB1A72430FEAB6EF2,0x1E61951A0D939C92,0xBF8B87F3F95D64B4,0x4B2E1D626868E114,0x1B9350B4F7E66,0x4BA24D9CE94FAD0A,0x7DDB90E6850D3FF3,0x29D8928C50BD4BDB,0x685D858DC86D03B2,0xCE3495945B200E29,0xE8225B1A6F24DCE7,0x9E342CF87D21,0x213AB48B00E29C07,0xBD99D3B0A4E3D108,0x72D5F5A36C3C1BCA,0xA2160210780DFD0D,0xE5C253444A62C137,0x336BBCBCE3E02F48,0x39976769AC8B,0x70208293363DB27,0xB76E699117DC4B41,0x4C242367D51CB9CB,0x1BFA0A4634F0EC83,0x264055BE553719EC,0xD633ABE9F149B2DA,0x139DE7B163FDD,0xDF015B00E02CAE31,0xA4422233AED1C4DB,0xC17F03068B13698D,0xC351D52F59457464,0x4B0F3DFBFF83AE22,0x9C56345ECF99E3D5,0x43AB6C66819,0x7DC26F4BD1960F45,0x3CBD87ADE9B3C4F1,0xE854B765FBE7BB0D,0x61C91F58FF53D063,0xB3B6B7B9B58D8803,0x518C774AFAF9FF96,0xFE7F74EA5E43,0x963CBF7E03089379,0x1A88FB2FF3BF8507,0x5D84C1ED51F66F3A,0xBA3ADDAAF898E5AA,0x68B961CE95E0FC37,0xC7B9D4C72A83C381,0x6FD8FA0007FF,0xBA58E3C76C574B9C,0xD0D70D0230A00F6B,0x1A48DB416926EB79,0x6C5EF2DD1E2E4BAF,0x66A397BDF4ED066B,0xEF7149A223DBF99D,0xE769A2E98C05,0x973B0EEA2F002CED,0xF2D2A8397C1C5BAF,0xAA6D49F949237D77,0xFF19DE1EB4096DD0,0x56B9A92BAC5E1E59,0x8DFADE056361B410,0x8F74F6AF1FCE,0xF9142769B8A5C09E,0xDC1F2B3A186BB2D5,0xADEA82112DDD1325,0x508C4A72AAE757FD,0x9FF573D72F37F55B,0x2488CA3628A5E53A,0x16E3834942F49,0xBEE741B69BAA3AEC,0x1F8721AE66510DFD,0xF884F9E41240EB21,0x823824184AEDA563,0x644B46F6356E3EE7,0x80E4D3D824CC0A62,0x169E9FBA08CB6,0x51EFB82D1D0131BB,0xE477017A6C5DA5DA,0x793626A82B9CEE31,0x4E38D75D33B126BA,0xDA6FD67FE1257D58,0xF103AF20358188C7,0x11D7AD28ED593,0xF27871CD147DCFC,0xD47A2BC5830170C1,0xB3DE6FA5D1632FDD,0xC5FBF22AE9BD163C,0xC3B3CE5E0207A080,0xDA3822D40E59F171,0x1FF4C77FDAB5C,0xEB2CF9E92B4E8E0A,0x39B9A2FCFE6BC6C9,0xD43DD07116F81FE5,0xB109F16D0F0DC03D,0xD305CFE34D8536E,0xFFBC2E2E815671CF,0x77F9B197B6F6,0x54D19C6803630BA8,0xB7AD1C069EF9B99C,0xA63D5752CA05A492,0xBABDC9EFA5CD01AA,0x4AD8BA7B28B04BE4,0x7EA42306B4CCD45B,0x185A8EF6A828,0x1D29779603072BCB,0x98BB0106F3C9D8E3,0xC0D167F257E375E6,0x3181C8BB9914977E,0x1832D37F1EBD500E,0x6602FE3329132F7A,0x21878A5E7D28E,0x9085B7210BE48927,0x3E17F4E03EE5CDD6,0xEB336FC34B4DFE05,0xE2727B1F4E2CB1AC,0xFEC489A704A10B97,0x63EAC340E55BC180,0x2054A25D35218,0x32990C07D5AD90B2,0xAE880262C4886912,0x5256D5C3FDDCA3E4,0x70CE3079DB211A56,0xEC27AFD26BCE1B6,0xA557BAE72F5DA394,0x98470924F022,0xB1EBD5D81A2F4E47,0xE02151561F7C3DAF,0xA59388D3E9C53D9E,0x7DED93F982310990,0x664A13B359E282B5,0x521AE1FA2D19EC7C,0x16E9438796653,0xBFF32C1221077F76,0xDA58DEC4ED98DEFA,0x627E16ECB9686843,0x94B1302E5771C6D1,0xC32C45ED1CCDADB1,0xB9A85FBC7B24C48A,0x168292646FA4B,0x86AB41486EC91E90,0xE58E4D03FC03E313,0x4CA1DC5FDA7A1DAD,0x5DD58A3C8596B0C3,0xCB67E6BDDCF97C6,0x5B72094C52940675,0x12213FE7A62FA,0x3E81990B60A5ACA3,0x9871473A15F8F7C0,0x5C5301D6AFEEFEA2,0xFA4BE332F527D55F,0xA26F8A21ACD9BA4C,0x9F855A6B18F9ED21,0x16C393B4E9CB5,0x4B05CB864F0456EC,0x98753687AE1A8A5B,0xCE765F036910524B,0x4BA2F4A05BB34D61,0xC6B7B22AF514A63E,0xF03D69A9CE6F2ADC,0x148606F00DE52,0x3101895E7B9E6BE7,0x37C69D6CC6897F9B,0xFF59C6354206328F,0x679622D90521BE0E,0x2BF9F1631A50E501,0x56491BA643794F51,0x12030D295213A,0xE8F52AD253191A21,0xF977091C3F2E5E8B,0xBA8018B198231FDC,0xB3B17B160842A288,0x33A6E34665EDEF73,0x1B739B93ACC609EE,0x9CB68EF014D1,0x2A18ACDAE36BF4BC,0x2F76A86A3D34093B,0x55DF43B6B843D61E,0xD5421B5744BA8528,0x362BF42D59A7CD20,0xD66C3F9AE3B7F9AA,0x6D69ABE591D9,0x8EA81473E6FC49AD,0xC34E5EA9F9A2C9F8,0xC450ADE6A14038D6,0x94EACE30DAF7917D,0xE238CA5AE201A54D,0xE244B0584FD63136,0x2033838E34C3B,0x106CBD9B10379484,0x94CDC7028E76F44F,0xDE460838AEC9E102,0x87B47DF1E4B8AA43,0xEE2AE0B906E9A809,0x198E4E7FDA9EC9C5,0xF84FFA1E9C5D,0x2792E12CFA1BDAF9,0x6D6ED1ED0C1166C0,0x3F860C4703C8DCAD,0x7F0467627ADBB954,0x4CB2F34F79B31DE,0x3A39C4F25052F25F,0x8701EDBD7232,0xC36425449FE4C2C,0xC76362E89FDEC027,0x46A425758F1CC8BC,0xCCB740616F7501E7,0x6DB2B232E18E2036,0xD9F27AF67D2FEF97,0x2EC52247DEBA,0x339D5A50C49563A1,0x4F113DBF1F569F36,0x29ACFAFBB6FF86A0,0x48499077B38808F1,0x4971EC79066AFE88,0x706D35C45E802E78,0x9157C002ECBB,0x802A14B1371B3C10,0xD6BD68E2A7072954,0x2A8A66D106B34131,0x1B378F31C2F00A52,0xD49B0EEAB656AD01,0x18D21280D2CD9A62,0x150782C4DA8C2,0x4C185D6CDAD40981,0xBEAF2C501AEC6AF6,0x2C4C794323194622,0x9B59FE6643D330FA,0x2ABA615C945D4302,0x3F93FFEEF2333049,0xF39DDD2CE7EE,0xC7EB97C631AF8F22,0x6DE161DDBC8FFFEA,0x2F604BDCD0CF2CB,0x7911C61A30FAABD7,0x662769E526546559,0xAF480BE487AF2407,0x8ED1939DE3DC,0x54CFC67BBB61113,0x8E0489F8EB20D77C,0x7278995178C64C84,0x7F569007FA69CC7,0x290136D3847345A2,0x8BE7C446AB69B89B,0x119A9D1C82F7C,0xB5DA38569EC56E6C,0x330818C075DA3049,0x834FAC0004757FFA,0x9E5A90BBBF465847,0x58E9F80C959C330E,0x72E2CCDDAFCD42B4,0xE2C47B491B2,0x2FA2C04A577EDE78,0x7C56DC7374D48815,0x2634710DF9332C1E,0xE246F88EC0DF0652,0x89DEC8D42C65770D,0xE43F4DE7E794EE9C,0x1572BA4F41ED,0xAA3004EADFBF43B8,0xAC48EAF6AAF0110F,0xB9894D0D6466E199,0xEFDE5F75DEC3B118,0x8C297E5D7A8B49E8,0x300A01DAB52DF6E7,0x223E71288B735,0x5D7C754D4AD7FFBA,0x29EE9E36557B3E25,0x89A84459244EF533,0xFA934AEA25AB6B03,0xE399D9709FD8A080,0x46B9373F348C9D12,0xFC791149256D,0xE0153A98EAFE1797,0xF0C5F04325829516,0x621B431D1958B148,0xFCFEE863C0B98076,0xEFF294DAC0C81277,0x5A8CF85DE0E74CD1,0x13008692F0DB3,0xDD007EE7296DE436,0xA9534C752958CACE,0x1251D48CD60E03B0,0xC157EA9B658D9461,0xB9638E2BDBF3467F,0xEFCACA4237DA0CDA,0xB5E2CFBE30B2,0x38D36B8053BCC653,0xD85BD7EFE1F32546,0x330B55E91F9A92B2,0x4F322F9BE10520A8,0x41A5EE76D9C45D7C,0x6B520E929A5E5DC3,0x1359C616D0054,0xCF3315DA9DD95176,0x1B353BE82F0ABE81,0xABC2FE39A18B8E7E,0xF6716D520C15B8D4,0xEE8A9AF5D2D66669,0xBB012036DE1E388,0x209090C2618E7,0xFA36260B659122F0,0xAE7C0A3B27334CA8,0x652C139D4C279113,0x7B1487A9062A4DB7,0x7056E71DDEB4CA2A,0x489A0AAF749271D7,0x39ABF87F7379,0xBB271BCE5CFD6A7,0xE99E513F3EE3BCEA,0xD3DDD265C8DAC8D5,0x6E668EF7F5E78B78,0x2661D03D07C7A0B3,0x44E4F91974384EF1,0x14BCBD26B54BA,0xD91A032CBE8C3FC0,0xFB34199A793BFC1C,0x1DD537B266C75081,0x70FEF01D6F992E5D,0x14E8C9EFDBF5EF2,0xD5404CF18A1C8179,0xF9F5F57D3EFF,0x41179C04A9EA15DC,0x1517ADB40AB39A05,0xA23F9C61796608B7,0x7791976BE3DDBBF5,0x4A8C5A70A03950FB,0x8B48BF3288AF2A86,0x84EC52498E01,0x154F7CED5C38297D,0xD1764B3C06C1ED2D,0x8DD7D7988677E62D,0x9AA33B4453385BD4,0xFACCDB4D9F57E93,0x54BE8CD8E7A26B8C,0x192A15F9CA469,0x5FC875D1D2411B07,0xB83467B2CF3C3432,0xB77F174E6D819AAC,0x23E83004F39FBB0B,0x4B057A99E52D25A4,0xE2BD8B8AF6AF931,0x1DFBE0BEEA2FB,0x3858EB9CF06AF55B,0x4D17EEFA279D9132,0xA9DA20C76DD0607C,0xD2723289AA5EAAB3,0xBF013F83B4C73E2B,0xFE6F84619C35D29B,0x199BC0C0F1BB2,0x2A43306D5F587AFF,0x643AB0BE50F208CE,0x84CABCB6C7B601AB,0x4B4AE1D4CD76F1DD,0x7CBFCFB445100515,0x889F1FA2BC49F987,0xB6D33770F694,0x7DFE4CC85617BCDA,0xF7A39EE2F0911B39,0x1846A91B99AA3390,0xD261EA7F818E44B8,0x719FAFA1C1F9CF1D,0xB162EDC868F587A2,0x1469FF4B24BD2,0xF9B7B78541FB028A,0x39CEDA9B1402EED3,0x947A14354412EF3,0x3E0A9FAE115D650,0x7317B924C3A53EBA,0xE0ABC9BE9FA74E5E,0x157592107C2D5,0xDF54738CB1EB1B0C,0x64EAE29DCC5DF9B2,0x7A7E70D3D78A124A,0x28C4445D770BA644,0x54CC77A69A56AC16,0x8C0992523F3051DD,0x16474C7BFCB7E,0x111BD3479C2C2F09,0x2C55B065B7AA31C9,0xF0AB63FDD1626773,0x4CCFF5B67E5708FE,0x6212902F3E0D93F5,0x2E5A391470B31C32,0x12AC647C4617A,0xB71A498BAD4FB1A7,0x6371CC037C9031F3,0xA0779521CCB2883E,0xF9C4126D5BDED025,0xDCA010E41B13808D,0x1A123CBBA47D1E7E,0x60374E7BFC13,0xE37046930D00E43F,0xDBA0A0A7D957A6CB,0x227259ED16911B90,0x7D160623DDCAD6D3,0xB124ED3B07581A52,0x7346851C986004DD,0x149CF3EBAABA,0x16794B5EE6CE6FFD,0x704CD6C33A419997,0xCDEBE2298C56376B,0xB8155FE008D4425B,0x34A1CE4B27B6074D,0xAA672F6FBEBE1DE2,0x3CFFF7A4B96B,0x7B06D906FB47A35D,0xADE436A48EBB807,0xF18E57FF265E4EEF,0x60302BBD1BCE78DB,0x19602D936D84E4A6,0xB6AB6D212BD2F9D4,0x1C5B2485F716B,0x9DCF7DB2A5F9E59B,0x8D8A494A926332E0,0x9281B900FB23898F,0x9134D0FB7634F044,0x8224E01B80C52C1F,0xDDA1411D2953900B,0x1CF6555A63B4,0x8A0C817BCC55C76F,0xC4660AA8C95FB91E,0xEA12C10331BA8333,0x2E465887D8F53D06,0xC1A0ADFA219FA9C8,0x68E1352733C6DEF5,0x10D032BED5BAA,0x7B400E3886A892EB,0xD6B2017F8D71C09E,0xF21C23E20A5989C8,0xAD85642A580575A7,0xCA9F5EF57F5CBF7,0x2DE38D73BFDC5896,0x31579B483392,0xC8E31C8408FD62BE,0x615F2A587D326DED,0x35BBE790881B5DB0,0x2474377BE12E4D81,0x1F92991A880FBCE2,0xC3296840AC5197AA,0x1B96DE3BE1277,0xDF31CFAC5CF38240,0x2246F0A68001ECA4,0xC543C5AEFD708B4E,0xA9841130894E515B,0x99E8F869235D4F5,0x567086B4A6E21CD1,0x68C14676D6B4,0xB0F5879F5A6AD51A,0xF882189AAFF010F,0x5EB4753424A2E873,0xD7E96198DC29001F,0xCD267581F4508F61,0xD4A238CA8D676C66,0x61A4488CBEDB,0x6B9BC9FEFDA108AC,0x16A62C4ACCF81540,0xEBB7DB22E95B7231,0x4162EE47EBCDCCAB,0x862D5B347DB2A225,0x1525BD1ECB660771,0x98720EF290A,0x314F30B470FE3F0D,0x338BBFDDE03DF2CD,0x9074C6CAA8B38E31,0xA1D8B24E63884E7A,0x69D0659C12603622,0xF88EE87B9408A359,0x1F07F51F2F0E3,0xD9C8FEC6A3F08986,0xED62FE3466666532,0xD489BC7570CDC0F4,0xB4C576AC1DFC0F12,0xABA97F744BD66893,0x3119EDEF1C555547,0x1AD80E5062B9F,0x407A1EE9CB63E0CD,0x1900ACA836590409,0xE7317182154DF3DE,0x10D13D02A93D0162,0x56869C87AB10A333,0xB6B57F2AA55D730F,0x223EE6CAA09BF,0x74EEF204797A8F9A,0x8BA811290B57D7D1,0x2BC288C7E0F5C1BC,0x553D9E8B56F23812,0x51AD3DC6D32D3F25,0x3DA5AFE68C218CCE,0x16B19BD4C7F90,0x20EB1F6A9F75249D,0x5E51810A76D82731,0xDEE25145CCC2A89B,0xAFC9E1E784078AD2,0x4715EDA1EA8D3046,0xE28B0989292CAA49,0xDAF0C2C394FE,0x66BBB3AEEC191F5A,0x3F577CE42DCC6BD7,0xF9A2555F007884B6,0xFEDCC3441943011B,0x2F4E5C339CA22480,0xBE711B951CAB347D,0x7C7C1EAC4BE6,0x8C3ED2178F61BCCC,0xB794A5D0D7C6D291,0xBC5BA1057256E5E7,0x4DA64479AA25E514,0xBD2880678A18C68F,0xA0B473A650C1C194,0x1BB4148163983,0x7B1CB08592DF5DD3,0x1289EF1DC12EDF63,0x286082817CA374C2,0xEF7F5F2C36446198,0x982F8091DEB2C6DC,0xD5472513D786A5F4,0x6B7CC6C1FCF3,0x9CEE29A194E20A0C,0xA0B3240D4C08EAF,0x65DA4DCF583F51D7,0x8C470F5DD0AA315C,0xEBAB10CD15186110,0x6F4D74E4E4073167,0x1213C3A8DD307,0x22F80C6BC5ECFEBD,0xAB636425D2E5100E,0x2528B82F30776A58,0xFA33F9E6A984628D,0xFD275E4D1DCC2533,0x8894357E48036712,0x13DC115D286CA,0xBF19B2C615BAA99A,0x36E3B9C711C7DFDC,0xF163DE740E4F6A24,0x285BA2CFEC04B7FA,0x6E9C84829BD4DF4F,0xE99833D48C12E37B,0x69421D613DD0,0xB66DF7C2D9BEB4D2,0x4B7EFF6758A66A21,0xD843A5DDE2D52E6E,0xB75253440E1C4844,0xBC78A529F0CC4A34,0x60F427956C1A5689,0x9D080E3C7124,0x68CE2F1039846242,0x4DE75D23853AD253,0xD32887575C6A69EA,0x4200E254443DECA5,0x44C7468F49E52FEE,0x1E5AE802C3E4C873,0x1C0D455EE0DE1,0xC090A9512785F30A,0xAA14387CF93518E4,0x9F41F62F4487BCED,0x853D124AE7A1E73D,0xA0E37851DEFE6C92,0x22611FF4B4655B5E,0x17566E4BE998,0x59D5D937A555132E,0xD4CD7AA1D3137769,0xFDA68F1203A60472,0x615480DB66A3138C,0xE761526C2B2C0802,0xD16997A470399790,0xE6C8FB813794,0x8B00E9A73C402414,0x52B5615CDB18D204,0xC6640AC763205E01,0xDBA6E3CDA161DA0B,0xF2C39B9AC73132BE,0xA6EAD438BA2513AD,0x18644360FDE28,0x766A55D8CED99B89,0xD39193B5800DB7AC,0x4ACA310B4550B75B,0x4DF4659035FD34D4,0x4F5DF3401DAEA9DA,0xF4DEB85808456E14,0xBB186F1B23FE,0x6290B485208D9584,0xDC324FD222CF8E42,0xB006A971AF1241B2,0xED4282834DA4DF95,0xE806D8F8EA89B4FC,0x944D24B41456D684,0x7CBFEEA7B1B9,0x472DFF1CAE75E082,0x2DA2E3A09BF155E,0xFE24D0378807973B,0x9E81F7952D4ADFA9,0x311126DBECA89069,0x532FC04E3E31721,0x1087726122F74,0x639EDFDEF93AC000,0x1A1DB7D2F8979DDA,0x44D860F411C416BD,0xFA52C938DADB0E9B,0xAA3A0DFBB01EFB92,0x21C94644EA9D6C14,0xE87956E25479,0x5B6D860AD125E564,0xCAE3CDA7842831F4,0xC34AADC750309EA5,0xAAB6545C578A139A,0xA09251CA46670049,0xF694790E60356661,0x11D20E5E09DC6,0xEC67FA2CE2E71EC7,0xA55DED0D162B1CA6,0x80A374BAF04BCFB3,0x54BD8018281AB414,0x5443F4B63B2D70D2,0x56B76AD775B6328A,0x1C49E57DCB7D3,0x7FA1CD60F05AEDE4,0xDB6354215F6BC102,0x69C71AA5DE5BD68,0xF4B09313CF5D134C,0xC28B51A7B3311857,0x9DD3B8049EA776E1,0x1FF6B187690B0,0x1FBE8EBC2C2781A5,0x6AB3E87542503B09,0x153C4215B098C95F,0xC097425FA7E250A7,0x2349EE8B1378501E,0x53D880DB41ED5E1D,0x7CFCE600E90D,0xE04BA3A855BF3775,0xAE48D93CC0D78BD1,0xAC9FC2FCC7BB7BB1,0x4474A2D9648C0663,0x8E6BB54CA9608D4,0xA638A5B521CDE7D3,0x15BB61EC1452D,0xF66A2A390432740D,0x24041B587EAAD54D,0xB2ACD7E62A729630,0x4645DF23E43B1A97,0x71A8886D23D5BDA,0x7DCACD2685A0CD99,0x54944A5C471D,0xD14ADFBAB0ADE782,0x6805D009BDF4D45,0x95E66FFC9C36083B,0x451F3597A8DB5BEE,0x79036FFC5D08913D,0x4EC5E1CED6EE5D3C,0x3C9AD96FF78C,0xCFC620C4C77DE808,0xD24AA92E23C4353F,0x720EADA550F5CCA9,0x778569A2A42C8373,0x5EB093E2BD539330,0xD11EC11D95FD2BD1,0x183CAB0BF1833,0x689DE2D325BD90A0,0x82D9462CBB120F0B,0xB8288FCDE5A417BB,0xD01CD9A89D95C49B,0x7230D01B545E0226,0xA771CC006784334C,0x74B54AD065B,0x3676198D6C9EA7E9,0x58F87DFA4D614823,0x2DF809FFDE284D4C,0xB2381EDF3AD632B5,0xA9BA06D0DBFC62C7,0x56058EC51BFD727E,0x10AC7B16AC66,0x68C5D308E692B551,0x3F1028AB40C1E7E3,0xFEDB13CCB50514C,0x5AD9BE12FEA70FF3,0xDD86F592FFB26EFE,0x3CE47FF60E3F66AA,0x105F527B0202,0x978AD4596DED992A,0x222AC0E1A6944BB7,0x91A2FC1F4FEDBB5D,0xB30539561F08A7B3,0xF8DABE0E8F5D9694,0xDF57E3681D3563D4,0x8A19AB5B20A0,0xF15F8A8D2156E89A,0xB98F319AF99CA9FB,0x2BD96964CEAFD5D3,0x4BECA91813CE3AF6,0x9338027D9F8AD664,0xE8E149A477C40812,0xEFE7212E30CE,0xB81BC60D07102D17,0x9BAB3EFC559D1FF2,0xB52806E97F58C857,0xAC37C9BCFAEA5252,0x29C8E4D90CE26A92,0x20EE1827DD97CE25,0x2A14B8BBA097,0xFD425E44482338D4,0x53D2947F6BBB456D,0x64DAA584D1C2AA9C,0x899B8ABDC8ECBEA5,0x324B2477FB55888A,0x4C6332B8D8E4B9D0,0x1CF9612C64174,0x83E072F0CAC436E7,0x405FD8145E960EFB,0x56C3482918687DF1,0xF1A9E74C025B9B33,0x24F63FB54AF7FB9D,0xE331F4A7A6AC4AF1,0x1E767CC4ED924,0x7BF76849CAEB8544,0xE1208E004A400DED,0x1908F1A1FAC2B9A6,0x5E034694609BD7C6,0x2FFFFA47A5E4BF4E,0x33486372FE6536C8,0x14497ED6FB095,0xF154C65B6C275AF9,0x4C534B84618800A9,0x36616CCC466FAD25,0xE07457878250053F,0x9A47EA287DDE080C,0xB17C6DF83B886DD1,0xBD4CD80B690D,0x8E92A9239289BEC5,0xAE9C47A5F3075CE6,0xAAEDC8C44C1C235C,0xDBD93E0F8707BFBD,0x820DDE891C1C9788,0x50462C5BB24B7B51,0x159866821E9D6,0x78AB72A2C6E0D1D1,0x8FDAF5D21C422510,0xF819E1D3C30245BA,0xA8CA110289EEC956,0xFD784AC9440DB1ED,0x96816BD864302962,0x3683B4648B4,0xA16DD045F01598BA,0x1B1C47A3FD7A5AEA,0x7EA0800342314C1F,0x805EDCBC4F7B379F,0x48077F966D0ED447,0x1A1B7123C6AC1F89,0x202D202F47F6A,0xA26D12DACA81ECAF,0x8C42F40677D86BE1,0x934AFEA3E509DAFE,0x98672E134C4CE323,0xBBF354DF0A96D5DC,0x9192634BB41FC7F1,0x1E4D253F11BA2,0xF1C3F9D975E757BB,0x176A3593C8113C08,0x92A0C734D0514F21,0x1D4205DADFA0067F,0x1F4FAE047134EB56,0xE8F9047588B4AD28,0x15AE0E4A5B86F,0x6B13C7330127D5FF,0x1628D39DBCF45B08,0x5061EC8B17C5A44E,0xF2A79EE2D54E1B17,0xDA78C3756398BF86,0xA28BD539340C6994,0x1FC42DD49A70,0x3974CBC17E6A1384,0x267BCACA0B56CBB1,0x88EB53F2F5208A62,0xBC0A673467FDC4D7,0xADE19CDC3B6A0459,0x7A1442313415047C,0x2FF37CA7CBC3,0x289E2ED5F7117CDD,0x99A62FE9B2011CA,0xB064B46988B4025B,0xCBDC2657225FE8A5,0xD15256EB546C6042,0xEC032740C176C556,0x1A124D7FE94B2,0x97DAF7951AA325B2,0xECECD8AAE30CDCEC,0xEC90E896AEFB72C9,0xB2D812D1DA7FF89A,0x25BCCBEDD09159EE,0x5112B516BD10D74,0xE323AD18C05B,0x18B8239D92E6E721,0x8121EB1749B44B95,0x837968F8E27ABD8B,0x3B8E4D4C0D0213E7,0xB3D4090B9814D20F,0xA351D90AA32A56AD,0x61DA8F57A3CE,0xE9C657BAE8C388CE,0x663089B223407364,0x38226E15A2DD1F95,0x594BF04805B697E0,0xC520A86DD322E87C,0xFF286B520F03A367,0xC457B7D5B332,0x7AB47C7DC2D8CAE1,0xACC74560CF99463,0x69AE6898499560A5,0xEC83A014181A4F97,0x2AAF629F250764C7,0x27C5BE99C310AC68,0x1D47A579711BC,0x7F44A99726BCE788,0xF16016A706E11CB7,0x71543F576565BA9B,0x13FF661C5E248E7C,0x28B8CB5A7845F0C8,0x599280A3236C95AD,0x1834B6DB5CA8C,0x92CE6C03263E1AF1,0xB793B45B20B635A,0x5768FFE68A242EB0,0x2E8F6437BB7FE37E,0xE498D0C4FA9B873E,0x71637DDF47B5F7AF,0x155A84114E7B0,0x46B01FE0C12C4439,0xD68CB66F23BC6052,0x62DE66ACE0DBF298,0x9034E2D0B1639E92,0x78D57510EE99AFCE,0x41FF377E391394D,0xAD5569D73086,0x74E5C68FF8D409C,0xA85A44A437338BC9,0xC11205338858DBDB,0x4705292DA0B7AA1E,0x1056B9429E7D21C3,0x3107824D7AE9182F,0x1BCF327FD1D24,0x2335A28843BB0C8C,0xC70FA0D8F35C726A,0xD318CAB7099B3171,0xD4A8C264021CC7AC,0xFFF84BBAAEA931BF,0x1D6CA1FE80C3EE41,0x1F61BEE9B9D3B,0xCA6FC2C9276E6718,0x2533F9B4F77F856F,0x53B0F548E4DE514F,0x2B0F3EF23DEBEFDC,0x782019AC1CF09276,0x840A03434FDB1367,0x7F54FF35A5FE,0x7952158FA4ABC4D,0xE651DE057D882E42,0x31645885A010B122,0x18C0AD07F7AF84ED,0x5598F85A8AF5A79F,0x7FA6A9ED2F7B67E6,0x3B385C69B9B9,0x22792FF02465707C,0xAAA24A5CCA38773B,0x4EED2F07DF1BEBEE,0xAA29EFC0680222C1,0x45D3E8A4CAB935EC,0x749AE2F6D69D5DD9,0x1976700A8DA2,0x8DD65119657873E4,0x49E4381F7897FC77,0x25B994367AB58273,0x13B1178EA9687F2B,0xF076FA8C1BE691D4,0x1A78E438AC095973,0x98AEE39BB700,0x69A7DB68955FE8CB,0xD3DD60E34CF07B5C,0x481F1B1008BBC796,0x11DE3E6E4F5AE4D5,0xCC5511BD54DCB9D7,0xEE83727AD5293D97,0x1C8F16C1C1C54,0xF2FF8E6D5085DAAA,0x679B5E171FB88DB,0x6D4F6531D5DBE81D,0x53205659E63B0E8D,0xFA60342317DC4F09,0xF4CCF522FCF50C94,0x217A6C1B8C13C,0x90E64050D3C794AE,0xEBA2E0DF66EF966F,0xAE46BF87A1DD32F2,0x643033DACB5F04EA,0x1A9975EAFB2A63DD,0x76F8E3F286C319D,0x16BD40FF55A37,0x765144A4B4233677,0xBE38135ED1C72E32,0xB48FE3529EAFCAF,0xECC81562F0C333A,0xCE016868671C26C9,0x6F58324D08911C7F,0x1C388F40B37BF,0x172A4779D40AA171,0xDB2C58A3F056B0DC,0xBF1ECDA696B3441A,0xE83E03D70F68D285,0x162915D3A2721D6A,0xAE5DC64E846A9882,0x70EFCFDA3F5F,0xC0E73D459E6512E8,0x3C4B3F221759926F,0xAB9666963BE9A481,0xA817C06067FC88A4,0x864175C4A020AD76,0x9E84FC3E6F225970,0xD72C109AE143,0x3F48D690F1D98791,0x505A1ED68320502E,0xFDE9D756CCAB5E05,0x1942FECC09E21CF6,0xF41E7647C77CFB46,0xEC31FAEF3E584920,0x191A987B33024,0x780986658F546AF7,0xAC27E5B89E641E82,0xD5F718C4D22B3B61,0xCF55EF2F46804FBD,0x4161753260E0AFAC,0xCBBA21419F318D8,0x7D15494B896E,0x3F8493E7A8F2E8E2,0x5CEA906E421BC420,0x9C579F75B48DAFA5,0x8760E85249544C26,0xD23F02B127322F39,0x7CA6685B2FE74A3E,0xEE302FD78DA0,0xF10FDA99FDDFA2A8,0x6549021F265AF50D,0xB03A864187C721C9,0x840E8AC91806388E,0x1D3E4F40815DE86D,0xFE044D815201B5BA,0x8418AEBC2248,0x8F4A3F4B440AFA7B,0xFC3F673A4470792A,0xC20F4C91672057B6,0xD55C22362CF234E3,0x5BD647B98BBE98F0,0xD6C3C351B4297957,0x5F9A8BCBA49A,0xA93C4E31E251CCAB,0x636B3F23905443D,0x193AB1F72D3E0E84,0x88D66FED9FFAF055,0x65D0A21E72C31434,0xD38BE6116F552071,0x1E0E4CD3FBA25,0xDCEC3A128A1217DE,0xD32335B3C577ABC9,0x2D01FDDB4849734B,0x9063F9F982DF9BB3,0xE2FFD8BFA029F255,0x1FBA6ECC25A10145,0x21E994ABF71BE,0x786C8B22E2C511C0,0x3AFE8A73B058D3B2,0xC915B0840F62C97D,0xD2C3B58934AE3F7E,0x1F13AAAE91E64503,0x386D0CC3BB72EAFD,0x1C3E842C7552,0x972B2674C97E9701,0x3E914D548D2EA6D7,0xB66A4C45BFB17029,0xA5AAF8B8CFF6F42F,0xE8614F5F715F8539,0xAA3078966D7698B6,0xFB7476B1E481,0xE15C6D5BD645CE6B,0xBC752249D24FF5E6,0xF96277795F82D2C1,0x375CB035C8B78D05,0x2766BB982FD7065A,0x1BD4805C5B49C98C,0x215549A0F30B8,0x86BE67255D3E6ECE,0xC990AE7C1012A4E8,0xCCCE9338597472FE,0x415B910E2DA22547,0xDE73E59EF90E8221,0x64F58E5BF95EFF96,0x1D0B9CD8C2AB8,0xB2483AD51EAF1DF6,0x74D0ADACD7F65791,0xCBD4DD5C8B9A5638,0xF9B74BCFAF318B66,0xEB1A7BD7103047B5,0xFB8B2C26EB526468,0x109A878151FBE,0xD5F8930669741E27,0xD42996CA09054D80,0x9EFA220F69426557,0x58413140F716ED1F,0xF7FD166E629E8B14,0x372BB27ADECD9FAF,0x696FBB68C776,0xDF3E4441C7BBD91E,0xAA427987A3C792AB,0x2981789041E39A63,0xABAE04114188C79B,0xBB48FCB317A706A8,0x57AE6CBEA8A98E21,0x45AA45F4C2B2,0x6E9329AE6A18E91A,0xB82A6D227C16B8E1,0xA5C3DDD3C394500C,0xBE5254564619BD76,0xFCDBDC7B060EB03,0xFAAC80075286A1FE,0x7B880066AF7C,0x9850295AAFECF554,0xB42A40D63FAE01B5,0x5F0A866010FC00A3,0x9B714F12E6AC3C2B,0xABD959078C77AA75,0x722DC6351D15EF1A,0x15FAEF15F485A,0xC0F8B1E9D27D5F44,0xE2407DFB93FA9500,0xDB4E11245994F204,0x708AA2C6C4509AC7,0x89E62248CFB5BDB2,0xDAD9780F8C8BE912,0x1B0DC22F57F80,0x635FA45713CC9146,0x8690AE1DF2BC66AC,0x596A298A6C3171EF,0x6F1B60D2D8E52100,0xD464C1B44FAEFD87,0xA13E2FBF7D88DBF9,0xC070B5E5F2F0,0x871AC5E3E2CEC727,0xF883050E5618CDEF,0xC34CE77987193D28,0xC72258569B80A9B7,0x879C1340E51BACD7,0x93EFACCFB1CD79A0,0x139FFADFBE3D9,0xB02E003DA6402F9D,0x136DA31DA4B48480,0xCD808491F24C727B,0x88AA8A314FCE5A8B,0x8CCF093A6B780973,0xB1AEAC8891432D0E,0x40CF978D62EE,0x930B3D73905C7F32,0xD351B5FA8B62DD0A,0xC594F6C3F9E2C9A,0x618D6C0C20DA14A0,0x32F0A6617DD1157D,0x4272EBE676C89A21,0x23211D5C165BC,0x91B0412EDB225148,0x472DF6B993B56792,0x33B301741BBAB3C9,0xB6404D6B5E86DF03,0x75084F8A7E08D2A6,0xF4F7EF658E6B9BCE,0xAC51FE56B22F,0xC7A9A6AC824367C0,0x1C17C05F872FA6E7,0x361ACA6415D0CC78,0xECDAEB01B2E1B768,0x3E3E831FA38F8106,0xA31B47CC69C94F88,0x12DE94FCB4512,0x87D6BAB67937BFB2,0x825CC2E0DBC393FE,0xE0DE95D420FA2F2D,0xAA5C18DB7A51628B,0x1C0FA1A9F10B301E,0xC7754BA989B19845,0x11EABF045EB73,0xA170789EEE685EBE,0xC291D01B9426EFC7,0x513D097082775CC2,0x9D38A0BD9E4E28C9,0x4DA983DD7FF4B63D,0xB494ABFC2E6822E,0x4399B9EF85F9,0x47F2B5A0FA7750D3,0x2816EF2E0BBCE1A1,0x14606C2DE9A94180,0xAB830DFD54EFC156,0x1917982A67005744,0xEA88DFDA3DA7A71F,0x1CAD0BA24371D,0xFE7C0917E5D99FA,0x41A514E8091B8E52,0x675872BE2F5F64E,0xE0C8F0D4E9F29279,0x4A5FA14D6F015856,0xE64208A22CB19110,0x1E5F35C60C233,0x4253B0E89882750,0x18B52DAB37DDCA7C,0x73138C99629DD117,0x34166A5C01FEB3D8,0x66377950B22A2872,0x325E011BB4D01C71,0x127EA378FC78E,0xAA6977B5B9EA730E,0x65304D375FD8D7AE,0x807191B726873FB9,0x8B12F68C809247A6,0xF03DDC4833F2BE30,0xE527F7615178EBD1,0x21E622C1D6090,0xEC1D361C7A890C34,0x5CE698C901CE3FBC,0xE8BC75D40DF61376,0xE7C43A2217820532,0x5E979EDB4668B9CC,0xB9E9F20A2C270929,0x130FFFAD19D25,0x4C894EDD11AD8CA4,0x9966F40801879111,0x47908D99215D47F,0x87CB2812772D1E32,0x3A849D8F454A7C38,0x5C0D0786DCD9608F,0x1126F8816CE4,0x336A23274A27758F,0x1123B20D112257CC,0xCD783FEDE10ED5CB,0x995E8C269863BE7A,0xF53F3EE07883A2A2,0x52F4EF1710EF3301,0x1BF06A53B8455,0x9EF9E61D120D2FD4,0xF4B381F014A7F54,0xA1F309735DC79222,0x742F9598CB2DE5C,0x4DEE0F7571F05488,0x80C287D0A708CEB6,0x1B54FBA6C0D93,0xE8563D6716EA3FAF,0x55A5EB5287936190,0xB3C6574FEA0CF38E,0x98EAEB2B181DCA36,0x82F40CC3C9EB2F99,0x836C38936B8297F2,0x195C3B3FD7696,0x54D500B00814AF95,0xF34A06783C5C77DC,0x40EB5BC84E3B3BC5,0x5799EC2DDBC6BF0E,0x6B4C3F78216A66E6,0xFE156EF85DDFC159,0x1937D6A73A3F,0xDCAACD3393CFF964,0x1C1D0BE65590504A,0x37EAFB91E5F60BD9,0x78E36FB614586402,0x30D11CDCBB6466F5,0xC282A821C4A11F0C,0x2945B30D20C,0xCFA87F5D23DD3DED,0x6F650324E7BB63BE,0xE9E5DF59DCA2F2F1,0x8C71C65D80253DB0,0xD07D5143EC6DD193,0xD21E6003EB03F127,0x203DF40CC6B93,0x54445960FF5AAB75,0x48EB7212BE9CA9E2,0xB3735DE56593D725,0xB9DB0678FAFA6DE7,0xAABDB1564F6CEE7B,0xF3DBC01AE07A9FC5,0x1DAF2C3A89724,0xE86D050B50BC6955,0x8673D6B5A882E00,0x751EBE97359B7F35,0xD83437A5CD119339,0x52A581A3B86CF2A5,0x29F4D58CD410F170,0x49AAA15D26AF,0x63BE3A412B5D12,0x601BB9545DA86433,0xD5307F699D6453F,0x8F97813A4D4E5564,0x88D4B75390B174D3,0x6F2564C2063A89FA,0x8A821FA61FD1,0x683B632153C6C3A7,0xE8F0ABA49115275F,0x595BD1419E2A49C3,0x9719D2E6DC959FF3,0xCB4A3748D3A99745,0x82BDF0347CEFA65B,0x17A4AA693DB19,0xDE6D1170F68D1753,0x62DB796FF24E5A84,0x860500D0FE2C57D5,0x371435874B5E5FC2,0x640E5835580DBDB3,0xEC8EAE76E0F1D9AC,0x141275DB3F6A2,0x20D47A2CF5B59F6F,0x7513D8ED9F485F51,0xB2B71D681339714B,0x4BF615D642C1D7A9,0xE55C671DAFF0D0DC,0xF3D46F1E942F63E1,0x111936874693B,0x277486519BFDFE8A,0x93E0A713168F865,0x212D6E1484C77E5A,0x233750831FBDB32E,0x2A74B93344E0D76F,0xD46B4D5755FB0AA6,0x2D1133C6821,0xFD017B4D00351139,0xEACDA4F8CC8ACBF,0xD63A289F25A0773C,0x8271C326A55E364E,0x9AED312DCACF4139,0xB5688D216CD952D0,0x211178C25787A,0x49B91B69309F0481,0xBD4A973390A0A032,0x4547C9DB90854819,0x66F5B337FF2A3725,0x7E24BF7FCC160DAE,0x26DEADB59FDB6213,0x486C1744E712,0xEAD7EECFC37C70FC,0xC4082617FF141D4F,0x83A1EB82CBC793DD,0x7C8584B20C8CF66A,0x6E6EF8E5FA042228,0x39AB8520FB1705BF,0x199BCBD0F6E8,0xB8B315BDE44F8722,0xD483EFF6B5B578C0,0x7201C0B040B9EB90,0xDA354460E04C33CB,0xCD72A76E042EEFF2,0xA8E0BB76BA6700D2,0x23312E5E86EEE,0xA02624AF5164F7BD,0x44B49B5914B6CF2C,0xAB9BEF84D559FFA1,0xB620E606B33A2E73,0xA0B6FFD44930271F,0x36BAB20180DC548F,0x2CE3AA2EBACD,0x50E20EE7001E2C06,0x2CC2FFFB34FB1A2E,0xC7A29E8B69A6A8EE,0xB4E67D1BD717E1E3,0xA6CF060C0EB9778D,0xA1AABA7E516CCE87,0x18B48809181DA,0xFF39D48809324E6C,0x96495B563E333333,0x497DA74CCB982F45,0x8EBC99C1EBE619BB,0x5D325E6801CF6E6C,0x5125A711ED8940F5,0x1F56C66B3C8B0,0xDE2254DFDD00CF3E,0x52AECE48BBB14221,0xE9D7EF23BD7EB884,0xDF03B7126660549E,0x7E40DB751E40BFB1,0xE3E2DADE2D49C974,0x17141DF9A10D5,0xC3F1903A24A65A30,0xE363F1387C00A879,0x8E27B7D72383DFB8,0xAE20C1524A4A1AE5,0xF85773EE4E313CF3,0xDB34183C31F949C0,0x1C1F2A8968791,0x1B9EDF9B892F7C54,0xF1A289967B629D8,0xD452191BF073252C,0xB6D66863DB9174D6,0x627FE69E99E4325,0xFCC7292E25027917,0x7EC08A5F5DD2,0x8E32F5695FE42B61,0x80DBD0476DB4EF87,0x4DF9C3C88D2E2733,0x34F414C97BE3BD3D,0x352DD8818639CE25,0x9D7FCB8A30F5B895,0x557BEE28E965,0xDB5F411AFF826F7B,0xBD752B641B004218,0x9FB258566D44BA45,0x66E477A33BAE5DF5,0x476DB1E418F98852,0x42ED10283E3B6483,0x18F02656A8EA1,0x71E0B1D5FA5C579C,0x6EE7E3373F725E82,0x1859EF3475678A35,0xE33355F44F55FD08,0xCA401AA86BCD89F0,0x7B37BD03D864925,0x20C9591B36642,0xF328F375B93D94ED,0x64DBC4B37935FBC8,0xA4F35031FF93833E,0xC9994BCE6B9C1C37,0xC1BB424715475B0,0xBEDD0C90CE35B1D6,0xAE75B5271115,0x7DA09F57B67B3B69,0xB048A6A3F7CD4B01,0x4CDA946395D9F2A0,0x15735732CE8CB482,0x894E6CBA4F5A1C9F,0xC454D0F46DAA993C,0x1433088A5E8F5,0x16EDD231121573F1,0x3C7228E46C445E18,0x9395ECA32F79EB9D,0x10DE17B0ABEE9969,0x329E7C7B2579FDAF,0x402D27D40F43D59A,0x21251EB60DAC8,0xCC1D9D208220D11,0xFE776C05BE7B49EA,0x88547A5003CAEB4E,0xD1C81DC64DD018A9,0xD372B9EEF2AF7A3,0x528BA68E73BC2C5E,0x1D8D8F5452A25,0x8B9693D162106637,0x860D40E49FA44B93,0xB45E03FBB0B6880D,0x9D8B7A97620345E7,0x954E77B1EBDF387,0xE6DB7A172A2E10B6,0xDCBD2232C20D,0x98CCD6719AB5522,0x2AB1823D9F773F5A,0x8107D1149B1AADCF,0x817292464845BE91,0x8CB812EE1AA0EF1B,0x5FDE8367B5A321C9,0x10A9EBE7EB1F5,0x998C4416B4CDCE74,0xCC8C7FB1A5BCE072,0xDD02D529159FC3CF,0x79243D97CBF3ED6A,0x5CADDF2C4C48687D,0xCE54ADA1651E9366,0x1702343B19A40,0x7D0260E0AE11DDA3,0x2C06C997E59B4134,0xD194E028CDAF04BF,0x484F5114D18506A8,0x65452147F4C9581F,0xCCBC443EE98BB3D,0x206A41B99E1D0,0x522E2D51AE951DCE,0x8B56D118B43F419F,0x3E37D7E9BB1AD1EB,0xE72D693CA698F2EF,0x72CEC13AD33E6535,0x58688A7B67245523,0x847C33634AE1,0x364000D4D4E7001,0xD1468C7A0336B225,0x269119303C8EDD45,0xE03096075F011BDF,0x89D78CA210D38B4A,0xA4640CC54DD85CE9,0x2232C24CE7099,0x4515E53B1C86875,0xB99D8ED4395F553A,0xF8C1D45FD5BBCB32,0x72B5B3D162B8FF33,0x97EF897AE189AEBA,0xE1B3AC57274F04E4,0x6E373EED4E4C,0xE9C4A032EF0EEDC6,0x2F8556A4BA269A66,0x2DC1202BE73ED028,0xB1AB548ACC01BC2A,0x6B9C9E0CDBA03C06,0x1D41F3016B3C6A94,0x925C716F4096,0xBEB656B25DCA9C22,0x6545BEC3CA212DE6,0x669BA147AD23BD88,0x475FE8FDA073DFC1,0x7BB8001812419FB6,0x6F86C73105A770AB,0x6B94DB6FB92A,0x8526A1A1092AD16,0xEA491C6522B8EF68,0x30FAA7E486EDB08E,0x4B5D2FA02254BA40,0x3CD74AE0A1867200,0x7821567E58968EFD,0x31265A358695,0x99E75DF32CD549FA,0xF89973A411A82AD4,0x61C0A1B4A7B7A5C6,0xE464C9E614545A4A,0x1EEE7F0D7719685,0x9709776EF0D36600,0x1934F7E3B7FB2,0xCCBFA8FEBD42C886,0x53A47C00688D65BB,0x84687B17CF2959A3,0xEC6B7F8E9A3D8C76,0x168415BCF2D2C9EB,0x53E022D143C2CD06,0xBF945126E3F0,0x186672174DF62E12,0x6A0E2520D0222785,0x32E01DD296C33872,0xDE043FBCF8B14D6B,0xEE232061F3DFC6EF,0x801C132B2729C7E5,0x1FB6C9D81931D,0xB30FA9146F9B82F3,0x82CF48A64750FE45,0x6CD678EC616A923C,0xD64FD3D5BFB76EA3,0x382CF64B4838AC16,0x1E0C4C7ED30B777E,0x2166147555E3A,0x446E623C50964A4B,0xDF88BC7AA4AC07F3,0x2FAC55450BE56B25,0xA577A8C300C181FF,0x2DEA04FED471D802,0xDEDB2EEC8B02E025,0x2A0E8B5E1438,0x575CB3A82EA7E973,0x2159D7D597B7C04E,0x72945394923E2E5E,0x54DF63004844A56E,0x1F4ACBD0FF66C76F,0x57F27BA958403D3C,0xAE52B2201BC,0x729FCDB137C072CC,0x183B1C54FA56EEBE,0x8F7F249F0F8840E,0xBA3ACE9577F29E2C,0x1491A104B0243A21,0x81929ECD7043521A,0x1835350FDFF64,0xEABB5B11E578BFD8,0xC3FCD2D4D7754CE7,0x5182E783A2777945,0xBCC93B9AA62A2025,0xE44D39097A14DB44,0x9D6A87332DC4F0FD,0xDEE6C41BA6CC,0x66CD22B5AEB13846,0x2ADD1658BBF4022C,0x473D392F7D88583,0x183E57002020EF05,0x4996D20FB031BAB7,0xDE5B1733D945F430,0xC3767432F1E8,0x6C6FE341ECA81B95,0xFF8D1C8E58CD70E3,0xD745E2289AE73B3F,0x3D1AA99D220FC798,0xF751DCBEC73C4B5A,0x3D23402D90770566,0x19480E46DA256,0xC935ACA301CB26BC,0xC71BBE51FD3403C0,0x8B404D75D67DB6A2,0x73E825B095250BBF,0xD9FEE44CDE0ABBB5,0xE677EB3FFEB2414D,0x1CB6F9A2F8056,0x9B584DCFA6CF4E85,0x55A868C72C852F5F,0x8AEDD96C1AB1E2F2,0x1326AA63922B267F,0x8C2E2720CE784887,0x502C665FB97D0778,0x24E23B0CAC34,0x3E4F7F5E0741C16C,0x5B98C66B0E3BAE1D,0xC68464ECD2A12E9D,0x51D1A7A2B96C5F05,0xE4B2D3E4735BD69D,0x18AE16C4300C1FC3,0xE88B001ED00E,0x6F9F1A6A0893FF6D,0x1ECC5C93CE737F3D,0x3009CB937ABB34BD,0xA3C26E927F981A0C,0x6965A2C204306C1C,0xE35413ED8B550DA0,0x23C8AC1599D3,0x754B6E36893D72CD,0x57603520E302D03B,0x3A436847C9D20CC8,0xA09ACD59E5BCF847,0xDE29947972DFD9FA,0x1830FE12509D5B05,0x2B87C9343083,0xA07F3559D74C3506,0xAB5324FCAA0C37F1,0x1B69EFAC9B386DCE,0x7710772B70183462,0xBB886C7343E2AA65,0x852EAD70F898AE25,0x1649023E22174,0xE5D039AA41AC81E5,0x6C199A9F32E8352F,0xD0EC323CA17C17E9,0xF85C5918F13C8745,0x23F83288D08901CE,0x9B47519758D25FC1,0x1328B8618D156,0xDAE53DB63EA6C876,0xC7B02ED386DBF867,0x2101028A9089C2F4,0xEA4E92E356D097D4,0x9A1A2255F107421C,0x4AFF4F8612F8DB16,0x15F54E06D8659,0x23CAB87569C4A7DE,0xDD464417E560E457,0xE8DFA358CD1182F7,0xE741B0EE9AEB2656,0xEF71EADFF7C71672,0x987902CAA437C6A1,0x1B930088EF38C,0xC19B46143828341D,0xBCA58ABE9B58034D,0x3CAFE3696387BF25,0xFBDFF875824E3E96,0x8FB4FCB050CC464E,0x5EC3AF169BF60EFE,0x54B0A710E1D4,0x66B2D71D412B8B90,0x75FB9B35A96FF364,0x4A41E7D803C1EA70,0xB442FDA50F47464A,0x95CB655B74F4EFC,0xE2762804A48B199C,0xAE8FB19F08BA,0xBBBC57318C40EE38,0x3B2F9ABA67F1715E,0x3350110AF668FEB9,0xCF66D80E08FEF3C0,0x6F75CE363116C8BD,0xA4EE479618C88353,0x2B4A5ED89A7C,0xB7484DC040DA8F83,0xB75ACFE34B3A7AA8,0x1956332485AE28B0,0x2B99A37A425E6421,0xE1DC5F9CF8F7044E,0x74C79C44E6E159BE,0x18C9681AB49E,0x5225BD883BED7D2D,0x20C2679D07DE313D,0x835E350129CEB477,0x297F2045416B3A55,0x29A316EF622C7E02,0x8641CBC3FE6C344B,0x77761E14B923,0xFD7C1004DF5E48D1,0x1E23DAD127BB0502,0x71346BC587CADD07,0x22C67404DEA995C4,0x873A517E769EC5BB,0x4C171F67DCDD078C,0x19147DC1C1227,0x27A80083D14317E1,0x240A41A7E1BD2A37,0x79D1FCD075407399,0x714CAB4801C29F6C,0xC7A6D14174FF641D,0x7D9EC7F8FA86048A,0x140527E30E434,0x29D4B2E1E17D7B84,0x50B5E6F4CE467103,0x80A35480DA7B3064,0x7B386EA9050CCC49,0xCA6CEAD20C69B168,0x7C698DD480EB4F02,0x1179D1D824946,0xB3B4B953B33A6421,0xD3539C35FB007C0B,0x3088628DA9D1C5D1,0xC15801B2F093C909,0x2A69FC067A01D86C,0x831B1D90A3CD6715,0xBB8B7BF2F1C9,0x362415A88CAAF813,0x599C275EBF585EA6,0x566443304B184C2A,0xAACF19AADCDE548E,0x2FC31249E4EFD541,0x4943B9E912D817CF,0x9417F1344643,0x19A20EDDB6F8A372,0x21628BEA52749860,0x512DC1C5069C06F4,0xB0D9A24E2540B14B,0x1F56ABD0A4A9BD4A,0x575C823C12A6A4AD,0x7BFA5A1CB962,0x115BB9306D93CDB9,0xA372BF6450397744,0x6D9707F65E58FB23,0xD6D748A9945488D3,0xE01E24A50A680097,0xD109CA16C90155FD,0x17C3355EB890F,0x49CF425C617200A4,0x901AFA9F50BA8F9,0x1016561509D7D78,0x89B24725DC851889,0x42AA720DD4178155,0xD683E1EAFD7B5C31,0xAE0E9F4983E3,0xF86BE32EBB1F903C,0xCE4DF9F435D4D8C8,0xC172B75FCA838603,0xC99518AC1EBF9AA4,0x222D02C4EBB4D106,0xFC7568A16B6C8FA8,0x13855819F5162,0x1FB7AAD0AE918EC5,0x79307E694D0420C8,0xBFBB5C4AD52D8856,0x76DD5D194A821BB8,0xDBD4D57415DBC2FD,0xDBB86CFB126DD3E8,0x1B90FBE352C47,0x4180170BF187DFD1,0x3D20F7A2E5731F0C,0x2B29B79096A9A245,0x442A49AC773CBC4F,0x8C9CF9859B34BF82,0x501AFEA57BADEA05,0x129D4ACA0106,0x178F98476D8AB26A,0x9DC62E679AA869E3,0xDECD81B5A2FF4287,0x8DD6832E6B6EF36C,0x8533453283620828,0x6A84A82FC6B3D1B5,0x197BAD1C5D477,0x93925BD16ABD042A,0x5B030E05B9063C53,0x824A1249B247A538,0xADFA2CAABAF7C92B,0x2E7A521CF9AFCEAB,0xBCC6B69EDDC323B4,0xCD3AF077FF29,0x53A04F6B25F6A6C8,0xC4A36CBC920C9A2A,0x4537BEC88ACAB5D5,0x4396585634C774FA,0x2388C0567C6C115,0x8960F3C52683471B,0x2159BABBE40A1,0x8E181A54A3EAC58E,0xCC830D29DA4B31C6,0x6031C923DBFAE762,0xB91C3EC32F9D5EF1,0x8A604191C14B20BA,0xE6896FC47296750D,0x721A410246D5,0x3ED39D51DAD842BD,0x4ABBE270659232C6,0x8A7C0E0FFE61205B,0xB8868DC0D026C4ED,0x328265C8CE6C19DC,0xB972C30EA38E6D0,0x122E113BF4DF2,0xD5410541D093064A,0xB70528420E86171,0x33365F520D16096F,0x936F54BA7D586F9C,0x3898A047C9023797,0x28C60F3579AED73F,0x138F0125D5C95,0xDFCBCDFB7C4805AD,0xB43E448B7E3E9DCC,0x60F9ED800A37E9A3,0xA03CEF6F6A56CABF,0x78629FDCE61E92A3,0x96A258EEBF131711,0x17E4757DEEC2F,0xCC10D4313A8135DD,0x75C5007A6B0DB985,0xC7056D9325A4B75B,0x1A1743AFA537A6FB,0x68F9031402CDD77F,0x209AA4971C2F352,0x1959DEB5D742E,0x7C0C76D84C26F44C,0xE1C1E195D3933817,0x14FA1E6C340617D5,0x3F638152E4EFC99A,0xB430912D9A805B00,0x9B9BA4847FFE9604,0x1C0449D2076F6,0xC7275B149DD1A5FE,0x6B24F1CC50DF7972,0xE286B78322F22C81,0x9946C35A7168E865,0xEE8E4BC759E7873F,0x8A64BEFB0A7F9A20,0x1CB18AA039C25,0x6D28336C58F73E30,0x725B50C04460EDC5,0x5CF888D713FCC1ED,0x2E54BE3E35D2DB2C,0x494295D0E946CFB8,0x7210AB8A084C7982,0x42FD9B30D5F6,0xEDA5D34BD1AAE0DA,0xAA56DE87CB3C6ECB,0x44D78A51DD85F9AC,0x807A46E94372867B,0xB81452500ACA79AC,0x9B390708D4F3BBBB,0x21366D30116E5,0x64184C2240999BBF,0x4565AB187EA5B623,0xFE53D23A14B3DC65,0x69C462B66449CB92,0xB55F63716A7F4B66,0xAA313D1E702C47FA,0x20B4B0C6D5E06,0x28213AEDB766D70A,0xBC56B59ECADB8DB6,0x345EAD5E9CB094E0,0x701D53B4419D8B16,0x726F4BA122BDB4F2,0x7F6277F789489550,0x147F55DD5F077,0xB5DABB19882D1CA6,0xD0653EBBA634DE3A,0x680D1248F27389DD,0xDB6C05E99BFA018E,0xBF3F75DB70F8995F,0x95C9F02E20BAE066,0x11D27A38FAC46,0xE5780B696D0047BC,0x7F2326E9BDDDF8A7,0x9603717332134044,0x7D4EBBB5D346896,0x532D2176AE4DC1D5,0x44A1882B27F3F864,0x1884B6C3617F0,0x6869618B986AB9E3,0x6AF03C28713E04B4,0x34E132B80B8B3F37,0x303E4DBD03F8A262,0xF1AC57DB3B80D94B,0xA1593F83270A71F,0x20FD628C05C6,0x195D43C00E0C5A7C,0xCFE78F2E07CED9BF,0x72DF12D5BA3F6638,0xBA30C462C8626994,0xB1EC4DC347193829,0x1D7F028D72B50CF2,0x903AB5EF5106,0x7606F8454C62194,0x46A4C41BE40D7849,0x39CE9CAA07070E7F,0x5C8FB682E872F843,0x56A05C134BAEF6E9,0xE4C90ED6FEC5AD4E,0x18B9769DDECF,0xB500561B57EBA29C,0x4D890A9534362A6A,0x54AEBC2DFFE806E8,0xC7AE4B3426DBCFA0,0x67CB4E28BD96981B,0xD1AEDA4BA1B1FB42,0x2228B040C2C5E,0x67CB873DF8A2C398,0x8B29E9790941BF7E,0xB506676E78C68431,0x76DFA9A848602182,0xB2651498D5519975,0x35771E20E10726F3,0x1135D89F893E1,0x6711E4468F956C38,0x6C66504A6FADBE92,0x620A2748025FE86,0x4351E917E9653395,0x6993BCA968EAEC3,0x4636DC368EEB8338,0x2E4F9A43444E,0x2688274E05EC15E3,0xC30290D7C2FD8F2C,0x9C3B86771EB6ECC6,0x5E42E53351BD3D1D,0xADF69AE1CF78FC3F,0xA75229CBA120E2C3,0xAA81D63AFAC9,0x56FB8635AB73ACBC,0x7B08C4A69B97F485,0x39BCE756FBF1BDA2,0x73C83B07755D36B3,0x883DC07CCBA9C937,0x7637206E603F2362,0x22EEEFFBDBAB4,0xA34FB49D5C921E6C,0x15D9C3506829EFFB,0x8017067261B8E6CA,0xBDE7D62EC1A1B5DA,0x9092286EB292FB89,0x5E1AB75A3CBBDA4F,0x103938A4B5459,0xAAD89605EF6DAF0C,0xFF9BE7486AF576A4,0x49C7767ECD4B434F,0xAB32040DAD7C4CF1,0x1080A5244019ACAE,0xE49CF12D166A85BD,0x36E406F47997,0xF86E6720378A8D76,0xB6C618C98EB2D0FD,0x595B15ECFDE582CD,0x9FCA16F68D4BBEAC,0x205478615AEB07E8,0xE59A63B3A2976B2E,0x2270C4E8162E3,0x9713C6BC945DF6A0,0x6AF51243CBB0BC5A,0x11A4B45CCA01545C,0x17EE1491BFEB3D07,0x954F158CC7A53D17,0xBF90F17673C96F75,0x46C02CEDECBE,0x71646DA3AD066784,0x5FF729C48A433944,0xD5D299F33EB51832,0x5A09CCB6E6729F56,0xD9CBB96E478F9741,0x2F3C87064670CFD8,0x1E98A0BA38AE1,0x6BA17E7A8F4A8960,0xB5D0BB0E76D75154,0x843B4D98327A053B,0x36C0DD431984DF23,0x6CFCAE77C78C641F,0xB0D48FAE4CB04739,0x1B7D921E5D9B3,0xFE2DD134E99A582,0x87478C7119D9C2A,0x73AD49C3B9AA77CA,0xAF88BBAC08974F3B,0xBDC2FA3DDAAC6D91,0xACAC4CE1DB25E4AB,0x20E1626F6CD7B,0xFEF58A08C03A8514,0xAD9C83CEE35C3094,0x63DC7C34740FAF1D,0x497FBB8C2A8F2416,0xA815A25221E4BCC4,0x12B51BBAC77A37CF,0x179009F926B0D,0x9E41417F1A1DA33A,0x973806A75E1DEF4A,0x6CF76B9285092309,0x123A7033869EC304,0x28A6110D1057831D,0xB98FDC2F6E1CF3F6,0xD56A39115A66,0x105D856279443F4A,0xA835B4D4696C59A2,0xCED4F6077405F53D,0x350513C07EB65D75,0x2470E3E8BE5F66F4,0xF432F04AD3DA6939,0x131B20AE96486,0x4490A47DB29FAE6C,0x28B76AAB02C717E7,0x790CC2B39994C5CC,0xE509A1D16AEC19AC,0x982A5787ABF5E521,0x2BA5E86482EAAA27,0x9DC7CFC39B45,0x23976930DF0492DF,0x7212140DAFAF2E04,0xDF00140D54E2F3B4,0x69C129B57C8FF828,0xF6D747AEC2296BE8,0x9E29B46DB988FA8E,0x6DEBE8C3D340,0xD3CB76800E25D461,0x162298699ADE5D6D,0xA6D9603BCD217A46,0x284E370F81BB51C9,0x5D059DCA27177796,0xB0D013EC70CF92A,0x14A875791DC1D,0x32C14C775D1D7F17,0x3A0F1AA1FCF4F7CB,0xF2018CF4B9AA522E,0x12AE8F660B59E7A8,0xA5625F52494198B9,0x3D362EAE0E84B0A3,0x16C708BEF4B2C,0x6F0AD69D242E212,0x9165D0186665A5AF,0x9D3538FCBFD479B0,0x17C924EC609C82D1,0xD48C7937407F3B8B,0xBD041F9AAF1016E1,0x2AF4A06F96CC,0xB9D79BE5D9AFF3A1,0x25B9251C88EB326D,0x83FAFD08B9F46F7E,0xEFF73AAC816458D7,0xC5FE567323AB9CBB,0xE9D215BBF220F7DE,0x119C19DD2A164,0x7BE48B8B2A004A55,0x769A26A3FA3AF6A2,0xDB4DE9C7A1229B78,0x8C8E28593579581D,0xB06837AD5EA3B976,0x9B6B75024B1C0D5C,0x4D2777704721,0xE4D60D23944C84BE,0xF5C54A183138D642,0x5AAE7624775DA793,0x8B36666716372609,0xDD43BFFA76CDACBF,0xCCC2F8DCBB3FB105,0x1CD44A2EE827D,0x4CF7EBE4E9EA3DE3,0x6A050C60C5657BD5,0x632920B8D3BDE5A1,0xF4FDEFE3DBF59A0B,0x13C715641257C5A4,0xB7C2EFA8380250A6,0x231FA490721A1,0xC7FF61DE6BC48ACB,0xE12E0BE469935E5E,0x216A909333808A52,0xAEBFFDDD33B3AB4C,0x74E028367EB7E146,0xF5A6FED7DE7390F9,0xA4F875E30A54,0x2DDB0C4467F3B096,0x4B9C804850027C87,0x3B3E059244E361EE,0xA3D4502A459C29D5,0x603BB792FD548782,0xE51C390EAB618784,0x661A72C3C300,0xA6F5019003F4F264,0xFA6C85AFB86104C5,0x18FCBC71ED0911B4,0x2696769E7AC50892,0x7CB769C44F7D6666,0x223F804B8DD0D5B1,0x222976E52520,0x623876015A08F9A9,0x99AC69724B747BC4,0xBE11143A85B3C619,0xAB9BE8A087DB4CB5,0x7B37D9312205027C,0xDC584993EC9CF80A,0x120488C2D9BD6,0xFDE273B8080D946A,0xDAA30AA10F1559AC,0x6B94BE96A8FAB087,0x543C42D82163D6C9,0x66F3845AC288428,0x6AA1DDEB83C5A2F7,0x12DCFDC0B3953,0xF7F5A78E18E118FF,0xEF6EDB87A851796A,0x4A883DB133BFF88B,0xA47854C72AFC211B,0xE271AADF9E9846A7,0x6BC058274F89287F,0xA85D9B922380,0x24DDA7B31D7B1D10,0xD74E2236DD04D7D7,0x14E80A0CCE4C6CCE,0x5EB981721804DFE4,0x195938B89CB494E6,0x7805883C70376DF8,0x19D235CF7902C,0x4A902CA33CF1D812,0xD18FDFE9CDDF5847,0xA0ECB268B6F6EB6B,0x70A51111AF2C3FA3,0x2F9404DDC202F4C7,0x6B8C4727D32B91EC,0x224CF9FF59193,0x7374681C9FAABAFA,0x2ABFA15FB82B854A,0xE084FBACE5D54483,0x9B88C3D7E7E634F8,0x1904E2F29A7AC1FC,0x7CB89FBC66BCE333,0x17699E62D31CA,0xE560358D51AA99A5,0x5FF39E30255C4DD7,0xFA9CF4CFC4B60F4B,0x9212D22E6F17C6F0,0x8D72E2C99DB0E6EE,0x21ED79954BCDD788,0xF90C6B97F33B,0x95C2E994239394C,0x45C7B5DFC6BD3990,0xF0410789606343E9,0x59F6302F2DAC96C6,0xB563075CA92A90F5,0xC1C6F269005A0A21,0x9C12C9131A90,0x1D58C456C349C408,0x5BCEE2CDD67EB50F,0xB448775323C48CF9,0xA542E8824557BD6D,0x718644FC8B1C7F6D,0xA5AEF86A8F55E260,0xBE3FC3D65CB2,0x9A94A6CF063B0255,0x2710C93FE7D25ADF,0x1A2E92EA2A556BAA,0x98CC2271A6C8C7BD,0x8FC771F85B4FF377,0x7CC5675A6CA3F1AF,0x79058ED91BA8,0x2C9273D40358B31F,0x20CD7753396FFE4F,0x20B42EF675B817A8,0x9CB9D697FD888D9C,0xB81A8B7BC33FE278,0x7D958DC68EC555C0,0x166BE6BC686D3,0x97187E81234D2BAE,0xA45F1A6BA1F05474,0x6F9268E4AFF49203,0x67A4E8C15853061F,0xCB4F8A2BB312CE9C,0xDCC508BD596F3D26,0xFEF3C4F12C59,0x39CA940207A643A8,0x3BEB2552177D48CB,0x7B5E25E7838FFFD7,0xE57119933A0F1713,0xBF6449423A797F6A,0x9FEE983A7AE99938,0xDAB0544DDA75,0x5D4959A63824A340,0x57FF7C86C234EE8C,0x5AFFF6FC812F86CE,0x859CD13884D658BB,0xCE80DDECE4FEEF62,0xEB54C071E32146DE,0xF3A10280F839,0x564C74EE16322B75,0x27321A8039395531,0xE2482625A82D6A62,0x72FC01212235FF14,0x1152054A9DDE66CE,0x51D3A2768B1EBF71,0x20CB7D51DC802,0x625334E81E1EAEC3,0x58E530990AC07825,0xC28A5106F4C72437,0xAF5CE43893912685,0x915686660CBE1A9D,0xA19A35CC6DA3062E,0x1FE7117DEFB04,0x2730062F37218F92,0xFB8B7DD32A666354,0xE6ACA64AB9FD390B,0xF2876F4AC7134EAF,0xA81D2730A964FF18,0x4FA14D823DC16E89,0x133DEB6AC0CE4,0xFDCE38DF5776A74A,0xF36A3A377F7761A4,0x57550DA64F48DDD9,0xFB58AE0583079826,0x758E9BFA748F4F9B,0xF4169799377EEC88,0x283FE4403AF7,0x13FA6DD518C2E57C,0x1C958342AF2332E,0xE94900CEBF4596CB,0x41357DFFA36C0443,0x77B1A85390754E45,0x3174C1456242293E,0x1BBDDA2ED65F4,0x901030CD5BCE363E,0xBBA0CC9C88FDA9D7,0x852756730F9DDC75,0x52DEC1D63C482424,0x3248E1B719F2923B,0x1B9B6C13CEECB776,0x2C5B2998F024,0x7459BD7BB172F50E,0xE6C94D8367A1543,0x9D735221343C331F,0x25EAA4E2E3265D64,0xBD97AD262D650F40,0x882A50EDE7A47237,0xE82B2DB80C28,0x15A9978F10B5F149,0x20E6B8E913DC56A7,0xE2D92AADD06AD899,0x65457E827E4B45F9,0xBEC45EE9002EECAD,0x42A30CFE21E10D12,0xC6EA7BCE5DE,0x80A2247D02D9EDD,0x353BF45A5A7FB676,0xA563551A561FF7ED,0xD2D635E40CFD5E28,0xDDE98AA30866C92B,0x408F81E566D14EFA,0x16D0AC3E59BFF,0xAEEF632F9D6DEDEB,0x34DB9B4AE389626E,0x3A9FE1FB2DDBB442,0x468A7ED382752B62,0xDFA4B3B1852E5EC9,0x5A9034F164F4986F,0x6FD8AA1DC59D,0x7EB84A6F578D6588,0x136AC6ACAFAC2448,0x3BEB2BE425D9CBBE,0xDD3FCE7E3C250143,0x8A477AE935454524,0xAC7429E6ACE4A868,0x110893EB9E1E1,0xAFCC82941858A893,0x89086F2628597463,0x6989ECACB625E20,0x8999E36659ECBAEF,0x40A939CC265E4ED8,0xC233121934122EAA,0x113AF794EA65A,0x538AC3F1E58338BC,0x239A175D00B11,0x5DB2232D115FE7D,0xA80AA129A3884CB0,0xA5AB7A431EF19FDB,0xA8D74FEB720A813,0x165DC276FA45B,0x8F03690FF465FF03,0x5EC5D2C0F887BB2F,0x3094AC0FBBE8FD72,0x83C61D82FBDAD1E4,0xE930110D43723E8C,0xE4BCCB5DF3D9B842,0x147404C913DE0,0xC62A03D77DBE44AA,0xBF8356DD5913C689,0xC3DB2BE3C88F4672,0x7CF5F97C9BE26321,0xF2EB7D587F67F4C8,0x487BEFFC4D46052C,0x1E8DBC62D6CA3,0x819955DC8E62F217,0xA6479CF341E0B79A,0xAA299A64ECAFD15C,0x9A5A07710D29FCF6,0x8CE34DB704A22582,0x3CEF0EC419A1FB03,0x1BEF9B709FFAB,0x6B60DAD219ABED73,0x33565DC6CD08C2AE,0xE41CEF457198E7ED,0x391CF2A0D11AB66D,0x2EA8A8AD13F29D64,0xCEEA64D04720540E,0x5D032645449,0xD8D0F5949F15E11D,0x56D074C7A2E10AF8,0xF16BB9B74A5FA619,0x44B1E6DCF3514B3,0xF6C19500DF474751,0xE864BD3298F2E401,0x1BD106DF50C68,0x8489EBE1F7796E43,0x85BB6AA3CBC6A44D,0xF32FED21684D2E0A,0x57CAB2D8A5A9B50D,0xE0DBB0A7C22A405F,0x31083324A8AD5120,0x1B9911F8219EE,0x31D243759B66193F,0x52812AE2B8AB4010,0x73BFB75372015E4F,0x5351C9B7211119F5,0x7311D4EBF112C8DF,0x25A11CF2E7859169,0x87E97F4BE176,0x6C457AFDB222FA30,0x8BAED55176F5A999,0xDE18E1DE61AC8638,0x76408DD51E3B690B,0x953D166B747A564A,0x27CEC7F2B93C24DD,0xF005FB4A29CE,0x38440B199EA98DCE,0x55782252509A43DB,0xCCF85A7D7F544F64,0x94A009ADE12A8A74,0x764B6A56CF61CE12,0x390B44602679FDD4,0x86922FDD349B,0xAB10C8A67A46DC2,0xCD14DBF6E938334A,0x261A59ABB1001D09,0x2DFC368D51A90772,0x152F59D693BE3AF4,0x6D0CA06267903B9,0x3A04AFCB55DA,0x440DBD2EF66CB98,0x9D0E3F493AAE536E,0x703FB141F017F570,0x6606CA05C45AB8B9,0x5C0E56AE06832260,0xD1CD517AD2BDDEF1,0x1E697CAE18DE2,0x6FCEDE3DCB0E0EBD,0xE0B8509B67A68FC1,0xF46D610F064420A0,0x4ADB1F760441D1B1,0x686126026EF6EB54,0xB3303BBD381C642D,0x134D95A66FB8C,0xD64F00C3A4BF622C,0xEF186CCC1E1CE01,0x1C6AE542C626CB4E,0x1367502D2391E0DB,0xA6BCA4C20980DC04,0x81CE3CCFCEAC0AEB,0x7DD923434123,0x610DFABD99E08D08,0x1F912EF04BBEF35A,0x9333ECB90EBE2278,0xFFFA1A665F9D5A65,0x3041DEEC29CFCCA8,0xCDE93D66A319E9B7,0x170E571177B1C,0x6045DD5D3548E45D,0xB7D85C0C7DCC45F8,0xC22A076DAFAD0818,0x7B5798466C0E7974,0xA3DA450658E672C8,0x5FB0E25C03632FAA,0x7345EA805A1E,0x84C490A48762E734,0xD9D5A2121C4C908D,0xCF569ADB4BD75E55,0x879EF9A86F62A25B,0x94E7D1FAE895D768,0xAAAEF80EF5C7B689,0x156FC4F666604,0xC83AE0309ACF3093,0xE415D4FF802D1981,0x43FC8714E3F2FC5B,0x4B74ACCC931B1DEE,0xB9BA7DFDB44C380,0x7CAB0703F445F953,0x3FAC4B84DE34,0x2AB70DB32CF691EF,0xE32BBC928E671EBF,0x509D4D50CD55D5,0x5AC6A9D0A95BDAA7,0x9F08A1752DBBFD74,0xC1F613F1468E523F,0x13ADEB76D9C76,0x676553185DCDC0C1,0x22144F92C76DEF69,0x1EAD0BC69BE314F,0x163B1483EA1A8412,0xB46D9EF1E23EEFB3,0x507B7F6835DB83FD,0x18F60862C39DC,0xCE6EC5D701FFDA58,0xA1FBA73BEBEFC1CB,0x54ED5ED4B46AFBD4,0x8ED0006D5C62CA92,0x8ECCE74F0206CEBC,0xDCD4B160F0DB294C,0x1FC74A2012289,0x15D7BF7B231B3B2D,0x8C60F596AE154F5F,0x473397219B641094,0x81B224823729B3D1,0x3E80D43D89A8F409,0x95052CCB6B5D278B,0x1CB58CDC1C7F2,0x64B62820CA177D9C,0xFEF641546ECC605D,0xA4782ADCD44E62D1,0xF4E07AD179F6B31F,0xBFA3CEB09208800,0x31469BB0854285EC,0xBBE2AD418D58,0x7BD4BF89B90ABB17,0x90C7D6F10E13D0E8,0x4AB2F5ADB26BB15A,0x316FA139F1811578,0xCD8C663833971F6F,0x912410B4F547BEAE,0x22F242C096F69,0x6AFC7A3956111B19,0xBD7D1F71CAE7DA08,0x3583DE86039E05FD,0xEBCFF53798112B0,0xDA8750490E90581A,0xB7D72138E308306D,0x1AFAEA7D6614D,0x85E89B97100CACA6,0x3F079D8DB1F4E6B0,0x173D89FD71DA43F3,0xDCEE209C573703A5,0x439A80DEAD55C807,0xB3AB5FA83C1509AE,0xE4D390FCF785,0xB3D007BB61F95CBA,0x9AA02AC203EEC6D6,0x2B62ECA981ED3B8,0xBD1741DE16F81E96,0x5133FCADFB4DC575,0xD8EE353A8A9E9B4A,0x219F432F2F5F1,0x5FD80ED39328EAA8,0x828920084ECC23DF,0xC7B70FC680A7F2F4,0xEDBBA6A2AF49A200,0x2837556E7BF08509,0x53E4939E9D52692F,0xF0E58CA5239A,0x880ECC469A28E883,0x741E40BD3DC27BC0,0x6242402AAF49D26E,0xB5A18BF816F889C9,0x27AC8971622526FB,0x449E6565E90532B5,0xE157A5B76473,0x2306BB9EEBADD94,0xD26CC55165CFF862,0x41C44BC7ECBEF521,0x279D4346A9AB1993,0x5834A05381E1BEC4,0x2FCC6C4D2AB1D6C2,0x115694C58F1FC,0x6893273708C4A99,0xD232DF10F4A6FCF1,0xCC2D8BD7EACF1F70,0xF75ED6C1A3D6E5E1,0x40A0271D4EDC7E8E,0x4B71D6399BB7E2AE,0x20684FC262E40,0xD01432A4E6069D4,0xD60D11DAF31B85EE,0xD3F0AD1E102A1C34,0xA8DD84D4C416915,0xFE15117696E4981,0x42F72041C83F654C,0x2304BDD439E69,0x72662CEE64944479,0xF51E4F59A076BC5D,0xA0E602CC617E13AD,0x20F525A8B8A2C177,0x7933A561F0225A78,0xE6D005B1525F0B75,0x16DA8348AAD0A,0xE17AA90E00F70D2,0x7C15C3102F907C77,0x3DFAB9326DF7267A,0x33C42F6CE6242D9F,0x4DB831C9F96B7F3D,0xF3DC012B1F45E9B2,0xB11C1AA79B2F,0x289E661949E50691,0xD783D5218400554F,0xD8D1C7071139D008,0xB8CBB0A809863858,0xD6F3E477E7EE63FA,0x84EB39ED482FDFC2,0x1D246FD07C84A,0x578B8322E3CC0AA8,0x40410D7EF9F37817,0xAE7383A6DD303821,0x2E378A4C14EF4859,0x55CCAA7FE6D4D5E8,0x54055055C8576F54,0x15442AA041818,0xEACEC6CFF1C13ECA,0xB33655B512557A74,0x785D6DF1EEDE19F,0x1C913285D217BE5F,0x5735C985BE91F88E,0xB209F1F1E74EC68E,0xE1BDD7D413C3,0x570481FB846B980,0xF88EB9E1CA6DBB65,0x9DE300573F03D9A4,0x691E9F0E95168822,0xF0FB7DE81B06DF9E,0xE7661D6A4DE3F6FA,0x1CE53A864B95E,0xC6CE50ADAC5F170A,0xC364D01ADF6CE67F,0x8808F3DF1A25542C,0xB50B445A931C4987,0x9BBE8B8398B266A2,0xAC73C0803C142EEC,0x50BA992F0E32,0x8C11C50CAC736F42,0xCA537214127CC969,0x9AC1381033D1E943,0xDC24EB7E8A24F988,0xF93E03E8E08CC4D1,0xF8ABA5716632E713,0x73BBD6997DDC,0xFE6F115917AF34E9,0x5E3C5CEB184220B5,0xB1C6657F42B99A3B,0x4E1ADAFF82DC3EC5,0x3E5E17A360AC849B,0xD06F7CE78B221AE4,0x738C71CD26FC,0x13605F0D867AEBC4,0xB5E4722A820E69C6,0x4360B7C0EE57D6ED,0x54A4D496BB748945,0x6C9051904827FAFB,0x89716C501BAFF2F2,0x1695EB6E43924,0x88DFC9E90259DB96,0x42D8ADE7F7ACBD2A,0xC641B5C706DE335C,0xA765A3F9D394D21E,0xF9076071A17E9CAF,0xACC75CD2BB073EBE,0x9EBBD417B6D1,0xBE0208C2816090C5,0x736EAC9AC72EA6A6,0x5E7D8895CC27B4E3,0xCADE7023A852DCD7,0xA027D4349DDADDF8,0xC78E7F141A9A5125,0x1E8F8D54BEF69,0xAEFFA99039E3911,0x8E085448BA7D8CF5,0xBE39F08833915C11,0x38D866119C5994BC,0xBD9CAD479BC4DE47,0x180C57C0011F1819,0x8F6154EBF6AD,0xABE356D428FF310F,0x3988BD8702A707D3,0xC7EF9B6138ED5C9D,0xCD7A1648ED63ABCF,0xEFAF0583343C7778,0x671873C9B670AEF1,0x2208C5AAA93DD,0xFAAE88E2908CAF83,0xEF9AB2BAE520824E,0x23B48A6042786DC8,0x3BFEAC78C7EFC697,0xF5BD779194FB2ABA,0x571F237144F88DB5,0x25A4869AC2BF,0xFCD331AF76248C8A,0xB5044EF1980DCA8,0x49AE3F3D66054348,0x3B3D576B6BC56634,0xD40B81DFC46CE3EF,0xE2E62163843C1FD4,0xCDC812C9FC46,0x9C86F8083CA481E0,0x79BC93A2853B21F9,0xBAF30392A81C50ED,0xB0ED84D591CB7D37,0x286007BF4479BD8C,0x2F57F6B56C74CAA6,0x1132F3DAE3C25,0xEAD9F6E76EED3405,0xD806830360716941,0x64024A4A5EA727EC,0x963B175433910641,0x3EA6A8496022192E,0x2B11DED50E582DB5,0x1C0E45727CB83,0x2C4D388900ED53EC,0x48706010BE84FB9D,0x3C2E31F01E5F0267,0x239428A6D2E164D2,0x53C378662DFA62C6,0xD74A5492C6917262,0x9E7E4F17124E,0x79FC1BB171F37F07,0x93C45C166CF5BE95,0x4D628DA3D54B8C75,0xD8F8803729FB535C,0x8F48A1E10C398D73,0x4A844C47C23BB090,0x1226BE0D1C402,0x7D5DA786568B8C54,0xA3DF804AB8BFA7BB,0xFEBF0CC1968A15CB,0xC95C9CA63FEDDEB2,0x87AAB35DC3231324,0xF035ADCD6174209C,0xA281E654CC9,0x37E9EDE31E5B9921,0xA8930C5C9B7E98D1,0x8F7269CFAF3B78C1,0x5314C97F9E4B943A,0x9719C6D314F944AD,0xB4509BB881C22951,0xF881E9380193,0x9CB932C367E5C920,0xF87D25FE50E9E8B9,0x5C59EC9BAF4A22C4,0x34F4BCF9E5F4F074,0xC5C2CFFA261D8BB9,0x42961720EF72E71F,0x140D7DE099BD7,0x47E645AE1A90E6A5,0x558353D1E87F3F64,0x91D82EBBAB40FF83,0xBEE46D51D4DF2014,0xEE4326540B67D0C3,0x816E9EE762B31FF0,0xC785773F834,0x33B62BB01BBB35D2,0x2EE05F2CC093A584,0xF2E79524C9FC683B,0x34B4FB05278D1AC,0x6C854E8038050419,0xE9810367FBD8EDE8,0x14ADC2949FBDD,0x7D3C082AE28015D9,0xD2D70DB48D783170,0x1A0F2D0137C96442,0x28CF8971876B910B,0x7C633E18CFE4F438,0x18B31B914F31CCCE,0x7433C89F3CDE,0x9A576139A43C9248,0x237DCD38A5155625,0x35A3A9F5D53214F5,0x29C596B44E33E646,0x826A284CE055154A,0x31F3FCC23A87F5E2,0x1ABB7BB26D494,0x22BD6A506C8DADD2,0xCAB166159D554FC5,0x5F16C89805898F00,0xBA8F4624571750B,0x677B30E150DC0641,0xBC2F91BF79806490,0x942BCE41B0B,0x16A871188DC33A67,0x468C5E9FA530F10B,0xC809F2C680D97F8E,0x9993D9A5F1C931CD,0x770173BF5488E51C,0xFDE9977F2851090D,0xC5283BFD319C,0xADCC685D2245C76B,0x933C9BA25ED19D84,0x317733B7937C2A20,0x5ACAF9638E6BEF73,0x2B9489BCB48633B3,0xC1C6446FC3A027B5,0x8A45A6F7BC32,0xF2C37D3D7F25F2FF,0x19028C677DF5C3FF,0xEBEAD4A3D6A0B2C1,0x69E07B7979344DB0,0xEC3BB7F297D13470,0x8DB0FA23653E12CB,0x1A9CD95B36A53,0xD93BBD37A00B84FE,0x38200A6F625B6702,0x326B07800E93D8D1,0x5D80781D3B8B2C60,0xEDBCE17E61AF76EA,0x4A5EB79FCB569703,0xB2E171217643,0x98E095DD245F8ED6,0x5A1EC3BB9FE8B756,0x9B6D434102891686,0xA6DA0A2229C3E542,0xAD7AA08C791CAB97,0x7B66DCCA5B01DC4F,0xBA9FB757FAC2,0x3E4AA8F04A5514B8,0x9A737743383B7B36,0x83724E3F58ABF0C3,0x8B9B3399124C1636,0x9F413A524B7F0B52,0xEE461CCD7A86FB2C,0x10415E0C01E5C,0xCBE32E8BDC2C5B3C,0x7FB62E643F39B36A,0x64CC72169F77E22,0xB27F30F0FED7CA86,0x8105C1EA608133CA,0xC9E6C6A04A4AF95E,0x2EC08BA9AA,0x4B84EC8A562DBA0D,0xF63BF8AE7D9DBA3D,0xCBF74B580CFF9867,0xE3233438E0F09F23,0x1B43FF93ACE5F27C,0xC76F6DF1846CB38D,0x6E1260243E67,0xDB96F56E6D5FB4DE,0x570DDDEFD01D876D,0xFDD41F669229DD67,0x33FEF9020503394D,0x10F60F3A8641D2D8,0x7A8DFB6305FBCDFC,0xD3C4389F2B34,0xA27981EEEF565A52,0xD6D4BDDE3403FC51,0x591054F45787DC63,0x6FC5DF0E3D752FEB,0xEEA97D066FA191B6,0xAEB40218DB5BF211,0x1F0683E337D2C,0xAAD1A549AF1202D7,0xF1306A05A9DBC9DE,0x9AA52E35515EFF40,0xC6FA6FC47515DC63,0xAAA81152432615BB,0x35666982877BE98E,0x1B25EEBA706AF,0xB3C35CF4027184FE,0xC343CCBE30FCB3A2,0xF5E16A370A229FF8,0x44F98C5AA1FA9BDB,0x72533790501CCA41,0x6E8693BFD3E078AE,0x162CE07764ACC,0x4AFDEF4AC3CC0065,0x95D51EDB15780033,0x6623DC3F037376B1,0xD5E3C524A2587A74,0xAFFD7FBFF398708E,0x3A207C878BD13B7A,0xB5BE4BEB023E,0xC75F3472B1A85173,0xD27CB8C55C01FD38,0x6B4F835618468DF4,0x4527D50D4A6540A7,0x466AECE50E780AB6,0xD54EE55250475210,0x7101341C39AA,0x350613C412A96AED,0x4B16B55D99D679E2,0xCA1512DC14BE8AEB,0x196E20EBEFE2C78D,0x265806727667FE29,0x717701038FEE44D6,0x14EF99956C3F7,0x63289C125127F6E5,0x3BB4FE8C3B45A889,0x95255476CF4AF87E,0x380DE838F25547E5,0xC870904690F5E3CF,0x3824BDA34883D375,0xE3135EE6A891,0xDC8DFB9AB25F1469,0x8FCA484C2EA49758,0x7FB3B8472D7B3736,0x97311D46024C7F8E,0x37D290D09703B0D8,0xCE534F7D754345D4,0x21662261F0882,0x88AF58645E60B132,0xE6362CE10825AA41,0x2B5E6DBFD5714041,0x964DD6315F28537,0x29D13030487433E8,0xBEE600EC88CA3728,0x1ACA2EF050790,0x74F1D69C5B3F4D58,0x8C10C59C61D797BF,0xE5633850DB56379,0xEE2B4F99657977D0,0x95A04AE55B92F9F5,0xA59A80E345573204,0x13D49103656D,0x1C3FFB79B7704A72,0xF63065C052F66998,0x2E2DA761E51E7FBB,0xB323D20294D4FA42,0x15D09718EEF30C49,0xE957F6CBB473DF7,0x214E68586AEF0,0x217386AACE7BBB89,0xCDE5D9710ECD2DF3,0xCD5349B0F8E20028,0x11BDB94C2D4A47A5,0xC4C932BBAC094222,0x43AE58D8662B12C0,0x36041B030FF8,0xA1735163DF2DBDA,0xA5441AF261AA50B,0x56F59115FA6E8503,0xAA94B68B84BAA63C,0x9FFA56A80E5D96F,0x30F6B8AF79106B3,0x1403943EC97FA,0x7D1F5D73F76FF357,0x6EE99E244997982B,0xE67C7786F0C50D2,0xF23B5420B52B53D0,0x5185FA1FD1C55841,0x49B82357EDC754F1,0x11F8A7D1D5F20,0xFB71F5822983BFA,0xF284640AF455C93,0xC494F0386DBE9757,0xD08543D94A0AECE3,0x1203B97CB0140430,0x7546BFE82F30379E,0x3EA06F568F1B,0x223E88FE6F1B4948,0x339B48072BE9C7A3,0x1EA706D1C8859C1B,0x5F831876181D4FD0,0x3F3E2572217AEC45,0x40E0091276BA683D,0x156D1D5F8631,0x2757117B91B4A7C6,0x5B74713C612B441B,0x655C7B3FE61D9918,0x4487AB3D88565A52,0x85B0B61A07F5DB6C,0x536F2BF750A5012F,0xE8C51EE4D880,0x83FBACD3310C1FD3,0x36343A80B72DA40A,0xD1E4262843C2547C,0x5EC1A041A5845F6F,0xCF8118A875686BE,0xB3C78B64C9732CD4,0x15540AD120B54,0x8F9E568C2EAE3564,0xF15224473912BF82,0xD392025F4CCCE263,0xCDC4A99B2D1BF052,0xBD48BCCBBED49568,0xEAC87A2B598291C9,0x1E34F39609869,0x4163A41980E0D379,0x604B4DAAC7CF5B22,0x9B5229AE4141BD1E,0x75B4B0AE3311241A,0xA880DC198B4AFB49,0x34F5D3924B68B4FF,0xD2028A37B19E,0x8BB5F010A26D0B7F,0xE8F7BB82ED4A0386,0x9881999C63CC797B,0x96474B95BC6091C4,0x464BDCEE5A7DE3BD,0x15EE33CD7ECDA385,0x1AF09D15DB217,0xE0EDFB4DAEAD63BA,0x1BF98BBDEFF8343C,0x9D51D945A94625F1,0x4B35A3D57BBA5F57,0xE07A7E80F06234AA,0xB7D8482C6B15B4DB,0x216E3A32EABFC,0xB72744394FC1402A,0x9452AAE9E0BEE368,0x18D278B5B4D8FD36,0x28682E3F6ABCD48,0xBE63E50DE3E550C1,0x1F2FF40381B39E33,0x95F3DE85694E,0x6B1F34DDD83C7961,0xE7BB51C53FF82E02,0x525A91A46DD7158B,0xDB309CB70801EE3B,0xD0E9499089EE02CD,0x5C3303C70475F419,0x156FACA6B70BE,0x336E137AFCB0688D,0xC2A904F54008251B,0x6C6CD3E186B20BB5,0xCC1CA12A596E70CE,0xD5125595C8EB573,0x8366E00C32C4AFA5,0x213B0F3DE25C3,0x12CB11387D0E1247,0x8B948D25DA6FE598,0x902319117F24C1EE,0x4508553946F08A6,0x4BB3717EBEFEE9CB,0xF54E357811E1AF7F,0x329996C7BF35,0x7E7EAB1F276BA202,0xC3973A35C211C8AC,0x6EB2090B5AA2E453,0xF1B87410E7AC7EA,0x60EEB312ED4ADFD6,0x10C8AAA64C32FC1B,0x9CD4B509E3A5,0xB4291A0F2B18B33C,0x1BB875C19FD7532F,0x66DD90D6B31DC899,0xAD1ECC3EDFF340BE,0x9C45FF1C7AF2E834,0xC65DE4096B73AB1F,0x1B8F5919A576F,0x8B04BE81E07153A0,0x88F8253763A9AB33,0x9C80AC40DA37EED5,0x8B87951C2013B2C4,0xB365456906124A4E,0x664CCE52F288162A,0x140335BA62950,0x7CA7727695D9877E,0x58D29315846AFB55,0xD1DBF983C0C3DDCF,0xF2AA13A3712277B5,0x597C56705DC0715F,0x8BF6E1746DA886B7,0x1E9B6C775182B,0x9B16C1BA89326063,0xBE2CD4F72C6A4A92,0x71419173D79FE08E,0xE99E797B397BB936,0x58C8EAF4F701CB25,0x43A2E99BAC8F3840,0x1D22B5ABDC838,0x479190BAD02A12B9,0xB826A9251F1EA32F,0x97C467172279BC97,0xCD198321795F6CB1,0x78D4ECF4DDC0F488,0xD688A43F3ED1837F,0x1410BB4D88E04,0xFF8A95D87E3AB393,0xAA1CEA7664FE34E2,0x2F259063046F10BF,0xB02352110CD1792E,0x6F0FC514C280554E,0x8C36872838D118CA,0x5F87D3D8FC0,0x508E9DD9959D345D,0x885BABF56F8DDB8C,0x43E5F9D7F24E126C,0x50C71FAD1343C1C0,0x739446F0303668DC,0xFB22E70EB764F178,0x14D95276FB744,0x3119AEC1557A2F56,0xB96AE4B28DC8020E,0x1B90776968C1F81,0xD4553930DD5F2AFC,0xB82A0FB3D9F2CD88,0x514BD178B749AB5A,0x1BD9360F4B7B4,0xFB66635521BB0144,0x9041F13044A105EF,0xB27967CB012F2301,0xE170F384569E5A65,0x7EC1F5F5C5D8ACFD,0x123783E4EFD327C3,0x2259D38D814BA,0x391C1AFF5FD14E0B,0xD638725C338178D8,0x24DB3260246DB696,0xBAA152DA021154C,0xB757A346B3BBA6AC,0x9E8776AB55F3D0E5,0x1ECA2D6A8D115,0xD4B9C377FB214ED,0xCAD696F160E5CB3A,0x8D398B7124F73E27,0x71DBC5C18D1D6563,0xFAD3572319F922C4,0x7D685880EEB7F568,0x7F06F4F77281,0x7018BB1BC6AA7371,0x5E815A2D3946FE0E,0x12D3A051AE462D1E,0x4ED345FC86737D8C,0x72D02ADB7ED4111D,0x65EC15639CAE25FC,0x309DD4DEA428,0xB67C28CAE3CBF105,0x35D9CAB0E94BCA90,0x59E51403A91DD0AF,0xF5D72DE70AC748F6,0x455A33A1E6509DB6,0xB58B1CE838A165,0x51F2CA32E819,0x7AD296680D645EDC,0xD81B5E62BC0E753C,0x5CC139220B356699,0xA6E45374731333E9,0xAE2644563893047F,0x5C57E74EE605F27F,0xDDCD68A72734,0xFE662224060714AA,0x43A3FF8740C53EA0,0x7BD45557C0956FF8,0xD36F361DF83B2130,0xBD7B1683D55A8352,0x9E48B079627A1284,0x1A0E2F91579,0x45361EC387127DA2,0x576F1984FCC930F7,0xC048884DE101CAEA,0x8CFA4D7343113100,0xAAB582F06072D5E2,0xAA879B59A19BDF49,0x168A5A15BA655,0x70CC5EABBEB91D86,0x9A6EC13F766325AA,0xA79AADFA7DBBA90C,0xA1F02B6F4F49EDCC,0xC09B45A4AFC0D78B,0x418A28578B188FA9,0x8BCF0DF49D4C,0xB50AC1052CFB2D09,0x95336E4721D5F575,0x3AC6AFA0126A0F51,0x32539F1B533A8A48,0x51209B2EF6A6C2C1,0x92A68753E00D1380,0x1262E5BA53BC0,0xC0EDAE3A2E1842EC,0xFB5390008AF2120A,0x9FB3E88F22750694,0xF70DCE62E8EC5CBD,0xE2D46171537DA5F,0xFACCCCD5DB1AF009,0x10C5190C89F4B,0xA16FE7C85927D588,0xB02E7A32B1763F7B,0xE872A8506D94E5FD,0xB170AC91911EC96A,0xED503A53AAEC667F,0x7AFC6F092F8718D5,0x6229BFF047E2,0x9C2A29520342893E,0xD3F05B617E923E01,0x8EACFFEC860B8ACB,0x5837072FE893DCC2,0x465EE85679AB088E,0xE64CECC56B684648,0x57E74402ED42,0xA50E3FAA673BE7A2,0xB300DFC39A63B637,0xD8D8B989272DBB16,0x7B6CD05BAA6D7B33,0x1182A9B34D0A4A97,0xF838789B88E61435,0xE7AA4074DB99,0xBBDFE20C0A6495B,0x55A4538A114C6F7A,0x51D2848E4FE49CF4,0xDF432149350D85D5,0x38032CF1D9879938,0x9E906C7EDDE99941,0x242F61AB1C73,0x3F599F3A9F26F569,0xED1D367CE7918B27,0x295FF6C96E63D8CA,0x2A367AA4CFF0D914,0x49E6E989142D8709,0xB1FCFB31C583E540,0x128464D679092,0x377587EF22DDF18E,0xDA4E55AD386999D4,0x46189A63F65B8802,0x22A7A43589C5BCB3,0x300C2D6AA0207A51,0x81B1A059B2E5CEBF,0x12418521FB33D,0xBC2B31CC408DE079,0xA536314A802DAAA0,0x42E078683EB512C4,0x95A83A1FF73B58D6,0x1AC1681B238ACAE,0x5FDCDE18C77C0B00,0x20A2A2198C256,0x55279B1AD61AFC13,0x63610837175E54F,0xF6E2C3D401BA1E61,0xB8F286CAAFB89A7B,0x3842AF650AC65E9,0xD0AD5850FEA2145C,0x20876275CBC5B,0xEAE295D9EDD7B326,0x4761292AB1312CE2,0xB4669DCCD6E09A1E,0x6C97B1724D72841A,0x8344E7039DACE577,0x1EA877C468AC3C91,0x1039C6B91B007,0xB9690741FE84ECEF,0x5764539F7F0EF535,0xDF1EE4CAC0E0534,0x74312C89440252FE,0x52804714E143758A,0x407860531A06D516,0xC87E67470986,0x5B80658DC4B343B4,0x38C506607A0E5645,0x78C629B89D18576B,0xAE7A204E0C1669D2,0xCDC4819ECB44D45,0xAB9ABBFE1D62E1E4,0x85B042B3C58,0xF29C84D3A99FC21,0x2788992E22887431,0xDE277C4A3189909,0x8B99E1C5B8B09BB8,0xE7E46B20B52D4234,0x69F1CDB8065C19DC,0x1E91F6E0026C7,0x3E0FC8AE16A75ADA,0xB64BAE4F190BBDEF,0xB1E65D7AE14E2B42,0x1DC36817DE7ADFA,0x2A4512F5F6C13155,0x8EBFA9A51438083A,0x16868589493B7,0x49E319249251885,0xB814374423A8FF4B,0xAA522D17B5C85A57,0xB11A1EA5F4F8D6E1,0xB9CE8A22092F4573,0x24C5F782A3CADB3E,0x289EB97AC61D,0x1E1F25FC05605745,0xFD3ED75E0DE25574,0xDC293B3F27D61AEB,0x9F2C2560420B34DD,0x7B1CF478FA889F39,0xDD115540E4690B35,0x993A5355136A,0x5A83F61CE953C1F6,0x2386B9C82089491A,0x6E30EC2BD969DB5F,0x7C8E21C7840DFA20,0x965C96B669DF17B0,0x6841D50D4E68588B,0x17BE06FC6062C,0x6F9B8A2F6F4AFC0A,0x191647DA8843EC2E,0x36599D639EF20E9E,0x59993A2070A7752D,0x625D0A1DD8375BD9,0x3D8128E5BE2DA1D2,0x2228E20866424,0xB2BEB6EB8179729B,0xE7ED7E5A417A8CFE,0xFEE6355C268AB0C,0x966D38CD87523F6F,0x9972C6B91E582BBB,0xA3B60D06080A7474,0x19699005E6C06,0x3051BE794A89906E,0xAB3047D40A1B9CC6,0x4DE53A4C2AFB3B47,0xB6AF41B110540A36,0xDE493C952C1BBEF2,0x3D14EEE0839E12F6,0xBE7A707766D5,0x30BCB28B309F2D2F,0x10FA08E511AF571B,0x9278D221296BFC4,0x4942C927E87D88EA,0x15D38FB2C60218C1,0x228C350DDFCE9BEA,0xF5D835AFFC3B,0xD1A6D2271819E06A,0x1FC05C9882718AC3,0x7DE8A19AD77CF9CC,0xEA1A106A59578ECA,0xC085F5611E2ED845,0x431A3BCE37B61C18,0xC0F75E4B0025,0x54F4E7A89CF333E5,0x1C12E05B63645AEB,0x7EF99D39E3EFA6E5,0x9F938C187BB9F0AB,0xC2281CC15CAF9ABD,0xC69A07595C632FA4,0x1690F631FAE0E,0x6D14EA997AB75624,0x2E98188D7362FD4,0x5066607C8139DE00,0xFBF13729EC133AF0,0x7E2369D35A95C408,0xF70A0A9EA9CC51EE,0x15BA503E4F717,0x6FF56BA88E348A1A,0x77FB43821C9838DB,0x7F861D698F3685C2,0xDE7134C378741E15,0xE77533961DCFC737,0x6EA8DDB18481F418,0x16837F4B47585,0x4C0EC2608C5F4A5F,0x433E5BF8B97ADE2,0xE1850F84F46C936,0xE7DD27D33372F3BC,0xFBF3F9F964BC8054,0xD0C4F7D399130BD4,0xB979DDD16B8E,0x9A9CD3B1191675E1,0x8F71F767171E7B3D,0x568BD28ACFD8E059,0xADC251A71E600FC,0x764FD3F6D1324DA6,0xA70B8E2090955D87,0x43AD3E69D53B,0x3594B02998B8C0C1,0x47F2AAAFCBCE7E2D,0xA7758E0C82225BF6,0x9117E3DA0A9EDB3C,0x38699684F1119CBE,0x6E61B0299F5303F2,0x10C053F27B826,0xF4BFA3A358BB3AB2,0x6F9E7AC36B0C5D74,0xF13F5346B6347D14,0xCEA80B891E38A90E,0xAAA83C5EDB41C5E8,0xCE223F1B15BA00F7,0x139D853E2C65C,0xABBEBC4BCBD278BC,0x2FC78B567ECF5C86,0x4B388B7E1126C42A,0xD3E73521A003DC2,0x7EDA45285C3B019A,0xCF31AC841D576F8,0x1896EC369779E,0xB0BD30EF357DC3EE,0x603CA34C73B295B3,0xA31A8360F0BAF0C,0x2B98B75544078DEB,0xD6A6CE7FDC067DE5,0xC9F6432B6D286FA7,0x1B59949113D41,0x8E8A73464230C913,0xC9469ECAE61132C4,0xD17981A7670700E5,0x22B911C0467A2B79,0x8865839A3411EEC3,0x9A483AACAD157AD6,0x19BAD161008B,0xE4778CDADAC0F8E9,0x1CB2A8B3D9225823,0xB66C74F14368FB28,0xE8053207EB4C5DE7,0x9AE18770B14A0026,0xBC9C11CE766D8F01,0xC7EE73A44C7A,0xA4DEAAA1E3AC7656,0xC823C0FB8F81DC87,0xF43975F83CA91A3B,0x3BEF213C08810FCC,0x9C838381ED436256,0x10C9EF87F5BF0046,0x5EA72636560F,0xC392349AFFE0A661,0xA301A660210FACC,0xA4374C2C9D70CEBF,0x57EBE97D64CDB70C,0xF979F083AEFCE306,0xFF142A2DF718AC42,0x190BEB881D898,0x1B0F7CAA27FF950B,0xA172B227B25D553A,0x5FA64569610FBC14,0xCB84223162C6CE8B,0xC51847F777ED8F2B,0xC481DBF4990F7E81,0xC46172243941,0xFB5AD75A16E7B16D,0xA00417348BCCDA77,0xE0A9166051AAF54A,0x870B78AB21B87BD9,0x16E3ADD9FEA90968,0x1534F94303958904,0xD480BCBC7273,0xE277F0A312BE2F34,0xA72BDB044B54B89,0xE67FC4F35DFCB723,0x7D7D9924A67BB0C0,0x4BA7080A09DDAAA8,0x94EB2AFBD55988EF,0x9749A7AC0189,0x15B32AC1DB037154,0x653CD0ABCAF7D9D5,0x28F9BDE0EB0E21FD,0x3C1878255832427B,0x99609E99D772064C,0xDE9C19FB263DDD10,0xEE92B0A7DC5A,0xC006547EA8869F3F,0x35CB546F8D1DFF6D,0xC0C48F622870677,0xAC63B3B968ED94ED,0xCED1975A0FDBE185,0x83D00F88061D6E06,0xBEA1FDF1C437,0xCB6DF8251E4C6EBB,0xA0083B2B3E95FC24,0x1D61FDDE199BAF95,0xCE4D37B9F8C65AF8,0x2106D3B669B23A41,0x76F70096DF822BD7,0x18251FAAC077D,0x13F9FF5F6A438A58,0xA1DBBDF9E893350F,0xF2688D9AC328A94D,0xE1244B6A23936329,0x7995E2EC3ED98C37,0x9D2BB3E620A30647,0x191010ED8C9B3,0xD442B7BE61FE45E0,0xF2B98DEDDE4E641A,0x94EE0710455BA129,0x6864D2E4634F6B22,0x778F2EC0C46BD9F,0x140CEFC11B1E74E1,0xA3EA52D43F4,0xBB7A41E68162E227,0xDFFD365A8231B9DB,0xC28391E9F09C2F33,0x69656DC317CEBF2A,0xAEB3E71F934B969,0xC091100AC1A903FB,0x88A76EC676E1,0x184C43C1A3F69E89,0xD6F6498BB920E965,0x4A92095F6D11EE54,0x34000A31B2E7CE65,0x98AA74D98073EA01,0x6C14F79B8A354EB7,0x16446A7A8ECE7,0x943B3BC9AD0BEB63,0x2A77870C5F892B4C,0x3BCD58AC25FD9AD2,0x57F8EA61F5CA4C87,0xCB535850F668141A,0xC76EE046815B8846,0xA2B65A3030B1,0x95FC77C3871844B,0x488F14B8C0DAEFE8,0x935AC1E67F8B0356,0x63A67778AE9FF6EC,0x6E184C908BE60FE3,0xDE4A27179C438444,0xF4311A34472C,0x159F52F71CBC1630,0x49AEBFA0A14DB119,0xCCB02206986B7458,0xE04D71D8CC415CDF,0xFEC92CF4A163148B,0x71B4D20FF24EE036,0x1682E0CE962ED,0xEDE9864477B63455,0x328C34F514592717,0x5F2C203B2A9E36DC,0x5CEBDDF689AE7281,0x48329BFBE000DC52,0x6D7FA8672EBC346A,0x1D7208BD6D5C1,0x2B745BF857E01C8C,0x32A92FC9C438B7BB,0x4691E739A487D9,0x6865CC2C918CAECF,0x6941C4F5AD74DEF,0x38936E9C9F4B123F,0xC9031F6DBEC2,0x627AA2F99975FEBD,0xF14F89D258BCA6BD,0xD546E0EF2BFBEF6C,0xBD1D12D4971A66F4,0x82EC126A2F1C5579,0x111060561067C519,0x67EE2AF50C10,0x81FEEBAC413FD990,0xE027C4C17298E33,0xF1ECC7C51FEDDC9C,0x5FC23F86A6E4F391,0x5B8473F3CBE29F89,0x6DD733CD6A5F9147,0x29810D16B130,0xB01CC21433DCD339,0xCE9407B8E7767CC9,0x2E72555A77D1B817,0x84BAD594E632743F,0xD458B9B491179F44,0x5DFEBD1186F02345,0x18FF7B6F18F7C,0x917CCDC9D8B1A6E,0xD36D02B3AFEF79A8,0x341CC5A2AB9355AB,0xD58A8E05B10A1E1E,0x1B00B79533ADAFFD,0xDCB18E50F93726ED,0x1E55273D50EC7,0x9F39EAEFB7845C63,0xE20443ACF0A40770,0x2984F76A610C5162,0xE5E0B51CDFB9F7D4,0x401F58F59F8A1879,0xDDE9BF76991750AE,0x3769A5304E1C,0xFF48D63D469A851F,0x6A73CBB7FB54A780,0x80AD8C6BB4638576,0x375B0AA5B69BDC3E,0x6DB4047000104B3D,0x8B4AD755EF483E99,0x1B8E8FFBB2812,0xFCE4881E5CB75EC6,0x6DAFE5172B914358,0x8AB0126B5DAB3EBC,0xC9BBD3B78CFD8436,0x8BF85F9D51EB92D2,0x32CD8C70B314498,0x164179BD25311,0x7906FE86FA6F1DD8,0x63ED266B14EE9474,0xCACADDD7E1DA78AA,0x688C2814E53B3121,0x74B214F0306C3AC3,0x46DC44BC9C7F683A,0xC1DF58651C3E,0x4ED9DB62A359D207,0x385282099CE643A2,0xB98DC4A7E527D634,0x94EB65C9BC377DD6,0x922CF69D72562547,0x6E9661EF8D05A2F6,0x5573AE77F24D,0xF2FA13063082C5D3,0xFF133884BD62E0B7,0x15A2B57F99173A32,0xB6BCA691B32850AB,0x22671291C0C69ED4,0xA7EEEA0975EAA822,0x1E5726A483B07,0xCFF91F58544288E8,0x8269FF777422222F,0x767A8D9624D00EC3,0xDA240CA06E6C2FEC,0xA936A9D860FC4AE4,0x431FAF790F0F4466,0x1C6EBF25B515D,0x15FFC9BD84D91AC4,0xEAC607FCCA8B25F3,0x50ADBCB093402A45,0xC6D72B5D9F6E1B86,0x671ABC45D511942E,0x542EB3C5B9DE4E71,0x1F7099C3F2D92,0x678C1B54F100EEFB,0xD0959C1575258109,0x31A512EFDD170157,0x47F82280C34BF51D,0xA439E47576AAA282,0x22F3E0A14D0B441E,0x25E4FF2007EE,0x641097D94411EBD9,0xC26D2BF4049D4386,0xB3CA1FC78D071AB5,0x3D86C3DA06F0F3E0,0x54B3BA7713013FA3,0x3F7C8AA230AFAB0C,0x15A72D5EA1D35,0xE9A27616627A338F,0xE9B0A4AEC6D790DA,0x1D9A51F724497A03,0xEF3772C36273A46,0xF09A6B9F324463CC,0xF6AF696FC4D3A683,0xAD865C3B5CDC,0x91FA89154C102405,0x693C3420965E6D80,0xF98D885133D0D191,0xA06063670B6F6CE5,0x4470F825AD8D0121,0x47D70D4C691BDE11,0x7EE30DEFC9A4,0x42FB9F89E581FA3B,0x4AC16F37C42609A8,0xE776AFAB1E876AC,0xF695D7DE53652F4C,0x2319CB649D05933A,0xD980EEA04DC0CD9A,0x8E09773BE8CE,0xF77E2526A6EB93CB,0xF5C034193D7DBF7A,0x3C78CA91C702E523,0x55E9DC3FDF08D3B5,0x6FB00AD71203960D,0x138B121A2A80803B,0xAE1167494FF3,0x6D50393F59E810EF,0x5F707669B58535A2,0x91148EDBF61602F4,0x1B547195EF496B12,0xA30C9A0CA64235F7,0x56FD145D11066029,0x13990AB37E9C,0x9D310B0B2AA1087A,0x743B46E2F42059D0,0x885B0656509F742B,0xBFB2465647E75F21,0xC5310EA3FBF8DA05,0x314C023AE917D984,0x18F953F37B7AC,0x3BD3649296B06205,0x67B4A6AD422BA116,0xBA311FCF087C6300,0xD0D7A703BECE148E,0xAA6E47F8301C4FE1,0x9321B27251BEFF2F,0x8D8821BFF361,0xE633AF6C2D478F4,0xC49914379D0DBA89,0x7DEB42B72E177036,0xF572530017EEFFC6,0x24750F593090C77C,0x4698FC267FB049BB,0x120C8383C83A4,0xB186FF1DC758A02B,0x63DFC531DD8621FB,0x7CA564E602D7D479,0xEB6C90C85E505F62,0xAA16CDAEDEB7283C,0xA49C857389660B64,0x142A5BC97D8E8,0x56F76C4A14D9B1F6,0xF5B9E1701D74BA16,0x582813FA2E6C9362,0xF3A9BF72B6D83F51,0x6498289CFEA60D88,0x4D949802B7A52E2C,0x157C253B27603,0xB0C78D9422EAC3E3,0x5A996A040DF60EC1,0x2CE38B8C1A68C3B2,0x324019DBC145D35,0x40AECED1D101AF7A,0xDF05F30D986BF31F,0x114F6738AE35A,0x8FBBE43323433F75,0xD9F70B1D48F6E919,0xA5FF95AA57796166,0x694488909C237501,0xECF5CD98D50D4D84,0x42305A893A3A5B89,0x22DB792019D51,0x648438DFD687BF57,0xF61E4C1B74B8835E,0x678F74D7EB0C181C,0xCEDB6CF1EA300B4E,0xDD2F2BC0FE7DF827,0x8BCB92A31D3E1EE,0x21401B6525345,0x7751DCF04B922749,0xE58042F531BDCCA1,0xE1CD1314F0867CD3,0xFC170A8E72428D48,0x34C70445EFCFDD1E,0xB97F8135FF785F85,0x21BD7B7590D51,0xC6567CA6DE8CB162,0xABBBD12F1DA57E3F,0x5D7F25655E080D7A,0x643C34DF6FDD02CF,0x789B7ADF973B2B11,0xDDFE75D018238F05,0x1409895FC3476,0xE5896DADFC183376,0x3A24F0A4EF5DE618,0xC76673AD21BFE5C2,0x741AF7A75EB424BB,0x3106AF6FB5149345,0xBDF38B4CCAF103D8,0x1420615E052BB,0xA555489CA8CF797D,0xDF01A3BECDB1135A,0xB2AB2EBFB8EEC55E,0x2F3CE2AAF36C7E5C,0x415A1876FEE609C6,0xF3F358872A897B1C,0x1D8826120B8FC,0xE88486451C96CBF7,0xE2E2D9A13ECAD72C,0xB1873801EAC5749B,0xF602973BC46AD717,0xAC54358A8E649D67,0xC1DABE98EB0C7531,0xA186651A1AE9,0x232CA3EC1C95A214,0x369D2A54946EB4FD,0xFDABB18D37FC7529,0xF6A30987EBE69942,0xC2C5EB35EED47C44,0x6EF117605909C441,0x21ADCFED14074,0x81E476377C6188A4,0x9342939A6EF03A67,0xD4200752A2BCE12C,0xD19864DC7369E144,0xB371267F42E0537F,0x3D050962A221243F,0x1B48DD7B16663,0xD8E246088769BE5,0xF24894222BECC047,0x89F28B19A3F184EC,0x2684DB124538EC09,0xACE5F2D7646BDA3A,0xC9F51B79DB4BABF2,0x6DB9E6B7BC7C,0x63BEAF31DD2A1A7,0x1B6CB88F1B41C035,0x169FBE6CEA1B5D26,0x84E9720E34938E7,0x90EFDDA4191EF9FE,0x5F846E6686BABA1F,0x203736769E832,0xBE3C06C737AE4D57,0x32EABD8FEEC2B84F,0x8935B6DF6D373D16,0x261665160CDABECC,0x77583FDBA470FE16,0x1EFAAB64FAC0EBE7,0x172B908CFA03D,0xB2495881F978EF98,0x1183B76919B0ACDC,0xE43DCBDA3692ECEE,0x4C397152BF2FC1E2,0xCCD10DC1AECDDE74,0x300E0A2F2DB03B93,0x16591C5A7B426,0xAF1346743AEFDF2,0x120724EED8F511E6,0x18DB81AB7BFE3F2A,0x31F9415AF3FEA6F,0x7241675D0B7813F5,0x4334E574BEF027AB,0x697E946D4273,0x59E13451A9CBFC14,0xF3BC74A0EA7B2899,0x2CA65A43270A7119,0xA903BB8303B36303,0x29799146D8019F6,0xD05A2F5CC427699E,0x366B08B954BB,0x26E173240FF300E3,0x1A529688A392BE7,0x7597ACDFC815832F,0xF8D7C0A9DEB8783A,0xD30C5E96DAE657FF,0xD82B537A1F26F64C,0xC1222A77CCB8,0x58FD170962DC4EE2,0xB84C1C815F4968F8,0x6F521842F9BD4DA1,0x337F6CE88C16A130,0xDBB55DBA14083F16,0x8D6A3E5B2A7127A1,0x36BAFABDE9B4,0xB8D58AC83C1A3F03,0xF087EE13D0076801,0xBE1ADE65A2BF4C35,0xC4B1B8C43C677524,0xD117A872F91B2125,0xE9CACAD7824C69BE,0x187253F46A3C5,0x841C7B4354D2FBDE,0x239C34E560DE5A9A,0x93C34C0010F9D970,0xEA2FECA543D49695,0xA00F362A0041F415,0xAAB308111E1943B9,0x1A9646162DA89,0xDCEC2004CCFF6502,0xE4997297B02EFA2C,0x3A67419596E3A325,0x995BA4FEB92B44DD,0xE3C8B51D8F37EE3B,0xFD82DD96AE77DF49,0x29C6345BABEE,0xFD3955F11FE5D8A1,0x85060110D4BE6B4C,0x515623082B7F5BB8,0xEDC9A26B02DA1613,0xD656F37B1BC0F505,0x53C690F513916B45,0x196F964148511,0xBCC6F9DFBDB2BD4B,0xD9118139A4AD6BE7,0xF181B1269CB8F856,0x7D4EBCE2E29704B9,0xDD66A21817B4673A,0x4D2B02527BA6E01B,0xAF8AB2E28B2F,0xBBE91FE634F92058,0xA64E05D027B8B7AD,0xDC9EB523AAE25ABF,0xCE93B1FA1D3B237,0x9049CA49C092607D,0xDADD5A2515F00B94,0x776E8B9F5FFB,0x7CD350DE75F2CC4C,0x40F7EF34345282E2,0x3232A18BAC233912,0x920B6F8EC6B1AD8F,0xA86E3E90D933DD47,0x4891C3E30951D81A,0x163D4D2E4C28,0x39FC6BB2C214DE28,0x8F1CEDA3A4713405,0xA0EF3116D15988CD,0x7085EB4980EB5A6E,0xB4A6E6633EAA1FEE,0x995153FEC491DC65,0x2B37DF7E4D10,0x24B0D641638BD998,0xE7EAD716E43F6E29,0xFC73EDCD235F61E2,0x4798D4223A44672A,0x95FF2BDBEA70F9DA,0xA534A49D6B082C33,0x160F307759A1F,0x9F412B3B7AF46077,0x34B6A3E08E76B835,0x32D83E43CB315E82,0x4E649DC0516678E4,0x2541188E52C5EE9D,0xD490678CE1CEA3EE,0x1CF8EB5FA889C,0x617F884924DA6938,0xB62C9030DA1649A6,0xAA5080BC30A967E6,0xE8B15B49E1E17B3E,0xD2738282E5DF72BA,0x82E63930CEE02392,0x1CE9CD7FDE3DA,0x1D8B02811A147165,0x12525EE6ED0B05EC,0x5CF6A56E1AD9EA1F,0x9D6A8C8D72CB5E34,0x49CDF00BD50305FD,0x82C622F53FD29248,0x16C1D4ED23A14,0xC9F3D410A1B88CAD,0x39A3AE807AB8BEC3,0xA429145FD730E79D,0x7FFF7DE82BBA3631,0xBDB1FA2E5A7B8155,0x1DE1D5ADA726930D,0xEC52C27DEE60,0x381B382AB2580E02,0x30C947F399D4E455,0x8F65DB13AA377953,0xBC0661E7B3571F67,0xBB862613135D68AA,0xC554A1C9B0FA1468,0x2324CAB533316,0x6E1C9A226AA2B0EA,0xF67D7604245789E5,0xFCA41A36D71C2440,0xAABB1D2F896D2A0C,0x3C21788F2E858023,0xEB6EC8C272269E4E,0xEF40FAB365DC,0x670F177E4E8D2407,0xD2D77112E87BD55C,0xC9134FF451BCCE12,0x56E30A297C86372D,0xB29D5BFFA5C10419,0xE630EFB50C627408,0xF0147452BC9F,0x24FC6D54FBB3D739,0x70D43985970805F6,0x64D9391F9C6E818E,0x4E24422D9CA6AC87,0x9614967652301C0,0xA1A82854AA548D72,0xB3D5F4ABBED9,0x947D3BF12AE4273F,0xB9A3EEF5C875361C,0x284625DB0D2133E1,0x5CC38FB493CDC07A,0x71A9420E8A8F1793,0xE1B260F50F4F6881,0x72B718AE0834,0x9114E93B8519CA92,0xCF05532E12D2E27F,0xE19A72906548F476,0xBE7C1B3A22ED283B,0xECE224A00354ED01,0xFEEDC2BE3885783F,0x2102030FF18CF,0xA7EB468A88159E08,0x7482B379A955232E,0x5089FBE36FCC14AF,0xC1AB095182B74587,0x9B7F40C899436F55,0x177061C6B102C724,0x201F19D2E0D7F,0x96F7D0952CF03BA4,0xBEF471936CDE93,0xAD6699D02C6FDFD0,0x191742F8A44FF8CE,0xD0DD627BE54CFCC6,0x471CFAA2DDB91F0E,0xB23B7B3A6ECB,0xD7E975EFDBC42532,0xBE7EBE32CECDC504,0xAC88BC71193B070A,0xDD1A7CE520B8BA88,0x31E33F5865B636BF,0x428A5F1DE0F7064A,0x1AD05C2ADC23A,0x81F61E90FDEE22D,0x9BDB22D3DBF4E904,0x172B4C424EE7A93B,0x595531CFB6719F51,0x90C3FFE5A860E188,0x766F9C816C34F80F,0xF49A34924046,0x7B997494716701CF,0x41945D900222B074,0x6A6B3BE957DC089C,0x2F81DC94D0DCEDF7,0x3F7E878FFC19D1B3,0xB3E93F0313270A80,0x2269612929807,0x636DE284C66ECE9D,0xB3E91BB39DE0A69C,0xDB02BC6650FAF782,0xC25447AAF53DD803,0xD6E0E79E6ADF998A,0xA7ADA45D03FA5F8,0x116C95797CF45,0xBA288815CBA9BBA3,0x3C159EA9B8C28219,0xB34F4D48BC2C454E,0x6746931AB531C735,0x2189814A657ADBBD,0x482D3A3B70088814,0x14911F072A8D2,0x89D7529D6D11BB2B,0x7453062C6BED4536,0x1D466047019F19C,0xD24C1CC4D21D0FF3,0x6F5D4F5A8D06D41B,0x5F63731C44665442,0xE505556A4F1D,0x15D87F526BBDECBF,0x60184D57E987294E,0x181AD10ED8AA40DC,0xE22B2FB40DDD1BA0,0x98E5140120736A97,0xCC803037F042A002,0x23213ECFF7758,0x27C400E011C82358,0xA47A808C64BDACFE,0xD1AB0D7A09CCDF81,0xD24AB97453F62288,0x2F135D5FB7070495,0xDDB21D053B275546,0x1F6FB2BDD890,0x7F18D6868F98B5B1,0x372D2A30E8045F77,0xED6FF463886180DC,0xDD50AA8122631E35,0xECDC6EC104EC6E4A,0x4110F6F15E3D8311,0x1DCCF1618A400,0x698BAF26E5FA1C58,0xBDB41ACA242CD208,0x16A41B92386B83F1,0xF48FEE4C5CEA43ED,0x13B20BC808DB262F,0xF37EB17A44A8873C,0x15B081629732,0xAE9E269F56D232CB,0xA55B2D25BCB62E9B,0xE72C7AC2F28C2F0A,0x7FABE9DA9F1BF11B,0x1D33F7F5B56A57A2,0x14E78D2817D45BA2,0x4C13BBA4AE0D,0x31988C6689F49DF,0x44BB54685FAD3030,0xFE817A2FA5FA16B1,0x94AE790A2F1EAEA5,0xCEFD375DA4283042,0x2CC375D363C1750F,0x18F6317D9EE23,0x96A7C878BB1EDAB7,0x40DEF16EE88927DC,0x47A11F4B4553DC3C,0x2EB5200C5BD46DCD,0xF11216909AE257ED,0x66F810F7D0FD0EDD,0x86C7EA89C724,0xBDC78AEDC89E1946,0x2507316DAD0A1760,0x92576B5A54C69E0E,0xF9DE8220301A0A2E,0xAB12032BA97BB634,0x8EAF464C50BB5DB2,0x983765B8AACA,0x7050FA6D8FD84449,0x2330EFB0379523FF,0x9B239F90F6B5D9AA,0x298B0BC8941CC5D6,0xA262BD196DCEE4BA,0xC1B2CB1F5A9E24A4,0xB6CE9234BD8D,0x55763A0860350B82,0x903F72F9E27F9EE9,0x92ECAF5A9AB163B0,0x53BE3141F78FE963,0x7FBBF55A04CE6753,0x70CCD8282A1EB302,0xFAB77E935280,0x97A1B799DE95E718,0x6735735B436FB210,0x4E5DF6EC779EC951,0xB11CD030BEF2FC0,0x8CAB7E6D905C43B8,0xDEECC7186BE02683,0xA2E8DE1FD05B,0xB1A5DD4A233B221B,0xA54D692FC443A4CE,0xC3535B32632AC1B8,0xBACDFC2DF8D132BF,0x7453DE996D1B65DC,0xD527370DD198C6CC,0x1372E31FA76C9,0xFC513FF7A1AA8275,0xEBAB567D19E6BD93,0x3E14E09E2AE10EBC,0x7CF23652EF97D46D,0x44EAC7F8D6D21756,0x71BBEF3AF4C514C6,0x1820E9F4C0A90,0xA0F70CD352E65EC6,0x5BA297366D5D06EC,0x172D0496867CA573,0xA974C0E61C1A78A3,0xFCB44912483BA062,0xCF9F033B3184EAF1,0xEC3A1A6FE572,0xBB001731A9241AFE,0xF3F2699215D3026F,0x51777995FAEC1563,0xDA9C83B345BF2D61,0x118DD8C6A754FD7B,0xF13C6A3E602F41A1,0x19431AEC3C434,0x795850F05E5E271F,0xF78498DDE7B4A2F3,0x70186C47C97999C6,0x32A19947F1861E91,0x9D050A4C6F565CB9,0x9987BFC34F526C9A,0x1A8AA2F68DE43,0x897EEA13B54D6AB3,0xBCC1C110541A25D8,0xD2F1D2FF4081C594,0x50B2B6126797689E,0xD6645C1F9E361219,0x192C50AC1D580066,0xE2B8044B69A1,0xD227EE067C9228FF,0x398A95CFA1DF1787,0x8675E243261FA767,0x391999E7E98E2FC1,0x7260A3AAEFC4D38C,0xA30F88EAEBEAFC08,0x16E680BA81CB3,0xD7B84672C191EC5E,0xF0793167F5641EF8,0xAFFB66DF5C034BCF,0x92B537D95D289E03,0xD8B2BD07B8A655A,0xC206A12B3EFADBDB,0x1B4C4A784C2BA,0xF44B63B96369A385,0xCF195BC8DC712FE3,0x76665BA9F80225FA,0x8764B3C5400B5893,0x6A71AEC475B579CE,0x5BD5F48DD10535B4,0x20E09A692ABAB,0x9591C582189350E3,0x5655D921F5E14D99,0x5FE29F23223A2237,0x38DC941C6B005D83,0xE193C91233855D92,0x3605D4392B5E7AD2,0x1FA62E6D34CFB,0xC7DE4DB15CDCB9D,0x44704A69EEA41F31,0x7FC700BF07FA5E28,0x5D61FA6AC409EB22,0x4033734C53B49E4A,0x90F078A1818B99C0,0x16C6F17B7AF83,0x40C880A34B0BAD26,0x847B1C9FD259A3A,0xED39541E937F52CB,0xCACD945B8B5AF23,0xD1C0C703C47553B2,0xCB902D4B90102100,0xD74B2826394F,0x25D73CE741F11368,0x668A44B950A0B31E,0x936D6EC7011D0B71,0xB47CAFF6F94A226E,0x85FB21E9B1524403,0xAA4EB71A1F2414CB,0x93B431756B9D,0x453869FE009B0F8,0x4937DDE2377E3E8,0x6D29E4F40ADCC056,0xB6723D52230A0940,0x761405E0A81BAFC6,0x1D7471F8BBC40EC9,0x166FAB17E2477,0x209E606C1B0B3E3E,0xF49F8DB334DE39BB,0xA9DB3C8C1B17EB94,0x7DC1BC406F22AA5,0x74A71B23C37C0386,0x8EF6AAF2EE75493D,0x3F6EBDBD400D,0x51608F8736C982FF,0x36A8D9B78374832,0xE221AAB047D504D,0x1AC87F47BF9597A8,0xFDF8AE43939A742,0x113A2720A82CBD7E,0x9CBA01A03D95,0xE6C37FD5E6B8619D,0x3F2DF8127C083994,0xF61D2A54A6A79A22,0x2C859508B25BCF71,0x50A09C9FCA27227A,0xB62F6322B85A83BF,0x88882CD1B74E,0x3833DBDDDEA6C29C,0xD9A5B3B2C3F49E0,0x417382E525C9AB20,0xC53E95FAB42B7D14,0x859D3723831EAAE7,0x5D3AC504FC0856E0,0x144A2428FA8F9,0x43CB6830595142A,0xC2E3864106321EA,0xADDA16615917617A,0xC7070F73589DCAE,0xF2D5C4C9B2E17C3C,0x26AFD6D7BAFC8C79,0x10AA03AC2CFCA,0x323DB09DF302E24,0x1ECE188A530F0B67,0x7A76296C95EE8EBD,0xA96774E6FBDEE30,0x3902867308187695,0x9A5D05EB4D17E866,0x580721F4A00,0xFBD6B23A80A6916,0xDBB752775100C19A,0x5FCE6BD6E51164B6,0xE72DBE46E280D41A,0x50DDCD7E786498F9,0x38BAE021D815751C,0x1B10314A0C4B5,0xB6F7260F0D2EC50C,0x8C16C335A0788B5E,0x4187D533017E9F88,0x7A0285C24F2B3EE7,0x4861931A336C69DA,0x40822BDC7AAD9A15,0x1B9316667023A,0x2D0831C265CF01C8,0x8647FF51B9F9B9A8,0xEF9BF6CD31463725,0x1258BDAEED556174,0xE2680C9B573401D0,0xB929BD1D9C5D4D52,0x23117CE03134C,0x6262E02EE28D7059,0x353CFC17FAC1C076,0x921EC6DD5CD7E640,0x8DD5CBA3C4CE8AEF,0x7F03BDD8B01CBD90,0xBD9A115E6E441948,0x21867CA90BD27,0x8EBD906C144421D2,0xB05AB9AC931F87BA,0x77A8C15D66339C85,0xF87BC517E134126C,0x49212AE9F55DEC35,0xE37E65DC8A7C4980,0x1C2C443258926,0x1B8CC19F4AD42E63,0x3E81133E706E2AB,0x8EFA02667FB1F21F,0xAB1C6271A90B1FEF,0x419DFEC03683B645,0x7FCE39E8387D6D7C,0xDCC31F749641,0xBA2D5EA8087C8BF7,0x82AA80517EA5226B,0x5CF30FF2DE0BEA79,0xE62D7E5CE2F6431C,0x1B4602E5103F8C54,0x7C49EFDF854C5073,0x10D19E8929D27,0x932ADF478F92CAE0,0x50146B75B82D7A53,0x6EF72291203F27BC,0x122E249A8A5C52D8,0x9B87B652A089C4A4,0xBD77548351CFFFAB,0xA77250DFAA1D,0xB88A7F23FD12A44C,0xF64162524CE2440B,0x8FAD24535B221134,0x937787FD0DB091EE,0x8A45ED4B551CA87F,0x870BA995B0F09F54,0xEC58B50C27AE,0xCAEFC33770AA6A7F,0xEE55B59C9E3AEECB,0x483AFE2EA15AC92F,0x64C5E54547A1F004,0xFABEBF9EB719C4F9,0xD0EF8BD9477BB451,0x10BD44605A8E8,0x33D4D9C5A00D5C,0xBC8C6598DDE090D8,0x9DEED6E852D1ABA8,0x91E751BFAB805FA1,0xB389BCF8C609318,0x390D7BA07091AF2B,0x13C05BBFD258F,0xD9D901171EDDC6EB,0x76D454CA4EA58188,0x2E1A4B118DBA4B62,0x1F59E15E3435AEFF,0x57C071915292870B,0x2BEBBF71CC803E5E,0x14C6DEAFA9A9B,0x269D2213AFE61828,0xCDE63325D2DB420A,0x3CBC0CD731C93E1D,0x4BB3829613C92480,0xCEA4E3ED7DA29B0B,0x4FB35394B2A137B7,0x196B2FA14D4D0,0xAFE964433BE893FF,0xD4AA932AC981C620,0x6B690FEE1880CC14,0xAEB48FD1248124B2,0xF563A102BED41524,0xD9E0B499D137B880,0x4B45BD575376,0x5BE060E58E3F7DFD,0x607DC5AD258A2103,0x183A33BFFAA7A857,0x8C5475C782644568,0x111FF3CF9AD73713,0xB330D7425903E61B,0x16FAB9DFB80D0,0x606D5E27C884A3C2,0x640BA774AAF57740,0xACCA4CBB219515C2,0x727C74F67E5A9DEC,0x78ED71F2185A349E,0x67BFE0852EB9BACC,0x109BB201E7939,0x621207936A3C01AB,0x76B64CAE9DFF6F2,0x211351B70AA126D4,0x9377C0BF444C945B,0x35ADB51DAD723C69,0x5929E31C8225711,0x1C2812F844174,0xCADEA7F50624BDD6,0xF1511E569AD59D8D,0x9957EB319143C99B,0x76BA8083F5080FAC,0xB8C798E5BEB4A189,0x380FBED1AD9AAAE1,0x1F556FF5337B1,0xB7D4CB9D5EC3A289,0xB6962C9DC550BBAA,0x70FAF7620F4DA69,0x4FC408CBBEFD197D,0x62F3EB31C964B6BA,0x25C15B846848BCE0,0xA794C7B283A8,0x556B4C55A7C45519,0xBBBA511DEDBE4BD4,0x89050314E5D246E0,0xBFF68D99076CD928,0x463F6AF3497D6143,0x1D28F7765EDDD509,0x22C3A44D6E41,0xD2B3288913F99990,0xBFE568477F99F884,0x2AC90C8FDC8C4694,0x3ED680E57EA01737,0x37A8A04181A5CE2A,0x6F91C2C1D6789700,0x19CA325FC9CA0,0x6EF23D9E4E6367E3,0x3E3C463169A8640B,0xD52840654E1F75DA,0xD39C12A1286CED3F,0x66304F5916FFE348,0x420D8054223BCEC0,0x891DA5BF56EE,0x27341CB39CD3BB74,0x96318E9BFDD0F94D,0x7092D288FCCD3D9A,0xB83892F3AF6C0C87,0x27DBF18BB7234B60,0xFD42DFE887D60230,0x78D065F963C7,0xE84B615FB6703AF2,0xE9A1487ABA8BFBEA,0xE82B88C77C222F24,0xBC019BF96F780F09,0xA70ED89964286B05,0x7023C916F74067,0x1920FFF6327C5,0xB3A576C1CAC64B68,0x7EA901C532BAB6,0x9246F9DF36769585,0x93D9C5DDC8D511A5,0x4B843872CB6412E2,0x887EECFC3EE6ED60,0x15CA5D4AC6B53,0x3A7C301AFE25D819,0xA77BCEB45B016D98,0x90FBFFB88B4723EF,0xB8DEF0AB99652012,0x1EA463C13BA2124E,0x229906DFFEBF7011,0x191348CD3B8B0,0x2B3FCF50CFED94B1,0x69956AF12F8A601D,0xC2E638C5F73CD7E5,0xBBC7CEE22D4CC26C,0xB722A4C41EEA74FE,0xD45FED19CC643C7F,0x233F9181761C3,0xD7D949125D41CADF,0xED747666A30CF71F,0xEF37CB05D30B12DA,0xC5DAFFD6681E5E4A,0x5AA73DF2D191A97C,0x6E2492A3BF57B5C,0x1D392A5D8E5CC,0x1E7208FB1A108004,0x4A158F72EA518CD5,0x6E3A0BD925DDC469,0x57DF2F8258BEAAAC,0x6618B5B6B099C81A,0x36ABA934111403F1,0x215765EBAAAC0,0x1777584763E783AA,0x5D0D93EF5FB17D5D,0x4FF28A157F52F31,0xC52F2A1F13CC2305,0x765644F242BC5D4B,0x46BA3099F6DA6B9C,0x9769D49729CC,0xE0B85949F7D999FD,0x31804BBE30DD8C1E,0xA648816639521A74,0xF7FD74F04EA74E0F,0xC91A45B1727D3917,0xD506B5A4F9A58AF4,0x194FF39690551,0x8C54BDD59FFB7D4D,0xB615D9EF09378BB2,0x34548ED338AFE48C,0xBC6BFBA837396E60,0x2A617CCE63A9F780,0x2EF32324407F648,0x232BAB331F562,0x5F0994E7E55D51,0x13A774B9433C1B57,0x8CA2A724F3F2D989,0x6DC499AFF0D60E43,0x12801D90C631EECC,0x5F4B872515C4FCAF,0x1637DBE7B6E23,0x822C45290E73A9A1,0x27EAE72AF51B424B,0x71AC2380A7148ED5,0xC48E1473EC348DAA,0xF8E1AB291C5351D2,0xACD0118D9D3FC9BC,0x1BA91D9C991BB,0xD54CA6F8F700C7EB,0xB3E3B6C798E64917,0x2D7581CF639A938B,0x9A8C153170DD161E,0x373749C58CD131DE,0xCC974A399F97A215,0x126413F19A2B5,0xD20C93FC471CDDDD,0xEA6753E7DB93279A,0xA8FD19573CBB7AEC,0x2BF80E8D33FD4AA3,0xC32C8DB67AAA1F0,0xF750C007C6386A98,0x133E3879FF276,0x506BBCAB910F5650,0x2876EE132A639FB7,0x406A3BC820EE1894,0x325F0986DB658A20,0x68C36C9C89F1D396,0x215DD51949392E34,0x1C9417D1919C7,0x168C7C08FD0638F5,0x512815EF0F09ECE3,0x5FEE261FDF0C4DC8,0x5026C5F538BE32B1,0x661AD44646BD010A,0x1B81A47A9B5BF56C,0x17D6E5AEF73C6,0x2840676EEEA61C84,0x667A23BE8BD05B17,0xE69E96C3E1750863,0x371044929F82D743,0xF71C905B74FB1A26,0xFF661F6D58D0CF30,0xADAB4F930271,0x11689AF0F1F44A6E,0x9182F5FCA5F526C8,0x8B891433ABD380D7,0x7F0642B49B36366C,0x90BFF49A9BC3F0BA,0xC928316D79A5768F,0xB1E02E69101A,0x355BED08FD7BB9FE,0x94000E87C8EAEAF7,0x603223C142FD519,0x1589C4E67E08A2B0,0x60D4C7E8A36E8AB4,0x5EDDCB0DCBAF8E32,0x1E6AA9E72FFF6,0xB659FE9BB927DC4C,0x94F02A04EA1224A4,0x93BE841E411A4305,0xB87A814CED357D9C,0x72F1ADB8758F2FD6,0x34A3EB9D278A7888,0x4CDE0DF95E3A,0x4475D9BB4042E344,0xFD5EA9272A9375E2,0x78E8DF894A22A4AA,0xD93D22995655090E,0x552FF88B55653A69,0x972A15474FA4B6AB,0x8190DD6DA589,0x36066EE3D62C212B,0xFE1EA5D78078EBBA,0x3B7C86DB8D7F7915,0xEEE231ABF631217,0xDBC64825272BD839,0x530741801B3C91A9,0x13A9C167843E7,0xC46BB040620BF5EE,0x9605392771578746,0x55E0FA1FE15E595E,0x775217CDF523B7B5,0xBE35CAF2EAF3EF6,0xFE5710BF1A5E30F6,0x14D066107DB1D,0xCB29E93463D819E2,0xCD7DA128E8798819,0x6E40403F712A7C4D,0x123EA947E85908AA,0xF2701B9895B74A19,0x1F850660F2BEFCDA,0x1C93866F0110F,0x783FFE8AAC2132BE,0x82163F31D3C7ED2,0xEFE04BFE48890763,0x30D0F991D335C231,0xDA0609E4CEB2112,0xC15C67C354C9CA38,0x168555425F115,0x481C9974C474A802,0x8F27F96E3D284951,0x101929E95AB3A412,0x6BE40F393DFA8339,0xDC03ED161BAB49ED,0x605A4810F88A1B1A,0x1FEDCC7D924A3,0xFB6529AE4167A7AC,0x8F497D89AC1ABFA5,0xDCCB096C15DFBF1D,0xB91EECE13086D895,0x802B732593F319E9,0xEE6A0CD350598AB1,0x19CD0F30AD370,0xE7C4EBDBABE37FAA,0x789A39824180D35C,0xE1A33A69BF2E27B3,0x34B7BD8C332620A0,0x181D57FC9EEBFAAA,0xA1F41DD827E9C5B8,0x1BB3CEECBEE97,0x8EAB3264A91ADFDA,0xA08B2B4BAF4FB1F,0xF633627A97C42887,0xBDBC20C1B1542ADA,0xBCB112D2BDB7B810,0x32B31D3378F1270B,0x20837A899DD05,0x8A5AA931D151DFFC,0xB8F13C4F2A7ED728,0x8DB193A1A88BF4C6,0xB1D4D72E6DF9235E,0x410266C817B0054A,0xF91EE8D47B80CE8B,0x57882ECFF68D,0x440072FEF2355474,0x8D938F776C8FA59D,0xED6E689255807F6D,0x652CE0B80CD157F,0x3FFAD39A60408079,0xFB4DF04F5D5CD028,0x1B6348AB1400F,0x10F944D5347C5280,0x998C70B87D759DBF,0x82665F73E54A79B6,0xAE079975FD435FE3,0x974D05010F2A0254,0x4025DCCA0AF3554C,0x11051B7F995B2,0x69A277BE7045C53C,0xE65A54254A92A66,0x2B2D4DC2D8FA051A,0x71399E41D23B328F,0x158839DA8D7E3EA6,0x160208393612030C,0x6260B1D3D771,0xBB5B9CD11C629FD6,0x3D5B571A3B0D3303,0x5A58E53962E6DA79,0xC431D8551FA94437,0xD6AFDB2E27991B6D,0x36B331AAC23DAD8,0x168CDA517C8F9,0xCE63368999CE75C5,0xD6ED29DEC3AD60BA,0x1721D879D55ED9F2,0xC7F8EB63FAC6BEFC,0xE235F27E35130F13,0xDB12254F1B759A0A,0x1F664C54FDA0E,0xD5C44428A5F16146,0x200A226F43032DB7,0x738EB9F65FF66D37,0xC76837F98BF693EF,0xF48F0A49BA62701D,0x50DF3BBC70ACF9C6,0x13D4441F9106,0x9279AA93F3AC7166,0xC0AB550D08F13B2C,0xBD8C3A95FF26FB28,0xD34CD9D9A9529D92,0x39E14EDEBC5B2400,0x5525083508B50D26,0xB372191BD318,0xFDEEB9A21EBEE2F6,0x90D72D214CE9A969,0x1CCCC55D7FBB336C,0xB095772CE0A1E824,0x6F825BD40036133,0x99DB04A28E74AE76,0xF1F943E318DD,0x6CCF0A27DA3D1D3F,0x296A4B6B3F425614,0x81593BA180F95DFD,0xA27CB10D067D93F3,0x84AE80B6CE1836ED,0x1AEDA6CB12A531E0,0xCC6CAA6E908F,0xE702A08630B3A6EA,0xA4A33CF9442B991A,0xB227F74AEEF489EB,0xD4DE729C44A61D15,0x69DDE402EF17A34C,0xC5DCAA05F9C63E77,0x1CF65E17EC9DF,0x23463A5BBC203F2E,0x35DA261E73ED5145,0x864E997F32ECC1F7,0x3CD72773AC77471F,0xB1471C8CBDF1F65B,0xC9CE72D5DB281200,0xB68C9626FB4F,0x6DD153831B23B6DA,0x9205B27A5B86B2B2,0x59C7EFED84131878,0x60A6B0B0BE13E104,0x573AA91138F30A49,0x8D0102053CAE2681,0x10035F7845622,0x1D5A6B5240EAA614,0x8E1382178DAD0CE9,0x8DFB211A42272121,0x773B80357E863913,0xCD54007C11571ACD,0x959B4A7A2FC52B6A,0x204C19E5AC9E6,0x627B4BEF3889BB26,0xD36FDF72F8CF1CE3,0xDE7C5E550CF8D086,0x120E7831A48E59CF,0x4554195EAC5779D6,0x1B18C63E1B30DE1D,0xD34495BFEE3D,0xCF45A904E91BCE09,0x585D386BE3119EF6,0xAEE6292C30FA8B7F,0x698D0A8AA447F6DF,0xA7452E45B26EF81D,0xA01CC28ECF20989F,0x80F3560E5D33,0xB284EA1F67C7BAEA,0x1C5869259894F879,0x62F43BAC11507FDF,0xC0A2CDB6096CFFEB,0xB80C89ADCBF38FE8,0xCF67432187251709,0x2D74ECF00FD7,0xE084247F403EB4B7,0x427785CB80D65A92,0xDCAFFB9401D25C8C,0x5C835695880B361A,0xE95FA77E49CB9988,0x514B97FBBAC3A4C9,0x1E1DE0134B959,0xE5A4C0223A7E2A36,0x2AED26C535019DC3,0x2B9118EB35000BB0,0x2293E9DB0D8777DD,0xCF5CCD0E590261FE,0x18A53BD77BF4BBC7,0x19F450C42843F,0xB3CE243598353115,0xB394420E270A2E92,0x20862B9521AB6389,0x9AF51328EFD30D85,0x726660514664BEFD,0xFAED4162E3ACBBEF,0x1A22272FD322D,0xA887403A9A330B28,0x7094D9DC988C2968,0xB864AC54E54C412A,0xD9A5D5FDECA7528C,0x9440F125E3BD588D,0xD3AFCA2AE06EA133,0x1F771A088BD16,0x727AF0A01A49F86,0x3845D0A2AA03E096,0xC5E263DA139A3EB6,0x40DC10875144EC34,0x59C548343E5B1A77,0x7AC1B156A7056D44,0x133342CE12F72,0xEDF92BF27D93FA26,0x6997773C4DF2533F,0x857B2448B54320D2,0xCBB9FD19F676F8AF,0x96A1F6BA660716FE,0x68AF5A4A666F7E64,0xBF26653067F4,0x8A52DBE2B12B747,0x5A295AA43D051A67,0x86592E633A342559,0xAFD804BC917DC463,0x76C187E4572B40B9,0xA94010D95D4FD5C0,0xF52854A82187,0x83D60C6D587CDCD3,0x37A5D9FAA80E6F97,0xAB9512167694998C,0x4483A204B581A85D,0x5B173C60FF010D19,0xF7501C42C7DA0F61,0x20B3238E75C8B,0xF70F4150B5AEFB91,0xF30CF74595704F72,0xEC22C752EBF5E923,0x8ED53C379B4F2EBB,0xB9601F65EB7B369F,0x801D0D8AE213065C,0x8E7FD4C9EA51,0x17ED201DC672353A,0xEB621473A21E84B8,0x1B056E60200BEA01,0x232B7705E57540FD,0xDC556FA4C3392B7B,0x15FCFC3E11F4E0C8,0x1A4B64BA1F38C,0x3A2D5CD0B2ACBF51,0x57D9AE5D4C8A2810,0x7955D7F34397CB8B,0x11B76E942912BA80,0x545D436E1B392C87,0x7A6693BFF3C5532D,0x953B2A1DEFA4,0x6E64D43A15947672,0x2F2626F68E0C2613,0x555AF0E146EFFABE,0xFD86755FCDE2AFF6,0xF63FAF17A07138FB,0x93A35E894175EF40,0x14ABDD39EEE4B,0x3BF383BF1D3AA573,0x7BB9A79B7AD6314,0x64178F71EA300451,0x1EFBDBFB13D75B1,0xDE1FB519946244D,0xF7BC99F68EB8DA74,0x17299CDC313A9,0xAAD33A7CA636763B,0x83E8C5540E6521DB,0xF4028F6A3EE124A2,0xE9A6F675F0FA7FBB,0x1A6FE37C7338D985,0xF659FB9F219A4198,0x534E592781D8,0x89E9C38AB46818B7,0x2D972C4E68010DF9,0xD01C291DBC832180,0x4F52CD52056C6A4,0x3ED8ED3E7010595B,0x1CE0F6059CECC7B9,0xDB5022F67FE,0x55F2502CDCEA5F08,0xA95BD1A142B1E114,0xF63E2E8DC03227C6,0x8FD3653C28AE276C,0xDD29AFDD6810A2D6,0xD9855498E83661A0,0x212944D76A217,0xF80C31E07636F014,0xD120BE05140A198F,0xA10A649AB8C91994,0x411C32709A19718C,0x356AFB6CC613A39A,0xD80BAF2480FFC662,0x6E8F9741C545,0xED53A4980BE88FF4,0xB16F17D0FFE7DB9A,0xB6A8FDD07AD64664,0x3BFE70CEBCF72019,0x992A915414C2830C,0xDA5D3A3C516BC1D4,0x75EA6E383DB6,0x72A172268BBC366B,0x3BB5A2B7ED4C1E5,0xDB6B151194AD2231,0x6DDBC606A105E9BA,0xE5163A9ACDC08961,0xCD69F7BAD9655383,0xA6ED89C28150,0x505934BF0EE09602,0x3951C25F2DBF55D3,0x25E7969F41F9F43F,0x6F9A98A2141EFF3,0x135B3E2E311BDCEE,0x7AB5114D73E13847,0x1FBE9B1010B85,0xA2A598A41EF32990,0xB158B588842BDFBF,0x65668480F2E5826B,0xA585E2C825CAEDFB,0xAB645FD32C5FBDE1,0xB1944FEC8AB7DCDD,0x717D08383149,0x723556DB4AE3F8D,0x3A061A967FC00EB,0x5F84F34A176EA7C8,0x13E833DA3CC0841,0x627D4459816CA516,0xE8EB5769190C3702,0x83AA284AE112,0x3E6267BC774B923,0x5277030A4A1239D3,0x3A0030370FCF05F3,0xC6621947968A0D6B,0x628FDEB8AB66231A,0x29BE3F5878401799,0x11DF91B775429,0xAE7FE94786A4C494,0xBD149244C10C0148,0xF5F4C3112E7008C0,0xCB4FECDCDD695806,0xE02718E4B9092EB6,0xB14DC9ABBAF84E08,0x1221DE60D111,0x48B2DD458D21C58E,0x4FC2EF2E46305264,0xBA6336E2B50DC468,0x2FB0106F3FA5FB3F,0xE87BAECD597BD335,0xBBB0EDFF19A0C876,0x1EF2ABA4C7340,0x5747CBFA67FB0BF6,0xBDD96F538500B095,0x3B031FDCC2B4667D,0x21614CE2B0A27969,0xA12325167CE5ADD,0xABF08DBED3BBBB2F,0x16610954F9E23,0xA81D391B48F4AED1,0x5B2EB16E4E73831B,0xB75C2465BE239,0x19D38A17F93D40A4,0x62FFA5F216D79968,0xDC8D25F1A4C4AA3,0x21A6C55FA0200,0xC585C31ACDF75F15,0x27AEF5C9140170B6,0xEE19086E5B6D0FC2,0x459FE7A758AEA172,0x464C922A87C7AE8C,0x1AAD2756DE22286A,0x2058561DCF1D7,0x22D33C1827F0FA12,0xA972F7AB4E5E7DAD,0xA18A447BA5560CE,0xFA2846360AE9B3C,0x40162C7D1730E5D8,0xEC87C83040A13683,0x1BD90BC4B5C5C,0xA9FCDE707AFB153D,0x9882EC56FBA1ED0C,0x30B8C93314824E75,0xB711DC79C48EF1E1,0xE7E1648B3C0579D8,0x636976ACF25C3E19,0x127043A177E6E,0xB5D192B3EE9A23E7,0x422BE6B774D4BC76,0xBB89520CC35CBC31,0xCBBE71FFB6BBD82E,0x5C54E6179AE0FCEC,0xFA4BDDCE75F52,0x1522E954633C8,0x33EE7862561A9B8D,0xA321A457A3DFF2BE,0x46F2A888940BCF25,0x2D41739A595AB351,0xF9294C0BE4A61FF0,0x19A2392C4E6AC3A0,0x2131D1926DFB9,0xCCFACA3944C3898A,0xF5B05B42B2DE8F6F,0x89DEBB951289B7D2,0x97528DA7388ED69,0x859690A39F79F237,0xA6B6A34A5B07BC,0x1DAC320D5BFFD,0x689B0C1B565DAD87,0x7064CA2681E3FD44,0xF9A4DCF17B951102,0x62B7979DD66B811A,0x7581E4C4DC1BBB4,0x3BCEDE81351EB32A,0x1F72D2966B8F2,0x26BF49EAC52E6C1,0x6349760EBCFACC98,0xF8BCED60063E8ED9,0x61038B47E6547121,0xE149AD2950B097EF,0xE3FA1422057A7270,0x8877EC664C6F,0x1F1BA7BC87514070,0x43225480FA8C4D6A,0xF00EB6ED8E468648,0x6A228895FA6E76B6,0x762D270715AC01B6,0x6376281A3BB490BD,0x87D038D8678A,0x76558A856D4DA426,0xAC6CC2503E0D7304,0x40952E707A390EF5,0x4D0CF6E3A949727A,0x5A810611C296F26A,0xCFF52FBE90D9FAF1,0x878D97151DCD,0xA94076878F5DC53D,0x37EB828ECE9D23BE,0xAD3F6CBD35639B4C,0xDF32276CF971921,0xD90787E6025C608A,0xF7038C9A82996366,0x102DE083D9254,0x2400E370B7729B1B,0x26ECD7FC92F71C71,0x841993E12C5644A8,0x691D65765393CC28,0x4A0105CEC34EF466,0x65B4B79219DBC296,0xB1C0757CAB50,0xA729F9FE21A2833B,0x8D016C0E66701591,0xBCAFA717E6FD9AC1,0x8BC651D1CBBE6CE9,0xB7B79D5FF0013053,0xDA621C1283A371F5,0xD0583B04F2FC,0xEE5C3CF3A5E454D,0x828F259FC973886A,0x7B2C0449948B1211,0x68160AFF2BDF4661,0x96BFFD773B5FA1BA,0xB9234054E335D4C8,0x13F93081AA31,0x52BE8C2BC0286178,0xC2E1C8B7FFCB7FB9,0xDBDF28BA5F060721,0xF22FE315FC712DE3,0xB3FC2AF310110311,0xAB1DB071336F4772,0x4F2531E9E9E3,0xDBF25825CE5B87DF,0xD0ACAA8D88CD14F8,0x8B44D62533B2EBDB,0xBB260A46BCA33C16,0xE6112ADDF9558226,0xE10024322A8A2948,0x1B9C476B5FABD,0xF56A8955251AEE46,0xAC55D6C5168541B4,0x48F477FEDE57ED19,0x213EA9EDDE4E05C1,0x1AFAAD1BA38CA514,0xB8E19AFC454A1B46,0x1D84E151F318,0xA30316B15BBAA8AF,0xCE50B8C68586AAD2,0xB10B661B82C96FB,0xE4CF1964EF27B9AF,0x253E26DEF62B06E9,0x57236BE50E85F81A,0x1C661090010E2,0x6289B56A0F6D2F27,0xCB2E182072C49989,0xF34A9846A9884331,0x6A21E060A0A6FC6E,0xA372633F46E99F4C,0x9FC4ED35B80068E3,0x1BDA432BF2B7E,0x4130D893109DB90D,0x2A02AB6CD2EB6C63,0x808E691F7059798E,0xF6AF801D71600AB,0x5802F3DD638D2869,0x9ADCBF1612C542EA,0x222FE8BB1ED83,0x2D5075833158727E,0x896551ACB849CE79,0xA681AF617647F788,0x1723B5EDDBE45009,0x9B5854E0EB1640D9,0xD62688D00D7182BA,0x19BA6EEC7E1CC,0x77514AC85F8FE7F,0xB1809AE785EA2C68,0x6C6ED5EABC9A5F7,0x70674BDF0E7404E,0xD56E93EFDDE8CAC9,0x2417F53265519D4C,0x1114DFBEBCAAE,0x5AD2B639D609EDD3,0xCF707AE8328321A0,0xFD0EB9CDBB3E9B67,0x3DCC84DB2747F7E,0x788F99E1AA81E624,0x89285D9EAD58FAF2,0x100EE7BAABF21,0x9ED5BAD1C958A533,0x8ECBEFD2F7E8004,0x2C51E8D9FF1986C,0x28BE4F7E03E8F96E,0x4E31D4A1E2E19C4D,0xE13532B544BFFD85,0x63807C8EBF63,0x4E48DB2718C189,0xF22832BC2DBD319,0x1014B3D4C276F007,0xAE0C8B5E22B206CC,0x94D40F9081FD4286,0x7BC1F98AC603050F,0x22D2C55CC94CA,0xCB6B8F9D520E9D80,0xE539AB381B667F8A,0x249202E9866E1249,0xF527C91DA98D2C27,0xAC19181EB83403FD,0xCC88C5918696F4A6,0x5001A3D48F32,0x3BD9746B54431F3C,0x82A5240D55F89DC4,0x4A1A73BB64F9B140,0x20F82C747F685C92,0xC19FE8486FDED4FE,0x2203D84CB49CC0CE,0x1E880FF73EE6D,0xC0C1055035534E38,0x6BF0F508C87BD876,0xC78F9C503C5EEF6E,0xD7092C550FBE8E50,0x750D9410A1E5619C,0xAACF3B982D1CEABF,0x3F8C7931F23A,0x10C502FB6A788788,0x6CAD8C424FB80F2A,0x281AFA60A1BD881C,0x425E76CB1AF1249A,0x4C7E2F9FA9F48D1F,0x5BBBFA8CAA3895DB,0x1CFD817D564A4,0x46F35660279179CA,0x11EF6F70B27392C0,0x4F590F0007CC38FC,0x9EA48BE3C4E4108D,0x77DABDD786EC26C3,0x6ECEE7012A0C6E3E,0x21D2708C80DE3,0xD028A05F7FBCDF3,0xE23F62E1D51277E7,0xA1049833C2C6D24F,0xF11D6272CF2134D0,0x53BABC624861B4B2,0x15B1DF06BB15C091,0x3A3A945AD4AC,0xDA12F91A68E8F80A,0x89BFB00F6CAEBB43,0x7D6ACCAF46B5503A,0xF968EE4748497AD9,0x8BC751C2CBC912FA,0x398B0B90322A413B,0x129537A34AB95,0x42DEF50ED03E12AC,0x92F10A9635CF4E9B,0x87AF82A7E5CFC7D4,0x605531A601F853C6,0xD406D851F92E5092,0x76D6EDB374B4D4D0,0x1A83105F8A7E7,0x3210B451A6540F70,0x2C4A1D0B5BBE7552,0xA1C9AF1457B75C3C,0xABD09E267401E097,0x395249825AFDF274,0x65011599A8ADB361,0xE23CA61F5963,0x1F06C1B2C8502415,0x97BFA56270C1F684,0x41042BE0EE32CECE,0xAD070A3AF8A7412E,0xB8E7F8ADD3B7AC00,0x8FF699112963E227,0x225C95920F085,0xE2DCD17133CAD0A6,0x2B9DD50EF829E74D,0xBAB9A951BBD39FD7,0x7B72851AF45FEE07,0x6437B201E68D797D,0x4E088B2032DD6F5B,0xCBDA2083597E,0xB7FA854786AF5AC0,0x7A5DC8209E7291C,0x1169AA6E93471504,0x6F86598C111C882A,0x632AB04739CE3699,0xDE9876A7284C1D84,0x84269C032152,0x5F8B038957660A8C,0x2845430517B568F7,0x8ECFC5848B09126D,0x7FA5CC4B07ADF64,0x8C729000AD353D9E,0x5C32F17F662D27AF,0x8BE8097F6AD6,0xAC776325C734F611,0x2FE2385BBBB71BA0,0x5CF4E1ED77737755,0x25FF01B2206AF6B7,0xC64C29BA5F5BE5CE,0x5D7325AD16951F5E,0xDBD3D98F45C4,0x38851C2247DC008E,0x5C2035037022E1B8,0x35331A89EBDDDB41,0x67AA9D69D5DA7510,0x2A0D1E22A301C79D,0x202B960831A9338C,0xF15BD902ACFC,0x1D5F98F9C510139,0x9699286F37571E35,0xC3654879975CF966,0x7F219D3601188814,0x4629FF37BB221CF3,0x42478C2A978FE177,0x196B840A07260,0x433ADA22100BD55B,0x56782481A63FA05E,0xBD70CCB1EC794FA4,0x34588790D6958E96,0x9404208C1DDD0B08,0x2904DF0945CE569,0x98D2D93ADD49,0x690C2682E502678C,0x4FF44A13A6B575AA,0x7BE7076FC43FCABD,0x2EB25C4B0E627C21,0xB79B9B03FC8AB4A3,0x801EB8F992107560,0x795E147F91FF,0x7C0429B65FDBABA1,0xD42E2FD3B0E69488,0x1D5F8DC0BCF43FC7,0x6BE9B306B508F1C9,0x6191F1A1C7994AAE,0xD701D04E58D22380,0x2081420012AB1,0x6464DFFF602399BD,0x2B4A8E2A574A8566,0x8A178A50375B0E47,0xBBEFA4479784FB5,0x4630EAAB397F1036,0xA3BE81440A0FD98B,0x18EE771405161,0x3300990E354CC128,0x6349E80F1B91B149,0xB9E1EE6A73FFE650,0x1FA782F82FDB52E,0xEF2024DBAFD18437,0x39847ED4720FA408,0x1CE5081E47C6F,0xA02F0C97DFAE7E47,0xB637B49189CB0814,0xC3F74B323BE03840,0xB31534EB340465AD,0x3234E3EB38BFDD68,0x7DC07CCB02369028,0x1F414C8FDA45B,0xBBDF17A1189F892E,0xE36B701711EBAEE3,0x9246EB958653A11B,0x83CDF571CC3ADD4C,0x2995DA9A115E2745,0x5E59E02C5DF39A62,0x6C681B957EEB,0x84EF739E89074420,0xF7756424F99DE4BE,0x4379E183A608DABB,0x7DBDEBBE169AEF00,0xBE1665D536F68CCF,0xE1CD3876BC9C435D,0x1055E38120373,0x390D402CE5D3DC44,0x8CFA0A3677B6515F,0x5B5D988515E05F4C,0x69F9F742A9C22D90,0x8E6B864D6F3E299B,0x5347C49B541CE666,0xED5A3DA86FB,0xF152430FBB0B1DBE,0xE3568C1C356BF46F,0x1364FC57DF940142,0x53A0739714D93EDF,0xA7E14F8906B19D5,0x84C8104B46A7DD44,0xDB2434BADA3B,0x1F3F7FE37E16E784,0x8E4743BF96285B45,0xCB2146A4B91A2E18,0xB3EAB0B7937035DD,0x96FF028F37DEFFB2,0x187B521A5CBDB905,0xC7786821C584,0x7C9E71B5088BD450,0xE377B6299A84EC23,0xB283FCD8652C0821,0x9ED4A93C9A53B760,0xD3895089CBB76193,0x248C2AB502D79774,0x86E9551D6334,0x4CDE82E4D578720E,0x897CEACA236F4406,0xD47C0EBAEAE4154D,0xF65FF050DA8B7E0A,0x78660BFEC5B21A5,0x98F50C69F901544A,0x17B78FF6A399B,0x529A98F90FB6B43,0x636BADD6951C6E5D,0xE3C69E3E5822D8BC,0x34CC5C43F5459C7,0x1CA19A068EAD9C4A,0xC5F411EA50E85259,0x1BA8FD736BDFC,0xAF809CF54BE3F6A2,0x4CD22CF0D1A9827D,0x81FB22D3234D3F38,0xD54BC4644CBDFFF7,0x2218D94F30060068,0x2F308E0193F0DDE3,0x42B77E4BB367,0x94747DE95EBA3DAD,0x6153BA2FA01D449D,0xE6CCB30D62922F3C,0x621940A023A262AF,0x1E1765A6F7EAAFAE,0x65084E461A39AA9,0xB6265C7D9074,0xE64A7E49BC1D0BAB,0xFD5F6F13C39DDE4,0x5768344CBABD07DB,0x3B24F6E7BBD4A16F,0x38D39AFDE3B06026,0x408D4F292B7FCB1F,0x1F704CF10ADC7,0x9F06FCDF010ED47,0x85623188384B6C75,0x1519B64C612EEE94,0xAF14519CB1ED965,0xE312A658F621B753,0xA73F4CDA462F0D37,0x2279A1DDA6C13,0xCA30355A6DE382BF,0xF5AC1F244DD80125,0x99FC0B644A578DBC,0xE0EEBD340362DAED,0x429284E4C3590DF8,0x3E80ABD06A82A75C,0xA70950492621,0x7804B0FE5705FFDD,0xBB4A15E989766C82,0x84033A2753BD5E7C,0x72D7362624013906,0xD19E7778B3BD49DD,0xDB56DB1412EBC326,0x1ADEEF60D6A58,0xFBF764916C8EA9AA,0xEAEFCD394012C09E,0xAADD625ABEC0F454,0x7AF0838DABF003CE,0x9B4868F0E6C4D282,0xDF0A63D100B3DC43,0x16FAE051DD39C,0xA6BE7DD4152463F6,0x684815D16113586F,0xACA6C2912CD33E4,0x4A2EE4DDAC35C32E,0x238BF66DB232AA57,0xE78EE0AB10B8AFFD,0x14D1A8692A152,0x62940A3930B9750E,0x9FAB4A381E83B99B,0x3B4349F3A96F3CA5,0xBDF20EABF8279605,0x69FF11F9497B171F,0xB754B8BE488D1AE5,0xA126632841F2,0xADB287FB8AFC8B02,0x2A53C3A01EE1F99B,0xD77E6E5CD5C1606F,0x942FD2705214D548,0x58D1A273371B6895,0xA76A78F45484E58C,0xE531D48FA704,0xC02E927D40A05C90,0x7DCE2C82CA8473D9,0x3DAB6F6FD45F6D51,0x2E17D123993311B5,0x27196142BA81E33,0x61AFB718AB8C2B4D,0x18672489CDC62,0x48E38BB704F6AEE4,0x4B8782D5C7941576,0x30DD7C8269D1B8E0,0xD893FF3CB1482ED9,0x3EE7C152F98D5687,0xABDDD372C3264441,0x7914E3317D28,0xA89E2EAD2F70DA66,0x8D1E248381491E19,0x73D99BA53084D9D,0xE57523BE9DA21504,0x40E63C14D205DAAA,0xB1219AF57EAFFDD0,0x874A3661CEBD,0x5CC2511E038D022C,0x8FE05A9BB10BA084,0xE0767D5AA5532E60,0x6669736EB1EA62FC,0xA95ED77561157EEB,0xF4760642D8316F95,0x15559C7EEB2F3,0xF6966A49A93AB734,0xFC88BD4B3946B069,0x2CB884198373CC28,0x77363BAC008101F7,0x8CFC4E02E0A837A7,0x580FFAF65F6EC0EC,0x12E9A21D5002C,0xD266E6BE728C9BF4,0xA542A87E398C9AA8,0x50F61BE614858C5,0x5DE4A9F7DA1A6FD7,0x52A08152EAF7DE3B,0xBA6AE916FAD47A5D,0x1B7BA0310D51,0x24116B47D1FE6659,0x4E32F7158FDCA0E5,0x1CF1C7111CBC8BA5,0x7D6C1BDAABB647F1,0xDBBEA064282D81B4,0x7D05EE18F2EE7D90,0x231FE745F73A8,0xC1A591DDECED3709,0xBF70B66A6AB2638B,0xCE7F185ED93BA38F,0xDBE994643E5AECE9,0xA492AE884E240DB0,0x87AD9344EEAD69DF,0x10E9DBBB4E870,0x47F1214544F1FDFC,0x26C218639D0C745E,0x361A32B9DB65AF82,0xD582A03B1CE0BEC7,0xD4895F190FA8E4AE,0x7A6EC93531DF50F0,0x1A91214205262,0x2880FCA6BFC771FC,0xE8A1ED366C5EAF41,0x26DEC38C89ACC9AE,0xF433245E87B28253,0xED43C4B08B4E1835,0xC605E763F1368AD7,0x15274B28B2E38,0x768B9867C45D4BC4,0x4D682CC1A9E19CCE,0x1F015815A86514D3,0x9801C170584526D2,0xAF7C9183EA0ED04B,0x12FC4454B033E85,0x122D555B3EB2A,0xFFFFFFFFFFFFC5E9,0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFF,0xA1397C7864FFFFFF,0xFC589FA8688D839,0xF86335AC52097FD0,0x1BDA7D338DBD6,0xBEF0CE72B6E9D1BD,0x92E0A4F2CA7CCCFF,0xA0B2035BB47215FC,0x6D36985E0F7DFC52,0x6BB989478F1D3A0F,0x8C5988BFDF5F4F7C,0x6E2AAF9B9D9E,0x15D9717B534B56F9,0x57ED5C5F28EEED33,0xAA224149D1792C1,0x2B663432FFF290C3,0x65B645B7893F3761,0x8C45450423A64BF5,0x2072BDFEECFDF,0x2E9A77227F6B4DC7,0x26A6145E0CD52870,0xE8E7255BCCFF6551,0xC04097B01E64680A,0x56B98EE30BBE9A40,0xF3BD5AE69AE90DEB,0x1ADE268AB0C0B,0x15D9717B534B56F9,0x57ED5C5F28EEED33,0xAA224149D1792C1,0x2B663432FFF290C3,0x65B645B7893F3761,0x8C45450423A64BF5,0x2072BDFEECFDF,0xD16588DD8094B238,0xD959EBA1F32AD78F,0x1718DAA433009AAE,0x3D80DECAC49B97F5,0x250CCD95259A1463,0x793F04EFE6DC126B,0x863CBE6C6738,0xFFFFFFFFFFFFC5E9,0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFF,0xA1397C7864FFFFFF,0xFC589FA8688D839,0xF86335AC52097FD0,0x1BDA7D338DBD6,0x410F318D49162E42,0x6D1F5B0D35833300,0x5F4DFCA44B8DEA03,0x908ADE1CD38203AD,0x100CD330A23B7494,0xE0A2D716A265D0DA,0x1C5F4777BD5A5,0x2880FCA6BFC771FC,0xE8A1ED366C5EAF41,0x26DEC38C89ACC9AE,0xF433245E87B28253,0xED43C4B08B4E1835,0xC605E763F1368AD7,0x15274B28B2E38,0x897467983BA2B43B,0xB297D33E561E6331,0xE0FEA7EA579AEB2C,0x65BFB50A8ABAD92D,0xCC49CAF44749DE58,0x6BCC9B9136C1E1D0,0x11149D163881A,0xC1A591DDECED3709,0xBF70B66A6AB2638B,0xCE7F185ED93BA38F,0xDBE994643E5AECE9,0xA492AE884E240DB0,0x87AD9344EEAD69DF,0x10E9DBBB4E870,0xB80EDEBABB0E0203,0xD93DE79C62F38BA1,0xC9E5CD46249A507D,0x283ED63FC61F4138,0xA73CFD5F21AFC9F5,0xF28D96A14FE5CF65,0x8B0D12F720E1,0x742C,0x0,0x0,0xB90FF404FC000000,0xD801A4FB559FACD4,0xE93254545F77410C,0xECEEA7BD2EDA,0x0,0x0,0x0,0x0,0x0,0x0,0x0,0xC1A591DDECED3709,0xBF70B66A6AB2638B,0xCE7F185ED93BA38F,0xDBE994643E5AECE9,0xA492AE884E240DB0,0x87AD9344EEAD69DF,0x10E9DBBB4E870,0x47F1214544F1FDFC,0x26C218639D0C745E,0x361A32B9DB65AF82,0xD582A03B1CE0BEC7,0xD4895F190FA8E4AE,0x7A6EC93531DF50F0,0x1A91214205262,0x2880FCA6BFC771FC,0xE8A1ED366C5EAF41,0x26DEC38C89ACC9AE,0xF433245E87B28253,0xED43C4B08B4E1835,0xC605E763F1368AD7,0x15274B28B2E38,0x768B9867C45D4BC4,0x4D682CC1A9E19CCE,0x1F015815A86514D3,0x9801C170584526D2,0xAF7C9183EA0ED04B,0x12FC4454B033E85,0x122D555B3EB2A,0xFFFFFFFFFFFFC5E9,0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFF,0xA1397C7864FFFFFF,0xFC589FA8688D839,0xF86335AC52097FD0,0x1BDA7D338DBD6,0xBEF0CE72B6E9D1BD,0x92E0A4F2CA7CCCFF,0xA0B2035BB47215FC,0x6D36985E0F7DFC52,0x6BB989478F1D3A0F,0x8C5988BFDF5F4F7C,0x6E2AAF9B9D9E,0x15D9717B534B56F9,0x57ED5C5F28EEED33,0xAA224149D1792C1,0x2B663432FFF290C3,0x65B645B7893F3761,0x8C45450423A64BF5,0x2072BDFEECFDF,0x2E9A77227F6B4DC7,0x26A6145E0CD52870,0xE8E7255BCCFF6551,0xC04097B01E64680A,0x56B98EE30BBE9A40,0xF3BD5AE69AE90DEB,0x1ADE268AB0C0B
			};
			static const uint64_t ph3_T2[DLEN_3*(ELL3_W>>1)*2*NWORDS64_FIELD] = {
				0x37ED7978F7323982,0x4936C76D51F21F5,0x16050C18A973EE88,0x72F7BF23AB9375D3,0x47C989C9C28BA7B6,0xDF611D74933C983D,0x1BC0CBF2A04F4,0x8A0EAC4B85F5303,0xB686A96CB86BBC16,0x8F36EBD4A443E09C,0x948222B2B2D156F6,0x8A66ACFAE2CCD48B,0x98DEEDA161F27B50,0x103F846A96DB5,0x694F92DFDA192A6A,0x53DB650F045FBB94,0x914D7DD5D24B8A7D,0x332A4681B4537855,0x77B22ED3C97AA0A8,0x29F51E43F01EEBF0,0x2BE99CC2EA3F,0x50AADD31483272BD,0xEF71AFF18F5CD2A7,0x2718B25973322EE9,0x4AA70233D7135E34,0x18E840E9D3D8137D,0xCC10E80D37099E81,0x611A286EE0D1,0xA9354A35B269FF9F,0x6FD04B074DF96AF6,0xC3106A5031DDC3E7,0x96D97DB085DE52CB,0x52E69CD3E0656B33,0x413B47B6FD5C630A,0x99B4D7906FA4,0x4773AACD7C006827,0x6E0D9C18E227CEB2,0x970AA29DDA4FC705,0x7DB2A73DC99EB,0x503DA274C8A57B9C,0x456519435726C6BD,0x1269589CECCC9,0xF03E85B4AA764A,0xD6A2D3F58CFCB874,0x92DA0409FD01266B,0xADCC6564A5D4F6F7,0xC02F15A74EC243E8,0x7730B32C01DE2F53,0x7E846A2516F7,0xE133F07C2ED3E1D8,0xB4326C2931139F04,0x6A51505413E95C84,0x736ACF58BEEF524F,0x538B2008AF8272BD,0x4BA606901671E978,0x177FF06B895C4,0xFDC3EBCDD657CD6E,0xA247E9F561464943,0x3C1009CB51346175,0xF002A2B6FA005AEF,0xEDDF64F46D2A177C,0x83518EC200172743,0xCCB04068DD44,0x45C5EF8ED846E9A1,0xC594E2935A404A15,0xF1FA86D9065998F9,0x3177F334A98AB972,0x683137B06187B6C5,0x763E818699AF9A82,0x25CDBC8D6DE3,0x27AE2A62883696B6,0x591D09EF41B582E3,0xAA5B4530E1582194,0x99CCB2BCAD4E2F47,0xAF7B2BBC1BFAEF2A,0xD2D3BCB820C00A6B,0x184CF997B843D,0xD149B9FF09F549CC,0xE7CFAEAB0F338F18,0x9EA6958AE87055F5,0x408E4C22C8263BC8,0x17E57438EEDDC7DB,0xD58BB12D4423F92F,0x1D7D0C433AD30,0xC7DCBE1AEC6D7C2D,0x8F6FB8EDACEED8A3,0xA92B7BA065B357F1,0xF6A92D93E4F369B4,0xD0B517306D4B0BAF,0xF53ABA5B52BEFB27,0xC56412AA646B,0x4B826A4831175010,0x67D08097812EF865,0x1950DA79789DA00D,0xED0B99011BB1215C,0x5E5B32AC7AE8C701,0x561D7DE509792047,0x1C21010F2BCCF,0xF057F3BC88DF9B21,0x8B8EE060C16EE48F,0x2FDAC5C9B23B3ECB,0x6AAEE97E18D02869,0x5EF58C8866A37082,0x9D42D0D7D776AAF7,0x9106AA9E5BF6,0x7F7F72CA5B66613B,0x49FF121C8796E2,0x446921D8A9193C4D,0xC46710A67EF0094C,0xEFEA22E111B76E16,0x2132DDE3208096C8,0x158537695C848,0xC9B7E0C49271E5E6,0x9232D78410486561,0x15CB972066F892E8,0xA221C57C5BC43FDF,0x3CD91966B44E4005,0x4313785071F6AAD4,0xCC4419E3E98C,0x83EF71190034677F,0xB73C0DA961FBF58C,0x2D2FA4D06846EF77,0x249169E3A3B98786,0xBC1076683BEB81A4,0x45D0E9EB5D1324FE,0x627AFD457D2E,0xA2D211FD38E81A4E,0xE74DF34D9BDAAD83,0xFA9276E47ABCF316,0xE306ACDDA72C24FA,0x5B68BF41E7F5BF84,0x322D98FD53020106,0x8902F84E1521,0x8C44089102E680E8,0x4CE3AD9309B0D800,0xE2AFBFE63C21F2F6,0x51D7493A7E0E5F24,0x5F4EBD5D1CCA229E,0x6B49826312FF74D8,0x91AB97F84071,0xC363D921AA5766B4,0x5F78F0E41B0F5359,0x4A43F68C67C5BCE2,0xFEAC89F6BF98C204,0x9D7AC06DBF4FC7B4,0x26989B3B2D44C774,0x1B6C805550CB5,0xD253F7739CCDB075,0x8318538FAD5D33A2,0xDBF3CCADFDE3674E,0xC6773568F794E12D,0xD6A71F2C69C17A5B,0x53FD7701967BFE2A,0x5CB62ADF7B66,0xF1876400E4AEBCEC,0xDB7D4068365A6399,0x8D2B3C07E3AA1D93,0xB763B70480DB937E,0x6B209F9E89C1C38E,0x99F233803462481C,0x1703585A8F941,0x53651286AF21B7BD,0xB5DC8DE712CB9D2B,0x8C8653F83462375E,0xA13993C7C4E915F2,0x964055D5E5966731,0x357C020389033773,0xCC534E50DED,0xFF72F3C9C8A650F0,0xCE49F4C2868EF761,0x458103FE62D5D7EC,0x5EDF0259B91D5E5F,0x1D08EB6816BD60E5,0xA946F4C5D79C5C1B,0x223E0CC250BC,0x2B51A049820E158D,0xB0AE40314ADDBCED,0x7ADA8CE98928AE66,0x54CB91E1C6E32237,0xC5BA6BBE489D3E92,0xC61076FBB6204480,0x16725A031428C,0xC9B7E0C49271E5E6,0x9232D78410486561,0x15CB972066F892E8,0xA221C57C5BC43FDF,0x3CD91966B44E4005,0x4313785071F6AAD4,0xCC4419E3E98C,0x83EF71190034677F,0xB73C0DA961FBF58C,0x2D2FA4D06846EF77,0x249169E3A3B98786,0xBC1076683BEB81A4,0x45D0E9EB5D1324FE,0x627AFD457D2E,0xC03CD433DBFD1940,0xAD1049F408EAFDC6,0x9413F0B3B5F855D2,0xB466E1E85E7E45E1,0xD353B33DE6AAFDBF,0xAF62EC1EE44B00A9,0xEE42010787DB,0xEF422805A8ECF8B2,0xEFC179F15231A992,0xBC9F1827039347A,0x60D8FBD734560515,0x60E378BBF16EA78D,0x96AF9D3C96D4CDAF,0xC16C643E295B,0x19CC0BB9E81D0F2D,0xC74613BF54DB4817,0x372D926DB39C3CDF,0x2CE188DAAE451706,0x2E9D1F86190D2D03,0x7AF6FEBA83E7E98,0x1239E39C886C9,0x18016CB681973207,0xFF2EEE394C8FA492,0x7015BF66C6DA1CB8,0x682A804E6C62CDBA,0x9ADB8BA7AC3527D2,0x5CD6BD2C7F57E672,0x1F7FCBBEAF368,0x67CBA14AA525913E,0x2A26621D55F2EC8B,0x3B31A5CE62B44712,0x213B2FEF1D0CFF9D,0x9E30F07CC0D2D53A,0x883F84866709C5ED,0x18DAEC58600EB,0x90378695AB6DC70F,0xF28CF06493FEF316,0xFBC1B50FE10764EE,0xF0FD9C62941B1F41,0x81FE1F9C80FBC203,0xC276A212D7C0E28A,0xA894531B6E73,0x82035719D951C650,0x5D73E4E48C7F1CF5,0xF661D60807952F30,0x3089A85460242F0C,0x2880A4DB87185002,0x999A96237AFA40D3,0x147336C9CC194,0x348C81B999814533,0x182CB5F120A91139,0x918C4131CA70B7FF,0x9078A6C623B9E8BE,0xAFB29263D67669ED,0x53C8A59C45318A16,0x189C6A62F5FE2,0xC9615E77B9A331CD,0x796546F9415693B6,0x49E2874740F92975,0xEEF96C473880186C,0x6BED96B40F3F4146,0x6F0183BCE2AC1B17,0x22050F405E829,0x65707E635ECB4E9D,0x8B6CB4F957CB894B,0xA548A3A39C9F6A88,0x4FBE409960FB15CB,0x2122ACF1EF2316A0,0xF5A6FB61350D3FD4,0x133795A80A6D8,0xECFBFB77B533B175,0xAEAB50CDE3FFEFFD,0x4409546B8780817E,0xCC08202B547B4343,0x8A8DE838FFAFB80F,0xD22E90C09A6FE7D9,0xA9F2F5FDCF0D,0x718C8E815F5758F7,0xA6E06FA6EAD8C201,0xCD15CB0EC91BE0C4,0x42ACCD25F3243E86,0x35E63846562FDC96,0x88D2A98AD796AD5C,0x2257C6CF6607A,0xE19E872603CA1A1C,0x268DB952AC008521,0x488E3BC0BC2AAD3E,0xCFEAFB68E859EEA3,0xC83D2F8A26E6C25C,0x14AE9DBA32791553,0x120681B9F27EE,0xED858188F54282DB,0x2A27EE3D8B657406,0x8D200617C2EA3B9F,0x4F0D0C886B0F473A,0xF0522717AFEE00B3,0xA45453EC4979385F,0x1206830AD6B2C,0x4CB72711CA850294,0xC211F90452DE238A,0x80E6D14309CB28FA,0xA30C6A97C9FF1E8A,0x2430A4CF8D66F69C,0x80AFE7DB642FD042,0x6868939979C,0x459FD7E2EF627373,0xBCE705FA699DA4ED,0x361BA49E03059B10,0xBA3AE86694E7AFC9,0xAECCF50BB95E2BD4,0xBD9B1FD6D6AFC93C,0x144E505989609,0xAA02D69A7AA9DD27,0x3BE7ED4F6FAC748C,0xDF08D3FFF1CE042,0x594A45EC41BBF34,0xAF092A2E24662426,0x32907C83259EF8A,0x1E654FD0601CE,0xD8AB9B4F1464637,0x914D439A7D3AD772,0x1A01657B9A7C2B06,0xC7AF199D6DC47DF1,0xB6948222D9D57B5E,0x3CFAE56B4E6A1BBB,0x1A6986F3D68E8,0x543D675586FE1C40,0x277125FD4BDE4D4C,0xBD3E505763231FF3,0xEBFA8EEA4AB8BB50,0x86ACD986B784D3A6,0x9C195DF7E8B7DF71,0x1C5B107CEE734,0xBEAFA7EA26F62970,0xA161A3CF8CA00C57,0x42638BCEF0EEE383,0xF43A6E5ECBC63EB6,0xBF1E53996CD8FA66,0x218EF3C352D484F9,0x1E849136E5513,0xE8C5A33992BDE5A,0x1E43F6F8ADC4687B,0xCEB1842C9087709B,0xEDB68CD7E5AD47E,0x6EE1502AB55169CE,0xA8AB5BF4CF196E9F,0xB16B329F570B,0xBBD392B020E491FD,0x94E0A3821045F058,0x4507C63B48F72909,0x7F06CB6D6E81C97D,0x8F1BC7547C099C62,0xA9EE5E02C3A26F1B,0x17005DFFDF107,0x8DC92873F960C071,0x12249643C8697FC7,0xF0D001E41AA019CA,0x638299852191FFC8,0xEE8C83B34DEBD97C,0x9F1E8DC7ECD7CBDB,0x21B0A9F9C368F,0xB0A5C2278BD9701F,0x54C3A8880A40BF84,0x5299B8BA210AF7C1,0x7D2CD9CA19C3E3E4,0x6C3DABCDFF981D6B,0x9D2C2A08F5928065,0x1095A87F8DC52,0xCF6D9E37ACA7FC41,0xE3BB6F4674DA80C7,0xDAAC5E9A0A44C2ED,0x7D5307AB1434786E,0x622342575645CD0F,0xE28C64BD43BFED4,0xB39968FC5CFB,0x938BBBEF5B9D2EDC,0x364AD80C20AFC700,0xEE2A7B39C1AC7E9A,0x66A182254E550751,0xD29EA3C001982C8B,0xBB51E4F503A3C906,0x1958524411AFC,0xD750AF1718FF1697,0xABAEF7EC842D0D3A,0xEF02AFF7EFB7CD4B,0xC5121B848AD9B8E5,0xD9363292FDF0BD28,0xBE7349B7CBCCC18C,0xE37C0A921C25,0xF81B601F370AB515,0x9718CB1D9BC77332,0x820B69809E9552AF,0x71469CE03D6C82D,0x82ED4CA05BB20F09,0xFA0EE0B3176D3AAE,0xF00F29F220C9,0x99718695CF56D1F3,0xFA0CF4A65E2853B,0x6BFC00CE66341DB5,0x9A1FD3D8BDE2168E,0xA66312AB6D1D1429,0xC4F6C545D87F3AB7,0x684ED280FF51,0xBB4C6122596AAFB4,0x9975728556A3FC4,0xF737FEBEFB08A1B9,0xE2179CF8F34CA308,0x1F733D8647D99B33,0x8F161334971B8D14,0x302DCAACE7F5,0x3CE82C92FCA36EED,0xAD86BC910427CBB2,0x3ED9D973F0A74442,0x1D5FA3CB5B76A6A8,0x72A273A14D634320,0x6323DA781445BA42,0x93D662E789C5,0x798CCEF93B4CDEE2,0x7F6C2D4D96E92EFC,0x6491DF60E516CE2A,0xE8401A029A644F0,0x3014E918D0C95BBE,0xBE324BC5C8BDD1B,0x1534511694B9E,0x1EE45FDBA7F4D0A5,0xA1CD76FD4DB4EB0A,0x7FFB78522A923AE5,0xACE69A5D8CD388AE,0x46A71C0069CD7FCB,0x66688F6CCD8075F6,0x10840992DC3A1,0x825FAB3D3068B880,0x2937434C524CE60B,0xFCDD3FACC3D70A74,0xA1B284E716631EB9,0x62FC10ED7CDFF026,0xE8C37CD20E024194,0x2127C297F67AF,0xE4E770D3A1998E55,0xE50F7A43CB8F3B7,0xCC0EE4F49DEE0608,0x696621C1C702829,0xEAAF7372E46AD1B1,0x6A697C9793AE126C,0xDCD786C808DD,0x62BC51904C27EC96,0x98659E35ED3CA154,0xCBEE68DBA5AEF1A9,0x84A6828DBE8486AD,0xB62C8CEBDA58823E,0xC0505E7F1EF6C439,0x3E7290CF7FA3,0xE7FA00100D00929E,0xE3DB9C87A19D44DF,0x39D3C0882DB52D98,0x8C44470CD6E81EA3,0xA7FC95CCEA497684,0x47DE411F14569424,0x1752EA982B5B6,0x7DAD95284EE41937,0x1E0416B96E7A5F8A,0x8C469B3A0B572529,0x4633E7F0D55E131D,0x88BC355428CBA688,0x8C158EF6F0E3684B,0xDBC4AA85FA49,0x4FCD82AF68488203,0x434ADECC0CBC9140,0xAB4F65432D70A4C5,0x32554FFD3D12AA5B,0x2B1E44AE79A19FFA,0xE890137A684FF01,0x4D89B9D12502,0x49F246888EE6329E,0xF17596FDC6C14127,0xABFC9B519BE18D18,0xC18702604BF0F5BA,0x9E139A8BAD27F11E,0x315E9799DD6EEF31,0x7ED9C9ED346E,0x259120C9EABD2A04,0x296350D8CC35E881,0x86F68443CA640C6F,0xA973BF6907A3D5D2,0x14037612F335A0E5,0xD8D2268298E642DF,0xD8266EEAE590,0x5249141997855FB9,0x9AC7EE090E433A1,0xED35F7C7E9A6E973,0x436D4670CC885A0C,0xEAE89063A007DE81,0x60485A975654EE3E,0x8A62063C2F5C,0x4CD74E53DAA52AD0,0x8B6F556B11A8B90E,0x20A9BC9880C1D9CF,0x10E64C7CFF911437,0x456570EF33273CF4,0x3BE32B31D3247ADA,0x642A69906731,0xD118CA3CAFCA633D,0x67F678D57401D949,0x14405A601FFA334,0xCFDB5AF2D0CD9DA6,0x13C98FE237000D4F,0x979D8CAEA95F896,0x1AFFA915B9008,0xB12A326ED9BBC676,0xD7DE5B0BD4522E47,0xA9E6D8842B2A1664,0xBEB0497CD50C458C,0xC5C316A98392C67B,0x4D9E4B5EF3567E40,0x1CB755BFD5674,0xB8C5831C6AA49F69,0xFEB8A0214FCC1E5,0x15D197F836B53533,0x24D813E652BB408A,0x6F2DECE9FCEAB2E2,0x6E516A4774407181,0x221F1B18E2342,0x20D3CA8D414DAB5E,0xF0258C41FC6BBB86,0x43433C374F87A54E,0x4774088F8B175EE5,0x213EA6CCF17658C1,0x227216743DCDE7C,0x155CE7D303C44,0xFD853D5D19CF4854,0x392508A64BDD3A54,0x894B13B458027F1D,0x8CA48D136C13B85B,0x5731520BD0C4AA8A,0x52CAA13C82DBE553,0x68EB8E1B6529,0x991D3CB9450ED822,0xE6D8228341C9800A,0x88C152836E6B96F2,0x6143BE170CF079AA,0xADE44C525EC2BE9A,0xA52C016B7528AE45,0x17913156216B2,0xF038877734157001,0x412D0AACCAFB28E4,0x7FF74C814593CC91,0x43A9B81BB791647,0xF22A302086992DE,0x7465164CC9E82926,0x295E9B47747B,0x8ADDEE86131054D7,0xE97A6178E6D79720,0x8EA7ABCFD67A1D62,0x2D22A45E719681C0,0x72AFBB8AE11C6C0A,0x3CA2C1D95465587A,0x1E3C708CF09EF,0xA2A1EAEF86053BA1,0xFD69E28F3A2E9CA8,0xFD6031FC1739E60C,0xBEDF49AE806DE468,0x506A2982151B920C,0x5BC815D2DD99C568,0x2DB77C0FCB84,0xB7F2C2639C5F4553,0x8DBFE37A13E726FD,0x1FB2C22BD0730383,0x670BA25D3F2D0FE,0x28090FBB3D0F1339,0xE601E04E4AC10937,0x119FDA27FD06,0x9ACECE4446A62F62,0xB55647F833B6BE4E,0x80303FB87C1BD961,0x7D6C1DC4BE8E38CB,0xD3C9366A3A099CD6,0x9F1E2D6904A29529,0x1EA55744F4F2D,0xE43EC119B8E89DFB,0x4DEB15DD1B9A0AB2,0x64829D8F572AD605,0x940936CCC608F92C,0x51F47E94B0861D24,0x492F7EBEAD3CCE12,0x1FC3876F37613,0x25C4A6F996DC1323,0x680A8A80AB9132FE,0x8E3C791DEB467E64,0x428699677462FC06,0x2D843757B703FADA,0x51B28D04B3643CFD,0x1850339234F22,0xD0D39F0D27C3F75,0x36A57108AA19DB81,0x3412A086F8A66907,0x4B78E3B9B09561F,0xE1B0121D7A7A8816,0xD25323C860715C66,0xADE8C97D8AD2,0xEA616353338D3792,0x4C6B66E7D1963131,0x1FE234D20B168EA2,0xE8A1C169B89571D3,0x3BD4A80643A3461D,0x4B7BBEB9AD84B95,0x18EE838EC0B41,0xEDAF05CADB8E0291,0xDAF47A15CC218ABA,0x61B556680B7725E3,0x53DCBE74C8924F35,0x7DD58616504C7A57,0x9B312880430480A9,0x1CEA9F68D9AEE,0x5A319CB0E0D6BAAE,0xB6BFD0B499A604DD,0xEEADB7BD07D44F62,0xE9A6CA161ADFFBA5,0x674C902305B8715C,0xF04F383D689837E2,0x16A41ABC094DA,0x17A3112EF9C267C8,0x5BEF4AEC399A35F2,0xD9D791D66F35CD4B,0x4BFE5DA02ADD348B,0xC235D0C6A8BAB24B,0xFF93B782B6B7B3FA,0x156430C23B60D,0xB6C4FCDE05363EFB,0x332D1402A997931B,0x580795E607F457DB,0x37483F148577486A,0x9EBCBF2A142B5717,0x1FB805332A8E4DA7,0x1DAAE25FF7308,0xDAC0756CB9A6388B,0x6C5E9371737E7ABB,0x10DBE7EACA81D420,0x6FAD339D714BADD8,0x1A0A29ACBE5C1BBC,0x85ED2D2523A62DD2,0xFC60934611FA,0x4E584C6FB9977354,0x17840354F7B6E37C,0x7347CA89776DFAA5,0x748CDD00CA7F6D4A,0x68B823C3EED214D7,0xE0D165BBAB35CB78,0x22F14B5CE6ACD,0x2FC304A5D47D947,0xE76F3E170042CB7B,0xEA1B19DE16729BB5,0x1C99DE37BD74AF94,0x7EA911A2BF31D874,0x4523575ACA0A3004,0xA1201EA47044,0x1D59FFFA2F6B49EF,0x9BC1933D4685E736,0xCBF48D315AF92B99,0xBD571321EFC03BB3,0x35CDBE486117B689,0xBBE90B54E3A84466,0xE5A421AD97F7,0xD1D273C0403DE823,0x781B73D97BB67E1,0x51F4D81C1E0AC129,0x9F339FE1A254FC7,0xB2996EE412AC0B92,0x1760A8A6E2FE0990,0xFBFC00C9B60D,0x4C91E4B4B33A0C23,0xFC39A597CC1E04FD,0xA7F7658285DB1A43,0x2A2B5772E3B3162D,0x1E4A550641B5E3FE,0x33C3B321B9B99780,0x17D6DE7225983,0xDC163718EC88FCAE,0x2E96A8F49BAC4E25,0xB5E2C4E0904984DB,0xE63C7AE8E9F26E3C,0xE0D827A74367AE7,0x6152BA01809BB89C,0x2353FC221195,0x59258A37906E6CD3,0xC9859BE3ECFA239B,0xB78EF80A9130F0C4,0xA9D17BB4E15803E8,0xEF5086C5A5D95CF9,0xAB935CD3A0AA7B02,0xF1228EC686C0,0xE13199B41023BBA7,0x244F8DAEF041E1C7,0x6081826C3CFBB069,0x1526E89CC84081F0,0xBBA7ED68CE80D381,0xF08F214B88FD8660,0x21B3AC0A52712,0x86ADD8B3DF54BC12,0x1023CD6D378A7A9,0xAE3E15AA4ABCA9B9,0x2E31C7BDA0AC2F2,0xCDBA7286B5DDFFC1,0xE319332FBAA5F892,0xA31657948561,0xECA480ACB43BCC1E,0xDD01AD6A9ED2785,0x92828D3EEABD6021,0x11DB3994B9A081A5,0x3984BA2D4AC653DF,0x44335D2844147169,0x9B7D558EC564,0x1F838D429EA62CC4,0x4A0D87DEEFC95AE6,0x4F8B5D39F65C7140,0x6AF51F2CF4EB68FB,0x4798FE25308722C4,0x3677CEA9F214ADF7,0x4A559E293A03,0x4DC6440C31E0EB3F,0x6F1D8AAF2E182C79,0x81437FA797B758FB,0xA9456EB48FB8CF37,0x67B6723F33797090,0xB2396AC924BE050C,0x12719F4D97819,0xBCE78B004E1D07E1,0xD2D80D01C9A1498D,0x5F807D8666AD6D1C,0x2AC1BCCF44AC8189,0x2944B20F5D4431FA,0x267DC20F4E492761,0x207E93D2C8EBC,0x1E90562B19A2C767,0x90916D779EC73093,0xBDC75EFA2077D0A,0xDBB814AC43FA58B1,0x25C788853A25949F,0x5696D094A05F13B6,0x23EE47B42381,0x2896837F061E5C95,0x18E00DFE453167BA,0x2187CAEB2813E16E,0x2CD5DD9E3855305F,0x8B53B210903EC228,0x40FCC08C8E186CDB,0x196AB10027441,0x87F7A48338880A4D,0xA6E0A086C9FC4CB2,0x9CDC64D79929978B,0xD398C716D9141C5A,0xB557A0B10A759E83,0x1A4F2A7E27C60654,0x1189951A3D03F,0x6FF1AF0B14CECD2,0x96B24B7A359B045F,0x3154A9746823F840,0xC650E752FBD93CA3,0x48CCEE4BC740509D,0x96D35EEE671378AF,0xBE5151481739,0x55E7DF5904F12E85,0x4819B3ABE47DA6F8,0xC8E940549D5B8CF5,0xDC83DDCCE72D9B52,0xB49A25AD17733066,0x5FBF9F9E3CC8A1F8,0x1A9D39A30116F,0x84D90559C8E97FC8,0x938FA91412A2D817,0xE2315CF1C1E69A96,0x69DFC18EEBB3511D,0xA409D353FD784C84,0x819AE43CB2377373,0x1DF9845A995E4,0x77CAF3A5ADA78ABA,0x4ED3731309C98E18,0x519CAAE8C44B7C48,0xA28E4F695DB08443,0xE64CF7E352D52720,0x68F6F5B8FEC87012,0xBBCC9954AEE1,0xA28E46C8E635C59B,0xF595CDB168EBD107,0xEFFFADE34390265D,0x6147D01D3C7D713,0x525D14CF19CDA61,0xB970104DEAFF2D3,0x1189373D47DC2,0xE5233DD25B708E9D,0x80A911745369550E,0xEC53FF0957DB8998,0x9813FE51521809FC,0xC94D771125922F97,0x18664277BA33253F,0x12C108CE12B5E,0x82426A0F7B4F73B,0xF48A31998E6AFBA2,0x15826D87451363BB,0xBCFADDEE30F5A468,0x1EB7B184AF185A90,0xD9863A0A8633C904,0x164CE6E66DAB1,0xB606B23D1A86DB65,0x8D5539CDBC6A4B05,0x5725B9F80B025881,0x374D5B24F660E747,0x75868CD69C75EE8B,0xE7D8DE35FA168C4E,0x1AC4AA828122B,0x48CE050DF7D161B5,0xD5E1F6ACBA619A11,0x244E813E118FA869,0x28003E5D99A3A441,0x3B68B7D319069EA1,0x983688F4C9C84166,0x168125925D30E,0x7B57AD1A506974CD,0xAFC1002B311AC2DF,0x894EDF5CDA376F59,0x369E940AF89751E6,0xC8708A28E151F660,0x5F334D17A492600F,0x15C130861CE70,0xFA66D8170B4AFECC,0x7827DAB5CF2234A2,0x11D4F694D1EF9DB9,0xF71F20D863B51CA1,0x2C7CA9F869E611BF,0x80EBED7536F5849A,0xF18BED30BA4F,0xFD3570F752C58B61,0x2FFD48C8F5697EC8,0xB6092D8F6E361607,0x85E6EB90A01000E3,0xC15D91CFBD1C38C1,0x18FA36D7C20D982A,0x2213827DB2A88,0xF172D86A90CF7483,0x19C3F9B4B519D6F4,0x256D2919381A3C51,0x190ACA76398932C0,0xABB7B0CFB4E028DE,0xF9CB3CFEE4A68C68,0x149AFAB09EEE6,0xB13947C39CF64B19,0x28674DEF15D919BA,0x5EDEAA1540F3E427,0xD6F262E411B3D89,0xBA7A2674093131FE,0x2BDE12EB794F83DB,0x118AD6EA8806,0x3CEF3CD3D46DD518,0x1EAE2B7C77E22AB4,0xD82F3F0CE40D71E,0x19FF46EE106BB60A,0x3097D41585C6E4AE,0x39E6556A40136136,0x1D1774685C42D,0xE086823136A031E9,0xD783A655B733EC4,0xBAEB57E3C170EF84,0x6B2CDC2AC6206BE0,0x6AB0E960A6B93666,0xD3502209D7178264,0x1B6D013167423,0x68D70D3C40F14E3C,0xF04F3808178834C4,0x40FDC0A8FBCE7B72,0xDF290F225A8B08EC,0x3BC16133555B89B7,0x8238E929398B3FF8,0x2303277DEEE13,0x50DDE40910E91AB1,0xB972596299C4DCE7,0x845055C681AF6CF8,0x8611DBA44EC8B872,0xDF7574438F28486F,0x8CAF377EE674F0E,0x1D67D2F5696CB,0xF0F1BFEFEBB366B8,0x7C848CED435462DD,0x3445A7AACB766136,0x3A3BC19FB15CD98B,0x4BC8BBD006CE136D,0x31A3F26468335B45,0x6309A17B6104,0x5C89902264D1CFF9,0xC98B69C47844418,0xC0635FD0E1A62316,0x95A8D4D2B332F943,0xC81D1D5859CD2276,0xD93A88F8F0690296,0x449124833F5E,0x3397CCC0C112AABD,0x8C6F06C424A734EE,0xA622809EC0EDF21E,0x552E40F79EC3F856,0x9FA096D2631D94B2,0xB272AF9DCB59F80F,0x17855976ACFA3,0x17518BD3E832C7BF,0xDDEDC4D02E12E80,0x9DCADB7A426937D1,0xF253E025696B4814,0x92EC17D7A3DD5B09,0x13ED83B9E9626ED7,0x108B9A7DB9D26,0xF230498CB6B37C52,0xF021883338542C28,0xD505B8FA2465D658,0x55A3B430E9F63460,0xCDD9EFCD02696B85,0xFFA9D550AB95EDC7,0x1A22336B3E5AB,0x89C1297040D3A474,0xFF71C4A765B4B4E7,0xCE64AA377C9CD1BF,0x639DBE419BE2C45F,0x4F98329F7FCA5FC8,0x665157168B2E4BC,0x210B727EBAA37,0x4E2EA413C071AC7,0xC9D6FC0B84D652CB,0xE7DF5DD3B5C472A5,0x17D11EB4637491BA,0xA95AA02F7E9B40EA,0xBCC421D0B684C06B,0x12A14E8F4040A,0x7195B72F9842BD37,0xBDC20346794656AE,0x298F37A257BB710E,0xDA77FC1ED5E8F5B1,0x473DFD2A02F30D51,0x4F0FD858BF4314F0,0x227BF05EDA151,0x4FE8034BF666BFB6,0x477CA71ECD43AA8D,0x445365BD406BB21,0xD8C4340E141B98D0,0xA100F9440D0DCE6F,0xBB69D18488A3E368,0xA7B58D26E2F3,0x610DC00581A673FF,0x7A2252C9C9C59EC1,0x68F4CAB28B0BDEA0,0xF09470EB9591CBFF,0xD613524F623913A1,0x392EAE8EE930C83A,0x2301ED4962761,0xDAC1D5074C7FAC11,0x1201094EA3567DBA,0x6035395B9EF45314,0x448EC3463D66D1BE,0xF87A2045402F682B,0xF1DA3881061A01A6,0x103C46E7BA7D4,0xC9798D6D92B88123,0xCDB3C4F571B7E949,0x5EA5404BEF683A2D,0x5A338AA2486A5669,0xBF743E15D71FE0EC,0x8E7EAD1288038D8B,0xBD2EEAC46788,0x39DF6D7E3A0E0B8C,0x5947E0F3299D2632,0xF65559E25161373C,0x501AC62BA145B962,0x9319B67104C4D29B,0xD14905102FBAD5BE,0x11AA0D9BA1338,0xAD59EF514C47481D,0xDD69DDA57766D86B,0xCCB58E187125009F,0xD7B83B91177A78BA,0x49CC41078F69A8D6,0x372296033AB2BEBA,0x15FFD6E4908A9,0x504C685562C71EEB,0xB7C8DDDE787D9355,0x9648516EE93400A1,0xEEE381D87A506BFB,0x154E8C3AB463D119,0x4523664CA4434C96,0x11A1B89565C87,0xFFB1C7745471413D,0x82814E0EA40A1DB2,0x6894EA66A68AA1AF,0x8208154BB53BC1FA,0x8F04BB227BBB6A2D,0x6A4F34802D71546B,0xCB6543ACEC72,0x3257407C69EAE7B8,0xCC142AF9C4F9ED2E,0x795AF9CF87455853,0x50B2ECCE460CC358,0xB4A3A4DD5B7116B,0x2A90E327E0A13F6D,0x105559901176,0xB5527838A4F3EAE0,0xB4D4C0FDDD233F15,0x1B5068BCDAB1D01B,0x11A3D020505B0A9E,0xD0FD6BC0F3DA73F7,0x893842BD0D53CC0A,0x17372A167015D,0x32956E857798056C,0xEA7FBDEAD42FC220,0xA18484F03AC52AD2,0x5A9453A6B267485B,0x772F0B236E715961,0x14F63E436B673F65,0xF5886CB43DD8,0xF398D8A08D3DF24C,0xA6FE4679BA91D7A3,0x21A5F32195AD76BF,0x872F0FDC511AFE4A,0x60589F4AC318B6C3,0x9DDDE7382C5A0BB8,0x1C5C82728598B,0xF97D97075C792DE1,0x64EC771D78859FA0,0x9F5DD6CADED67A66,0x68A9B844B86E01E0,0x2F5B863278E6B2C2,0x28458C2E31D8DE4,0x1958DAAC84549,0x993B0EBEFB56A8C4,0xC08562C18402597C,0x591E067D5C6FB9DD,0xA28E4AE7A5E491F8,0x41F2D1732A9F9D8E,0x89567F6E064E8F56,0x160EDDB96DB62,0x95F031DEAB939ED3,0x68BDBB684EA98320,0x827CB6B6BF2C9C58,0xB7FC54DEBC89C71D,0xB51E046855C44A5B,0x938ECE19F1A2DAC6,0xBC7E6DFAF66A,0x1CE1942FC86F43B3,0x412A183762AA4F78,0x6A955CA6AAD6A2BF,0x9BB60F27242351E,0x1EC587FC4172573D,0xDABA4191BB36FB86,0xEEE0F3561D87,0xD36089A15329244E,0x78E65A6EB6A05CBA,0x452AE8C5BA97C3E6,0x2283E9011B9B2970,0xF7224332DC0F6EA9,0x21546220F3BE9B7,0x1ECAA9A5EDBFA,0xF681BE1811B16745,0x2A43C9620637EC81,0x6DBD82344400CDD0,0x2F3CF0C5467947F5,0x2B1D36C44388929,0x674B88CA2729EB78,0x4700327DA938,0x3AD966E3A12BE29F,0x989FBF1CD71B5E11,0x8D53C2978E8F58E2,0xF8D08B8A2CFE7A39,0xF5EE8016059AA8FD,0xCF0884447BA4A36A,0xCDE1636EAD,0x6DAF1543650FE939,0x56E3F5B5A2FB1756,0x7EDE9FEAE3701221,0x11E76CD322EA40B5,0xA1654212EA5451B3,0x626666D3D8DE14C1,0x96CAAF3C17D8,0x615E38D241FD05FC,0x3C0062D34EB7DE11,0x7D2F5F0FB354BBAE,0x85E2C91EC41BCD3F,0x9952EC9522F453E0,0x3A72483C32D12963,0x973ECBC9FA6F,0x802532DCA3936A71,0xD211F94280C82BF1,0x4A6054A68499D109,0x4C262CC601B2A25F,0xD0B64B666BC4074C,0xFB3A08A247BE4181,0xCE88AD9EE8E3,0x181614747D9CFFCB,0xDCFAE009C245C63D,0xEFC20498067923ED,0xDB6272138A8635AE,0x91D58A03136BB088,0x6662D0317389DBA6,0x1F397816FA65C,0xAC5DB3F8A9DDEC46,0x9244396D61E5957A,0xE0EE324364425AF8,0xBE71B718B4692F0B,0xDB0BCDCB37795CE9,0x1FF9B333707DA3FC,0xA507737ED80,0xBC2F4DEF28503752,0xDBCB89B2D7CF897,0x66451D05753258FA,0x7D285958C7A09C70,0x5617C75F685D49D3,0xC812CF22C0564F82,0x1C735413CB2EB,0x7213F4CF3366024,0x56F5CA85C3760117,0xADCF03DE500D8847,0xD7636A9936771DDB,0x14FB141137680E4D,0xA631E6877B7995B9,0x1584B36DAD310,0x6E360A690D4984F4,0xB7C8A1ABC388A32E,0xD14CF9D7D6B4F746,0x8E8EB352352522C,0x10A30C45AD941E8B,0xE682E674E598F5B4,0x3F75BF43525D,0xEC2DE2829A236D76,0xA89B5061BC31D514,0x4D30FEB3F4B046D8,0xEFC79370701D6D99,0xF6B38D4F70A430EF,0x88A3C8920DD3DD17,0x1AACB7AD6D963,0x108FC2682F8B1309,0xF0EBFA41C1CF1296,0xD47CB36AD7B53903,0xD5DE9B9AED7EF00B,0xB9BA0155897C6D43,0x1A56AE3261D669AC,0x11DB629347049,0xA8E51920064BD18B,0x70BD02EEB72DA9C2,0xE8D0C69C7185DB72,0x94E9BC90A510CA4C,0x8572FF106CE4BD0D,0xFCA6CCF817FD6FAF,0x121A45BBAEB9A,0xA8B2A7B857406FB8,0x798537AAFEC12B82,0xD287608502456F71,0x900D19280B58AD1D,0x71A96014DED1721D,0x825A27B891E2257D,0x13D5850A6D764,0x51C31D1011EC45E7,0x80B0C5BC24A496EB,0x4FC2F5CB9209CC4F,0x9F652E1CEAC10E1B,0xFA2BEAF4BD1B3FA9,0xE59523A40E4B1336,0x1103FFF370977,0x12C3EBB49B0C017B,0xE3D2D2BC9C341724,0xA192381B97F6E38E,0x6B3069CBAC53BAE4,0x61617F4893060627,0x60169606ADA9B57D,0x751B54C7934C,0x49FD8CDB826492BB,0xE7FEE9E8A1A31984,0x5FA80EA5ADCA30F0,0x6BB0607E3B5445D8,0x59460FBA1C339609,0xDBB9FDAA8D1886F2,0x2253D19B47F6F,0x9C2D01F96C71C0EE,0x986698E464F17F4B,0xA2377AE4588DFA3F,0xCA1A99EAA06D4452,0xC094406AF2F7D088,0xF852B42D12C2E0,0x455ABFC4A688,0x143CF32B1679030,0x2CB23353ABF72C7D,0x144B52446C03208B,0xFCDD9E26F3B141AA,0x4240EE04E607829B,0xC44816C6FBFA480B,0xEE5BF73ED2E0,0xD15BF1F8A80F45C3,0xF12AE28C50376FDB,0x7E894671BE19BAB2,0xF25C688850C77B0C,0xF30CD998FC02FBC5,0xB87A664E869402B2,0x1101302E33111,0x6DFEE4BF141AE97B,0x9967B49E300983D9,0xB37BEEA11F8423,0x8C0E8E3F926575AF,0x6BCAC3667DD778CA,0xECF5614D26457C14,0x146ADCCF684FF,0xEC0FBB225E07EC04,0x851A1629E73C0362,0xA2C66664392400D5,0x3B91A5DB0F97A0A0,0x7FBA66950A40A8F7,0xA577D7B81D063754,0x3558282BCE71,0x8CC232D0029FAE41,0x1FF0EE8967F2B586,0x89A2096A166B9876,0xE51E7BC5430ADD25,0x14E96D994B63501B,0x5A13BAB060EB7E1D,0x1526C87977071,0x18CDCDB0CA56A7F3,0x19B29E0E63247456,0x1072F30249E8642D,0xB27BDBDD8458B19F,0xE2BB359016E681CD,0xF7349DC8464DA410,0x122530AF1B8C2,0x2F60EA7D67C95718,0xD080934779E9A09A,0xBD338F2A923A08AB,0xD399D9478208F96D,0x1EDFA1097646140E,0xE694939290EFC5C9,0x17637E708D9E,0xEF2985109439F01,0x8756891DDD1A9E47,0xDDAC94A6A48480D6,0x746D16635DE2DAC2,0x554E6FDF021EDAD5,0x12146F6669FB4B19,0x11D04B047B2EF,0x3EC9A3A04B5831E8,0x5B8C8D3FA2F93BE0,0x87F3854794B5CC29,0x89F3D8C4EB27C93,0x7DF551EBA1E4B3EE,0x8315000E064B8DC0,0x1BE4DD7697E11,0x9E23AE7CE75625FB,0xC447976868368186,0x2AC959C1A148FE92,0x5715F79D1A34B88D,0xEFCA811FBEB30134,0x40EC78621AA16B9F,0x2172A0B3BF1DE,0xD4DB80B99E71F195,0x1FD6C73CEC487A44,0xCB8EC53D4D98D038,0x8F9599E7AF896C9D,0x9DE70E37D7D3404F,0x25918F1C61F2E8A3,0x2EE4FA576F89,0x6C3404ACA8916BED,0xB3C0CC23AD3DD17,0xB147775302AD42D7,0xC830CAD9EE1DE07,0xF74B80B2AD672AB7,0xF21C66A894C741D2,0x169F44056AACE,0x8CBE4F8BB6B6B222,0x14D73147DD48A483,0xA5EE9A0C38C2D38A,0x428E4666C3E4962A,0x36E538FFBE9BCFA5,0x35C9E7FDF4767AF8,0x1751BC481BD5F,0x984BB981CFA7F6CF,0xCE1EF877871BBFE,0x2470A7704EFD5C56,0xDBDEEC9992820075,0xAA6794C980132A27,0x4316456A0E35B1E6,0x1D73A70681DDA,0x9B3F61FEE0ABFB03,0xBF70F2BC3C323D4C,0xC1528598CE32C6BD,0x7DEF265161A307DA,0xF2886238D52CEC15,0xE1E9BB4BDD68A269,0x92588332E5D,0x5CC77FA16AEF2681,0xB67178E6C7513C27,0x225B2E4C08339FDB,0xC29432766F501C05,0x9BF2D9DEC3BF2D80,0x891CCB6F95243C0F,0x1A0D680E7376B,0x782AA001ED5C273A,0x94534ADDEE1AD0BC,0xA932622F63DFD642,0xDD36A4287C543BCD,0x2E58D3A64FE37AB2,0xD4ADE5D69AE2A1A3,0x14D849B2B0E0F,0x7941F92334062F5E,0xEE919F3BD8899FC0,0x7359C3F6E7CD2AF,0x6F2DC6966C225853,0xB032677D6956D66D,0x253BF07A196CCC69,0x1BD627B2E2A72,0xAE1A278CFD8D503,0x646C05085881F413,0x95AEEC497420FFB7,0x44432EAACF0C5BC,0x6D0C27F428482A7B,0x36BF0F73795AA970,0x19B5D7D1CB06A,0x8CF1AAAA1528F977,0x30E061E79074CB0C,0xA707BCACAF089926,0xC4433A96484B2488,0x3EFC2B893CC11E21,0x3E9C16A82D794CEA,0x1AA78C0619F6,0x386E11EF3C9ADE33,0x3F345CA904720525,0xFBF85C12161FC42,0x713870D555AD571B,0x19BC840403F1788,0xFB2C55A15F0606D,0xA62E0ACE7494,0x9A2A6064FEEBA7B6,0x817A5BD526F00A2C,0x1A0D6D7E501029D8,0xF9EE0E9F6248D694,0x37F1AD16EEF97EF3,0x765B35CBF5E84A24,0x1F2469B0ED462,0x914DDC06CD058D0A,0xB1CB521AA015341F,0x764554932AD560BF,0xDDADA204BB4EB61,0xDC5D65101FD1723D,0xED233BF74AFC0753,0x11C386BC088B1,0xD24F36F7EE6B2C64,0x390FC1516A72ED6B,0x38B9D3E1DD99197D,0xF686100F229E99AE,0x62183576EE814323,0xBCE11102ED020DCD,0x4E15AB76609B,0x6081FF811443FFA9,0x350E77FDCAB2D6FD,0x59AB0E00DCE80940,0xCA3DDE7ED63EE6E9,0xBC6DBA8E805469F4,0x9E00842703BDF5CE,0x12E18815CBAB0,0xDD8EC29C8CA1B3E,0x294FDFB431DF7FF7,0x5161BE8DC4F9B74E,0x2D1DA9FADA3A0D19,0x6F8E5E8E319CDC8D,0x141B4E90AB5C35D7,0x209DCD53ADEC5,0x5AD8C2FF2000614D,0x6A0009899485922,0xF60BE8649878A9D3,0x824413699961ECF3,0xC64207D8E56AE56B,0x7CC487A119E3657A,0x9C6B456EB547,0xD79164F0AE4832C2,0x18AB4B04FF516A95,0x78E7EF83FE6F7F14,0xB49D5BDC652DB352,0xC0DCE40B3129F558,0x8F6D4E0D78C014E,0x8AEA7CDB1E42,0x915EC782EF6E7980,0xC3891DDF0857BEC4,0xC5E30E839374A7BC,0x1780BA1F410ECF91,0x7122972E82E2A865,0x86995FDC92D0B278,0x71CC3D2A1CC1,0x164434AB40812A11,0x394BEDC32B5A1595,0xA1CD4DFED75A0BA8,0x5085BE2FEC402595,0x823CCD0231DCEE8A,0xB6D712B7F5D773D9,0x40EB00321CF5,0x48BAEDD5FD2B1800,0xF0169DBE8161293E,0xB6552BB0BDD53BDB,0xE65FE0A7DA66E23D,0x1E1131F481DD48A9,0x2D1D20B8601744B0,0x5C6C4116B51D,0xAF3F77272DBFC895,0x5566C60CE4E13FF2,0x6A2F921B830DD150,0x9713F5E579DB2A7B,0xEB5DB9E5173B29D5,0xE149A82C1F4354DD,0x229123D77F20D,0xCF80A2D006EFF4D3,0xD813D9B71034683A,0x2DF47B0858B74D19,0x699B2E6987F5B937,0xB59A67979BD19A8F,0x9FF50417A0C7C51C,0x1452867161821,0x671FA199D11B27AA,0xB394846CC8902719,0x1214B1962D43EDFD,0x9FB04AAF55E979D4,0xA0B3646A2A6962E1,0xE14AB1763C13124A,0x16F7E75B2F925,0x648E314A441606C,0x70ABA22B1EF8AD4B,0x43653F156C44AB8A,0xDE655728B459BE6A,0x9356F6CCC7ED6214,0xE09D74AF2B543587,0x13D3D7A473A08,0x6D3145B3A15DC693,0x87EE74A0119DC5C3,0x6EB3EAD2D399D3C1,0x8241C7A29D1BD979,0xFD6CDDE8773013A3,0xD10EF0D36DB31DFF,0x1421DE85F2E67,0xB67EF8C12A0F95BD,0x7D3E0BDECBDF5615,0x2708B77D19CF90E,0x23E07BCCACF17B47,0x8D47EB82DC6A1F7C,0x10A673E279430B65,0x14CF38522E666,0x1A096E1EEB270528,0x933038843B893F38,0x88B613C19DBB7467,0xFEC6DA1CA0713103,0x240180E56E9B7710,0xBF2E5E38057F4D63,0xAB82D1ADAF55,0x502DF4088EEDECB0,0x1410FB3E67C58CB8,0x198656792C37A4E8,0x4DE0AFE6029F1E72,0xDFE0B7C02849BDF2,0x61D8A089A990E61C,0x15CBF26EC7D9B,0xC604F3808AD861A7,0xFC5221165FBEBDBE,0x7096DFF6A69A2FCC,0xB1F040801D15E3F4,0xBAB7C15F02815853,0xBAC3173868CF2F86,0x108C09198BD98,0xD24EFCEFACBB4507,0xE859FEA5FF78FD89,0x920BB19909373B75,0x7FFF9DF9C2465F5,0xC307CDC3BC25FC30,0xA63D94EF63913399,0xED7041483B84,0x2A113CD82EF74316,0x25F3CD1F99B6A469,0x91F37212B85EC349,0x41469F7DA76CA706,0x66CC52A81908A5D,0x4E7666AFC4487BA0,0xA6A111E280D7,0x8E4F12C60387301,0x2C0327333350C305,0x654B3E84E69C975E,0x1B252CD1230EE46D,0x93EC13BB0E071435,0xD6312AA2237AA214,0xA941117BF44C,0xD06B9C5F1D59DA2E,0x363E8F00AA1AC71A,0xA18023F2861D16B4,0x948F5EBD6D24EEEF,0x666F55D48D856AE4,0xC627819F38C6ED2C,0x70C8E4209D9D,0xD3F26F1F9527B83D,0x850B832791ACB96A,0xFFBDE8F888140FC,0x3CE0491FA5944FF3,0xFC36A1819BF21EEB,0xC0B781BE5BD3AC3,0xEFA0DB34FD95,0xFF6345E89328BCE0,0x8D79BB07F0AD184E,0x35139A24BD461173,0x1734448FC20A9C8B,0xBF667217CA7A09A1,0xE8EFB503E7196C5B,0x22E314F4872E1,0xBAE666426C00C42F,0x26CF0420FD295407,0xC0BA27DB997BBB9F,0x8BD7F03A41C0A6EF,0x591FC2A97A36A65E,0x435F8D4B5A43EAD5,0x158C556A6127D,0x992296776C5492E3,0xA56A349270346E93,0xB05E3B37CB4CCF2,0xA157AF55BAE8A3BF,0x9C48223428C3A094,0xDCCFC7F3EA67352E,0x158EDDF305929,0xD7C86C5604239010,0xEF2033BB69E3D360,0xF80A504D47EED8EE,0x2F2982FFC989BDBB,0x21F9500849894B38,0x3348D62BCB154C8,0xFB91EA9C41DF,0xBE43F1042591D7FC,0x68B2F489C88B0656,0xA53B0B23780395F0,0xC1B3FEFFE0350652,0xF947D5D589987687,0xA5B0EE0ACEF62D10,0x96A8BE06F151,0x6EB8AF1144E66885,0xA0664F384B2C74A1,0xBE5C5B9C67C50569,0x7D763D5E67E0301,0x97D9BB30D21837FE,0x7B8E0AC0EBB6AA84,0x1D3E30FEC320C,0x386503F306755803,0xF57F11BE69F5A7F,0x5AD5D1AF653FC2C1,0xEF66E6557A3CB2B2,0x97E4B94FF03DA531,0x1386737CE662693B,0xF8971469D755,0xC8064799BF1E72B9,0x6670AA87661EA130,0xCDEC0DFD48524E0,0x85DECB5CDEBEFCE,0x7D661BF603E36C9,0x1A8C271E48D9592D,0xFA7B15FACA96,0x2754E6C7634B47DD,0xAD4DE2593DB70BDC,0xEFE555CECABFF708,0xAFE9C22F4A5E5726,0xB64977A84CFFCEC9,0xA875076F76816D33,0x46B42F82B90A,0x1D1EF4891A10A797,0x9D0A5A2174C87CD,0x1E59582514DE9704,0x3940A491B435AF17,0xDE8396F3E6C72405,0x250323D82A61F880,0x16CCC90282BBE,0x220E11B0467833C8,0x4703B9A764E4B8BF,0x7C98652DEEC49AF5,0x72D69564C062BDF3,0xDAC71C23822469C4,0x1F35A9B6730B49E1,0x9618C1E62999,0xF04D0E8F962ECD7,0xB56E44FE43EAA2C2,0x170AFB64F110B4A8,0x7A8306ABE1822C25,0x7CB27D274B8FF21,0xE3FE38F8CE6A7B52,0x172EA02AA50B4,0x643C9B06E08CC66,0x2B87BDAF4E9388DC,0xD71922CF1672D873,0xE28B6BF1D657B51E,0x4EE86292969E26C8,0x774FA57CEFDBEBD3,0x22356CA9281DC,0x79D5C53DAB84F2F4,0xA9E38E3B640CACA5,0x8EB5FA0E9D457179,0xEA83DFFFCEE2D34F,0xB7756A9A698C6A56,0x1C2E63E5F5F94EBE,0x1C70A050C84E9,0x861DFF7E8086EA85,0xD0D3D9DBE0B60B87,0x6FC7EBD7B7EA4D0F,0xACD957CA1EFC597D,0x674AB3DD0A8A4A68,0x64FECB128E6267A8,0x21980E790245F,0xD03F684814D93079,0x76170D91C18AE862,0x715CE4F2661E8422,0x3B151C3DF3DEC28C,0x2138500C7294A963,0x4C03650748673F92,0x11D21D53E443,0x215DF9EE691F61F4,0xB09E653C8FA78755,0xEF4393609883DC89,0x29D78911DA4A1113,0xBD63E4727C7E28F9,0xA3D4F7564B901073,0x10F1BB1A30409,0xCF4DDC5549300A,0xE9413CA6F4CD4100,0xF7A542F8CF3404BA,0x439D8AC2A6FE8751,0x20ED88D8D653E5A6,0x3A1D37F43C306D15,0x19CB3FB40E64E,0x30EF0546FB21C004,0x3B818951DE4FB322,0xC765E13150B1C2D5,0xD678825CF33E22EF,0x32982733B22CB893,0x103D3E24F664B502,0x1C359727DB042,0xF74EBD503067A42C,0x91FD5E66FD9F2807,0x88C546CE93DEB7CC,0x6D142D5052D8131D,0x28B693F95AAF253C,0x9C71D15A25A76204,0x93296544C54F,0xA402E80F39BCAAA3,0xA375A97EC1ED7F27,0x7F41AA440EB9EBC2,0x442CB4563CBC85DD,0xAA7C95FD16A78B27,0x3A9C4258D739E100,0x314022980AFC,0xEF6420DB718E8CCA,0x22FA53308FA56AB,0x942E1FABB5C82FBC,0x1F024937482AC7B8,0xA13BA11324609DB1,0xEFDB248CA66B462A,0x72F06D79F8E3,0x9A4DB208DB96B3C6,0xF193B9FD64A6E6E0,0x681FA9DC20DB5D3E,0x104B991877536BAA,0x6F1EFDFB41FEDF60,0xDF88781860A4D51B,0x22A463C0BEBCB,0x608DA1520FD3520C,0xE9B12ED60CC101FE,0xDAF96698575A7191,0xC93D41DA54C3199B,0x5074D286FE59F326,0x3685F42CC3076EBC,0x113B64618EF3,0xB839FD952957E967,0xD1C800823F554F89,0x59556DA71A539131,0x8D7923878A5CC54,0x42E592A7AF0E25EE,0xD9093304DDD65945,0x4E497043E3C3,0x5DB62A851952AC1B,0x9B3F62695628E9AE,0x6E6B76578038E04C,0xE21CA7DDBDD5830C,0xA3D21C6FECCFBFD4,0x48CF797C096E0740,0x80A32146236A,0x7CE03DBFEE575E24,0xB5502FA503B9635C,0xF6D7020410A9C247,0xAE731401F0227220,0x626FB4B1155D4A2C,0xD3B4121F356F92B6,0x118FCAED5598B,0xEAFA2FB66B470175,0x30E87AE005121328,0x2F88A31577FF52B1,0xCFEFEF54BE7FA45D,0x61AD5E1CB48AB19D,0x2B593EBE6F21953B,0x7C62F48E67E5,0x1B7B3581BAEF1BFA,0x50CE15FFBBF8B6F4,0xF7D5EF289E4852CC,0x57AE0BB595B89A5D,0x13AAD32E2B7E9620,0x11DBF55DC1950696,0xA2891670CEFE,0x9D3FD5BB56310B5B,0x4FCDBB4A5468EA54,0x1269F3873398856A,0xD0B6E5A3B049CE2B,0x25EB4C992C5CB3A2,0x78042A54DDF046E8,0x6FDF68CD1D74,0x8044A6344D792F29,0xE748807D4F09BB63,0x45CF59B6352E0DE,0xFF05E523AB8CCCEC,0xD3A6B9CA4FBFCA12,0x29E9FB38FB4D68C6,0xBA8785175432,0x85B3E3F08124B64E,0xF104133C18139ACC,0x28902AADDF6361F3,0xE8DD45B08B900D27,0x9FFDBAB1C0E62AA4,0x264C51B5BE14E180,0x69BDB668EDA8,0x719DE1EDC7D0A421,0x6BC959BD69519685,0xB107B97645155E0A,0xACC868A7936881,0x50F46B9C022275E,0x2C3D1284963EF67A,0x21ACE55D2850A,0x898BE48131F65BB3,0xE15839DF988EEA12,0x509C88CD5E733E4E,0x1609EBC1EB79CA03,0xC96F28E85057AFBA,0x54921BDAFFDC67FD,0x1B4A11D1BDF08,0xD4A697FBBA3F840,0x480C56FAE876C1CC,0x6634C7A90BE98D90,0x67C8F1A197CFB02,0xC557CDCA3E1D08BC,0x5E8B15E7518D36C4,0x5C5260A46CD6,0x5178990EC7416753,0xD3E7A43B8689D25E,0xA6D639D7A2E9B2F6,0x9FABC1EC98A4C1C0,0x348B109318082F2B,0x180A660A6213AFB3,0x19058A1E7E29,0x46B123EBEC2A9864,0x8C82F5C34E70065A,0x5F47608551B7D269,0x8CD6DD23FEF5C5E,0xAE00D387FCCFC68B,0xDD7B97B185757FDD,0x218A212A11D4B,0x876A9B62899AE70A,0xA597032DBCEE2638,0xF80BDFDD8D6D24D9,0x6E8C7C9277CA1A72,0x9F6B8CF39318805A,0xDFBABD16529CB1A2,0x12CECB1521941,0xE5E32920DCF5C7E2,0x977B6818D9225CF8,0x4C2C1A90AA10045,0x3F6BB1CDAE29B31,0x616EA6FF7C682817,0xF6520D1AFB4CBBB8,0x543E4A838328,0x366A448CF82E8BAD,0xB48D561EEAE6DFDC,0xC6084E09627A76FB,0x485270F129E3ACB2,0x1114B2C643E7D25D,0xBAA7E8A0FC60DC7,0xEF3B4AD3A921,0x1A42F823AED24BB2,0xB1E32354683F7DB4,0x6929E5646DBA982C,0x2A850BA0A4EE8B42,0x2D0D895254DC4910,0x8D0ABF026E117420,0x9B37B29A1358,0xECE5D3FDDA92015A,0xD9C92BFA89C61D08,0x2E7136099ED22997,0xE2850DE7E04222C6,0xC99575BC27EB2079,0x688653C67AB4DB6B,0x21C76FF9D2E20,0xD18702F2B5286947,0xE42F676580E34921,0x7CB042D32758D79E,0xE6E1BDFF0C816C94,0x84D92C1005EEBD21,0xDA1F19D93E39DA25,0x1D0E3BB26034B,0x2D820EDDC0651F7B,0x62013B028741C437,0x1B35E956F9177AEC,0x935EFADB19A405C0,0x12DFDAF2C5348559,0x5279F1FE21C8FBE,0x363BB39BE8AF,0x6CD3519BAFED8441,0xB38340789DEB45BD,0xAED44626481A1A53,0x46B118FCABF566E1,0x29AFBF3B872A027B,0x8E2B400D687BF795,0x146EA999A9953,0x7F85C0742D06D762,0x16ADE4369A703CF8,0x5C0616EBE9B1EB1,0x8345B4C4C381909F,0x74588C4BB01ECA31,0x590037DED94BE0AA,0x3AB7DE3CAAE2,0xBB08F29FA22D81CC,0x1FA616B97E05C828,0x9131B905B629A13A,0x6CE6FDFE8586509,0x59C56F398A650985,0x837987E14D7FE08D,0x5618821A3DE8,0xCE16FD8336A5F12B,0x2D0BD200C05DCF17,0x759D564FCC3D1D5F,0xB050DA5152AC93D2,0x4A05AA3503015256,0xA9C2FC326D770AB7,0x1F88396C8570F,0xFDBF0DA4DDB14AEF,0xE31C97CAAC2B1AE9,0xC1EE53A7AD179B6A,0xEAFBF7A9FAFFAF8B,0xFA0A8C02A4F7B0B3,0xB763F72ECB7E482B,0x1E53148120D31,0xE342611E59B31042,0x34A38160F4B0FE89,0x236AB126F9D5418D,0x569662C622778FF3,0x6E1BFF617F506E58,0x6457E0518CBCAC2A,0xE90DCECD11B2,0x6DE0A64DBD620FFA,0xEA04616A6DA4B554,0x35BEC09E9F32EACD,0x3C364BF667956041,0x7F550D0B76DD9D96,0x9BF6F57974B48DB7,0x146F2E393C1B1,0x57FDAE3875A8750D,0x6D18736EA187ED61,0x6C8AFDCA13C92443,0xF118921FBBFEE1ED,0x1D5626A7695F376B,0x599DE0B91A589A2A,0x1FBAD61F71587,0x72A9B33F06B83EE7,0x3D791D5E45E1824C,0x2D533C8F6861664D,0xFBF508F845035F20,0x7E4C27072918DBA0,0xEBE602F6C29E73E1,0x22FCC4BDD7B42,0x5D68418C13B9D614,0xFFCF3AE8611FB3D3,0xAC676226B445E60C,0xF5F378EA091C9F19,0x33BAC1DF195C59DA,0xAFD9915202890D1B,0x1D0784F8F4F42,0x4DDD8FBDCE7A2644,0x5243541912B284AA,0xD09BDA56895D92AF,0xC116CAA4ABA08A3,0xBE99D0D49254454A,0x85C5B53BDB411939,0x16F29BB48E401,0xEF1DE09B50A4F245,0xD0B95B3635D0A2D8,0xD3608B43A0657567,0x91EEFF77F7C3A82A,0x34F13298B6272CAD,0xF4E8173A373712A4,0x5434636112E,0x55E11A8D3D251857,0xCB68B3160FD39797,0x9A93FC78DB3107A,0x868FB7FFBE8D63AA,0x611AA5A0B2B597,0x4E9C9A3FE51803D0,0x1CA96D81BFF51,0x78CDE4F9D6051327,0xCF10B1620B661D24,0x98D4FB1F5781A67C,0x909C5A7FB50BDDCF,0xFF01BF5791090F1D,0x8DFF2D3118FBCB63,0x7F70C09D2EFD,0xCF2EDABA8295FD20,0xA6F8DF8CC583746,0xFDA751D82BF865CA,0x2FBE1C53DE2C0424,0x4D55ED1A61339F79,0xB34234F2BFE697C7,0x2E5856979B09,0xFBCFF766FFBE437E,0x754CE7B163CB222F,0xB849A5BC4A0DC7A0,0x9F60F6EC4FC9C4A2,0x7CF33BFE698E35EA,0x7615FAC56115E26C,0x1616D76E20FE1,0xDE61615D11EF01B,0x46BE25F4077721C9,0x5723C28E5F160AE1,0xF657500AAFD90A98,0x598C4E7005A8AB92,0xDB08BF38D602C320,0xB9C975129CE5,0x4F198A137324A1B7,0x6FF849C5830D410C,0x1FB006DAA4951D77,0xE4460B76BF982F39,0xAD1F12B1E3BE38F9,0xAB5D8D3B524B72CA,0x1DD0AB1E6EC9,0x7FBF1B94D9CF1640,0xC26D6D0911A74398,0x65A9B26124832F31,0x4AE23D0C025E4BA9,0x68D9789E237F222D,0xB7ECE01E56835E60,0x1EF2B8F05273B,0x8816C79C8ADB0ED4,0x9339FF345CA7C8A8,0x535DC632C3F597BB,0x3EF3D718CACCF0AE,0x89B774C401AB4891,0xB5934191E05CB2BC,0xEC1FA11D8B68,0x8AB1A1F70385C52D,0xB94556CF9BF3D0AC,0xB4AA88E6819E5E0A,0x6D319409884BFA33,0xE4189021F087C029,0x15A89A91B12344D8,0x1B2710E8F329D,0x646A04449FFBBBF7,0xF4D923532B1AAD91,0x2DCFAFCD6E36A52D,0x45DB3B53F6F5985B,0xFC4EBD3EDB941B7C,0x4863EA26EE5D538C,0x14EDC74BAFFD4,0x2C3992457820CCD,0x316E66A30F7D2EF2,0xFC8CC1FB57BA847E,0xF1CF9478A8778DD2,0x161D24DC1B9D2378,0xC59E79FF5E4B2302,0x18EDF6CE1FEE2,0x8DC9EE7EB81E5AA4,0x92B7F26FF3C5EE36,0xA01A474EF621E132,0xD1EA064372AEE4EE,0xA7B95DE88EC51F00,0x6A495DB60D987EC6,0x1DA53C8DBC4D4,0x7F811980684FB3CF,0x601B85D72F0AF064,0xAB4F8993D471163,0xF1919D90D9B43D34,0x59BA8DA39CAE9CCB,0x5BF9FEADED16F1A4,0x196A4A17D748A,0x5298F797A27F5150,0x92CA69210336675D,0x5DB8C32FAB48C020,0x1480138C4FC22048,0xE6325C0E69BE2148,0xEE1FDA3493C7CF34,0x197B8AD2FB15,0x287360F85DE134D0,0xF7BF6A242A2034F4,0x9398781FB7AA3F70,0x5B9D567F433BA060,0xC330328C0593C768,0xCE880153ADCDB51E,0x4BBE3955FFAF,0x2AD522BB777BE40,0x80263C347B9F5777,0xFE1F088263BC556E,0xA28B531AEAE0F7D4,0xF1CE3D837313B1B6,0x7BEF1B9C771173DF,0x6BA43B0B093E,0x38220332E878527B,0x9A612E4711BEC953,0x67C533B0CD3EC95B,0x6E4B9E9FF4E905D0,0x887DA87DD651C10A,0xA62CA7BE213157F0,0x20D5EB054E1D8,0xB7F9BB65D8BAF28C,0x4C7BFA14B8D3156E,0x34FA213E1E1075F2,0x1F94CDD1AB61A46B,0x9D63052543E45312,0x2C9AC5A3C4E98741,0x1079E5A2858B0,0x286787778402A292,0x145C63904754316C,0x1FACFCC71C294471,0x14E7B408D394AF92,0x1843A9224D6B7912,0xE88598FF4C7A96BF,0x22F1F3142A93F,0xD8CF10B3BCB02B84,0x5F03E923BB2245A6,0x8FB431F769123CDC,0xCD3B2F7858717658,0x11B42B005E4FF596,0x11F2BAB32FF606BC,0x5A2C828DBD16,0xB1DDB6DEBCDEC9,0x61128324155D3CEF,0xB0C0EBD93BF11C5E,0x730EE92BBC1A4EF4,0x86A7878F26DDA852,0xA24C2F5E269110A2,0x1E5EBCF959B28,0x1153FFFCCB348CD5,0xCEBEFAEA41DF75C6,0xCA8748480B814868,0x79E2C99FBE9E5880,0x2759ADF9E306DFE7,0x5446CBC43774951B,0x1C2895200D4DC,0xDACECCAAA3C2A7CD,0xE56712DACAB019F6,0x7955D43F30F61063,0x9AC9F05FA4ED2841,0x65B5339DB6048BE2,0x260CC3B4AF77C5A,0x2F347551F8DE,0xDDE27C39E5626FE9,0xDFC551A81025E113,0x94D22CFB476F509A,0xC8839688DCE81025,0x5815193C09AD94B6,0x9F98099882067310,0x1511315FEB6DA,0x4A05DC3328CC6BE,0xB92582C9DB6E4C07,0x97030A5E1650353B,0xD450739D812BADA2,0x69747662902CEACB,0x142A9446FB95E706,0x1081DB29E64F2,0x28731C1602BB8CB5,0x98060A1758A65AD2,0x2B053FAFF8DC9C65,0xF4B92909AAB802F0,0x17AC3B78A878368A,0x847C549FD7402549,0x12F73F2112FC6,0xA531425205889461,0x9B72AE4C7E6EF72E,0x751D6A2951A6D4EF,0x1B05128457B83BB4,0x6526ABD5941C1F6D,0x3BD6D42ECFBF4D40,0xA09FC53B9DF8,0x8D58D309EEDC486F,0x112C5BD86E9E0FB1,0x858BBAA717F59232,0x658B8AF8E7B62FBF,0x88FB5C91F4AE85EA,0x22A6C230F9C90815,0x11C19CAB72E67,0x1D90480AC175AFAF,0x7FFC8D58B9BC7EB3,0x827BD1874A612D07,0x9908D51007A0EBA5,0x2A5D23B758579A71,0xBDFEFC70358620A0,0x1738C2B2FF3D1,0x4590F8BDD6C13983,0xA2D75E7F48950BF7,0x8186A357BF717C27,0x53C8A21AA095680E,0x298C8D12A626CA1E,0xF94CDF412036FC38,0x5E0EB58BB81F,0x49785E3D56341207,0x6A39F43FEAE891D,0xEEE6890001BF13D0,0xFEAFD0A3ACC0DBDA,0x2E0C3A506C4DB6B1,0xFC1FF88539EA2773,0x116C45D2CC742,0x81997CBD6C43A5CE,0x6B539C95A3EE9769,0xBC4EAAF59F6D6F34,0x7BA1C6E5C28A0D5B,0xB328EE3DF3467CD0,0x492B6580F8058DC4,0x339CB0EA72EB,0x8960EAE09DE7D867,0xF13485A36824DE5B,0x117E035CBC85732E,0xA58E19C9DAF0DC40,0x3BAF7EEC5AF1F40F,0x404F34B63C74D894,0xA23F8914C830,0xC983335328420E5D,0x79C286A5071BF4D2,0xDFBDC55D634338E8,0x697511718C4AA293,0xAF5CF0B482CC875C,0x1EF2E2D292D0C957,0x977C97D9279E,0xC9F66A0683B8C3A,0x54EEE9D8BDE994E5,0x7ED22AFD0860F6AD,0x711346E4E1C858C0,0xA692118CF50453E3,0x3036C4A5D963CA83,0x1A132C155E430,0x7DF5DE1683A44824,0x243D598920FE9A7C,0x692A4A2B22F3996,0xD6B279FEA06DC015,0x9965C4B8F2C09939,0xF7FF91ECDC4E32F4,0x13C650257ED8E,0xD64E8329AD199D87,0xF66ECACFF297A171,0x957460EC027D5615,0xEA476FB14CCCADD,0xF608151FFEF3694E,0xD27CD80139E3DCB3,0x3D625C6A9E8,0x870988C29F0030F,0x5DE315F6369F31DA,0xE0560638A01894E7,0xA491C7A21DC59FC2,0x4C3F60076317783C,0xF1BDD6255B597B43,0x1B33DB5C62C49,0xB3BE86D29967A865,0xDB38E3F69BAF6457,0xAEE18EFF08AFA4B,0x9DEABC64475EFFA7,0xBA0A3B5BAE292ED5,0x9BBA7332BF019FB7,0x16A7299C4E8F,0xE6861A1FADEB275,0x2075B10C5ADE9821,0x46A86544757505B1,0x6BBB476E6583AA41,0x31223859B2B39372,0x19F39E6BEC13D373,0x8C68B695D2AA,0xCC2FED47AE78DC2A,0xBC9A2CA6BD18F4C2,0xCF9E50B65018980A,0xBC6A2DF109C0CA18,0x2D664A7D1B84B2EE,0x25D97DBC2F591D05,0x10EFBE11F08C8,0xD1E29BD5C1C4F144,0xDE77536B090618A8,0xB0DE55BD147671BC,0x34916BD6BAB49704,0xE395ED1D34017B21,0x54A21D25EFA3D59E,0x195D2C3C8C102,0xC33E3978EEC4F1FA,0x11957BC9EF62BB9C,0x43AFAEB972EE9F0E,0x1DA8E0CFF9EABA06,0x24DB30F770F137D,0x99575EF21086A3B4,0x1EDCA430703E2,0xDA29A1DD58C7AE34,0x977E44C3210B70B4,0xC51126523427AA05,0xA2B05B44C4803A63,0x378EAE846D0312BD,0xD72C2F48D8745C23,0x613C40A1117A,0x9F9D362976F40D96,0xD99BBE928EFAE86F,0x77C1E91EFA831B,0x7E8A0807C5E5500E,0xFAEF3C566C9D024F,0xB9329BD4D006F227,0x117B661F0BB6A,0x7A135BEE40757BBE,0x306CC8D10A3AF9A0,0xF7D6743580432AFB,0x5E59F7518EA17D7A,0x75AE14659521DE88,0x2F83381C01AFC99E,0x99BE21C5E3E1,0x42B0D5B05F7F5087,0x6ED6B0E99ED7EAA0,0x52E9F33C0CB23852,0x6928168169CE5570,0x56E206EAA9BDCCA5,0x9F6DE5A69FAB6A13,0x164B9AEC6F6D4,0x5A8F68B8B0E78675,0xF92F400CBFCE87DE,0x6D35E78E1BDE0830,0x3C1AB45B5BC2EDF,0xD1E2E249C9988A2,0x420461DEA4EA9A6B,0xABD4794DD096,0x535590A143DB7CEA,0xF61CEA217FC37F9F,0x6DB8E94682A16E04,0x21E012004C8470B1,0x6A070AB20BA37D2C,0x6C9460E143D3AA3A,0xA37222A352A5,0x1A495FA9BB748780,0xE40F4340E993AE9C,0x461369E6EC949FEC,0x67E587023F337697,0xFBFF368A111842A,0xB27EB190E5596253,0x219F1F119B1F4,0x7A5895E4C84400DB,0xD4B450064E1A526,0xE44B762CBEA29082,0xFC0364D3B6558CA6,0xCD1B3734D1F9E573,0x140B822AE785BF18,0x156BF932A8183,0x7E9EBB009BDC03B2,0xFFB2998DA699BF21,0x264181A5891555F2,0xF5599C8E3E3C2150,0x816F4A3CD7DC8741,0xEE6A45E2E3482597,0x1781AEC94BE10,0x382370D315ED6170,0xA10372EE5D6FCAE1,0x5526E3D7FB2E880B,0xD656C5FBB7D832F7,0x22E319B35FEE36EC,0x42B6B4B9C267CA56,0x7C770D535553,0xD86C4BDE8031F281,0x6BF12C0B987168F8,0xA7F557020FE57AF,0x69FB1836B6380366,0xCBE1E7CA931C623D,0x323D1C387A1B4789,0xB0F1296A5E2B,0x7338F8B5C7E9B29D,0x426B5C75DE8C8DCE,0xB67063F4FDC3F23D,0x7C5B6B7971793736,0xCC4208DEEFDE9453,0x25166CFEF257497,0x132B040B9AB59,0xC3C749855A1D8A16,0x4F1A9C294B3565B7,0x512EEAE9BFBE5AAC,0x43E3E4DFF1C62B53,0x7613DC64194928B6,0xF34CDC95BB824E42,0x163EA87C166C,0x312B87369225CCFA,0xEF8962E3690FF351,0x47D33612FCA91366,0x2AF7FDDF9E684737,0x22730257678BFBD2,0x9CD0114800B112C,0x16A4D8B353ED6,0xCAE78280E308E6EA,0xD681804330FEE1E3,0xA4AB00EE90531A81,0xCD4CAA4DB872E204,0x3A9551A227132D23,0x3BE4C3178C0B5B1D,0x111F7F53F5C77,0x9008BD4792E54A19,0x8AC7367A4422C0F8,0xB4005E7E7EE88D56,0x3F1546C8B7446D8F,0xA4A1CCB16AAE96E,0x67D08471B9BA6AD8,0x144C3F0FFCECF,0x8736318961941A53,0xED6E0E9635E86131,0x7CA40E67B2E9AB36,0xBE606F2AC109A825,0x4F2B1198250D908E,0x3225091C04B09195,0xEC693D0AC3E3,0x4F277C9E73F2A111,0xAB607A5008F2D8C2,0x69A89D6B649D42FB,0x88E482741647DEE0,0x371280C3D5B669C8,0x8DD7727C23D5AFE8,0x5D2544E7489D,0xEDEC5B680AF3D500,0xD4520660EF4B55F,0x8C043F029D8FC5A3,0xFC6C2E01FF04BC37,0x217D05F4DE288E89,0xC7D30EEC1016D348,0x155A08E78628D,0x447BF4C225226839,0xB883ED3FB3A52461,0xBB0555C0B007E0B6,0x31C048B559B46B16,0xD44E01207410D663,0x5BB01420B81EBBA6,0x60C946F33E05,0xB5D3140942BE75D5,0xC8109C48B10D9888,0x1BC5A987D09F47C4,0xB8BF2C0F76AF23B5,0xE0E2A04664660647,0x6D58B7EEDD3DAE04,0x129BA59C57913,0x8BAED70AB64391D,0xC77787C77AA099C8,0xE1D51DFD8AD7979A,0x48325767596374EE,0xC59B6B28E7333F64,0x61E346FC955B6458,0x2205DA5BEAA17,0x9754E0126F38BF6A,0xF390592EEBBD8E61,0xA2402F58D7DF42BD,0x1CDC8CAC3014CAAB,0x8A2BA7F2186E0731,0xFF5703BD43832136,0x13E474D088E22,0x34DA9AB4DE5F1482,0x9F66274A5100F430,0x3D1F1DB437CEF4FA,0xD949DCA4ADD35587,0x30B90F9AAF9B6AE3,0xFECE70370DEAC3C7,0x3C9F1FCAEFA2,0xADBA6E6B20629EAD,0x319EAD085588C04,0x1AC2609CD6311FEA,0xC3D7A842AFD7673A,0x739DC942269EFA3F,0xEC0D7F0ABC7BE75A,0xB51F4FDF4A2D,0x85A237F40A3CFB55,0xD929B119CB741757,0xE5D7B98797CB6472,0x3233D6ED5389865E,0x9EF2B1D80F6046BA,0x90EF4087C0DA6CD8,0x52465F3106A,0xE1A67052E24D71DB,0x47D6B754C481AC8F,0x30E28090C771C4CB,0x48BFA5E75E58CD12,0xB396C1633BB3D263,0x9265A047DD665B54,0x1F9D27150DEC,0x6A6A33A4EB026227,0x4F502D7EAFFE6205,0x29909B58EA9795CD,0x2E5A376EC650CF1C,0x8481563236BDAE35,0x5D75DB648B1E8C91,0x5B21E1AF5724,0xB62D1B7813BB7095,0x3ED57C0D4ED4EE8D,0x1CBAC805E34C9084,0x539A41CE72DC3BF6,0x641B8E073281CE87,0xC76D2220D2C0AA35,0xD71B7BFA8064,0xCACC954B5C202809,0x814F68FB164B5348,0x6794C7912F47FDAF,0x519E5BFF29E0D48A,0xBE84C80F45F363AC,0x2E690DF1D38F8CC8,0x1BBE38BBABF2C,0x1358569E74C3819,0x61701E48C29C038A,0xF15BCE69DCAC9B95,0x2F0CE8A2759C728F,0x70FA1C5C1D3A1D73,0x53A549584551E036,0x2E7F470BCB03,0xA22689292CB603F4,0xA2AB8B080CB6D5CB,0xCC062DFB57B7F088,0x94AD1D973A7E748,0x9A6541E2DB7EE6F1,0x8FCA855F8D648D9F,0x13B33A2C2C92A,0xE2D0CBDB512B7007,0x9E7D3E105A165E5B,0x9BF6C4B99F8161FB,0x5F1CAEC6DF2A6FDA,0x516E72C1DA25196,0xF83F3FE5C9AD8BD0,0x1D5D00062857B,0x55B4889775BC067E,0x19CD451D7DF4B125,0x5D5E1A13E9AC06D1,0x91EC48DCBA22266D,0x542D1CEE574ADB37,0xBDB1A669991B93D7,0x21B396C35DD86,0xC253424B7400FACD,0x12E90BEE17A04E06,0x4B7100D70E6BC850,0x5AA98B79CA67F24,0xCC83F8CB15238D55,0x27FBFD6C5AC30905,0xCF74A9FBE99F,0xD95FA749CC2AAD5,0x8F2A986435C362A8,0xDDA94651449BB1A4,0x2F2B49E69A206DD5,0xC946B1E876C78A04,0x7B99EFC8855928F3,0xCC448E5A1AE,0xCF19198A0C367FAC,0x3C90232BBAC99377,0xEB13848694390E2,0x4B85229FEC74DB8,0x21978AAF0E812A2A,0xF249CB7045C3B1E1,0x7B170A4480A6,0x2A14A627C8E7D3A3,0x9592D3608A2D2640,0xCFDC65013201699,0xFFD90242B7637E39,0x216E72B056ED0F5C,0xDDBD4E097A99BEE7,0xF086C3AAF972,0x3D5DB71F2E0781D6,0x465E6D4788427071,0x6B31D00BED4DC6CD,0x18B2365A3AD488DA,0x74183D5CE7B6642E,0x2535A80523E21DA0,0x129C4720694A5,0xB84837BD4B4B5B06,0xAC2374CF7B462242,0x1C57870F4B6FA099,0xC69988D9529B1273,0x7A0D22837A340B5D,0x21EB1511AE38025B,0xDE868A311361,0x1763890E4113B713,0x97DD0C1BEBADDEB9,0xB71200ED3F02FDF4,0xA41F1B44F94954F7,0x46F2EE5BBE823E48,0x7C10B74F106CD574,0xBB68146229EB,0x9EC9C6696D1EF2D6,0x29F41CB297CF38A2,0x375654F3339B49AF,0x4053A19D6751A39E,0x4020609D75D2BD3C,0x5CC066BC1375FAD2,0xBA841CF64C0,0x24F22285DAC19C7E,0xF3777471B8DFB6C9,0x8F7845AEF2416084,0x72D3161A169957CC,0x87EDF60EC7CCDE6A,0x31976374B60FA750,0x290B66B05C0D,0x9120C07565546675,0xCE2891EDEE9A1CA2,0xD5676DA332F6BE73,0x671E4541BCEBFF35,0x242AED038DFF4C5C,0x3A69E28E95A7592D,0x77A380B1599C,0xC442A090CEF0940D,0xC96003F230F5DFD3,0xD972CF2A68980D92,0x290B01C1BD385482,0x381F00D306E39997,0x10A4E26E839828B,0xED5D2A0EC0CD,0x62519D1E8C71C486,0x6CC0C599118E1EBE,0x3323A21CAA787BF4,0x42D7CA8BB2A4AA35,0xCCF02347BC6B55ED,0x831721472598BB6D,0x19E58C2D29482,0x3151195BBA32D637,0x84F3565D83881644,0x93FE121B97AE6AE,0x6DD8C9F9F4165DAF,0x5338DA46203365DF,0x1E8F5599728AB5D,0x8C77606E66D2,0x1B82C38E1D05EA96,0xA9E1FD5F156EF29D,0xC09B9760CAD7ECA,0xDF85514541A4E9B6,0xB9A46DC1F48D7FD7,0x37C7EAD7A25A16C3,0x8B848E829A58,0xBB948AC7647F95DB,0x2BC3E282693CFE73,0x39D6DA4D7A206F07,0x845F4C1ED9901CD5,0x630F130A33A6E54E,0x962599D90A70697F,0x1014F0432FBBD,0x9CF0E7C4B8A5465F,0x50350A811AC57590,0x884C66425C9C9259,0x3F1FDCE4565E1A04,0x75FF00229018CFBD,0x429253DBA75C36BA,0xEE660DB6E671,0xBD6175FAC221EFA2,0x4B22C573815E1153,0xAAE777F553A08562,0xD14B74A0C4EA1FF6,0xC353AB0F4737D834,0xE7DB7615A5E6F905,0x1831190D56D12,0x6BEBB99876899EFC,0x567162E7C4EA2F4C,0xD3D387E3FFB6C8A5,0xCC1DD593EBC2F535,0x9EF6DD7E87B66D9,0x6807A5EB956883D1,0x1C35DAA2E4770,0xCCAA61E9864982A5,0xF43B9182A0814DB2,0xE819AB728F47ED5,0x7948D9D986F98292,0xF7D58765948EDCA5,0x32CCEEA126B2BE6,0x2052F68023386,0xEEE200809F944896,0x26AA48F4A6AD5970,0xB8026C1A283E771C,0xCE2C17A3453E89EA,0xE137224507DCB55E,0x1C4C29E6F74A5D43,0x171D10B64B3C6,0x748BB5B8DA24B2E4,0xC3C697987F3E6F84,0x128CD9117ABDA64A,0x6042044E4E8575E5,0xB6C9261B195580F9,0x1B74A2F841FB4701,0x1C529528C0B47,0xF1FF09AF0A0EFE7,0x378A1BAB980108BA,0x7BB3144340ED44B8,0x5CD7F7BA5995638C,0xD5D6CCBC7BA84192,0xB9A1871A2B6D2DC5,0x187C186A962E9,0x182935E4F6661A44,0xED1FA57AD444D8B6,0xB2817CFC86EB7B9D,0xF27BA78E19D3F6D,0x8CA6988F25CB8F50,0xA62E1A0F6B46F3B8,0x1D0E7C858F6B6,0x8D1C7818D04382EA,0x19A4DE6D0E912AC3,0x5C7BF61F80264FD3,0xA650F9F54BC86321,0x8F3C6400DFC78293,0x1CFEEC739B8AD6EE,0x80F69214F75C,0x8C691331EE657930,0xD114BE667A9ECF81,0xAD207E62316131C8,0x409A44B127B66CFC,0x7D8648F94DD2993C,0x15DF5D48B2086273,0x1A9EA12312E41,0x280D58B75219C1FE,0x6F92974A8F0586D1,0x4C59281F207044D1,0xD8328DD8B129725F,0xC79751FFAAEE3342,0xEFC71549E60557A4,0x1B58EDC25971F,0xF9A065D5C7D2DEE4,0x20924F853C5BED1E,0x3BAE976C9CB825D6,0xC44559710311BCFB,0x5E7F29A533DC34AA,0x3F263B184CB6946A,0x18D06BEA0C8F1,0x36FB060D33FDB4FC,0x4988BA19A8B06990,0x8D22B53069145B75,0xAF7FFF7291C122EE,0x3B9C0936AB940FCC,0x85BB81E424183351,0x3E0BECCF184B,0xE85D6930BE6FE72C,0xC80B8F443322C878,0xE9C2D55B597F1E69,0x5B9B098E9B057487,0xDDD1A8FD742EE48A,0x2D9C81DB9A3CC4DD,0x2341876BBA9AF,0xA2D713CA4DA69538,0xEFA983630A202D49,0xF58532E4A24C3A1F,0xA07DBA43E5532E40,0xAF7492709CE1A512,0x2A26EEEF9B7C9FC4,0x13E1530C78719,0x1703DA3AABD2C2BB,0x2A3BDE28A6F5F34E,0x36B9090EB0B1CDF8,0xB6E7958976925FB1,0x2F9329EB65B67DB8,0x6C1434CF347BAD48,0x1B9BBE5CC8774,0x8A11FE770267C03E,0xEBDEC10B2F148F7A,0xDB9661298D66C84,0xC1301D10C237C197,0xD42C55B228A79A6E,0xA41DAB18B18A7140,0xD299CE6E5C,0xF84F68FF780FBD01,0x75A6C70295F9C502,0x7DAFB1E3B41314AF,0xB2E3C53BA34946AA,0x89A7C83EC1F56106,0x7CAB1A32BCD4ED3D,0xFE592925C5B5,0x824E2D8F22AAC089,0x788CD4D40123BA3B,0x44CE79BE677C2E23,0xD87E0BE777178390,0x19C0288B88D4D75D,0x9E2A14544B4E1DBE,0x2082473812168,0xE0E26F23FF29A66F,0x8FE04D27039CAE77,0x8E7F0F5F6E1D63BD,0x50A05F46864DCC87,0x5A20616366659FBF,0xF98F63A203063AE,0x2296B7724AE91,0xCDF1E9A176A9FB83,0xD0D821C77909DA59,0xB95C5CC74F1513D9,0x1899086C3FD8A3B9,0xE5B4E4F7862DEDEC,0x72227ADF6DD4E281,0x136FC0FBC098C,0x4CF38718DD5766A,0x2109D8E061455D9,0x29313E6AF349D606,0xFC6191AF6431F9CC,0x3C43844146DA3F76,0xBB4B3F0CFD8C5368,0x185A7C9FA00EA,0xCE2EB5763FA15596,0xA7C1CC3BE4D45B81,0x40C2AE24131F7C82,0x545AD9042DEA6F83,0xB6E3E3C91956EE18,0x472C9300CF69734D,0x186C43E7AD5CE,0x5BBF9CFDE7BB8453,0x8CEB3A4B1D2267F0,0x17453AB197C81076,0x1F0AF6F2284F8921,0xCBDB65A4BAC4FB55,0x520638A4E48EE6D9,0x15D40997C7A8A,0x1C107925B8ADB925,0xCB68EBC192DE6C15,0x436746CB827CA8EB,0x33FE20395242911E,0x8B39847C6AA6BB2C,0x23DFED045DB248A6,0xD7644D44F90D,0x2E089CF9935A4672,0x7FC8F4556BB1194,0x7AD2837DECC6176,0x1EE96BA08F355F31,0x189FFCD45A5E4ED3,0xE9BC5D7F4E0E3231,0xB675D32F43E7,0x722682B08E26249B,0x625C205CB137373C,0x773850FB6267C440,0x117CAE101635E96F,0x400FA1C9D733496D,0x43CB9098CA3E5E7C,0x23A7CFBCA4A8,0x38C1D813D4A5DC6B,0x3CBFA38771160F12,0xB12ED79654978E0C,0x5A8D48941C7B7B9E,0xDB2E3C237A13F7B1,0x6DB25EFC590A5910,0x82220A665531,0xA3AA4256AD780BA4,0x10209F9AF6776A40,0xE22D306BE4835C4A,0x842EF921B4B9BB51,0xC0A670380ACB29E4,0x7157E1F87E752C2A,0x2068E149E54C0,0x6CDE861409C8C890,0x14FACB12CB84AC04,0xAC4045A116D2BE7D,0xEDB99688BBC3B9D3,0x53C6C6365329DE0E,0x5368F48EC3966CFE,0x39378A557AE,0x2799F97DF597D592,0x49C07A3BB43ACA35,0xADE8D932A952DED5,0xDE1624F6D647D759,0x4DD12913D2175293,0x84239BF5DA2FBC54,0x185717BC7D9CF,0xF17073C74E5F367,0x75D5B957D5FA422F,0xF6A1161B2393EE19,0xE45E2F3F23312D25,0x669931B8C46F1548,0x6AE817EA5D47ABE7,0x9E66081536A0,0xAE6F0B8EC93A5EFA,0x4E8BE6453050A7F,0x7AF891413407F3E0,0xFB92AB099949633A,0xEB8293FC1015EA5A,0x3EC8EB2A30A403CD,0x1EFF3BF3FEB1D,0x6C1384F34C25765F,0x98448695EB511E99,0x23E92E9E05006990,0xA3912E8922FC4E24,0xEA4AA2E4CB19C120,0x17232578D35C569C,0x192BB46F70239,0x64CC26D0FA313211,0x9B9BA33A5D033059,0x184062EAB4B72582,0x3679E17D8C6EAD22,0x3591B6809D2D5AE5,0x5BA93B62EFA11615,0xA6EBC90BF042,0x75E13850F9DFEF4D,0xEC3B071747D5086D,0x4FA68D20F6918B59,0x90814F459C92E08A,0x410E8ACA808CA92F,0x2F345B1ABF4FEF55,0x31F65E0FBA6E,0x56AD002F898AA856,0x47CE4C8146BA86B1,0x82C088F0FA423455,0x1C49290CCEC6D070,0x5C2655A64BE2D5AE,0x36F041AF16954C0E,0x12548E36E4FD1,0xD370B4509C649F77,0x25A24D9C15ED432F,0xD324EBCAFCA09E3,0x23E99B1D66E4628D,0x82D2E73F726BA270,0xDDB6F62C7BDDC930,0x10DDF32FD84CE,0xDF0C37C30DA02B6E,0x98E314C539EF301E,0xADEE4C61BD1CF84A,0xFC991C6121ACBB53,0x88AFF9F166E949D0,0xD58BA8FE0910B487,0x1136C6E818864,0x50E4F4B98821D4D1,0xB575CB814F310365,0x2ADC07558EC10159,0xA6A89B5061EE63C5,0xABE4C0C41FFEF67F,0x2AE3D2951337E71C,0xF6050E9D52F4,0x513998B93FB7B8F4,0xF6BF005E2B8A1C25,0xFA82E07DEE27C541,0x887FC0B87FD94250,0xDF1F205D4C14A304,0x54C72C8DEF7202BF,0x12F43D5E5181D,0x7EEDCE6A8D5DE95D,0xF5FA1324371EE79C,0x574EE6A0A004BE6A,0x60A8FF27DB44BFD2,0x7669530B4BBB843F,0x5BBA13842841ECB6,0xA0936330E281,0x57D384A28AD152CA,0x3ABD779394FE3448,0xDA76E8F3DE0B4037,0x4BDC0627FC8CCB6B,0x88E2E791F5210CC9,0x983FF1E7F337E5CD,0x1367FA28F1C14,0xB2F25861353EE767,0x821513E0D5F05C43,0x937AB8406890A79B,0x90768C9385F9C289,0xDF8CF5E03D458B7C,0x286FF32582442D5E,0x3AF3D28A71CA,0x2D5AC72A28422BD6,0xA4C1F2E50D20F228,0x14860E038C5C201B,0x806E53838B9577B1,0xB64BB187DB6A7D32,0x2760A99AE13C7432,0xFCD4031A1A36,0x3CFB605CE4A17332,0xAAEA040B4B405D5D,0x878034393AB5BFCB,0x58A78023319CB60,0x7A59CBFDF4CBDE89,0xCBF4F757C7001F7A,0x199287F695931,0xE3354273A95E742F,0x72A24FE402EEE0D8,0x5CBFFD0C1E600473,0x46CA38E9B3DAEB14,0xE14C54A93AA29A51,0x25FC090248E113A7,0x116C76B8D8C5A,0x8FFEDCBE1105DDB3,0xC95EA9E01F0463A4,0x4CC4E73BB4042D82,0x7E414D4A282F207A,0xD2F440795E72F08A,0xFEE339E71F605281,0x1C57EE652C1CE,0x5ACEA5059F0950BD,0x90A8B09E52F4971A,0x59805C135A6052DF,0x38E8609B40E36F82,0xE0706C600BBF0621,0x8E5F37EC4BB379D,0x194853E1B7C61,0xFEAB9992EA633ECC,0x79532EC650DD929E,0xFA9DC95CCE02922E,0x75ACD5118F98DD87,0xB77FF15F827B5980,0xE860085573D4738E,0xEA5C6CF82A4,0x3355B17CF772812E,0xA7D3FD2709D03A4C,0xA2DB950491509D06,0xB40DA33192A50A9D,0x66EBFF39C3A9DA2B,0xEFD0613FDE56A09B,0x2183AC6C921CC,0xBC31759460FEEC2B,0x34F66F615176310A,0x568703F737BE1B08,0x178950B77374354A,0x1ECD99422103E64B,0x3C3641CABB5FD1F3,0xAD35C5A69ED5,0x4B92C984A3361EA4,0x3BB460616BD402D7,0x83740FCCB35EFB85,0xECE0BF66C6349E56,0x45C583D7E4787BEC,0x93DC001AC9B3041F,0x1FD22E6A24FCB,0x7F35D5669F5EA85F,0x64356B6BB6857660,0x5E25CE0D9121E63C,0x60552CFDCB048A40,0xC11F94AC55C335C3,0x55D3F4C025659DD5,0x1B8F01380775F,0x61A1350E812C742F,0xE7A68B2BF960139,0xC0CACC1ED26874EF,0xF2C6781DC082AA5C,0xBFDC165B8789D248,0x851058D7009C8A37,0x1CEC589E05125,0x450C732825D39F42,0x6DA82924CDDA3398,0x92533B055624D79D,0x97540199CF72E85F,0xC5CFBCCE9E3D8665,0x3741EC05F2B37C8D,0x110AC4C8AC67D,0xA7C03B7A794A2C0,0x3900722857C30881,0xA8C441EF2E641D12,0x753C800DA8B95AAE,0x2E33B7FB49ED6FE2,0xEA829A0C10C83864,0xCFA2133FD629,0x6DB607BF53970F31,0xBC2FC39D8C8BB594,0x123DBBD1B0A48FE0,0x68B9F03C7C0D2E6,0x587ADEDD473A2E65,0x6D3FA7B18FDEFE8A,0x18592929B8E28,0x378AD36708A1B4A,0xA06A533FD430F3F6,0x80CC5F1DE9304DAE,0x932F25E295CD7C68,0xBD82D3F91212C91,0xDF480638B3006AC2,0x85A9C36EF42B,0x1ED7C0957C3538E5,0x78BA6FF2201FB760,0x481BC0C01367CCF9,0xB865DAECCC43959C,0x77F1F7806DC8C70F,0x1C2EC92AE51CEAED,0xA5C02B6C10BA,0xF31CC9E6FD0813E8,0x2344462F5EAF3A07,0x751319931D916906,0xAA885BC474B9245F,0x1C9C806D7D7DE4A2,0xAAA760C170A5B67F,0x58178A7C5D25,0xD5D4A678944F4E9F,0xCC01254A0ED2209E,0x89A2CA5DA9F0EAB3,0x78375858E080F17B,0xAECFFAA0EAD00949,0xA2F1068357325D3B,0x14A5FA0FED09D,0xD6C64977072BAA28,0x44B3BB43ACD8624F,0x6A0D011485A169A0,0xEA5162B8220D33C4,0x5A77B3BE17E6C173,0x6BB26282A27016D2,0x1C3752AD9AB3B,0xDE0A1934549B2B0,0xF02E56BE925E1630,0x642039BA1B07DD17,0x43AC1AFE50ECB59B,0xB7C96A8E2FAFE340,0xF1B8338A67298B8D,0xB102A3F4C0EB,0x9578F5AEA81D0456,0xACA98AC201478857,0xBF43693331F05E35,0x38162A9EA8CE96C8,0xA7D77A9D727F9644,0xB8BCBFCB96C1A022,0x12C352E40ED55,0xD6BBD20CE1C81E15,0xEC53B43C0463B2FC,0x9C4E4E29E2EE4238,0xC58788FD508558E3,0x793D47E589D42F02,0xF17E896FE3E8B436,0x211B7687797D8,0x62845FBFDD3605D6,0x9F9AB96CB199CEA0,0xFF027B962B5355EF,0xDFE590443F995C37,0x48A2C8C8C081BDB6,0x576E85E1A9492922,0x4485D2DE7492,0x7F6E5C3A7496BA14,0x655C1142DC33BF6C,0xE0C23F7A01FD12EC,0x1F69B23AD6B4995F,0xDAE79DC2BEE26ADA,0xF5A7F2CF65ED2D84,0x7047D6855A2A,0x7E85838327123996,0xFBEA6792EF9E1379,0xF750E910CA4D48E6,0x4A231D72FA241D18,0x638C1678074CD8BB,0xDAFEB076D30BEDA6,0xD2CE89C66BB1,0x670ED66A85B397CC,0x631098BFFD72911B,0x335A6D7C255174D5,0x35A13EFBCD2D2FAA,0x87A2465FF6CD1902,0xF1B9D495B489891C,0x1B00B77821658,0x6F18AF722BB30C7F,0x1B71AD600A9E9A35,0x2C3847E0BE8AE564,0x1015E7C5CFEEC10E,0x31D04C669767CD91,0x57924F67E80A428C,0x70F982C892E1,0x441AAF50A7765ADC,0xC85FFB9509936729,0x1B483388F704517D,0xD9B3181A1C1604CC,0xF77B1FA22F46E894,0x9F9DD17A3BB7C5E4,0x2227E953EC04C,0xDA9A4A4A683A9E56,0xD65D4EFA9745D31C,0x18C3B0D12319785,0xA52A2F09E88C3E12,0x7921B8F458A4AD9C,0xD2FBBE08F54B445C,0xECEFD08658B2,0xC6EC4458A5484DB8,0xBBD8DD2EFBAF67D9,0xC5731DD7EECF8430,0xFDBC56BE5E02D136,0x6252AEC9FA679E1A,0x79DCFE66A6A49FC1,0x17B0992E0EA28,0x79161446C4635D23,0xA1860B263FCF6746,0xECC9687C4B07A5F7,0xDB2260BA5106BD8D,0xEEDE0A3B79EE73A8,0x1535C455F59BDAAE,0x1A87D443D7E3,0x9F682159F576447B,0xC97C783E52FBC9A,0xE06DB304763D580B,0xF363577676C0E14E,0xDA198797E7D7CCFA,0x8B27ADD97AF00D31,0x139B2E237EBCB,0x545EB8D360A26F97,0x2EA8E39E523E1148,0xA0AC32EF0C968684,0xE3E368B616D43921,0xA99F764714D65734,0xE25D2DA2A18F9251,0x1F426851A11BC,0x1AF9CCF2CA16A3B,0xC6ACBF40B74B1B79,0x1A1AADC8E4AFD078,0xA73825A7A740B8FA,0x4ECFD80154708C14,0x9A05F2F76E39048B,0x199EDAC6297BF,0xF26CE6B3C9F32FD4,0x30EED38F8D91A2A9,0xA296F524BF913859,0x889ED41B52B989B8,0xF1CFF0B92ED0CB5A,0x30CD703206864ED6,0x190640EDCF66B,0x5374868976F20FFE,0x384627122BFED35D,0xAB3B711576FA871C,0x79461D41FB3B6DA3,0xA88C50272CE5D2F4,0x7ABA5A27D8A64CE7,0x1D69A43412569,0x3AFE70696BCBC677,0xE59BE0C5B05D8D25,0xC0B08E301B9765C9,0x9A3B377C55B138EF,0xA246597C08200679,0xAEAD737A3A2BF481,0x1415A08DFCF64,0x213AB48B00E29C07,0xBD99D3B0A4E3D108,0x72D5F5A36C3C1BCA,0xA2160210780DFD0D,0xE5C253444A62C137,0x336BBCBCE3E02F48,0x39976769AC8B,0x70208293363DB27,0xB76E699117DC4B41,0x4C242367D51CB9CB,0x1BFA0A4634F0EC83,0x264055BE553719EC,0xD633ABE9F149B2DA,0x139DE7B163FDD,0x62BFE384532FF23,0x84B01FE7AE4B064A,0xB4C24B2A805A7694,0x69FC06E4C0FB9926,0xFB86330E05CAAC3B,0x93BC96F121FB6B01,0x2E7DFD6EC296,0xF56533AD11BB02ED,0xE481719AACFC35F8,0x870A371206C32EFC,0xDB062A0447041124,0xA9B9EEEFE223052A,0x4030323A73C0BADD,0x1BE03E8172B46,0xF27871CD147DCFC,0xD47A2BC5830170C1,0xB3DE6FA5D1632FDD,0xC5FBF22AE9BD163C,0xC3B3CE5E0207A080,0xDA3822D40E59F171,0x1FF4C77FDAB5C,0xEB2CF9E92B4E8E0A,0x39B9A2FCFE6BC6C9,0xD43DD07116F81FE5,0xB109F16D0F0DC03D,0xD305CFE34D8536E,0xFFBC2E2E815671CF,0x77F9B197B6F6,0x3597147BE0A486CF,0x3DFB00F4D8A60AD5,0x23D098A22C8568AE,0x6AF1491960EF6104,0x39DED34CD7668E31,0x48CBF8DF96D33591,0x5DB3312C047D,0x915F847D99432F38,0xEB56CF680A227E54,0x3F4ECEB841F1F52,0x8F96647EA3F66659,0x93063E22D114B88A,0xE6B02B681D6D1786,0x119EC297F8DD7,0x524C8C288E278086,0xDF04C6DA4587213,0x660255D300937D3C,0xB6AE96481FB45AFF,0xA468E33EC50667F8,0xD1F973D1F424641B,0x1E1E440BD06BF,0x3078004CB2F87442,0xA4BA955F4F4B4343,0x23E926231076C8EE,0x56AE3DD996773DC6,0xEA66091504D4AFC1,0x45527FCFAD8A809,0x283185584F74,0x54D19C6803630BA8,0xB7AD1C069EF9B99C,0xA63D5752CA05A492,0xBABDC9EFA5CD01AA,0x4AD8BA7B28B04BE4,0x7EA42306B4CCD45B,0x185A8EF6A828,0x1D29779603072BCB,0x98BB0106F3C9D8E3,0xC0D167F257E375E6,0x3181C8BB9914977E,0x1832D37F1EBD500E,0x6602FE3329132F7A,0x21878A5E7D28E,0xC65BD888F99BEDD7,0x7D26C7318F3A0E7,0x9E0CA6E0434D5BE3,0x21D30A8C5BB5F9B2,0x91AD824606253D73,0xDBFB9198F674B2E3,0x740A04085FA0,0x14F9A7208C10929F,0x46F94158C3637051,0xF9F7C912B0143E06,0x8D18FBAC3B3A3FB9,0x495A0FF5B3E4A969,0x78965291F22826CE,0x149BC2A39E791,0xAFACCCC9628B0A5A,0x99211AE8671B897F,0xD5ADC5684C39F9F6,0x2B0AF838D79024B0,0x9066FC62D57A5D4A,0x1A319D46811EE49C,0x1C5E8CA970ECF,0x70E53AFAA2869878,0x729D84B0A8A88FB8,0xDA8D24C0793FDF84,0xDBFA064747430AC8,0x613550286308B4D7,0xD7B6021AEDA051F4,0x103F488AE58EC,0x9085B7210BE48927,0x3E17F4E03EE5CDD6,0xEB336FC34B4DFE05,0xE2727B1F4E2CB1AC,0xFEC489A704A10B97,0x63EAC340E55BC180,0x2054A25D35218,0x32990C07D5AD90B2,0xAE880262C4886912,0x5256D5C3FDDCA3E4,0x70CE3079DB211A56,0xEC27AFD26BCE1B6,0xA557BAE72F5DA394,0x98470924F022,0xB54CAD89D35A3C99,0x1CF69AD6368174A0,0xECE1DFC0681983F4,0x5810B4812E61B175,0xC6629BDB99D17CD1,0x59F7F24926477E61,0xF7F779F4A1B,0xCDB210BFE962D26D,0xA226D3AAF626DFD9,0x6607410A1F2EDBD7,0xAEA460E065DE1EF,0x2FBAD039AEF856B1,0x14E3BE5245D83582,0x19BA37C46BE60,0x364327560492AA2C,0xC84B059FA29E5DB1,0x2CDE4EA25E789377,0xFD44F12959BBE89B,0x7CB817DB0955C0CB,0xFD6A959BBAD2DA8E,0x180A34C77F930,0x7EA8B583A46866B5,0xCFC11F56593DD9DF,0x199A05B022A54DC6,0xE7C61FC71F29EA76,0x2ADAB8F037EFDF10,0x8177702C62D00345,0x1AF1986C9489,0xB1EBD5D81A2F4E47,0xE02151561F7C3DAF,0xA59388D3E9C53D9E,0x7DED93F982310990,0x664A13B359E282B5,0x521AE1FA2D19EC7C,0x16E9438796653,0xBFF32C1221077F76,0xDA58DEC4ED98DEFA,0x627E16ECB9686843,0x94B1302E5771C6D1,0xC32C45ED1CCDADB1,0xB9A85FBC7B24C48A,0x168292646FA4B,0xDB801166802EFC61,0xB6786BFF6227C682,0x18C184B660242CDD,0xB63C60C2970F8AAB,0xA264A0B43810B3CE,0xC3B06C0707E8AB47,0x20BF291B57C9E,0x13D38AA995F066D1,0xB42471DEC81D28E4,0x14FC13B1B335FFC7,0x5066F8967935DFAA,0xADC100AB670A6C6,0xE1DBE76A8844E6EE,0x1EE0720B328E7,0x2792E12CFA1BDAF9,0x6D6ED1ED0C1166C0,0x3F860C4703C8DCAD,0x7F0467627ADBB954,0x4CB2F34F79B31DE,0x3A39C4F25052F25F,0x8701EDBD7232,0xC36425449FE4C2C,0xC76362E89FDEC027,0x46A425758F1CC8BC,0xCCB740616F7501E7,0x6DB2B232E18E2036,0xD9F27AF67D2FEF97,0x2EC52247DEBA,0x3165E43C9EF4E28A,0x3F54531E75DC5FBD,0xB6834BF14E11EF0E,0xA3B3D9843B95EA6F,0x89DF4389AC202180,0x73785A8A70E43185,0x3022D3AC25EB,0x439043239E1823AD,0x3379189B848EAC1,0xAC83150649D53514,0x1C8B10530579AD77,0x9D9CAD7BF862CA1B,0x1A79ECF406DC0118,0x14EEB51674AC2,0x5D7C754D4AD7FFBA,0x29EE9E36557B3E25,0x89A84459244EF533,0xFA934AEA25AB6B03,0xE399D9709FD8A080,0x46B9373F348C9D12,0xFC791149256D,0xE0153A98EAFE1797,0xF0C5F04325829516,0x621B431D1958B148,0xFCFEE863C0B98076,0xEFF294DAC0C81277,0x5A8CF85DE0E74CD1,0x13008692F0DB3,0xD138C6092900F7D5,0x6CF167F3D6252D3F,0x1F38ECD3D89A4145,0x3D4D0CE679208286,0x13FA0174C16DEB38,0x4047F2D26D5F8754,0xD5E2EA545425,0x92AF256C2000E1F6,0xE9B0490F4B817FB2,0xA5CCF281548099DB,0x1A78D3E551B58732,0x4B9AD8CCBB23B839,0xDDC0FED384DED559,0x16E36E71AFC4D,0x1E593AB3913E4319,0x74CCE6911A18CA26,0x1E850CD0682DF4C6,0xC578E6468138ED08,0x184BF9CCDCB44609,0xA75F965FD47E2259,0x4E8502B50F41,0x3D21C903135A0D00,0x432DBE75A7B781C3,0xAB72FE06290507E4,0x8E874ED23C406307,0x474BF58053A709C6,0x1F0CE2843BE9E31,0x13001EEF4280E,0xDD007EE7296DE436,0xA9534C752958CACE,0x1251D48CD60E03B0,0xC157EA9B658D9461,0xB9638E2BDBF3467F,0xEFCACA4237DA0CDA,0xB5E2CFBE30B2,0x38D36B8053BCC653,0xD85BD7EFE1F32546,0x330B55E91F9A92B2,0x4F322F9BE10520A8,0x41A5EE76D9C45D7C,0x6B520E929A5E5DC3,0x1359C616D0054,0x2A45CB5ED45F7020,0x4D10FEE8E47DC291,0x17CB3ED6F7B60E82,0xF960C3D7AB24CB6,0xA612D8F6B7820208,0x55118A0B2563E3DB,0x147B3C3A0BCB8,0x4C4E62E255B1D9C4,0x7F0762813284DB52,0x3BE9A46C8288943C,0x71FD8B0EAD12878D,0x949EF0FA54D8C003,0x6581DBCFF131B9CF,0x9404B176E372,0xD2498670FEEE637A,0x2C38BD9D877D29D5,0xD4A12CC5572894F1,0x332552DBFA14302C,0xA49BF9798DDEFE4E,0xA17670BC6FD4A3E8,0x1D421F862F9DB,0x64A0FBC4CC5B148E,0x83DE2E19558BC4F0,0xA1BD604D47E07FB2,0xE5C33C89EDB983D5,0x50CEB657B781D4BF,0xEFCE25011D0CE8C3,0xDE80ABBC3BDC,0xCF3315DA9DD95176,0x1B353BE82F0ABE81,0xABC2FE39A18B8E7E,0xF6716D520C15B8D4,0xEE8A9AF5D2D66669,0xBB012036DE1E388,0x209090C2618E7,0xFA36260B659122F0,0xAE7C0A3B27334CA8,0x652C139D4C279113,0x7B1487A9062A4DB7,0x7056E71DDEB4CA2A,0x489A0AAF749271D7,0x39ABF87F7379,0xCAF39CDA9BB8D025,0xF3296314A5DF6890,0x40C4D2816003E57F,0x5222E7820878794,0x4A7ED7B6C55D9A27,0xB70D012705188579,0x1C1F9029973CF,0x5CDE3814F98CC950,0x59A6E97B748CA914,0x2A7F8417CE332B89,0x9F283932EF56D47E,0x14533E5CC2F364FC,0xD1DBF368BBAD9A7A,0xA35BCD967FBE,0x719B7DFEAEDD8444,0xC56DF8C55268BCEA,0x97501039B9862B9B,0x4D93398B9C6BC0E1,0x4A37097CD3C24744,0x3EE9BD5DD619D14F,0xE1C083488637,0xEF980C268FCD2BF4,0xF775207C2FB9209D,0x870296080E155A49,0x2B8F66E690D3D7EA,0x7A04759D0C5D250A,0x563AD45B6921BB43,0xB94B845D879,0xBB271BCE5CFD6A7,0xE99E513F3EE3BCEA,0xD3DDD265C8DAC8D5,0x6E668EF7F5E78B78,0x2661D03D07C7A0B3,0x44E4F91974384EF1,0x14BCBD26B54BA,0xD91A032CBE8C3FC0,0xFB34199A793BFC1C,0x1DD537B266C75081,0x70FEF01D6F992E5D,0x14E8C9EFDBF5EF2,0xD5404CF18A1C8179,0xF9F5F57D3EFF,0xF78235B9E46BB023,0x7A0D0BF10C323E58,0xC890BF06D98D81BF,0x65CBEAC1582E8CAA,0xAC8D9DDC98431BBF,0xE54BACA47F38E2CB,0xE54D89C0B14C,0xAC650D5AFFBBA37A,0x53D457E5636EB44C,0xC00A3AFFF4452601,0x97F7275AEA70B296,0xFA6A80DEB855E394,0xF86A5FF7AE9C7A84,0x2252853BCA147,0x111BD3479C2C2F09,0x2C55B065B7AA31C9,0xF0AB63FDD1626773,0x4CCFF5B67E5708FE,0x6212902F3E0D93F5,0x2E5A391470B31C32,0x12AC647C4617A,0xB71A498BAD4FB1A7,0x6371CC037C9031F3,0xA0779521CCB2883E,0xF9C4126D5BDED025,0xDCA010E41B13808D,0x1A123CBBA47D1E7E,0x60374E7BFC13,0xE3E490D4269BFCE7,0x758CE98C7E069AC4,0xB53F83D6254E64AF,0x477285B896925C04,0x98B6A4565C4F7AB8,0xF1125A1705D0E58,0x1052F0BE42897,0xE6446B056E856F52,0xA989FBB3AF3A6967,0x637FB13000C003AE,0x6778C2AA9291FA9C,0x36339D88CDBE2B05,0x35E0F30BD4329E6E,0x39B25E51ACD,0xB0F5879F5A6AD51A,0xF882189AAFF010F,0x5EB4753424A2E873,0xD7E96198DC29001F,0xCD267581F4508F61,0xD4A238CA8D676C66,0x61A4488CBEDB,0x6B9BC9FEFDA108AC,0x16A62C4ACCF81540,0xEBB7DB22E95B7231,0x4162EE47EBCDCCAB,0x862D5B347DB2A225,0x1525BD1ECB660771,0x98720EF290A,0xBA830E03927851BD,0xC1D69AF357362CDC,0xE4BEA80339642CD4,0x6F83869037EEF91E,0x76A7A2A92BB8503F,0xEE3D3A9EB1DF6C80,0x49AF730BDCC9,0x11CD502D33BBD4D6,0x56D69E4D9D402603,0xE67F0E1788EF06CC,0x622B8E5586880B07,0xFAB293679FB11521,0x40530623F2630E5E,0x1CCAF3DB73EE,0xC970370C7E22CE1A,0x48D3126E3B234889,0x7E76A161C71117E9,0x47A43571753D5D11,0x4D5F937940A2E771,0x1CDD4AAEE6A894B0,0x1FD0CD62C516B,0xCB1E412181A981F3,0x39375CCA89D4BB1,0x25EC0A191489D9B5,0x26675D96E7646790,0x8199B7D472C8BF1C,0xB4392B4201421B1E,0x139BB646A881C,0x314F30B470FE3F0D,0x338BBFDDE03DF2CD,0x9074C6CAA8B38E31,0xA1D8B24E63884E7A,0x69D0659C12603622,0xF88EE87B9408A359,0x1F07F51F2F0E3,0xD9C8FEC6A3F08986,0xED62FE3466666532,0xD489BC7570CDC0F4,0xB4C576AC1DFC0F12,0xABA97F744BD66893,0x3119EDEF1C555547,0x1AD80E5062B9F,0xF1410B36D90C42E6,0xF1CA12750FE91BB5,0x86923CB542F47EE8,0x1BF8CAA57DC0331F,0xF4A42B21A48E030C,0x2A96F22E854F5D12,0x158C7B0120F70,0x22C61711B71545E9,0x58F4A0FC3836BF2B,0x739F1DC00FDEB27,0xDBAFC41CAB11A61F,0xB38DAD8D3BC143A1,0xE335AF36A03ED538,0xB278932FB062,0x7A03CD76ABDCA45F,0x660CBA9822180053,0x6013ED8CDD639CD6,0xD54932B7E5253459,0xACF6DB95730A98E0,0xB76899251C69106B,0x1B676903D0D21,0xAB9D67BB28678B32,0xB1006443FC1776AE,0x96CB331CE40306D,0x89C780ED7109323,0xB80E75366C299FED,0xA153690F255D31EB,0x1DA01EB5EBB0C,0x407A1EE9CB63E0CD,0x1900ACA836590409,0xE7317182154DF3DE,0x10D13D02A93D0162,0x56869C87AB10A333,0xB6B57F2AA55D730F,0x223EE6CAA09BF,0x74EEF204797A8F9A,0x8BA811290B57D7D1,0x2BC288C7E0F5C1BC,0x553D9E8B56F23812,0x51AD3DC6D32D3F25,0x3DA5AFE68C218CCE,0x16B19BD4C7F90,0x2DE9D998EFA8271A,0x2FD7C3D557AE3817,0x7643654D1087B46F,0x2B6FD3F08ACB4B5A,0x55086A459F9C5814,0x69D2EB0F7C8683D6,0xC9233FD4EC,0x53A9E21DAB8757E8,0xF0CC331BC5932303,0x416DC7B691236D59,0x87E23D7314B86C6,0x11D4E8FE8454EC1C,0x5E53AD08B571C6EC,0x1F81EDD57BBAB,0x9AD8980E1941A000,0x7ACCD6BDE9C9FAA,0x973C4088280E3789,0x7775CAC519E9E282,0x3A827A156914D66F,0x8452AD8BB8133E6B,0x2FA57562139,0xB43CE5993D462E6A,0x7D7B472983403C58,0xAE4172DAAFA36F64,0x97EA72401CC730D2,0xDB24C8579B1CB3FE,0xCDBB11F41088C298,0xFA85EF29E0C3,0x20EB1F6A9F75249D,0x5E51810A76D82731,0xDEE25145CCC2A89B,0xAFC9E1E784078AD2,0x4715EDA1EA8D3046,0xE28B0989292CAA49,0xDAF0C2C394FE,0x66BBB3AEEC191F5A,0x3F577CE42DCC6BD7,0xF9A2555F007884B6,0xFEDCC3441943011B,0x2F4E5C339CA22480,0xBE711B951CAB347D,0x7C7C1EAC4BE6,0x6C2816762D25D19A,0x3351AEED1B7C39B5,0x3C430DCB2C6FB19F,0xA92961C495ADBDF7,0x9E155A5A1F85AF92,0xD7905CB8D64DF350,0x18F88F7BC284E,0x91459AA75335EBD5,0xF4CC867AA1DFA218,0x5800E16097D30B93,0x6A93DBCD8DB14D79,0x23B5BE2395DE2095,0xEBB445DC39824CE2,0xB12F965CFAAC,0x59D5D937A555132E,0xD4CD7AA1D3137769,0xFDA68F1203A60472,0x615480DB66A3138C,0xE761526C2B2C0802,0xD16997A470399790,0xE6C8FB813794,0x8B00E9A73C402414,0x52B5615CDB18D204,0xC6640AC763205E01,0xDBA6E3CDA161DA0B,0xF2C39B9AC73132BE,0xA6EAD438BA2513AD,0x18644360FDE28,0xE471196316987C52,0x256FBA204E605154,0xDE545757769A68C1,0x79AA090DCDE27BD5,0x7A679F8B1CF42195,0x9FDF7949981D292C,0x172F6FFCA138E,0x8E8BAA1C588456D3,0xB1179BEDA74CACAF,0x8FFEA7D98003A01B,0x963E4A555649F77A,0xA7C7603370BE579D,0xAF32A161E44658F2,0x29AE42839E98,0xE04BA3A855BF3775,0xAE48D93CC0D78BD1,0xAC9FC2FCC7BB7BB1,0x4474A2D9648C0663,0x8E6BB54CA9608D4,0xA638A5B521CDE7D3,0x15BB61EC1452D,0xF66A2A390432740D,0x24041B587EAAD54D,0xB2ACD7E62A729630,0x4645DF23E43B1A97,0x71A8886D23D5BDA,0x7DCACD2685A0CD99,0x54944A5C471D,0x3EB591EED4C38DBD,0xBE896899E2FF29F0,0x2B9285F034386121,0xB80B784C644DC8C7,0x709D74AA163C61B8,0x8E93A60B109CFB3D,0x143836877A9AA,0xFF4B4D8C06436544,0x67C5A91E807BB0B8,0x7E441C2B09E60C0C,0x21D4193F9D1E7CE5,0xFCAD5BD42BA81717,0xB9A06BBAA4B175F3,0x161AEBC05B405,0x8A6D6B4C72A4474A,0x601E49CD19BC56A3,0x5F935BD26B1321DC,0x107A5863E0B3E610,0xA5BDBE085EC166C0,0x145AA9B6B368F30F,0x9F8B2103AF4E,0x35FDC28021170FB3,0x4E40A386F9CEC5E0,0xE5A7B13F888BC18A,0x9240A82DF72AF9B3,0x4B78ECDE0B9C64BD,0xC4273550D867CEDB,0x22FD7BF3F36D7,0xD14ADFBAB0ADE782,0x6805D009BDF4D45,0x95E66FFC9C36083B,0x451F3597A8DB5BEE,0x79036FFC5D08913D,0x4EC5E1CED6EE5D3C,0x3C9AD96FF78C,0xCFC620C4C77DE808,0xD24AA92E23C4353F,0x720EADA550F5CCA9,0x778569A2A42C8373,0x5EB093E2BD539330,0xD11EC11D95FD2BD1,0x183CAB0BF1833,0x7AC6AEBE8E274CB,0xB62D8A7584D0F2BF,0x8B9EDC49325F89CF,0xE0B92650A064E221,0x603DAA5B3C0D4A55,0x8C7EA2473C230061,0x21812F34FBA85,0xD2820EED2D214F0D,0x4ED249FC2FC3D5E3,0xA6174CB7884AC92,0xA70749FA316D6DAF,0x882058EDA4BB840C,0x709B34EF639C582F,0xC5B991C8C3BF,0xC7C939ACD02D06EA,0xE4BD99D8162E82E5,0x618AC96883D5674C,0xB8C1D815DBD9E789,0x19F33D0F4BB492AD,0xFAF6D9B34140D642,0xB181923289BA,0x19E0408820E5020C,0x508FD352D19F94B7,0x1802203C28999415,0xEAC5EF34258617D9,0xA53E430B28003C5F,0xE40A87D53489C62B,0xE97E2F0D9230,0x689DE2D325BD90A0,0x82D9462CBB120F0B,0xB8288FCDE5A417BB,0xD01CD9A89D95C49B,0x7230D01B545E0226,0xA771CC006784334C,0x74B54AD065B,0x3676198D6C9EA7E9,0x58F87DFA4D614823,0x2DF809FFDE284D4C,0xB2381EDF3AD632B5,0xA9BA06D0DBFC62C7,0x56058EC51BFD727E,0x10AC7B16AC66,0xF2AD3383A52FC68F,0x92D3CEEF037E820C,0x5CB1AD96AE45967C,0xDF459B66331088A9,0xB64D9D41774724D,0xAB0D979B88C1BC29,0xB0532EF5B463,0x3F733ED37D37F919,0xC1228B9C2996F61D,0xFEB083478205F499,0x3BC4DE02434B4A5A,0x39CFBD2A923697F1,0xB19D16C5664267F5,0x232F5AC98FBEF,0x6996A29E45B86620,0xD67F1DFE3CB1C4D5,0x2B03EB293D00E677,0xD4891DB30E100F65,0x6034B75ABC204E1D,0xF6D4A79997DCD11E,0x427662E4D65A,0xF70D1F83046A9AAC,0x9F0F144384273584,0x312A09FE9A5A81D3,0x76734D0D3965F1B1,0xA373D36001C626C5,0x435BC8EB18315747,0x1D42D5FAED57B,0x68C5D308E692B551,0x3F1028AB40C1E7E3,0xFEDB13CCB50514C,0x5AD9BE12FEA70FF3,0xDD86F592FFB26EFE,0x3CE47FF60E3F66AA,0x105F527B0202,0x978AD4596DED992A,0x222AC0E1A6944BB7,0x91A2FC1F4FEDBB5D,0xB30539561F08A7B3,0xF8DABE0E8F5D9694,0xDF57E3681D3563D4,0x8A19AB5B20A0,0x8517BCAA31C78014,0x39E70400CFBC6236,0x5412BA9B4497AA51,0x831AFE58CAC05FD5,0xA6F4EB4A9C89FCB4,0x321A3C6D9DAA7E90,0x113F019228FA0,0x5A96D5BC11424398,0xF9740DF3FDE17F26,0x177611017C68CB0F,0xD3FAB77F513213CC,0x2D89C8EBEB6B981E,0xC84E0E12F7B1C3B7,0x1F409BEE37C79,0xA16DD045F01598BA,0x1B1C47A3FD7A5AEA,0x7EA0800342314C1F,0x805EDCBC4F7B379F,0x48077F966D0ED447,0x1A1B7123C6AC1F89,0x202D202F47F6A,0xA26D12DACA81ECAF,0x8C42F40677D86BE1,0x934AFEA3E509DAFE,0x98672E134C4CE323,0xBBF354DF0A96D5DC,0x9192634BB41FC7F1,0x1E4D253F11BA2,0xDBBF5472CDC573B7,0xD05F79C625764DCF,0x7BAAFFE5C376B184,0x6F31174697C67EF5,0xD415F0BE701692EB,0x5FC75498660A66D2,0x6E4C8DA6C39,0xE3A41CFBE98187BD,0x5F3DF54D46BD71AE,0x7C42BF730B0E7F56,0xEB10E5DD280E3E23,0xB99A27175BE3E2E4,0xC13955C8075B8769,0x8C1AC288D7BC,0x7F44A99726BCE788,0xF16016A706E11CB7,0x71543F576565BA9B,0x13FF661C5E248E7C,0x28B8CB5A7845F0C8,0x599280A3236C95AD,0x1834B6DB5CA8C,0x92CE6C03263E1AF1,0xB793B45B20B635A,0x5768FFE68A242EB0,0x2E8F6437BB7FE37E,0xE498D0C4FA9B873E,0x71637DDF47B5F7AF,0x155A84114E7B0,0xCB3D700C2C70B529,0xD909FA6BA5A2EBD8,0xD70BFB4B44B1809F,0xCD545EF71F01C1AC,0xC5453DB2E14FF047,0x77D2DE789C1C35D8,0x1692BB2BCEB2A,0xA8C0D3FBE5286C63,0xAD995CA0750F828E,0x6F04FFC73F09627C,0x8106DC516BA7D8B1,0x925A19FABE5E7D21,0x473680B57169ACFD,0x1DD4BF88D4F61,0xC747FD91B58C38,0x1D90763306470357,0x690DBD5825238C87,0xA66EC28FD37B2260,0x1C594102CD0594B1,0xE19197ECB40A500B,0x18D5F7C345BA0,0x322169D04839E559,0xE08EC42965C08575,0x3C541D90AAFA89BA,0xB489CD09421D79DE,0xD487813FDB8B10D5,0x7B99C718ED1A60FB,0xA391607BD417,0x46B01FE0C12C4439,0xD68CB66F23BC6052,0x62DE66ACE0DBF298,0x9034E2D0B1639E92,0x78D57510EE99AFCE,0x41FF377E391394D,0xAD5569D73086,0x74E5C68FF8D409C,0xA85A44A437338BC9,0xC11205338858DBDB,0x4705292DA0B7AA1E,0x1056B9429E7D21C3,0x3107824D7AE9182F,0x1BCF327FD1D24,0xCBD743117FBA3BA7,0x7F1275F62774FCFC,0xD799CAA05AA7EDD3,0x611636E61E49D45F,0x791AED983672965A,0xAEC985C7D8EB2C76,0x1FE9D5D7F33C6,0x9ECBF2ACEF9D53AF,0x92F2A50B87DE7B28,0x981DF92CDB1DA2BD,0x7340CA5CC1C95A8B,0xF7A3402469AB2476,0xDAC83BC2A720424E,0x140F93F5F7C8F,0xCEF9534E595D5924,0xD4280A97A21CD844,0xBE4BA8AE7442B8CF,0x35784AFB46A3F8F2,0xBA491E8A2D1FB157,0xFAE4727EC8DAB1C6,0x12AB84182712B,0xD940008B60A7B5D1,0x8E66F87650CA8B07,0xF177480A4AC38AB4,0x15A777FDC1F34ADD,0x3437FF7C45114797,0x448EF6B10949A227,0xAC98EF724C0F,0x2335A28843BB0C8C,0xC70FA0D8F35C726A,0xD318CAB7099B3171,0xD4A8C264021CC7AC,0xFFF84BBAAEA931BF,0x1D6CA1FE80C3EE41,0x1F61BEE9B9D3B,0xCA6FC2C9276E6718,0x2533F9B4F77F856F,0x53B0F548E4DE514F,0x2B0F3EF23DEBEFDC,0x782019AC1CF09276,0x840A03434FDB1367,0x7F54FF35A5FE,0x4429EAA13608ED6C,0x9D813919C2CFB8B9,0xE61BEC5E900E1554,0x9CAF2AFB92512C07,0xD3D006E56F8F5A56,0x94F86B135AC9AACE,0x101B8CAC6313,0x9BA66377710EC368,0x51F467ED9D218A89,0xB748489D9EB1D88A,0xBB2CF0397FC1C669,0x7CA05263C36C8E22,0xB253D89ECB207719,0x4B9FA67F9002,0x5B67A00BCA672788,0xAD9DA18A9ED20D4F,0xE6A035AFCB5E6F41,0x97FE873DDCC9DCBF,0xE31F3EDD1E238225,0x5FCCE7D3F41B4DD6,0x162C82BA5FB24,0xD1317F2479FA3F7F,0xBBC4F7A03D572A6D,0xDD334B07F5A1590D,0x352987AB2EBBAD8C,0x690BC7235FF00978,0x297655F890C05E1C,0x3E7DBCC10A98,0x7952158FA4ABC4D,0xE651DE057D882E42,0x31645885A010B122,0x18C0AD07F7AF84ED,0x5598F85A8AF5A79F,0x7FA6A9ED2F7B67E6,0x3B385C69B9B9,0x22792FF02465707C,0xAAA24A5CCA38773B,0x4EED2F07DF1BEBEE,0xAA29EFC0680222C1,0x45D3E8A4CAB935EC,0x749AE2F6D69D5DD9,0x1976700A8DA2,0xC4ED848B79F74FD6,0xC50843D1D5BC73CE,0x8906D6BC82629C06,0xA49992374B8CB022,0x5B7B2B372A5E2C08,0xBF3C71A4BD7D5204,0xFD220BEA7675,0x4247CF39651D1F12,0xC9B1EB5AE6B49D77,0x3A89ED8F4911DF70,0xF6FEDB82E4A49933,0x93F61F7DAA71DFC7,0xE5FDB11C5E28E92F,0x151FF50433B17,0x780986658F546AF7,0xAC27E5B89E641E82,0xD5F718C4D22B3B61,0xCF55EF2F46804FBD,0x4161753260E0AFAC,0xCBBA21419F318D8,0x7D15494B896E,0x3F8493E7A8F2E8E2,0x5CEA906E421BC420,0x9C579F75B48DAFA5,0x8760E85249544C26,0xD23F02B127322F39,0x7CA6685B2FE74A3E,0xEE302FD78DA0,0xC8CFD2B6DD5AB2C9,0xCD73DC2BD85EC68D,0xF77296B8567E16A2,0xABBA143B17B366EF,0x8AA949A354A36740,0x55BCC2785783E1E5,0x621B4E85156A,0x69B0EA552F7A490B,0x8F6F92D627E7A83C,0x9773A9D7AC893212,0xE79D9369EEACCFA1,0xFADAC84F31D562E1,0xEE15E2A22B3AEC5E,0xD84479DD30F8,0xB2483AD51EAF1DF6,0x74D0ADACD7F65791,0xCBD4DD5C8B9A5638,0xF9B74BCFAF318B66,0xEB1A7BD7103047B5,0xFB8B2C26EB526468,0x109A878151FBE,0xD5F8930669741E27,0xD42996CA09054D80,0x9EFA220F69426557,0x58413140F716ED1F,0xF7FD166E629E8B14,0x372BB27ADECD9FAF,0x696FBB68C776,0x2DB515030C3277DB,0xAF9305AE3D126152,0x1205F01198376E96,0xD5594B3F86F6404A,0x7618F977A5923244,0x40F4B1D315D06EAA,0xA7D77F5FBF9,0x4BF7334FD26020A1,0x6835153D9B52AB89,0xF14F41B2C8F47F34,0x3B63E8B8416AB58,0x53C8D42042DF1C72,0x50B0176CF3551B9E,0x1F6C97A0F8CED,0x44E7170BE981A33,0xEDB0B5DE43615BFB,0xB5BD82049CBCE3F,0xE5E3F3B2F5DC2456,0x59F2A0F85621618D,0x4CF956AF71E16905,0x230C5334E2205,0xFAED045AA7A91A18,0x98F038035F5FD5C4,0x59DA72477DD61AB6,0x711CE8D75E5F57D1,0x31B98573CE21ECCA,0x54D18BD9EBB00491,0xF704A10205CA,0xDF3E4441C7BBD91E,0xAA427987A3C792AB,0x2981789041E39A63,0xABAE04114188C79B,0xBB48FCB317A706A8,0x57AE6CBEA8A98E21,0x45AA45F4C2B2,0x6E9329AE6A18E91A,0xB82A6D227C16B8E1,0xA5C3DDD3C394500C,0xBE5254564619BD76,0xFCDBDC7B060EB03,0xFAAC80075286A1FE,0x7B880066AF7C,0x682E25F0690F9ECE,0xFAAB97BD78648CD9,0xDAF232DA79F448BC,0xBC9CFF3C12FE75F3,0x97239B34E3C1386F,0xF7F4AF4FEB6D9633,0x1C6343CAE0CB7,0x2BD0432CE6C07663,0x4AEDE5B2621F2B28,0xA5EA203D3EAEE18D,0xC670B44E970256BD,0x251A66A4958CC3E5,0xBB028203BC60F71A,0x1998924BFFD8E,0xD25BF0C1E545EE92,0x7D3B8DB6FDCC1F3B,0x68A46FC7F4834F,0x17CABBF9647E2908,0x1852EB038B34A6BC,0xCFB2DC089689E4F1,0x1C3C2309B23C0,0x5919AF22BBDB60BF,0x30785305EAD1A9A4,0x56FE98DAFDF4B974,0xF779B82352EE7393,0xAB3C32D5A8E1FC62,0x833E6D3B7BADEBD8,0xDEE17FA81851,0x9850295AAFECF554,0xB42A40D63FAE01B5,0x5F0A866010FC00A3,0x9B714F12E6AC3C2B,0xABD959078C77AA75,0x722DC6351D15EF1A,0x15FAEF15F485A,0xC0F8B1E9D27D5F44,0xE2407DFB93FA9500,0xDB4E11245994F204,0x708AA2C6C4509AC7,0x89E62248CFB5BDB2,0xDAD9780F8C8BE912,0x1B0DC22F57F80,0x4CF5EC00622DDD4C,0x1368F34E37A46943,0xCE832E7E47C5ED70,0xB06A53BDE05B30BC,0x258349F3D0836501,0x39DA7CE0229638FE,0xD59EDEA62322,0xDB940C74C6D185CE,0xAA0F38610C1BA650,0xF8A851294622A505,0xE42195BF1539F7B8,0xD2B84E9AE4825504,0xA3BE324F1EE22472,0x229B099A97468,0x383D60ABB6B4764A,0x2F3773BF67CC51B7,0x79D231B65A3F4291,0xF131FDEF09803AEA,0x122FCA1BBA2F3541,0xB59E1E2DE70D7FC4,0x7DC5C4FD9CD8,0xF74EB1A87B3986FE,0x72EA78AF21313414,0x2D1B614D7EF4CD44,0xE7C306C10555B9ED,0x589223CCB267A1DB,0x624E22F421EF20CA,0x7436C499F199,0x635FA45713CC9146,0x8690AE1DF2BC66AC,0x596A298A6C3171EF,0x6F1B60D2D8E52100,0xD464C1B44FAEFD87,0xA13E2FBF7D88DBF9,0xC070B5E5F2F0,0x871AC5E3E2CEC727,0xF883050E5618CDEF,0xC34CE77987193D28,0xC72258569B80A9B7,0x879C1340E51BACD7,0x93EFACCFB1CD79A0,0x139FFADFBE3D9,0x82DA4A89F7D5B908,0x7375C55FC1531025,0x19656AE4BBAE5B08,0x3F2ADDB299A56F63,0xA7E49EDBBACF215F,0x654AAACDB7146887,0x20AC88FC283E3,0xC85DA8FE176D305E,0x77CB6D84DF0BFF3F,0xC4011ADEE8175A08,0xE9AA3A36810E5D4D,0x7EA23F0714752048,0xCFAEA8FBADC805A4,0xEEE88C8B04C3,0x4253B0E89882750,0x18B52DAB37DDCA7C,0x73138C99629DD117,0x34166A5C01FEB3D8,0x66377950B22A2872,0x325E011BB4D01C71,0x127EA378FC78E,0xAA6977B5B9EA730E,0x65304D375FD8D7AE,0x807191B726873FB9,0x8B12F68C809247A6,0xF03DDC4833F2BE30,0xE527F7615178EBD1,0x21E622C1D6090,0x55A7129ADA77F149,0xFC6F99A7AE0BB0E9,0xB3AE905BD0BD6C2E,0x6ED5F6BC5340B4F4,0xE919EF90D9D95DA,0x3E9B26FC4F615436,0x1B76368C5B612,0x3630A7D1AC02F244,0x6610C6DAC5431669,0x9A640D8B5E5B048D,0x40202A4FD5D78B00,0xF1E7DE7C0013DC0A,0x8140AAE3EFADC4EF,0xADC5A1215342,0x54445960FF5AAB75,0x48EB7212BE9CA9E2,0xB3735DE56593D725,0xB9DB0678FAFA6DE7,0xAABDB1564F6CEE7B,0xF3DBC01AE07A9FC5,0x1DAF2C3A89724,0xE86D050B50BC6955,0x8673D6B5A882E00,0x751EBE97359B7F35,0xD83437A5CD119339,0x52A581A3B86CF2A5,0x29F4D58CD410F170,0x49AAA15D26AF,0x2D60E27C2DF331D5,0x395E70DD664A32BD,0x18FEB032C47D1256,0x8A9A077D966A66BB,0xA29AF608B2FB02F0,0x8FF8DB51B7F5CA18,0x4CDE342ED549,0x952FFD5C832669EF,0xCF774D8D4DD033F0,0xB9671CE50468DCA2,0xAE16FA72FE3CD08B,0x4DD9F593067ABCDD,0x46BD5905098C810F,0x7D455308F204,0x76D3F219E6169BC2,0xF71CB88D26350C66,0xC8BA0C039EBC659F,0x735985E8CA00D290,0xE37D9DA2EA561A43,0x1353EB2145264630,0x1B463BF18521E,0xF064725EED9A7087,0x40B438DA6051DB29,0x13D2491FBE3330B6,0x8E52BA1398C2D695,0x37FC499F4ADB4C8E,0x248FF72F27749CA3,0x14E0D62EDD309,0x63BE3A412B5D12,0x601BB9545DA86433,0xD5307F699D6453F,0x8F97813A4D4E5564,0x88D4B75390B174D3,0x6F2564C2063A89FA,0x8A821FA61FD1,0x683B632153C6C3A7,0xE8F0ABA49115275F,0x595BD1419E2A49C3,0x9719D2E6DC959FF3,0xCB4A3748D3A99745,0x82BDF0347CEFA65B,0x17A4AA693DB19,0xBF04C478FA4D76B0,0x7BFE551C27F4BAE5,0x3047865050BA8C21,0x3CDF6868389D486C,0x58E83263E14B7F12,0x7A7B95564DAEEBC4,0xA8E065838B98,0xEB8BFA92F6C564A4,0x7DB0032AB4B8AF13,0xEF59463AB5E388DB,0x6AFBC48DBF953258,0xA8A5312137F9E3F,0x491F7957B54F5F9D,0x9410C6FC5BFC,0xF7CB1062873FA5C5,0x17A17DEEF45B0A2A,0x4B42CE552FA07F5C,0x223BAC106B69F336,0x15B1DA0EBD185DEB,0xEB18832D0B385D98,0xF084DE286BF3,0xA52AEFA2A801D8A9,0xD2258FC27C31253A,0xA91977CD27AB51BB,0x15C9577E67FA623C,0x6F8561270F3A108F,0xB1F40B21BFB57BD0,0x1C923394909A9,0xDE6D1170F68D1753,0x62DB796FF24E5A84,0x860500D0FE2C57D5,0x371435874B5E5FC2,0x640E5835580DBDB3,0xEC8EAE76E0F1D9AC,0x141275DB3F6A2,0x20D47A2CF5B59F6F,0x7513D8ED9F485F51,0xB2B71D681339714B,0x4BF615D642C1D7A9,0xE55C671DAFF0D0DC,0xF3D46F1E942F63E1,0x111936874693B,0x7A1FEADE41F4FBC8,0x1F307B96304A40D0,0x2C7026C9BB216EC5,0x18ED4521AB8ADE09,0x599906A1E8B64EC4,0x4430B68075753ABD,0x13FF2B73D7A96,0xE81CAC44BE18B72F,0x51B7B65FD2A4D846,0xC0294A617AD87EA5,0xF6D5ADCD8A996E7,0x2C4E13821EAD2A59,0x5325F984382063F7,0x14DB8660B0167,0x8D1D825A5D3D866D,0xD2520326848CBEEA,0x555DFFD75ED4FD37,0x498ED1700DBDB0B8,0xDC06E2B0B57D1B06,0xBEBB224B2FB975BD,0xC0A4DD8A8558,0x66FBE984C2C6A161,0x7A91B3C3BD398A65,0xA96925CA957813A4,0x1D8E769298672893,0xC10CE8BF1C5AE52A,0x4A8A990D7BDDB92D,0x63CAB50A4B7,0x277486519BFDFE8A,0x93E0A713168F865,0x212D6E1484C77E5A,0x233750831FBDB32E,0x2A74B93344E0D76F,0xD46B4D5755FB0AA6,0x2D1133C6821,0xFD017B4D00351139,0xEACDA4F8CC8ACBF,0xD63A289F25A0773C,0x8271C326A55E364E,0x9AED312DCACF4139,0xB5688D216CD952D0,0x211178C25787A,0xD9405E8880A2EB44,0x5340BB31ECAC975E,0x232E8A3FF61D73DE,0xFC41EE2EDAA7E8C8,0x27E28EA70E0B2693,0xA3A66E5E34D5E070,0x242BB806D8C5,0xC10145F777952D2F,0x6252F7597346B7D0,0xBDD64769F9DFA224,0xFA8417231FAAF7FE,0xDF451C82C47F5880,0xA257E74C58F4593F,0xE1D34D3036BE,0x1B9EDF9B892F7C54,0xF1A289967B629D8,0xD452191BF073252C,0xB6D66863DB9174D6,0x627FE69E99E4325,0xFCC7292E25027917,0x7EC08A5F5DD2,0x8E32F5695FE42B61,0x80DBD0476DB4EF87,0x4DF9C3C88D2E2733,0x34F414C97BE3BD3D,0x352DD8818639CE25,0x9D7FCB8A30F5B895,0x557BEE28E965,0x24479ECBF919CCD6,0xE2E55513494ACC07,0xD72072FD8D482E26,0x5CB0250720DF1E1,0x488A1B9AEB6D667,0xE3161FB69EC12D84,0x22C50D305EE3C,0x9ED4BF50183A09A2,0x84A5AC08551E2BCF,0xD8D46452EF15EB4A,0x804CE86BA79D2680,0x8094D37367A4F659,0x65E4032164A92AC,0xE3FCF2D5DC89,0x998C4416B4CDCE74,0xCC8C7FB1A5BCE072,0xDD02D529159FC3CF,0x79243D97CBF3ED6A,0x5CADDF2C4C48687D,0xCE54ADA1651E9366,0x1702343B19A40,0x7D0260E0AE11DDA3,0x2C06C997E59B4134,0xD194E028CDAF04BF,0x484F5114D18506A8,0x65452147F4C9581F,0xCCBC443EE98BB3D,0x206A41B99E1D0,0xA081D9C4B259BB0A,0x5C6F1B2BC71245A,0x58EFC61BAF0859AD,0xF2526D1F3500137A,0x810B8EA3FAE12D72,0xE06A58A8EFD7DEC,0x1001CEC9BA0B8,0x23E93E6C953DAC43,0xF3F5CC11B613293F,0x5334FC18BD66A4F3,0xA369ECCD9FD66A3D,0x74983EBD662EFB15,0x9C0E854B8ED52E65,0xAA55672DB2AC,0x6D17DD0E36C635FA,0x3582CAA1E084AD9,0x9A29D65A79D6F70F,0xB64931C36A77496B,0xC506A54438B6DF93,0x402F3A2C67F0357A,0x14BE8B121E809,0x15B5777D80CF875,0xB430AB3B8E06CCB4,0xA30C7D371DF6ADC6,0x6B2190987A27B152,0x3C2C8AD91459EAF5,0x858B9477782AE2DB,0x11DEB0CA281C0,0x522E2D51AE951DCE,0x8B56D118B43F419F,0x3E37D7E9BB1AD1EB,0xE72D693CA698F2EF,0x72CEC13AD33E6535,0x58688A7B67245523,0x847C33634AE1,0x364000D4D4E7001,0xD1468C7A0336B225,0x269119303C8EDD45,0xE03096075F011BDF,0x89D78CA210D38B4A,0xA4640CC54DD85CE9,0x2232C24CE7099,0x2E148A6893A3377F,0x85FF66B19FC30926,0xDF8D30EA233AC9E7,0x8B68A0FFD9003CE4,0x84CC5854D5F81C7E,0x5EE273D8F3FC1D99,0x22818E157686F,0xE5DB26D183ECB3C8,0x9B25BC9377F48B19,0xF55C46ADF5B9DD58,0x2F384E23EF962CBB,0xD952CC9D9BC6EAF2,0x1DEA899A669AE29,0x128E597E73A7F,0xEAEC517F34486095,0x91E31162693384E7,0xAFB4C7CC9E5E75BC,0x7E3BD8A4C5D8CB91,0x73384C6B61630C4F,0x73A490C2A240C6E3,0x18C23A7BF5A7C,0x520D5763C4B18AB9,0x5649F3C4F58B22E9,0x726ED35C14619AEA,0x75858C5604802933,0x8EBEF6FC8514DABA,0xEAF0AF812C7C7E0E,0x30C97406B409,0x4515E53B1C86875,0xB99D8ED4395F553A,0xF8C1D45FD5BBCB32,0x72B5B3D162B8FF33,0x97EF897AE189AEBA,0xE1B3AC57274F04E4,0x6E373EED4E4C,0xE9C4A032EF0EEDC6,0x2F8556A4BA269A66,0x2DC1202BE73ED028,0xB1AB548ACC01BC2A,0x6B9C9E0CDBA03C06,0x1D41F3016B3C6A94,0x925C716F4096,0xA3D3C6887C4D3B50,0x82DD63EC6F8D6AD6,0x9C1030E36EC27891,0x82DE48DA284A6971,0x953D436E92D0B90E,0x70B48A25662DD99B,0xFE24850202D1,0xD41FF4ED3E1F980A,0xE662593FABE819A,0xF1B6B51E0D789697,0x1B455D09AA448D2B,0xA5D2BC8F31F67E32,0x26256100CEF5A41B,0x1E7A5FCF5879,0x5031776B1B9293C6,0xBA3E795A1148165D,0x59C1817B3D973F97,0x7A1E125549AA30AE,0xD01F3948B68BBA88,0xE180D22877250A09,0x1FD0E6793538E,0xA45E7CF4ABAA2CFA,0xF7B9D6BD82D103E3,0xDCC36B28CC5AA224,0x5D7F4B012A8AAB1B,0x1952D93262BAB62E,0x362604A672DF189,0x21FA318616606,0xBEB656B25DCA9C22,0x6545BEC3CA212DE6,0x669BA147AD23BD88,0x475FE8FDA073DFC1,0x7BB8001812419FB6,0x6F86C73105A770AB,0x6B94DB6FB92A,0x8526A1A1092AD16,0xEA491C6522B8EF68,0x30FAA7E486EDB08E,0x4B5D2FA02254BA40,0x3CD74AE0A1867200,0x7821567E58968EFD,0x31265A358695,0xE4F6028AFDE0CF14,0x22A334DD59B0FA21,0x855D1C7987AFA5AB,0xD4FAFAEB81AED276,0x9DA77E66B3C9FC9C,0x8D75D654240D941C,0x15253D8AAE0B9,0x34CA14BE825B9889,0x4DA052355D87540A,0xD38990A96DBFB0A7,0xE307BC0474701C33,0x82D14C0AACED3AC8,0x3BE4238B937ABE5B,0x6283016EFA74,0x66CD22B5AEB13846,0x2ADD1658BBF4022C,0x473D392F7D88583,0x183E57002020EF05,0x4996D20FB031BAB7,0xDE5B1733D945F430,0xC3767432F1E8,0x6C6FE341ECA81B95,0xFF8D1C8E58CD70E3,0xD745E2289AE73B3F,0x3D1AA99D220FC798,0xF751DCBEC73C4B5A,0x3D23402D90770566,0x19480E46DA256,0x9DB40B6E6D676311,0x864C569011F224B6,0xAD6D4E9DD1E2573E,0x344CD73B3327C72B,0xBAA477B6394CFCCF,0xB3E8A98255C473B6,0x14B4185A0A4E6,0xD0A89978C493AB56,0x66E788CAC51D7635,0xDB551D179D7AB76,0xDCCEA0E8314CF5DD,0xDAF9C9A6434D0D4D,0x345C3BD317ED429B,0x1006B8A3B38B3,0x23CAB87569C4A7DE,0xDD464417E560E457,0xE8DFA358CD1182F7,0xE741B0EE9AEB2656,0xEF71EADFF7C71672,0x987902CAA437C6A1,0x1B930088EF38C,0xC19B46143828341D,0xBCA58ABE9B58034D,0x3CAFE3696387BF25,0xFBDFF875824E3E96,0x8FB4FCB050CC464E,0x5EC3AF169BF60EFE,0x54B0A710E1D4,0x8791C5B6AA3938B7,0xC134A67484228077,0x462482CBD489D474,0xE732CC166A44B4F6,0xF00272A42EAFCA06,0x99E535FA723F4F96,0xA7CF67C7132,0x39280D345C13D4B9,0xBE5A85978FE52EA5,0x2A890212980070F3,0x94B859385073B111,0xC53D2936B5A6A5EA,0x2A7935A75668DA96,0x16B77798A40CF,0x8B6D71CC9AB823C6,0x171AEB39ECAF1773,0x972DAE6D76B2B3E4,0xDEC21A33DA07E101,0x7D102C3551D153FF,0x3F1E19F96E77AD46,0x81E348E4D5B0,0x86FF926A6FFE038E,0x39F5BF79CAA4BA03,0x607AE5BE3C94FB4D,0x4920F6AB3CBCFD6D,0x8AD06F08246B6512,0xAC96FEF1F41E5AF3,0x6367183B8884,0x66B2D71D412B8B90,0x75FB9B35A96FF364,0x4A41E7D803C1EA70,0xB442FDA50F47464A,0x95CB655B74F4EFC,0xE2762804A48B199C,0xAE8FB19F08BA,0xBBBC57318C40EE38,0x3B2F9ABA67F1715E,0x3350110AF668FEB9,0xCF66D80E08FEF3C0,0x6F75CE363116C8BD,0xA4EE479618C88353,0x2B4A5ED89A7C,0xC2115D6DD296CED6,0xAEAC2F46CDA54B3A,0xEBA88C89DAC25DDA,0x9F369123723F8A15,0xA1E020038D255BE5,0x12C286378DDCA7E6,0x13D60C41402BA,0x84F486AD6C0C821A,0x94BE225ED8C87391,0xA6461171A40E6CD9,0x6983336F2DFBB172,0x29135F9AF8180BBF,0x31AAB42A9012EDE0,0x91535C067EE0,0xCF366D9687191537,0xDA2F3C118F50411D,0x8A8EB896B37DFB35,0xCE26075F12F63CEB,0x667128E4E6924100,0x8D6C7CDEB8BBE655,0x7284D6C47B42,0xFE31E7B144FACB0D,0x51C05D4E0F9FA631,0xD475499851D677C6,0x75C4A8A6DCA30D75,0x777936E42C24FE01,0x7F7AF5EBF7B1D377,0x1C0A58C1969D6,0xB7484DC040DA8F83,0xB75ACFE34B3A7AA8,0x1956332485AE28B0,0x2B99A37A425E6421,0xE1DC5F9CF8F7044E,0x74C79C44E6E159BE,0x18C9681AB49E,0x5225BD883BED7D2D,0x20C2679D07DE313D,0x835E350129CEB477,0x297F2045416B3A55,0x29A316EF622C7E02,0x8641CBC3FE6C344B,0x77761E14B923,0xA651B079193847EC,0x23B25278AB30EAD3,0xBCBB1DB498AD6C46,0xBC3745046F374A31,0x22DAF18E411CD83D,0x3F85D330369B32B9,0x1505F17631AB8,0xFD7B718ECE04AB47,0x52386C463A1FA49,0x3D2216726C6110E8,0x320B26E3B6B03D7,0x7A797C0273085473,0x8CB4061AEAD848CD,0xE90FEBA7D12F,0x46B47C318ECDC97D,0xD322DD012F880529,0xD46B51F5A2D2AA87,0x2F7AE7F1A319B8B1,0x3876D3E943F00B2B,0xC0FEE2A49AAC0C16,0x18D8E6CC1C0DC,0xE5D4FB78856DC1FB,0x5206264F7F524A20,0x2D6F72D23CCBEE76,0xFC79BBFDC870FF48,0xB5149C68BE78D6AC,0x277A951D716BA172,0xEEF78C6477D5,0xFD7C1004DF5E48D1,0x1E23DAD127BB0502,0x71346BC587CADD07,0x22C67404DEA995C4,0x873A517E769EC5BB,0x4C171F67DCDD078C,0x19147DC1C1227,0x27A80083D14317E1,0x240A41A7E1BD2A37,0x79D1FCD075407399,0x714CAB4801C29F6C,0xC7A6D14174FF641D,0x7D9EC7F8FA86048A,0x140527E30E434,0xA76B8CA804FAC010,0xB8C79ABE37CC7560,0x12504797490065A7,0x6C925098719F22B8,0x14A4CB612968B59E,0xD5DD65E053033B9C,0x1900A426225E3,0xCCA9FC9D8E0C0DB9,0xC93E6DC98FBACD81,0x6225EEE44BD00F33,0x46328F3266D906EF,0x50C78CAF24E42316,0x58F20D684C36DB12,0x105DE98D22064,0x4180170BF187DFD1,0x3D20F7A2E5731F0C,0x2B29B79096A9A245,0x442A49AC773CBC4F,0x8C9CF9859B34BF82,0x501AFEA57BADEA05,0x129D4ACA0106,0x178F98476D8AB26A,0x9DC62E679AA869E3,0xDECD81B5A2FF4287,0x8DD6832E6B6EF36C,0x8533453283620828,0x6A84A82FC6B3D1B5,0x197BAD1C5D477,0xF1E82AFFC4C50508,0xAE354D7BD7DC6933,0x2A8DE17C679540CF,0x8E26B095C6D8A9DE,0x8D0A113A7A6F427E,0xD8CD72938C04510D,0xA64A81231004,0x7432DC877C25109,0xE0CC7C5A034FB449,0xE4367FF97286C8C1,0x25FDE79DF2E100,0x62FA18932EFEBDC2,0xBAE28B811516F4EB,0x13C7EF76D7BE8,0xC7275B149DD1A5FE,0x6B24F1CC50DF7972,0xE286B78322F22C81,0x9946C35A7168E865,0xEE8E4BC759E7873F,0x8A64BEFB0A7F9A20,0x1CB18AA039C25,0x6D28336C58F73E30,0x725B50C04460EDC5,0x5CF888D713FCC1ED,0x2E54BE3E35D2DB2C,0x494295D0E946CFB8,0x7210AB8A084C7982,0x42FD9B30D5F6,0xC09DFD096ECD7E8A,0x43597BF397F85CE4,0x1F3257EEF2E21E01,0xEB5B80C76CFFAF95,0xEA381C7104064E67,0xC184A403A5E61FD7,0x60186EC4DDE5,0x954F21E01BF9998D,0x5F8814C46D5A94D2,0xC3657DE3C0ACD7D3,0xFDD94C076A58312A,0x79BC2684753DBDEB,0x11E51A878D1223BE,0x2066798641AEE,0xAAC0A24B40C3F4E1,0x5E2C91F223972BA9,0x398D99D44BD2529D,0x5F741BFC3676D886,0xA232F12DD967CA41,0x91D8DF7B73C02EC0,0x5A7384A12DBE,0xEF281AFB4A39CBFE,0x4A981793D090C953,0x38C48EC5F72F47F6,0x3913D4E8DF5F3B47,0x99BBFCD1ECD2F722,0x6B1BE4AF21A60A80,0x9D12C5CC4145,0xEDA5D34BD1AAE0DA,0xAA56DE87CB3C6ECB,0x44D78A51DD85F9AC,0x807A46E94372867B,0xB81452500ACA79AC,0x9B390708D4F3BBBB,0x21366D30116E5,0x64184C2240999BBF,0x4565AB187EA5B623,0xFE53D23A14B3DC65,0x69C462B66449CB92,0xB55F63716A7F4B66,0xAA313D1E702C47FA,0x20B4B0C6D5E06,0x505B06E0F48374F3,0x6074667B9B283405,0x7719C8419E6EF341,0x61B9742BD7042596,0xA47F6A4AE0883F71,0xEC213D98DE13DF22,0x11A989DF3D244,0xAC66F85AE7158131,0x1751DC219609C485,0x9815D2EDC39B7041,0xEA4CA2F167EA9429,0xAD283327405B8787,0x9DFB28B3D05297AB,0x193D00B345EDA,0x35BD17F4BA5A4BA7,0x70E977AFB4C9F0FF,0x7BE7200057B27F5C,0xF8A2AE784251FA74,0xF9F05C1A3CD11318,0xEFC59D8DA48A5E3C,0x5111E582F972,0xBCBC2549EC68FCDE,0x823DC23B9928FAAE,0xCA67DFB037298E6F,0xAC70AF2E2F59CB0F,0xF303D3F9FEA1DE0F,0x1FEF6966CEC1C6D8,0xA45351E4C3BB,0x28213AEDB766D70A,0xBC56B59ECADB8DB6,0x345EAD5E9CB094E0,0x701D53B4419D8B16,0x726F4BA122BDB4F2,0x7F6277F789489550,0x147F55DD5F077,0xB5DABB19882D1CA6,0xD0653EBBA634DE3A,0x680D1248F27389DD,0xDB6C05E99BFA018E,0xBF3F75DB70F8995F,0x95C9F02E20BAE066,0x11D27A38FAC46,0x9068144FAC93B614,0x6D1DD96E8210110A,0x1062BA34C040EB33,0xADDDC5FAD8EC3AF7,0xC40E23557263EDDE,0x7D3D921FA93F38F7,0x1CBDDCDFD9568,0xFB0B041651094E3A,0xB95E746D698367C9,0xCCD8C9E511672B3D,0x6DAF7DD38741973B,0xEDA96C3D57AC590D,0x42CD602A9DF63B5E,0x1273C95C0B467,0xCBB68BABE871D287,0x34169DEEA7B7E71E,0x958E889C3212FCE5,0xC7ECE44F146520BA,0x5B30C9E65DC31CC8,0x9A789A950510C3DF,0x1B0BCC1343BC0,0x8DC76D5585B4468F,0xB1D1CAEC523E4056,0x90108679BA558D28,0xADBD10025EBEFCA9,0x4824559AF61592C7,0x19E7BD08BBA1E056,0x6E4E5425EC76,0xE5780B696D0047BC,0x7F2326E9BDDDF8A7,0x9603717332134044,0x7D4EBBB5D346896,0x532D2176AE4DC1D5,0x44A1882B27F3F864,0x1884B6C3617F0,0x6869618B986AB9E3,0x6AF03C28713E04B4,0x34E132B80B8B3F37,0x303E4DBD03F8A262,0xF1AC57DB3B80D94B,0xA1593F83270A71F,0x20FD628C05C6,0x9A89E43CA0689CBD,0x640233F6E9D471D2,0x2173B938FAF79BA9,0x7AEBB22E721876A5,0xB504AB80EB2ECAB9,0xAA02AC5DE27F18DD,0xA092B9F8841E,0x8914B44FA242FD87,0xA305924D88CCA294,0x39D5C0CCD696D33F,0xFDF020F1EDAEFB71,0x464A508097E3D6AC,0x3D9B72CD97694F22,0x17FA978604D14,0xAAD89605EF6DAF0C,0xFF9BE7486AF576A4,0x49C7767ECD4B434F,0xAB32040DAD7C4CF1,0x1080A5244019ACAE,0xE49CF12D166A85BD,0x36E406F47997,0xF86E6720378A8D76,0xB6C618C98EB2D0FD,0x595B15ECFDE582CD,0x9FCA16F68D4BBEAC,0x205478615AEB07E8,0xE59A63B3A2976B2E,0x2270C4E8162E3,0x7FB67BAE054FCDA,0xD0760527AB909AE7,0x9102FD59C83A6623,0xFE8E6CE603F8A9EB,0x2C33A5D9863F6C40,0x924DD334D9E84CC7,0xB245B8D01AE0,0xAA53791584D728A6,0xCDF4EFD7CAA0FEB5,0xE8A8CE859D2A8BC1,0xCCD75E9B50B9ECAF,0x2795BCE93EF7A2A0,0xFDC933DC920D00A6,0x139649A3F2F16,0x23976930DF0492DF,0x7212140DAFAF2E04,0xDF00140D54E2F3B4,0x69C129B57C8FF828,0xF6D747AEC2296BE8,0x9E29B46DB988FA8E,0x6DEBE8C3D340,0xD3CB76800E25D461,0x162298699ADE5D6D,0xA6D9603BCD217A46,0x284E370F81BB51C9,0x5D059DCA27177796,0xB0D013EC70CF92A,0x14A875791DC1D,0x46AF50ED4241771B,0xD86E0239DBC02426,0x42829678B9424ECD,0xB063D73874D7BEF4,0x8D3A6DD88F76CE4C,0xE70EBD0B8BF92771,0x535B4421407F,0x4C1281082A366ADD,0x3C34A9F83DD489C7,0x85D957A854606CE1,0x595663BB5FA0291D,0x81FFCBFB907427A5,0x50851421B9C6A1E7,0x5667AA50F0A5,0x500A55AB054C1891,0x2175302348E2854D,0xFB3756DD9770184C,0x74E416B857D53D33,0xD7FED4D83D845CE9,0xAB394F2E13D3AFB2,0x1E016B121CF38,0x8CCBA32940728311,0x1A3267B6810A3280,0xE9E4CD9D0EDC9F8E,0x536F22D5F05B9D22,0x1B482BE79EAC3837,0x4108189BA9A1A52E,0x11405EA3C013B,0x32C14C775D1D7F17,0x3A0F1AA1FCF4F7CB,0xF2018CF4B9AA522E,0x12AE8F660B59E7A8,0xA5625F52494198B9,0x3D362EAE0E84B0A3,0x16C708BEF4B2C,0x6F0AD69D242E212,0x9165D0186665A5AF,0x9D3538FCBFD479B0,0x17C924EC609C82D1,0xD48C7937407F3B8B,0xBD041F9AAF1016E1,0x2AF4A06F96CC,0xF5B77F2B8195AB19,0x123757CFF7E1E8CE,0xD63CB3A7CE3AA60B,0x899536E72DEF38D1,0x433C8B233FEA7FC9,0x48B4F7176794AB2F,0x10971C9E957CC,0xB41F3B1D4A1D93C2,0x868B03C56348689C,0x640EB189E020E0D,0x99ACFCA66B41985F,0xB34CE6DE2A08D9A,0xAC558D3667A632D3,0x34B40C82106A,0xFB6F2A72FCBE5959,0xAF320D1E294C5937,0xA25480638757B201,0xFFF60C9B283FEA81,0xFFDA6F6DA1980364,0x8EE8352D9527B4FB,0x62C13AFBF9C0,0xB8C8A6098B2DE393,0x83F2CED7AB8E0CD6,0x6BBD0280A51F217D,0x54301B4940B4A922,0xE9F191EC15E11A5E,0x653C7B0B0CFFCFE9,0xFD29D443089E,0xB9D79BE5D9AFF3A1,0x25B9251C88EB326D,0x83FAFD08B9F46F7E,0xEFF73AAC816458D7,0xC5FE567323AB9CBB,0xE9D215BBF220F7DE,0x119C19DD2A164,0x7BE48B8B2A004A55,0x769A26A3FA3AF6A2,0xDB4DE9C7A1229B78,0x8C8E28593579581D,0xB06837AD5EA3B976,0x9B6B75024B1C0D5C,0x4D2777704721,0x59EE194CE5CF8A3B,0x6F845AA691287777,0xE545DA2729ECACD8,0x8375AAC90E8908E3,0x26E452139449CC5D,0xE9529FD58435A664,0x159848F135BCB,0x7DB7CAA86B756FA9,0xE80E08AB7EFCDDAA,0x50B3774962C96339,0x953DB3965B3341A2,0x2D3C3F3B6A7657E9,0xA55E602F8C3C227C,0xB5D218DF3DBA,0x9B95E5058891CB51,0xCEB2394819878E9F,0x9F99C0BB63981570,0x3138DCE48938AC67,0x9FB0B27DB7CB1F7D,0x565685F888D52E75,0x1CB224DEB0BA5,0x33CDA9A15A4632B8,0x51E771FB02BFD0B2,0xD57CAD02A947994F,0xB9047EEF3470430A,0x12B6DB69C544FA87,0x5CAEBD388562E302,0x2271B6607BD9A,0xE4D60D23944C84BE,0xF5C54A183138D642,0x5AAE7624775DA793,0x8B36666716372609,0xDD43BFFA76CDACBF,0xCCC2F8DCBB3FB105,0x1CD44A2EE827D,0x4CF7EBE4E9EA3DE3,0x6A050C60C5657BD5,0x632920B8D3BDE5A1,0xF4FDEFE3DBF59A0B,0x13C715641257C5A4,0xB7C2EFA8380250A6,0x231FA490721A1,0x1FF96CECC1BCD6C,0x152A42098844B1CA,0xD7AA606C69533617,0xE6045ACD5018F279,0xCC04674F36E84247,0x77CD66AA9C6E7E6C,0xD4A16915EC7D,0x1929F14CE050A996,0xCD9B591CBCDE3644,0xC13075B43DB9D607,0x75FC99DFBB0CF922,0xB44F9AE8B8B479C3,0x84B5B8D5B857CCBC,0x1C643F2505049,0x7374681C9FAABAFA,0x2ABFA15FB82B854A,0xE084FBACE5D54483,0x9B88C3D7E7E634F8,0x1904E2F29A7AC1FC,0x7CB89FBC66BCE333,0x17699E62D31CA,0xE560358D51AA99A5,0x5FF39E30255C4DD7,0xFA9CF4CFC4B60F4B,0x9212D22E6F17C6F0,0x8D72E2C99DB0E6EE,0x21ED79954BCDD788,0xF90C6B97F33B,0xEF21E425DF5A59C0,0xA655B17D41F13DBC,0x29347808D5862E6B,0x4F427EEDD7CDC84E,0x8E4F4510984E9361,0x2A3A9E046C3336D8,0xE903F930ACB5,0x785BADBDDEBE2C63,0xC93EBF8FF9CFFB48,0x82E4BD717353896F,0x44E6DCD789E0C98D,0xACF924307DC30C2D,0xC7A062AD7DC49921,0xC83FED7D9EB1,0x625334E81E1EAEC3,0x58E530990AC07825,0xC28A5106F4C72437,0xAF5CE43893912685,0x915686660CBE1A9D,0xA19A35CC6DA3062E,0x1FE7117DEFB04,0x2730062F37218F92,0xFB8B7DD32A666354,0xE6ACA64AB9FD390B,0xF2876F4AC7134EAF,0xA81D2730A964FF18,0x4FA14D823DC16E89,0x133DEB6AC0CE4,0x948BE6FD77274D8A,0x210600A750854B70,0x7AA2A56D9DFBB530,0x82FD6C3E866EE956,0x2BFC653FC1CFB9CC,0x6682E8ACD34E515A,0x5B5FF5F4F4DB,0x17E8B98B9048163F,0x4343FA0A89CCE6A8,0xF92779B99B7B22A7,0xD63A3E1C35F210AE,0x7685F722683FB64A,0x4100249DD94CF82E,0xF2E73F768C36,0xB58B332CAB916F88,0x201122344D934319,0x148034C9905EC039,0x6E2883936216ED99,0xF0981D63EC7AB680,0x912592C2566286DA,0x13A8ECAAE8640,0x3018179E6DE7CA53,0x1ABE6A61A32CC61A,0x1327D80FEDB7174D,0x9270D61C4AEB50D6,0xCABA1C716DE19107,0x984F75C458FD8777,0x223AECF03E547,0xFDCE38DF5776A74A,0xF36A3A377F7761A4,0x57550DA64F48DDD9,0xFB58AE0583079826,0x758E9BFA748F4F9B,0xF4169799377EEC88,0x283FE4403AF7,0x13FA6DD518C2E57C,0x1C958342AF2332E,0xE94900CEBF4596CB,0x41357DFFA36C0443,0x77B1A85390754E45,0x3174C1456242293E,0x1BBDDA2ED65F4,0xE370EFE3C9BAE341,0x9262FE52C1A3F9C7,0xBDCADC838A2F681D,0xA4F7BE580B6B6DFC,0xA638741A1415AA4D,0xDF4110D0423D487C,0x18DDA57DFBADC,0xE9BAF3DD27456D2A,0xA7C5B99D240B1CF,0x1A78CF6E54CF5D95,0xA11CC6AB86A19F0D,0xDACCE7B152877055,0x6B595311E89A170C,0x60FDB8A988BD,0x8686ED997CD1DA19,0x5FBB786764D3C29E,0x63479A2A942829F1,0xD72BFE4181B771A0,0x11D9E45A3CE5A2BB,0x8EDE34CC3B8644E3,0x16A2319FEF3DA,0x54DBC7860169D1AC,0x9D06546CC79782BA,0x111A7B21AFD7802B,0x54164B1B88E13F4A,0xE351B8DE675C35,0x8525FFD6BD6BDC9E,0x1B4E63EBF811B,0x901030CD5BCE363E,0xBBA0CC9C88FDA9D7,0x852756730F9DDC75,0x52DEC1D63C482424,0x3248E1B719F2923B,0x1B9B6C13CEECB776,0x2C5B2998F024,0x7459BD7BB172F50E,0xE6C94D8367A1543,0x9D735221343C331F,0x25EAA4E2E3265D64,0xBD97AD262D650F40,0x882A50EDE7A47237,0xE82B2DB80C28,0x8D5ECAD5F0CB875F,0x8D378B4F8A83785C,0x963701055C701E6,0x669E3D346EA6F7FA,0xE8B20EC1716591B3,0x21FBDC5A0409649D,0x1321E4ECA0CF8,0x952B76180D98F4E3,0xEB2CE804640A0DF9,0x29023A1E2BBD1E0E,0xE8D62DA7B41DED14,0xA5005901DD91E7D4,0xB374E36707711661,0x1DFBAB9CF2247,0x482342E3BC9BE557,0x63B5942D7B81B4A9,0xBEC0173690C2E281,0x43B178DBDCFB7C61,0xA3EAD90600551D42,0x5084ECADC636CFE,0x1A9D2E844D7EA,0x664C83B31850E1E,0x74535ECB6153CBBA,0xB8CC07C4AA469864,0xEC37F6AD48E16271,0xB11E8CE2F865A1A7,0x2C5A3A1F2E766C27,0x170BEF101DF5E,0x15A9978F10B5F149,0x20E6B8E913DC56A7,0xE2D92AADD06AD899,0x65457E827E4B45F9,0xBEC45EE9002EECAD,0x42A30CFE21E10D12,0xC6EA7BCE5DE,0x80A2247D02D9EDD,0x353BF45A5A7FB676,0xA563551A561FF7ED,0xD2D635E40CFD5E28,0xDDE98AA30866C92B,0x408F81E566D14EFA,0x16D0AC3E59BFF,0x29F8F3C171C9EA,0x37CFE4AF5449E515,0x4B41930F877523,0xA6A45C62EF5092B0,0x369A32A319046E7A,0xA8E497236911D6F4,0xD19212EE39CB,0x38FE34CDFD98F8FC,0x7E5C0CCC2FCEF40B,0x814B9013CB10E18E,0xB3525F5C3D862E,0x50A10D51764E930C,0xDD4561197135C7F7,0x178FB85105D81,0xD8D0F5949F15E11D,0x56D074C7A2E10AF8,0xF16BB9B74A5FA619,0x44B1E6DCF3514B3,0xF6C19500DF474751,0xE864BD3298F2E401,0x1BD106DF50C68,0x8489EBE1F7796E43,0x85BB6AA3CBC6A44D,0xF32FED21684D2E0A,0x57CAB2D8A5A9B50D,0xE0DBB0A7C22A405F,0x31083324A8AD5120,0x1B9911F8219EE,0xFB21DE1BC3C76C50,0x148574768D9D3CB,0xC8358562F9EC979F,0x6EF2B795777DEC3E,0x27252888B7C9E875,0x12BC62778391A3F2,0x1EC67969D6A49,0x98F26CFC864424CC,0x65804B13CCAB836D,0x5A7309DD73CFC68D,0x5EED2C7A1F2715C8,0x84283F0787738E47,0x147C73CB03DBA50C,0x209494E64E36A,0x6045DD5D3548E45D,0xB7D85C0C7DCC45F8,0xC22A076DAFAD0818,0x7B5798466C0E7974,0xA3DA450658E672C8,0x5FB0E25C03632FAA,0x7345EA805A1E,0x84C490A48762E734,0xD9D5A2121C4C908D,0xCF569ADB4BD75E55,0x879EF9A86F62A25B,0x94E7D1FAE895D768,0xAAAEF80EF5C7B689,0x156FC4F666604,0xC0D8C890147D406,0xA1E0E0CA234F4990,0x8B53CFEE9B4897A4,0x478267E3D87989F0,0x72B4FCE27C685BB7,0x4FB88FCC8A62FC6D,0x3B77F4D18B5A,0x771678BD7F9B8AD4,0xDBD1298F8F06B240,0x451624C67894DABA,0xC11C6C92C606E12E,0xBBE551C4095827D0,0x17AA5FF49A2F58DF,0x26D299E0AD20,0x472391455F2AEDF1,0xB6587C7FA7C91C69,0x504AC33B69F23F90,0x14A31EFE9971DE52,0x36FC8D8009A256A6,0x402AF6C5B19A9EDF,0x1DADA1D60F871,0x5532BFC96A6AE251,0xCD38C8CB17B6C639,0xD711C9CE3A7328DB,0x791261B11B9641E7,0xF1E67B9A3FC38C5D,0x96034F295B89C82F,0x50E8D7EB4868,0xC83AE0309ACF3093,0xE415D4FF802D1981,0x43FC8714E3F2FC5B,0x4B74ACCC931B1DEE,0xB9BA7DFDB44C380,0x7CAB0703F445F953,0x3FAC4B84DE34,0x2AB70DB32CF691EF,0xE32BBC928E671EBF,0x509D4D50CD55D5,0x5AC6A9D0A95BDAA7,0x9F08A1752DBBFD74,0xC1F613F1468E523F,0x13ADEB76D9C76,0x717E23CEAEB482B6,0x238B0AC28ABADB0,0xDF864E5D965E8BBD,0x6A498321F7F6D14F,0x818421E33286F140,0xDAEEAA06A0421DEE,0x96CB15B58C1D,0x37E850480C398380,0x9380062B28507DC4,0x6936C24BE9A79DD6,0x30441A6E015F3C5B,0x4F4653738172647D,0x3FE23048788A9CB2,0x26C467BB4289,0x6E3262BDEE58D514,0x9A537427E7776207,0xE3182BF1DBC2DCDB,0xA524A93FA30C02CA,0xD3039DBC62A96F24,0x19D07004FB855468,0xEEC3403CA6D8,0x34063DB588ABF0C5,0x444B2EC9534E4B8C,0xEF5F1A4698C8741D,0x2A60574B8DFC2AD3,0x6A36978D1D75C62F,0x763452BF6108618C,0x177D583EFE43F,0x676553185DCDC0C1,0x22144F92C76DEF69,0x1EAD0BC69BE314F,0x163B1483EA1A8412,0xB46D9EF1E23EEFB3,0x507B7F6835DB83FD,0x18F60862C39DC,0xCE6EC5D701FFDA58,0xA1FBA73BEBEFC1CB,0x54ED5ED4B46AFBD4,0x8ED0006D5C62CA92,0x8ECCE74F0206CEBC,0xDCD4B160F0DB294C,0x1FC74A2012289,0x96B9798753BCBB16,0x3E1846CB6DD8A800,0x63DCC0F18F082D9,0x238D9F47DDBC734E,0x7875AD3B8E3E0C7D,0xF1097D24844ACFC3,0x5739C11E70EF,0x9F0748E8778E52CC,0x14B6D3A93D153BFD,0xE2080A331B867DBE,0x68FF669C489CA2FF,0xD2F4396606CD2F8E,0x6C2E6F9A44BA1CDF,0x1FF9A8A25DCF8,0x20CDAFB6F42AA9D2,0xBD990C7196D5DD41,0xEEC035970046B361,0xA76FB28693E36642,0xDCC311658E2F7D30,0x4A655CED0B6AAA2,0xA817AFB237A3,0x3F52CD947CCDB0C9,0x8FCC309AFCB6C653,0x83CB787BEFCB671C,0x2BA40F340C42BA5C,0x528696E77C1E31CA,0xC69C31B5DC2FE425,0x21BC39B630F7B,0x15D7BF7B231B3B2D,0x8C60F596AE154F5F,0x473397219B641094,0x81B224823729B3D1,0x3E80D43D89A8F409,0x95052CCB6B5D278B,0x1CB58CDC1C7F2,0x64B62820CA177D9C,0xFEF641546ECC605D,0xA4782ADCD44E62D1,0xF4E07AD179F6B31F,0xBFA3CEB09208800,0x31469BB0854285EC,0xBBE2AD418D58,0xFD1A8EDBB4BD8BBC,0x7EB997C5FD733497,0x4464CB2F9A598CC3,0x926372A9095C2AB5,0x6FEA1D9A93842E65,0x3232366FAB271C16,0x10261EB54A2FA,0x55947CB8C00AD2D5,0x31885E6E8C0C237C,0xE6E0E6A5D1C49F2F,0x90C42A399AAB306C,0x19C80C674B93E099,0x8C68A96096EFC8C4,0xDD6C401CB511,0xD01432A4E6069D4,0xD60D11DAF31B85EE,0xD3F0AD1E102A1C34,0xA8DD84D4C416915,0xFE15117696E4981,0x42F72041C83F654C,0x2304BDD439E69,0x72662CEE64944479,0xF51E4F59A076BC5D,0xA0E602CC617E13AD,0x20F525A8B8A2C177,0x7933A561F0225A78,0xE6D005B1525F0B75,0x16DA8348AAD0A,0x756744C323F443BC,0x38A37292276C2DD6,0x5E3CC4DB23D2C5D3,0x44026A64112D502E,0x43CA664169DECDA5,0xEEA0B6B6E53E1146,0xF664567486F9,0x4EA9B7A6BD99B6C7,0x80729D8CA5F41EC6,0x7B8609548AF56330,0x354444BDCA16BBED,0xA69F1C9789C9F0BF,0x21224C226C40146,0x113ACF8CE7D13,0x13605F0D867AEBC4,0xB5E4722A820E69C6,0x4360B7C0EE57D6ED,0x54A4D496BB748945,0x6C9051904827FAFB,0x89716C501BAFF2F2,0x1695EB6E43924,0x88DFC9E90259DB96,0x42D8ADE7F7ACBD2A,0xC641B5C706DE335C,0xA765A3F9D394D21E,0xF9076071A17E9CAF,0xACC75CD2BB073EBE,0x9EBBD417B6D1,0x8163D6EF0FC09E28,0x3B95661D45D64906,0x773E0AE79C2E2589,0x47CB19BEDDFEFF33,0xEB618B197B5F28E7,0xE5E5B9F8513B38F7,0x18DCADE21593,0x56D7B9030B3E6BB2,0x35DA2EB8ABED53B6,0x57B4532682BE42C3,0x25F93153AB803781,0x8E9DE6116D67878D,0xF7E977E8F9C1B01C,0x211BC8265C2A4,0x3B6D025BDD4BDF09,0x21C43009A1420DFF,0x60878EADCA98EBE2,0x35C75B4CE8B8C878,0x3C615CE9E76B24EA,0x8D297D0BEBB0FC26,0x21BE01D065C3A,0xD83DDC0448820AA0,0x17D44264FA9548F9,0xB9ABD804C8AB0A02,0xE9ECAB01FBA7472E,0x7088EB64455B81EA,0x9665C2EDD6237DD5,0x1460582DB9DBF,0xBE0208C2816090C5,0x736EAC9AC72EA6A6,0x5E7D8895CC27B4E3,0xCADE7023A852DCD7,0xA027D4349DDADDF8,0xC78E7F141A9A5125,0x1E8F8D54BEF69,0xAEFFA99039E3911,0x8E085448BA7D8CF5,0xBE39F08833915C11,0x38D866119C5994BC,0xBD9CAD479BC4DE47,0x180C57C0011F1819,0x8F6154EBF6AD,0x4FE3B6486308B2D8,0x8A6F886B9057EB17,0x4D14D0C12CC84255,0x3B39E63310BDF55,0x46570699BF360706,0xBEFD17128A5F55F1,0x1D1DE42CA95E0,0x92224E9CFDA0A325,0x5A3A1EA900B43EB0,0x2F3096CB3FF32515,0x79919C391E8ED1E9,0xC68B2CAECB6510F5,0x92D133C648769E08,0x1B2F7DC4D2038,0xC569E30160D273B4,0x6406236BB473C2C2,0x6CD01C46DD4FFB12,0x3A43456B61D1AA0F,0x8E8C677B59D6B1FF,0x4DEE41DD1CAD2C79,0x4BF8FA6855C9,0x2B38A5DCDF547A8C,0xAB31195F9ACF7531,0x7B08BDF885065B3B,0xEB1074C5A4A20157,0xDBC40C2C1B012023,0xD1FCDC8ECAA6A7D3,0x19CDE9149C343,0xABE356D428FF310F,0x3988BD8702A707D3,0xC7EF9B6138ED5C9D,0xCD7A1648ED63ABCF,0xEFAF0583343C7778,0x671873C9B670AEF1,0x2208C5AAA93DD,0xFAAE88E2908CAF83,0xEF9AB2BAE520824E,0x23B48A6042786DC8,0x3BFEAC78C7EFC697,0xF5BD779194FB2ABA,0x571F237144F88DB5,0x25A4869AC2BF,0xB2247E97FDE5FF41,0xC03355CA688D1D44,0xAEBCE9E59B4FB10,0xF5A97DC0586D6803,0xF5023633CD5B9642,0xC73C55A7D52E8AC2,0xFB047C64BFE5,0xCBFB5B3EA0D6347E,0xB69F33C51950D74C,0xF7DF76840D3C5CEB,0x2246493A346E885F,0x4753BFAA754FDFCE,0xBD5FDF9057473D4,0x4F076558D29C,0xAC5B12C9F2F49B43,0x5175F26B8AE16FC9,0x67B08D52D8200AE4,0x31EB6A57D2C9599E,0xFAC1BB1012D79089,0x1789BBFC4CC28B17,0xD536AD11408F,0xE6B166DB54DEECCF,0xAA0725609C4B2236,0x93BFEE98A220D288,0x5F5B3DBAEFCDF13C,0xB9CEFA1F2B7F8500,0x44597B03562753B1,0x15515678A02D6,0xFCD331AF76248C8A,0xB5044EF1980DCA8,0x49AE3F3D66054348,0x3B3D576B6BC56634,0xD40B81DFC46CE3EF,0xE2E62163843C1FD4,0xCDC812C9FC46,0x9C86F8083CA481E0,0x79BC93A2853B21F9,0xBAF30392A81C50ED,0xB0ED84D591CB7D37,0x286007BF4479BD8C,0x2F57F6B56C74CAA6,0x1132F3DAE3C25,0x59A7D21EA5F353EC,0x77BC1F2C96E50A58,0xC776EC933E7486CD,0xD409C9702FA02D81,0xE9D0AB136A3DC9EC,0xC835E860152138B8,0xD6BD90BE633,0x65C4D2526B924344,0xCECCFCC3133A9ADE,0xEE00E79894AF2483,0x667F371919AFCE61,0x91F82CBD4AA64218,0xEB072A56B2B95417,0x1CCDCB9E7BDD,0x7D3C082AE28015D9,0xD2D70DB48D783170,0x1A0F2D0137C96442,0x28CF8971876B910B,0x7C633E18CFE4F438,0x18B31B914F31CCCE,0x7433C89F3CDE,0x9A576139A43C9248,0x237DCD38A5155625,0x35A3A9F5D53214F5,0x29C596B44E33E646,0x826A284CE055154A,0x31F3FCC23A87F5E2,0x1ABB7BB26D494,0xB632CB4A189B52CD,0x8FBC824223ABDE4D,0xCA784F68ACB6FC59,0x47F1BA7803E48701,0x3E2B7FADDB3D9183,0x129B34614B9B277B,0x8F6580FAF51E,0xA197C25829C37274,0x68BCA862D7620C34,0x52C84117A450952,0xA020AE1AEBD926A4,0xCF4E0FB4AA47DA2D,0x195A49497F4C05DB,0x8F5EC4888F51,0x4B84EC8A562DBA0D,0xF63BF8AE7D9DBA3D,0xCBF74B580CFF9867,0xE3233438E0F09F23,0x1B43FF93ACE5F27C,0xC76F6DF1846CB38D,0x6E1260243E67,0xDB96F56E6D5FB4DE,0x570DDDEFD01D876D,0xFDD41F669229DD67,0x33FEF9020503394D,0x10F60F3A8641D2D8,0x7A8DFB6305FBCDFC,0xD3C4389F2B34,0x9584515BD4E03DEB,0xB8DE3C147ED7BFF2,0xBE0066C28B50C4A7,0xC70B8E616A791C0,0xE9E86BC0F65DC583,0x43C7838C628A19FC,0x69C75BBBDB19,0xF9211F2899C71025,0x886EDBACA5AD717,0x2E1465395E681767,0x726024F86768F704,0xDF097B01AD209E9,0x37A88C27CAEE1B43,0x1AB95D7D98301,0xC8337B8BF47262DF,0x770AE04C066D8CF2,0x20015C68912D507,0x375D0B0C0163A8F1,0x1594E66608741E94,0xD1324A31C65B9CAE,0x13A65129C1ADE,0x7A7FAF2CFC3B47C4,0xC69ED56564610FD3,0x14D70F612A0D0A02,0x5F39AF19214471F,0xBE4A577822DF8853,0x738C504286BE9E8B,0xFA55200410D0,0xA27981EEEF565A52,0xD6D4BDDE3403FC51,0x591054F45787DC63,0x6FC5DF0E3D752FEB,0xEEA97D066FA191B6,0xAEB40218DB5BF211,0x1F0683E337D2C,0xAAD1A549AF1202D7,0xF1306A05A9DBC9DE,0x9AA52E35515EFF40,0xC6FA6FC47515DC63,0xAAA81152432615BB,0x35666982877BE98E,0x1B25EEBA706AF,0x64863CB1B2C29C9D,0xBC688B14C6848AB,0x4025513FF5AFEDB9,0x970E13A38E4B049F,0xE96B1541B53A000E,0x5C15DB14CFB677B6,0x20D9976F18FAB,0x200F56B7EF4BE92B,0x46A2DC6DFDD3B87A,0x84F5BBB4B9C6BB3E,0x99FFA345B5B54747,0x4BF1A22A125A5A9E,0x4CE6702927132840,0x19C00F3790FC1,0x13089ACF10472DE8,0xFCFB0A62F8F99650,0x26339341D6530CB1,0xF9C05F3D1BCCA0FF,0xC1313F4B94953E4B,0x683CD078C10DB4A5,0x1D306BD6C62B,0xF4024A2172F4742C,0x3294D6C3CAE55FEA,0xB4C947D3440E3078,0x2079EC63FEAB3665,0x11F2730E5111C7C1,0xD6F185DC47961CE3,0x12C07AF3E200C,0xB3C35CF4027184FE,0xC343CCBE30FCB3A2,0xF5E16A370A229FF8,0x44F98C5AA1FA9BDB,0x72533790501CCA41,0x6E8693BFD3E078AE,0x162CE07764ACC,0x4AFDEF4AC3CC0065,0x95D51EDB15780033,0x6623DC3F037376B1,0xD5E3C524A2587A74,0xAFFD7FBFF398708E,0x3A207C878BD13B7A,0xB5BE4BEB023E,0xDAC795F02144083A,0x59CB1448C6E0CA21,0x87B0D3016A55A59C,0x4FDE4D1A1C0B5C2E,0x1C7EDADDA115E3C4,0x7CDD884658915F11,0x21B428DF943E1,0xA324AF7974DAD0A4,0x35D2B72CAC1A9020,0xC6B8F51B3289607A,0x8D8EBE9FF6413293,0xA865F42BE8E718AF,0xA761DD51C4383201,0x159F04F21E802,0xD3BAAC91D44683F1,0x259F2FC8FC244361,0x1D9E9C1E4422B812,0x2FF4DA9A6DF0E864,0x40DB955E784E41D2,0x22CFF497BE206106,0xFA8663C03847,0x6E587A165FD68240,0x23B494B3204846A4,0x54A0E74B1C253DB9,0x571BB1D210CFCCF8,0x9945A5B078B50995,0xC3CCDE1B99A1F6C8,0x444B7EC6D5BB,0xC75F3472B1A85173,0xD27CB8C55C01FD38,0x6B4F835618468DF4,0x4527D50D4A6540A7,0x466AECE50E780AB6,0xD54EE55250475210,0x7101341C39AA,0x350613C412A96AED,0x4B16B55D99D679E2,0xCA1512DC14BE8AEB,0x196E20EBEFE2C78D,0x265806727667FE29,0x717701038FEE44D6,0x14EF99956C3F7,0x3DD1A9DB83C6418A,0x1856E198A79211F8,0x3D26BF9BDFBF4174,0xC65AF978DC5F0BF6,0x325D1A2F77B6632F,0x30002DDA4735B0C1,0x1FFE862A04E96,0xB2B52F9FEE8C8CCB,0x8C9793AF60C93752,0x989E9DE63F77F57C,0x7128EDACACBE70B1,0x5D8795DE50EEEFA,0x1BE9619223F2E3CA,0x4A7E7ECD9C5F,0xFB71F5822983BFA,0xF284640AF455C93,0xC494F0386DBE9757,0xD08543D94A0AECE3,0x1203B97CB0140430,0x7546BFE82F30379E,0x3EA06F568F1B,0x223E88FE6F1B4948,0x339B48072BE9C7A3,0x1EA706D1C8859C1B,0x5F831876181D4FD0,0x3F3E2572217AEC45,0x40E0091276BA683D,0x156D1D5F8631,0xFF75D7924545923D,0x9916651856E7AC42,0x32C757D08652F92F,0xD3CE147415D17515,0xBF8F9942A9F25CFF,0x380521F4199EDDEE,0x15EA26F5AD22C,0x6B71BB951B198EB8,0xA6DEEBDC99524E87,0x607F27B9494ED99F,0x5D34F47C9B54B432,0x27C05F161B96998A,0x84B77EEA8F457660,0xAE562FF4539,0x336E137AFCB0688D,0xC2A904F54008251B,0x6C6CD3E186B20BB5,0xCC1CA12A596E70CE,0xD5125595C8EB573,0x8366E00C32C4AFA5,0x213B0F3DE25C3,0x12CB11387D0E1247,0x8B948D25DA6FE598,0x902319117F24C1EE,0x4508553946F08A6,0x4BB3717EBEFEE9CB,0xF54E357811E1AF7F,0x329996C7BF35,0x1D3093CB097651F1,0xAFDB67BE06B37202,0xE12EEFB1F65B8128,0x8AD735F9AB9FB91F,0x5C14EA01ADE5D9F7,0x7470BEBD21E507C0,0x9B50371C9B55,0x502F5AB0EFEAA3BD,0x1CDA16ABC6764C5D,0x16F4BF04B74992ED,0x70D49C9CAA8CFE2F,0xF6305338D9956E24,0x892F8B806E341F8D,0x19E255FC7817E,0x75D3146D75362AD9,0xE6C63513EDE0BC41,0xBBB7A748088F336D,0xAC0F776DF2096CBB,0x39E1E095DA3AD9BD,0x298A78C680ED0AF4,0x1AAF08105DA2B,0x5812024A085BCB8C,0xB5DA0EA4C6680E13,0x4B892D4E32702364,0xCD43FBBC14754F1A,0x4C4176F37722BA4D,0xFB43389413ADE03,0x21A8ECD99834,0x7E7EAB1F276BA202,0xC3973A35C211C8AC,0x6EB2090B5AA2E453,0xF1B87410E7AC7EA,0x60EEB312ED4ADFD6,0x10C8AAA64C32FC1B,0x9CD4B509E3A5,0xB4291A0F2B18B33C,0x1BB875C19FD7532F,0x66DD90D6B31DC899,0xAD1ECC3EDFF340BE,0x9C45FF1C7AF2E834,0xC65DE4096B73AB1F,0x1B8F5919A576F,0x8B9E7C9218B022ED,0x14D4A6823CEA4525,0x921D28209435A015,0x51B71BFAE00522A6,0x947B14DAE0D1BD95,0xE8F73E4FBB4B5462,0x165505363EE4B,0x41670CC5D7B3A682,0x177AA740637DEE2,0xE00F62AB186EBB61,0xB82271BE0FBFE5D9,0xC31114F14B8F0CB,0x29C51F52D2E13C8,0x1B1857D3E5445,0xE745464C0B8E8A54,0x414B1193BD85C84F,0x795FFCFA808A6FCE,0xE2EC2A7C5D0A2044,0xCEED293E9AB5D3F7,0x13703E6498C088E9,0x20150F4839511,0x3187F1D8FC5B5D96,0x2DF1093D539C1761,0xD38A2E54AE9CD49F,0xBDC3F5E753214FAC,0x64A0958F9461798E,0x69609581BD201F10,0x17EC8FDA33788,0x8B04BE81E07153A0,0x88F8253763A9AB33,0x9C80AC40DA37EED5,0x8B87951C2013B2C4,0xB365456906124A4E,0x664CCE52F288162A,0x140335BA62950,0x7CA7727695D9877E,0x58D29315846AFB55,0xD1DBF983C0C3DDCF,0xF2AA13A3712277B5,0x597C56705DC0715F,0x8BF6E1746DA886B7,0x1E9B6C775182B,0x79830FF0C584729D,0x6EEE28887724A9D5,0x6FEE78FCA3A0DCCF,0xC7AAEE962F402056,0xA3799D431FA5B308,0xC051E23F447D63AA,0xCB3F1B27BCFD,0x800C845C054E5563,0xFFDACFD250BDD96F,0xE68CE6E45F7B9D5E,0xFFBE47025391CD4C,0xAE170D8D6A0978E9,0xDB5AE93E15CD11CD,0x151E68589C827,0x21E44A9FCD46E2A6,0xF16AFDA678E0AFB2,0x9B7BB00EFAF5335F,0x36FD6869A3B7D6DB,0xD70A69FB744D7B24,0xAB64E3B2DFBB40A4,0xBE65258E717C,0xE8F3B2352357AEDD,0xB80A895AEB3FE706,0xB7BB6225E15CC8EB,0x7F34DBCEC08B72B8,0x675EB381ECB98125,0xDCAFB7AEF81029E6,0x1AD4286E5CFA,0x9B16C1BA89326063,0xBE2CD4F72C6A4A92,0x71419173D79FE08E,0xE99E797B397BB936,0x58C8EAF4F701CB25,0x43A2E99BAC8F3840,0x1D22B5ABDC838,0x479190BAD02A12B9,0xB826A9251F1EA32F,0x97C467172279BC97,0xCD198321795F6CB1,0x78D4ECF4DDC0F488,0xD688A43F3ED1837F,0x1410BB4D88E04,0xD22FA3AEA51440F2,0xC05213C2F70538CB,0xECA39C2FC49C922A,0xE592BABF7F2901B6,0x71822667F8B0F7EE,0x759A4517F05F47B1,0x1DA99588C251C,0xBBBBC746DB4F892F,0x5C0584C4D72BA1E7,0x764BEAEC281F602A,0x73F758BCA47CD5E0,0x9C7D87C43764B6DB,0x3A88965EE2D98A44,0x26E9F2A4CE03,0x7AD296680D645EDC,0xD81B5E62BC0E753C,0x5CC139220B356699,0xA6E45374731333E9,0xAE2644563893047F,0x5C57E74EE605F27F,0xDDCD68A72734,0xFE662224060714AA,0x43A3FF8740C53EA0,0x7BD45557C0956FF8,0xD36F361DF83B2130,0xBD7B1683D55A8352,0x9E48B079627A1284,0x1A0E2F91579,0x5DE9CC4FFA0637E5,0x9AF2ADB6B4BC53F4,0x732CBE4D7DF38977,0x5806A0647ED76990,0xA6A1CB4F1EA1D4FA,0x3D2A54350C927632,0x16E212911D833,0xA8AAD85C97574823,0x4E6432756D8506B0,0x24D365E8992B3FFA,0xE93C4145E0CB776A,0x3033470C71605FAF,0x379ACE4B00463AA0,0x1B0FDCF31A387,0x3F599F3A9F26F569,0xED1D367CE7918B27,0x295FF6C96E63D8CA,0x2A367AA4CFF0D914,0x49E6E989142D8709,0xB1FCFB31C583E540,0x128464D679092,0x377587EF22DDF18E,0xDA4E55AD386999D4,0x46189A63F65B8802,0x22A7A43589C5BCB3,0x300C2D6AA0207A51,0x81B1A059B2E5CEBF,0x12418521FB33D,0xE83BC98DB14DF0F6,0xFF7B10D1B5733D6E,0xFE007606F1896A58,0xCEF7ABA2C766A0A5,0xA26746BA598BA9A9,0x44D8F01FCF1BC6A6,0xD41ECCA97911,0xE56C3DF53A8D5E41,0xD292D4AE1C2FC2DF,0xA436F1641F426F99,0xBDE17790AF0AB83C,0x208373FA458DCE58,0xD68C70202D18821E,0x20DF3A68EC990,0x7609110401A74906,0xAA6928283D7C0D5F,0xD79E62E850FBBF57,0x928A1D21B5CA71B9,0x12D258F23A8DA7FD,0xF51AE005EC81AF04,0x21874B8C31DF7,0x12EFFA4B7A01788D,0xA24BC5302ED2B21C,0x8A3072D57697D6EA,0xED2B5BA5AF534CF1,0xABC88617DB74396F,0x2CDDE9074C16CF1B,0x2146694C4412E,0xBC2B31CC408DE079,0xA536314A802DAAA0,0x42E078683EB512C4,0x95A83A1FF73B58D6,0x1AC1681B238ACAE,0x5FDCDE18C77C0B00,0x20A2A2198C256,0x55279B1AD61AFC13,0x63610837175E54F,0xF6E2C3D401BA1E61,0xB8F286CAAFB89A7B,0x3842AF650AC65E9,0xD0AD5850FEA2145C,0x20876275CBC5B,0x242BD3BBE83F267F,0xCB2AB8813B4E6431,0x57E4CD87546929DA,0x214D92D67295934E,0xAE8B204B38E32A34,0x38B46DAE8F7D3244,0xD177F1631E2D,0x69BD86E1C0DABAD1,0x48F2ED211B8922BC,0xB7F03954E0D2027,0x3CD14E05BC2F9E86,0xFD71D4B2C79093AF,0x7C85E606EAB40FB,0x7B5F0002FA2F,0xED36355EACE4AA69,0xF936D4C3D379C54D,0x4F8BB99D5231774E,0x57DE717FF9A49265,0x8CA9AD721B5DBC0C,0xD0E575D6269BDBCC,0x102C24B89A99E,0x8D62BDD245308E64,0xBC82175CB26D75A4,0x1E148785EF3D9AC4,0x59CDAE3C5E460BFD,0x4B43D4BD9CAD24F4,0xE1689094523EF6CB,0x24BC47342670,0xEAE295D9EDD7B326,0x4761292AB1312CE2,0xB4669DCCD6E09A1E,0x6C97B1724D72841A,0x8344E7039DACE577,0x1EA877C468AC3C91,0x1039C6B91B007,0xB9690741FE84ECEF,0x5764539F7F0EF535,0xDF1EE4CAC0E0534,0x74312C89440252FE,0x52804714E143758A,0x407860531A06D516,0xC87E67470986,0xB1FE72D22FAD9021,0xD456B59FD018EC13,0x32CF5D474F78D32F,0x4067196FA3E1C4F4,0xA74307CC4861DFE0,0xEA68958514197E59,0x1FAF0FA967C8F,0x88ECA0229531518D,0x51A3737724786A14,0x91D59D071EF1EB01,0xA2CCA7DD91EE125D,0x19D1789DAF1AAF30,0x1C37D3FB481B84B2,0x14C56A3980481,0x887682916B78EA60,0x9216C34D5632F95F,0x77DF5ED25A66555C,0xDCC5719D917A890,0xE3E0FB7B757469D2,0x5472100DA71AF33E,0xB54418D59C70,0x8B7ADBB4AE73616C,0x2151BF1CF74DDC15,0x27534369BE915B75,0xED08D73CF3E8C1FB,0x29A86E4B2B69BB41,0xCF5696EDC9BFAB6B,0x10265EEF16709,0x5B80658DC4B343B4,0x38C506607A0E5645,0x78C629B89D18576B,0xAE7A204E0C1669D2,0xCDC4819ECB44D45,0xAB9ABBFE1D62E1E4,0x85B042B3C58,0xF29C84D3A99FC21,0x2788992E22887431,0xDE277C4A3189909,0x8B99E1C5B8B09BB8,0xE7E46B20B52D4234,0x69F1CDB8065C19DC,0x1E91F6E0026C7,0x72DDC2A683609E7C,0xBC3443169958AC44,0x2A122325DA5469B6,0x200ABBB9849C54E8,0x243EE3DEAEC7614D,0x49AC6F4D976814D3,0x76645B81F571,0x15C8C73F0EB4A15,0xBBB8C98D4C2CDDFE,0x2604D79424678FD6,0xC5DBE53F081BEF82,0xA321C91EBA3DBFD1,0x136F8E7DB16D114D,0x21425FD199782,0xD1A6D2271819E06A,0x1FC05C9882718AC3,0x7DE8A19AD77CF9CC,0xEA1A106A59578ECA,0xC085F5611E2ED845,0x431A3BCE37B61C18,0xC0F75E4B0025,0x54F4E7A89CF333E5,0x1C12E05B63645AEB,0x7EF99D39E3EFA6E5,0x9F938C187BB9F0AB,0xC2281CC15CAF9ABD,0xC69A07595C632FA4,0x1690F631FAE0E,0xCD9B8344BAE2973D,0x9E245EBD282001D6,0x170651EE8A55E6BF,0x855831B56D4A0F84,0x73DE76474627D8B3,0xF01D2D0A2181787C,0x11AA6DB04FDB3,0xB0FB7B2C676C2BB4,0x82548415333A5936,0x5847E40640FAF1A4,0x14EA3776F5C1295A,0x9117C238B05D451A,0xCFAC44BFC25FFA98,0x4D05CA7537A9,0x8E8A73464230C913,0xC9469ECAE61132C4,0xD17981A7670700E5,0x22B911C0467A2B79,0x8865839A3411EEC3,0x9A483AACAD157AD6,0x19BAD161008B,0xE4778CDADAC0F8E9,0x1CB2A8B3D9225823,0xB66C74F14368FB28,0xE8053207EB4C5DE7,0x9AE18770B14A0026,0xBC9C11CE766D8F01,0xC7EE73A44C7A,0xEFA0EEEB9FCB8707,0x7F670309CDBC0794,0x144B71EDB1589C11,0xEE97D607C698F8F1,0x380926DCAC8FB92E,0x5028A802FFEEA43E,0x1190DA46DB521,0x425B210CED6F4DE4,0xF46091ADDE2F4BC5,0xF1865D27ABDA6189,0xE62A482545BB1526,0x3E1E2812AFC5195F,0x51C01C15CAA68D4,0x3AAAF42F260,0x5C4081E72F7B4670,0x4FB40FD093D1446A,0x545446D1DE9824EF,0x6A6ECD1F55398B04,0xE8F7B1976ACA02C9,0x648BF2EE1CB5F881,0xC142A061F2CB,0x5FCCEE86B09F2D95,0x9C39E8846C557086,0x1CEFC36E3FE0FB7F,0x7927535204C3647E,0xC0B1B4DA01E8973F,0x7C8136411C0C1DB3,0x8E3BC337083F,0xA4DEAAA1E3AC7656,0xC823C0FB8F81DC87,0xF43975F83CA91A3B,0x3BEF213C08810FCC,0x9C838381ED436256,0x10C9EF87F5BF0046,0x5EA72636560F,0xC392349AFFE0A661,0xA301A660210FACC,0xA4374C2C9D70CEBF,0x57EBE97D64CDB70C,0xF979F083AEFCE306,0xFF142A2DF718AC42,0x190BEB881D898,0xE00129A87864718E,0x482ADAE811E42F81,0x15AA4E8D7E035D6C,0x3F8D255304A51171,0x6B9CFC017FF0B61C,0xFD3440B631CBDF5,0x21BDD4904E046,0x7778D70CC0B3DCB2,0x891CA6F2D45C1219,0x44E40F27646F5050,0x565B8DEAFDCF5C63,0x5487E4A2C97531AD,0xF9C4D86FEBC2E7DD,0xF4541B38BE58,0xE58588403111FCA4,0x3558B1DA7EAA39DB,0x8BDA15F4A5B48350,0xBEEA4915AD852E,0xE7B8FAE69D87F7CD,0xEB90F5AEC5393776,0x41AA873F92AE,0xF0DFEB9584F98D64,0xD3526E0705D55621,0xEB55E33642E14EC4,0xEBE1A92694551A4F,0x58F1F9B939DF3D8F,0x1EBB583FADF38EF6,0x1800EE97490EE,0x1B0F7CAA27FF950B,0xA172B227B25D553A,0x5FA64569610FBC14,0xCB84223162C6CE8B,0xC51847F777ED8F2B,0xC481DBF4990F7E81,0xC46172243941,0xFB5AD75A16E7B16D,0xA00417348BCCDA77,0xE0A9166051AAF54A,0x870B78AB21B87BD9,0x16E3ADD9FEA90968,0x1534F94303958904,0xD480BCBC7273,0x2E19C78A03A660BB,0xA5F252DA9E13924E,0x93863A84A863BCBB,0x53D84E32435A0C71,0x4EFC2875559C07C7,0x958EEF3DAD062DE3,0x2505EC949E53,0x6BA30AF53BDD2EFF,0xA3FF536404BF46A6,0xD689A44221279075,0xEAFC7B159FF6B0B7,0x2A429C32FD186F69,0xE281731B0C76F36F,0x1598C6E4EFF61,0xA1067B7F9EF62063,0xB066E9B03880A34E,0x13F6BDCFC4467AAB,0xBA252479955F26D7,0xCC8F5381537401D3,0xAF16CB6C76B811D1,0x213EE7EA100FA,0x6B79A27C635B1A2,0x6FA23C5AA002A849,0xDB5559D433E7ADD2,0x7E3D816B4B35CE8D,0x875BF7FA2AB290E0,0x82325EDB1AA0F026,0x39C241968C60,0xE277F0A312BE2F34,0xA72BDB044B54B89,0xE67FC4F35DFCB723,0x7D7D9924A67BB0C0,0x4BA7080A09DDAAA8,0x94EB2AFBD55988EF,0x9749A7AC0189,0x15B32AC1DB037154,0x653CD0ABCAF7D9D5,0x28F9BDE0EB0E21FD,0x3C1878255832427B,0x99609E99D772064C,0xDE9C19FB263DDD10,0xEE92B0A7DC5A,0x144AB71822AE3A73,0x197398706EF7ABDF,0x3FAF0621321FC028,0x8BAA3090310FC75C,0x414AB1DB09EAA9A6,0x8F1D35E617A89822,0x81517F263130,0x2823C7898F144348,0xD1AF23A8221F9EE5,0xE1A483ED9F0B9618,0xC3BA39C010F07415,0x3ACEB16831929E9,0x5842BE358776E0C4,0x458E55A850A2,0x159F52F71CBC1630,0x49AEBFA0A14DB119,0xCCB02206986B7458,0xE04D71D8CC415CDF,0xFEC92CF4A163148B,0x71B4D20FF24EE036,0x1682E0CE962ED,0xEDE9864477B63455,0x328C34F514592717,0x5F2C203B2A9E36DC,0x5CEBDDF689AE7281,0x48329BFBE000DC52,0x6D7FA8672EBC346A,0x1D7208BD6D5C1,0x643D1C5F5F42618C,0xD8B66E3231D5295D,0xC845F9F201AE6495,0x6DB5222874141A4C,0x2AE95B78CB46770A,0x6C2F2C7ACA0EEDF8,0x1ED5654AA287E,0x5D4D1439B0948D87,0x9C01F06555C5CA03,0x99AD1E6525323879,0xAA3DCCA6F5A6DCC6,0x5567A2408062C2D4,0x2F6C28601748A38E,0x386C1FCC4948,0x7906FE86FA6F1DD8,0x63ED266B14EE9474,0xCACADDD7E1DA78AA,0x688C2814E53B3121,0x74B214F0306C3AC3,0x46DC44BC9C7F683A,0xC1DF58651C3E,0x4ED9DB62A359D207,0x385282099CE643A2,0xB98DC4A7E527D634,0x94EB65C9BC377DD6,0x922CF69D72562547,0x6E9661EF8D05A2F6,0x5573AE77F24D,0xF575686154311EB0,0xE938BB510A854427,0xBD1FABC99BF51B20,0xDAA713CE71911487,0x719476B5C232E4E2,0x9FEBCB0E095A3378,0x2187672897A46,0x54280FCD0137681C,0xC3076F709E6AC9D2,0x45FB36E98A2B6362,0xF92BCD101855B865,0x24E3810077CF4CA7,0xBBB0CAF5CDC8E7CC,0x11290394A79FF,0x83C7A7C654457A72,0x4C4B601001EE67CE,0x66C0AB430077FB61,0x558319611AEB138C,0x98470DB5344602A,0x2B2F9191A69200D9,0x122ABCF332EAB,0x2B5CADEA4814172,0x67B2CBB3F4D92E91,0xD89B87AFECE265D,0x32D06291986E6ACD,0x5F651B0B3265BB8C,0xDE2AE7E8B8A85D26,0x1087E1E7C45D0,0xF2FA13063082C5D3,0xFF133884BD62E0B7,0x15A2B57F99173A32,0xB6BCA691B32850AB,0x22671291C0C69ED4,0xA7EEEA0975EAA822,0x1E5726A483B07,0xCFF91F58544288E8,0x8269FF777422222F,0x767A8D9624D00EC3,0xDA240CA06E6C2FEC,0xA936A9D860FC4AE4,0x431FAF790F0F4466,0x1C6EBF25B515D,0x14EEE0422CF5BD0A,0xAC6D590037AB7A1B,0x4919FD75BA03090A,0x43B3E99F764BE6FD,0xCAFFC2E38C918B39,0x6EEAEB3EF7F245E7,0x1D16B40C07D9E,0xA3F9E05FBFB9F5E9,0xC0A0712AD8330945,0x7B81594FA8166BD,0xE7FE92051989254A,0x6351F4D7303B34F3,0xBF8FE9BF41AC55DD,0x81D27B4603F8,0x1470D302647BACA0,0xD9558ACE090B2DA,0xB1B81C5BAC203265,0x76A343045FAFDB1B,0x223B86ABB9736253,0x4917DE367221FC36,0x488396FF98A9,0x203304A2BBB40E3C,0xCCFA074B8C0567F5,0xBDCAA38F7A37BD74,0xFC1440918608AF4A,0x69A4F830546DB077,0x95D0852A9FA357C2,0xB118752371F6,0x15FFC9BD84D91AC4,0xEAC607FCCA8B25F3,0x50ADBCB093402A45,0xC6D72B5D9F6E1B86,0x671ABC45D511942E,0x542EB3C5B9DE4E71,0x1F7099C3F2D92,0x678C1B54F100EEFB,0xD0959C1575258109,0x31A512EFDD170157,0x47F82280C34BF51D,0xA439E47576AAA282,0x22F3E0A14D0B441E,0x25E4FF2007EE,0xCE918E46E4FBE876,0x9801E19FC93DCB52,0x91C5A3B5CBD381B4,0x6C6C844D44E58AB7,0x8EA5214C7F50C8E6,0xD021BA31244B395,0x2031E1EBA90E4,0x2440531EB8EEE489,0x2610E6235CB71365,0x1469AEFA0E78AF6F,0x7B132079740FE04,0xF71D198B518F3341,0xC94D22BF5ED53FFC,0x19F5AE3B6E1E0,0x2A4371CEEF216890,0xEAD8B7E0FF053D6B,0x2750B2993D6A6450,0x99C13FBC5AAB40E1,0x2A1190EE3F326149,0xE214C7062FA8D6D4,0x22EE586B1736C,0xE927E82DDFA88CF5,0xC847F2338956BE09,0xE47ED865F0989744,0xB79C9A7F1FE1157D,0xEC822D4E9A02354E,0xD5783E1EAB683951,0xC7CDF3CE2165,0x641097D94411EBD9,0xC26D2BF4049D4386,0xB3CA1FC78D071AB5,0x3D86C3DA06F0F3E0,0x54B3BA7713013FA3,0x3F7C8AA230AFAB0C,0x15A72D5EA1D35,0xE9A27616627A338F,0xE9B0A4AEC6D790DA,0x1D9A51F724497A03,0xEF3772C36273A46,0xF09A6B9F324463CC,0xF6AF696FC4D3A683,0xAD865C3B5CDC,0x6A33AC1DE02E65ED,0xCC782BC8677AD36E,0x2A1F5CF72A977F4E,0x1286862CE67DA7F0,0xFFEFD5EEF48EF03E,0x1662D2EBFC8C75FE,0x17C9C16489560,0x7C83510F1E073F8A,0x552CBDDCAA422FD4,0x4B0CE3064B69B192,0x26E774F4501B8FF,0x72126FF5CAD44775,0x551487290B5B822C,0x6BE37157D248,0x56F76C4A14D9B1F6,0xF5B9E1701D74BA16,0x582813FA2E6C9362,0xF3A9BF72B6D83F51,0x6498289CFEA60D88,0x4D949802B7A52E2C,0x157C253B27603,0xB0C78D9422EAC3E3,0x5A996A040DF60EC1,0x2CE38B8C1A68C3B2,0x324019DBC145D35,0x40AECED1D101AF7A,0xDF05F30D986BF31F,0x114F6738AE35A,0xBED90BE650A877D,0x5C40BF201960B3C1,0xD17075AA640F9809,0x6148F86F25E28E3E,0xF5BCA0F9F89D36F7,0x6D9FE8FFD881816D,0x13EF37025263F,0x5BC7982E46851B6,0x67D9F82BE2E45299,0xFC66FB83767F7A0A,0x8EEC5EC7B3825C50,0xA51B890543A5CF16,0x48841759DD689D20,0xE67013873BA,0x81E476377C6188A4,0x9342939A6EF03A67,0xD4200752A2BCE12C,0xD19864DC7369E144,0xB371267F42E0537F,0x3D050962A221243F,0x1B48DD7B16663,0xD8E246088769BE5,0xF24894222BECC047,0x89F28B19A3F184EC,0x2684DB124538EC09,0xACE5F2D7646BDA3A,0xC9F51B79DB4BABF2,0x6DB9E6B7BC7C,0x4F650848B594887,0x9C642AA7AFFA326F,0x8D0DD386768CF96,0xA7D0B6AB3199E103,0x2E3088A93452CC61,0x2DD75BD7C7ED2D15,0x1239EA61E5FE0,0xE5E983D21D2EE596,0x6048597F75455ABE,0x63C258EBD356ECC4,0xAF45C93F42E7EBFE,0x4135E87D0892E951,0xC3B3F168DAE19C09,0x21606F03E6E86,0x13AEFEEE61CEF623,0x395524FC30BB0C61,0xE604D87A398B00CB,0x7C4B3F85E46BA97C,0xF407037C179B16EA,0x297929D94F009B66,0x21C0C34466C5B,0xAC7C30402F5AAA58,0x5CD217BAAE6C8B31,0xA85EFE7D5ACC2ABE,0xCF4807B272FEDB40,0x7EB9C24AE2A1DE06,0x917D14CEE6B09889,0x3357CAEF37B5,0x63BEAF31DD2A1A7,0x1B6CB88F1B41C035,0x169FBE6CEA1B5D26,0x84E9720E34938E7,0x90EFDDA4191EF9FE,0x5F846E6686BABA1F,0x203736769E832,0xBE3C06C737AE4D57,0x32EABD8FEEC2B84F,0x8935B6DF6D373D16,0x261665160CDABECC,0x77583FDBA470FE16,0x1EFAAB64FAC0EBE7,0x172B908CFA03D,0xE9DC5C8794C3C281,0x6C4864085F84D70F,0xAC7E698F6F625800,0xA4FD8A6276964766,0xB50EE00C9B27606B,0xFCD661FC3163286E,0x20110427F0256,0xDC2FD40F34446091,0x857A99B0A93AA21F,0xBC2DA259AF77F691,0x24A763DC85AC633,0x1AADAF1BF5D95813,0xB09FB1D12D9828FB,0xC81610C1FEED,0xC8BDE0E7B1BAA639,0xE5D47327A23F994A,0xB608D08200ABAB11,0xAFF071DC1FFE0381,0x7ACDC15A5BE5558A,0xAA3F94D433A778B6,0xA593879DF50A,0x972D32AE1F1D0C54,0x6C4C1386428A874D,0xEFDC02FCA2B1C57E,0x98FEC9F150BCBDB7,0x4E00631AE9064,0xDD1B278130A99049,0x2F76EA5B7097,0xB2495881F978EF98,0x1183B76919B0ACDC,0xE43DCBDA3692ECEE,0x4C397152BF2FC1E2,0xCCD10DC1AECDDE74,0x300E0A2F2DB03B93,0x16591C5A7B426,0xAF1346743AEFDF2,0x120724EED8F511E6,0x18DB81AB7BFE3F2A,0x31F9415AF3FEA6F,0x7241675D0B7813F5,0x4334E574BEF027AB,0x697E946D4273,0xDB6E7CAA695DE8FE,0x76780C9B0C6376BA,0x4422D2C4B3FE422D,0x5C6C8372783A6161,0xB70CF4F39EB155BD,0x47F5F15308B05B3B,0xFC6BFDD5A3B0,0x5C9F770B70DE8331,0xF2622FC9BD22BDB4,0x8CEFFD7AB7665591,0x455EBC78AC23643F,0xEA2DBD29417ADCA4,0x4712D7141BB2CA8B,0xDB4FBF60D970,0x38E6157E35F2CB8F,0x3866FD31EA04B738,0xD11BD2F70A745F6,0x13A97B197861906F,0xB19041A71D5C372F,0x992F0B47DC711B02,0x646332028466,0x85BC7AE84CF2C04C,0xC7CDA0B86677927A,0xF88048DF421AEA5C,0xD136404A6B5E8E26,0xCA2E8C2B5B9CD583,0x5498FC2AA64BEE5E,0x186CA7CB911D2,0x59E13451A9CBFC14,0xF3BC74A0EA7B2899,0x2CA65A43270A7119,0xA903BB8303B36303,0x29799146D8019F6,0xD05A2F5CC427699E,0x366B08B954BB,0x26E173240FF300E3,0x1A529688A392BE7,0x7597ACDFC815832F,0xF8D7C0A9DEB8783A,0xD30C5E96DAE657FF,0xD82B537A1F26F64C,0xC1222A77CCB8,0x2944C364B05DC02A,0xE2322AD2873EC495,0xC17A76DB90297359,0x4407FDAE8AEAEB8F,0x81C713A27B790385,0x6B6722CDB5C98343,0x16DC3F5B3CF0B,0x24E3707F5CA5D83A,0x867BDBD3976B002E,0x4FEBE139539998F,0x2CD0000FBCCBD3C0,0x45F1695F0039E002,0xDC5311C8C16F17E3,0xE4002CBCFCA0,0x39FC6BB2C214DE28,0x8F1CEDA3A4713405,0xA0EF3116D15988CD,0x7085EB4980EB5A6E,0xB4A6E6633EAA1FEE,0x995153FEC491DC65,0x2B37DF7E4D10,0x24B0D641638BD998,0xE7EAD716E43F6E29,0xFC73EDCD235F61E2,0x4798D4223A44672A,0x95FF2BDBEA70F9DA,0xA534A49D6B082C33,0x160F307759A1F,0xB5F76080AB80FC25,0x2D67570F46E0B10,0x5651883A4979E4F4,0x6E7D291CC7B1CD26,0x38C72A4F70A8BCDA,0xB0F96AB3B69EA827,0x1C9CA6BECDF7E,0xD0AB519CBCE12B9E,0x970C4871727AEBA8,0x8420A6C60B992CAA,0xB7AB00A6D54E8F9F,0x36906344EF88E016,0xB06153C84F5153DC,0x15F0263200204,0x947D3BF12AE4273F,0xB9A3EEF5C875361C,0x284625DB0D2133E1,0x5CC38FB493CDC07A,0x71A9420E8A8F1793,0xE1B260F50F4F6881,0x72B718AE0834,0x9114E93B8519CA92,0xCF05532E12D2E27F,0xE19A72906548F476,0xBE7C1B3A22ED283B,0xECE224A00354ED01,0xFEEDC2BE3885783F,0x2102030FF18CF,0xF0638805C1648B84,0x11C11F041EB94A39,0x8FDBC1975D96510A,0x842A86277D88BE11,0x2746337DE50737B2,0x29DB1393F202C78F,0x92DEED39232C,0xED23597169FD6AD2,0x1987E66990D89E67,0x517F18DFA2A1C894,0x906CEF66D99C266D,0x843724DF0544A69B,0x8F93DDDE788BDA05,0x19100A6FCAE0D,0xB78529C97C1B9D1D,0xA2FA1D041F03CDF3,0xC306AABEB32D17CC,0x902DF2B861006A40,0xFAD36C4AEAEEBE09,0x21D201A5D6C7B0D,0x19C232776B4A5,0x8D0439A0B7BEF8ED,0x96F939776F043211,0x5C2D209133653AD6,0x4D62838B446BDC78,0xEFB52824F7555C5B,0x49726205E0B243B9,0x7D933B77598,0xA7EB468A88159E08,0x7482B379A955232E,0x5089FBE36FCC14AF,0xC1AB095182B74587,0x9B7F40C899436F55,0x177061C6B102C724,0x201F19D2E0D7F,0x96F7D0952CF03BA4,0xBEF471936CDE93,0xAD6699D02C6FDFD0,0x191742F8A44FF8CE,0xD0DD627BE54CFCC6,0x471CFAA2DDB91F0E,0xB23B7B3A6ECB,0x531E932729049C33,0x167BFEBED836B594,0x7872362032BB2B9D,0x24E7CEEA27C682E1,0xF37987AA3F6B5BD2,0xF914BD046608BFE8,0x687FC018D77B,0xD3CD014DB630B6B3,0xBF9C1FC22A8376F2,0x7D264855527860C6,0xE6EC229E1C21F041,0x3662575ABE9A3414,0xB877AC931F71C0E7,0x1D8FF2B97BB16,0xC0EC2ADD71FE54E5,0xFCC59F0FDF5D87BD,0x2E54962D8E3E578,0x7560EAEE85D16A13,0xFB08B6423A000879,0x6D3992268C2CC613,0xB99911FF1AC1,0xD4A01A5D1CE22D75,0xABC2590AD6D13193,0x73BBFAD3093793A8,0x2B4AF6BB47B12F0F,0xD1776BC36EB540E4,0xA9F3B3D977EFB8F8,0x3560B7E97399,0xD7E975EFDBC42532,0xBE7EBE32CECDC504,0xAC88BC71193B070A,0xDD1A7CE520B8BA88,0x31E33F5865B636BF,0x428A5F1DE0F7064A,0x1AD05C2ADC23A,0x81F61E90FDEE22D,0x9BDB22D3DBF4E904,0x172B4C424EE7A93B,0x595531CFB6719F51,0x90C3FFE5A860E188,0x766F9C816C34F80F,0xF49A34924046,0x4DA5D5474836A32D,0xAE85D5165BDA380E,0x7B6DE67A451FD04B,0x408E3271F6B5B569,0xFC509F10D8998DD7,0x45DEDA7A023B832B,0x18E1EBE6CC26A,0xDC624CD0159331EF,0x47BDE4456CDD494C,0x35CDB385FF3B375A,0x833B80A92F899D0B,0x362057F03D11B6E,0x37BEA8973B8D8C7B,0xF6D974CC3EE1,0xEBB89A768FC89724,0x5BF7DB48E8575EA,0x36944A271FC12241,0x37E5B434875E2D9,0x40061D52A7129E72,0xDD0B4DB74CFB499,0xF2CE5EA2D1B1,0x6F6854212D926629,0x1B48496F573FF254,0x3C58708AB076C28F,0xA58E58F1BA47C9C5,0x8A6F269E8E150478,0x2A6BF95D3E16BB8,0x12E3D6ABC066C,0x7B997494716701CF,0x41945D900222B074,0x6A6B3BE957DC089C,0x2F81DC94D0DCEDF7,0x3F7E878FFC19D1B3,0xB3E93F0313270A80,0x2269612929807,0x636DE284C66ECE9D,0xB3E91BB39DE0A69C,0xDB02BC6650FAF782,0xC25447AAF53DD803,0xD6E0E79E6ADF998A,0xA7ADA45D03FA5F8,0x116C95797CF45,0xB71768CD629BC746,0x9D9A5DCD384E6D27,0xB8C38F254642448A,0xD2C7C73DAEF347CB,0xB37252F32D0C7421,0xF96A46B61A3862E8,0xBDED5A29F435,0x9C70E45C25CABAA3,0xC50298F0F354923A,0xAD95FF2BC8130204,0xE3D5C75C05464412,0x11ECA686864000E7,0x24480461D0CCCB12,0x1ECEB6464DBE6,0x96A7C878BB1EDAB7,0x40DEF16EE88927DC,0x47A11F4B4553DC3C,0x2EB5200C5BD46DCD,0xF11216909AE257ED,0x66F810F7D0FD0EDD,0x86C7EA89C724,0xBDC78AEDC89E1946,0x2507316DAD0A1760,0x92576B5A54C69E0E,0xF9DE8220301A0A2E,0xAB12032BA97BB634,0x8EAF464C50BB5DB2,0x983765B8AACA,0xD8005AD4F804801,0xE058FB63BFCF106D,0xDBF519557BE9F8FF,0x9F530927320D4D7B,0x6905617CC9DF8585,0x6AB39D33D37EAFB0,0x3B77B2A6F32,0x567B378070571330,0x9DB3787BB63D0303,0xD4175E3D6A50C5E1,0x4B5646CECE8C5442,0xFA33B4A79C414C44,0xDCDB30D98B8F948C,0xA0077B05D7AB,0x897EEA13B54D6AB3,0xBCC1C110541A25D8,0xD2F1D2FF4081C594,0x50B2B6126797689E,0xD6645C1F9E361219,0x192C50AC1D580066,0xE2B8044B69A1,0xD227EE067C9228FF,0x398A95CFA1DF1787,0x8675E243261FA767,0x391999E7E98E2FC1,0x7260A3AAEFC4D38C,0xA30F88EAEBEAFC08,0x16E680BA81CB3,0xD6F558286D6D1C26,0x7C327190A872EF5E,0xC49F5346F70302F7,0x600E595F6643AE4,0x1CDDEC6C70AE5831,0x48F58C7939077EEB,0x10151BB268143,0xD7228AF8BB8E730,0x7985B965BF4EDD18,0x963FA8E6B4640AB,0xB45CC5EE71161CE5,0x1D8E2BE8B954F2F3,0xD42AF0154827A033,0x8EC70D5DF042,0x8866F2DF175C0BD3,0xC3E9B251B8203A4C,0x2095C095934D7AB,0x256FAC5CEDE39A08,0x1D0FE32E85A9FB2B,0x45457A4F36E03F63,0x9C11A97CD941,0x5F937052BB7B7069,0x4F6D88D0C3D49C7D,0xD5E07AE405E40016,0x1C848E15EF14527B,0xDA4847BFF97592D9,0x2C9E21BCE151441D,0xFDD63D59BAA4,0xD7B84672C191EC5E,0xF0793167F5641EF8,0xAFFB66DF5C034BCF,0x92B537D95D289E03,0xD8B2BD07B8A655A,0xC206A12B3EFADBDB,0x1B4C4A784C2BA,0xF44B63B96369A385,0xCF195BC8DC712FE3,0x76665BA9F80225FA,0x8764B3C5400B5893,0x6A71AEC475B579CE,0x5BD5F48DD10535B4,0x20E09A692ABAB,0xEFCF26CE68C0C9B1,0xC27CEA271D875FED,0xC974D69F7BF52A2A,0x7C1A2D9ECF7B7E71,0x6743FE73C8F9A780,0x56DBED5FFC683A90,0xB8C4466581C8,0x688AD43A797D0C74,0x6DF256CF1FF0AAFE,0xB696466371BB5762,0x90985A758748703A,0x1F0F3D052BCC89CB,0x780AC9D8FDA74046,0x13C02237B1120,0x7434AE2BCC0DA77D,0x4E16688AB1B1555,0x1843BB89F78C5337,0x7ADFD538B3E10656,0xEC6E762729166C40,0xB52D6FFDB09DA181,0x116D99DCC7BF6,0x2DB9969A25475946,0x11234E267F8013E7,0x4534C989C4EEFAF5,0xB78674DD2E8DBC23,0x13C87CB3E2E23E9A,0x7F7262CA6F63FFC2,0x1C0830F70140C,0x9591C582189350E3,0x5655D921F5E14D99,0x5FE29F23223A2237,0x38DC941C6B005D83,0xE193C91233855D92,0x3605D4392B5E7AD2,0x1FA62E6D34CFB,0xC7DE4DB15CDCB9D,0x44704A69EEA41F31,0x7FC700BF07FA5E28,0x5D61FA6AC409EB22,0x4033734C53B49E4A,0x90F078A1818B99C0,0x16C6F17B7AF83,0xD6C73DF7F9A14E4,0xA82882BDD2299B06,0xC55D711CFAD8B1F9,0x86FEC7D7C06490F6,0xC9B08CA66BE8931B,0xF6BCEBDD0999E259,0x19AEDC7120E4,0x5F5C338BDA36F6DC,0x1DA2FB3044138428,0x7506A757261CE364,0xDE8F082CB207BB82,0xB3FFA869A40ABE55,0x16032E76B65AAA09,0x229A92F293F33,0x3DD7EBB34CDAACF3,0x124CC111F9C5B56C,0x29F3385182726105,0xCAD5BABEB87AC9BD,0x91B4A0DAD81A4496,0xAF07D771EDAC948E,0x10BF45902CDD4,0xA4354FB46557557A,0xF0A839CE210D6976,0x47742B7931007233,0x121B8E746086DDD9,0x70FA79E8EC06A888,0xBF9B52C132DDC8F2,0x1E67759505122,0x40C880A34B0BAD26,0x847B1C9FD259A3A,0xED39541E937F52CB,0xCACD945B8B5AF23,0xD1C0C703C47553B2,0xCB902D4B90102100,0xD74B2826394F,0x25D73CE741F11368,0x668A44B950A0B31E,0x936D6EC7011D0B71,0xB47CAFF6F94A226E,0x85FB21E9B1524403,0xAA4EB71A1F2414CB,0x93B431756B9D,0x39A5DF70C8F8D3B9,0xE0A67C2A26D66E5C,0xA4B64C22F1892472,0x933710592A159A1F,0xD0F74685856AC788,0xA7A285F98245C4A7,0xB0C9C88B01AF,0x4A4DCF3A437ADEB0,0xD0945A384AD67351,0x6338E2A5280215EC,0x9ED08D1E0645EDD8,0x5118A311B7618F28,0x9E742BC8A15AFEBC,0x11E94D9CBCE31,0xB6F7260F0D2EC50C,0x8C16C335A0788B5E,0x4187D533017E9F88,0x7A0285C24F2B3EE7,0x4861931A336C69DA,0x40822BDC7AAD9A15,0x1B9316667023A,0x2D0831C265CF01C8,0x8647FF51B9F9B9A8,0xEF9BF6CD31463725,0x1258BDAEED556174,0xE2680C9B573401D0,0xB929BD1D9C5D4D52,0x23117CE03134C,0x60779DAF4F64CE11,0x9D9A3AEA37D1DBC9,0x49CBEF970AD0D782,0xC101D76D9C3E2AB0,0x8ECDBDF6BE660651,0x61F38982CDFC3BA7,0x49E641DF9909,0x3DF877EDE686474F,0x892D0F31F6041AE1,0x95DE46832D8348F1,0x4FA27A8769B88843,0xCB362C1961DDA761,0xE6BC9DF2095BADF2,0x1910EBF4C0548,0xD9D901171EDDC6EB,0x76D454CA4EA58188,0x2E1A4B118DBA4B62,0x1F59E15E3435AEFF,0x57C071915292870B,0x2BEBBF71CC803E5E,0x14C6DEAFA9A9B,0x269D2213AFE61828,0xCDE63325D2DB420A,0x3CBC0CD731C93E1D,0x4BB3829613C92480,0xCEA4E3ED7DA29B0B,0x4FB35394B2A137B7,0x196B2FA14D4D0,0x7CC40D915DDC4BCE,0x2F43C0ADA51643F4,0x998DE241DD076287,0x28B4EA098CBC9605,0x44F5E0D274166361,0x8557C2C2BA559DF,0x861585FF9E24,0xBF441C5DA6D4CF32,0x6B7DB418FDBFB536,0x2FB9F3EFE2445BE2,0xC1C4FD4E3670D12C,0x7D9A4156058C38A3,0x38D72C1235A97437,0x19BFB1DFC4F07,0xFB5D7E397FCA37E1,0x8437C36448C2DF47,0x5130A204DC0CDEF8,0xD42701B210EE3E5A,0x8602206D2C49E71C,0x7AF042D155BC5DEC,0x21CB090AC055,0xF82CC93A140E1D0D,0x38EDDC1A8F6A2749,0x21C26A115C20654F,0x3448F4312B8CD47,0xB74383E301A53CA5,0xB1420F9A92984949,0x1EEE2EFF064B0,0xAFE964433BE893FF,0xD4AA932AC981C620,0x6B690FEE1880CC14,0xAEB48FD1248124B2,0xF563A102BED41524,0xD9E0B499D137B880,0x4B45BD575376,0x5BE060E58E3F7DFD,0x607DC5AD258A2103,0x183A33BFFAA7A857,0x8C5475C782644568,0x111FF3CF9AD73713,0xB330D7425903E61B,0x16FAB9DFB80D0,0xF39E3C5020EA3B9C,0x4FBA36B855DFA26C,0x38331F719CB0B0A8,0xDB9420EA58F252C5,0x4529255376662427,0x35A6CE922B10C3F7,0xA50BE8FF88D5,0x84439342C38A98ED,0xA98E483FA39F7CD,0x941E4D2E436AEE1F,0xC9A5BAD761F6954C,0xD98DE683CAF1FE03,0x4A2133972A31DA32,0x191C3451707C2,0xF8951463B4929007,0x1D2BCF73710BBC5C,0xD2D152C01D028226,0x2A8B7A57DDAAF1DC,0x2718ABF21E6F776C,0x4B7CD08EE272C95D,0x948E5854837F,0x2E3652A672CE3B37,0xF8FF08579BAB49C1,0x50BFD4948027ECA8,0x5A45616C7AEE384C,0x3C6CC9853E9C5849,0x98F280152B900E0E,0x1E6D084451421,0x606D5E27C884A3C2,0x640BA774AAF57740,0xACCA4CBB219515C2,0x727C74F67E5A9DEC,0x78ED71F2185A349E,0x67BFE0852EB9BACC,0x109BB201E7939,0x621207936A3C01AB,0x76B64CAE9DFF6F2,0x211351B70AA126D4,0x9377C0BF444C945B,0x35ADB51DAD723C69,0x5929E31C8225711,0x1C2812F844174,0x60E60866049C2603,0x4145B9E3DEE77B8D,0xB9B1605568DAD3F6,0xF29721F18769C8F,0x30A74A0DB5186193,0xBFBB1E8618CD4536,0x59AE09428F5A,0x77638BFB8EFC1172,0x8F23948A56937313,0x489B1B17EC532982,0x584507B5879AB4AB,0xD02298BF65C6FCDC,0x43C553FEB4C98D20,0xF05637A2E6DA,0xF6E616EFBB25C1C2,0x9F175EF203BFBDEC,0xBEB19414EBE54ECA,0xBDDE40F6AE76846F,0xDADA0BD88123DE46,0x19D7160B1C0E14B7,0x128F2F986D8F,0x8E1E5890EECAE8BC,0x6C485261398CB132,0x7E4DE37D8A8B15D6,0xF210C8E2B874DF1F,0x52A353A7C4D49F2B,0xD5F3C5BEBB6DCB4F,0xEDBAE54E0DF6,0xCADEA7F50624BDD6,0xF1511E569AD59D8D,0x9957EB319143C99B,0x76BA8083F5080FAC,0xB8C798E5BEB4A189,0x380FBED1AD9AAAE1,0x1F556FF5337B1,0xB7D4CB9D5EC3A289,0xB6962C9DC550BBAA,0x70FAF7620F4DA69,0x4FC408CBBEFD197D,0x62F3EB31C964B6BA,0x25C15B846848BCE0,0xA794C7B283A8,0x15E671537B82978B,0xC8F93EA779B2D61D,0xA11C209843A55DC0,0xBC97A76E0EE2F027,0xB68D7396DBC3FED,0x9A4938377B1F6ADE,0x8FB40E390C1,0x22F0C49C75B401E5,0x3B958830ED9762F1,0x6F53589AF61239C,0xA0D5FDA566A77AF4,0xFE740B57AFBE6B29,0xCD7C8E5557B7F5C1,0x19B0EE0ED76D0,0xD7D949125D41CADF,0xED747666A30CF71F,0xEF37CB05D30B12DA,0xC5DAFFD6681E5E4A,0x5AA73DF2D191A97C,0x6E2492A3BF57B5C,0x1D392A5D8E5CC,0x1E7208FB1A108004,0x4A158F72EA518CD5,0x6E3A0BD925DDC469,0x57DF2F8258BEAAAC,0x6618B5B6B099C81A,0x36ABA934111403F1,0x215765EBAAAC0,0xCF90CF78B98A021D,0xCF8E749D08055281,0xC2A03263F2E1CA72,0xBAC660D3509FC570,0x96257150287DA6D,0x7604DF89E18EB8D9,0x1980788C52BBE,0xBEA0D78A1AD416F9,0x1F93F9A3FDDAA911,0x37678238E0C3776,0xE52CF324EB93B75A,0xBBD3B9359D0C974,0xD87498AAB9952C83,0x19597164EFD57,0x168C7C08FD0638F5,0x512815EF0F09ECE3,0x5FEE261FDF0C4DC8,0x5026C5F538BE32B1,0x661AD44646BD010A,0x1B81A47A9B5BF56C,0x17D6E5AEF73C6,0x2840676EEEA61C84,0x667A23BE8BD05B17,0xE69E96C3E1750863,0x371044929F82D743,0xF71C905B74FB1A26,0xFF661F6D58D0CF30,0xADAB4F930271,0xA3E7D2FF0BBEF097,0x3418AB9FD626E998,0xE0D702853D3202A6,0x7632E44073D1FB73,0x7A6C974037D53902,0x36717A97C597917F,0x11007BD4B2856,0x7EAF980FB28C4CD8,0x27E9A2099C755B1E,0x9749A7D090FA00F5,0xC3D23AF4A8B1100,0xD94905081A25C29B,0x570ADCB990AC7DE8,0x2CCCB9A0B9C0,0xEEE13F58585E9A97,0x629F645090E83074,0x394929D19003DD66,0xCFC85B4F465BE0F5,0x1E17BABF34A3B997,0x1C6D0E428D7FACC0,0x1238A9FF458A2,0x1189D1C521D9F0EE,0x5F99BA77BF461E26,0xA5B6787334B7B948,0x759DC916C70B68B7,0xDFFAB86B121B422D,0xE47888B90AA22402,0xB1D97CD6F070,0x11689AF0F1F44A6E,0x9182F5FCA5F526C8,0x8B891433ABD380D7,0x7F0642B49B36366C,0x90BFF49A9BC3F0BA,0xC928316D79A5768F,0xB1E02E69101A,0x355BED08FD7BB9FE,0x94000E87C8EAEAF7,0x603223C142FD519,0x1589C4E67E08A2B0,0x60D4C7E8A36E8AB4,0x5EDDCB0DCBAF8E32,0x1E6AA9E72FFF6,0x43441F661D611094,0xD6DD653ECF6351CE,0x52EC1A6E39D5D5BD,0xBDCD21F126E3004F,0x42104C7A2478961C,0x17D339706A36C877,0x139B3342D48C5,0x35D2B8BBC5A1019E,0x6D84194D7EE2153D,0xC3C5C1C5017F2077,0x52AAD05DA55C6EE3,0xC4308CF5CEBCB9E0,0x9EC259BA2711E492,0xF05F942757E5,0xEC1ADC416300EADE,0x4574970F12D4A873,0x49C260B3BB18AD94,0xAEA566B143D38C29,0x72DF3D8860CF92D1,0x7464BD9D35034EB5,0x13C0CE2C146A4,0xEB87A6538B71A430,0xDD99F9847F0B2B87,0xB980C42B103249EB,0xA96E0E5C0D9F88CF,0x39D7D9BB6E5FDF35,0x8CB548AD7E218B2E,0xF03901832F4D,0xB659FE9BB927DC4C,0x94F02A04EA1224A4,0x93BE841E411A4305,0xB87A814CED357D9C,0x72F1ADB8758F2FD6,0x34A3EB9D278A7888,0x4CDE0DF95E3A,0x4475D9BB4042E344,0xFD5EA9272A9375E2,0x78E8DF894A22A4AA,0xD93D22995655090E,0x552FF88B55653A69,0x972A15474FA4B6AB,0x8190DD6DA589,0xC937818F072C4C88,0x24FBB4B7D5B85C33,0xD6F9C09443F80D13,0xAAF9B4BA381D70A1,0x9EED0D779B175710,0x94316E84E1B9573E,0x210F377BA8743,0xF24FF57223A1F761,0x2B8E2E8A817BE364,0x9C0122BA9E342EDD,0xED4AAEFAD1619062,0x508C3400C0BB73BE,0xB5613374714A2B28,0x11625C484A29D,0x180A52CA60EE16E3,0x85D88FB3D4E7F9BD,0x751822E2749970D7,0x8F141818E576E6F8,0xD8634FFB003AFE7F,0x4A52142BABE21888,0x5B7A8DBB0702,0x4A9FAE311B872277,0x5053C5AB0E35B428,0xEBEBD7827E9FC29E,0xC4590ADB12F24BD,0xC30F3B7E37E27135,0xA5C44D8D9DE5C468,0x556CE81AD2DD,0x36066EE3D62C212B,0xFE1EA5D78078EBBA,0x3B7C86DB8D7F7915,0xEEE231ABF631217,0xDBC64825272BD839,0x530741801B3C91A9,0x13A9C167843E7,0xC46BB040620BF5EE,0x9605392771578746,0x55E0FA1FE15E595E,0x775217CDF523B7B5,0xBE35CAF2EAF3EF6,0xFE5710BF1A5E30F6,0x14D066107DB1D,0x28A73813110F8CE,0x459A2CB2B8F0600,0xB394AF578D3FF64D,0x1D93568B1A08633C,0x17EC24587E77CD7B,0xE5D81083E937052D,0x10AC95DA87556,0x532CDF6E3FED03C1,0xDD62CAAA9269E049,0xAE4EF2F5A329318C,0x3B2BBF2A55CB6DEB,0x537F1DA82896B20,0x15B2060ABEA734E,0x12784F9F084A2,0x10F944D5347C5280,0x998C70B87D759DBF,0x82665F73E54A79B6,0xAE079975FD435FE3,0x974D05010F2A0254,0x4025DCCA0AF3554C,0x11051B7F995B2,0x69A277BE7045C53C,0xE65A54254A92A66,0x2B2D4DC2D8FA051A,0x71399E41D23B328F,0x158839DA8D7E3EA6,0x160208393612030C,0x6260B1D3D771,0x39794B1BF29D1909,0xC7A6A2E3E1AD841E,0xD3808657789020D5,0xC8E5833C2A22080,0x6A28F58AD598B36,0xF8530692C128E474,0x184EC50800DE3,0x2AB5865EE64510BD,0x3ADC5A8820AAE1A7,0xA7E33A72795BD704,0x1568ECFDAD0AE541,0x9B4D54EF0F4D073C,0xDD19B2ED4FAE6FCE,0xFA966EB18D29,0x6DD153831B23B6DA,0x9205B27A5B86B2B2,0x59C7EFED84131878,0x60A6B0B0BE13E104,0x573AA91138F30A49,0x8D0102053CAE2681,0x10035F7845622,0x1D5A6B5240EAA614,0x8E1382178DAD0CE9,0x8DFB211A42272121,0x773B80357E863913,0xCD54007C11571ACD,0x959B4A7A2FC52B6A,0x204C19E5AC9E6,0xFB63FA0A9D0B7A49,0x3813EA3725BF80CB,0x5C0DB1E25B43E35A,0x8D9302BC3A7CED5,0x9C4EF5D38BAAC458,0x56527E9FE07A2459,0x1A060D4241E2D,0xECF6A91292689E68,0x20F3AC188327427B,0xA4A3270566893DDD,0x3D648A71249B0F05,0xA826195E22C0E4D4,0x7F666BA1DBFD31AB,0x1276F98A29A71,0x8B0E38690DC88C07,0xCDA0D6DFEEAC7390,0x3C8D9E147FC761EC,0xF06CB113B0AD56F5,0x9D581256B9BFB386,0x7686CDF715219762,0x1780F87A9C0E2,0xA0C5F70AA604E6AD,0xD5AF3C9A7B649B03,0x8EF1FC237FBD7B8,0x5EA5111E35D526BD,0x2BF16C5A99B76779,0x65642210704DEDC,0x7452CF9B9D9E,0x627B4BEF3889BB26,0xD36FDF72F8CF1CE3,0xDE7C5E550CF8D086,0x120E7831A48E59CF,0x4554195EAC5779D6,0x1B18C63E1B30DE1D,0xD34495BFEE3D,0xCF45A904E91BCE09,0x585D386BE3119EF6,0xAEE6292C30FA8B7F,0x698D0A8AA447F6DF,0xA7452E45B26EF81D,0xA01CC28ECF20989F,0x80F3560E5D33,0xDF9DA4CF31D2D60A,0xDC22C680CAE08D06,0x9EEB361FBE04A3B,0x7EC878F235DDC11E,0xAE987EBF22832D29,0xE8455E68EE76BF6C,0x1A738EF06AEB,0x21C67B7EC90FFBFA,0xD90B0E8E7FA9AE95,0x44E2BB66252DCB7,0x2F63C5CE0A385C7A,0xDA41BB6E0C1A0D3,0xB9E081DF7F378A94,0x3106F0E62F19,0xA79A2527B7DA91C6,0x92F2CEED95D707DA,0x7AED6E603862294,0xE832521B8CBA8F9B,0x910C75EED2C825B2,0xF3A7B9322214A3D5,0xEC76811F770F,0xA9E6F9896BE6C8D1,0x4FA6073E82281AA2,0x64FD58B90DC4D435,0x4551B52901BE9F27,0xDEB82327A4296B7F,0x680D883F6EF91FDE,0x170E3A06ACC3B,0xB284EA1F67C7BAEA,0x1C5869259894F879,0x62F43BAC11507FDF,0xC0A2CDB6096CFFEB,0xB80C89ADCBF38FE8,0xCF67432187251709,0x2D74ECF00FD7,0xE084247F403EB4B7,0x427785CB80D65A92,0xDCAFFB9401D25C8C,0x5C835695880B361A,0xE95FA77E49CB9988,0x514B97FBBAC3A4C9,0x1E1DE0134B959,0x66C58D3BE396F341,0x2AF6EA7788C46069,0x3B6541B120928060,0x8AC6D64E2455E268,0x3AA9BDDB8B766AD4,0xA1B2F575457C8272,0x6EB75C37DEB4,0xEA95302133114F08,0xDC4617102CDC64C7,0x60C456FB372BE44,0xB90DD6D56DF6CD41,0xD23872BB18FC1A3D,0x531D9B8FF3CCD9E6,0xDEFFB9A976EC,0xE3009F8B55BEC82A,0x91A7E9257880CDD1,0x4BE9E288673FA852,0x68ADD3203C2C1577,0xE29EDF34F207E2DB,0xD3898B444CEA5477,0x14B00B3B03B36,0x6A9357E9DF80FEA8,0xDE1804456C09C7F5,0xD1F65385B82A6CA1,0x487F5C8D48042B38,0x34018BE8BDF29CF9,0x14065DD162E01CE3,0x1BC0430B46664,0xE5A4C0223A7E2A36,0x2AED26C535019DC3,0x2B9118EB35000BB0,0x2293E9DB0D8777DD,0xCF5CCD0E590261FE,0x18A53BD77BF4BBC7,0x19F450C42843F,0xB3CE243598353115,0xB394420E270A2E92,0x20862B9521AB6389,0x9AF51328EFD30D85,0x726660514664BEFD,0xFAED4162E3ACBBEF,0x1A22272FD322D,0xB3E8E4AAD70E2479,0xFF683DB32FC93288,0xE5D4AFC7F16DB12D,0x8D963521F4D20765,0x6A14913AE8BB6A68,0x614FD14F3ED18BC7,0x13FD9AF6118F3,0xE9BB5A6ED9729F77,0x41DF3DCD8397A1A5,0x8F89193D0AB2ED71,0x61C66ABD4B21404F,0x221195969AA4E05C,0xFBE0D4E7821A3AF6,0x839A9C93E018,0x6E64D43A15947672,0x2F2626F68E0C2613,0x555AF0E146EFFABE,0xFD86755FCDE2AFF6,0xF63FAF17A07138FB,0x93A35E894175EF40,0x14ABDD39EEE4B,0x3BF383BF1D3AA573,0x7BB9A79B7AD6314,0x64178F71EA300451,0x1EFBDBFB13D75B1,0xDE1FB519946244D,0xF7BC99F68EB8DA74,0x17299CDC313A9,0xE500085387FD8F0A,0x10737F09D8E1391,0xF4C574986AC443E5,0xEFA2F01E1ACAFE00,0x3578653A0CE28CCA,0x3B41DA89AB2FF7EB,0x1786493C08C1F,0xD7356292AE120CEC,0x93653B1A029701CA,0xE76F82D445443E1F,0x6D61703B496D2BC7,0x54457F15C8ACB9F0,0x1DA8288F48FFCB8D,0x1105F14642209,0x723556DB4AE3F8D,0x3A061A967FC00EB,0x5F84F34A176EA7C8,0x13E833DA3CC0841,0x627D4459816CA516,0xE8EB5769190C3702,0x83AA284AE112,0x3E6267BC774B923,0x5277030A4A1239D3,0x3A0030370FCF05F3,0xC6621947968A0D6B,0x628FDEB8AB66231A,0x29BE3F5878401799,0x11DF91B775429,0xF44E86F8417C7DC,0x42787E2414C678D6,0xD2D5BD0F109B4C65,0xDE535F260C70D941,0x9310BAC5DC84037C,0x148060B7840F701D,0x106C65A4181F4,0xEE55B957620CB5C1,0x55E1166F0B57B7D6,0x77A405BF51CE4F25,0x3EB1349410E014CB,0xE25AD538613BD115,0x52FE81D84C2470E3,0x982BE4285B72,0x9C865941AEC9AD25,0xD9DD6942F1538C8B,0x8CB3A51A9CA9473B,0xFAA9FE4E18265FD0,0x5DABDD06C4C50B24,0x1DB098B9A76B1632,0x800AE5A9E529,0x1BC707105953EF09,0x646E416ECF160E46,0x1AF50785636CB7FC,0xBBB3094B88AF9DDC,0x64EE343627235580,0x121A81C260B1DF31,0x1C553CA56AE57,0xAE7FE94786A4C494,0xBD149244C10C0148,0xF5F4C3112E7008C0,0xCB4FECDCDD695806,0xE02718E4B9092EB6,0xB14DC9ABBAF84E08,0x1221DE60D111,0x48B2DD458D21C58E,0x4FC2EF2E46305264,0xBA6336E2B50DC468,0x2FB0106F3FA5FB3F,0xE87BAECD597BD335,0xBBB0EDFF19A0C876,0x1EF2ABA4C7340,0xD5EF7AE21B9ABEFF,0x80A299F8608AD207,0xCDFACAAD71ACA129,0x6D67A4EBB3E13319,0x6EF921E4507BF6D7,0xD7103C9E40CB1519,0x128FEE9E919AC,0xB7F5A65DCA063AB0,0x8AD8EB0D845B15C3,0x2992D04893C7D982,0x3A4FA3BE64830D60,0x389FC24092D6E4C1,0xA7918D68BB357EFC,0x194E1903D65CD,0xE3B916CBDBD39506,0x58A4B4AFA78464EF,0x4BA018E826F3105A,0xCB63805393E0DB28,0x772B24965505BAC0,0x9F4490D66B71B41A,0x9D0603560CC6,0x589B15D879D1BDD,0xC5757CC100B06F9E,0x560603E1B5CC688F,0x1C53AF49D7C7F7EA,0x2F7DA4B1E0FE59C2,0xB7246B158AEA5BA4,0x1C5C3284E3A1D,0x5747CBFA67FB0BF6,0xBDD96F538500B095,0x3B031FDCC2B4667D,0x21614CE2B0A27969,0xA12325167CE5ADD,0xABF08DBED3BBBB2F,0x16610954F9E23,0xA81D391B48F4AED1,0x5B2EB16E4E73831B,0xB75C2465BE239,0x19D38A17F93D40A4,0x62FFA5F216D79968,0xDC8D25F1A4C4AA3,0x21A6C55FA0200,0xA067665D810878C4,0xF08C8F61B896E7F1,0x38876433424B3563,0x65E81A5D0ABDB9F3,0x1BA11DC5D7449D93,0xE2609F60380E6505,0x1DE0F8C286776,0x9D2ED0B9BFFDA9A8,0x7280F663EF99BF97,0x5B9A9684A4230E26,0x45623485877112B2,0x20D34CCD8D34C055,0xA99FAEC806D400AD,0x94F218C148CE,0x35320879258D281B,0xB9BAD39419393798,0xBEC1F84599E93267,0xAF7D33B4494A6AE,0x44D8FBE144675DA8,0x162CDAA02B9199AC,0x4D1282AB93C8,0x47DCBCFF51C1108B,0xC7BA4ECFE1B70442,0x5A69211C6A2F4999,0xD02158C69F7C491A,0x9F154D72D4E7E927,0xB4780F6EB64438F2,0xF2D3CC4BC008,0xC585C31ACDF75F15,0x27AEF5C9140170B6,0xEE19086E5B6D0FC2,0x459FE7A758AEA172,0x464C922A87C7AE8C,0x1AAD2756DE22286A,0x2058561DCF1D7,0x22D33C1827F0FA12,0xA972F7AB4E5E7DAD,0xA18A447BA5560CE,0xFA2846360AE9B3C,0x40162C7D1730E5D8,0xEC87C83040A13683,0x1BD90BC4B5C5C,0x29FA4C5DCC44608B,0x24DAEBC77417B7CF,0x4C253DC60E7AB933,0x260CA68A4515CE0,0x192996D60DA8F59B,0xA90103BDED2D0E1,0x219EB679D9A17,0x17AB8C3C0D3A3152,0x2A00D6647C8BC8C9,0xC274BC857AABF6D1,0xDE5D622A78114CED,0xE1029D3E448A5F4D,0x2680532FBA77486B,0x1D88034DEC60D,0xA94076878F5DC53D,0x37EB828ECE9D23BE,0xAD3F6CBD35639B4C,0xDF32276CF971921,0xD90787E6025C608A,0xF7038C9A82996366,0x102DE083D9254,0x2400E370B7729B1B,0x26ECD7FC92F71C71,0x841993E12C5644A8,0x691D65765393CC28,0x4A0105CEC34EF466,0x65B4B79219DBC296,0xB1C0757CAB50,0x14A969BEC5DB1A59,0x871FB1862F336291,0x907B04C14A86C72,0x657488FE38173755,0x9970D26425E9A739,0x4533E0AA24CDAE32,0x1F75D331082A7,0x5B26601583C13C5C,0xE09A5165D719A7F,0xAD8AA538602B9DE7,0x1E4D4579CBBFAB8C,0x7F3461C7638FE52B,0x5DAA64204AD523DC,0x145A69699EE94,0x2D5075833158727E,0x896551ACB849CE79,0xA681AF617647F788,0x1723B5EDDBE45009,0x9B5854E0EB1640D9,0xD62688D00D7182BA,0x19BA6EEC7E1CC,0x77514AC85F8FE7F,0xB1809AE785EA2C68,0x6C6ED5EABC9A5F7,0x70674BDF0E7404E,0xD56E93EFDDE8CAC9,0x2417F53265519D4C,0x1114DFBEBCAAE,0x1B1FA3A4C05C2E92,0x1A828F0C63D0C314,0x862A472BAEC50157,0x3736320CDEAAD20C,0x96E9DD3B0F1136D4,0x9C3B665979C3DC81,0xBA7257F0E70D,0xCF79FDD3CD205172,0x44327DB60363A751,0xBB823E50DCCEB37C,0x1D7F17F861449549,0x42A182EE2558B6B4,0x2D2CB0F6D9C70974,0x1913675156E90,0xFC2AD0CAE142EAC8,0xF3028E6768A050DD,0x9FAC46D3CCCBC00D,0x42051099504F85B,0x6E5E01D26513973A,0xE08A22090462C3F7,0x1FCB27F6F058D,0xD442A2710D7279C8,0x47C9353107967853,0x3186B539B65ECCA3,0x2F9040AC3FAAEC11,0xFC81BC15040A0242,0x33421E047DB53C84,0x181EF4D06E884,0x5AD2B639D609EDD3,0xCF707AE8328321A0,0xFD0EB9CDBB3E9B67,0x3DCC84DB2747F7E,0x788F99E1AA81E624,0x89285D9EAD58FAF2,0x100EE7BAABF21,0x9ED5BAD1C958A533,0x8ECBEFD2F7E8004,0x2C51E8D9FF1986C,0x28BE4F7E03E8F96E,0x4E31D4A1E2E19C4D,0xE13532B544BFFD85,0x63807C8EBF63,0xCBA318968A0080EA,0x6E1EA446FDF32F6E,0x3918FDEE0945EEFB,0x7395E3591BED04B3,0x9BF369DF4E1D1F68,0x20B04F1EDA6EC7A9,0x1F65E0BA40D36,0xA30A39CEDD4A222B,0xFA4283DC48E5BA9A,0x28F21B66EEDEA5A7,0xF281B97309A457FD,0xCE8146C6C9506F0F,0xF137C2F24953699C,0x1C832AE77E493,0x10D1C03A6D49BC4D,0xFBED2EDEAE19ED12,0x2300FF1D15A1FCE5,0xEEA2B1ABD907BB86,0x1B749CB24E5DF0D7,0x7F2B3B838935CAC7,0x15CCF8AF3C88E,0x8957207CAB35FDA7,0x3F202E983A7C2251,0xF84C77733F94A993,0xD5E9BAC3F2480FAB,0x2AB0A873EA9DB538,0x22279784D7692684,0x3E9DB4C486CF,0x4E48DB2718C189,0xF22832BC2DBD319,0x1014B3D4C276F007,0xAE0C8B5E22B206CC,0x94D40F9081FD4286,0x7BC1F98AC603050F,0x22D2C55CC94CA,0xCB6B8F9D520E9D80,0xE539AB381B667F8A,0x249202E9866E1249,0xF527C91DA98D2C27,0xAC19181EB83403FD,0xCC88C5918696F4A6,0x5001A3D48F32,0x4AAD16DDA72E0B3D,0x4FE27BA2AD11DB48,0x33B1070F26174171,0x7C301CB1AC90C1DB,0x261244E5F9079EEE,0xE8FE75871A7C68AD,0x44A1C4ABA16A,0x11676A8DDDBD0340,0x5CB2C0CEB9AFA00F,0xDACF20980C55013A,0x3072DF6D1397017C,0xCD8FD549DBBDED8F,0x52AECF1796CC84D3,0x49D28C2E88AE,0xB65B1735CA655C4D,0x6F76CCD8558A35E5,0x4756098172764FAE,0x133672ABE7B8AC79,0xA77420038C2DF576,0x5C8602487789F5F6,0xCD45E977BA33,0x4405E9F98CDA4908,0x7CCCA71AEF54E06F,0x6CFAF82FED48388,0xBFF533C830EF401F,0x35379607CB533986,0x4312D54C6C5ABA66,0x1CB73871DFE06,0x3BD9746B54431F3C,0x82A5240D55F89DC4,0x4A1A73BB64F9B140,0x20F82C747F685C92,0xC19FE8486FDED4FE,0x2203D84CB49CC0CE,0x1E880FF73EE6D,0xC0C1055035534E38,0x6BF0F508C87BD876,0xC78F9C503C5EEF6E,0xD7092C550FBE8E50,0x750D9410A1E5619C,0xAACF3B982D1CEABF,0x3F8C7931F23A,0x68B429BAD6B2774E,0x9748A385107A0501,0xE8C1C619F4E25A52,0x11C290A2752B1858,0xF03F885523B1325E,0x614F000D3A10634F,0x11A786BC97D6A,0x9B3A5EF04A5AE789,0xC560859153F5E0D8,0xE0CBAC4448FFDCDE,0x28E77ED61B6E805F,0x6D1A53063B71CD8,0xBCB2A1656BB817B7,0x4249D8400C62,0xB7FA854786AF5AC0,0x7A5DC8209E7291C,0x1169AA6E93471504,0x6F86598C111C882A,0x632AB04739CE3699,0xDE9876A7284C1D84,0x84269C032152,0x5F8B038957660A8C,0x2845430517B568F7,0x8ECFC5848B09126D,0x7FA5CC4B07ADF64,0x8C729000AD353D9E,0x5C32F17F662D27AF,0x8BE8097F6AD6,0x7D89D3929A3FB996,0x73895E614D1731B,0x69525E899F54687B,0xD420758D66EF5494,0xDB58FC5E91390710,0xB5137A7326DFDBD3,0x34DBC48E8154,0xD214787A42B34E0B,0x5BAB3A90CA91799F,0x66AF1457F338AE10,0xE34D45E593BA63CF,0x4629FC74E87A2409,0xC9C511732392B012,0x1188E1D7E9A74,0xA02F0C97DFAE7E47,0xB637B49189CB0814,0xC3F74B323BE03840,0xB31534EB340465AD,0x3234E3EB38BFDD68,0x7DC07CCB02369028,0x1F414C8FDA45B,0xBBDF17A1189F892E,0xE36B701711EBAEE3,0x9246EB958653A11B,0x83CDF571CC3ADD4C,0x2995DA9A115E2745,0x5E59E02C5DF39A62,0x6C681B957EEB,0xD79C7A3713401C3E,0x3C15A60271E5BA08,0x5A75226D8E7FE905,0xD133222B3A45B328,0xAF446C79B7610870,0x3EDCEBFDC33B0E46,0x1E5C107CD7990,0x7B89CFDF854F487B,0xEBEE5F2F24618E2B,0x75A442065C2CB258,0x73C059841B74E5E8,0x67AC618123BBA529,0x27580ED44099CF33,0x1C523C646E1BB,0x5548036B6B86B283,0xAACB716B6ABC7050,0x89CF0F9F1F6552C0,0x9A665F8400D87BA4,0x84D2AA1F1DF63636,0x6CDCEC2BE880938E,0x2AB398292840,0x2B94F69AAD082CD3,0xB13850F11C95D4E7,0xE7192BD8FACF99D2,0x7E9D90C86D4400AE,0x32AC85A534E2B4AF,0xB91249979A9882A8,0xB6295FB3D086,0x84EF739E89074420,0xF7756424F99DE4BE,0x4379E183A608DABB,0x7DBDEBBE169AEF00,0xBE1665D536F68CCF,0xE1CD3876BC9C435D,0x1055E38120373,0x390D402CE5D3DC44,0x8CFA0A3677B6515F,0x5B5D988515E05F4C,0x69F9F742A9C22D90,0x8E6B864D6F3E299B,0x5347C49B541CE666,0xED5A3DA86FB,0xF6DC78F623D7E6F7,0xF6222F21867029C1,0x58A4C0701A4B488E,0x4E3B8B8547AE74B0,0xD5D3C557DF93FFF0,0x8846F7445BE15891,0x1795E70C5F6B0,0xEC33500459DAEDE2,0xFA19D820518F3376,0xB73A6315501F729D,0x90B0F2390AE91B33,0x7D7DE108A2D5305F,0xA731403DDD7DB5D7,0x1E61328D3B3A3,0x9907AE5C8C7C09E2,0x829A77D65E771750,0xEBCB0DA23A387DF2,0xC554C6EACC8C41CF,0x5C6FC6CF17434CDC,0x8B9AB73D5E274CE3,0x1D6D62BADC253,0x75FC836641BC3BA0,0x66160AC7D263AF87,0xF272A8ACFEC28D22,0x29AEC47D8B0F2B2D,0x8AEDAE0C343E97DF,0x8F4D42A7272DAF84,0x1024177DE4DBE,0xF152430FBB0B1DBE,0xE3568C1C356BF46F,0x1364FC57DF940142,0x53A0739714D93EDF,0xA7E14F8906B19D5,0x84C8104B46A7DD44,0xDB2434BADA3B,0x1F3F7FE37E16E784,0x8E4743BF96285B45,0xCB2146A4B91A2E18,0xB3EAB0B7937035DD,0x96FF028F37DEFFB2,0x187B521A5CBDB905,0xC7786821C584,0x15D2C69187FE7E9B,0x5E1E586E1C413B8F,0x377A2CD34B273EE5,0x444AF3ECD606D007,0x448C5CFB70FB3AFA,0xA77280D2FB06E05F,0xF2F0B9DD6B2,0xF7A15794731CA711,0x1DA89839EBAB4015,0x6A628685D8AAC42C,0x68ADE67B3E83DFA8,0xDEE7A96D338D637A,0xBAE6D774CCB3EEB0,0x1CB7AC39710BD,0xB18E188A83606185,0x19DD500E793EB5F9,0x12C76A87E568F3A7,0x568D47E6FEEDA55D,0xE7CD608C95E2A322,0x31C5EEAD9A5A8DDF,0x1D44A00359883,0xBB9629B9F11E02B2,0x871EF19A1D125468,0x6AE239228B48B4F4,0x6AB3A7C2C690FDDE,0x58749FDB9E9DC64D,0x19F6B937A2FB5E4,0x134DF49A77166,0x7C9E71B5088BD450,0xE377B6299A84EC23,0xB283FCD8652C0821,0x9ED4A93C9A53B760,0xD3895089CBB76193,0x248C2AB502D79774,0x86E9551D6334,0x4CDE82E4D578720E,0x897CEACA236F4406,0xD47C0EBAEAE4154D,0xF65FF050DA8B7E0A,0x78660BFEC5B21A5,0x98F50C69F901544A,0x17B78FF6A399B,0x4E1FF36CD7E657A6,0xABB7C9BBBDB37F28,0x23967188A4B2E87C,0xD5F12C0E14AD5D4F,0xAE8217C059F26235,0x18215528BFB615F5,0x17D9FFA6E5F1E,0x8F3EF16B6462AEB4,0xBB37519662D63BDF,0xD74C619945D19BBB,0x1F802B44351DE23B,0xD7D3D26D97C6642E,0x45B6B0644571C9,0x376A39B9CF1E,0xA6BE7DD4152463F6,0x684815D16113586F,0xACA6C2912CD33E4,0x4A2EE4DDAC35C32E,0x238BF66DB232AA57,0xE78EE0AB10B8AFFD,0x14D1A8692A152,0x62940A3930B9750E,0x9FAB4A381E83B99B,0x3B4349F3A96F3CA5,0xBDF20EABF8279605,0x69FF11F9497B171F,0xB754B8BE488D1AE5,0xA126632841F2,0xA1D6E2C81467F02A,0x6787FC23330482B1,0x12C08B821649EC27,0x27770AD80F13E3D0,0xA8482E97530FA1AB,0x2A6B9482E37BF91F,0xCD9C8BA6CF19,0xCB00E6CD755E2339,0xF28E1D920152F636,0xEA6F07B885ACD928,0x802F25F50676AC51,0xB8CE8BC8DF3FCB48,0x4B50101647FA93D5,0x856A3D3F007B,0xC1A591DDECED3709,0xBF70B66A6AB2638B,0xCE7F185ED93BA38F,0xDBE994643E5AECE9,0xA492AE884E240DB0,0x87AD9344EEAD69DF,0x10E9DBBB4E870,0x47F1214544F1FDFC,0x26C218639D0C745E,0x361A32B9DB65AF82,0xD582A03B1CE0BEC7,0xD4895F190FA8E4AE,0x7A6EC93531DF50F0,0x1A91214205262,0x1DD92154C952A9E3,0x7253E6EB1F5DD09,0x2950BA533570EAB8,0xB4F99783D14D82F3,0xFA8DE285BDAFB52C,0x621CC3C8DFF26687,0xC1236623F3CD,0xA28066213CFF86D8,0x8519BE9066E70EA9,0xADA3B90C9C9D0822,0x78260120D02319E,0x63456744FA182ED4,0xB0D8F625E224DB95,0xAC3C875FC878,0x785EC3168B24B327,0xA652B12B6F4863CA,0xFE9336F4BC8B2ABC,0x449DC91E15F6FA6D,0x565C4F0DEE478F13,0x9ADEE49B0BB957A5,0xBF4841C7170C,0x261266AEB6489F86,0x70F47B3C026A7CD7,0xD53F719F592D2695,0xC8EF9E057CF90CDC,0x207ABC26FBB82D57,0x5F34AB5AD3B00CDD,0xFAC95A6BE005,0x2880FCA6BFC771FC,0xE8A1ED366C5EAF41,0x26DEC38C89ACC9AE,0xF433245E87B28253,0xED43C4B08B4E1835,0xC605E763F1368AD7,0x15274B28B2E38,0x768B9867C45D4BC4,0x4D682CC1A9E19CCE,0x1F015815A86514D3,0x9801C170584526D2,0xAF7C9183EA0ED04B,0x12FC4454B033E85,0x122D555B3EB2A,0xAF3432BF24DD5D01,0x88931E27357EA51,0xB4BC4DEC5765FFC9,0xBAB5802D1F2DBE30,0x77945CF510624591,0xDF905FF2B98F9D53,0xC541ECFC19A6,0xD3CF88CEDDA56BA8,0xB611D9A6A8622C83,0x19986B1EC1D394D3,0xB45D8120DD6D0E2D,0x4DC51083A59F29EE,0xDD31431E3A000D25,0x1E1A8F196D140,0x16EC5885A01EEFEA,0x4E449BDA055D1DDA,0x58F2FA8D2B55E29F,0x17B02F3EDC4EFCAE,0x50582C76E701EFE2,0xD4114A56CFA73858,0x7FE05C9B4776,0x7B69B57C0CD4A1AE,0x11723646A501876A,0x6221F4F809449E57,0x3A5C8C953FEFDCC,0x74B91C97DE053B8A,0x5E7673BA3120D0F5,0x1218FDE7C33D4,0xFFFFFFFFFFFFC5E9,0xFFFFFFFFFFFFFFFF,0xFFFFFFFFFFFFFFFF,0xA1397C7864FFFFFF,0xFC589FA8688D839,0xF86335AC52097FD0,0x1BDA7D338DBD6,0xBEF0CE72B6E9D1BD,0x92E0A4F2CA7CCCFF,0xA0B2035BB47215FC,0x6D36985E0F7DFC52,0x6BB989478F1D3A0F,0x8C5988BFDF5F4F7C,0x6E2AAF9B9D9E,0x425529A64ABCAC1F,0x49734E54998F89B6,0x9C429949C1DCE97C,0x9BE2625E5A7B4023,0x7E239939824146A,0xB15C34D4A1653801,0x672443E98A7A,0x18D5AB42DC1B2CA0,0x71C6EC0E867DCDCF,0x26DEAB045FD561B1,0x45B3BA1D5BD767C7,0xABA0A9E948A246A,0xA721BAFBE893B610,0x80697B53F1E1,0xAEF4EA78C6BAB2D4,0x8FEED1FA59A392FC,0x388326919250140F,0x1B94EB75B4BE5DFF,0x5BE9D0EBCDE6C767,0x63006B60E4B989E3,0xA140AE748A84,0x8CEA2016847486F,0xC383BC14A70F364D,0x2F2963663C26BBAA,0x342E5B2BD6F661DB,0x94F684BAC65F5EA6,0x91E13307F205794F,0x15C3EB457D0C5,0x15D9717B534B56F9,0x57ED5C5F28EEED33,0xAA224149D1792C1,0x2B663432FFF290C3,0x65B645B7893F3761,0x8C45450423A64BF5,0x2072BDFEECFDF,0x2E9A77227F6B4DC7,0x26A6145E0CD52870,0xE8E7255BCCFF6551,0xC04097B01E64680A,0x56B98EE30BBE9A40,0xF3BD5AE69AE90DEB,0x1ADE268AB0C0B,0x69C81B94AB88A2F5,0x52881065DEC1BF2C,0xD81C0EB80E03EA8B,0x42A15D8FBBB829E,0x2ADC2AE485616A64,0x7000B77296196229,0xB3B37F2C686A,0x8392008D794918AE,0xEBDABCAB9B836E2D,0x279BB892BC901E72,0xC16D3DF36FF6DB3E,0xBD3554E2019FFE83,0xAE5BB534F18B3147,0x4E8CD30C178C
			};
		#endif
	#endif 
#endif	
