/********************************************************************************************
* Supersingular Isogeny Key Encapsulation Library
*
* Abstract: supersingular isogeny parameters and generation of functions for P751_compressed
*********************************************************************************************/

#include <oqs/rand.h>
#include "../../oqs_namespace_sike_compressed.h"
#include "P751_compressed_api.h"
#define COMPRESS
#include "P751_internal.h"
#include "../internal.h"

// defines moved from P751_compressed_api.h
#define CRYPTO_SECRETKEYBYTES 602 // MSG_BYTES + SECRETKEY_A_BYTES + CRYPTO_PUBLICKEYBYTES + FP2_ENCODED_BYTES bytes
#define CRYPTO_PUBLICKEYBYTES 335 // 3*ORDER_B_ENCODED_BYTES + FP2_ENCODED_BYTES + 3 bytes for shared elligator
#define CRYPTO_BYTES 32
#define CRYPTO_CIPHERTEXTBYTES 410 // PARTIALLY_COMPRESSED_CHUNK_CT + MSG_BYTES bytes     
#define SIDH_SECRETKEYBYTES_A 47
#define SIDH_SECRETKEYBYTES_B 48
#define SIDH_PUBLICKEYBYTES 335
#define SIDH_BYTES 188

// Encoding of field elements, elements over Z_order, elements over GF(p^2) and elliptic curve points:
// --------------------------------------------------------------------------------------------------
// Elements over GF(p) and Z_order are encoded with the least significant octet (and digit) located at the leftmost position (i.e., little endian format).
// Elements (a+b*i) over GF(p^2), where a and b are defined over GF(p), are encoded as {a, b}, with a in the least significant position.
// Elliptic curve points P = (x,y) are encoded as {x, y}, with x in the least significant position.
// Internally, the number of digits used to represent all these elements is obtained by approximating the number of bits to the immediately greater multiple of 32.
// For example, a 751-bit field element is represented with Ceil(751 / 64) = 12 64-bit digits or Ceil(751 / 32) = 24 32-bit digits.

//
// Curve isogeny system "SIDHp751". Base curve: Montgomery curve By^2 = Cx^3 + Ax^2 + Cx defined over GF(p751^2), where A=6, B=1, C=1 and p751 = 2^372*3^239-1
//

const uint64_t p751[NWORDS64_FIELD] = {0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xEEAFFFFFFFFFFFFF,
                                              0xE3EC968549F878A8, 0xDA959B1A13F7CC76, 0x084E9867D6EBE876, 0x8562B5045CB25748, 0x0E12909F97BADC66, 0x00006FE5D541F71C
                                             };
const uint64_t p751x2[NWORDS64_FIELD] = {0xFFFFFFFFFFFFFFFE, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xDD5FFFFFFFFFFFFF,
                                                0xC7D92D0A93F0F151, 0xB52B363427EF98ED, 0x109D30CFADD7D0ED, 0x0AC56A08B964AE90, 0x1C25213F2F75B8CD, 0x0000DFCBAA83EE38
                                               };
const uint64_t p751x4[NWORDS64_FIELD]            = { 0xFFFFFFFFFFFFFFFC, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xBABFFFFFFFFFFFFF, 
                                                     0x8FB25A1527E1E2A3, 0x6A566C684FDF31DB, 0x213A619F5BAFA1DB, 0x158AD41172C95D20, 0x384A427E5EEB719A, 0x0001BF975507DC70 }; 
const uint64_t p751p1[NWORDS64_FIELD]            = { 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0xEEB0000000000000,
                                                     0xE3EC968549F878A8, 0xDA959B1A13F7CC76, 0x084E9867D6EBE876, 0x8562B5045CB25748, 0x0E12909F97BADC66, 0x00006FE5D541F71C };   
/* OQS note: commented out to avoid unused error
static const uint64_t p751x16p[2*NWORDS64_FIELD]        = { 0x0000000000000010, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x2A00000000000000, 
                                                     0x826D2F56C0F0EAE2, 0xAD4C9CBD81067123, 0xF62CF3052282F124, 0x53A95F7469B516FE, 0x3DADEC0D08A4732F, 0x58AD934557C11C7E, 
                                                     0x7F731B89B2DA43F2, 0x51AE9F5F5F6AFF3B, 0xD74319A6C9BCA375, 0x5BAB790796CF84D4, 0xA421554FE2E49CA8, 0x20AD617C8DF437CF, 
                                                     0x3AB06E7A12F5FF7B, 0x70A25E037E40347E, 0x51F1D323FB4C1151, 0xAE0D99AA4835FED9, 0xDF5429960D2536B6, 0x000000030E91D466 }; 
*/
// Order of Alice's subgroup
static const uint64_t Alice_order[NWORDS64_ORDER] = {0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0010000000000000};
// Order of Bob's subgroup
static const uint64_t Bob_order[NWORDS64_ORDER] = {0xC968549F878A8EEB, 0x59B1A13F7CC76E3E, 0xE9867D6EBE876DA9, 0x2B5045CB25748084, 0x2909F97BADC66856, 0x06FE5D541F71C0E1};

/* Basis for Alice on A = 6, expressed in Montgomery representation */
static const uint64_t A_gen[6 * NWORDS64_FIELD] = {0x2584350E0C33C304, 0x51E9C29E234DC61E, 0xC6E65A7BF90ACC05, 0xB1333E2E19B3A930, 0xA4F7CA2F7F66909F, 0xE01E9E6F6704BF9E,
                                                   0xE2345D48C0219D6D, 0x70F37AD9933FC182, 0x7B9D4D5870CFACA3, 0x3B8DAF20190D460D, 0xB02D6FF9AAFA0C7, 0x15A435D19526, // XPA0
                                                   0xE85E3F2B4EDDAF22, 0x4824EDCA0A253CB2, 0x65C70852876C50A0, 0x917389F0D88B919, 0x93FBE011EFA068E5, 0x72703759A4651388,
                                                   0xA266A6AEE1213EE0, 0xC496ABC50E388B6E, 0x564CB9FE0EBD72B5, 0x88B483157D3BADC8, 0x326D337A76B5317, 0x440F6F4F2D5A, // XPA1
                                                   0xCDD55D2646A1DE32, 0xAA056CDD8B80E53E, 0xAA87189B3A885C53, 0x9F6D9809057564A1, 0xC59794A13E1D38B8, 0x97F8ED39F3FA7DE5,
                                                   0xFC0CAF68C8B95129, 0x393F28B240A42FFE, 0xCD99B2F9792DEF96, 0xF1036825CBF416B9, 0x877B835F0533F2AE, 0xCDFFE238E18, // XQA0
                                                   0xAC4EF1B17010B136, 0xEC411E1B5AD8A667, 0x7737372EDB66A1AF, 0x43593ECED672CF87, 0x1E418547C7B8A975, 0x8CC78DCB18BD469A,
                                                   0x6C9FB93FD2EF8496, 0x8A4AC42666AB8545, 0x8A973B8387C15F1D, 0xC1657503D4BB4ADA, 0x22F49E4311D7BBF0, 0x1299B8FDA94C, // XQA1
                                                   0xC04B8957D3A4748F, 0xF3FB80F19063629F, 0x595434555D4EBE94, 0x8E1FEF11BFD1E0DA, 0xE31E3377248C0BB4, 0x9A05DEFF75EA51BA,
                                                   0x398686FBB343398A, 0x20331307B470DA54, 0x964FA62AD10005C5, 0x9EA5CC4D64E5D9EE, 0xC84675CF9B96060F, 0x1DECCB78CFAC, // XRA0
                                                   0x6B20FF684759DDC2, 0xD50EB91730DEAFBF, 0xAA5CA048E2DAF488, 0xE29708E28654FC18, 0x542928AD1F445359, 0xA311B83D79E73FF6,
                                                   0x850B7F5926826B22, 0x2D46731863BDB99D, 0x467A80CD8320B69D, 0xC046B12F05BFD513, 0x35D9B2FF794BDB40, 0x633276495B85
                                                  }; // XRA1

/* Basis for Bob on A = 6, expressed in Montgomery representation */
static const uint64_t B_gen[6 * NWORDS64_FIELD] = {0x110F4508C6634CCB, 0x31910BC05E296F4C, 0xED17AB0D6C029EA6, 0x9C863AB6172B9974, 0x5C15236CDB216F99, 0xDC025064818EC7D7,
                                                   0xC2180F387487EBF0, 0x946B1D0F025CBC3B, 0x5AE34395A520CB46, 0xB52034F98A879F2C, 0x3D2FAE10A22AB7C7, 0x174CD090DA3D, // XPB0
                                                   0xC3C6A839776171F2, 0x5883AFB529C8E50A, 0xDE1622BBD192925, 0x64CCE86B1826A21, 0x441AF1ABE9F6568E, 0x3F29EEC0BC6F962D,
                                                   0xA7845A0127159975, 0x109DCD6D92B0C3F2, 0x462438CD0100EE2E, 0xFB7869F2B1DF80EB, 0x563B0C55F0EEDC53, 0x1958C37D4721, // XPB1
                                                   0x110F4508C6634CCB, 0x31910BC05E296F4C, 0xED17AB0D6C029EA6, 0x9C863AB6172B9974, 0x5C15236CDB216F99, 0xDC025064818EC7D7,
                                                   0xC2180F387487EBF0, 0x946B1D0F025CBC3B, 0x5AE34395A520CB46, 0xB52034F98A879F2C, 0x3D2FAE10A22AB7C7, 0x174CD090DA3D, // XQB0
                                                   0x3C3957C6889E8E0D, 0xA77C504AD6371AF5, 0xF21E9DD442E6D6DA, 0xF9B331794E7D95DE, 0xBBE50E541609A971, 0xAF86113F439069D2,
                                                   0x3C683C8422E2DF33, 0xC9F7CDAC81470884, 0xC22A5F9AD5EAFA48, 0x89EA4B11AAD2D65C, 0xB7D78449A6CC0012, 0x568D11C4AFFA, // XQB1
                                                   0x31BB0964DFBDC34F, 0xFDC65CF4959AB106, 0xA3071E4B8B04D8FF, 0x9B68CFCE270DE486, 0x2339E590896E0095, 0xFC753508AD83E33E,
                                                   0x73A274E4A6908387, 0x88D1B207BBE8E2DC, 0xA6D0583233DC71F, 0xCF7F2ECC609DE5BE, 0xB8AF0669FBD1CF01, 0x1F3EF25DD512, // XRB0
                                                   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
                                                  };                                                       // XRB1

/* The x-coordinate of lB^(eB-1)*QB, expressed in Montgomery representation */
static const uint64_t XQB3[2 * NWORDS64_FIELD] = {0x96DCB5BD39AA8288, 0x81E7BA9AA19C1C53, 0x7021384077E4FEB6, 0xA1490C7A753E8E74, 0xA31871E13D80831C, 0x7E653AD5931BC82E,
                                                  0x8A0C230821766ECA, 0x1CE14E07EB29B669, 0xF985F42F404E45EF, 0x44AF699D9546F9B, 0x8A8C976EE869BE6B, 0xEC5AA84C390,
                                                  0xF16AEFC3FA025CCB, 0x205E4E1559261A6E, 0xFA6EEE97610CA4AA, 0x4B29DB4DF10E7D9, 0x83BC8A148E0FBB2, 0x1065373142BD8B81,
                                                  0xD7C4A7221ED3E403, 0x1071E13BCEB89A2A, 0x54A18AEA86CD9D51, 0x39BF04224B598D62, 0xD4B89FF2AA465908, 0x5A0F792EB1A3
                                                 };

static const uint64_t A_basis_zero[8 * NWORDS64_FIELD] = {0x2584350E0C38565F, 0x51E9C29E234DC61E, 0xC6E65A7BF90ACC05, 0xB1333E2E19B3A930, 0xA4F7CA2F7F66909F, 0xF78E9E6F6704BF9E,
                                                          0xA8972A8BE4E1FD91, 0x694D0F3DFDE63EAC, 0x48A2455F29C6089, 0x4F74628F6069C173, 0x1EDF6B38ED1A5433, 0x74AA0969CE,
                                                          0xE85E3F2B4EDDAF22, 0x4824EDCA0A253CB2, 0x65C70852876C50A0, 0x917389F0D88B919, 0x93FBE011EFA068E5, 0x72703759A4651388,
                                                          0xA266A6AEE1213EE0, 0xC496ABC50E388B6E, 0x564CB9FE0EBD72B5, 0x88B483157D3BADC8, 0x326D337A76B5317, 0x440F6F4F2D5A,
                                                          0xCF298A24AB4EECC8, 0x426BE362D17B58E1, 0xDEC9E4AB0C0813E4, 0xBB213D92B1A23DEC, 0x73F9337EBF1AFB1E, 0x22A428421E3F369D,
                                                          0x4C504FBA8D4C1F41, 0x97C03C026B64E556, 0x94524150E5242247, 0x8F397D005B7D0F3, 0x2EEEFB40C2D1D40F, 0x49FFA7880CC0,
                                                          0xC735AE6A6D9AB879, 0x4431FCF02097BF97, 0xCD6C8982E0B17062, 0xD38791B330DBF671, 0xBBE57BF59A8D9150, 0x58F968F33F45A7D8,
                                                          0x903068B77EC83B26, 0x7DADADD772211F21, 0x2A8DEA498D8A12BB, 0xAE73B6AE3E7657F, 0x11695A4A18565FCD, 0x303FE0D52CDB,
                                                          0xCDD55D2646A6718C, 0xAA056CDD8B80E53E, 0xAA87189B3A885C53, 0x9F6D9809057564A1, 0xC59794A13E1D38B8, 0x9E18ED39F3FA7DE5,
                                                          0xA65C1331377229F6, 0xC2E5830BF42799F, 0x5ED5225ED1E68BF3, 0x8A4CD0997002E967, 0xA96AA837EF598281, 0x6796479D59DC,
                                                          0xAC4EF1B17010B136, 0xEC411E1B5AD8A667, 0x7737372EDB66A1AF, 0x43593ECED672CF87, 0x1E418547C7B8A975, 0x8CC78DCB18BD469A,
                                                          0x6C9FB93FD2EF8496, 0x8A4AC42666AB8545, 0x8A973B8387C15F1D, 0xC1657503D4BB4ADA, 0x22F49E4311D7BBF0, 0x1299B8FDA94C,
                                                          0xD976D6B7056D9BE, 0x269169101B22C490, 0x485B7EFC9E5BBC09, 0xD0B55BEABD8EF10, 0x985F36CD657501DD, 0x19FD14B27B195F17,
                                                          0xF0F7E577A00E0E42, 0x1A179D4A080F58D2, 0x97603A54D8CFB3E5, 0x96E69A6743A68CE1, 0xF04C8D027EB4B382, 0xA8E741B2F5C,
                                                          0xBBF34629BA51923B, 0x94250E3ECE841C7C, 0x56CD785FA63E19F9, 0x126F4144FC6B39F4, 0x4BB54DEE559E01E3, 0x3195B2E094E0D92,
                                                          0x1F40204597E534A, 0x7F9B8D473586B966, 0x38281C1E6F54E744, 0x215C5889B91D5D63, 0x9CEA26343AD2B060, 0x22374E0C0255
                                                         };

/* Full 3-torsion for Bob on A = 0, expressed in Montgomery representation */
static const uint64_t B_gen_3_tors[16 * NWORDS64_FIELD] = {0x7BB7A4A07B0788DC, 0xDC36A3F6607B21B0, 0x4750E18EE74CF2F0, 0x464E319D0B7AB806, 0xC25AA44C04F758FF, 0x392E8521A46E0A68,
                                                           0xFC4E76B63EFF37DF, 0x1F3566D892E67DD8, 0xF8D2EB0F73295E65, 0x457B13EBC470BCCB, 0xFDA1CC9EFEF5BE33, 0x5DBF3D92CC02,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0xE7E952E9A260914, 0x92C98EDCD6E01424, 0x8F2B207221D33C50, 0xAEF2728CA22F318E, 0x5FD0E29877DC3D43, 0x1BDC7B7367E4E124,
                                                           0xFF5BC4186BDDCAE7, 0xD20A34710EE8B44B, 0x2E0866A5B9467EBD, 0x876EFC05B732A6AD, 0x9D39159E24822ABB, 0x658F31DC0D6E,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x84485B5F84F87723, 0x23C95C099F84DE4F, 0xB8AF1E7118B30D0F, 0xB9B1CE62F48547F9, 0x3DA55BB3FB08A700, 0xB5817ADE5B91F597,
                                                           0xE79E1FCF0AF940C9, 0xBB60344181114E9D, 0xF7BAD5863C28A11, 0x3FE7A11898419A7C, 0x1070C40098C51E33, 0x122697AF2B19,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0xE7E952E9A260914, 0x92C98EDCD6E01424, 0x8F2B207221D33C50, 0xAEF2728CA22F318E, 0x5FD0E29877DC3D43, 0x1BDC7B7367E4E124,
                                                           0xFF5BC4186BDDCAE7, 0xD20A34710EE8B44B, 0x2E0866A5B9467EBD, 0x876EFC05B732A6AD, 0x9D39159E24822ABB, 0x658F31DC0D6E,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x6491DDA56B18F762, 0xB50239F87A977F80, 0xD061934416F37D7C, 0xD79D328C01810FEC, 0x4F4BB58557C706AF, 0xEFE0C6963007AABB,
                                                           0x20EA47B4FDE821A5, 0xDE5144774B84AE51, 0x262E82E15B85FF1B, 0x7A0820D88294BDEE, 0xA36653F0DFF6FE6E, 0x239BB2213058,
                                                           0xA8C40A6A4ACECF3A, 0xE988877E7DF4D938, 0xC2353D8E5C2F0BAB, 0xC4758A5B1437A358, 0xC5CEE9FD3A3B088D, 0x86B2876D6520F51,
                                                           0xD31476314E3DAD08, 0x517E8096137AFBC9, 0xE52ADD7D2B85BB6C, 0x43A92AC98B1E5F2B, 0x4F32E4103B80C703, 0x68BCEB2391D2,
                                                           0x573BF595B53130C5, 0x16777881820B26C7, 0x3DCAC271A3D0F454, 0x3B8A75A4EBC85CA7, 0x3A311602C5C4F772, 0xE644D78929ADF0AE,
                                                           0x10D82053FBBACBA0, 0x89171A84007CD0AD, 0x2323BAEAAB662D0A, 0x41B98A3AD193F81C, 0xBEDFAC8F5C3A1563, 0x728EA1E6549,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x9B6E225A94E7089D, 0x4AFDC6078568807F, 0x2F9E6CBBE90C8283, 0x2862CD73FE7EF013, 0xB0B44A7AA838F950, 0xFECF3969CFF85544,
                                                           0xC3024ED04C105702, 0xFC4456A2C8731E25, 0xE22015867B65E95A, 0xB5A942BDA1D9959, 0x6AAC3CAEB7C3DDF8, 0x4C4A2320C6C3,
                                                           0xA8C40A6A4ACECF3A, 0xE988877E7DF4D938, 0xC2353D8E5C2F0BAB, 0xC4758A5B1437A358, 0xC5CEE9FD3A3B088D, 0x86B2876D6520F51,
                                                           0xD31476314E3DAD08, 0x517E8096137AFBC9, 0xE52ADD7D2B85BB6C, 0x43A92AC98B1E5F2B, 0x4F32E4103B80C703, 0x68BCEB2391D2,
                                                           0xA8C40A6A4ACECF3A, 0xE988877E7DF4D938, 0xC2353D8E5C2F0BAB, 0xC4758A5B1437A358, 0xC5CEE9FD3A3B088D, 0x86B2876D6520F51,
                                                           0xD31476314E3DAD08, 0x517E8096137AFBC9, 0xE52ADD7D2B85BB6C, 0x43A92AC98B1E5F2B, 0x4F32E4103B80C703, 0x68BCEB2391D2
                                                          };

/* Pre-computed pairing ReducedTatePairing(R0,S3,lB^eB) on A = 0 */
static const uint64_t g_R_S_im[NWORDS64_FIELD] = {0xCB6E5ADE9CD78AF1, 0x40F3DD4D50CE0E29, 0x38109C203BF27F5B, 0x50A4863D3A9F473A, 0x518C38F09EC0418E, 0xC2429D6AC98DE417,
                                                  0x1A2DC3684817A3CB, 0xF7E83EC334E40005, 0xC560B1CA4C837125, 0xCECA2F86BEB1A124, 0x563DDE23E947A71E, 0x34BDF9FF47AA
                                                 };

// Montgomery constant Montgomery_R2 = (2^768)^2 mod p751
static const uint64_t Montgomery_R2[NWORDS64_FIELD] = {0x233046449DAD4058, 0xDB010161A696452A, 0x5E36941472E3FD8E, 0xF40BFE2082A2E706, 0x4932CCA8904F8751, 0x1F735F1F1EE7FC81,
                                                       0xA24F4D80C1048E18, 0xB56C383CCDB607C5, 0x441DD47B735F9C90, 0x5673ED2C6A6AC82A, 0x06C905261132294B, 0x000041AD830F1F35
                                                      };
// Value one in Montgomery representation
static const uint64_t Montgomery_one[NWORDS64_FIELD] = {0x00000000000249ad, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x8310000000000000,
                                                        0x5527b1e4375c6c66, 0x697797bf3f4f24d0, 0xc89db7b2ac5c4e2e, 0x4ca4b439d2076956, 0x10f7926c7512c7e9, 0x00002d5b24bce5e2
                                                       };

// constant Montgomery_RB1 = (2^NBITS_ORDER)^2 mod Bob_order
static const uint64_t Montgomery_RB1[NWORDS64_FIELD] = {0x1A55482318541298, 0x70A6370DFA12A03, 0xCB1658E0E3823A40, 0xB3B7384EB5DEF3F9, 0xCBCA952F7006EA33, 0x569EF8EC94864C};

// constant Montgomery_RB2 = (-Bob_order)^-1 mod 2^NBITS_ORDER
static const uint64_t Montgomery_RB2[NWORDS64_FIELD] = {0x48062A91D3AB563D, 0x6CE572751303C2F5, 0x5D1319F3F160EC9D, 0xE35554E8C2D5623A, 0xCA29300232BC79A5, 0x8AAD843D646D78C5};

/* OQS note: unused
// 1/3 mod p
static const uint64_t threeinv[NWORDS64_FIELD] = {0x555555555556188F, 0x5555555555555555, 0x5555555555555555, 0x5555555555555555,
                                                  0x5555555555555555, 0x8105555555555555, 0x1C6290A167C97977, 0xCDD287EA6A6FB6F0,
                                                  0x42DF3D3B8EC96F64, 0x198C3C1346027872, 0xB0528624270642A3, 0xF1E61944CA0
                                                 };
*/

// Fixed parameters for isogeny tree computation
static const unsigned int strat_Alice[MAX_Alice - 1] = {
	80, 48, 27, 15, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 7, 4, 2, 1, 1, 2, 1,
	1, 3, 2, 1, 1, 1, 1, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1,
	1, 1, 2, 1, 1, 1, 21, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1,
	1, 1, 1, 2, 1, 1, 1, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1,
	33, 20, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1, 1, 1, 2, 1,
	1, 1, 8, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 2, 1, 1, 16, 8, 4, 2, 1, 1,
	1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1
};

static const unsigned int strat_Bob[MAX_Bob - 1] = {
	112, 63, 32, 16, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1,
	1, 4, 2, 1, 1, 2, 1, 1, 16, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2,
	1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 31, 16, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2,
	1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 15, 8, 4, 2, 1, 1, 2, 1, 1, 4,
	2, 1, 1, 2, 1, 1, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 49, 31, 16, 8, 4, 2,
	1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1,
	15, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1,
	1, 1, 1, 21, 12, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 5, 3, 2, 1, 1, 1, 1,
	2, 1, 1, 1, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1
};

// Fixed traversal strategies for Pohlig-Hellman discrete logs

static const unsigned int ph2_path[PLEN_2] = { // w_2 = 4
#ifdef COMPRESSED_TABLES
  #ifdef ELL2_TORUS
    #if W_2 == 4
      0, 0, 1, 2, 3, 3, 4, 4, 5, 6, 6, 7, 8, 9, 9, 9, 10, 11, 12, 13, 13, 13, 14, 14, 15, 16, 17, 18, 19, 19, 19, 19, 20, 21, 22, 22, 23, 24, 25, 26, 27, 27, 28, 28, 28, 28, 28, 29, 30, 31, 32, 33, 34, 34, 35, 36, 37, 38, 39, 40, 40, 40, 40, 41, 42, 42, 42, 42, 42, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52, 52, 53, 54, 55, 56, 57, 58, 59, 59, 59, 59, 59, 59, 59
    #endif
  #endif
#endif
};

static const unsigned int ph3_path[PLEN_3] = { 
#ifdef COMPRESSED_TABLES
  #ifdef ELL3_FULL_SIGNED
    #if W_3 == 3
      0, 0, 1, 2, 3, 3, 4, 4, 5, 6, 6, 7, 8, 9, 9, 9, 10, 11, 12, 13, 13, 13, 13, 14, 15, 16, 17, 18, 19, 19, 19, 19, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 28, 28, 28, 28, 28, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 41, 41, 41, 41, 41, 41, 41, 41, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, 52
    #endif
  #endif
#endif
};

// Entangled bases related static tables and parameters

// Constants u and u0 where u = u0^2 in F_{p^2} \ F_p used in entagled basis generation
// For the 2^eA-torsion basis generation:
//      Two tables of 17 elements each for the values r in F_p such that v = 1/(1+ur^2) where r is already converted to Montgomery representation
//      Also, 2 tables for the quadratic residues (qr) and quadratric non residues (qnr) v in F_{p^2} with 17 GF(p^2) elements each.
// For the 3^eB-torsion basis generation:
//      A table of size 20 for values v = 1/(1+U*r^2) where U = 4+i

static const uint64_t u_entang[2 * NWORDS64_FIELD] = {
	0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
	0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000,
	0x000000000004935a, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0620000000000000,
	0xaa4f63c86eb8d8cd, 0xd2ef2f7e7e9e49a0, 0x913b6f6558b89c5c, 0x99496873a40ed2ad, 0x21ef24d8ea258fd2, 0x00005ab64979cbc4
};

/* OQS note: unused
static const uint64_t u0_entang[2 * NWORDS64_FIELD] = {
	0x00000000000249ad, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x8310000000000000,
	0x5527b1e4375c6c66, 0x697797bf3f4f24d0, 0xc89db7b2ac5c4e2e, 0x4ca4b439d2076956, 0x10f7926c7512c7e9, 0x00002d5b24bce5e2,
	0x00000000000249ad, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x8310000000000000,
	0x5527b1e4375c6c66, 0x697797bf3f4f24d0, 0xc89db7b2ac5c4e2e, 0x4ca4b439d2076956, 0x10f7926c7512c7e9, 0x00002d5b24bce5e2
};
*/

// Elligator constant U = min{u0+k} for k=1,2... such that u0+k is a square in F_p^2 for generating 3^n torsion bases
static const uint64_t U3[2*NWORDS64_FIELD] = {0x926B5,0x0,0x0,0x0,0x0,0x1D90000000000000,0x70B2310B937938F1,0xCB48C3E2E944C6CA,0x1A284662DA855042,0xAD301BE2EB6B4E13,0x35CBB9123C90433E,0x4586BDB1A06C,0x249AD,0x0,0x0,0x0,0x0,0x8310000000000000,0x5527B1E4375C6C66,0x697797BF3F4F24D0,0xC89DB7B2AC5C4E2E,0x4CA4B439D2076956,0x10F7926C7512C7E9,0x2D5B24BCE5E2};

// Tables for quadratic residues and quadratic non residues v with 17 elements each.

static const uint64_t table_r_qr[][NWORDS64_FIELD] = {
	{0x249AD, 0x0, 0x0, 0x0, 0x0, 0x8310000000000000, 0x5527B1E4375C6C66, 0x697797BF3F4F24D0, 0xC89DB7B2AC5C4E2E, 0x4CA4B439D2076956, 0x10F7926C7512C7E9, 0x2D5B24BCE5E2},
	{0x926B5, 0x0, 0x0, 0x0, 0x0, 0x1D90000000000000, 0x70B2310B937938F1, 0xCB48C3E2E944C6CA, 0x1A284662DA855042, 0xAD301BE2EB6B4E13, 0x35CBB9123C90433E, 0x4586BDB1A06C},
	{0xDBA10, 0x0, 0x0, 0x0, 0x0, 0x3500000000000000, 0x3714FE4EB8399915, 0xC3A2584753EB43F4, 0xA3151D605C520428, 0xC116CF5232C7C978, 0x49A84D4B8EFAF6AA, 0x305731E97514},
	{0x1003BD, 0x0, 0x0, 0x0, 0x0, 0xB810000000000000, 0x8C3CB032EF96057B, 0x2D19F006933A68C4, 0x6BB2D51308AE5257, 0xDBB838C04CF32CF, 0x5A9FDFB8040DBE94, 0x5DB256A65AF6},
	{0x124D6B, 0x0, 0x0, 0x0, 0x0, 0x4C70000000000000, 0xFD77CB91DCF9F939, 0xBBFBECABBE91C11D, 0x2C01F45DDE1EB80E, 0xD4FD82C17A2444DE, 0x5D84E184E165AA16, 0x1B27A62149BC},
	{0x149718, 0x0, 0x0, 0x0, 0x0, 0xCF80000000000000, 0x529F7D761456659F, 0x2573846AFDE0E5EE, 0xF49FAC108A7B063D, 0x21A236FB4C2BAE34, 0x6E7C73F156787200, 0x4882CADE2F9E},
	{0x192A73, 0x0, 0x0, 0x0, 0x0, 0xE6F0000000000000, 0x19024AB93916C5C3, 0x1DCD18CF68876318, 0x7D8C830E0C47BA23, 0x3588EA6A9388299A, 0x8259082AA8E3256C, 0x33533F160446},
	{0x1DBDCE, 0x0, 0x0, 0x0, 0x0, 0xFE60000000000000, 0xDF6517FC5DD725E7, 0x1626AD33D32DE041, 0x6795A0B8E146E09, 0x496F9DD9DAE4A500, 0x96359C63FB4DD8D8, 0x1E23B34DD8EE},
	{0x249AD6, 0x0, 0x0, 0x0, 0x0, 0x98E0000000000000, 0xFAEF9723B9F3F272, 0x77F7D9577D23823B, 0x5803E8BBBC3D701D, 0xA9FB0582F44889BC, 0xBB09C309C2CB542D, 0x364F4C429378},
	{0x292E31, 0x0, 0x0, 0x0, 0x0, 0xB050000000000000, 0xC1526466DEB45296, 0x70516DBBE7C9FF65, 0xE0F0BFB93E0A2403, 0xBDE1B8F23BA50521, 0xCEE6574315360799, 0x211FC07A6820},
	{0x2B77DE, 0x0, 0x0, 0x0, 0x0, 0x3360000000000000, 0x167A164B1610BEFD, 0xD9C9057B27192436, 0xA98E776BEA667231, 0xA866D2C0DAC6E78, 0xDFDDE9AF8A48CF83, 0x4E7AE5374E02},
	{0x2DC18C, 0x0, 0x0, 0x0, 0x0, 0xC7C0000000000000, 0x87B531AA0374B2BA, 0x68AB022052707C8F, 0x69DD96B6BFD6D7E9, 0xD1C86C6183018087, 0xE2C2EB7C67A0BB05, 0xBF034B23CC8},
	{0x300B39, 0x0, 0x0, 0x0, 0x0, 0x4AD0000000000000, 0xDCDCE38E3AD11F21, 0xD22299DF91BFA15F, 0x327B4E696C332617, 0x1E6D209B5508E9DE, 0xF3BA7DE8DCB382EF, 0x394B596F22AA},
	{0x3254E6, 0x0, 0x0, 0x0, 0x0, 0xCDE0000000000000, 0x32049572722D8B87, 0x3B9A319ED10EC630, 0xFB19061C188F7446, 0x6B11D4D527105334, 0x4B2105551C64AD8, 0x66A67E2C088D},
	{0x349E94, 0x0, 0x0, 0x0, 0x0, 0x6240000000000000, 0xA33FB0D15F917F45, 0xCA7C2E43FC661E89, 0xBB682566EDFFD9FD, 0x3253D40A9C656543, 0x79712222F1E365B, 0x241BCDA6F753},
	{0x3B7B9C, 0x0, 0x0, 0x0, 0x0, 0xFCC0000000000000, 0xBECA2FF8BBAE4BCF, 0x2C4D5A67A65BC083, 0xCF2B4171C28DC12, 0x92DF3BB3B5C94A00, 0x2C6B38C7F69BB1B0, 0x3C47669BB1DD},
	{0x3DC549, 0x0, 0x0, 0x0, 0x0, 0x7FD0000000000000, 0x13F1E1DCF30AB836, 0x95C4F226E5AAE554, 0xD5906BC9C8852A40, 0xDF83EFED87D0B356, 0x3D62CB346BAE7999, 0x69A28B5897BF}
};

static const uint64_t table_r_qnr[17][NWORDS64_FIELD] = {
	{0x4935A, 0x0, 0x0, 0x0, 0x0, 0x620000000000000, 0xAA4F63C86EB8D8CD, 0xD2EF2F7E7E9E49A0, 0x913B6F6558B89C5C, 0x99496873A40ED2AD, 0x21EF24D8EA258FD2, 0x5AB64979CBC4},
	{0x6DD08, 0x0, 0x0, 0x0, 0x0, 0x9A80000000000000, 0x1B8A7F275C1CCC8A, 0x61D12C23A9F5A1FA, 0x518A8EB02E290214, 0x608B67A91963E4BC, 0x24D426A5C77D7B55, 0x182B98F4BA8A},
	{0xB7063, 0x0, 0x0, 0x0, 0x0, 0xB1F0000000000000, 0xE1ED4C6A80DD2CAE, 0x5A2AC088149C1F23, 0xDA7765ADAFF5B5FA, 0x74721B1860C06021, 0x38B0BADF19E82EC1, 0x2FC0D2C8F32},
	{0x16E0C6, 0x0, 0x0, 0x0, 0x0, 0x63E0000000000000, 0xC3DA98D501BA595D, 0xB455811029383E47, 0xB4EECB5B5FEB6BF4, 0xE8E43630C180C043, 0x716175BE33D05D82, 0x5F81A591E64},
	{0x1B7420, 0x0, 0x0, 0x0, 0x0, 0x6A00000000000000, 0x6E29FC9D7073322A, 0x8744B08EA7D687E8, 0x462A3AC0B8A40851, 0x822D9EA4658F92F1, 0x93509A971DF5ED55, 0x60AE63D2EA28},
	{0x20077B, 0x0, 0x0, 0x0, 0x0, 0x8170000000000000, 0x348CC9E09533924E, 0x7F9E44F3127D0512, 0xCF1711BE3A70BC37, 0x96145213ACEC0E56, 0xA72D2ED07060A0C1, 0x4B7ED80ABED0},
	{0x225129, 0x0, 0x0, 0x0, 0x0, 0x15D0000000000000, 0xA5C7E53F8297860C, 0xE8041983DD45D6B, 0x8F6631090FE121EF, 0x5D56514922412065, 0xAA12309D4DB88C44, 0x8F42785AD96},
	{0x26E483, 0x0, 0x0, 0x0, 0x0, 0x1BF0000000000000, 0x50174907F1505ED9, 0xE16F7116BC72A70C, 0x20A1A06E6899BE4B, 0xF69FB9BCC64FF313, 0xCC01557637DE1C16, 0x63AA70FF795A},
	{0x36E841, 0x0, 0x0, 0x0, 0x0, 0xE550000000000000, 0xF86762B596EDEBAB, 0x33F3C6033BB54359, 0x8405DD199A5C282C, 0x7EF888446E6CCE9A, 0x188EA48EA430FE44, 0x5176F263DD35},
	{0x3931EF, 0x0, 0x0, 0x0, 0x0, 0x79B0000000000000, 0x69A27E148451DF69, 0xC2D5C2A8670C9BB3, 0x4454FC646FCC8DE3, 0x463A8779E3C1E0A9, 0x1B73A65B8188E9C7, 0xEEC41DECBFB},
	{0x400EF7, 0x0, 0x0, 0x0, 0x0, 0x1430000000000000, 0x852CFD3BE06EABF4, 0x24A6EECC11023DAD, 0x95DF8B149DF58FF8, 0xA6C5EF22FD25C565, 0x4047CD014906651C, 0x2717DAD38685},
	{0x4258A4, 0x0, 0x0, 0x0, 0x0, 0x9740000000000000, 0xDA54AF2017CB185A, 0x8E1E868B5051627D, 0x5E7D42C74A51DE26, 0xF36AA35CCF2D2EBC, 0x513F5F6DBE192D05, 0x5472FF906C67},
	{0x44A252, 0x0, 0x0, 0x0, 0x0, 0x2BA0000000000000, 0x4B8FCA7F052F0C18, 0x1D0083307BA8BAD7, 0x1ECC62121FC243DE, 0xBAACA292448240CB, 0x5424613A9B711888, 0x11E84F0B5B2D},
	{0x46EBFF, 0x0, 0x0, 0x0, 0x0, 0xAEB0000000000000, 0xA0B77C633C8B787E, 0x86781AEFBAF7DFA7, 0xE76A19C4CC1E920C, 0x75156CC1689AA21, 0x651BF3A71083E072, 0x3F4373C8410F},
	{0x4935AC, 0x0, 0x0, 0x0, 0x0, 0x31C0000000000000, 0xF5DF2E4773E7E4E5, 0xEFEFB2AEFA470477, 0xB007D177787AE03A, 0x53F60B05E8911378, 0x761386138596A85B, 0x6C9E988526F1},
	{0x4DC907, 0x0, 0x0, 0x0, 0x0, 0x4930000000000000, 0xBC41FB8A98A84509, 0xE849471364ED81A1, 0x38F4A874FA479420, 0x67DCBE752FED8EDE, 0x89F01A4CD8015BC7, 0x576F0CBCFB99},
	{0x525C62, 0x0, 0x0, 0x0, 0x0, 0x60A0000000000000, 0x82A4C8CDBD68A52D, 0xE0A2DB77CF93FECB, 0xC1E17F727C144806, 0x7BC371E4774A0A43, 0x9DCCAE862A6C0F33, 0x423F80F4D041}
};

static const uint64_t table_v_qr[TABLE_V_LEN][NWORDS64_FIELD] = {
	{0xCCCCCCCCCCCD41EF, 0xCCCCCCCCCCCCCCCC, 0xCCCCCCCCCCCCCCCC, 0xCCCCCCCCCCCCCCCC, 0xCCCCCCCCCCCCCCCC, 0x467CCCCCCCCCCCCC, 0x5CCF962F575DF8B, 0x3953C2CA47D95926, 0xC50BC81A120A6C9F, 0xDE486C73B57C04C7, 0x3C38F0BBEDB4E68A, 0x35D48FD9C405},
	{0x6666666666657C21, 0x6666666666666666, 0x6666666666666666, 0x6666666666666666, 0x6666666666666666, 0x61B6666666666666, 0xD852A3BF5F0CB992, 0x67EE158584451A2A, 0x7E370833B2D70F38, 0xC8D1DC1CF1BA4DB8, 0x95A0AF27BC510F50, 0x43CB58E6F11},
	{0xB891DB891DB8926D, 0xDB891DB891DB891D, 0x1DB891DB891DB891, 0x91DB891DB891DB89, 0x891DB891DB891DB8, 0xB7A1DB891DB891DB, 0x7667EA3E359A528F, 0x920F68111BC2395B, 0x73153DF2CE971E2C, 0x16EDA0D863AC7DA5, 0xBF837289A3FC18BF, 0x5ADF5C37BA11},
	{0xEDC48EDC48EDB246, 0x8EDC48EDC48EDC48, 0x48EDC48EDC48EDC4, 0xC48EDC48EDC48EDC, 0xDC48EDC48EDC48ED, 0x49A48EDC48EDC48E, 0x590A01C2CFF1EF31, 0xF144BE828EE398A4, 0x7553BC320111D67F, 0xAE524764F48D28A2, 0x7D745F00E9774A86, 0x16C21BBD6A1},
	{0x5E40E05A0E84063C, 0x1F4D776F0A51A9CB, 0xCB5E40E05A0E8406, 0x61F4D776F0A51A9, 0xA9CB5E40E05A0E84, 0x72C61F4D776F0A51, 0x618B216FB9C76A1E, 0xE22409F266B16463, 0x894A7D245553FD6B, 0x4E89F364CCE00FA7, 0x77A8C743F3083EC9, 0x188BE18D4AC5},
	{0x7DC0E6ABEADE3F10, 0x323668C519083ECD, 0xCD7DC0E6ABEADE47, 0x47323668C519083E, 0x3ECD7DC0E6ABEADE, 0xA347323668C51908, 0xCFA800E85F71B1FE, 0xF36E5745D988B7A, 0xE7F654456F1F4116, 0x3F5EDBEC2C210D61, 0x39B0FEDD215C1DCA, 0x1705E4626A37},
	{0x791C39AD37694088, 0x40791C39AD376940, 0x6940791C39AD3769, 0x376940791C39AD37, 0xAD376940791C39AD, 0x6C2D376940791C39, 0x4A463885B9946D81, 0x736B37A93397FE42, 0x3874F55B29BB4D54, 0x5F7A8A5F8D7543C5, 0x228CCBA8F76BD619, 0x5096624D7492},
	{0xA331EBB0C9B54BA9, 0x51A331EBB0C9B551, 0xB551A331EBB0C9B5, 0xC9B551A331EBB0C9, 0xB0C9B551A331EBB0, 0xC980C9B551A331EB, 0xC7BC1BC679178B4E, 0x7072B575C98A5F98, 0xB10657E6A1B9DEA2, 0x71773BA18E92437A, 0xAD4025945D8B2AB8, 0x2F2C83A4E8D6},
	{0x732233773223377C, 0x3773223377322337, 0x2337732233773223, 0x3223377322337732, 0x7732233773223377, 0xC637322337732233, 0x2695C5C06B69B3F6, 0x2330B3C47A0D9F3, 0x1FF319B45BEC27C0, 0xA128906C609EE6B9, 0x736A7518122E449F, 0x1FCE86244E63},
	{0x6EE64466EE6441DB, 0x466EE64466EE6446, 0x64466EE64466EE64, 0xEE64466EE64466EE, 0x66EE64466EE64466, 0x63D6EE64466EE644, 0xA64EE10DFC0F7507, 0x7E19CCA11263939B, 0x39CF2CD41A04B92C, 0xB2FBF371165340D9, 0x53745B07D6DF8B00, 0x44F4C0618554},
	{0x70E5437D550C15E9, 0x514AB52CDC183AC2, 0xA4D63DFA7AA6594A, 0x46DAD39BC3F3F19, 0x82FD5B4A38522B72, 0xAD58A5AA7E8D2ECF, 0x523BAAB70A135BE2, 0x45782546050175F7, 0xB24F33FAA2472B55, 0xEBA7A14C1F8593CD, 0x20600FBAB2D24E65, 0x133D309F7EC7},
	{0x8EEB4AB02E5A2272, 0x8EB9599CB8AAD0F4, 0xB06CC77E62BB7EF8, 0x32986176DFFA11C5, 0x1BAC3B085C0081D9, 0x69272A1BEAA860AF, 0xE41E6EBFB6ED0D08, 0xF255608B042DB67E, 0x1279C6C0D0C40109, 0x76B7BA4E30FA03B6, 0xD3DDD4F6F5A7EB7, 0x106A8E48CD0F},
	{0xD5120148FEAF2C06, 0x255365F9DAA4C4B8, 0xCC272EFC74777061, 0xD9E68887C98D5C50, 0x383B3424E91729B, 0xE0B5120148FEAF2C, 0x1BE4FA2B9F7CFD78, 0xDA98A1C587680DD9, 0xACC09E7217DDD41D, 0xBAB5E239D699C58C, 0x41FD047089438309, 0x858395E2C66},
	{0x94FAC8FF3E686241, 0xB72999CF503E0946, 0x2F59559E717C42A, 0x412EFA3785EBB9F, 0xAD808B51BA81A8AC, 0x4BE4FAC8FF3E6864, 0x8C1AAEA7B9495845, 0x94D196357B071596, 0x4FBF87DA63CFBC99, 0x666394A603DFBB5D, 0xAA328976810C7C69, 0x6AA898DF5E6A},
	{0x40E70907368FAAA3, 0x2531EC3903CE9F2C, 0xF8AB9BFACF708D95, 0xD2DE3A9B1838859A, 0x4E47B12D7C682948, 0x281BDC83C4953F02, 0xDE0EF99ABED7AFEA, 0xB81783A3E8034AE6, 0x3E73219AAC32B265, 0xA6F25C569E80F4D3, 0x6961E44D2E61E550, 0x54AF9D7748CE},
	{0x4EF61479F650B3CA, 0xE4161CB8F931D792, 0xAD6C0EDA1D651114, 0x96969F3A055F9963, 0xA55811F1BE797DD9, 0xBB36DA0672F2CEF4, 0xBCD0F6F9FFB265C5, 0x868F4BABA3739212, 0xDA96099F90F8EC8D, 0xF6B719FF6C14091F, 0xEF5131B07D9BA208, 0x15F75A78FB86},
	{0xB5CA528D6B5CA529, 0xCA528D6B5CA528D6, 0x528D6B5CA528D6B5, 0x8D6B5CA528D6B5CA, 0x6B5CA528D6B5CA52, 0xB27528D6B5CA528D, 0xD7C00A95FC590145, 0x4E4610B31250410D, 0x92EB9C10C371F0FC, 0x71D124AD03784C6D, 0xFBA87309E1A6D857, 0x2DF3B0F268CD},
	{0x6B5AE52946B5AD2D, 0x5AE52946B5AE5294, 0xE52946B5AE52946B, 0x2946B5AE52946B5A, 0x46B5AE52946B5AE5, 0xD0BE52946B5AE529, 0x5BEAE3E345C8E7D7, 0x9D2D7059D4BB6A4E, 0x18F7C0F408EA159, 0x4E11D89176651539, 0x48671FC6C152F99C, 0x530CE08D1028},
	{0xCE57AA6B0A6E427F, 0x9B8F883433A0EB6A, 0xBD90869A824E3109, 0xD1840719519276EA, 0xF601CD0A72F12455, 0x8904C6A15653D646, 0x72DB02908565884, 0x71294DC40DC2EB13, 0x361608701CEA745E, 0x1279CC6C51A4F800, 0xDD923715582E9EDE, 0x6578E5CC44C5},
	{0xB218A10D98E7AC3C, 0x3CAF3BDD50AC19A5, 0x2A2B48E62A13DFAE, 0xA9CE07E98542FDD0, 0x4B70FD8D0D9C06C5, 0x6829379D7BC99C61, 0x57C3D248CC251489, 0x9F1367A307C698E9, 0x2CC0B2B8746361EF, 0x9A6A53D6400CB8EA, 0x7888C0894B30195B, 0x29E026757F97},
	{0xB7B49E77DD01B05A, 0xC7FCDE689EA7A30E, 0x3B83E48534D253E7, 0xE1F4F28B24C70D5F, 0xC226AF798CDAE97, 0x29592FFABA44E0BF, 0x451BFFACF7902A0B, 0xEF0C4136DB713262, 0x7F066B8D7CD90C93, 0x193C99BB6A2BD9E4, 0xDCCEB7DD7547E7EE, 0x1BC6242C97DB},
	{0xE49911F2B13CA178, 0xF8D4BCF08B36207D, 0x2605805106CF5C4D, 0xBB2BF392469C496A, 0xC6EE51B30BE997A4, 0x3636AEDEA9BE252F, 0x5962BFE5CE1E434D, 0x80A1219F24FBAA71, 0x9727D3F901B9FA59, 0x9C7DB47FBDB8D837, 0x260B21A1EA96DA92, 0x58C7EC9D75A5},
	{0x11372FCC7943203A, 0x5C4549B345DE13E1, 0xB35E08AA984D8EEB, 0x9AD22A4F3D9CF45D, 0x84408FBE68108C13, 0xD6F34279BA1D5079, 0xFD86BCE1D485CE9C, 0x7547DCC9E6A72B9A, 0x82BAC6E5640D1B8, 0xA0AA9060525B5FA0, 0x9DF6B6008023369A, 0x64534002FC0F},
	{0x338AA1050E3B47F2, 0xA779AFC5AA01E0AA, 0x7A24EAE40DA1607F, 0x2F3BC85F75845B2F, 0xB63ECCFACC4A42BC, 0xB970439A64648442, 0xFC7FE81D54BCBFAD, 0x8F070C333697311A, 0xC3F49A6F3F169CEA, 0x5D4853A9EA5F7ED, 0xD518DE0B200B8454, 0x527415AF5FE1},
	{0x29D6C703462657AD, 0xDC60DB370D32F7B9, 0x7ECF292BC2C418DD, 0x37B10E0D29478FC0, 0x2DFAFEF9EA3B8E16, 0xE9716B45FA8908A4, 0x6CB32B342A6375FF, 0xFD73F31AB04928C5, 0xB397940272F73C0C, 0x5B001264620355C2, 0x5803F26169F9945B, 0x5C5620099D9B},
	{0xDA0656B84FE5EB1E, 0xBA4CBC548666840D, 0x1A44273AF862539E, 0x1FFD96A7C772BACB, 0x953D86F6FED07774, 0x7BBC68E6D3E03A59, 0xA78131504FEA94B4, 0x5FF580356C9EF9E7, 0xDF4576DB17102D3C, 0xCA73668DEBAABC08, 0xC73232385D81A0CB, 0x14DC1275B692},
	{0xE1C98F3C564A49CD, 0x76847854DFFF1979, 0x6DD3A9A2B019D474, 0xFEB9A85944A3C836, 0x4C554E80A25DEBB9, 0xC247DBF7209CBA59, 0xE606E4DA4D403E68, 0x5AABAE35231E6C78, 0x1E08F32D70B4CF5A, 0x8EDD7B4A0D3949FA, 0xA549CBAE571FABD6, 0x345B562BB1C2},
	{0x3DDA63D9B718EF13, 0xDB18FF110367AB22, 0xB7A690D61E54A79F, 0xD1FB6E7474B2F230, 0x5D6F299A0CDCA8B4, 0xBDB8418CAF5F5E57, 0x87950067C9C186CF, 0xF190BC388A72A53B, 0x213C2BED23C24CA9, 0xD23020B606F4B8B5, 0xE7CFBF2E0DE63E1D, 0x85488821CA3},
	{0x2E87222F0886018B, 0x2006A348352327DC, 0x19561FC0617F066, 0x390FD49917EAD64, 0xC8962E05BE5C859E, 0x9098F7D823688E17, 0xA367361FE7730947, 0x4C6AE2F141441363, 0x6012B07C3CA41E86, 0xB27005C079FEB1B6, 0x825B9BFC26C91F27, 0x3F1183F8513F},
	{0xB584B99EC62D9D35, 0xA4912F9464B54407, 0x74A10C6ED11079EF, 0x42C935F4B27768B1, 0x355CC43459FC8F5, 0x83FB4BDA9E4C1B5, 0x362D54FAF7050831, 0xED24CAC0D587AF78, 0x520C08B595DEF623, 0x9C385DB3FD4B1742, 0x12B09C148B913905, 0x4C96EA957BF4},
	{0xCCB2CE7886EF239D, 0x47C704F6F492B661, 0x63CDC17F473B3143, 0xE43575F9DEDACC50, 0xED36745A83436E23, 0xB43E3A2A47D36381, 0x6FE523A476BF9F48, 0x82B1F402B3BD2A0, 0xDBE6E728B539ADAE, 0xA37C34807A76FF4E, 0x538EE747772483FB, 0xDB5A5E658A6},
	{0xEFAD9377610BEA32, 0xECEDC7C4592CCB7E, 0xE95A17CFCF63D4AC, 0xC5A8F05F0C78F770, 0x386981FAC3E25272, 0xDD7CD0B4AB9A79D3, 0xD9455DDAD31D0592, 0x994588049EBE35C8, 0x7820458445B7DF6, 0x160C1A4D5730403C, 0xD554560F2A51E183, 0x27B420420DC7},
	{0xED6F9F18F5F91D9B, 0xD0140983A4C9F4FA, 0x4B28E37741FCEC6E, 0x58485D7E32AF47F7, 0xBFCAD3B8EF182455, 0xAA683462489724D5, 0x4F7F01FFFD1C1E61, 0xDB4F6A13AEBB1ACA, 0x40272AE951D7AD1B, 0x430F04D3C2402C98, 0x67C05BFAD21BAAF1, 0x53D2C0CD2C47},
	{0xBA47E3D71B355EF5, 0xEDE1D03F79CAC2E3, 0xF12082CA2D8570E2, 0x33DB874355B82195, 0xAED62CBE488111EF, 0x8FF5A83E933036A1, 0x5FC5432547B62106, 0x3675123E1B5D6362, 0x187E4E847EDF9ACC, 0x92E55D9773F0AB6E, 0x2FB79EE22F3B6FA7, 0x59F9721DDC1B}
};

static const uint64_t table_v_qnr[TABLE_V_LEN][NWORDS64_FIELD] = {
	{0xE85E85E85E85F161, 0x5E85E85E85E85E85, 0x85E85E85E85E85E8, 0xE85E85E85E85E85E, 0x5E85E85E85E85E85, 0x65385E85E85E85E8, 0x85F916A5F6E08638, 0x2609C462575E3A01, 0xD55BC460BFFDB5F7, 0x394FD56C08877041, 0x90A4C64E64B48269, 0x155B182E978E},
	{0xBD0BD0BD0BD074F6, 0xBD0BD0BD0BD0BD0, 0xD0BD0BD0BD0BD0BD, 0xBD0BD0BD0BD0BD0B, 0xBD0BD0BD0BD0BD0, 0xB39D0BD0BD0BD0BD, 0x981077DADCECBF8E, 0x84DD13216CFDC8E1, 0x65BF0DC9ADEA2134, 0x4046BEA875292C81, 0x96FEEECC09D1A583, 0x34F2E90F31C3},
	{0x94DFB461AC813046, 0x794DFB461AC812E7, 0xE794DFB461AC812E, 0x2E794DFB461AC812, 0x12E794DFB461AC81, 0x3D0E794DFB461AC8, 0x70595ED790F5300A, 0x8AC21DF01D745314, 0x960E4F84DA8D1645, 0x5B7E64183952E46B, 0xF22C180908C76B1F, 0x4768EB30E5C6},
	{0x88455121DEEA9B08, 0x788455121DEEABB7, 0xB788455121DEEABB, 0xBB788455121DEEAB, 0xABB788455121DEEA, 0xE53B788455121DEE, 0xC8CE631746684732, 0x7D5D2A56DD6FBE22, 0xD6AD8D86B52354AE, 0xD1BD7280508807CF, 0xA1C516D87EBCCC99, 0x396575A76D53},
	{0xD62298ECA7759C89, 0x4D62298ECA7759C4, 0xC4D62298ECA7759C, 0x9C4D62298ECA7759, 0x59C4D62298ECA775, 0x796C4D62298ECA77, 0xAE6DBC543FEF825C, 0xEC1F7E95B33ED029, 0xCA719FAF0D52EACA, 0xF0376BE32D0BF058, 0xF65BA4732A06CB30, 0x253E29FDA895},
	{0x2D3E21C74B076D2D, 0xE2D3E21C74B0778E, 0x8E2D3E21C74B0778, 0x78E2D3E21C74B077, 0x778E2D3E21C74B07, 0x2288E2D3E21C74B0, 0x114736656CB88D28, 0x65C8937E512FEBC1, 0x306EEB4AB789442, 0xF0BAF2EB5B82DA71, 0xD1557C1ADE14F33F, 0x281EF5D67B92},
	{0x55A7CC22BD034BC5, 0xCF664D2B55543DB8, 0xAD0509637903BB29, 0xCC764B6DB31CC125, 0xA38DE755C3A64634, 0xC5DBC1418728812C, 0x54CDBEA5DAAB0EBF, 0x784DA3FDEE94F653, 0xACD92FABB346E2C0, 0x309E575438F3301C, 0xDE6DA98747E3BC84, 0x78998A0688},
	{0x14E884DC556CCE17, 0xF813B625562FC7FD, 0xD410AA497515C755, 0x4395124C11891A90, 0x392344FF261926C0, 0x5B0104CE685B1520, 0xA32FA4F27454F2D6, 0xDDED7CB7AF975B5C, 0xFEA15A43C78AC218, 0x89AE7B37DEB4C0E0, 0x48661EEF6DCF9520, 0x11ADE16CDC2E},
	{0xD2261143E5B763C9, 0x7BAF0692270E1A07, 0x3E5B763C797E0B76, 0x2270E1A07D226114, 0xC797E0B767BAF069, 0x8FA2261143E5B763, 0x6F563606E110ED58, 0xCD1C300154910AA4, 0xB61EDFD60C5C6096, 0x12CADF965191532E, 0xA045CA5ADE8C8CA6, 0x2CDC07A09F1D},
	{0x952C939D91AFBD6C, 0xDB189B941022B733, 0xD91AFBF752331AB4, 0x41022B733952C939, 0x752331AB4DB189B9, 0x91552C939D91AFBF, 0x6376CA8518BA8EA, 0x4C144651E91CACE2, 0xE0E13E3F7AF8AC34, 0x4C7E7CDE3D51F816, 0x11E5E216628DA39F, 0x3C9C0D30F762},
	{0xC605DAAC2686111, 0x653B01B6DAB28C3B, 0xC50A3A32DE5449A4, 0xB6EBC1C42E96427E, 0x8FEFFAE8A88084D9, 0x58EA39884624C641, 0x9EC005ED22C5BC45, 0x1633B0FBD77D6D9D, 0xA57C267B6FBA36DB, 0xF94465FAC0B899A2, 0x34A4BE281ED17637, 0x55D1BDD26591},
	{0xC709286502B5CCB, 0xFDA560011E994595, 0x4856E21B8EEF3C44, 0xE6FF4B98A9EA2DE2, 0x9887CBBDFB34929F, 0x7E47E75497B06B9B, 0xE723E79ABA5D9470, 0x2C05C07824B8E6E, 0x5E4E45149E3C5445, 0x2452AE29E3055EA4, 0xEF90DE3635F81ED5, 0x2811146405A3},
	{0x9807214A3A1C1CF3, 0x7A0E708928E1F773, 0x6B79A82BEC4DCC5F, 0xE98AFBE2931E7695, 0xF1BFAF7F0D6FE513, 0x729355B40CC74834, 0x7355193ADF5FC317, 0xAABE07A1788CB696, 0x22D720814DB72B6B, 0x37DF09E734DE45BA, 0xFD056FDA4F240CBA, 0x68ABBB7B5C8E},
	{0xC3777B85DA951B35, 0x729E2EE622CB06CE, 0x142662CA9F3EC02B, 0x79B13BB964738D59, 0xD0D82AA614F52FF, 0x2073598189B312EF, 0x187D2FB5FF497E99, 0x55FEA80CEF2A4B0E, 0x6B197F79D5FAF833, 0x254447C18292F844, 0x60F938B568EE1543, 0x71826A2ADCA},
	{0x9C02260C68B4DBCD, 0xEAD5B56BFB777BE, 0x3BC3BE5177278FD6, 0xDAB6622B90CD39DE, 0x90CC3EC6ACA6DDD0, 0x6E158FF767CE5D2C, 0xA8B669D3FF7489EA, 0xAF133837592C7ADA, 0xFBF80D62062EC026, 0xC3512D9A00005FBB, 0xB759F49E11D399A6, 0x2B5BC9341B5D},
	{0xC32617FB97A7BA46, 0xDC97C62323C3A3A3, 0x100C4C10F8AD3EB2, 0x303659A310A3583D, 0x12DA436E2F3F2E72, 0x4D50F367A011A161, 0x8328C3F9F44A3084, 0xF982C9DA22658656, 0x5EB7238A18DD90A7, 0xB9156E1D1B343898, 0x57243EC083BFEF2F, 0x3DE52126EF4},
	{0xBBD06880C51C680B, 0x7A81A7C7078464C3, 0xFDECD7041B4ECF3E, 0x2C5C3D04ED2F2A03, 0x5D174A6A4E7F7DB4, 0x1B52D2539D1EA2E8, 0xA5CC20018839AB95, 0x67625D02AE4CF980, 0x186074DCBD384821, 0x9E5E42872A3C7A83, 0x545832703A5A56B9, 0x27681CF8D9FB},
	{0xD629BC89002BCCEA, 0xB88D005E2C3A8F32, 0x56386D851D5B66D8, 0x60ED69D4ABC2EE09, 0x1731219EC24A5538, 0x966D87BCF622EA5D, 0x62A6B07E4C8B3CEF, 0x6EA16F4B509B81A3, 0x7A97D35B5CDE162D, 0xE25FB4ACF2AF1E4F, 0xC48E64140BD3474E, 0x21FDD6C8FB82},
	{0x9288477180A6C951, 0xB2D2DE965A033D61, 0x3BA1694AB827311A, 0x70A5B57FC1C70CDC, 0xBFD11385625A2E5B, 0x2DDDAFB04A38853F, 0xF21823FF1B225FDC, 0x5F007A2E1641AEB8, 0xC3E5684AC220ECAA, 0x78179AF27DD10F63, 0x96DC98F67C36644F, 0x7D6D89BB770},
	{0x829327C9D19D0226, 0xD65D25DC7C2E4992, 0xD5DBE128D0A23BA2, 0xF6DFC62FD21334A4, 0x651EAEB5C3A9A583, 0x180C253598055EB9, 0x3F73FA2EEF9168FA, 0x4318BBEE307725D0, 0x54DD26A9FC5450F7, 0x76ABA5758680DBA0, 0x353ED351A6A5F735, 0x2FF3A6573DE4},
	{0xB9993936AA2301F8, 0x7B319EACE5E9B127, 0x12E616BD697BA4E, 0x811B1F35F96913B0, 0x474DB18B6AB00C82, 0xC1FFF493FE2C0901, 0x796424D7A11AEDE9, 0x61076638D85C4C48, 0x64774660A3BE93D0, 0x43FA3F433B9D067D, 0x47A0B426F92F7600, 0xA280AE8D170},
	{0x3581912DE993F071, 0x70141CFFC8A2ECAF, 0xC3EB4B7D9EAADF4D, 0x39E0D5685C6769F8, 0x44208A1289B360A9, 0x1495F58B3248D82B, 0xCBC66BB98CDA2E99, 0xCC3F8763F217FA81, 0x495812161E7D7E11, 0x255FA1A2A5D70135, 0x240F5A6F7EAE5BA4, 0x4C1F4DD638F5},
	{0x5B9845DD9A79003D, 0xEF65F03C8F56ECBD, 0x30F64FF52B455D62, 0xD45579863FAD5D92, 0xC798CD9C72EB564D, 0xDD15C4F63ED0D645, 0x9F35EC5BCF419520, 0x2759D2CB3306DFB3, 0x6A4CEE0C84723002, 0x62AE428E1405808C, 0xA93787CE06A12315, 0x4885205A0A7F},
	{0x3184F5FF10FC6AF3, 0x143B921A36E08BDC, 0x4DA6A729B2406BF7, 0xE6678BF19EEF357A, 0x96091214F1C2F4A2, 0x9F07E61747E02986, 0x48FA86D281046604, 0xFF69050DD5E29481, 0xF97F744AB920445E, 0x16B828177FE35D68, 0x2A44107C12A0C668, 0x65CD488721CA},
	{0x4E5420F1827B439D, 0xF487A4ED360B1C4D, 0xC55EFC7F8D8112D0, 0xED67A91DBC5910C6, 0x9E9A78EFA44FC34F, 0xD40F65D4211D8623, 0x5C21F3F0FE48DBC9, 0x9CEA4A83954032F, 0x68A443A160CCF02, 0xBA2D35302646DABE, 0xB27CC0AAD03BA372, 0x2E3F49905A08},
	{0x40785DE28D4C9530, 0xA6405C1C01E10071, 0x3C209F250C73B2C8, 0xBF22E6EBADC20A64, 0xD1DDAB04AF2AB602, 0xA33C04772868D58C, 0x98DA3CE309D547DA, 0x4DD504F6F173D486, 0x283B355D938C1316, 0x99001A20441BA13A, 0xE8C99EC8D3BB4B46, 0x6F688BD1C61},
	{0x459A64577F98DD28, 0x3FD6890B60AA7022, 0x32B84AB229B33D20, 0x7CA6E1524F6EBF03, 0x4F0242DD469A4D8B, 0x53EAA4C1EBC77C19, 0x62C19F1C435AC30D, 0xF230AE4E1F1BF94B, 0xCF7E083E29C16D7E, 0xCA42C2833B78234E, 0xD01EFFD11DB566A8, 0x224E1F2FFD2E},
	{0x6EDAA71406539762, 0xB74F189440621EB1, 0x345F3262EC4F13E0, 0x23185407A289E9FD, 0xD105FEB3ED85CEC2, 0x11F70813D04E49FC, 0xD99654FECB39AB46, 0x734ECF9475139205, 0x54E95C9BD579AED5, 0xE044D2C9270C1251, 0xE60CAFCEA4BB3508, 0x552B57BCA931},
	{0xD91B6C9B924D91B6, 0x9B924D91B6C9B924, 0x91B6C9B924D91B6C, 0xB924D91B6C9B924D, 0x1B6C9B924D91B6C9, 0x312D91B6C9B924D9, 0x17D193651E9DCA60, 0x8DF224D4272EC01E, 0xCBD2BC92A7A778F6, 0xA59CF2B6356CD90E, 0x9C712EB8F35468FD, 0x5C2C61931F7C},
	{0x249B236D93724969, 0x6D937249B236D937, 0x49B236D937249B23, 0xD937249B236D9372, 0x9B236D937249B236, 0x7D4249B236D93724, 0x6F78BF6586101FB3, 0xDEBA77A01FF372B9, 0x2809BF0C10D132CA, 0x15E12E137474B550, 0x32E144163D1A7A6C, 0x83BCB784FD},
	{0xA014E250439995F, 0x45FC65F1FFF85333, 0x6BCB9B2F717EC6B4, 0xE17F29C4A3AAB256, 0x84007F11103DBAF3, 0x8F00A6A9F3A093F6, 0x9726358A43891D51, 0x897A6562E06B392, 0x9C43741CF4F314B9, 0xF134466DE6EFE2AB, 0xF564982D2C7FD077, 0x1314C36F69C9},
	{0xA43641B1D7CEDC7D, 0xF0874E7045509B0D, 0x792E7B86FF0D73E7, 0x7B8EC819E2656382, 0xDB846DE5527F9573, 0xCBAED11FBDC7B5A7, 0x9CF8B6F734116414, 0xAB4016ED22D2B78C, 0x901482B46AAC884D, 0x6F313D2D60B58E49, 0x8226CE453AFF56F5, 0x541B0694E059},
	{0xE39251FB5F896DD7, 0x861156ADC66BF25B, 0xD3B6E410C9EB77F6, 0x149A5B2E22C5FF6, 0x39BF19B234BC83FE, 0x6FC6DDCDE9500C15, 0x33D635192E7CA410, 0xF40BB7C04811EFE5, 0xB95BDBAD0678F5E3, 0xFFEE8DDD2907E2DA, 0x71BB095C5885E70A, 0x3006B1C7034E},
	{0xD681EED8B087DAC7, 0x90726086FB0A1D9F, 0x643AD6038FE15FF2, 0xF6524CCDFEB45CE0, 0x511BD3AA0B479432, 0xFDAA3B25B585A917, 0x16C0CA87EB477009, 0x5DCAD9871ABB0038, 0x5BA46393DEEC08B5, 0x9AC8A0B793849197, 0xA8F7FEDA5FA4EA5A, 0x3A742CC9DEB9}
};

static const uint64_t v_3_torsion[TABLE_V3_LEN][2 * NWORDS64_FIELD] = {
	{0xD89D89D89D8A493E, 0x9D89D89D89D89D89, 0x89D89D89D89D89D8, 0xD89D89D89D89D89D, 0x9D89D89D89D89D89, 0x8B389D89D89D89D8, 0xB4C6B9529F01D8C3, 0x1399AE2626262260, 0xDE85321D9D8185DF, 0x8451DC3FDF91784A, 0xFAEFD5E487381389, 0x6319EE6373CB, 0x3B13B13B13B1248D, 0x13B13B13B13B13B1, 0xB13B13B13B13B13B, 0x3B13B13B13B13B13, 0x13B13B13B13B13B1, 0x7A4B13B13B13B13B, 0xA30792A3BBCAECC7, 0x8E3262972F905537, 0x6EC1E1420B7BAD51, 0x3369C4F4190692FF, 0x9D6D588BD01A282C, 0x28F2E2C80A9},
	{0x673D45AA630B09FB, 0x5673D45AA630AE95, 0x95673D45AA630AE9, 0xE95673D45AA630AE, 0xAE95673D45AA630A, 0xD4595673D45AA630, 0x6D7B45385F91ACC, 0x7FDE0F3B044702EF, 0xF1BFC4766FC2E18A, 0xD67138B8790E3167, 0x9EF75C6F0552A406, 0x2C5A25459904, 0xF6C4681424EE5801, 0xAF6C4681424EE5FA, 0xFAF6C4681424EE5F, 0x5FAF6C4681424EE5, 0xE5FAF6C4681424EE, 0x7FAFAF6C4681424E, 0x71E7F021FC96E1CF, 0xBD317223E4665091, 0x4D1B06EED834E4E, 0xF423F47B02EEF4B8, 0xA418B4D55C23C88, 0x94F95559E31},
	{0xBC8EA75EFC1DB814, 0xDC562840F895B742, 0xEA75EFC1DA922F50, 0x62840F895B742BC8, 0x5EFC1DA922F50DC5, 0x1B3895B742BC8EA7, 0xEF4E35288013377, 0x7034C1EDC7954F9D, 0x404E07FFDCD9C4B4, 0x6C0D3C3112E9FC8B, 0xE93593735AD57C40, 0x5E74FB7D527D, 0x5C835ABF61CF41ED, 0xDF2950278C2E9C12, 0x35ABF61CF458FB68, 0x950278C2E9C125C8, 0xBF61CF458FB68DF2, 0x7E9C2E9C125C835A, 0xCA8064284EDAE000, 0xB011CE81CE7860C5, 0xF2D3F4033A9B8AE2, 0x2FF7880A1DE06DD0, 0x7182DD9DC3FCB499, 0xA4A77D973AB},
	{0x8AE21EF6EA938B09, 0x37966C6ED5186501, 0x69DCB18F74253FB3, 0x3188241B31A56443, 0x9156C7D752C811D, 0x9CFAE79AFE751DE1, 0x8BF44341193A82F9, 0x34DAFBE7EE16D274, 0x272F0852BA272D8, 0xA0F247E8810D0FE5, 0x48E124BE4CC4D188, 0x4FDE2C9A5389, 0x44369DCB18F7403C, 0x11D3188241B31A56, 0xDE109156C7D752C8, 0x993912AE79AFE751, 0x34E708BDAC04CC86, 0x8C7E4CE5A9BBC962, 0xF269836E4C98DDD3, 0x45188E7965A38A0, 0xDAB83EB832EB62A0, 0xCCEFE1931E93559A, 0x775966B73B29F6D5, 0x45DBC312B900},
	{0xB71EE1F7F5C9327, 0x8A5569830CE94076, 0x12961C17A5932157, 0xDD79C7E8EBB6086B, 0x2BEF8AB644F43951, 0x37277CB68D98CE33, 0xDE633F0A815B7506, 0x61740E19C3F2EFA4, 0x61B1E49FB5E4EABE, 0xA015247782FF386E, 0xE25E2C7B9F068D08, 0x5A5611C2C7A3, 0xD213F53791E68D00, 0x2EDE2F635D1EFA2E, 0xDA72EC5F8871204D, 0xDAEBFEACFFF3E491, 0x5F948C8E72BBF1CF, 0x4A440214B6EE743F, 0xD224FC96C70931D9, 0xEA00C3F488138831, 0x9D46231784C01FDC, 0xB12AA781AEC3B474, 0xE312A59BD123CE5A, 0x1FED35710EB4},
	{0x72F7284A69378371, 0x3D4A5A89547E058, 0xB12975E3DAE2E152, 0x599E71CBADA705F9, 0x4ACD0B0B6F42CE9A, 0xE5285A0245577966, 0x6C038D909BB02FA1, 0x124374A6F770AF89, 0x99103DFE37CA79AF, 0x147E8A248FDBCAAE, 0xEB9B0A578010CFB9, 0x37F0B3678B6F, 0x5D470CF2D274DD99, 0xC8516258CB0C2B2B, 0xB43C53B94E11D2A6, 0x5ABE2DE626199F2D, 0x29D6C7C473EDC98, 0xE0376A892583E1DC, 0xC0CAA571F277C6A0, 0xD4F6AD7F88406328, 0xC9A2779B2F2431FD, 0xD7757D28491C9C9C, 0x3710D688A0ED993D, 0x4ADB9FE768DE},
	{0x81F1C30C09FD33FC, 0xDBF51DCE3A9149C9, 0xE9CF78A0524C35B8, 0xC02099DC646FDE01, 0xB0594156B5517D78, 0xCDF2CD6BD4510E26, 0xA62E43C6D4ABA685, 0x6282F81A9AB8658, 0xD6D76B986C3AFC2D, 0x82003479185AC489, 0x68A227ED5A5D8D62, 0x4CB44C725522, 0x248D49A3572E2D8B, 0xD7958CBA82F9C02A, 0x8CAA9BD2ECD45A98, 0xD568F26B9E3F23BE, 0x6C2D5F812C90C433, 0xF1565E72D2FCBB81, 0x27E816D21960503D, 0xC557356BF112F986, 0x4C0F35DCE29E0B8B, 0xB5D66BBE22866AEA, 0xA93F73DFFDAB2E0A, 0x615CF5C957D3},
	{0x516EE2A1BE021E2A, 0xF602D6E151658AD7, 0x26B24FB3189298B0, 0xD965FAF4CE85C95, 0xEFEF3DB9B3A4B775, 0xD851DD3FA8E08D31, 0x2DA32E186C78CE68, 0x143617DD482EE422, 0x667237E94436FC75, 0x1AC3465D96DCFDBB, 0x56D74ABF63E8D5D3, 0x5D7C60C0D2E4, 0xA30146117F0077FD, 0x9BE366E0CADBC188, 0xEF64080B2EACAD26, 0xA8FD83BF0D513C6, 0x877CB3DDB56170B2, 0xB35C3C73A225B6FC, 0xBF79CBB465569B88, 0x498384E224080C8B, 0xB2F0B94988649443, 0xB3DCD435A017EB4F, 0xE9941938991157FB, 0x21C1C22E5E5B},
	{0xD40E007BD838CEA, 0xA6EFBD2F6C46FEC8, 0xDC1072B787D7CD36, 0x7204C616DD691C29, 0xF3C600AA941422BD, 0xB810967F5BD8F3A2, 0x53CCFBAE4D3DD147, 0xCC071945DE01CFF7, 0x6DD53CCD52E214A3, 0xB2C2FB56192E2E38, 0xD2B99B61C093CFD5, 0x51BE0034F0EA, 0x8107763EA961F872, 0x74B86F2CE50392D0, 0xDB4527F030D9F82D, 0xEC3F66A797FB122F, 0xF4DE67CF2F609807, 0xEFC422F586A5B1F2, 0xB765C11D7152D2E1, 0xB0A5BFC4FD2AC62B, 0x589DF808D4BB351A, 0x1165BA9E7035FA93, 0xE9B487C366AEC785, 0x586D2918F33C},
	{0x6E3302C6FE0C0C01, 0xDFAC66DC6CEDA28C, 0xDED64C5EF280083F, 0x8B39B4C197F785CD, 0x11B93A8C64FD9206, 0x9AC84CC06813FCFF, 0xCF85682DC9D9B864, 0xC23496D96970C301, 0x5194BE86F6E26D90, 0x4DB579722DE163E0, 0x66DAEB4660E533AE, 0x15B525F2A541, 0x7C758374FFD93EC3, 0x2C7354CAF23E0942, 0xBD93A801DBC916D7, 0x47E4D764786D5DAC, 0x712497FC82C0471, 0x87311F4B3AF3FE32, 0xEFAEC8155C91650E, 0xE2169C5AD02053FE, 0xC96489E29375ADEB, 0x5E7D1381050F03CE, 0xB88C1A56DDD05F31, 0x6070D9E2F41},
	{0x9ED7073D4D892D5E, 0x7A8ACF5D59CC31D7, 0x149DB0BEB6B3A2E1, 0xC059D51CC46D3357, 0xEE43544B33B041DD, 0x5CC5EB8BC024538F, 0x4B759EA5B9EAB631, 0xA335C8BC0AEEAFBC, 0xB03F67E326524765, 0x2C790108FA72A4B0, 0x789B5DA805C4192F, 0x64DEC5C9AB82, 0x36DAB776C6A6E1AD, 0x70788D740F12BB16, 0x678109B1BC9AEB8, 0x6CC747669CD9ABEF, 0xCBF25E90DDFDB9BA, 0x9186ABE61B8B423E, 0x29E06834AA38279D, 0x45176543D221EEC1, 0x761F084D3249B19E, 0xB968904364D0E464, 0x5DB85DA70B052289, 0x521E2DDB50A5},
	{0x395783F790D29774, 0x112CEC9A5FB0FD8E, 0x2286B13D7C636B50, 0x3BB1C1FAABC848AE, 0xCEBD9A801EAB03DC, 0xD0D0F3042DC47BD3, 0xBA7CC91C76FBFC6F, 0x7E4B81D9BC252CC4, 0x791DA3C97B45B52E, 0x76E4B2A8663A7CDB, 0xB18F9D7CDF16DE77, 0x6738B89E9A90, 0xC5C400C8DEF67ACF, 0x7DB5C96374C369C2, 0x4DE65F1497C0251B, 0x3D6A0B0AA5A7DC1D, 0xCFF42CDA41FF0694, 0x13585BF9A20DF747, 0x29F5EA2387657EA2, 0x85F982D144BC9D02, 0x70BA26172B51F920, 0x7CDC5B7AD3069E68, 0xDA452B30229CE4DE, 0x5308C36EA06B},
	{0xAC4F402D28A1B7D9, 0x1F9692D5D204DA19, 0x7B09C49F36556C65, 0x74E03DD0CC93BAA4, 0x2857AF2E2C697F78, 0xB5B97969D9A74314, 0x9B0B4C1B8B9C5804, 0x12FE23955636F0E, 0xB8E8FEC4456834F9, 0xF6E7C5B893C1F2AC, 0x5A45A12C0CF9B64, 0x4242B7D98CD1, 0x895BC47E5E99FE58, 0xCFD306BF78ED7A, 0x43394EBA61FCD38C, 0x83E94C74E7C77585, 0x9E944616E2627346, 0x9BA6856CF3432676, 0x2185C6AFF5533077, 0x67F7CB233436A703, 0x52D795D168AD54FF, 0xBB7BF59763DC0670, 0x79561A9B5335EFFB, 0x6F8E14E2EA1D},
	{0x52170E4682673A4B, 0xAD4577028668C00, 0x105830C4E5B35043, 0xE44B518DCE2097C, 0x1DC57628AFA086C9, 0x7017D10868A9652D, 0xF52DAC7D308B4DED, 0xE5F8D6B44798B8A4, 0x860E5513916A8B0B, 0xEF1A2AE69403037, 0xFD7E1676F8DB8D34, 0x1DA40260AF34, 0x438DF92CAED981E9, 0xE0EC927187AF9BD2, 0x2C47A2EB752165A, 0x10A814019E74674B, 0xAD43DCF5C2393FE, 0x9EF40A6C20918C14, 0x827A35EE6A2A622C, 0xC5367CAADECCABBA, 0xDEC9E5F60CC017A0, 0x89393E2776CC5F44, 0x7AD261CF40C946E0, 0x69EC298FC2C5},
	{0xE370D28A643127FE, 0x6484E527E4F15551, 0x3F59642BD5A34627, 0x85A03E71ECC4BDC1, 0x7B2E731CBACBC6B9, 0x70F5FDBD8FDA90B6, 0x24E12F3C9683795B, 0x35FAA7C6879A3C7C, 0xFF53C1E88A24C6E7, 0xE5DD61BDDC562FD5, 0xE68B1C39660B3261, 0x6259E2A86D74, 0x2982F0136CCDEF7A, 0x8848824CB25EDE31, 0x7BF8BC5D9CE7B4CD, 0x5A3A0205A555D497, 0x9240434A161CAB24, 0xFDB3CBB75FE9869F, 0xC97131D9A60D18C9, 0x9DB915412B8CB0B8, 0x5B7E4834AD58F24F, 0xDF6E9178CA871824, 0xF1259454624AC092, 0x1327C8999EB4},
	{0x666280546751F204, 0x3124828432952042, 0x628126D710FA6B0A, 0x1D3B96F1B28EB258, 0xA3E698E6B79C0621, 0x5CFD4588D9A09890, 0x43A7B8C256D49EBA, 0xE2F94301C0032B74, 0x8953956AC6CA7C77, 0x7F8015DC765F106C, 0x82E4DDDD36ED8083, 0x133AE4EA2F71, 0xDBF063D1F1AF17B9, 0xCEC32E934E871629, 0xE2671C831AFAA693, 0x960B975DBBED5813, 0x203E4A33C527B48A, 0x423CD368EF5C02DB, 0x361633B23B9CEB74, 0x5F11CA4E0B5ECBDD, 0xAD9328EC4662A397, 0xEA2509539AEEDDBA, 0x15BB2346F14D8146, 0x2F2B9A05EEE},
	{0x795D912BDEF015D7, 0x65E6C0F35599987E, 0x8BE785422F397093, 0xCD93F70F07BAA7DB, 0xF1A8945586B3C277, 0x26353E447C9BF58B, 0x4CD879A487B985CA, 0x9E680A1F08FF3F3F, 0xEA67CD5F51356E12, 0x501DDEE09936B7D1, 0x559A868F99683C3A, 0x23FA0B67D681, 0xEDDAB05F06B5A605, 0xC6E7FE44E0E20ACE, 0xDA2D7FCAF5C0C0EB, 0x7507BC508BE5F429, 0x7516F14BC559D8F0, 0x55DB0692A07CC717, 0x12F7611288A3B08B, 0x34ED3DD86C14B639, 0x2BCFE7EA6234CAAA, 0xA20E37F49CD7B313, 0xED785E90CA869C5F, 0x524FB4A9B68D},
	{0xB24323DEA264A910, 0xFBB77EB1969BA34D, 0x6DFE6BD2DF0E62D, 0x4BA1D9212AAC1737, 0x6604456CC168CE0D, 0x1F045C2902E70B1A, 0xC30F3CF005747BBB, 0xD0808A1E60231D09, 0x616E5202B8B68AB9, 0x91279DE1E1F59C61, 0x17039A54DE74E351, 0x483F7B722965, 0xCA63AE6778B0F84E, 0x3920204D38EBDF19, 0xFB1FE6BAD523F1EC, 0x8B34EDE136AC2733, 0x7409325BA76B71BB, 0x987FACF98158EF9B, 0x3EBAB81D7CC6988B, 0x518A2D688F371575, 0xC77B4766A9D72BCA, 0xAF7343FE6E77A7C4, 0x54516072DE9FE449, 0x52E493B8BEDC},
	{0x95B9273E98BEE101, 0x6CF2C2CC7C2F0D29, 0xEFE3B76E594AA11B, 0x3DDEDCDD1FE42241, 0xF7E15F986E39CC89, 0x3CF9E152B6FD333B, 0xE7F2CF0844AD69DD, 0xB792DFF3C762D02E, 0x3888F4A332FD9030, 0x8A4CC0E4C437575A, 0x833E2BA7BAF41403, 0x135FE5EBD4BB, 0x657D9769E52DAD91, 0x84951CC10B514173, 0x7678CDEC0CC5511B, 0xD4E7DB99FC763848, 0xC6ADD473DF8087CE, 0x842DE2D06829FA76, 0xD086DB2A4651BE48, 0xD0399255E5DAD344, 0x6B2EAEB21B8BB524, 0xF6DE9148F0694AEF, 0x5A85093194755805, 0x2570D86C9FCF},
	{0xD665CD614A703CBD, 0x7251A4FFE04E2B30, 0x8AD6A13EAA0B07BC, 0x2AB5112D91260BE2, 0xF31D78441E75FDE5, 0x981D1D465A8768E6, 0x7AD08CCEE352CCDD, 0x31C6C60ACD409AC7, 0xDFC10AD642C330AA, 0x16DA3C495AE40C44, 0x89AB4B294D700C6D, 0x13081265A555, 0xB649623190FAD2EC, 0x9E0A9F4A626C11FD, 0xEF8A6A8092D66371, 0xEB9370EA38CC1EED, 0x74BF8D8667FFF12C, 0xF931EE21E90FE5CA, 0x5E180EC10EC59AE0, 0xBA6729A7EF221E52, 0xAEAB0D0AC6ED85F9, 0x2401EAF62859B015, 0xD309B49CD60C1B34, 0x2CBA9B452CC8}
};

// Setting up macro defines and including GF(p), GF(p^2), curve, isogeny and kex functions
#define fpcopy fpcopy751
#define fpzero fpzero751
#define fpadd fpadd751
#define fpsub fpsub751
#define fpneg fpneg751
#define fpdiv2 fpdiv2_751
#define fpcorrection fpcorrection751
#define fpmul_mont fpmul751_mont
#define fpsqr_mont fpsqr751_mont
#define fpinv_mont fpinv751_mont
#define fpinv_chain_mont fpinv751_chain_mont
#define fpinv_mont_bingcd fpinv751_mont_bingcd
#define fp2copy fp2copy751
#define fp2zero fp2zero751
#define fp2add fp2add751
#define fp2sub fp2sub751
#define mp_sub_p2 mp_sub751_p2
#define mp_sub_p4 mp_sub751_p4
#define sub_p4 mp_sub_p4
#define fp2neg fp2neg751
#define fp2div2 fp2div2_751
#define fp2correction fp2correction751
#define fp2mul_mont fp2mul751_mont
#define fp2sqr_mont fp2sqr751_mont
#define fp2inv_mont fp2inv751_mont
#define fp2inv_mont_bingcd fp2inv751_mont_bingcd
#define fpequal_non_constant_time fpequal751_non_constant_time
#define mp_add_asm oqs_kem_sike_mp_add751_asm
#define mp_subaddx2_asm oqs_kem_sike_mp_subadd751x2_asm
#define mp_dblsubx2_asm oqs_kem_sike_mp_dblsub751x2_asm
#define crypto_kem_keypair OQS_KEM_sike_p751_compressed_keypair
#define crypto_kem_enc OQS_KEM_sike_p751_compressed_encaps
#define crypto_kem_dec OQS_KEM_sike_p751_compressed_decaps
#define random_mod_order_A oqs_kem_sidh_p751_compressed_random_mod_order_A
#define random_mod_order_B oqs_kem_sidh_p751_compressed_random_mod_order_B
#define EphemeralKeyGeneration_A oqs_kem_sidh_p751_compressed_EphemeralKeyGeneration_A
#define EphemeralKeyGeneration_B oqs_kem_sidh_p751_compressed_EphemeralKeyGeneration_B
#define EphemeralSecretAgreement_A oqs_kem_sidh_p751_compressed_EphemeralSecretAgreement_A
#define EphemeralSecretAgreement_B oqs_kem_sidh_p751_compressed_EphemeralSecretAgreement_B
#ifdef USE_SIKEP751_ASM
#define USE_SIKE_ASM
#endif

#if defined(_AMD64_) && defined(USE_SIKE_ASM)
#include "AMD64/fp_x64.c"
#elif defined(_ARM64_)
#include "ARM64/fp_arm64.c"
#else
#include "generic/fp_generic.c"
#endif

#include "../fpx.c"
#include "../ec_isogeny.c"
#include "../compression/torsion_basis.c"
#include "P751_compressed_pair_tables.c"
#include "../compression/pairing.c"
#include "P751_compressed_dlog_tables.c"
#include "../compression/dlog.c"
#include "../compression/sidh_compressed.c"
#include "../compression/sike_compressed.c"
