/********************************************************************************************
* Supersingular Isogeny Key Encapsulation Library
*
* Abstract: supersingular isogeny parameters and generation of functions for P610_compressed
*********************************************************************************************/

#include <oqs/rand.h>
#include "../../oqs_namespace_sike_compressed.h"
#include "P610_compressed_api.h"
#define COMPRESS
#include "P610_internal.h"
#include "../internal.h"

// defines moved from P610_compressed_api.h
#define CRYPTO_SECRETKEYBYTES 491 // MSG_BYTES + SECRETKEY_A_BYTES + CRYPTO_PUBLICKEYBYTES + FP2_ENCODED_BYTES bytes
#define CRYPTO_PUBLICKEYBYTES 274 // 3*ORDER_B_ENCODED_BYTES + FP2_ENCODED_BYTES + 2 bytes for shared elligator
#define CRYPTO_BYTES 24
#define CRYPTO_CIPHERTEXTBYTES 336 // PARTIALLY_COMPRESSED_CHUNK_CT + MSG_BYTES bytes
#define SIDH_SECRETKEYBYTES_A 39
#define SIDH_SECRETKEYBYTES_B 38
#define SIDH_PUBLICKEYBYTES 274
#define SIDH_BYTES 154

// Encoding of field elements, elements over Z_order, elements over GF(p^2) and elliptic curve points:
// --------------------------------------------------------------------------------------------------
// Elements over GF(p) and Z_order are encoded with the least significant octet (and digit) located at the leftmost position (i.e., little endian format).
// Elements (a+b*i) over GF(p^2), where a and b are defined over GF(p), are encoded as {a, b}, with a in the least significant position.
// Elliptic curve points P = (x,y) are encoded as {x, y}, with x in the least significant position.
// Internally, the number of digits used to represent all these elements is obtained by approximating the number of bits to the immediately greater multiple of 32.
// For example, a 610-bit field element is represented with Ceil(610 / 64) = 10 64-bit digits or Ceil(610 / 32) = 20 32-bit digits.

//
// Curve isogeny system "SIDHp610". Base curve: Montgomery curve By^2 = Cx^3 + Ax^2 + Cx defined over GF(p610^2), where A=6, B=1, C=1 and p610 = 2^305*3^192-1
//

const uint64_t p610[NWORDS64_FIELD] = {0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0x6E01FFFFFFFFFFFF,
                                              0xB1784DE8AA5AB02E, 0x9AE7BF45048FF9AB, 0xB255B2FA10C4252A, 0x819010C251E7D88C, 0x000000027BF6A768
                                             };
const uint64_t p610x2[NWORDS64_FIELD] = {0xFFFFFFFFFFFFFFFE, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xDC03FFFFFFFFFFFF,
                                                0x62F09BD154B5605C, 0x35CF7E8A091FF357, 0x64AB65F421884A55, 0x03202184A3CFB119, 0x00000004F7ED4ED1
                                               };
const uint64_t p610x4[NWORDS64_FIELD]            = { 0xFFFFFFFFFFFFFFFC, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xFFFFFFFFFFFFFFFF, 0xB807FFFFFFFFFFFF, 
                                                     0xC5E137A2A96AC0B9, 0x6B9EFD14123FE6AE, 0xC956CBE8431094AA, 0x06404309479F6232, 0x00000009EFDA9DA2 };
const uint64_t p610p1[NWORDS64_FIELD]            = { 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x6E02000000000000,
                                                     0xB1784DE8AA5AB02E, 0x9AE7BF45048FF9AB, 0xB255B2FA10C4252A, 0x819010C251E7D88C, 0x000000027BF6A768 };   
/* OQS note: commented out to avoid unused error
static const uint64_t p610x16p[2*NWORDS64_FIELD]        = { 0x0000000000000010, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x3FC0000000000000, 
                                                     0xD0F642EAB4A9FA32, 0xA308175F6E00CA89, 0xB549A0BDE77B5AAC, 0xCDFDE7B5C304EE69, 0x7FDB7FF0812B12EF, 
                                                     0xE09BA529B9FE1167, 0xD249C196DAB8CD7F, 0xD4E22754A3F20928, 0x97825638B19A7CCE, 0x05E04550FC4CCE0D, 
                                                     0x8FB5DA1152CDE50C, 0xF9649BA3EA408644, 0x4473C93E6441063D, 0xBE190269D1337B7B, 0x0000000000000062 };
*/
// Order of Alice's subgroup
static const uint64_t Alice_order[NWORDS64_ORDER] = {0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x0002000000000000};
// Order of Bob's subgroup
static const uint64_t Bob_order[NWORDS64_ORDER] = {0x26F4552D58173701, 0xDFA28247FCD5D8BC, 0xD97D086212954D73, 0x086128F3EC46592A, 0x00013DFB53B440C8};

/* Basis for Alice on A = 6, expressed in Montgomery representation */
static const uint64_t A_gen[6 * NWORDS64_FIELD] = {0x31C8AF7FFC0DE9FA, 0x8A8AD55D2AC8A709, 0x95A4DC49B64E5B2C, 0xF08C77AAE90ABE83, 0x675E4FF97C95845D,
                                                   0xF8A22591248401F0, 0x73F573A4FF34A84A, 0x37D18A6C3D989158, 0xEE73973862A3E95, 0x24084FCCB, // XPA0
                                                   0x4B8C9CED6DEF0B8B, 0x652C800D926AB992, 0x3DFA6D6B8FD37D80, 0xA30C578CD98EFD79, 0x9FC067E58CCBD32E,
                                                   0x2B0599AEAF150FDB, 0xBA321B31886F3292, 0xE0011F56247547A1, 0x28CA0747910BFAE2, 0xFC020A14, // XPA1
                                                   0x2728178178DEAFBD, 0xD377C4656DBC71F0, 0x968642007B807932, 0xB8B04B1039062A21, 0xF824771B468A977C,
                                                   0x260F1C50354F46AB, 0x78A3D37CDBBD4DC5, 0x1FB1BAC6851BA175, 0xA73444F1CAC4A10, 0xF3A5C2BB, // XQA0
                                                   0x4F828B752E825BB4, 0x82CEA210AC766C69, 0x8B1BBC87DAD8BEDD, 0x9BFC5B9CE215B423, 0xF7E1BCC0C541177C,
                                                   0x7727E3A0F1A1AF24, 0xFBCFE4177D2B0221, 0xBB15BDCC160D902A, 0x3FE1467B4A911446, 0x1A495CB35, // XQA1
                                                   0x38687702D78D1A93, 0x58C09FD23B1E1B56, 0xC54917327D5C0FAB, 0xB6D55B7BE801A3C, 0xEB3AE21C8B93E9E9,
                                                   0xECB45AD6D24FF76A, 0x850645B4F39EC5F2, 0xE6F78202586C9B3A, 0x2923209A250F7F66, 0x26FB150F, // XRA0
                                                   0x5AC7B27F9096F718, 0x487DDD2820132C83, 0x6B21AC48569E12D8, 0x57B54E5A827D1CD9, 0xDB7C4BEB143E4130,
                                                   0xB6781CA1DA245EAD, 0xCC09878A2A6D7C45, 0x980726C5232C75E5, 0x50D3A7350792C35F, 0x172B595DB
                                                  }; // XRA1

/* Basis for Bob on A = 6, expressed in Montgomery representation */
static const uint64_t B_gen[6 * NWORDS64_FIELD] = {0xD4A2CF040BC56F2C, 0x58F1D1D2B190EDE7, 0x2229F10D3BC7BA47, 0x769AB0F0EDD86AA4, 0x97F1214B80D8463,
                                                   0x9B23774D13ED3EEE, 0x9A182E846DAA95C6, 0x343741369B273442, 0x61FB37462569D4BB, 0x1815EF8B9, // XPB0
                                                   0xF380CA27C26BF32E, 0xD594C3EA0698D298, 0x21D388E632D1CA2E, 0xDD1E0B34330E0AB0, 0xEA7B89CAD59CA8C2,
                                                   0x28C129BFC584BEC1, 0x48D1E802FC7418CF, 0x11F3A548C5DFFDF7, 0xDB0E9AF98D314F67, 0x219918D2B, // XPB1
                                                   0xD4A2CF040BC56F2C, 0x58F1D1D2B190EDE7, 0x2229F10D3BC7BA47, 0x769AB0F0EDD86AA4, 0x97F1214B80D8463,
                                                   0x9B23774D13ED3EEE, 0x9A182E846DAA95C6, 0x343741369B273442, 0x61FB37462569D4BB, 0x1815EF8B9, // XQB0
                                                   0xC7F35D83D940CD1, 0x2A6B3C15F9672D67, 0xDE2C7719CD2E35D1, 0x22E1F4CBCCF1F54F, 0x838676352A63573D,
                                                   0x88B72428E4D5F16C, 0x5215D742081BE0DC, 0xA0620DB14AE42733, 0xA68175C8C4B68925, 0x62651A3C, // XQB1
                                                   0x4F62205A5DAFB369, 0xA2B75D5BC06C691F, 0x6B82C9B893D51C38, 0x2C2467D7AB7DAA2C, 0x8A8D5AC13C2C5ADD,
                                                   0xBC3AEC544F8953F5, 0xBC43C1BE1B1DC069, 0xB8CDA0908AEBCD84, 0xA213356DB0FBFCFF, 0x15F063030, // XRB0
                                                   0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0
                                                  };                                           // XRB1

/* The x-coordinate of lB^(eB-1)*QB, expressed in Montgomery representation */
static const uint64_t XQB3[2 * NWORDS64_FIELD] = {0x4E8DA387C42D31D0, 0x83AFE0DEB4FCE84B, 0xE819FEA7B6AE32CB, 0xA3DA371FFE9EE0F2, 0x4765959DBFECDE2D,
                                                  0xDB446F20746721E0, 0xE2596FFE42E2D7A8, 0x1FC46DF0BC2B8759, 0x9F5106F3AD0A27F8, 0x161083808,
                                                  0xA88E809BEF28310C, 0x506DB48241A76019, 0xE9FB9219833DF071, 0x342D697582036AEE, 0x25F078F10F357AFB,
                                                  0x43CE96BEE1668B06, 0xC540E958A6494D2, 0x46EB300B952B437D, 0x898338D77E9F811D, 0x1FC9961CB
                                                 };

static const uint64_t A_basis_zero[8 * NWORDS64_FIELD] = {0x31C8AF80CA277BC7, 0x8A8AD55D2AC8A709, 0x95A4DC49B64E5B2C, 0xF08C77AAE90ABE83, 0x2DC44FF97C95845D,
                                                          0xE25D5D22CB0C4641, 0xEDA1E4D340EC56D7, 0x2AE81B9DCDAA0FF1, 0xB4292D8E409BC484, 0x1D5B8282A,
                                                          0x4B8C9CED6DEF0B8B, 0x652C800D926AB992, 0x3DFA6D6B8FD37D80, 0xA30C578CD98EFD79, 0x9FC067E58CCBD32E,
                                                          0x2B0599AEAF150FDB, 0xBA321B31886F3292, 0xE0011F56247547A1, 0x28CA0747910BFAE2, 0xFC020A14,
                                                          0x3CA84837D69D8728, 0xB2BDFE3304CB7401, 0x8C840937950AD3E9, 0xCE8094A539AA6C49, 0xF0802AAE490F29A0,
                                                          0x5458A8E61BB9D01F, 0x3592A73DE4758511, 0x7DEA75B85A60F316, 0xF835EEAC9B12CC1D, 0x11C4E0162,
                                                          0x87A90900552B058, 0xF34899FE9411DC6A, 0x3807CF5B95B0168, 0xC986BAF1E3FFDED4, 0x1D10EAC33AA0781A,
                                                          0xD9569230F9A2D512, 0xF8295F6189DBAAF3, 0x26B44D4CECB1A5E8, 0x9CA4CE754143DAA4, 0xAF517DFC,
                                                          0x2728178246F8418A, 0xD377C4656DBC71F0, 0x968642007B807932, 0xB8B04B1039062A21, 0xBE8A771B468A977C,
                                                          0xFCA53E1DBD78AFC, 0xF25044AB1D74FC52, 0x12C84BF8152D200E, 0xAFB53869D71DCFFF, 0x88D8EE1A,
                                                          0x4F828B752E825BB4, 0x82CEA210AC766C69, 0x8B1BBC87DAD8BEDD, 0x9BFC5B9CE215B423, 0xF7E1BCC0C541177C,
                                                          0x7727E3A0F1A1AF24, 0xFBCFE4177D2B0221, 0xBB15BDCC160D902A, 0x3FE1467B4A911446, 0x1A495CB35,
                                                          0x59DD6DAF69D05BED, 0xD76AC76FF586B73A, 0x7FE9D57BA40A785A, 0xD14A8A25B5B1A187, 0xA96A1F6746EC4DCB,
                                                          0x3D12E7FA86078B0A, 0x37816E342D89C86B, 0xEF93BD5D8E68FE5, 0x582A083E04E4D83D, 0x1A402EB75,
                                                          0x439B3817B316DC12, 0xAAC33FB9DB2CFE99, 0x103161162202C633, 0x3C38BFDA7559D6F5, 0xB4BC209633AB8D22,
                                                          0xA1724B6332EFE4FE, 0x81919EE777E6B9E1, 0xD15DFC6FAB242CF4, 0xD69C41AF15E2B7FC, 0x274D72579
                                                         };

/* Full 3-torsion for Bob on A = 0, expressed in Montgomery representation */
static const uint64_t B_gen_3_tors[16 * NWORDS64_FIELD] = {0x1930B476A230B7BE, 0x3A42EE16DC949E2, 0x838416CC861CB74D, 0xE43303F18BFF483E, 0x30C6C49AC1829C11,
                                                           0x40DD7E521AF62273, 0x33AA27D882B23CB2, 0xA970F369B7A93C6, 0xDBAC55CF2FEFF179, 0x1967E10A3,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x4709DA258D7CF239, 0x68142BD88204D1EA, 0xAE14B528AE909088, 0x8D813E124214BFD6, 0x8E6949403E0D509B,
                                                           0xF795CE2041AC8CA6, 0x20F1E1F2A851B359, 0x89B489710FA8BD76, 0x1608B910A609EED8, 0x1F957263F,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0xE6CF4B895DCF4841, 0xFC5BD11E9236B61D, 0x7C7BE93379E348B2, 0x1BCCFC0E7400B7C1, 0x3D3B3B653E7D63EE,
                                                           0x709ACF968F648DBB, 0x673D976C81DDBCF9, 0xA7BEA3C375499164, 0xA5E3BAF321F7E713, 0xE57896C4,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0x4709DA258D7CF239, 0x68142BD88204D1EA, 0xAE14B528AE909088, 0x8D813E124214BFD6, 0x8E6949403E0D509B,
                                                           0xF795CE2041AC8CA6, 0x20F1E1F2A851B359, 0x89B489710FA8BD76, 0x1608B910A609EED8, 0x1F957263F,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0xF4209DFFD5BA5848, 0x5B5A41268B033DBF, 0x7487D67F15941658, 0xE0C6754A260143AA, 0xDC40C6C153BD4F30,
                                                           0xA37675E3DD939202, 0x71830795095B5791, 0x204F7BB4612B47A, 0x196818C06A9FA46F, 0x1C82644E6,
                                                           0xC41ECDCAECF46668, 0x3303A35DC32C0272, 0x6A10EEBB3E7C798D, 0x800B26C6B6F0F2B7, 0xE0AA44F811ECA6C6,
                                                           0x38BB298B83C41423, 0x5C8E988E2C1B6909, 0xD61B6FE6260A9672, 0xE249E98EF30D4401, 0xA91ADF39,
                                                           0x3BE13235130B9997, 0xCCFC5CA23CD3FD8D, 0x95EF1144C1838672, 0x7FF4D939490F0D48, 0x8D57BB07EE135939,
                                                           0x78BD245D26969C0A, 0x3E5926B6D87490A2, 0xDC3A4313EAB98EB8, 0x9F4627335EDA948A, 0x1D2DBC82E,
                                                           0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0,
                                                           0xBDF62002A45A7B7, 0xA4A5BED974FCC240, 0x8B782980EA6BE9A7, 0x1F398AB5D9FEBC55, 0x91C1393EAC42B0CF,
                                                           0xE01D804CCC71E2B, 0x2964B7AFFB34A21A, 0xB050BB3ECAB170B0, 0x6827F801E748341D, 0xB3D06282,
                                                           0xC41ECDCAECF46668, 0x3303A35DC32C0272, 0x6A10EEBB3E7C798D, 0x800B26C6B6F0F2B7, 0xE0AA44F811ECA6C6,
                                                           0x38BB298B83C41423, 0x5C8E988E2C1B6909, 0xD61B6FE6260A9672, 0xE249E98EF30D4401, 0xA91ADF39,
                                                           0xC41ECDCAECF46668, 0x3303A35DC32C0272, 0x6A10EEBB3E7C798D, 0x800B26C6B6F0F2B7, 0xE0AA44F811ECA6C6,
                                                           0x38BB298B83C41423, 0x5C8E988E2C1B6909, 0xD61B6FE6260A9672, 0xE249E98EF30D4401, 0xA91ADF39
                                                          };

/* Pre-computed pairing ReducedTatePairing(R0,S3,lB^eB) on A = 0 */
static const uint64_t g_R_S_im[NWORDS64_FIELD] = {0x58B92E3BB6DC9E31, 0x3E280F90A5818BDA, 0x8BF300AC24A8E69A, 0x2E12E47000B08F86, 0xB01B3531200990E9,
                                                  0xF63C539B47B5A4FE, 0x9F70EF0C3FFAB9BA, 0xCFBD59EBE2438F9B, 0x1E7E8AD9F5361552, 0xC2DDA200
                                                 };

// Montgomery constant Montgomery_R2 = (2^640)^2 mod p610
static const uint64_t Montgomery_R2[NWORDS64_FIELD] = {0xE75F5D201A197727, 0xE0B85963B627392E, 0x6BC1707818DE493D, 0xDC7F419940D1A0C5, 0x7358030979EDE54A,
                                                       0x84F4BEBDEED75A5C, 0x7ECCA66E13427B47, 0xC5BB4E65280080B3, 0x7019950F516DA19A, 0x000000008E290FF3
                                                      };

// constant Montgomery_RB1 = (2^NBITS_ORDER)^2 mod Bob_order
static const uint64_t Montgomery_RB1[NWORDS64_FIELD] = {0x4FDA7CB300F00BFB, 0x8559FEAD08AF7CE1, 0x6C728679071D8D77, 0xFBFABAAF1825440F, 0x8D0536062AB9};

// constant Montgomery_RB2 = -(3^OBOB_EXP)^-1 mod 2^NBITS_ORDER
static const uint64_t Montgomery_RB2[NWORDS64_FIELD] = {0x5BAEA880514636FF, 0x32B4BD3B3F4460CB, 0x4802A8E49DA81538, 0x9F99CD971298B1A8, 0xE333BBA90FAF20FE};

// Value one in Montgomery representation
static const uint64_t Montgomery_one[NWORDS64_FIELD] = {0x00000000670CC8E6, 0x0000000000000000, 0x0000000000000000, 0x0000000000000000, 0x9A34000000000000,
                                                        0x4D99C2BD28717A3F, 0x0A4A1839A323D41C, 0xD2B62215D06AD1E2, 0x1369026E862CAF3D, 0x000000010894E964
                                                       };

/* OQS note: unused
// 1/3 mod p
static const uint64_t threeinv[NWORDS64_FIELD] = {
	0x5555555577AEEDA2, 0x5555555555555555, 0x5555555555555555, 0x5555555555555555, 0xDE11555555555555, 0xC488963F0D7B28BF, 0xAE18B2BDE10BF15E, 0x463CB6074578F0A0, 0x5BCDAB7A2CB98FBF, 0x5831A321
};
*/

// Fixed parameters for isogeny tree computation
static const unsigned int strat_Alice[MAX_Alice - 1] = {
  67, 37, 21, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1, 16, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 33, 16, 8, 5, 2, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 16, 8, 4, 2, 1, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1
};

static const unsigned int strat_Bob[MAX_Bob - 1] = {
  86, 48, 27, 15, 8, 4, 2, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 21, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1, 38, 21, 12, 7, 4, 2, 1, 1, 2, 1, 1, 3, 2, 1, 1, 1, 1, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1, 17, 9, 5, 3, 2, 1, 1, 1, 1, 2, 1, 1, 1, 4, 2, 1, 1, 1, 2, 1, 1, 8, 4, 2, 1, 1, 1, 2, 1, 1, 4, 2, 1, 1, 2, 1, 1
};

// Fixed traversal strategies for Pohlig-Hellman discrete logs

static const unsigned int ph2_path[PLEN_2] = { // w_2 = 5
#ifdef COMPRESSED_TABLES
  #ifdef ELL2_TORUS
    #if (W_2 == 5)
      0, 0, 1, 2, 3, 4, 4, 5, 5, 6, 7, 7, 8, 9, 10, 10, 11, 12, 13, 13, 14, 14, 15, 16, 17, 18, 18, 18, 19, 20, 20, 21, 22, 23, 24, 25, 25, 25, 25, 26, 27, 28, 29, 29, 30, 31, 32, 33, 34, 35, 35, 35, 35, 35, 36, 37, 38, 39, 40, 41, 41, 42
    #endif
  #endif
#endif
};

static const unsigned int ph3_path[PLEN_3] = {
#ifdef COMPRESSED_TABLES
  #ifdef ELL3_FULL_SIGNED
    #if (W_3 == 3)
      0, 0, 1, 2, 3, 3, 4, 4, 5, 6, 6, 7, 8, 9, 9, 9, 10, 11, 12, 13, 13, 13, 13, 14, 15, 16, 17, 18, 19, 19, 19, 19, 19, 20, 21, 22, 23, 24, 25, 26, 27, 28, 28, 28, 28, 28, 28, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 41, 41, 41, 41
    #endif
  #endif
#endif
};

// Entangled bases related static tables and parameters

// Constants u and u0 where u = u0^2 in F_{p^2} \ F_p used in entagled basis generation
// For the 2^eA-torsion basis generation:
//      Two tables of 17 elements each for the values r in F_p such that v = 1/(1+ur^2) where r is already converted to Montgomery representation
//      Also, 2 tables for the quadratic residues (qr) and quadratric non residues (qnr) v in F_{p^2} with 17 GF(p^2) elements each.
// For the 3^eB-torsion basis generation:
//      A table of size 20 for values v = 1/(1+U*r^2) where U = 4+i

static const uint64_t u_entang[2 * NWORDS64_FIELD] = {
	0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0x0, 0xCE1991CC, 0x0, 0x0, 0x0, 0x3468000000000000, 0x9B33857A50E2F47F, 0x149430734647A838, 0xA56C442BA0D5A3C4, 0x26D204DD0C595E7B, 0x21129D2C8
};

/* OQS note: unused
static const uint64_t u0_entang[2 * NWORDS64_FIELD] = {
	0x670CC8E6, 0x0, 0x0, 0x0, 0x9A34000000000000, 0x4D99C2BD28717A3F, 0xA4A1839A323D41C, 0xD2B62215D06AD1E2, 0x1369026E862CAF3D, 0x10894E964, 0x670CC8E6, 0x0, 0x0, 0x0, 0x9A34000000000000,
	0x4D99C2BD28717A3F, 0xA4A1839A323D41C, 0xD2B62215D06AD1E2, 0x1369026E862CAF3D, 0x10894E964
};
*/

// Elligator constant U = min{u0+k} for k=1,2... such that u0+k is a square in F_p^2 for generating 3^n torsion bases
static const uint64_t U3[2*NWORDS64_FIELD] = {0x2033FEC80,0x0,0x0,0x0,0x2700000000000000,0x211031E0758202E1,0xFDA2FA9626933136,0xB8E34478F08DCF14,0x5DECEAA3FB0FBB1B,0x32FB4023,0x670CC8E6,0x0,0x0,0x0,0x9A34000000000000,0x4D99C2BD28717A3F,0xA4A1839A323D41C,0xD2B62215D06AD1E2,0x1369026E862CAF3D,0x10894E964};

// Tables for quadratic residues and quadratic non residues v with 17 elements each.

static const uint64_t table_r_qr[TABLE_R_LEN][NWORDS64_FIELD] = {
	{0x670CC8E6, 0x0, 0x0, 0x0, 0x9A34000000000000, 0x4D99C2BD28717A3F, 0xA4A1839A323D41C, 0xD2B62215D06AD1E2, 0x1369026E862CAF3D, 0x10894E964},
	{0x19C332399, 0x0, 0x0, 0x0, 0xFACE000000000000, 0x84EEBD0BF76B38CF, 0x8E40A1A187FF56C5, 0x9882D55D30E7225D, 0xCC13F8F7C6CAE46A, 0x1A65CFE27},
	{0x338664733, 0x0, 0x0, 0x0, 0x879A000000000000, 0x58652C2F447BC171, 0x819983FE0B6EB3DF, 0x7EAFF7C0510A1F90, 0x1697E12D3BADF048, 0xD0C354E7},
	{0x39F731019, 0x0, 0x0, 0x0, 0x21CE000000000000, 0xA5FEEEEC6CED3BB1, 0x8BE39C37AE9287FB, 0x516619D62174F172, 0x2A00E39BC1DA9F86, 0x1D9583E4B},
	{0x4067FD900, 0x0, 0x0, 0x0, 0x4E00000000000000, 0x422063C0EB0405C2, 0xFB45F52C4D26626C, 0x71C688F1E11B9E29, 0xBBD9D547F61F7637, 0x65F68046},
	{0x46D8CA1E6, 0x0, 0x0, 0x0, 0xE834000000000000, 0x8FBA267E13758001, 0x5900D65F04A3688, 0x447CAB07B186700C, 0xCF42D7B67C4C2575, 0x16E8B69AA},
	{0x53BA633B3, 0x0, 0x0, 0x0, 0xAE9A000000000000, 0x79755E0FB9FDC452, 0x7F3C7E943201E515, 0x37933C394197EEA5, 0x7484CBD136BDAB64, 0x103BE950A},
	{0x5A2B2FC99, 0x0, 0x0, 0x0, 0x48CE000000000000, 0xC70F20CCE26F3E92, 0x898696CDD525B931, 0xA495E4F1202C087, 0x87EDCE3FBCEA5AA2, 0x20C537E6E},
	{0x609BFC580, 0x0, 0x0, 0x0, 0x7500000000000000, 0x633095A1608608A3, 0xF8E8EFC273B993A2, 0x2AA9CD6AD1A96D3E, 0x19C6BFEBF12F3153, 0x98F1C06A},
	{0x7A5F2E919, 0x0, 0x0, 0x0, 0x6FCE000000000000, 0xE81F52AD57F14173, 0x87299163FBB8EA67, 0xC32CA2C802908F9C, 0xE5DAB8E3B7FA15BD, 0x23F4EBE91},
	{0x9A932D599, 0x0, 0x0, 0x0, 0x96CE000000000000, 0x92F848DCD734454, 0x84CC8BFA224C1B9E, 0x7C0FE740F31E5EB1, 0x43C7A387B309D0D9, 0x27249FEB5},
	{0xA103F9E80, 0x0, 0x0, 0x0, 0xC300000000000000, 0xA550F9624B8A0E65, 0xF42EE4EEC0DFF60E, 0x9C70565CB2C50B68, 0xD5A09533E74EA78A, 0xFEE840B0},
	{0xA774C6766, 0x0, 0x0, 0x0, 0x5D34000000000000, 0xF2EABC1F73FB88A5, 0xFE78FD286403CA2A, 0x6F267872832FDD4A, 0xE90997A26D7B56C8, 0x2077D2A14},
	{0xADE59304D, 0x0, 0x0, 0x0, 0x8966000000000000, 0x8F0C30F3F21252B6, 0x6DDB561D0297A49B, 0x8F86E78E42D68A02, 0x7AE2894EA1C02D79, 0x941B6C10},
	{0xB4565F933, 0x0, 0x0, 0x0, 0x239A000000000000, 0xDCA5F3B11A83CCF6, 0x78256E56A5BB78B7, 0x623D09A413415BE4, 0x8E4B8BBD27ECDCB7, 0x19CB05574},
	{0xBAC72C21A, 0x0, 0x0, 0x0, 0x4FCC000000000000, 0x78C76885989A9707, 0xE787C74B444F5328, 0x829D78BFD2E8089B, 0x20247D695C31B368, 0x294E9770},
	{0xC137F8B00, 0x0, 0x0, 0x0, 0xEA00000000000000, 0xC6612B42C10C1146, 0xF1D1DF84E7732744, 0x55539AD5A352DA7D, 0x338D7FD7E25E62A6, 0x131E380D4}
};

static const uint64_t table_r_qnr[TABLE_R_LEN][NWORDS64_FIELD] = {
	{0xCE1991CC, 0x0, 0x0, 0x0, 0x3468000000000000, 0x9B33857A50E2F47F, 0x149430734647A838, 0xA56C442BA0D5A3C4, 0x26D204DD0C595E7B, 0x21129D2C8},
	{0x135265AB3, 0x0, 0x0, 0x0, 0x609A000000000000, 0x3754FA4ECEF9BE90, 0x83F68967E4DB82A9, 0xC5CCB347607C507B, 0xB8AAF689409E352C, 0x9DC814C3},
	{0x2033FEC80, 0x0, 0x0, 0x0, 0x2700000000000000, 0x211031E0758202E1, 0xFDA2FA9626933136, 0xB8E34478F08DCF14, 0x5DECEAA3FB0FBB1B, 0x32FB4023},
	{0x26A4CB566, 0x0, 0x0, 0x0, 0xC134000000000000, 0x6EA9F49D9DF37D20, 0x7ED12CFC9B70552, 0x8B99668EC0F8A0F7, 0x7155ED12813C6A59, 0x13B902987},
	{0x2D1597E4C, 0x0, 0x0, 0x0, 0x5B68000000000000, 0xBC43B75AC664F760, 0x12372B096CDAD96E, 0x5E4F88A4916372D9, 0x84BEEF8107691997, 0x2442512EB},
	{0x4D4996ACC, 0x0, 0x0, 0x0, 0x8268000000000000, 0xDD53E93B3BE6FA41, 0xFDA259F936E0AA4, 0x1732CD1D81F141EE, 0xE2ABDA250278D4B3, 0x27720530E},
	{0x670CC8E66, 0x0, 0x0, 0x0, 0xF34000000000000, 0xB0CA585E88F782E3, 0x33307FC16DD67BE, 0xFD5FEF80A2143F21, 0x2D2FC25A775BE090, 0x1A186A9CE},
	{0x6D7D9574D, 0x0, 0x0, 0x0, 0x3B66000000000000, 0x4CEBCD33070E4CF4, 0x729560F0B571422F, 0x1DC05E9C61BAEBD8, 0xBF08B406ABA0B742, 0x2E24EBC9},
	{0x73EE62033, 0x0, 0x0, 0x0, 0xD59A000000000000, 0x9A858FF02F7FC733, 0x7CDF792A5895164B, 0xF07680B23225BDBA, 0xD271B67531CD667F, 0x136B9D52D},
	{0x80CFFB200, 0x0, 0x0, 0x0, 0x9C00000000000000, 0x8440C781D6080B84, 0xF68BEA589A4CC4D8, 0xE38D11E3C2373C53, 0x77B3AA8FEC3EEC6E, 0xCBED008D},
	{0x8740C7AE6, 0x0, 0x0, 0x0, 0x3634000000000000, 0xD1DA8A3EFE7985C4, 0xD602923D7098F4, 0xB64333F992A20E36, 0x8B1CACFE726B9BAC, 0x1D481E9F1},
	{0x8DB1943CD, 0x0, 0x0, 0x0, 0x6266000000000000, 0x6DFBFF137C904FD5, 0x70385B86DC047365, 0xD6A3A3155248BAED, 0x1CF59EAAA6B0725D, 0x61202BED},
	{0x942260CB3, 0x0, 0x0, 0x0, 0xFC9A000000000000, 0xBB95C1D0A501CA14, 0x7A8273C07F284781, 0xA959C52B22B38CCF, 0x305EA1192CDD219B, 0x169B51551},
	{0xCE1991CCD, 0x0, 0x0, 0x0, 0xB066000000000000, 0xB01C62D467945597, 0x6B7E50B3292AD5D1, 0x486A2C0733645917, 0xD8CF73F29CCFE895, 0xC716AC33},
	{0xD48A5E5B3, 0x0, 0x0, 0x0, 0x4A9A000000000000, 0xFDB625919005CFD7, 0x75C868ECCC4EA9ED, 0x1B204E1D03CF2AF9, 0xEC38766122FC97D3, 0x1CFAB9597},
	{0xDAFB2AE9A, 0x0, 0x0, 0x0, 0x76CC000000000000, 0x99D79A660E1C99E8, 0xE52AC1E16AE2845E, 0x3B80BD38C375D7B0, 0x7E11680D57416E84, 0x5C49D793},
	{0xE7DCC4066, 0x0, 0x0, 0x0, 0xAB34000000000000, 0x350B1FE05EFF8E67, 0xF9BEF254B12A2C97, 0xE0ED0164644B7B74, 0xA4E36CEA639ACCFF, 0x26D73AA5B}
};

static const uint64_t table_v_qr[TABLE_V_LEN][NWORDS64_FIELD] = {
	{0xCCCCCCCCE168F4FA, 0xCCCCCCCCCCCCCCCC, 0xCCCCCCCCCCCCCCCC, 0xCCCCCCCCCCCCCCCC, 0x76D8CCCCCCCCCCCC, 0x3E565465D2C0BCB, 0xB12E9DDC577A588F, 0x1F35C932D0B247B5, 0x521DAB18F8F36B0, 0x231AFE7CE},
	{0x666666663D2E160A, 0x6666666666666666, 0x6666666666666666, 0x6666666666666666, 0xEE52666666666666, 0x5B25D1449A5D48C5, 0xD37242D15A2B4239, 0x263FD38E8023BAE9, 0xF8DC6C2184B143B9, 0x948D7F34},
	{0x2B7522B7524531E5, 0x22B7522B7522B752, 0x522B7522B7522B75, 0x7522B7522B7522B7, 0xED882B7522B7522B, 0xF1639645024AE01, 0x4F035B12DE5E50A0, 0x73C4980ACDD7F1A1, 0xA42840C6580DE412, 0x1DD62C7C0},
	{0x915BA915B759C347, 0xA915BA915BA915BA, 0xBA915BA915BA915B, 0x5BA915BA915BA915, 0xD2C915BA915BA91, 0x71F86F2E9E45744B, 0x40364B61A6454DC2, 0xF1CB7911E8296CFF, 0x22098A2EFDE7A36E, 0x26EBD6120},
	{0xAD9149BAD91637DA, 0x9BAD9149BAD9149B, 0x149BAD9149BAD914, 0xD9149BAD9149BAD9, 0x9F25149BAD9149BA, 0xD62B18FB0FE869F1, 0xC55BADFBDB1E3966, 0xD3983013C35F72D3, 0xCDCBA5EC343A38C7, 0xA9663F71},
	{0x375B229374E412DD, 0x29375B229375B229, 0xB229375B229375B2, 0x75B229375B229375, 0x77BBB229375B2293, 0x2DE6294756311D89, 0x7FD8288210926F0E, 0x959F6E4E9B17AAF0, 0xD0DF547715963752, 0x23F992A62},
	{0x95AB46517CB6558, 0xF00AC4C8322D018D, 0xC4BBB5E824213413, 0xFE54FF999E981B8B, 0x457485163ADB696E, 0xB3837BF2C2444F67, 0xAE6D4CC9D80D6369, 0x16560F053370AC84, 0x23263644FE26E906, 0x70C8ECA4},
	{0x1499D806F149DE33, 0x192F79503F8504C0, 0x8136E31922FD0B62, 0xE3640C9A3BE918D, 0x827DC7EEC12747C3, 0x816C63BC5D0BB5E8, 0x2B161316CBD65F8F, 0x113FC50958EB09DD, 0x6F238E5A72A41216, 0xABCB36F8},
	{0x84665F30F3CF95E5, 0xC3B84BD34725A330, 0x3A6AF0FAC575EE0A, 0x9C71AFCF5C78A658, 0xB84BA4EE269DDBC3, 0x8E25FEED6714479A, 0x6FA95E8F73EA96FC, 0x4E8F67FDBB4C1B14, 0x9F723476F170C736, 0xBA3A5C22},
	{0x9005A1C185D2E4DD, 0x1804C2F06A988218, 0x5C73BC15BBDE0797, 0xC72EA5FFC1BE0B12, 0x5F5D25F1D4AC4F2D, 0xD90ACB26BB0EA9D3, 0x771734D57DE89542, 0xB9B7016789BF68DA, 0x4AFADFDC405345FF, 0x114409A09},
	{0x69E362D42702ACE6, 0x9FC36EB7EA12F1E0, 0x9CCA7DF3BC005DFF, 0xD8FB897860F89087, 0x684E603A54569145, 0x59A0620287865C95, 0xA1EE61177C586BBF, 0xDDBC6767DA8F6EE1, 0x9C092A6A8396897F, 0x27991721C},
	{0xE70C93731F788DA2, 0xF9415624BA1759DB, 0xC894EF9847A7245A, 0xE2380C3455075FCD, 0x63CD08DC462AABF8, 0xA1CFF8CECB706A7, 0x5BBB070AC484296B, 0xF8E970373DD450FB, 0xF985BF0108DCBA20, 0x243AC61BE},
	{0x886DC36536F16AC0, 0xCE46420B9809323C, 0x8F0CE1E1A569DA62, 0x574F7A978263CAAD, 0x93BE116009D2DB5C, 0xE47CE9FDE2BCB17A, 0xAEFCD1BFDE2FB352, 0x6596D37050A6FE96, 0xA936C53A6B044AC2, 0xBAE5C165},
	{0xDF14045D75410E1C, 0xA73CCCB143DBAC13, 0x20FDC4139A3DA98B, 0xB91023F5D83E66D9, 0xE7D50F33079A6014, 0xA6117DAD2E4A7E4B, 0x78B9C7A26D42C39F, 0x8856BF64102261B1, 0x31F624C6B2E5E24B, 0x1A9000C99},
	{0x4D1B7C9879B8F55E, 0x2A149D570FBDDE34, 0xB2B6E11841366015, 0xBCF5C4F2C545EF76, 0x939396F37CA988F0, 0x2F66041FC5F19298, 0x60D9027E6FDDE5C8, 0x594434D0D278B451, 0xDABB15ADAEA1D3E7, 0x24D1C996F},
	{0xEDE936859CC846A4, 0x906F12AFE543BFE9, 0x57F752DC24BCDF97, 0xA7AA6C41ECE9523E, 0x70D8DB291C664F5E, 0xC2D8762B1F0A0862, 0xF538248730EF2597, 0xE1699BCD8E10BBF7, 0x4A5EA2590DDB6DCC, 0x22EEB1551},
	{0xF2A95939B0FDB788, 0x220AC9E23E53C0A, 0x37C7FB2D043F1549, 0x17A39CE70F0B8E54, 0xF8D84DD2D87A4331, 0xCC560AC3C2D4ECB1, 0x29AFE290E4B8470D, 0x5FD3854406911AD1, 0x16284324DEA12E26, 0x265A1E3E1},
	{0x72512896E203613D, 0x42909204E70C74C1, 0xF2787ADE891C95AA, 0x726631D78DAFCFF5, 0x812D335F7915E3B4, 0x602D763CFDD14E0D, 0xF09DBF94B4B8587D, 0x95CEDD56F23A03B9, 0x35067361D5A1D79E, 0x1FD8BE4EB},
	{0x401CA1FC04E9C123, 0x90C26307F5CC0A0F, 0x485D5BE49E59B0C3, 0x4FE729949842C2E, 0xDFE318C2AE76911B, 0xD12311AD4DD78EA1, 0xE40CA0745241900F, 0x3D124917B409965B, 0xBFD7D5AFE6476F0, 0x6D74FB8C},
	{0x2F3F273A24BD4ACD, 0xBBC4B38CC68BA0FD, 0xE8B2D539670B7871, 0xEA60CBAEA93B6577, 0x487E2AEFF59AC1CB, 0xD2D62FD5A457C222, 0x778BDCA33D649C39, 0xD6463F406CBC16A4, 0x726CA44086DD4B26, 0x1D3824D0D},
	{0xBDB1260A3702AD09, 0x2D6B1A339991FEBC, 0xAA9F6C83305EA847, 0xA046C8AD7DFDC0E3, 0xD44F15831B61D266, 0xE5069559BBEFA954, 0xD78E56E41C84CC4E, 0x760438E0D334C5E, 0xAD70552405CB2442, 0x26FADBB00},
	{0x62D4D20873F55316, 0x9E0A17CCEF05AEAA, 0x3297B1A6560ABFB3, 0xC178F3490A1BFF80, 0x30F33204C7CD322E, 0xBF67D1323DD3AF36, 0xC26B4425CDECACD6, 0xFD1804929AF46345, 0x6C6AD7A3773EB3E9, 0xA0F5703F},
	{0x7B5586A15EEE987A, 0x712704F03450E84C, 0x6FDC36F98F02AC0C, 0xA1B5EBF84C82B74F, 0x70B06DBC6C6FBB4, 0xF49DA4BBBBB4A081, 0xD539B581C8C78E70, 0xC505DC913CDC308A, 0x79302B7361615850, 0x1FDAD6EA8},
	{0xC864A00E76FB7862, 0x7F79E3208CF1BA8D, 0xCEBB9173B4F3EB3F, 0x65B5CD9A69BCEE1F, 0x2FFC82DF6866F802, 0x372294003EFDDFB6, 0x2B4A606D7D21B67A, 0xFCCF899A749472DD, 0x60AD6F3318245C4F, 0x8A94EC6B},
	{0x3B540A505CFDE08A, 0x1F4FA2605E47B5FD, 0x545260EAD2FAFEEF, 0xEACCE2AB9D6A73E1, 0x4DE929F08605608A, 0x7FD4E8C4330111E9, 0xACA7C41ABA9F32CB, 0x81CE18FFB987259B, 0xCF27FAA2E00078FC, 0x13C2D3CF2},
	{0xAC298794E336248F, 0xA36E730E1546DE9E, 0xACF027D5C26DA122, 0xF5F2E5A8A5CC01FA, 0xBBDC81BC339A2257, 0x70D7847A7AC291EB, 0x653A274765BB19A2, 0x4AD77CDA41E716D9, 0x909862AA50015453, 0x214843766},
	{0x28DCFDDEE7E75520, 0x926112237D3B1FF5, 0x1C7EF82595F8671, 0x1506781F7F3CBDFF, 0x6085630169990327, 0xE99E1AB62629523A, 0xF5124A3DA6D4B381, 0x6238E3E6BA5CD147, 0xFF73FD00F4AE2505, 0x267B4E1A7},
	{0x4562207B3C7D2A3C, 0x5326B1E0C543FDBD, 0xDB4DEB9EFDF44D2C, 0x4127DC9D580DE9A7, 0xB5A21F4968C0B6A, 0x551995E644DD6F9D, 0xA410258D02B6BD1A, 0x38AD3333153D132C, 0xBC05805A3AF78F5E, 0x118462D9D},
	{0x6BCC76AD0919255E, 0x1CE349F8E0113FDD, 0x9FDDB44D41754041, 0x17B14F71175D7A86, 0xFF025B1D9403E728, 0xB066EF99669BAC1, 0x823A70C11307F544, 0xE302A74F2C217B92, 0x44959A86BE1185DF, 0x1AAAE7CC1},
	{0xBBA91C2845FB1C23, 0xFDAEBA39658D3CB, 0xD20FA6CF11D6712F, 0xE1F96B50E371876C, 0x8FCBEAD568182A6E, 0x7470BD4B6F6ECAC0, 0x854B204F3377B675, 0x1A4069A8A9B7FEE1, 0xD94275FEBB51BF6B, 0x278D27918},
	{0x6426BB8949C81154, 0xF8E70C86ACAADADD, 0x254018F73FF1D13, 0x299DE9B99FB375F3, 0xFAD85F5B9455BC3D, 0x75B38230DECC8C86, 0x84D8352EC4455A3E, 0xC63E8F4D14896C61, 0xABDCAF24FBE2620C, 0x18DD5BB6F},
	{0xF983D3F93B7E21FB, 0xA1EFB325856DFD63, 0xB40DBF77DDD2F2C6, 0x90765A64B6E3095A, 0x5C97784B64B13624, 0xA66B10ECDB13601E, 0x56B0519E348B08D7, 0x377A62607E2C784, 0xBD9E2E4BF13D82CB, 0x1FF451525},
	{0x10BEA3C10F74187C, 0xD1A28B48103C996B, 0x71DE9B27B2DEE637, 0xCCD9CF6314DF24F6, 0xA3509CFF5B01D17C, 0xBD870219DBDE553D, 0xC24847905D600F2, 0x3F6029BE326908F1, 0x956902BB0608AF31, 0x19FBAFFCD},
	{0x43909A8B57B3D387, 0x11CAD4DD5E94732, 0x5ACD10DE50BD4786, 0xA485CF553F041B2F, 0x78621C881B370A7F, 0x54D76CE547ACAC13, 0xD2BAF160F13C3B9E, 0x4FDE5E64A3480204, 0x2501DA6825E2CC9A, 0xD63111F9}
};

static const uint64_t table_v_qnr[TABLE_V_LEN][NWORDS64_FIELD] = {
	{0xAD4AD4AD4C6A88FF, 0x4AD4AD4AD4AD4AD4, 0xD4AD4AD4AD4AD4AD, 0xAD4AD4AD4AD4AD4A, 0xA6D6AD4AD4AD4AD4, 0x9AEFF6AA14208728, 0x25F686D3A6B965DD, 0x89CFD24F7B8B05D6, 0x3277900023742A0A, 0x181A60983},
	{0x95A95A959CABB803, 0xA95A95A95A95A95A, 0x5A95A95A95A95A95, 0x95A95A95A95A95A9, 0xEF5495A95A95A95A, 0x9FD9D03AB2C137A2, 0xD6D285BBE104B16D, 0x2D2DEC66777C8B23, 0xF413D3CA7DE5EA6B, 0x5EA0F8F0},
	{0x890EF755DC154E99, 0xA890EF755DBC422A, 0x2A890EF755DBC422, 0x22A890EF755DBC42, 0xE6F8890EF755DBC4, 0x992CEA18AE912E4D, 0x78C13445EFF8450A, 0xFC7F0B40B903DCD5, 0x3DEB2380F4A72BD3, 0x14B83DE77},
	{0x5CF29BF686807934, 0x25CF29BF68C35902, 0x25CF29BF68C3590, 0x9025CF29BF68C359, 0xE9A5CF29BF68C35, 0x298A955A6155A056, 0x8F77CBC74E2AE5F6, 0x366A3337A563ECA7, 0xB5182885FF4D6097, 0x1885CE5B2},
	{0x1060A07FBF2642F, 0xE01060A07FBE7D7E, 0x7E01060A07FBE7D7, 0xD7E01060A07FBE7D, 0x832001060A07FBE7, 0xED7D83B8855A105B, 0x61124019E05FD124, 0x17593771A74013B3, 0x65ED650971772921, 0x27986C521},
	{0xCCD20A70CAA86EA0, 0x3CCD20A70CCB7D63, 0x63CCD20A70CCB7D6, 0xD63CCD20A70CCB7D, 0xE023CCD20A70CCB7, 0x46FB7D673A1F3731, 0x4BB2D66D1167EA52, 0x455020A49BCB6949, 0x65C58A1BD6024304, 0x79DA31E3},
	{0xC8A714319269CBF, 0x3EE6C5324FB03682, 0x820C8A7143192186, 0x863EE6C5324FB036, 0xEB040C8A71431921, 0x136A320C0C040BB6, 0x168E5DED014E95FF, 0x823E3A98411D42EC, 0xBF3B909BE84CA071, 0x20313C368},
	{0x79102520ED23EA0D, 0x4F1889D99670AB6C, 0x6C79102520EE923E, 0x3E4F1889D99670AB, 0x415279102520EE92, 0x70DBE13BE1C34F41, 0x5B5EAA3FAF145AD4, 0x783FC4CF8CF9BE6B, 0x137330EF8AE1C880, 0x1B4499FA0},
	{0x4634145CDBBC7D65, 0xBE4634145CDBB9BE, 0xB9BE4634145CDBB9, 0xDBB9BE4634145CDB, 0xFC11B9BE4634145C, 0x2FE8FEBF758B8693, 0x62828E9265ECF9A0, 0x1C4ED813B3BC7207, 0x846F272559B2E04B, 0xB0BF2182},
	{0x20103473E1D7FF3A, 0x2920103473E2E529, 0xE52920103473E2E5, 0xE2E52920103473E2, 0x896EE52920103473, 0x11F70027A281C06D, 0x3B5E57817B07BF75, 0xAB30DBCF075069D5, 0x7934D8F49EE1D2A2, 0x1E5CF7B77},
	{0x9A1999347725E051, 0x99B2E2369C46A0F7, 0x477258EE57C21979, 0x69C46A0F79A19993, 0x56DA2197999B2E23, 0xCE319B8B5D5BF252, 0x5FFDE836E4511966, 0x3C4EE6DEE16E12B3, 0xF9D395517385C677, 0xC56DA3A3},
	{0xA333A4F9F563A486, 0x16C18290308AE972, 0x9F5BF3DD45A35733, 0x308AE972A333A4F, 0xF74E3573316C1829, 0x6C7C6702DC7153B0, 0x77D40082BF6132B4, 0xD95D312C4920072E, 0x7E9DE8AED502DB52, 0x1795EC244},
	{0xBFAD90149BFAF2C4, 0xAD90149BFAD90149, 0x90149BFAD90149BF, 0x149BFAD90149BFAD, 0x7E72D90149BFAD90, 0x6866E05B5982496, 0xC0A9956428181AA3, 0x692DDBE0498CF749, 0x8D9D8E6C9C0C1C9F, 0x123FACCAD},
	{0xA4DFD6C80A1A7714, 0xDFD6C80A4DFD6C80, 0xD6C80A4DFD6C80A4, 0xC80A4DFD6C80A4DF, 0x8425FD6C80A4DFD6, 0x8E0BC711DB513DD0, 0x7A798B500484E43A, 0xB493DF45AE3AFC5, 0x35B299EB698262E2, 0x251C8F93C},
	{0xF106BA2361C0BBFC, 0x267751BCCB188710, 0xCF04E54D6488BC95, 0xE9C02260C68B4DBC, 0xA0F2D5B56BFB777B, 0xC62C2D54EEB3DD6D, 0x8C3B7A47FF6C7E8D, 0xE7AD379C550443FF, 0xE820CAF499D9CD06, 0xBB9BA44B},
	{0xCECFBC1D4AD7905D, 0x269973BD729F0BBF, 0x96F247430346373F, 0x3C32617FB97A7BB4, 0x170F7C62323C3A3A, 0x1B9DB0A6D879BD7E, 0xDA82AC8B59360996, 0xA26FE113F634C7A, 0x7130F589581CBC64, 0x13858DF6C},
	{0x7E5D2D52571E03D0, 0xF75D1072B445F79B, 0x706331C41D88D1D5, 0x265B33A4D24975E8, 0xC97488655D632ABE, 0x6D82AC11CD7F6048, 0xB030079E0A9EBAD4, 0xA7E87B1B62EAFB16, 0x9522F6F141D35D44, 0x82B24B60},
	{0x242547937C0658FA, 0xDC6E5DA7AEE53E68, 0x84EA07953DACDA65, 0xE92546CBB60D8BA3, 0xA612BF6B9CFBCEAE, 0xB2353AB9010C800F, 0x1BBAD6158193C629, 0x5464119263DFE2DF, 0x50E7B308405743DC, 0x20FCCD43A},
	{0x6D72564707F57DA7, 0x44E62AEA7A94D72F, 0xC60D55CE4D3ABF28, 0xC819D3C0BC081DE5, 0x161317D2034B91E3, 0x9A84FE10A373C3D5, 0xD73276492F6D4036, 0x5332ED7A2830AE9E, 0xCC28BE76FB2DF00A, 0x120CA3CCB},
	{0xFAB2620720D754B4, 0xB0B9E340EEDF8BC9, 0x1653DB4EA86AA228, 0xAF4A45B466A291F5, 0x6F2D8FB5B3D8182E, 0x7774D2D36F2E881F, 0xC3D2516E474E557D, 0x92B45DD23EC32833, 0x3984A07B8619C7C9, 0x1CAB80BBA},
	{0xE7CC90E6EE298EA, 0x99D3392BBC06C468, 0xE8F51B6B2D8C36DD, 0x7477C602BFC793C0, 0x4CC9A5B9824BBB8D, 0x66B8B5F5427D3453, 0xCC9FA954AA043A0, 0xE68FEBFA58A0024E, 0x1C4E709FC346F24B, 0xE0867832},
	{0x16134C45F74D2894, 0x645075230AF517E, 0x638788BD12EAF884, 0xBB57CA874264F15D, 0x83B306DD1713D2A4, 0x623C0C908B287197, 0xBA8FC5C2734F4A9E, 0xFC971C3F17C55B04, 0x61546665082121F8, 0x18551EAB8},
	{0x47236352CCCCD4B, 0x807700F6D1EDDCEE, 0x17537341BAEA6CBA, 0x504419A64B50C73D, 0x2591E7171ACC64DE, 0xEB04A5C122391E85, 0xD96557BF89D24F8E, 0x780CD38FF64726CB, 0xFDAFF780CC4EB54E, 0x11861577},
	{0x302306EE9997BC4D, 0x3E045AB634949BFF, 0xCC742F753994DECA, 0x7E7F0333368EA0FF, 0x8A8430A2AB229760, 0xE1D75408F952D6F, 0x4F3D600657F280B3, 0xDE89E80487A538B0, 0x68D1E57A19D64620, 0xD0027A4E},
	{0x5970FB914A91F4D4, 0xEF97BEC7A9F0D144, 0x693EE802E571FE58, 0xCE320E8569E81BC7, 0x2FE521976533C96E, 0xE5D7F9268E7D0519, 0xFAFD246C31C4833, 0x7218544447E13899, 0xEA1FFC738CF3C5D3, 0x95EB5BCC},
	{0x5B105189D0CA2ADE, 0xCEDD8AD3AABF2386, 0xA052407BEE2D471, 0xD51FFCA04EBD31DF, 0x7CFF2C4FBFF9840F, 0x68E863DD7BE0FB3F, 0x71ADC0F419B76B2F, 0x9F2486034EC06308, 0xEE7F083774F1E588, 0x1782E172E},
	{0x26B06F653E426CA3, 0x653E426B06F653E4, 0x6B06F653E426B06F, 0x53E426B06F653E42, 0xCD29653E426B06F6, 0xE25934934CDB47A9, 0x368EE07F45083838, 0xFF81E126E2784509, 0xB7FBA2C6FB2C0307, 0x66F35905},
	{0x7C84D60DEC9AE6B4, 0xDECA7C84D60DECA, 0xC84D60DECA7C84D6, 0xDECA7C84D60DECA7, 0x5F6E0DECA7C84D60, 0x5E606F56135F2DCD, 0x6D881359A8F60397, 0x381AE539FC37E98C, 0x29BF8428D894967B, 0x12918EDC8},
	{0xBF924F1BF0EB7ADA, 0xC78EDA5EA1CEF780, 0x48C247D7D53CFE2, 0x3FE9A831F3E25AF8, 0xC4223C70A27E8F90, 0x51DCF86B78B6D4C0, 0x29A14DBF09A8A42, 0x35B0B186FFA03AA9, 0xB34DDE1D41C9D131, 0xF56A46E3},
	{0x253AF4484C94CA03, 0x32A226E35D2A4AA2, 0x4FB18C5BBCF1589A, 0x3E17070314360DE9, 0xB94BC9B9873E96C0, 0x8A3E6EFEA06EB882, 0xC06DE286FA2EECCD, 0x124298F6432B070E, 0x25CB59755BA0A18B, 0x14B1CD916},
	{0xBF83218A8CCC3E78, 0x5BD5A5F17E801598, 0x2F948F576364D3E7, 0xF9D2270EC3408B6F, 0xC00D05257D2C78BC, 0x3CA32AC0114D7EAF, 0x16D9296AC7B7D2D4, 0x6E661F03BA457337, 0x3F69FD67E3F62F31, 0xA70AFB41},
	{0x67B914B86B6BD1E0, 0x9E7D53018B3CF47E, 0x4A5172C659663E8A, 0xCE0F42ACA114BBDA, 0xCF29856D865D954F, 0xDC1EF1F853824B06, 0x8D28378F21C8EB01, 0x8140F83D8E9FBBA9, 0xEF3B3FBF4817BA, 0x1CEA072EF},
	{0xB04ACFA73EA0C46C, 0xCF933F68BB9B3F4A, 0x15AE5CE2F27DB04C, 0x1695B1823A1F6D59, 0xD753A316CA51CB6, 0x7445E2D99B0BA50B, 0xB6F9A9A580E786, 0x51545996DACD4F17, 0x112B452323F4F0A3, 0xAFBF4B5F},
	{0xA8982A5E43B37B3, 0x4D1DFB947C1F2BC7, 0x7A938628C766F64A, 0x545AB97381CCDA04, 0xD8ACCB93F83D4C5B, 0xCA5959634A186221, 0x9E72DEA20E161486, 0x3FB2EF9969B4F6C, 0xA62EC7356550C78, 0x1C0F7BD25}
};

static const uint64_t v_3_torsion[TABLE_V3_LEN][2 * NWORDS64_FIELD] = {
	{0x6EB3E454124AAF4, 0xEB3E45306EB3E453, 0x3E45306EB3E45306, 0x45306EB3E45306EB, 0x286B3E45306EB3E, 0x375E62F5BC7C82AF, 0x650BCBF202B9AE38, 0x3FDD465F450DFCFD, 0xC63A99445C986900, 0x250ED785C, 0x2983759F1FCF38D7, 0x83759F22983759F2, 0x759F22983759F229, 0x9F22983759F22983, 0x4EEA3759F2298375, 0x87430979EEB424F2, 0x4D75F319566660DA, 0x30CD05437A146239, 0xA2641600BDE8C04, 0x712AA3BE},
	{0xCF43C7FB897F2FE2, 0x3C7FB84C2F0E011E, 0xFB84C2F0E011ECF4, 0x4C2F0E011ECF43C7, 0x751C11ECF43C7FB8, 0x1D007C113B5D095, 0x4235B0D5E66773CE, 0x9E42B6C94350F39E, 0xB98F10656ABF493B, 0xA2738C79, 0xC023D9E87818892A, 0x3D9E878FF70985E1, 0xE878FF70985E1C02, 0x8FF70985E1C023D9, 0x52445E1C023D9E87, 0xC4A56A1DB3037852, 0xBE1F9B0F8C463146, 0xD7DB04211C8A255B, 0x87C4C4B67CF93C1C, 0x96C2BFBD},
	{0xBB5E4448409D1B64, 0x9650668A321ACCD8, 0x48D39ADBC2F1339E, 0x8F7071B9DA7AD093, 0xDC2BFF88AE51552D, 0x67B12046E4504FA2, 0xDAC40568A6E7ACA6, 0x62ACCE83A2BE7FE7, 0xCEBB880D5DC0D19E, 0x25FA2A502, 0x12FE408DB093593F, 0xAAF044FB38FAC1A9, 0x2726CD500D0CEF1, 0x1605D37D0757467F, 0x1B66F6542117A77, 0x1350C0D1B1A7435E, 0x51F0069F69135871, 0x203791450029A4BC, 0x42B7E3FCB870BE19, 0x1CDC6F300},
	{0x7B5CB802B56792C7, 0xBD1E27BB33DC664B, 0xC2399B484A347FD4, 0x5CB802B42E1D844C, 0x7699BB33DC664B7B, 0xA62220120D0CB91A, 0x24AAF6ED9C3527F0, 0x1FA962A3FE0A08F3, 0x219A48CC422DF1B1, 0x118AF1785, 0x2085638DAFEB8A83, 0xFD68A923B68BC5DE, 0x9743A21DF7A9C724, 0x85638DB029756DC4, 0x9A323B68BC5DE20, 0xDC5A0F3CC5799C18, 0x3CDBDDF993C3E933, 0x399816785890182E, 0x3DDF5FEAA7F4990C, 0xAC3F3421},
	{0x63BDA9468BD342F5, 0x7403E8DA50E41EF7, 0x7B32C72BE3C39E62, 0xA4AEF030E69A6DAF, 0xC8E9EC938255BB45, 0x5AE36274AC3BB919, 0xAF73A8FEA01C6E98, 0x6C9E0C391AEA075A, 0xDE3F1DFEA825A09A, 0x495C0A99, 0xCDBBE48241B79B59, 0x29FF396B0854C507, 0xC91466B43E7DC219, 0xA46772E1FAE15C03, 0xC113B29586A5D8CF, 0x44FE6A6E5B9FB7B0, 0x11825F97D892585E, 0xC1F3209A0E0006B4, 0x313425C4139E19C4, 0x14DB5D37A},
	{0x3011EB8A0C610EAE, 0x8632968FCBEC6D1D, 0xEF749BB7DE6283EB, 0x1038A58CDB855B10, 0x5574052BEBB46182, 0xB3E1CF3C83597C0F, 0xA718ADD06AD56610, 0x359FB5AA447520F6, 0xCDF6E662B8EB669C, 0x9CC50ACC, 0x4E213592827CF602, 0xB80C916F6BF2EBB6, 0xD05F9D238FE11743, 0x5FC7793D16D042BF, 0xBD58FD8E99F594EA, 0xC1100B4C07C4A897, 0x29F16A0996A54969, 0x3A7EB43A7FC5530D, 0xB3EF41DC7BE3CE6C, 0x4A3A4407},
	{0x32CEC987C02E1E16, 0xB7E44C9BBD497EE5, 0x59C0313782E83DF1, 0x2C71AF09B871F5DE, 0xE44DFBF49F9CEA51, 0xC4BD0DF1CC341EB3, 0x7EC7C21EE53DE39D, 0x2B1B1BB9C6276C2C, 0xF86DE3A5ABE9539E, 0x2302FFE8B, 0x28DF22C8DCD7982F, 0x426541E94DCF04E7, 0x98CA1BA8F9B2855A, 0xF4067E6F7EA49E1D, 0xD03034D68F095472, 0x7D26821B325A3A85, 0x87DB2B11EC3322D6, 0x73BAAA33C117FB57, 0x92225964AC014B47, 0x12E0D7FE7},
	{0xDF2479A490E3F3DA, 0xE07894F7FF684C36, 0x9A5E7229EF62380A, 0xE88DF8EE954C69A4, 0x800A8000EAE4D453, 0xAE77A73EF2BD6B7D, 0xDA18723BBC2CA8D7, 0xF186FA30727D2A31, 0x291452F619104C5B, 0x1E7F1399, 0x3732F21D651B1FEE, 0x8DDCCB2BDD071E43, 0x9641B49A54A8A116, 0xDA68148CA4D341DD, 0xC32B7A049A23B8F4, 0x24932D3124EB482F, 0x84AE00B5C46870B9, 0xCEAE14C3C3D55562, 0x97148EC2C4193A5, 0x1D94CB43},
	{0x5DDBBCF90B937BD9, 0x3E84379AA3442E8, 0xCC25DC2808E0C528, 0xFB444A0D98465DEB, 0xD69F5A96BE71AC2C, 0x13938F1D6E581318, 0x37AE0B4F7DAC04ED, 0x863F2AD318BF5A7F, 0x397673E27BAC9DEA, 0x1216A9621, 0x904B65B8E15A658, 0xC7BB97BD7B0C45E2, 0x93CD91DECA863B07, 0x7E159E553BF8497, 0x5075FEF394A53914, 0xE7E384970BABE75A, 0x5968B82EF98ECFC6, 0x29B5FE78CEF214F8, 0xE3F63B7F9DF9725C, 0xFDF9B378},
	{0x57CC8084740AF239, 0x5703E093ED1DEC26, 0x700B10597FE6838D, 0xBDADB8A8DE7E5988, 0x3EEB9E3F634109D8, 0xC632284A295D87A2, 0xFF7ED47380421F0, 0x46A42D85372D6D51, 0xCEB02C53FD6561AD, 0x68700397, 0x94083C312FDD1CCF, 0x8D8BBEBC2683FDD4, 0xDF6AA14583575AC1, 0xDF3FF19FF44AEEA3, 0xCEBEB4FB032ECD7D, 0x9CE783658636286, 0x7B3D4444C53B3D32, 0xB8749FA7F3A9070, 0xC415DCDB6A0AD61E, 0xB7869319},
	{0x139CF42869D49482, 0xBC6F0C720A8973B, 0x28E407583814A9E0, 0xF00EFBF2ED7F95AF, 0x7D480A57AEF5BFD3, 0x7EC46EDF97D9EAC2, 0xB3F987339979273B, 0x699F07E12CA4729D, 0x7A1D6F1D94D877DD, 0x24F8BFBDA, 0x971EC29363AFF4EF, 0xA525D0504E28FD9, 0xB1499925B9A9134F, 0x3790DCE6BA9C9907, 0x9E2EB2BD7D2BDE58, 0xB1A37E98CE909692, 0x4C9EB9AB58C0D4A9, 0x77A16CF26A2492C6, 0xB2121E58A818AFB2, 0x5FF8FB81},
	{0xD59C657588B26B4C, 0xB824E399449753E5, 0xB2EA59D067504894, 0x2E77C98265BE8415, 0x3A34AD15D89E7895, 0x6AA2E3EAB0F65905, 0x3A464491A81B6AD6, 0x64AFCC145F1EE37C, 0xEF30841D92867D4C, 0x1653C11AC, 0xC4A62DC72662944A, 0x4BC6EC7683E05B2E, 0xDE0AB70705491613, 0x7FC5FD6400FBAE3A, 0xBE093F12B7441F4E, 0x754D61F85BE9B748, 0xED144F4AD57B89A0, 0x4B6DFB51DECE2551, 0x67980426FCD240D0, 0x75699852},
	{0xD3D4BB413D7061BE, 0x18AEC9D6854ECB25, 0x74912785EEF65EF3, 0x428B21B602840579, 0x7BEDFBC19961AB28, 0xE537045DFA2E6F0C, 0xAF709EC9946F55BB, 0xD4633E3CA2BBFE3D, 0xDDF84B965ACD8170, 0x1837A2775, 0xFB8249F345205225, 0xDC81B2C34B0B30FC, 0xF5B9D98AC4766CBA, 0x8B37B96AC565EDF5, 0xAB88EC455AB3E730, 0xF99410A2815AE199, 0x926C8A85493EB093, 0xC7082D622A51444B, 0xFE11910D6EC752C1, 0x22D0E6336},
	{0xFBDC842A96D96B13, 0xEE3034ABA5D2EC9A, 0x5D4BFE83ECA3D306, 0x10CE1C3DB0210EC, 0x645EB45E9AC10E46, 0x99085D8D171BC2BE, 0x9F361AF1FC456B18, 0x8D9D38D8C416A25, 0xB071C0B7D59CE33C, 0xD89DA6CB, 0x770A7B6239055C1F, 0xE9767FC8D42B73D3, 0xFCC1954DC3AC1EE7, 0x1251746D17D4CA40, 0xD83FEEE6736971D, 0xB0210B1F8D0AB262, 0x3DE49D676CD13FB7, 0x1657FEBB10C212AF, 0x2F61A2D8A3F81312, 0x18E3342F8},
	{0xD08AB92DAE048DB1, 0xAFDFB841D45B6CDB, 0x4E16B9F649DBDAD4, 0x8F7BD80A86CD6FE4, 0xE9D860EE31368282, 0xAB87FA97EF9B4B4E, 0x595B5FD6D5CD3C22, 0x8B2AF45E9CDA91B3, 0x2DFE5B32A817912D, 0x1FB079EE, 0x6B3EC443091B482D, 0x9248D66829014899, 0x638D81CC4AE199D, 0x5E8DE73EEB6DE96A, 0x55771848724D17F0, 0x734CB651B4339600, 0xFD2432CB79D99C1B, 0x373869CEA55DDEC4, 0x39D83CB6F4815AF5, 0xD31D84E9},
	{0x8FC08908D008677E, 0xF50D53E90F7118D4, 0x38984342BC5C5F12, 0x9EF8ADEE9D890053, 0x2E9134B510F28488, 0x8951AC1CA56CA72C, 0x7CFED4FEED36EDEC, 0xCDF3038414AC384C, 0xA2C3889B433C8C6F, 0x27A01960F, 0xA3526DB518930119, 0xDC37B0E13644ED3F, 0x6E31E890F08A37BD, 0xA37A5E57024A8AA0, 0x40B194BE09CD892F, 0xF8D9FD2F6BFCA6BD, 0x20DFBAEF4B42549B, 0x45F0B1D7FBC11721, 0x5E7E3BB644063BE0, 0x171428956},
	{0xB637BBC305F67791, 0x204FF0D2AEDF672E, 0xF92D3782494289DB, 0x24F7EB7320299A80, 0x977CC127123CD922, 0xB6C51C2947B68DB4, 0xBEFC8E5B0AE18F09, 0xFABDF4A65AAD3791, 0xAEBB5AA66A4EC660, 0x9F9CF17F, 0x110C3AF5E13E1951, 0xF40DC401CC686B06, 0x68EEA45A46B10505, 0x325395A25A0F0DE9, 0x1C92E56F56D28E24, 0xD0F8A55787894E1F, 0xEEEC7393357C5523, 0x5FB696B3518CC792, 0xD39D05F3A9382E5A, 0x26B74CDB1},
	{0xBDF8A208F713F4A9, 0x18481280856177D9, 0x10BD738108662F2C, 0x1EF37C1AB5531600, 0x98D3C52040B3E9DB, 0x8E988D26C011FF69, 0x18AE98AA255688B, 0x8E60CDC1B136648B, 0x7E91BFE825833F56, 0x13DB93F4A, 0xC90DED2C7313B98, 0x82F6CE5855C8D3AC, 0xD22BF07E7380D124, 0xE912C49BBF549FD8, 0xBD34790D9C351C45, 0x4A1A372896AB66BF, 0x44E6AD3D3CB7D968, 0x473B9962DF10B605, 0x2DC4E87221525887, 0xD61465},
	{0x3E77A9B726C4E35A, 0x72508865383E24E9, 0x703A850A1795B24B, 0xB744F39B78F82E6, 0xBCA3D4CB44C8DE88, 0x98BB9A1523EDC1A1, 0x1055B9EF0ED241A1, 0xDF619C4A7A3305B, 0xD3CAD19AFBE71CE8, 0x8A810A03, 0x78553A5AD2AFD1FC, 0x87C486E5D053F7C5, 0xD49A2C8D3988259, 0xD059B9BB567D1CB1, 0x5D10F35FBB64BF63, 0xB6A58DE0F78BF9DD, 0xCD0DEBAFEA4B0478, 0xDAF46A0964D3FD0A, 0x8E33C74FE2F70AFC, 0x21CAAD2F8},
	{0x3BABE989C4B7C8C6, 0x2BF8F447EF3BDD7D, 0x20EAAD8C5457CF18, 0x8C176F27BA3E5B6D, 0x6994DF1010BAB10B, 0xC508E30E46C0CA1E, 0xA48AA3C16405257B, 0xFDA00D371BEC4681, 0x9D86B52EDF6EC9F8, 0x1C34BB8B6, 0xADB4F379816C780A, 0xC25F504173CEACF5, 0x1DA0375FEB0BBE8C, 0xB87264B37860585, 0xE0E1B8080C0215AB, 0xA4E2CD906374B7FC, 0x5326260B7A3021ED, 0x168C806A9FD020B4, 0xDD2E356F2066C037, 0x110BD582B}
};

// Setting up macro defines and including GF(p), GF(p^2), curve, isogeny and kex functions
#define fpcopy fpcopy610
#define fpzero fpzero610
#define fpadd fpadd610
#define fpsub fpsub610
#define fpneg fpneg610
#define fpdiv2 fpdiv2_610
#define fpcorrection fpcorrection610
#define fpmul_mont fpmul610_mont
#define fpsqr_mont fpsqr610_mont
#define fpinv_mont fpinv610_mont
#define fpinv_chain_mont fpinv610_chain_mont
#define fpinv_mont_bingcd fpinv610_mont_bingcd
#define fp2copy fp2copy610
#define fp2zero fp2zero610
#define fp2add fp2add610
#define fp2sub fp2sub610
#define mp_sub_p2 mp_sub610_p2
#define mp_sub_p4 mp_sub610_p4
#define sub_p4 mp_sub_p4
#define fp2neg fp2neg610
#define fp2div2 fp2div2_610
#define fp2correction fp2correction610
#define fp2mul_mont fp2mul610_mont
#define fp2sqr_mont fp2sqr610_mont
#define fp2inv_mont fp2inv610_mont
#define fp2inv_mont_bingcd fp2inv610_mont_bingcd
#define fpequal_non_constant_time fpequal610_non_constant_time
#define mp_add_asm oqs_kem_sike_mp_add610_asm
#define mp_subaddx2_asm oqs_kem_sike_mp_subadd610x2_asm
#define mp_dblsubx2_asm oqs_kem_sike_mp_dblsub610x2_asm
#define crypto_kem_keypair OQS_KEM_sike_p610_compressed_keypair
#define crypto_kem_enc OQS_KEM_sike_p610_compressed_encaps
#define crypto_kem_dec OQS_KEM_sike_p610_compressed_decaps
#define random_mod_order_A oqs_kem_sidh_p610_compressed_random_mod_order_A
#define random_mod_order_B oqs_kem_sidh_p610_compressed_random_mod_order_B
#define EphemeralKeyGeneration_A oqs_kem_sidh_p610_compressed_EphemeralKeyGeneration_A
#define EphemeralKeyGeneration_B oqs_kem_sidh_p610_compressed_EphemeralKeyGeneration_B
#define EphemeralSecretAgreement_A oqs_kem_sidh_p610_compressed_EphemeralSecretAgreement_A
#define EphemeralSecretAgreement_B oqs_kem_sidh_p610_compressed_EphemeralSecretAgreement_B
#ifdef USE_SIKEP610_ASM
#define USE_SIKE_ASM
#endif

#if defined(_AMD64_) && defined(USE_SIKE_ASM)
#include "AMD64/fp_x64.c"
#elif defined(_ARM64_)
#include "ARM64/fp_arm64.c"
#else
#include "generic/fp_generic.c"
#endif

#include "../fpx.c"
#include "../ec_isogeny.c"
#include "../compression/torsion_basis.c"
#include "P610_compressed_pair_tables.c"
#include "../compression/pairing.c"
#include "P610_compressed_dlog_tables.c"
#include "../compression/dlog.c"
#include "../compression/sidh_compressed.c"
#include "../compression/sike_compressed.c"
