use super::{Report, ReportConfig, ReportData};
use crate::util::percentage;
use anyhow::Error;
use fehler::throws;
use serde::Deserialize;
use std::{fs::File, path::Path};

#[derive(Debug)]
pub(super) struct TopCrateInfo {
    name: String,
    version: semver::Version,
    id: String,
}

impl Report {
    #[throws]
    pub(super) async fn top_crates(&self, _config: &ReportConfig) -> Vec<TopCrateInfo> {
        let path = self.data_dir.join("crate-information.json");
        if path.exists() {
            tokio::task::spawn_blocking(move || load_top_crates(&path)).await??
        } else {
            log::warn!("no crate-information.json file found");
            vec![]
        }
    }

    #[throws]
    pub(super) fn write_top_crates(&self, _config: &ReportConfig, data: &ReportData) {
        let stable_crates = data
            .top_crates
            .iter()
            .filter(|c| c.version.major >= 1)
            .count() as u64;
        let total_crates = data.top_crates.len() as u64;
        let percentage = percentage(stable_crates, total_crates);

        let output = self.output_dir().join("top_crate.csv");
        let output_file = &mut File::create(output)?;
        let mut csv = csv::Writer::from_writer(output_file);
        csv.write_record(&["Measurement".to_string(), "Value".to_string()])?;
        csv.write_record(&[
            "Total 'significant' crates".to_string(),
            total_crates.to_string(),
        ])?;
        csv.write_record(&[
            "crates at 1.0 or higher".to_string(),
            percentage.to_string(),
        ])?;
    }
}

/// Loads the `crate-information.json` generated by the [top-crates playground code][pc].
///
/// [pc]: https://github.com/integer32llc/rust-playground/tree/master/top-crates
#[throws]
fn load_top_crates(path: &Path) -> Vec<TopCrateInfo> {
    #[derive(Deserialize)]
    struct TC {
        name: String,
        version: String,
        id: String,
    }
    let data = std::fs::read_to_string(&path)?;
    let data: Vec<TC> = serde_json::from_str(&data)?;
    let mut vec = Vec::new();
    for TC { name, version, id } in data {
        let version = semver::Version::parse(&version)?;
        vec.push(TopCrateInfo { name, version, id });
    }
    vec
}
