//! See [Solver].

use crate::core::stop_criterion::StopCriterion;

use super::{Objective, Problem};

/// A pair of (Solution, Value) generated by the [Solver]
pub type Iteration<P> = (<P as Problem>::Solution, <P as Problem>::Value);

/// A solver is a procedure which seeks to find a good solution for a given [Problem].
///
/// There's two kinds of solvers:
/// - Metaheuristics
///     - They are generic over the [Problem] being solved.
/// - Heuristics
///     - They're [Problem]-specific.
pub trait Solver<SC: StopCriterion<Self::P>> {
    /// The problem being solved
    type P: Problem;

    /// A iteration consists on a step to generate a candidate solution
    fn iterate(&mut self, stop_criterion: &mut SC) -> Option<Iteration<Self::P>>;

    /// Execute the whole process defined by the solver to achieve a good solution
    ///
    /// By default, it executes [iterate][Self::iterate] while the stop criterion isn't met and returns
    /// the best solution found among all iterations.
    fn solve(&mut self, stop_criterion: &mut SC) -> Option<<Self::P as Problem>::Solution> {
        let (mut best_solution, mut best_value) = self.iterate(stop_criterion)?;
        stop_criterion.update(best_value);

        while !stop_criterion.should_stop() {
            let (candidate_solution, candidate_value) = {
                match self.iterate(stop_criterion) {
                    Some(s) => s,
                    None => break,
                }
            };

            if candidate_is_better::<Self::P, _>(&best_value, &candidate_value) {
                best_solution = candidate_solution;
                best_value = candidate_value;
            }

            stop_criterion.update(candidate_value);
        }

        Some(best_solution)
    }
}

fn candidate_is_better<P: Problem, T: Ord>(current: &T, candidate: &T) -> bool {
    match P::OBJECTIVE {
        Objective::Min => candidate < current,
        Objective::Max => candidate > current,
    }
}
