//! # opt_args
//!
//! This crate lets you easily derive macros to call functions and instantiate structs
//! without having to specify all of their arguments.
mod parser;


use proc_macro::TokenStream;
use std::collections::HashMap;
use quote::quote;
use syn::{FnArg, Item, Pat};
use itertools::Itertools;
use crate::parser::{OptArgs, OptArgsItem};


/// Apply to a function or struct to generate a macro that can be called with named optional arguments.
///
/// You can provide default values for optional arguments or leave them blank:
/// if left blank, Default::default() will be used (if possible, otherwise the macro won't work).
/// For example, blank default values are usually not available for references.
/// In that case you must specify the default value for the argument.
///
/// Examples below are about functions but the same goes for structs: same rules, same syntax.
///
/// Optional arguments must cover all the final positions from the first optional argument to the
/// last function argument. For example:
/// ```
/// #[opt_args(c = 5)]
/// fn f(a: i32, b: i32, c: i32, d: i32, e: i32) {
///     println!("a = {}, b = {}, c = {}, d = {}, e = {}", a, b, c, d, e);
/// }
/// ```
/// is not a valid macro call since arguments `d` and `e` are not optional,
/// but they come after `c` which is optional.
/// Since all the arguments after the first optional must be optional,
/// one correct macro call (w.r.t. `c` being the first optional) would be the following:
/// ```
/// #[opt_args(c = 5, d, e)]
/// fn f(a: i32, b: i32, c: i32, d: i32, e: i32) {
///     println!("a = {}, b = {}, c = {}, d = {}, e = {}", a, b, c, d, e);
/// }
/// ```
/// where all arguments after the first optional (`c`) are optional.
/// In other words: if you mark `n` arguments as optional, they **must be the last** `n` arguments
/// of the function.
///
/// Optional arguments can be passed to the macro in any order,
/// in both the `opt_args` and the function macro calls.
/// For example:
/// ```
/// #[opt_args(d, c = 5, e)]
/// fn f(a: i32, b: i32, c: i32, d: i32, e: i32) {
///     println!("a = {}, b = {}, c = {}, d = {}, e = {}", a, b, c, d, e);
/// }
/// ```
/// would be a correct macro call, and the code generated would be the same as before.
///
/// To use the function macro, simply use the name of the function as a macro and pass first the
/// positional required arguments, then the named optional arguments (in any order),
/// like in the following (referred to previous examples):
/// ```
/// f!(1, 2, e = 6, c = 3);
/// ```
/// In this case we would have `d = 0`, since no default value was provided for the optional `d`.
///
/// Once an argument is marked as optional, it cannot be used as positional in the function macro:
/// ```
/// f!(1, 2, 3, d = 5);
/// ```
/// is not a valid call since `c` is used as positional (with value `3`).
/// The correct call would be the following:
/// ```
/// f!(1, 2, c = 3, d = 5);
/// ```
///
/// The macro is made in such a way that it is possible to use the function macro inside the
/// original function:
/// ```
/// #[opt_args(d, c = 5, e)]
/// fn f(a: i32, b: i32, c: i32, d: i32, e: i32) {
///     if a == 3 {
///         f!(1, b, c = c, d = d, e = e);
///     } else {
///         println!("a = {}, b = {}, c = {}, d = {}, e = {}", a, b, c, d, e);
///     }
/// }
/// ```
/// which would result in a recursive call.
///
/// The macro generated by `opt_args` is annotated with `#[macro_export]`, meaning that you can use
/// it outside the module in which the function is defined. Anyway, the actual possibility to use
/// the macro depends on the possibility to use the function: if you don't expose the function
/// to the outside, the macro will be visible but the code won't compile.
#[proc_macro_attribute]
pub fn opt_args(attr: TokenStream, item: TokenStream) -> TokenStream {
    let opt_args = syn::parse_macro_input!(attr as OptArgs);
    let input = syn::parse_macro_input!(item as OptArgsItem);
    let combinations = combinations(
        opt_args
            .attrs
            .iter()
            .map(ToString::to_string)
            .collect()
    );
    let (name, args, is_function) = match &input {
        OptArgsItem::Function(input) => {
            let name: String = input.sig.ident.to_string();
            let args: Vec<String> = input
                .sig
                .inputs
                .iter()
                .map(fn_arg_to_name)
                .collect();
            (name, args, true)
        }
        OptArgsItem::Struct(input) => {
            let name: String = input.ident.to_string();
            let args: Vec<String> = input
                .fields
                .iter()
                .map(|f| f.ident.as_ref().unwrap().to_string())
                .collect();
            (name, args, false)
        }
    };
    let macro_branches = macro_branches(
        &name,
        args,
        combinations,
        opt_args,
        is_function,
    );
    let macro_body = macro_branches
        .iter()
        .map(|(p, b)| format!("({}) => {{{}}}", p, b))
        .join(";");
    let macro_token: Item = syn::parse_str(
        &format!(
            "#[macro_export]macro_rules! {} {{{}}}",
            name,
            macro_body
        )
    ).unwrap();
    let result = quote! {
        #macro_token

        #input
    };
    result.into()
}


///converts a single FnArg object into a string representing its ident
fn fn_arg_to_name(arg: &FnArg) -> String {
    match arg {
        FnArg::Typed(pat_type) => {
            if let Pat::Ident(pat_ident) = &*pat_type.pat {
                return pat_ident.ident.to_string();
            }
            panic!("If you can read this, please open a PR")
        }
        _ => panic!("If you can read this, please open a PR")
    }
}


/// returns the vector of all combinations for optional arguments
fn combinations(args: Vec<String>) -> Vec<Vec<String>> {
    let mut result = vec![];
    for i in 0..=args.len() {
        args
            .iter()
            .permutations(i)
            .for_each(
                |combination| result.push(
                    combination
                        .iter()
                        .map(ToString::to_string)
                        .collect()
                )
            )
    }
    result
}


/// creates couples (pattern, branch) where:
/// pattern is a pattern of the final macro
/// branch is the code that will substitute the macro call
fn macro_branches(
    name: &str,
    args: Vec<String>,
    combinations: Vec<Vec<String>>,
    opt_args: OptArgs,
    is_function: bool,
) -> Vec<(String, String)> {
    let required_args_formatter = if is_function {
        |a: &String| format!("${}", a)
    } else {
        |a: &String| format!("{}: ${}", a, a)
    };
    let opt_args_formatter = if is_function {
        |a: &String, v: &String, c: &Vec<String>| if c.contains(a) {
            format!("${}", a)
        } else {
            v.to_string()
        }
    } else {
        |a: &String, v: &String, c: &Vec<String>| if c.contains(a) {
            format!("{}: ${}", a, a)
        } else {
            format!("{}: {}", a, v)
        }
    };
    let required_args_num = args.len() - combinations[combinations.len() - 1].len();
    if required_args_num == args.len() {
        panic!("Provide at least one optional parameter");
    }
    let args_indexes: HashMap<String, usize> = args
        .iter()
        .enumerate()
        .map(|(i, a)| (a.clone(), i))
        .collect();
    opt_args
        .iter()
        .for_each(
            |a| if args_indexes[&a.name] < required_args_num {
                panic!("All the arguments after the first optional must be optional")
            }
        );


    let required_args: Vec<String> = args
        .into_iter()
        .take(required_args_num)
        .collect();
    let opt_arg_to_value: Vec<(String, String)> = opt_args
        .iter()
        .map(|a| (a.name.clone(), a.value.clone()))
        .sorted_by_key(|(a, _)| args_indexes[a])
        .collect();
    let required_args_pattern = required_args
        .iter()
        .map(|a| format!("${}:expr", a))
        .join(",");
    let mut opt_args_pattern: String;
    let required_args_branch = required_args
        .iter()
        .map(required_args_formatter)
        .join(",");
    let mut opt_args_branch: String;
    let mut result: Vec<(String, String)> = vec![];


    for combination in combinations {
        opt_args_pattern = combination
            .iter()
            .map(|a| format!("{} = ${}:expr", a, a))
            .join(",");
        let pattern = vec![&required_args_pattern, &opt_args_pattern]
            .iter()
            .filter(|e| !e.is_empty())
            .join(",");
        opt_args_branch = opt_arg_to_value
            .iter()
            .map(|(a, v)| opt_args_formatter(a, v, &combination))
            .join(",");
        let branch = vec![&required_args_branch, &opt_args_branch]
            .iter()
            .filter(|e| !e.is_empty())
            .join(",");
        result.push((
            pattern,
            if is_function {
                format!("{}({})", name, branch)
            } else {
                format!("{}{{{}}}", name, branch)
            }
        ));
    }
    result
}
