pub mod discrete_observation;
use crate::InternalData as Data;
pub mod util;
use core::fmt::Debug;
use discrete_observation::DiscreteObservation;
use md_continuous_observation::ContinuousObservation;
pub mod md_continuous_observation;
use serde::{Deserialize, Serialize};

/// trait of an observation which are simple transformation of `Data` struct
pub trait Observation {
    /// unique identifier of the observation
    fn to_int(&self) -> Option<usize> {
        None
    }
    /// observation feature the order is: tree observation (tree_1, tree_2 ...), excavator speed, excavator angular speed, stick extension, stick angle, finger state
    fn to_vec(&self) -> Option<Vec<f32>> {
        None
    }
    /// number of trees at proximity of the operator
    fn tree_observation(&self) -> TreeObservation;
    fn excavator_speed_observation(&self) -> String;
    fn excavator_speed_direction_observation(&self) -> Option<String> {
        None
    }
    fn excavator_angular_speed_observation(&self) -> String;
    fn stick_extension_observation(&self) -> String;
    fn stick_angle_observation(&self) -> String;
    fn finger_state_observation(&self) -> String;
}

impl Debug for dyn Observation {
    fn fmt(&self, f: &mut core::fmt::Formatter<'_>) -> core::fmt::Result {
        let excavator_speed_direction =
            if let Some(direction) = self.excavator_speed_direction_observation() {
                direction
            } else {
                String::from("unavailable")
            };
        write!(f, "Observation {{ tree_observation: {tree_observation:?}, excavator_speed: {excavator_speed},excavator_speed_direction: {excavator_speed_direction}, excavator_angular_speed: {excavator_angular_speed} stick_extension:{stick_extension}, stick_angle: {stick_angle}, finger_state:{finger_state} }} ",
         tree_observation = self.tree_observation(),
         excavator_speed = self.excavator_speed_observation(),
         excavator_speed_direction = excavator_speed_direction,
         excavator_angular_speed = self.excavator_angular_speed_observation(),
         stick_extension = self.stick_extension_observation(),
         stick_angle = self.stick_angle_observation(),
         finger_state = self.finger_state_observation()

        )
    }
}

#[derive(Debug)]
/// type of tree observation
pub enum TreeObservation {
    NumberOfTreesClose(String),
    DistanceOfTrees(Vec<String>),
}

#[derive(Clone, Deserialize, Serialize)]
pub enum ObservationType {
    DiscreteObservation,
    ContinuousObservation,
}

pub fn new_observation(
    observation_type: &ObservationType,
    data: &[Data; 2],
) -> Box<dyn Observation> {
    match observation_type {
        ObservationType::DiscreteObservation => Box::new(DiscreteObservation::new(data)),
        ObservationType::ContinuousObservation => Box::new(ContinuousObservation::new(data)),
    }
}

pub fn new_observation_vector(
    data: Vec<Data>,
    observation_type: &ObservationType,
    n_step: usize,
) -> Result<Vec<Box<dyn Observation>>, &'static str> {
    if data.len() < 2 {
        return Err("the len of data must be at least 2");
    }
    let mut resp: Vec<Box<dyn Observation>> = Vec::with_capacity((data.len() / 2) + 1);
    let mut prev_data = data[0].clone();
    let iter = data.iter().skip(1).step_by(n_step);
    for el in iter {
        let current_data = [prev_data, el.clone()];
        let obs = new_observation(&observation_type, &current_data);
        prev_data = el.clone();
        resp.push(obs);
    }
    Ok(resp)
}

pub fn new_observation_feature_vector(
    data: Vec<Data>,
    observation_type: ObservationType,
    n_step: usize,
) -> Result<Vec<Vec<f32>>, &'static str> {
    if data.len() < 2 {
        return Err("the len of data must be at least 2");
    } else if is_discrete(&observation_type) {
        return Err("must be a continuous type");
    }
    let mut resp: Vec<Vec<f32>> = Vec::with_capacity((data.len() / 2) + 1);
    let mut prev_data = data[0].clone();
    let iter = data.iter().skip(1).step_by(n_step);
    for el in iter {
        let current_data = [prev_data, el.clone()];
        let obs = new_observation(&observation_type, &current_data);
        prev_data = el.clone();
        resp.push(obs.to_vec().unwrap());
    }
    Ok(resp)
}

/// return the observation and the number of possible state
pub fn new_observation_int_vector(
    data: Vec<Data>,
    observation_type: ObservationType,
    n_step: usize,
) -> Result<(Vec<usize>, usize), &'static str> {
    if data.len() < 2 {
        return Err("the len of data must be at least 2");
    } else if !is_discrete(&observation_type) {
        return Err("must be a discrete type");
    }
    let mut resp = Vec::with_capacity((data.len() / 2) + 1);
    let mut prev_data = data[0].clone();
    let iter = data.iter().skip(1).step_by(n_step);
    for el in iter {
        let current_data = [prev_data, el.clone()];
        let obs = new_observation(&observation_type, &current_data);
        prev_data = el.clone();
        resp.push(obs.to_int().unwrap());
    }
    Ok((
        resp,
        number_possible_observation(&observation_type).unwrap(),
    ))
}

fn is_discrete(observation_type: &ObservationType) -> bool {
    match observation_type {
        ObservationType::DiscreteObservation => true,
        ObservationType::ContinuousObservation => false,
    }
}

pub fn number_possible_observation(observation_type: &ObservationType) -> Option<usize> {
    match observation_type {
        ObservationType::DiscreteObservation => {
            Some(DiscreteObservation::number_possible_observation())
        }
        ObservationType::ContinuousObservation => None,
    }
}
