// SPDX-FileCopyrightText: 2021 Heiko Schaefer <heiko@schaefer.name>
// SPDX-License-Identifier: MIT OR Apache-2.0

use std::convert::TryFrom;

use anyhow::Result;
use nom::{branch, bytes::complete as bytes, combinator, multi, sequence};
use nom::branch::alt;
use nom::combinator::map;

use crate::KeyType;
use crate::parse::algo_attrs;
use crate::parse::algo_attrs::Algo;

#[derive(Debug, Clone, Eq, PartialEq)]
pub struct AlgoInfo(
    Vec<(KeyType, Algo)>
);

impl AlgoInfo {
    pub fn get_by_keytype(&self, kt: KeyType) -> Vec<&Algo> {
        self.0
            .iter()
            .filter(|(k, _)| *k == kt)
            .map(|(_, a)| a)
            .collect()
    }
}

fn key_type(input: &[u8]) -> nom::IResult<&[u8], KeyType> {
    alt((
        map(bytes::tag([0xc1]), |_| KeyType::Signing),
        map(bytes::tag([0xc2]), |_| KeyType::Decryption),
        map(bytes::tag([0xc3]), |_| KeyType::Authentication),
        map(bytes::tag([0xda]), |_| KeyType::Attestation),
    ))(input)
}

fn parse_one(input: &[u8]) -> nom::IResult<&[u8], Algo> {
    let (x, a) = combinator::map(
        combinator::flat_map(crate::tlv::length::length, bytes::take),
        |i| combinator::all_consuming(algo_attrs::parse)(i),
    )(input)?;

    Ok((x, a?.1))
}

fn parse_list(input: &[u8]) -> nom::IResult<&[u8], Vec<(KeyType, Algo)>> {
    multi::many0(sequence::pair(key_type, parse_one))(input)
}

fn parse_tl_list(input: &[u8]) -> nom::IResult<&[u8], Vec<(KeyType, Algo)>> {
    let (input, (_, _, list)) =
        sequence::tuple((
            bytes::tag([0xfa]),
            crate::tlv::length::length,
            parse_list))(input)?;

    Ok((input, list))
}

pub(self) fn parse(input: &[u8]) -> nom::IResult<&[u8], Vec<(KeyType, Algo)>> {
    // Handle two variations of input format:
    // a) TLV format (e.g. Yubikey 5)
    // b) Plain list (e.g. Gnuk, FOSS-Store Smartcard 3.4)

    // -- Gnuk: do_alg_info (uint16_t tag, int with_tag)

    branch::alt(
        (combinator::all_consuming(parse_list),
         combinator::all_consuming(parse_tl_list))
    )(input)
}

impl TryFrom<&[u8]> for AlgoInfo {
    type Error = anyhow::Error;

    fn try_from(input: &[u8]) -> Result<Self> {
        Ok(AlgoInfo(crate::parse::complete(parse(input))?))
    }
}


// test


#[cfg(test)]
mod test {
    use std::convert::TryFrom;

    use crate::KeyType::*;
    use crate::parse::algo_attrs::*;
    use crate::parse::algo_attrs::Algo::*;
    use crate::parse::algo_attrs::Curve::*;
    use crate::parse::algo_info::AlgoInfo;

    #[test]
    fn test_gnuk() {
        let data = [0xc1, 0x6, 0x1, 0x8, 0x0, 0x0, 0x20, 0x0, 0xc1, 0x6,
            0x1, 0x10, 0x0, 0x0, 0x20, 0x0, 0xc1, 0x9, 0x13, 0x2a, 0x86,
            0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0xc1, 0x6, 0x13, 0x2b, 0x81,
            0x4, 0x0, 0xa, 0xc1, 0xa, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1,
            0xda, 0x47, 0xf, 0x1, 0xc2, 0x6, 0x1, 0x8, 0x0, 0x0, 0x20,
            0x0, 0xc2, 0x6, 0x1, 0x10, 0x0, 0x0, 0x20, 0x0, 0xc2, 0x9,
            0x13, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0xc2, 0x6,
            0x13, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xc2, 0xb, 0x12, 0x2b, 0x6,
            0x1, 0x4, 0x1, 0x97, 0x55, 0x1, 0x5, 0x1, 0xc3, 0x6, 0x1, 0x8,
            0x0, 0x0, 0x20, 0x0, 0xc3, 0x6, 0x1, 0x10, 0x0, 0x0, 0x20,
            0x0, 0xc3, 0x9, 0x13, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3, 0x1,
            0x7, 0xc3, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xc3, 0xa,
            0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xda, 0x47, 0xf, 0x1];

        let ai = AlgoInfo::try_from(data.to_vec()).unwrap();

        assert_eq!(
            ai, AlgoInfo(
                vec![
                    (Signing, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Signing, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Signing, Eddsa(EddsaAttrs::new(Ed25519, None))),
                    (Decryption, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Decryption, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Decryption, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Decryption, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(Cv25519, None))),
                    (Authentication, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Authentication, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Authentication, Eddsa(EddsaAttrs::new(Ed25519, None)))
                ]
            )
        );
    }

    #[test]
    fn test_opgp_card_34() {
        let data = [0xc1, 0x6, 0x1, 0x8, 0x0, 0x0, 0x20, 0x0, 0xc1, 0x6,
            0x1, 0xc, 0x0, 0x0, 0x20, 0x0, 0xc1, 0x6, 0x1, 0x10, 0x0,
            0x0, 0x20, 0x0, 0xc1, 0x9, 0x13, 0x2a, 0x86, 0x48, 0xce,
            0x3d, 0x3, 0x1, 0x7, 0xc1, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0,
            0x22, 0xc1, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0x23, 0xc1, 0xa,
            0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc1,
            0xa, 0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb,
            0xc1, 0xa, 0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1,
            0xd, 0xc2, 0x6, 0x1, 0x8, 0x0, 0x0, 0x20, 0x0, 0xc2, 0x6,
            0x1, 0xc, 0x0, 0x0, 0x20, 0x0, 0xc2, 0x6, 0x1, 0x10, 0x0,
            0x0, 0x20, 0x0, 0xc2, 0x9, 0x12, 0x2a, 0x86, 0x48, 0xce,
            0x3d, 0x3, 0x1, 0x7, 0xc2, 0x6, 0x12, 0x2b, 0x81, 0x4, 0x0,
            0x22, 0xc2, 0x6, 0x12, 0x2b, 0x81, 0x4, 0x0, 0x23, 0xc2, 0xa,
            0x12, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc2,
            0xa, 0x12, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb,
            0xc2, 0xa, 0x12, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1,
            0xd, 0xc3, 0x6, 0x1, 0x8, 0x0, 0x0, 0x20, 0x0, 0xc3, 0x6, 0x1,
            0xc, 0x0, 0x0, 0x20, 0x0, 0xc3, 0x6, 0x1, 0x10, 0x0, 0x0,
            0x20, 0x0, 0xc3, 0x9, 0x13, 0x2a, 0x86, 0x48, 0xce, 0x3d, 0x3,
            0x1, 0x7, 0xc3, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0x22, 0xc3,
            0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0x23, 0xc3, 0xa, 0x13, 0x2b,
            0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc3, 0xa, 0x13,
            0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb, 0xc3, 0xa,
            0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xd];

        let ai = AlgoInfo::try_from(data.to_vec()).unwrap();

        assert_eq!(
            ai, AlgoInfo(
                vec![
                    (Signing, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Signing, Rsa(RsaAttrs { len_n: 3072, len_e: 32, import_format: 0 })),
                    (Signing, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP384r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP521r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP256r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP384r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP512r1, None))),
                    (Decryption, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Decryption, Rsa(RsaAttrs { len_n: 3072, len_e: 32, import_format: 0 })),
                    (Decryption, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP256r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP384r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP521r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP256r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP384r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP512r1, None))),
                    (Authentication, Rsa(RsaAttrs { len_n: 2048, len_e: 32, import_format: 0 })),
                    (Authentication, Rsa(RsaAttrs { len_n: 3072, len_e: 32, import_format: 0 })),
                    (Authentication, Rsa(RsaAttrs { len_n: 4096, len_e: 32, import_format: 0 })),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP384r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP521r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP256r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP384r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP512r1, None)))
                ]
            )
        );
    }


    #[test]
    fn test_yk5() {
        let data = [0xfa, 0x82, 0x1, 0xe2, 0xc1, 0x6, 0x1, 0x8, 0x0, 0x0,
            0x11, 0x0, 0xc1, 0x6, 0x1, 0xc, 0x0, 0x0, 0x11, 0x0, 0xc1,
            0x6, 0x1, 0x10, 0x0, 0x0, 0x11, 0x0, 0xc1, 0x9, 0x13, 0x2a,
            0x86, 0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0xc1, 0x6, 0x13, 0x2b,
            0x81, 0x4, 0x0, 0x22, 0xc1, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0,
            0x23, 0xc1, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xc1, 0xa,
            0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc1,
            0xa, 0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb,
            0xc1, 0xa, 0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1,
            0xd, 0xc1, 0xa, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xda, 0x47,
            0xf, 0x1, 0xc1, 0xb, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0x97,
            0x55, 0x1, 0x5, 0x1, 0xc2, 0x6, 0x1, 0x8, 0x0, 0x0, 0x11,
            0x0, 0xc2, 0x6, 0x1, 0xc, 0x0, 0x0, 0x11, 0x0, 0xc2, 0x6,
            0x1, 0x10, 0x0, 0x0, 0x11, 0x0, 0xc2, 0x9, 0x12, 0x2a, 0x86,
            0x48, 0xce, 0x3d, 0x3, 0x1, 0x7, 0xc2, 0x6, 0x12, 0x2b, 0x81,
            0x4, 0x0, 0x22, 0xc2, 0x6, 0x12, 0x2b, 0x81, 0x4, 0x0, 0x23,
            0xc2, 0x6, 0x12, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xc2, 0xa, 0x12,
            0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc2, 0xa,
            0x12, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb, 0xc2,
            0xa, 0x12, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xd,
            0xc2, 0xa, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xda, 0x47, 0xf,
            0x1, 0xc2, 0xb, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0x97, 0x55,
            0x1, 0x5, 0x1, 0xc3, 0x6, 0x1, 0x8, 0x0, 0x0, 0x11, 0x0,
            0xc3, 0x6, 0x1, 0xc, 0x0, 0x0, 0x11, 0x0, 0xc3, 0x6, 0x1,
            0x10, 0x0, 0x0, 0x11, 0x0, 0xc3, 0x9, 0x13, 0x2a, 0x86, 0x48,
            0xce, 0x3d, 0x3, 0x1, 0x7, 0xc3, 0x6, 0x13, 0x2b, 0x81, 0x4,
            0x0, 0x22, 0xc3, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0x23, 0xc3,
            0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xc3, 0xa, 0x13, 0x2b,
            0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xc3, 0xa, 0x13,
            0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb, 0xc3, 0xa,
            0x13, 0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xd, 0xc3,
            0xa, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xda, 0x47, 0xf, 0x1,
            0xc3, 0xb, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0x97, 0x55, 0x1,
            0x5, 0x1, 0xda, 0x6, 0x1, 0x8, 0x0, 0x0, 0x11, 0x0, 0xda,
            0x6, 0x1, 0xc, 0x0, 0x0, 0x11, 0x0, 0xda, 0x6, 0x1, 0x10,
            0x0, 0x0, 0x11, 0x0, 0xda, 0x9, 0x13, 0x2a, 0x86, 0x48, 0xce,
            0x3d, 0x3, 0x1, 0x7, 0xda, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0,
            0x22, 0xda, 0x6, 0x13, 0x2b, 0x81, 0x4, 0x0, 0x23, 0xda, 0x6,
            0x13, 0x2b, 0x81, 0x4, 0x0, 0xa, 0xda, 0xa, 0x13, 0x2b, 0x24,
            0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0x7, 0xda, 0xa, 0x13, 0x2b,
            0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xb, 0xda, 0xa, 0x13,
            0x2b, 0x24, 0x3, 0x3, 0x2, 0x8, 0x1, 0x1, 0xd, 0xda, 0xa,
            0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0xda, 0x47, 0xf, 0x1, 0xda,
            0xb, 0x16, 0x2b, 0x6, 0x1, 0x4, 0x1, 0x97, 0x55, 0x1, 0x5, 0x1];

        let ai = AlgoInfo::try_from(data.to_vec()).unwrap();

        assert_eq!(
            ai, AlgoInfo(
                vec![
                    (Signing, Rsa(RsaAttrs { len_n: 2048, len_e: 17, import_format: 0 })),
                    (Signing, Rsa(RsaAttrs { len_n: 3072, len_e: 17, import_format: 0 })),
                    (Signing, Rsa(RsaAttrs { len_n: 4096, len_e: 17, import_format: 0 })),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP384r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(NistP521r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP256r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP384r1, None))),
                    (Signing, Ecdsa(EcdsaAttrs::new(BrainpoolP512r1, None))),
                    (Signing, Eddsa(EddsaAttrs::new(Ed25519, None))),
                    (Signing, Eddsa(EddsaAttrs::new(Cv25519, None))),
                    (Decryption, Rsa(RsaAttrs { len_n: 2048, len_e: 17, import_format: 0 })),
                    (Decryption, Rsa(RsaAttrs { len_n: 3072, len_e: 17, import_format: 0 })),
                    (Decryption, Rsa(RsaAttrs { len_n: 4096, len_e: 17, import_format: 0 })),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP256r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP384r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(NistP521r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(Secp256k1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP256r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP384r1, None))),
                    (Decryption, Ecdh(EcdhAttrs::new(BrainpoolP512r1, None))),
                    (Decryption, Eddsa(EddsaAttrs::new(Ed25519, None))),
                    (Decryption, Eddsa(EddsaAttrs::new(Cv25519, None))),
                    (Authentication, Rsa(RsaAttrs { len_n: 2048, len_e: 17, import_format: 0 })),
                    (Authentication, Rsa(RsaAttrs { len_n: 3072, len_e: 17, import_format: 0 })),
                    (Authentication, Rsa(RsaAttrs { len_n: 4096, len_e: 17, import_format: 0 })),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP384r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(NistP521r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP256r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP384r1, None))),
                    (Authentication, Ecdsa(EcdsaAttrs::new(BrainpoolP512r1, None))),
                    (Authentication, Eddsa(EddsaAttrs::new(Ed25519, None))),
                    (Authentication, Eddsa(EddsaAttrs::new(Cv25519, None))),
                    (Attestation, Rsa(RsaAttrs { len_n: 2048, len_e: 17, import_format: 0 })),
                    (Attestation, Rsa(RsaAttrs { len_n: 3072, len_e: 17, import_format: 0 })),
                    (Attestation, Rsa(RsaAttrs { len_n: 4096, len_e: 17, import_format: 0 })),
                    (Attestation, Ecdsa(EcdsaAttrs::new(NistP256r1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(NistP384r1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(NistP521r1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(Secp256k1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(BrainpoolP256r1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(BrainpoolP384r1, None))),
                    (Attestation, Ecdsa(EcdsaAttrs::new(BrainpoolP512r1, None))),
                    (Attestation, Eddsa(EddsaAttrs::new(Ed25519, None))),
                    (Attestation, Eddsa(EddsaAttrs::new(Cv25519, None)))
                ]
            )
        );
    }
}
