use log::debug;
use openmls_traits::crypto::OpenMlsCrypto;
use tls_codec::Deserialize;

use crate::ciphersuite::signable::Verifiable;
use crate::extensions::ExtensionType;
use crate::group::{core_group::*, *};
use crate::key_packages::*;
use crate::messages::*;
use crate::schedule::*;
use crate::treesync::node::Node;

impl CoreGroup {
    // Join a group from a welcome message
    pub fn new_from_welcome(
        welcome: Welcome,
        nodes_option: Option<Vec<Option<Node>>>,
        key_package_bundle: KeyPackageBundle,
        backend: &impl OpenMlsCryptoProvider,
    ) -> Result<Self, WelcomeError> {
        log::debug!("CoreGroup::new_from_welcome_internal");
        let mls_version = *welcome.version();
        if !Config::supported_versions().contains(&mls_version) {
            return Err(WelcomeError::UnsupportedMlsVersion);
        }

        let ciphersuite_name = welcome.ciphersuite();
        let ciphersuite = Config::ciphersuite(ciphersuite_name)?;

        // Find key_package in welcome secrets
        let egs = if let Some(egs) = Self::find_key_package_from_welcome_secrets(
            key_package_bundle.key_package(),
            welcome.secrets(),
            backend,
        )? {
            egs
        } else {
            return Err(WelcomeError::JoinerSecretNotFound);
        };
        if ciphersuite_name != key_package_bundle.key_package().ciphersuite_name() {
            let e = WelcomeError::CiphersuiteMismatch;
            debug!("new_from_welcome {:?}", e);
            return Err(e);
        }

        let group_secrets_bytes = backend.crypto().hpke_open(
            ciphersuite.hpke_config(),
            &egs.encrypted_group_secrets,
            key_package_bundle.private_key().as_slice(),
            &[],
            &[],
        )?;
        let group_secrets = GroupSecrets::tls_deserialize(&mut group_secrets_bytes.as_slice())?
            .config(ciphersuite, mls_version);
        let joiner_secret = group_secrets.joiner_secret;

        // Prepare the PskSecret
        let psk_secret = PskSecret::new(ciphersuite, backend, group_secrets.psks.psks())?;

        // Create key schedule
        let mut key_schedule = KeySchedule::init(ciphersuite, backend, joiner_secret, psk_secret)?;

        // Derive welcome key & nonce from the key schedule
        let (welcome_key, welcome_nonce) = key_schedule
            .welcome(backend)?
            .derive_welcome_key_nonce(backend)?;

        let group_info_bytes = welcome_key
            .aead_open(backend, welcome.encrypted_group_info(), &[], &welcome_nonce)
            .map_err(|_| WelcomeError::GroupInfoDecryptionFailure)?;
        let group_info = GroupInfo::tls_deserialize(&mut group_info_bytes.as_slice())?;

        // Make sure that we can support the required capabilities in the group info.
        let group_context_extensions = group_info.group_context_extensions();
        let required_capabilities = group_context_extensions
            .iter()
            .find(|&extension| extension.extension_type() == ExtensionType::RequiredCapabilities);
        if let Some(required_capabilities) = required_capabilities {
            let required_capabilities =
                required_capabilities.as_required_capabilities_extension()?;
            check_required_capabilities_support(required_capabilities)?;
            // Also check that our key package actually supports the extensions.
            // Per spec the sender must have checked this. But you never know.
            key_package_bundle
                .key_package()
                .check_extension_support(required_capabilities.extensions())?
        }

        let path_secret_option = group_secrets.path_secret;

        // Build the ratchet tree

        // Set nodes either from the extension or from the `nodes_option`.
        // If we got a ratchet tree extension in the welcome, we enable it for
        // this group. Note that this is not strictly necessary. But there's
        // currently no other mechanism to enable the extension.
        let (nodes, enable_ratchet_tree_extension) =
            match try_nodes_from_extensions(group_info.other_extensions(), backend.crypto())? {
                Some(nodes) => (nodes, true),
                None => match nodes_option {
                    Some(n) => (n, false),
                    None => return Err(WelcomeError::MissingRatchetTree),
                },
            };

        // Commit secret is ignored when joining a group, since we already have
        // the joiner_secret.
        let (tree, _commit_secret_option) = TreeSync::from_nodes_with_secrets(
            backend,
            ciphersuite,
            &nodes,
            group_info.signer(),
            path_secret_option,
            key_package_bundle,
        )?;

        let signer_key_package = tree
            .leaf_from_id(group_info.signer())?
            .ok_or(WelcomeError::UnknownSender)?
            .key_package();

        // Verify GroupInfo signature
        group_info
            .verify_no_out(backend, signer_key_package.credential())
            .map_err(|_| WelcomeError::InvalidGroupInfoSignature)?;

        // Compute state
        let group_context = GroupContext::new(
            group_info.group_id().clone(),
            group_info.epoch(),
            tree.tree_hash().to_vec(),
            group_info.confirmed_transcript_hash().to_vec(),
            group_context_extensions,
        )?;

        let serialized_group_context = group_context.tls_serialize_detached()?;
        // TODO #751: Implement PSK
        key_schedule.add_context(backend, &serialized_group_context)?;
        let epoch_secrets = key_schedule.epoch_secrets(backend)?;

        let (group_epoch_secrets, message_secrets) =
            epoch_secrets.split_secrets(serialized_group_context, tree.leaf_count()?);

        let confirmation_tag = message_secrets
            .confirmation_key()
            .tag(backend, group_context.confirmed_transcript_hash())?;
        let interim_transcript_hash = update_interim_transcript_hash(
            ciphersuite,
            backend,
            &MlsPlaintextCommitAuthData::from(&confirmation_tag),
            group_context.confirmed_transcript_hash(),
        )?;

        // Verify confirmation tag
        if &confirmation_tag != group_info.confirmation_tag() {
            log::error!("Confirmation tag mismatch");
            log_crypto!(trace, "  Got:      {:x?}", confirmation_tag);
            log_crypto!(trace, "  Expected: {:x?}", group_info.confirmation_tag());
            Err(WelcomeError::ConfirmationTagMismatch)
        } else {
            let message_secrets_store = MessageSecretsStore::new_with_secret(0, message_secrets);

            Ok(CoreGroup {
                ciphersuite,
                group_context,
                group_epoch_secrets,
                tree,
                interim_transcript_hash,
                use_ratchet_tree_extension: enable_ratchet_tree_extension,
                mls_version,
                message_secrets_store,
            })
        }
    }

    // Helper functions

    pub(crate) fn find_key_package_from_welcome_secrets(
        key_package: &KeyPackage,
        welcome_secrets: &[EncryptedGroupSecrets],
        backend: &impl OpenMlsCryptoProvider,
    ) -> Result<Option<EncryptedGroupSecrets>, KeyPackageError> {
        for egs in welcome_secrets {
            if key_package.hash_ref(backend.crypto())?.as_slice() == egs.new_member.as_slice() {
                return Ok(Some(egs.clone()));
            }
        }
        Ok(None)
    }
}
