//!
//! This example showcases the process of integrating with the
//! [GitLab OpenID Connect](https://docs.gitlab.com/ee/integration/openid_connect_provider.html)
//! provider.
//!
//! Before running it, you'll need to generate your own
//! [GitLab Application](https://docs.gitlab.com/ee/integration/oauth_provider.html).
//! The application needs `openid`, `profile` and `email` permission.
//!
//! In order to run the example call:
//!
//! ```sh
//! GITLAB_CLIENT_ID=xxx GITLAB_CLIENT_SECRET=yyy cargo run --example gitlab
//! ```
//!
//! ...and follow the instructions.
//!

use std::env;
use std::io::{BufRead, BufReader, Write};
use std::net::TcpListener;
use std::process::exit;

use url::Url;

use openidconnect::core::{
    CoreClient, CoreGenderClaim, CoreIdTokenClaims, CoreIdTokenVerifier, CoreProviderMetadata,
    CoreResponseType,
};

use openidconnect::reqwest::http_client;
use openidconnect::{AdditionalClaims, UserInfoClaims};
use openidconnect::{
    AuthenticationFlow, AuthorizationCode, ClientId, ClientSecret, CsrfToken, IssuerUrl, Nonce,
    OAuth2TokenResponse, RedirectUrl, Scope,
};

use serde::{Deserialize, Serialize};

#[derive(Debug, Deserialize, Serialize)]
struct GitLabClaims {
    // Deprecated and thus optional as it might be removed in the futre
    sub_legacy: Option<String>,
    groups: Vec<String>,
}
impl AdditionalClaims for GitLabClaims {}

fn handle_error<T: std::error::Error>(fail: &T, msg: &'static str) {
    let mut err_msg = format!("ERROR: {}", msg);
    let mut cur_fail: Option<&dyn std::error::Error> = Some(fail);
    while let Some(cause) = cur_fail {
        err_msg += &format!("\n    caused by: {}", cause);
        cur_fail = cause.source();
    }
    println!("{}", err_msg);
    exit(1);
}

fn main() {
    env_logger::init();

    let gitlab_client_id = ClientId::new(
        env::var("GITLAB_CLIENT_ID").expect("Missing the GITLAB_CLIENT_ID environment variable."),
    );
    let gitlab_client_secret = ClientSecret::new(
        env::var("GITLAB_CLIENT_SECRET")
            .expect("Missing the GITLAB_CLIENT_SECRET environment variable."),
    );
    let issuer_url = IssuerUrl::new("https://gitlab.com".to_string()).expect("Invalid issuer URL");

    // Fetch GitLab's OpenID Connect discovery document.
    //
    let provider_metadata = CoreProviderMetadata::discover(&issuer_url, http_client)
        .unwrap_or_else(|err| {
            handle_error(&err, "Failed to discover OpenID Provider");
            unreachable!();
        });

    // Set up the config for the GitLab OAuth2 process.
    let client = CoreClient::from_provider_metadata(
        provider_metadata,
        gitlab_client_id,
        Some(gitlab_client_secret),
    )
    // This example will be running its own server at localhost:8080.
    // See below for the server implementation.
    .set_redirect_uri(
        RedirectUrl::new("http://localhost:8080".to_string()).expect("Invalid redirect URL"),
    );

    // Generate the authorization URL to which we'll redirect the user.
    let (authorize_url, csrf_state, nonce) = client
        .authorize_url(
            AuthenticationFlow::<CoreResponseType>::AuthorizationCode,
            CsrfToken::new_random,
            Nonce::new_random,
        )
        // This example is requesting access to the the user's profile including email.
        .add_scope(Scope::new("email".to_string()))
        .add_scope(Scope::new("profile".to_string()))
        .url();

    println!(
        "Open this URL in your browser:\n{}\n",
        authorize_url.to_string()
    );

    // A very naive implementation of the redirect server.
    let listener = TcpListener::bind("127.0.0.1:8080").unwrap();
    for stream in listener.incoming() {
        if let Ok(mut stream) = stream {
            let code;
            let state;
            {
                let mut reader = BufReader::new(&stream);

                let mut request_line = String::new();
                reader.read_line(&mut request_line).unwrap();

                let redirect_url = request_line.split_whitespace().nth(1).unwrap();
                let url = Url::parse(&("http://localhost".to_string() + redirect_url)).unwrap();

                let code_pair = url
                    .query_pairs()
                    .find(|pair| {
                        let &(ref key, _) = pair;
                        key == "code"
                    })
                    .unwrap();

                let (_, value) = code_pair;
                code = AuthorizationCode::new(value.into_owned());

                let state_pair = url
                    .query_pairs()
                    .find(|pair| {
                        let &(ref key, _) = pair;
                        key == "state"
                    })
                    .unwrap();

                let (_, value) = state_pair;
                state = CsrfToken::new(value.into_owned());
            }

            let message = "Go back to your terminal :)";
            let response = format!(
                "HTTP/1.1 200 OK\r\ncontent-length: {}\r\n\r\n{}",
                message.len(),
                message
            );
            stream.write_all(response.as_bytes()).unwrap();

            println!("GitLab returned the following code:\n{}\n", code.secret());
            println!(
                "GitLab returned the following state:\n{} (expected `{}`)\n",
                state.secret(),
                csrf_state.secret()
            );

            // Exchange the code with a token.
            let token_response = client
                .exchange_code(code)
                .request(http_client)
                .unwrap_or_else(|err| {
                    handle_error(&err, "Failed to contact token endpoint");
                    unreachable!();
                });

            println!(
                "GitLab returned access token:\n{}\n",
                token_response.access_token().secret()
            );
            println!("GitLab returned scopes: {:?}", token_response.scopes());

            let id_token_verifier: CoreIdTokenVerifier = client.id_token_verifier();
            let id_token_claims: &CoreIdTokenClaims = token_response
                .extra_fields()
                .id_token()
                .expect("Server did not return an ID token")
                .claims(&id_token_verifier, &nonce)
                .unwrap_or_else(|err| {
                    handle_error(&err, "Failed to verify ID token");
                    unreachable!();
                });
            println!("GitLab returned ID token: {:?}\n", id_token_claims);

            let userinfo_claims: UserInfoClaims<GitLabClaims, CoreGenderClaim> = client
                .user_info(token_response.access_token().to_owned(), None)
                .unwrap_or_else(|err| {
                    handle_error(&err, "No user info endpoint");
                    unreachable!();
                })
                .request(http_client)
                .unwrap_or_else(|err| {
                    handle_error(&err, "Failed requesting user info");
                    unreachable!();
                });
            println!("GitLab returned UserInfo: {:?}", userinfo_claims);

            // The server will terminate itself
            break;
        }
    }
}
