use crate::simulations::{
    model::{nodes::NodeNames, params::GenParams, MetaModel, Model, ModelSet, ModelTrait},
    scenario::Scenarios,
};

#[test]
fn test_model() {
    let mut m = Model::new("aaaa");
    m.input(
        NodeNames::LossEventFrequency,
        GenParams::PertParams {
            min: 20.0,
            mean: 100.0,
            max: 900.0,
        },
    )
    .unwrap();
    m.input(
        NodeNames::PrimaryLoss,
        GenParams::PertParams {
            min: 3000000.0,
            mean: 3500000.0,
            max: 5000000.0,
        },
    )
    .unwrap();
    m.input(
        NodeNames::SecondaryLoss,
        GenParams::ConstParams { val: 3500000.0 },
    )
    .unwrap();
    m.simulate(100).unwrap();
    assert!(true);
}

#[test]
fn test_scenario() {
    let data = std::fs::read_to_string("data/input.json").expect("Unable to read file");
    let json: serde_json::Value = serde_json::from_str(&data).expect("JSON was not well-formatted");
    let scenario = json.as_object().unwrap().get("scenario").unwrap();
    match Scenarios::from_json(scenario) {
        Ok(mut s) => {
            s.simulate(3).unwrap();
            assert!(true);
        }
        Err(_) => {
            assert!(false);
        }
    }
}

#[test]
fn test_metamodel() {
    let mut m1 = Model::new("m1");
    m1.input(
        NodeNames::LossEventFrequency,
        GenParams::PertParams {
            min: 20.0,
            mean: 100.0,
            max: 900.0,
        },
    )
    .unwrap();
    m1.input(
        NodeNames::PrimaryLoss,
        GenParams::PertParams {
            min: 3000000.0,
            mean: 3500000.0,
            max: 5000000.0,
        },
    )
    .unwrap();
    m1.input(
        NodeNames::SecondaryLoss,
        GenParams::ConstParams { val: 3500000.0 },
    )
    .unwrap();
    let mut m2 = Model::new("m2");
    m2.input(
        NodeNames::LossEventFrequency,
        GenParams::PertParams {
            min: 20.0,
            mean: 100.0,
            max: 900.0,
        },
    )
    .unwrap();
    m2.input(
        NodeNames::PrimaryLoss,
        GenParams::PertParams {
            min: 3000000.0,
            mean: 3500000.0,
            max: 5000000.0,
        },
    )
    .unwrap();
    m2.input(
        NodeNames::SecondaryLoss,
        GenParams::ConstParams { val: 3500000.0 },
    )
    .unwrap();
    let mut m3 = Model::new("m3");
    m3.input(
        NodeNames::LossEventFrequency,
        GenParams::PertParams {
            min: 20.0,
            mean: 100.0,
            max: 900.0,
        },
    )
    .unwrap();
    m3.input(
        NodeNames::PrimaryLoss,
        GenParams::PertParams {
            min: 3000000.0,
            mean: 3500000.0,
            max: 5000000.0,
        },
    )
    .unwrap();
    m3.input(
        NodeNames::SecondaryLoss,
        GenParams::ConstParams { val: 3500000.0 },
    )
    .unwrap();
    let mm1 = MetaModel::new("mm1", vec![Box::new(m1.clone()), Box::new(m2.clone())]);
    let mut mm2 = MetaModel::new(
        "mm1",
        vec![
            Box::new(m1.clone()),
            Box::new(m3.clone()),
            Box::new(mm1.clone()),
        ],
    );
    mm2.simulate(100).unwrap();
    assert!(true);
}

#[test]
fn test_metamodel_from_json() {
    let json: serde_json::Value = serde_json::from_str(
        r###"
{
  "currency":"USD",
  "simulation_count": 1000,
  "models":[
    {
      "settings": {
        "Primary Loss": {
          "min": 3000000,
          "mean": 3500000,
          "max": 5000000
        },
        "Secondary Loss":{
          "val": 3500000
        },
        "Loss Event Frequency": {
          "min": 20,
          "mean": 100,
          "max": 900
        }
      },
      "name": "Sample Model 1",
      "type": "model"
    },
    {
      "settings": {
        "Loss Magnitude": {
          "min": 2000000000,
          "mean": 3000000000,
          "max": 5000000000
        },
        "Loss Event Frequency": {
          "mean": 0.3,
          "dev": 0.1
        }
      },
      "name": "Sample Model 2",
      "type": "model"
    },
    {
      "name": "Meta Model 1",
      "type": "meta model",
      "models": ["Sample Model 1", "Sample Model 2"]
    }
  ]
}
"###,
    )
    .expect("JSON was not well-formatted");
    let mut mm = ModelSet::from_json(&json).unwrap();
    mm.simulate_all().unwrap();
    println!("{}", serde_json::to_string_pretty(&mm).unwrap());
    assert!(true);
}
