pub(crate) mod calculations;
pub(crate) mod nodes;
pub(crate) mod params;
pub(crate) mod tree;

use self::{
    calculations::{std_deviation, std_max, std_mean, std_min},
    nodes::NodeNames,
    params::GenParams,
    tree::Tree,
};
use crate::{error::Error, Result};
use serde::{Deserialize, Serialize};
use serde_json::Value;

#[derive(Serialize, Deserialize, Debug, Clone)]
pub struct Model {
    pub name: String,
    tree: Tree,
    values: Option<Vec<f32>>,
    mean: Option<f32>,
    dev: Option<f32>,
    min: Option<f32>,
    max: Option<f32>,
}

impl Model {
    pub fn new(name: &str) -> Model {
        Model {
            name: name.to_string(),
            tree: Tree::new(),
            values: None,
            mean: None,
            dev: None,
            min: None,
            max: None,
        }
    }

    pub fn from_json(v: &Value) -> Result<Model> {
        if v["type"].as_str().is_none() {
            return Err(Error::TypeFieldNotPresent);
        }
        if v["type"].as_str().unwrap() != "model" {
            return Err(Error::InvalidTypeField(
                v["type"].as_str().unwrap().to_string(),
            ));
        }

        let mut m = Model::new(v["name"].as_str().ok_or(Error::NameFieldNotPresent)?);
        let settings = v["settings"]
            .as_object()
            .ok_or(Error::SettingsFieldNotPresent)?;
        for s in settings {
            m.input(NodeNames::from(s.0)?, GenParams::from_json(s.1)?)?;
        }
        Ok(m)
    }

    pub fn input(&mut self, input_name: NodeNames, gp: GenParams) -> Result<()> {
        match self.tree.search(&input_name) {
            Ok(e) => e.set_param(gp),
            Err(e) => Err(e),
        }
    }

    #[allow(dead_code)]
    pub fn get_mean(&mut self, input_name: NodeNames) -> Result<f32> {
        match self.tree.search(&input_name) {
            Ok(e) => Ok(e.mean.unwrap()),
            Err(e) => Err(e),
        }
    }

    pub fn get(&mut self, name: &NodeNames) -> Result<(f32, f32, f32, f32)> {
        self.tree.get(&name)
    }
}

#[derive(Serialize, Clone)]
pub struct ModelSet {
    // currency: String,
    simulation_count: u32,
    models: Vec<Model>,
    metamodels: Vec<MetaModel>,
}

impl ModelSet {
    pub fn new() -> ModelSet {
        ModelSet {
            models: vec![],
            metamodels: vec![],
            // currency: "USD".to_string(),
            simulation_count: 10000,
        }
    }

    pub fn from_json(v: &Value) -> Result<ModelSet> {
        let mut ms = ModelSet::new();
        ms.simulation_count = v["simulation_count"]
            .as_u64()
            .ok_or(Error::InvalidSimulationCount)? as u32;
        // ms.currency = v["currency"]
        //     .as_str()
        //     .ok_or(Error::InvalidCurrency)?
        //     .to_string();
        let model_list = v["models"].as_array().ok_or(Error::RequiredModelArrays)?;
        for m in model_list {
            let mtype = m["type"].as_str().ok_or_else(|| {
                Error::InvalidTypeField(m["type"].as_str().unwrap_or_default().to_string())
            })?;
            match mtype {
                "model" => ms.models.push(Model::from_json(m)?),
                "meta model" => {
                    let mm = MetaModel::from_json(m, &mut ms)?;
                    ms.metamodels.push(mm);
                }
                _ => return Err(Error::InvalidTypeField(mtype.to_string())),
            }
        }
        Ok(ms)
    }

    pub fn get(&self, mname: &str) -> Result<Box<dyn ModelTrait>> {
        for m in &self.models {
            if m.name == mname {
                return Ok(Box::new(m.clone()));
            }
        }
        for m in &self.metamodels {
            if m.name == mname {
                return Ok(Box::new(m.clone()));
            }
        }
        Err(Error::UndefinedModelReference(mname.to_string()))
    }

    pub fn simulate_all(&mut self) -> Result<&ModelSet> {
        for m in &mut self.models {
            m.simulate(self.simulation_count)?;
        }
        for m in &mut self.metamodels {
            m.simulate(self.simulation_count)?;
        }
        Ok(self)
    }
}

pub trait ModelClone {
    fn clone_box(&self) -> Box<dyn ModelTrait>;
}

pub trait ModelTrait: ModelClone {
    fn get_name(&'_ self) -> Result<String>;
    fn simulate(&'_ mut self, n: u32) -> Result<(f32, f32, f32, f32)>;
    fn get_result(&'_ mut self) -> Result<(f32, f32, f32, f32)>;
    fn get_full_result(&'_ mut self) -> Result<(f32, f32, f32, f32, &Vec<f32>)>;
}

impl<T> ModelClone for T
where
    T: 'static + ModelTrait + Clone,
{
    fn clone_box(&self) -> Box<dyn ModelTrait> {
        Box::new(self.clone())
    }
}

impl Clone for Box<dyn ModelTrait> {
    fn clone(&self) -> Box<dyn ModelTrait> {
        self.clone_box()
    }
}

impl ModelTrait for Model {
    fn get_name(&'_ self) -> Result<String> {
        Ok(self.name.clone())
    }

    fn simulate(&'_ mut self, n: u32) -> Result<(f32, f32, f32, f32)> {
        self.tree.calc(n)?;
        let s = self.get_full_result()?;
        Ok((s.0, s.1, s.2, s.3))
    }

    fn get_result(&'_ mut self) -> Result<(f32, f32, f32, f32)> {
        self.tree.get(&NodeNames::Risk)
    }

    fn get_full_result(&'_ mut self) -> Result<(f32, f32, f32, f32, &Vec<f32>)> {
        match self.tree.search(&NodeNames::Risk) {
            Ok(e) => {
                self.min = e.min;
                self.mean = e.mean;
                self.max = e.max;
                self.dev = e.dev;
                self.values = Some(e.values.clone());
                Ok((
                    e.min.unwrap(),
                    e.mean.unwrap(),
                    e.max.unwrap(),
                    e.dev.unwrap(),
                    &e.values,
                ))
            }
            Err(e) => Err(e),
        }
    }
}

#[derive(Serialize, Clone)]
pub struct MetaModel {
    pub name: String,
    models_names: Vec<String>,
    #[serde(skip_serializing)]
    models: Vec<Box<dyn ModelTrait>>,
    values: Option<Vec<f32>>,
    mean: Option<f32>,
    dev: Option<f32>,
    min: Option<f32>,
    max: Option<f32>,
}

impl MetaModel {
    pub fn new(name: &str, models: Vec<Box<dyn ModelTrait>>) -> MetaModel {
        let mut mmn: Vec<String> = vec![];
        for m in &models {
            mmn.push(m.get_name().unwrap());
        }
        MetaModel {
            name: name.to_string(),
            models_names: mmn,
            models,
            min: None,
            max: None,
            mean: None,
            dev: None,
            values: None,
        }
    }

    pub fn from_json(v: &Value, ms: &mut ModelSet) -> Result<MetaModel> {
        let mname = v["name"].as_str().ok_or_else(|| {
            Error::InvalidNameField(v["name"].as_str().unwrap_or_default().to_string())
        })?;
        let mut vv: Vec<Box<dyn ModelTrait>> = vec![];
        let content = v["models"].as_array().ok_or(Error::IncorrectModelSet)?;
        for c in content {
            vv.push(ms.get(c.as_str().ok_or_else(|| {
                Error::IncorrectModelRef(c.as_str().unwrap_or_default().to_string())
            })?)?)
        }
        Ok(MetaModel::new(mname, vv))
    }
}

impl ModelTrait for MetaModel {
    fn get_name(&'_ self) -> Result<String> {
        Ok(self.name.clone())
    }

    fn simulate(&'_ mut self, n: u32) -> Result<(f32, f32, f32, f32)> {
        for m in &mut self.models {
            m.simulate(n)?;
            let mr = m.get_full_result()?;
            match &mut self.values {
                None => {
                    self.values = Some(mr.4.clone());
                }
                Some(rrt) => {
                    for i in 0..rrt.len() {
                        rrt[i] += mr.4[i];
                    }
                }
            }
        }
        self.min = Some(std_min(&self.values.as_ref().unwrap()).unwrap());
        self.max = Some(std_max(&self.values.as_ref().unwrap()).unwrap());
        self.mean = Some(std_mean(&self.values.as_ref().unwrap()).unwrap());
        self.dev =
            Some(std_deviation(&self.values.as_ref().unwrap(), &self.mean.unwrap()).unwrap());
        Ok((
            self.min.unwrap(),
            self.mean.unwrap(),
            self.max.unwrap(),
            self.dev.unwrap(),
        ))
    }
    fn get_result(&'_ mut self) -> Result<(f32, f32, f32, f32)> {
        Ok((
            self.min.unwrap(),
            self.mean.unwrap(),
            self.max.unwrap(),
            self.dev.unwrap(),
        ))
    }
    fn get_full_result(&'_ mut self) -> Result<(f32, f32, f32, f32, &Vec<f32>)> {
        Ok((
            self.min.unwrap(),
            self.mean.unwrap(),
            self.max.unwrap(),
            self.dev.unwrap(),
            &self.values.as_ref().unwrap(),
        ))
    }
}

pub fn simulate_model(input_json: &Value) -> Result<Value> {
    let model = input_json
        .as_object()
        .ok_or(Error::InvalidInput)?
        .get("model")
        .ok_or(Error::ModelNotFound)?;

    match ModelSet::from_json(&model) {
        Ok(mut s) => Ok(serde_json::to_value(&s.simulate_all()?)?),
        Err(e) => Err(e),
    }
}
