use crate::{error::Error, Result};

pub fn mul_vecs(v: &mut Vec<f32>, p1: &[f32], p2: &[f32]) -> Result<()> {
    if p1.len() != p2.len() {
        return Err(Error::DifferentSizeVectorsMultiplication {
            v1: p1.len(),
            v2: p2.len(),
        });
    }
    v.clear();
    for i in 0..p1.len() {
        v.push(p1[i] * p2[i]);
    }
    Ok(())
}

pub fn add_vecs(v: &mut Vec<f32>, p1: &[f32], p2: &[f32]) -> Result<()> {
    if p1.len() != p2.len() {
        return Err(Error::DifferentSizeVectorsAddition {
            v1: p1.len(),
            v2: p2.len(),
        });
    }
    v.clear();
    for i in 0..p1.len() {
        v.push(p1[i] + p2[i]);
    }
    Ok(())
}

pub fn step_vecs(v: &mut Vec<f32>, p1: &[f32], p2: &[f32]) -> Result<()> {
    if p1.len() != p2.len() {
        return Err(Error::DifferentSizeVectors {
            v1: p1.len(),
            v2: p2.len(),
        });
    }
    v.clear();
    let mut val: f32 = 0.0;
    for i in 0..p1.len() {
        val += if p1[i] > p2[i] { 1.0 } else { 0.0 };
    }
    val /= p1.len() as f32;
    for _i in 0..p1.len() {
        v.push(val);
    }
    Ok(())
}

pub fn std_mean(data: &[f32]) -> Result<f32> {
    let sum = data.iter().sum::<f32>() as f32;
    let count = data.len();
    Ok(sum / count as f32)
}

pub fn std_min(data: &[f32]) -> Result<f32> {
    let mut res: Option<f32> = None;
    for i in data {
        match res {
            Some(s) => {
                if &s > i {
                    res = Some(*i)
                }
            }
            None => {
                res = Some(*i);
            }
        }
    }
    Ok(res.unwrap())
}

pub fn std_max(data: &[f32]) -> Result<f32> {
    let mut res: Option<f32> = None;
    for i in data {
        match res {
            Some(s) => {
                if &s < i {
                    res = Some(*i)
                }
            }
            None => {
                res = Some(*i);
            }
        }
    }
    Ok(res.unwrap())
}

pub fn std_deviation(data: &[f32], mean: &f32) -> Result<f32> {
    let variance = data
        .iter()
        .map(|value| {
            let diff = mean - (*value as f32);
            diff * diff
        })
        .sum::<f32>()
        / data.len() as f32;
    Ok(variance.sqrt())
}
