/// OpenFAIR (Factor Analysis of Information Risk)
/// The Open Group has created standards and best practices in the area of
/// Security and Risk for 20+ years. The Open Group is a consensus-based
/// and member-driven organization.
///  Open FAIR provides a risk analysis methodology that describes the how and
/// why of risk analysis. Open FAIR enhances higher level risk management
/// frameworks from ISO, NIST, and other organizations by providing a means to
/// more effectively analyse and measure risk. It improves consistency in
/// undertaking analyses. The Open FAIR taxonomy and method provide the basis for
/// meaningful metrics. It is flexible and can be used at different levels of
/// abstraction to match the need, the available resources, and available data.
/// The Open FAIR risk analysis method provides a more rigorous approach that
/// helps to reduce gaps and analyst bias. It improves the ability to defend
/// conclusions and recommendations.
#[cfg(test)]
mod tests;

pub mod error;
pub type Result<T> = std::result::Result<T, error::Error>;

mod simulations;
use self::simulations::{
    charts::{model_set_to_d3js, scenario_to_d3js},
    model::simulate_model,
    scenario::simulate_scenario,
};

use serde_json::{json, Value};

pub fn simulate(input_json: &Value) -> Result<Value> {
    //! Simulate Scenarios and Model using the OpenFAIR
    //! framework. Use the returned data with any charting tool
    //!
    //! # Example usage
    //!```rust
    //! use openfair::{simulate, Result};
    //! use std::fs::read_to_string;
    //!
    //! fn main() -> Result<()> {
    //!
    //!    let input = read_to_string("data/input.json")?;
    //!    let result = simulate(&serde_json::from_str(&input)?)?;
    //!
    //!    println!("{:#?}", result);
    //!
    //!    Ok(())
    //! }
    //! ```
    Ok(json!({
        "scenario": simulate_scenario(input_json)?,
        "model": simulate_model(input_json)?,
    }))
}

pub fn simulate_with_charts(input_json: &Value) -> Result<Value> {
    //! Simulate Scenarios and Model using the OpenFAIR
    //! framework, and return generated chart data that can be used with d3js
    //!
    //! # Example usage
    //!```rust
    //! use openfair::{simulate_with_charts, Result};
    //! use std::fs::read_to_string;
    //!
    //! fn main() -> Result<()> {
    //!
    //!    let input = read_to_string("data/input.json")?;
    //!    let result = simulate_with_charts(&serde_json::from_str(&input)?)?;
    //!
    //!    println!("{:#?}", result);
    //!
    //!    Ok(())
    //! }
    //! ```
    let scenario_result = serde_json::to_string(&simulate_scenario(input_json)?)?;
    let model_result = serde_json::to_string(&simulate_model(input_json)?)?;

    Ok(json!({
        "scenario": scenario_to_d3js(&scenario_result, vec!["risk", "v", "lef"])?,
        "model": model_set_to_d3js(&model_result)?,
    }))
}
