use crate::simulations::model::nodes::NodeNames;

#[derive(Debug, thiserror::Error)]
pub enum Error {
    // Model Errors
    #[error("Incorrect model set")]
    IncorrectModelSet,
    #[error("Incorrect model ref: {0}")]
    IncorrectModelRef(String),
    #[error("Reference to undefined model: {0}")]
    UndefinedModelReference(String),
    #[error("Incorrect model: {n:?}")]
    IncorrectModel { n: NodeNames },
    #[error("Invalid model: Node {n:?} has {c} children")]
    InvalidModel { n: NodeNames, c: usize },
    #[error("Invalid input field: {e}")]
    InvalidInputField { e: String },
    #[error("Unsupported Distribution Setting: {e:?}")]
    UnsupportedDistributionSetting { e: serde_json::Value },
    #[error("Multiplication of vectors with different sizes {v1} {v2}")]
    DifferentSizeVectorsMultiplication { v1: usize, v2: usize },
    #[error("Addition of vectors with different sizes {v1} {v2}")]
    DifferentSizeVectorsAddition { v1: usize, v2: usize },
    #[error("Vectors of different sizes {v1} {v2}")]
    DifferentSizeVectors { v1: usize, v2: usize },
    #[error("Simulation count not found")]
    InvalidSimulationCount,
    #[error("Currency not found")]
    InvalidCurrency,
    #[error("Required Arrays of Models")]
    RequiredModelArrays,
    #[error("Required Field 'name' is not present in Model input")]
    NameFieldNotPresent,
    #[error("Required Field 'settings' is not present in Model input")]
    SettingsFieldNotPresent,
    #[error("Required Field 'type' is not present in Model input")]
    TypeFieldNotPresent,
    #[error("'type=model' not `{0}`")]
    InvalidTypeField(String),
    #[error("Invalid name: {0}")]
    InvalidNameField(String),

    // Scenario Errors
    #[error("Threats field not found")]
    ThreatsFieldNotFound,
    #[error("Threat not found in threats: `{0}`")]
    ThreatNotFound(String),
    #[error("Threat Id not found: `{0}`")]
    InvalidThreatId(String),
    #[error("Params field not found")]
    ParamsFieldNotFound,
    #[error("Expected Frequency field not found")]
    ExpectedFrequencyFieldNotFound,
    #[error("Expected Frequency should be a string for threat: {0}")]
    InvalidExpectedFrequency(String),
    #[error("Expected Frequency must contain `min`, `mean`, `max`")]
    ExpectedFrequencyMinMeanMaxNotFound,
    #[error("Capability field not found")]
    CapabilityFieldNotFound,
    #[error("Capability should be a string for threat: {0}")]
    InvalidCapability(String),
    #[error("Capability must contain `min`, `mean`, `max`")]
    CapabilityMinMeanMaxNotFound,
    #[error("Loss Magnitude field not found")]
    LossMagnitudeFieldNotFound,
    #[error("Loss Magnitude should be a string for threat: {0}")]
    InvalidLossMagnitude(String),
    #[error("Loss Magnitude must contain `min`, `mean`, `max`")]
    LossMagnitudeMinMeanMaxNotFound,
    #[error("Difficulty field not found")]
    DifficultyFieldNotFound,
    #[error("Difficulty should be a string for control: {0}")]
    InvalidDifficulty(String),
    #[error("Difficulty must contain `min`, `mean`, `max`")]
    DifficultyMinMeanMaxNotFound,
    #[error("Controls field not found")]
    ControlsFieldNotFound,
    #[error("Control Ids field not found")]
    ControlIdsNotFound,
    #[error("Control not found in controls: `{0}`")]
    InvalidControl(String),
    #[error("Invalid Control Id: `{0}`")]
    InvalidControlId(String),
    #[error("Scenarios field not found")]
    ScenariosFieldNotFound,

    // Charting errors
    #[error("Invalid Control in Chart: {0}")]
    InvalidChartControl(String),
    #[error("Invalid Mean in Chart: {0}")]
    InvalidChartMean(f64),
    #[error("Simulation Result not found")]
    SimulationResultNotFound,

    // Input read errors
    #[error("Input Json not valid/error")]
    InvalidInput,
    #[error("Scenario field not found")]
    ScenarioNotFound,
    #[error("Model field not found")]
    ModelNotFound,

    // Generic Errors
    #[error("Serde json error")]
    Json(#[from] serde_json::Error),
    #[error("{0}")]
    Io(#[from] std::io::Error),
    #[error("{0}")]
    Generic(String),
    #[error("Unknown Error")]
    Unknown,
}
