use crate::core::{
    attribute::TypedAttribute,
    channel_list::{ChannelListRef, ChannelListRefMut},
    cppstd::CppString,
    error::Error,
    preview_image::{PreviewImage, PreviewImageRef},
    refptr::{OpaquePtr, Ref, RefMut},
    tile_description::TileDescription,
    Compression, LineOrder,
};

use openexr_sys as sys;

use imath_traits::{Bound2, Vec2};

type Result<T, E = Error> = std::result::Result<T, E>;

use std::alloc::{GlobalAlloc, Layout, System};
use std::ffi::{CStr, CString};

/// The `Header` represents the header in the OpenEXR file and is typically read
/// or written when the file is first opened.
///
/// There will be one header for each *part* of an OpenEXR file and it describes
/// the structure of the associated image.
///
/// The header must always contain the following required attributes, accessible
/// by convenience methods:
/// * [`display_window()`](Header::display_window) - "displayWindow" - the rectangle describing the viewable
/// portion of the image plane.
/// * [`data_window()`](Header::data_window) - "dataWindow" - the rectangle describing the portion of
/// the image plane that contains data.
/// * [`pixel_aspect_ratio()`](Header::pixel_aspect_ratio) - "pixelAspectRatio" - the ratio of a pixel's
/// width to height when the image is projected at the correct ratio
/// * [`channels()`](Header::channels) - "channels" - the description of
/// [`Channel`](crate::core::channel_list::Channel)s in the image.
/// * [`compression()`](Header::compression) - "compression" - which compression method is applied to
/// the pixel data when it is stored.
/// * [`line_order()`](Header::line_order) - "lineOrder" - the order in which scan lines are stored
/// in the file. Note this does not change the interpretation of "up", which is
/// always down in screen space, this is just an optimization hint for readers.
/// * [`screen_window_width()`](Header::screen_window_width) - "screenWindowWidth" - the width of the screen
/// window for the perspective projection that produced the image. If the image
/// wasn't produced by a perspective projection this should be set to 1.0.
/// * [`screen_window_center()`](Header::screen_window_center) - "screenWindowCenter" - the center of the screen
/// window for the perspective projection that produced the image. If the image
/// wasn't produced by a perspective projection this should be set to 1.0.
/// * "tiles" - required for tiled images only. [`TileDescription`]
/// describes the size, level mode and level rounding mode of the image.
/// * [`name()`](Header::name) - "name" - required for multi-part images only. The name of the
/// part, which must be unique between all parts in the image.
/// * [`image_type()`](Header::image_type) - "type" - required for deep images only, optional for
/// others. Can be either "scanlineimage", "tiledimage", "deepscanline", or
/// "deeptile"
/// * "maxSamplesPerPixel" - this attribute is
/// automatically added to deep files, being set when the file is written. This
/// represents the maximum number of samples contained by any pixel in the image
/// and is intended as an optimization hint for readers for pre-allocating storage.
///
#[repr(transparent)]
pub struct Header(pub(crate) Box<sys::Imf_Header_t>);

#[repr(transparent)]
pub struct HeaderSlice(pub(crate) Box<[sys::Imf_Header_t]>);

unsafe impl OpaquePtr for Header {
    type SysPointee = sys::Imf_Header_t;
    type Pointee = Header;
}

pub type HeaderRef<'a, P = Header> = Ref<'a, P>;
pub type HeaderRefMut<'a, P = Header> = RefMut<'a, P>;

impl Header {
    /// Construct a new [`Header`] with the given attributes.
    ///
    /// # Arguments
    /// * `data_window` - The window which contains data. Typically will be the
    /// same as the display window, but may be smaller to represent a crop region
    /// or larger to represent overscan.
    /// * `display_window` - The window which represents the size of the
    /// displayable image. Typically will be [[0, width-1], [0, height-1]].
    /// * `pixel_aspect_ratio` - The ratio of the pixel `width/height`, e.g. 2.0
    /// for anamorphic.
    /// * `screen_window_center` - The center of the screen window. Will be (0,0)
    /// for images that were not generated by perspective projection.
    /// * `screen_window_width` - The width of the screen window. Will be 1.0 for
    /// images that were not generated by perspective projection
    /// * `line_order` - The vertical order in which scanlines are stored. This
    /// is a hint for readers and may not be respected.
    /// * `compression` - The compression scheme to use to store all image data.
    ///
    /// ## Errors
    /// * [`Error::InvalidArgument`] - If the pixel aspect ratio is negative, or
    /// if the width or height of the display window is less than 1.
    ///
    pub fn new<B, V>(
        data_window: B,
        display_window: B,
        pixel_aspect_ratio: f32,
        screen_window_center: V,
        screen_window_width: f32,
        line_order: LineOrder,
        compression: Compression,
    ) -> Result<Header>
    where
        B: Bound2<i32>,
        V: Vec2<f32>,
    {
        unsafe {
            // We use the system allocator here as Header is not movable, but
            // is exposed as opaquebytes so that we can create arrays for it.
            // That means that we need Header to wrap a Box<sys::Imf_Header_t>
            // so that the C struct will never move.
            // The only way to do this is with Box::from_raw
            //
            let header = System.alloc(Layout::new::<sys::Imf_Header_t>())
                as *mut sys::Imf_Header_t;

            sys::Imf_Header_ctor(
                header,
                data_window.as_ptr() as *const sys::Imath_Box2i_t,
                display_window.as_ptr() as *const sys::Imath_Box2i_t,
                pixel_aspect_ratio,
                screen_window_center.as_ptr() as *const sys::Imath_V2f_t,
                screen_window_width,
                line_order.into(),
                compression.into(),
            )
            .into_result()?;

            Ok(Header(Box::from_raw(header)))
        }
    }

    /// Construct a new [`Header`] with the given attributes.
    ///
    /// # Arguments
    /// * `width` - The width of the image, setting both data and display
    /// windows
    /// * `height` - The height of the image, setting both data and display
    /// windows
    /// displayable image. Typically will be [[0, width-1], [0, height-1]].
    /// * `pixel_aspect_ratio` - The ratio of the pixel `width/height`, e.g. 2.0
    /// for anamorphic.
    /// * `screen_window_center` - The center of the screen window. Will be (0,0)
    /// for images that were not generated by perspective projection.
    /// * `screen_window_width` - The width of the screen window. Will be 1.0 for
    /// images that were not generated by perspective projection
    /// * `line_order` - The vertical order in which scanlines are stored. This
    /// is a hint for readers and may not be respected.
    /// * `compression` - The compression scheme to use to store all image data.
    ///
    /// ## Errors
    /// * [`Error::InvalidArgument`] - If the pixel aspect ratio is negative, or
    /// if the width or height of the display window is less than 1.
    ///
    pub fn with_dimensions<V>(
        width: i32,
        height: i32,
        pixel_aspect_ratio: f32,
        screen_window_center: V,
        screen_window_width: f32,
        line_order: LineOrder,
        compression: Compression,
    ) -> Result<Header>
    where
        V: Vec2<f32>,
    {
        unsafe {
            // We use the system allocator here as Header is not movable, but
            // is exposed as opaquebytes so that we can create arrays for it.
            // That means that we need Header to wrap a Box<sys::Imf_Header_t>
            // so that the C struct will never move.
            // The only way to do this is with Box::from_raw
            //
            let header = System.alloc(Layout::new::<sys::Imf_Header_t>())
                as *mut sys::Imf_Header_t;

            sys::Imf_Header_with_dimensions(
                header,
                width,
                height,
                pixel_aspect_ratio,
                screen_window_center.as_ptr() as *const sys::Imath_V2f_t,
                screen_window_width,
                line_order.into(),
                compression.into(),
            )
            .into_result()?;

            Ok(Header(Box::from_raw(header)))
        }
    }

    /// Construct a new [`HeaderSlice`], i.e. an array of `num` [`Header`]s,
    /// suitable for passing to the constructor of e.g.
    /// [`MultiPartOutputFile`](`crate::multi_part::multi_part_output_file::MultiPartOutputFile`).
    ///
    /// All the resulting [`Header`]s are default-initialized so you should
    /// manually iterate over and set their attributes after construction.
    ///
    pub fn new_array(num: usize) -> HeaderSlice {
        unsafe {
            let ptr = System
                .alloc(Layout::array::<sys::Imf_Header_t>(num).unwrap())
                as *mut sys::Imf_Header_t;

            dbg!(std::mem::size_of::<sys::Imf_Header_t>());

            let mut array =
                Box::from_raw(std::slice::from_raw_parts_mut(ptr, num));

            for header in array.iter_mut() {
                println!("header: {:p}", header as *mut _);
                sys::Imf_Header_with_dimensions(
                    header,
                    64,
                    64,
                    1.0f32,
                    [0.0f32, 0.0f32].as_ptr() as *const sys::Imath_V2f_t,
                    1.0f32,
                    LineOrder::IncreasingY.into(),
                    Compression::Zip.into(),
                );
            }

            HeaderSlice(array)
        }
    }

    /// Shortcut to construct a new [`Header`] with just the dimensions and
    /// everything else Default
    ///
    pub fn from_dimensions(width: i32, height: i32) -> Header {
        let mut header = Header::default();
        header.set_dimensions(width, height);
        header
    }

    /// Shortcut to construct a new [`Header`] with just the data and display
    /// windows and everything else Default
    ///
    pub fn from_windows<B: Bound2<i32>>(
        data_window: B,
        display_window: B,
    ) -> Header {
        let mut header = Header::default();
        *header.data_window_mut() = data_window;
        *header.display_window_mut() = display_window;
        header
    }

    /// Examines the header and returns an error if it finds something wrong
    /// with the attributes (e.g. empty display window, negative pixel aspect
    /// ratio etc.)
    ///
    /// # Arguments
    /// * `is_tiled` - This header should represent a tiled file
    /// * `is_multi_part` - This header should represent a multi-part file
    ///
    /// ## Errors
    /// * [`Error::InvalidArgument`] - If any of the header attributes are invalid
    ///
    pub fn sanity_check(
        &self,
        is_tiled: bool,
        is_multi_part: bool,
    ) -> Result<()> {
        unsafe {
            sys::Imf_Header_sanityCheck(
                self.0.as_ref(),
                is_tiled,
                is_multi_part,
            )
            .into_result()?;
        }
        Ok(())
    }

    /// [`Header::sanity_check()`] will throw an exception if the width or
    /// height of the data window exceeds the maximum image width or height, or
    /// if the size of a tile exceeds the maximum tile width or height.
    ///
    /// At program startup the maximum image and tile width and height
    /// are set to zero, meaning that width and height are unlimited.
    ///
    /// Limiting image and tile width and height limits how much memory
    /// will be allocated when a file is opened.  This can help protect
    /// applications from running out of memory while trying to read
    /// a damaged image file.
    ///
    pub fn set_max_image_size(max_width: i32, max_height: i32) {
        unsafe {
            sys::Imf_Header_setMaxImageSize(max_width, max_height)
                .into_result()
                .unwrap();
        }
    }

    /// [`Header::sanity_check()`] will throw an exception if the width or
    /// height of the data window exceeds the maximum image width or height, or
    /// if the size of a tile exceeds the maximum tile width or height.
    ///
    /// At program startup the maximum image and tile width and height
    /// are set to zero, meaning that width and height are unlimited.
    ///
    /// Limiting image and tile width and height limits how much memory
    /// will be allocated when a file is opened.  This can help protect
    /// applications from running out of memory while trying to read
    /// a damaged image file.
    ///
    pub fn set_max_tile_size(max_width: i32, max_height: i32) {
        unsafe {
            sys::Imf_Header_setMaxTileSize(max_width, max_height)
                .into_result()
                .unwrap();
        }
    }

    /// Check if the header reads nothing
    ///
    /// FIXME: This should be a const method in C++ but it's not - patch OpenEXR?
    ///
    pub fn reads_nothing(&mut self) -> bool {
        let mut result = false;
        unsafe {
            sys::Imf_Header_readsNothing(self.0.as_mut(), &mut result)
                .into_result()
                .unwrap()
        };
        result
    }
}

impl Default for Header {
    /// Creates a default header
    ///
    /// The resulting header has parameters:
    /// * `width` = 64
    /// * `height` = 64
    /// * `pixel_aspect_ratio` = 1.0
    /// * `screen_window_center` = [0.0, 0.0]
    /// * `screen_window_width` = 1.0
    /// * `line_order` = LineOrder::IncreasingY
    /// * `compression` = Compression::Zip
    ///
    fn default() -> Header {
        Header::with_dimensions(
            64,
            64,
            1.0f32,
            [0.0f32, 0.0f32],
            1.0f32,
            LineOrder::IncreasingY,
            Compression::Zip,
        )
        .unwrap()
    }
}

impl Header {
    //! # Standard attributes
    //!
    //! These methods can be used to get and set the required attributes of a
    //! standard OpenEXR file. The attributes `name`, `type` and
    //! `maxSamplesPerPixel` are only required for deep and multi-part images.

    /// Get a reference to the display window.
    ///
    /// The display window represents the rectangular region in pixel space that
    /// we wish to display. This typically correlates to what we normally think
    /// of as the "width" and "height" of the image, such that the display
    /// window rectangle is defined as a min/max inclusive pair of points
    /// (0, 0), (width-1, height-1).
    ///
    /// The display window must be the same for all parts in a file.
    ///
    /// Pixel space is a 2D coordinate system with X increasing from left to
    /// right and Y increasing from top to bottom.
    ///
    pub fn display_window<B>(&self) -> &B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_displayWindow_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            &*(ptr as *const sys::Imath_Box2i_t as *const B)
        }
    }

    /// Get a mutable reference to the display window.
    ///
    /// The display window represents the rectangular region in pixel space that
    /// we wish to display. This typically correlates to what we normally think
    /// of as the "width" and "height" of the image, such that the display
    /// window rectangle is defined as a min/max inclusive pair of points
    /// (0, 0), (width-1, height-1).
    ///
    /// The display window must be the same for all parts in a file.
    ///
    /// Pixel space is a 2D coordinate system with X increasing from left to
    /// right and Y increasing from top to bottom.
    ///
    pub fn display_window_mut<B>(&mut self) -> &mut B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_displayWindow(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            &mut *(ptr as *mut sys::Imath_Box2i_t as *mut B)
        }
    }

    /// Get a reference to the data window
    ///
    /// The data window represents the rectangular region of the image for which
    /// pixel data is defined in the file. Attempting to read or write data
    /// outside of that region is an error. For a "normal" image, the data
    /// window corresponds exactly to the display window, but for special cases
    /// may be different. For example it is common to only render a small section
    /// of the image ("crop region"), in which case the data window will be
    /// smaller than the display window, or to to render extra pixels outside of
    /// the display window ("overscan"), in which case the data window will be
    /// larger than the display window.
    ///
    pub fn data_window<B>(&self) -> &B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_dataWindow_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();

            &*(ptr as *const sys::Imath_Box2i_t as *const B)
        }
    }

    /// Get a mutable reference to the data window
    ///
    /// The data window represents the rectangular region of the image for which
    /// pixel data is defined in the file. Attempting to read or write data
    /// outside of that region is an error. For a "normal" image, the data
    /// window corresponds exactly to the display window, but for special cases
    /// may be different. For example it is common to only render a small section
    /// of the image ("crop region"), in which case the data window will be
    /// smaller than the display window, or to to render extra pixels outside of
    /// the display window ("overscan"), in which case the data window will be
    /// larger than the display window.
    ///
    pub fn data_window_mut<B>(&mut self) -> &mut B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_dataWindow(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            &mut *(ptr as *mut sys::Imath_Box2i_t as *mut B)
        }
    }

    /// Set both display and data windows to [[0, 0], [width-1, height-1]]
    ///
    pub fn set_dimensions(&mut self, width: i32, height: i32) {
        *self.data_window_mut() = [0, 0, width - 1, height - 1];
        *self.display_window_mut() = [0, 0, width - 1, height - 1];
    }

    /// Get the pixel aspect ratio
    ///
    /// Given d_x, the difference between pixel locations (x, y) and (x+1, y),
    /// and d_y, difference between pixel locations (x, y) and (x, y+1) on the
    /// the display, the pixel aspect ratio is the ratio d_x / d_y when the image
    /// is displayed dusch that the aspect ratio width/height is as intended.
    ///
    /// The pixel aspect ratio must be the same for all parts in a file.
    ///
    /// A normal image thus has a pixel aspect ratio of 1.0, while it is 2.0
    /// for an anamorphic image.
    ///
    pub fn pixel_aspect_ratio(&self) -> f32 {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_pixelAspectRatio_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            *ptr
        }
    }

    /// Set the pixel aspect ratio
    ///
    /// Given d_x, the difference between pixel locations (x, y) and (x+1, y),
    /// and d_y, difference between pixel locations (x, y) and (x, y+1) on the
    /// the display, the pixel aspect ratio is the ratio d_x / d_y when the image
    /// is displayed dusch that the aspect ratio width/height is as intended.
    ///
    /// The pixel aspect ratio must be the same for all parts in a file.
    ///
    /// A normal image thus has a pixel aspect ratio of 1.0, while it is 2.0
    /// for an anamorphic image.
    ///
    pub fn set_pixel_aspect_ratio(&mut self, par: f32) {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_pixelAspectRatio(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            *ptr = par;
        }
    }

    /// Get a reference to the screen window center
    ///
    /// The screen window represents the bounding rectangle of the image on the
    /// `z=1` plane assuming the image was generated by perspective projection
    /// with a width, `W`, and a center, `C`. The height of the window can be
    /// derived from the center and the pixel aspect ratio.
    ///
    /// Images that were not generated by perspective projection should have
    /// their screen window width set to 1 and their center to (0,0).
    ///
    pub fn screen_window_center<B>(&self) -> &B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_screenWindowCenter_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            &*(ptr as *const sys::Imath_Box2i_t as *const B)
        }
    }

    /// Get a mutable reference to the screen window center
    ///
    /// The screen window represents the bounding rectangle of the image on the
    /// `z=1` plane assuming the image was generated by perspective projection
    /// with a width, `W`, and a center, `C`. The height of the window can be
    /// derived from the center and the pixel aspect ratio.
    ///
    /// Images that were not generated by perspective projection should have
    /// their screen window width set to 1 and their center to (0,0).
    ///
    pub fn screen_window_center_mut<B>(&mut self) -> &mut B
    where
        B: Bound2<i32>,
    {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_screenWindowCenter(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            &mut *(ptr as *mut sys::Imath_Box2i_t as *mut B)
        }
    }

    /// Get a reference to the screen window width
    ///
    /// The screen window represents the bounding rectangle of the image on the
    /// `z=1` plane assuming the image was generated by perspective projection
    /// with a width, `W`, and a center, `C`. The height of the window can be
    /// derived from the center and the pixel aspect ratio.
    ///
    /// Images that were not generated by perspective projection should have
    /// their screen window width set to 1 and their center to (0,0).
    ///
    pub fn screen_window_width(&self) -> &f32 {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_screenWindowWidth_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            &*ptr
        }
    }

    /// Get a mutable reference to the screen window width
    ///
    /// The screen window represents the bounding rectangle of the image on the
    /// `z=1` plane assuming the image was generated by perspective projection
    /// with a width, `W`, and a center, `C`. The height of the window can be
    /// derived from the center and the pixel aspect ratio.
    ///
    /// Images that were not generated by perspective projection should have
    /// their screen window width set to 1 and their center to (0,0).
    ///
    pub fn screen_window_width_mut(&mut self) -> &f32 {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_screenWindowWidth(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            &mut *ptr
        }
    }

    /// Get a reference to the list of channels in the header
    pub fn channels(&self) -> ChannelListRef {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_channels_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            ChannelListRef::new(ptr)
        }
    }

    /// Get a mutable reference to the list of channels in the header
    pub fn channels_mut(&mut self) -> ChannelListRefMut {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_channels(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            ChannelListRefMut::new(ptr)
        }
    }

    /// Get the line order from the header
    ///
    /// Specifies the order in which rows of pixels are stored in the file,
    /// either [`LineOrder::IncreasingY`], [`LineOrder::DecreasingY`] or
    /// [`LineOrder::RandomY`] for tiled images.
    ///
    /// This does not affect the pixel space coordinates, only the order in
    /// which the data is stored.
    ///
    pub fn line_order(&self) -> LineOrder {
        let mut ptr = std::ptr::null();
        unsafe {
            sys::Imf_Header_lineOrder_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            (*ptr).into()
        }
    }

    /// Set the line order in the header
    ///
    /// Specifies the order in which rows of pixels are stored in the file,
    /// either [`LineOrder::IncreasingY`], [`LineOrder::DecreasingY`] or
    /// [`LineOrder::RandomY`] for tiled images.
    ///
    /// This does not affect the pixel space coordinates, only the order in
    /// which the data is stored.
    ///
    pub fn set_line_order(&mut self, lo: LineOrder) {
        unsafe {
            let mut ptr = std::ptr::null_mut();
            sys::Imf_Header_lineOrder(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            *ptr = lo.into();
        };
    }

    /// Get the compression type from the header
    ///
    /// Defines the compression scheme used to store all pixel data.
    ///
    pub fn compression(&self) -> Compression {
        let mut ptr = std::ptr::null();
        unsafe {
            sys::Imf_Header_compression_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .unwrap();
            (*ptr).into()
        }
    }

    /// Set the compression type in the header
    ///
    /// Defines the compression scheme used to store all pixel data.
    ///
    pub fn set_compression(&mut self, cmp: Compression) {
        let mut ptr = std::ptr::null_mut();
        unsafe {
            sys::Imf_Header_compression(self.0.as_mut(), &mut ptr)
                .into_result()
                .unwrap();
            *ptr = cmp.into();
        }
    }
}

impl Header {
    //! # Required attributes for multi-part files
    //!
    //! These attributes are all mandatory for multi-part files and optional
    //! for single-part files.

    /// Get the name of this part from the header
    ///
    /// Names must be unique, that is no two parts in the same file may share
    /// the same name.
    ///
    pub fn name(&self) -> Result<String> {
        unsafe {
            let mut s = std::ptr::null();
            sys::Imf_Header_name_const(self.0.as_ref(), &mut s)
                .into_result()
                .map(|_| {
                    let mut cptr = std::ptr::null();
                    sys::std_string_c_str(s, &mut cptr).into_result().unwrap();

                    CStr::from_ptr(cptr).to_string_lossy().to_string()
                })
                .map_err(Error::from)
        }
    }

    /// Set the name of this part in the header
    ///
    /// Names must be unique, that is no two parts in the same file may share
    /// the same name.
    ///
    pub fn set_name(&mut self, name: &str) {
        unsafe {
            let s = CppString::new(name);
            sys::Imf_Header_setName(self.0.as_mut(), s.0);
        }
    }

    /// Does the file/part have a name?
    pub fn has_name(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasName(self.0.as_ref(), &mut v);
            v
        }
    }

    /// Get the image type of this part from the header
    ///
    /// This must be one of:
    /// * `scanlineimage` - Flat, scanline-based.
    /// * `tiledimage` - Flat, tiled.
    /// * `deepscanline` - Deep, scanline-based.
    /// * `deeptile` - Deep, tiled.
    ///
    pub fn image_type(&self) -> Result<ImageType> {
        unsafe {
            let mut s = std::ptr::null();
            sys::Imf_Header_type_const(self.0.as_ref(), &mut s)
                .into_result()
                .map(|_| {
                    let mut cptr = std::ptr::null();
                    sys::std_string_c_str(s, &mut cptr);
                    match CStr::from_ptr(cptr).to_str().unwrap() {
                        "scanlineimage" => ImageType::Scanline,
                        "tiledimage" => ImageType::Tiled,
                        "deepscanline" => ImageType::DeepScanline,
                        "deeptile" => ImageType::DeepTiled,
                        _ => panic!("bad value for image type"),
                    }
                })
                .map_err(Error::from)
        }
    }

    /// Set the image type of this part in the header
    ///
    /// This must be one of:
    /// * `scanlineimage` - Flat, scanline-based.
    /// * `tiledimage` - Flat, tiled.
    /// * `deepscanline` - Deep, scanline-based.
    /// * `deeptile` - Deep, tiled.
    ///
    pub fn set_image_type(&mut self, image_type: ImageType) {
        unsafe {
            let s = match image_type {
                ImageType::Scanline => CppString::new("scanlineimage"),
                ImageType::Tiled => CppString::new("tiledimage"),
                ImageType::DeepScanline => CppString::new("deepscanline"),
                ImageType::DeepTiled => CppString::new("deeptile"),
            };
            sys::Imf_Header_setType(self.0.as_mut(), s.0)
                .into_result()
                .expect("Unexpected exception from Imf_Header_setType");
        }
    }

    /// Does the file/part have a type?
    pub fn has_image_type(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasType(self.0.as_ref(), &mut v);
            v
        }
    }

    /// Get the version of the file
    ///
    pub fn version(&self) -> Result<i32> {
        unsafe {
            let mut v = std::ptr::null();
            sys::Imf_Header_version_const(self.0.as_ref(), &mut v)
                .into_result()
                .map(|_| *v)
                .map_err(Error::from)
        }
    }

    /// Set the version of the file
    ///
    pub fn set_version(&mut self, v: i32) {
        unsafe {
            sys::Imf_Header_setVersion(self.0.as_mut(), v);
        }
    }

    /// Does the file have its version specified?
    pub fn has_version(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasVersion(self.0.as_ref(), &mut v);
            v
        }
    }
}

impl Header {
    //! # Chunk count
    //!
    //! Chunk count is set automatically when writing the file

    /// Does the file have its chunk count specified?
    pub fn has_chunk_count(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasChunkCount(self.0.as_ref(), &mut v);
            v
        }
    }

    /// Get the chunk_count of the file
    ///
    pub fn chunk_count(&self) -> Result<i32> {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_chunkCount_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .map(|_| *ptr)
                .map_err(Error::from)
        }
    }
}

impl Header {
    //! # Views
    //!
    //! View names must be unique, that is no two parts in the same file may share
    //! the same view. Only supported for multi-part files, deprecated for
    //! single-part files.

    /// Get the view of this part from the header
    ///
    pub fn view(&self) -> Result<String> {
        unsafe {
            let mut s = std::ptr::null();
            sys::Imf_Header_view_const(self.0.as_ref(), &mut s)
                .into_result()
                .map(|_| {
                    let mut cptr = std::ptr::null();
                    sys::std_string_c_str(s, &mut cptr);
                    CStr::from_ptr(cptr).to_string_lossy().to_string()
                })
                .map_err(Error::from)
        }
    }

    /// Set the view of this part in the header
    ///
    pub fn set_view(&mut self, view: &str) {
        unsafe {
            let s = CppString::new(view);
            sys::Imf_Header_setView(self.0.as_mut(), s.0);
        }
    }

    /// Does the part have a view specified?
    pub fn has_view(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasView(self.0.as_ref(), &mut v);
            v
        }
    }
}

impl Header {
    //! # Tile Description
    //!
    //! The tile description is a
    //! [`TileDescriptionAttribute`](crate::core::attribute::TileDescriptionAttribute) whose name is
    //! `"tiles"`. It is mandatory for tiled files. The [`TileDescription`]
    //! describes various properties of the tiles that make up the image file.

    /// Get the tile description from the header
    ///
    pub fn tile_description(&self) -> Result<TileDescription> {
        let mut ptr = std::ptr::null();
        unsafe {
            sys::Imf_Header_tileDescription_const(self.0.as_ref(), &mut ptr)
                .into_result()
                .map(|_| (*ptr).clone().into())
                .map_err(Error::from)
        }
    }

    /// Set the tile description in the header
    ///
    pub fn set_tile_description(&mut self, td: &TileDescription) {
        unsafe {
            let td = (*td).into();
            sys::Imf_Header_setTileDescription(self.0.as_mut(), &td);
        }
    }

    /// Does the part have a tile description?
    ///
    pub fn has_tile_description(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasTileDescription(self.0.as_ref(), &mut v);
            v
        }
    }
}

impl Header {
    //! # Preview Image
    //!
    //! The preview image ias a [`PreviewImageAttribute`](crate::core::attribute::PreviewImageAttribute) whose name is
    //! `"preview"`.
    //! This attribute is special -- while an image file is being written,
    //! the pixels of the preview image can be changed repeatedly by calling
    //! [`update_preview_image()`](crate::core::output_file::OutputFile::update_preview_image)

    /// Get the preview image from the header
    ///
    /// # Errors
    /// * [`Error::InvalidType`] - If the preview image attribute is not of the expected type
    /// * [`Error::Base`] - If any other error occurs
    ///
    pub fn preview_image(&self) -> Result<PreviewImageRef> {
        unsafe {
            let mut ptr = std::ptr::null();
            sys::Imf_Header_previewImage_const(self.0.as_ref(), &mut ptr)
                .into_result()?;

            Ok(PreviewImageRef::new(ptr))
        }
    }

    /// Set the preview image in the header
    ///
    /// # Errors
    /// * [`Error::InvalidType`] - If the preview image attribute can not be assigned the expected type
    /// * [`Error::Base`] - If any other error occurs
    pub fn set_preview_image(&mut self, pi: &PreviewImage) -> Result<()> {
        unsafe {
            sys::Imf_Header_setPreviewImage(self.0.as_mut(), pi.0)
                .into_result()?;

            Ok(())
        }
    }

    /// Does the part have a preview image?
    ///
    pub fn has_preview_image(&self) -> bool {
        unsafe {
            let mut v = false;
            sys::Imf_Header_hasPreviewImage(self.0.as_ref(), &mut v);
            v
        }
    }
}

use paste::paste;

macro_rules! make_find_typed_attribute {
    ($tn:ident, $sfx:ident) => {
        paste! {
            use crate::core::attribute::{[<$tn AttributeRef>], [<$tn AttributeRefMut>]};
            impl Header {
                /// Get a reference to the typed Attribute with the given name
                ///
                pub fn [<find_typed_attribute_ $sfx>](
                    &self,
                    name: &str,
                ) -> Option<[<$tn AttributeRef>]> {
                    let c_name = CString::new(name).expect("Invalid UTF-8 in name");
                    let mut attr_ptr = std::ptr::null();
                    unsafe {
                        sys::[<Imf_Header_findTypedAttribute_ $tn _const>](
                            self.0.as_ref(),
                            &mut attr_ptr,
                            c_name.as_ptr(),
                        )
                    };

                    if !attr_ptr.is_null() {
                        Some([<$tn AttributeRef>]::new(attr_ptr))
                    } else {
                        None
                    }
                }

                /// Get a mutable reference to the typed Attribute with the given name
                ///
                pub fn [<find_typed_attribute_ $sfx _mut>](
                    &mut self,
                    name: &str,
                ) -> Option<[<$tn AttributeRefMut>]> {
                    let c_name = CString::new(name).expect("Invalid UTF-8 in name");
                    let mut attr_ptr = std::ptr::null_mut();
                    unsafe {
                        sys::[<Imf_Header_findTypedAttribute_ $tn>](
                            self.0.as_mut(),
                            &mut attr_ptr,
                            c_name.as_ptr(),
                        )
                    };

                    if !attr_ptr.is_null() {
                        Some([<$tn AttributeRefMut>]::new(attr_ptr))
                    } else {
                        None
                    }
                }
            }
        }
    }
}

impl Header {
    //! # Modifying user attributes

    /// Inserts the given metadata attribute with the given name
    ///
    /// ## Errors
    /// * [`Error::InvalidType`] - If the attribute to be inserted matches an
    /// attribute that is already present but with a different type.
    /// * [`Error::InvalidArgument`] - If the attribute name is the empty string
    ///
    pub fn insert<A>(&mut self, name: &str, attribute: &A) -> Result<()>
    where
        A: TypedAttribute,
    {
        let c_name = CString::new(name).expect("Invalid UTF-8 in name");
        unsafe {
            sys::Imf_Header_insert(
                self.0.as_mut(),
                c_name.as_ptr(),
                attribute.as_attribute_ptr(),
            )
            .into_result()?;
        }

        Ok(())
    }

    /// Erases the attribute with the given name.
    ///
    /// If no attribute with `name` exists, the [`Header`] is unchanged.
    ///
    pub fn erase(&mut self, name: &str) -> Result<()> {
        let c_name = CString::new(name).expect("Invalid UTF-8 in name");
        unsafe {
            sys::Imf_Header_erase(self.0.as_mut(), c_name.as_ptr())
                .into_result()?;
        }
        Ok(())
    }
}

make_find_typed_attribute!(Int, int);
make_find_typed_attribute!(Float, float);
make_find_typed_attribute!(Double, double);

make_find_typed_attribute!(Chromaticities, chromaticities);
make_find_typed_attribute!(Compression, compression);
make_find_typed_attribute!(DeepImageState, deep_image_state);
make_find_typed_attribute!(Envmap, envmap);
make_find_typed_attribute!(ChannelList, channel_list);
make_find_typed_attribute!(CppVectorFloat, vector_float);
make_find_typed_attribute!(CppVectorString, vector_string);
make_find_typed_attribute!(CppString, string);
make_find_typed_attribute!(LineOrder, line_order);

make_find_typed_attribute!(V2i, v2i);
make_find_typed_attribute!(V2f, v2f);
make_find_typed_attribute!(V2d, v2d);

make_find_typed_attribute!(V3i, v3i);
make_find_typed_attribute!(V3f, v3f);
make_find_typed_attribute!(V3d, v3d);

make_find_typed_attribute!(Box2i, box2i);
make_find_typed_attribute!(Box2f, box2f);

make_find_typed_attribute!(M33f, m33f);
make_find_typed_attribute!(M33d, m33d);

make_find_typed_attribute!(M44f, m44f);
make_find_typed_attribute!(M44d, m44d);

impl Drop for Header {
    fn drop(&mut self) {
        unsafe {
            sys::Imf_Header_dtor(self.0.as_mut());
        }
    }
}

impl HeaderSlice {
    pub fn iter(&self) -> HeaderSliceIter {
        HeaderSliceIter {
            curr: 0,
            len: self.0.len(),
            header_slice: self,
        }
    }

    pub fn iter_mut(&mut self) -> HeaderSliceIterMut {
        HeaderSliceIterMut {
            curr: 0,
            len: self.0.len(),
            header_slice: self,
        }
    }
}

pub struct HeaderSliceIter<'s> {
    curr: usize,
    len: usize,
    header_slice: &'s HeaderSlice,
}

impl<'s> Iterator for HeaderSliceIter<'s> {
    type Item = HeaderRef<'s>;

    fn next(&mut self) -> Option<Self::Item> {
        if self.curr == self.len {
            None
        } else {
            let ptr = &((*self.header_slice.0)[self.curr])
                as *const sys::Imf_Header_t;

            self.curr += 1;

            Some(HeaderRef::new(ptr))
        }
    }
}

pub struct HeaderSliceIterMut<'s> {
    curr: usize,
    len: usize,
    header_slice: &'s mut HeaderSlice,
}

impl<'s> Iterator for HeaderSliceIterMut<'s> {
    type Item = HeaderRefMut<'s>;

    fn next(&mut self) -> Option<Self::Item> {
        if self.curr == self.len {
            None
        } else {
            let ptr = &mut ((*self.header_slice.0)[self.curr])
                as *mut sys::Imf_Header_t;

            self.curr += 1;

            Some(HeaderRefMut::new(ptr))
        }
    }
}

/// Used to set (or inspect) the type of an image in the header
///
#[derive(Copy, Clone, Debug, PartialEq, Eq, PartialOrd, Ord, Hash)]
pub enum ImageType {
    Scanline,
    Tiled,
    DeepScanline,
    DeepTiled,
}

#[cfg(test)]
#[test]
fn header_rtrip1() -> Result<()> {
    use crate::tests::load_ferris;
    use crate::{
        core::{
            attribute::{
                CompressionAttribute, CppStringAttribute,
                CppVectorFloatAttribute, CppVectorStringAttribute,
                DeepImageStateAttribute, DoubleAttribute, EnvmapAttribute,
                FloatAttribute, IntAttribute,
            },
            cppstd::{CppVectorFloat, CppVectorString},
            Envmap,
        },
        deep::DeepImageState,
        rgba::{
            rgba::RgbaChannels,
            rgba_file::{RgbaInputFile, RgbaOutputFile},
        },
    };

    let (pixels, width, height) = load_ferris();

    let mut header = Header::from_dimensions(width, height);

    header.insert("at_int", &IntAttribute::from_value(17))?;
    header.insert("at_float", &FloatAttribute::from_value(42.0))?;
    header.insert("at_double", &DoubleAttribute::from_value(127.0))?;

    header.insert(
        "at_compression",
        &CompressionAttribute::from_value(&Compression::Dwaa),
    )?;

    header.insert(
        "at_deep_image_state",
        &DeepImageStateAttribute::from_value(&DeepImageState::NonOverlapping),
    )?;

    header
        .insert("at_envmap", &EnvmapAttribute::from_value(&Envmap::Latlong))?;

    header.insert(
        "at_vector_float",
        &CppVectorFloatAttribute::from_value(&CppVectorFloat::from_slice(&[
            1.0, 2.0, 3.0, 4.0,
        ])),
    )?;

    header.insert(
        "at_vector_string",
        &CppVectorStringAttribute::from_value(&CppVectorString::from_slice(&[
            "a", "b", "c", "d",
        ])),
    )?;

    header
        .insert("at_string", &CppStringAttribute::from_value("lorem ipsum"))?;

    let mut file = RgbaOutputFile::new(
        "header_rtrip1.exr",
        &header,
        RgbaChannels::WriteRgba,
        1,
    )?;

    file.set_frame_buffer(&pixels, 1, width as usize)?;
    unsafe {
        file.write_pixels(height)?;
    }

    std::mem::drop(file);

    let file = RgbaInputFile::new("header_rtrip1.exr", 4)?;

    assert_eq!(
        file.header()
            .find_typed_attribute_int("at_int")
            .unwrap()
            .value(),
        &17
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_float("at_float")
            .unwrap()
            .value(),
        &42.0
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_double("at_double")
            .unwrap()
            .value(),
        &127.0
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_compression("at_compression")
            .unwrap()
            .value(),
        Compression::Dwaa
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_deep_image_state("at_deep_image_state")
            .unwrap()
            .value(),
        DeepImageState::NonOverlapping,
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_envmap("at_envmap")
            .unwrap()
            .value(),
        Envmap::Latlong,
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_vector_float("at_vector_float")
            .unwrap()
            .value()
            .as_slice(),
        &[1.0f32, 2.0, 3.0, 4.0],
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_vector_string("at_vector_string")
            .unwrap()
            .value()
            .to_vec(),
        &["a", "b", "c", "d"],
    );

    assert_eq!(
        file.header()
            .find_typed_attribute_string("at_string")
            .unwrap()
            .value(),
        "lorem ipsum",
    );

    assert!(file.header().version().is_err());
    assert!(file.header().image_type().is_err());
    assert!(file.header().preview_image().is_err());
    assert!(file.header().name().is_err());
    assert!(file.header().view().is_err());

    Ok(())
}
