#![allow(
	unused_parens,
	clippy::excessive_precision,
	clippy::missing_safety_doc,
	clippy::not_unsafe_ptr_arg_deref,
	clippy::should_implement_trait,
	clippy::too_many_arguments,
	clippy::unused_unit,
)]
//! # Super Resolution
//! 
//! The Super Resolution module contains a set of functions and classes that can be used to solve the
//! problem of resolution enhancement. There are a few methods implemented, most of them are described in
//! the papers [Farsiu03](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Farsiu03) and [Mitzel09](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Mitzel09) .
use crate::{mod_prelude::*, core, sys, types};
pub mod prelude {
	pub use { super::Superres_DenseOpticalFlowExtConst, super::Superres_DenseOpticalFlowExt, super::Superres_FarnebackOpticalFlowConst, super::Superres_FarnebackOpticalFlow, super::Superres_DualTVL1OpticalFlowConst, super::Superres_DualTVL1OpticalFlow, super::Superres_BroxOpticalFlowConst, super::Superres_BroxOpticalFlow, super::Superres_PyrLKOpticalFlowConst, super::Superres_PyrLKOpticalFlow, super::Superres_FrameSourceConst, super::Superres_FrameSource, super::Superres_SuperResolutionConst, super::Superres_SuperResolution };
}

/// ## C++ default parameters
/// * device_id: 0
// createFrameSource_Camera(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:80
#[inline]
pub fn create_frame_source_camera(device_id: i32) -> Result<core::Ptr<dyn crate::superres::Superres_FrameSource>> {
	let ret = unsafe { sys::cv_superres_createFrameSource_Camera_int(device_id) }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FrameSource>::opencv_from_extern(ret) };
	Ok(ret)
}

// createFrameSource_Empty() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:75
#[inline]
pub fn create_frame_source_empty() -> Result<core::Ptr<dyn crate::superres::Superres_FrameSource>> {
	let ret = unsafe { sys::cv_superres_createFrameSource_Empty() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FrameSource>::opencv_from_extern(ret) };
	Ok(ret)
}

// createFrameSource_Video_CUDA(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:78
#[inline]
pub fn create_frame_source_video_cuda(file_name: &str) -> Result<core::Ptr<dyn crate::superres::Superres_FrameSource>> {
	extern_container_arg!(file_name);
	let ret = unsafe { sys::cv_superres_createFrameSource_Video_CUDA_const_StringR(file_name.opencv_as_extern()) }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FrameSource>::opencv_from_extern(ret) };
	Ok(ret)
}

// createFrameSource_Video(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:77
#[inline]
pub fn create_frame_source_video(file_name: &str) -> Result<core::Ptr<dyn crate::superres::Superres_FrameSource>> {
	extern_container_arg!(file_name);
	let ret = unsafe { sys::cv_superres_createFrameSource_Video_const_StringR(file_name.opencv_as_extern()) }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FrameSource>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_Brox_CUDA() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:177
#[inline]
pub fn create_opt_flow_brox_cuda() -> Result<core::Ptr<dyn crate::superres::Superres_BroxOpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_Brox_CUDA() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_BroxOpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_DualTVL1() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:139
#[inline]
pub fn create_opt_flow_dual_tvl1() -> Result<core::Ptr<dyn crate::superres::Superres_DualTVL1OpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_DualTVL1() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_DualTVL1OpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_DualTVL1_CUDA() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:140
#[inline]
pub fn create_opt_flow_dual_tvl1_cuda() -> Result<core::Ptr<dyn crate::superres::Superres_DualTVL1OpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_DualTVL1_CUDA() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_DualTVL1OpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_Farneback() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:96
#[inline]
pub fn create_opt_flow_farneback() -> Result<core::Ptr<dyn crate::superres::Superres_FarnebackOpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_Farneback() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FarnebackOpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_Farneback_CUDA() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:97
#[inline]
pub fn create_opt_flow_farneback_cuda() -> Result<core::Ptr<dyn crate::superres::Superres_FarnebackOpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_Farneback_CUDA() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_FarnebackOpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

// createOptFlow_PyrLK_CUDA() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:196
#[inline]
pub fn create_opt_flow_pyr_lk_cuda() -> Result<core::Ptr<dyn crate::superres::Superres_PyrLKOpticalFlow>> {
	let ret = unsafe { sys::cv_superres_createOptFlow_PyrLK_CUDA() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_PyrLKOpticalFlow>::opencv_from_extern(ret) };
	Ok(ret)
}

/// Create Bilateral TV-L1 Super Resolution.
/// 
/// This class implements Super Resolution algorithm described in the papers [Farsiu03](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Farsiu03) and
/// [Mitzel09](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Mitzel09) .
/// 
/// Here are important members of the class that control the algorithm, which you can set after
/// constructing the class instance:
/// 
/// *   **int scale** Scale factor.
/// *   **int iterations** Iteration count.
/// *   **double tau** Asymptotic value of steepest descent method.
/// *   **double lambda** Weight parameter to balance data term and smoothness term.
/// *   **double alpha** Parameter of spacial distribution in Bilateral-TV.
/// *   **int btvKernelSize** Kernel size of Bilateral-TV filter.
/// *   **int blurKernelSize** Gaussian blur kernel size.
/// *   **double blurSigma** Gaussian blur sigma.
/// *   **int temporalAreaRadius** Radius of the temporal search area.
/// *   **Ptr\<DenseOpticalFlowExt\> opticalFlow** Dense optical flow algorithm.
// createSuperResolution_BTVL1() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:199
#[inline]
pub fn create_super_resolution_btvl1() -> Result<core::Ptr<dyn crate::superres::Superres_SuperResolution>> {
	let ret = unsafe { sys::cv_superres_createSuperResolution_BTVL1() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_SuperResolution>::opencv_from_extern(ret) };
	Ok(ret)
}

// createSuperResolution_BTVL1_CUDA() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:200
#[inline]
pub fn create_super_resolution_btvl1_cuda() -> Result<core::Ptr<dyn crate::superres::Superres_SuperResolution>> {
	let ret = unsafe { sys::cv_superres_createSuperResolution_BTVL1_CUDA() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_SuperResolution>::opencv_from_extern(ret) };
	Ok(ret)
}

// BroxOpticalFlow /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:143
pub trait Superres_BroxOpticalFlowConst: crate::superres::Superres_DenseOpticalFlowExtConst {
	fn as_raw_Superres_BroxOpticalFlow(&self) -> *const c_void;

	/// Flow smoothness
	/// ## See also
	/// setAlpha
	// getAlpha() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:148
	#[inline]
	fn get_alpha(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getAlpha_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// Gradient constancy importance
	/// ## See also
	/// setGamma
	// getGamma() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:153
	#[inline]
	fn get_gamma(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getGamma_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// Pyramid scale factor
	/// ## See also
	/// setScaleFactor
	// getScaleFactor() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:158
	#[inline]
	fn get_scale_factor(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getScaleFactor_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of lagged non-linearity iterations (inner loop)
	/// ## See also
	/// setInnerIterations
	// getInnerIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:163
	#[inline]
	fn get_inner_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getInnerIterations_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of warping iterations (number of pyramid levels)
	/// ## See also
	/// setOuterIterations
	// getOuterIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:168
	#[inline]
	fn get_outer_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getOuterIterations_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of linear system solver iterations
	/// ## See also
	/// setSolverIterations
	// getSolverIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:173
	#[inline]
	fn get_solver_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_getSolverIterations_const(self.as_raw_Superres_BroxOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait Superres_BroxOpticalFlow: crate::superres::Superres_BroxOpticalFlowConst + crate::superres::Superres_DenseOpticalFlowExt {
	fn as_raw_mut_Superres_BroxOpticalFlow(&mut self) -> *mut c_void;

	/// Flow smoothness
	/// ## See also
	/// setAlpha getAlpha
	// setAlpha(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:150
	#[inline]
	fn set_alpha(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setAlpha_double(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Gradient constancy importance
	/// ## See also
	/// setGamma getGamma
	// setGamma(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:155
	#[inline]
	fn set_gamma(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setGamma_double(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Pyramid scale factor
	/// ## See also
	/// setScaleFactor getScaleFactor
	// setScaleFactor(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:160
	#[inline]
	fn set_scale_factor(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setScaleFactor_double(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of lagged non-linearity iterations (inner loop)
	/// ## See also
	/// setInnerIterations getInnerIterations
	// setInnerIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:165
	#[inline]
	fn set_inner_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setInnerIterations_int(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of warping iterations (number of pyramid levels)
	/// ## See also
	/// setOuterIterations getOuterIterations
	// setOuterIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:170
	#[inline]
	fn set_outer_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setOuterIterations_int(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Number of linear system solver iterations
	/// ## See also
	/// setSolverIterations getSolverIterations
	// setSolverIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:175
	#[inline]
	fn set_solver_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_BroxOpticalFlow_setSolverIterations_int(self.as_raw_mut_Superres_BroxOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
}

// DenseOpticalFlowExt /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:56
pub trait Superres_DenseOpticalFlowExtConst: core::AlgorithmTraitConst {
	fn as_raw_Superres_DenseOpticalFlowExt(&self) -> *const c_void;

}

pub trait Superres_DenseOpticalFlowExt: core::AlgorithmTrait + crate::superres::Superres_DenseOpticalFlowExtConst {
	fn as_raw_mut_Superres_DenseOpticalFlowExt(&mut self) -> *mut c_void;

	/// ## C++ default parameters
	/// * flow2: noArray()
	// calc(cv::InputArray, cv::InputArray, cv::OutputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:59
	#[inline]
	fn calc(&mut self, frame0: &dyn core::ToInputArray, frame1: &dyn core::ToInputArray, flow1: &mut dyn core::ToOutputArray, flow2: &mut dyn core::ToOutputArray) -> Result<()> {
		input_array_arg!(frame0);
		input_array_arg!(frame1);
		output_array_arg!(flow1);
		output_array_arg!(flow2);
		let ret = unsafe { sys::cv_superres_DenseOpticalFlowExt_calc_const__InputArrayR_const__InputArrayR_const__OutputArrayR_const__OutputArrayR(self.as_raw_mut_Superres_DenseOpticalFlowExt(), frame0.as_raw__InputArray(), frame1.as_raw__InputArray(), flow1.as_raw__OutputArray(), flow2.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	// collectGarbage() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:60
	#[inline]
	fn collect_garbage(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DenseOpticalFlowExt_collectGarbage(self.as_raw_mut_Superres_DenseOpticalFlowExt()) }.into_result()?;
		Ok(ret)
	}
	
}

// DualTVL1OpticalFlow /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:103
pub trait Superres_DualTVL1OpticalFlowConst: crate::superres::Superres_DenseOpticalFlowExtConst {
	fn as_raw_Superres_DualTVL1OpticalFlow(&self) -> *const c_void;

	/// ## See also
	/// setTau
	// getTau() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:107
	#[inline]
	fn get_tau(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getTau_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setLambda
	// getLambda() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:111
	#[inline]
	fn get_lambda(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getLambda_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setTheta
	// getTheta() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:115
	#[inline]
	fn get_theta(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getTheta_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setScalesNumber
	// getScalesNumber() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:119
	#[inline]
	fn get_scales_number(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getScalesNumber_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setWarpingsNumber
	// getWarpingsNumber() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:123
	#[inline]
	fn get_warpings_number(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getWarpingsNumber_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setEpsilon
	// getEpsilon() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:127
	#[inline]
	fn get_epsilon(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getEpsilon_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations
	// getIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:131
	#[inline]
	fn get_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getIterations_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setUseInitialFlow
	// getUseInitialFlow() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:135
	#[inline]
	fn get_use_initial_flow(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_getUseInitialFlow_const(self.as_raw_Superres_DualTVL1OpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait Superres_DualTVL1OpticalFlow: crate::superres::Superres_DenseOpticalFlowExt + crate::superres::Superres_DualTVL1OpticalFlowConst {
	fn as_raw_mut_Superres_DualTVL1OpticalFlow(&mut self) -> *mut c_void;

	/// ## See also
	/// setTau getTau
	// setTau(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:109
	#[inline]
	fn set_tau(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setTau_double(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setLambda getLambda
	// setLambda(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:113
	#[inline]
	fn set_lambda(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setLambda_double(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setTheta getTheta
	// setTheta(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:117
	#[inline]
	fn set_theta(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setTheta_double(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setScalesNumber getScalesNumber
	// setScalesNumber(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:121
	#[inline]
	fn set_scales_number(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setScalesNumber_int(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setWarpingsNumber getWarpingsNumber
	// setWarpingsNumber(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:125
	#[inline]
	fn set_warpings_number(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setWarpingsNumber_int(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setEpsilon getEpsilon
	// setEpsilon(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:129
	#[inline]
	fn set_epsilon(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setEpsilon_double(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations getIterations
	// setIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:133
	#[inline]
	fn set_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setIterations_int(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setUseInitialFlow getUseInitialFlow
	// setUseInitialFlow(bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:137
	#[inline]
	fn set_use_initial_flow(&mut self, val: bool) -> Result<()> {
		let ret = unsafe { sys::cv_superres_DualTVL1OpticalFlow_setUseInitialFlow_bool(self.as_raw_mut_Superres_DualTVL1OpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
}

// FarnebackOpticalFlow /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:64
pub trait Superres_FarnebackOpticalFlowConst: crate::superres::Superres_DenseOpticalFlowExtConst {
	fn as_raw_Superres_FarnebackOpticalFlow(&self) -> *const c_void;

	/// ## See also
	/// setPyrScale
	// getPyrScale() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:68
	#[inline]
	fn get_pyr_scale(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getPyrScale_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setLevelsNumber
	// getLevelsNumber() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:72
	#[inline]
	fn get_levels_number(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getLevelsNumber_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setWindowSize
	// getWindowSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:76
	#[inline]
	fn get_window_size(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getWindowSize_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations
	// getIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:80
	#[inline]
	fn get_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getIterations_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setPolyN
	// getPolyN() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:84
	#[inline]
	fn get_poly_n(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getPolyN_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setPolySigma
	// getPolySigma() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:88
	#[inline]
	fn get_poly_sigma(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getPolySigma_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setFlags
	// getFlags() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:92
	#[inline]
	fn get_flags(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_getFlags_const(self.as_raw_Superres_FarnebackOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait Superres_FarnebackOpticalFlow: crate::superres::Superres_DenseOpticalFlowExt + crate::superres::Superres_FarnebackOpticalFlowConst {
	fn as_raw_mut_Superres_FarnebackOpticalFlow(&mut self) -> *mut c_void;

	/// ## See also
	/// setPyrScale getPyrScale
	// setPyrScale(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:70
	#[inline]
	fn set_pyr_scale(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setPyrScale_double(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setLevelsNumber getLevelsNumber
	// setLevelsNumber(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:74
	#[inline]
	fn set_levels_number(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setLevelsNumber_int(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setWindowSize getWindowSize
	// setWindowSize(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:78
	#[inline]
	fn set_window_size(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setWindowSize_int(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations getIterations
	// setIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:82
	#[inline]
	fn set_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setIterations_int(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setPolyN getPolyN
	// setPolyN(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:86
	#[inline]
	fn set_poly_n(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setPolyN_int(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setPolySigma getPolySigma
	// setPolySigma(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:90
	#[inline]
	fn set_poly_sigma(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setPolySigma_double(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setFlags getFlags
	// setFlags(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:94
	#[inline]
	fn set_flags(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FarnebackOpticalFlow_setFlags_int(self.as_raw_mut_Superres_FarnebackOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
}

// FrameSource /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:66
pub trait Superres_FrameSourceConst {
	fn as_raw_Superres_FrameSource(&self) -> *const c_void;

}

pub trait Superres_FrameSource: crate::superres::Superres_FrameSourceConst {
	fn as_raw_mut_Superres_FrameSource(&mut self) -> *mut c_void;

	// nextFrame(cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:71
	#[inline]
	fn next_frame(&mut self, frame: &mut dyn core::ToOutputArray) -> Result<()> {
		output_array_arg!(frame);
		let ret = unsafe { sys::cv_superres_FrameSource_nextFrame_const__OutputArrayR(self.as_raw_mut_Superres_FrameSource(), frame.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	// reset() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:72
	#[inline]
	fn reset(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_superres_FrameSource_reset(self.as_raw_mut_Superres_FrameSource()) }.into_result()?;
		Ok(ret)
	}
	
}

// PyrLKOpticalFlow /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:180
pub trait Superres_PyrLKOpticalFlowConst: crate::superres::Superres_DenseOpticalFlowExtConst {
	fn as_raw_Superres_PyrLKOpticalFlow(&self) -> *const c_void;

	/// ## See also
	/// setWindowSize
	// getWindowSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:184
	#[inline]
	fn get_window_size(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_getWindowSize_const(self.as_raw_Superres_PyrLKOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setMaxLevel
	// getMaxLevel() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:188
	#[inline]
	fn get_max_level(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_getMaxLevel_const(self.as_raw_Superres_PyrLKOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations
	// getIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:192
	#[inline]
	fn get_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_getIterations_const(self.as_raw_Superres_PyrLKOpticalFlow()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait Superres_PyrLKOpticalFlow: crate::superres::Superres_DenseOpticalFlowExt + crate::superres::Superres_PyrLKOpticalFlowConst {
	fn as_raw_mut_Superres_PyrLKOpticalFlow(&mut self) -> *mut c_void;

	/// ## See also
	/// setWindowSize getWindowSize
	// setWindowSize(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:186
	#[inline]
	fn set_window_size(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_setWindowSize_int(self.as_raw_mut_Superres_PyrLKOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setMaxLevel getMaxLevel
	// setMaxLevel(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:190
	#[inline]
	fn set_max_level(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_setMaxLevel_int(self.as_raw_mut_Superres_PyrLKOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// ## See also
	/// setIterations getIterations
	// setIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres/optical_flow.hpp:194
	#[inline]
	fn set_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_PyrLKOpticalFlow_setIterations_int(self.as_raw_mut_Superres_PyrLKOpticalFlow(), val) }.into_result()?;
		Ok(ret)
	}
	
}

/// Base class for Super Resolution algorithms.
/// 
/// The class is only used to define the common interface for the whole family of Super Resolution
/// algorithms.
// SuperResolution /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:87
pub trait Superres_SuperResolutionConst: core::AlgorithmTraitConst + crate::superres::Superres_FrameSourceConst {
	fn as_raw_Superres_SuperResolution(&self) -> *const c_void;

	/// Scale factor
	/// ## See also
	/// setScale
	// getScale() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:109
	#[inline]
	fn get_scale(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getScale_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Iterations count
	/// ## See also
	/// setIterations
	// getIterations() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:115
	#[inline]
	fn get_iterations(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getIterations_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Asymptotic value of steepest descent method
	/// ## See also
	/// setTau
	// getTau() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:121
	#[inline]
	fn get_tau(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getTau_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Weight parameter to balance data term and smoothness term
	/// ## See also
	/// setLambda
	// getLambda() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:127
	#[inline]
	fn get_lambda(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getLambda_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Parameter of spacial distribution in Bilateral-TV
	/// ## See also
	/// setAlpha
	// getAlpha() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:133
	#[inline]
	fn get_alpha(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getAlpha_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Kernel size of Bilateral-TV filter
	/// ## See also
	/// setKernelSize
	// getKernelSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:139
	#[inline]
	fn get_kernel_size(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getKernelSize_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Gaussian blur kernel size
	/// ## See also
	/// setBlurKernelSize
	// getBlurKernelSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:145
	#[inline]
	fn get_blur_kernel_size(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getBlurKernelSize_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Gaussian blur sigma
	/// ## See also
	/// setBlurSigma
	// getBlurSigma() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:151
	#[inline]
	fn get_blur_sigma(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getBlurSigma_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Radius of the temporal search area
	/// ## See also
	/// setTemporalAreaRadius
	// getTemporalAreaRadius() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:157
	#[inline]
	fn get_temporal_area_radius(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getTemporalAreaRadius_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Dense optical flow algorithm
	/// ## See also
	/// setOpticalFlow
	// getOpticalFlow() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:163
	#[inline]
	fn get_optical_flow(&self) -> Result<core::Ptr<dyn crate::superres::Superres_DenseOpticalFlowExt>> {
		let ret = unsafe { sys::cv_superres_SuperResolution_getOpticalFlow_const(self.as_raw_Superres_SuperResolution()) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::superres::Superres_DenseOpticalFlowExt>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

pub trait Superres_SuperResolution: core::AlgorithmTrait + crate::superres::Superres_FrameSource + crate::superres::Superres_SuperResolutionConst {
	fn as_raw_mut_Superres_SuperResolution(&mut self) -> *mut c_void;

	/// Set input frame source for Super Resolution algorithm.
	/// 
	/// ## Parameters
	/// * frameSource: Input frame source
	// setInput(const Ptr<cv::superres::FrameSource> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:94
	#[inline]
	fn set_input(&mut self, frame_source: &core::Ptr<dyn crate::superres::Superres_FrameSource>) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setInput_const_Ptr_FrameSource_R(self.as_raw_mut_Superres_SuperResolution(), frame_source.as_raw_PtrOfSuperres_FrameSource()) }.into_result()?;
		Ok(ret)
	}
	
	/// Process next frame from input and return output result.
	/// 
	/// ## Parameters
	/// * frame: Output result
	// nextFrame(cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:100
	#[inline]
	fn next_frame(&mut self, frame: &mut dyn core::ToOutputArray) -> Result<()> {
		output_array_arg!(frame);
		let ret = unsafe { sys::cv_superres_SuperResolution_nextFrame_const__OutputArrayR(self.as_raw_mut_Superres_SuperResolution(), frame.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	// reset() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:101
	#[inline]
	fn reset(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_reset(self.as_raw_mut_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Clear all inner buffers.
	// collectGarbage() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:105
	#[inline]
	fn collect_garbage(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_collectGarbage(self.as_raw_mut_Superres_SuperResolution()) }.into_result()?;
		Ok(ret)
	}
	
	/// Scale factor
	/// ## See also
	/// setScale getScale
	// setScale(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:111
	#[inline]
	fn set_scale(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setScale_int(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Iterations count
	/// ## See also
	/// setIterations getIterations
	// setIterations(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:117
	#[inline]
	fn set_iterations(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setIterations_int(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Asymptotic value of steepest descent method
	/// ## See also
	/// setTau getTau
	// setTau(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:123
	#[inline]
	fn set_tau(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setTau_double(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Weight parameter to balance data term and smoothness term
	/// ## See also
	/// setLambda getLambda
	// setLambda(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:129
	#[inline]
	fn set_lambda(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setLambda_double(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Parameter of spacial distribution in Bilateral-TV
	/// ## See also
	/// setAlpha getAlpha
	// setAlpha(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:135
	#[inline]
	fn set_alpha(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setAlpha_double(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Kernel size of Bilateral-TV filter
	/// ## See also
	/// setKernelSize getKernelSize
	// setKernelSize(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:141
	#[inline]
	fn set_kernel_size(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setKernelSize_int(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Gaussian blur kernel size
	/// ## See also
	/// setBlurKernelSize getBlurKernelSize
	// setBlurKernelSize(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:147
	#[inline]
	fn set_blur_kernel_size(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setBlurKernelSize_int(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Gaussian blur sigma
	/// ## See also
	/// setBlurSigma getBlurSigma
	// setBlurSigma(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:153
	#[inline]
	fn set_blur_sigma(&mut self, val: f64) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setBlurSigma_double(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Radius of the temporal search area
	/// ## See also
	/// setTemporalAreaRadius getTemporalAreaRadius
	// setTemporalAreaRadius(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:159
	#[inline]
	fn set_temporal_area_radius(&mut self, val: i32) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setTemporalAreaRadius_int(self.as_raw_mut_Superres_SuperResolution(), val) }.into_result()?;
		Ok(ret)
	}
	
	/// Dense optical flow algorithm
	/// ## See also
	/// setOpticalFlow getOpticalFlow
	// setOpticalFlow(const Ptr<cv::superres::DenseOpticalFlowExt> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/superres.hpp:165
	#[inline]
	fn set_optical_flow(&mut self, val: &core::Ptr<dyn crate::superres::Superres_DenseOpticalFlowExt>) -> Result<()> {
		let ret = unsafe { sys::cv_superres_SuperResolution_setOpticalFlow_const_Ptr_DenseOpticalFlowExt_R(self.as_raw_mut_Superres_SuperResolution(), val.as_raw_PtrOfSuperres_DenseOpticalFlowExt()) }.into_result()?;
		Ok(ret)
	}
	
}
