#![allow(
	unused_parens,
	clippy::excessive_precision,
	clippy::missing_safety_doc,
	clippy::not_unsafe_ptr_arg_deref,
	clippy::should_implement_trait,
	clippy::too_many_arguments,
	clippy::unused_unit,
)]
//! # OGRE 3D Visualiser
//! 
//! ovis is a simplified rendering wrapper around [ogre3d](https://www.ogre3d.org/).
//! The [Ogre terminology](https://ogrecave.github.io/ogre/api/latest/_the-_core-_objects.html) is used in the API
//! and [Ogre Script](https://ogrecave.github.io/ogre/api/latest/_scripts.html) is assumed to be used for advanced customization.
//! 
//! Besides the API you see here, there are several environment variables that control the behavior of ovis.
//! They are documented in @ref createWindow.
//! 
//! ## Loading geometry
//! 
//! You can create geometry [on the fly](@ref createTriangleMesh) or by loading Ogre `.mesh` files.
//! 
//! ### Blender
//! For converting/ creating geometry [Blender](https://www.blender.org/) is recommended.
//! - Blender 2.7x is better tested, but Blender 2.8x should work too
//! - install [blender2ogre](https://github.com/OGRECave/blender2ogre) matching your Blender version
//! - download the [Ogre MSVC SDK](https://www.ogre3d.org/download/sdk/sdk-ogre) which contains `OgreXMLConverter.exe` (in `bin/`) and set the path in the blender2ogre settings
//! - get [ogre-meshviewer](https://github.com/OGRECave/ogre-meshviewer) to enable the preview function in blender2ogre as well as for verifying the exported files
//! - in case the exported materials are not exactly how you want them, consult the [Ogre Manual](https://ogrecave.github.io/ogre/api/latest/_material-_scripts.html)
//! 
//! ### Assimp
//! When using Ogre 1.12.9 or later, enabling the Assimp plugin allows to load arbitrary geometry.
//! Simply pass `bunny.obj` instead of `bunny.mesh` as `meshname` in @ref WindowScene::createEntity.
//! 
//! You should still use ogre-meshviewer to verify that the geometry is converted correctly.
use crate::{mod_prelude::*, core, sys, types};
pub mod prelude {
	pub use { super::WindowSceneConst, super::WindowScene };
}

// ENTITY_AABB_WORLD /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:78
pub const ENTITY_AABB_WORLD: i32 = 2;
// ENTITY_ANIMBLEND_MODE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:79
pub const ENTITY_ANIMBLEND_MODE: i32 = 3;
// ENTITY_CAST_SHADOWS /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:80
pub const ENTITY_CAST_SHADOWS: i32 = 4;
// ENTITY_MATERIAL /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:76
pub const ENTITY_MATERIAL: i32 = 0;
// ENTITY_SCALE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:77
pub const ENTITY_SCALE: i32 = 1;
// MATERIAL_DIFFUSE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:66
pub const MATERIAL_DIFFUSE: i32 = 4;
// MATERIAL_EMISSIVE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:65
pub const MATERIAL_EMISSIVE: i32 = 3;
// MATERIAL_LINE_WIDTH /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:63
pub const MATERIAL_LINE_WIDTH: i32 = 1;
// MATERIAL_OPACITY /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:64
pub const MATERIAL_OPACITY: i32 = 2;
// MATERIAL_POINT_SIZE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:62
pub const MATERIAL_POINT_SIZE: i32 = 0;
// MATERIAL_TEXTURE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:68
pub const MATERIAL_TEXTURE: i32 = 5;
// MATERIAL_TEXTURE0 /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:67
pub const MATERIAL_TEXTURE0: i32 = 5;
// MATERIAL_TEXTURE1 /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:69
pub const MATERIAL_TEXTURE1: i32 = 6;
// MATERIAL_TEXTURE2 /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:70
pub const MATERIAL_TEXTURE2: i32 = 7;
// MATERIAL_TEXTURE3 /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:71
pub const MATERIAL_TEXTURE3: i32 = 8;
/// Apply anti-aliasing. The first window determines the setting for all windows.
// SCENE_AA /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:53
pub const SCENE_AA: i32 = 8;
/// allow the user to control the camera.
// SCENE_INTERACTIVE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:49
pub const SCENE_INTERACTIVE: i32 = 2;
/// Render off-screen without a window. Allows separate AA setting. Requires manual update via @ref WindowScene::update
// SCENE_OFFSCREEN /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:55
pub const SCENE_OFFSCREEN: i32 = 16;
/// the window will use a separate scene. The scene will be shared otherwise.
// SCENE_SEPARATE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:47
pub const SCENE_SEPARATE: i32 = 1;
/// Enable real-time shadows in the scene. All entities cast shadows by default. Control via @ref ENTITY_CAST_SHADOWS
// SCENE_SHADOWS /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:57
pub const SCENE_SHADOWS: i32 = 32;
/// draw coordinate system crosses for debugging
// SCENE_SHOW_CS_CROSS /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:51
pub const SCENE_SHOW_CS_CROSS: i32 = 4;
// EntityProperty /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:74
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum EntityProperty {
	ENTITY_MATERIAL = 0,
	ENTITY_SCALE = 1,
	ENTITY_AABB_WORLD = 2,
	ENTITY_ANIMBLEND_MODE = 3,
	ENTITY_CAST_SHADOWS = 4,
}

opencv_type_enum! { crate::ovis::EntityProperty }

// MaterialProperty /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:60
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum MaterialProperty {
	MATERIAL_POINT_SIZE = 0,
	MATERIAL_LINE_WIDTH = 1,
	MATERIAL_OPACITY = 2,
	MATERIAL_EMISSIVE = 3,
	MATERIAL_DIFFUSE = 4,
	MATERIAL_TEXTURE0 = 5,
	// MATERIAL_TEXTURE = 5 as isize, // duplicate discriminant
	MATERIAL_TEXTURE1 = 6,
	MATERIAL_TEXTURE2 = 7,
	MATERIAL_TEXTURE3 = 8,
}

opencv_type_enum! { crate::ovis::MaterialProperty }

// SceneSettings /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:44
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum SceneSettings {
	/// the window will use a separate scene. The scene will be shared otherwise.
	SCENE_SEPARATE = 1,
	/// allow the user to control the camera.
	SCENE_INTERACTIVE = 2,
	/// draw coordinate system crosses for debugging
	SCENE_SHOW_CS_CROSS = 4,
	/// Apply anti-aliasing. The first window determines the setting for all windows.
	SCENE_AA = 8,
	/// Render off-screen without a window. Allows separate AA setting. Requires manual update via @ref WindowScene::update
	SCENE_OFFSCREEN = 16,
	/// Enable real-time shadows in the scene. All entities cast shadows by default. Control via @ref ENTITY_CAST_SHADOWS
	SCENE_SHADOWS = 32,
}

opencv_type_enum! { crate::ovis::SceneSettings }

/// Add an additional resource location that is search for meshes, textures and materials
/// 
/// must be called before the first createWindow. If give path does not exist, retries inside
/// Ogre Media Directory.
/// ## Parameters
/// * path: folder or Zip archive.
// addResourceLocation(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:323
#[inline]
pub fn add_resource_location(path: &str) -> Result<()> {
	extern_container_arg!(path);
	let ret = unsafe { sys::cv_ovis_addResourceLocation_const_StringR(path.opencv_as_extern()) }.into_result()?;
	Ok(ret)
}

/// creates a grid
/// 
/// creates a material with the same name
/// ## Parameters
/// * name: name of the mesh
/// * size: extents of the grid
/// * segments: number of segments per side
/// 
/// ## C++ default parameters
/// * segments: Size(1,1)
// createGridMesh(const cv::String &, const cv::Size2f &, const cv::Size &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:403
#[inline]
pub fn create_grid_mesh(name: &str, size: core::Size2f, segments: core::Size) -> Result<()> {
	extern_container_arg!(name);
	let ret = unsafe { sys::cv_ovis_createGridMesh_const_StringR_const_Size2fR_const_SizeR(name.opencv_as_extern(), &size, &segments) }.into_result()?;
	Ok(ret)
}

/// create a 2D plane, X right, Y down, Z up
/// 
/// creates a material with the same name
/// ## Parameters
/// * name: name of the mesh
/// * size: size in world units
/// * image: optional texture
/// 
/// ## C++ default parameters
/// * image: noArray()
// createPlaneMesh(const cv::String &, const cv::Size2f &, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:383
#[inline]
pub fn create_plane_mesh(name: &str, size: core::Size2f, image: &dyn core::ToInputArray) -> Result<()> {
	extern_container_arg!(name);
	input_array_arg!(image);
	let ret = unsafe { sys::cv_ovis_createPlaneMesh_const_StringR_const_Size2fR_const__InputArrayR(name.opencv_as_extern(), &size, image.as_raw__InputArray()) }.into_result()?;
	Ok(ret)
}

/// creates a point cloud mesh
/// 
/// creates a material with the same name
/// ## Parameters
/// * name: name of the mesh
/// * vertices: float vector of positions
/// * colors: uchar vector of colors
/// 
/// ## C++ default parameters
/// * colors: noArray()
// createPointCloudMesh(const cv::String &, cv::InputArray, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:393
#[inline]
pub fn create_point_cloud_mesh(name: &str, vertices: &dyn core::ToInputArray, colors: &dyn core::ToInputArray) -> Result<()> {
	extern_container_arg!(name);
	input_array_arg!(vertices);
	input_array_arg!(colors);
	let ret = unsafe { sys::cv_ovis_createPointCloudMesh_const_StringR_const__InputArrayR_const__InputArrayR(name.opencv_as_extern(), vertices.as_raw__InputArray(), colors.as_raw__InputArray()) }.into_result()?;
	Ok(ret)
}

/// creates a triangle mesh from vertex-vertex or face-vertex representation
/// 
/// creates a material with the same name
/// ## Parameters
/// * name: name of the mesh
/// * vertices: float vector of positions
/// * normals: float vector of normals
/// * indices: int vector of indices
/// 
/// ## C++ default parameters
/// * normals: noArray()
/// * indices: noArray()
// createTriangleMesh(const cv::String &, cv::InputArray, cv::InputArray, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:414
#[inline]
pub fn create_triangle_mesh(name: &str, vertices: &dyn core::ToInputArray, normals: &dyn core::ToInputArray, indices: &dyn core::ToInputArray) -> Result<()> {
	extern_container_arg!(name);
	input_array_arg!(vertices);
	input_array_arg!(normals);
	input_array_arg!(indices);
	let ret = unsafe { sys::cv_ovis_createTriangleMesh_const_StringR_const__InputArrayR_const__InputArrayR_const__InputArrayR(name.opencv_as_extern(), vertices.as_raw__InputArray(), normals.as_raw__InputArray(), indices.as_raw__InputArray()) }.into_result()?;
	Ok(ret)
}

/// create a new rendering window/ viewport
/// ## Parameters
/// * title: window title
/// * size: size of the window
/// * flags: a combination of @ref SceneSettings
/// 
/// Furthermore, the behavior is controlled by the following environment variables
/// - OPENCV_OVIS_VERBOSE_LOG: print all of OGRE log output
/// - OPENCV_OVIS_RENDERSYSTEM: the name of the OGRE RenderSystem to use
/// - OPENCV_OVIS_NOVSYNC: disable VSYNC for all windows
/// 
/// ## C++ default parameters
/// * flags: SCENE_INTERACTIVE|SCENE_AA
// createWindow(const cv::String &, const cv::Size &, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:336
#[inline]
pub fn create_window(title: &str, size: core::Size, flags: i32) -> Result<core::Ptr<dyn crate::ovis::WindowScene>> {
	extern_container_arg!(title);
	let ret = unsafe { sys::cv_ovis_createWindow_const_StringR_const_SizeR_int(title.opencv_as_extern(), &size, flags) }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::ovis::WindowScene>::opencv_from_extern(ret) };
	Ok(ret)
}

/// set the shader property of a material to the given value
/// ## Parameters
/// * name: material name
/// * prop: property name
/// * value: the value
// setMaterialProperty(const cv::String &, const cv::String &, const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:373
#[inline]
pub fn set_material_property_2(name: &str, prop: &str, value: core::Scalar) -> Result<()> {
	extern_container_arg!(name);
	extern_container_arg!(prop);
	let ret = unsafe { sys::cv_ovis_setMaterialProperty_const_StringR_const_StringR_const_ScalarR(name.opencv_as_extern(), prop.opencv_as_extern(), &value) }.into_result()?;
	Ok(ret)
}

/// set the property of a material to the given value
/// ## Parameters
/// * name: material name
/// * prop: @ref MaterialProperty
/// * value: the value
// setMaterialProperty(const cv::String &, int, const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:354
#[inline]
pub fn set_material_property(name: &str, prop: i32, value: core::Scalar) -> Result<()> {
	extern_container_arg!(name);
	let ret = unsafe { sys::cv_ovis_setMaterialProperty_const_StringR_int_const_ScalarR(name.opencv_as_extern(), prop, &value) }.into_result()?;
	Ok(ret)
}

/// set the shader property of a material to the given value
/// ## Parameters
/// * name: material name
/// * prop: property name
/// * value: the value
/// 
/// ## Overloaded parameters
// setMaterialProperty(const cv::String &, int, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:357
#[inline]
pub fn set_material_property_1(name: &str, prop: i32, value: &str) -> Result<()> {
	extern_container_arg!(name);
	extern_container_arg!(value);
	let ret = unsafe { sys::cv_ovis_setMaterialProperty_const_StringR_int_const_StringR(name.opencv_as_extern(), prop, value.opencv_as_extern()) }.into_result()?;
	Ok(ret)
}

/// set the texture of a material to the given value
/// ## Parameters
/// * name: material name
/// * prop: @ref MaterialProperty
/// * value: the texture data
// setMaterialProperty(const cv::String &, int, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:365
#[inline]
pub fn set_material_texture(name: &str, prop: i32, value: &dyn core::ToInputArray) -> Result<()> {
	extern_container_arg!(name);
	input_array_arg!(value);
	let ret = unsafe { sys::cv_ovis_setMaterialProperty_const_StringR_int_const__InputArrayR(name.opencv_as_extern(), prop, value.as_raw__InputArray()) }.into_result()?;
	Ok(ret)
}

/// 
/// **Deprecated**: use setMaterialProperty
#[deprecated = "use setMaterialProperty"]
// updateTexture(const cv::String &, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:417
#[inline]
pub fn update_texture(name: &str, image: &dyn core::ToInputArray) -> Result<()> {
	extern_container_arg!(name);
	input_array_arg!(image);
	let ret = unsafe { sys::cv_ovis_updateTexture_const_StringR_const__InputArrayR(name.opencv_as_extern(), image.as_raw__InputArray()) }.into_result()?;
	Ok(ret)
}

/// update all windows and wait for keyboard event
/// 
/// ## Parameters
/// * delay: 0 is the special value that means "forever".
///        Any positive number returns after sync to blank (typically 16ms).
/// ## Returns
/// the code of the pressed key or -1 if no key was pressed
/// 
/// ## C++ default parameters
/// * delay: 0
// waitKey(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:346
#[inline]
pub fn wait_key(delay: i32) -> Result<i32> {
	let ret = unsafe { sys::cv_ovis_waitKey_int(delay) }.into_result()?;
	Ok(ret)
}

/// A 3D viewport and the associated scene
// WindowScene /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:86
pub trait WindowSceneConst {
	fn as_raw_WindowScene(&self) -> *const c_void;

}

pub trait WindowScene: crate::ovis::WindowSceneConst {
	fn as_raw_mut_WindowScene(&mut self) -> *mut c_void;

	/// set window background to custom image
	/// ## Parameters
	/// * image: 
	// setBackground(cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:94
	#[inline]
	fn set_background(&mut self, image: &dyn core::ToInputArray) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_ovis_WindowScene_setBackground_const__InputArrayR(self.as_raw_mut_WindowScene(), image.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// set window background to custom image
	/// ## Parameters
	/// * image: 
	/// 
	/// ## Overloaded parameters
	// setBackground(const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:97
	#[inline]
	fn set_background_color(&mut self, color: core::Scalar) -> Result<()> {
		let ret = unsafe { sys::cv_ovis_WindowScene_setBackground_const_ScalarR(self.as_raw_mut_WindowScene(), &color) }.into_result()?;
		Ok(ret)
	}
	
	/// enable an ordered chain of full-screen post processing effects
	/// 
	/// this way you can add distortion or SSAO effects.
	/// The effects themselves must be defined inside Ogre .compositor scripts.
	/// ## See also
	/// addResourceLocation
	/// ## Parameters
	/// * names: compositor names that will be applied in order of appearance
	// setCompositors(const std::vector<String> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:107
	#[inline]
	fn set_compositors(&mut self, names: &core::Vector<String>) -> Result<()> {
		let ret = unsafe { sys::cv_ovis_WindowScene_setCompositors_const_vector_String_R(self.as_raw_mut_WindowScene(), names.as_raw_VectorOfString()) }.into_result()?;
		Ok(ret)
	}
	
	/// place an entity of a mesh in the scene
	/// 
	/// the mesh needs to be created beforehand. Either programmatically
	/// by e.g. @ref createPointCloudMesh or by placing an Ogre .mesh file in a resource location.
	/// ## See also
	/// addResourceLocation
	/// ## Parameters
	/// * name: entity name
	/// * meshname: mesh name
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	// createEntity(const cv::String &, const cv::String &, cv::InputArray, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:120
	#[inline]
	fn create_entity(&mut self, name: &str, meshname: &str, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray) -> Result<()> {
		extern_container_arg!(name);
		extern_container_arg!(meshname);
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_createEntity_const_StringR_const_StringR_const__InputArrayR_const__InputArrayR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), meshname.opencv_as_extern(), tvec.as_raw__InputArray(), rot.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// remove an entity from the scene
	/// ## Parameters
	/// * name: entity name
	// removeEntity(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:127
	#[inline]
	fn remove_entity(&mut self, name: &str) -> Result<()> {
		extern_container_arg!(name);
		let ret = unsafe { sys::cv_ovis_WindowScene_removeEntity_const_StringR(self.as_raw_mut_WindowScene(), name.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// set the property of an entity to the given value
	/// ## Parameters
	/// * name: entity name
	/// * prop: @ref EntityProperty
	/// * value: the value
	// setEntityProperty(const cv::String &, int, const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:135
	#[inline]
	fn set_entity_property(&mut self, name: &str, prop: i32, value: core::Scalar) -> Result<()> {
		extern_container_arg!(name);
		let ret = unsafe { sys::cv_ovis_WindowScene_setEntityProperty_const_StringR_int_const_ScalarR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), prop, &value) }.into_result()?;
		Ok(ret)
	}
	
	/// set the property of an entity to the given value
	/// ## Parameters
	/// * name: entity name
	/// * prop: @ref EntityProperty
	/// * value: the value
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * sub_entity_idx: -1
	// setEntityProperty(const cv::String &, int, const cv::String &, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:138
	#[inline]
	fn set_entity_property_1(&mut self, name: &str, prop: i32, value: &str, sub_entity_idx: i32) -> Result<()> {
		extern_container_arg!(name);
		extern_container_arg!(value);
		let ret = unsafe { sys::cv_ovis_WindowScene_setEntityProperty_const_StringR_int_const_StringR_int(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), prop, value.opencv_as_extern(), sub_entity_idx) }.into_result()?;
		Ok(ret)
	}
	
	/// get the property of an entity
	/// ## Parameters
	/// * name: entity name
	/// * prop: @ref EntityProperty
	/// * value: the value
	// getEntityProperty(const cv::String &, int, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:147
	#[inline]
	fn get_entity_property(&mut self, name: &str, prop: i32, value: &mut dyn core::ToOutputArray) -> Result<()> {
		extern_container_arg!(name);
		output_array_arg!(value);
		let ret = unsafe { sys::cv_ovis_WindowScene_getEntityProperty_const_StringR_int_const__OutputArrayR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), prop, value.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// convenience method to visualize a camera position
	/// 
	/// ## Parameters
	/// * name: entity name
	/// * K: intrinsic matrix
	/// * imsize: image size
	/// * zFar: far plane in camera coordinates
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// * color: line color
	/// ## Returns
	/// the extents of the Frustum at far plane, where the top left corner denotes the principal
	/// point offset
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	/// * color: Scalar::all(1)
	// createCameraEntity(const cv::String &, cv::InputArray, const cv::Size &, float, cv::InputArray, cv::InputArray, const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:162
	#[inline]
	fn create_camera_entity(&mut self, name: &str, k: &dyn core::ToInputArray, imsize: core::Size, z_far: f32, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray, color: core::Scalar) -> Result<core::Rect2d> {
		extern_container_arg!(name);
		input_array_arg!(k);
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_createCameraEntity_const_StringR_const__InputArrayR_const_SizeR_float_const__InputArrayR_const__InputArrayR_const_ScalarR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), k.as_raw__InputArray(), &imsize, z_far, tvec.as_raw__InputArray(), rot.as_raw__InputArray(), &color) }.into_result()?;
		Ok(ret)
	}
	
	/// creates a point light in the scene
	/// ## Parameters
	/// * name: entity name
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// * diffuseColor: 
	/// * specularColor: 
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	/// * diffuse_color: Scalar::all(1)
	/// * specular_color: Scalar::all(1)
	// createLightEntity(const cv::String &, cv::InputArray, cv::InputArray, const cv::Scalar &, const cv::Scalar &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:175
	#[inline]
	fn create_light_entity(&mut self, name: &str, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray, diffuse_color: core::Scalar, specular_color: core::Scalar) -> Result<()> {
		extern_container_arg!(name);
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_createLightEntity_const_StringR_const__InputArrayR_const__InputArrayR_const_ScalarR_const_ScalarR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), tvec.as_raw__InputArray(), rot.as_raw__InputArray(), &diffuse_color, &specular_color) }.into_result()?;
		Ok(ret)
	}
	
	/// update entity pose by transformation in the parent coordinate space. (pre-rotation)
	/// ## Parameters
	/// * name: entity name
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	// updateEntityPose(const cv::String &, cv::InputArray, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:186
	#[inline]
	fn update_entity_pose(&mut self, name: &str, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray) -> Result<()> {
		extern_container_arg!(name);
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_updateEntityPose_const_StringR_const__InputArrayR_const__InputArrayR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), tvec.as_raw__InputArray(), rot.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// set entity pose in the world coordinate space.
	/// ## Parameters
	/// * name: enitity name
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// * invert: use the inverse of the given pose
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	/// * invert: false
	// setEntityPose(const cv::String &, cv::InputArray, cv::InputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:196
	#[inline]
	fn set_entity_pose(&mut self, name: &str, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray, invert: bool) -> Result<()> {
		extern_container_arg!(name);
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_setEntityPose_const_StringR_const__InputArrayR_const__InputArrayR_bool(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), tvec.as_raw__InputArray(), rot.as_raw__InputArray(), invert) }.into_result()?;
		Ok(ret)
	}
	
	/// Retrieves the current pose of an entity
	/// ## Parameters
	/// * name: entity name
	/// * R: 3x3 rotation matrix
	/// * tvec: translation vector
	/// * invert: return the inverted pose
	/// 
	/// ## C++ default parameters
	/// * r: noArray()
	/// * tvec: noArray()
	/// * invert: false
	// getEntityPose(const cv::String &, cv::OutputArray, cv::OutputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:206
	#[inline]
	fn get_entity_pose(&mut self, name: &str, r: &mut dyn core::ToOutputArray, tvec: &mut dyn core::ToOutputArray, invert: bool) -> Result<()> {
		extern_container_arg!(name);
		output_array_arg!(r);
		output_array_arg!(tvec);
		let ret = unsafe { sys::cv_ovis_WindowScene_getEntityPose_const_StringR_const__OutputArrayR_const__OutputArrayR_bool(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), r.as_raw__OutputArray(), tvec.as_raw__OutputArray(), invert) }.into_result()?;
		Ok(ret)
	}
	
	/// get a list of available entity animations
	/// ## Parameters
	/// * name: entity name
	/// * out: the animation names
	// getEntityAnimations(const cv::String &, std::vector<String> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:214
	#[inline]
	fn get_entity_animations(&mut self, name: &str, out: &mut core::Vector<String>) -> Result<()> {
		extern_container_arg!(name);
		let ret = unsafe { sys::cv_ovis_WindowScene_getEntityAnimations_const_StringR_vector_String_R(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), out.as_raw_mut_VectorOfString()) }.into_result()?;
		Ok(ret)
	}
	
	/// play entity animation
	/// ## Parameters
	/// * name: entity name
	/// * animname: animation name
	/// * loop: enable or disable animation loop
	/// ## See also
	/// getEntityAnimations
	/// 
	/// ## C++ default parameters
	/// * loop_: true
	// playEntityAnimation(const cv::String &, const cv::String &, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:223
	#[inline]
	fn play_entity_animation(&mut self, name: &str, animname: &str, loop_: bool) -> Result<()> {
		extern_container_arg!(name);
		extern_container_arg!(animname);
		let ret = unsafe { sys::cv_ovis_WindowScene_playEntityAnimation_const_StringR_const_StringR_bool(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), animname.opencv_as_extern(), loop_) }.into_result()?;
		Ok(ret)
	}
	
	/// stop entity animation
	/// ## Parameters
	/// * name: enitity name
	/// * animname: animation name
	// stopEntityAnimation(const cv::String &, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:231
	#[inline]
	fn stop_entity_animation(&mut self, name: &str, animname: &str) -> Result<()> {
		extern_container_arg!(name);
		extern_container_arg!(animname);
		let ret = unsafe { sys::cv_ovis_WindowScene_stopEntityAnimation_const_StringR_const_StringR(self.as_raw_mut_WindowScene(), name.opencv_as_extern(), animname.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// read back the image generated by the last call to @ref waitKey
	// getScreenshot(cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:236
	#[inline]
	fn get_screenshot(&mut self, frame: &mut dyn core::ToOutputArray) -> Result<()> {
		output_array_arg!(frame);
		let ret = unsafe { sys::cv_ovis_WindowScene_getScreenshot_const__OutputArrayR(self.as_raw_mut_WindowScene(), frame.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// read back the texture of an active compositor
	/// ## Parameters
	/// * compname: name of the compositor
	/// * texname: name of the texture inside the compositor
	/// * mrtIndex: if texture is a MRT, specifies the attachment
	/// * out: the texture contents
	/// 
	/// ## C++ default parameters
	/// * mrt_index: 0
	// getCompositorTexture(const cv::String &, const cv::String &, cv::OutputArray, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:245
	#[inline]
	fn get_compositor_texture(&mut self, compname: &str, texname: &str, out: &mut dyn core::ToOutputArray, mrt_index: i32) -> Result<()> {
		extern_container_arg!(compname);
		extern_container_arg!(texname);
		output_array_arg!(out);
		let ret = unsafe { sys::cv_ovis_WindowScene_getCompositorTexture_const_StringR_const_StringR_const__OutputArrayR_int(self.as_raw_mut_WindowScene(), compname.opencv_as_extern(), texname.opencv_as_extern(), out.as_raw__OutputArray(), mrt_index) }.into_result()?;
		Ok(ret)
	}
	
	/// get the depth for the current frame.
	/// 
	/// return the per pixel distance to the camera in world units
	// getDepth(cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:253
	#[inline]
	fn get_depth(&mut self, depth: &mut dyn core::ToOutputArray) -> Result<()> {
		output_array_arg!(depth);
		let ret = unsafe { sys::cv_ovis_WindowScene_getDepth_const__OutputArrayR(self.as_raw_mut_WindowScene(), depth.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// convenience method to force the "up" axis to stay fixed
	/// 
	/// works with both programmatic changes and SCENE_INTERACTIVE
	/// ## Parameters
	/// * useFixed: whether to enforce the fixed yaw axis
	/// * up: the axis to be fixed
	/// 
	/// ## C++ default parameters
	/// * up: noArray()
	// fixCameraYawAxis(bool, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:262
	#[inline]
	fn fix_camera_yaw_axis(&mut self, use_fixed: bool, up: &dyn core::ToInputArray) -> Result<()> {
		input_array_arg!(up);
		let ret = unsafe { sys::cv_ovis_WindowScene_fixCameraYawAxis_bool_const__InputArrayR(self.as_raw_mut_WindowScene(), use_fixed, up.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Sets the current camera pose
	/// ## Parameters
	/// * rot: @ref Rodrigues vector or 3x3 rotation matrix
	/// * tvec: translation
	/// * invert: use the inverse of the given pose
	/// 
	/// ## C++ default parameters
	/// * tvec: noArray()
	/// * rot: noArray()
	/// * invert: false
	// setCameraPose(cv::InputArray, cv::InputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:270
	#[inline]
	fn set_camera_pose(&mut self, tvec: &dyn core::ToInputArray, rot: &dyn core::ToInputArray, invert: bool) -> Result<()> {
		input_array_arg!(tvec);
		input_array_arg!(rot);
		let ret = unsafe { sys::cv_ovis_WindowScene_setCameraPose_const__InputArrayR_const__InputArrayR_bool(self.as_raw_mut_WindowScene(), tvec.as_raw__InputArray(), rot.as_raw__InputArray(), invert) }.into_result()?;
		Ok(ret)
	}
	
	/// convenience method to orient the camera to a specific entity
	/// ## Parameters
	/// * target: entity name
	/// * offset: offset from entity centre
	/// 
	/// ## C++ default parameters
	/// * offset: noArray()
	// setCameraLookAt(const cv::String &, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:278
	#[inline]
	fn set_camera_look_at(&mut self, target: &str, offset: &dyn core::ToInputArray) -> Result<()> {
		extern_container_arg!(target);
		input_array_arg!(offset);
		let ret = unsafe { sys::cv_ovis_WindowScene_setCameraLookAt_const_StringR_const__InputArrayR(self.as_raw_mut_WindowScene(), target.opencv_as_extern(), offset.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// convenience method to orient an entity to a specific entity.
	/// If target is an empty string the entity looks at the given offset point
	/// ## Parameters
	/// * origin: entity to make look at
	/// * target: name of target entity
	/// * offset: offset from entity centre
	/// 
	/// ## C++ default parameters
	/// * offset: noArray()
	// setEntityLookAt(const cv::String &, const cv::String &, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:287
	#[inline]
	fn set_entity_look_at(&mut self, origin: &str, target: &str, offset: &dyn core::ToInputArray) -> Result<()> {
		extern_container_arg!(origin);
		extern_container_arg!(target);
		input_array_arg!(offset);
		let ret = unsafe { sys::cv_ovis_WindowScene_setEntityLookAt_const_StringR_const_StringR_const__InputArrayR(self.as_raw_mut_WindowScene(), origin.opencv_as_extern(), target.opencv_as_extern(), offset.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Retrieves the current camera pose
	/// ## Parameters
	/// * R: 3x3 rotation matrix
	/// * tvec: translation vector
	/// * invert: return the inverted pose
	/// 
	/// ## C++ default parameters
	/// * r: noArray()
	/// * tvec: noArray()
	/// * invert: false
	// getCameraPose(cv::OutputArray, cv::OutputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:295
	#[inline]
	fn get_camera_pose(&mut self, r: &mut dyn core::ToOutputArray, tvec: &mut dyn core::ToOutputArray, invert: bool) -> Result<()> {
		output_array_arg!(r);
		output_array_arg!(tvec);
		let ret = unsafe { sys::cv_ovis_WindowScene_getCameraPose_const__OutputArrayR_const__OutputArrayR_bool(self.as_raw_mut_WindowScene(), r.as_raw__OutputArray(), tvec.as_raw__OutputArray(), invert) }.into_result()?;
		Ok(ret)
	}
	
	/// set intrinsics of the camera
	/// 
	/// ## Parameters
	/// * K: intrinsic matrix or noArray(). If noArray() is specified, imsize
	/// is ignored and zNear/ zFar can be set separately.
	/// * imsize: image size
	/// * zNear: near clip distance or -1 to keep the current
	/// * zFar: far clip distance or -1 to keep the current
	/// 
	/// ## C++ default parameters
	/// * z_near: -1
	/// * z_far: -1
	// setCameraIntrinsics(cv::InputArray, const cv::Size &, float, float) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:307
	#[inline]
	fn set_camera_intrinsics(&mut self, k: &dyn core::ToInputArray, imsize: core::Size, z_near: f32, z_far: f32) -> Result<()> {
		input_array_arg!(k);
		let ret = unsafe { sys::cv_ovis_WindowScene_setCameraIntrinsics_const__InputArrayR_const_SizeR_float_float(self.as_raw_mut_WindowScene(), k.as_raw__InputArray(), &imsize, z_near, z_far) }.into_result()?;
		Ok(ret)
	}
	
	/// render this window, but do not swap buffers. Automatically called by @ref ovis::waitKey
	// update() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/ovis.hpp:313
	#[inline]
	fn update(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_ovis_WindowScene_update(self.as_raw_mut_WindowScene()) }.into_result()?;
		Ok(ret)
	}
	
}
