#![allow(
	unused_parens,
	clippy::excessive_precision,
	clippy::missing_safety_doc,
	clippy::not_unsafe_ptr_arg_deref,
	clippy::should_implement_trait,
	clippy::too_many_arguments,
	clippy::unused_unit,
)]
//! # Object Detection
//! 
//! Haar Feature-based Cascade Classifier for Object Detection
//! ----------------------------------------------------------
//! 
//! The object detector described below has been initially proposed by Paul Viola [Viola01](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Viola01) and
//! improved by Rainer Lienhart [Lienhart02](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Lienhart02) .
//! 
//! First, a classifier (namely a *cascade of boosted classifiers working with haar-like features*) is
//! trained with a few hundred sample views of a particular object (i.e., a face or a car), called
//! positive examples, that are scaled to the same size (say, 20x20), and negative examples - arbitrary
//! images of the same size.
//! 
//! After a classifier is trained, it can be applied to a region of interest (of the same size as used
//! during the training) in an input image. The classifier outputs a "1" if the region is likely to show
//! the object (i.e., face/car), and "0" otherwise. To search for the object in the whole image one can
//! move the search window across the image and check every location using the classifier. The
//! classifier is designed so that it can be easily "resized" in order to be able to find the objects of
//! interest at different sizes, which is more efficient than resizing the image itself. So, to find an
//! object of an unknown size in the image the scan procedure should be done several times at different
//! scales.
//! 
//! The word "cascade" in the classifier name means that the resultant classifier consists of several
//! simpler classifiers (*stages*) that are applied subsequently to a region of interest until at some
//! stage the candidate is rejected or all the stages are passed. The word "boosted" means that the
//! classifiers at every stage of the cascade are complex themselves and they are built out of basic
//! classifiers using one of four different boosting techniques (weighted voting). Currently Discrete
//! Adaboost, Real Adaboost, Gentle Adaboost and Logitboost are supported. The basic classifiers are
//! decision-tree classifiers with at least 2 leaves. Haar-like features are the input to the basic
//! classifiers, and are calculated as described below. The current algorithm uses the following
//! Haar-like features:
//! 
//! ![image](https://docs.opencv.org/4.5.4/haarfeatures.png)
//! 
//! The feature used in a particular classifier is specified by its shape (1a, 2b etc.), position within
//! the region of interest and the scale (this scale is not the same as the scale used at the detection
//! stage, though these two scales are multiplied). For example, in the case of the third line feature
//! (2c) the response is calculated as the difference between the sum of image pixels under the
//! rectangle covering the whole feature (including the two white stripes and the black stripe in the
//! middle) and the sum of the image pixels under the black stripe multiplied by 3 in order to
//! compensate for the differences in the size of areas. The sums of pixel values over a rectangular
//! regions are calculated rapidly using integral images (see below and the integral description).
//! 
//! To see the object detector at work, have a look at the facedetect demo:
//! <https://github.com/opencv/opencv/tree/master/samples/cpp/dbt_face_detection.cpp>
//! 
//! The following reference is for the detection part only. There is a separate application called
//! opencv_traincascade that can train a cascade of boosted classifiers from a set of samples.
//! 
//! 
//! Note: In the new C++ interface it is also possible to use LBP (local binary pattern) features in
//! addition to Haar-like features. .. [Viola01] Paul Viola and Michael J. Jones. Rapid Object Detection
//! using a Boosted Cascade of Simple Features. IEEE CVPR, 2001. The paper is available online at
//! <http://research.microsoft.com/en-us/um/people/viola/Pubs/Detect/violaJones_CVPR2001.pdf>
//!    # C API
use crate::{mod_prelude::*, core, sys, types};
pub mod prelude {
	pub use { super::SimilarRectsTraitConst, super::SimilarRectsTrait, super::BaseCascadeClassifier_MaskGeneratorConst, super::BaseCascadeClassifier_MaskGenerator, super::BaseCascadeClassifierConst, super::BaseCascadeClassifier, super::CascadeClassifierTraitConst, super::CascadeClassifierTrait, super::DetectionROITraitConst, super::DetectionROITrait, super::HOGDescriptorTraitConst, super::HOGDescriptorTrait, super::QRCodeDetectorTraitConst, super::QRCodeDetectorTrait, super::DetectionBasedTracker_ParametersTraitConst, super::DetectionBasedTracker_ParametersTrait, super::DetectionBasedTracker_IDetectorConst, super::DetectionBasedTracker_IDetector, super::DetectionBasedTracker_ExtObjectTraitConst, super::DetectionBasedTracker_ExtObjectTrait, super::DetectionBasedTrackerTraitConst, super::DetectionBasedTrackerTrait, super::FaceDetectorYNConst, super::FaceDetectorYN, super::FaceRecognizerSFConst, super::FaceRecognizerSF };
}

// CASCADE_DO_CANNY_PRUNING /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:168
pub const CASCADE_DO_CANNY_PRUNING: i32 = 1;
// CASCADE_DO_ROUGH_SEARCH /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:171
pub const CASCADE_DO_ROUGH_SEARCH: i32 = 8;
// CASCADE_FIND_BIGGEST_OBJECT /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:170
pub const CASCADE_FIND_BIGGEST_OBJECT: i32 = 4;
// CASCADE_SCALE_IMAGE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:169
pub const CASCADE_SCALE_IMAGE: i32 = 2;
/// Default nlevels value.
// DEFAULT_NLEVELS /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:377
pub const HOGDescriptor_DEFAULT_NLEVELS: i32 = 64;
// ObjectStatus /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:138
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum DetectionBasedTracker_ObjectStatus {
	DETECTED_NOT_SHOWN_YET = 0,
	DETECTED = 1,
	DETECTED_TEMPORARY_LOST = 2,
	WRONG_OBJECT = 3,
}

opencv_type_enum! { crate::objdetect::DetectionBasedTracker_ObjectStatus }

/// Definition of distance used for calculating the distance between two face features
// DisType /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:92
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum FaceRecognizerSF_DisType {
	FR_COSINE = 0,
	FR_NORM_L2 = 1,
}

opencv_type_enum! { crate::objdetect::FaceRecognizerSF_DisType }

// DescriptorStorageFormat /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:379
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum HOGDescriptor_DescriptorStorageFormat {
	DESCR_FORMAT_COL_BY_COL = 0,
	DESCR_FORMAT_ROW_BY_ROW = 1,
}

opencv_type_enum! { crate::objdetect::HOGDescriptor_DescriptorStorageFormat }

// HistogramNormType /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:375
#[repr(C)]
#[derive(Copy, Clone, Debug, PartialEq)]
pub enum HOGDescriptor_HistogramNormType {
	/// Default histogramNormType
	L2Hys = 0,
}

opencv_type_enum! { crate::objdetect::HOGDescriptor_HistogramNormType }

// createFaceDetectionMaskGenerator() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:340
#[inline]
pub fn create_face_detection_mask_generator() -> Result<core::Ptr<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>> {
	let ret = unsafe { sys::cv_createFaceDetectionMaskGenerator() }.into_result()?;
	let ret = unsafe { core::Ptr::<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>::opencv_from_extern(ret) };
	Ok(ret)
}

/// ## C++ default parameters
/// * detect_threshold: 0.0
/// * win_det_size: Size(64,128)
// groupRectangles_meanshift(std::vector<Rect> &, std::vector<double> &, std::vector<double> &, double, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:162
#[inline]
pub fn group_rectangles_meanshift(rect_list: &mut core::Vector<core::Rect>, found_weights: &mut core::Vector<f64>, found_scales: &mut core::Vector<f64>, detect_threshold: f64, win_det_size: core::Size) -> Result<()> {
	let ret = unsafe { sys::cv_groupRectangles_meanshift_vector_Rect_R_vector_double_R_vector_double_R_double_Size(rect_list.as_raw_mut_VectorOfRect(), found_weights.as_raw_mut_VectorOff64(), found_scales.as_raw_mut_VectorOff64(), detect_threshold, win_det_size.opencv_as_extern()) }.into_result()?;
	Ok(ret)
}

/// Groups the object candidate rectangles.
/// 
/// ## Parameters
/// * rectList: Input/output vector of rectangles. Output vector includes retained and grouped
/// rectangles. (The Python list is not modified in place.)
/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a
/// group of rectangles to retain it.
/// * eps: Relative difference between sides of the rectangles to merge them into a group.
/// 
/// The function is a wrapper for the generic function partition . It clusters all the input rectangles
/// using the rectangle equivalence criteria that combines rectangles with similar sizes and similar
/// locations. The similarity is defined by eps. When eps=0 , no clustering is done at all. If
/// ![inline formula](https://latex.codecogs.com/png.latex?%5Ctexttt%7Beps%7D%5Crightarrow%20%2B%5Cinf) , all the rectangles are put in one cluster. Then, the small
/// clusters containing less than or equal to groupThreshold rectangles are rejected. In each other
/// cluster, the average rectangle is computed and put into the output rectangle list.
/// 
/// ## C++ default parameters
/// * eps: 0.2
// groupRectangles(std::vector<Rect> &, int, double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:151
#[inline]
pub fn group_rectangles(rect_list: &mut core::Vector<core::Rect>, group_threshold: i32, eps: f64) -> Result<()> {
	let ret = unsafe { sys::cv_groupRectangles_vector_Rect_R_int_double(rect_list.as_raw_mut_VectorOfRect(), group_threshold, eps) }.into_result()?;
	Ok(ret)
}

/// Groups the object candidate rectangles.
/// 
/// ## Parameters
/// * rectList: Input/output vector of rectangles. Output vector includes retained and grouped
/// rectangles. (The Python list is not modified in place.)
/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a
/// group of rectangles to retain it.
/// * eps: Relative difference between sides of the rectangles to merge them into a group.
/// 
/// The function is a wrapper for the generic function partition . It clusters all the input rectangles
/// using the rectangle equivalence criteria that combines rectangles with similar sizes and similar
/// locations. The similarity is defined by eps. When eps=0 , no clustering is done at all. If
/// ![inline formula](https://latex.codecogs.com/png.latex?%5Ctexttt%7Beps%7D%5Crightarrow%20%2B%5Cinf) , all the rectangles are put in one cluster. Then, the small
/// clusters containing less than or equal to groupThreshold rectangles are rejected. In each other
/// cluster, the average rectangle is computed and put into the output rectangle list.
/// 
/// ## Overloaded parameters
// groupRectangles(std::vector<Rect> &, int, double, std::vector<int> *, std::vector<double> *) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:156
#[inline]
pub fn group_rectangles_levelweights(rect_list: &mut core::Vector<core::Rect>, group_threshold: i32, eps: f64, weights: &mut core::Vector<i32>, level_weights: &mut core::Vector<f64>) -> Result<()> {
	let ret = unsafe { sys::cv_groupRectangles_vector_Rect_R_int_double_vector_int_X_vector_double_X(rect_list.as_raw_mut_VectorOfRect(), group_threshold, eps, weights.as_raw_mut_VectorOfi32(), level_weights.as_raw_mut_VectorOff64()) }.into_result()?;
	Ok(ret)
}

/// Groups the object candidate rectangles.
/// 
/// ## Parameters
/// * rectList: Input/output vector of rectangles. Output vector includes retained and grouped
/// rectangles. (The Python list is not modified in place.)
/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a
/// group of rectangles to retain it.
/// * eps: Relative difference between sides of the rectangles to merge them into a group.
/// 
/// The function is a wrapper for the generic function partition . It clusters all the input rectangles
/// using the rectangle equivalence criteria that combines rectangles with similar sizes and similar
/// locations. The similarity is defined by eps. When eps=0 , no clustering is done at all. If
/// ![inline formula](https://latex.codecogs.com/png.latex?%5Ctexttt%7Beps%7D%5Crightarrow%20%2B%5Cinf) , all the rectangles are put in one cluster. Then, the small
/// clusters containing less than or equal to groupThreshold rectangles are rejected. In each other
/// cluster, the average rectangle is computed and put into the output rectangle list.
/// 
/// ## Overloaded parameters
/// 
/// ## C++ default parameters
/// * eps: 0.2
// groupRectangles(std::vector<Rect> &, std::vector<int> &, int, double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:153
#[inline]
pub fn group_rectangles_weights(rect_list: &mut core::Vector<core::Rect>, weights: &mut core::Vector<i32>, group_threshold: i32, eps: f64) -> Result<()> {
	let ret = unsafe { sys::cv_groupRectangles_vector_Rect_R_vector_int_R_int_double(rect_list.as_raw_mut_VectorOfRect(), weights.as_raw_mut_VectorOfi32(), group_threshold, eps) }.into_result()?;
	Ok(ret)
}

/// Groups the object candidate rectangles.
/// 
/// ## Parameters
/// * rectList: Input/output vector of rectangles. Output vector includes retained and grouped
/// rectangles. (The Python list is not modified in place.)
/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a
/// group of rectangles to retain it.
/// * eps: Relative difference between sides of the rectangles to merge them into a group.
/// 
/// The function is a wrapper for the generic function partition . It clusters all the input rectangles
/// using the rectangle equivalence criteria that combines rectangles with similar sizes and similar
/// locations. The similarity is defined by eps. When eps=0 , no clustering is done at all. If
/// ![inline formula](https://latex.codecogs.com/png.latex?%5Ctexttt%7Beps%7D%5Crightarrow%20%2B%5Cinf) , all the rectangles are put in one cluster. Then, the small
/// clusters containing less than or equal to groupThreshold rectangles are rejected. In each other
/// cluster, the average rectangle is computed and put into the output rectangle list.
/// 
/// ## Overloaded parameters
/// 
/// ## C++ default parameters
/// * eps: 0.2
// groupRectangles(std::vector<Rect> &, std::vector<int> &, std::vector<double> &, int, double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:159
#[inline]
pub fn group_rectangles_levels(rect_list: &mut core::Vector<core::Rect>, reject_levels: &mut core::Vector<i32>, level_weights: &mut core::Vector<f64>, group_threshold: i32, eps: f64) -> Result<()> {
	let ret = unsafe { sys::cv_groupRectangles_vector_Rect_R_vector_int_R_vector_double_R_int_double(rect_list.as_raw_mut_VectorOfRect(), reject_levels.as_raw_mut_VectorOfi32(), level_weights.as_raw_mut_VectorOff64(), group_threshold, eps) }.into_result()?;
	Ok(ret)
}

// BaseCascadeClassifier /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:174
pub trait BaseCascadeClassifierConst: core::AlgorithmTraitConst {
	fn as_raw_BaseCascadeClassifier(&self) -> *const c_void;

	// empty() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:178
	#[inline]
	fn empty(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_empty_const(self.as_raw_BaseCascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// isOldFormatCascade() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:202
	#[inline]
	fn is_old_format_cascade(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_isOldFormatCascade_const(self.as_raw_BaseCascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// getOriginalWindowSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:203
	#[inline]
	fn get_original_window_size(&self) -> Result<core::Size> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_getOriginalWindowSize_const(self.as_raw_BaseCascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// getFeatureType() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:204
	#[inline]
	fn get_feature_type(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_getFeatureType_const(self.as_raw_BaseCascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait BaseCascadeClassifier: core::AlgorithmTrait + crate::objdetect::BaseCascadeClassifierConst {
	fn as_raw_mut_BaseCascadeClassifier(&mut self) -> *mut c_void;

	// load(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:179
	#[inline]
	fn load(&mut self, filename: &str) -> Result<bool> {
		extern_container_arg!(filename);
		let ret = unsafe { sys::cv_BaseCascadeClassifier_load_const_StringR(self.as_raw_mut_BaseCascadeClassifier(), filename.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, double, int, int, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:180
	#[inline]
	fn detect_multi_scale(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_BaseCascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_double_int_int_Size_Size(self.as_raw_mut_BaseCascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, std::vector<int> &, double, int, int, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:186
	#[inline]
	fn detect_multi_scale_num(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, num_detections: &mut core::Vector<i32>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_BaseCascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_vector_int_R_double_int_int_Size_Size(self.as_raw_mut_BaseCascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), num_detections.as_raw_mut_VectorOfi32(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, std::vector<int> &, std::vector<double> &, double, int, int, cv::Size, cv::Size, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:193
	#[inline]
	fn detect_multi_scale_levels(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, reject_levels: &mut core::Vector<i32>, level_weights: &mut core::Vector<f64>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size, output_reject_levels: bool) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_BaseCascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_vector_int_R_vector_double_R_double_int_int_Size_Size_bool(self.as_raw_mut_BaseCascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), reject_levels.as_raw_mut_VectorOfi32(), level_weights.as_raw_mut_VectorOff64(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern(), output_reject_levels) }.into_result()?;
		Ok(ret)
	}
	
	// getOldCascade() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:205
	#[inline]
	fn get_old_cascade(&mut self) -> Result<*mut c_void> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_getOldCascade(self.as_raw_mut_BaseCascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// setMaskGenerator(const Ptr<cv::BaseCascadeClassifier::MaskGenerator> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:214
	#[inline]
	fn set_mask_generator(&mut self, mask_generator: &core::Ptr<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>) -> Result<()> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_setMaskGenerator_const_Ptr_MaskGenerator_R(self.as_raw_mut_BaseCascadeClassifier(), mask_generator.as_raw_PtrOfBaseCascadeClassifier_MaskGenerator()) }.into_result()?;
		Ok(ret)
	}
	
	// getMaskGenerator() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:215
	#[inline]
	fn get_mask_generator(&mut self) -> Result<core::Ptr<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_getMaskGenerator(self.as_raw_mut_BaseCascadeClassifier()) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

// MaskGenerator /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:207
pub trait BaseCascadeClassifier_MaskGeneratorConst {
	fn as_raw_BaseCascadeClassifier_MaskGenerator(&self) -> *const c_void;

}

pub trait BaseCascadeClassifier_MaskGenerator: crate::objdetect::BaseCascadeClassifier_MaskGeneratorConst {
	fn as_raw_mut_BaseCascadeClassifier_MaskGenerator(&mut self) -> *mut c_void;

	// generateMask(const cv::Mat &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:211
	#[inline]
	fn generate_mask(&mut self, src: &core::Mat) -> Result<core::Mat> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_MaskGenerator_generateMask_const_MatR(self.as_raw_mut_BaseCascadeClassifier_MaskGenerator(), src.as_raw_Mat()) }.into_result()?;
		let ret = unsafe { core::Mat::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	// initializeMask(const cv::Mat &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:212
	#[inline]
	fn initialize_mask(&mut self, unnamed: &core::Mat) -> Result<()> {
		let ret = unsafe { sys::cv_BaseCascadeClassifier_MaskGenerator_initializeMask_const_MatR(self.as_raw_mut_BaseCascadeClassifier_MaskGenerator(), unnamed.as_raw_Mat()) }.into_result()?;
		Ok(ret)
	}
	
}

/// @example samples/cpp/facedetect.cpp
/// This program demonstrates usage of the Cascade classifier class
/// \image html Cascade_Classifier_Tutorial_Result_Haar.jpg "Sample screenshot" width=321 height=254
/// 
/// Cascade classifier class for object detection.
// CascadeClassifier /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:224
pub trait CascadeClassifierTraitConst {
	fn as_raw_CascadeClassifier(&self) -> *const c_void;

	/// Checks whether the classifier has been loaded.
	// empty() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:236
	#[inline]
	fn empty(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_CascadeClassifier_empty_const(self.as_raw_CascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// isOldFormatCascade() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:327
	#[inline]
	fn is_old_format_cascade(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_CascadeClassifier_isOldFormatCascade_const(self.as_raw_CascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// getOriginalWindowSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:328
	#[inline]
	fn get_original_window_size(&self) -> Result<core::Size> {
		let ret = unsafe { sys::cv_CascadeClassifier_getOriginalWindowSize_const(self.as_raw_CascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// getFeatureType() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:329
	#[inline]
	fn get_feature_type(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_CascadeClassifier_getFeatureType_const(self.as_raw_CascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait CascadeClassifierTrait: crate::objdetect::CascadeClassifierTraitConst {
	fn as_raw_mut_CascadeClassifier(&mut self) -> *mut c_void;

	// cc /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:337
	#[inline]
	fn cc(&mut self) -> core::Ptr<dyn crate::objdetect::BaseCascadeClassifier> {
		let ret = unsafe { sys::cv_CascadeClassifier_getPropCc(self.as_raw_mut_CascadeClassifier()) };
		let ret = unsafe { core::Ptr::<dyn crate::objdetect::BaseCascadeClassifier>::opencv_from_extern(ret) };
		ret
	}
	
	// cc /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:337
	#[inline]
	fn set_cc(&mut self, mut val: core::Ptr<dyn crate::objdetect::BaseCascadeClassifier>) {
		let ret = unsafe { sys::cv_CascadeClassifier_setPropCc_Ptr_BaseCascadeClassifier_(self.as_raw_mut_CascadeClassifier(), val.as_raw_mut_PtrOfBaseCascadeClassifier()) };
		ret
	}
	
	/// Loads a classifier from a file.
	/// 
	/// ## Parameters
	/// * filename: Name of the file from which the classifier is loaded. The file may contain an old
	/// HAAR classifier trained by the haartraining application or a new cascade classifier trained by the
	/// traincascade application.
	// load(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:243
	#[inline]
	fn load(&mut self, filename: &str) -> Result<bool> {
		extern_container_arg!(filename);
		let ret = unsafe { sys::cv_CascadeClassifier_load_const_StringR(self.as_raw_mut_CascadeClassifier(), filename.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// Reads a classifier from a FileStorage node.
	/// 
	/// 
	/// Note: The file may contain a new cascade classifier (trained traincascade application) only.
	// read(const cv::FileNode &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:248
	#[inline]
	fn read(&mut self, node: &core::FileNode) -> Result<bool> {
		let ret = unsafe { sys::cv_CascadeClassifier_read_const_FileNodeR(self.as_raw_mut_CascadeClassifier(), node.as_raw_FileNode()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects objects of different sizes in the input image. The detected objects are returned as a list
	/// of rectangles.
	/// 
	/// ## Parameters
	/// * image: Matrix of the type CV_8U containing an image where objects are detected.
	/// * objects: Vector of rectangles where each rectangle contains the detected object, the
	/// rectangles may be partially outside the original image.
	/// * scaleFactor: Parameter specifying how much the image size is reduced at each image scale.
	/// * minNeighbors: Parameter specifying how many neighbors each candidate rectangle should have
	/// to retain it.
	/// * flags: Parameter with the same meaning for an old cascade as in the function
	/// cvHaarDetectObjects. It is not used for a new cascade.
	/// * minSize: Minimum possible object size. Objects smaller than that are ignored.
	/// * maxSize: Maximum possible object size. Objects larger than that are ignored. If `maxSize == minSize` model is evaluated on single scale.
	/// 
	/// The function is parallelized with the TBB library.
	/// 
	/// 
	/// Note:
	///    *   (Python) A face detection example using cascade classifiers can be found at
	///        opencv_source_code/samples/python/facedetect.py
	/// 
	/// ## C++ default parameters
	/// * scale_factor: 1.1
	/// * min_neighbors: 3
	/// * flags: 0
	/// * min_size: Size()
	/// * max_size: Size()
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, double, int, int, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:270
	#[inline]
	fn detect_multi_scale(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_CascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_double_int_int_Size_Size(self.as_raw_mut_CascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects objects of different sizes in the input image. The detected objects are returned as a list
	/// of rectangles.
	/// 
	/// ## Parameters
	/// * image: Matrix of the type CV_8U containing an image where objects are detected.
	/// * objects: Vector of rectangles where each rectangle contains the detected object, the
	/// rectangles may be partially outside the original image.
	/// * scaleFactor: Parameter specifying how much the image size is reduced at each image scale.
	/// * minNeighbors: Parameter specifying how many neighbors each candidate rectangle should have
	/// to retain it.
	/// * flags: Parameter with the same meaning for an old cascade as in the function
	/// cvHaarDetectObjects. It is not used for a new cascade.
	/// * minSize: Minimum possible object size. Objects smaller than that are ignored.
	/// * maxSize: Maximum possible object size. Objects larger than that are ignored. If `maxSize == minSize` model is evaluated on single scale.
	/// 
	/// The function is parallelized with the TBB library.
	/// 
	/// 
	/// Note:
	///    *   (Python) A face detection example using cascade classifiers can be found at
	///        opencv_source_code/samples/python/facedetect.py
	/// 
	/// ## Overloaded parameters
	/// 
	/// * image: Matrix of the type CV_8U containing an image where objects are detected.
	/// * objects: Vector of rectangles where each rectangle contains the detected object, the
	///    rectangles may be partially outside the original image.
	/// * numDetections: Vector of detection numbers for the corresponding objects. An object's number
	///    of detections is the number of neighboring positively classified rectangles that were joined
	///    together to form the object.
	/// * scaleFactor: Parameter specifying how much the image size is reduced at each image scale.
	/// * minNeighbors: Parameter specifying how many neighbors each candidate rectangle should have
	///    to retain it.
	/// * flags: Parameter with the same meaning for an old cascade as in the function
	///    cvHaarDetectObjects. It is not used for a new cascade.
	/// * minSize: Minimum possible object size. Objects smaller than that are ignored.
	/// * maxSize: Maximum possible object size. Objects larger than that are ignored. If `maxSize == minSize` model is evaluated on single scale.
	/// 
	/// ## C++ default parameters
	/// * scale_factor: 1.1
	/// * min_neighbors: 3
	/// * flags: 0
	/// * min_size: Size()
	/// * max_size: Size()
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, std::vector<int> &, double, int, int, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:292
	#[inline]
	fn detect_multi_scale2(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, num_detections: &mut core::Vector<i32>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_CascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_vector_int_R_double_int_int_Size_Size(self.as_raw_mut_CascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), num_detections.as_raw_mut_VectorOfi32(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects objects of different sizes in the input image. The detected objects are returned as a list
	/// of rectangles.
	/// 
	/// ## Parameters
	/// * image: Matrix of the type CV_8U containing an image where objects are detected.
	/// * objects: Vector of rectangles where each rectangle contains the detected object, the
	/// rectangles may be partially outside the original image.
	/// * scaleFactor: Parameter specifying how much the image size is reduced at each image scale.
	/// * minNeighbors: Parameter specifying how many neighbors each candidate rectangle should have
	/// to retain it.
	/// * flags: Parameter with the same meaning for an old cascade as in the function
	/// cvHaarDetectObjects. It is not used for a new cascade.
	/// * minSize: Minimum possible object size. Objects smaller than that are ignored.
	/// * maxSize: Maximum possible object size. Objects larger than that are ignored. If `maxSize == minSize` model is evaluated on single scale.
	/// 
	/// The function is parallelized with the TBB library.
	/// 
	/// 
	/// Note:
	///    *   (Python) A face detection example using cascade classifiers can be found at
	///        opencv_source_code/samples/python/facedetect.py
	/// 
	/// ## Overloaded parameters
	/// 
	///    This function allows you to retrieve the final stage decision certainty of classification.
	///    For this, one needs to set `outputRejectLevels` on true and provide the `rejectLevels` and `levelWeights` parameter.
	///    For each resulting detection, `levelWeights` will then contain the certainty of classification at the final stage.
	///    This value can then be used to separate strong from weaker classifications.
	/// 
	///    A code sample on how to use it efficiently can be found below:
	///    ```ignore
	///    Mat img;
	///    vector<double> weights;
	///    vector<int> levels;
	///    vector<Rect> detections;
	///    CascadeClassifier model("/path/to/your/model.xml");
	///    model.detectMultiScale(img, detections, levels, weights, 1.1, 3, 0, Size(), Size(), true);
	///    cerr << "Detection " << detections[0] << " with weight " << weights[0] << endl;
	///    ```
	/// 
	/// 
	/// ## C++ default parameters
	/// * scale_factor: 1.1
	/// * min_neighbors: 3
	/// * flags: 0
	/// * min_size: Size()
	/// * max_size: Size()
	/// * output_reject_levels: false
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, std::vector<int> &, std::vector<double> &, double, int, int, cv::Size, cv::Size, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:317
	#[inline]
	fn detect_multi_scale3(&mut self, image: &dyn core::ToInputArray, objects: &mut core::Vector<core::Rect>, reject_levels: &mut core::Vector<i32>, level_weights: &mut core::Vector<f64>, scale_factor: f64, min_neighbors: i32, flags: i32, min_size: core::Size, max_size: core::Size, output_reject_levels: bool) -> Result<()> {
		input_array_arg!(image);
		let ret = unsafe { sys::cv_CascadeClassifier_detectMultiScale_const__InputArrayR_vector_Rect_R_vector_int_R_vector_double_R_double_int_int_Size_Size_bool(self.as_raw_mut_CascadeClassifier(), image.as_raw__InputArray(), objects.as_raw_mut_VectorOfRect(), reject_levels.as_raw_mut_VectorOfi32(), level_weights.as_raw_mut_VectorOff64(), scale_factor, min_neighbors, flags, min_size.opencv_as_extern(), max_size.opencv_as_extern(), output_reject_levels) }.into_result()?;
		Ok(ret)
	}
	
	// getOldCascade() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:330
	#[inline]
	fn get_old_cascade(&mut self) -> Result<*mut c_void> {
		let ret = unsafe { sys::cv_CascadeClassifier_getOldCascade(self.as_raw_mut_CascadeClassifier()) }.into_result()?;
		Ok(ret)
	}
	
	// setMaskGenerator(const Ptr<BaseCascadeClassifier::MaskGenerator> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:334
	#[inline]
	fn set_mask_generator(&mut self, mask_generator: &core::Ptr<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>) -> Result<()> {
		let ret = unsafe { sys::cv_CascadeClassifier_setMaskGenerator_const_Ptr_MaskGenerator_R(self.as_raw_mut_CascadeClassifier(), mask_generator.as_raw_PtrOfBaseCascadeClassifier_MaskGenerator()) }.into_result()?;
		Ok(ret)
	}
	
	// getMaskGenerator() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:335
	#[inline]
	fn get_mask_generator(&mut self) -> Result<core::Ptr<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>> {
		let ret = unsafe { sys::cv_CascadeClassifier_getMaskGenerator(self.as_raw_mut_CascadeClassifier()) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::objdetect::BaseCascadeClassifier_MaskGenerator>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

/// @example samples/cpp/facedetect.cpp
/// This program demonstrates usage of the Cascade classifier class
/// \image html Cascade_Classifier_Tutorial_Result_Haar.jpg "Sample screenshot" width=321 height=254
/// 
/// Cascade classifier class for object detection.
// CascadeClassifier /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:224
pub struct CascadeClassifier {
	ptr: *mut c_void
}

opencv_type_boxed! { CascadeClassifier }

impl Drop for CascadeClassifier {
	fn drop(&mut self) {
		extern "C" { fn cv_CascadeClassifier_delete(instance: *mut c_void); }
		unsafe { cv_CascadeClassifier_delete(self.as_raw_mut_CascadeClassifier()) };
	}
}

unsafe impl Send for CascadeClassifier {}

impl crate::objdetect::CascadeClassifierTraitConst for CascadeClassifier {
	#[inline] fn as_raw_CascadeClassifier(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::CascadeClassifierTrait for CascadeClassifier {
	#[inline] fn as_raw_mut_CascadeClassifier(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl CascadeClassifier {
	// CascadeClassifier() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:227
	#[inline]
	pub fn default() -> Result<crate::objdetect::CascadeClassifier> {
		let ret = unsafe { sys::cv_CascadeClassifier_CascadeClassifier() }.into_result()?;
		let ret = unsafe { crate::objdetect::CascadeClassifier::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Loads a classifier from a file.
	/// 
	/// ## Parameters
	/// * filename: Name of the file from which the classifier is loaded.
	// CascadeClassifier(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:232
	#[inline]
	pub fn new(filename: &str) -> Result<crate::objdetect::CascadeClassifier> {
		extern_container_arg!(filename);
		let ret = unsafe { sys::cv_CascadeClassifier_CascadeClassifier_const_StringR(filename.opencv_as_extern()) }.into_result()?;
		let ret = unsafe { crate::objdetect::CascadeClassifier::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	// convert(const cv::String &, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:332
	#[inline]
	pub fn convert(oldcascade: &str, newcascade: &str) -> Result<bool> {
		extern_container_arg!(oldcascade);
		extern_container_arg!(newcascade);
		let ret = unsafe { sys::cv_CascadeClassifier_convert_const_StringR_const_StringR(oldcascade.opencv_as_extern(), newcascade.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
}

// DetectionBasedTracker /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:57
pub trait DetectionBasedTrackerTraitConst {
	fn as_raw_DetectionBasedTracker(&self) -> *const c_void;

	// getParameters() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:131
	#[inline]
	fn get_parameters(&self) -> Result<crate::objdetect::DetectionBasedTracker_Parameters> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_getParameters_const(self.as_raw_DetectionBasedTracker()) }.into_result()?;
		let ret = unsafe { crate::objdetect::DetectionBasedTracker_Parameters::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	// getObjects(std::vector<cv::Rect> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:135
	#[inline]
	fn get_objects(&self, result: &mut core::Vector<core::Rect>) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_getObjects_const_vector_Rect_R(self.as_raw_DetectionBasedTracker(), result.as_raw_mut_VectorOfRect()) }.into_result()?;
		Ok(ret)
	}
	
	// getObjects(std::vector<ExtObject> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:155
	#[inline]
	fn get_objects_1(&self, result: &mut core::Vector<crate::objdetect::DetectionBasedTracker_ExtObject>) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_getObjects_const_vector_ExtObject_R(self.as_raw_DetectionBasedTracker(), result.as_raw_mut_VectorOfDetectionBasedTracker_ExtObject()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait DetectionBasedTrackerTrait: crate::objdetect::DetectionBasedTrackerTraitConst {
	fn as_raw_mut_DetectionBasedTracker(&mut self) -> *mut c_void;

	// run() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:124
	#[inline]
	fn run(&mut self) -> Result<bool> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_run(self.as_raw_mut_DetectionBasedTracker()) }.into_result()?;
		Ok(ret)
	}
	
	// stop() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:125
	#[inline]
	fn stop(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_stop(self.as_raw_mut_DetectionBasedTracker()) }.into_result()?;
		Ok(ret)
	}
	
	// resetTracking() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:126
	#[inline]
	fn reset_tracking(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_resetTracking(self.as_raw_mut_DetectionBasedTracker()) }.into_result()?;
		Ok(ret)
	}
	
	// process(const cv::Mat &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:128
	#[inline]
	fn process(&mut self, image_gray: &core::Mat) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_process_const_MatR(self.as_raw_mut_DetectionBasedTracker(), image_gray.as_raw_Mat()) }.into_result()?;
		Ok(ret)
	}
	
	// setParameters(const cv::DetectionBasedTracker::Parameters &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:130
	#[inline]
	fn set_parameters(&mut self, params: &crate::objdetect::DetectionBasedTracker_Parameters) -> Result<bool> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_setParameters_const_ParametersR(self.as_raw_mut_DetectionBasedTracker(), params.as_raw_DetectionBasedTracker_Parameters()) }.into_result()?;
		Ok(ret)
	}
	
	// addObject(const cv::Rect &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:158
	#[inline]
	fn add_object(&mut self, location: core::Rect) -> Result<i32> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_addObject_const_RectR(self.as_raw_mut_DetectionBasedTracker(), &location) }.into_result()?;
		Ok(ret)
	}
	
}

// DetectionBasedTracker /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:57
pub struct DetectionBasedTracker {
	ptr: *mut c_void
}

opencv_type_boxed! { DetectionBasedTracker }

impl Drop for DetectionBasedTracker {
	fn drop(&mut self) {
		extern "C" { fn cv_DetectionBasedTracker_delete(instance: *mut c_void); }
		unsafe { cv_DetectionBasedTracker_delete(self.as_raw_mut_DetectionBasedTracker()) };
	}
}

unsafe impl Send for DetectionBasedTracker {}

impl crate::objdetect::DetectionBasedTrackerTraitConst for DetectionBasedTracker {
	#[inline] fn as_raw_DetectionBasedTracker(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::DetectionBasedTrackerTrait for DetectionBasedTracker {
	#[inline] fn as_raw_mut_DetectionBasedTracker(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl DetectionBasedTracker {
	// DetectionBasedTracker(cv::Ptr<IDetector>, cv::Ptr<IDetector>, const cv::DetectionBasedTracker::Parameters &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:121
	#[inline]
	pub fn new(mut main_detector: core::Ptr<dyn crate::objdetect::DetectionBasedTracker_IDetector>, mut tracking_detector: core::Ptr<dyn crate::objdetect::DetectionBasedTracker_IDetector>, params: &crate::objdetect::DetectionBasedTracker_Parameters) -> Result<crate::objdetect::DetectionBasedTracker> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_DetectionBasedTracker_Ptr_IDetector__Ptr_IDetector__const_ParametersR(main_detector.as_raw_mut_PtrOfDetectionBasedTracker_IDetector(), tracking_detector.as_raw_mut_PtrOfDetectionBasedTracker_IDetector(), params.as_raw_DetectionBasedTracker_Parameters()) }.into_result()?;
		let ret = unsafe { crate::objdetect::DetectionBasedTracker::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

// ExtObject /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:145
pub trait DetectionBasedTracker_ExtObjectTraitConst {
	fn as_raw_DetectionBasedTracker_ExtObject(&self) -> *const c_void;

	// id /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:147
	#[inline]
	fn id(&self) -> i32 {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_getPropId_const(self.as_raw_DetectionBasedTracker_ExtObject()) };
		ret
	}
	
	// location /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:148
	#[inline]
	fn location(&self) -> core::Rect {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_getPropLocation_const(self.as_raw_DetectionBasedTracker_ExtObject()) };
		ret
	}
	
	// status /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:149
	#[inline]
	fn status(&self) -> crate::objdetect::DetectionBasedTracker_ObjectStatus {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_getPropStatus_const(self.as_raw_DetectionBasedTracker_ExtObject()) };
		ret
	}
	
}

pub trait DetectionBasedTracker_ExtObjectTrait: crate::objdetect::DetectionBasedTracker_ExtObjectTraitConst {
	fn as_raw_mut_DetectionBasedTracker_ExtObject(&mut self) -> *mut c_void;

	// id /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:147
	#[inline]
	fn set_id(&mut self, val: i32) {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_setPropId_int(self.as_raw_mut_DetectionBasedTracker_ExtObject(), val) };
		ret
	}
	
	// location /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:148
	#[inline]
	fn set_location(&mut self, val: core::Rect) {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_setPropLocation_Rect(self.as_raw_mut_DetectionBasedTracker_ExtObject(), val.opencv_as_extern()) };
		ret
	}
	
	// status /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:149
	#[inline]
	fn set_status(&mut self, val: crate::objdetect::DetectionBasedTracker_ObjectStatus) {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_setPropStatus_ObjectStatus(self.as_raw_mut_DetectionBasedTracker_ExtObject(), val) };
		ret
	}
	
}

// ExtObject /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:145
pub struct DetectionBasedTracker_ExtObject {
	ptr: *mut c_void
}

opencv_type_boxed! { DetectionBasedTracker_ExtObject }

impl Drop for DetectionBasedTracker_ExtObject {
	fn drop(&mut self) {
		extern "C" { fn cv_DetectionBasedTracker_ExtObject_delete(instance: *mut c_void); }
		unsafe { cv_DetectionBasedTracker_ExtObject_delete(self.as_raw_mut_DetectionBasedTracker_ExtObject()) };
	}
}

unsafe impl Send for DetectionBasedTracker_ExtObject {}

impl crate::objdetect::DetectionBasedTracker_ExtObjectTraitConst for DetectionBasedTracker_ExtObject {
	#[inline] fn as_raw_DetectionBasedTracker_ExtObject(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::DetectionBasedTracker_ExtObjectTrait for DetectionBasedTracker_ExtObject {
	#[inline] fn as_raw_mut_DetectionBasedTracker_ExtObject(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl DetectionBasedTracker_ExtObject {
	// ExtObject(int, cv::Rect, cv::DetectionBasedTracker::ObjectStatus) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:150
	#[inline]
	pub fn new(_id: i32, _location: core::Rect, _status: crate::objdetect::DetectionBasedTracker_ObjectStatus) -> Result<crate::objdetect::DetectionBasedTracker_ExtObject> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_ExtObject_ExtObject_int_Rect_ObjectStatus(_id, _location.opencv_as_extern(), _status) }.into_result()?;
		let ret = unsafe { crate::objdetect::DetectionBasedTracker_ExtObject::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

// IDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:68
pub trait DetectionBasedTracker_IDetectorConst {
	fn as_raw_DetectionBasedTracker_IDetector(&self) -> *const c_void;

	// getMinObjectSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:88
	#[inline]
	fn get_min_object_size(&self) -> Result<core::Size> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_getMinObjectSize_const(self.as_raw_DetectionBasedTracker_IDetector()) }.into_result()?;
		Ok(ret)
	}
	
	// getMaxObjectSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:92
	#[inline]
	fn get_max_object_size(&self) -> Result<core::Size> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_getMaxObjectSize_const(self.as_raw_DetectionBasedTracker_IDetector()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait DetectionBasedTracker_IDetector: crate::objdetect::DetectionBasedTracker_IDetectorConst {
	fn as_raw_mut_DetectionBasedTracker_IDetector(&mut self) -> *mut c_void;

	// detect(const cv::Mat &, std::vector<cv::Rect> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:78
	#[inline]
	fn detect(&mut self, image: &core::Mat, objects: &mut core::Vector<core::Rect>) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_detect_const_MatR_vector_Rect_R(self.as_raw_mut_DetectionBasedTracker_IDetector(), image.as_raw_Mat(), objects.as_raw_mut_VectorOfRect()) }.into_result()?;
		Ok(ret)
	}
	
	// setMinObjectSize(const cv::Size &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:80
	#[inline]
	fn set_min_object_size(&mut self, min: core::Size) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_setMinObjectSize_const_SizeR(self.as_raw_mut_DetectionBasedTracker_IDetector(), &min) }.into_result()?;
		Ok(ret)
	}
	
	// setMaxObjectSize(const cv::Size &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:84
	#[inline]
	fn set_max_object_size(&mut self, max: core::Size) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_setMaxObjectSize_const_SizeR(self.as_raw_mut_DetectionBasedTracker_IDetector(), &max) }.into_result()?;
		Ok(ret)
	}
	
	// getScaleFactor() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:96
	#[inline]
	fn get_scale_factor(&mut self) -> Result<f32> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_getScaleFactor(self.as_raw_mut_DetectionBasedTracker_IDetector()) }.into_result()?;
		Ok(ret)
	}
	
	// setScaleFactor(float) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:100
	#[inline]
	fn set_scale_factor(&mut self, value: f32) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_setScaleFactor_float(self.as_raw_mut_DetectionBasedTracker_IDetector(), value) }.into_result()?;
		Ok(ret)
	}
	
	// getMinNeighbours() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:104
	#[inline]
	fn get_min_neighbours(&mut self) -> Result<i32> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_getMinNeighbours(self.as_raw_mut_DetectionBasedTracker_IDetector()) }.into_result()?;
		Ok(ret)
	}
	
	// setMinNeighbours(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:108
	#[inline]
	fn set_min_neighbours(&mut self, value: i32) -> Result<()> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_IDetector_setMinNeighbours_int(self.as_raw_mut_DetectionBasedTracker_IDetector(), value) }.into_result()?;
		Ok(ret)
	}
	
}

// Parameters /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:60
pub trait DetectionBasedTracker_ParametersTraitConst {
	fn as_raw_DetectionBasedTracker_Parameters(&self) -> *const c_void;

	// maxTrackLifetime /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:62
	#[inline]
	fn max_track_lifetime(&self) -> i32 {
		let ret = unsafe { sys::cv_DetectionBasedTracker_Parameters_getPropMaxTrackLifetime_const(self.as_raw_DetectionBasedTracker_Parameters()) };
		ret
	}
	
	// minDetectionPeriod /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:63
	#[inline]
	fn min_detection_period(&self) -> i32 {
		let ret = unsafe { sys::cv_DetectionBasedTracker_Parameters_getPropMinDetectionPeriod_const(self.as_raw_DetectionBasedTracker_Parameters()) };
		ret
	}
	
}

pub trait DetectionBasedTracker_ParametersTrait: crate::objdetect::DetectionBasedTracker_ParametersTraitConst {
	fn as_raw_mut_DetectionBasedTracker_Parameters(&mut self) -> *mut c_void;

	// maxTrackLifetime /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:62
	#[inline]
	fn set_max_track_lifetime(&mut self, val: i32) {
		let ret = unsafe { sys::cv_DetectionBasedTracker_Parameters_setPropMaxTrackLifetime_int(self.as_raw_mut_DetectionBasedTracker_Parameters(), val) };
		ret
	}
	
	// minDetectionPeriod /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:63
	#[inline]
	fn set_min_detection_period(&mut self, val: i32) {
		let ret = unsafe { sys::cv_DetectionBasedTracker_Parameters_setPropMinDetectionPeriod_int(self.as_raw_mut_DetectionBasedTracker_Parameters(), val) };
		ret
	}
	
}

// Parameters /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:60
pub struct DetectionBasedTracker_Parameters {
	ptr: *mut c_void
}

opencv_type_boxed! { DetectionBasedTracker_Parameters }

impl Drop for DetectionBasedTracker_Parameters {
	fn drop(&mut self) {
		extern "C" { fn cv_DetectionBasedTracker_Parameters_delete(instance: *mut c_void); }
		unsafe { cv_DetectionBasedTracker_Parameters_delete(self.as_raw_mut_DetectionBasedTracker_Parameters()) };
	}
}

unsafe impl Send for DetectionBasedTracker_Parameters {}

impl crate::objdetect::DetectionBasedTracker_ParametersTraitConst for DetectionBasedTracker_Parameters {
	#[inline] fn as_raw_DetectionBasedTracker_Parameters(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::DetectionBasedTracker_ParametersTrait for DetectionBasedTracker_Parameters {
	#[inline] fn as_raw_mut_DetectionBasedTracker_Parameters(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl DetectionBasedTracker_Parameters {
	// Parameters() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/detection_based_tracker.hpp:65
	#[inline]
	pub fn default() -> Result<crate::objdetect::DetectionBasedTracker_Parameters> {
		let ret = unsafe { sys::cv_DetectionBasedTracker_Parameters_Parameters() }.into_result()?;
		let ret = unsafe { crate::objdetect::DetectionBasedTracker_Parameters::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

/// struct for detection region of interest (ROI)
// DetectionROI /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:345
pub trait DetectionROITraitConst {
	fn as_raw_DetectionROI(&self) -> *const c_void;

	/// scale(size) of the bounding box
	// scale /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:348
	#[inline]
	fn scale(&self) -> f64 {
		let ret = unsafe { sys::cv_DetectionROI_getPropScale_const(self.as_raw_DetectionROI()) };
		ret
	}
	
	/// set of requested locations to be evaluated
	// locations /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:350
	#[inline]
	fn locations(&self) -> core::Vector<core::Point> {
		let ret = unsafe { sys::cv_DetectionROI_getPropLocations_const(self.as_raw_DetectionROI()) };
		let ret = unsafe { core::Vector::<core::Point>::opencv_from_extern(ret) };
		ret
	}
	
	/// vector that will contain confidence values for each location
	// confidences /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:352
	#[inline]
	fn confidences(&self) -> core::Vector<f64> {
		let ret = unsafe { sys::cv_DetectionROI_getPropConfidences_const(self.as_raw_DetectionROI()) };
		let ret = unsafe { core::Vector::<f64>::opencv_from_extern(ret) };
		ret
	}
	
}

pub trait DetectionROITrait: crate::objdetect::DetectionROITraitConst {
	fn as_raw_mut_DetectionROI(&mut self) -> *mut c_void;

	/// scale(size) of the bounding box
	// scale /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:348
	#[inline]
	fn set_scale(&mut self, val: f64) {
		let ret = unsafe { sys::cv_DetectionROI_setPropScale_double(self.as_raw_mut_DetectionROI(), val) };
		ret
	}
	
	/// set of requested locations to be evaluated
	// locations /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:350
	#[inline]
	fn set_locations(&mut self, mut val: core::Vector<core::Point>) {
		let ret = unsafe { sys::cv_DetectionROI_setPropLocations_vector_Point_(self.as_raw_mut_DetectionROI(), val.as_raw_mut_VectorOfPoint()) };
		ret
	}
	
	/// vector that will contain confidence values for each location
	// confidences /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:352
	#[inline]
	fn set_confidences(&mut self, mut val: core::Vector<f64>) {
		let ret = unsafe { sys::cv_DetectionROI_setPropConfidences_vector_double_(self.as_raw_mut_DetectionROI(), val.as_raw_mut_VectorOff64()) };
		ret
	}
	
}

/// struct for detection region of interest (ROI)
// DetectionROI /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:345
pub struct DetectionROI {
	ptr: *mut c_void
}

opencv_type_boxed! { DetectionROI }

impl Drop for DetectionROI {
	fn drop(&mut self) {
		extern "C" { fn cv_DetectionROI_delete(instance: *mut c_void); }
		unsafe { cv_DetectionROI_delete(self.as_raw_mut_DetectionROI()) };
	}
}

unsafe impl Send for DetectionROI {}

impl crate::objdetect::DetectionROITraitConst for DetectionROI {
	#[inline] fn as_raw_DetectionROI(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::DetectionROITrait for DetectionROI {
	#[inline] fn as_raw_mut_DetectionROI(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl DetectionROI {
}

/// DNN-based face detector, model download link: https://github.com/ShiqiYu/libfacedetection.train/tree/master/tasks/task1/onnx.
// FaceDetectorYN /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:18
pub trait FaceDetectorYNConst {
	fn as_raw_FaceDetectorYN(&self) -> *const c_void;

}

pub trait FaceDetectorYN: crate::objdetect::FaceDetectorYNConst {
	fn as_raw_mut_FaceDetectorYN(&mut self) -> *mut c_void;

	/// Set the size for the network input, which overwrites the input size of creating model. Call this method when the size of input image does not match the input size when creating model
	/// 
	/// ## Parameters
	/// * input_size: the size of the input image
	// setInputSize(const cv::Size &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:27
	#[inline]
	fn set_input_size(&mut self, input_size: core::Size) -> Result<()> {
		let ret = unsafe { sys::cv_FaceDetectorYN_setInputSize_const_SizeR(self.as_raw_mut_FaceDetectorYN(), &input_size) }.into_result()?;
		Ok(ret)
	}
	
	// getInputSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:29
	#[inline]
	fn get_input_size(&mut self) -> Result<core::Size> {
		let ret = unsafe { sys::cv_FaceDetectorYN_getInputSize(self.as_raw_mut_FaceDetectorYN()) }.into_result()?;
		Ok(ret)
	}
	
	/// Set the score threshold to filter out bounding boxes of score less than the given value
	/// 
	/// ## Parameters
	/// * score_threshold: threshold for filtering out bounding boxes
	// setScoreThreshold(float) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:35
	#[inline]
	fn set_score_threshold(&mut self, score_threshold: f32) -> Result<()> {
		let ret = unsafe { sys::cv_FaceDetectorYN_setScoreThreshold_float(self.as_raw_mut_FaceDetectorYN(), score_threshold) }.into_result()?;
		Ok(ret)
	}
	
	// getScoreThreshold() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:37
	#[inline]
	fn get_score_threshold(&mut self) -> Result<f32> {
		let ret = unsafe { sys::cv_FaceDetectorYN_getScoreThreshold(self.as_raw_mut_FaceDetectorYN()) }.into_result()?;
		Ok(ret)
	}
	
	/// Set the Non-maximum-suppression threshold to suppress bounding boxes that have IoU greater than the given value
	/// 
	/// ## Parameters
	/// * nms_threshold: threshold for NMS operation
	// setNMSThreshold(float) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:43
	#[inline]
	fn set_nms_threshold(&mut self, nms_threshold: f32) -> Result<()> {
		let ret = unsafe { sys::cv_FaceDetectorYN_setNMSThreshold_float(self.as_raw_mut_FaceDetectorYN(), nms_threshold) }.into_result()?;
		Ok(ret)
	}
	
	// getNMSThreshold() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:45
	#[inline]
	fn get_nms_threshold(&mut self) -> Result<f32> {
		let ret = unsafe { sys::cv_FaceDetectorYN_getNMSThreshold(self.as_raw_mut_FaceDetectorYN()) }.into_result()?;
		Ok(ret)
	}
	
	/// Set the number of bounding boxes preserved before NMS
	/// 
	/// ## Parameters
	/// * top_k: the number of bounding boxes to preserve from top rank based on score
	// setTopK(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:51
	#[inline]
	fn set_top_k(&mut self, top_k: i32) -> Result<()> {
		let ret = unsafe { sys::cv_FaceDetectorYN_setTopK_int(self.as_raw_mut_FaceDetectorYN(), top_k) }.into_result()?;
		Ok(ret)
	}
	
	// getTopK() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:53
	#[inline]
	fn get_top_k(&mut self) -> Result<i32> {
		let ret = unsafe { sys::cv_FaceDetectorYN_getTopK(self.as_raw_mut_FaceDetectorYN()) }.into_result()?;
		Ok(ret)
	}
	
	/// A simple interface to detect face from given image
	/// 
	/// ## Parameters
	/// * image: an image to detect
	/// * faces: detection results stored in a cv::Mat
	// detect(cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:60
	#[inline]
	fn detect(&mut self, image: &dyn core::ToInputArray, faces: &mut dyn core::ToOutputArray) -> Result<i32> {
		input_array_arg!(image);
		output_array_arg!(faces);
		let ret = unsafe { sys::cv_FaceDetectorYN_detect_const__InputArrayR_const__OutputArrayR(self.as_raw_mut_FaceDetectorYN(), image.as_raw__InputArray(), faces.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
}

impl dyn FaceDetectorYN + '_ {
	/// Creates an instance of this class with given parameters
	/// 
	/// ## Parameters
	/// * model: the path to the requested model
	/// * config: the path to the config file for compability, which is not requested for ONNX models
	/// * input_size: the size of the input image
	/// * score_threshold: the threshold to filter out bounding boxes of score smaller than the given value
	/// * nms_threshold: the threshold to suppress bounding boxes of IoU bigger than the given value
	/// * top_k: keep top K bboxes before NMS
	/// * backend_id: the id of backend
	/// * target_id: the id of target device
	/// 
	/// ## C++ default parameters
	/// * score_threshold: 0.9f
	/// * nms_threshold: 0.3f
	/// * top_k: 5000
	/// * backend_id: 0
	/// * target_id: 0
	// create(const cv::String &, const cv::String &, const cv::Size &, float, float, int, int, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:73
	#[inline]
	pub fn create(model: &str, config: &str, input_size: core::Size, score_threshold: f32, nms_threshold: f32, top_k: i32, backend_id: i32, target_id: i32) -> Result<core::Ptr<dyn crate::objdetect::FaceDetectorYN>> {
		extern_container_arg!(model);
		extern_container_arg!(config);
		let ret = unsafe { sys::cv_FaceDetectorYN_create_const_StringR_const_StringR_const_SizeR_float_float_int_int_int(model.opencv_as_extern(), config.opencv_as_extern(), &input_size, score_threshold, nms_threshold, top_k, backend_id, target_id) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::objdetect::FaceDetectorYN>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}
/// DNN-based face recognizer, model download link: https://drive.google.com/file/d/1ClK9WiB492c5OZFKveF3XiHCejoOxINW/view.
// FaceRecognizerSF /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:85
pub trait FaceRecognizerSFConst {
	fn as_raw_FaceRecognizerSF(&self) -> *const c_void;

	/// Aligning image to put face on the standard position
	/// ## Parameters
	/// * src_img: input image
	/// * face_box: the detection result used for indicate face in input image
	/// * aligned_img: output aligned image
	// alignCrop(cv::InputArray, cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:99
	#[inline]
	fn align_crop(&self, src_img: &dyn core::ToInputArray, face_box: &dyn core::ToInputArray, aligned_img: &mut dyn core::ToOutputArray) -> Result<()> {
		input_array_arg!(src_img);
		input_array_arg!(face_box);
		output_array_arg!(aligned_img);
		let ret = unsafe { sys::cv_FaceRecognizerSF_alignCrop_const_const__InputArrayR_const__InputArrayR_const__OutputArrayR(self.as_raw_FaceRecognizerSF(), src_img.as_raw__InputArray(), face_box.as_raw__InputArray(), aligned_img.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Calculating the distance between two face features
	/// ## Parameters
	/// * _face_feature1: the first input feature
	/// * _face_feature2: the second input feature of the same size and the same type as _face_feature1
	/// * dis_type: defining the similarity with optional values "FR_OSINE" or "FR_NORM_L2"
	/// 
	/// ## C++ default parameters
	/// * dis_type: FaceRecognizerSF::FR_COSINE
	// match(cv::InputArray, cv::InputArray, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:112
	#[inline]
	fn match_(&self, _face_feature1: &dyn core::ToInputArray, _face_feature2: &dyn core::ToInputArray, dis_type: i32) -> Result<f64> {
		input_array_arg!(_face_feature1);
		input_array_arg!(_face_feature2);
		let ret = unsafe { sys::cv_FaceRecognizerSF_match_const_const__InputArrayR_const__InputArrayR_int(self.as_raw_FaceRecognizerSF(), _face_feature1.as_raw__InputArray(), _face_feature2.as_raw__InputArray(), dis_type) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait FaceRecognizerSF: crate::objdetect::FaceRecognizerSFConst {
	fn as_raw_mut_FaceRecognizerSF(&mut self) -> *mut c_void;

	/// Extracting face feature from aligned image
	/// ## Parameters
	/// * aligned_img: input aligned image
	/// * face_feature: output face feature
	// feature(cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:105
	#[inline]
	fn feature(&mut self, aligned_img: &dyn core::ToInputArray, face_feature: &mut dyn core::ToOutputArray) -> Result<()> {
		input_array_arg!(aligned_img);
		output_array_arg!(face_feature);
		let ret = unsafe { sys::cv_FaceRecognizerSF_feature_const__InputArrayR_const__OutputArrayR(self.as_raw_mut_FaceRecognizerSF(), aligned_img.as_raw__InputArray(), face_feature.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
}

impl dyn FaceRecognizerSF + '_ {
	/// Creates an instance of this class with given parameters
	/// ## Parameters
	/// * model: the path of the onnx model used for face recognition
	/// * config: the path to the config file for compability, which is not requested for ONNX models
	/// * backend_id: the id of backend
	/// * target_id: the id of target device
	/// 
	/// ## C++ default parameters
	/// * backend_id: 0
	/// * target_id: 0
	// create(const cv::String &, const cv::String &, int, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect/face.hpp:120
	#[inline]
	pub fn create(model: &str, config: &str, backend_id: i32, target_id: i32) -> Result<core::Ptr<dyn crate::objdetect::FaceRecognizerSF>> {
		extern_container_arg!(model);
		extern_container_arg!(config);
		let ret = unsafe { sys::cv_FaceRecognizerSF_create_const_StringR_const_StringR_int_int(model.opencv_as_extern(), config.opencv_as_extern(), backend_id, target_id) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::objdetect::FaceRecognizerSF>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}
/// Implementation of HOG (Histogram of Oriented Gradients) descriptor and object detector.
/// 
/// the HOG descriptor algorithm introduced by Navneet Dalal and Bill Triggs [Dalal2005](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Dalal2005) .
/// 
/// useful links:
/// 
/// https://hal.inria.fr/inria-00548512/document/
/// 
/// https://en.wikipedia.org/wiki/Histogram_of_oriented_gradients
/// 
/// https://software.intel.com/en-us/ipp-dev-reference-histogram-of-oriented-gradients-hog-descriptor
/// 
/// http://www.learnopencv.com/histogram-of-oriented-gradients
/// 
/// http://www.learnopencv.com/handwritten-digits-classification-an-opencv-c-python-tutorial
// HOGDescriptor /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:372
pub trait HOGDescriptorTraitConst {
	fn as_raw_HOGDescriptor(&self) -> *const c_void;

	/// Detection window size. Align to block size and block stride. Default value is Size(64,128).
	// winSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:587
	#[inline]
	fn win_size(&self) -> core::Size {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropWinSize_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Block size in pixels. Align to cell size. Default value is Size(16,16).
	// blockSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:590
	#[inline]
	fn block_size(&self) -> core::Size {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropBlockSize_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Block stride. It must be a multiple of cell size. Default value is Size(8,8).
	// blockStride /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:593
	#[inline]
	fn block_stride(&self) -> core::Size {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropBlockStride_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Cell size. Default value is Size(8,8).
	// cellSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:596
	#[inline]
	fn cell_size(&self) -> core::Size {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropCellSize_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Number of bins used in the calculation of histogram of gradients. Default value is 9.
	// nbins /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:599
	#[inline]
	fn nbins(&self) -> i32 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropNbins_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// not documented
	// derivAperture /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:602
	#[inline]
	fn deriv_aperture(&self) -> i32 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropDerivAperture_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Gaussian smoothing window parameter.
	// winSigma /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:605
	#[inline]
	fn win_sigma(&self) -> f64 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropWinSigma_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// histogramNormType
	// histogramNormType /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:608
	#[inline]
	fn histogram_norm_type(&self) -> crate::objdetect::HOGDescriptor_HistogramNormType {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropHistogramNormType_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// L2-Hys normalization method shrinkage.
	// L2HysThreshold /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:611
	#[inline]
	fn l2_hys_threshold(&self) -> f64 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropL2HysThreshold_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Flag to specify whether the gamma correction preprocessing is required or not.
	// gammaCorrection /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:614
	#[inline]
	fn gamma_correction(&self) -> bool {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropGammaCorrection_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// coefficients for the linear SVM classifier.
	// svmDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:617
	#[inline]
	fn svm_detector(&self) -> core::Vector<f32> {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropSvmDetector_const(self.as_raw_HOGDescriptor()) };
		let ret = unsafe { core::Vector::<f32>::opencv_from_extern(ret) };
		ret
	}
	
	/// coefficients for the linear SVM classifier used when OpenCL is enabled
	// oclSvmDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:620
	#[inline]
	fn ocl_svm_detector(&self) -> core::UMat {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropOclSvmDetector_const(self.as_raw_HOGDescriptor()) };
		let ret = unsafe { core::UMat::opencv_from_extern(ret) };
		ret
	}
	
	/// not documented
	// free_coef /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:623
	#[inline]
	fn free_coef(&self) -> f32 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropFree_coef_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Maximum number of detection window increases. Default value is 64
	// nlevels /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:626
	#[inline]
	fn nlevels(&self) -> i32 {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropNlevels_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Indicates signed gradient will be used or not
	// signedGradient /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:629
	#[inline]
	fn signed_gradient(&self) -> bool {
		let ret = unsafe { sys::cv_HOGDescriptor_getPropSignedGradient_const(self.as_raw_HOGDescriptor()) };
		ret
	}
	
	/// Returns the number of coefficients required for the classification.
	// getDescriptorSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:438
	#[inline]
	fn get_descriptor_size(&self) -> Result<size_t> {
		let ret = unsafe { sys::cv_HOGDescriptor_getDescriptorSize_const(self.as_raw_HOGDescriptor()) }.into_result()?;
		Ok(ret)
	}
	
	/// Checks if detector size equal to descriptor size.
	// checkDetectorSize() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:442
	#[inline]
	fn check_detector_size(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_HOGDescriptor_checkDetectorSize_const(self.as_raw_HOGDescriptor()) }.into_result()?;
		Ok(ret)
	}
	
	/// Returns winSigma value
	// getWinSigma() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:446
	#[inline]
	fn get_win_sigma(&self) -> Result<f64> {
		let ret = unsafe { sys::cv_HOGDescriptor_getWinSigma_const(self.as_raw_HOGDescriptor()) }.into_result()?;
		Ok(ret)
	}
	
	/// Stores HOGDescriptor parameters in a cv::FileStorage.
	/// ## Parameters
	/// * fs: File storage
	/// * objname: Object name
	// write(cv::FileStorage &, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:464
	#[inline]
	fn write(&self, fs: &mut core::FileStorage, objname: &str) -> Result<()> {
		extern_container_arg!(objname);
		let ret = unsafe { sys::cv_HOGDescriptor_write_const_FileStorageR_const_StringR(self.as_raw_HOGDescriptor(), fs.as_raw_mut_FileStorage(), objname.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// saves HOGDescriptor parameters and coefficients for the linear SVM classifier to a file
	/// ## Parameters
	/// * filename: File name
	/// * objname: Object name
	/// 
	/// ## C++ default parameters
	/// * objname: String()
	// save(const cv::String &, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:476
	#[inline]
	fn save(&self, filename: &str, objname: &str) -> Result<()> {
		extern_container_arg!(filename);
		extern_container_arg!(objname);
		let ret = unsafe { sys::cv_HOGDescriptor_save_const_const_StringR_const_StringR(self.as_raw_HOGDescriptor(), filename.opencv_as_extern(), objname.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// clones the HOGDescriptor
	/// ## Parameters
	/// * c: cloned HOGDescriptor
	// copyTo(cv::HOGDescriptor &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:481
	#[inline]
	fn copy_to(&self, c: &mut crate::objdetect::HOGDescriptor) -> Result<()> {
		let ret = unsafe { sys::cv_HOGDescriptor_copyTo_const_HOGDescriptorR(self.as_raw_HOGDescriptor(), c.as_raw_mut_HOGDescriptor()) }.into_result()?;
		Ok(ret)
	}
	
	/// @example samples/cpp/train_HOG.cpp
	/// /
	/// Computes HOG descriptors of given image.
	/// ## Parameters
	/// * img: Matrix of the type CV_8U containing an image where HOG features will be calculated.
	/// * descriptors: Matrix of the type CV_32F
	/// * winStride: Window stride. It must be a multiple of block stride.
	/// * padding: Padding
	/// * locations: Vector of Point
	/// 
	/// ## C++ default parameters
	/// * win_stride: Size()
	/// * padding: Size()
	/// * locations: std::vector<Point>()
	// compute(cv::InputArray, std::vector<float> &, cv::Size, cv::Size, const std::vector<Point> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:492
	#[inline]
	fn compute(&self, img: &dyn core::ToInputArray, descriptors: &mut core::Vector<f32>, win_stride: core::Size, padding: core::Size, locations: &core::Vector<core::Point>) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_compute_const_const__InputArrayR_vector_float_R_Size_Size_const_vector_Point_R(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), descriptors.as_raw_mut_VectorOff32(), win_stride.opencv_as_extern(), padding.opencv_as_extern(), locations.as_raw_VectorOfPoint()) }.into_result()?;
		Ok(ret)
	}
	
	/// Performs object detection without a multi-scale window.
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * foundLocations: Vector of point where each point contains left-top corner point of detected object boundaries.
	/// * weights: Vector that will contain confidence values for each detected object.
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane.
	/// Usually it is 0 and should be specified in the detector coefficients (as the last free coefficient).
	/// But if the free coefficient is omitted (which is allowed), you can specify it manually here.
	/// * winStride: Window stride. It must be a multiple of block stride.
	/// * padding: Padding
	/// * searchLocations: Vector of Point includes set of requested locations to be evaluated.
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * win_stride: Size()
	/// * padding: Size()
	/// * search_locations: std::vector<Point>()
	// detect(cv::InputArray, std::vector<Point> &, std::vector<double> &, double, cv::Size, cv::Size, const std::vector<Point> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:508
	#[inline]
	fn detect_weights(&self, img: &dyn core::ToInputArray, found_locations: &mut core::Vector<core::Point>, weights: &mut core::Vector<f64>, hit_threshold: f64, win_stride: core::Size, padding: core::Size, search_locations: &core::Vector<core::Point>) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detect_const_const__InputArrayR_vector_Point_R_vector_double_R_double_Size_Size_const_vector_Point_R(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), found_locations.as_raw_mut_VectorOfPoint(), weights.as_raw_mut_VectorOff64(), hit_threshold, win_stride.opencv_as_extern(), padding.opencv_as_extern(), search_locations.as_raw_VectorOfPoint()) }.into_result()?;
		Ok(ret)
	}
	
	/// Performs object detection without a multi-scale window.
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * foundLocations: Vector of point where each point contains left-top corner point of detected object boundaries.
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane.
	/// Usually it is 0 and should be specified in the detector coefficients (as the last free coefficient).
	/// But if the free coefficient is omitted (which is allowed), you can specify it manually here.
	/// * winStride: Window stride. It must be a multiple of block stride.
	/// * padding: Padding
	/// * searchLocations: Vector of Point includes locations to search.
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * win_stride: Size()
	/// * padding: Size()
	/// * search_locations: std::vector<Point>()
	// detect(cv::InputArray, std::vector<Point> &, double, cv::Size, cv::Size, const std::vector<Point> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:524
	#[inline]
	fn detect(&self, img: &dyn core::ToInputArray, found_locations: &mut core::Vector<core::Point>, hit_threshold: f64, win_stride: core::Size, padding: core::Size, search_locations: &core::Vector<core::Point>) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detect_const_const__InputArrayR_vector_Point_R_double_Size_Size_const_vector_Point_R(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), found_locations.as_raw_mut_VectorOfPoint(), hit_threshold, win_stride.opencv_as_extern(), padding.opencv_as_extern(), search_locations.as_raw_VectorOfPoint()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects objects of different sizes in the input image. The detected objects are returned as a list
	/// of rectangles.
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * foundLocations: Vector of rectangles where each rectangle contains the detected object.
	/// * foundWeights: Vector that will contain confidence values for each detected object.
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane.
	/// Usually it is 0 and should be specified in the detector coefficients (as the last free coefficient).
	/// But if the free coefficient is omitted (which is allowed), you can specify it manually here.
	/// * winStride: Window stride. It must be a multiple of block stride.
	/// * padding: Padding
	/// * scale: Coefficient of the detection window increase.
	/// * finalThreshold: Final threshold
	/// * useMeanshiftGrouping: indicates grouping algorithm
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * win_stride: Size()
	/// * padding: Size()
	/// * scale: 1.05
	/// * final_threshold: 2.0
	/// * use_meanshift_grouping: false
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, std::vector<double> &, double, cv::Size, cv::Size, double, double, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:543
	#[inline]
	fn detect_multi_scale_weights(&self, img: &dyn core::ToInputArray, found_locations: &mut core::Vector<core::Rect>, found_weights: &mut core::Vector<f64>, hit_threshold: f64, win_stride: core::Size, padding: core::Size, scale: f64, final_threshold: f64, use_meanshift_grouping: bool) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detectMultiScale_const_const__InputArrayR_vector_Rect_R_vector_double_R_double_Size_Size_double_double_bool(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), found_locations.as_raw_mut_VectorOfRect(), found_weights.as_raw_mut_VectorOff64(), hit_threshold, win_stride.opencv_as_extern(), padding.opencv_as_extern(), scale, final_threshold, use_meanshift_grouping) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects objects of different sizes in the input image. The detected objects are returned as a list
	/// of rectangles.
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * foundLocations: Vector of rectangles where each rectangle contains the detected object.
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane.
	/// Usually it is 0 and should be specified in the detector coefficients (as the last free coefficient).
	/// But if the free coefficient is omitted (which is allowed), you can specify it manually here.
	/// * winStride: Window stride. It must be a multiple of block stride.
	/// * padding: Padding
	/// * scale: Coefficient of the detection window increase.
	/// * finalThreshold: Final threshold
	/// * useMeanshiftGrouping: indicates grouping algorithm
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * win_stride: Size()
	/// * padding: Size()
	/// * scale: 1.05
	/// * final_threshold: 2.0
	/// * use_meanshift_grouping: false
	// detectMultiScale(cv::InputArray, std::vector<Rect> &, double, cv::Size, cv::Size, double, double, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:561
	#[inline]
	fn detect_multi_scale(&self, img: &dyn core::ToInputArray, found_locations: &mut core::Vector<core::Rect>, hit_threshold: f64, win_stride: core::Size, padding: core::Size, scale: f64, final_threshold: f64, use_meanshift_grouping: bool) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detectMultiScale_const_const__InputArrayR_vector_Rect_R_double_Size_Size_double_double_bool(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), found_locations.as_raw_mut_VectorOfRect(), hit_threshold, win_stride.opencv_as_extern(), padding.opencv_as_extern(), scale, final_threshold, use_meanshift_grouping) }.into_result()?;
		Ok(ret)
	}
	
	///  Computes gradients and quantized gradient orientations.
	/// ## Parameters
	/// * img: Matrix contains the image to be computed
	/// * grad: Matrix of type CV_32FC2 contains computed gradients
	/// * angleOfs: Matrix of type CV_8UC2 contains quantized gradient orientations
	/// * paddingTL: Padding from top-left
	/// * paddingBR: Padding from bottom-right
	/// 
	/// ## C++ default parameters
	/// * padding_tl: Size()
	/// * padding_br: Size()
	// computeGradient(cv::InputArray, cv::InputOutputArray, cv::InputOutputArray, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:573
	#[inline]
	fn compute_gradient(&self, img: &dyn core::ToInputArray, grad: &mut dyn core::ToInputOutputArray, angle_ofs: &mut dyn core::ToInputOutputArray, padding_tl: core::Size, padding_br: core::Size) -> Result<()> {
		input_array_arg!(img);
		input_output_array_arg!(grad);
		input_output_array_arg!(angle_ofs);
		let ret = unsafe { sys::cv_HOGDescriptor_computeGradient_const_const__InputArrayR_const__InputOutputArrayR_const__InputOutputArrayR_Size_Size(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), grad.as_raw__InputOutputArray(), angle_ofs.as_raw__InputOutputArray(), padding_tl.opencv_as_extern(), padding_br.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// evaluate specified ROI and return confidence value for each location
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * locations: Vector of Point
	/// * foundLocations: Vector of Point where each Point is detected object's top-left point.
	/// * confidences: confidences
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane. Usually
	/// it is 0 and should be specified in the detector coefficients (as the last free coefficient). But if
	/// the free coefficient is omitted (which is allowed), you can specify it manually here
	/// * winStride: winStride
	/// * padding: padding
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * win_stride: Size()
	/// * padding: Size()
	// detectROI(cv::InputArray, const std::vector<cv::Point> &, std::vector<cv::Point> &, std::vector<double> &, double, cv::Size, cv::Size) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:642
	#[inline]
	fn detect_roi(&self, img: &dyn core::ToInputArray, locations: &core::Vector<core::Point>, found_locations: &mut core::Vector<core::Point>, confidences: &mut core::Vector<f64>, hit_threshold: f64, win_stride: core::Size, padding: core::Size) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detectROI_const_const__InputArrayR_const_vector_Point_R_vector_Point_R_vector_double_R_double_Size_Size(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), locations.as_raw_VectorOfPoint(), found_locations.as_raw_mut_VectorOfPoint(), confidences.as_raw_mut_VectorOff64(), hit_threshold, win_stride.opencv_as_extern(), padding.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
	/// evaluate specified ROI and return confidence value for each location in multiple scales
	/// ## Parameters
	/// * img: Matrix of the type CV_8U or CV_8UC3 containing an image where objects are detected.
	/// * foundLocations: Vector of rectangles where each rectangle contains the detected object.
	/// * locations: Vector of DetectionROI
	/// * hitThreshold: Threshold for the distance between features and SVM classifying plane. Usually it is 0 and should be specified
	/// in the detector coefficients (as the last free coefficient). But if the free coefficient is omitted (which is allowed), you can specify it manually here.
	/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a group of rectangles to retain it.
	/// 
	/// ## C++ default parameters
	/// * hit_threshold: 0
	/// * group_threshold: 0
	// detectMultiScaleROI(cv::InputArray, std::vector<cv::Rect> &, std::vector<DetectionROI> &, double, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:655
	#[inline]
	fn detect_multi_scale_roi(&self, img: &dyn core::ToInputArray, found_locations: &mut core::Vector<core::Rect>, locations: &mut core::Vector<crate::objdetect::DetectionROI>, hit_threshold: f64, group_threshold: i32) -> Result<()> {
		input_array_arg!(img);
		let ret = unsafe { sys::cv_HOGDescriptor_detectMultiScaleROI_const_const__InputArrayR_vector_Rect_R_vector_DetectionROI_R_double_int(self.as_raw_HOGDescriptor(), img.as_raw__InputArray(), found_locations.as_raw_mut_VectorOfRect(), locations.as_raw_mut_VectorOfDetectionROI(), hit_threshold, group_threshold) }.into_result()?;
		Ok(ret)
	}
	
	/// Groups the object candidate rectangles.
	/// ## Parameters
	/// * rectList: Input/output vector of rectangles. Output vector includes retained and grouped rectangles. (The Python list is not modified in place.)
	/// * weights: Input/output vector of weights of rectangles. Output vector includes weights of retained and grouped rectangles. (The Python list is not modified in place.)
	/// * groupThreshold: Minimum possible number of rectangles minus 1. The threshold is used in a group of rectangles to retain it.
	/// * eps: Relative difference between sides of the rectangles to merge them into a group.
	// groupRectangles(std::vector<cv::Rect> &, std::vector<double> &, int, double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:667
	#[inline]
	fn group_rectangles(&self, rect_list: &mut core::Vector<core::Rect>, weights: &mut core::Vector<f64>, group_threshold: i32, eps: f64) -> Result<()> {
		let ret = unsafe { sys::cv_HOGDescriptor_groupRectangles_const_vector_Rect_R_vector_double_R_int_double(self.as_raw_HOGDescriptor(), rect_list.as_raw_mut_VectorOfRect(), weights.as_raw_mut_VectorOff64(), group_threshold, eps) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait HOGDescriptorTrait: crate::objdetect::HOGDescriptorTraitConst {
	fn as_raw_mut_HOGDescriptor(&mut self) -> *mut c_void;

	/// Detection window size. Align to block size and block stride. Default value is Size(64,128).
	// winSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:587
	#[inline]
	fn set_win_size(&mut self, val: core::Size) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropWinSize_Size(self.as_raw_mut_HOGDescriptor(), val.opencv_as_extern()) };
		ret
	}
	
	/// Block size in pixels. Align to cell size. Default value is Size(16,16).
	// blockSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:590
	#[inline]
	fn set_block_size(&mut self, val: core::Size) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropBlockSize_Size(self.as_raw_mut_HOGDescriptor(), val.opencv_as_extern()) };
		ret
	}
	
	/// Block stride. It must be a multiple of cell size. Default value is Size(8,8).
	// blockStride /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:593
	#[inline]
	fn set_block_stride(&mut self, val: core::Size) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropBlockStride_Size(self.as_raw_mut_HOGDescriptor(), val.opencv_as_extern()) };
		ret
	}
	
	/// Cell size. Default value is Size(8,8).
	// cellSize /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:596
	#[inline]
	fn set_cell_size(&mut self, val: core::Size) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropCellSize_Size(self.as_raw_mut_HOGDescriptor(), val.opencv_as_extern()) };
		ret
	}
	
	/// Number of bins used in the calculation of histogram of gradients. Default value is 9.
	// nbins /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:599
	#[inline]
	fn set_nbins(&mut self, val: i32) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropNbins_int(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// not documented
	// derivAperture /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:602
	#[inline]
	fn set_deriv_aperture(&mut self, val: i32) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropDerivAperture_int(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// Gaussian smoothing window parameter.
	// winSigma /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:605
	#[inline]
	fn set_win_sigma(&mut self, val: f64) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropWinSigma_double(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// histogramNormType
	// histogramNormType /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:608
	#[inline]
	fn set_histogram_norm_type(&mut self, val: crate::objdetect::HOGDescriptor_HistogramNormType) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropHistogramNormType_HistogramNormType(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// L2-Hys normalization method shrinkage.
	// L2HysThreshold /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:611
	#[inline]
	fn set_l2_hys_threshold(&mut self, val: f64) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropL2HysThreshold_double(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// Flag to specify whether the gamma correction preprocessing is required or not.
	// gammaCorrection /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:614
	#[inline]
	fn set_gamma_correction(&mut self, val: bool) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropGammaCorrection_bool(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// coefficients for the linear SVM classifier.
	// svmDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:617
	#[inline]
	fn set_svm_detector_vec(&mut self, mut val: core::Vector<f32>) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropSvmDetector_vector_float_(self.as_raw_mut_HOGDescriptor(), val.as_raw_mut_VectorOff32()) };
		ret
	}
	
	/// coefficients for the linear SVM classifier used when OpenCL is enabled
	// oclSvmDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:620
	#[inline]
	fn set_ocl_svm_detector(&mut self, mut val: core::UMat) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropOclSvmDetector_UMat(self.as_raw_mut_HOGDescriptor(), val.as_raw_mut_UMat()) };
		ret
	}
	
	/// not documented
	// free_coef /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:623
	#[inline]
	fn set_free_coef(&mut self, val: f32) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropFree_coef_float(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// Maximum number of detection window increases. Default value is 64
	// nlevels /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:626
	#[inline]
	fn set_nlevels(&mut self, val: i32) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropNlevels_int(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// Indicates signed gradient will be used or not
	// signedGradient /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:629
	#[inline]
	fn set_signed_gradient(&mut self, val: bool) {
		let ret = unsafe { sys::cv_HOGDescriptor_setPropSignedGradient_bool(self.as_raw_mut_HOGDescriptor(), val) };
		ret
	}
	
	/// @example samples/cpp/peopledetect.cpp
	/// /
	/// Sets coefficients for the linear SVM classifier.
	/// ## Parameters
	/// * svmdetector: coefficients for the linear SVM classifier.
	// setSVMDetector(cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:453
	#[inline]
	fn set_svm_detector(&mut self, svmdetector: &dyn core::ToInputArray) -> Result<()> {
		input_array_arg!(svmdetector);
		let ret = unsafe { sys::cv_HOGDescriptor_setSVMDetector_const__InputArrayR(self.as_raw_mut_HOGDescriptor(), svmdetector.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Reads HOGDescriptor parameters from a cv::FileNode.
	/// ## Parameters
	/// * fn: File node
	// read(cv::FileNode &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:458
	#[inline]
	fn read(&mut self, fn_: &mut core::FileNode) -> Result<bool> {
		let ret = unsafe { sys::cv_HOGDescriptor_read_FileNodeR(self.as_raw_mut_HOGDescriptor(), fn_.as_raw_mut_FileNode()) }.into_result()?;
		Ok(ret)
	}
	
	/// loads HOGDescriptor parameters and coefficients for the linear SVM classifier from a file.
	/// ## Parameters
	/// * filename: Path of the file to read.
	/// * objname: The optional name of the node to read (if empty, the first top-level node will be used).
	/// 
	/// ## C++ default parameters
	/// * objname: String()
	// load(const cv::String &, const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:470
	#[inline]
	fn load(&mut self, filename: &str, objname: &str) -> Result<bool> {
		extern_container_arg!(filename);
		extern_container_arg!(objname);
		let ret = unsafe { sys::cv_HOGDescriptor_load_const_StringR_const_StringR(self.as_raw_mut_HOGDescriptor(), filename.opencv_as_extern(), objname.opencv_as_extern()) }.into_result()?;
		Ok(ret)
	}
	
}

/// Implementation of HOG (Histogram of Oriented Gradients) descriptor and object detector.
/// 
/// the HOG descriptor algorithm introduced by Navneet Dalal and Bill Triggs [Dalal2005](https://docs.opencv.org/4.5.4/d0/de3/citelist.html#CITEREF_Dalal2005) .
/// 
/// useful links:
/// 
/// https://hal.inria.fr/inria-00548512/document/
/// 
/// https://en.wikipedia.org/wiki/Histogram_of_oriented_gradients
/// 
/// https://software.intel.com/en-us/ipp-dev-reference-histogram-of-oriented-gradients-hog-descriptor
/// 
/// http://www.learnopencv.com/histogram-of-oriented-gradients
/// 
/// http://www.learnopencv.com/handwritten-digits-classification-an-opencv-c-python-tutorial
// HOGDescriptor /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:372
pub struct HOGDescriptor {
	ptr: *mut c_void
}

opencv_type_boxed! { HOGDescriptor }

impl Drop for HOGDescriptor {
	fn drop(&mut self) {
		extern "C" { fn cv_HOGDescriptor_delete(instance: *mut c_void); }
		unsafe { cv_HOGDescriptor_delete(self.as_raw_mut_HOGDescriptor()) };
	}
}

unsafe impl Send for HOGDescriptor {}

impl crate::objdetect::HOGDescriptorTraitConst for HOGDescriptor {
	#[inline] fn as_raw_HOGDescriptor(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::HOGDescriptorTrait for HOGDescriptor {
	#[inline] fn as_raw_mut_HOGDescriptor(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl HOGDescriptor {
	/// Creates the HOG descriptor and detector with default params.
	/// 
	/// aqual to HOGDescriptor(Size(64,128), Size(16,16), Size(8,8), Size(8,8), 9 )
	// HOGDescriptor() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:385
	#[inline]
	pub fn default() -> Result<crate::objdetect::HOGDescriptor> {
		let ret = unsafe { sys::cv_HOGDescriptor_HOGDescriptor() }.into_result()?;
		let ret = unsafe { crate::objdetect::HOGDescriptor::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Creates the HOG descriptor and detector with default params.
	/// 
	/// aqual to HOGDescriptor(Size(64,128), Size(16,16), Size(8,8), Size(8,8), 9 )
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## Parameters
	/// * _winSize: sets winSize with given value.
	/// * _blockSize: sets blockSize with given value.
	/// * _blockStride: sets blockStride with given value.
	/// * _cellSize: sets cellSize with given value.
	/// * _nbins: sets nbins with given value.
	/// * _derivAperture: sets derivAperture with given value.
	/// * _winSigma: sets winSigma with given value.
	/// * _histogramNormType: sets histogramNormType with given value.
	/// * _L2HysThreshold: sets L2HysThreshold with given value.
	/// * _gammaCorrection: sets gammaCorrection with given value.
	/// * _nlevels: sets nlevels with given value.
	/// * _signedGradient: sets signedGradient with given value.
	/// 
	/// ## C++ default parameters
	/// * _deriv_aperture: 1
	/// * _win_sigma: -1
	/// * _histogram_norm_type: HOGDescriptor::L2Hys
	/// * _l2_hys_threshold: 0.2
	/// * _gamma_correction: false
	/// * _nlevels: HOGDescriptor::DEFAULT_NLEVELS
	/// * _signed_gradient: false
	// HOGDescriptor(cv::Size, cv::Size, cv::Size, cv::Size, int, int, double, HOGDescriptor::HistogramNormType, double, bool, int, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:405
	#[inline]
	pub fn new(_win_size: core::Size, _block_size: core::Size, _block_stride: core::Size, _cell_size: core::Size, _nbins: i32, _deriv_aperture: i32, _win_sigma: f64, _histogram_norm_type: crate::objdetect::HOGDescriptor_HistogramNormType, _l2_hys_threshold: f64, _gamma_correction: bool, _nlevels: i32, _signed_gradient: bool) -> Result<crate::objdetect::HOGDescriptor> {
		let ret = unsafe { sys::cv_HOGDescriptor_HOGDescriptor_Size_Size_Size_Size_int_int_double_HistogramNormType_double_bool_int_bool(_win_size.opencv_as_extern(), _block_size.opencv_as_extern(), _block_stride.opencv_as_extern(), _cell_size.opencv_as_extern(), _nbins, _deriv_aperture, _win_sigma, _histogram_norm_type, _l2_hys_threshold, _gamma_correction, _nlevels, _signed_gradient) }.into_result()?;
		let ret = unsafe { crate::objdetect::HOGDescriptor::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Creates the HOG descriptor and detector with default params.
	/// 
	/// aqual to HOGDescriptor(Size(64,128), Size(16,16), Size(8,8), Size(8,8), 9 )
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## Parameters
	/// * filename: The file name containing HOGDescriptor properties and coefficients for the linear SVM classifier.
	// HOGDescriptor(const cv::String &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:419
	#[inline]
	pub fn new_from_file(filename: &str) -> Result<crate::objdetect::HOGDescriptor> {
		extern_container_arg!(filename);
		let ret = unsafe { sys::cv_HOGDescriptor_HOGDescriptor_const_StringR(filename.opencv_as_extern()) }.into_result()?;
		let ret = unsafe { crate::objdetect::HOGDescriptor::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Creates the HOG descriptor and detector with default params.
	/// 
	/// aqual to HOGDescriptor(Size(64,128), Size(16,16), Size(8,8), Size(8,8), 9 )
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## Parameters
	/// * d: the HOGDescriptor which cloned to create a new one.
	// HOGDescriptor(const cv::HOGDescriptor &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:427
	#[inline]
	pub fn copy(d: &crate::objdetect::HOGDescriptor) -> Result<crate::objdetect::HOGDescriptor> {
		let ret = unsafe { sys::cv_HOGDescriptor_HOGDescriptor_const_HOGDescriptorR(d.as_raw_HOGDescriptor()) }.into_result()?;
		let ret = unsafe { crate::objdetect::HOGDescriptor::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Returns coefficients of the classifier trained for people detection (for 64x128 windows).
	// getDefaultPeopleDetector() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:578
	#[inline]
	pub fn get_default_people_detector() -> Result<core::Vector<f32>> {
		let ret = unsafe { sys::cv_HOGDescriptor_getDefaultPeopleDetector() }.into_result()?;
		let ret = unsafe { core::Vector::<f32>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// @example samples/tapi/hog.cpp
	/// /
	/// Returns coefficients of the classifier trained for people detection (for 48x96 windows).
	// getDaimlerPeopleDetector() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:584
	#[inline]
	pub fn get_daimler_people_detector() -> Result<core::Vector<f32>> {
		let ret = unsafe { sys::cv_HOGDescriptor_getDaimlerPeopleDetector() }.into_result()?;
		let ret = unsafe { core::Vector::<f32>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

// QRCodeDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:670
pub trait QRCodeDetectorTraitConst {
	fn as_raw_QRCodeDetector(&self) -> *const c_void;

	/// Detects QR code in image and returns the quadrangle containing the code.
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing (or not) QR code.
	/// * points: Output vector of vertices of the minimum-area quadrangle containing the code.
	// detect(cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:691
	#[inline]
	fn detect(&self, img: &dyn core::ToInputArray, points: &mut dyn core::ToOutputArray) -> Result<bool> {
		input_array_arg!(img);
		output_array_arg!(points);
		let ret = unsafe { sys::cv_QRCodeDetector_detect_const_const__InputArrayR_const__OutputArrayR(self.as_raw_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects QR codes in image and returns the vector of the quadrangles containing the codes.
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing (or not) QR codes.
	/// * points: Output vector of vector of vertices of the minimum-area quadrangle containing the codes.
	// detectMulti(cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:734
	#[inline]
	fn detect_multi(&self, img: &dyn core::ToInputArray, points: &mut dyn core::ToOutputArray) -> Result<bool> {
		input_array_arg!(img);
		output_array_arg!(points);
		let ret = unsafe { sys::cv_QRCodeDetector_detectMulti_const_const__InputArrayR_const__OutputArrayR(self.as_raw_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Decodes QR codes in image once it's found by the detect() method.
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR codes.
	/// * decoded_info: UTF8-encoded output vector of string or empty vector of string if the codes cannot be decoded.
	/// * points: vector of Quadrangle vertices found by detect() method (or some other algorithm).
	/// * straight_qrcode: The optional output vector of images containing rectified and binarized QR codes
	/// 
	/// ## C++ default parameters
	/// * straight_qrcode: noArray()
	// decodeMulti(cv::InputArray, cv::InputArray, std::vector<std::string> &, cv::OutputArrayOfArrays) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:743
	#[inline]
	fn decode_multi(&self, img: &dyn core::ToInputArray, points: &dyn core::ToInputArray, decoded_info: &mut core::Vector<String>, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<bool> {
		input_array_arg!(img);
		input_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_decodeMulti_const_const__InputArrayR_const__InputArrayR_vector_string_R_const__OutputArrayR(self.as_raw_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__InputArray(), decoded_info.as_raw_mut_VectorOfString(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Both detects and decodes QR codes
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR codes.
	/// * decoded_info: UTF8-encoded output vector of string or empty vector of string if the codes cannot be decoded.
	/// * points: optional output vector of vertices of the found QR code quadrangles. Will be empty if not found.
	/// * straight_qrcode: The optional output vector of images containing rectified and binarized QR codes
	/// 
	/// ## C++ default parameters
	/// * points: noArray()
	/// * straight_qrcode: noArray()
	// detectAndDecodeMulti(cv::InputArray, std::vector<std::string> &, cv::OutputArray, cv::OutputArrayOfArrays) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:756
	#[inline]
	fn detect_and_decode_multi(&self, img: &dyn core::ToInputArray, decoded_info: &mut core::Vector<String>, points: &mut dyn core::ToOutputArray, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<bool> {
		input_array_arg!(img);
		output_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_detectAndDecodeMulti_const_const__InputArrayR_vector_string_R_const__OutputArrayR_const__OutputArrayR(self.as_raw_QRCodeDetector(), img.as_raw__InputArray(), decoded_info.as_raw_mut_VectorOfString(), points.as_raw__OutputArray(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait QRCodeDetectorTrait: crate::objdetect::QRCodeDetectorTraitConst {
	fn as_raw_mut_QRCodeDetector(&mut self) -> *mut c_void;

	/// sets the epsilon used during the horizontal scan of QR code stop marker detection.
	/// ## Parameters
	/// * epsX: Epsilon neighborhood, which allows you to determine the horizontal pattern
	/// of the scheme 1:1:3:1:1 according to QR code standard.
	// setEpsX(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:680
	#[inline]
	fn set_eps_x(&mut self, eps_x: f64) -> Result<()> {
		let ret = unsafe { sys::cv_QRCodeDetector_setEpsX_double(self.as_raw_mut_QRCodeDetector(), eps_x) }.into_result()?;
		Ok(ret)
	}
	
	/// sets the epsilon used during the vertical scan of QR code stop marker detection.
	/// ## Parameters
	/// * epsY: Epsilon neighborhood, which allows you to determine the vertical pattern
	/// of the scheme 1:1:3:1:1 according to QR code standard.
	// setEpsY(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:685
	#[inline]
	fn set_eps_y(&mut self, eps_y: f64) -> Result<()> {
		let ret = unsafe { sys::cv_QRCodeDetector_setEpsY_double(self.as_raw_mut_QRCodeDetector(), eps_y) }.into_result()?;
		Ok(ret)
	}
	
	/// Decodes QR code in image once it's found by the detect() method.
	/// 
	/// Returns UTF8-encoded output string or empty string if the code cannot be decoded.
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR code.
	/// * points: Quadrangle vertices found by detect() method (or some other algorithm).
	/// * straight_qrcode: The optional output image containing rectified and binarized QR code
	/// 
	/// ## C++ default parameters
	/// * straight_qrcode: noArray()
	// decode(cv::InputArray, cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:700
	#[inline]
	fn decode(&mut self, img: &dyn core::ToInputArray, points: &dyn core::ToInputArray, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<String> {
		input_array_arg!(img);
		input_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_decode_const__InputArrayR_const__InputArrayR_const__OutputArrayR(self.as_raw_mut_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__InputArray(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		let ret = unsafe { String::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Decodes QR code on a curved surface in image once it's found by the detect() method.
	/// 
	/// Returns UTF8-encoded output string or empty string if the code cannot be decoded.
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR code.
	/// * points: Quadrangle vertices found by detect() method (or some other algorithm).
	/// * straight_qrcode: The optional output image containing rectified and binarized QR code
	/// 
	/// ## C++ default parameters
	/// * straight_qrcode: noArray()
	// decodeCurved(cv::InputArray, cv::InputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:709
	#[inline]
	fn decode_curved(&mut self, img: &dyn core::ToInputArray, points: &dyn core::ToInputArray, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<String> {
		input_array_arg!(img);
		input_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_decodeCurved_const__InputArrayR_const__InputArrayR_const__OutputArrayR(self.as_raw_mut_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__InputArray(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		let ret = unsafe { String::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Both detects and decodes QR code
	/// 
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR code.
	/// * points: optional output array of vertices of the found QR code quadrangle. Will be empty if not found.
	/// * straight_qrcode: The optional output image containing rectified and binarized QR code
	/// 
	/// ## C++ default parameters
	/// * points: noArray()
	/// * straight_qrcode: noArray()
	// detectAndDecode(cv::InputArray, cv::OutputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:717
	#[inline]
	fn detect_and_decode(&mut self, img: &dyn core::ToInputArray, points: &mut dyn core::ToOutputArray, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<String> {
		input_array_arg!(img);
		output_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_detectAndDecode_const__InputArrayR_const__OutputArrayR_const__OutputArrayR(self.as_raw_mut_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__OutputArray(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		let ret = unsafe { String::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Both detects and decodes QR code on a curved surface
	/// 
	/// ## Parameters
	/// * img: grayscale or color (BGR) image containing QR code.
	/// * points: optional output array of vertices of the found QR code quadrangle. Will be empty if not found.
	/// * straight_qrcode: The optional output image containing rectified and binarized QR code
	/// 
	/// ## C++ default parameters
	/// * points: noArray()
	/// * straight_qrcode: noArray()
	// detectAndDecodeCurved(cv::InputArray, cv::OutputArray, cv::OutputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:726
	#[inline]
	fn detect_and_decode_curved(&mut self, img: &dyn core::ToInputArray, points: &mut dyn core::ToOutputArray, straight_qrcode: &mut dyn core::ToOutputArray) -> Result<String> {
		input_array_arg!(img);
		output_array_arg!(points);
		output_array_arg!(straight_qrcode);
		let ret = unsafe { sys::cv_QRCodeDetector_detectAndDecodeCurved_const__InputArrayR_const__OutputArrayR_const__OutputArrayR(self.as_raw_mut_QRCodeDetector(), img.as_raw__InputArray(), points.as_raw__OutputArray(), straight_qrcode.as_raw__OutputArray()) }.into_result()?;
		let ret = unsafe { String::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

// QRCodeDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:670
pub struct QRCodeDetector {
	ptr: *mut c_void
}

opencv_type_boxed! { QRCodeDetector }

impl Drop for QRCodeDetector {
	fn drop(&mut self) {
		extern "C" { fn cv_QRCodeDetector_delete(instance: *mut c_void); }
		unsafe { cv_QRCodeDetector_delete(self.as_raw_mut_QRCodeDetector()) };
	}
}

unsafe impl Send for QRCodeDetector {}

impl crate::objdetect::QRCodeDetectorTraitConst for QRCodeDetector {
	#[inline] fn as_raw_QRCodeDetector(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::QRCodeDetectorTrait for QRCodeDetector {
	#[inline] fn as_raw_mut_QRCodeDetector(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl QRCodeDetector {
	// QRCodeDetector() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:673
	#[inline]
	pub fn default() -> Result<crate::objdetect::QRCodeDetector> {
		let ret = unsafe { sys::cv_QRCodeDetector_QRCodeDetector() }.into_result()?;
		let ret = unsafe { crate::objdetect::QRCodeDetector::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}

/// class for grouping object candidates, detected by Cascade Classifier, HOG etc.
/// instance of the class is to be passed to cv::partition (see cxoperations.hpp)
// SimilarRects /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:121
pub trait SimilarRectsTraitConst {
	fn as_raw_SimilarRects(&self) -> *const c_void;

	// eps /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:133
	#[inline]
	fn eps(&self) -> f64 {
		let ret = unsafe { sys::cv_SimilarRects_getPropEps_const(self.as_raw_SimilarRects()) };
		ret
	}
	
}

pub trait SimilarRectsTrait: crate::objdetect::SimilarRectsTraitConst {
	fn as_raw_mut_SimilarRects(&mut self) -> *mut c_void;

	// eps /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:133
	#[inline]
	fn set_eps(&mut self, val: f64) {
		let ret = unsafe { sys::cv_SimilarRects_setPropEps_double(self.as_raw_mut_SimilarRects(), val) };
		ret
	}
	
}

/// class for grouping object candidates, detected by Cascade Classifier, HOG etc.
/// instance of the class is to be passed to cv::partition (see cxoperations.hpp)
// SimilarRects /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:121
pub struct SimilarRects {
	ptr: *mut c_void
}

opencv_type_boxed! { SimilarRects }

impl Drop for SimilarRects {
	fn drop(&mut self) {
		extern "C" { fn cv_SimilarRects_delete(instance: *mut c_void); }
		unsafe { cv_SimilarRects_delete(self.as_raw_mut_SimilarRects()) };
	}
}

unsafe impl Send for SimilarRects {}

impl crate::objdetect::SimilarRectsTraitConst for SimilarRects {
	#[inline] fn as_raw_SimilarRects(&self) -> *const c_void { self.as_raw() }
}

impl crate::objdetect::SimilarRectsTrait for SimilarRects {
	#[inline] fn as_raw_mut_SimilarRects(&mut self) -> *mut c_void { self.as_raw_mut() }
}

impl SimilarRects {
	// SimilarRects(double) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/objdetect.hpp:124
	#[inline]
	pub fn new(_eps: f64) -> Result<crate::objdetect::SimilarRects> {
		let ret = unsafe { sys::cv_SimilarRects_SimilarRects_double(_eps) }.into_result()?;
		let ret = unsafe { crate::objdetect::SimilarRects::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}
