#![allow(
	unused_parens,
	clippy::excessive_precision,
	clippy::missing_safety_doc,
	clippy::not_unsafe_ptr_arg_deref,
	clippy::should_implement_trait,
	clippy::too_many_arguments,
	clippy::unused_unit,
)]
//! # Feature Detection and Description
use crate::{mod_prelude::*, core, sys, types};
pub mod prelude {
	pub use { super::CUDA_DescriptorMatcherConst, super::CUDA_DescriptorMatcher, super::CUDA_Feature2DAsyncConst, super::CUDA_Feature2DAsync, super::CUDA_FastFeatureDetectorConst, super::CUDA_FastFeatureDetector, super::CUDA_ORBConst, super::CUDA_ORB };
}

/// Abstract base class for matching keypoint descriptors.
/// 
/// It has two groups of match methods: for matching descriptors of an image with another image or with
/// an image set.
// DescriptorMatcher /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:75
pub trait CUDA_DescriptorMatcherConst: core::AlgorithmTraitConst {
	fn as_raw_CUDA_DescriptorMatcher(&self) -> *const c_void;

	/// Returns true if the descriptor matcher supports masking permissible matches.
	// isMaskSupported() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:100
	#[inline]
	fn is_mask_supported(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_isMaskSupported_const(self.as_raw_CUDA_DescriptorMatcher()) }.into_result()?;
		Ok(ret)
	}
	
	/// Returns a constant link to the train descriptor collection.
	// getTrainDescriptors() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:117
	#[inline]
	fn get_train_descriptors(&self) -> Result<core::Vector<core::GpuMat>> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_getTrainDescriptors_const(self.as_raw_CUDA_DescriptorMatcher()) }.into_result()?;
		let ret = unsafe { core::Vector::<core::GpuMat>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
	/// Returns true if there are no train descriptors in the collection.
	// empty() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:125
	#[inline]
	fn empty(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_empty_const(self.as_raw_CUDA_DescriptorMatcher()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait CUDA_DescriptorMatcher: core::AlgorithmTrait + crate::cudafeatures2d::CUDA_DescriptorMatcherConst {
	fn as_raw_mut_CUDA_DescriptorMatcher(&mut self) -> *mut c_void;

	/// Adds descriptors to train a descriptor collection.
	/// 
	/// If the collection is not empty, the new descriptors are added to existing train descriptors.
	/// 
	/// ## Parameters
	/// * descriptors: Descriptors to add. Each descriptors[i] is a set of descriptors from the same
	/// train image.
	// add(const std::vector<GpuMat> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:113
	#[inline]
	fn add(&mut self, descriptors: &core::Vector<core::GpuMat>) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_add_const_vector_GpuMat_R(self.as_raw_mut_CUDA_DescriptorMatcher(), descriptors.as_raw_VectorOfGpuMat()) }.into_result()?;
		Ok(ret)
	}
	
	/// Clears the train descriptor collection.
	// clear() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:121
	#[inline]
	fn clear(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_clear(self.as_raw_mut_CUDA_DescriptorMatcher()) }.into_result()?;
		Ok(ret)
	}
	
	/// Trains a descriptor matcher.
	/// 
	/// Trains a descriptor matcher (for example, the flann index). In all methods to match, the method
	/// train() is run every time before matching.
	// train() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:132
	#[inline]
	fn train(&mut self) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_train(self.as_raw_mut_CUDA_DescriptorMatcher()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the best match for each descriptor from a query set (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches. If a query descriptor is masked out in mask , no match is added for this
	/// descriptor. So, matches size may be smaller than the query descriptors count.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// 
	/// In the first variant of this method, the train descriptors are passed as an input argument. In the
	/// second variant of the method, train descriptors collection that was set by DescriptorMatcher::add is
	/// used. Optional mask (or masks) can be passed to specify which query and training descriptors can be
	/// matched. Namely, queryDescriptors[i] can be matched with trainDescriptors[j] only if
	/// mask.at\<uchar\>(i,j) is non-zero.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	// match(cv::InputArray, cv::InputArray, std::vector<DMatch> &, cv::InputArray) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:154
	#[inline]
	fn match_(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::DMatch>, mask: &dyn core::ToInputArray) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_match_const__InputArrayR_const__InputArrayR_vector_DMatch_R_const__InputArrayR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfDMatch(), mask.as_raw__InputArray()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the best match for each descriptor from a query set (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches. If a query descriptor is masked out in mask , no match is added for this
	/// descriptor. So, matches size may be smaller than the query descriptors count.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// 
	/// In the first variant of this method, the train descriptors are passed as an input argument. In the
	/// second variant of the method, train descriptors collection that was set by DescriptorMatcher::add is
	/// used. Optional mask (or masks) can be passed to specify which query and training descriptors can be
	/// matched. Namely, queryDescriptors[i] can be matched with trainDescriptors[j] only if
	/// mask.at\<uchar\>(i,j) is non-zero.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	// match(cv::InputArray, std::vector<DMatch> &, const std::vector<GpuMat> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:160
	#[inline]
	fn match__1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::DMatch>, masks: &core::Vector<core::GpuMat>) -> Result<()> {
		input_array_arg!(query_descriptors);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_match_const__InputArrayR_vector_DMatch_R_const_vector_GpuMat_R(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfDMatch(), masks.as_raw_VectorOfGpuMat()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the best match for each descriptor from a query set (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::matchConvert method to retrieve results in standard representation.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// In the first variant of this method, the train descriptors are passed as an input argument. In the
	/// second variant of the method, train descriptors collection that was set by DescriptorMatcher::add is
	/// used. Optional mask (or masks) can be passed to specify which query and training descriptors can be
	/// matched. Namely, queryDescriptors[i] can be matched with trainDescriptors[j] only if
	/// mask.at\<uchar\>(i,j) is non-zero.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * stream: Stream::Null()
	// matchAsync(cv::InputArray, cv::InputArray, cv::OutputArray, cv::InputArray, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:181
	#[inline]
	fn match_async(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, mask: &dyn core::ToInputArray, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		output_array_arg!(matches);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_matchAsync_const__InputArrayR_const__InputArrayR_const__OutputArrayR_const__InputArrayR_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), mask.as_raw__InputArray(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the best match for each descriptor from a query set (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::matchConvert method to retrieve results in standard representation.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// In the first variant of this method, the train descriptors are passed as an input argument. In the
	/// second variant of the method, train descriptors collection that was set by DescriptorMatcher::add is
	/// used. Optional mask (or masks) can be passed to specify which query and training descriptors can be
	/// matched. Namely, queryDescriptors[i] can be matched with trainDescriptors[j] only if
	/// mask.at\<uchar\>(i,j) is non-zero.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	/// * stream: Stream::Null()
	// matchAsync(cv::InputArray, cv::OutputArray, const std::vector<GpuMat> &, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:188
	#[inline]
	fn match_async_1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, masks: &core::Vector<core::GpuMat>, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		output_array_arg!(matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_matchAsync_const__InputArrayR_const__OutputArrayR_const_vector_GpuMat_R_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), masks.as_raw_VectorOfGpuMat(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Converts matches array from internal representation to standard matches vector.
	/// 
	/// The method is supposed to be used with DescriptorMatcher::matchAsync to get final result.
	/// Call this method only after DescriptorMatcher::matchAsync is completed (ie. after synchronization).
	/// 
	/// ## Parameters
	/// * gpu_matches: Matches, returned from DescriptorMatcher::matchAsync.
	/// * matches: Vector of DMatch objects.
	// matchConvert(cv::InputArray, std::vector<DMatch> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:201
	#[inline]
	fn match_convert(&mut self, gpu_matches: &dyn core::ToInputArray, matches: &mut core::Vector<core::DMatch>) -> Result<()> {
		input_array_arg!(gpu_matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_matchConvert_const__InputArrayR_vector_DMatch_R(self.as_raw_mut_CUDA_DescriptorMatcher(), gpu_matches.as_raw__InputArray(), matches.as_raw_mut_VectorOfDMatch()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the k best matches for each descriptor from a query set (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches. Each matches[i] is k or less matches for the same query descriptor.
	/// * k: Count of best matches found per each query descriptor or less if a query descriptor has
	/// less than k possible matches in total.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// These extended variants of DescriptorMatcher::match methods find several best matches for each query
	/// descriptor. The matches are returned in the distance increasing order. See DescriptorMatcher::match
	/// for the details about query and train descriptors.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * compact_result: false
	// knnMatch(cv::InputArray, cv::InputArray, std::vector<std::vector<DMatch>> &, int, cv::InputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:226
	#[inline]
	fn knn_match(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, k: i32, mask: &dyn core::ToInputArray, compact_result: bool) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_knnMatch_const__InputArrayR_const__InputArrayR_vector_vector_DMatch__R_int_const__InputArrayR_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), k, mask.as_raw__InputArray(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the k best matches for each descriptor from a query set (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches. Each matches[i] is k or less matches for the same query descriptor.
	/// * k: Count of best matches found per each query descriptor or less if a query descriptor has
	/// less than k possible matches in total.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// These extended variants of DescriptorMatcher::match methods find several best matches for each query
	/// descriptor. The matches are returned in the distance increasing order. See DescriptorMatcher::match
	/// for the details about query and train descriptors.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	/// * compact_result: false
	// knnMatch(cv::InputArray, std::vector<std::vector<DMatch>> &, int, const std::vector<GpuMat> &, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:234
	#[inline]
	fn knn_match_1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, k: i32, masks: &core::Vector<core::GpuMat>, compact_result: bool) -> Result<()> {
		input_array_arg!(query_descriptors);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_knnMatch_const__InputArrayR_vector_vector_DMatch__R_int_const_vector_GpuMat_R_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), k, masks.as_raw_VectorOfGpuMat(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the k best matches for each descriptor from a query set (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::knnMatchConvert method to retrieve results in standard representation.
	/// * k: Count of best matches found per each query descriptor or less if a query descriptor has
	/// less than k possible matches in total.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// These extended variants of DescriptorMatcher::matchAsync methods find several best matches for each query
	/// descriptor. The matches are returned in the distance increasing order. See DescriptorMatcher::matchAsync
	/// for the details about query and train descriptors.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * stream: Stream::Null()
	// knnMatchAsync(cv::InputArray, cv::InputArray, cv::OutputArray, int, cv::InputArray, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:257
	#[inline]
	fn knn_match_async(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, k: i32, mask: &dyn core::ToInputArray, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		output_array_arg!(matches);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_knnMatchAsync_const__InputArrayR_const__InputArrayR_const__OutputArrayR_int_const__InputArrayR_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), k, mask.as_raw__InputArray(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Finds the k best matches for each descriptor from a query set (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::knnMatchConvert method to retrieve results in standard representation.
	/// * k: Count of best matches found per each query descriptor or less if a query descriptor has
	/// less than k possible matches in total.
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// These extended variants of DescriptorMatcher::matchAsync methods find several best matches for each query
	/// descriptor. The matches are returned in the distance increasing order. See DescriptorMatcher::matchAsync
	/// for the details about query and train descriptors.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	/// * stream: Stream::Null()
	// knnMatchAsync(cv::InputArray, cv::OutputArray, int, const std::vector<GpuMat> &, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:265
	#[inline]
	fn knn_match_async_1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, k: i32, masks: &core::Vector<core::GpuMat>, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		output_array_arg!(matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_knnMatchAsync_const__InputArrayR_const__OutputArrayR_int_const_vector_GpuMat_R_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), k, masks.as_raw_VectorOfGpuMat(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Converts matches array from internal representation to standard matches vector.
	/// 
	/// The method is supposed to be used with DescriptorMatcher::knnMatchAsync to get final result.
	/// Call this method only after DescriptorMatcher::knnMatchAsync is completed (ie. after synchronization).
	/// 
	/// ## Parameters
	/// * gpu_matches: Matches, returned from DescriptorMatcher::knnMatchAsync.
	/// * matches: Vector of DMatch objects.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// ## C++ default parameters
	/// * compact_result: false
	// knnMatchConvert(cv::InputArray, std::vector<std::vector<DMatch>> &, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:282
	#[inline]
	fn knn_match_convert(&mut self, gpu_matches: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, compact_result: bool) -> Result<()> {
		input_array_arg!(gpu_matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_knnMatchConvert_const__InputArrayR_vector_vector_DMatch__R_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), gpu_matches.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
	/// For each query descriptor, finds the training descriptors not farther than the specified distance (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Found matches.
	/// * maxDistance: Threshold for the distance between matched descriptors. Distance means here
	/// metric distance (e.g. Hamming distance), not the distance between coordinates (which is measured
	/// in Pixels)!
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// For each query descriptor, the methods find such training descriptors that the distance between the
	/// query descriptor and the training descriptor is equal or smaller than maxDistance. Found matches are
	/// returned in the distance increasing order.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * compact_result: false
	// radiusMatch(cv::InputArray, cv::InputArray, std::vector<std::vector<DMatch>> &, float, cv::InputArray, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:309
	#[inline]
	fn radius_match(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, max_distance: f32, mask: &dyn core::ToInputArray, compact_result: bool) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_radiusMatch_const__InputArrayR_const__InputArrayR_vector_vector_DMatch__R_float_const__InputArrayR_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), max_distance, mask.as_raw__InputArray(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
	/// For each query descriptor, finds the training descriptors not farther than the specified distance (blocking version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Found matches.
	/// * maxDistance: Threshold for the distance between matched descriptors. Distance means here
	/// metric distance (e.g. Hamming distance), not the distance between coordinates (which is measured
	/// in Pixels)!
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// For each query descriptor, the methods find such training descriptors that the distance between the
	/// query descriptor and the training descriptor is equal or smaller than maxDistance. Found matches are
	/// returned in the distance increasing order.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	/// * compact_result: false
	// radiusMatch(cv::InputArray, std::vector<std::vector<DMatch>> &, float, const std::vector<GpuMat> &, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:317
	#[inline]
	fn radius_match_1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, max_distance: f32, masks: &core::Vector<core::GpuMat>, compact_result: bool) -> Result<()> {
		input_array_arg!(query_descriptors);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_radiusMatch_const__InputArrayR_vector_vector_DMatch__R_float_const_vector_GpuMat_R_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), max_distance, masks.as_raw_VectorOfGpuMat(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
	/// For each query descriptor, finds the training descriptors not farther than the specified distance (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::radiusMatchConvert method to retrieve results in standard representation.
	/// * maxDistance: Threshold for the distance between matched descriptors. Distance means here
	/// metric distance (e.g. Hamming distance), not the distance between coordinates (which is measured
	/// in Pixels)!
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// For each query descriptor, the methods find such training descriptors that the distance between the
	/// query descriptor and the training descriptor is equal or smaller than maxDistance. Found matches are
	/// returned in the distance increasing order.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * stream: Stream::Null()
	// radiusMatchAsync(cv::InputArray, cv::InputArray, cv::OutputArray, float, cv::InputArray, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:341
	#[inline]
	fn radius_match_async(&mut self, query_descriptors: &dyn core::ToInputArray, train_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, max_distance: f32, mask: &dyn core::ToInputArray, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		input_array_arg!(train_descriptors);
		output_array_arg!(matches);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_radiusMatchAsync_const__InputArrayR_const__InputArrayR_const__OutputArrayR_float_const__InputArrayR_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), train_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), max_distance, mask.as_raw__InputArray(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// For each query descriptor, finds the training descriptors not farther than the specified distance (asynchronous version).
	/// 
	/// ## Parameters
	/// * queryDescriptors: Query set of descriptors.
	/// * trainDescriptors: Train set of descriptors. This set is not added to the train descriptors
	/// collection stored in the class object.
	/// * matches: Matches array stored in GPU memory. Internal representation is not defined.
	/// Use DescriptorMatcher::radiusMatchConvert method to retrieve results in standard representation.
	/// * maxDistance: Threshold for the distance between matched descriptors. Distance means here
	/// metric distance (e.g. Hamming distance), not the distance between coordinates (which is measured
	/// in Pixels)!
	/// * mask: Mask specifying permissible matches between an input query and train matrices of
	/// descriptors.
	/// * stream: CUDA stream.
	/// 
	/// For each query descriptor, the methods find such training descriptors that the distance between the
	/// query descriptor and the training descriptor is equal or smaller than maxDistance. Found matches are
	/// returned in the distance increasing order.
	/// 
	/// ## Overloaded parameters
	/// 
	/// ## C++ default parameters
	/// * masks: std::vector<GpuMat>()
	/// * stream: Stream::Null()
	// radiusMatchAsync(cv::InputArray, cv::OutputArray, float, const std::vector<GpuMat> &, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:349
	#[inline]
	fn radius_match_async_1(&mut self, query_descriptors: &dyn core::ToInputArray, matches: &mut dyn core::ToOutputArray, max_distance: f32, masks: &core::Vector<core::GpuMat>, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(query_descriptors);
		output_array_arg!(matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_radiusMatchAsync_const__InputArrayR_const__OutputArrayR_float_const_vector_GpuMat_R_StreamR(self.as_raw_mut_CUDA_DescriptorMatcher(), query_descriptors.as_raw__InputArray(), matches.as_raw__OutputArray(), max_distance, masks.as_raw_VectorOfGpuMat(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Converts matches array from internal representation to standard matches vector.
	/// 
	/// The method is supposed to be used with DescriptorMatcher::radiusMatchAsync to get final result.
	/// Call this method only after DescriptorMatcher::radiusMatchAsync is completed (ie. after synchronization).
	/// 
	/// ## Parameters
	/// * gpu_matches: Matches, returned from DescriptorMatcher::radiusMatchAsync.
	/// * matches: Vector of DMatch objects.
	/// * compactResult: Parameter used when the mask (or masks) is not empty. If compactResult is
	/// false, the matches vector has the same size as queryDescriptors rows. If compactResult is true,
	/// the matches vector does not contain matches for fully masked-out query descriptors.
	/// 
	/// ## C++ default parameters
	/// * compact_result: false
	// radiusMatchConvert(cv::InputArray, std::vector<std::vector<DMatch>> &, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:366
	#[inline]
	fn radius_match_convert(&mut self, gpu_matches: &dyn core::ToInputArray, matches: &mut core::Vector<core::Vector<core::DMatch>>, compact_result: bool) -> Result<()> {
		input_array_arg!(gpu_matches);
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_radiusMatchConvert_const__InputArrayR_vector_vector_DMatch__R_bool(self.as_raw_mut_CUDA_DescriptorMatcher(), gpu_matches.as_raw__InputArray(), matches.as_raw_mut_VectorOfVectorOfDMatch(), compact_result) }.into_result()?;
		Ok(ret)
	}
	
}

impl dyn CUDA_DescriptorMatcher + '_ {
	/// Brute-force descriptor matcher.
	/// 
	/// For each descriptor in the first set, this matcher finds the closest descriptor in the second set
	/// by trying each one. This descriptor matcher supports masking permissible matches of descriptor
	/// sets.
	/// 
	/// ## Parameters
	/// * normType: One of NORM_L1, NORM_L2, NORM_HAMMING. L1 and L2 norms are
	/// preferable choices for SIFT and SURF descriptors, NORM_HAMMING should be used with ORB, BRISK and
	/// BRIEF).
	/// 
	/// ## C++ default parameters
	/// * norm_type: cv::NORM_L2
	// createBFMatcher(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:92
	#[inline]
	pub fn create_bf_matcher(norm_type: i32) -> Result<core::Ptr<dyn crate::cudafeatures2d::CUDA_DescriptorMatcher>> {
		let ret = unsafe { sys::cv_cuda_DescriptorMatcher_createBFMatcher_int(norm_type) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::cudafeatures2d::CUDA_DescriptorMatcher>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}
/// Wrapping class for feature detection using the FAST method.
// FastFeatureDetector /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:426
pub trait CUDA_FastFeatureDetectorConst: crate::cudafeatures2d::CUDA_Feature2DAsyncConst {
	fn as_raw_CUDA_FastFeatureDetector(&self) -> *const c_void;

	// getMaxNumPoints() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:441
	#[inline]
	fn get_max_num_points(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_cuda_FastFeatureDetector_getMaxNumPoints_const(self.as_raw_CUDA_FastFeatureDetector()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait CUDA_FastFeatureDetector: crate::cudafeatures2d::CUDA_FastFeatureDetectorConst + crate::cudafeatures2d::CUDA_Feature2DAsync {
	fn as_raw_mut_CUDA_FastFeatureDetector(&mut self) -> *mut c_void;

	// setThreshold(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:438
	#[inline]
	fn set_threshold(&mut self, threshold: i32) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_FastFeatureDetector_setThreshold_int(self.as_raw_mut_CUDA_FastFeatureDetector(), threshold) }.into_result()?;
		Ok(ret)
	}
	
	// setMaxNumPoints(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:440
	#[inline]
	fn set_max_num_points(&mut self, max_npoints: i32) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_FastFeatureDetector_setMaxNumPoints_int(self.as_raw_mut_CUDA_FastFeatureDetector(), max_npoints) }.into_result()?;
		Ok(ret)
	}
	
}

impl dyn CUDA_FastFeatureDetector + '_ {
	// LOCATION_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:429
	pub const LOCATION_ROW: i32 = 0;
	// RESPONSE_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:430
	pub const RESPONSE_ROW: i32 = 1;
	// ROWS_COUNT /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:431
	pub const ROWS_COUNT: i32 = 2;
	// FEATURE_SIZE /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:432
	pub const FEATURE_SIZE: i32 = 7;
	/// ## C++ default parameters
	/// * threshold: 10
	/// * nonmax_suppression: true
	/// * typ: cv::FastFeatureDetector::TYPE_9_16
	/// * max_npoints: 5000
	// create(int, bool, int, int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:434
	#[inline]
	pub fn create(threshold: i32, nonmax_suppression: bool, typ: i32, max_npoints: i32) -> Result<core::Ptr<dyn crate::cudafeatures2d::CUDA_FastFeatureDetector>> {
		let ret = unsafe { sys::cv_cuda_FastFeatureDetector_create_int_bool_int_int(threshold, nonmax_suppression, typ, max_npoints) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::cudafeatures2d::CUDA_FastFeatureDetector>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}
/// Abstract base class for CUDA asynchronous 2D image feature detectors and descriptor extractors.
// Feature2DAsync /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:377
pub trait CUDA_Feature2DAsyncConst: crate::features2d::Feature2DTraitConst {
	fn as_raw_CUDA_Feature2DAsync(&self) -> *const c_void;

}

pub trait CUDA_Feature2DAsync: crate::cudafeatures2d::CUDA_Feature2DAsyncConst + crate::features2d::Feature2DTrait {
	fn as_raw_mut_CUDA_Feature2DAsync(&mut self) -> *mut c_void;

	/// Detects keypoints in an image.
	/// 
	/// ## Parameters
	/// * image: Image.
	/// * keypoints: The detected keypoints.
	/// * mask: Mask specifying where to look for keypoints (optional). It must be a 8-bit integer
	/// matrix with non-zero values in the region of interest.
	/// * stream: CUDA stream.
	/// 
	/// ## C++ default parameters
	/// * mask: noArray()
	/// * stream: Stream::Null()
	// detectAsync(cv::InputArray, cv::OutputArray, cv::InputArray, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:390
	#[inline]
	fn detect_async(&mut self, image: &dyn core::ToInputArray, keypoints: &mut dyn core::ToOutputArray, mask: &dyn core::ToInputArray, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(image);
		output_array_arg!(keypoints);
		input_array_arg!(mask);
		let ret = unsafe { sys::cv_cuda_Feature2DAsync_detectAsync_const__InputArrayR_const__OutputArrayR_const__InputArrayR_StreamR(self.as_raw_mut_CUDA_Feature2DAsync(), image.as_raw__InputArray(), keypoints.as_raw__OutputArray(), mask.as_raw__InputArray(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Computes the descriptors for a set of keypoints detected in an image.
	/// 
	/// ## Parameters
	/// * image: Image.
	/// * keypoints: Input collection of keypoints.
	/// * descriptors: Computed descriptors. Row j is the descriptor for j-th keypoint.
	/// * stream: CUDA stream.
	/// 
	/// ## C++ default parameters
	/// * stream: Stream::Null()
	// computeAsync(cv::InputArray, cv::OutputArray, cv::OutputArray, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:402
	#[inline]
	fn compute_async(&mut self, image: &dyn core::ToInputArray, keypoints: &mut dyn core::ToOutputArray, descriptors: &mut dyn core::ToOutputArray, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(image);
		output_array_arg!(keypoints);
		output_array_arg!(descriptors);
		let ret = unsafe { sys::cv_cuda_Feature2DAsync_computeAsync_const__InputArrayR_const__OutputArrayR_const__OutputArrayR_StreamR(self.as_raw_mut_CUDA_Feature2DAsync(), image.as_raw__InputArray(), keypoints.as_raw__OutputArray(), descriptors.as_raw__OutputArray(), stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Detects keypoints and computes the descriptors.
	/// 
	/// ## C++ default parameters
	/// * use_provided_keypoints: false
	/// * stream: Stream::Null()
	// detectAndComputeAsync(cv::InputArray, cv::InputArray, cv::OutputArray, cv::OutputArray, bool, cv::cuda::Stream &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:408
	#[inline]
	fn detect_and_compute_async(&mut self, image: &dyn core::ToInputArray, mask: &dyn core::ToInputArray, keypoints: &mut dyn core::ToOutputArray, descriptors: &mut dyn core::ToOutputArray, use_provided_keypoints: bool, stream: &mut core::Stream) -> Result<()> {
		input_array_arg!(image);
		input_array_arg!(mask);
		output_array_arg!(keypoints);
		output_array_arg!(descriptors);
		let ret = unsafe { sys::cv_cuda_Feature2DAsync_detectAndComputeAsync_const__InputArrayR_const__InputArrayR_const__OutputArrayR_const__OutputArrayR_bool_StreamR(self.as_raw_mut_CUDA_Feature2DAsync(), image.as_raw__InputArray(), mask.as_raw__InputArray(), keypoints.as_raw__OutputArray(), descriptors.as_raw__OutputArray(), use_provided_keypoints, stream.as_raw_mut_Stream()) }.into_result()?;
		Ok(ret)
	}
	
	/// Converts keypoints array from internal representation to standard vector.
	// convert(cv::InputArray, std::vector<KeyPoint> &) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:416
	#[inline]
	fn convert(&mut self, gpu_keypoints: &dyn core::ToInputArray, keypoints: &mut core::Vector<core::KeyPoint>) -> Result<()> {
		input_array_arg!(gpu_keypoints);
		let ret = unsafe { sys::cv_cuda_Feature2DAsync_convert_const__InputArrayR_vector_KeyPoint_R(self.as_raw_mut_CUDA_Feature2DAsync(), gpu_keypoints.as_raw__InputArray(), keypoints.as_raw_mut_VectorOfKeyPoint()) }.into_result()?;
		Ok(ret)
	}
	
}

/// Class implementing the ORB (*oriented BRIEF*) keypoint detector and descriptor extractor
/// ## See also
/// cv::ORB
// ORB /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:452
pub trait CUDA_ORBConst: crate::cudafeatures2d::CUDA_Feature2DAsyncConst {
	fn as_raw_CUDA_ORB(&self) -> *const c_void;

	// getBlurForDescriptor() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:476
	#[inline]
	fn get_blur_for_descriptor(&self) -> Result<bool> {
		let ret = unsafe { sys::cv_cuda_ORB_getBlurForDescriptor_const(self.as_raw_CUDA_ORB()) }.into_result()?;
		Ok(ret)
	}
	
	// getFastThreshold() /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:479
	#[inline]
	fn get_fast_threshold(&self) -> Result<i32> {
		let ret = unsafe { sys::cv_cuda_ORB_getFastThreshold_const(self.as_raw_CUDA_ORB()) }.into_result()?;
		Ok(ret)
	}
	
}

pub trait CUDA_ORB: crate::cudafeatures2d::CUDA_Feature2DAsync + crate::cudafeatures2d::CUDA_ORBConst {
	fn as_raw_mut_CUDA_ORB(&mut self) -> *mut c_void;

	/// if true, image will be blurred before descriptors calculation
	// setBlurForDescriptor(bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:475
	#[inline]
	fn set_blur_for_descriptor(&mut self, blur_for_descriptor: bool) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_ORB_setBlurForDescriptor_bool(self.as_raw_mut_CUDA_ORB(), blur_for_descriptor) }.into_result()?;
		Ok(ret)
	}
	
	// setFastThreshold(int) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:478
	#[inline]
	fn set_fast_threshold(&mut self, fast_threshold: i32) -> Result<()> {
		let ret = unsafe { sys::cv_cuda_ORB_setFastThreshold_int(self.as_raw_mut_CUDA_ORB(), fast_threshold) }.into_result()?;
		Ok(ret)
	}
	
}

impl dyn CUDA_ORB + '_ {
	// X_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:455
	pub const X_ROW: i32 = 0;
	// Y_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:456
	pub const Y_ROW: i32 = 1;
	// RESPONSE_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:457
	pub const RESPONSE_ROW: i32 = 2;
	// ANGLE_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:458
	pub const ANGLE_ROW: i32 = 3;
	// OCTAVE_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:459
	pub const OCTAVE_ROW: i32 = 4;
	// SIZE_ROW /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:460
	pub const SIZE_ROW: i32 = 5;
	// ROWS_COUNT /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:461
	pub const ROWS_COUNT: i32 = 6;
	/// ## C++ default parameters
	/// * nfeatures: 500
	/// * scale_factor: 1.2f
	/// * nlevels: 8
	/// * edge_threshold: 31
	/// * first_level: 0
	/// * wta_k: 2
	/// * score_type: cv::ORB::HARRIS_SCORE
	/// * patch_size: 31
	/// * fast_threshold: 20
	/// * blur_for_descriptor: false
	// create(int, float, int, int, int, int, int, int, int, bool) /home/pro/projects/opencv-lib/opencv-4.5/install/include/opencv4/opencv2/cudafeatures2d.hpp:463
	#[inline]
	pub fn create(nfeatures: i32, scale_factor: f32, nlevels: i32, edge_threshold: i32, first_level: i32, wta_k: i32, score_type: i32, patch_size: i32, fast_threshold: i32, blur_for_descriptor: bool) -> Result<core::Ptr<dyn crate::cudafeatures2d::CUDA_ORB>> {
		let ret = unsafe { sys::cv_cuda_ORB_create_int_float_int_int_int_int_int_int_int_bool(nfeatures, scale_factor, nlevels, edge_threshold, first_level, wta_k, score_type, patch_size, fast_threshold, blur_for_descriptor) }.into_result()?;
		let ret = unsafe { core::Ptr::<dyn crate::cudafeatures2d::CUDA_ORB>::opencv_from_extern(ret) };
		Ok(ret)
	}
	
}