use schemars::{JsonSchema, JsonSchema_repr};
use serde::{Deserialize, Serialize};
use serde_repr::*;

#[derive(Serialize, Deserialize, Debug, JsonSchema)]
/// Should be handled as additional information for e.g. debugging.
/// Contains information if an object is valid.
/// If not, a status code and maybe an additional message inform about what is wrong.
pub struct Status {
    /// Describes status in short.
    /// Default is 200, no errors detected.
    pub code: StatusCode,

    /// Optional message string contains additional information to better understand the status code.
    pub message: Option<String>,
}

#[derive(Serialize_repr, Deserialize_repr, Debug, JsonSchema_repr, PartialEq)]
#[repr(u16)]
/// Defines status codes on product and scene level.
/// 200 is default. All other codes than 200 are errors.
/// Codes <= 1000 are reserved except the predefined ones.
/// Codes > 1000 are user defined error codes.
pub enum StatusCode {
    /// No errors detected.
    Ok = 200,

    /// Not sufficient defined or invalid parameters.
    BadRequest = 400,

    /// Data for further processing not found.
    NotFound = 404,

    /// One or more errors occurred.
    Conflict = 409,

    /// An error during data processing occurred.
    InternalServerError = 500,

    /// Service for data processing/gathering not available.
    ServiceUnavailable = 503,
}
