use anyhow::Result;
use argh::FromArgs;
use serde::Deserialize;
use std::fs;
use std::path::PathBuf;
use std::str::FromStr;
use ureq::{self, OrAnyStatus, Response};
use url::Url;

#[derive(FromArgs, Debug)]
/// OpenAPI fuzzer
struct Args {
    /// path to finding generated by OpenAPI fuzzer
    #[argh(positional)]
    file: PathBuf,

    /// extra header
    #[argh(option, short = 'H')]
    header: Vec<Header>,
}

#[derive(Debug)]
pub struct Header(String, String);

impl FromStr for Header {
    type Err = String;

    fn from_str(s: &str) -> Result<Self, Self::Err> {
        let parts = s.splitn(2, ':').collect::<Vec<_>>();
        if parts.len() != 2 {
            return Err("invalid header format".to_string());
        }
        Ok(Header {
            0: parts[0].to_string().to_lowercase(),
            1: parts[1].to_string(),
        })
    }
}

impl From<Header> for (String, String) {
    fn from(val: Header) -> Self {
        (val.0, val.1)
    }
}

#[derive(Debug, Deserialize)]
struct Payload {
    url: Url,
    method: String,
    path: String,
    query_params: Vec<(String, String)>,
    path_params: Vec<(String, String)>,
    headers: Vec<(String, String)>,
    body: Vec<serde_json::Value>,
}

#[derive(Debug, Deserialize)]
struct ResultFile {
    payload: Payload,
}

impl Payload {
    fn send(&mut self, extra_headers: Vec<(String, String)>) -> Result<Response> {
        let mut path_with_params = self.path.to_owned();
        for (name, value) in self.path_params.iter() {
            path_with_params = path_with_params.replace(&format!("{{{}}}", name), value);
        }
        let mut request = ureq::request_url(
            &self.method,
            &self.url.join(path_with_params.trim_start_matches('/'))?,
        );

        for (param, value) in self.query_params.iter() {
            request = request.query(param, value)
        }

        for (name, value) in extra_headers {
            let index = self
                .headers
                .iter()
                .position(|(header_name, _)| header_name.to_lowercase() == name);
            match index {
                Some(i) => self.headers[i] = (name, value),
                None => self.headers.push((name, value)),
            }
        }

        for (header, value) in self.headers.iter() {
            request = request.set(header, value)
        }

        if !self.body.is_empty() {
            Ok(request.send_json(self.body[0].clone()).or_any_status()?)
        } else {
            request.call().or_any_status().map_err(|e| e.into())
        }
    }
}

fn main() -> Result<()> {
    let args: Args = argh::from_env();
    let mut result: ResultFile = serde_json::from_str(&fs::read_to_string(&args.file)?)?;

    let resp = result
        .payload
        .send(args.header.into_iter().map(|h| h.into()).collect())?;

    println!("{:?}", &resp);
    if resp.content_type().to_lowercase().contains("json") {
        println!(
            "{}",
            serde_json::to_string_pretty(&resp.into_json::<serde_json::Value>()?)?
        )
    } else {
        println!("{}", &resp.into_string()?);
    }
    Ok(())
}
